"""
Init command to configure environment variables for SpotifySaver.
"""

from pathlib import Path
import click


@click.command("init", short_help="Initialize SpotifySaver configuration")
def init():
    """Initialize SpotifySaver configuration by setting up environment variables."""

    # Create the .spotify-saver directory in user's home
    config_dir = Path.home() / ".spotify-saver"
    config_dir.mkdir(exist_ok=True)

    env_file = config_dir / ".env"

    click.echo("🎵 SpotifySaver Configuration Setup")
    click.echo("=" * 40)

    # Prompt for environment variables
    spotify_client_id = click.prompt("Enter your Spotify Client ID", type=str)
    spotify_client_secret = click.prompt(
        "Enter your Spotify Client Secret", type=str, hide_input=True
    )
    spotify_redirect_uri = click.prompt(
        "Enter your Spotify Redirect URI",
        default="http://localhost:8888/callback",
        type=str,
    )

    # Optional variables with defaults
    download_path = click.prompt(
        "Enter download path",
        default=str(Path.home() / "Downloads" / "SpotifySaver"),
        type=str,
    )

    youtube_dl_cookies_path = click.prompt(
        "Enter YouTube DL cookies path (optional)",
        default="cookies.txt",
        type=str,
    )

    api_port = click.prompt(
        "Enter API port",
        default=8000,
        type=int,
    )
    api_host = click.prompt(
        "Enter API host",
        default="0.0.0.0",
        type=str,
    )

    # Write to .env file
    env_content = f"""# SpotifySaver Configuration generated by init command
SPOTIFY_REDIRECT_URI={spotify_redirect_uri}
SPOTIFY_CLIENT_ID={spotify_client_id}
SPOTIFY_CLIENT_SECRET={spotify_client_secret}
SPOTIFYSAVER_OUTPUT_DIR={download_path}
YTDLP_COOKIES_PATH={youtube_dl_cookies_path or ""}
API_PORT={api_port}
API_HOST={api_host}
"""

    with open(env_file, "w", encoding="utf-8") as f:
        f.write(env_content)
    # Set restrictive permissions (chmod 600) to the .env file
    import os
    os.chmod(env_file, 0o600)

    click.echo(f"\n✅ Configuration saved to: {env_file}")
    click.echo("You can now run the SpotifySaver commands with your configured settings.")
    click.echo("You can run init again to update settings or create a new configuration.")
