#!/usr/bin/python3
# -*- coding: utf-8 -*-

"""
OpenCSP
=======

This module contains several tools to work with data that comes from the
2 Flies Tracker software found in OpenCSP.

"""

from os import path
import numpy as np
import pandas as pd

# from ..tools import OneEuroFilter


class OpenCSPOutput(object):

    """
    OpenCSP Output
    ==============

    Use this class to work with all data contained in the CSV file
    generated by the OpenCSP '2 Flies Tracker (no arena)' software.

    You can access both flies data through either the `male` or the
    `female` attribute. These are pandas DataFrame objects.

    :param str csvfile: CSV file generated by the tracking software

    :param int fbi: female blob index (must be 0 or 1)

    # FIXME remove the following
    :param int fps: video FPS (frames per second)
    :param float pxmm: number of pixels corresponding to 1 mm
    """

    # These values are the default for the current output format
    SKIPROWS = [0, 1, 2, 4, 5]
    SKIPFOOTER = 2
    USELESS_COLUMNS = [
        # 'frame',
        # 'seconds',
        # 'blob index',
        # 'x',
        # 'y',
        # 'rotation',
        # 'head x',
        # 'head y',
        # 'tail x',
        # 'tail y',
        # 'velocity',
        # 'velocity.1',
        # 'velocity x',
        # 'velocity y',
        # 'velocity x.1',
        # 'velocity y.1',
        # 'angular velocity',
        # 'angular velocity.1',
        # 'translational angle',
        # 'angle between flies',
        # 'line angle between flies',
        # 'Distance between blobs'
    ]

    def __init__(self, csvfile=None):

        self.female = None
        self.male = None

        self.fps = None  # NOT USED FIXME
        self.pxmm = None  # NOT USED FIXME

        if csvfile is not None:
            self.load_from_csv(csvfile)

    def load_from_csv(self, csvfile, **kwargs):
        """Read CSV file resulting from OpenCSP 2 Flies Tracker
        (no arena) into a Pandas DataFrame object.

        :param str csvfile: CSV file generated by the tracking software
        Other `kwargs` are passed to `pd.read_csv`.
        """
        self.filename = path.basename(csvfile)

        self.raw_data = pd.read_csv(
            csvfile,
            # Some initial rows are useless
            skiprows=kwargs.pop('skiprows', self.SKIPROWS),
            engine='python',
            skipfooter=kwargs.pop('skipfooter', self.SKIPFOOTER),
            **kwargs
        )

    def remove_unnecessary_columns(self, column_names=[]):
        """Remove columns from the `raw_data` DataFrame loaded from
        the CSV. If the `column_names` list passed is empty, this will
        remove the USELESS_COLUMNS.

        :param list column_names: column names to remove
        """

        if not column_names:
            column_names = self.USELESS_COLUMNS

        for column_name in column_names:
            self.raw_data.drop(column_name, axis=1, inplace=True)

    def fix_column_names(self):
        """Fix column names.
        For all column names in `raw_data`, strip the strings and
        replace the spaces for underscores.
        """
        self.raw_data.rename(
            columns={
                name: name.strip().replace(" ", "_")
                for name in self.raw_data.columns[:]
            },
            inplace=True
        )

    def split_by_gender(self, gender_col, index_col='frame', fbi=0, mbi=1):
        """Split `raw_data` by gender (coded in the column
        'blob index').
        The splitting is followed by a resetting of the index to the
        frame number.

        :param int fbi: female blob index
        :param int fbi: male blob index
        """
        self.female_blob_index = fbi
        self.male_blob_index = mbi

        self.female = self.raw_data[
            self.raw_data[gender_col] == self.female_blob_index
        ]
        if self.female.empty:
            raise ValueError("fbi provided could not originate a DataFrame")

        self.male = self.raw_data[
            self.raw_data[gender_col] == self.male_blob_index
        ]
        if self.male.empty:
            raise ValueError("mbi provided could not originate a DataFrame")

        self.female = self.female.set_index(index_col)
        self.male = self.male.set_index(index_col)

        # Fill in the missing frames where tracker failed
        # assuming it is the same for the male...
        new_index = pd.Index(np.arange(self.female.first_valid_index(),
                                       self.female.last_valid_index(),
                                       1, dtype=int))
        self.female = self.female.reindex(new_index)
        self.male = self.male.reindex(new_index)

        # reindex loses the name, so we must set it again
        self.female.index.name = index_col
        self.male.index.name = index_col

        # Drop the column that identifies male and female cause it is redundant
        self.female.drop(gender_col, axis=1, inplace=True)
        self.male.drop(gender_col, axis=1, inplace=True)
