;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; file-formats.clj, part of nben, a mathematics library for the JVM.
;; This namespace provides import/export mechanisms for neuroscience file formats such as NifTI and
;; MGH.
;; 
;; Copyright (C) 2016 Noah C. Benson
;;
;; This file is part of the nben clojure library.
;;
;; The nben clojure library is free software: you can redistribute it and/or modify it under the 
;; terms of the GNU General Public License as published by the Free Software Foundation, either 
;; version 3 of the License, or (at your option) any later version.
;;
;; The nben clojure library is distributed in the hope that it will be useful, but WITHOUT ANY 
;; WARRANTY; without even the implied warranty of  MERCHANTABILITY or FITNESS FOR A PARTICULAR
;; PURPOSE.  See the GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License along with the nben clojure
;; library.  If not, see <http:;;www.gnu.org/licenses/>.
;;

(ns nben.neuroscience.file-formats
  (:use nben.util)
  (:use nben.sys)
  (:require [org.clojars.smee.binary.core :as smee]))

;; #MGH and #MGZ files ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod file-format-of "MGH" [_] "MGH")
(defmethod file-format-of "MGZ" [_] "MGZ")

(def ^{:doc "MGH-image-buffer-type->int is a map that translates the MGH image buffer types into
             their appropriate integer representations."}
  MGH-image-buffer-type->int
  {:ubyte  0
   :int    1
   :double 3
   :short  4})
(def ^{:doc "int->MGH-image-buffer-type is a map that translates to MGH image buffer types from
             their appropriate integer representations."}
  int->MGH-image-buffer-type
  (map-invert MGH-image-buffer-type->int))
(def ^{:doc "MGH-image-buffer-types is a set of image buffer types that are understood by nben."}
  MGH-image-buffer-types
  (key-set MGH-image-buffer-type->int))

;; the MGHDataHeader type
(defprotocol PMGHDataHeader
  "The PMGHDataHeader protocol is a protocol that should be extended by anything that wishes to be
   exportable as an MGH/MGZ file header. An IPersistentMap object that with the appropriate fields
   automatically counts as a PMGHDataHeader; these fields are the same as the members of the
   PMGHDataHeader protocol (with the additional prefix of MGH-header-):"
  (MGH-header? [_] "yields true if given object is a valid MGH header")
  (MGH-header-dimensions [_] "yields the [x y z] dimensions of the MGH header data")
  (MGH-header-frames [_] "yields the number of frames in the MGH header data")
  (MGH-header-image-buffer-type [_] "yields the image buffer type of the data in the associated
                                     MGH obect")
  (MGH-header-degrees-of-freedom [_] "yields the degrees of freedom in the MGH header data")
  (MGH-header-VOX-to-RAS-matrix [_] "yields the VOX-to-RAS matrix for the given MGH header data")
  (MGH-header-spacings [_] "yields the [x y z] spacings for the given MGH header data"))

(def- mgh-header-map-impl
  {:MGH-header?                   #(every? (partial contains? %)
                                           [:dimensions :frame :image-buffer-type
                                            :degrees-of-freedom
                                            :VOX-to-RAS-matrix :spacings])
   :MGH-header-dimensions         :dimensions
   :MGH-header-frames             :frames
   :MGH-header-image-buffer-type  :image-buffer-type
   :MGH-header-degrees-of-freedom :degrees-of-freedom
   :MGH-header-VOX-to-RAS-matrix  :VOX-to-RAS-matrix
   :MGH-header-spacings           :spacings})
(extend-protocol PMGHDataHeader
  Object
  (MGH-header? [obj]
    (if (map? obj)
      (do (extend (class obj) PMGHDataHeader mgh-header-map-impl)
          (MGH-header? obj))
      false))
  (MGH-header-dimensions [obj]
    (when (map? obj)
      (do (extend (class obj) PMGHDataHeader mgh-header-map-impl)
          (MGH-header-dimensions obj))))
  (MGH-header-frames [obj]
    (when (map? obj)
      (do (extend (class obj) PMGHDataHeader mgh-header-map-impl)
          (MGH-header-frames obj))))
  (MGH-header-image-buffer-type [obj]
    (when (map? obj)
      (do (extend (class obj) PMGHDataHeader mgh-header-map-impl)
          (MGH-header-image-buffer-type obj))))
  (MGH-header-degrees-of-freedom [obj]
    (when (map? obj)
      (do (extend (class obj) PMGHDataHeader mgh-header-map-impl)
          (MGH-header-degrees-of-freedom obj))))
  (MGH-header-VOX-to-RAS-matrix [obj]
    (when (map? obj)
      (do (extend (class obj) PMGHDataHeader mgh-header-map-impl)
          (MGH-header-VOX-to-RAS-matrix obj))))
  (MGH-header-spacings [obj]
    (when (map? obj)
      (do (extend (class obj) PMGHDataHeader mgh-header-map-impl)
          (MGH-header-spacings obj)))))

;; the MGHDataFooter type
(defprotocol PMGHDataFooter
  "The PMGHDataFooter protocol is a protocol that should be extended by anything that wishes to be
   exportable as an MGH/MGZ file footer. An IPersistentMap object that with the appropriate fields
   automatically counts as a PMGHDataFooterer; these fields are the same as the members of the
   MGHDataFooter record."
  (MGH-footer? [_] "yields true if given object is a valid MGH footer")
  (MGH-footer-TR [_] "yields the TR stored in the mgh footer")
  (MGH-footer-flip-angle [_] "yields the flip angle stored in the mgh footer")
  (MGH-footer-TE [_] "yields the TE value stored in the mgh footer")
  (MGH-footer-TI [_] "yields the TI value stored in the mgh footer")
  (MGH-footer-FoV [_] "yields the FoV value stored in the mgh footer"))

(def- mgh-footer-map-impl
  {:MGH-footer?           (fn [m] (every? (partial contains? m) [:flip-angle :TR :TE :TI]))
   :MGH-footer-TR         :TR
   :MGH-footer-flip-angle :flip-angle
   :MGH-footer-TE         :TE
   :MGH-footer-TI         :TI
   :MGH-footer-FoV        :FoV})
(extend-protocol PMGHDataFooter
  Object
  (MGH-footer? [obj]
    (if (map? obj)
      (do (extend (class obj) PMGHDataFooter mgh-footer-map-impl) (MGH-footer? obj))
      false))
  (MGH-footer-TR [obj]
    (when (map? obj)
      (do (extend (class obj) PMGHDataFooter mgh-footer-map-impl) (MGH-footer-TR obj))))
  (MGH-footer-flip-angle [obj]
    (when (map? obj)
      (do (extend (class obj) PMGHDataFooter mgh-footer-map-impl) (MGH-footer-flip-angle obj))))
  (MGH-footer-TE [obj]
    (when (map? obj)
      (do (extend (class obj) PMGHDataFooter mgh-footer-map-impl) (MGH-footer-TE obj))))
  (MGH-footer-TI [obj]
    (when (map? obj)
      (do (extend (class obj) PMGHDataFooter mgh-footer-map-impl) (MGH-footer-TI obj))))
  (MGH-footer-FoV [obj]
    (when (map? obj)
      (do (extend (class obj) PMGHDataFooter mgh-footer-map-impl) (MGH-footer-FoV obj)))))

;; the MGHData type
(defprotocol PMGHData
  "The PMGHData protocol is a protocol that should be extended by anything that wishes to be
   exportable as an MGH/MGZ file. Any object that implements the IPersistentMap protocol and that
   has the appropriate fields automatically are considered PMGHData objects. These keys are, which
   share the names of the functions in the PMGHData protocol (all of which additionally bear the
   MGH- prefix):
     :header must yield a map of MGH header data; the map must contain the following:
       :dimensions must yield the [width height depth] of the volume,
       :frames must yield a positive integer number of frames in the volume,
       :image-buffer-type must hield the type (see MGH-image-buffer-types),
       :degrees-of-freedom must yield an integer,
       :VOX-to-RAS-matrix must yield the matrix that translates VOX coordinates to RAS coordinates,
       :spacing must yield a seq of the [x, y, z] spacings between voxels
   PMGHDataFooter protocol (with the additional prefix of MGH-footer-):
     :TR must yield the TR length,
     :flip-angle must yield the flip angle,
     :TE, :TI, and :FoV should all yield their appropriate values.
     :frames must yield the frame vector,
     :footer must optionally yield a PMGHFooterData object.
  An MGH-header "
  (MGH? [_] "yields true if given object is a valid MGH object")
  (MGH-header [_] "(MGH-header o) yields the PMGHDataHeader object that represents o's header
                     information.")
  (MGH-frames [_] "(MGH-frames o) yields the tensor of frames with dimensionality
                     (frames x width x depth x height)")
  (MGH-footer [_] "(MGH-footer o) yields the PMGHDataFooter object that represents o's footer
                     information, or nil if no footer is specified."))
(def- mgh-map-impl
  {:MGH? (fn [m] (and (contains? m :header) (MGH-header? (:header m))
                      (contains? m :frames)))
   :MGH-header #(get % :header %)
   :MGH-frames :frames
   :MGH-footer #(get % :footer %)})
(extend-protocol PMGHDataFooter
  Object
  (MGH? [o] (if (map? o) (do (extend (class o) PMGHData mgh-map-impl) (MGH? o)) false))
  (MGH-header [o] (when (map? o) (do (extend (class o) PMGHData mgh-map-impl) (MGH-header o))))
  (MGH-frames [o] (when (map? o) (do (extend (class o) PMGHData mgh-map-impl) (MGH-frames o))))
  (MGH-footer [o] (when (map? o) (do (extend (class o) PMGHData mgh-map-impl) (MGH-footer o)))))

;; the codec for an MGH file
;(def MGH-file-format-codec
;  (smee/compile-codec

;(defmethod import-stream "MGH"
;  (fn [stream _ opts]
    

