# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""
.. _`LARSIM`: http://www.larsim.de/en/the-model/

|meteo_glob_morsim| applies the equations given in :cite:t:`ref-LARSIM` and
:cite:t:`ref-LARSIM-Hilfe` to estimate global radiation.  It is pretty similar to
|meteo_glob_fao56| for daily simulations.  However, it is more complicated for hourly
(or other short) timesteps, as it implements an approach of adjusting hourly global
radiation values to directly calculated daily global radiation sums.  The documentation
on method |Calc_GlobalRadiation_V2| explains this in more detail.

Users that strive for high consistency with *LARSIM's* results should note
|meteo_glob_morsim| works slightly differently regarding the mentioned approach for
adjusting hourly global radiation values.  *LARSIM* normalises them so that their sum
is identical with the directly calculated global radiation sum of the current calendar
day.  Hence, estimating the global radiation for the early morning takes the sunshine
duration measured for the late evening into account.  *HydPy* generally does not
support looking into the future for calculating current states or fluxes.  Instead, we
decided to let |meteo_glob_morsim| always uses data of the last 24 hours for this
adjustment process.  Hence, there are noticeable differences in individual hourly
radiation estimates of *LARSIM* and |meteo_glob_morsim|, but these are random short
term fluctuations that do not introduce any long term bias.

Integration tests
=================

.. how_to_understand_integration_tests::

Application model |meteo_glob_morsim| calculates multiple meteorological factors
hydrological models could require.  We design the following integration tests so that
their output can serve as input for the integration tests of |lland_knauf|.  Among
others, |lland_knauf| requires global radiation and possible sunshine duration as
input, which is why we select the corresponding sequences
|meteo_fluxes.GlobalRadiation| and |meteo_factors.PossibleSunshineDuration| as outputs.
We hand them over to the |Node| instances `node1` and `node2`:

>>> from hydpy import Element, Node
>>> from hydpy.aliases import meteo_fluxes_GlobalRadiation, meteo_factors_PossibleSunshineDuration
>>> node1 = Node("node1", variable=meteo_fluxes_GlobalRadiation)
>>> node2 = Node("node2", variable=meteo_factors_PossibleSunshineDuration)

We prepare a |meteo_glob_morsim| instance and assign it to an element connected to
those nodes:

>>> from hydpy.models.meteo_glob_morsim import *
>>> parameterstep()
>>> element = Element("element", outputs=(node1, node2))
>>> element.model = model

We will use the same coordinates and Ångström coefficients in all examples:

>>> latitude(54.1)
>>> longitude(9.7)
>>> angstromconstant(0.25)
>>> angstromfactor(0.5)
>>> angstromalternative(0.15)

.. _meteo_glob_morsim_daily_simulation_summer:

daily simulation summer
_______________________

The first example takes place in a summer period, for which we calculate daily values
of global radiation and possible sunshine duration.  Besides preparing an
|IntegrationTest| instance for running the test, we only need to provide the time
series of (measured) sunshine duration:

>>> from hydpy import IntegrationTest, pub, round_
>>> pub.timegrids = "1997-08-01", "1997-09-01", "1d"
>>> test = IntegrationTest(element)
>>> test.axis1 = (fluxes.globalradiation,)
>>> test.axis2 = (factors.possiblesunshineduration,)
>>> test.dateformat = "%Y-%d-%m"

>>> inputs.sunshineduration.series = (
...     6.3, 1.7, 4.5, 12.4, 13.9, 13.0, 13.8, 12.3, 13.1, 12.8, 13.1, 13.3, 12.7,
...     10.2, 9.4, 10.3, 11.1, 11.0, 8.5, 11.3, 12.4, 0.1, 6.7, 10.4, 6.5, 4.9, 6.6,
...     0.3, 0.1, 5.0, 3.8)

.. integration-test::

    >>> test("meteo_glob_morsim_daily_summer")
    |       date | sunshineduration | earthsundistance | solardeclination | timeofsunrise | timeofsunset | possiblesunshineduration | dailypossiblesunshineduration | dailysunshineduration | portiondailyradiation | extraterrestrialradiation | clearskysolarradiation | unadjustedglobalradiation | dailyglobalradiation | globalradiation |      node1 |     node2 |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 |              6.3 |         0.971564 |         0.307429 |      4.507941 |    20.198726 |                15.690785 |                     15.690785 |                   6.3 |                 100.0 |                422.073153 |             316.554865 |                 190.25149 |            190.25149 |       190.25149 |  190.25149 | 15.690785 |
    | 1997-02-08 |              1.7 |         0.971855 |         0.302714 |      4.538923 |    20.167744 |                15.628821 |                     15.628821 |                   1.7 |                 100.0 |                419.403284 |             314.552463 |                127.660782 |           127.660782 |      127.660782 | 127.660782 | 15.628821 |
    | 1997-03-08 |              4.5 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                15.566134 |                     15.566134 |                   4.5 |                 100.0 |                416.686259 |             312.514694 |                164.401298 |           164.401298 |      164.401298 | 164.401298 | 15.566134 |
    | 1997-04-08 |             12.4 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                15.502754 |                     15.502754 |                  12.4 |                 100.0 |                413.922951 |             310.442213 |                269.020502 |           269.020502 |      269.020502 | 269.020502 | 15.502754 |
    | 1997-05-08 |             13.9 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                15.438712 |                     15.438712 |                  13.9 |                 100.0 |                411.114245 |             308.335683 |                287.848667 |           287.848667 |      287.848667 | 287.848667 | 15.438712 |
    | 1997-06-08 |             13.0 |         0.973104 |         0.282971 |      4.666316 |     20.04035 |                15.374034 |                     15.374034 |                  13.0 |                 100.0 |                408.261035 |             306.195777 |                274.674263 |           274.674263 |      274.674263 | 274.674263 | 15.374034 |
    | 1997-07-08 |             13.8 |         0.973436 |         0.277822 |      4.698958 |    20.007708 |                 15.30875 |                      15.30875 |                  13.8 |                 100.0 |                405.364232 |             304.023174 |                284.047892 |           284.047892 |      284.047892 | 284.047892 |  15.30875 |
    | 1997-08-08 |             12.3 |         0.973776 |         0.272591 |      4.731891 |    19.974776 |                15.242885 |                     15.242885 |                  12.3 |                 100.0 |                402.424754 |             301.818566 |                262.971263 |           262.971263 |      262.971263 | 262.971263 | 15.242885 |
    | 1997-09-08 |             13.1 |         0.974124 |         0.267279 |        4.7651 |    19.941567 |                15.176467 |                     15.176467 |                  13.1 |                 100.0 |                399.443534 |              299.58265 |                272.256414 |           272.256414 |      272.256414 | 272.256414 | 15.176467 |
    | 1997-10-08 |             12.8 |         0.974479 |         0.261887 |      4.798573 |    19.908094 |                15.109521 |                     15.109521 |                  12.8 |                 100.0 |                396.421514 |             297.316136 |                267.019218 |           267.019218 |      267.019218 | 267.019218 | 15.109521 |
    | 1997-11-08 |             13.1 |         0.974842 |         0.256418 |      4.832298 |    19.874369 |                15.042071 |                     15.042071 |                  13.1 |                 100.0 |                393.359651 |             295.019738 |                269.626547 |           269.626547 |      269.626547 | 269.626547 | 15.042071 |
    | 1997-12-08 |             13.3 |         0.975213 |         0.250873 |      4.866263 |    19.840404 |                14.974141 |                     14.974141 |                  13.3 |                 100.0 |                390.258912 |             292.694184 |                270.878291 |           270.878291 |      270.878291 | 270.878291 | 14.974141 |
    | 1997-13-08 |             12.7 |          0.97559 |         0.245254 |      4.900456 |    19.806211 |                14.905755 |                     14.905755 |                  12.7 |                 100.0 |                387.120277 |             290.340208 |                261.697164 |           261.697164 |      261.697164 | 261.697164 | 14.905755 |
    | 1997-14-08 |             10.2 |         0.975975 |         0.239562 |      4.934866 |      19.7718 |                14.836934 |                     14.836934 |                  10.2 |                 100.0 |                383.944737 |             287.958553 |                227.962115 |           227.962115 |      227.962115 | 227.962115 | 14.836934 |
    | 1997-15-08 |              9.4 |         0.976367 |         0.233799 |      4.969483 |    19.737183 |                  14.7677 |                       14.7677 |                   9.4 |                 100.0 |                380.733298 |             285.549974 |                216.356321 |           216.356321 |      216.356321 | 216.356321 |   14.7677 |
    | 1997-16-08 |             10.3 |         0.976766 |         0.227967 |      5.004296 |    19.702371 |                14.698075 |                     14.698075 |                  10.3 |                 100.0 |                377.486975 |             283.115231 |                226.637904 |           226.637904 |      226.637904 | 226.637904 | 14.698075 |
    | 1997-17-08 |             11.1 |         0.977172 |         0.222067 |      5.039294 |    19.667372 |                14.628078 |                     14.628078 |                  11.1 |                 100.0 |                374.206797 |             280.655098 |                235.528503 |           235.528503 |      235.528503 | 235.528503 | 14.628078 |
    | 1997-18-08 |             11.0 |         0.977584 |         0.216102 |      5.074469 |    19.632198 |                14.557728 |                     14.557728 |                  11.0 |                 100.0 |                370.893805 |             278.170354 |                232.849428 |           232.849428 |      232.849428 | 232.849428 | 14.557728 |
    | 1997-19-08 |              8.5 |         0.978003 |         0.210072 |       5.10981 |    19.596856 |                14.487046 |                     14.487046 |                   8.5 |                 100.0 |                367.549053 |              275.66179 |                 199.71349 |            199.71349 |       199.71349 |  199.71349 | 14.487046 |
    | 1997-20-08 |             11.3 |         0.978429 |          0.20398 |      5.145309 |    19.561357 |                14.416048 |                     14.416048 |                  11.3 |                 100.0 |                364.173607 |             273.130205 |                233.771902 |           233.771902 |      233.771902 | 233.771902 | 14.416048 |
    | 1997-21-08 |             12.4 |          0.97886 |         0.197828 |      5.180957 |    19.525709 |                14.344752 |                     14.344752 |                  12.4 |                 100.0 |                360.768544 |             270.576408 |                246.121286 |           246.121286 |      246.121286 | 246.121286 | 14.344752 |
    | 1997-22-08 |              0.1 |         0.979299 |         0.191617 |      5.216746 |    19.489921 |                14.273175 |                     14.273175 |                   0.1 |                 100.0 |                357.334956 |             268.001217 |                  90.58551 |             90.58551 |        90.58551 |   90.58551 | 14.273175 |
    | 1997-23-08 |              6.7 |         0.979743 |          0.18535 |      5.252667 |       19.454 |                14.201333 |                     14.201333 |                   6.7 |                 100.0 |                353.873944 |             265.405458 |                171.944993 |           171.944993 |      171.944993 | 171.944993 | 14.201333 |
    | 1997-24-08 |             10.4 |         0.980193 |         0.179027 |      5.288712 |    19.417955 |                14.129243 |                     14.129243 |                  10.4 |                 100.0 |                350.386625 |             262.789968 |                216.549813 |           216.549813 |      216.549813 | 216.549813 | 14.129243 |
    | 1997-25-08 |              6.5 |         0.980649 |         0.172651 |      5.324874 |    19.381792 |                14.056918 |                     14.056918 |                   6.5 |                 100.0 |                346.874124 |             260.155593 |                 166.91683 |            166.91683 |       166.91683 |  166.91683 | 14.056918 |
    | 1997-26-08 |              4.9 |         0.981111 |         0.166225 |      5.361147 |     19.34552 |                13.984374 |                     13.984374 |                   4.9 |                 100.0 |                 343.33758 |             257.503185 |                145.985611 |           145.985611 |      145.985611 | 145.985611 | 13.984374 |
    | 1997-27-08 |              6.6 |         0.981578 |         0.159748 |      5.397522 |    19.309145 |                13.911624 |                     13.911624 |                   6.6 |                 100.0 |                339.778145 |             254.833609 |                 165.54389 |            165.54389 |       165.54389 |  165.54389 | 13.911624 |
    | 1997-28-08 |              0.3 |         0.982051 |         0.153225 |      5.433993 |    19.272674 |                13.838682 |                     13.838682 |                   0.3 |                 100.0 |                 336.19698 |             252.147735 |                 87.693346 |            87.693346 |       87.693346 |  87.693346 | 13.838682 |
    | 1997-29-08 |              0.1 |         0.982529 |         0.146656 |      5.470553 |    19.236114 |                13.765561 |                     13.765561 |                   0.1 |                 100.0 |                332.595259 |             249.446444 |                 84.356885 |            84.356885 |       84.356885 |  84.356885 | 13.765561 |
    | 1997-30-08 |              5.0 |         0.983012 |         0.140044 |      5.507197 |     19.19947 |                13.692273 |                     13.692273 |                   5.0 |                 100.0 |                328.974167 |             246.730625 |                142.309203 |           142.309203 |      142.309203 | 142.309203 | 13.692273 |
    | 1997-31-08 |              3.8 |           0.9835 |          0.13339 |      5.543918 |    19.162749 |                13.618831 |                     13.618831 |                   3.8 |                 100.0 |                325.334897 |             244.001173 |                126.722078 |           126.722078 |      126.722078 | 126.722078 | 13.618831 |

All getters specified by the |RadiationModel_V1| interface return the correct data:

>>> round_(model.get_possiblesunshineduration())
13.618831
>>> round_(model.get_sunshineduration())
3.8
>>> round_(model.get_clearskysolarradiation())
244.001173
>>> round_(model.get_globalradiation())
126.722078

.. _meteo_glob_morsim_daily_simulation_winter:

daily simulation winter
_______________________

This test is similar to the :ref:`meteo_glob_morsim_daily_simulation_summer` example
but deals with winter conditions:

>>> pub.timegrids = "2010-12-01", "2011-01-01", "1d"

>>> inputs.sunshineduration.series = (
...     3.5, 0.0, 0.1, 3.7, 0.0, 0.1, 0.0, 0.0, 4.4, 0.1, 0.2, 0.7, 0.0, 1.2, 3.0, 0.0,
...     0.0, 0.0, 0.0, 0.2, 0.8, 0.0, 0.0, 0.0, 3.2, 4.3, 0.0, 3.4, 0.0, 0.0, 4.0)

.. integration-test::

    >>> test("meteo_glob_morsim_daily_winter")
    |       date | sunshineduration | earthsundistance | solardeclination | timeofsunrise | timeofsunset | possiblesunshineduration | dailypossiblesunshineduration | dailysunshineduration | portiondailyradiation | extraterrestrialradiation | clearskysolarradiation | unadjustedglobalradiation | dailyglobalradiation | globalradiation |     node1 |    node2 |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |              3.5 |         1.028719 |        -0.389384 |      8.532819 |    16.173848 |                 7.641029 |                      7.641029 |                   3.5 |                 100.0 |                 66.682705 |              50.012029 |                 31.942798 |            31.942798 |       31.942798 | 31.942798 | 7.641029 |
    | 2010-02-12 |              0.0 |         1.028994 |        -0.391536 |      8.548511 |    16.158156 |                 7.609644 |                      7.609644 |                   0.0 |                 100.0 |                 65.825989 |              49.369492 |                  9.873898 |             9.873898 |        9.873898 |  9.873898 | 7.609644 |
    | 2010-03-12 |              0.1 |          1.02926 |        -0.393572 |      8.563421 |    16.143245 |                 7.579824 |                      7.579824 |                   0.1 |                 100.0 |                 65.018433 |              48.763825 |                   16.6835 |              16.6835 |         16.6835 |   16.6835 | 7.579824 |
    | 2010-04-12 |              3.7 |         1.029518 |        -0.395492 |      8.577535 |    16.129131 |                 7.551596 |                      7.551596 |                   3.7 |                 100.0 |                 64.259854 |               48.19489 |                 31.807427 |            31.807427 |       31.807427 | 31.807427 | 7.551596 |
    | 2010-05-12 |              0.0 |         1.029767 |        -0.397294 |      8.590839 |    16.115828 |                 7.524989 |                      7.524989 |                   0.0 |                 100.0 |                  63.55007 |              47.662553 |                  9.532511 |             9.532511 |        9.532511 |  9.532511 | 7.524989 |
    | 2010-06-12 |              0.1 |         1.030007 |        -0.398979 |      8.603319 |    16.103347 |                 7.500028 |                      7.500028 |                   0.1 |                 100.0 |                 62.888906 |               47.16668 |                 16.141484 |            16.141484 |       16.141484 | 16.141484 | 7.500028 |
    | 2010-07-12 |              0.0 |         1.030238 |        -0.400545 |      8.614964 |    16.091703 |                 7.476739 |                      7.476739 |                   0.0 |                 100.0 |                 62.276192 |              46.707144 |                  9.341429 |             9.341429 |        9.341429 |  9.341429 | 7.476739 |
    | 2010-08-12 |              0.0 |         1.030461 |        -0.401992 |      8.625761 |    16.080906 |                 7.455145 |                      7.455145 |                   0.0 |                 100.0 |                 61.711763 |              46.283822 |                  9.256764 |             9.256764 |        9.256764 |  9.256764 | 7.455145 |
    | 2010-09-12 |              4.4 |         1.030674 |        -0.403321 |      8.635699 |    16.070967 |                 7.435268 |                      7.435268 |                   4.4 |                 100.0 |                 61.195464 |              45.896598 |                 33.405815 |            33.405815 |       33.405815 | 33.405815 | 7.435268 |
    | 2010-10-12 |              0.1 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                  7.41713 |                       7.41713 |                   0.1 |                 100.0 |                  60.72715 |              45.545363 |                 15.591158 |            15.591158 |       15.591158 | 15.591158 |  7.41713 |
    | 2010-11-12 |              0.2 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                 7.400749 |                      7.400749 |                   0.2 |                 100.0 |                 60.306686 |              45.230014 |                 15.891544 |            15.891544 |       15.891544 | 15.891544 | 7.400749 |
    | 2010-12-12 |              0.7 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                 7.386143 |                      7.386143 |                   0.7 |                 100.0 |                 59.933948 |              44.950461 |                 17.823519 |            17.823519 |       17.823519 | 17.823519 | 7.386143 |
    | 2010-13-12 |              0.0 |         1.031437 |        -0.407436 |      8.666669 |    16.039997 |                 7.373328 |                      7.373328 |                   0.0 |                 100.0 |                 59.608826 |               44.70662 |                  8.941324 |             8.941324 |        8.941324 |  8.941324 | 7.373328 |
    | 2010-14-12 |              1.2 |         1.031605 |        -0.408164 |      8.672175 |    16.034492 |                 7.362317 |                      7.362317 |                   1.2 |                 100.0 |                 59.331223 |              44.498417 |                 19.668068 |            19.668068 |       19.668068 | 19.668068 | 7.362317 |
    | 2010-15-12 |              3.0 |         1.031763 |         -0.40877 |      8.676772 |    16.029895 |                 7.353123 |                      7.353123 |                   3.0 |                 100.0 |                 59.101055 |              44.325792 |                 26.831581 |            26.831581 |       26.831581 | 26.831581 | 7.353123 |
    | 2010-16-12 |              0.0 |         1.031912 |        -0.409256 |      8.680456 |    16.026211 |                 7.345755 |                      7.345755 |                   0.0 |                 100.0 |                 58.918256 |              44.188692 |                  8.837738 |             8.837738 |        8.837738 |  8.837738 | 7.345755 |
    | 2010-17-12 |              0.0 |         1.032051 |         -0.40962 |      8.683222 |    16.023444 |                 7.340222 |                      7.340222 |                   0.0 |                 100.0 |                 58.782773 |               44.08708 |                  8.817416 |             8.817416 |        8.817416 |  8.817416 | 7.340222 |
    | 2010-18-12 |              0.0 |         1.032182 |        -0.409863 |      8.685068 |    16.021598 |                  7.33653 |                       7.33653 |                   0.0 |                 100.0 |                 58.694569 |              44.020927 |                  8.804185 |             8.804185 |        8.804185 |  8.804185 |  7.33653 |
    | 2010-19-12 |              0.0 |         1.032302 |        -0.409985 |      8.685992 |    16.020675 |                 7.334683 |                      7.334683 |                   0.0 |                 100.0 |                 58.653626 |              43.990219 |                  8.798044 |             8.798044 |        8.798044 |  8.798044 | 7.334683 |
    | 2010-20-12 |              0.2 |         1.032413 |        -0.409985 |      8.685992 |    16.020675 |                 7.334683 |                      7.334683 |                   0.2 |                 100.0 |                 58.659939 |              43.994954 |                 15.464746 |            15.464746 |       15.464746 | 15.464746 | 7.334683 |
    | 2010-21-12 |              0.8 |         1.032515 |        -0.409863 |      8.685068 |    16.021598 |                  7.33653 |                       7.33653 |                   0.8 |                 100.0 |                 58.713523 |              44.035142 |                 17.879541 |            17.879541 |       17.879541 | 17.879541 |  7.33653 |
    | 2010-22-12 |              0.0 |         1.032607 |         -0.40962 |      8.683222 |    16.023444 |                 7.340222 |                      7.340222 |                   0.0 |                 100.0 |                 58.814408 |              44.110806 |                  8.822161 |             8.822161 |        8.822161 |  8.822161 | 7.340222 |
    | 2010-23-12 |              0.0 |         1.032689 |        -0.409256 |      8.680456 |    16.026211 |                 7.345755 |                      7.345755 |                   0.0 |                 100.0 |                  58.96264 |               44.22198 |                  8.844396 |             8.844396 |        8.844396 |  8.844396 | 7.345755 |
    | 2010-24-12 |              0.0 |         1.032762 |         -0.40877 |      8.676772 |    16.029895 |                 7.353123 |                      7.353123 |                   0.0 |                 100.0 |                 59.158284 |              44.368713 |                  8.873743 |             8.873743 |        8.873743 |  8.873743 | 7.353123 |
    | 2010-25-12 |              3.2 |         1.032825 |        -0.408164 |      8.672175 |    16.034492 |                 7.362317 |                      7.362317 |                   3.2 |                 100.0 |                 59.401417 |              44.551063 |                 27.759642 |            27.759642 |       27.759642 | 27.759642 | 7.362317 |
    | 2010-26-12 |              4.3 |         1.032879 |        -0.407436 |      8.666669 |    16.039997 |                 7.373328 |                      7.373328 |                   4.3 |                 100.0 |                 59.692135 |              44.769102 |                 32.328755 |            32.328755 |       32.328755 | 32.328755 | 7.373328 |
    | 2010-27-12 |              0.0 |         1.032922 |        -0.406588 |      8.660262 |    16.046405 |                 7.386143 |                      7.386143 |                   0.0 |                 100.0 |                 60.030549 |              45.022912 |                  9.004582 |             9.004582 |        9.004582 |  9.004582 | 7.386143 |
    | 2010-28-12 |              3.4 |         1.032956 |        -0.405619 |      8.652959 |    16.053708 |                 7.400749 |                      7.400749 |                   3.4 |                 100.0 |                 60.416781 |              45.312585 |                 28.982321 |            28.982321 |       28.982321 | 28.982321 | 7.400749 |
    | 2010-29-12 |              0.0 |         1.032981 |         -0.40453 |      8.644768 |    16.061898 |                  7.41713 |                       7.41713 |                   0.0 |                 100.0 |                 60.850969 |              45.638227 |                  9.127645 |             9.127645 |        9.127645 |  9.127645 |  7.41713 |
    | 2010-30-12 |              0.0 |         1.032995 |        -0.403321 |      8.635699 |    16.070967 |                 7.435268 |                      7.435268 |                   0.0 |                 100.0 |                 61.333263 |              45.999948 |                   9.19999 |              9.19999 |         9.19999 |   9.19999 | 7.435268 |
    | 2010-31-12 |              4.0 |            1.033 |        -0.401992 |      8.625761 |    16.080906 |                 7.455145 |                      7.455145 |                   4.0 |                 100.0 |                 61.863826 |              46.397869 |                 32.062234 |            32.062234 |       32.062234 | 32.062234 | 7.455145 |

.. _meteo_glob_morsim_hourly_simulation_summer:

hourly simulation summer
________________________

When choosing a simulation timestep smaller one day, we additionally must provide
sunshine duration and "unadjusted" global radiation values for the last 24 hours, which
we pass to the sequences |LoggedSunshineDuration| and |LoggedUnadjustedGlobalRadiation|:

>>> pub.timegrids = "1997-08-03", "1997-08-06", "1h"
>>> test = IntegrationTest(element)
>>> test.axis1 = (fluxes.globalradiation,)
>>> test.axis2 = (factors.possiblesunshineduration,)
>>> test.dateformat = "%Y-%d-%m %H:00"

>>> inputs.sunshineduration.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.5, 0.7, 0.8, 0.5, 0.4, 0.5,
...     0.5, 0.3, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.9,
...     1.0, 1.0, 0.9, 0.8, 0.9, 0.8, 0.9, 0.9, 0.9, 1.0, 1.0, 1.0, 0.3, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.6, 0.9, 1.0, 0.9, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
...     1.0, 1.0, 1.0, 1.0, 0.5, 0.0, 0.0, 0.0)

>>> test.inits = (
...     (logs.loggedsunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.1, 0.2, 0.1, 0.2, 0.2, 0.3, 0.0,
...       0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (logs.loggedunadjustedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 27.777778, 55.555556, 138.888889, 222.222222,
...       305.555556, 333.333333, 388.888889, 527.777778, 444.444444, 250.0,
...       222.222222, 166.666667, 111.111111, 55.555556, 27.777778, 0.0, 0.0, 0.0,
...       0.0]))

.. integration-test::

    >>> test("meteo_glob_morsim_hourly_summer")
    |             date | sunshineduration | earthsundistance | solardeclination | timeofsunrise | timeofsunset | possiblesunshineduration | dailypossiblesunshineduration | dailysunshineduration | portiondailyradiation | extraterrestrialradiation | clearskysolarradiation | unadjustedglobalradiation | dailyglobalradiation | globalradiation |      node1 |    node2 |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-03-08 00:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      0.0 |                     15.566134 |                   1.3 |                   0.0 |                416.686259 |                    0.0 |                       0.0 |           121.571265 |             0.0 |        0.0 |      0.0 |
    | 1997-03-08 01:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      0.0 |                     15.566134 |                   1.3 |                   0.0 |                416.686259 |                    0.0 |                       0.0 |           121.571265 |             0.0 |        0.0 |      0.0 |
    | 1997-03-08 02:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      0.0 |                     15.566134 |                   1.3 |                   0.0 |                416.686259 |                    0.0 |                       0.0 |           121.571265 |             0.0 |        0.0 |      0.0 |
    | 1997-03-08 03:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      0.0 |                     15.566134 |                   1.3 |                   0.0 |                416.686259 |                    0.0 |                       0.0 |           121.571265 |             0.0 |        0.0 |      0.0 |
    | 1997-03-08 04:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                 0.429734 |                     15.566134 |                   1.3 |              0.086656 |                416.686259 |               6.499471 |                   2.16649 |           121.571265 |        1.943686 |   1.943686 | 0.429734 |
    | 1997-03-08 05:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   1.3 |              0.968392 |                416.686259 |              72.632793 |                 24.210931 |           121.571265 |       21.932441 |  21.932441 |      1.0 |
    | 1997-03-08 06:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   1.3 |                2.4897 |                416.686259 |             186.736268 |                 62.245423 |           121.571265 |       57.256187 |  57.256187 |      1.0 |
    | 1997-03-08 07:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   1.3 |              4.679733 |                416.686259 |             350.996451 |                116.998817 |           121.571265 |      109.332844 | 109.332844 |      1.0 |
    | 1997-03-08 08:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   1.3 |              7.219274 |                416.686259 |             541.470994 |                180.490331 |           121.571265 |      170.949152 | 170.949152 |      1.0 |
    | 1997-03-08 09:00 |              0.2 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   1.5 |              9.442017 |                416.686259 |             708.184581 |                330.486138 |           124.248142 |      311.762624 | 311.762624 |      1.0 |
    | 1997-03-08 10:00 |              0.5 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   1.7 |             10.709164 |                416.686259 |               803.2251 |                  535.4834 |           126.925019 |      501.583299 | 501.583299 |      1.0 |
    | 1997-03-08 11:00 |              0.7 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   2.2 |             10.773868 |                416.686259 |             808.078107 |                646.462485 |           133.617212 |      615.018727 | 615.018727 |      1.0 |
    | 1997-03-08 12:00 |              0.8 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   2.8 |             10.364714 |                416.686259 |             777.390094 |                673.738082 |           141.647843 |      626.544326 | 626.544326 |      1.0 |
    | 1997-03-08 13:00 |              0.5 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   3.2 |             10.872489 |                416.686259 |             815.475032 |                543.650021 |           147.001597 |      496.133417 | 496.133417 |      1.0 |
    | 1997-03-08 14:00 |              0.4 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   3.4 |             10.461143 |                416.686259 |             784.622637 |                470.773582 |           149.678474 |      419.520994 | 419.520994 |      1.0 |
    | 1997-03-08 15:00 |              0.5 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   3.8 |              8.865382 |                416.686259 |             664.934948 |                443.289965 |           155.032228 |      387.887354 | 387.887354 |      1.0 |
    | 1997-03-08 16:00 |              0.5 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   4.1 |              6.474205 |                416.686259 |             485.588212 |                323.725475 |           159.047544 |      278.496873 | 278.496873 |      1.0 |
    | 1997-03-08 17:00 |              0.3 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   4.4 |              3.977658 |                416.686259 |             298.338414 |                159.113821 |           163.062859 |      137.138608 | 137.138608 |      1.0 |
    | 1997-03-08 18:00 |              0.1 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   4.5 |              1.972911 |                416.686259 |             147.975255 |                 59.190102 |           164.401298 |       51.080715 |  51.080715 |      1.0 |
    | 1997-03-08 19:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      1.0 |                     15.566134 |                   4.5 |              0.634046 |                416.686259 |               47.55565 |                 15.851883 |           164.401298 |       13.632816 |  13.632816 |      1.0 |
    | 1997-03-08 20:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                   0.1364 |                     15.566134 |                   4.5 |              0.008648 |                416.686259 |               0.648661 |                   0.21622 |           164.401298 |        0.185943 |   0.185943 |   0.1364 |
    | 1997-03-08 21:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      0.0 |                     15.566134 |                   4.5 |                   0.0 |                416.686259 |                    0.0 |                       0.0 |           164.401298 |             0.0 |        0.0 |      0.0 |
    | 1997-03-08 22:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      0.0 |                     15.566134 |                   4.5 |                   0.0 |                416.686259 |                    0.0 |                       0.0 |           164.401298 |             0.0 |        0.0 |      0.0 |
    | 1997-03-08 23:00 |              0.0 |         0.972155 |         0.297909 |      4.570266 |      20.1364 |                      0.0 |                     15.566134 |                   4.5 |                   0.0 |                416.686259 |                    0.0 |                       0.0 |           164.401298 |             0.0 |        0.0 |      0.0 |
    | 1997-04-08 00:00 |              0.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      0.0 |                     15.502754 |                   4.5 |                   0.0 |                413.922951 |                    0.0 |                       0.0 |           163.555652 |             0.0 |        0.0 |      0.0 |
    | 1997-04-08 01:00 |              0.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      0.0 |                     15.502754 |                   4.5 |                   0.0 |                413.922951 |                    0.0 |                       0.0 |           163.555652 |             0.0 |        0.0 |      0.0 |
    | 1997-04-08 02:00 |              0.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      0.0 |                     15.502754 |                   4.5 |                   0.0 |                413.922951 |                    0.0 |                       0.0 |           163.555652 |             0.0 |        0.0 |      0.0 |
    | 1997-04-08 03:00 |              0.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      0.0 |                     15.502754 |                   4.5 |                   0.0 |                413.922951 |                    0.0 |                       0.0 |           163.555652 |             0.0 |        0.0 |      0.0 |
    | 1997-04-08 04:00 |              0.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                 0.398044 |                     15.502754 |                   4.5 |               0.07485 |                413.922951 |               5.576807 |                  1.858936 |           163.555652 |        1.590516 |   1.590516 | 0.398044 |
    | 1997-04-08 05:00 |              0.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   4.5 |              0.938714 |                413.922951 |              69.939921 |                 23.313307 |           163.555652 |       19.950903 |  19.950903 |      1.0 |
    | 1997-04-08 06:00 |              0.1 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   4.6 |              2.455474 |                413.922951 |             182.947859 |                 73.179144 |            164.89065 |       62.985774 |  62.985774 |      1.0 |
    | 1997-04-08 07:00 |              0.9 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   5.5 |              4.650125 |                413.922951 |             346.462853 |                 323.36533 |           176.905633 |      285.776389 | 285.776389 |      1.0 |
    | 1997-04-08 08:00 |              1.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   6.5 |              7.209034 |                413.922951 |             537.117229 |                537.117229 |           190.255614 |      475.225247 | 475.225247 |      1.0 |
    | 1997-04-08 09:00 |              1.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   7.3 |              9.458797 |                413.922951 |             704.738349 |                704.738349 |           200.935599 |      614.006719 | 614.006719 |      1.0 |
    | 1997-04-08 10:00 |              0.9 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   7.7 |             10.747438 |                413.922951 |             800.750043 |                747.366707 |           206.275591 |      643.806508 | 643.806508 |      1.0 |
    | 1997-04-08 11:00 |              0.8 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   7.8 |             10.819272 |                413.922951 |             806.102134 |                698.621849 |           207.610589 |       600.26298 |  600.26298 |      1.0 |
    | 1997-04-08 12:00 |              0.9 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   7.9 |             10.408076 |                413.922951 |             775.465449 |                723.767752 |           208.945587 |       620.51414 |  620.51414 |      1.0 |
    | 1997-04-08 13:00 |              0.8 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   8.2 |             10.917404 |                413.922951 |             813.413559 |                704.958418 |           212.950582 |      599.441386 | 599.441386 |      1.0 |
    | 1997-04-08 14:00 |              0.9 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   8.7 |             10.494383 |                413.922951 |             781.895886 |                729.769494 |           219.625572 |      613.551223 | 613.551223 |      1.0 |
    | 1997-04-08 15:00 |              0.9 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   9.1 |              8.874215 |                413.922951 |             661.183412 |                617.104518 |           224.965565 |      517.107047 | 517.107047 |      1.0 |
    | 1997-04-08 16:00 |              0.9 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                   9.5 |               6.45697 |                413.922951 |             481.083834 |                449.011578 |           230.305557 |      377.836616 | 377.836616 |      1.0 |
    | 1997-04-08 17:00 |              1.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                  10.2 |              3.945159 |                413.922951 |              293.93856 |                 293.93856 |           239.650544 |      252.204639 | 252.204639 |      1.0 |
    | 1997-04-08 18:00 |              1.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                  11.1 |              1.939642 |                413.922951 |             144.515231 |                144.515231 |           251.665526 |      128.576715 | 128.576715 |      1.0 |
    | 1997-04-08 19:00 |              1.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      1.0 |                     15.502754 |                  12.1 |               0.60531 |                413.922951 |              45.099317 |                 45.099317 |           265.015507 |       42.072585 |  42.072585 |      1.0 |
    | 1997-04-08 20:00 |              0.3 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                 0.104711 |                     15.502754 |                  12.4 |              0.005136 |                413.922951 |               0.382679 |                  0.858487 |           269.020502 |        0.812898 |   0.812898 | 0.104711 |
    | 1997-04-08 21:00 |              0.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      0.0 |                     15.502754 |                  12.4 |                   0.0 |                413.922951 |                    0.0 |                       0.0 |           269.020502 |             0.0 |        0.0 |      0.0 |
    | 1997-04-08 22:00 |              0.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      0.0 |                     15.502754 |                  12.4 |                   0.0 |                413.922951 |                    0.0 |                       0.0 |           269.020502 |             0.0 |        0.0 |      0.0 |
    | 1997-04-08 23:00 |              0.0 |         0.972464 |         0.293016 |      4.601956 |    20.104711 |                      0.0 |                     15.502754 |                  12.4 |                   0.0 |                413.922951 |                    0.0 |                       0.0 |           269.020502 |             0.0 |        0.0 |      0.0 |
    | 1997-05-08 00:00 |              0.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      0.0 |                     15.438712 |                  12.4 |                   0.0 |                411.114245 |                    0.0 |                       0.0 |           267.877073 |             0.0 |        0.0 |      0.0 |
    | 1997-05-08 01:00 |              0.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      0.0 |                     15.438712 |                  12.4 |                   0.0 |                411.114245 |                    0.0 |                       0.0 |           267.877073 |             0.0 |        0.0 |      0.0 |
    | 1997-05-08 02:00 |              0.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      0.0 |                     15.438712 |                  12.4 |                   0.0 |                411.114245 |                    0.0 |                       0.0 |           267.877073 |             0.0 |        0.0 |      0.0 |
    | 1997-05-08 03:00 |              0.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      0.0 |                     15.438712 |                  12.4 |                   0.0 |                411.114245 |                    0.0 |                       0.0 |           267.877073 |             0.0 |        0.0 |      0.0 |
    | 1997-05-08 04:00 |              0.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                 0.366022 |                     15.438712 |                  12.4 |              0.063733 |                411.114245 |               4.716292 |                  1.572097 |           267.877073 |        1.482348 |   1.482348 | 0.366022 |
    | 1997-05-08 05:00 |              0.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  12.4 |              0.908598 |                411.114245 |              67.236784 |                 22.412261 |           267.877073 |       21.135568 |  21.135568 |      1.0 |
    | 1997-05-08 06:00 |              0.6 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  12.9 |              2.420659 |                411.114245 |             179.130154 |                131.362113 |           274.534271 |      125.883451 | 125.883451 |      1.0 |
    | 1997-05-08 07:00 |              0.9 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  12.9 |              4.619722 |                411.114245 |             341.862027 |                319.071225 |           274.534271 |      305.954943 | 305.954943 |      1.0 |
    | 1997-05-08 08:00 |              1.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  12.9 |              7.198198 |                411.114245 |             532.670693 |                532.670693 |           274.534271 |      511.104575 | 511.104575 |      1.0 |
    | 1997-05-08 09:00 |              0.9 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  12.8 |              9.475573 |                411.114245 |             701.197754 |                654.451237 |           273.202831 |       629.51924 |  629.51924 |      1.0 |
    | 1997-05-08 10:00 |              1.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  12.9 |             10.786309 |                411.114245 |             798.192977 |                798.192977 |           274.534271 |       765.81659 |  765.81659 |      1.0 |
    | 1997-05-08 11:00 |              1.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  13.1 |             10.865527 |                411.114245 |             804.055141 |                804.055141 |            277.19715 |      767.145843 | 767.145843 |      1.0 |
    | 1997-05-08 12:00 |              1.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  13.2 |              10.45226 |                411.114245 |             773.473117 |                773.473117 |            278.52859 |      736.263854 | 736.263854 |      1.0 |
    | 1997-05-08 13:00 |              1.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  13.4 |              10.96314 |                411.114245 |             811.278551 |                811.278551 |           281.191469 |      768.006183 | 768.006183 |      1.0 |
    | 1997-05-08 14:00 |              1.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  13.5 |             10.528069 |                411.114245 |             779.083066 |                779.083066 |           282.522909 |       735.92937 |  735.92937 |      1.0 |
    | 1997-05-08 15:00 |              1.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  13.6 |              8.882849 |                411.114245 |             657.335808 |                657.335808 |           283.854348 |      620.374953 | 620.374953 |      1.0 |
    | 1997-05-08 16:00 |              1.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  13.7 |              6.439028 |                411.114245 |             476.491717 |                476.491717 |           285.185788 |      450.095263 | 450.095263 |      1.0 |
    | 1997-05-08 17:00 |              1.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  13.7 |              3.911896 |                411.114245 |             289.482482 |                289.482482 |           285.185788 |      273.614141 | 273.614141 |      1.0 |
    | 1997-05-08 18:00 |              1.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  13.7 |              1.905864 |                411.114245 |             141.035017 |                141.035017 |           285.185788 |      133.368104 | 133.368104 |      1.0 |
    | 1997-05-08 19:00 |              1.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      1.0 |                     15.438712 |                  13.7 |              0.576079 |                411.114245 |              42.630198 |                 42.630198 |           285.185788 |       40.326502 |  40.326502 |      1.0 |
    | 1997-05-08 20:00 |              0.5 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                 0.072689 |                     15.438712 |                  13.9 |              0.002495 |                411.114245 |               0.184625 |                  0.908183 |           287.848667 |        0.867122 |   0.867122 | 0.072689 |
    | 1997-05-08 21:00 |              0.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      0.0 |                     15.438712 |                  13.9 |                   0.0 |                411.114245 |                    0.0 |                       0.0 |           287.848667 |             0.0 |        0.0 |      0.0 |
    | 1997-05-08 22:00 |              0.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      0.0 |                     15.438712 |                  13.9 |                   0.0 |                411.114245 |                    0.0 |                       0.0 |           287.848667 |             0.0 |        0.0 |      0.0 |
    | 1997-05-08 23:00 |              0.0 |          0.97278 |         0.288036 |      4.633978 |    20.072689 |                      0.0 |                     15.438712 |                  13.9 |                   0.0 |                411.114245 |                    0.0 |                       0.0 |           287.848667 |             0.0 |        0.0 |      0.0 |

.. _meteo_glob_morsim_hourly_simulation_winter:

hourly simulation winter
________________________

This test is similar to the :ref:`meteo_glob_morsim_hourly_simulation_summer` example but
deals with winter conditions:

>>> pub.timegrids = "2010-12-10", "2010-12-13", "1h"

>>> inputs.sunshineduration.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.2,
...     0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0)

>>> test.inits = (
...     (logs.loggedsunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.04, 0.25, 0.59, 0.91,
...       0.97, 1.0, 0.65, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (logs.loggedunadjustedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 27.777778, 55.555556,
...       111.111111, 166.666667, 138.888889, 55.555556, 0.0,0.0, 0.0, 0.0, 0.0, 0.0,
...       0.0, 0.0, 0.0]))

.. integration-test::

    >>> test("meteo_glob_morsim_hourly_winter")
    |             date | sunshineduration | earthsundistance | solardeclination | timeofsunrise | timeofsunset | possiblesunshineduration | dailypossiblesunshineduration | dailysunshineduration | portiondailyradiation | extraterrestrialradiation | clearskysolarradiation | unadjustedglobalradiation | dailyglobalradiation | globalradiation |      node1 |    node2 |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-10-12 00:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                  4.41 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            33.235046 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 01:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                  4.41 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            33.235046 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 02:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                  4.41 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            33.235046 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 03:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                  4.41 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            33.235046 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 04:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                  4.41 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            33.235046 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 05:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                  4.41 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            33.235046 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 06:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                  4.41 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            33.235046 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 07:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                  3.76 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            30.574135 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 08:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                 0.355232 |                       7.41713 |                  2.76 |              0.266196 |                  60.72715 |                2.90976 |                   0.96992 |            26.480426 |         1.10761 |    1.10761 | 0.355232 |
    | 2010-10-12 09:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      1.0 |                       7.41713 |                  1.79 |              4.919884 |                  60.72715 |              53.778697 |                 17.926232 |            22.509527 |       18.663203 |  18.663203 |      1.0 |
    | 2010-10-12 10:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      1.0 |                       7.41713 |                  0.88 |             14.873457 |                  60.72715 |             162.580073 |                 54.193358 |            18.784252 |       56.267907 |  56.267907 |      1.0 |
    | 2010-10-12 11:00 |              0.1 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      1.0 |                       7.41713 |                  0.39 |             22.189872 |                  60.72715 |             242.554983 |                 97.021993 |            16.778334 |      107.168237 | 107.168237 |      1.0 |
    | 2010-10-12 12:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      1.0 |                       7.41713 |                  0.14 |             22.197307 |                  60.72715 |             242.636259 |                 80.878753 |            15.754907 |       91.473333 |  91.473333 |      1.0 |
    | 2010-10-12 13:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      1.0 |                       7.41713 |                   0.1 |             20.962863 |                  60.72715 |             229.142692 |                 76.380897 |            15.591158 |       80.475535 |  80.475535 |      1.0 |
    | 2010-10-12 14:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      1.0 |                       7.41713 |                   0.1 |             11.702396 |                  60.72715 |              127.91757 |                  42.63919 |            15.591158 |       43.120591 |  43.120591 |      1.0 |
    | 2010-10-12 15:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      1.0 |                       7.41713 |                   0.1 |              2.880181 |                  60.72715 |               31.48293 |                  10.49431 |            15.591158 |       10.320091 |  10.320091 |      1.0 |
    | 2010-10-12 16:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                 0.061898 |                       7.41713 |                   0.1 |              0.007844 |                  60.72715 |               0.085736 |                  0.028579 |            15.591158 |        0.028102 |   0.028102 | 0.061898 |
    | 2010-10-12 17:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                   0.1 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            15.591158 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 18:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                   0.1 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            15.591158 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 19:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                   0.1 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            15.591158 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 20:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                   0.1 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            15.591158 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 21:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                   0.1 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            15.591158 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 22:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                   0.1 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            15.591158 |             0.0 |        0.0 |      0.0 |
    | 2010-10-12 23:00 |              0.0 |         1.030879 |         -0.40453 |      8.644768 |    16.061898 |                      0.0 |                       7.41713 |                   0.1 |                   0.0 |                  60.72715 |                    0.0 |                       0.0 |            15.591158 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 00:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 01:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 02:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 03:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 04:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 05:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 06:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 07:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 08:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                 0.347041 |                      7.400749 |                   0.1 |              0.254871 |                 60.306686 |               2.766676 |                  0.922225 |            15.484108 |        0.900734 |   0.900734 | 0.347041 |
    | 2010-11-12 09:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      1.0 |                      7.400749 |                   0.1 |              4.882681 |                 60.306686 |              53.002491 |                 17.667497 |            15.484108 |        17.26751 |   17.26751 |      1.0 |
    | 2010-11-12 10:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      1.0 |                      7.400749 |                   0.1 |             14.862249 |                 60.306686 |              161.33274 |                  53.77758 |            15.484108 |        52.61761 |   52.61761 |      1.0 |
    | 2010-11-12 11:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      1.0 |                      7.400749 |                   0.0 |              22.23196 |                 60.306686 |             241.332453 |                 80.444151 |            15.076671 |       80.135633 |  80.135633 |      1.0 |
    | 2010-11-12 12:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      1.0 |                      7.400749 |                   0.0 |             22.247803 |                 60.306686 |              241.50443 |                 80.501477 |            15.076671 |       80.276118 |  80.276118 |      1.0 |
    | 2010-11-12 13:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      1.0 |                      7.400749 |                   0.0 |             20.992726 |                 60.306686 |             227.880311 |                 75.960104 |            15.076671 |       75.835403 |  75.835403 |      1.0 |
    | 2010-11-12 14:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      1.0 |                      7.400749 |                   0.0 |             11.677032 |                 60.306686 |             126.756554 |                 42.252185 |            15.076671 |       42.227911 |  42.227911 |      1.0 |
    | 2010-11-12 15:00 |              0.1 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      1.0 |                      7.400749 |                   0.1 |              2.844748 |                 60.306686 |              30.880321 |                 12.352128 |            15.484108 |       12.613921 |  12.613921 |      1.0 |
    | 2010-11-12 16:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                 0.053708 |                      7.400749 |                   0.1 |               0.00593 |                 60.306686 |                0.06437 |                  0.021457 |            15.484108 |        0.021912 |   0.021912 | 0.053708 |
    | 2010-11-12 17:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 18:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 19:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 20:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 21:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 22:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-11-12 23:00 |              0.0 |         1.031074 |        -0.405619 |      8.652959 |    16.053708 |                      0.0 |                      7.400749 |                   0.1 |                   0.0 |                 60.306686 |                    0.0 |                       0.0 |            15.484108 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 00:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.1 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |            15.389206 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 01:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.1 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |            15.389206 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 02:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.1 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |            15.389206 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 03:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.1 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |            15.389206 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 04:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.1 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |            15.389206 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 05:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.1 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |            15.389206 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 06:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.1 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |            15.389206 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 07:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.1 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |            15.389206 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 08:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                 0.339738 |                      7.386143 |                   0.1 |              0.244956 |                 59.933948 |               2.642609 |                   0.88087 |            15.389206 |        0.894145 |   0.894145 | 0.339738 |
    | 2010-12-12 09:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      1.0 |                      7.386143 |                   0.1 |              4.849391 |                 59.933948 |              52.315768 |                 17.438589 |            15.389206 |        17.71254 |   17.71254 |      1.0 |
    | 2010-12-12 10:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      1.0 |                      7.386143 |                   0.1 |             14.852029 |                 59.933948 |              160.22533 |                 53.408443 |            15.389206 |       54.302585 |  54.302585 |      1.0 |
    | 2010-12-12 11:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      1.0 |                      7.386143 |                   0.1 |             22.269583 |                 59.933948 |             240.246723 |                 80.082241 |            15.389206 |       81.504146 |  81.504146 |      1.0 |
    | 2010-12-12 12:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      1.0 |                      7.386143 |                   0.1 |             22.293018 |                 59.933948 |              240.49954 |                 80.166513 |            15.389206 |       81.665294 |  81.665294 |      1.0 |
    | 2010-12-12 13:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      1.0 |                      7.386143 |                   0.1 |             21.019348 |                 59.933948 |             226.759052 |                 75.586351 |            15.389206 |       77.078959 |  77.078959 |      1.0 |
    | 2010-12-12 14:00 |              0.2 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      1.0 |                      7.386143 |                   0.3 |             11.654171 |                 59.933948 |             125.726485 |                  58.67236 |            16.200644 |       60.254042 |  60.254042 |      1.0 |
    | 2010-12-12 15:00 |              0.2 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      1.0 |                      7.386143 |                   0.4 |              2.813062 |                 59.933948 |              30.347619 |                 14.162222 |            16.606362 |       14.837301 |  14.837301 |      1.0 |
    | 2010-12-12 16:00 |              0.2 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                 0.046405 |                      7.386143 |                   0.6 |              0.004444 |                 59.933948 |               0.047938 |                  0.153718 |              17.4178 |        0.168856 |   0.168856 | 0.046405 |
    | 2010-12-12 17:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.6 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |              17.4178 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 18:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.6 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |              17.4178 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 19:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.6 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |              17.4178 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 20:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.6 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |              17.4178 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 21:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.6 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |              17.4178 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 22:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.6 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |              17.4178 |             0.0 |        0.0 |      0.0 |
    | 2010-12-12 23:00 |              0.0 |          1.03126 |        -0.406588 |      8.660262 |    16.046405 |                      0.0 |                      7.386143 |                   0.6 |                   0.0 |                 59.933948 |                    0.0 |                       0.0 |              17.4178 |             0.0 |        0.0 |      0.0 |
"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import modeltools
from hydpy.interfaces import radiationinterfaces
from hydpy.models.meteo import meteo_model


class Model(modeltools.AdHocModel, radiationinterfaces.RadiationModel_V1):
    """|meteo_glob_morsim.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="Meteo-Glob-MORSIM",
        description="global radiation estimation adopted from MORECS/LARSIM",
    )

    INLET_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        meteo_model.Calc_EarthSunDistance_V1,
        meteo_model.Calc_SolarDeclination_V2,
        meteo_model.Calc_TimeOfSunrise_TimeOfSunset_V1,
        meteo_model.Calc_DailyPossibleSunshineDuration_V1,
        meteo_model.Calc_PossibleSunshineDuration_V2,
        meteo_model.Update_LoggedSunshineDuration_V1,
        meteo_model.Calc_DailySunshineDuration_V1,
        meteo_model.Calc_ExtraterrestrialRadiation_V2,
        meteo_model.Calc_ClearSkySolarRadiation_V1,
        meteo_model.Calc_DailyGlobalRadiation_V1,
        meteo_model.Calc_PortionDailyRadiation_V1,
        meteo_model.Adjust_ClearSkySolarRadiation_V1,
        meteo_model.Calc_UnadjustedGlobalRadiation_V1,
        meteo_model.Update_LoggedUnadjustedGlobalRadiation_V1,
        meteo_model.Calc_GlobalRadiation_V2,
    )
    ADD_METHODS = (meteo_model.Return_DailyGlobalRadiation_V1,)
    INTERFACE_METHODS = (
        meteo_model.Process_Radiation_V1,
        meteo_model.Get_PossibleSunshineDuration_V1,
        meteo_model.Get_SunshineDuration_V2,
        meteo_model.Get_ClearSkySolarRadiation_V1,
        meteo_model.Get_GlobalRadiation_V1,
    )
    OUTLET_METHODS = ()
    SENDER_METHODS = ()
    SUBMODELINTERFACES = ()
    SUBMODELS = ()


tester = Tester()
cythonizer = Cythonizer()
