# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""
.. _`German Federal Institute of Hydrology (BfG)`: https://www.bafg.de/EN
.. _`LIAS issue on GitHub`: https://github.com/hydpy-dev/hydpy/issues/98

The |sw1d.DOCNAME.long| model family member |sw1d_channel| allows combining different
storage and routing submodels for representing the 1-dimensional flow processes within
a single channel reach.

|sw1d_channel| is a "user model" whose task lies seldom in performing actual
calculations but merely in specifying individual reaches of large channel networks on
an |Element| basis, even if they must be solved later simultaneously (by the "composite
model" |sw1d_network|).  Both these models were initially developed on behalf of the
`German Federal Institute of Hydrology (BfG)`_ to apply a 1-dimensional implementation
of the "local inertial approximation of the shallow water equations" introduced by
:cite:t:`ref-Bates2010` and "stabilised" by :cite:t:`ref-Almeida2012`.  Still, they
should also apply to other approaches that discretise the shallow water equations in a
finite volume staggered grid manner.

Integration tests
=================

.. how_to_understand_integration_tests::

We select a simulation period of five hours and an "external" simulation step size of
five minutes for all integration tests:

>>> from hydpy import pub
>>> pub.timegrids = "2000-01-01 00:00", "2000-01-01 05:00", "5m"

The considered 20 km channel consists of eight segments with alternating lengths of two
and three kilometres:

>>> from hydpy.models.sw1d_channel import *
>>> parameterstep()
>>> nmbsegments(8)
>>> lengths = 2.0, 3.0, 2.0, 3.0, 2.0, 3.0, 2.0, 3.0

A valid |sw1d_channel| configuration requires one storage model that complies with the
|StorageModel_V1| at each segment.  We prepare |sw1d_storage| submodels with identical
channel bottom elevations and rectangular profiles.  As for other submodels of the
|sw1d.DOCNAME.long| model family, one specifies such geometries by sub-submodels that
comply with |CrossSectionModel_V2| interface.  Here, we select |wq_trapeze|:

>>> for i, length_ in enumerate(lengths):
...     with model.add_storagemodel_v1("sw1d_storage", position=i):
...         length(length_)
...         with model.add_crosssection_v2("wq_trapeze"):
...             nmbtrapezes(1)
...             bottomlevels(5.0)
...             bottomwidths(5.0)
...             sideslopes(0.0)

Additionally, |sw1d_channel| requires one routing model that complies with the
|RoutingModel_V2| interface between each pair of segments.  So, our example model
requires a total of seven routing models.  We select |sw1d_lias| and parametrise its
bottom elevation and channel profile as explained above.  Besides that, we need to
define the Strickler coefficient and two factors for stabilising the numerical
integration:

>>> for i in range(1, nmbsegments.value):
...     with model.add_routingmodel_v2("sw1d_lias", position=i):
...         lengthupstream(2.0 if i % 2 else 3.0)
...         lengthdownstream(3.0 if i % 2 else 2.0)
...         stricklercoefficient(1.0/0.03)
...         timestepfactor(0.7)
...         diffusionfactor(0.2)
...         with model.add_crosssection_v2("wq_trapeze"):
...             nmbtrapezes(1)
...             bottomlevels(5.0)
...             bottomwidths(5.0)
...             sideslopes(0.0)

The prepared model has neither a submodel for receiving nor releasing flow at the first
or last channel segment. We leave it like that to assign it to an |Element| instance
that is neither connected to any inlet nor outlet nodes:

>>> from hydpy import Element
>>> channel = Element("channel")
>>> channel.model = model

Next, we prepare a test function object to control the following test runs:

>>> from hydpy.core.testtools import IntegrationTest
>>> test = IntegrationTest(channel)
>>> test.plotting_options.axis1 = (factors.waterlevels,)

For convenience, we also define a function that prepares the storage models' initial
water volumes and the routing models' "old" discharges based on general or individual
water depth and discharge values:

>>> def prepare_inits(hs, qs):
...     if isinstance(hs, float):
...         hs = nmbsegments.value * [hs]
...     if isinstance(qs, float):
...         qs = (nmbsegments.value + 1) * [qs]
...     inits = []
...     for h, s in zip(hs, model.storagemodels):
...         length = s.parameters.control.length
...         c = s.crosssection.parameters.control
...         v = h * (c.bottomwidths[0] + h * c.sideslopes[0]) * length
...         inits.append((s.sequences.states.watervolume, v))
...     for q, r in zip(qs, model.routingmodels):
...         if r is not None:
...             inits.append((r.sequences.states.discharge, q))
...     test.inits = inits

.. _sw1d_channel_zero_inflow_and_outflow:

Zero inflow and outflow
_______________________

As mentioned above, there is no routing model at the inlet or outlet position, so there
can be no inflow or outflow.  Instead, we set the initial depths of 3 m for the first
four and 1 m for the last four segments to enforce water movement in this first
example:

>>> prepare_inits(hs=[3.0, 3.0, 3.0, 3.0, 1.0, 1.0, 1.0, 1.0], qs=0.0)

At the end of the simulation period, all segments' water levels have nearly reached the
average depth of 2 m.  Closer inspection reveals a small "overshooting" with higher
water levels in the last than in the first segments that reached its maximum after
about four hours:

.. integration-test::

    >>> conditions = test("sw1d_channel_zero_inflow_and_outflow", get_conditions="2000-01-01 00:00")
    |                date |  timestep |                                                                       waterlevels |                                                                                   discharges |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 41.932744 |      8.0       8.0   7.99689  7.659739   6.51142  6.001387       6.0          6.0 | 0.0        0.0        0.0   0.103665  17.116697   0.069362        0.0        0.0         0.0 |
    | 2000-01-01 00:05:00 | 41.932744 | 7.999992  7.996361  7.824697  7.649492  6.587536  6.077925  6.002109          6.0 | 0.0    0.00028   0.182249   5.922032   6.434402   3.897203   0.070336   0.000021         0.0 |
    | 2000-01-01 00:10:00 |  41.86989 | 7.993352  7.931422   7.78946  7.595997  6.715088   6.12184  6.026127      6.00139 | 0.0   0.221329   3.468262   4.642813   7.317577   3.065842   0.870068   0.069492         0.0 |
    | 2000-01-01 00:15:00 | 40.570615 | 7.942995  7.887898  7.761077  7.556577  6.807278   6.17491   6.04788     6.007794 | 0.0   1.678558   3.854736   4.800826   6.771817   3.698807    1.04529   0.320178         0.0 |
    | 2000-01-01 00:20:00 | 38.464288 | 7.876758   7.85486  7.731448  7.524701  6.879592    6.2307  6.069909     6.017934 | 0.0   2.207898   3.859806   4.847457   6.441228   4.030779   1.241302   0.506993         0.0 |
    | 2000-01-01 00:25:00 | 36.060713 | 7.818283   7.81743  7.702268  7.498342  6.934081  6.288067  6.093754      6.03057 | 0.0   1.949184   3.820718   4.793387   6.111343   4.295028    1.42665   0.631814         0.0 |
    | 2000-01-01 00:30:00 | 33.726739 | 7.771332   7.77554  7.671519  7.475733  6.974537  6.345257  6.120048      6.04518 | 0.0   1.565018   3.659513   4.684456   5.814913   4.466393   1.606941   0.730501         0.0 |
    | 2000-01-01 00:35:00 | 31.578426 |  7.73188  7.733155  7.638895  7.455195    7.0041  6.400921  6.148744     6.061651 | 0.0   1.315079   3.434327   4.521823   5.548742   4.563291   1.780091   0.823557         0.0 |
    | 2000-01-01 00:40:00 | 29.590331 | 7.695544  7.693043  7.605781  7.435459  7.025353  6.454008  6.179562     6.079997 | 0.0   1.211187   3.216789   4.320583   5.307354   4.598933   1.944567   0.917288         0.0 |
    | 2000-01-01 00:45:00 | 27.697934 | 7.660109  7.655911  7.573675  7.415932  7.040288  6.503779  6.212122     6.100249 | 0.0   1.181168   3.037773   4.107951   5.084325   4.586478   2.097941   1.012595         0.0 |
    | 2000-01-01 00:50:00 | 25.854608 | 7.625278  7.621259  7.543316  7.396526  7.050449  6.549766  6.245998     6.122422 | 0.0   1.161016   2.893595   3.905593   4.875904   4.537222    2.23787   1.108668         0.0 |
    | 2000-01-01 00:55:00 | 24.044617 | 7.591516  7.588403  7.514686  7.377366  7.057039  6.591734  6.280746     6.146505 | 0.0   1.125422   2.768196   3.722499   4.680467   4.460811     2.3624   1.204147         0.0 |
    | 2000-01-01 01:00:00 | 22.271186 | 7.559223  7.556936  7.487453  7.358578  7.060989  6.629639  6.315927     6.172454 | 0.0   1.076444   2.649834     3.5576   4.497034   4.365364    2.47012   1.297422         0.0 |
    | 2000-01-01 01:05:00 | 20.541621 | 7.528507  7.526715  7.461316  7.340213  7.062994  6.663585  6.351131     6.200189 | 0.0   1.023846   2.534864   3.406128   4.324343   4.257504   2.560214   1.386749         0.0 |
    | 2000-01-01 01:10:00 | 18.859766 | 7.499272  7.497721  7.436118  7.322279  7.063549  6.693783  6.385994     6.229596 | 0.0   0.974495   2.424218   3.264124   4.160846   4.142345   2.632447    1.47035         0.0 |
    | 2000-01-01 01:15:00 | 17.225223 | 7.471356   7.46994  7.411816  7.304769  7.062995  6.720512  6.420212     6.260525 | 0.0   0.930529   2.319569   3.129639    4.00511   4.023586   2.687091   1.546498         0.0 |
    | 2000-01-01 01:20:00 | 15.635338 | 7.444619  7.443332  7.388403  7.287689  7.061565  6.744091  6.453548     6.292798 | 0.0   0.891234   2.221637   3.002079   3.856098   3.903739    2.72482   1.613616         0.0 |
    | 2000-01-01 01:25:00 | 14.087165 | 7.418965  7.417833  7.365868  7.271051  7.059429  6.764847  6.485835     6.326205 | 0.0   0.855147   2.130122   2.881273   3.713189   3.784411   2.746582    1.67035         0.0 |
    | 2000-01-01 01:30:00 | 12.578396 |  7.39433  7.393371  7.344187  7.254869  7.056713  6.783109   6.51697     6.360518 | 0.0    0.82117   2.044235    2.76695   3.576035   3.666568   2.753481   1.715636         0.0 |
    | 2000-01-01 01:35:00 | 11.107466 | 7.370666  7.369884  7.323322  7.239154  7.053523   6.79919  6.546909     6.395492 | 0.0   0.788783   1.963163   2.658648   3.444403   3.550737   2.746682   1.748731         0.0 |
    | 2000-01-01 01:40:00 |   9.67329 | 7.347931  7.347315  7.303237  7.223909   7.04995  6.813386  6.575652     6.430877 | 0.0   0.757842   1.886273   2.555805   3.318058   3.437146   2.727335   1.769232         0.0 |
    | 2000-01-01 01:45:00 |  8.274999 | 7.326081  7.325619  7.283894  7.209132  7.046077  6.825972  6.603235     6.466418 | 0.0   0.728329   1.813123   2.457882   3.196719   3.325814   2.696526   1.777071         0.0 |
    | 2000-01-01 01:50:00 |   6.91179 | 7.305075  7.304755  7.265263  7.194819   7.04198    6.8372  6.629718     6.501868 | 0.0   0.700211   1.743393   2.364414   3.080056   3.216616   2.655249   1.772492         0.0 |
    | 2000-01-01 01:55:00 |  5.582889 | 7.284873  7.284687  7.247317  7.180966  7.037731  6.847298  6.655169     6.536988 | 0.0   0.673412    1.67682   2.275012   2.967695   3.109324   2.604399   1.756018         0.0 |
    | 2000-01-01 02:00:00 |  4.287568 | 7.265438   7.26538  7.230032  7.167568  7.033399  6.856475  6.679661     6.571556 | 0.0    0.64782    1.61316   2.189331   2.859236   3.003643   2.544779   1.728405         0.0 |
    | 2000-01-01 02:05:00 |  3.025166 | 7.246738  7.246803  7.213386  7.154623   7.02905  6.864918  6.703256     6.605368 | 0.0   0.623323   1.552172   2.107049   2.754256   2.899238   2.477123   1.690598         0.0 |
    | 2000-01-01 02:10:00 |  1.795106 | 7.228744  7.228927  7.197359  7.142134  7.024747   6.87279   6.72601     6.638242 | 0.0   0.599814   1.493619   2.027846   2.652318   2.795754   2.402115   1.643674         0.0 |
    | 2000-01-01 02:15:00 |  0.596896 | 7.211428  7.211726  7.181935  7.130102  7.020551  6.880239  6.747958     6.670018 | 0.0   0.577198   1.437266   1.951396   2.552973   2.692831   2.320423   1.588801         0.0 |
    | 2000-01-01 02:20:00 |     300.0 | 7.194761  7.195181  7.167095  7.118539  7.016521  6.887386  6.769131     6.700556 | 0.0   0.555555   1.382809   1.877496   2.455693   2.590029   2.232663   1.526904         0.0 |
    | 2000-01-01 02:25:00 |     300.0 | 7.179143  7.178767  7.153074  7.107295  7.012697  6.894305  6.789218     6.730212 | 0.0   0.520605    1.34131   1.808668   2.370851   2.498317   2.152358   1.482801         0.0 |
    | 2000-01-01 02:30:00 |     300.0 | 7.164041  7.163107  7.139327  7.096616  7.009094  6.901059  6.808633     6.758487 | 0.0   0.503419    1.28639   1.744609   2.278535   2.398625   2.060935   1.413763         0.0 |
    | 2000-01-01 02:35:00 |     300.0 |  7.14924  7.148327  7.125966  7.086446  7.005787  6.907742  6.827426     6.785206 | 0.0   0.493359   1.232396   1.677772   2.186283   2.296518   1.962357   1.335919         0.0 |
    | 2000-01-01 02:40:00 |     300.0 | 7.134803   7.13427  7.113223  7.076738  7.002818  6.914435  6.845526      6.81031 | 0.0   0.481213   1.184035   1.608792   2.094209   2.193177   1.858547   1.255216         0.0 |
    | 2000-01-01 02:45:00 |     300.0 | 7.120879  7.120776  7.101168   7.06752  7.000211  6.921182  6.862846     6.833787 | 0.0   0.464144    1.13886   1.540716   2.001626   2.088528   1.751174   1.173856         0.0 |
    | 2000-01-01 02:50:00 |     300.0 | 7.107569  7.107779  7.089733  7.058843  6.997996  6.927994  6.879314     6.855642 | 0.0   0.443671   1.093526   1.474669   1.908479   1.982306   1.641693   1.092748         0.0 |
    | 2000-01-01 02:55:00 |     300.0 |   7.0949  7.095291  7.078844   7.05074  6.996207  6.934858   6.89488      6.87589 | 0.0   0.422306   1.046697   1.409677   1.814849   1.874506   1.531302   1.012416         0.0 |
    | 2000-01-01 03:00:00 |     300.0 |  7.08285  7.083351  7.068477  7.043214  6.994861  6.941745  6.909512     6.894556 | 0.0   0.401668   0.998683   1.344234   1.720515   1.765371   1.421011   0.933302         0.0 |
    | 2000-01-01 03:05:00 |     300.0 | 7.071386   7.07199  7.058659  7.036262  6.993955  6.948615  6.923187     6.911674 | 0.0    0.38214   0.950157   1.277423   1.625017   1.655202   1.311701   0.855859         0.0 |
    | 2000-01-01 03:10:00 |     300.0 | 7.060487  7.061228   7.04943  7.029884  6.993466  6.955417  6.935895     6.927284 | 0.0   0.363293   0.901399   1.209035   1.527933   1.544235   1.204145    0.78054         0.0 |
    | 2000-01-01 03:15:00 |     300.0 | 7.050154  7.051073  7.040821  7.024087  6.993358   6.96209  6.947632      6.94144 | 0.0   0.344439   0.852209   1.139185   1.429057   1.432655   1.098997   0.707776         0.0 |
    | 2000-01-01 03:20:00 |     300.0 | 7.040403   7.04153  7.032847  7.018878  6.993591  6.968568  6.958397     6.954198 | 0.0   0.325005   0.802146   1.067939   1.328408   1.320664   0.996774   0.637926         0.0 |
    | 2000-01-01 03:25:00 |     300.0 | 7.031263  7.032609  7.025514  7.014258  6.994119  6.974782  6.968196     6.965623 | 0.0   0.304682   0.750747   0.995179    1.22614   1.208538   0.897862   0.571242         0.0 |
    | 2000-01-01 03:30:00 |     300.0 | 7.022761  7.024324  7.018824  7.010222  6.994894  6.980664  6.977036      6.97578 | 0.0   0.283391   0.697651   0.920653   1.122462   1.096635   0.802523   0.507842         0.0 |
    | 2000-01-01 03:35:00 |     300.0 | 7.014926  7.016695   7.01278  7.006752  6.995861  6.986153  6.984932     6.984735 | 0.0   0.261178   0.642619   0.844086   1.017589   0.985357   0.710905    0.44772         0.0 |
    | 2000-01-01 03:40:00 |     300.0 | 7.007783  7.009747  7.007387  7.003823   6.99696  6.991194    6.9919      6.99255 | 0.0   0.238099   0.585506   0.765278   0.911751   0.875101   0.623051   0.390767         0.0 |
    | 2000-01-01 03:45:00 |     300.0 | 7.001358  7.003505  7.002649  7.001402   6.99813   6.99574  6.997963     6.999286 | 0.0   0.214166   0.526232   0.684152   0.805203   0.766215   0.538901   0.336802         0.0 |
    | 2000-01-01 03:50:00 |     300.0 | 6.995678  6.997997   6.99857  6.999452  6.999308  6.999754  7.003144     7.004998 | 0.0   0.189335   0.464764    0.60075   0.698243   0.658976   0.458301   0.285605         0.0 |
    | 2000-01-01 03:55:00 |     300.0 | 6.990772  6.993245  6.995147  6.997932  7.000436  7.003205  7.007467     7.009737 | 0.0   0.163546    0.40111   0.515219   0.591202   0.553589   0.381022   0.236926         0.0 |
    | 2000-01-01 04:00:00 |     300.0 | 6.986669  6.989274  6.992373  6.996799  7.001463  7.006073  7.010955     7.013547 | 0.0    0.13676   0.335337   0.427783   0.484427   0.450191   0.306782   0.190502         0.0 |
    | 2000-01-01 04:05:00 |     300.0 | 6.983399  6.986102  6.990239  6.996009  7.002345  7.008345  7.013632     7.016468 | 0.0   0.108995   0.267588    0.33873   0.378258   0.348872   0.235278   0.146056         0.0 |
    | 2000-01-01 04:10:00 |     300.0 | 6.980989  6.983747   6.98873  6.995517  7.003045  7.010015  7.015519     7.018534 | 0.0   0.080337   0.198098   0.248407   0.273014   0.249683   0.166206   0.103317         0.0 |
    | 2000-01-01 04:15:00 |     300.0 |  6.97946  6.982221  6.987829  6.995282  7.003535  7.011082  7.016636     7.019775 | 0.0   0.050946   0.127216   0.157227    0.16898   0.152653   0.099283   0.062028         0.0 |
    | 2000-01-01 04:20:00 |     300.0 | 6.978829  6.981534  6.987521  6.995267  7.003794  7.011553  7.017005     7.020214 | 0.0    0.02104   0.055406   0.065681   0.066424   0.057791   0.034262   0.021969         0.0 |
    | 2000-01-01 04:25:00 |     300.0 | 6.979102  6.981687  6.987788  6.995442  7.003809  7.011436  7.016645     7.019873 | 0.0  -0.009105  -0.016742  -0.025635  -0.034386  -0.034891  -0.029054  -0.017033         0.0 |
    | 2000-01-01 04:30:00 |     300.0 | 6.980272  6.982663  6.988592   6.99577  7.003578  7.010761  7.015596      7.01878 | 0.0  -0.038998   -0.08782  -0.114627  -0.131043  -0.123366  -0.089609   -0.05467         0.0 |
    | 2000-01-01 04:35:00 |     300.0 |  6.98229  6.984355  6.989816    6.9962  7.003153  7.009639  7.013962     7.016992 | 0.0   -0.06725  -0.151826  -0.192607  -0.214119  -0.199935  -0.143875  -0.089381         0.0 |
    | 2000-01-01 04:40:00 |     300.0 | 6.985039  6.986604  6.991318   6.99669  7.002605  7.008212  7.011873     7.014604 | 0.0  -0.091648  -0.204122   -0.25421  -0.278681  -0.260404   -0.18903  -0.119416         0.0 |
    | 2000-01-01 04:45:00 |     300.0 |  6.98833  6.989243  6.992985  6.997212     7.002  7.006607  7.009473     7.011746 | 0.0   -0.10968  -0.241625  -0.297188  -0.323311  -0.303171  -0.222919  -0.142907         0.0 |
    | 2000-01-01 04:50:00 |     300.0 | 6.991916  6.992114  6.994745  6.997758  7.001387  7.004922  7.006892     7.008579 | 0.0  -0.119553  -0.263108  -0.321762  -0.349079  -0.328638  -0.244374  -0.158353         0.0 |
    | 2000-01-01 04:55:00 |     300.0 | 6.995544  6.995078  6.996558  6.998332  7.000792   7.00322  7.004246     7.005277 | 0.0  -0.120926    -0.2691  -0.329544  -0.358235   -0.33839  -0.253296  -0.165073         0.0 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_channel_higher_precision:

Higher precision
________________

Some water level trajectories calculated in the
:ref:`sw1d_channel_zero_inflow_and_outflow` example show some "edges" in the first ten
minutes, which indicates possible improvements by increasing numerical precision.  We
do this by setting |TimeStepFactor| to the rather small value of 0.1:

>>> for routingmodel in model.routingmodels.submodels[1:-1]:
...     routingmodel.parameters.control.timestepfactor(0.1)

Now, all water level trajectories have a smooth appearance:

.. integration-test::

    >>> test("sw1d_channel_higher_precision")
    |                date |  timestep |                                                                       waterlevels |                                                                             discharges |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  5.055286 | 7.999359  7.996438  7.970465  7.914823  6.134216  6.016356  6.004034     6.000333 | 0.0   0.021356  0.199435  1.183919  5.442766  0.968903  0.151096  0.016637         0.0 |
    | 2000-01-01 00:05:00 |  4.697458 | 7.986212  7.975735  7.915509  7.847529  6.267498  6.045112   6.02026     6.005304 | 0.0   0.438228   1.47342  3.305312  6.670002  2.227249  0.789434  0.248559         0.0 |
    | 2000-01-01 00:10:00 |  3.581945 | 7.956047  7.947591  7.870189  7.792268  6.390078  6.076048  6.038611     6.014142 | 0.0   1.005517  2.412676  3.923344  6.686387  2.600387  1.053587  0.441905         0.0 |
    | 2000-01-01 00:15:00 |  1.949215 | 7.917942  7.916689  7.830781  7.743287  6.500739  6.109846  6.058508     6.024865 | 0.0   1.270155  2.815262  4.128856  6.577921  2.889252  1.199389  0.536133         0.0 |
    | 2000-01-01 00:20:00 |  0.094214 | 7.878259  7.884197  7.793829  7.698516  6.598885  6.146288  6.080006     6.037014 | 0.0   1.322779  2.947389  4.179134  6.417709  3.146168  1.324069  0.607487         0.0 |
    | 2000-01-01 00:25:00 | 36.003106 | 7.839418  7.850838  7.758049   7.65722  6.684292  6.184883  6.103065      6.05051 | 0.0   1.294695  2.962627  4.155295  6.220066  3.373167  1.443412  0.674757         0.0 |
    | 2000-01-01 00:30:00 | 34.308016 | 7.802277  7.816802  7.722947  7.617961  6.759264  6.225717  6.127213     6.065053 | 0.0   1.238042  2.939838  4.109898  6.072866  3.573782  1.532068  0.727156         0.0 |
    | 2000-01-01 00:35:00 | 32.617505 | 7.766466  7.782843  7.688801  7.581676  6.822391  6.267623  6.152796     6.080889 | 0.0   1.193686  2.891649  4.029838  5.844068   3.73985  1.644575  0.791808         0.0 |
    | 2000-01-01 00:40:00 | 30.936158 | 7.731846  7.749315    7.6558  7.548064  6.874703  6.309935  6.179611     6.098046 | 0.0   1.154013  2.830421  3.930452  5.611056  3.867321  1.751701  0.857852         0.0 |
    | 2000-01-01 00:45:00 | 29.267696 | 7.698335  7.716424  7.624015  7.516868   6.91739  6.352068  6.207468     6.116501 | 0.0    1.11702  2.761583  3.821099  5.380888  3.957966  1.851305  0.922739         0.0 |
    | 2000-01-01 00:50:00 |  27.61513 | 7.665887  7.684299  7.593461  7.487843  6.951679  6.393507  6.236172     6.136218 | 0.0   1.081616  2.687876  3.706324  5.157569  4.014603  1.942654  0.985864         0.0 |
    | 2000-01-01 00:55:00 | 25.981161 | 7.634473   7.65302  7.564121  7.460754  6.978763  6.433818  6.265526     6.157154 | 0.0   1.047111  2.611064  3.589087  4.943565  4.040786  2.025257  1.046797         0.0 |
    | 2000-01-01 01:00:00 |  24.36822 | 7.604078  7.622633  7.535951   7.43538  6.999756  6.472651  6.295335     6.179256 | 0.0   1.013178  2.532494  3.471468  4.740189  4.040408  2.098742  1.105089         0.0 |
    | 2000-01-01 01:05:00 | 22.778413 | 7.574686  7.593163  7.508901   7.41152  7.015671  6.509742  6.325412     6.202461 | 0.0   0.979745  2.453253  3.354931  4.547895  4.017397  2.162842  1.160267         0.0 |
    | 2000-01-01 01:10:00 | 21.213465 |  7.54628  7.564616  7.482913  7.388999  7.027402  6.544904  6.355579     6.226698 | 0.0   0.946873   2.37421   3.24048  4.366537  3.975509  2.217393  1.211848         0.0 |
    | 2000-01-01 01:15:00 | 19.674726 | 7.518839  7.536989  7.457931  7.367663  7.035724  6.578022  6.385668     6.251885 | 0.0    0.91467  2.296039  3.128776  4.195581  3.918193   2.26233   1.25935         0.0 |
    | 2000-01-01 01:20:00 | 18.163205 | 7.492342  7.510269  7.433901  7.347382  7.041296  6.609038  6.415529     6.277931 | 0.0   0.883242  2.219239  3.020219  4.034275  3.848515  2.297681  1.302311         0.0 |
    | 2000-01-01 01:25:00 |  16.67962 | 7.466762  7.484439  7.410775  7.328047  7.044675   6.63795  6.445027     6.304737 | 0.0   0.852673  2.144158   2.91502  3.881765  3.769137  2.323563  1.340304         0.0 |
    | 2000-01-01 01:30:00 | 15.224463 | 7.442072  7.459479  7.388509  7.309568  7.046321  6.664793  6.474044     6.332196 | 0.0   0.823013  2.071019   2.81325  3.737182  3.682316  2.340174  1.372947         0.0 |
    | 2000-01-01 01:35:00 | 13.798047 | 7.418243  7.435366  7.367061  7.291872  7.046614  6.689636   6.50248     6.360194 | 0.0   0.794287  1.999949  2.714884  3.599688  3.589933  2.347782  1.399916         0.0 |
    | 2000-01-01 01:40:00 | 12.400562 | 7.395248  7.412076  7.346396  7.274899  7.045863  6.712572  6.530253     6.388614 | 0.0   0.766492  1.930999   2.61983  3.468505  3.493523  2.346713  1.420955         0.0 |
    | 2000-01-01 01:45:00 | 11.032112 |  7.37306  7.389585  7.326482  7.258599  7.044322  6.733711  6.557296     6.417331 | 0.0    0.73961  1.864162  2.527954  3.342929   3.39432  2.337345  1.435884         0.0 |
    | 2000-01-01 01:50:00 |  9.692751 | 7.351652  7.367869  7.307291  7.242934  7.042193  6.753175  6.583561     6.446223 | 0.0   0.713606  1.799392  2.439091  3.222331  3.293296  2.320092    1.4446         0.0 |
    | 2000-01-01 01:55:00 |  8.382503 | 7.330999  7.346906  7.288797  7.227873  7.039641  6.771092  6.609011     6.475165 | 0.0   0.688439  1.736611  2.353064  3.106152  3.191203  2.295401  1.447082         0.0 |
    | 2000-01-01 02:00:00 |   7.10139 | 7.311077  7.326672  7.270978  7.213388    7.0368  6.787589  6.633621     6.504033 | 0.0   0.664059  1.675724  2.269684    2.9939  3.088608  2.263734   1.44339         0.0 |
    | 2000-01-01 02:05:00 |  5.849441 | 7.291864  7.307148  7.253814  7.199461  7.033776  6.802796  6.657378     6.532706 | 0.0   0.640415  1.616621   2.18876  2.885135  2.985928  2.225569  1.433657         0.0 |
    | 2000-01-01 02:10:00 |  4.626708 | 7.273341  7.288314  7.237287  7.186074  7.030656  6.816838  6.680277     6.561068 | 0.0   0.617453  1.559184  2.110101  2.779468  2.883458  2.181388  1.418094         0.0 |
    | 2000-01-01 02:15:00 |  3.433274 | 7.255487   7.27015   7.22138  7.173213  7.027511  6.829832  6.702318     6.589007 | 0.0   0.595119  1.503291  2.033519  2.676551  2.781396  2.131677  1.396972         0.0 |
    | 2000-01-01 02:20:00 |   2.26926 | 7.238286  7.252641   7.20608  7.160868  7.024397  6.841891  6.723507      6.61642 | 0.0   0.573358  1.448817  1.958829  2.576069   2.67987  2.076919  1.370624         0.0 |
    | 2000-01-01 02:25:00 |  1.134831 | 7.221723  7.235771  7.191373   7.14903  7.021361  6.853118  6.743852     6.643208 | 0.0   0.552116  1.395637  1.885851   2.47774  2.578949  2.017595  1.339429         0.0 |
    | 2000-01-01 02:30:00 |  0.030204 | 7.205783  7.219525   7.17725  7.137692   7.01844  6.863608  6.763363     6.669284 | 0.0   0.531341  1.343627   1.81441  2.381304  2.478665  1.954178  1.303806         0.0 |
    | 2000-01-01 02:35:00 | 42.026752 | 7.190447    7.2039   7.16369  7.126856  7.015668  6.873439  6.782063      6.69456 | 0.0   0.511191   1.29245  1.744437  2.286257  2.378664   1.88708  1.263766         0.0 |
    | 2000-01-01 02:40:00 |   41.1302 | 7.175893  7.188582  7.150726  7.116253  7.012942  6.882806  6.799802     6.719449 | 0.0   0.485131  1.250992  1.683134  2.213299  2.304143  1.835794  1.244491         0.0 |
    | 2000-01-01 02:45:00 | 40.258369 | 7.161825  7.173889  7.138193  7.106152  7.010403  6.891622  6.816874     6.743473 | 0.0   0.468947   1.20361  1.621361  2.126415  2.211055  1.770262  1.201191         0.0 |
    | 2000-01-01 02:50:00 | 39.410071 | 7.148228  7.159822  7.126175  7.096566  7.008084  6.899964  6.833241     6.766496 | 0.0   0.453232  1.156578  1.557195  2.036478  2.113777  1.696705   1.15113         0.0 |
    | 2000-01-01 02:55:00 | 38.586127 | 7.135129  7.146361  7.114695  7.087491  7.005999  6.907891  6.848879     6.788455 | 0.0   0.436619  1.109659  1.492331  1.946098  2.015593  1.619227  1.097956         0.0 |
    | 2000-01-01 03:00:00 |  37.78789 | 7.122556  7.133489  7.103756  7.078918  7.004155  6.915452  6.863774     6.809315 | 0.0   0.419114  1.062741  1.427371  1.856024  1.917491  1.539457  1.042973         0.0 |
    | 2000-01-01 03:05:00 |  37.01666 | 7.110526  7.121192  7.093356  7.070839  7.002554  6.922683  6.877915     6.829052 | 0.0   0.400989  1.015829   1.36248  1.766418  1.819781  1.458242   0.98686         0.0 |
    | 2000-01-01 03:10:00 |  36.27356 | 7.099051  7.109463  7.083494  7.063247  7.001195  6.929611  6.891297     6.847654 | 0.0   0.382491  0.968943  1.297685   1.67727  1.722577  1.376166    0.9301         0.0 |
    | 2000-01-01 03:15:00 | 35.559559 | 7.088138  7.098297  7.074168  7.056136  7.000073  6.936255  6.903916     6.865116 | 0.0   0.363786  0.922086  1.232962  1.588529  1.625939  1.293714  0.873099         0.0 |
    | 2000-01-01 03:20:00 | 34.875529 | 7.077789  7.087692  7.065377  7.049498   6.99918  6.942627  6.915769      6.88144 | 0.0   0.344963  0.875238  1.168278  1.500149  1.529927  1.211317  0.816216         0.0 |
    | 2000-01-01 03:25:00 | 34.222293 | 7.068007  7.077645   7.05712  7.043328  6.998504  6.948732  6.926857     6.896636 | 0.0   0.326063  0.828366  1.103596  1.412106  1.434617   1.12937  0.759776         0.0 |
    | 2000-01-01 03:30:00 |  33.60065 | 7.058794  7.068159  7.049396  7.037618  6.998033   6.95457  6.937182     6.910717 | 0.0   0.307096  0.781429  1.038884  1.324398  1.340102  1.048234   0.70407         0.0 |
    | 2000-01-01 03:35:00 | 33.011392 | 7.050152  7.059232  7.042204  7.032359   6.99775  6.960135  6.946748     6.923704 | 0.0   0.288061  0.734392  0.974123  1.237053  1.246497   0.96824  0.649353         0.0 |
    | 2000-01-01 03:40:00 | 32.455306 | 7.042084  7.050867  7.035542  7.027543  6.997636   6.96542  6.955563     6.935621 | 0.0   0.268949  0.687224  0.909307  1.150122  1.153931  0.889684   0.59585         0.0 |
    | 2000-01-01 03:45:00 | 31.933167 | 7.034591  7.043064  7.029405  7.023158   6.99767  6.970414  6.963636     6.946496 | 0.0   0.249756   0.63991  0.844451   1.06369  1.062552  0.812833  0.543749         0.0 |
    | 2000-01-01 03:50:00 | 31.445726 | 7.027677  7.035824  7.023791  7.019193   6.99783  6.975106  6.970977      6.95636 | 0.0   0.230482  0.592454   0.77959  0.977865  0.972521  0.737919   0.49321         0.0 |
    | 2000-01-01 03:55:00 | 30.993694 | 7.021342  7.029149  7.018694  7.015633  6.998093  6.979484    6.9776     6.965247 | 0.0   0.211139  0.544879  0.714788  0.892787  0.884011  0.665145  0.444362         0.0 |
    | 2000-01-01 04:00:00 | 30.577717 |  7.01559   7.02304  7.014107  7.012463  6.998436  6.983534  6.983522     6.973194 | 0.0   0.191745  0.497231  0.650138  0.808625   0.79721  0.594688  0.397308         0.0 |
    | 2000-01-01 04:05:00 | 30.198359 |  7.01042  7.017495  7.010021  7.009667  6.998834  6.987246  6.988759     6.980236 | 0.0   0.172332  0.449585  0.585763  0.725577  0.712313  0.526697  0.352128         0.0 |
    | 2000-01-01 04:10:00 | 29.860333 | 7.005832  7.012513  7.006428  7.007226  6.999264  6.990611  6.993331     6.986414 | 0.0   0.152945  0.402043  0.521822  0.643868   0.62953  0.461302  0.308883         0.0 |
    | 2000-01-01 04:15:00 | 29.589266 | 7.001823  7.008091  7.003315  7.005121  6.999704   6.99362  6.997261     6.991766 | 0.0   0.133644  0.354738  0.458509  0.563752  0.549078  0.398616  0.267617         0.0 |
    | 2000-01-01 04:20:00 | 29.356723 | 6.998388  7.004224  7.000668  7.003332  7.000134  6.996269  7.000573     6.996334 | 0.0     0.1145  0.307845  0.396061  0.485521  0.471197  0.338743  0.228371         0.0 |
    | 2000-01-01 04:25:00 |  29.18498 | 6.995519  7.000905  6.998473  7.001837  7.000535  6.998556  7.003291     7.000157 | 0.0   0.095606  0.261566  0.334758  0.409494  0.396135  0.281786  0.191174         0.0 |
    | 2000-01-01 04:30:00 | 29.291738 | 6.993207  6.998123  6.996709  7.000615   7.00089  7.000482  7.005445     7.003278 | 0.0    0.07708  0.216139  0.274916  0.336007  0.324148  0.227844  0.156044         0.0 |
    | 2000-01-01 04:35:00 | 29.448203 | 6.991436  6.995867  6.995358  6.999644  7.001188  7.002053  7.007064     7.005739 | 0.0   0.059028   0.17187  0.216905  0.265472  0.255553  0.177023  0.123041         0.0 |
    | 2000-01-01 04:40:00 | 29.565954 | 6.990188  6.994117  6.994396  6.998901  7.001417  7.003277  7.008182     7.007583 | 0.0   0.041601  0.129085  0.161149  0.198314  0.190673  0.129459  0.092207         0.0 |
    | 2000-01-01 04:45:00 | 29.647677 | 6.989439  6.992853  6.993798  6.998363  7.001571  7.004168  7.008833     7.008855 | 0.0   0.024971  0.088151  0.108111  0.134991  0.129852  0.085302  0.063592         0.0 |
    | 2000-01-01 04:50:00 | 29.696155 | 6.989159   6.99205  6.993533  6.998009  7.001647  7.004743  7.009057       7.0096 | 0.0    0.00932  0.049481    0.0583     0.076  0.073473  0.044724   0.03727         0.0 |
    | 2000-01-01 04:55:00 | 29.714293 | 6.989461  6.991538  6.993633  6.997815  7.001644   7.00503  7.008885     7.009868 | 0.0  -0.010055  0.015535  0.012206  0.021919  0.022023  0.007684  0.013407         0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_channel_no_additional_diffusion:

No additional diffusion
_______________________

However, despite the smoother appearance, one might still not be completely satisfied
with the results gained in the :ref:`sw1d_channel_higher_precision` example.  First,
the levelling out of the water depths is slightly slowed down.  Second, the first and
second channel segments' water level trajectories intersect after a few minutes.  Please
refer to the `LIAS issue on GitHub`_, where we discuss this and related problems in
more detail and hopefully find ways for improvement.  Here, we show the simple solution
of omitting the additional diffusion introduced by :cite:t:`ref-Almeida2012` by setting
|DiffusionFactor| to zero:

>>> for routingmodel in model.routingmodels.submodels[1:-1]:
...     routingmodel.parameters.control.diffusionfactor(0.0)

Now, everything looks as expected.  Hence, omitting additional diffusion and setting
the time step factor to small values seems like the solution to achieving good results.
Still, one needs to remember that it comes with the cost of additional computation
time:

.. integration-test::

    >>> test("sw1d_channel_no_additional_diffusion")
    |                date |  timestep |                                                                       waterlevels |                                                                                   discharges |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  5.065796 | 7.999999  7.999887  7.991889  7.870648  6.197894   6.00293   6.00002          6.0 | 0.0   0.000036   0.005663   0.276036   6.743655   0.147178   0.000663   0.000002         0.0 |
    | 2000-01-01 00:05:00 |  5.042995 |   7.9998  7.995718   7.93679  7.746357  6.411724  6.025152  6.000833     6.000008 | 0.0   0.006618   0.215091   2.051727   8.266236   1.138593   0.027514   0.000417         0.0 |
    | 2000-01-01 00:10:00 |  4.807163 | 7.996774  7.974025  7.855656  7.667033  6.578773   6.06747  6.005777     6.000152 | 0.0   0.100887   1.185541   3.890007    7.85624   2.287923   0.171993    0.00719         0.0 |
    | 2000-01-01 00:15:00 |  3.966838 | 7.981655   7.93136  7.798521  7.608942  6.709932   6.11968  6.018387     6.001021 | 0.0   0.503973   2.637238   4.541731   7.446246   3.074296   0.463797   0.043453         0.0 |
    | 2000-01-01 00:20:00 |   2.31299 | 7.944131  7.885437   7.76457   7.56192  6.811523   6.17672  6.037862     6.003866 | 0.0   1.250786   3.546899   4.678589   7.029729   3.643358   0.791376   0.142222         0.0 |
    | 2000-01-01 00:25:00 | 37.638557 | 7.887266   7.84776   7.73529  7.524712  6.888021  6.236618  6.060806     6.009988 | 0.0   1.895506   3.779353   4.755375    6.61578   4.065851   1.070954   0.306128         0.0 |
    | 2000-01-01 00:30:00 | 35.332825 | 7.825932  7.813687  7.704449  7.495573   6.94416  6.297645  6.085397     6.019803 | 0.0   2.044446   3.748141   4.776152    6.23307   4.361757   1.310394   0.490719         0.0 |
    | 2000-01-01 00:35:00 | 33.002023 | 7.772477  7.776112  7.672988  7.471848  6.984546  6.357988  6.111793     6.032849 | 0.0   1.781864   3.660598   4.709309   5.895557   4.549351   1.532207   0.652336         0.0 |
    | 2000-01-01 00:40:00 | 30.812599 | 7.730275  7.733768  7.641421  7.451305  7.013188  6.416117  6.140556     6.048516 | 0.0   1.406731   3.523903   4.576149   5.603291   4.648575   1.742123   0.783356         0.0 |
    | 2000-01-01 00:45:00 | 28.806719 | 7.695922  7.690122  7.609172  7.432369  7.033286  6.470943  6.171789     6.066454 | 0.0   1.145091   3.327391    4.40235    5.34918   4.679249   1.937958   0.896866         0.0 |
    | 2000-01-01 00:50:00 | 26.939197 | 7.663844  7.649407  7.576289  7.413916  7.047235  6.521782  6.205148     6.086551 | 0.0   1.069273   3.105041   4.201141   5.123791   4.658795   2.116847   1.004861         0.0 |
    | 2000-01-01 00:55:00 |  25.13268 | 7.630307  7.613615  7.543966  7.395262  7.056722  6.568272  6.240082       6.1088 | 0.0   1.117881   2.907502   3.984933   4.917619   4.601393   2.276881   1.112431         0.0 |
    | 2000-01-01 01:00:00 | 23.332882 | 7.595034  7.581893  7.513663  7.376237  7.062882  6.610292  6.275994     6.133197 | 0.0   1.175784   2.761883   3.771981   4.723244   4.517922   2.416936   1.219875         0.0 |
    | 2000-01-01 01:05:00 | 21.533371 |  7.55999  7.552087  7.486006  7.357105  7.066495  6.647892  6.312324     6.159705 | 0.0   1.168141   2.658424   3.580318   4.536886   4.416457   2.536433   1.325405         0.0 |
    | 2000-01-01 01:10:00 | 19.761038 | 7.527161  7.522619  7.460496  7.338302   7.06815  6.681249  6.348581     6.188239 | 0.0   1.094293   2.567701   3.418049   4.358229   4.303067   2.635237   1.426668         0.0 |
    | 2000-01-01 01:15:00 | 18.046165 | 7.497273  7.493201  7.436103  7.320135  7.068331  6.710627   6.38435     6.218665 | 0.0   0.996265   2.467146   3.280228   4.188586   4.182556   2.713637   1.521339         0.0 |
    | 2000-01-01 01:20:00 | 16.402282 | 7.469829  7.464427  7.412067  7.302653  7.067428  6.736358  6.419301     6.250811 | 0.0   0.914783   2.353479   3.154694    4.02876   4.058835   2.772331   1.607305         0.0 |
    | 2000-01-01 01:25:00 | 14.823334 | 7.443836   7.43701  7.388252  7.285716  7.065721  6.758808   6.45319     6.284466 | 0.0   0.866447   2.237311   3.031135   3.878005   3.934926   2.812378   1.682736         0.0 |
    | 2000-01-01 01:30:00 | 13.293018 | 7.418519  7.411267  7.364994  7.269153   7.06337  6.778363   6.48586     6.319389 | 0.0   0.843893   2.131023   2.906303   3.734455   3.812825   2.835108   1.746126         0.0 |
    | 2000-01-01 01:35:00 | 11.796777 | 7.393647  7.387043  7.342706  7.252891  7.060454  6.795393  6.517229     6.355316 | 0.0   0.829073   2.040278   2.783182   3.596291    3.69349   2.841989   1.796339         0.0 |
    | 2000-01-01 01:40:00 | 10.328646 | 7.369417  7.363943  7.321585  7.236973   7.05702  6.810245  6.547285     6.391968 | 0.0   0.807673   1.962692    2.66673   3.462628   3.577087   2.834491   1.832627         0.0 |
    | 2000-01-01 01:45:00 |  8.890761 | 7.346141  7.341624  7.301541  7.221499  7.053127  6.823233  6.576065     6.429061 | 0.0   0.775864   1.891798   2.559936    3.33362   3.463375    2.81397   1.854638         0.0 |
    | 2000-01-01 01:50:00 |  7.488483 | 7.323994  7.319947  7.282332  7.206549  7.048863  6.834642  6.603642     6.466309 | 0.0   0.738238   1.822078   2.462384   3.209899   3.352051   2.781613   1.862389         0.0 |
    | 2000-01-01 01:55:00 |  6.125578 | 7.302941   7.29895  7.263741  7.192138  7.044332  6.844731  6.630108     6.503433 | 0.0   0.701761   1.751625   2.371331   3.091883   3.242919   2.738436   1.856236         0.0 |
    | 2000-01-01 02:00:00 |  4.802115 | 7.282814  7.278734  7.245677  7.178227  7.039637  6.853742  6.655563      6.54017 | 0.0   0.670893     1.6817   2.283829   2.979382   3.135863   2.685311   1.836823         0.0 |
    | 2000-01-01 02:05:00 |  3.515238 | 7.263433  7.259362  7.228168  7.164759  7.034865  6.861897  6.680101     6.576271 | 0.0   0.646049    1.61465   2.198281   2.871656   3.030719   2.623003   1.805042         0.0 |
    | 2000-01-01 02:10:00 |  2.261336 | 7.244687  7.240819  7.211288  7.151698  7.030083  6.869396  6.703808      6.61151 | 0.0   0.624845   1.552015   2.114697   2.767752   2.927175   2.552207   1.761985         0.0 |
    | 2000-01-01 02:15:00 |  1.037912 | 7.226555   7.22303  7.195084   7.13904  7.025345   6.87642  6.726748     6.645688 | 0.0   0.604394   1.493822   2.033959   2.666842   2.824757   2.473564   1.708899         0.0 |
    | 2000-01-01 02:20:00 |   42.7955 | 7.209065   7.20591  7.179548  7.126809   7.02071  6.883124  6.748965     6.678631 | 0.0   0.583007   1.438994   1.956843   2.568391   2.722909   2.387694   1.647142         0.0 |
    | 2000-01-01 02:25:00 | 41.796677 | 7.192247  7.189398  7.164632  7.115036  7.016239  6.889642  6.770477     6.710194 | 0.0    0.56061   1.386232   1.883444   2.472103   2.621114   2.295242   1.578149         0.0 |
    | 2000-01-01 02:30:00 | 40.824444 | 7.176103  7.173468  7.150278  7.103745  7.012003  6.896083  6.791283      6.74026 | 0.0   0.538123   1.334634   1.813073   2.377624   2.518838   2.196785   1.503278         0.0 |
    | 2000-01-01 02:35:00 | 39.879199 | 7.160606   7.15812  7.136454  7.092949  7.008068  6.902537  6.811357     6.768738 | 0.0    0.51658    1.28396   1.744777   2.284589   2.415751   2.093054   1.423905         0.0 |
    | 2000-01-01 02:40:00 | 38.960799 | 7.145711  7.143363  7.123154  7.082654  7.004494  6.909071  6.830663     6.795564 | 0.0   0.496497   1.234358   1.677695   2.192468   2.311586    1.98486   1.341321         0.0 |
    | 2000-01-01 02:45:00 | 38.068784 | 7.131379  7.129197  7.110395  7.072867   7.00133  6.915731  6.849155     6.820698 | 0.0    0.47773      1.186   1.611291   2.100639   2.206117   1.873109    1.25671         0.0 |
    | 2000-01-01 02:50:00 |  37.20278 | 7.117587  7.115615    7.0982  7.063603   6.99861  6.922538  6.866785     6.844121 | 0.0    0.45972   1.138822   1.545326   2.008489   2.099146   1.758792   1.171143         0.0 |
    | 2000-01-01 02:55:00 | 36.362811 | 7.104332  7.102603  7.086584  7.054887   6.99636   6.92949  6.883506     6.865833 | 0.0    0.44186   1.092484   1.479674   1.915498   1.990516    1.64295    1.08558         0.0 |
    | 2000-01-01 03:00:00 | 35.549418 | 7.091619  7.090148  7.075555  7.046744  6.994594   6.93656  6.899278      6.88585 | 0.0   0.423744   1.046474   1.414133    1.82127   1.880144   1.526632   1.000874         0.0 |
    | 2000-01-01 03:05:00 |  34.76358 | 7.079462  7.078248  7.065112  7.039201  6.993318  6.943704  6.914071     6.904206 | 0.0   0.405236   1.000267   1.348338   1.725511    1.76805   1.410855   0.917773         0.0 |
    | 2000-01-01 03:10:00 | 34.006561 | 7.067871  7.066907  7.055261  7.032276  6.992526   6.95086   6.92786     6.920944 | 0.0   0.386381   0.953421   1.281786   1.627998    1.65438    1.29656    0.83693         0.0 |
    | 2000-01-01 03:15:00 | 33.279787 | 7.056853   7.05614  7.046012  7.025984  6.992201  6.957956  6.940631     6.936122 | 0.0   0.367251   0.905603   1.213932   1.528547   1.539395   1.184589   0.758889         0.0 |
    | 2000-01-01 03:20:00 | 32.584808 | 7.046418  7.045966   7.03738  7.020329  6.992308  6.964912  6.952378     6.949804 | 0.0   0.347834   0.856551   1.144287   1.427018   1.423448   1.075658   0.684084         0.0 |
    | 2000-01-01 03:25:00 | 31.923346 | 7.036578  7.036405  7.029386  7.015312    6.9928  6.971644  6.963103      6.96206 | 0.0   0.327996   0.806009   1.072469   1.323339   1.306945   0.970337   0.612826         0.0 |
    | 2000-01-01 03:30:00 | 31.297375 | 7.027353  7.027482   7.02205  7.010926  6.993616   6.97807  6.972816     6.972966 | 0.0   0.307512   0.753696    0.99821   1.217531   1.190317   0.869039     0.5453         0.0 |
    | 2000-01-01 03:35:00 | 30.709183 | 7.018769  7.019218   7.01539  7.007157  6.994688   6.98411  6.981529     6.982598 | 0.0   0.286133   0.699302   0.921321   1.109731   1.073994   0.772005   0.481559         0.0 |
    | 2000-01-01 03:40:00 | 30.161387 |  7.01086  7.011641  7.009415  7.003987  6.995942  6.989691  6.989262     6.991028 | 0.0    0.26365   0.642513   0.841657   1.000191   0.958391   0.679302   0.421528         0.0 |
    | 2000-01-01 03:45:00 |  29.65689 | 7.003662  7.004778  7.004134  7.001384  6.997303  6.994753  6.996037     6.998328 | 0.0   0.239918   0.583049   0.759103    0.88925   0.843893   0.590831    0.36501         0.0 |
    | 2000-01-01 03:50:00 | 29.259424 | 6.997216  6.998662  6.999547  6.999309  6.998697  6.999243  7.001876     7.004562 | 0.0   0.214866   0.520692   0.673578   0.777308   0.730845   0.506348   0.311701         0.0 |
    | 2000-01-01 03:55:00 | 29.418752 | 6.991562  6.993325  6.995655  6.997715  7.000054  7.003123  7.006805     7.009787 | 0.0   0.188468   0.455317   0.585063   0.664781   0.619535   0.425503   0.261219         0.0 |
    | 2000-01-01 04:00:00 | 29.735213 |  6.98674  6.988802  6.992453  6.996546  7.001311  7.006369  7.010847     7.014049 | 0.0   0.160718   0.386879   0.493616   0.552047   0.510152   0.347857   0.213124         0.0 |
    | 2000-01-01 04:05:00 | 29.989964 | 6.982792  6.985125  6.989934  6.995745  7.002411  7.008966  7.014026     7.017389 | 0.0   0.131623   0.315432   0.399398   0.439438   0.402782   0.272939   0.166964         0.0 |
    | 2000-01-01 04:10:00 | 30.185391 | 6.979756  6.982327  6.988087  6.995255  7.003305  7.010908  7.016365     7.019835 | 0.0   0.101206   0.241131   0.302681   0.327211   0.297393   0.200274   0.122312         0.0 |
    | 2000-01-01 04:15:00 |  30.32345 |  6.97767  6.980433    6.9869  6.995021  7.003956  7.012197  7.017883     7.021411 | 0.0   0.069525   0.164234   0.203824   0.215531   0.193826   0.129402   0.078795         0.0 |
    | 2000-01-01 04:20:00 | 30.405672 | 6.976569  6.979464  6.986355  6.994997  7.004336  7.012835  7.018597     7.022133 | 0.0   0.036698    0.08511    0.10326   0.104456   0.091805   0.059893   0.036106         0.0 |
    | 2000-01-01 04:25:00 | 30.433162 | 6.976481  6.979438  6.986438  6.995147  7.004425  7.012828  7.018517     7.022013 | 0.0   0.002924   0.004258   0.001483  -0.006028  -0.009004  -0.008631  -0.005993         0.0 |
    | 2000-01-01 04:30:00 | 30.406829 | 6.977421  6.980344  6.987115  6.995436  7.004222  7.012199  7.017672     7.021067 | 0.0  -0.031322  -0.076628  -0.099177   -0.11363  -0.106869  -0.075452  -0.047273         0.0 |
    | 2000-01-01 04:35:00 | 30.329086 |  6.97936  6.982093  6.988283  6.995822  7.003772  7.011043  7.016141     7.019338 | 0.0  -0.064634  -0.152083  -0.191026  -0.210304   -0.19528  -0.137486  -0.086455         0.0 |
    | 2000-01-01 04:40:00 | 30.205261 | 6.982202  6.984518  6.989801  6.996269  7.003153  7.009506  7.014051     7.016903 | 0.0  -0.094742  -0.215986  -0.266598  -0.288952  -0.268326  -0.191436  -0.121757         0.0 |
    | 2000-01-01 04:45:00 | 30.043101 | 6.985767   6.98742   6.99154  6.996757   7.00245   7.00774  7.011547     7.013881 | 0.0   -0.11882  -0.263919  -0.321869  -0.346299  -0.322859  -0.234594  -0.151127         0.0 |
    | 2000-01-01 04:50:00 | 29.851901 | 6.989793  6.990611  6.993411  6.997283  7.001728  7.005875  7.008772     7.010429 | 0.0  -0.134195  -0.293745  -0.356102  -0.382389  -0.358331  -0.265065  -0.172585         0.0 |
    | 2000-01-01 04:55:00 | 29.641651 | 6.993973  6.993934  6.995367  6.997853   7.00103  7.003993  7.005856     7.006734 | 0.0  -0.139356  -0.305526  -0.370742  -0.399264  -0.376017   -0.28193  -0.184726         0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_channel_longitudinal_inflow:

Longitudinal inflow
___________________

So far, the considered channel has received no inflow.  We could accomplish this by
adding another |sw1d_lias| model upstream of the first channel segment.  However, then
we would have to connect the existing channel with at least one other channel, as
|sw1d_lias| only works "between" channel segments.  We demonstrate such couplings in
the |sw1d_network| documentation.  Here, we select the |sw1d_q_in| routing model, which
allows using observed or previously simulated discharge series as inflow:

>>> with model.add_routingmodel_v1("sw1d_q_in", position=0):
...     lengthdownstream(2.0)
...     timestepfactor(0.7)
...     with model.add_crosssection_v2("wq_trapeze"):
...         nmbtrapezes(1)
...         bottomlevels(5.0)
...         bottomwidths(5.0)
...         sideslopes(0.0)

No matter if adding |sw1d_lias|, |sw1d_q_in|, or another routing model to the first
position, we must now add an inlet node connectible to the inlet variable
|sw1d_inlets.LongQ| to ensure *HydPy* can build the required connections to other
models (in case of |sw1d_lias|) or provide the inflow time series (in case of
|sw1d_q_in|):

>>> from hydpy import Node
>>> inflow = Node("inflow", variable="LongQ")
>>> channel = Element("channel", inlets=inflow)
>>> channel.model = model

This extension of our project setting requires a fresh |IntegrationTest| instance:

>>> test = IntegrationTest(channel)
>>> test.plotting_options.axis1 = (factors.waterlevels,)

We reset the settings of the remaining routing models to the initial
:ref:`sw1d_channel_zero_inflow_and_outflow` example:

>>> for routingmodel in model.routingmodels.submodels[1:-1]:
...     routingmodel.parameters.control.timestepfactor(0.7)
...     routingmodel.parameters.control.diffusionfactor(0.2)

The constant inflow is 1 m³/s, and the initial water depth is 1 m:

>>> inflow.sequences.sim.series = 1.0
>>> prepare_inits(hs=1.0, qs=0.0)

The simulated water levels rise as expected:

.. integration-test::

    >>> conditions = test("sw1d_channel_longitudinal_inflow", get_conditions="2000-01-01 00:00")
    |                date | timestep |                                                                       waterlevels |                                                                            discharges | inflow |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |    300.0 |     6.03       6.0       6.0       6.0       6.0       6.0       6.0          6.0 | 1.0       0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |    1.0 |
    | 2000-01-01 00:05:00 |    300.0 | 6.054607  6.003595       6.0       6.0       6.0       6.0       6.0          6.0 | 1.0  0.179758       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |    1.0 |
    | 2000-01-01 00:10:00 |    300.0 | 6.070015  6.012899  6.000636       6.0       6.0       6.0       6.0          6.0 | 1.0  0.486393  0.021192       0.0       0.0       0.0       0.0       0.0         0.0 |    1.0 |
    | 2000-01-01 00:15:00 |    300.0 | 6.081535  6.022503  6.004598  6.000075       6.0       6.0       6.0          6.0 | 1.0  0.616016  0.135813  0.003743       0.0       0.0       0.0       0.0         0.0 |    1.0 |
    | 2000-01-01 00:20:00 |    300.0 | 6.091495  6.030868  6.010796  6.000929  6.000013       6.0       6.0          6.0 | 1.0   0.66799   0.24975  0.043139  0.000441       0.0       0.0       0.0         0.0 |    1.0 |
    | 2000-01-01 00:25:00 |    300.0 | 6.100528  6.038324   6.01716  6.003007  6.000313  6.000002       6.0          6.0 | 1.0    0.6989  0.326119  0.113982  0.010054  0.000078       0.0       0.0         0.0 |    1.0 |
    | 2000-01-01 00:30:00 |    300.0 | 6.108864  6.045292   6.02263  6.006134  6.001277  6.000059       6.0          6.0 | 1.0  0.722135  0.373698  0.191367  0.035042  0.002898  0.000009       0.0         0.0 |    1.0 |
    | 2000-01-01 00:35:00 |    300.0 | 6.116644  6.051932  6.027283  6.009726  6.003108  6.000306   6.00002          6.0 | 1.0  0.740664  0.408688  0.253589  0.074003  0.012964  0.000645  0.000002         0.0 |    1.0 |
    | 2000-01-01 00:40:00 |    300.0 | 6.123957  6.058289  6.031538    6.0133  6.005624  6.000916  6.000119     6.000004 | 1.0  0.756245  0.438392  0.296574  0.117876  0.033999  0.003495  0.000181         0.0 |    1.0 |
    | 2000-01-01 00:45:00 |    300.0 | 6.130859  6.064402   6.03568  6.016669  6.008397  6.002005   6.00041     6.000027 | 1.0  0.769932  0.464299  0.326229  0.157758  0.065325  0.010875  0.001173         0.0 |    1.0 |
    | 2000-01-01 00:50:00 |    300.0 | 6.137395   6.07031  6.039817  6.019839   6.01106  6.003545  6.001018     6.000113 | 1.0  0.782126  0.486691  0.348785  0.190274  0.101529  0.024525  0.004276         0.0 |    1.0 |
    | 2000-01-01 00:55:00 |    300.0 | 6.143606  6.076048  6.043963  6.022868  6.013473  6.005381  6.002009     6.000336 | 1.0  0.792962  0.506102  0.367899   0.21646  0.136012  0.044211  0.011164         0.0 |    1.0 |
    | 2000-01-01 01:00:00 |    300.0 |  6.14953  6.081634  6.048101  6.025806  6.015682  6.007325  6.003345     6.000796 | 1.0  0.802553  0.523225  0.385282  0.238399  0.164768  0.067539     0.023         0.0 |    1.0 |
    | 2000-01-01 01:05:00 |    300.0 | 6.155198  6.087082  6.052212  6.028689   6.01779  6.009244    6.0049     6.001585 | 1.0  0.811049  0.538667  0.401636  0.257467  0.187214  0.091286  0.039444         0.0 |    1.0 |
    | 2000-01-01 01:10:00 |    300.0 |  6.16064  6.092397  6.056283  6.031547  6.019876  6.011079  6.006539     6.002752 | 1.0  0.818617  0.552867  0.417184  0.274292  0.204741  0.112984  0.058372         0.0 |    1.0 |
    | 2000-01-01 01:15:00 |    300.0 | 6.165877  6.097583  6.060308  6.034401   6.02198  6.012831  6.008183     6.004287 | 1.0  0.825412  0.566098  0.431938  0.289225  0.219089  0.131517  0.076716         0.0 |    1.0 |
    | 2000-01-01 01:20:00 |    300.0 |  6.17093  6.102644  6.064287  6.037266  6.024112  6.014527  6.009829     6.006123 | 1.0  0.831567  0.578512  0.445861  0.302615   0.23156  0.146726  0.091839         0.0 |    1.0 |
    | 2000-01-01 01:25:00 |    300.0 | 6.175815  6.107584  6.068224  6.040148   6.02627  6.016209   6.01152     6.008172 | 1.0  0.837188  0.590188  0.458945  0.314826  0.242872  0.158799  0.102458         0.0 |    1.0 |
    | 2000-01-01 01:30:00 |    300.0 | 6.180544  6.112408  6.072123   6.04305  6.028457  6.017915  6.013298     6.010346 | 1.0  0.842353  0.601171  0.471236  0.326175  0.253303  0.167979   0.10869         0.0 |    1.0 |
    | 2000-01-01 01:35:00 |    300.0 |  6.18513   6.11712  6.075983  6.045968  6.030676  6.019682  6.015189     6.012577 | 1.0  0.847121  0.611501  0.482808  0.336868  0.262882  0.174547  0.111513         0.0 |    1.0 |
    | 2000-01-01 01:40:00 |    300.0 | 6.189584  6.121727  6.079808  6.048903  6.032939  6.021535  6.017191      6.01482 | 1.0   0.85154   0.62122  0.493739  0.346995  0.271553    0.1789  0.112164         0.0 |    1.0 |
    | 2000-01-01 01:45:00 |    300.0 | 6.193915  6.126232  6.083596  6.051854  6.035258  6.023489  6.019286     6.017054 | 1.0  0.855646  0.630372  0.504091  0.356561  0.279281  0.181562  0.111731         0.0 |    1.0 |
    | 2000-01-01 01:50:00 |    300.0 | 6.198131  6.130641  6.087349  6.054821  6.037641  6.025549   6.02145     6.019274 | 1.0  0.859472  0.639003  0.513903  0.365533  0.286104  0.183119  0.110989         0.0 |    1.0 |
    | 2000-01-01 01:55:00 |    300.0 | 6.202239  6.134959  6.091068  6.057808  6.040093  6.027709  6.023662     6.021482 | 1.0  0.863046  0.647151  0.523199  0.373883  0.292136  0.184131  0.110404         0.0 |    1.0 |
    | 2000-01-01 02:00:00 |    300.0 | 6.206248   6.13919  6.094754  6.060815  6.042615  6.029959  6.025908     6.023686 | 1.0  0.866392   0.65485  0.531988   0.38161  0.297539  0.185056  0.110198         0.0 |    1.0 |
    | 2000-01-01 02:05:00 |    300.0 | 6.210162  6.143338  6.098409  6.063846  6.045203  6.032285  6.028181     6.025895 | 1.0  0.869529  0.662123  0.540283  0.388754  0.302491  0.186205  0.110444         0.0 |    1.0 |
    | 2000-01-01 02:10:00 |    300.0 | 6.213987  6.147408  6.102036    6.0669   6.04785  6.034673  6.030479     6.028118 | 1.0  0.872476  0.668993    0.5481  0.395382  0.307158  0.187744  0.111131         0.0 |    1.0 |
    | 2000-01-01 02:15:00 |    300.0 |  6.21773  6.151403  6.105636  6.069978  6.050547  6.037112  6.032804     6.030362 | 1.0  0.875245  0.675476  0.555462  0.401581  0.311669  0.189719  0.112212         0.0 |    1.0 |
    | 2000-01-01 02:20:00 |    300.0 | 6.221395  6.155329  6.109212  6.073077  6.053287  6.039592  6.035158     6.032634 | 1.0  0.877849  0.681591  0.562401  0.407436  0.316113  0.192093  0.113622         0.0 |    1.0 |
    | 2000-01-01 02:25:00 |    300.0 | 6.224986  6.159187  6.112764  6.076196  6.056061  6.042107  6.037543      6.03494 | 1.0  0.880297  0.687357  0.568951  0.413026   0.32054  0.194784  0.115289         0.0 |    1.0 |
    | 2000-01-01 02:30:00 |    300.0 | 6.228508  6.162983  6.116293   6.07933  6.058865  6.044653   6.03996     6.037283 | 1.0    0.8826  0.692794  0.575149  0.418414  0.324968  0.197698  0.117141         0.0 |    1.0 |
    | 2000-01-01 02:35:00 |    300.0 | 6.231965   6.16672    6.1198  6.082478  6.061692  6.047226  6.042409     6.039665 | 1.0  0.884765  0.697924  0.581029  0.423642  0.329393  0.200746  0.119104         0.0 |    1.0 |
    | 2000-01-01 02:40:00 |    300.0 | 6.235361  6.170401  6.123284  6.085636  6.064541  6.049825  6.044891     6.042087 | 1.0  0.886801  0.702768  0.586622  0.428738    0.3338  0.203852  0.121115         0.0 |    1.0 |
    | 2000-01-01 02:45:00 |    300.0 | 6.238699  6.174028  6.126746  6.088801  6.067407  6.052449  6.047406      6.04455 | 1.0  0.888718  0.707349  0.591956  0.433716   0.33817  0.206955  0.123121         0.0 |    1.0 |
    | 2000-01-01 02:50:00 |    300.0 | 6.241984  6.177605  6.130185   6.09197   6.07029  6.055098  6.049954     6.047051 | 1.0  0.890523  0.711688  0.597053  0.438578   0.34248  0.210013  0.125082         0.0 |    1.0 |
    | 2000-01-01 02:55:00 |    300.0 | 6.245217  6.181133  6.133601  6.095142  6.073188  6.057773  6.052535     6.049591 | 1.0  0.892225  0.715803  0.601932  0.443323  0.346712  0.212994  0.126975         0.0 |    1.0 |
    | 2000-01-01 03:00:00 |    300.0 | 6.248402  6.184616  6.136995  6.098315  6.076101  6.060472  6.055147     6.052167 | 1.0  0.893831  0.719713  0.606606  0.447947   0.35085   0.21588  0.128789         0.0 |    1.0 |
    | 2000-01-01 03:05:00 |    300.0 | 6.251541  6.188054  6.140365  6.101488  6.079028  6.063196  6.057792     6.054777 | 1.0   0.89535  0.723432  0.611088  0.452444  0.354881   0.21866  0.130519         0.0 |    1.0 |
    | 2000-01-01 03:10:00 |    300.0 | 6.254638   6.19145  6.143713   6.10466  6.081968  6.065946  6.060467     6.057421 | 1.0  0.896788  0.726975  0.615388  0.456809  0.358796  0.221333   0.13217         0.0 |    1.0 |
    | 2000-01-01 03:15:00 |    300.0 | 6.257693  6.194806  6.147038  6.107829  6.084922   6.06872  6.063171     6.060095 | 1.0  0.898151  0.730353  0.619514  0.461037  0.362591  0.223901  0.133747         0.0 |    1.0 |
    | 2000-01-01 03:20:00 |    300.0 |  6.26071  6.198124  6.150341  6.110996  6.087888  6.071517  6.065904     6.062801 | 1.0  0.899445  0.733577  0.623473  0.465127  0.366261  0.226368  0.135259         0.0 |    1.0 |
    | 2000-01-01 03:25:00 |    300.0 |  6.26369  6.201404  6.153622   6.11416  6.090866  6.074339  6.068665     6.065535 | 1.0  0.900675  0.736656  0.627272  0.469077  0.369808  0.228742  0.136712         0.0 |    1.0 |
    | 2000-01-01 03:30:00 |    300.0 | 6.266634  6.204649  6.156883  6.117321  6.093855  6.077183  6.071453     6.068297 | 1.0  0.901844  0.739598  0.630916  0.472887  0.373231  0.231029  0.138112         0.0 |    1.0 |
    | 2000-01-01 03:35:00 |    300.0 | 6.269545   6.20786  6.160123  6.120478  6.096856  6.080049  6.074266     6.071086 | 1.0  0.902958   0.74241  0.634411   0.47656  0.376534  0.233236  0.139464         0.0 |    1.0 |
    | 2000-01-01 03:40:00 |    300.0 | 6.272425  6.211038  6.163343  6.123631  6.099868  6.082936  6.077104     6.073902 | 1.0  0.904019  0.745099  0.637763  0.480097  0.379721  0.235368  0.140772         0.0 |    1.0 |
    | 2000-01-01 03:45:00 |    300.0 | 6.275274  6.214186  6.166544  6.126781  6.102889  6.085843  6.079966     6.076743 | 1.0  0.905031   0.74767  0.640977  0.483503  0.382793  0.237428  0.142037         0.0 |    1.0 |
    | 2000-01-01 03:50:00 |    300.0 | 6.278094  6.217303  6.169726  6.129926   6.10592   6.08877   6.08285     6.079608 | 1.0  0.905996  0.750128  0.644057   0.48678  0.385756   0.23942  0.143263         0.0 |    1.0 |
    | 2000-01-01 03:55:00 |    300.0 | 6.280887  6.220392   6.17289  6.133068  6.108959  6.091715  6.085757     6.082497 | 1.0  0.906916  0.752479   0.64701  0.489933  0.388612  0.241345   0.14445         0.0 |    1.0 |
    | 2000-01-01 04:00:00 |    300.0 | 6.283653  6.223453  6.176036  6.136205  6.112007  6.094678  6.088686     6.085409 | 1.0  0.907794  0.754726  0.649838  0.492965  0.391364  0.243206  0.145598         0.0 |    1.0 |
    | 2000-01-01 04:05:00 |    300.0 | 6.286394  6.226488  6.179166  6.139339  6.115063  6.097659  6.091634     6.088343 | 1.0  0.908631  0.756875  0.652548   0.49588  0.394015  0.245004  0.146708         0.0 |    1.0 |
    | 2000-01-01 04:10:00 |    300.0 | 6.289111  6.229498   6.18228  6.142468  6.118126  6.100655  6.094603     6.091299 | 1.0   0.90943  0.758929  0.655144  0.498682  0.396569  0.246739  0.147781         0.0 |    1.0 |
    | 2000-01-01 04:15:00 |    300.0 | 6.291805  6.232484  6.185378  6.145593  6.121197  6.103667  6.097591     6.094275 | 1.0  0.910192  0.760892  0.657629  0.501374  0.399027  0.248412  0.148817         0.0 |    1.0 |
    | 2000-01-01 04:20:00 |    300.0 | 6.294478  6.235447  6.188461  6.148714  6.124274  6.106695  6.100597     6.097271 | 1.0  0.910919  0.762768  0.660008   0.50396  0.401391  0.250025  0.149815         0.0 |    1.0 |
    | 2000-01-01 04:25:00 |    300.0 | 6.297129  6.238388  6.191529  6.151831  6.127357  6.109736  6.103621     6.100287 | 1.0  0.911612   0.76456  0.662285  0.506442  0.403665  0.251578  0.150777         0.0 |    1.0 |
    | 2000-01-01 04:30:00 |    300.0 | 6.299761  6.241308  6.194583  6.154944  6.130446  6.112792  6.106662     6.103321 | 1.0  0.912273  0.766272  0.664463  0.508825   0.40585  0.253072  0.151703         0.0 |    1.0 |
    | 2000-01-01 04:35:00 |    300.0 | 6.302374  6.244208  6.197624  6.158052  6.133541  6.115861   6.10972     6.106373 | 1.0  0.912902  0.767906  0.666546   0.51111  0.407949  0.254509  0.152593         0.0 |    1.0 |
    | 2000-01-01 04:40:00 |    300.0 | 6.304969  6.247089  6.200652  6.161157  6.136641  6.118942  6.112793     6.109442 | 1.0  0.913502  0.769466  0.668538    0.5133  0.409963  0.255889  0.153448         0.0 |    1.0 |
    | 2000-01-01 04:45:00 |    300.0 | 6.307547  6.249951  6.203667  6.164258  6.139747  6.122036  6.115881     6.112527 | 1.0  0.914072  0.770954   0.67044  0.515399  0.411895  0.257213  0.154269         0.0 |    1.0 |
    | 2000-01-01 04:50:00 |    300.0 | 6.310108  6.252796  6.206671  6.167355  6.142856  6.125141  6.118984     6.115628 | 1.0  0.914616  0.772373  0.672257  0.517408  0.413747  0.258484  0.155057         0.0 |    1.0 |
    | 2000-01-01 04:55:00 |    300.0 | 6.312654  6.255624  6.209663  6.170448  6.145971  6.128258  6.122101     6.118744 | 1.0  0.915132  0.773725  0.673992   0.51933   0.41552  0.259701  0.155812         0.0 |    1.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0


.. _sw1d_channel_lateral_flow:

Lateral flow
____________

In the :ref:`sw1d_channel_longitudinal_inflow` example, the |sw1d_q_in| adds
"longitudinal" inflow to the channel's first segment.  With "longitudinal", we mean
that the entering water is already moving in the same direction as the water already in
the channel.  This configuration makes sense when simulating the inflow from an
upstream channel.

The |sw1d_storage| models offers the alternative of adding (or subtracting) lateral
flow via an additional node connected to its inlet sequence |sw1d_inlets.LatQ|.  One
can use it for considering inflow from adjacent areas or other sources that let their
water flow vertically to the channel's direction.  At the moment, all lateral inflow
affects only the first channel segments' water amount.  More flexible approaches (e.g.
distributing it evenly) are under discussion (see the `LIAS issue on GitHub`).

Additionally, one can query the water level (also, currently only of the first channel
segment) via a node connectible to the sender |sw1d_senders.WaterLevel| sequence.  This
information helps if feedback to the above model is required, such as letting a pump
stop working to prevent exceeding a dangerous water level.

We add the two mentioned nodes to our |Element| instance:

>>> latflow = Node("latflow", variable="LatQ")
>>> waterlevel = Node("waterlevel", variable="WaterLevel")
>>> channel = Element("channel", inlets=(inflow, latflow), senders=waterlevel)
>>> channel.model = model
>>> test = IntegrationTest(channel)
>>> test.plotting_options.axis1 = (factors.waterlevels,)

We set the same initial and inflow values as in the
:ref:`sw1d_channel_longitudinal_inflow` example, except letting the constant inflow
enter "laterally" instead of "longitudinally":

>>> inflow.sequences.sim.series = 0.0
>>> latflow.sequences.sim.series = 1.0
>>> prepare_inits(hs=1.0, qs=0.0)

The results are similar.  However, the first segment's water level is now about 1 cm
higher than in the :ref:`sw1d_channel_longitudinal_inflow` example due to assuming that
the lateral's flow velocity component into the channel direction is zero (note that the
extent of this difference vastly depends on the value of |DiffusionFactor|):

.. integration-test::

    >>> test("sw1d_channel_lateral_flow")
    |                date | timestep |                                                                       waterlevels |                                                                            discharges | inflow | latflow | waterlevel |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |    300.0 |     6.03       6.0       6.0       6.0       6.0       6.0       6.0          6.0 | 0.0       0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |    0.0 |     1.0 |       6.03 |
    | 2000-01-01 00:05:00 |    300.0 | 6.054607  6.003595       6.0       6.0       6.0       6.0       6.0          6.0 | 0.0  0.179758       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |    0.0 |     1.0 |   6.054607 |
    | 2000-01-01 00:10:00 |    300.0 | 6.072648  6.011144  6.000636       6.0       6.0       6.0       6.0          6.0 | 0.0  0.398649  0.021192       0.0       0.0       0.0       0.0       0.0         0.0 |    0.0 |     1.0 |   6.072648 |
    | 2000-01-01 00:15:00 |    300.0 | 6.086507  6.019566  6.004031  6.000075       6.0       6.0       6.0          6.0 | 0.0  0.538024  0.116934  0.003743       0.0       0.0       0.0       0.0         0.0 |    0.0 |     1.0 |   6.086507 |
    | 2000-01-01 00:20:00 |    300.0 | 6.098069  6.027473  6.009471  6.000824  6.000013       6.0       6.0          6.0 | 0.0  0.614589  0.219246  0.037917  0.000441       0.0       0.0       0.0         0.0 |    0.0 |     1.0 |   6.098069 |
    | 2000-01-01 00:25:00 |    300.0 | 6.108208   6.03479  6.015346  6.002654  6.000278  6.000002       6.0          6.0 | 0.0  0.662057  0.296202  0.100383  0.008917  0.000078       0.0       0.0         0.0 |    0.0 |     1.0 |   6.108208 |
    | 2000-01-01 00:30:00 |    300.0 | 6.117337  6.041736  6.020631  6.005479  6.001129  6.000053       6.0          6.0 | 0.0  0.695699  0.348411  0.172231   0.03095  0.002584  0.000009       0.0         0.0 |    0.0 |     1.0 |   6.117337 |
    | 2000-01-01 00:35:00 |    300.0 | 6.125702   6.04841  6.025243  6.008829  6.002772  6.000271  6.000018          6.0 | 0.0  0.721172  0.387442  0.233735  0.066227  0.011478  0.000576  0.000002         0.0 |    0.0 |     1.0 |   6.125702 |
    | 2000-01-01 00:40:00 |    300.0 | 6.133455  6.054839  6.029486  6.012252  6.005087  6.000815  6.000106     6.000003 | 0.0  0.741537   0.42011  0.278646  0.107496  0.030321    0.0031  0.000162         0.0 |    0.0 |     1.0 |   6.133455 |
    | 2000-01-01 00:45:00 |    300.0 | 6.140698  6.061044  6.033618  6.015535  6.007711  6.001803  6.000365     6.000024 | 0.0  0.758594  0.448359  0.310652  0.146536  0.059068  0.009683  0.001042         0.0 |    0.0 |     1.0 |   6.140698 |
    | 2000-01-01 00:50:00 |    300.0 | 6.147501  6.067054  6.037744   6.01865  6.010294  6.003228  6.000912       6.0001 | 0.0  0.773234  0.472711   0.33515  0.179394  0.093311  0.022054  0.003805         0.0 |    0.0 |     1.0 |   6.147501 |
    | 2000-01-01 00:55:00 |    300.0 | 6.153924  6.072897  6.041885  6.021639  6.012673  6.004963  6.001821       6.0003 | 0.0  0.785892  0.493771  0.355742  0.206311  0.127001  0.040277  0.010001         0.0 |    0.0 |     1.0 |   6.153924 |
    | 2000-01-01 01:00:00 |    300.0 | 6.160018  6.078588  6.046026  6.024544  6.014864  6.006834  6.003068     6.000717 | 0.0  0.796867  0.512272  0.374262  0.228973  0.155948  0.062401  0.020816         0.0 |    0.0 |     1.0 |   6.160018 |
    | 2000-01-01 01:05:00 |    300.0 | 6.165825   6.08414  6.050146  6.027401  6.016952  6.008705  6.004548     6.001439 | 0.0  0.806426  0.528867  0.391529  0.248684  0.179061  0.085476  0.036143         0.0 |    0.0 |     1.0 |   6.165825 |
    | 2000-01-01 01:10:00 |    300.0 | 6.171381  6.089555  6.054232  6.030236  6.019016  6.010511  6.006133     6.002524 | 0.0  0.814817  0.544038  0.407845  0.266098  0.197321  0.107024  0.054206         0.0 |    0.0 |     1.0 |   6.171381 |
    | 2000-01-01 01:15:00 |    300.0 | 6.176713  6.094838  6.058276   6.03307  6.021094  6.012242   6.00774     6.003967 | 0.0  0.822258  0.558096  0.423276  0.281579  0.212288  0.125751  0.072193         0.0 |    0.0 |     1.0 |   6.176713 |
    | 2000-01-01 01:20:00 |    300.0 | 6.181845  6.099992  6.062279  6.035917  6.023201   6.01392  6.009355     6.005717 | 0.0  0.828923  0.571225   0.43781  0.295462  0.225237   0.14133  0.087493         0.0 |    0.0 |     1.0 |   6.181845 |
    | 2000-01-01 01:25:00 |    300.0 | 6.186797  6.105021  6.066241  6.038784  6.025337  6.015582  6.011011      6.00769 | 0.0  0.834952  0.583526  0.451453  0.308106  0.236923  0.153857  0.098654         0.0 |    0.0 |     1.0 |   6.186797 |
    | 2000-01-01 01:30:00 |    300.0 | 6.191583  6.109929  6.070165  6.041672  6.027501  6.017264   6.01275     6.009802 | 0.0  0.840448  0.595062  0.464252  0.319828  0.247672  0.163531  0.105562         0.0 |    0.0 |     1.0 |   6.191583 |
    | 2000-01-01 01:35:00 |    300.0 | 6.196219  6.114721  6.074053  6.044581    6.0297  6.019003  6.014597     6.011982 | 0.0  0.845487  0.605883  0.476284  0.330849   0.25755  0.170604   0.10902         0.0 |    0.0 |     1.0 |   6.196219 |
    | 2000-01-01 01:40:00 |    300.0 | 6.200715  6.119403  6.077905  6.047508  6.031943  6.020825  6.016555     6.014185 | 0.0   0.85013   0.61604  0.487633  0.341275  0.266519   0.17544  0.110172         0.0 |    0.0 |     1.0 |   6.200715 |
    | 2000-01-01 01:45:00 |    300.0 | 6.205082  6.123979  6.081722  6.050453   6.03424  6.022745  6.018609     6.016387 | 0.0  0.854425  0.625585  0.498365  0.351125  0.274545  0.178526  0.110085         0.0 |    0.0 |     1.0 |   6.205082 |
    | 2000-01-01 01:50:00 |    300.0 |  6.20933  6.128456  6.085503  6.053416  6.036602   6.02477  6.020735     6.018578 | 0.0  0.858411  0.634569  0.508528  0.360373  0.281661  0.180427  0.109558         0.0 |    0.0 |     1.0 |    6.20933 |
    | 2000-01-01 01:55:00 |    300.0 | 6.213466  6.132838   6.08925  6.056399  6.039033  6.026895  6.022913      6.02076 | 0.0   0.86212  0.643036  0.518147  0.368994  0.287967  0.181695  0.109091         0.0 |    0.0 |     1.0 |   6.213466 |
    | 2000-01-01 02:00:00 |    300.0 | 6.217498  6.137129  6.092963  6.059404  6.041534  6.029112  6.025128     6.022939 | 0.0  0.865583  0.651024  0.527239  0.376985  0.293619  0.182793  0.108942         0.0 |    0.0 |     1.0 |   6.217498 |
    | 2000-01-01 02:05:00 |    300.0 | 6.221434  6.141334  6.096646  6.062433  6.044101  6.031407  6.027373     6.025123 | 0.0  0.868823  0.658561  0.535818  0.384381   0.29879   0.18405  0.109211         0.0 |    0.0 |     1.0 |   6.221434 |
    | 2000-01-01 02:10:00 |    300.0 | 6.225278  6.145458  6.100299  6.065486  6.046729  6.033767  6.029646     6.027321 | 0.0  0.871858  0.665673  0.543901  0.391246  0.303646  0.185654  0.109905         0.0 |    0.0 |     1.0 |   6.225278 |
    | 2000-01-01 02:15:00 |    300.0 | 6.229037  6.149505  6.103925  6.068563   6.04941   6.03618  6.031946     6.029541 | 0.0  0.874707   0.67238  0.551513  0.397663   0.30832  0.187669  0.110988         0.0 |    0.0 |     1.0 |   6.229037 |
    | 2000-01-01 02:20:00 |    300.0 | 6.232715  6.153478  6.107525  6.071663  6.052134  6.038636  6.034277     6.031789 | 0.0  0.877381  0.678702  0.558685  0.403718  0.312906  0.190073  0.112401         0.0 |    0.0 |     1.0 |   6.232715 |
    | 2000-01-01 02:25:00 |    300.0 | 6.236319  6.157383  6.111102  6.074782  6.054895  6.041129  6.036638     6.034071 | 0.0  0.879893   0.68466  0.565453  0.409491   0.31746  0.192796  0.114076         0.0 |    0.0 |     1.0 |   6.236319 |
    | 2000-01-01 02:30:00 |    300.0 | 6.239851  6.161222  6.114654  6.077918  6.057686  6.043655  6.039032     6.036389 | 0.0  0.882253  0.690276  0.571853  0.415046  0.322004  0.195749  0.115941         0.0 |    0.0 |     1.0 |   6.239851 |
    | 2000-01-01 02:35:00 |    300.0 | 6.243317     6.165  6.118184  6.081068  6.060503  6.046209   6.04146     6.038748 | 0.0   0.88447  0.695571  0.577922  0.420428   0.32654  0.198842  0.117925         0.0 |    0.0 |     1.0 |   6.243317 |
    | 2000-01-01 02:40:00 |    300.0 |  6.24672   6.16872   6.12169  6.084228  6.063341   6.04879  6.043921     6.041147 | 0.0  0.886555  0.700569  0.583691  0.425668  0.331054  0.202001  0.119965         0.0 |    0.0 |     1.0 |    6.24672 |
    | 2000-01-01 02:45:00 |    300.0 | 6.250065  6.172384  6.125173  6.087396  6.066199  6.051397  6.046416     6.043587 | 0.0  0.888515  0.705294  0.589189   0.43078  0.335528  0.205164  0.122006         0.0 |    0.0 |     1.0 |   6.250065 |
    | 2000-01-01 02:50:00 |    300.0 | 6.253354  6.175996  6.128633   6.09057  6.069074   6.05403  6.048944     6.046067 | 0.0  0.890361  0.709765   0.59444  0.435772  0.339941  0.208285  0.124007         0.0 |    0.0 |     1.0 |   6.253354 |
    | 2000-01-01 02:55:00 |    300.0 | 6.256591  6.179558  6.132069  6.093746  6.071965  6.056689  6.051506     6.048586 | 0.0    0.8921  0.714005  0.599462  0.440641  0.344274  0.211333  0.125943         0.0 |    0.0 |     1.0 |   6.256591 |
    | 2000-01-01 03:00:00 |    300.0 | 6.259779  6.183072  6.135482  6.096924  6.074871  6.059373    6.0541     6.051142 | 0.0  0.893741   0.71803  0.604272  0.445384  0.348511  0.214285  0.127799         0.0 |    0.0 |     1.0 |   6.259779 |
    | 2000-01-01 03:05:00 |    300.0 |  6.26292  6.186541  6.138871  6.100102  6.077791  6.062083  6.056727     6.053734 | 0.0  0.895292  0.721858  0.608882  0.449995   0.35264  0.217133  0.129573         0.0 |    0.0 |     1.0 |    6.26292 |
    | 2000-01-01 03:10:00 |    300.0 | 6.266017  6.189966  6.142237  6.103278  6.080726  6.064819  6.059385     6.056359 | 0.0  0.896759  0.725502  0.613304  0.454471  0.356651  0.219872  0.131266         0.0 |    0.0 |     1.0 |   6.266017 |
    | 2000-01-01 03:15:00 |    300.0 | 6.269073   6.19335   6.14558  6.106453  6.083674   6.06758  6.062074     6.059017 | 0.0   0.89815  0.728976  0.617545  0.458808  0.360539  0.222503  0.132883         0.0 |    0.0 |     1.0 |   6.269073 |
    | 2000-01-01 03:20:00 |    300.0 | 6.272089  6.196693    6.1489  6.109625  6.086635  6.070365  6.064792     6.061705 | 0.0  0.899469   0.73229  0.621613  0.463001    0.3643  0.225032  0.134433         0.0 |    0.0 |     1.0 |   6.272089 |
    | 2000-01-01 03:25:00 |    300.0 | 6.275067  6.199999  6.152199  6.112795  6.089609  6.073175  6.067538     6.064424 | 0.0  0.900723  0.735453  0.625515  0.467052  0.367934  0.227465  0.135923         0.0 |    0.0 |     1.0 |   6.275067 |
    | 2000-01-01 03:30:00 |    300.0 |  6.27801  6.203268  6.155475  6.115961  6.092594  6.076007  6.070312     6.067171 | 0.0  0.901916  0.738475  0.629258  0.470959  0.371443  0.229808  0.137357         0.0 |    0.0 |     1.0 |    6.27801 |
    | 2000-01-01 03:35:00 |    300.0 | 6.280918  6.206501  6.158731  6.119123  6.095591  6.078862  6.073111     6.069946 | 0.0  0.903051  0.741363  0.632848  0.474726  0.374828  0.232069  0.138742         0.0 |    0.0 |     1.0 |   6.280918 |
    | 2000-01-01 03:40:00 |    300.0 | 6.283794  6.209701  6.161966  6.122282  6.098599  6.081739  6.075937     6.072747 | 0.0  0.904132  0.744123   0.63629  0.478353  0.378094  0.234253  0.140081         0.0 |    0.0 |     1.0 |   6.283794 |
    | 2000-01-01 03:45:00 |    300.0 | 6.286639  6.212869  6.165181  6.125437  6.101617  6.084637  6.078786     6.075575 | 0.0  0.905163  0.746762  0.639589  0.481846  0.381243  0.236363  0.141377         0.0 |    0.0 |     1.0 |   6.286639 |
    | 2000-01-01 03:50:00 |    300.0 | 6.289455  6.216007  6.168377  6.128588  6.104645  6.087554  6.081659     6.078428 | 0.0  0.906145  0.749285  0.642752  0.485207  0.384279  0.238402  0.142631         0.0 |    0.0 |     1.0 |   6.289455 |
    | 2000-01-01 03:55:00 |    300.0 | 6.292242  6.219114  6.171554  6.131734  6.107682  6.090491  6.084555     6.081305 | 0.0  0.907082  0.751697  0.645782   0.48844  0.387206  0.240374  0.143846         0.0 |    0.0 |     1.0 |   6.292242 |
    | 2000-01-01 04:00:00 |    300.0 | 6.295003  6.222194  6.174714  6.134877  6.110728  6.093446  6.087473     6.084205 | 0.0  0.907976  0.754003  0.648686  0.491549  0.390026   0.24228  0.145022         0.0 |    0.0 |     1.0 |   6.295003 |
    | 2000-01-01 04:05:00 |    300.0 | 6.297738  6.225246  6.177856  6.138016  6.113781  6.096418  6.090412     6.087128 | 0.0  0.908829  0.756208  0.651467  0.494539  0.392744  0.244121  0.146159         0.0 |    0.0 |     1.0 |   6.297738 |
    | 2000-01-01 04:10:00 |    300.0 | 6.300449  6.228273  6.180981   6.14115  6.116843  6.099408  6.093371     6.090073 | 0.0  0.909643  0.758315  0.654131  0.497412  0.395361  0.245898  0.147258         0.0 |    0.0 |     1.0 |   6.300449 |
    | 2000-01-01 04:15:00 |    300.0 | 6.303136  6.231275  6.184091   6.14428  6.119912  6.102413   6.09635      6.09304 | 0.0  0.910419  0.760329  0.656682  0.500174  0.397881  0.247613  0.148318         0.0 |    0.0 |     1.0 |   6.303136 |
    | 2000-01-01 04:20:00 |    300.0 | 6.305802  6.234253  6.187185  6.147406  6.122987  6.105434  6.099348     6.096026 | 0.0  0.911159  0.762254  0.659124  0.502826  0.400306  0.249265  0.149341         0.0 |    0.0 |     1.0 |   6.305802 |
    | 2000-01-01 04:25:00 |    300.0 | 6.308446  6.237208  6.190264  6.150528  6.126069  6.108469  6.102363     6.099033 | 0.0  0.911864  0.764092  0.661461  0.505373  0.402637  0.250857  0.150327         0.0 |    0.0 |     1.0 |   6.308446 |
    | 2000-01-01 04:30:00 |    300.0 |  6.31107  6.240142  6.193328  6.153646  6.129157  6.111519  6.105397     6.102059 | 0.0  0.912537  0.765848  0.663697  0.507817  0.404879  0.252389  0.151276         0.0 |    0.0 |     1.0 |    6.31107 |
    | 2000-01-01 04:35:00 |    300.0 | 6.313674  6.243055  6.196379  6.156759  6.132251  6.114583  6.108447     6.105102 | 0.0  0.913178  0.767524  0.665835  0.510162  0.407032  0.253862  0.152188         0.0 |    0.0 |     1.0 |   6.313674 |
    | 2000-01-01 04:40:00 |    300.0 | 6.316261  6.245948  6.199416  6.159868  6.135351  6.117659  6.111513     6.108164 | 0.0  0.913788  0.769124  0.667879   0.51241  0.409098  0.255278  0.153066         0.0 |    0.0 |     1.0 |   6.316261 |
    | 2000-01-01 04:45:00 |    300.0 | 6.318829  6.248823  6.202441  6.162974  6.138455  6.120748  6.114595     6.111242 | 0.0   0.91437  0.770651  0.669833  0.514565  0.411081  0.256636  0.153908         0.0 |    0.0 |     1.0 |   6.318829 |
    | 2000-01-01 04:50:00 |    300.0 | 6.321382  6.251679  6.205453  6.166075  6.141565  6.123849  6.117692     6.114336 | 0.0  0.914923  0.772107  0.671699  0.516627  0.412981   0.25794  0.154716         0.0 |    0.0 |     1.0 |   6.321382 |
    | 2000-01-01 04:55:00 |    300.0 | 6.323918  6.254518  6.208453  6.169173  6.144679  6.126961  6.120803     6.117446 | 0.0  0.915449  0.773495   0.67348  0.518601  0.414802   0.25919  0.155491         0.0 |    0.0 |     1.0 |   6.323918 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_channel_weir_outflow:

Weir outflow
____________

Adding an outflow routing model works similarly to adding an inflow routing model.
Again, we could select a model like |sw1d_lias| if we were interested in coupling this
channel to another one.  But here, we choose the outflow routing model |sw1d_weir_out|,
which simulates free weir flow (that does not consider the downstream water level):

>>> with model.add_routingmodel_v3("sw1d_weir_out", position=8):
...     lengthupstream(lengths[-1])
...     crestheight(7.0)
...     crestwidth(5.0)
...     flowcoefficient(0.58)
...     timestepfactor(0.7)

This time, we must add an outlet node responsible for the outlet variable
|sw1d_outlets.LongQ|, which necessitates another refreshing of the |IntegrationTest|
instance:

>>> outflow = Node("outflow", variable="LongQ")
>>> channel = Element("channel", inlets=inflow, outlets=outflow)
>>> channel.model = model
>>> test = IntegrationTest(channel)
>>> test.plotting_options.axis1 = (factors.waterlevels,)

The constant (longitudinal) inflow is again 1 m³/s, but the initial water depth is 2 m,
corresponding to the weir's crest level:

>>> inflow.sequences.sim.series = 1.0
>>> latflow.sequences.sim.series = 0.0
>>> prepare_inits(hs=2.0, qs=0.0)

Sticking to the time step factor of 0.7 would result in numerical inaccuracies
at the beginning of the simulation period, even more notable than in the
:ref:`sw1d_channel_zero_inflow_and_outflow` example.  We skip showing them and directly
reduce the time step factor as in the :ref:`sw1d_channel_higher_precision` example:

>>> for routingmodel in model.routingmodels.submodels:
...     routingmodel.parameters.control.timestepfactor(0.1)

After a certain settling process, the water level profiles take a regular shape.
However, even after five hours, their weir flow is still below 0.4 n³/s, showing that
conditions are still not stationary:

.. integration-test::

    >>> conditions = test("sw1d_channel_weir_outflow", get_conditions="2000-01-01 00:00")
    |                date |  timestep |                                                                       waterlevels |                                                                            discharges | inflow | latflow |  outflow | waterlevel |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 29.522473 | 7.024137  7.003412  7.000685  7.000038  7.000002       7.0       7.0          7.0 | 1.0  0.195422    0.0248  0.001968  0.000084  0.000002       0.0       0.0         0.0 |    1.0 |     0.0 |      0.0 |   7.024137 |
    | 2000-01-01 00:05:00 | 30.437257 | 7.036519  7.011036  7.005204  7.000941  7.000266  7.000027  7.000005          7.0 | 1.0  0.587274  0.206115  0.055485  0.010297  0.001527  0.000166  0.000014         0.0 |    1.0 |     0.0 |      0.0 |   7.036519 |
    | 2000-01-01 00:10:00 | 31.047327 |  7.04402  7.018286  7.011516  7.003326   7.00151  7.000285   7.00009     7.000013 | 1.0  0.749984  0.387467  0.177078  0.057861  0.016386  0.003503  0.000643         0.0 |    1.0 |     0.0 |      0.0 |    7.04402 |
    | 2000-01-01 00:15:00 | 31.487924 | 7.049239   7.02458   7.01767  7.006723  7.003915  7.001055  7.000472      7.00011 | 1.0  0.826011  0.511293  0.306155   0.13627  0.056095  0.017583  0.004856    0.000003 |    1.0 |     0.0 | 0.000003 |   7.049239 |
    | 2000-01-01 00:20:00 | 31.831217 | 7.053307  7.029995  7.023053  7.010545  7.007106  7.002453  7.001376     7.000444 | 1.0  0.864394   0.59365  0.414189  0.223131  0.116769  0.046862   0.01673    0.000031 |    1.0 |     0.0 | 0.000031 |   7.053307 |
    | 2000-01-01 00:25:00 | 32.115718 | 7.056761  7.034699  7.027639  7.014403  7.010619  7.004414   7.00291     7.001193 | 1.0  0.884894  0.649681  0.496818  0.303889  0.186801  0.088777  0.037634    0.000173 |    1.0 |     0.0 | 0.000173 |   7.056761 |
    | 2000-01-01 00:30:00 | 32.362765 | 7.059865  7.038851  7.031572  7.018106  7.014134  7.006792  7.005052     7.002486 | 1.0  0.896522  0.688955  0.557864  0.372712  0.255527   0.13661  0.065221    0.000611 |    1.0 |     0.0 | 0.000611 |   7.059865 |
    | 2000-01-01 00:35:00 | 32.584688 | 7.062756  7.042577  7.035017  7.021586  7.017504  7.009448  7.007698     7.004358 | 1.0  0.903618  0.717313  0.602498  0.428534  0.316215  0.183427  0.095237    0.001597 |    1.0 |     0.0 | 0.001597 |   7.062756 |
    | 2000-01-01 00:40:00 | 32.788919 | 7.065507  7.045977  7.038114  7.024842  7.020696  7.012278  7.010717     7.006765 | 1.0  0.908324  0.738324  0.635081  0.472255  0.365839  0.224336  0.123708    0.003371 |    1.0 |     0.0 | 0.003371 |   7.065507 |
    | 2000-01-01 00:45:00 | 32.980171 | 7.068156  7.049127  7.040975   7.02791   7.02374  7.015217  7.013983     7.009606 | 1.0  0.911696  0.754166  0.658805  0.505411  0.403938  0.256991  0.148114    0.006095 |    1.0 |     0.0 | 0.006095 |   7.068156 |
    | 2000-01-01 00:50:00 | 33.161638 | 7.070729  7.052089  7.043683  7.030835  7.026684  7.018225  7.017391     7.012759 | 1.0  0.914231  0.766166  0.675898   0.52966  0.431527    0.2811  0.167494    0.009817 |    1.0 |     0.0 | 0.009817 |   7.070729 |
    | 2000-01-01 00:55:00 | 33.335665 | 7.073244  7.054909    7.0463  7.033661  7.029573  7.021279  7.020859      7.01611 | 1.0  0.916149  0.775157  0.687924  0.546602  0.450302  0.297624  0.182022    0.014488 |    1.0 |     0.0 | 0.014488 |   7.073244 |
    | 2000-01-01 01:00:00 |   33.5041 | 7.075718  7.057626   7.04887  7.036427  7.032441   7.02436  7.024327     7.019559 | 1.0   0.91754  0.781696  0.696023  0.557716  0.462138  0.308081  0.192474    0.019988 |    1.0 |     0.0 | 0.019988 |   7.075718 |
    | 2000-01-01 01:05:00 | 33.668478 | 7.078165  7.060271  7.051424  7.039162  7.035306  7.027455  7.027755     7.023033 | 1.0  0.918453  0.786203  0.701073  0.564322  0.468803  0.314081  0.199832     0.02617 |    1.0 |     0.0 |  0.02617 |   7.078165 |
    | 2000-01-01 01:10:00 | 33.830091 | 7.080597  7.062868  7.053981  7.041887  7.038179   7.03055  7.031116     7.026476 | 1.0   0.91893  0.789032  0.703781  0.567571  0.471817  0.317069  0.205041    0.032882 |    1.0 |     0.0 | 0.032882 |   7.080597 |
    | 2000-01-01 01:15:00 | 33.990019 | 7.083026  7.065439  7.056554  7.044613  7.041059  7.033633  7.034395     7.029854 | 1.0  0.919026  0.790503  0.704735  0.568434  0.472419   0.31822  0.208898    0.039987 |    1.0 |     0.0 | 0.039987 |   7.083026 |
    | 2000-01-01 01:20:00 | 34.149135 | 7.085462  7.067997  7.059149  7.047347  7.043944  7.036696  7.037587     7.033147 | 1.0  0.918808  0.790913  0.704432  0.567713   0.47157  0.318424  0.212024    0.047372 |    1.0 |     0.0 | 0.047372 |   7.085462 |
    | 2000-01-01 01:25:00 | 34.308116 | 7.087911  7.070553  7.061766  7.050092  7.046825   7.03973  7.040691     7.036346 | 1.0  0.918352  0.790537  0.703286  0.566046  0.469988  0.318316  0.214866    0.054949 |    1.0 |     0.0 | 0.054949 |   7.087911 |
    | 2000-01-01 01:30:00 | 34.467448 | 7.090379  7.073115  7.064406  7.052846  7.049698  7.042727  7.043709     7.039447 | 1.0  0.917734  0.789622   0.70164  0.563931  0.468192  0.318323   0.21773    0.062652 |    1.0 |     0.0 | 0.062652 |   7.090379 |
    | 2000-01-01 01:35:00 | 34.627446 | 7.092868  7.075688  7.067064  7.055606  7.052554  7.045684  7.046646     7.042455 | 1.0  0.917028  0.788385   0.69977   0.56174  0.466536  0.318709  0.220809    0.070436 |    1.0 |     0.0 | 0.070436 |   7.092868 |
    | 2000-01-01 01:40:00 | 34.788273 | 7.095379  7.078274  7.069738   7.05837  7.055388  7.048596  7.049508     7.045373 | 1.0  0.916298  0.787006  0.697896  0.559739  0.465253  0.319621  0.224213    0.078272 |    1.0 |     0.0 | 0.078272 |   7.095379 |
    | 2000-01-01 01:45:00 | 34.949971 | 7.097911  7.080873  7.072421  7.061131  7.058197  7.051464  7.052302      7.04821 | 1.0  0.915599  0.785634  0.696178   0.55811  0.464481  0.321123  0.227994    0.086141 |    1.0 |     0.0 | 0.086141 |   7.097911 |
    | 2000-01-01 01:50:00 | 35.112477 | 7.100462  7.083485  7.075111  7.063886  7.060977  7.054285  7.055033     7.050973 | 1.0  0.914973  0.784379  0.694734  0.556964  0.464291  0.323224  0.232164    0.094032 |    1.0 |     0.0 | 0.094032 |   7.100462 |
    | 2000-01-01 01:55:00 | 35.275657 | 7.103029  7.086108  7.077801  7.066632  7.063727  7.057061  7.057709     7.053668 | 1.0  0.914449  0.783325  0.693639  0.556359  0.464704  0.325899  0.236713    0.101942 |    1.0 |     0.0 | 0.101942 |   7.103029 |
    | 2000-01-01 02:00:00 | 35.439325 | 7.105607  7.088738  7.080489  7.069364  7.066445  7.059793  7.060334     7.056303 | 1.0  0.914047  0.782524  0.692936  0.556315  0.465708  0.329102  0.241612    0.109869 |    1.0 |     0.0 | 0.109869 |   7.105607 |
    | 2000-01-01 02:05:00 | 35.603263 | 7.108194  7.091374   7.08317  7.072081  7.069132  7.062483  7.062912     7.058884 | 1.0  0.913777  0.782009  0.692642  0.556821   0.46727  0.332778  0.246827    0.117813 |    1.0 |     0.0 | 0.117813 |   7.108194 |
    | 2000-01-01 02:10:00 | 35.767241 | 7.110785  7.094011  7.085841  7.074779  7.071787  7.065132  7.065449     7.061415 | 1.0  0.913642  0.781789  0.692753  0.557846  0.469343  0.336871  0.252323    0.125776 |    1.0 |     0.0 | 0.125776 |   7.110785 |
    | 2000-01-01 02:15:00 | 35.931023 | 7.113376  7.096646  7.088499  7.077457  7.074411  7.067744  7.067946     7.063901 | 1.0  0.913638  0.781863  0.693251  0.559348  0.471874  0.341321  0.258063    0.133758 |    1.0 |     0.0 | 0.133758 |   7.113376 |
    | 2000-01-01 02:20:00 | 36.094387 | 7.115963  7.099277  7.091143  7.080114  7.077005  7.070318  7.070408     7.066346 | 1.0  0.913756  0.782216  0.694107  0.561277   0.47481  0.346077  0.264012    0.141761 |    1.0 |     0.0 | 0.141761 |   7.115963 |
    | 2000-01-01 02:25:00 | 36.257123 | 7.118543    7.1019  7.093769  7.082748   7.07957  7.072858  7.072837     7.068753 | 1.0  0.913987  0.782829  0.695287  0.563583  0.478095   0.35109  0.270139    0.149785 |    1.0 |     0.0 | 0.149785 |   7.118543 |
    | 2000-01-01 02:30:00 | 36.419042 | 7.121114  7.104513  7.096376  7.085359  7.082106  7.075366  7.075234     7.071124 | 1.0  0.914318  0.783674  0.696754  0.566213  0.481679  0.356318  0.276416     0.15783 |    1.0 |     0.0 |  0.15783 |   7.121114 |
    | 2000-01-01 02:35:00 | 36.579976 | 7.123672  7.107113  7.098964  7.087946  7.084614  7.077841  7.077601     7.073463 | 1.0  0.914736  0.784726  0.698471  0.569119  0.485514  0.361721  0.282818    0.165895 |    1.0 |     0.0 | 0.165895 |   7.123672 |
    | 2000-01-01 02:40:00 | 36.739783 | 7.126215  7.109699  7.101531  7.090508  7.087095  7.080287   7.07994      7.07577 | 1.0  0.915229  0.785957  0.700401  0.572257  0.489558  0.367269  0.289322     0.17398 |    1.0 |     0.0 |  0.17398 |   7.126215 |
    | 2000-01-01 02:45:00 |  36.89834 | 7.128741  7.112267  7.104076  7.093047  7.089549  7.082704   7.08225     7.078046 | 1.0  0.915784   0.78734  0.702511  0.575586  0.493773  0.372932  0.295911    0.182081 |    1.0 |     0.0 | 0.182081 |   7.128741 |
    | 2000-01-01 02:50:00 | 37.055547 |  7.13125  7.114818  7.106598  7.095561  7.091977  7.085093  7.084534     7.080294 | 1.0  0.916392   0.78885   0.70477   0.57907  0.498128  0.378688  0.302566    0.190197 |    1.0 |     0.0 | 0.190197 |    7.13125 |
    | 2000-01-01 02:55:00 |  37.21132 | 7.133738   7.11735  7.109098   7.09805   7.09438  7.087454  7.086791     7.082513 | 1.0  0.917041  0.790466  0.707149  0.582678  0.502593  0.384516  0.309275    0.198324 |    1.0 |     0.0 | 0.198324 |   7.133738 |
    | 2000-01-01 03:00:00 | 37.365596 | 7.136207  7.119861  7.111574  7.100515  7.096757  7.089789  7.089022     7.084704 | 1.0  0.917724  0.792167  0.709627  0.586386  0.507145  0.390399  0.316024     0.20646 |    1.0 |     0.0 |  0.20646 |   7.136207 |
    | 2000-01-01 03:05:00 | 37.518323 | 7.138654  7.122351  7.114026  7.102955  7.099109  7.092098  7.091228     7.086868 | 1.0  0.918433  0.793936  0.712181  0.590169  0.511765  0.396323  0.322803      0.2146 |    1.0 |     0.0 |   0.2146 |   7.138654 |
    | 2000-01-01 03:10:00 | 37.669464 | 7.141079  7.124819  7.116455  7.105371  7.101437  7.094381  7.093408     7.089005 | 1.0  0.919162  0.795759  0.714795  0.594011  0.516434  0.402277  0.329603    0.222742 |    1.0 |     0.0 | 0.222742 |   7.141079 |
    | 2000-01-01 03:15:00 |  37.81899 | 7.143482  7.127265   7.11886  7.107762  7.103739  7.096639  7.095563     7.091116 | 1.0  0.919907  0.797623  0.717455  0.597895   0.52114  0.408251  0.336416    0.230882 |    1.0 |     0.0 | 0.230882 |   7.143482 |
    | 2000-01-01 03:20:00 | 37.966882 | 7.145862  7.129687  7.121242  7.110129  7.106017  7.098872  7.097693       7.0932 | 1.0  0.920663  0.799519  0.720148   0.60181  0.525871  0.414236  0.343235    0.239015 |    1.0 |     0.0 | 0.239015 |   7.145862 |
    | 2000-01-01 03:25:00 | 38.113129 | 7.148219  7.132087  7.123599  7.112471  7.108271   7.10108  7.099798     7.095259 | 1.0  0.921427  0.801437  0.722865  0.605746  0.530618  0.420226  0.350055    0.247138 |    1.0 |     0.0 | 0.247138 |   7.148219 |
    | 2000-01-01 03:30:00 | 38.257724 | 7.150553  7.134464  7.125932   7.11479  7.110501  7.103263  7.101879     7.097291 | 1.0  0.922196  0.803372  0.725598  0.609694  0.535374  0.426215  0.356869    0.255248 |    1.0 |     0.0 | 0.255248 |   7.150553 |
    | 2000-01-01 03:35:00 | 38.400665 | 7.152864  7.136817  7.128241  7.117083  7.112706  7.105421  7.103934     7.099298 | 1.0  0.922968  0.805317  0.728342  0.613648  0.540132  0.432198  0.363675    0.263341 |    1.0 |     0.0 | 0.263341 |   7.152864 |
    | 2000-01-01 03:40:00 | 38.541953 | 7.155152  7.139146  7.130527  7.119353  7.114888  7.107556  7.105966     7.101279 | 1.0  0.923743  0.807269  0.731091  0.617604  0.544888  0.438172  0.370467    0.271413 |    1.0 |     0.0 | 0.271413 |   7.155152 |
    | 2000-01-01 03:45:00 | 38.681592 | 7.157416  7.141452  7.132788  7.121599  7.117045  7.109666  7.107972     7.103235 | 1.0  0.924518  0.809224  0.733841  0.621557  0.549637  0.444133  0.377243    0.279461 |    1.0 |     0.0 | 0.279461 |   7.157416 |
    | 2000-01-01 03:50:00 |  38.81959 | 7.159657  7.143734  7.135026  7.123821  7.119179  7.111752  7.109955     7.105165 | 1.0  0.925292   0.81118  0.736589  0.625503  0.554377  0.450078  0.383999    0.287482 |    1.0 |     0.0 | 0.287482 |   7.159657 |
    | 2000-01-01 03:55:00 | 38.955953 | 7.161875  7.145993   7.13724  7.126018  7.121289  7.113814  7.111913      7.10707 | 1.0  0.926065  0.813133  0.739334  0.629441  0.559105  0.456005  0.390732    0.295473 |    1.0 |     0.0 | 0.295473 |   7.161875 |
    | 2000-01-01 04:00:00 | 39.090692 |  7.16407  7.148228   7.13943  7.128192  7.123376  7.115852  7.113847      7.10895 | 1.0  0.926837  0.815084  0.742072  0.633368  0.563818  0.461911   0.39744     0.30343 |    1.0 |     0.0 |  0.30343 |    7.16407 |
    | 2000-01-01 04:05:00 | 39.223815 | 7.166242   7.15044  7.141597  7.130343  7.125439  7.117866  7.115757     7.110806 | 1.0  0.927606   0.81703  0.744802  0.637282  0.568515  0.467794  0.404121    0.311352 |    1.0 |     0.0 | 0.311352 |   7.166242 |
    | 2000-01-01 04:10:00 | 39.355334 | 7.168391  7.152628   7.14374   7.13247  7.127478  7.119857  7.117644     7.112636 | 1.0  0.928373   0.81897  0.747524  0.641182  0.573194  0.473653  0.410772    0.319235 |    1.0 |     0.0 | 0.319235 |   7.168391 |
    | 2000-01-01 04:15:00 |  39.48526 | 7.170517  7.154792   7.14586  7.134573  7.129495  7.121825  7.119506     7.114443 | 1.0  0.929137  0.820904  0.750237  0.645068  0.577854  0.479486  0.417392    0.327078 |    1.0 |     0.0 | 0.327078 |   7.170517 |
    | 2000-01-01 04:20:00 | 39.613604 |  7.17262  7.156934  7.147957  7.136653  7.131488  7.123769  7.121346     7.116225 | 1.0  0.929899  0.822831  0.752938  0.648937  0.582494  0.485292  0.423979    0.334877 |    1.0 |     0.0 | 0.334877 |    7.17262 |
    | 2000-01-01 04:25:00 | 39.740379 |   7.1747  7.159052  7.150031   7.13871  7.133458  7.125689  7.123162     7.117983 | 1.0  0.930657   0.82475  0.755628  0.652788  0.587111  0.491068   0.43053    0.342631 |    1.0 |     0.0 | 0.342631 |     7.1747 |
    | 2000-01-01 04:30:00 | 39.865596 | 7.176758  7.161147  7.152081  7.140744  7.135406  7.127587  7.124955     7.119717 | 1.0  0.931412  0.826661  0.758306  0.656622  0.591706  0.496814  0.437045    0.350339 |    1.0 |     0.0 | 0.350339 |   7.176758 |
    | 2000-01-01 04:35:00 | 39.989269 | 7.178793  7.163219  7.154109  7.142754  7.137331  7.129462  7.126725     7.121427 | 1.0  0.932164  0.828564  0.760972  0.660437  0.596278  0.502528  0.443522    0.357997 |    1.0 |     0.0 | 0.357997 |   7.178793 |
    | 2000-01-01 04:40:00 | 40.111408 | 7.180805  7.165268  7.156114  7.144742  7.139233  7.131315  7.128473     7.123114 | 1.0  0.932912  0.830457  0.763624  0.664232  0.600824   0.50821   0.44996    0.365604 |    1.0 |     0.0 | 0.365604 |   7.180805 |
    | 2000-01-01 04:45:00 | 40.232028 | 7.182796  7.167294  7.158096  7.146707  7.141113  7.133144  7.130198     7.124778 | 1.0  0.933656  0.832341  0.766263  0.668006  0.605345  0.513857  0.456357     0.37316 |    1.0 |     0.0 |  0.37316 |   7.182796 |
    | 2000-01-01 04:50:00 | 40.351141 | 7.184764  7.169298  7.160056   7.14865   7.14297  7.134952    7.1319     7.126419 | 1.0  0.934396  0.834215  0.768888   0.67176   0.60984   0.51947  0.462712    0.380661 |    1.0 |     0.0 | 0.380661 |   7.184764 |
    | 2000-01-01 04:55:00 |  40.46876 |  7.18671  7.171279  7.161994   7.15057  7.144806  7.136737  7.133581     7.128038 | 1.0  0.935133  0.836079  0.771498  0.675491  0.614308  0.525046  0.469025    0.388108 |    1.0 |     0.0 | 0.388108 |    7.18671 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_channel_initial_flow:

Initial flow
____________

In all of the above examples, the initial flow is zero.  One can set other values,
which in real projects usually happens automatically via reading and writing condition
files and allows one to stop and resume a simulation without impairing the results.
Here, we repeat the :ref:`sw1d_channel_weir_outflow` example with an initial flow of
1 m³/s:

>>> prepare_inits(hs=2.0, qs=1.0)

Due to the water's inertia, some of the water already stored in the channel now
directly moves in the weir's direction, which results in a water level rise at the
weir's front and a start of weir flow much earlier than in the
:ref:`sw1d_channel_weir_outflow` example:

.. integration-test::

    >>> test("sw1d_channel_initial_flow")
    |                date |  timestep |                                                                       waterlevels |                                                                            discharges | inflow | latflow |  outflow | waterlevel |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 29.297819 | 7.002831   7.00034   7.00006  7.000007  7.000086  7.000554   7.00495     7.013705 | 1.0  0.905635  0.888654  0.886644  0.886285  0.883431  0.855742  0.690758    0.005524 |    1.0 |     0.0 | 0.005524 |   7.002831 |
    | 2000-01-01 00:05:00 | 29.842493 | 7.007904  7.001757  7.000667  7.000244  7.001091  7.002901   7.01199     7.020178 | 1.0  0.830914  0.760068  0.739837  0.727986  0.694475  0.577118  0.342421    0.018745 |    1.0 |     0.0 | 0.018745 |   7.007904 |
    | 2000-01-01 00:10:00 | 30.237048 | 7.013565   7.00411  7.002215  7.001124  7.003233  7.006015  7.017107     7.023633 | 1.0   0.81129  0.693616  0.642038  0.598044  0.526631  0.370947  0.200391    0.027634 |    1.0 |     0.0 | 0.027634 |   7.013565 |
    | 2000-01-01 00:15:00 | 30.486654 | 7.019217   7.00717  7.004754  7.002809   7.00614  7.009228  7.020379     7.025438 | 1.0  0.811591   0.65857  0.573923  0.489673  0.392778  0.232129  0.123073    0.032858 |    1.0 |     0.0 | 0.032858 |   7.019217 |
    | 2000-01-01 00:20:00 | 30.633003 | 7.024687  7.010762  7.008136  7.005253   7.00946   7.01226  7.022333     7.026241 | 1.0  0.817677  0.638103  0.525367   0.40319  0.292514  0.140895  0.075748     0.03558 |    1.0 |     0.0 |  0.03558 |   7.024687 |
    | 2000-01-01 00:25:00 | 30.711932 | 7.029971  7.014751  7.012137  7.008308   7.01294  7.015018  7.023465      7.02644 | 1.0  0.823865  0.624393  0.491013  0.338249  0.222254  0.084328  0.046613    0.036656 |    1.0 |     0.0 | 0.036656 |   7.029971 |
    | 2000-01-01 00:30:00 | 30.818157 | 7.035116  7.019032  7.016532  7.011794   7.01641  7.017502  7.024158     7.026319 | 1.0  0.828485  0.614451  0.467962  0.293679  0.178011  0.053817  0.030697    0.036745 |    1.0 |     0.0 | 0.036745 |   7.035116 |
    | 2000-01-01 00:35:00 | 31.127012 | 7.040167  7.023512  7.021123  7.015532  7.019768  7.019755  7.024694     7.026097 | 1.0  0.831649  0.607636   0.45462  0.267694   0.15576  0.043106  0.025243     0.03637 |    1.0 |     0.0 |  0.03637 |   7.040167 |
    | 2000-01-01 00:40:00 | 31.445807 | 7.045143   7.02811   7.02575  7.019372  7.022964   7.02184  7.025265     7.025941 | 1.0  0.834118  0.604222  0.449963  0.257963  0.151435  0.047192   0.02815    0.035943 |    1.0 |     0.0 | 0.035943 |   7.045143 |
    | 2000-01-01 00:45:00 | 31.762917 | 7.050043  7.032751  7.030298  7.023198  7.025987  7.023822  7.025997     7.025976 | 1.0  0.836695  0.604652  0.453053  0.261753  0.161005  0.061913  0.037511    0.035772 |    1.0 |     0.0 | 0.035772 |   7.050043 |
    | 2000-01-01 00:50:00 |  32.07606 | 7.054843  7.037368  7.034688  7.026932  7.028851   7.02576  7.026963     7.026284 | 1.0  0.839984  0.609121  0.462792  0.276074   0.18061  0.083704  0.051503    0.036076 |    1.0 |     0.0 | 0.036076 |   7.054843 |
    | 2000-01-01 00:55:00 |  32.38263 | 7.059514  7.041906  7.038876  7.030532  7.031586  7.027704  7.028197     7.026913 | 1.0  0.844299   0.61743  0.477852  0.297895  0.206724  0.109553  0.068439       0.037 |    1.0 |     0.0 |    0.037 |   7.059514 |
    | 2000-01-01 01:00:00 | 32.680106 | 7.064024  7.046319  7.042843  7.033979  7.034227  7.029689  7.029702     7.027877 | 1.0  0.849667  0.628999  0.496739  0.324359  0.236302  0.137019  0.086845    0.038628 |    1.0 |     0.0 | 0.038628 |   7.064024 |
    | 2000-01-01 01:05:00 | 32.966395 | 7.068347  7.050577  7.046595  7.037278   7.03681  7.031742  7.031463     7.029168 | 1.0  0.855898  0.642978  0.517931  0.352982  0.266888  0.164245  0.105537    0.040999 |    1.0 |     0.0 | 0.040999 |   7.068347 |
    | 2000-01-01 01:10:00 | 33.240053 | 7.072467  7.054663  7.050146  7.040443  7.039364  7.033876  7.033452     7.030759 | 1.0  0.862671    0.6584  0.540016  0.381764  0.296652  0.189942  0.123643    0.044116 |    1.0 |     0.0 | 0.044116 |   7.072467 |
    | 2000-01-01 01:15:00 | 33.500371 | 7.076378  7.058569  7.053522  7.043495   7.04191  7.036097  7.035634     7.032611 | 1.0  0.869619  0.674323   0.56181  0.409241  0.324362  0.213325  0.140599    0.047956 |    1.0 |     0.0 | 0.047956 |   7.076378 |
    | 2000-01-01 01:20:00 | 33.747334 | 7.080086  7.062298  7.056747  7.046454  7.044464  7.038403  7.037972     7.034684 | 1.0  0.876402  0.689941  0.582418  0.434456  0.349304  0.234027  0.156103    0.052474 |    1.0 |     0.0 | 0.052474 |   7.080086 |
    | 2000-01-01 01:25:00 | 33.981507 | 7.083604   7.06586  7.059849  7.049341  7.047036  7.040786   7.04043     7.036933 | 1.0  0.882751   0.70464  0.601246   0.45689  0.371176  0.251993  0.170061    0.057612 |    1.0 |     0.0 | 0.057612 |   7.083604 |
    | 2000-01-01 01:30:00 | 34.203868 | 7.086949   7.06927  7.062851  7.052173  7.049627  7.043238  7.042976     7.039317 | 1.0  0.888484  0.718013  0.617967  0.476363  0.389976   0.26738  0.182522    0.063304 |    1.0 |     0.0 | 0.063304 |   7.086949 |
    | 2000-01-01 01:35:00 | 34.415647 | 7.090144  7.072543  7.065772  7.054964  7.052239  7.045747  7.045581       7.0418 | 1.0  0.893504  0.729847  0.632478  0.492944  0.405901  0.280472  0.193629     0.06948 |    1.0 |     0.0 |  0.06948 |   7.090144 |
    | 2000-01-01 01:40:00 | 34.618178 |  7.09321  7.075697  7.068629  7.057723  7.054867    7.0483  7.048222      7.04435 | 1.0   0.89779   0.74008  0.644836  0.506856  0.419262   0.29161  0.203569    0.076069 |    1.0 |     0.0 | 0.076069 |    7.09321 |
    | 2000-01-01 01:45:00 | 34.812792 | 7.096169  7.078749  7.071436  7.060459  7.057506  7.050885   7.05088     7.046941 | 1.0  0.901372  0.748767  0.655207  0.518413  0.430425  0.301147  0.212545    0.083005 |    1.0 |     0.0 | 0.083005 |   7.096169 |
    | 2000-01-01 01:50:00 | 35.000732 |  7.09904  7.081715  7.074202  7.063176  7.060152  7.053492   7.05354     7.049552 | 1.0  0.904314  0.756033  0.663817  0.527964   0.43976  0.309414  0.220753    0.090229 |    1.0 |     0.0 | 0.090229 |    7.09904 |
    | 2000-01-01 01:55:00 | 35.183114 | 7.101839  7.084608  7.076936  7.065877    7.0628  7.056111   7.05619     7.052165 | 1.0  0.906702  0.762046   0.67092  0.535859   0.44762  0.316708  0.228369    0.097686 |    1.0 |     0.0 | 0.097686 |   7.101839 |
    | 2000-01-01 02:00:00 | 35.360898 |  7.10458   7.08744  7.079643  7.068564  7.065443  7.058731  7.058822      7.05477 | 1.0  0.908623  0.766993  0.676768  0.542424  0.454322  0.323282  0.235547    0.105331 |    1.0 |     0.0 | 0.105331 |    7.10458 |
    | 2000-01-01 02:05:00 | 35.534886 | 7.107275  7.090223  7.082326  7.071237  7.068077  7.061347   7.06143     7.057355 | 1.0  0.910166  0.771056  0.681603  0.547951  0.460141  0.329341  0.242412    0.113124 |    1.0 |     0.0 | 0.113124 |   7.107275 |
    | 2000-01-01 02:10:00 |  35.70573 | 7.109933  7.092963  7.084989  7.073896  7.070698  7.063953   7.06401     7.059916 | 1.0  0.911412  0.774408  0.685641  0.552692  0.465307   0.33505  0.249066    0.121034 |    1.0 |     0.0 | 0.121034 |   7.109933 |
    | 2000-01-01 02:15:00 | 35.873945 |  7.11256  7.095667  7.087633  7.076541  7.073304  7.066542  7.066558     7.062447 | 1.0   0.91243  0.777203  0.689067  0.556857  0.470008  0.340539  0.255586    0.129033 |    1.0 |     0.0 | 0.129033 |    7.11256 |
    | 2000-01-01 02:20:00 | 36.039926 | 7.115161  7.098341  7.090259  7.079169  7.075891  7.069112  7.069074     7.064946 | 1.0   0.91328  0.779575  0.692042  0.560617  0.474394  0.345902  0.262031    0.137101 |    1.0 |     0.0 | 0.137101 |   7.115161 |
    | 2000-01-01 02:25:00 | 36.203969 | 7.117741  7.100989  7.092868  7.081781  7.078456  7.071659  7.071557      7.06741 | 1.0  0.914011  0.781635  0.694695  0.564109  0.478582   0.35121  0.268446    0.145221 |    1.0 |     0.0 | 0.145221 |   7.117741 |
    | 2000-01-01 02:30:00 | 36.366286 | 7.120301  7.103613  7.095458  7.084375     7.081  7.074182  7.074007      7.06984 | 1.0  0.914663  0.783475  0.697131  0.567437   0.48266  0.356514   0.27486     0.15338 |    1.0 |     0.0 |  0.15338 |   7.120301 |
    | 2000-01-01 02:35:00 | 36.527023 | 7.122843  7.106215   7.09803   7.08695  7.083519  7.076679  7.076423     7.072234 | 1.0  0.915266   0.78517  0.699431  0.570682  0.486694  0.361846  0.281294    0.161566 |    1.0 |     0.0 | 0.161566 |   7.122843 |
    | 2000-01-01 02:40:00 | 36.686275 | 7.125368  7.108796  7.100584  7.089505  7.086014  7.079149  7.078807     7.074594 | 1.0  0.915844  0.786776  0.701659  0.573901   0.49073  0.367227  0.287761    0.169771 |    1.0 |     0.0 | 0.169771 |   7.125368 |
    | 2000-01-01 02:45:00 | 36.844095 | 7.127876  7.111357  7.103118  7.092039  7.088485  7.081592   7.08116      7.07692 | 1.0  0.916412  0.788336  0.703859  0.577136  0.494799   0.37267  0.294267    0.177989 |    1.0 |     0.0 | 0.177989 |   7.127876 |
    | 2000-01-01 02:50:00 |  37.00051 | 7.130366  7.113899  7.105632  7.094552  7.090929  7.084007   7.08348     7.079212 | 1.0  0.916984  0.789883  0.706064  0.580412  0.498921  0.378179  0.300815    0.186213 |    1.0 |     0.0 | 0.186213 |   7.130366 |
    | 2000-01-01 02:55:00 | 37.155526 | 7.132839  7.116422  7.108127  7.097043  7.093348  7.086394  7.085771     7.081471 | 1.0  0.917567  0.791438  0.708298  0.583747  0.503106  0.383755  0.307403    0.194439 |    1.0 |     0.0 | 0.194439 |   7.132839 |
    | 2000-01-01 03:00:00 | 37.309133 | 7.135294  7.118925    7.1106  7.099512  7.095742  7.088753  7.088032     7.083698 | 1.0  0.918166  0.793017  0.710573  0.587148  0.507361  0.389393   0.31403    0.202664 |    1.0 |     0.0 | 0.202664 |   7.135294 |
    | 2000-01-01 03:05:00 | 37.461314 | 7.137731  7.121408  7.113052  7.101957   7.09811  7.091085  7.090264     7.085894 | 1.0  0.918784  0.794628  0.712897  0.590619  0.511683  0.395089  0.320692    0.210883 |    1.0 |     0.0 | 0.210883 |   7.137731 |
    | 2000-01-01 03:10:00 | 37.612048 | 7.140148  7.123871  7.115482   7.10438  7.100453   7.09339  7.092467      7.08806 | 1.0  0.919422  0.796277  0.715273  0.594158   0.51607  0.400836  0.327383    0.219094 |    1.0 |     0.0 | 0.219094 |   7.140148 |
    | 2000-01-01 03:15:00 |  37.76131 | 7.142546  7.126313   7.11789  7.106778   7.10277  7.095667  7.094643     7.090196 | 1.0  0.920078  0.797964    0.7177   0.59776  0.520516  0.406626  0.334097    0.227293 |    1.0 |     0.0 | 0.227293 |   7.142546 |
    | 2000-01-01 03:20:00 | 37.909077 | 7.144923  7.128735  7.120275  7.109154  7.105062  7.097919  7.096792     7.092303 | 1.0  0.920753  0.799689  0.720175   0.60142  0.525014  0.412452   0.34083    0.235479 |    1.0 |     0.0 | 0.235479 |   7.144923 |
    | 2000-01-01 03:25:00 | 38.055328 |  7.14728  7.131134  7.122638  7.111505  7.107329  7.100144  7.098914     7.094382 | 1.0  0.921444  0.801449  0.722694   0.60513  0.529558  0.418307  0.347576    0.243647 |    1.0 |     0.0 | 0.243647 |    7.14728 |
    | 2000-01-01 03:30:00 | 38.200043 | 7.149615  7.133513  7.124978  7.113832  7.109572  7.102343  7.101009     7.096433 | 1.0  0.922149  0.803241  0.725253  0.608883  0.534138  0.424183  0.354329    0.251796 |    1.0 |     0.0 | 0.251796 |   7.149615 |
    | 2000-01-01 03:35:00 | 38.343206 | 7.151929  7.135869  7.127294  7.116136   7.11179  7.104516  7.103079     7.098456 | 1.0  0.922867  0.805061  0.727844  0.612671  0.538748  0.430074  0.361083    0.259922 |    1.0 |     0.0 | 0.259922 |   7.151929 |
    | 2000-01-01 03:40:00 | 38.484805 | 7.154221  7.138203  7.129587  7.118415  7.113983  7.106664  7.105123     7.100452 | 1.0  0.923595  0.806905  0.730464  0.616488  0.543381  0.435974  0.367835    0.268024 |    1.0 |     0.0 | 0.268024 |   7.154221 |
    | 2000-01-01 03:45:00 | 38.624829 | 7.156491  7.140514  7.131857  7.120671  7.116152  7.108787  7.107142     7.102422 | 1.0  0.924331  0.808768  0.733106  0.620327  0.548031  0.441876  0.374579    0.276098 |    1.0 |     0.0 | 0.276098 |   7.156491 |
    | 2000-01-01 03:50:00 | 38.763272 | 7.158739  7.142802  7.134104  7.122903  7.118296  7.110886  7.109136     7.104365 | 1.0  0.925075  0.810648  0.735766  0.624182  0.552691  0.447776  0.381312    0.284143 |    1.0 |     0.0 | 0.284143 |   7.158739 |
    | 2000-01-01 03:55:00 |  38.90013 | 7.160964  7.145068  7.136327   7.12511  7.120417  7.112959  7.111105     7.106282 | 1.0  0.925823  0.812539  0.738438  0.628047  0.557356   0.45367   0.38803    0.292156 |    1.0 |     0.0 | 0.292156 |   7.160964 |
    | 2000-01-01 04:00:00 | 39.035403 | 7.163167  7.147311  7.138526  7.127294  7.122514  7.115009   7.11305     7.108174 | 1.0  0.926575   0.81444   0.74112  0.631917  0.562021  0.459552  0.394728    0.300135 |    1.0 |     0.0 | 0.300135 |   7.163167 |
    | 2000-01-01 04:05:00 | 39.169091 | 7.165347   7.14953  7.140703  7.129455  7.124587  7.117034  7.114971     7.110041 | 1.0  0.927329  0.816347  0.743806  0.635789  0.566683  0.465421  0.401404    0.308076 |    1.0 |     0.0 | 0.308076 |   7.165347 |
    | 2000-01-01 04:10:00 | 39.301197 | 7.167505  7.151727  7.142856  7.131591  7.126637  7.119035  7.116867     7.111882 | 1.0  0.928085  0.818257  0.746494  0.639658  0.571336  0.471271  0.408055    0.315979 |    1.0 |     0.0 | 0.315979 |   7.167505 |
    | 2000-01-01 04:15:00 | 39.431728 |  7.16964    7.1539  7.144985  7.133705  7.128663  7.121013   7.11874     7.113699 | 1.0  0.928841  0.820168  0.749182  0.643521  0.575979  0.477101  0.414679    0.323841 |    1.0 |     0.0 | 0.323841 |    7.16964 |
    | 2000-01-01 04:20:00 | 39.560689 | 7.171752  7.156051  7.147091  7.135794  7.130666  7.122967  7.120589     7.115491 | 1.0  0.929596  0.822078  0.751866  0.647375  0.580609  0.482907  0.421272     0.33166 |    1.0 |     0.0 |  0.33166 |   7.171752 |
    | 2000-01-01 04:25:00 | 39.688087 | 7.173841  7.158178  7.149175  7.137861  7.132646  7.124898  7.122415     7.117259 | 1.0  0.930351  0.823985  0.754544  0.651219  0.585222  0.488688  0.427833    0.339434 |    1.0 |     0.0 | 0.339434 |   7.173841 |
    | 2000-01-01 04:30:00 | 39.813933 | 7.175908  7.160282  7.151235  7.139904  7.134603  7.126805  7.124217     7.119003 | 1.0  0.931103  0.825889  0.757216  0.655048  0.589816  0.494442  0.434359    0.347161 |    1.0 |     0.0 | 0.347161 |   7.175908 |
    | 2000-01-01 04:35:00 | 39.938235 | 7.177952  7.162364  7.153272  7.141925  7.136537   7.12869  7.125996     7.120724 | 1.0  0.931854  0.827787  0.759878  0.658863  0.594391  0.500166  0.440849    0.354839 |    1.0 |     0.0 | 0.354839 |   7.177952 |
    | 2000-01-01 04:40:00 | 40.061004 | 7.179974  7.164422  7.155287  7.143922  7.138449  7.130551  7.127753      7.12242 | 1.0  0.932602  0.829678   0.76253  0.662661  0.598943  0.505859  0.447301    0.362468 |    1.0 |     0.0 | 0.362468 |   7.179974 |
    | 2000-01-01 04:45:00 | 40.182252 | 7.181974  7.166458  7.157278  7.145897  7.140338   7.13239  7.129487     7.124094 | 1.0  0.933346  0.831562   0.76517  0.666441  0.603472  0.511519  0.453714    0.370044 |    1.0 |     0.0 | 0.370044 |   7.181974 |
    | 2000-01-01 04:50:00 | 40.301989 | 7.183951  7.168471  7.159248  7.147849  7.142204  7.134207  7.131199     7.125744 | 1.0  0.934088  0.833438  0.767798  0.670201  0.607976  0.517146  0.460086    0.377568 |    1.0 |     0.0 | 0.377568 |   7.183951 |
    | 2000-01-01 04:55:00 | 40.420228 | 7.185907  7.170461  7.161194  7.149778  7.144049  7.136001  7.132889     7.127372 | 1.0  0.934826  0.835305  0.770413  0.673941  0.612454  0.522737  0.466416    0.385037 |    1.0 |     0.0 | 0.385037 |   7.185907 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_channel_unequal_profiles:

Unequal profiles
________________

This experiment deals with non-identical channel profiles.  To demonstrate the effect
of a rapid change in geometry, we increase the bottom width of the last four channel
segments from 5 m to 10 m:

>>> for storagemodel in model.storagemodels[4:]:
...     storagemodel.crosssection.parameters.control.bottomwidths(10.0)

As the |sw1d_lias| models need to know the channel geometry as well, we set the bottom
width of the central one to the average value of 7.5 m and the bottom width of the last
three to 10 m:

>>> model.routingmodels[4].crosssection.parameters.control.bottomwidths(7.5)
>>> for routingmodel in model.routingmodels[5:-1]:
...     routingmodel.crosssection.parameters.control.bottomwidths(10.0)

We reset the initial depth to 1 m and the initial discharge to zero:

>>> prepare_inits(hs=1.0, qs=0.0)

For water depths of about 1 m, no weir flow occurs.  Hence, when using the same time
step factor, the following results are readily comparable with the ones of the
:ref:`sw1d_channel_longitudinal_inflow` example:

>>> for routingmodel in model.routingmodels.submodels:
...     routingmodel.parameters.control.timestepfactor(0.7)

Compared to the :ref:`sw1d_channel_longitudinal_inflow` example, there is now a more
marked difference between the water level gradient of the first (narrower) and last
(wider) channel segments:

.. integration-test::

    >>> conditions = test("sw1d_channel_unequal_profiles", get_conditions="2000-01-01 00:00")
    |                date | timestep |                                                                       waterlevels |                                                                            discharges | inflow | latflow | outflow | waterlevel |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |    300.0 |     6.03       6.0       6.0       6.0       6.0       6.0       6.0          6.0 | 1.0       0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |    1.0 |     0.0 |     0.0 |       6.03 |
    | 2000-01-01 00:05:00 |    300.0 | 6.054607  6.003595       6.0       6.0       6.0       6.0       6.0          6.0 | 1.0  0.179758       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |    1.0 |     0.0 |     0.0 |   6.054607 |
    | 2000-01-01 00:10:00 |    300.0 | 6.070015  6.012899  6.000636       6.0       6.0       6.0       6.0          6.0 | 1.0  0.486393  0.021192       0.0       0.0       0.0       0.0       0.0         0.0 |    1.0 |     0.0 |     0.0 |   6.070015 |
    | 2000-01-01 00:15:00 |    300.0 | 6.081535  6.022503  6.004598  6.000075       6.0       6.0       6.0          6.0 | 1.0  0.616016  0.135813  0.003743       0.0       0.0       0.0       0.0         0.0 |    1.0 |     0.0 |     0.0 |   6.081535 |
    | 2000-01-01 00:20:00 |    300.0 | 6.091495  6.030868  6.010796  6.000924   6.00001       6.0       6.0          6.0 | 1.0   0.66799   0.24975  0.043139  0.000661       0.0       0.0       0.0         0.0 |    1.0 |     0.0 |     0.0 |   6.091495 |
    | 2000-01-01 00:25:00 |    300.0 | 6.100528  6.038324  6.017159  6.002947  6.000202  6.000001       6.0          6.0 | 1.0    0.6989  0.326119  0.114028  0.012916  0.000117       0.0       0.0         0.0 |    1.0 |     0.0 |     0.0 |   6.100528 |
    | 2000-01-01 00:30:00 |    300.0 | 6.108864  6.045292  6.022611  6.005872  6.000832  6.000039       6.0          6.0 | 1.0  0.722135  0.373708  0.191977  0.045727  0.003748  0.000014       0.0         0.0 |    1.0 |     0.0 |     0.0 |   6.108864 |
    | 2000-01-01 00:35:00 |    300.0 | 6.116644  6.051929  6.027193  6.009021  6.002057  6.000199  6.000013          6.0 | 1.0  0.740666  0.408826  0.256071   0.09861  0.016892  0.000837  0.000002         0.0 |    1.0 |     0.0 |     0.0 |   6.116644 |
    | 2000-01-01 00:40:00 |    300.0 | 6.123956  6.058274   6.03128  6.011869  6.003787  6.000605  6.000077     6.000002 | 1.0  0.756266  0.439014  0.302794  0.160384  0.045106  0.004551  0.000235         0.0 |    1.0 |     0.0 |     0.0 |   6.123956 |
    | 2000-01-01 00:45:00 |    300.0 | 6.130855  6.064354  6.035122  6.014263  6.005729  6.001349   6.00027     6.000018 | 1.0  0.770032  0.466033  0.337972  0.218258  0.088785  0.014361  0.001527         0.0 |    1.0 |     0.0 |     0.0 |   6.130855 |
    | 2000-01-01 00:50:00 |    300.0 | 6.137382  6.070196  6.038818   6.01629  6.007588  6.002437  6.000682     6.000074 | 1.0  0.782429  0.490313  0.367099  0.265768  0.141835  0.033069  0.005626         0.0 |    1.0 |     0.0 |     0.0 |   6.137382 |
    | 2000-01-01 00:55:00 |    300.0 | 6.143573  6.075823  6.042403  6.018085  6.009207  6.003775  6.001376     6.000223 | 1.0  0.793644  0.512324  0.392848  0.303076  0.195088  0.061207  0.014936         0.0 |    1.0 |     0.0 |     0.0 |   6.143573 |
    | 2000-01-01 01:00:00 |    300.0 | 6.149459  6.081249  6.045892  6.019749  6.010585  6.005225  6.002348     6.000538 | 1.0  0.803817  0.532516  0.416208  0.333038  0.241233  0.096301   0.03149         0.0 |    1.0 |     0.0 |     0.0 |   6.149459 |
    | 2000-01-01 01:05:00 |    300.0 | 6.155066  6.086486  6.049299  6.021338  6.011797  6.006658  6.003524     6.001094 | 1.0  0.813078  0.551196   0.43764  0.358158  0.277349  0.133972  0.055571         0.0 |    1.0 |     0.0 |     0.0 |   6.155066 |
    | 2000-01-01 01:10:00 |    300.0 |  6.16042  6.091546  6.052632  6.022892   6.01293  6.008002  6.004799     6.001943 | 1.0  0.821547  0.568565  0.457461  0.379804  0.304245  0.169933  0.084947         0.0 |    1.0 |     0.0 |     0.0 |    6.16042 |
    | 2000-01-01 01:15:00 |    300.0 |  6.16554  6.096437  6.055898  6.024436  6.014043  6.009232  6.006088     6.003097 | 1.0  0.829321   0.58477  0.475881  0.398647  0.324442  0.201387  0.115401         0.0 |    1.0 |     0.0 |     0.0 |    6.16554 |
    | 2000-01-01 01:20:00 |    300.0 | 6.170446  6.101168  6.059105  6.025993  6.015163  6.010366  6.007361     6.004521 | 1.0  0.836485  0.599923  0.493021  0.415181  0.340502  0.227158  0.142351         0.0 |    1.0 |     0.0 |     0.0 |   6.170446 |
    | 2000-01-01 01:25:00 |    300.0 | 6.175152  6.105748   6.06226  6.027574  6.016299  6.011437  6.008629     6.006146 | 1.0  0.843106  0.614111  0.508973  0.429934  0.354238  0.247071  0.162521         0.0 |    1.0 |     0.0 |     0.0 |   6.175152 |
    | 2000-01-01 01:30:00 |    300.0 | 6.179675  6.110185  6.065366  6.029182  6.017451   6.01249  6.009928     6.007893 | 1.0  0.849242  0.627401  0.523847  0.443429  0.366582  0.261348  0.174735         0.0 |    1.0 |     0.0 |     0.0 |   6.179675 |
    | 2000-01-01 01:35:00 |    300.0 | 6.184027  6.114487  6.068429  6.030816  6.018626  6.013564  6.011288      6.00969 | 1.0   0.85494  0.639857  0.537772  0.456065  0.377774  0.270371  0.179688         0.0 |    1.0 |     0.0 |     0.0 |   6.184027 |
    | 2000-01-01 01:40:00 |    300.0 |  6.18822  6.118661  6.071449  6.032473  6.019832  6.014693  6.012721     6.011482 | 1.0  0.860241  0.651544  0.550879  0.468053  0.387668  0.274702  0.179165         0.0 |    1.0 |     0.0 |     0.0 |    6.18822 |
    | 2000-01-01 01:45:00 |    300.0 | 6.192264  6.122714  6.074426   6.03415  6.021083  6.015902   6.01422     6.013234 | 1.0  0.865181  0.662531  0.563275  0.479414  0.396013  0.275155  0.175234         0.0 |    1.0 |     0.0 |     0.0 |   6.192264 |
    | 2000-01-01 01:50:00 |    300.0 | 6.196171  6.126652  6.077362   6.03585  6.022394  6.017201  6.015766     6.014931 | 1.0  0.869793  0.672883  0.575025  0.490051  0.402655  0.272769   0.16971         0.0 |    1.0 |     0.0 |     0.0 |   6.196171 |
    | 2000-01-01 01:55:00 |    300.0 | 6.199947  6.130481  6.080257  6.037576  6.023776   6.01859  6.017338     6.016571 | 1.0  0.874107  0.682655  0.586159  0.499831  0.407642  0.268694  0.163924         0.0 |    1.0 |     0.0 |     0.0 |   6.199947 |
    | 2000-01-01 02:00:00 |    300.0 | 6.203603  6.134206  6.083113  6.039336  6.025238  6.020062  6.018917     6.018158 | 1.0  0.878152  0.691892  0.596678  0.508665   0.41123  0.264035   0.15871         0.0 |    1.0 |     0.0 |     0.0 |   6.203603 |
    | 2000-01-01 02:05:00 |    300.0 | 6.207144  6.137833  6.085935  6.041137  6.026779  6.021603  6.020495     6.019703 | 1.0   0.88195  0.700624  0.606579  0.516551  0.413827  0.259716  0.154514         0.0 |    1.0 |     0.0 |     0.0 |   6.207144 |
    | 2000-01-01 02:10:00 |    300.0 | 6.210579  6.141366  6.088725  6.042983  6.028394  6.023198  6.022069     6.021218 | 1.0  0.885523  0.708874  0.615863  0.523578  0.415904  0.256391   0.15152         0.0 |    1.0 |     0.0 |     0.0 |   6.210579 |
    | 2000-01-01 02:15:00 |    300.0 | 6.213912   6.14481  6.091488  6.044875  6.030074  6.024833  6.023639     6.022716 | 1.0  0.888885  0.716657  0.624549  0.529912  0.417914  0.254433  0.149759         0.0 |    1.0 |     0.0 |     0.0 |   6.213912 |
    | 2000-01-01 02:20:00 |    300.0 | 6.217151  6.148171  6.094228  6.046814  6.031807  6.026496   6.02521     6.024207 | 1.0  0.892049   0.72399  0.632674  0.535754  0.420222   0.25396  0.149179         0.0 |    1.0 |     0.0 |     0.0 |   6.217151 |
    | 2000-01-01 02:25:00 |    300.0 |   6.2203  6.151454  6.096946  6.048793   6.03358  6.028177  6.026789     6.025704 | 1.0  0.895027  0.730891   0.64029  0.541308   0.42307  0.254905  0.149676         0.0 |    1.0 |     0.0 |     0.0 |     6.2203 |
    | 2000-01-01 02:30:00 |    300.0 | 6.223365  6.154663  6.099643  6.050808  6.035383  6.029872  6.028378     6.027215 | 1.0  0.897827  0.737383  0.647462  0.546749  0.426573  0.257083  0.151103         0.0 |    1.0 |     0.0 |     0.0 |   6.223365 |
    | 2000-01-01 02:35:00 |    300.0 | 6.226351  6.157802  6.102321  6.052849  6.037205  6.031577  6.029983     6.028748 | 1.0  0.900458  0.743493  0.654254  0.552205  0.430733  0.260252  0.153284         0.0 |    1.0 |     0.0 |     0.0 |   6.226351 |
    | 2000-01-01 02:40:00 |    300.0 | 6.229263  6.160876  6.104976  6.054908  6.039039   6.03329  6.031605     6.030308 | 1.0  0.902932  0.749253  0.660727  0.557749  0.435472  0.264155   0.15602         0.0 |    1.0 |     0.0 |     0.0 |   6.229263 |
    | 2000-01-01 02:45:00 |    300.0 | 6.232106  6.163887  6.107609  6.056979   6.04088  6.035011  6.033247     6.031899 | 1.0  0.905258  0.754695  0.666935  0.563405  0.440663  0.268546   0.15911         0.0 |    1.0 |     0.0 |     0.0 |   6.232106 |
    | 2000-01-01 02:50:00 |    300.0 | 6.234882  6.166839  6.110217  6.059054  6.042725  6.036741  6.034909     6.033523 | 1.0  0.907448  0.759854  0.672917  0.569155  0.446158    0.2732  0.162366         0.0 |    1.0 |     0.0 |     0.0 |   6.234882 |
    | 2000-01-01 02:55:00 |    300.0 | 6.237597  6.169734  6.112799  6.061129  6.044573   6.03848  6.036593     6.035179 | 1.0  0.909514   0.76476  0.678705  0.574959  0.451809  0.277927   0.16563         0.0 |    1.0 |     0.0 |     0.0 |   6.237597 |
    | 2000-01-01 03:00:00 |    300.0 | 6.240253  6.172575  6.115353    6.0632  6.046422  6.040229    6.0383     6.036867 | 1.0  0.911467   0.76944  0.684316  0.580762  0.457484  0.282569   0.16878         0.0 |    1.0 |     0.0 |     0.0 |   6.240253 |
    | 2000-01-01 03:05:00 |    300.0 | 6.242853  6.175363  6.117877  6.065265  6.048273   6.04199  6.040029     6.038584 | 1.0  0.913318  0.773918   0.68976  0.586507  0.463071  0.287007  0.171733         0.0 |    1.0 |     0.0 |     0.0 |   6.242853 |
    | 2000-01-01 03:10:00 |    300.0 | 6.245401    6.1781  6.120373  6.067323  6.050128  6.043763   6.04178     6.040329 | 1.0  0.915078  0.778215  0.695044  0.592143  0.468483  0.291161  0.174442         0.0 |    1.0 |     0.0 |     0.0 |   6.245401 |
    | 2000-01-01 03:15:00 |    300.0 | 6.247898  6.180788  6.122838  6.069374  6.051988   6.04555  6.043552     6.042098 | 1.0  0.916756  0.782344  0.700166  0.597624  0.473658  0.294985  0.176886         0.0 |    1.0 |     0.0 |     0.0 |   6.247898 |
    | 2000-01-01 03:20:00 |    300.0 | 6.250347  6.183429  6.125274  6.071418  6.053853  6.047351  6.045343     6.043888 | 1.0  0.918359  0.786319  0.705125  0.602919  0.478558  0.298466  0.179067         0.0 |    1.0 |     0.0 |     0.0 |   6.250347 |
    | 2000-01-01 03:25:00 |    300.0 |  6.25275  6.186024  6.127681  6.073456  6.055726  6.049166  6.047152     6.045698 | 1.0  0.919894  0.790148  0.709918  0.608005  0.483164  0.301614  0.181003         0.0 |    1.0 |     0.0 |     0.0 |    6.25275 |
    | 2000-01-01 03:30:00 |    300.0 | 6.255109  6.188574  6.130059   6.07549  6.057607  6.050996  6.048978     6.047526 | 1.0  0.921368  0.793838  0.714544  0.612868  0.487476  0.304455   0.18272         0.0 |    1.0 |     0.0 |     0.0 |   6.255109 |
    | 2000-01-01 03:35:00 |    300.0 | 6.257426  6.191082  6.132411   6.07752  6.059497  6.052841  6.050819     6.049368 | 1.0  0.922783  0.797393  0.719001  0.617504  0.491506  0.307026   0.18425         0.0 |    1.0 |     0.0 |     0.0 |   6.257426 |
    | 2000-01-01 03:40:00 |    300.0 | 6.259702  6.193549  6.134737  6.079547  6.061396    6.0547  6.052676     6.051224 | 1.0  0.924143  0.800817  0.723289  0.621916  0.495275  0.309369  0.185627         0.0 |    1.0 |     0.0 |     0.0 |   6.259702 |
    | 2000-01-01 03:45:00 |    300.0 | 6.261938  6.195975  6.137038  6.081573  6.063306  6.056573  6.054545     6.053093 | 1.0  0.925452  0.804114  0.727409  0.626111  0.498808  0.311524  0.186883         0.0 |    1.0 |     0.0 |     0.0 |   6.261938 |
    | 2000-01-01 03:50:00 |    300.0 | 6.264137  6.198364  6.139316  6.083598  6.065225  6.058459  6.056427     6.054974 | 1.0   0.92671  0.807286  0.731364  0.630101  0.502134  0.313529  0.188047         0.0 |    1.0 |     0.0 |     0.0 |   6.264137 |
    | 2000-01-01 03:55:00 |    300.0 | 6.266299  6.200716  6.141571  6.085624  6.067155  6.060358  6.058322     6.056865 | 1.0  0.927921  0.810335  0.735158  0.633899  0.505279  0.315418  0.189145         0.0 |    1.0 |     0.0 |     0.0 |   6.266299 |
    | 2000-01-01 04:00:00 |    300.0 | 6.268427  6.203032  6.143805  6.087649  6.069093  6.062268  6.060227     6.058767 | 1.0  0.929085  0.813264  0.738794  0.637519  0.508268  0.317219  0.190195         0.0 |    1.0 |     0.0 |     0.0 |   6.268427 |
    | 2000-01-01 04:05:00 |    300.0 | 6.270521  6.205315  6.146019  6.089675  6.071041   6.06419  6.062143     6.060679 | 1.0  0.930203  0.816077   0.74228  0.640976  0.511123  0.318952  0.191214         0.0 |    1.0 |     0.0 |     0.0 |   6.270521 |
    | 2000-01-01 04:10:00 |    300.0 | 6.272582  6.207565  6.148214  6.091702  6.072997  6.066122  6.064069     6.062601 | 1.0  0.931277  0.818776   0.74562  0.644282   0.51386  0.320633  0.192211         0.0 |    1.0 |     0.0 |     0.0 |   6.272582 |
    | 2000-01-01 04:15:00 |    300.0 | 6.274613  6.209784   6.15039  6.093729  6.074962  6.068064  6.066006     6.064533 | 1.0  0.932308  0.821365  0.748822  0.647449  0.516495  0.322273  0.193191         0.0 |    1.0 |     0.0 |     0.0 |   6.274613 |
    | 2000-01-01 04:20:00 |    300.0 | 6.276614  6.211973  6.152549  6.095757  6.076934  6.070016  6.067951     6.066475 | 1.0  0.933297  0.823847   0.75189  0.650489  0.519038  0.323876  0.194157         0.0 |    1.0 |     0.0 |     0.0 |   6.276614 |
    | 2000-01-01 04:25:00 |    300.0 | 6.278587  6.214133  6.154691  6.097786  6.078912  6.071976  6.069906     6.068426 | 1.0  0.934246  0.826226  0.754831  0.653409  0.521496  0.325447   0.19511         0.0 |    1.0 |     0.0 |     0.0 |   6.278587 |
    | 2000-01-01 04:30:00 |    300.0 | 6.280532  6.216266  6.156816  6.099815  6.080897  6.073945   6.07187     6.070386 | 1.0  0.935155  0.828507  0.757651  0.656217  0.523874  0.326983  0.196048         0.0 |    1.0 |     0.0 |     0.0 |   6.280532 |
    | 2000-01-01 04:35:00 |    300.0 | 6.282451  6.218373  6.158926  6.101843  6.082889  6.075922  6.073843     6.072356 | 1.0  0.936026  0.830692  0.760356  0.658918  0.526176  0.328484  0.196967         0.0 |    1.0 |     0.0 |     0.0 |   6.282451 |
    | 2000-01-01 04:40:00 |    300.0 | 6.284345  6.220454  6.161022  6.103872  6.084885  6.077907  6.075824     6.074335 | 1.0   0.93686  0.832785   0.76295  0.661519  0.528402  0.329945  0.197865         0.0 |    1.0 |     0.0 |     0.0 |   6.284345 |
    | 2000-01-01 04:45:00 |    300.0 | 6.286216  6.222511  6.163102    6.1059  6.086887  6.079899  6.077814     6.076322 | 1.0   0.93766  0.834791  0.765439  0.664021  0.530554  0.331365  0.198739         0.0 |    1.0 |     0.0 |     0.0 |   6.286216 |
    | 2000-01-01 04:50:00 |    300.0 | 6.288063  6.224546  6.165169  6.107928  6.088894  6.081898  6.079811     6.078318 | 1.0  0.938425  0.836713  0.767826  0.666429  0.532631  0.332739  0.199585         0.0 |    1.0 |     0.0 |     0.0 |   6.288063 |
    | 2000-01-01 04:55:00 |    300.0 | 6.289888  6.226558  6.167222  6.109956  6.090906  6.083903  6.081816     6.080322 | 1.0  0.939157  0.838555  0.770116  0.668745  0.534633  0.334065  0.200402         0.0 |    1.0 |     0.0 |     0.0 |   6.289888 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_dry_channel:

Dry channel
___________

The propagation of a flood wave through an initially dry channel brings extra
challenges for any routing algorithm.  Monitor the numerical accuracy (and stability)
closely for such situations.  Here, we again reduce the internal numerical step sizes
for this reason:

>>> for routingmodel in model.routingmodels.submodels:
...     routingmodel.parameters.control.timestepfactor(0.1)

For simplicity, we reset all profiles to the same width:

>>> for storagemodel in model.storagemodels[4:]:
...     storagemodel.crosssection.parameters.control.bottomwidths(5.0)
>>> for routingmodel in model.routingmodels[4:-1]:
...     routingmodel.crosssection.parameters.control.bottomwidths(5.0)

At least when using |sw1d_lias| as the routing submodel, starting with zero initial
water depth does not result in apparent errors (e.g. zero divisions).  The water level
trajectories look stable and plausible.  However, inspecting the discharge between the
first and the second segment reveals a slight tendency to instability (which magnifies
when increasing the time step factor):

.. integration-test::

    >>> prepare_inits(hs=0.0, qs=0.0)
    >>> conditions = test("sw1d_channel_dry_channel", get_conditions="2000-01-01 00:00")
    |                date |   timestep |                                                             waterlevels |                                                                  discharges | inflow | latflow | outflow | waterlevel |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |      300.0 |     5.03       5.0       5.0       5.0       5.0  5.0  5.0          5.0 | 1.0       0.0       0.0       0.0       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |       5.03 |
    | 2000-01-01 00:05:00 |  32.922721 | 5.059238  5.000485  5.000034  5.000001       5.0  5.0  5.0          5.0 | 1.0  0.025395  0.001158  0.000029  0.000001       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.059238 |
    | 2000-01-01 00:10:00 |   3.140263 | 5.088345  5.001074  5.000043  5.000001       5.0  5.0  5.0          5.0 | 1.0  0.029774  0.000307  0.000014  0.000001       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.088345 |
    | 2000-01-01 00:15:00 |   2.343509 | 5.117256  5.001797  5.000047  5.000001       5.0  5.0  5.0          5.0 | 1.0  0.036308   0.00016  0.000004       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.117256 |
    | 2000-01-01 00:20:00 |  76.483711 | 5.145949  5.002667  5.000049  5.000001       5.0  5.0  5.0          5.0 | 1.0  0.043556  0.000053  0.000001       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.145949 |
    | 2000-01-01 00:25:00 |  22.029162 | 5.174262  5.003789  5.000053  5.000001       5.0  5.0  5.0          5.0 | 1.0  0.056223  0.000135  0.000003       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.174262 |
    | 2000-01-01 00:30:00 |  84.897296 |  5.20238  5.005041  5.000057  5.000001       5.0  5.0  5.0          5.0 | 1.0  0.062757   0.00013  0.000001       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |    5.20238 |
    | 2000-01-01 00:35:00 |   50.34963 | 5.230025  5.006607  5.000062  5.000001       5.0  5.0  5.0          5.0 | 1.0  0.078489  0.000189  0.000004       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.230025 |
    | 2000-01-01 00:40:00 |  16.285185 | 5.257395  5.008355  5.000071  5.000001       5.0  5.0  5.0          5.0 | 1.0  0.087657  0.000286  0.000002       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.257395 |
    | 2000-01-01 00:45:00 | 145.562892 | 5.284469  5.010295  5.000087  5.000001       5.0  5.0  5.0          5.0 | 1.0  0.097537   0.00053  0.000002       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.284469 |
    | 2000-01-01 00:50:00 | 147.269604 | 5.310886  5.012672  5.000104  5.000002       5.0  5.0  5.0          5.0 | 1.0  0.119443  0.000608   0.00002  0.000001       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.310886 |
    | 2000-01-01 00:55:00 |  11.226983 | 5.336609  5.015506  5.000129  5.000002       5.0  5.0  5.0          5.0 | 1.0  0.142565  0.000852  0.000035  0.000002       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.336609 |
    | 2000-01-01 01:00:00 |  22.622175 | 5.361796   5.01869  5.000166  5.000002       5.0  5.0  5.0          5.0 | 1.0  0.160431   0.00125  0.000002  0.000001       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.361796 |
    | 2000-01-01 01:05:00 |   32.64561 | 5.386276  5.022343  5.000205  5.000002       5.0  5.0  5.0          5.0 | 1.0  0.184005  0.001309  0.000002       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.386276 |
    | 2000-01-01 01:10:00 |  41.539704 | 5.410034  5.026471  5.000257  5.000002       5.0  5.0  5.0          5.0 | 1.0  0.208065  0.001712  0.000001       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.410034 |
    | 2000-01-01 01:15:00 |  49.491413 | 5.433062  5.031077   5.00032  5.000002       5.0  5.0  5.0          5.0 | 1.0  0.232394  0.002095  0.000002       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.433062 |
    | 2000-01-01 01:20:00 |  56.648831 | 5.455351  5.036165  5.000398  5.000002       5.0  5.0  5.0          5.0 | 1.0  0.257047  0.002613  0.000002       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.455351 |
    | 2000-01-01 01:25:00 |   63.12824 | 5.476891   5.04174  5.000494  5.000002       5.0  5.0  5.0          5.0 | 1.0  0.281982  0.003225  0.000004       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.476891 |
    | 2000-01-01 01:30:00 |  69.023937 | 5.497676  5.047805  5.000612  5.000003       5.0  5.0  5.0          5.0 | 1.0  0.307169  0.003938  0.000004       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.497676 |
    | 2000-01-01 01:35:00 |  74.412568 |   5.5177   5.05436  5.000756  5.000003       5.0  5.0  5.0          5.0 | 1.0  0.332534  0.004783  0.000007       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |     5.5177 |
    | 2000-01-01 01:40:00 |  79.357614 |  5.53696  5.061405  5.000928  5.000003       5.0  5.0  5.0          5.0 | 1.0  0.358008  0.005749  0.000008       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |    5.53696 |
    | 2000-01-01 01:45:00 |  83.912002 | 5.555455  5.068938  5.001134  5.000003       5.0  5.0  5.0          5.0 | 1.0  0.383505  0.006864  0.000013       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.555455 |
    | 2000-01-01 01:50:00 |  88.120373 | 5.573186  5.076954  5.001377  5.000003       5.0  5.0  5.0          5.0 | 1.0  0.408944  0.008127  0.000015       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.573186 |
    | 2000-01-01 01:55:00 |  92.020666 | 5.590159  5.085448  5.001663  5.000004       5.0  5.0  5.0          5.0 | 1.0  0.434238  0.009557  0.000023       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.590159 |
    | 2000-01-01 02:00:00 |  95.645383 |  5.60638  5.094411  5.001997  5.000004       5.0  5.0  5.0          5.0 | 1.0  0.459302  0.011162  0.000026  0.000001       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |    5.60638 |
    | 2000-01-01 02:05:00 |  99.022565 | 5.621859  5.103833  5.002384  5.000005       5.0  5.0  5.0          5.0 | 1.0  0.484054  0.012955   0.00004       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.621859 |
    | 2000-01-01 02:10:00 |   4.009331 | 5.636608  5.113687  5.002852  5.000006       5.0  5.0  5.0          5.0 | 1.0  0.508342   0.01563  0.000057  0.000001       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.636608 |
    | 2000-01-01 02:15:00 |   8.379898 | 5.650626  5.123996  5.003367  5.000008       5.0  5.0  5.0          5.0 | 1.0  0.532736  0.017264  0.000074  0.000001       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.650626 |
    | 2000-01-01 02:20:00 |   12.48123 | 5.663958  5.134709  5.003964   5.00001       5.0  5.0  5.0          5.0 | 1.0  0.555603   0.01997  0.000095  0.000001       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.663958 |
    | 2000-01-01 02:25:00 |  16.338488 | 5.676624  5.145806  5.004648  5.000012       5.0  5.0  5.0          5.0 | 1.0  0.577789  0.022926  0.000127  0.000001       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.676624 |
    | 2000-01-01 02:30:00 |  19.973015 | 5.688647  5.157268  5.005427  5.000015       5.0  5.0  5.0          5.0 | 1.0  0.599245  0.026149  0.000166       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.688647 |
    | 2000-01-01 02:35:00 |  23.403758 | 5.700049  5.169074   5.00631   5.00002       5.0  5.0  5.0          5.0 | 1.0  0.619947  0.029653   0.00021       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.700049 |
    | 2000-01-01 02:40:00 |  26.647588 | 5.710853  5.181203  5.007306  5.000025       5.0  5.0  5.0          5.0 | 1.0  0.639869  0.033448  0.000261       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.710853 |
    | 2000-01-01 02:45:00 |  29.719586 | 5.721083  5.193631  5.008422  5.000031       5.0  5.0  5.0          5.0 | 1.0  0.658989  0.037543  0.000324       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.721083 |
    | 2000-01-01 02:50:00 |  32.633293 | 5.730764  5.206338  5.009669  5.000039       5.0  5.0  5.0          5.0 | 1.0  0.677287  0.041947  0.000401       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.730764 |
    | 2000-01-01 02:55:00 |  35.400897 | 5.739922    5.2193  5.011054  5.000049       5.0  5.0  5.0          5.0 | 1.0  0.694747  0.046666  0.000496       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.739922 |
    | 2000-01-01 03:00:00 |  38.033405 | 5.748581  5.232493  5.012587  5.000061       5.0  5.0  5.0          5.0 | 1.0  0.711357  0.051708  0.000609       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.748581 |
    | 2000-01-01 03:05:00 |  40.540776 | 5.756768  5.245893  5.014277  5.000076       5.0  5.0  5.0          5.0 | 1.0   0.72711  0.057081  0.000744       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.756768 |
    | 2000-01-01 03:10:00 |  42.932048 | 5.764508  5.259478  5.016134  5.000094       5.0  5.0  5.0          5.0 | 1.0     0.742  0.062792  0.000903       0.0       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.764508 |
    | 2000-01-01 03:15:00 |  45.215431 | 5.771827  5.273221  5.018166  5.000116       5.0  5.0  5.0          5.0 | 1.0  0.756027  0.068847  0.001089  0.000001       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.771827 |
    | 2000-01-01 03:20:00 |  47.398398 | 5.778751    5.2871  5.020385  5.000142  5.000001  5.0  5.0          5.0 | 1.0  0.769195  0.075255  0.001306  0.000002       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.778751 |
    | 2000-01-01 03:25:00 |  49.487763 | 5.785306   5.30109  5.022799  5.000173  5.000001  5.0  5.0          5.0 | 1.0   0.78151  0.082021  0.001558  0.000003       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.785306 |
    | 2000-01-01 03:30:00 |  51.489743 | 5.791516  5.315166  5.025418   5.00021  5.000001  5.0  5.0          5.0 | 1.0  0.792985  0.089151   0.00185  0.000005       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.791516 |
    | 2000-01-01 03:35:00 |  53.410021 | 5.797407  5.329306  5.028252  5.000254  5.000001  5.0  5.0          5.0 | 1.0  0.803632  0.096651  0.002187  0.000007       0.0  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.797407 |
    | 2000-01-01 03:40:00 |  55.253791 | 5.803003  5.343485   5.03131  5.000305  5.000001  5.0  5.0          5.0 | 1.0  0.813469  0.104526  0.002574   0.00001  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.803003 |
    | 2000-01-01 03:45:00 |  57.025811 | 5.808328   5.35768  5.034603  5.000365  5.000002  5.0  5.0          5.0 | 1.0  0.822518  0.112778  0.003018  0.000012  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.808328 |
    | 2000-01-01 03:50:00 |  58.730434 | 5.813404  5.371867   5.03814  5.000435  5.000002  5.0  5.0          5.0 | 1.0    0.8308  0.121411  0.003525  0.000014  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.813404 |
    | 2000-01-01 03:55:00 |   60.37165 | 5.818254  5.386026  5.041929  5.000517  5.000002  5.0  5.0          5.0 | 1.0  0.838341  0.130427  0.004102  0.000015  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.818254 |
    | 2000-01-01 04:00:00 |  61.953117 | 5.822899  5.400133  5.045981  5.000612  5.000003  5.0  5.0          5.0 | 1.0  0.845171  0.139825  0.004757  0.000017  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.822899 |
    | 2000-01-01 04:05:00 |  63.478186 | 5.827359  5.414167  5.050305  5.000722  5.000003  5.0  5.0          5.0 | 1.0   0.85132  0.149605  0.005497  0.000018  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.827359 |
    | 2000-01-01 04:10:00 |  64.949931 | 5.831654  5.428108  5.054908  5.000848  5.000004  5.0  5.0          5.0 | 1.0  0.856819  0.159765  0.006332   0.00002  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.831654 |
    | 2000-01-01 04:15:00 |  66.371171 | 5.835803  5.441936  5.059799  5.000993  5.000005  5.0  5.0          5.0 | 1.0  0.861704  0.170302  0.007271  0.000021  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.835803 |
    | 2000-01-01 04:20:00 |  67.744489 | 5.839823  5.455632  5.064985  5.001159  5.000005  5.0  5.0          5.0 | 1.0   0.86601   0.18121  0.008323  0.000023  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.839823 |
    | 2000-01-01 04:25:00 |  69.072253 |  5.84373  5.469178  5.070475  5.001348  5.000006  5.0  5.0          5.0 | 1.0  0.869773  0.192483  0.009501  0.000025  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |    5.84373 |
    | 2000-01-01 04:30:00 |  70.356636 | 5.847539  5.482556  5.076274  5.001564  5.000007  5.0  5.0          5.0 | 1.0  0.873033  0.204113  0.010816  0.000027  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.847539 |
    | 2000-01-01 04:35:00 |  71.599629 | 5.851264  5.495751  5.082388  5.001809  5.000008  5.0  5.0          5.0 | 1.0  0.875828   0.21609  0.012279   0.00003  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.851264 |
    | 2000-01-01 04:40:00 |  72.803055 | 5.854918  5.508747  5.088823  5.002086  5.000009  5.0  5.0          5.0 | 1.0  0.878198  0.228403  0.013905  0.000034  0.000001  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.854918 |
    | 2000-01-01 04:45:00 |  73.968589 | 5.858513   5.52153  5.095583    5.0024   5.00001  5.0  5.0          5.0 | 1.0  0.880181   0.24104  0.015706  0.000039  0.000002  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.858513 |
    | 2000-01-01 04:50:00 |   0.314461 | 5.862061  5.534083  5.102672  5.002753  5.000011  5.0  5.0          5.0 | 1.0  0.881725  0.254045  0.017729  0.000048  0.000002  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.862061 |
    | 2000-01-01 04:55:00 |   1.774741 | 5.865742  5.546294  5.110091  5.003139  5.000016  5.0  5.0          5.0 | 1.0  0.877277  0.266755  0.019464  0.000179  0.000006  0.0  0.0         0.0 |    1.0 |     0.0 |     0.0 |   5.865742 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_channel_internal_negative_volumes:

Internal negative volumes
_________________________

The internal water movement within the channel could result in undershooting a water
depth of zero for individual segments, caused, for example, by strongly varying bottom
levels.  We create such a potentially problematic case by increasing the height of the
first, the fourth, and the last segment by one metre:

>>> for i in [0, 3, 7]:
...     model.storagemodels[i].crosssection.parameters.control.bottomlevels(6.0)

We set all inflow to zero and the initial water depth to only 0.1 m:

>>> inflow.sequences.sim.series = 0.0
>>> prepare_inits(hs=0.1, qs=0.0)

The selected submodels decide how they handle potentially negative water depths.
|sw1d_lias| relies on method |Update_Discharge_V1| to prevent or at least limit taking
too much water from a channel segment, and |sw1d_storage| uses the sub-submodel
|wq_trapeze|, whose method |wq_model.Calc_WaterDepth_V2| determines a depth of zero
even for negative volumes.  So, the following results look impeccable:

.. integration-test::

    >>> conditions = test("sw1d_channel_internal_negative_volumes", get_conditions="2000-01-01 00:00")
    |                date |  timestep |                                                                       waterlevels |                                                                                 discharges | inflow | latflow | outflow | waterlevel |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 70.563903 | 6.063082  5.126061  5.122812  6.066717  5.123866  5.101405  5.123011     6.083969 | 0.0  1.230604  -0.072463  -0.832849    0.8313   0.035754  -0.034515  -0.801558         0.0 |    0.0 |     0.0 |     0.0 |   6.063082 |
    | 2000-01-01 00:05:00 | 68.582282 | 6.025784  5.153196  5.144745   6.03295  5.147203  5.104013  5.145753     6.067518 | 0.0  1.243268   -0.11345   -0.84456    0.8438   0.065897  -0.064486  -0.822532         0.0 |    0.0 |     0.0 |     0.0 |   6.025784 |
    | 2000-01-01 00:10:00 | 66.644541 |      6.0  5.172879  5.165554  6.000223  5.169486  5.107009  5.168131     6.051108 | 0.0  0.859461  -0.124696  -0.818336  0.817983   0.075228  -0.074554  -0.820507         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 00:15:00 | 65.841766 |      6.0   5.17366  5.164718       6.0  5.168364  5.109666  5.190113     6.034768 | 0.0       0.0  -0.039043  -0.011171  0.011171    0.04858  -0.084281  -0.817011         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 00:20:00 | 65.874117 |      6.0  5.173702  5.164654       6.0  5.167353  5.112216   5.21166     6.018527 | 0.0       0.0  -0.002126        0.0       0.0   0.033695  -0.093801  -0.812027         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 00:25:00 | 65.873517 |      6.0  5.173595  5.164815       6.0  5.166331  5.114997  5.232687      6.00241 | 0.0       0.0   0.005372        0.0       0.0   0.034063  -0.104975  -0.805867         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 00:30:00 | 65.864957 |      6.0  5.173448  5.165036       6.0  5.165334  5.117328  5.233801          6.0 | 0.0       0.0   0.007354        0.0       0.0   0.033234  -0.083346  -0.120498         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 00:35:00 |   65.8545 |      6.0  5.173296  5.165264       6.0  5.164473  5.119041  5.232093          6.0 | 0.0       0.0   0.007598        0.0       0.0   0.028688  -0.056956        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 00:40:00 | 65.843908 |      6.0  5.173147  5.165487       6.0  5.163643  5.120705  5.230427          6.0 | 0.0       0.0   0.007438        0.0       0.0    0.02769  -0.055517        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 00:45:00 | 65.833584 |      6.0  5.173003  5.165703       6.0  5.162833  5.122333  5.228795          6.0 | 0.0       0.0   0.007193        0.0       0.0   0.027001    -0.0544        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 00:50:00 | 65.823611 |      6.0  5.172864  5.165911       6.0  5.162043  5.123926  5.227196          6.0 | 0.0       0.0   0.006933        0.0       0.0   0.026316  -0.053302        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 00:55:00 | 65.814001 |      6.0  5.172731  5.166111       6.0  5.161274  5.125482   5.22563          6.0 | 0.0       0.0   0.006674        0.0       0.0   0.025629  -0.052216        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:00:00 |  65.80475 |      6.0  5.172603  5.166304       6.0  5.160526  5.127004  5.224095          6.0 | 0.0       0.0   0.006419        0.0       0.0   0.024939  -0.051143        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:05:00 | 65.795854 |      6.0  5.172479  5.166489       6.0  5.159799  5.128491  5.222593          6.0 | 0.0       0.0    0.00617        0.0       0.0   0.024246  -0.050082        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:10:00 | 65.787303 |      6.0  5.172361  5.166666       6.0  5.159092  5.129942  5.221122          6.0 | 0.0       0.0   0.005926        0.0       0.0   0.023551  -0.049033        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:15:00 |  65.77909 |      6.0  5.172247  5.166837       6.0  5.158407  5.131359  5.219682          6.0 | 0.0       0.0   0.005689        0.0       0.0   0.022853  -0.047996        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:20:00 | 65.771207 |      6.0  5.172138  5.167001       6.0  5.157742  5.132742  5.218273          6.0 | 0.0       0.0   0.005457        0.0       0.0   0.022152   -0.04697        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:25:00 | 65.763646 |      6.0  5.172033  5.167158       6.0  5.157099   5.13409  5.216894          6.0 | 0.0       0.0   0.005231        0.0       0.0   0.021449  -0.045956        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:30:00 | 65.756398 |      6.0  5.171933  5.167308       6.0  5.156476  5.135404  5.215546          6.0 | 0.0       0.0   0.005011        0.0       0.0   0.020743  -0.044953        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:35:00 | 65.749456 |      6.0  5.171837  5.167452       6.0  5.155875  5.136684  5.214227          6.0 | 0.0       0.0   0.004797        0.0       0.0   0.020035  -0.043961        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:40:00 | 65.742811 |      6.0  5.171745   5.16759       6.0  5.155296   5.13793  5.212937          6.0 | 0.0       0.0   0.004589        0.0       0.0   0.019324  -0.042981        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:45:00 | 65.736455 |      6.0  5.171657  5.167721       6.0  5.154737  5.139142  5.211677          6.0 | 0.0       0.0   0.004387        0.0       0.0    0.01861  -0.042012        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:50:00 | 65.730378 |      6.0  5.171574  5.167847       6.0  5.154201  5.140321  5.210445          6.0 | 0.0       0.0   0.004192        0.0       0.0   0.017894  -0.041054        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 01:55:00 | 65.724574 |      6.0  5.171494  5.167967       6.0  5.153685  5.141467  5.209242          6.0 | 0.0       0.0   0.004002        0.0       0.0   0.017175  -0.040106        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:00:00 | 65.719032 |      6.0  5.171417  5.168082       6.0  5.153192  5.142579  5.208067          6.0 | 0.0       0.0   0.003818        0.0       0.0   0.016453   -0.03917        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:05:00 | 65.713746 |      6.0  5.171344  5.168191       6.0   5.15272  5.143659   5.20692          6.0 | 0.0       0.0    0.00364        0.0       0.0   0.015728  -0.038244        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:10:00 | 65.708707 |      6.0  5.171275  5.168295       6.0   5.15227  5.144705    5.2058          6.0 | 0.0       0.0   0.003468        0.0       0.0      0.015  -0.037329        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:15:00 | 65.703905 |      6.0  5.171209  5.168394       6.0  5.151842  5.145719  5.204707          6.0 | 0.0       0.0   0.003303        0.0       0.0    0.01427  -0.036425        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:20:00 | 65.699334 |      6.0  5.171146  5.168488       6.0  5.151436    5.1467  5.203641          6.0 | 0.0       0.0   0.003143        0.0       0.0   0.013536  -0.035531        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:25:00 | 65.694985 |      6.0  5.171086  5.168578       6.0  5.151052  5.147649  5.202602          6.0 | 0.0       0.0   0.002989        0.0       0.0   0.012799  -0.034648        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:30:00 | 65.690849 |      6.0   5.17103  5.168663       6.0   5.15069  5.148566  5.201588          6.0 | 0.0       0.0    0.00284        0.0       0.0   0.012059  -0.033776        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:35:00 | 65.686919 |      6.0  5.170976  5.168744       6.0  5.150351  5.149451  5.200601          6.0 | 0.0       0.0   0.002697        0.0       0.0   0.011316  -0.032915        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:40:00 | 65.683187 |      6.0  5.170924  5.168821       6.0  5.150033  5.150303  5.199639          6.0 | 0.0       0.0    0.00256        0.0       0.0   0.010569  -0.032064        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:45:00 | 65.679645 |      6.0  5.170876  5.168894       6.0  5.149739  5.151124  5.198702          6.0 | 0.0       0.0   0.002429        0.0       0.0   0.009818  -0.031223        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:50:00 | 65.676286 |      6.0   5.17083  5.168963       6.0  5.149467  5.151913  5.197791          6.0 | 0.0       0.0   0.002302        0.0       0.0   0.009064  -0.030394        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 02:55:00 | 65.673102 |      6.0  5.170786  5.169028       6.0  5.149218  5.152671  5.196903          6.0 | 0.0       0.0   0.002182        0.0       0.0   0.008305  -0.029575        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:00:00 | 65.670085 |      6.0  5.170745   5.16909       6.0  5.148992  5.153397   5.19604          6.0 | 0.0       0.0   0.002066        0.0       0.0   0.007542  -0.028767        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:05:00 | 65.667229 |      6.0  5.170706  5.169149       6.0  5.148788  5.154092  5.195201          6.0 | 0.0       0.0   0.001955        0.0       0.0   0.006775   -0.02797        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:10:00 | 65.664526 |      6.0  5.170669  5.169204       6.0  5.148608  5.154756  5.194386          6.0 | 0.0       0.0   0.001849        0.0       0.0   0.006002  -0.027183        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:15:00 | 65.661969 |      6.0  5.170634  5.169257       6.0  5.148452  5.155388  5.193593          6.0 | 0.0       0.0   0.001748        0.0       0.0   0.005224  -0.026408        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:20:00 | 65.659553 |      6.0  5.170601  5.169306       6.0  5.148318   5.15599  5.192824          6.0 | 0.0       0.0   0.001652        0.0       0.0   0.004441  -0.025644        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:25:00 |  65.65727 |      6.0   5.17057  5.169353       6.0  5.148209  5.156561  5.192077          6.0 | 0.0       0.0    0.00156        0.0       0.0   0.003651   -0.02489        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:30:00 | 65.655114 |      6.0   5.17054  5.169397       6.0  5.148123  5.157101  5.191353          6.0 | 0.0       0.0   0.001472        0.0       0.0   0.002854  -0.024148        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:35:00 | 65.653079 |      6.0  5.170512  5.169439       6.0  5.148062   5.15761   5.19065          6.0 | 0.0       0.0   0.001389        0.0       0.0   0.002049  -0.023418        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:40:00 |  65.65116 |      6.0  5.170486  5.169478       6.0  5.148025  5.158089   5.18997          6.0 | 0.0       0.0    0.00131        0.0       0.0   0.001236  -0.022698        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:45:00 |  65.64935 |      6.0  5.170461  5.169515       6.0  5.148012  5.158537   5.18931          6.0 | 0.0       0.0   0.001235        0.0       0.0   0.000414   -0.02199        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:50:00 | 65.647644 |      6.0  5.170438   5.16955       6.0  5.148176   5.15886  5.188661          6.0 | 0.0       0.0   0.001164        0.0       0.0  -0.005471  -0.021631        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 03:55:00 | 65.646036 |      6.0  5.170416  5.169583       6.0   5.14854  5.159068  5.187986          6.0 | 0.0       0.0   0.001096        0.0       0.0  -0.012135  -0.022501        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:00:00 | 65.644522 |      6.0  5.170396  5.169614       6.0  5.148932  5.159258  5.187309          6.0 | 0.0       0.0   0.001032        0.0       0.0  -0.013052  -0.022566        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:05:00 | 65.643097 |      6.0  5.170376  5.169643       6.0  5.149323  5.159441  5.186643          6.0 | 0.0       0.0   0.000971        0.0       0.0   -0.01305  -0.022199        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:10:00 | 65.641757 |      6.0  5.170358  5.169671       6.0  5.149711  5.159618   5.18599          6.0 | 0.0       0.0   0.000913        0.0       0.0  -0.012912   -0.02175        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:15:00 | 65.640495 |      6.0  5.170341  5.169696       6.0  5.150093  5.159788  5.185352          6.0 | 0.0       0.0   0.000859        0.0       0.0  -0.012752  -0.021295        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:20:00 | 65.639309 |      6.0  5.170325  5.169721       6.0  5.150471  5.159954  5.184726          6.0 | 0.0       0.0   0.000808        0.0       0.0  -0.012586  -0.020843        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:25:00 | 65.638194 |      6.0  5.170309  5.169743       6.0  5.150844  5.160113  5.184114          6.0 | 0.0       0.0   0.000759        0.0       0.0  -0.012418  -0.020398        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:30:00 | 65.637146 |      6.0  5.170295  5.169765       6.0  5.151211  5.160267  5.183516          6.0 | 0.0       0.0   0.000713        0.0       0.0  -0.012248   -0.01996        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:35:00 | 65.636162 |      6.0  5.170282  5.169785       6.0  5.151573  5.160416   5.18293          6.0 | 0.0       0.0    0.00067        0.0       0.0  -0.012076  -0.019528        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:40:00 | 65.635238 |      6.0  5.170269  5.169804       6.0   5.15193   5.16056  5.182357          6.0 | 0.0       0.0   0.000629        0.0       0.0  -0.011903  -0.019102        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:45:00 |  65.63437 |      6.0  5.170257  5.169821       6.0  5.152282    5.1607  5.181796          6.0 | 0.0       0.0    0.00059        0.0       0.0  -0.011727  -0.018682        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:50:00 | 65.633555 |      6.0  5.170246  5.169838       6.0  5.152629  5.160834  5.181248          6.0 | 0.0       0.0   0.000554        0.0       0.0  -0.011551  -0.018269        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |
    | 2000-01-01 04:55:00 |  65.63279 |      6.0  5.170236  5.169854       6.0   5.15297  5.160964  5.180712          6.0 | 0.0       0.0    0.00052        0.0       0.0  -0.011372  -0.017862        0.0         0.0 |    0.0 |     0.0 |     0.0 |        6.0 |

Also, there is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

However, looking into the details, one finds the final water volumes of the third
segment to be slightly negative:

>>> from hydpy import print_vector
>>> print_vector(s.sequences.states.watervolume.value for s in model.storagemodels)
0.0, 2.553539, 1.698536, -0.003351, 1.529698, 2.414455, 1.807123, 0.0

This deficit is due to the limitation of method |Update_Discharge_V1| only to consider
the water loss caused by one routing model and not all involved routing models.
Implementing a safer approach would likely increase computation time and complexity for
little benefit.  Hence, we keep it that way until applications suggest an improvement
is necessary.

.. _sw1d_channel_excessive_water_withdrawal:

Excessive water withdrawal
__________________________

More significant deficits can result from misleading inflow and outflow data used as
external forcing.  To apply such forcings in all possible ways, we replace the
|sw1d_weir_out| submodel at the channel's outlet with a |sw1d_q_out| submodel.  The
:ref:`sw1d_network_bifurcations` example of |sw1d_network| explains the following
necessary steps:

>>> channel.model.storagemodels[-1].routingmodelsdownstream.number = 0
>>> channel.model.routingmodels[-2].routingmodelsdownstream.number = 0

>>> from hydpy.models import sw1d_q_out
>>> with channel.model.add_routingmodel_v3(sw1d_q_out, position=8):
...     lengthupstream(2.0)
...     timestepfactor(0.7)
...     with model.add_crosssection_v2("wq_trapeze"):
...         nmbtrapezes(1)
...         bottomlevels(5.0)
...         bottomwidths(5.0)
...         sideslopes(0.0)
>>> channel.model.connect()

>>> test = IntegrationTest()

>>> outflow.deploymode = "oldsim_bi"

We reset all bottom levels to 5 m for simplicity:

>>> for i in [0, 3, 7]:
...     model.storagemodels[i].crosssection.parameters.control.bottomlevels(5.0)

The simulation starts with a consistent water depth of a half metre:

>>> prepare_inits(hs=0.5, qs=0.0)

The longitudinal inflow and the lateral flow are each -0.5 m³/s (subtracting 1 m³/s
from the first segment), and the longitudinal outflow is 1 m³/s (removing this amount
of water from the last segment):

>>> inflow.sequences.sim.series = -0.5
>>> latflow.sequences.sim.series = -0.5
>>> outflow.sequences.sim.series = 1.0

Despite the symmetry of the external forcing, the first segment's water depth reaches
zero a little earlier due to its smaller lengths and the correspondingly smaller
initial water volume:

.. integration-test::

    >>> conditions = test("sw1d_channel_excessive_water_withdrawal", get_conditions="2000-01-01 00:00")
    |                date |  timestep |                                                                       waterlevels |                                                                                 discharges | inflow | latflow | outflow | waterlevel |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 29.085815 | 5.471039  5.499322  5.499978       5.5       5.5  5.499977  5.498408     5.481084 | -0.5  -0.034635   -0.00072        0.0        0.0       0.0  0.001152  0.054224         1.0 |   -0.5 |    -0.5 |     1.0 |   5.471039 |
    | 2000-01-01 00:05:00 | 29.081855 | 5.446176  5.496526  5.499136  5.499922  5.499855  5.499095  5.492599     5.465946 | -0.5  -0.171228  -0.031463  -0.003392   0.000509  0.005351  0.049452  0.243094         1.0 |   -0.5 |    -0.5 |     1.0 |   5.446176 |
    | 2000-01-01 00:10:00 |  29.01749 | 5.422669  5.493614  5.497568  5.499408   5.49929  5.497626  5.487339      5.45144 | -0.5  -0.216446  -0.070849  -0.018586   0.007125  0.025939  0.099382    0.2747         1.0 |   -0.5 |    -0.5 |     1.0 |   5.422669 |
    | 2000-01-01 00:15:00 | 28.835099 | 5.399744  5.490771  5.495883   5.49824  5.498506   5.49611  5.482501     5.437124 | -0.5  -0.235826  -0.093629  -0.037444   0.020963  0.047097  0.122905  0.284185         1.0 |   -0.5 |    -0.5 |     1.0 |   5.399744 |
    | 2000-01-01 00:20:00 | 28.556319 | 5.377164  5.487961  5.494251  5.496495  5.497672  5.494644  5.477879     5.422923 | -0.5  -0.247332  -0.106857   -0.05248   0.034775  0.062594  0.135897  0.289937         1.0 |   -0.5 |    -0.5 |     1.0 |   5.377164 |
    | 2000-01-01 00:25:00 | 28.213481 | 5.354797  5.485177  5.492651  5.494377  5.496801  5.493209  5.473399     5.408805 | -0.5  -0.254446  -0.115233  -0.061888   0.043987  0.073023  0.144755  0.294102         1.0 |   -0.5 |    -0.5 |     1.0 |   5.354797 |
    | 2000-01-01 00:30:00 |  27.82749 | 5.332549  5.482421   5.49103   5.49208  5.495858  5.491775  5.469026     5.394749 | -0.5  -0.258385  -0.120606  -0.066562   0.048305  0.079731  0.151437  0.297207         1.0 |   -0.5 |    -0.5 |     1.0 |   5.332549 |
    | 2000-01-01 00:35:00 | 27.420133 | 5.310343  5.479704  5.489355  5.489736  5.494816  5.490319  5.464743     5.380738 | -0.5  -0.259788  -0.123903  -0.068066   0.049131  0.083876  0.156694  0.299453         1.0 |   -0.5 |    -0.5 |     1.0 |   5.310343 |
    | 2000-01-01 00:40:00 | 27.067062 | 5.288115  5.477037  5.487615  5.487422  5.493663  5.488827  5.460542     5.366757 | -0.5  -0.259073   -0.12573  -0.067745   0.047929   0.08635  0.160926  0.300972         1.0 |   -0.5 |    -0.5 |     1.0 |   5.288115 |
    | 2000-01-01 00:45:00 | 26.702544 | 5.265811  5.474436  5.485816  5.485177  5.492402  5.487295  5.456418     5.352794 | -0.5  -0.256536  -0.126486  -0.066512   0.045748  0.087774  0.164383  0.301825         1.0 |   -0.5 |    -0.5 |     1.0 |   5.265811 |
    | 2000-01-01 00:50:00 | 26.314234 | 5.243383  5.471917   5.48397  5.483014  5.491043  5.485721  5.452373     5.338836 | -0.5  -0.252395   -0.12644  -0.064912   0.043242  0.088555  0.167235   0.30208         1.0 |   -0.5 |    -0.5 |     1.0 |   5.243383 |
    | 2000-01-01 00:55:00 | 25.903677 | 5.220787  5.469495  5.482095  5.480933  5.489598  5.484109  5.448407     5.324871 | -0.5  -0.246821   -0.12576  -0.063244   0.040782   0.08895  0.169597  0.301792         1.0 |   -0.5 |    -0.5 |     1.0 |   5.220787 |
    | 2000-01-01 01:00:00 | 25.473148 | 5.197986  5.467187  5.480207  5.478929  5.488081   5.48246  5.444524     5.310892 | -0.5  -0.239946  -0.124552  -0.061643   0.038559  0.089116  0.171549  0.301005         1.0 |   -0.5 |    -0.5 |     1.0 |   5.197986 |
    | 2000-01-01 01:05:00 | 25.025176 | 5.174942  5.465007  5.478325  5.476993  5.486506   5.48078  5.440725     5.296887 | -0.5  -0.231882  -0.122881  -0.060151   0.036648  0.089148  0.173146  0.299759         1.0 |   -0.5 |    -0.5 |     1.0 |   5.174942 |
    | 2000-01-01 01:10:00 |  24.56226 | 5.151624  5.462969  5.476465  5.475117  5.484885  5.479073  5.437016     5.282848 | -0.5  -0.222723  -0.120782  -0.058756   0.035058  0.089101  0.174432  0.298086         1.0 |   -0.5 |    -0.5 |     1.0 |   5.151624 |
    | 2000-01-01 01:15:00 | 24.086705 |    5.128  5.461083  5.474639  5.473293  5.483227  5.477345  5.433398     5.268769 | -0.5  -0.212553  -0.118278   -0.05742   0.033764  0.089004  0.175437  0.296015         1.0 |   -0.5 |    -0.5 |     1.0 |      5.128 |
    | 2000-01-01 01:20:00 | 23.600553 | 5.104044  5.459362   5.47286  5.471517  5.481543  5.475598  5.429877      5.25464 | -0.5  -0.201451  -0.115376  -0.056092   0.032722  0.088872  0.176188  0.293573         1.0 |   -0.5 |    -0.5 |     1.0 |   5.104044 |
    | 2000-01-01 01:25:00 | 23.105559 | 5.079729  5.457813   5.47114  5.469785  5.479838  5.473838  5.426454     5.240456 | -0.5  -0.189495   -0.11208  -0.054721   0.031883   0.08871  0.176706  0.290782         1.0 |   -0.5 |    -0.5 |     1.0 |   5.079729 |
    | 2000-01-01 01:30:00 | 42.990151 |  5.05506  5.456432  5.469491  5.468068  5.478137  5.472069  5.423046     5.226276 | -0.5  -0.177698  -0.108635  -0.053694   0.032183  0.088903  0.177395  0.291026         1.0 |   -0.5 |    -0.5 |     1.0 |    5.05506 |
    | 2000-01-01 01:35:00 |  29.14383 | 5.030184  5.455138  5.467964  5.466208  5.476515  5.470289  5.419263     5.212414 | -0.5  -0.170819  -0.106093  -0.055189   0.037781  0.091837  0.180807  0.306882         1.0 |   -0.5 |    -0.5 |     1.0 |   5.030184 |
    | 2000-01-01 01:40:00 |  3.040211 |  5.00556  5.453808   5.46666  5.463597  5.475036  5.468417   5.41451     5.199667 | -0.5  -0.179188  -0.112691  -0.069232   0.061327  0.110615  0.204226  0.362656         1.0 |   -0.5 |    -0.5 |     1.0 |    5.00556 |
    | 2000-01-01 01:45:00 | 39.152023 |      5.0  5.455549  5.466636  5.453209   5.47401  5.467807  5.413606     5.186857 | -0.5  -0.168541   -0.25561  -0.254781   0.264631   0.29884  0.329337  0.359488         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 01:50:00 | 20.944995 |      5.0  5.455636  5.466006  5.448272  5.472809  5.466934  5.411512     5.172371 | -0.5  -0.141291  -0.145631  -0.124632   0.122193   0.16222  0.205878  0.275685         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 01:55:00 | 20.560819 |      5.0  5.454978  5.464491    5.4462  5.470887  5.465459  5.408989     5.157766 | -0.5  -0.139054  -0.106188  -0.055702   0.047902  0.111968  0.185703  0.269778         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:00:00 | 19.995923 |      5.0  5.453995  5.462427  5.445649  5.468385  5.463687  5.406433     5.143063 | -0.5  -0.137878  -0.088716  -0.019919   0.007642  0.091043  0.179635  0.264832         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:05:00 | 19.239103 |      5.0  5.452819  5.460267  5.446629  5.464214  5.461841  5.403931     5.128258 | -0.5  -0.136806  -0.077978   -0.00597   -0.05499  0.084054  0.176374  0.259776         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:10:00 | 18.208774 |      5.0  5.451585  5.458523  5.447752  5.459442  5.460069  5.401559      5.11335 | -0.5  -0.135879  -0.074186  -0.016046  -0.072175  0.086872  0.175465  0.254562         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:15:00 | 17.116554 |      5.0  5.450378  5.457064  5.448608  5.454867  5.458253  5.399313     5.098333 | -0.5  -0.135112  -0.074761  -0.026148   -0.06897  0.083543  0.174323  0.249161         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:20:00 | 16.059958 |      5.0   5.44921  5.455752  5.449169   5.45075  5.456346  5.397176     5.083203 | -0.5  -0.134394  -0.076024  -0.032274  -0.060318  0.076916  0.172275  0.243515         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:25:00 | 15.356243 |      5.0  5.448072  5.454508  5.449466  5.447157  5.454344  5.395136     5.067957 | -0.5  -0.133707  -0.076791  -0.035329  -0.050167  0.069612  0.169716  0.237717         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:30:00 | 15.149787 |      5.0   5.44695  5.453293  5.449535  5.444075  5.452261  5.393188     5.052594 | -0.5  -0.133067  -0.076978  -0.036495  -0.039949  0.062773  0.166913   0.23184         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:35:00 | 14.921425 |      5.0  5.445836  5.452088   5.44941  5.441462  5.450118  5.391335      5.03711 | -0.5   -0.13243  -0.076734   -0.03656  -0.030297  0.056808  0.163997  0.225778         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:40:00 | 14.670191 |      5.0  5.444725  5.450882   5.44912  5.439259  5.447934  5.389581       5.0215 | -0.5  -0.131784  -0.076231   -0.03604  -0.021542    0.0519  0.161058  0.219535         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:45:00 |  7.142901 |      5.0   5.44353  5.449671  5.448739  5.437157  5.445831  5.387472     5.006057 | -0.5  -0.136853  -0.077113  -0.036734  -0.017678  0.052369  0.157545  0.227837         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:50:00 |  66.98234 |      5.0  5.444645  5.449367  5.448623  5.424823  5.444859   5.38758          5.0 | -0.5  -0.153398  -0.209143  -0.199021  -0.193242  0.217913  0.266496  0.262895         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 02:55:00 | 14.135988 |      5.0  5.444897  5.449487  5.448093  5.417105  5.443687  5.386986          5.0 | -0.5  -0.135207  -0.147807  -0.151798  -0.125274  0.131972  0.190563  0.210366         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:00:00 | 14.029428 |      5.0  5.444257  5.448929  5.447144  5.414496  5.441537  5.385612          5.0 | -0.5  -0.132237  -0.100238  -0.081633  -0.034211  0.052767   0.16031   0.20611         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:05:00 | 13.749097 |      5.0  5.443307  5.447936  5.445489  5.414914  5.438911  5.383971          5.0 | -0.5  -0.131171  -0.083664  -0.050564   0.032202   0.01825   0.14952  0.204219         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:10:00 | 13.305941 |      5.0  5.442245  5.447061  5.443002  5.416587  5.436425  5.382247          5.0 | -0.5  -0.130321  -0.077196   -0.04804    0.07633  0.020583  0.144916  0.202377         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:15:00 | 12.825124 |      5.0  5.441178  5.446242   5.44039   5.41804  5.434198  5.380561          5.0 | -0.5  -0.129651  -0.076305  -0.048979   0.081595   0.03317    0.1445  0.200706         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:20:00 | 12.482928 |      5.0   5.44012  5.445352  5.437878  5.419127  5.432161  5.378928          5.0 | -0.5  -0.129029  -0.076126  -0.046458   0.079131  0.042878   0.14473  0.199156         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:25:00 | 12.166178 |      5.0   5.43906  5.444357   5.43554  5.419886  5.430255  5.377338          5.0 | -0.5  -0.128399  -0.075392  -0.042254   0.074676  0.049384  0.144656  0.197654         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:30:00 |  11.82704 |      5.0  5.437988  5.443259  5.433395  5.420363  5.428446  5.375782          5.0 | -0.5  -0.127748  -0.074163  -0.037565   0.069689  0.053795  0.144288  0.196171         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:35:00 | 11.465151 |      5.0  5.436899  5.442069   5.43144  5.420598  5.426709  5.374253          5.0 | -0.5  -0.127078  -0.072655  -0.032987   0.064762  0.056933  0.143748  0.194704         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:40:00 | 11.082353 |      5.0  5.435792  5.440803  5.429659  5.420625  5.425032  5.372749          5.0 | -0.5  -0.126393  -0.071025  -0.028819   0.060187  0.059277  0.143118  0.193252         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:45:00 | 10.681468 |      5.0  5.434666  5.439478  5.428033  5.420475  5.423406  5.371268          5.0 | -0.5  -0.125695  -0.069381  -0.025203   0.056106    0.0611  0.142443  0.191816         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:50:00 | 10.265716 |      5.0  5.433522   5.43811  5.426538  5.420175  5.421822  5.369808          5.0 | -0.5  -0.124989  -0.067792  -0.022197   0.052575  0.062566  0.141748  0.190395         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 03:55:00 |   9.83838 |      5.0  5.432362  5.436715   5.42515   5.41975  5.420277   5.36837          5.0 | -0.5  -0.124277  -0.066304  -0.019801   0.049599  0.063776  0.141047  0.188991         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:00:00 |  9.402586 |      5.0   5.43119  5.435306  5.423847   5.41922  5.418766  5.366952          5.0 | -0.5  -0.123561  -0.064946  -0.017982    0.04715  0.064802  0.140346  0.187603         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:05:00 |  8.961168 |      5.0  5.430008  5.433895   5.42261  5.418605  5.417286  5.365555          5.0 | -0.5  -0.122844  -0.063732   -0.01668    0.04518  0.065689  0.139651  0.186233         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:10:00 |  8.516578 |      5.0  5.428819  5.432489  5.421421   5.41792  5.415837  5.364177          5.0 | -0.5  -0.122128  -0.062667  -0.015821   0.043628  0.066469  0.138964  0.184881         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:15:00 |  8.070833 |      5.0  5.427625  5.431097  5.420266  5.417178  5.414414  5.362819          5.0 | -0.5  -0.121414  -0.061743  -0.015323   0.042426  0.067163  0.138287  0.183547         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:20:00 |   7.62551 |      5.0   5.42643  5.429721  5.419134   5.41639  5.413017  5.361481          5.0 | -0.5  -0.120703   -0.06095  -0.015105   0.041506  0.067783  0.137619  0.182232         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:25:00 |  7.181763 |      5.0  5.425236  5.428366  5.418016  5.415564  5.411645  5.360162          5.0 | -0.5  -0.119996  -0.060269   -0.01509   0.040805  0.068335  0.136961  0.180935         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:30:00 |  6.740371 |      5.0  5.424043  5.427032  5.416906  5.414707  5.410295  5.358862          5.0 | -0.5  -0.119294  -0.059684  -0.015213   0.040266  0.068825   0.13631  0.179656         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:35:00 |  6.301794 |      5.0  5.422855  5.425719  5.415801  5.413825  5.408967   5.35758          5.0 | -0.5  -0.118596  -0.059176  -0.015417   0.039843  0.069253  0.135667  0.178395         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:40:00 |  5.866243 |      5.0  5.421672  5.424427  5.414698  5.412921  5.407659  5.356316          5.0 | -0.5  -0.117904  -0.058728  -0.015662   0.039497  0.069621  0.135028  0.177152         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:45:00 |  5.433749 |      5.0  5.420494  5.423155  5.413596  5.411999  5.406369   5.35507          5.0 | -0.5  -0.117217  -0.058326  -0.015915     0.0392  0.069932  0.134395  0.175927         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:50:00 |  5.004213 |      5.0  5.419322  5.421901  5.412494  5.411061  5.405098  5.353841          5.0 | -0.5  -0.116535  -0.057957  -0.016157   0.038934  0.070186  0.133765  0.174717         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |
    | 2000-01-01 04:55:00 |  4.577466 |      5.0  5.418157  5.420663  5.411393   5.41011  5.403843   5.35263          5.0 | -0.5  -0.115857  -0.057613  -0.016376   0.038684  0.070387  0.133137  0.173525         1.0 |   -0.5 |    -0.5 |     1.0 |        5.0 |

Again, there seems to be no error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

However, the first and the last segments "bookmark" the impossible subtractions as
negative water volumes:

>>> from hydpy import print_vector
>>> print_vector((s.sequences.states.watervolume.value for s in model.storagemodels), width=80)
-10.126869, 6.272359, 4.206635, 6.17089, 4.101102, 6.057642, 3.526298, -6.208057

Hence, when applying external time series that extract water from channels, one has to
ensure such problems never emerge to a problematic extent.


.. _sw1d_channel_flood_wave:

Flood wave
__________

All previous examples dealt with constant inflows and outflows, rectangular channel
profiles, and zero channel slopes.  Here, we demonstrate the propagation in a flood
wave through a trapezoidal, inclined channel.

We take the following configuration from the :ref:`musk_mct_base_example` on |musk_mct|
to allow for comparison between both routing approaches.

The simulation period and step size are identical:

>>> pub.timegrids = "2000-01-01", "2000-01-05", "30m"

We also prepare a 50 km long channel and divide it into 50 equally long and shaped
segments.  |musk_mct|, we cannot set the bottom slope directly but must ensure the
individual bottom levels agree with it:

>>> nmbsegments(50)
>>> for i in range(50):
...     with model.add_storagemodel_v1("sw1d_storage", position=i):
...         length(2.0)
...         with model.add_crosssection_v2("wq_trapeze"):
...             nmbtrapezes(1)
...             bottomlevels(-1000.0 * (i * 2.0 + 1.0) * 0.00025)
...             bottomwidths(15.0)
...             sideslopes(5.0)
>>> for i in range(1, 50):
...     with model.add_routingmodel_v2("sw1d_lias", position=i):
...         lengthupstream(2.0)
...         lengthdownstream(2.0)
...         stricklercoefficient(1.0/0.035)
...         timestepfactor(0.7)
...         diffusionfactor(0.2)
...         with model.add_crosssection_v2("wq_trapeze"):
...             nmbtrapezes(1)
...             bottomlevels(-1000.0 * (i * 2.0) * 0.00025)
...             bottomwidths(15.0)
...             sideslopes(5.0)

We use a |sw1d_q_in| submodel to provide the channel inflow:

>>> with model.add_routingmodel_v1("sw1d_q_in", position=0):
...     lengthdownstream(2.0)
...     timestepfactor(0.7)
...     with model.add_crosssection_v2("wq_trapeze"):
...         nmbtrapezes(1)
...         bottomlevels(0.0)
...         bottomwidths(15.0)
...         sideslopes(5.0)

In contrast to |musk_mct|, we also need an explicit assumption for the channel outflow
and select the |sw1d_weir_out| submodel for this purpose.  We set its crest height
equal to the channel bottom, its crest width to 10 m, and its flow coefficient so that
its the weir flow is 100 m³/s for the initial water depth defined below:

>>> with model.add_routingmodel_v3("sw1d_weir_out", position=50):
...     lengthupstream(2.0)
...     crestheight(-1000.0 * (50 * 2.0) * 0.00025)
...     crestwidth(10.0)
...     flowcoefficient(0.472396985)
...     timestepfactor(0.7)

Now we reset the outflow node's |Node.deploymode|, refresh all connections, and prepare
a new |IntegrationTest| instance:

>>> outflow.deploymode = "newsim"
>>> channel.model = model
>>> from hydpy.core.testtools import IntegrationTest
>>> test = IntegrationTest(channel)
>>> test.plotting_options.axis1 = (fluxes.discharges,)

The simulation starts with a base flow of 100 m³/s and a sufficiently consistent water
depth:

>>> prepare_inits(hs=3.717832, qs=100.0)

We calculate the inflow time series as in the :ref:`musk_mct_base_example` on
|musk_mct|.

>>> import numpy
>>> q_base, q_peak, t_peak, beta = 100.0, 900.0, 24.0, 16.0
>>> ts = pub.timegrids.init.to_timepoints()
>>> inflow.sequences.sim.series = q_base + (
...     (q_peak - q_base) * ((ts / t_peak) * numpy.exp(1.0 - ts / t_peak)) ** beta)
>>> latflow.sequences.sim.series = 0.0

There is a good agreement between the results of |musk_mct| and the following results
for most of the given discharge time series.  Striking differences occur only for those
segments in the vicinity of the weir:

.. integration-test::

    >>> conditions = test("sw1d_flood_wave", get_conditions="2000-01-01 00:00")
    |                date |   timestep |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           waterlevels |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         discharges |     inflow | latflow |    outflow | waterlevel |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 177.268435 | 3.467832  2.967832  2.467832  1.967832  1.467832  0.967832  0.467832  -0.032168  -0.532168  -1.032168  -1.532168  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032168  -12.532168  -13.032168  -13.532168  -14.032168  -14.532168  -15.032168  -15.532168  -16.032168  -16.532168  -17.032168  -17.532168  -18.032171  -18.532208  -19.032474  -19.533931  -20.040201  -20.562465   -21.131609 |      100.0   99.999978   99.999966    99.99996   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999958   99.999966  100.000164  100.002461  100.020209  100.122392   100.58776  102.339155  108.049422 |      100.0 |     0.0 | 108.049422 |   3.467832 |
    | 2000-01-01 00:30:00 | 177.268486 | 3.467833  2.967832  2.467832  1.967832  1.467832  0.967832  0.467832  -0.032168  -0.532168  -1.032168  -1.532168  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032168  -12.532168  -13.032168  -13.532168  -14.032168  -14.532168  -15.032168  -15.532168  -16.032168  -16.532169  -17.032176  -17.532206  -18.032336  -18.532828  -19.034477  -19.539459  -20.053178  -20.588374   -21.176244 |      100.0   99.999992   99.999982   99.999972   99.999965    99.99996   99.999958   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999958   99.999969  100.000045  100.000484  100.002686  100.012253   100.04819  100.164265  100.484508  101.234739  102.724388  105.251717 |      100.0 |     0.0 | 105.251717 |   3.467833 |
    | 2000-01-01 01:00:00 | 177.268514 | 3.467833  2.967832  2.467832  1.967832  1.467832  0.967832  0.467832  -0.032168  -0.532168  -1.032168  -1.532168  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032168  -12.532168  -13.032168  -13.532168  -14.032168  -14.532168  -15.032168  -15.532169  -16.032172  -16.532184  -17.032226  -17.532364  -18.032788  -18.533988  -19.037161   -19.54504   -20.06362  -20.606083   -21.204098 |      100.0   99.999996   99.999989   99.999982   99.999975   99.999968   99.999964    99.99996   99.999958   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999959    99.99997  100.000021  100.000236  100.001069  100.003973  100.013145  100.039304  100.106561  100.262085  100.585025  101.187384  102.201267  103.767276 |      100.0 |     0.0 | 103.767276 |   3.467833 |
    | 2000-01-01 01:30:00 | 177.268531 | 3.467833  2.967833  2.467832  1.967832  1.467832  0.967832  0.467832  -0.032168  -0.532168  -1.032168  -1.532168  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032168  -12.532168  -13.032168  -13.532168  -14.032168  -14.532168  -15.032169  -15.532173  -16.032185  -16.532223  -17.032332   -17.53263  -18.033402    -18.5353  -19.039752  -19.549764  -20.071596  -20.618668   -21.223159 |      100.0   99.999997   99.999993   99.999988   99.999982   99.999976    99.99997   99.999966   99.999962    99.99996   99.999958   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999958   99.999963   99.999984  100.000057  100.000306  100.001088  100.003364  100.009511  100.024889  100.060473  100.136536  100.286579  100.559645  101.018948  101.737382  102.804006 |      100.0 |     0.0 | 102.804006 |   3.467833 |
    | 2000-01-01 02:00:00 | 177.268542 | 3.467833  2.967833  2.467833  1.967832  1.467832  0.967832  0.467832  -0.032168  -0.532168  -1.032168  -1.532168  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032168  -12.532168  -13.032168  -13.532168  -14.032168   -14.53217  -15.032173  -15.532183  -16.032211  -16.532286  -17.032479   -17.53295  -18.034056  -18.536558  -19.042016  -19.553589  -20.077684  -20.627883   -21.236821 |      100.0   99.999998   99.999995   99.999991   99.999987   99.999982   99.999976   99.999971   99.999967   99.999964   99.999961   99.999959   99.999958   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999959   99.999966   99.999988  100.000057  100.000261  100.000829  100.002316  100.005985  100.014495  100.033045  100.071001  100.143835  100.274834  100.495737  100.845786  101.370683  102.132699 |      100.0 |     0.0 | 102.132699 |   3.467833 |
    | 2000-01-01 02:30:00 |  177.26855 | 3.467833  2.967833  2.467833  1.967833  1.467832  0.967832  0.467832  -0.032168  -0.532168  -1.032168  -1.532168  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032168  -12.532168  -13.032168  -13.532168  -14.032169  -14.532172  -15.032179  -15.532198  -16.032247  -16.532366  -17.032645  -17.533282  -18.034684  -18.537679  -19.043917  -19.556645  -20.082363  -20.634775   -21.246897 |      100.0   99.999999   99.999997   99.999994    99.99999   99.999986   99.999981   99.999977   99.999972   99.999968   99.999965   99.999962    99.99996   99.999959   99.999958   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999959   99.999966   99.999984  100.000037   100.00018  100.000547  100.001445  100.003532  100.008144   100.01782  100.037082  100.073431  100.138407  100.248402  100.424828  100.693634  101.085609  101.646312 |      100.0 |     0.0 | 101.646312 |   3.467833 |
    | 2000-01-01 03:00:00 | 177.268558 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967832  0.467832  -0.032168  -0.532168  -1.032168  -1.532168  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032168  -12.532168  -13.032168  -13.532169  -14.032171  -14.532176  -15.032188  -15.532218   -16.03229  -16.532453  -17.032816  -17.533601  -18.035252  -18.538646  -19.045487  -19.559081   -20.08599  -20.640012   -21.254477 | 100.000003  100.000001   99.999998   99.999996   99.999993   99.999989   99.999985   99.999981   99.999977   99.999973   99.999969   99.999966   99.999963   99.999961    99.99996   99.999958   99.999958   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999959   99.999964   99.999978  100.000014  100.000106  100.000329   100.00085  100.002011  100.004492  100.009561  100.019465   100.03795  100.070885  100.126872  100.217658  100.358177  100.566349   100.86387  101.284945 | 100.000003 |     0.0 | 101.284945 |   3.467833 |
    | 2000-01-01 03:30:00 | 177.268583 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467832  -0.032168  -0.532168  -1.032168  -1.532168  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032168  -12.532168  -13.032169   -13.53217  -14.032173  -14.532181  -15.032199  -15.532241  -16.032336  -16.532542  -17.032981  -17.533893  -18.035751  -18.539464  -19.046772  -19.561024  -20.088823  -20.644042   -21.260264 | 100.000029  100.000015  100.000005   99.999999   99.999996   99.999992   99.999989   99.999985   99.999981   99.999977   99.999973    99.99997   99.999967   99.999964   99.999962    99.99996   99.999959   99.999958   99.999958   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999959   99.999962   99.999971   99.999995  100.000051  100.000182  100.000477  100.001116  100.002448  100.005117  100.010261  100.019789  100.036736  100.065661  100.113016  100.187365  100.299398  100.461884  100.690589  101.011625 | 100.000029 |     0.0 | 101.011625 |   3.467833 |
    | 2000-01-01 04:00:00 | 177.268716 | 3.467834  2.967834  2.467833  1.967833  1.467833  0.967833  0.467833  -0.032168  -0.532168  -1.032168  -1.532168  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032168  -12.532169  -13.032169  -13.532171  -14.032176  -14.532187  -15.032212  -15.532266  -16.032382  -16.532628  -17.033133  -17.534154  -18.036182  -18.540149  -19.047822  -19.562577   -20.09105  -20.647173   -21.264734 | 100.000175  100.000095  100.000045  100.000018  100.000005   99.999997   99.999992   99.999988   99.999985   99.999981   99.999977   99.999974    99.99997   99.999967   99.999965   99.999963   99.999961    99.99996   99.999959   99.999958   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999958    99.99996   99.999966   99.999981  100.000014   100.00009  100.000255  100.000604  100.001319  100.002731  100.005424  100.010386  100.019202  100.034306  100.059239  100.098877  100.159562  100.249104   100.37682   100.55442  100.802087 | 100.000175 |     0.0 | 100.802087 |   3.467834 |
    | 2000-01-01 04:30:00 | 177.269343 | 3.467841  2.967838  2.467835  1.967834  1.467833  0.967833  0.467833  -0.032167  -0.532168  -1.032168  -1.532168  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032168  -12.532169   -13.03217  -13.532173  -14.032179  -14.532194  -15.032225   -15.53229  -16.032427  -16.532707   -17.03327  -17.534382  -18.036548  -18.540719  -19.048678  -19.563823  -20.092813  -20.649627   -21.268217 | 100.000826  100.000465  100.000237  100.000112  100.000049  100.000019  100.000004   99.999995   99.999989   99.999985   99.999981   99.999977   99.999974   99.999971   99.999968   99.999965   99.999963   99.999962    99.99996   99.999959   99.999958   99.999958   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999957   99.999958   99.999959   99.999963   99.999971   99.999991  100.000034  100.000125  100.000316    100.0007   100.00145  100.002869  100.005473  100.010098  100.018048  100.031265  100.052505  100.085487  100.134967  100.206749  100.307754  100.446828  100.639719 | 100.000826 |     0.0 | 100.639719 |   3.467841 |
    | 2000-01-01 05:00:00 | 177.271794 | 3.467864  2.967853  2.467844  1.967838  1.467835  0.967834  0.467833  -0.032167  -0.532167  -1.032167  -1.532168  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032169  -12.532169  -13.032171  -13.532175  -14.032183  -14.532201  -15.032238  -15.532314  -16.032469   -16.53278  -17.033391  -17.534579  -18.036858  -18.541191  -19.049377  -19.564824  -20.094214  -20.651562   -21.270953 | 100.003194  100.001854  100.000991  100.000499  100.000244  100.000116  100.000052   100.00002  100.000002   99.999993   99.999986   99.999982   99.999978   99.999974   99.999971   99.999968   99.999966   99.999964   99.999962   99.999961   99.999959   99.999959   99.999958   99.999958   99.999957   99.999957   99.999957   99.999957   99.999957   99.999958    99.99996   99.999965   99.999977  100.000001  100.000051  100.000155  100.000362  100.000762  100.001514  100.002889  100.005331  100.009541  100.016584  100.028007  100.045966  100.073317  100.113669  100.171392  100.251707  100.361382  100.512801 | 100.003194 |     0.0 | 100.512801 |   3.467864 |
    | 2000-01-01 05:30:00 | 177.279943 | 3.467937  2.967901  2.467872  1.967854  1.467844  0.967838  0.467835  -0.032166  -0.532167  -1.032167  -1.532167  -2.032168  -2.532168  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032169   -12.53217  -13.032172  -13.532177  -14.032187  -14.532207   -15.03225  -15.532336  -16.032508  -16.532845  -17.033498  -17.534748  -18.037119  -18.541582  -19.049947  -19.565633  -20.095335  -20.653099   -21.273117 | 100.010515  100.006285  100.003501   100.00186  100.000967  100.000497  100.000251  100.000122  100.000054  100.000019  100.000001    99.99999   99.999984   99.999979   99.999975   99.999972   99.999969   99.999966   99.999964   99.999963   99.999961    99.99996   99.999959   99.999958   99.999958   99.999957   99.999957   99.999957   99.999958   99.999959   99.999962   99.999968   99.999982   100.00001  100.000066  100.000177  100.000391  100.000792  100.001522  100.002818  100.005058  100.008826  100.014987  100.024777  100.039886  100.062526  100.095462  100.142023  100.206193  100.293206  100.412865 | 100.010515 |     0.0 | 100.412865 |   3.467937 |
    | 2000-01-01 06:00:00 | 177.303555 | 3.468138  2.968036  2.467955  1.967901  1.467869  0.967852  0.467843  -0.032162  -0.532165  -1.032166  -1.532167  -2.032167  -2.532167  -3.032168  -3.532168  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532168  -12.032169   -12.53217  -13.032173  -13.532179   -14.03219  -14.532214  -15.032262  -15.532357  -16.032543  -16.532902   -17.03359  -17.534892  -18.037338  -18.541906  -19.050412  -19.566287  -20.096234  -20.654324   -21.274837 | 100.030315  100.018632  100.010776  100.005997  100.003277  100.001774  100.000948  100.000496  100.000252  100.000123  100.000055  100.000019   99.999999   99.999988   99.999981   99.999976   99.999973    99.99997   99.999967   99.999965   99.999963   99.999962    99.99996   99.999959   99.999959   99.999958   99.999958   99.999958   99.999958    99.99996   99.999963   99.999971   99.999986  100.000017  100.000077  100.000192  100.000405  100.000795  100.001486  100.002683  100.004706  100.008036  100.013377  100.021716  100.034387  100.053112  100.080028  100.117695  100.169184  100.238581  100.333684 | 100.030315 |     0.0 | 100.333684 |   3.468138 |
    | 2000-01-01 06:30:00 | 177.364491 | 3.468636  2.968377  2.468169  1.968027  1.467943  0.967895  0.467867  -0.032148  -0.532157  -1.032162  -1.532165  -2.032166  -2.532167  -3.032167  -3.532167  -4.032168  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532169  -12.032169  -12.532171  -13.032174  -13.532181  -14.032194  -14.532221  -15.032273  -15.532376  -16.032574  -16.532953  -17.033669  -17.535014  -18.037521  -18.542174  -19.050794  -19.566817  -20.096958  -20.655306   -21.276211 | 100.078179  100.049335  100.029542  100.017126  100.009776  100.005528  100.003087  100.001696  100.000915  100.000484  100.000249  100.000122  100.000054  100.000017   99.999997   99.999986   99.999979   99.999974    99.99997   99.999968   99.999965   99.999964   99.999962   99.999961    99.99996   99.999959   99.999959   99.999959   99.999959   99.999961   99.999965   99.999973    99.99999  100.000022  100.000084  100.000199  100.000407  100.000776  100.001418  100.002507   100.00431  100.007226  100.011825  100.018899  100.029503  100.044986  100.067013   100.09757  100.139042  100.194644  100.270609 | 100.078179 |     0.0 | 100.270609 |   3.468636 |
    | 2000-01-01 07:00:00 | 177.506897 | 3.469752  2.969159  2.468673  1.968337  1.468129  0.968006  0.467933   -0.03211  -0.532135  -1.032149  -1.532158  -2.032162  -2.532165  -3.032166  -3.532167  -4.032167  -4.532168  -5.032168  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532169   -12.03217  -12.532171  -13.032175  -13.532183  -14.032197  -14.532226  -15.032283  -15.532393  -16.032601  -16.532996  -17.033737  -17.535117  -18.037674  -18.542395  -19.051106  -19.567249  -20.097543  -20.656097   -21.277314 | 100.183351   100.11864  100.073347  100.044112  100.026169  100.015376  100.008927  100.005107  100.002878  100.001598  100.000872  100.000467  100.000242  100.000119  100.000052  100.000016   99.999995   99.999984   99.999977   99.999972   99.999969   99.999966   99.999964   99.999962   99.999961    99.99996    99.99996   99.999959    99.99996   99.999962   99.999966   99.999975   99.999993  100.000026  100.000087  100.000199  100.000397  100.000743   100.00133  100.002308  100.003901  100.006435  100.010376  100.016358   100.02522  100.038026  100.056079  100.080931  100.114451   100.15918  100.220127 | 100.183351 |     0.0 | 100.220127 |   3.469752 |
    | 2000-01-01 07:30:00 | 177.812302 | 3.472056  2.970803  2.469763  1.969029   1.46856  0.968273  0.468097  -0.032011  -0.532076  -1.032115  -1.532137  -2.032151  -2.532158  -3.032162  -3.532165  -4.032166  -4.532167  -5.032167  -5.532168  -6.032168  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032168  -11.532169   -12.03217  -12.532172  -13.032176  -13.532184  -14.032201  -14.532232  -15.032292  -15.532408  -16.032626  -16.533034  -17.033795  -17.535204  -18.037802  -18.542578  -19.051363  -19.567601  -20.098018  -20.656735   -21.278203 | 100.396214  100.262535  100.167169  100.103938  100.063821  100.038812  100.023328  100.013833  100.008092  100.004671   100.00266  100.001492  100.000823  100.000444  100.000232  100.000114  100.000049  100.000013   99.999993   99.999982   99.999975    99.99997   99.999967   99.999965   99.999963   99.999962   99.999961   99.999961   99.999961   99.999963   99.999968   99.999977   99.999995  100.000028  100.000088  100.000194   100.00038  100.000698  100.001229    100.0021  100.003496  100.005687  100.009051  100.014098  100.021499  100.032095  100.046913  100.067174  100.094351  100.130466  100.179554 | 100.396214 |     0.0 | 100.179554 |   3.472056 |
    | 2000-01-01 08:00:00 | 178.419716 | 3.476483  2.974021  2.471952  1.970461  1.469483  0.968863   0.46847  -0.031778  -0.531932  -1.032027  -1.532085  -2.032119   -2.53214  -3.032152  -3.532159  -4.032163  -4.532165  -5.032166  -5.532167  -6.032167  -6.532168  -7.032168  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032169  -11.532169   -12.03217  -12.532173  -13.032177  -13.532186  -14.032204  -14.532237  -15.032301  -15.532421  -16.032647  -16.533067  -17.033845  -17.535278  -18.037909   -18.54273  -19.051574  -19.567888  -20.098404  -20.657252   -21.278921 | 100.797305   100.54033  100.353591  100.226604  100.143542  100.090055   100.05586  100.034214  100.020692  100.012361  100.007293  100.004247   100.00244  100.001381  100.000768  100.000418  100.000219  100.000108  100.000046  100.000011   99.999991    99.99998   99.999973   99.999969   99.999966   99.999964   99.999963   99.999962   99.999962   99.999964   99.999969   99.999978   99.999996  100.000028  100.000085  100.000186  100.000358  100.000647  100.001122  100.001891  100.003108  100.004994  100.007857  100.012109  100.018287   100.02706  100.039242  100.055798  100.077894  100.107149  100.146825 | 100.797305 |     0.0 | 100.146825 |   3.476483 |
    | 2000-01-01 08:30:00 | 179.549451 |  3.48446  2.979927  2.476068  1.973231  1.471321  0.970073  0.469259   -0.03127   -0.53161  -1.031825  -1.531959  -2.032042  -2.532093  -3.032124  -3.532142  -4.032153  -4.532159  -5.032163  -5.532165  -6.032166  -6.532167  -7.032167  -7.532168  -8.032168  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032169  -11.532169  -12.032171  -12.532173  -13.032178  -13.532188  -14.032206  -14.532241  -15.032308  -15.532433  -16.032665  -16.533095  -17.033887  -17.535339  -18.037998  -18.542855  -19.051747  -19.568124  -20.098719  -20.657672   -21.279504 | 101.507032   101.04341  100.700396   100.46148  100.300736  100.194113  100.123921  100.078172  100.048731  100.030026  100.018285  100.011003  100.006541  100.003839  100.002224  100.001269  100.000711  100.000389  100.000205    100.0001  100.000041  100.000008   99.999989   99.999978   99.999972   99.999968   99.999965   99.999964   99.999964   99.999966    99.99997   99.999979   99.999996  100.000027  100.000081  100.000174  100.000331  100.000592  100.001015  100.001689  100.002745  100.004363  100.006794  100.010373  100.015529  100.022798  100.032828  100.046384  100.064396  100.088164  100.120334 | 101.507032 |     0.0 | 100.120334 |    3.48446 |
    | 2000-01-01 09:00:00 | 181.527241 | 3.498035  2.990161  2.483369  1.978278  1.474762  0.972401  0.470819  -0.030238  -0.530935  -1.031389  -1.531681  -2.031867  -2.531984  -3.032056  -3.532101  -4.032128  -4.532145  -5.032154   -5.53216  -6.032163  -6.532165  -7.032166  -7.532167  -8.032167  -8.532168  -9.032168  -9.532168  -10.032168  -10.532168  -11.032169  -11.532169  -12.032171  -12.532174  -13.032179  -13.532189  -14.032209  -14.532245  -15.032314  -15.532443   -16.03268  -16.533119  -17.033922  -17.535391  -18.038072  -18.542959   -19.05189  -19.568317  -20.098976  -20.658014   -21.279977 | 102.694853  101.904349   101.30907  100.884856   100.59184  100.392107  100.257027  100.166584  100.106759  100.067666  100.042413  100.026287  100.016107  100.009757   100.00584  100.003453  100.002015  100.001158  100.000654  100.000359   100.00019  100.000092  100.000037  100.000005   99.999987   99.999977   99.999971   99.999968   99.999967   99.999968   99.999972    99.99998   99.999996  100.000025  100.000076  100.000161  100.000303  100.000536   100.00091  100.001499   100.00241  100.003795  100.005857  100.008866   100.01317  100.019198  100.027467  100.038588  100.053307  100.072668  100.098827 | 102.694853 |     0.0 | 100.098827 |   3.498035 |
    | 2000-01-01 09:30:00 | 184.802383 | 3.519958  3.006997  2.495655  1.986988  1.480858   0.97663  0.473723  -0.028267  -0.529613  -1.030512  -1.531106  -2.031494  -2.531744  -3.031904  -3.532006  -4.032069  -4.532108  -5.032132  -5.532147  -6.032156  -6.532161  -7.032164  -7.532166  -8.032167  -8.532167  -9.032168  -9.532168  -10.032168  -10.532168  -11.032169   -11.53217  -12.032171  -12.532174   -13.03218  -13.532191  -14.032211  -14.532249   -15.03232  -15.532452  -16.032694  -16.533139  -17.033952  -17.535435  -18.038134  -18.543046  -19.052009  -19.568475  -20.099187  -20.658293   -21.280363 | 104.586356  103.305278  102.323473  101.608219  101.101809  100.747767  100.502272  100.333747  100.219404  100.142718  100.091855  100.058489  100.036843  100.022957  100.014148  100.008622  100.005194  100.003092  100.001816  100.001051  100.000596  100.000329  100.000174  100.000084  100.000032  100.000003   99.999986   99.999977   99.999972   99.999971   99.999974   99.999982   99.999996  100.000023  100.000069  100.000147  100.000275  100.000481   100.00081  100.001322  100.002107  100.003289  100.005035  100.007564  100.011158   100.01616  100.022987  100.032128  100.044181  100.059993  100.081319 | 104.586356 |     0.0 | 100.081319 |   3.519958 |
    | 2000-01-01 10:00:00 | 189.950605 | 3.553705  3.033406  2.515368  2.001308  1.491125  0.983926  0.478853  -0.024701  -0.527163  -1.028846  -1.529985  -2.030748  -2.531253  -3.031585    -3.5318  -4.031938  -4.532026  -5.032081  -5.532115  -6.032136  -6.532149  -7.032157  -7.532161  -8.032164  -8.532166  -9.032167  -9.532167  -10.032168  -10.532168  -11.032169   -11.53217  -12.032171  -12.532174   -13.03218  -13.532192  -14.032213  -14.532252  -15.032325  -15.532459  -16.032706  -16.533156  -17.033978  -17.535472  -18.038186  -18.543118  -19.052106  -19.568606   -20.09936  -20.658522   -21.280679 | 107.466641  105.484251  103.937749   102.78661  101.952207  101.354848   100.93086  100.632954  100.425996  100.283814  100.187176  100.122187  100.078946  100.050483  100.031948  100.020006  100.012396  100.007597  100.004604  100.002757  100.001629  100.000948   100.00054  100.000299  100.000157  100.000075  100.000028  100.000001   99.999986    99.99998   99.999979   99.999984   99.999997  100.000021  100.000063  100.000132  100.000246  100.000429  100.000716   100.00116  100.001834  100.002842  100.004319  100.006444  100.009445  100.013601  100.019244   100.02677  100.036661  100.049602   100.06703 | 107.466641 |     0.0 |  100.06703 |   3.553705 |
    | 2000-01-01 10:30:00 | 197.650209 | 3.603366  3.073048  2.545641   2.02383  1.507658  0.995943  0.487494  -0.018559  -0.522844  -1.025842  -1.527917  -2.029339  -2.530304  -3.030951  -3.531381  -4.031664  -4.531849  -5.031967  -5.532043  -6.032091  -6.532121   -7.03214  -7.532151  -8.032158  -8.532162  -9.032164  -9.532166  -10.032167  -10.532168  -11.032169   -11.53217  -12.032171  -12.532175  -13.032181  -13.532193  -14.032215  -14.532255  -15.032329  -15.532466  -16.032716  -16.533171  -17.033999  -17.535503  -18.038229  -18.543177  -19.052187  -19.568714  -20.099502   -20.65871   -21.280938 | 111.678559  108.738346  106.402474   104.62632  103.309392  102.345127  101.645504  101.143075  100.786246  100.535555  100.361254  100.241305  100.159606  100.104531  100.067784  100.043519   100.02766  100.017402  100.010835  100.006675  100.004067  100.002449  100.001455  100.000851  100.000487   100.00027  100.000142  100.000068  100.000025  100.000002   99.999992   99.999991       100.0   100.00002  100.000057  100.000119  100.000219   100.00038   100.00063  100.001013  100.001591  100.002449  100.003697  100.005483   100.00799  100.011445  100.016116  100.022323  100.030454  100.041068  100.055342 | 111.678559 |     0.0 | 100.055342 |   3.603366 |
    | 2000-01-01 11:00:00 | 208.621317 | 3.673392  3.130131  2.590267  2.057832  1.533198   1.01492  0.501436  -0.008434  -0.515573  -1.020675  -1.524282  -2.026808  -2.528559   -3.02976  -3.530577  -4.031126  -4.531492  -5.031734  -5.531891  -6.031994  -6.532059    -7.0321  -7.532127  -8.032143  -8.532153  -9.032159  -9.532163  -10.032165  -10.532167  -11.032168  -11.532169  -12.032171  -12.532175  -13.032182  -13.532194  -14.032216  -14.532257  -15.032333  -15.532472  -16.032724  -16.533184  -17.034017  -17.535528  -18.038265  -18.543227  -19.052254  -19.568803  -20.099619  -20.658864    -21.28115 | 117.614836  113.422347  110.028046   107.39116  105.392259  103.896612  102.788451  101.976005   101.38691  100.964294  100.664185  100.453193  100.306326  100.205107  100.136036  100.089369   100.05815  100.037472  100.023913   100.01511  100.009451  100.005851  100.003583  100.002168  100.001294   100.00076  100.000437  100.000243  100.000129  100.000063  100.000027  100.000011   100.00001  100.000023  100.000053  100.000106  100.000195  100.000334  100.000551  100.000882  100.001376  100.002105   100.00316   100.00466  100.006756   100.00963  100.013502  100.018627  100.025324  100.034046  100.045761 | 117.614836 |     0.0 | 100.045761 |   3.673392 |
    | 2000-01-01 11:30:00 | 223.525514 | 3.768179   3.20914  2.653558  2.107241  1.571174  1.043759  0.523072   0.007604  -0.503818   -1.01215  -1.518163  -2.022459  -2.525498  -3.027627  -3.529104  -4.030119   -4.53081  -5.031276  -5.531588  -6.031794  -6.531929  -7.032016  -7.532073  -8.032109  -8.532131  -9.032146  -9.532155   -10.03216  -10.532164  -11.032166  -11.532168  -12.032171  -12.532175  -13.032182  -13.532194  -14.032217  -14.532259  -15.032336  -15.532476  -16.032731  -16.533194  -17.034033   -17.53555  -18.038295  -18.543269   -19.05231  -19.568877  -20.099715  -20.658991   -21.281324 | 125.703736  119.941924  115.184156  111.406421  108.478929  106.241605  104.549937  103.284877  102.349388  101.664948  101.169219  100.813681  100.561152  100.383505  100.259725  100.174299  100.115901  100.076361  100.049845  100.032233  100.020647  100.013099  100.008229  100.005117  100.003148  100.001914  100.001148  100.000677  100.000391   100.00022   100.00012  100.000065   100.00004  100.000038  100.000057  100.000099  100.000174  100.000294  100.000481  100.000765  100.001187  100.001806  100.002697  100.003957   100.00571  100.008104  100.011316  100.015555  100.021078  100.028258  100.037892 | 125.703736 |     0.0 | 100.037892 |   3.768179 |
    | 2000-01-01 12:00:00 |  21.864496 | 3.891626  3.314396  2.740031  2.176455   1.62564   1.08604  0.555462   0.032102  -0.485507  -0.998611  -1.508255   -2.01528  -2.520347  -3.023966  -3.526526  -4.028321  -4.529567  -5.030425   -5.53101  -6.031405   -6.53167  -7.031846  -7.531961  -8.032037  -8.532085  -9.032116  -9.532136  -10.032149  -10.532157  -11.032162  -11.532166   -12.03217  -12.532174  -13.032182  -13.532195  -14.032218  -14.532261  -15.032339  -15.532481  -16.032738  -16.533203  -17.034046  -17.535568  -18.038321  -18.543303  -19.052356  -19.568938  -20.099795  -20.659096   -21.281468 | 136.388647  128.735071  122.289016  117.055656  112.909113  109.672389   107.17553  105.271996  103.837576  102.768348  101.979405  101.402933  100.985711  100.686563  100.474052  100.324471  100.220147  100.148053  100.098688  100.065196  100.042685  100.027693  100.017801  100.011336   100.00715  100.004464  100.002758  100.001684  100.001015  100.000603  100.000353  100.000205  100.000122  100.000084  100.000079  100.000103  100.000161  100.000262  100.000421  100.000663  100.001023  100.001548    100.0023  100.003359  100.004826  100.006821  100.009489  100.013001  100.017565  100.023487  100.031409 | 136.388647 |     0.0 | 100.031409 |   3.891626 |
    | 2000-01-01 12:30:00 |  49.453947 | 4.046382  3.449525  2.854004  2.270073  1.701122  1.145977  0.602368   0.068304  -0.457916  -0.977824  -1.492761  -2.003847  -2.511992  -3.017917  -3.522187  -4.025236  -4.527394  -5.028908   -5.52996  -6.030685  -6.531181  -7.031516  -7.531741  -8.031891   -8.53199  -9.032054  -9.532096  -10.032123   -10.53214  -11.032152   -11.53216  -12.032166  -12.532172  -13.032181  -13.532195  -14.032219  -14.532262  -15.032341  -15.532484  -16.032743  -16.533211  -17.034057  -17.535584  -18.038342  -18.543332  -19.052394  -19.568988  -20.099861  -20.659182   -21.281586 | 150.102726  140.268859  131.812811  124.788255  119.094953  114.554808  110.981191  108.203883  106.071862  104.453726  103.238438  102.334698  101.668995  101.183136  100.831722  100.579807  100.400811  100.274744  100.186737  100.125841  100.084076  100.055688  100.036563  100.023795  100.015347  100.009809   100.00621  100.003893  100.002416  100.001483    100.0009  100.000542  100.000327  100.000206  100.000147  100.000137  100.000167  100.000243  100.000373  100.000577  100.000881  100.001325   100.00196   100.00285  100.004078  100.005742  100.007961  100.010873  100.014648  100.019537  100.026071 | 150.102726 |     0.0 | 100.026071 |   4.046382 |
    | 2000-01-01 13:00:00 |  81.972093 |  4.23355  3.616896  2.999014  2.392427  1.802303  1.228236  0.668173   0.120154  -0.417622  -0.946896   -1.46929  -1.986222  -2.498886  -3.008264  -3.515142  -4.020139  -4.523739  -5.026309  -5.528128  -6.029405  -6.530294  -7.030907  -7.531326  -8.031611  -8.531802   -9.03193  -9.532014   -10.03207  -10.532106   -11.03213  -11.532146  -12.032157  -12.532167  -13.032178  -13.532193  -14.032219  -14.532263  -15.032343  -15.532487  -16.032747  -16.533217  -17.034066  -17.535597   -18.03836  -18.543356  -19.052426   -19.56903  -20.099916  -20.659253   -21.281684 | 167.240349  154.991305  144.236421  135.090346  127.502683  121.317466  116.348001  112.410441  109.331612  106.953247  105.136154  103.762025  102.732873  101.969172  101.407485  100.997954  100.701911  100.489719  100.338912  100.232639  100.158383  100.106939  100.071605  100.047543    100.0313   100.02043  100.013218  100.008477  100.005386  100.003391  100.002115  100.001308  100.000805  100.000499  100.000324  100.000237  100.000216  100.000252  100.000346   100.00051  100.000763  100.001136  100.001669  100.002416  100.003444  100.004833  100.006679  100.009096  100.012223  100.016266  100.021667 | 167.240349 |     0.0 | 100.021667 |    4.23355 |
    | 2000-01-01 13:30:00 |  118.68865 | 4.452472  3.817194   3.17722  2.546946  1.933483  1.337563  0.757689   0.192228  -0.360469  -0.902191  -1.434749  -1.959834  -2.478936  -2.993325  -3.504058  -4.011988  -4.517796  -5.022012  -5.525047  -6.027214  -6.528748  -7.029826  -7.530576  -8.031095   -8.53145  -9.031692  -9.531855  -10.031964  -10.532036  -11.032084  -11.532116  -12.032138  -12.532155   -13.03217  -13.532189  -14.032216  -14.532262  -15.032344  -15.532489  -16.032751  -16.533223  -17.034074  -17.535608  -18.038375  -18.543376  -19.052453  -19.569065  -20.099961  -20.659313   -21.281765 | 188.127569  173.306489  160.026633  148.466786  138.646796  130.458405  123.737388  118.303875   113.97443   110.57003  107.924729  105.891479  104.344304  103.178046  102.306753  101.661403  101.187394  100.842092  100.592595  100.413782  100.286668   100.19704   100.13436  100.090886  100.060982  100.040584  100.026787  100.017533  100.011379  100.007321   100.00467  100.002954  100.001856  100.001163  100.000737  100.000488   100.00036  100.000323  100.000362  100.000476  100.000675  100.000982  100.001426   100.00205  100.002909  100.004068  100.005605  100.007613  100.010206  100.013554  100.018027 | 188.127569 |     0.0 | 100.018027 |   4.452472 |
    | 2000-01-01 14:00:00 | 158.535506 | 4.700811  4.049254  3.388977  2.735561  2.097943  1.478215  0.875726   0.289486  -0.281679   -0.83933  -1.385281  -1.921382   -2.44938   -2.97084  -3.487113  -3.999332  -4.508424  -5.015129  -5.520033  -6.023591  -6.526151  -7.027978  -7.529273  -8.030182  -8.530817  -9.031255  -9.531556  -10.031761    -10.5319  -11.031994  -11.532056  -12.032099   -12.53213  -13.032154  -13.532179   -14.03221  -14.532259  -15.032342   -15.53249  -16.032753  -16.533227   -17.03408  -17.535616  -18.038387  -18.543393  -19.052475  -19.569094  -20.099998  -20.659362   -21.281832 | 212.993962  195.536736  179.593642  165.398599  153.052152  142.516283  133.674093  126.373217  120.439555  115.686873  111.929756  108.994627  106.726229   104.99038  103.674222  102.684909   101.94743  101.402095  101.002018  100.710797  100.500459  100.349719  100.242535  100.166917  100.113992  100.077243  100.051931  100.034637  100.022917   100.01504  100.009791  100.006322  100.004052  100.002582  100.001643  100.001058  100.000711  100.000527  100.000464  100.000501  100.000634   100.00087  100.001231  100.001747  100.002462  100.003427  100.004707  100.006376  100.008529  100.011305  100.015014 | 212.993962 |     0.0 | 100.015014 |   4.700811 |
    | 2000-01-01 14:30:00 |   2.770336 | 4.974887  4.310184  3.632663  2.958244  2.297294  1.653267  1.026474   0.416792   -0.17615  -0.753331  -1.316269  -1.866765  -2.406688   -2.93784  -3.461863  -3.980192  -4.494041  -5.004411   -5.51211  -6.017779   -6.52192  -7.024922  -7.527082  -8.028624  -8.529716  -9.030485  -9.531021  -10.031392  -10.531647  -11.031822  -11.531941  -12.032022  -12.532079  -13.032121  -13.532157  -14.032197  -14.532251  -15.032338  -15.532488  -16.032754  -16.533229  -17.034085  -17.535624  -18.038397  -18.543406  -19.052494  -19.569118   -20.10003  -20.659403   -21.281888 | 241.948195  221.886301  203.245141  186.293207  171.207911  158.033896  146.722202   137.17301  129.247337   122.77363  117.562411  113.422313  110.171995  107.647473  105.705751  104.225687  103.107003  102.268227   101.64418    101.1834   100.84571  100.600066  100.422708  100.295607  100.205208  100.141399  100.096703  100.065637  100.044211   100.02955  100.019598  100.012897  100.008424  100.005463  100.003526  100.002275  100.001485  100.001008  100.000748  100.000648  100.000676  100.000823  100.001095  100.001509  100.002095  100.002895  100.003958  100.005346  100.007134  100.009438  100.012516 | 241.948195 |     0.0 | 100.012516 |   4.974887 |
    | 2000-01-01 15:00:00 |  50.426809 | 5.270406  4.595751  3.904762  3.212752  2.530937  1.863869    1.2127   0.578195  -0.039005  -0.638958  -1.222518  -1.791112  -2.346492  -2.890541  -3.425112  -3.951924  -4.472498  -4.988132  -5.499908  -6.008704  -6.515221  -7.020014  -7.523512  -8.026046  -8.527868  -9.029169  -9.530092  -10.030741  -10.531194  -11.031509  -11.531726  -12.031876   -12.53198  -13.032055  -13.532114  -14.032168  -14.532233  -15.032327  -15.532482  -16.032751   -16.53323  -17.034087  -17.535629  -18.038405  -18.543418  -19.052509  -19.569138  -20.100056  -20.659436   -21.281934 |  274.95935  252.392682  231.124381  211.416183  193.500262  177.497995  163.437813  151.292581  140.985237  132.387111  125.329161  119.620115  115.063267   111.46934  108.665127  106.498219  104.838557  103.577826  102.627553  101.916613  101.388581  100.999195  100.714088  100.506814    100.3572  100.249982  100.173702  100.119831  100.082067  100.055792  100.037649  100.025218  100.016766  100.011067  100.007259  100.004739  100.003095  100.002045    100.0014  100.001038  100.000883  100.000893  100.001048  100.001351  100.001814  100.002466  100.003343  100.004493  100.005977  100.007891  100.010439 |  274.95935 |     0.0 | 100.010439 |   5.270406 |
    | 2000-01-01 15:30:00 |   97.54219 | 5.581711  4.900687  4.200481  3.495203  2.796277  2.109017  1.435177   0.776194   0.133688  -0.491297  -1.098627  -1.688977   -2.26363  -2.824271  -3.372782  -3.911067  -4.440919  -4.963948  -5.481542  -5.994865  -6.504872   -7.01233  -7.517847  -8.021899  -8.524853  -9.026993  -9.528532   -10.02963   -10.53041  -11.030958  -11.531343  -12.031611  -12.531798  -13.031931   -13.53203  -14.032113  -14.532196  -15.032303  -15.532467  -16.032742  -16.533226  -17.034087  -17.535631  -18.038411  -18.543426  -19.052521  -19.569155  -20.100077  -20.659464   -21.281972 | 311.845554  287.024037  263.312814  240.961019  220.235433  201.326685  184.339411   169.32486   156.28508  145.161044  135.833995  128.139781  121.886918  116.873811  112.903078  109.791781  107.377381  105.520131  104.102933  103.029652  102.222651  101.620077  101.173212  100.844058  100.603248  100.428268  100.301992  100.211496  100.147096  100.101592   100.06967  100.047439  100.032071  100.021529  100.014354  100.009512  100.006278  100.004145  100.002766  100.001907  100.001411   100.00118  100.001156   100.00131  100.001635  100.002141  100.002848  100.003791  100.005018  100.006603   100.00872 | 311.845554 |     0.0 |  100.00872 |   5.581711 |
    | 2000-01-01 16:00:00 | 143.123291 | 5.903451   5.21937  4.514078   3.80015  3.088651  2.385305  1.692159   1.010946   0.343972  -0.306688  -0.939764  -1.554874  -2.152446  -2.733588  -3.299898  -3.853245   -4.39557  -4.928741  -5.454457  -5.974201  -6.489227  -7.000571   -7.50907  -8.015393  -8.520063  -9.023489  -9.525985  -10.027793  -10.529092  -11.030021   -11.53068  -12.031146  -12.531474  -13.031707  -13.531876  -14.032008  -14.532125  -15.032256  -15.532436  -16.032723  -16.533214  -17.034081   -17.53563  -18.038413  -18.543432   -19.05253  -19.569168  -20.100095  -20.659487   -21.282004 | 352.270855  325.485655  299.601391  274.836684  251.465327  229.724203  209.778806  191.751013  175.726385  161.733568  149.731932  139.616235  131.230187  124.383723  118.871109  114.486978  111.038286  108.351692  106.276951  104.687374  103.478425  102.565322  101.880256   101.36962  100.991449  100.713181  100.509746   100.36199  100.255382  100.178977  100.124591  100.086143  100.059154  100.040342  100.027327  100.018391  100.012306  100.008203   100.00547  100.003684  100.002553  100.001884  100.001551  100.001474  100.001613  100.001949  100.002485  100.003237  100.004236  100.005542  100.007295 | 352.270855 |     0.0 | 100.007295 |   5.903451 |
    | 2000-01-01 16:30:00 |   9.299088 | 6.230527  5.546226   4.83955  4.121385  3.402042  2.687358   1.97943   1.279882   0.591253  -0.083681   -0.74265  -1.384127  -2.007412  -2.612646  -3.200734  -3.773156  -4.331745   -4.87847  -5.415268  -5.943929  -6.466035  -6.982937  -7.495757  -8.005408  -8.512623  -9.017982  -9.521934  -10.024832  -10.526942  -11.028469  -11.529568  -12.030354  -12.530914  -13.031313  -13.531602  -14.031818  -14.531995  -15.032167  -15.532376  -16.032683  -16.533189  -17.034066  -17.535622   -18.03841  -18.543434  -19.052536  -19.569177  -20.100108  -20.659506    -21.28203 | 395.750601   367.34233  339.622041  312.764047  287.026764  262.668142  239.892218  218.871039  199.760387  182.677051  167.674124  154.732912  143.766209  134.628953  127.134761  121.075079  116.237017  112.417219  109.430912  107.116521  105.336892  103.978267  102.947996  102.171713  101.590435  101.157837  100.837852  100.602617  100.430755  100.305981  100.215971  100.151459  100.105526  100.073041  100.050224  100.034311  100.023295   100.01573  100.010583  100.007122  100.004835  100.003367  100.002478  100.002007  100.001854  100.001959  100.002291  100.002843  100.003629  100.004685   100.00612 | 395.750601 |     0.0 |  100.00612 |   6.230527 |
    | 2000-01-01 17:00:00 |   54.27145 | 6.558241  5.875966  5.171009  4.452488  3.729759  3.008558  2.290869   1.577928   0.872061    0.17627  -0.506553  -1.173957  -1.824102  -2.455931   -3.06927  -3.664779  -4.243787  -4.808061  -5.359583   -5.90035  -6.432244   -6.95695   -7.47592   -7.99037  -8.501297  -9.009504  -9.515628  -10.020169  -10.523515  -11.025966   -11.52775  -12.029043  -12.529974  -13.030644  -13.531128  -14.031485  -14.531763  -15.032006  -15.532266  -16.032609  -16.533139  -17.034033  -17.535602  -18.038399  -18.543429  -19.052537  -19.569183  -20.100118   -20.65952   -21.282051 | 441.664962  412.009029   382.84134  354.279651  326.551862  299.912457  274.580667  250.755024  228.636996  208.413708  190.227842  174.159114  160.215625   148.33258  138.379915  130.178015  123.517602  118.179274  113.949481  110.631529  108.051696  106.061426   104.53682  103.376535  102.498944  101.839106  101.345874   100.97931  100.708464  100.509515  100.364247  100.258817  100.182771  100.128264  100.089447  100.061984  100.042686  100.029222  100.019901  100.013505  100.009166  100.006268  100.004384  100.003217  100.002567  100.002306  100.002352  100.002661  100.003216  100.004028  100.005167 | 441.664962 |     0.0 | 100.005167 |   6.558241 |
    | 2000-01-01 17:30:00 |  95.933524 | 6.881609  6.203455  5.502871  4.787414  4.065316  3.342046  2.619477   1.898374   1.180554   0.468811  -0.233828  -0.924435  -1.600338   -2.25936  -2.900076  -3.521963  -4.125391  -4.711493  -5.281939  -5.838697  -6.383815  -6.919265  -7.446837  -7.968092  -8.484346  -8.996688  -9.505996  -10.012972  -10.518169  -11.022017  -11.524851  -12.026926  -12.528438  -13.029536  -13.530334  -14.030919  -14.531362  -15.031725   -15.53207  -16.032473  -16.533046   -17.03397  -17.535561  -18.038373  -18.543414   -19.05253  -19.569182  -20.100123   -20.65953   -21.282066 | 489.279607  458.847329  428.683064  398.863588  369.576412  341.063265   313.54935  287.241886  262.360514  239.133335  217.769595  198.437022  181.245489  166.233952  153.364324  142.525927  133.549726  126.228412  120.337521  115.653679  111.967869  109.093376  106.869206  105.160274  103.855591  102.865453  102.118286  101.557581  101.139104  100.828487  100.599206  100.430913  100.308093  100.218984  100.154719  100.108654  100.075844  100.052626  100.036308  100.024925  100.017052  100.011663   100.00803  100.005636  100.004122  100.003244  100.002837  100.002795  100.003056  100.003593  100.004431 | 489.279607 |     0.0 | 100.004431 |   6.881609 |
    | 2000-01-01 18:00:00 | 134.151612 | 7.196594   6.52424  5.830133    5.1205  4.402414  3.680955  2.957915   2.233637   1.509298     0.7872   0.070118  -0.639024  -1.337229  -2.021596   -2.68963  -3.339529  -3.970386  -4.582237  -5.175967  -5.753106  -6.315581  -6.865475  -7.404842  -7.935579  -8.459361  -8.977615  -9.491525  -10.002057  -10.509981  -11.015909  -11.520319  -12.023582  -12.525984  -13.027746  -13.529035  -14.029982  -14.530691  -15.031247  -15.531731  -16.032235   -16.53288  -17.033856  -17.535483  -18.038321  -18.543381   -19.05251  -19.569171   -20.10012  -20.659533   -21.282075 |  537.77215  507.011831  476.309636  445.705166  415.342608  385.438718  356.211574  327.862882  300.609719  274.694036  250.361991  227.842431  207.330064  188.967356   172.82713  158.902258  147.106402  137.285776  129.238665  122.737797  117.550926  113.456659  110.254582  107.770243  105.856264  104.390964  103.275638  102.431317  101.795506  101.319185  100.964187  100.700988  100.506883  100.364506   100.26065  100.185322  100.131004  100.092071  100.064339  100.044715  100.030926  100.021317  100.014686  100.010172  100.007162  100.005222  100.004055  100.003455  100.003285   100.00346  100.003934 |  537.77215 |     0.0 | 100.003934 |   7.196594 |
    | 2000-01-01 18:30:00 |   7.292819 | 7.499696  6.834444  6.148439  5.446778  4.735418  4.018972  3.299185   2.576092   1.850244   1.123337   0.397668  -0.324197  -1.039417  -1.744916  -2.437604  -3.114691     -3.774   -4.41421  -5.034976  -5.636882  -6.221269  -6.789983   -7.34512  -7.888803  -8.423039  -8.949621  -9.470093   -9.985747  -10.497639  -11.006619  -11.513363    -12.0184  -12.522144  -13.024915  -13.526959  -14.028468  -14.529593  -15.030455  -15.531164  -16.031831  -16.532594  -17.033655  -17.535343  -18.038225  -18.543316  -19.052467  -19.569145  -20.100105  -20.659527   -21.282074 | 586.262607  555.621865  524.845065  493.936526  462.999425  432.219063  401.797676  371.923075  342.795905  314.649773  287.739138  262.321657  238.645304    216.9326  197.359769  180.036098  164.989704  152.163918  141.425565  132.583041  125.409399  119.665165  115.116956  111.550182  108.775967   106.63349  104.989232  103.734462   102.78194  102.062476    101.5217  101.117203   100.81612  100.593127  100.428808  100.308357  100.220534  100.156855  100.110946  100.078044  100.054612  100.038037  100.026402  100.018311  100.012752  100.009001  100.006542   100.00501  100.004152  100.003782  100.003745 | 586.262607 |     0.0 | 100.003745 |   7.499696 |
    | 2000-01-01 19:00:00 |  41.671269 | 7.787963  7.130759  6.454083  5.761984  5.059418   4.35052  3.636992   2.918667   2.195576   1.468818   0.740145   0.011598  -0.714447  -1.435208  -2.147571  -2.848311  -3.534378  -4.203226  -4.853107  -5.483247  -6.093877  -6.686089  -7.261603  -7.822493   -8.37094  -8.909051  -9.438741   -9.961679  -10.479273  -10.992681  -11.502839  -12.010493  -12.516233  -13.020517  -13.523703   -14.02607  -14.527835  -15.029174  -15.530235  -16.031161  -16.532114  -17.033314  -17.535101  -18.038056  -18.543198  -19.052387  -19.569091   -20.10007  -20.659504   -21.282059 | 633.846018  603.757917  573.360596    542.6221  511.611544   480.48083   449.40835  418.562517  388.121335  358.296507  329.329549  301.478942  275.011893  250.194793  227.276663  206.467137  187.913754  171.683838  157.755884  146.023385  136.310384  128.394482  122.031557  116.977297  113.002841  109.904084  107.505615   105.66085  104.249879  103.176189  102.363065  101.750115  101.290165  100.946604  100.691173  100.502168  100.362999  100.261042  100.186736  100.132875  100.094053  100.066238  100.046437  100.032443  100.022636   100.01584    100.0112  100.008101  100.006098  100.004849  100.004016 | 633.846018 |     0.0 | 100.004016 |   7.787963 |
    | 2000-01-01 19:30:00 |  72.477978 | 8.058405  7.410125  6.743743  6.062475   5.37037  4.671026  3.966085    3.25531   2.538366   1.815849      1.089   0.359319  -0.371404  -1.100945  -1.826601  -2.545282  -3.253679   -3.94853  -4.626944  -5.286735   -5.92667  -6.546564    -7.1472  -7.730115  -8.297318  -8.851023  -9.393429   -9.926575  -10.452261  -10.972018  -11.487118  -11.998591  -12.507265  -13.013791  -13.518682  -14.022338  -14.525075  -15.027142  -15.528747  -16.030077  -16.531329  -17.032747  -17.534696  -18.037767  -18.542994  -19.052244  -19.568992  -20.100001  -20.659457    -21.28202 | 679.625318  650.558224  621.006079   590.91715  560.326358   529.35247   498.15376    466.8843  435.700046  404.784273  374.352188  344.643052  315.915709  288.445303  262.514169   238.39536  216.330835  196.507402  179.034915  163.932356  151.126308  140.462777  131.729061  124.679813  119.061459  114.631135  111.168839  108.483412  106.413887  104.827897  103.618528  102.700588    102.0069  101.484924  101.093835  100.802081  100.585397  100.425203   100.30733  100.221024  100.158152  100.112595  100.079771  100.056263  100.039541  100.027736  100.019481  100.013769  100.009858  100.007153  100.005021 | 679.625318 |     0.0 | 100.005021 |   8.058405 |
    | 2000-01-01 20:00:00 |  99.872883 | 8.308838  7.670158  7.014776  6.345249  5.664824  4.976551  4.281975   3.580875    2.87269   2.157618   1.436474   0.710295   -0.01968  -0.751847  -1.484126  -2.213943  -2.938282  -3.653812  -4.357111  -5.044983  -5.714817  -6.364894  -6.994561  -7.604222  -8.195165  -8.769285  -9.328793   -9.875964  -10.412957  -10.941708  -11.463882  -11.980874  -12.493819  -13.003634  -13.511042  -14.016618   -14.52081  -15.023976  -15.526407  -16.028356  -16.530069   -17.03183  -17.534031  -18.037288  -18.542651  -19.051999  -19.568818  -20.099878  -20.659366   -21.281943 | 722.742733   695.10151  666.815338  637.817005  608.113645  577.792769  546.989795  515.842861  484.486981  453.077154  421.798709  390.862924  360.504218  330.981001  302.573733  275.576633  250.282953  226.964478  205.846824  187.084489  170.741661  156.784623  145.088523  135.456757  127.647457  121.400514  116.460034  112.589843  109.582022  107.259889  105.477249  104.115524  103.079972  102.295753  101.704258  101.259903  100.927431  100.679694  100.495881  100.360098  100.260254  100.187187  100.133984  100.095447  100.067692  100.047821  100.033686  100.023692  100.016633   100.01152  100.007237 | 722.742733 |     0.0 | 100.007237 |   8.308838 |
    | 2000-01-01 20:30:00 | 124.032744 | 8.537482  7.908918   7.26505  6.607898  5.940022  5.263931  4.581033   3.891167   3.193666    2.48844   1.775968   1.056925   0.332101  -0.397454  -1.130292  -1.864499  -2.597667  -3.326905  -4.048915  -4.760181  -5.457255  -6.137131  -6.797596  -7.437486  -8.056758  -8.656375  -9.238054   -9.803959  -10.356417  -10.897695   -11.42987   -11.95475  -12.473861  -12.988457  -13.499553  -14.007955  -14.514305  -15.019111  -15.522784  -16.025669  -16.528085  -17.030371  -17.532963   -18.03651  -18.542087  -19.051592  -19.568525  -20.099665  -20.659206   -21.281801 | 762.408181  736.563378    709.9296  682.427557  654.042259  624.833321  594.913163  564.404269  533.423274  502.098948  470.586245  439.065283  407.739171  376.836366  346.613111   317.35183   289.35446  262.930248  238.377217  215.957757  195.871522  178.231485  163.049858  150.238591  139.624667  130.975724  124.029047  118.517588  114.189259  110.818612   108.21196  106.207833  104.674606  103.506772  102.620818  101.951261  101.447128  101.068963   100.78637  100.576026  100.420097  100.304999  100.220415  100.158546   100.11351   100.08089   100.05738  100.040493  100.028324  100.019272  100.011449 | 762.408181 |     0.0 | 100.011449 |   8.537482 |
    | 2000-01-01 21:00:00 |  145.13634 | 8.742939   8.12488  7.492888  6.848538  6.193801  5.530678  4.860399   4.182875   3.497424   2.803766   2.102131   1.392923   0.676601  -0.046221  -0.774632  -1.507347  -2.242639  -2.978296  -3.711586  -4.439305  -5.157909  -5.863769  -6.553533  -7.224506  -7.874982  -8.504408  -9.113345   -9.703255  -10.276198  -10.834515   -11.38057  -11.916575  -12.444488  -12.965978   -13.48243  -13.994967  -14.504493  -15.011727  -15.517247  -16.021533  -16.525007  -17.028091   -17.53128  -18.035273  -18.541181  -19.050931  -19.568042   -20.09931  -20.658934   -21.281555 | 797.923464  774.204168  749.567865  723.926059  697.246593    669.5651  640.971839  611.574104  581.473016  550.774769  519.606176  488.116807  456.477275  424.881482  393.550855  362.736787  332.719868  303.805435  276.314096  250.565575  226.855548  205.427621   186.44555  169.972787  155.965742  144.283336  134.710079  126.985978  120.835839  115.992683  112.213163  109.285432  107.031267  105.304501  103.987508  102.986955  102.229553  101.658203  101.228691  100.906935  100.666771  100.488181  100.355898  100.258313  100.186629  100.134194  100.095987  100.068203  100.047883   100.03249  100.018925 | 797.923464 |     0.0 | 100.018925 |   8.742939 |
    | 2000-01-01 21:30:00 |  15.255807 | 8.924242  8.316938  7.697063  7.065772  6.424536  5.774894  5.117884   4.453466   3.780993   3.100084   2.410793   1.713332   1.007927   0.294881  -0.425309  -1.151865  -1.883668  -2.619194  -3.356441  -4.092873  -4.825428  -5.550605  -6.264673  -6.963998  -7.645441  -8.306746  -8.946797   -9.565673  -10.164499  -10.745159  -11.309961  -11.861337  -12.401627  -12.932944   -13.45711  -13.975652  -14.489819  -15.000621  -15.508873   -16.01524  -16.520295  -17.024574  -17.528666  -18.033336  -18.539751  -19.049876  -19.567265  -20.098732  -20.658484   -21.281144 | 828.701418  807.380965  785.036367   761.56766  736.930445  711.141608  684.270327  656.407243  627.639916  598.056648  567.760019   536.87164  505.531581  473.899951  442.161127  410.527884  379.243908  348.584377  318.853772  290.379055  263.496251  238.529426  215.763039  195.411635  177.593648  162.316963    149.4812   138.89633  130.311938  123.449104  118.028029  113.787741  110.497378  107.960555  106.015005  104.529505  103.399615   102.54318  101.896137  101.408849   101.04305  100.769354   100.56527  100.413631  100.301369  100.218559  100.157662  100.112919   100.07981  100.054376  100.031665 | 828.701418 |     0.0 | 100.031665 |   8.924242 |
    | 2000-01-01 22:00:00 |  32.068288 | 9.080616  8.484284  7.876679   7.25859  6.631055  5.995204  5.351866   4.701035   4.042136   3.374748   2.698807   2.014388   1.321558   0.620413  -0.088837  -0.805789  -1.529799  -2.259922  -2.994835  -3.732756  -4.471369  -5.207801   -5.93866  -6.660194   -7.36857  -8.060265  -8.732489    -9.38353  -10.012913  -10.621337  -11.210425  -11.782388  -12.339689   -12.88478  -13.419924  -13.947109  -14.468016  -14.984035  -15.496302  -16.005745  -16.513146  -17.019209  -17.524652  -18.030342  -18.537523  -19.048222  -19.566035  -20.097807  -20.657758   -21.280471 |  854.27953  835.597387   815.79355  794.764765  772.457567  748.873545  724.064008  698.105256  671.072773  643.038916  614.084967   584.30811  553.821704  522.755718  491.259953  459.508448  427.703404  396.078248  364.899603   334.46704  305.108756  277.171149   251.00069  226.917918  205.186096  185.980466  169.366244  155.292481  143.604183  134.068956  126.410218  120.338758  115.577127  111.875046  109.016708  106.822128  105.144859  103.867885  102.898947  102.166007  101.613224  101.197534  100.885866  100.652903  100.479309  100.350343  100.254762  100.183928  100.130994  100.089869  100.052747 |  854.27953 |     0.0 | 100.052747 |   9.080616 |
    | 2000-01-01 22:30:00 |  46.086845 | 9.211597   8.62639  8.031128  7.426298  6.812567  6.190694  5.561276   4.924311    4.27932   3.625899   2.963916   2.293351   1.614171   0.926336   0.229868   -0.47509  -1.188231  -1.909045  -2.636752  -3.370228   -4.10791  -4.847721  -5.587013  -6.322564  -7.050682  -7.767428  -8.468972   -9.152035   -9.814301   -10.45468  -11.073356  -11.671596  -12.251438  -12.815322  -13.365784  -13.905234  -14.435829  -14.959419  -15.477555  -15.991519  -16.502386  -17.011095   -17.51855  -18.025765  -18.534098  -19.045662  -19.564118  -20.096353  -20.656604   -21.279395 | 874.327871  858.470223  841.409172  823.040114  803.301888  782.182307  759.716311  735.967914  711.003998  684.885513  657.676619  629.453289  600.305116  570.335465  539.663659   508.42875  476.793473  444.948018  413.113653  381.545826  350.535534  320.407346   291.51205  264.211908  238.857353  215.756207  195.139974  177.135098  161.747829  148.868026  138.290896  129.749777  122.950854  117.602407   113.43498  110.212374  107.735361  105.840595  104.396916  103.300601  102.470532  101.843777  101.371793  101.017299  100.751746  100.553311  100.405293  100.294807  100.211566  100.146293  100.086838 | 874.327871 |     0.0 | 100.086838 |   9.211597 |
    | 2000-01-01 23:00:00 |  57.468361 | 9.317084  8.743089  8.160184  7.568585  6.968646  6.360795  5.745394    5.12242   4.491495   3.852268   3.204572   2.548326   1.883431   1.209767    0.52724   -0.16417  -0.864385  -1.573194  -2.290211  -3.014814  -3.746063  -4.482617  -5.222645  -5.963749  -6.702928  -7.436625  -8.160889   -8.871682   -9.565307  -10.238861  -10.890607  -11.520128  -12.128238  -12.716707  -13.287883   -13.84435  -14.388645  -14.923099  -15.449746  -15.970319  -16.486282    -16.9989   -17.50934  -18.018826   -18.52888   -19.04174  -19.561162  -20.094097    -20.6548   -21.277699 | 888.651531   875.74192  861.563255  846.011135  829.019717  810.566633  790.672872  769.390171  746.776313  722.882071  697.757458  671.461069   644.06301  615.644731  586.300062  556.138067  525.286655  493.896287    462.1439  430.237037  398.417718  366.965114  336.195626  306.458496  278.124685  251.567109  227.131923  205.103761  185.671752  168.905605  154.749705  143.037537  133.521723  125.910505   119.90156  115.207455  111.571136  108.772755  106.630353  104.996916  103.755732  102.815318  102.104621   101.56881  101.165759   100.86318  100.636306  100.465966  100.336765  100.234671  100.140978 | 888.651531 |     0.0 | 100.140978 |   9.317084 |
    | 2000-01-01 23:30:00 |  66.354142 | 9.397216  8.834472  8.263882  7.685429  7.099188  6.505299  5.903896   5.294911   4.678066   4.053085   3.419797   2.778077   2.127785   1.468757   0.800836   0.123901   -0.56211  -1.257182  -1.961193  -2.673871  -3.394734  -4.123018  -4.857595  -5.596882  -6.338746  -7.080444   -7.81862   -8.549404   -9.268655   -9.972351   -10.65706  -11.320387   -11.96125  -12.579911  -13.177765  -13.756989  -14.320153    -14.8699   -15.40873  -15.938878   -16.46229  -16.980659  -17.495512  -18.008368  -18.520984   -19.03578  -19.556649  -20.090634  -20.652013   -21.275062 | 897.187983  887.298677  876.090328  863.459821  849.338674  833.697019  816.544269  797.920909  777.876154  756.452025   733.68674  709.624191  684.317935  657.831066  630.236359  601.618156  572.075286  541.724323  510.703104  479.174723   447.33193  415.401547  383.648174  352.376059  321.927401  292.674827  265.005657  239.296535  215.879565  195.005106  176.810174  161.302209  148.364016  137.778336  129.264027  122.513585  117.224071  113.117863   109.95346  107.528639  105.678737  104.272371  103.206249  102.400004  101.791557  101.333153   100.98806  100.727762  100.529269  100.371438  100.225707 | 897.187983 |     0.0 | 100.225707 |   9.397216 |
    | 2000-01-02 00:00:00 |  72.871944 | 9.452358  8.900858  8.342501  7.777056  7.204358  6.624294  6.036777   5.441681   4.838821    4.22801   3.609091   2.981914   2.346312     1.7021    1.04909   0.387109  -0.283989  -0.964307  -1.653891  -2.352696   -3.06055  -3.777098  -4.501739  -5.233547  -5.971178  -6.712772  -7.455867    -8.19736   -8.933546    -9.66029  -10.373356  -11.068867  -11.743797  -12.396363  -13.026183  -13.634172  -14.222228  -14.792841   -15.34872  -15.892526  -16.426713  -16.953484  -17.474832  -17.992674  -18.509094  -19.026775  -19.549804  -20.085357  -20.647746   -21.271004 |      900.0  893.153286  884.952674  875.297224  864.117404  851.378023  837.079101  821.251017   803.93489  785.165046   764.96987  743.380969  720.437963  696.188586  670.688233  644.000982  616.201819  587.379341  557.638694  527.104941  495.927016  464.282218  432.380941  400.471149  368.841676  337.822863  307.782398  279.113796  252.215273  227.458552   205.15075  185.496986  168.574201  154.325047  142.574203  133.061551  125.481803  119.520595  114.881085  111.299645   108.55234  106.455036  104.859868  103.650107  102.734689  102.043108  101.520898  101.125635  100.822981   100.58115  100.356772 |      900.0 |     0.0 | 100.356772 |   9.452358 |
    | 2000-01-02 00:30:00 |    77.1369 | 9.483077  8.942774  8.396531  7.843918  7.284559  6.718117  6.144302    5.56292   4.973866   4.377053    3.77235   3.159594   2.538602    1.90918   1.271132   0.624262  -0.031617  -0.696678  -1.371063  -2.054869  -2.748117  -3.450723  -4.162445  -4.882832  -5.611143  -6.346264  -7.086601   -7.829988   -8.573611   -9.313998  -10.047123  -10.768657  -11.474388  -12.160725   -12.82518  -13.466658  -14.085485  -14.683173   -15.26204   -15.82481  -16.374292  -16.913186  -17.444016  -17.969197   -18.49125  -19.013217  -19.539467  -20.077362  -20.641255   -21.264809 | 897.264896  893.436062  888.232972  881.557612  873.340631  863.543281  852.158138  839.206676  824.722353  808.732518  791.257248  772.317905  751.942431  730.165722  707.028828  682.579289  656.872721  629.974999  601.964633   572.93544   542.99975  512.292255  480.974456  449.239556  417.317435  385.479019  354.038817  323.353738  293.815658  265.835014  239.813688  216.108283  194.989412     176.607  160.972532  147.964656  137.356218  128.853614  122.137062  116.893258  112.836815  109.721009   107.34049  105.528995  104.154526  103.113655  102.325841  101.728007  101.268857  100.900635   100.55772 | 897.264896 |     0.0 |  100.55772 |   9.483077 |
    | 2000-01-02 01:00:00 |  79.252408 | 9.490115  8.960928  8.426647  7.886658  7.340393  6.787322  6.226967   5.659059   5.083571   4.500511   3.909788   3.311231   2.704647   2.089844   1.466629   0.834805   0.194166  -0.455491  -1.114362  -1.782624  -2.460414  -3.147811  -3.844806  -4.551259  -5.266851  -5.991012  -6.722838   -7.460995   -8.203611   -8.948183   -9.691527  -10.429812  -11.158738  -11.873883  -12.571194  -13.247511  -13.900995  -14.531285  -15.139393  -15.727364  -16.297875  -16.853869  -17.398327  -17.934204  -18.464546  -18.992863  -19.523905  -20.065291  -20.631425   -21.255402 | 889.260928  888.382156   886.12254  882.387603  877.109139  870.246254  861.785775  851.741459  840.139538  827.000863  812.338356  796.164822  778.498426  759.363323  738.788682  716.808525  693.462779  668.799002  642.874285   615.75731  587.530806  558.294588  528.169234   497.30042  465.863812  434.070321  402.171203  370.462059  339.284142  309.020637   280.08505  252.899082  227.859289   205.29578  185.431352   168.35271  154.003667  142.202903  132.679923  125.117678  119.190928  114.594058   111.05709  108.351913  106.291861  104.727539    103.5409  102.638552  101.943979  101.385458  100.863894 | 889.260928 |     0.0 | 100.863894 |   9.490115 |
    | 2000-01-02 01:30:00 |   79.31108 | 9.474371  8.956187  8.433687  7.906083  7.372635  6.832644  6.285459    5.73074   5.168527   4.598924   4.021881   3.437227   2.844764   2.244305    1.63567   1.018669   0.393101  -0.241252  -0.884608  -1.537184  -2.199178  -2.870765  -3.552072  -4.243154  -4.943961  -5.654291  -6.373724   -7.101552   -7.836682   -8.577529   -9.321909  -10.066958  -10.809114  -11.544216  -12.267768  -12.975376  -13.663295  -14.328939    -14.9712  -15.590479  -16.188444  -16.767653  -17.331174  -17.882346  -18.424734  -18.962386  -19.500524  -20.047106  -20.616577   -21.241163 | 876.351729   878.31663  878.907066  878.032575  875.626835  871.647918  866.078344  858.925286   850.20818  839.941717  828.132531   814.78624  799.912834  783.527452  765.649364  746.301485  725.511007  703.310741  679.740635  654.849345   628.69606  601.352775  572.907153  543.466003  513.159409  482.145526  450.615913  418.801077  386.975523  355.461022  324.626083  294.878859   266.65041  240.366226  216.406947  195.064266  176.502964  160.741235  147.656436  137.014152  128.510429  121.814689  116.604141  112.585972  109.508072  107.161217  105.375772  104.015072  102.965645  102.119981  101.328619 | 876.351729 |     0.0 | 101.328619 |   9.474371 |
    | 2000-01-02 02:00:00 |  77.395712 | 9.436875  8.929551  8.418624  7.903141  7.382208  6.854972  6.320628   5.778773   5.229505   4.673022   4.109314   3.538214   2.959523   2.373061   1.778665   1.176163   0.565361  -0.053953  -0.682004   -1.31902  -1.965231  -2.620858  -3.286102  -3.961132  -4.646059   -5.34091  -6.045581   -6.759789      -7.483   -8.214343   -8.952503   -9.695612  -10.441146  -11.185861  -11.925828  -12.656607   -13.37361  -14.072619  -14.750357  -15.404941  -16.036078   -16.64496  -17.233963  -17.806297   -18.36579   -18.91695  -19.465496  -20.019763  -20.594185   -21.219644 | 858.969595  863.638193  866.950799  868.821318  869.185859  868.002336  865.250162  860.930952  855.057774  847.639411   838.67688  828.169731  816.121104  802.538552  787.433041  770.818289   752.71109  733.132348  712.108288   689.67165  665.863018   640.73248  614.341757  586.766859  558.101324  528.460108   497.98418  466.845799  435.254255  403.461594  371.767341  340.520559  310.116752  280.986444  253.572439  228.294713  205.506217   185.44855  168.220336  153.769261  141.910744  132.366285  124.808827  118.903296  114.335564  110.828587  108.147711  106.097795  104.512928  103.232932  102.032771 | 858.969595 |     0.0 | 102.032771 |   9.436875 |
    | 2000-01-02 02:30:00 |  73.834064 | 9.378767  8.882133  8.382547    7.8789  7.370152  6.855319  6.333458   5.804108   5.267427   4.723696   4.172945   3.615012   3.049698   2.476836   1.896279   1.307876    0.71145   0.106799  -0.506294  -1.128059  -1.758729  -2.398545  -3.047743  -3.706549  -4.375167  -5.053753  -5.742398   -6.441085    -7.14965   -7.867711   -8.594597   -9.329245  -10.070089  -10.814946  -11.560925  -12.304398  -13.041099  -13.766393  -14.475737  -15.165264   -15.83234  -16.475936  -17.096699   -17.69679  -18.279629  -18.849789  -19.413298  -19.978778  -20.560472   -21.187159 |   837.5984  844.802365  850.679974  855.149727  858.150562  859.640903  859.598054   858.01897  854.910743   850.27647  844.112084  836.411985  827.173571  816.397984  804.089181  790.253193  774.898264  758.035632  739.680494  719.852916  698.578787   675.89101  651.831043  626.450875  599.815456  572.005677   543.12199  513.288757  482.659343  451.421869  419.805292  388.085124  356.587462  325.689228  295.811625  267.403398  240.911404  216.739163  195.199591  176.473708  160.588669   147.42315  136.737902  128.220437  121.530245  116.334461  112.329687  109.249686  106.858598  104.920965  103.099403 |   837.5984 |     0.0 | 103.099403 |   9.378767 |
    | 2000-01-02 03:00:00 |  70.012121 | 9.301281   8.81514  8.326639   7.83452   7.33761  6.834804  6.325039   5.807812   5.283334   4.751963   4.213765   3.668584   3.116223   2.556527    1.98937    1.41462   0.832121   0.241686   -0.35689  -0.963824  -1.579345  -2.203694  -2.837117  -3.479864  -4.132181  -4.794296  -5.466407   -6.148656   -6.841101   -7.543678   -8.256142   -8.977996   -9.708406  -10.446087  -11.189191   -11.93519  -12.680809  -13.422052  -14.154379  -14.873084  -15.573855  -16.253396  -16.909959  -17.543638  -18.156402  -18.752048   -19.33633   -19.91775  -20.509893   -21.138203 | 812.756821  822.304828   830.56628  837.464381  842.941168  846.956078  849.485159  850.521511  850.066573  848.117913  844.666837  839.703374  833.220104  825.212742  815.679275  804.619261  792.033851  777.926385  762.303149  745.174071  726.553398  706.460533  684.921144  661.968595  637.645736  612.007108  585.121666  557.076112  527.978959  497.965367  467.202754  435.896981  404.298663  372.708601  341.480647   311.01936  281.769026   254.19067   228.72561  205.748708  185.520738  168.153705  153.601251  141.677575  132.097409    124.5231  118.605333  114.008417  110.413375  107.482753  104.715487 | 812.756821 |     0.0 | 104.715487 |   9.301281 |
    | 2000-01-02 03:30:00 |  66.053675 | 9.205723  8.729853  8.252159  7.771241  7.285801  6.794624  6.296548   5.791038   5.278362   4.758942   4.232874   3.700008   3.160154   2.613167   2.058942   1.497369    0.92831   0.351598  -0.232956  -0.825554  -1.426412  -2.035759  -2.653837    -3.2809  -3.917208   -4.56302  -5.218587   -5.884135   -6.559853   -7.245861   -7.942178   -8.648677   -9.365015  -10.090561   -10.82429  -11.564665  -12.309524  -13.055973  -13.800357  -14.538358  -15.265276  -15.976534  -16.668326  -17.338288  -17.986026  -18.613437  -19.224949  -19.828032  -20.434597   -21.064755 | 784.982427  796.665404  807.110777  816.246493  824.017853  830.385666  835.325376  838.826849  840.886178  841.495607  840.642231  838.312357  834.494516    829.1798  822.361092   814.03245  804.189095  792.827865  779.947809  765.550692  749.641489  732.228957  713.326428  692.952833  671.134005  647.904273   623.30846  597.404343  570.265716  541.986137  512.683458   482.50518  451.634562  420.297182  388.767283  357.372569  326.495264   296.56628  268.048766  241.408102  217.068701  195.363842  176.491009   160.48718   147.23288  136.482539  127.908295  121.139614  115.779335  111.363843  107.162191 | 784.982427 |     0.0 | 107.162191 |   9.205723 |
    | 2000-01-02 04:00:00 |  62.189815 | 9.093457  8.627607  8.160422  7.690357  7.216001   6.73604  6.249226   5.755011   5.253719   4.745826   4.231451    3.71045   3.182639   2.647888   2.106111   1.557217   1.001091   0.437582   -0.13348   -0.71228  -1.299016    -1.8939  -2.497163  -3.109048  -3.729812  -4.359722  -4.999049   -5.648057   -6.307001   -6.976103   -7.655534   -8.345386   -9.045629   -9.756057  -10.476214  -11.205308  -11.942094  -12.684758  -13.430798  -14.176954  -14.919231  -15.653102  -16.373926  -17.077597  -17.761315  -18.424336  -19.068569  -19.699024  -20.324146   -20.955644 | 754.817086  768.413317  780.829011   791.99682  801.865383   810.39713   817.56737  823.363219  827.775707  830.792003  832.395389  832.568996  831.297938  828.569355  824.371736  818.694438   811.52769  802.862949  792.693338  781.014021  767.822534  753.119194  736.907651   719.19562  699.995816  679.327099  657.215908  633.698048  608.820918  582.646294  555.253761  526.744899  497.248321  466.925547  435.977586  404.651759  373.247816  342.121588  311.683473  282.388064  254.711201  229.112526  205.986453  185.611133  168.109926  153.437981  141.396225  131.660279  123.795876  117.204822  110.852357 | 754.817086 |     0.0 | 110.852357 |   9.093457 |
    | 2000-01-02 04:30:00 |  58.428398 | 8.965883  8.509781  8.052782  7.593205  7.129532  6.660356  6.184361   5.701005   5.210667   4.713865   4.210736   3.701138   3.184897   2.661898   2.132072   1.595351   1.051636   0.500795  -0.057324  -0.622888  -1.196074  -1.777077  -2.366108  -2.963396  -3.569187  -4.183737  -4.807319   -5.440208   -6.082684   -6.735019   -7.397464   -8.070232   -8.753474   -9.447244  -10.151448  -10.865787  -11.589673  -12.322127  -13.061663  -13.806168  -14.552804  -15.297984  -16.037497  -16.766846  -17.481846  -18.179415  -18.858357  -19.519757  -20.166083   -20.796527 |  722.79402     738.074  752.237573  765.222004  776.979254  787.473518  796.680307  804.584332   811.17184  816.425224   820.32434  822.849607  823.983239  823.708979  822.011517  818.876177  814.288964   808.23685  800.708086  791.692457  781.181503  769.168804  755.650369  740.625155  724.095732  706.069097  686.557699  665.580708   643.16562  619.350266  594.185322  567.737425  540.093009   511.36298  481.688265  451.246217  420.257601  388.993493   357.78078   327.00403  297.100402  268.543566   241.81322      217.35  195.501523  176.471506    160.2846  146.769055  135.533746  125.865773  116.364898 |  722.79402 |     0.0 | 116.364898 |   8.965883 |
    | 2000-01-02 05:00:00 |  54.679715 | 8.824432  8.377779  7.930623  7.481149   7.02774  6.568902  6.103272   5.630328   5.150503   4.664348   4.172006   3.673341    3.16819   2.656453   2.138078   1.613016   1.081184   0.542469  -0.003264  -0.556163  -1.116387   -1.68411  -2.259523  -2.842837  -3.434279  -4.034092  -4.642535   -5.259879   -5.886407   -6.522406   -7.168162   -7.823949   -8.490015   -9.166557   -9.853699  -10.551444  -11.259627  -11.977838  -12.705337  -13.440943  -14.182916  -14.928851  -15.675622  -16.419433  -17.156049   -17.88122  -18.591107  -19.282078  -19.948042   -20.571451 | 689.426711  706.157509   721.84207  736.422121  749.852873  762.100353  773.140471  782.955742  791.527888  798.834888  804.853559   809.56201  812.939979  814.968269  815.628281  814.901861  812.771432  809.220217   804.23244   797.79348  789.890014  780.510216   769.64403  757.283538  743.423427  728.061568  711.199733  692.844499  673.008384   651.71128  628.982253  604.861795  579.404636  552.683238  524.792069  495.852788  466.020336  435.489829  404.503819  373.358972  342.410396  312.070763  282.800248  255.082898  229.386427  206.106223  185.498703  167.607015  152.160296  138.368159  124.448443 | 689.426711 |     0.0 | 124.448443 |   8.824432 |
    | 2000-01-02 05:30:00 |  50.967805 | 8.670544  8.233015  7.795337  7.355564  6.911984  6.463025  6.007292   5.544303   5.074543   4.598581   4.116563   3.628357   3.133811   2.632842   2.125415   1.611496    1.09102   0.563887    0.02998  -0.510829  -1.058682  -1.613732  -2.176152  -2.746131  -3.323877  -3.909614  -4.503583   -5.106041    -5.71726   -6.337524   -6.967128   -7.606366   -8.255531   -8.914895   -9.584697  -10.265116  -10.956238  -11.658009  -12.370179  -13.092216  -13.823215  -14.561785  -15.305931  -16.052959  -16.799402  -17.540942  -18.272044   -18.98436  -19.661445   -20.266931 | 655.199767  673.149032  690.127249  706.080425  720.966896  734.754534  747.419729  758.942892  769.301472  778.469426  786.420936  793.132144  798.580655  802.744718  805.602872  807.133971  807.317397  806.133231  803.562359  799.586536  794.188477  787.351997  779.062203  769.305752  758.071182  745.349306  731.133725  715.421455  698.213735  679.517032   659.34432  637.716684   614.66533  590.234118  564.482705  537.490446   509.36115  480.228789  450.264105   419.68188  388.748188  357.786228  327.178267  297.359745  268.800055  241.963229  217.240408  194.840703  174.606519  155.674381  135.945495 | 655.199767 |     0.0 | 135.945495 |   8.670544 |
    | 2000-01-02 06:00:00 |  47.318875 | 8.505663  8.076907   7.64832  7.217827  6.783625  6.344072  5.897761   5.444263    4.98411   4.517882   4.045718   3.567492   3.083064   2.592368   2.095386   1.592096   1.082447   0.566357   0.043722  -0.485569  -1.021639  -1.564623  -2.114673   -2.67196   -3.23667  -3.809007  -4.389193   -4.977463   -5.574074   -6.179297   -6.793416   -7.416728   -8.049536   -8.692143   -9.344844  -10.007908  -10.681561  -11.365957  -12.061138  -12.766984  -13.483145  -14.208944  -14.943263  -15.684369  -16.429654  -17.175089  -17.913891  -18.633053  -19.305276   -19.877995 | 620.561767  639.501268  657.548949  674.654886  690.780369  705.895064  719.975821  733.000961  744.943919  755.775072  765.466349  773.992226   781.32845  787.451038  792.336037  795.959736  798.298936  799.331088  799.034304  797.387354  794.369718   789.96168  784.144477  776.900474  768.213385  758.068553  746.453296  733.357354  718.773448  702.697989  685.131974  666.082118  645.562278  623.595246   600.21501  575.469576  549.424483  522.167128  493.812003  464.506845  434.439552  403.845266  373.012355  342.284616  312.054747  282.740042  254.723895  228.232328  203.089292  178.289439  151.593125 | 620.561767 |     0.0 | 151.593125 |   8.505663 |
    | 2000-01-02 06:30:00 |  43.698082 | 8.331224  7.910864  7.490958  7.069304  6.644013  6.213382  5.776008   5.331529   4.880522   4.423562   3.960778    3.49205   3.017253   2.536336   2.049296   1.556124   1.056781   0.551196   0.039284  -0.479053   -1.00392  -1.535435  -2.073732  -2.618962  -3.171292  -3.730905  -4.298001   -4.872797   -5.455526   -6.046441   -6.645811   -7.253918   -7.871061   -8.497548    -9.13369   -9.779797  -10.436163  -11.103049  -11.780663   -12.46912  -13.168397  -13.878252  -14.598102  -15.326807  -16.062219  -16.800193  -17.532261  -18.240368  -18.887492   -19.412822 | 585.920022  605.628683  624.527858  642.571448  659.723491  675.955357  691.244246  705.566333  718.891319  731.186469  742.421837  752.570478  761.606512  769.503988  776.236795  781.779026   786.10531  789.190897  791.011595  791.543715  790.764088  788.650143  785.180014  780.332659  774.088007  766.427142  757.332545  746.788401  734.780983  721.299156  706.334993  689.884574  671.948977  652.535548  631.659485  609.345844   585.63205   560.57101  534.234957  506.720054  478.151745  448.690523  418.537211  387.935613  357.167744  326.531526  296.280157  266.483655    236.7562  205.869838  171.733418 | 585.920022 |     0.0 | 171.733418 |   8.331224 |
    | 2000-01-02 07:00:00 |  40.126442 | 8.148642  7.736275  7.324616  6.911342  6.494481  6.072274  5.643345   5.207409   4.765079   4.316916   3.863036   3.403324   2.937669   2.466038   1.988442   1.504881   1.015326   0.519719   0.017988  -0.489949  -1.004183  -1.524815  -2.051964  -2.585761  -3.126355  -3.673908  -4.228601   -4.790629   -5.360205   -5.937558   -6.522937   -7.116608   -7.718852   -8.329965   -8.950258   -9.580047   -10.21965  -10.869377   -11.52951  -12.200281  -12.881822  -13.574094   -14.27673  -14.988728  -15.707787  -16.428817  -17.140691   -17.81999   -18.42323   -18.891849 |  551.63713  571.903605  591.445137  610.219178  628.192302  645.337454  661.631996   677.04987  691.557339  705.119039  717.703506   729.28258  739.828938  749.314897  757.712478  764.993916  771.132021  776.100212  779.872391   782.42284  783.726231  783.757687  782.492856  779.907997  775.980064  770.686829  764.007041  755.920638  746.409012  735.455353   723.04508  709.166392  693.810961  676.974808  658.659414  638.873102  617.632778  594.966075  570.913964  545.533826  518.902862  491.121391  462.314904  432.632268  402.234314  371.260937  339.753781  307.499197  273.776389  237.178966  196.093563 |  551.63713 |     0.0 | 196.093563 |   8.148642 |
    | 2000-01-02 07:30:00 |  36.608922 | 7.959308  7.554502  7.150634  6.745261  6.336331  5.922042   5.50106   5.073186   4.639059   4.199217   3.753761    3.30258   2.845581   2.382748     1.9141   1.439648    0.95937   0.473221   -0.01886  -0.516942  -1.021101  -1.531423  -2.048012  -2.570984  -3.100469  -3.636611  -4.179571   -4.729524   -5.286661   -5.851191    -6.42334   -7.003352   -7.591487   -8.188023   -8.793255   -9.407489  -10.031039  -10.664222  -11.307335  -11.960635  -12.624283  -13.298236  -13.982039  -14.674373  -15.372103  -16.068292  -16.748369  -17.384287  -17.931232   -18.341339 | 518.029154  538.653901  558.639616  577.947033  596.545013  614.407902  631.512974  647.831849  663.327687  677.962984  691.705073  704.524754  716.393408  727.281826   737.16043    745.9999  753.771539  760.447256  765.999377  770.400529   773.62363  775.641946  776.429149  775.959363  774.207215  771.147905  766.757309  761.012123  753.890043  745.369995  735.432433  724.059714  711.236572  696.950713  681.193569  663.961212  645.255491   625.08536  603.468401  580.432366  556.016412  530.271179  503.255929  475.029067  445.624802  415.002709  382.950267  348.925247  311.900766  270.503137  223.811245 | 518.029154 |     0.0 | 223.811245 |   7.959308 |
    | 2000-01-02 08:00:00 |  33.116929 | 7.764579  7.366874  6.970316  6.572345  6.170832  5.763944  5.350405   4.930107   4.503704   4.071705   3.634191   3.191058   2.742231   2.287708   1.827518   1.361678   0.890175   0.412972  -0.069981  -0.558739  -1.053367  -1.553938  -2.060542  -2.573278  -3.092262  -3.617622  -4.149499   -4.688049   -5.233443    -5.78587   -6.345532   -6.912651   -7.487466   -8.070229   -8.661214   -9.260703   -9.868988   -10.48636  -11.113086  -11.749371  -12.395279  -13.050581  -13.714435  -14.384758  -15.056974  -15.721669  -16.360827  -16.943943  -17.430211   -17.786035 | 485.365238  506.162139  526.406516  546.062076  565.099727  583.495016  601.224833  618.258394  634.556122  650.078844  664.792979  678.668418  691.675375  703.783303  714.961273  725.178686  734.405613  742.612729  749.771075  755.851933  760.826812  764.667502  767.346113  768.835099  769.107278  768.135863  765.894534  762.357523  757.499734  751.296897  743.725761  734.764343  724.392237  712.590996  699.344592  684.639957  668.467574  650.822055  631.702545  611.112616  589.058934  565.547368  540.573903  514.105646  486.043857  456.157529  423.979289  388.689621  349.127744  304.206832  253.696968 | 485.365238 |     0.0 | 253.696968 |   7.764579 |
    | 2000-01-02 08:30:00 |  29.690379 | 7.565774  7.174681  6.784928   6.39384  5.999215  5.599199  5.192596   4.779383   4.360221   3.935582   3.505526   3.069959   2.628822   2.182127   1.729909   1.272192   0.808968   0.340207  -0.134128  -0.614082   -1.09971  -1.591073  -2.088247   -2.59132  -3.100393  -3.615577  -4.136997   -4.664792   -5.199114   -5.740132   -6.288027   -6.842997   -7.405257   -7.975036   -8.552575   -9.138125   -9.731938  -10.334251   -10.94525  -11.565013  -12.193389  -12.829767   -13.47264  -14.118781   -14.76177  -15.389613  -15.981822  -16.508397  -16.936153   -17.244968 |  453.86844  474.666026  494.997447  514.829098  534.133682  552.887707   571.06736  588.639415  605.561949  621.794569  637.303016  652.056383  666.023828  679.173641  691.473777  702.892603  713.399209  722.963273  731.554804  739.144006  745.701275  751.197245  755.602814  758.889146  761.027669  761.990084  761.748405  760.275004  757.542693  753.524808   748.19533  741.529028  733.501632  724.090028  713.272457  701.028676      687.34  672.189054  655.558911  637.431038  617.780958  596.569756  573.728182  549.128289  522.535776   493.53811  461.460177  425.333979  384.083788  337.055967  284.545427 |  453.86844 |     0.0 | 284.545427 |   7.565774 |
    | 2000-01-02 09:00:00 |  26.289405 | 7.364167  6.979167  6.595688  6.210943  5.822662  5.428984  5.028802    4.62218   4.209772   3.792006   3.368925   2.940442   2.506516    2.06717   1.622445   1.172367   0.716934   0.256124  -0.210093  -0.681752  -1.158895  -1.641577  -2.129862  -2.623826  -3.123555  -3.629148  -4.140714   -4.658377   -5.182271   -5.712545   -6.249361   -6.792894   -7.343333   -7.900878   -8.465735   -9.038111   -9.618198  -10.206145  -10.802008  -11.405653  -12.016587   -12.63364  -13.254403  -13.874263  -14.484871   -15.07214  -15.614743  -16.085827  -16.461039   -16.730732 | 423.717566  444.359781  464.621397  484.471194  503.883414  522.835252   541.30187   559.24761  576.628627  593.403712  609.538125  625.000239  639.758266  653.779502  667.031001  679.480326  691.095801  701.846319  711.701073  720.629429  728.600923  735.585308  741.552561  746.472874  750.316629  753.054396  754.656941  755.095255  754.340583  752.364468  749.138795  744.635844  738.828327  731.689389  723.192522  713.311303  702.018797  689.286328  675.081143  659.362122  642.072139  623.124873  602.382764  579.621986    554.4815  526.401894   494.58715  458.076146  416.040344    368.2576  315.336098 | 423.717566 |     0.0 | 315.336098 |   7.364167 |
    | 2000-01-02 09:30:00 |   22.94393 | 7.160984  6.781527  6.403767  6.024805  5.642309  5.254425  4.860147   4.459616    4.05347   3.642089   3.225497    2.80362    2.37643   1.943959   1.506252   1.063337   0.615215   0.161872  -0.296715  -0.760572  -1.229736  -1.704249  -2.184167  -2.669555  -3.160488  -3.657053  -4.159345   -4.667472   -5.181552   -5.701717   -6.228108   -6.760877   -7.300185   -7.846199   -8.399081   -8.958979      -9.526  -10.100171  -10.681354  -11.269124  -11.862525  -12.459679  -13.057126  -13.648812  -14.224726  -14.769603  -15.262997  -15.682802  -16.012841   -16.250616 | 395.049792  415.396489  435.446734    455.1714  474.546025  493.548205  512.151761  530.318682   548.00362  565.164911  581.767485  597.779068  613.167022  627.897822  641.937823   655.25399  667.814069  679.586357  690.539425  700.642005     709.863  718.171513  725.536853  731.928498  737.316072  741.669315  744.958085  747.152346  748.222173  748.137737  746.869282  744.387071  740.661268  735.661717  729.357519  721.716259  712.702654  702.276202  690.387206  676.970139  661.932824  645.139274  626.383608  605.353277  581.584594  554.427734   523.06761  486.677044  444.736933  397.362037  345.299755 | 395.049792 |     0.0 | 345.299755 |   7.160984 |
    | 2000-01-02 10:00:00 |  19.641095 | 6.957398  6.582903  6.210281  5.836522  5.459238  5.076597  4.687701   4.292758   3.892378   3.486891   3.076306   2.660556    2.23963   1.813565   1.382408   0.946187   0.504906   0.058556  -0.392878  -0.849418   -1.31109  -1.777931  -2.249988  -2.727317  -3.209983  -3.698059  -4.191631   -4.690791   -5.195643   -5.706299   -6.222882   -6.745517   -7.274335   -7.809461   -8.351003   -8.899032   -9.453543  -10.014395  -10.581204  -11.153166  -11.728771  -12.305339  -12.878326  -13.440387  -13.980375  -14.482919  -14.929786  -15.304075  -15.596225   -15.808156 | 367.963873  387.890918  407.603699  427.074903  446.281115  465.200009  483.803795  502.052258  519.898972  537.302124  554.226421     570.639  586.506481  601.794683  616.469452  630.497338   643.84568   656.48234   668.37544  679.493267  689.804281  699.277142  707.880694  715.583928  722.355936  728.165875  732.982946  736.776355  739.515279  741.168789  741.705746  741.094616  739.303177  736.298018  732.043715  726.501481  719.626955  711.366667  701.652408  690.392451  677.458185  662.664511  645.742921  626.308878  603.832797  577.639854  546.984622  511.243614   470.18724  424.149034  373.902146 | 367.963873 |     0.0 | 373.902146 |   6.957398 |
    | 2000-01-02 10:30:00 |  18.872077 | 6.754513  6.384378  6.016288  5.647136  5.274479  4.896523  4.512482   4.122619   3.727506    3.32742   2.922358   2.512263   2.097132   1.677009   1.251939    0.82195   0.387048  -0.052773  -0.497521  -0.947212  -1.401868  -1.861519  -2.326205  -2.795972  -3.270878  -3.750984  -4.236365   -4.727101   -5.223279   -5.724994   -6.232344   -6.745429   -7.264342   -7.789159   -8.319919   -8.856593   -9.399035   -9.946896  -10.499487  -11.055568  -11.613012  -12.168314  -12.715934  -13.247548  -13.751532  -14.213363  -14.617761  -14.952678  -15.213445   -15.404483 | 342.523746  361.923901  381.188903  400.293458  419.214939  437.930768  456.411486  474.614999  492.494168    510.0072  527.118648  543.795124  560.002639  575.706517  590.872274  605.466201  619.455376  632.807377  645.490032  657.471345  668.719505  679.202902  688.890094  697.749764  705.750656  712.861534  719.051127  724.288069  728.540801  731.777439  733.965553  735.071841  735.061614  733.897977  731.540561  727.943538  723.052548  716.800009  709.098046  699.828072  688.825948  675.862019  660.616923    642.6584  621.433186  596.301042  566.644284  532.057288  492.538387  448.539334  400.799808 | 342.523746 |     0.0 | 400.799808 |   6.754513 |
    | 2000-01-02 11:00:00 |  46.132977 |  6.55334  6.186961  5.822787  5.457632  5.089008  4.715168  4.335446   3.950152   3.559807   3.164633   2.764612   2.359699   1.949899   1.535259   1.115821   0.691609   0.262633  -0.171111  -0.609628  -1.052929  -1.501031  -1.953959  -2.411746  -2.874432  -3.342064  -3.814698  -4.292393   -4.775219   -5.263246   -5.756549   -6.255203   -6.759274    -7.26881   -7.783828   -8.304282   -8.830022   -9.360731   -9.895813  -10.434233  -10.974274  -11.513175  -12.046664  -12.568389  -13.069455  -13.538406  -13.962245  -14.328849  -14.630253  -14.865167    -15.03938 | 318.762374  337.547368  356.270486  374.910087  393.444318  411.850701  430.098751  448.144611  465.940134  483.443547  500.619507   517.43435  533.853632  549.842316  565.365708  580.389972  594.882063  608.809413  622.139701  634.840791   646.88075  658.227841  668.850486  678.717204  687.796539  696.056998   703.46697  709.994619  715.607739  720.273523  723.958228  726.626675  728.241483  728.761917   728.14215  726.328649  723.256311  718.842788  712.980365  705.524641  696.279566  684.979309  671.269977  654.699278  634.730114  610.800725  582.446214  549.456705  511.989014  470.541137  425.803488 | 318.762374 |     0.0 | 425.803488 |    6.55334 |
    | 2000-01-02 11:30:00 |  77.019007 | 6.354971  5.991662  5.630746  5.268944   4.90373  4.533429  4.157494    3.77626    3.39018   2.999423   2.603964   2.203764   1.798837   1.389226    0.97497    0.55609   0.132594  -0.295518  -0.728249  -1.165605  -1.607602  -2.054259  -2.505604   -2.96167  -3.422498  -3.888133  -4.358624   -4.834026   -5.314395   -5.799782   -6.290233   -6.785777    -7.28641   -7.792075   -8.302626   -8.817772   -9.336996   -9.859428  -10.383661  -10.907488   -11.42755  -11.938917  -12.434684  -12.905806  -13.341514  -13.730701  -14.064219  -14.337311  -14.550904    -14.71127 |  296.68568  314.768383  332.864651   350.95236  369.010468  387.016183   404.93631  422.724464  440.332382  457.719122  474.849536  491.689561  508.204309  524.358517   540.11746  555.447342  570.315125  584.688215   598.53426  611.821114  624.516844  636.589716  648.008135  658.740573  668.755482  678.021206  686.505874  694.177238  701.002457  706.947773  711.978038  716.056027  719.141414  721.189285  722.147944  721.955757  720.536624  717.793625  713.600354  707.789587  700.139559  690.359596  678.079844  662.854662  644.194327  621.639193  594.873936  563.843194  528.800097  490.243708  448.775478 |  296.68568 |     0.0 | 448.775478 |   6.354971 |
    | 2000-01-02 12:00:00 |  103.44243 |  6.16033  5.799374  5.441038   5.08193  4.719501  4.352166  3.979485    3.60179   3.219465   2.832633   2.441257   2.045307   1.644799   1.239769   0.830252   0.416264  -0.002187  -0.425101  -0.852477  -1.284324   -1.72065  -2.161474  -2.606818   -3.05671  -3.511182  -3.970271  -4.434016   -4.902455   -5.375626   -5.853556   -6.336259   -6.823718   -7.315868   -7.812569   -8.313558   -8.818383    -9.32631   -9.836175  -10.346191  -10.853681  -11.354758  -11.843993  -12.314202  -12.756563  -13.161356   -13.51946  -13.824319  -14.073577  -14.269543   -14.418445 | 276.276442   293.59445  310.997563  328.463101  345.970589  363.496094  381.003626  398.445522  415.774292  432.949721  449.936493  466.700036  483.205142  499.416532   515.29963   530.82079  545.947074  560.645953  574.885161  588.632658  601.856628  614.525445  626.607597  638.071601  648.885895   659.01873  668.438018  677.111118   685.00454  692.083518  698.311396  703.648749  708.052117  711.472193  713.851258  715.119578  715.190427  713.953413  711.265823  706.942134  700.742693  692.364384  681.439089  667.549354  650.272315  629.256746  604.318704  575.515373  543.150605  507.699798  469.680868 | 276.276442 |     0.0 | 469.680868 |    6.16033 |
    | 2000-01-02 12:30:00 | 126.025012 | 5.970259  5.610927  5.254472  4.897388  4.537119  4.172178  3.802212   3.427531   3.048451   2.665049   2.277283   1.885123   1.488583   1.087693   0.682479   0.272953   -0.14088  -0.559019  -0.981465  -1.408223  -1.839301  -2.274716  -2.714484  -3.158629  -3.607175   -4.06015  -4.517581   -4.979491   -5.445896   -5.916797   -6.392168   -6.871942   -7.355986   -7.844062   -8.335779   -8.830512   -9.327294   -9.824663  -10.320462  -10.811589  -11.293729  -11.761133  -12.206597   -12.62182   -12.99832  -13.328876  -13.609072  -13.838281  -14.019621   -14.159011 | 257.498082  274.003215   290.66019  307.446426  324.341568  341.319938  358.343137  375.363078  392.333319  409.214269   425.97044  442.566907  458.968338  475.139582  491.046292  506.655027  521.933011  536.847878  551.367547  565.460188  579.094199  592.238147  604.860685  616.930441  628.415907  639.285285  649.506298  659.045916   667.86998  675.942668  683.225744  689.677499  695.251266  699.893342  703.540125  706.114216  707.519244  707.633216  706.300431  703.322553  698.450515  691.380817  681.762222  669.220816  653.409747  634.080579  611.155813  584.768232  555.241055  523.013848  488.534349 | 257.498082 |     0.0 | 488.534349 |   5.970259 |
    | 2000-01-02 13:00:00 |   8.109091 | 5.785498  5.427072  5.071787  4.716053  4.357322  3.994202  3.626405   3.254211   2.877866   2.497406   2.112779   1.723953   1.330936    0.93375   0.532409   0.126922   -0.28271  -0.696488  -1.114414  -1.536493  -1.962733  -2.393146  -2.827749  -3.266556  -3.709588  -4.156861  -4.608388   -5.064178   -5.524221   -5.988488   -6.456912   -6.929366   -7.405641   -7.885395   -8.368097   -8.852939   -9.338718   -9.823681  -10.305328  -10.780196  -11.243657  -11.689824  -12.111694  -12.501685  -12.852612  -13.158959  -13.418007  -13.630336   -13.79949   -13.931052 | 240.298231  255.957474  271.828255  287.890586  304.123557  320.499511  336.978513  353.512794  370.056935  386.571778  403.021707  419.371637  435.586305  451.630823  467.471153  483.074132  498.407232  513.438341  528.135652  542.467621   556.40293   569.91041  582.958937  595.517315  607.554132   619.03758  629.935217  640.213631  649.837989  658.771402  666.974042  674.401938  681.005295  686.726228  691.495699  695.229494  697.823081  699.145348  699.031537  697.276414  693.629832  687.798535   679.45971  668.291882  654.024844   636.50017  615.721102  591.866761  565.259706  536.299177  505.385381 | 240.298231 |     0.0 | 505.385381 |   5.785498 |
    | 2000-01-02 13:30:00 |  25.718244 | 5.606729   5.24848   4.89365    4.5386  4.180797  3.818931  3.452757   3.082519   2.708397   2.330389   1.948432   1.562488   1.172555   0.778641   0.380754  -0.021109  -0.426949   -0.83677  -1.250576  -1.668374  -2.090172  -2.515981  -2.945811  -3.379676  -3.817586  -4.259548  -4.705563   -5.155617   -5.609678   -6.067681    -6.52951    -6.99498   -7.463793   -7.935499   -8.409421   -8.884563   -9.359491   -9.832177  -10.299821  -10.758676  -11.203923  -11.629698  -12.029376  -12.396201  -12.724247  -13.009459  -13.250422  -13.448526  -13.607499   -13.732612 | 224.612057  239.402418  254.459256  269.764668  285.296393  301.024851  316.909415  332.903432  348.962981  365.049583   381.12757  397.161665   413.11664  428.957881  444.651729  460.165399  475.466712  490.523894  505.305489   519.78031  533.917387  547.685871  561.054918  573.993553  586.470501  598.453974  609.911387  620.808966  631.111223  640.780225  649.774614  658.048262  665.548469  672.213557  677.969738  682.727121  686.374838  688.775463  689.759337  689.120133  686.614127  681.966907  674.891936  665.123929  652.464691  636.830521  618.282928  597.026041  573.369387  547.671346  520.290669 | 224.612057 |     0.0 | 520.290669 |   5.606729 |
    | 2000-01-02 14:00:00 |   40.60365 | 5.434685  5.075838  4.720726  4.365661  4.008144  3.646945  3.281851   2.913051   2.540654   2.164621   1.784873   1.401364   1.014079   0.623015   0.228165  -0.170479  -0.572926  -0.979185  -1.389262  -1.803166  -2.220906  -2.642491  -3.067931  -3.497231  -3.930396  -4.367422  -4.808294   -5.252978   -5.701414   -6.153499   -6.609067   -7.067864   -7.529506   -7.993425   -8.458796   -8.924441   -9.388704    -9.84931  -10.303208  -10.746448  -11.174138  -11.580568  -11.959607  -12.305364  -12.613063  -12.879866  -13.105343  -13.291418  -13.441847   -13.561541 | 210.365288  224.264037  238.482329  253.002565  267.802502  282.851094  298.106091  313.520504  329.050963  344.659505   360.31081   375.97002   391.60246  407.174073  422.651675   438.00287  453.195847  468.199223  482.981954  497.513279  511.762633  525.699544  539.293501  552.513796  565.329339  577.708405  589.618313  601.024983  611.892341  622.181527  631.849817  640.849199  649.124488  656.610877  663.230838  668.890316   673.47432  676.842255  678.823819  679.217035  677.790874   674.29575  668.484889  660.147166  649.146556  635.456925  619.178071   600.52392  579.785664   557.28401   533.32851 | 210.365288 |     0.0 |  533.32851 |   5.434685 |
    | 2000-01-02 14:30:00 |  53.087599 | 5.269903  4.909692  4.553564  4.197799  3.839938  3.478828  3.114265   2.746377   2.375204   2.000668   1.622674   1.241161   0.856099   0.467469   0.075251  -0.320573  -0.720018  -1.123098  -1.529826  -1.940212  -2.354266  -2.771997  -3.193412  -3.618512  -4.047293  -4.479741  -4.915823   -5.355484   -5.798634   -6.245128   -6.694749   -7.147171   -7.601918   -8.058306   -8.515363   -8.971733   -9.425559    -9.87435   -10.31486  -10.743017  -11.153953  -11.542221  -11.902247  -12.228989  -12.518694  -12.769509  -12.981734  -13.157625  -13.300871   -13.416017 |  197.47692  210.480558  223.851233  237.571737  251.619429  265.961885   280.55615   295.35596   310.31903  325.407934  340.587472   355.82295   371.08007  386.325308  401.526053  416.650466   431.66727  446.545601  461.254922  475.764954  490.045589  504.066778  517.798383  531.210015  544.270812  556.949169  569.212373  581.026116  592.353856  603.155961   613.38858  623.002167  631.939573  640.133626  647.504165  653.954547  659.367841  663.603202  666.493404  667.845175  667.444644  665.070459  660.516212  653.620798  644.300653  632.573581  618.564055  602.486088  584.609188  565.219303  544.585336 |  197.47692 |     0.0 | 544.585336 |   5.269903 |
    | 2000-01-02 15:00:00 |  63.440271 | 5.112849  4.750518  4.392652  4.035509  3.676685   3.31509  2.950515   2.583017   2.212571    1.83906   1.462367   1.082416   0.699157   0.312552  -0.077432  -0.470823  -0.867646  -1.267923   -1.67167  -2.078903  -2.489632  -2.903866  -3.321609   -3.74286  -4.167604  -4.595815  -5.027444   -5.462414     -5.9006   -6.341818   -6.785792   -7.232126   -7.680251   -8.129365   -8.578356   -9.025704   -9.469369   -9.906682  -10.334259  -10.747978   -11.14308  -11.514453  -11.857108    -12.1668  -12.440658  -12.677615  -12.878498  -13.045756  -13.182957   -13.294296 | 185.861607  197.974587   210.49647   223.41078  236.694478  250.314008  264.226201  278.385843  292.751843  307.287403  321.957617   336.72813  351.565138  366.435691  381.307753  396.150038  410.931818  425.622789  440.192976  454.612656  468.852251  482.882199  496.672806  510.194053  523.415375   536.30535  548.831316  560.958851   572.65109  583.867845  594.564437  604.690212  614.186659  622.985088  631.003875  638.145379  644.292837  649.307835  653.029437  655.276547  655.855525  654.574803  651.266865  645.814941  638.178048  628.405861  616.636743  603.078511  587.978253  571.590693  554.151034 | 185.861607 |     0.0 | 554.151034 |   5.112849 |
    | 2000-01-02 15:30:00 |   71.88991 | 4.963904  4.598719  4.238408  3.879227  3.518831  3.156187  2.791062    2.42344   2.053229   1.680276   1.304439   0.925621    0.54375    0.15877  -0.229369  -0.620708  -1.015281  -1.413119  -1.814245  -2.218676  -2.626429  -3.037511  -3.451925  -3.869663  -4.290703  -4.715007  -5.142508   -5.573104   -6.006637    -6.44288   -6.881501   -7.322028   -7.763803   -8.205911   -8.647105   -9.085715   -9.519545   -9.945787  -10.360971   -10.76099  -11.141261  -11.497047  -11.823942  -12.118426  -12.378373  -12.603335  -12.794513   -12.95445  -13.086566   -13.194733 | 175.431719  186.666133  198.345561  210.454621  222.970018  235.857447     249.074  262.575764  276.322895  290.279299  304.410502   318.68264  333.062517  347.517832   362.01718  376.529877  391.025793  405.475222  419.848792  434.117367  448.251939  462.223482  476.002796  489.560302  502.865798  515.888137  528.594817  540.951438  552.921004  564.463018  575.532322   586.07763  596.039722  605.349271  613.924373  621.667939  628.465328  634.182917  638.668662  641.756142  643.273627  643.059207  640.981271  636.961004  630.990907  623.142748  613.561059   602.44376  590.016076  576.505028  562.117632 | 175.431719 |     0.0 | 562.117632 |   4.963904 |
    | 2000-01-02 16:00:00 |  78.629824 | 4.823359  4.454622  4.091184  3.729325  3.366768   3.00252  2.636316   2.268063   1.897605    1.52475   1.149331   0.771225   0.390336   0.006586  -0.380089  -0.769747  -1.162434  -1.558188  -1.957041  -2.359015   -2.76413  -3.172394  -3.583807  -3.998358  -4.416016   -4.83673  -5.260415   -5.686943   -6.116125   -6.547687    -6.98124   -7.416243   -7.851945   -8.287326   -8.721015   -9.151208    -9.57558   -9.991223  -10.394626  -10.781753  -11.148242  -11.489754  -11.802437  -12.083419  -12.331191  -12.545785  -12.728662  -12.882395  -13.010244   -13.115786 | 166.099094  176.473863  187.323873  198.635219  210.384618  222.537488  235.051652  247.884709  260.998126  274.356599   287.92623  301.673778   315.56675  329.573568  343.663511  357.806558  371.973229  386.134464  400.261535  414.325936  428.299267  442.153084  455.858729  469.387121  482.708497  495.792079  508.605641  521.114969  533.283143  545.069642  556.429205  567.310421  577.654032  587.390959  596.440159  604.706523  612.079262  618.431484  623.622002  627.500621  629.918038  630.740666  629.868971   627.25574  622.919047  616.945132   609.47935  600.707781  590.835039  580.063743  568.577124 | 166.099094 |     0.0 | 568.577124 |   4.823359 |
    | 2000-01-02 16:30:00 |  83.824356 | 4.691416  4.318473  3.951265  3.586118   3.22083   2.85444  2.486639   2.117257    1.74608   1.372874   0.997442   0.619633   0.239327  -0.143576  -0.529162  -0.917502  -1.308657  -1.702674  -2.099592  -2.499443  -2.902248  -3.308017  -3.716749  -4.128427  -4.543014  -4.960444  -5.380615   -5.803374   -6.228497   -6.655666   -7.084438   -7.514199   -7.944114   -8.373063   -8.799563   -9.221696   -9.637034  -10.042603  -10.434894  -10.809985  -11.163763  -11.492288  -11.792227  -12.061269  -12.298416  -12.504065  -12.679853  -12.828339  -12.952621   -13.056014 | 157.776495   167.31651  177.356012   187.88297  198.874498  210.296269  222.107291  234.266877  246.737887   259.48592  272.477794  285.681016  299.063888  312.595611  326.246211  339.986381   353.78734  367.620727  381.458498  395.272818  409.035938   422.72004  436.297062  449.738486  463.015063   476.09648  488.950924  501.544529  513.840677  525.799121  537.374885  548.516944  559.166664  569.256059  578.705998  587.424623  595.306442  602.232808   608.07472  612.698954  615.978234  617.805169  618.108148  616.865658  614.114744  609.950352  604.515093  597.982426  590.537945  582.362855  573.620149 | 157.776495 |     0.0 | 573.620149 |   4.691416 |
    | 2000-01-02 17:00:00 |  87.613734 | 4.568188  4.190443  3.818865  3.449856  3.081295  2.712244  2.342344   1.971352   1.598993   1.224995   0.849129   0.471214   0.091101  -0.291334  -0.676195  -1.063573  -1.453539  -1.846156  -2.241471  -2.639522  -3.040335  -3.443923  -3.850282  -4.259393  -4.671209  -5.085653  -5.502605   -5.921884   -6.343235   -6.766298   -7.190573    -7.61538   -8.039805   -8.462634   -8.882287    -9.29675   -9.703519  -10.099582  -10.481473  -10.845409  -11.187546  -11.504331    -11.7929  -12.051425  -12.279326  -12.477277  -12.647027  -12.791093  -12.912415   -13.014076 | 150.378786  159.114064   168.36703  178.127956  188.374815  199.074115  210.186487  221.673159  233.498492  245.629119  258.032722  270.677674  283.533153  296.569203  309.756649  323.066951  336.472076  349.944402  363.456612  376.981581  390.492253  403.961484  417.361864  430.665498  443.843738  456.866839  469.703531  482.320475  494.681585  506.747179  518.472957  529.808773  540.697242  551.072248  560.857508  569.965487  578.297144  585.743161  592.187491   597.51396  601.616258  604.410633  605.849266  605.931114  604.706851  602.275876    598.7758  594.367414  589.219004  583.493022  577.335096 | 150.378786 |     0.0 | 577.335096 |   4.568188 |
    | 2000-01-02 17:30:00 |  90.118017 | 4.453694  4.070622  3.694131  3.320732  2.948389  2.576181  2.203699    1.83063   1.456643   1.081425   0.704713   0.326295  -0.054006  -0.436339  -0.820833  -1.207593  -1.596709  -1.988254  -2.382287  -2.778853  -3.177984  -3.579694  -3.983981   -4.39082  -4.800158  -5.211905  -5.625923   -6.042008   -6.459871   -6.879111   -7.299176   -7.719323   -8.138563   -8.555602   -8.968774   -9.375987   -9.774684  -10.161848  -10.534076  -10.887751  -11.219297   -11.52553  -11.804006  -12.053309  -12.273192   -12.46454  -12.629168  -12.769532  -12.888426   -12.988723 | 143.823868  151.788714  160.283412  169.300966   178.82074   188.81067  199.233434  210.052382   221.23351  232.744639   244.55446  256.632331  268.948388  281.473579  294.179578   307.03866  320.023589   333.10752  346.263897  359.466349   372.68855  385.904077  399.086223  412.207788  425.240804  438.156199  450.923382  463.509717  475.879871  487.995022   499.81191   511.28173  522.348922  532.949923  543.012095  552.453096  561.181195  569.097098     576.098  582.084345  586.969306  590.690034  593.218632  594.570046  594.804368  594.022395  592.355412   589.95195  586.964685  583.539655  579.807493 | 143.823868 |     0.0 | 579.807493 |   4.453694 |
    | 2000-01-02 18:00:00 |  91.440297 | 4.347871  3.959024  3.577143  3.198875   2.82228   2.44645  2.070928   1.695334   1.319287   0.942434   0.564477   0.185172  -0.195689  -0.578281  -0.962754  -1.349234  -1.737827   -2.12862  -2.521683   -2.91707  -3.314819  -3.714947  -4.117453  -4.522308  -4.929453  -5.338786   -5.75015    -6.16332   -6.577976   -6.993676   -7.409821   -7.825608   -8.239983   -8.651577   -9.058654   -9.459063   -9.850213  -10.229105  -10.592424  -10.936725  -11.258702  -11.555508  -11.825071  -12.066324  -12.279286  -12.464997  -12.625306    -12.7626  -12.879532   -12.978794 | 138.033385  145.265579  153.033838  161.334314  170.148317  179.445808  189.191919  199.352337  209.894853  220.788639  232.003554  243.510055  255.279316   267.28325  279.494438  291.886014  304.431578  317.105098  329.880821  342.733155   355.63655  368.565344  381.493589  394.394841  407.241891  420.006447  432.658729  445.166968  457.496803  469.610541  481.466303  493.017047  504.209535   514.98335  525.270145   534.99344  544.069395  552.409097   559.92289  566.527033   572.15246  576.754515  580.321733  582.881323  584.499531   585.27643  585.336296  584.816076  583.854453  582.583098  581.119609 | 138.033385 |     0.0 | 581.119609 |   4.347871 |
    | 2000-01-02 18:30:00 |  91.669313 | 4.250569  3.855588  3.467914  3.084359  2.703086  2.323204   1.94421   1.565667   1.187147   0.808259   0.428671   0.048105  -0.333677  -0.716877  -1.101667  -1.488196  -1.876585  -2.266938  -2.659335  -3.053841    -3.4505  -3.849334  -4.250342  -4.653493  -5.058722  -5.465916  -5.874903   -6.285434   -6.697161   -7.109606   -7.522123   -7.933858   -8.343694   -8.750204   -9.151591   -9.545661    -9.92981  -10.301074  -10.656239  -10.992042   -11.30543  -11.593867  -11.855606  -12.089869  -12.296891  -12.477822   -12.63452  -12.769301   -12.88469   -12.983212 | 132.933235  139.473243   146.54976  154.162451  162.295136  170.920364   180.00611  189.520622  199.433666  209.715946  220.338645  231.273438  242.492617  253.969129   265.67651  277.588803  289.680481  301.926362  314.301521  326.781185  339.340609  351.954935  364.599024  377.247245   389.87323  402.449561   414.94739  427.335977  439.582128  451.649529  463.497984   475.08258  486.352833   497.25195  507.716381  517.675978  527.055139    535.7754  543.759863  550.939573  557.261406  562.696337  567.246285  570.947674  573.870448  576.112519  577.790921  579.031831  579.961438  580.698827   581.35021 | 132.933235 |     0.0 |  581.35021 |   4.250569 |
    | 2000-01-02 19:00:00 |   90.88159 | 4.161564  3.760183  3.366393  2.977198  2.590873  2.206547  1.823682   1.441792   1.060407   0.679102    0.29751  -0.084678   -0.46773  -0.851876  -1.237314   -1.62421  -2.012707  -2.402921  -2.794949  -3.188863  -3.584716  -3.982536  -4.382322  -4.784043  -5.187627  -5.592952  -5.999831   -6.407997   -6.817072   -7.226546   -7.635731   -8.043726   -8.449363    -8.85116   -9.247277   -9.635488  -10.013195  -10.377481  -10.725245    -11.0534  -11.359135  -11.640195  -11.895111  -12.123347  -12.325308  -12.502221  -12.655941  -12.788706  -12.902924   -13.000976 | 128.453911  134.344121    140.7658  147.722421  155.200828  163.176684  171.621167  180.505313  189.801058  199.480831  209.517299  219.883471  230.552851  241.499481  252.697911  264.123137  275.750541  287.555823  299.514917  311.603895  323.798853  336.075772  348.410363  360.777869  373.152828  385.508784  397.817939  410.050719  422.175275  434.156889  445.957314  457.534072  468.839785  479.821659  490.421308  500.575212  510.216142  519.275928   527.68984  535.402534  542.375032  548.591597  554.064934  558.838247  562.983344  566.595068  569.783276  572.664238  575.353022  577.957673  580.574428 | 128.453911 |     0.0 | 580.574428 |   4.161564 |
    | 2000-01-02 19:30:00 |  89.143206 | 4.080563  3.672612  3.272468  2.877349  2.485654  2.096539  1.709441   1.323835   0.939218   0.555132   0.171182  -0.212976  -0.597637  -0.983058  -1.369461  -1.757035  -2.145941  -2.536312  -2.928257  -3.321862  -3.717186  -4.114263  -4.513096  -4.913653  -5.315857  -5.719579  -6.124617   -6.530687   -6.937386   -7.344173   -7.750326   -8.154899   -8.556682   -8.954149   -9.345425   -9.728271    -10.1001  -10.458059  -10.799159   -11.12049  -11.419463  -11.694069  -11.943083  -12.166168  -12.363859  -12.537439  -12.688744   -12.81994  -12.933326   -13.031155 | 124.530701  129.814695  135.620028  141.954166  148.807424  156.159038  163.983705   172.25548  180.948674  190.037636   199.49668  209.300274  219.423225  229.840743  240.528446   251.46233  262.618722  273.974235  285.505688  297.190026   309.00421  320.925096  332.929284  344.992936   357.09156  369.199743  381.290828  393.336535  405.306512  417.167822  428.884382  440.416403  451.719885  462.746304  473.442682   483.75227  493.616175  502.976199  511.779048  519.981771  527.557824  534.502678  540.837646  546.610792  551.894469  556.779915  561.370064  565.772157  570.091375  574.426072  578.863717 | 124.530701 |     0.0 | 578.863717 |   4.080563 |
    | 2000-01-02 20:00:00 |  86.511247 | 4.007216   3.59262  3.185969  2.784717  2.387394  1.993195  1.601541   1.211884   0.823694   0.436487    0.04984  -0.336618  -0.723212  -1.110225  -1.497903  -1.886456  -2.276063  -2.666877   -3.05902  -3.452589  -3.847653  -4.244253  -4.642396  -5.042048  -5.443132   -5.84551   -6.24897    -6.65321   -7.057808   -7.462194   -7.865615   -8.267089   -8.665369   -9.058895   -9.445769   -9.823745  -10.190265   -10.54254    -10.8777  -11.192998  -11.486049  -11.755065  -11.999024  -12.217753  -12.411892  -12.582756  -12.732156  -12.862187  -12.975052   -13.072886 | 121.103761  125.825629  131.054111  136.800722  143.059591  149.813896   157.04213  164.721578  172.829135  181.341266    190.2341   199.48369   209.06624  218.958206  229.136335  239.577666  250.259513  261.159426  272.255137  283.524486  294.945326  306.495413  318.152269  329.893018  341.694188  353.531471  365.379441  377.211211  388.998052  400.708956  412.310182  423.764817  435.032435  446.068969  456.826969  467.256477  477.306768  486.929165  496.081009  504.730532  512.862041  520.480407   527.61376  534.313539  540.651709  546.715644  552.601731  558.409027  564.233939    570.1663   576.28587 | 121.103761 |     0.0 |  576.28587 |   4.007216 |
    | 2000-01-02 20:30:00 |  83.035021 | 3.941122  3.519898  3.106675  2.699156   2.29601  1.896484  1.499999   1.105992   0.713918   0.323274  -0.066386   -0.45546  -0.844298  -1.233207  -1.622457  -2.012279  -2.402873  -2.794407  -3.187019  -3.580819  -3.975886  -4.372267  -4.769975  -5.168977  -5.569195  -5.970484  -6.372624   -6.775299   -7.178068   -7.580338   -7.981329   -8.380029   -8.775162   -9.165141   -9.548054    -9.92166  -10.283434  -10.630659  -10.960578  -11.270601  -11.558526  -11.822754   -12.06244  -12.277543   -12.46878  -12.637488  -12.785446  -12.914683  -13.027312   -13.125365 | 118.118088  122.321801  127.013383  132.208317  137.904766  144.090113  150.746863  157.855719  165.396362  173.347567  181.687439   190.39375  199.444202  208.816577  218.488814  228.439046  238.645606  249.087017  259.741944  270.589143  281.607381  292.775336  304.071478  315.473927  326.960273  338.507367  350.091074  361.685987  373.265096  384.799436  396.257735  407.606094  418.807788  429.823301  440.610745  451.126861  461.328819  471.176942   480.63837   489.69136  498.329645  506.565951   514.43377  521.986788  529.295951  536.444704   543.52333  550.623508   557.83386  565.236726  572.905088 | 118.118088 |     0.0 | 572.905088 |   3.941122 |
    | 2000-01-02 21:00:00 |  78.757067 | 3.881845  3.454095  3.034316  2.620471  2.211375  1.806336   1.40479   1.006175    0.60994    0.21557  -0.177399  -0.569384   -0.96076  -1.351857  -1.742964  -2.134336  -2.526191  -2.918714  -3.312059  -3.706349  -4.101674  -4.498088   -4.89561  -5.294211   -5.69381  -6.094262  -6.495337   -6.896707   -7.297918   -7.698356   -8.097219   -8.493472   -8.885814   -9.272642   -9.652037   -10.02177  -10.379355  -10.722149  -11.047506  -11.352976  -11.636522  -11.896713  -12.132847  -12.344993  -12.533925  -12.700989  -12.847929  -12.976713  -13.089373   -13.187847 | 115.523427  119.252256  123.446841  128.126408  133.293223  138.939027  145.050479  151.611855  158.605805  166.013597  173.815472  181.991052  190.519644  199.380441  208.552642  218.015524  227.748482  237.731036  247.942812  258.363502    268.9728  279.750327  290.675523  301.727526  312.885018  324.126049  335.427829  346.766484  358.116794  369.451922  380.743146  391.959663  403.068514  414.034758  424.822015  435.393555  445.714098  455.752397  465.484561  474.897795  483.993999  492.792449  501.330837  509.664255   517.86223   526.00435  534.175269   542.46004   550.94039  559.692055  568.782078 | 115.523427 |     0.0 | 568.782078 |   3.881845 |
    | 2000-01-02 21:30:00 |  73.714011 | 3.828924  3.394825  2.968585  2.548427  2.133319  1.722639  1.315853   0.912415   0.511777   0.113423  -0.283124  -0.678297  -1.072488  -1.466048  -1.859287  -2.252478  -2.645859  -3.039632  -3.433966  -3.828997  -4.224827   -4.62152  -5.019099  -5.417541  -5.816766  -6.216625  -6.616885   -7.017209   -7.417129   -7.816017   -8.213053   -8.607185   -8.997094   -9.381166   -9.757483  -10.123836   -10.47778  -10.816745   -11.13819  -11.439796   -11.71967  -11.976523  -12.209775  -12.419579  -12.606754  -12.772644  -12.918959  -13.047607  -13.160548   -13.259637 | 113.274094  116.570121  120.307102  124.507657  129.178079  134.314497   139.90778  145.945888  152.414592  159.297816  166.578095  174.237027  182.255638  190.614618  199.294497  208.275753  217.538883  227.064438  236.833026  246.825292  257.021877  267.403355  277.950151  288.642437  299.460011  310.382147  321.387428  332.453554  343.557141  354.673523  365.776577  376.838626  387.830489  398.721756  409.481438  420.079099  430.486615  440.680583   450.64528  460.375875  469.881337  479.186411  488.332072  497.374206  506.380699  515.427442  524.593945  533.959336  543.599238  553.583568  563.974181 | 113.274094 |     0.0 | 563.974181 |   3.828924 |
    | 2000-01-02 22:00:00 |  67.937278 | 3.781883  3.341675  2.909139   2.48275  2.061632  1.645243  1.233092    0.82466   0.419417   0.016851  -0.383517  -0.782131  -1.179395  -1.575678  -1.971309  -2.366577  -2.761738  -3.157012  -3.552582  -3.948599  -4.345174  -4.742384  -5.140257  -5.538777  -5.937865  -6.337373  -6.737064   -7.136595   -7.535489   -7.933108   -8.328617   -8.720951   -9.108781   -9.490491   -9.864166  -10.227622  -10.578459   -10.91418  -11.232339  -11.530735  -11.807602  -12.061771  -12.292763  -12.500798  -12.686722  -12.851876   -12.99793  -13.126739  -13.240198   -13.340088 | 111.328777  114.232462  117.550317  121.307881  125.515271  130.172898  135.275816  140.815719   146.78161  153.160213  159.936492  167.094152  174.616042  182.484448  190.681302  199.188343  207.987216  217.059534  226.386914  235.950973  245.733308  255.715459  265.878844  276.204681  286.673891  297.266988  307.963943  318.744049  329.585777  340.466646  351.363141  362.250706  373.103889  383.896725  394.603441  405.199614  415.663857  425.980017  436.139796  446.145435  456.012033  465.768913  475.459624  485.140419  494.877444   504.74312  514.812295  525.158814  535.852902  546.959351  558.535514 | 111.328777 |     0.0 | 558.535514 |   3.781883 |
    | 2000-01-02 22:30:00 |  61.453708 | 3.740242  3.294219   2.85561  2.423135  1.996072  1.573963  1.156375   0.742826   0.332815  -0.074155  -0.478557  -0.880839  -1.281415  -1.680665  -2.078931  -2.476522  -2.873708  -3.270724  -3.667769  -4.065007  -4.462562  -4.860519  -5.258918  -5.657747  -6.056931  -6.456323  -6.855686   -7.254674   -7.652803   -8.049428   -8.443706   -8.834563   -9.220667     -9.6004   -9.971863  -10.332896  -10.681147  -11.014187  -11.329659  -11.625467  -11.899955  -12.152054  -12.381367  -12.588164   -12.77331  -12.938134   -13.08427  -13.213521  -13.327726   -13.428598 | 109.650285  112.200131  115.136057  118.485978  122.263497  126.473089  131.113872  136.181259  141.667547  147.562369  153.853236  160.526078  167.565675     174.956   182.68047  190.722146  199.063868  207.688351  216.578243  225.716152  235.084647  244.666238  254.443342  264.398224  274.512933  284.769221  295.148451  305.631507  316.198709  326.829755   337.50371  348.199089  358.894094   369.56706  380.197221  390.765857  401.257875  411.663812  421.982087  432.221237  442.401683  452.556609  462.731597  472.982982  483.375171   493.97738  504.860261  516.092967  527.740961  539.864518  552.517126 | 109.650285 |     0.0 | 552.517126 |   3.740242 |
    | 2000-01-02 23:00:00 |  54.286068 | 3.703526  3.252021  2.807614  2.369253  1.936365  1.508583  1.085537   0.666796   0.251897  -0.159632  -0.568251  -0.974403  -1.378505  -1.780945  -2.182077  -2.582222  -2.981664  -3.380653  -3.779403   -4.17809  -4.576851  -4.975781   -5.37493  -5.774292  -6.173798  -6.573305  -6.972577   -7.371266   -7.768887    -8.16479   -8.558128   -8.947825   -9.332548   -9.710686  -10.080356  -10.439429  -10.785598    -11.1165  -11.429858  -11.723669   -11.99637  -12.246977  -12.475155  -12.681212  -12.866023  -13.030899  -13.177441  -13.307403  -13.422571   -13.524604 | 108.205287  110.437577  113.027196  116.003835  119.384151  123.176362  127.383434  132.004411  137.034893  142.467485  148.292342  154.497718   161.07043  167.996227  175.260093  182.846476  190.739465  198.922914  207.380535  216.095944  225.052691  234.234267  243.624085  253.205457  262.961554  272.875355  282.929603  293.106757  303.388962  313.758055   324.19563  334.683196  345.202484  355.735966  366.267636  376.784142  387.276254  397.740645  408.181822  418.613922  429.062018  439.562576   450.16281  460.918966  471.893783  483.153559  494.765198  506.793704  519.300345  532.341424  545.967155 | 108.205287 |     0.0 | 545.967155 |   3.703526 |
    | 2000-01-02 23:30:00 |  46.453505 | 3.671272  3.214646  2.764754  2.320754  1.882213  1.448856  1.020383   0.596424   0.176561  -0.239644  -0.652628  -1.062824  -1.470644  -1.876478  -2.280687  -2.683602   -3.08552  -3.486703  -3.887377  -4.287732  -4.687917  -5.088039  -5.488155  -5.888269  -6.288319  -6.688166  -7.087577   -7.486207   -7.883571   -8.279018   -8.671703    -9.06055    -9.44423   -9.821146  -10.189433  -10.546993  -10.891568  -11.220852  -11.532645  -11.825021  -12.096497  -12.346155  -12.573709  -12.779495  -12.964389   -13.12968  -13.276935  -13.407866   -13.52421   -13.627579 | 106.964036  108.912651  111.189766  113.826231  116.841254  120.246381  124.048139  128.249034  132.847918  137.840382  143.219287  148.975296   155.09735  161.573066  168.389072  175.531273  182.985063  190.735479  198.767326  207.065251    215.6138  224.397445  233.400597  242.607608  252.002754  261.570229  271.294129  281.158455  291.147135   301.24409  311.433361  321.699345  332.027159  342.403207  352.815985  363.257161  373.722936  384.215609  394.745194  405.330828  416.001677  426.797028  437.765441  448.962988  460.450866  472.292748  484.552221  497.290637  510.565581   524.42987  538.930999 | 106.964036 |     0.0 | 538.930999 |   3.671272 |
    | 2000-01-03 00:00:00 |  37.971929 | 3.643038  3.181666  2.726631  2.277277  1.833299  1.394513  0.960695   0.531537   0.106676  -0.314282  -0.731746  -1.146128   -1.55783  -1.967239  -2.374719  -2.780605  -3.185204  -3.588789  -3.991598  -4.393831  -4.795651  -5.197174  -5.598468  -5.999547  -6.400355  -6.800761  -7.200537   -7.599341   -7.996693   -8.391946   -8.784256   -9.172556   -9.555525   -9.931581  -10.298882  -10.655364  -10.998814  -11.326982  -11.637731  -11.929206  -12.199988  -12.449212  -12.676628  -12.882586  -13.067957  -13.234009  -13.382274  -13.514423  -13.632152   -13.737032 | 105.900084  107.596403  109.592811  111.920731  114.601368  117.649126  121.073725  124.880901  129.072636  133.647476  138.601007  143.926357  149.614666    155.6555  162.037209  168.747224  175.772299  183.098706  190.712374  198.599007  206.744158  215.133288  223.751804  232.585084  241.618495  250.837418   260.22727  269.773548  279.461904  289.278267   299.20903  309.241345  319.363546  329.565754  339.840682  350.184673  360.598939  371.090925   381.67563  392.376675  403.226849  414.267908  425.549558  437.127681  449.062091  461.414143  474.244505   487.61134  501.569013  516.167289  531.451474 | 105.900084 |     0.0 | 531.451474 |   3.643038 |
    | 2000-01-03 00:30:00 |  28.854347 | 3.618402  3.152669  2.692849  2.238453  1.789291  1.345266  0.906226   0.471934   0.042086  -0.383663  -0.805685  -1.224364  -1.640086  -2.053228   -2.46415  -2.873191  -3.280662  -3.686845  -4.091986  -4.496299  -4.899955  -5.303082  -5.705758  -6.108007  -6.509782  -6.910959  -7.311319   -7.710525   -8.108104   -8.503416   -8.895625   -9.283671   -9.666251  -10.041798  -10.408498   -10.76432  -11.107095  -11.434624   -11.74483  -12.035912  -12.306504  -12.555782  -12.783519  -12.990072  -13.176298  -13.343443  -13.493005  -13.626616  -13.745933   -13.852502 |  104.99001  106.462861  108.208237  110.257573  112.633522  115.352826  118.427979  121.867651   125.67677  129.856748  134.405873  139.319762  144.591808  150.213594  156.175262  162.465838  169.073498  175.985791  183.189815  190.672353  198.419986  206.419168  214.656297  223.117767  231.790015  240.659577  249.713147  258.937665  268.320438   277.84931  287.512911  297.301001  307.204944  317.218343  327.337839  337.564105  347.902962  358.366547  368.974384  379.754128  390.741806  401.981373  413.523553  425.424076  437.741557  450.535336  463.863546  477.781581  492.341007  507.588932  523.568982 |  104.99001 |     0.0 | 523.568982 |   3.618402 |
    | 2000-01-03 01:00:00 |  19.111158 | 3.596973  3.127258  2.663021  2.203916  1.749852  1.300811  0.856714   0.417394   -0.01739  -0.447928   -0.87455  -1.297604  -1.717454  -2.134461  -2.548977  -2.961337  -3.371854  -3.780816  -4.188478   -4.59506  -5.000744  -5.405669  -5.809924  -6.213541  -6.616484  -7.018639  -7.419795   -7.819624   -8.217662   -8.613279   -9.005649   -9.393729   -9.776229  -10.151607  -10.518075  -10.873639  -11.216171  -11.543521  -11.853658  -12.144831  -12.415714   -12.66551  -12.894007  -13.101556  -13.289002   -13.45756  -13.608697  -13.744009  -13.865119   -13.973555 | 104.213141  105.488824  107.010645  108.809559  110.909127  113.327899  116.080688  119.178746  122.629708  126.437707  130.603662  135.125668  139.999399  145.218502  150.774976  156.659502  162.861738  169.370563  176.174281   183.26079   190.61771  198.232502  206.092552  214.185255  222.498095  231.018727  239.735077  248.635464  257.708768  266.944647  276.333837  285.868547  295.542972  305.353949  315.301759  325.391058  335.631889  346.040671   356.64103   367.46427  378.549344  389.942199   401.69451  413.861927  426.502073  439.672571   453.42938  467.825515  482.910131  498.728036  515.321666 | 104.213141 |     0.0 | 515.321666 |   3.596973 |
    | 2000-01-03 01:30:00 |   8.750417 | 3.578386   3.10506  2.636775  2.173302  1.714638  1.260834   0.81188   0.367676  -0.071953  -0.507241  -0.938466  -1.365942  -1.789998  -2.210975  -2.629211  -3.045035  -3.458756  -3.870665   -4.28102   -4.69005  -5.097945  -5.504854  -5.910876  -6.316052  -6.720358   -7.12369  -7.525846   -7.926512   -8.325234   -8.721393    -9.11418   -9.502568   -9.885287  -10.260822  -10.627414  -10.983105  -11.325807  -11.653413  -11.963935   -12.25566  -12.527291  -12.778049  -13.007724  -13.216659  -13.405674  -13.575958  -13.728942  -13.866192  -13.989298   -14.099783 | 103.551298  104.653645  105.977171  107.551932  109.401893  111.546894   114.00359  116.785429  119.902467  123.361353  127.165525  131.315502  135.809232   140.64246  145.809085    151.3015    157.1109  163.227545  169.640991  176.340277  183.314092  190.550908  198.039101  205.767055  213.723277  221.896503  230.275838  238.850912   247.61209  256.550733  265.659547  274.933016  284.367956  293.964192  303.725345  313.659722  323.781217  334.110147  344.673863  355.506985  366.651135  378.154104  390.068491  402.449973  415.355391  428.840951  442.960779   457.76587  473.303326  489.615983  506.745569 | 103.551298 |     0.0 | 506.745569 |   3.578386 |
    | 2000-01-03 02:00:00 | 136.813595 | 3.562306  3.085726  2.613753  2.146259   1.68331  1.225019  0.771435   0.322524  -0.121824  -0.561785  -0.997585  -1.429491  -1.857801  -2.282826  -2.704886  -3.124296   -3.54136  -3.956366  -4.369575   -4.78122  -5.191498  -5.600567  -6.008534  -6.415453  -6.821308  -7.226008  -7.629364   -8.031072   -8.430693   -8.827624   -9.221073   -9.610033   -9.993259  -10.369263  -10.736318  -11.092503  -11.435769  -11.764046  -12.075385  -12.368101  -12.640916  -12.893061  -13.124317  -13.335011  -13.525936  -13.698249  -13.853351  -13.992774   -14.11808   -14.230787 | 102.988541  103.939105  105.087416  106.462365  108.087875  109.984581  112.170509  114.660894  117.467879  120.600392  124.064223   127.86222  131.994569  136.459112  141.251677  146.366416  151.796109  157.532445  163.566274  169.887815  176.486848  183.352871  190.475249  197.843348   205.44667  213.274997  221.318557  229.568221  238.015746  246.654079  255.477749  264.483339  273.670072  283.040504  292.601309  302.364111  312.346307  322.571751  333.071181  343.882243  355.049016  366.621027  378.651812  391.197193  404.313454  418.055672  432.476464  447.625146  463.547091  480.283472  497.873909 | 102.988541 |     0.0 | 497.873909 |   3.562306 |
    | 2000-01-03 02:30:00 | 126.146668 | 3.548432  3.068939  2.593627  2.122451  1.655537  1.193046  0.735083   0.281671  -0.167239  -0.611762  -1.052072  -1.488389  -1.920969  -2.350092  -2.776052  -3.199149  -3.619675  -4.037911  -4.454119  -4.868532  -5.281354  -5.692748  -6.102832  -6.511667  -6.919251  -7.325503  -7.730248   -8.133196   -8.533923   -8.931846   -9.326191   -9.715975  -10.099982   -10.47675  -10.844592  -11.201619   -11.54582  -11.875163  -12.187727  -12.481851  -12.756267  -13.010204  -13.243431   -13.45625  -13.649423  -13.824071  -13.981563  -14.123397  -14.251111   -14.366168 |  102.51094  103.328797  104.322441  105.519663  106.943998  108.616383  110.555703  112.778544  115.298781  118.127344  121.272171  124.738308  128.528101  132.641459  137.076149  141.828105  146.891733  152.260195  157.925674  163.879604   170.11288  176.616045  183.379455  190.393445  197.648491  205.135381  212.845411  220.770614   228.90404    237.2401  245.774989  254.507201  263.438138  272.572824  281.920676  291.496306  301.320255  311.419558   321.82801  332.586005  343.739895  355.340876  367.443539  380.104192  393.379013  407.322405  421.986006  437.417778  453.661389  470.756507  488.734462 |  102.51094 |     0.0 | 488.734462 |   3.548432 |
    | 2000-01-03 03:00:00 | 114.923475 | 3.536484  3.054393  2.576076  2.101554  1.630996  1.164603  0.702526    0.24484  -0.208445  -0.657391  -1.102115  -1.542789  -1.979623  -2.412864  -2.842777  -3.269637  -3.693722  -4.115304  -4.534639  -4.951961  -5.367474  -5.781348  -6.193708  -6.604626  -7.014108  -7.422088  -7.828403    -8.23278   -8.634812   -9.033936   -9.429401   -9.820251  -10.205297  -10.583114  -10.952047  -11.310247  -11.655739  -11.986523  -12.300704  -12.596636  -12.873054  -13.129174  -13.364746  -13.580039   -13.77578  -13.953057    -14.1132   -14.25768  -14.388009   -14.505601 | 102.106358  102.808974  103.667508  104.707885   105.95289  107.423501  109.139173   111.11748  113.373658  115.920306  118.767257  121.921576   125.38766  129.167429  133.260567  137.664805  142.376203  147.389435  152.698059  158.294759  164.171579  170.320121  176.731745  183.397753  190.309575  197.458972  204.838257  212.440556  220.260121   228.29271   236.53604   244.99034  253.658985  262.549227  271.672957  281.047483  290.696191  300.649022  310.942593  321.619896  332.729486  344.324216  356.459617  369.192154  382.577627  396.669856  411.519566  427.173581  443.674646  461.061388  479.364497 | 102.106358 |     0.0 | 479.364497 |   3.536484 |
    | 2000-01-03 03:30:00 | 103.143875 | 3.526218  3.041818   2.56081  2.083263  1.609378  1.139383  0.673468   0.211754  -0.245701  -0.698901  -1.147914   -1.59286  -2.033906  -2.471257  -2.905145   -3.33582  -3.763541  -4.188564  -4.611137  -5.031491  -5.449831  -5.866328  -6.281114  -6.694269  -7.105812  -7.515686  -7.923744    -8.32973   -8.733256    -9.13378   -9.530579   -9.922722  -10.309055  -10.688187  -11.058501  -11.418186  -11.765304  -12.097883  -12.414051   -12.71217   -12.99097   -13.24965  -13.487928  -13.706037  -13.904666  -14.084863  -14.247923  -14.395286  -14.528443   -14.648762 | 101.764251   102.36664  103.106864  104.008975  105.094725  106.384691  107.898458  109.654202  111.668183  113.954377  116.524253  119.386684  122.547967  126.011937   129.78015  133.852116  138.225563  142.896703  147.860499  153.110924  158.641193  164.443995  170.511703  176.836582  183.411002  190.227664  197.279849  204.561705  212.068596  219.797504  227.747523  235.920434  244.321375  252.959577  261.849133  271.009749  280.467356  290.254484  300.410277   310.98004  322.014311  333.567485  345.696146  358.457314  371.906871  386.098327  401.081823   416.90344  433.605079  451.224521  469.792183 | 101.764251 |     0.0 | 469.792183 |   3.526218 |
    | 2000-01-03 04:00:00 |  90.807283 | 3.517417  3.030974  2.547566  2.067297  1.590388   1.11709  0.647618   0.182132  -0.279266  -0.736532  -1.189683   -1.63879  -2.083974  -2.525398  -2.963259  -3.397777  -3.829185  -4.257725   -4.68363  -5.107124   -5.52841   -5.94766  -6.365011  -6.780549  -7.194303  -7.606228  -8.016193   -8.423959   -8.829159   -9.231273   -9.629605  -10.023257  -10.411109  -10.791807  -11.163773  -11.525237  -11.874295  -12.209004  -12.527508  -12.828175  -13.109721  -13.371321  -13.612657  -13.833918   -14.03575  -14.219158  -14.385401  -14.535889  -14.672091   -14.795335 |  101.47548  101.991006   102.62797  103.408534  104.353255  105.482034  106.814237  108.368272  110.161061  112.207623  114.520785  117.111002  119.986296  123.152294  126.612345  130.367703  134.417753  138.760258  143.391612  148.307099  153.501141  158.967531  164.699668  170.690783  176.934175  183.423457   190.15283  197.117403  204.313556  211.739384  219.395211  227.284199  235.413035  243.792686  252.439175  261.374304  270.626236  280.229797  290.226396  300.663471  311.593441  323.072246  335.157614  347.907289  361.377464  375.621592  390.689502  406.626798  423.474802  441.270676  460.044987 |  101.47548 |     0.0 | 460.044987 |   3.517417 |
    | 2000-01-03 04:30:00 |  77.912239 | 3.509886  3.021642  2.536102  2.053396  1.573755  1.097442  0.624693   0.155698  -0.309405  -0.770529  -1.227645  -1.680776  -2.129999  -2.575434  -3.017239  -3.455601  -3.890727  -4.322836  -4.752147  -5.178872  -5.603209  -6.025329   -6.44537  -6.863426  -7.279532  -7.693656  -8.105682   -8.515389    -8.92243   -9.326312   -9.726368  -10.121731  -10.511318  -10.893817   -11.26769  -11.631207  -11.982498  -12.319651  -12.640822  -12.944377  -13.229018  -13.493886  -13.738621  -13.963363  -14.168709   -14.35562  -14.525314  -14.679172  -14.818639   -14.945018 | 101.232151  101.672594  102.219696  102.893738   103.71397   104.69941  105.868936  107.240868  108.832457  110.659442  112.735705   115.07303  117.680963   120.56677  123.735489  127.190057  130.931487  134.959085  139.270692  143.862925  148.731432  153.871135  159.276473   164.94165  170.860884  177.028675  183.440109  190.091187  196.979228  204.103326  211.464898  219.068313  226.921605  235.037241  243.432903  252.132209   261.16526  270.568902  280.386572  290.667658  301.466361  312.840151  324.847978  337.548466  350.998335  365.251236  380.356943  396.360852  413.303959  431.223043   450.14924 | 101.232151 |     0.0 |  450.14924 |   3.509886 |
    | 2000-01-03 05:00:00 |  64.456407 | 3.503454  3.013627  2.526201  2.041321  1.559224  1.080175  0.604424   0.132183  -0.336379  -0.801141   -1.26203  -1.719028  -2.172167  -2.621524  -3.067219  -3.509402  -3.948252  -4.383962  -4.816732  -5.246761  -5.674238   -6.09933  -6.522175  -6.942869  -7.361457  -7.777918  -8.192149   -8.603948   -9.012989   -9.418806   -9.820762  -10.218025   -10.60955  -10.994068  -11.370083  -11.735907  -12.089707  -12.429596  -12.753743   -13.06051  -13.348577  -13.617049  -13.865515  -14.094061  -14.303229  -14.493933  -14.667349  -14.824826  -14.967789   -15.097517 | 101.027458  101.403153  101.872257  102.453216  103.163903  104.022306  105.046594  106.254739  107.664023  109.290597  111.149107  113.252399  115.611314  118.234579  121.128791  124.298477   127.74623  131.472876  135.477699  139.758663  144.312661  149.135761   154.22346  159.570938  165.173328  171.026005  177.124897  183.466849  190.050035  196.874448  203.942466  211.259501  218.834738  226.681915  234.820122  243.274516  252.076857  261.265741  270.886399  280.989994  291.632405  302.872617  314.770868  327.386809  340.777898  354.998229  370.097775  386.121983  403.111775  421.103771   440.13011 | 101.027458 |     0.0 |  440.13011 |   3.503454 |
    | 2000-01-03 05:30:00 |  50.436708 | 3.497969  3.006756  2.517668  2.030858   1.54656  1.065041  0.586555   0.111329  -0.360445  -0.828614  -1.293073  -1.753762  -2.210672   -2.66384  -3.113346  -3.559304   -4.00186   -4.44118  -4.877443  -5.310831   -5.74152   -6.16967  -6.595417  -7.018859  -7.440048  -7.858972  -8.275541   -8.689571   -9.100761   -9.508669   -9.912689  -10.312027  -10.705678  -11.092416  -11.470791  -11.839158   -12.19572  -12.538617   -12.86603  -13.176314  -13.468123  -13.740519  -13.993038  -14.225704  -14.439002   -14.63379  -14.811202  -14.972554  -15.119245   -15.252549 | 100.855551  101.175531  101.577099  102.076938  102.691517  103.437691  104.332758  105.394123  106.638845  108.083209  109.742342  111.629884  113.757734   116.13588  118.772319  121.673057  124.842185  128.282012  131.993243   135.97519     140.226  144.742906  149.522472  154.560866  159.854138  165.398517  171.190748   177.22847  183.510644  190.038066  196.813948  203.844597  211.140159  218.715431  226.590663  234.792287   243.35345  252.314239  261.721446  271.627828   282.09084  293.170953  304.929755   317.42806  330.724264  344.873141  359.925115  375.925935  392.916694  410.934079  430.011676 | 100.855551 |     0.0 | 430.011676 |   3.497969 |
    | 2000-01-03 06:00:00 |  35.849475 |   3.4933  3.000877  2.510328  2.021809  1.535549  1.051808  0.570842   0.092887  -0.381851  -0.853194  -1.321007  -1.785197  -2.245717  -2.702565  -3.155781  -3.605444  -4.051667  -4.494585  -4.934353  -5.371134  -5.805088  -6.236366    -6.6651  -7.091387  -7.515283  -7.936784  -8.355814   -8.772205   -9.185679   -9.595821  -10.002058  -10.403632  -10.799582  -11.188725   -11.56966  -11.940785  -12.300343    -12.6465  -12.977448  -13.291535  -13.587385  -13.864013  -14.120897  -14.357995  -14.575725   -14.77489  -14.956575   -15.12206  -15.272723   -15.409841 | 100.711411  100.983545  101.326769   101.75609  102.286586  102.933916  103.714391  104.644677  105.741395   107.02072  108.497999  110.187406  112.101657  114.251785  116.647011   119.29468  122.200284  125.367547  128.798562  132.493973  136.453183  140.674595  145.155855  149.894126  154.886373  160.129676  165.621571  171.360447  177.345993   183.57972  190.065565   196.81058  203.825706  211.126589  218.734407  226.676613   234.98748  243.708323  252.887268  262.578485  272.840886  283.736396  295.327998  307.677792  320.845302  334.886212  349.851642   365.78789  382.736462  400.734338  419.817009 | 100.711411 |     0.0 | 419.817009 |     3.4933 |
    | 2000-01-03 06:30:00 |  20.690623 | 3.489332  2.995854  2.504025  2.013999  1.525996  1.040266  0.557063   0.076622  -0.400836  -0.875119  -1.346066  -1.813557   -2.27751  -2.737888  -3.194694  -3.647972  -4.097799  -4.544281  -4.987547  -5.427735   -5.86499  -6.299451   -6.73124  -7.160454  -7.587149   -8.01133  -8.432931   -8.851802   -9.267685   -9.680193  -10.088786  -10.492745   -10.89115  -11.282867  -11.666543  -12.040623  -12.403389  -12.753036  -13.087769  -13.405924  -13.706097  -13.987252  -14.248802  -14.490636  -14.713101  -14.916935  -15.103174  -15.273059  -15.427943   -15.569125 | 100.590745  100.821868  101.114792  101.482959  101.940083  102.500609  103.179781  103.993401  104.957478  106.087865  107.399894  108.908031  110.625567  112.564363  114.734672   117.14502   119.80218  122.711204  125.875517  129.297062   132.97649  136.913371  141.106442  145.553866  150.253527  155.203347  160.401638  165.847506   171.54131  177.485204  183.683744  190.144602  196.879348  203.904286  211.241304  218.918631  226.971407  235.441925  244.379413  253.839264  263.881722  274.570115  285.968851  298.141425  311.148664  325.047382  339.889638  355.722521  372.588188  390.524142  409.568252 | 100.590745 |     0.0 | 409.568252 |   3.489332 |
    | 2000-01-03 07:00:00 |   4.955827 | 3.485963  2.991569  2.498622  2.007271  1.517724  1.030222  0.545006   0.062315  -0.417628  -0.894618  -1.368478  -1.839063  -2.306262  -2.770005  -3.230264  -3.687046  -4.140396  -4.590388  -5.037122  -5.480713  -5.921286  -6.358965  -6.793863  -7.226071  -7.655645  -8.082595  -8.506866   -8.928323   -9.346728   -9.761723  -10.172799  -10.579276  -10.980279  -11.374722  -11.761302  -12.138513   -12.50468  -12.858027  -13.196771  -13.519242  -13.824002  -14.109964  -14.376471  -14.623338  -14.850835  -15.059632   -15.25071  -15.425268   -15.58463   -15.730137 | 100.489885  100.685918  100.935566  101.250814  101.644071  102.128571  102.718452  103.428564   104.27417  105.270625  106.433042  107.775953  109.312999  111.056649  113.017987  115.206551  117.630254  120.295369  123.206577  126.367077  129.778734  133.442282  137.357541  141.523683  145.939513  150.603791  155.515589  160.674703  166.082122  171.740571   177.65514  183.833999  190.289201  197.037532  204.101381   211.50952  219.297711  227.508975  236.193406  245.407425  255.212467  265.673195  276.855456  288.824226  301.641789  315.366309  330.051015  345.744025  362.488391  380.322362  399.286695 | 100.489885 |     0.0 | 399.286695 |   3.485963 |
    | 2000-01-03 07:30:00 | 140.226743 | 3.483107  2.987919  2.493998  2.001485  1.510576  1.021497  0.534481    0.04976  -0.432442  -0.911914  -1.388466  -1.861934  -2.332184  -2.799119  -3.262678  -3.722837  -4.179609  -4.633036   -5.08319   -5.53016   -5.97405  -6.414966  -6.853009  -7.288263  -7.720781  -8.150577  -8.577604    -9.00174    -9.42277   -9.840359  -10.254031  -10.663147  -11.066876  -11.464181  -11.853811   -12.23431  -12.604049  -12.961283  -13.304247  -13.631259  -13.940851  -14.231883  -14.503628  -14.755817  -14.988641  -15.202695  -15.398899  -15.578408  -15.742514   -15.892519 | 100.405708  100.571859  100.784364   101.05394  101.391762  101.809882  102.321323  102.939927  103.680113  104.556601  105.584109  106.777038  108.149166  109.713358  111.481327  113.463441  115.668601  118.104175  120.776008  123.688483  126.844646  130.246363  133.894536  137.789338  141.930497  146.317607  150.950494  155.829623  160.956573  166.334589  171.969214  177.869016  184.046399  190.518467  197.307919  204.443854  211.962418  219.907113  228.328664  237.284312  246.836525  257.051211  267.995638  279.736329   292.33716  305.857836  320.352995  335.872011  352.459064  370.153401  388.987362 | 100.405708 |     0.0 | 388.987362 |   3.483107 |
    | 2000-01-03 08:00:00 |  124.76231 | 3.480689  2.984815  2.490047  1.996517   1.50441  1.013935  0.525314   0.038769  -0.445477  -0.927213  -1.406241  -1.882381  -2.355484  -2.825427  -3.292123  -3.755519  -4.215595  -4.672366  -5.125873  -5.576179  -6.023365  -6.467519  -6.908727  -7.347062  -7.782575  -8.215279  -8.645135   -9.072032   -9.495775   -9.916053  -10.332424  -10.744284  -11.150851  -11.551136  -11.943943  -12.327868  -12.701328  -13.062615   -13.40998  -13.741737  -14.056387  -14.352735  -14.629982  -14.887776  -15.126221  -15.345832  -15.547463  -15.732215  -15.901344   -16.056078 |  100.33556  100.476112  100.656691  100.886827  101.176543  101.536752  101.979365   102.51719  103.163739  103.932992  104.839131  105.896257  107.118094  108.517706  110.107239  111.897707  113.898834  116.118947  118.564943  121.242311  124.155214  127.306619  130.698483  134.331974  138.207734  142.326191  146.687911  151.294007  156.146618  161.249455  166.608444  172.232449  178.134095   184.33065  190.844928  197.706136  204.950556  212.621919  220.771323  229.456599  238.741064  248.691755  259.377346  270.866025  283.223534  296.511525  310.786628  326.099994  342.496981   360.01781  378.691997 |  100.33556 |     0.0 | 378.691997 |   3.480689 |
    | 2000-01-03 08:30:00 | 108.762979 | 3.478644  2.982177  2.486674  1.992258  1.499099  1.007392  0.517343   0.029167  -0.456922  -0.940714  -1.422007  -1.900613  -2.376368  -2.849131  -3.318792   -3.78527   -4.24852  -4.708526  -5.165301  -5.618883  -6.069328  -6.516702  -6.961077  -7.402514  -7.841058  -8.276718  -8.709462    -9.13919   -9.565721   -9.988772  -10.407929  -10.822628  -11.232131  -11.635501  -12.031595  -12.419065  -12.796377  -13.161862   -13.51379  -13.850476  -14.170391  -14.472286  -14.755287  -15.018953  -15.263297  -15.488752  -15.696097  -15.886377  -16.060809   -16.220552 | 100.277186  100.396027   100.54941  100.745716  100.993894  101.303793  101.686267  102.153099  102.716853  103.390691  104.188148  105.122877  106.208383   107.45774  108.883342  110.496664  112.308082  114.326734  116.560445   119.01571  121.697738  124.610546  127.757119  131.139599   134.75954  138.618196  142.716872  147.057323  151.642227  156.475735  161.564104  166.916429  172.545467  178.468527  184.708395  191.294211  198.262197  205.656093  213.527162  221.933628  230.939511  240.612894  251.023789  262.241873  274.334411  287.364701   301.39107  316.466243  332.637313  349.946177  368.425236 | 100.277186 |     0.0 | 368.425236 |   3.478644 |
    | 2000-01-03 09:00:00 |  92.224979 | 3.476916  2.979939  2.483799  1.988611  1.494532  1.001739  0.510427   0.020795  -0.466949    -0.9526  -1.435956  -1.916826  -2.395035  -2.870428  -3.342875  -3.812274  -4.278554  -4.741673  -5.201616  -5.658396  -6.112043  -6.562605  -7.010133  -7.454676  -7.896269  -8.334921  -8.770598   -9.203212   -9.632596  -10.058489  -10.480508  -10.898126  -11.310649  -11.717192  -12.116668  -12.507783  -12.889057  -13.258863  -13.615496  -13.957271  -14.282635  -14.590286  -14.879274  -15.149069  -15.399588  -15.631172  -15.844524  -16.040625   -16.22065     -16.3857 | 100.228681  100.329034  100.459155   100.62641   100.83875  101.105014  101.435031  101.839578  102.330267  102.919398  103.619779  104.444503   105.40671  106.519325  107.794813  109.244934  110.880549  112.711448  114.746251  116.992346  119.455901  122.141919  125.054366  128.196333  131.570265  135.178238  139.022285  143.104794  147.428968  151.999365  156.822533  161.907733  167.267768  172.919873  178.886665  185.197044  191.886972   198.99998  206.587249  214.707145  223.424127   232.80707  242.927158  253.855591   265.66144  278.409951  292.161436  306.970605  322.886451  339.952632   358.20485 | 100.228681 |     0.0 |  358.20485 |   3.476916 |
    | 2000-01-03 09:30:00 |  75.146084 | 3.475458  2.978041   2.48135  1.985493  1.490609  0.996864  0.504434    0.01351  -0.475715  -0.963041  -1.448269  -1.931208  -2.411676  -2.889509  -3.364561  -3.836713  -4.305871  -4.771968  -5.234965  -5.694849  -6.151629  -6.605327  -7.055977  -7.503614  -7.948261  -8.389923  -8.828566   -9.264107   -9.696395  -10.125188  -10.550132  -10.970736  -11.386349  -11.796139  -12.199073  -12.593917  -12.979243  -13.353471  -13.714925  -14.061927  -14.392901  -14.706497   -15.00169  -15.277856  -15.534815  -15.772812  -15.992465  -16.194687  -16.380603   -16.551269 | 100.188431  100.273078   100.38334  100.525676  100.707134  100.935613  101.219971  101.570009   101.99639  102.510535  103.124466  103.850634  104.701697  105.690291  106.828795  108.129089  109.602351  111.258871  113.107919  115.157655  117.415103  119.886177  122.575763  125.487863  128.625792  131.992427  135.590524  139.423092  143.493846  147.807733  152.371555  157.194693  162.289918  167.674302  173.370162  179.406002  185.817335  192.647255  199.946623  207.773696  216.193149  225.274477  235.089928  245.712202   257.21223  269.657338      283.11  297.627116  313.259821  330.053773  348.048463 | 100.188431 |     0.0 | 348.048463 |   3.475458 |
    | 2000-01-03 10:00:00 |  57.525565 |  3.47423  2.976434  2.479268   1.98283  1.487245  0.992665  0.499252   0.007181  -0.483364  -0.972193  -1.459113  -1.943934  -2.426474  -2.906559  -3.384036  -3.858768  -4.330644  -4.799575  -5.265501  -5.728384  -6.188208  -6.644977  -7.098703  -7.549406  -7.997096  -8.441772  -8.883398   -9.321896   -9.757125  -10.188863  -10.616781  -11.040424  -11.459185   -11.87228  -12.278734   -12.67737   -13.06682   -13.44555   -13.81192  -14.164262  -14.500985  -14.820692  -15.122287  -15.405053  -15.668707  -15.913395  -16.139642  -16.348288  -16.540402   -16.717009 | 100.155078  100.226402  100.319735  100.440732  100.595619  100.791432  101.036122  101.338557  101.708478  102.156416  102.693583  103.331719  104.082913  104.959396   105.97332  107.136538  108.460388  109.955504   111.63166  113.497661  115.561277  117.829237  120.307285   123.00028  125.912369  129.047212  132.408273  135.999175  139.824131  143.888462  148.199203  152.765809  157.600972  162.721511   168.14934  173.912428  180.045661    186.5915  193.600251  201.129827  209.244885  218.015336   227.51434  237.816015   248.99315  261.115238   274.24707  288.447936  303.771347  320.265228  337.973368 | 100.155078 |     0.0 | 337.973368 |    3.47423 |
    | 2000-01-03 10:30:00 |  39.364241 | 3.473195  2.975075    2.4775  1.980558  1.484364  0.989054  0.494776   0.001692  -0.490027    -0.9802  -1.468643   -1.95517    -2.4396  -2.921757  -3.401478  -3.878617  -4.353047  -4.824662  -5.293381  -5.759144  -6.221914  -6.681672  -7.138413  -7.592138  -8.042847  -8.490525  -8.935137   -9.376607   -9.814802  -10.249516  -10.680446  -11.107169   -11.52912  -11.945565  -12.355584  -12.758061  -13.151686  -13.534978  -13.906335  -14.264107  -14.606693  -14.932656  -15.240828  -15.530405  -15.800998  -16.052645  -16.285778  -16.501153  -16.699778   -16.882666 | 100.127477  100.187515  100.266439  100.369187  100.501247  100.668867  100.879155  101.140104  101.460563  101.850192  102.319376  102.879108  103.540838  104.316297  105.217298  106.255527  107.442341  108.788569  110.304348  111.998991  113.880896  115.957511   118.23535  120.720071  123.416609  126.329378  129.462527  132.820282  136.407347  140.229404  144.293702  148.609747  153.190101  158.051272  163.214681  168.707639  174.564258  180.826175  187.542928  194.771844  202.577318   211.02945  220.202123  230.170745  241.009919  252.791352  265.582284  279.444576  294.434353  310.602074  327.996372 | 100.127477 |     0.0 | 327.996372 |   3.473195 |
    | 2000-01-03 11:00:00 |  20.664761 | 3.472325  2.973926  2.475999  1.978622  1.481899  0.985952  0.490916  -0.003061   -0.49582  -0.987192  -1.477001  -1.965069  -2.451217  -2.935269   -3.41706  -3.896434  -4.373252  -4.847395  -5.318763  -5.787279  -6.252883  -6.715538  -7.175218  -7.631908  -8.085593  -8.536251  -8.983838    -9.42828   -9.869451   -10.30716  -10.741127  -11.170959   -11.59613  -12.015955   -12.42957  -12.835919  -13.233752  -13.621645  -13.998039  -14.361307  -14.709846  -15.042183  -15.357088  -15.653667  -15.931425  -16.190293  -16.430597  -16.653008  -16.858464   -17.047986 | 100.104666  100.155156  100.221831  100.308995  100.421474  100.564799  100.745305  100.970164  101.247383  101.585776  101.994906  102.484998  103.066818  103.751523   104.55049  105.475127  106.536675  107.746023  109.113532  110.648887  112.360993  114.257906  116.346824  118.634129  121.125492  123.826037  126.740579  129.873926  133.231263  136.818627  140.643468  144.715322  149.046593  153.653429  158.556685  163.782917  169.365333  175.344575  181.769208  188.695741  196.188053  204.316183  213.154517  222.779585  233.267716  244.692837  257.124734  270.628014  285.261636  301.078871  318.133776 | 100.104666 |     0.0 | 318.133776 |   3.472325 |
    | 2000-01-03 11:30:00 |   1.431931 | 3.471594  2.972956  2.474727  1.976975  1.479793  0.983291  0.487591  -0.007171  -0.500849  -0.993287  -1.484318  -1.973772  -2.461476  -2.947256  -3.430945  -3.912384  -4.391426  -4.867939  -5.341809  -5.812941  -6.281259  -6.746705  -7.209237   -7.66882  -8.125428  -8.579025  -9.029562   -9.476964    -9.92111   -10.36182  -10.798835  -11.231791    -11.6602  -12.083423   -12.50065  -12.910885  -13.312943  -13.705457  -14.086916  -14.455722  -14.810278  -15.149083  -15.470851    -15.7746  -16.059734  -16.326069  -16.573825  -16.803577  -17.016189   -17.212713 | 100.085838   100.12826  100.184535  100.258408  100.354114  100.476534  100.631297  100.824818  101.064312  101.357781  101.713986   102.14238  102.653018  103.256432  103.963488  104.785216  105.732632  106.816555   108.04744  109.435215  110.989166  112.717846  114.629036  116.729763  119.026367  121.524639  124.230022  127.147888  130.283891  133.644411  137.237094  141.071495   145.15984  149.517881  154.165858  159.129493  164.440973  170.139802  176.273382  182.897173  190.074287  197.874433  206.372254  215.645195  225.771175  236.826302  248.882978  262.008698  276.265469  291.709657  308.401358 | 100.085838 |     0.0 | 308.401358 |   3.471594 |
    | 2000-01-03 12:00:00 | 147.810877 |  3.47098  2.972138  2.473649  1.975573  1.477995  0.981011  0.484731   -0.01072  -0.505209  -0.998592  -1.490713   -1.98141  -2.470518  -2.957867  -3.443292  -3.926632  -4.407734  -4.886457  -5.362678  -5.836285  -6.307188  -6.775311  -7.240594  -7.702989  -8.162451  -8.618935  -9.072385    -9.52272   -9.969826  -10.413531  -10.853594  -11.289677  -11.721328  -12.147953  -12.568796   -12.98292  -13.389202  -13.786337  -14.172868  -14.547232  -14.907844  -15.253185  -15.581919  -15.892985  -16.185684  -16.459718  -16.715196  -16.952593  -17.172687   -17.376401 | 100.070317  100.105964   100.15343  100.215995  100.297367  100.401845  100.534416  100.700801  100.907473  101.161672  101.471383    101.8453  102.292754  102.823623  103.448202  104.177068  105.020912  105.990377  107.095884  108.347485  109.754723   111.32653  113.071171  114.996225   117.10863  119.414787  121.920727  124.632359  127.555788   130.69773  134.066022  137.670237   141.52241  145.637885  150.036248   154.74233  159.787215  165.209138  171.054173  177.376519  184.238262   191.70851  199.861881  208.776492  218.531664  229.205597  240.873328  253.605356  267.466926   282.51778  298.805486 | 100.070317 |     0.0 | 298.805486 |    3.47098 |
    | 2000-01-03 12:30:00 | 129.410282 | 3.470464  2.971449  2.472737  1.974383  1.476461  0.979059  0.482275   -0.01378  -0.508984  -1.003202  -1.496291    -1.9881  -2.478471   -2.96724  -3.454246  -3.939327  -4.422329  -4.903106  -5.381524  -5.857462  -6.330815  -6.801493   -7.26942   -7.73453  -8.196767  -8.656074  -9.112385   -9.565616  -10.015653  -10.462334  -10.905431  -11.344633  -11.779519  -12.209538  -12.633985  -13.051983  -13.462471   -13.86421  -14.255798  -14.635715  -15.002395  -15.354312  -15.690086  -16.008587  -16.309017  -16.590968  -16.854432  -17.099777  -17.327688    -17.53895 | 100.057539  100.087431  100.127414  100.180335  100.249436  100.338494  100.451913  100.594776  100.772878  100.992737  101.261603  101.587428  101.978828  102.445004  102.995648  103.640821  104.390811  105.255988  106.246636  107.372811  108.644196  110.069992  111.658838  113.418772   115.35725  117.481217  119.797242  122.311735  125.031228   127.96276  131.114348  134.495573  138.118275  141.997365  146.151746  150.605303  155.387931  160.536477  166.095514  172.117763  178.664024  185.802505  193.607498  202.157527  211.533134  221.814557  233.079645  245.402321  258.851382  273.490034  289.369879 | 100.057539 |     0.0 | 289.369879 |   3.470464 |
    | 2000-01-03 13:00:00 | 110.558484 | 3.470032  2.970869  2.471966  1.973373  1.475155   0.97739  0.480166  -0.016416  -0.512247  -1.007202  -1.501151  -1.993951  -2.485454  -2.975503  -3.463943  -3.950614  -4.435363  -4.918039  -5.398503  -5.876625   -6.35229  -6.825393  -7.295846  -7.763569  -8.228491  -8.690544  -9.149652   -9.605728  -10.058657  -10.508282  -10.954389  -11.396689  -11.834791  -12.268184  -12.696213  -13.118059  -13.532724  -13.939033  -14.335644  -14.721089  -15.093826  -15.452333  -15.795197  -16.121223  -16.429521  -16.719575  -16.991259  -17.244831   -17.48088   -17.700066 | 100.047032  100.072099   100.10578  100.150532  100.209182  100.285044  100.382005   100.50457  100.657907  100.847867  101.081003  101.364559  101.706446  102.115187  102.599847  103.169928  103.835255   104.60584  105.491737    106.5029  107.649043  108.939522  110.383245  111.988613  113.763514  115.715365  117.851222  120.177952  122.702493  125.432194  128.375252  131.541263  134.941881  138.591595  142.508626  146.715906  151.242102  156.122609  161.400389  167.126523  173.360316   180.16882  187.625723  195.809618  204.801843  214.684174  225.536747  237.436366  250.455317  264.660891  280.111012 | 100.047032 |     0.0 | 280.111012 |   3.470032 |
    | 2000-01-03 13:30:00 |  91.270398 |  3.46967   2.97038  2.471314  1.972515  1.474042  0.975964  0.478358  -0.018685  -0.515064  -1.010669  -1.505378   -1.99906  -2.491574  -2.982775  -3.472511  -3.960629  -4.446975  -4.931401  -5.413761  -5.893922  -6.371757  -6.847153  -7.320009  -7.790233  -8.257741  -8.722452  -9.184284   -9.643142   -10.09891  -10.551437  -11.000519  -11.445885  -11.887173   -12.32391  -12.755487   -13.18114  -13.599937  -14.010767  -14.412352  -14.803277  -15.182038   -15.54712  -15.897092  -16.230702  -16.546978  -16.845297  -17.125422  -17.387493  -17.632001   -17.859495 | 100.038402  100.059405  100.087752  100.125568  100.175312  100.239884  100.322703  100.427754  100.559631  100.723572  100.925478   101.17192  101.470128  101.827959  102.253837  102.756682  103.345806  104.030803  104.821415  105.727401  106.758405  107.923836  109.232764  110.693855  112.315341  114.105034  116.070412  118.218755  120.557372    123.0939  125.836711  128.795416  131.981487  135.409003  139.095504  143.062958  147.338782  151.956852  156.958419  162.392765  168.317467   174.79812  181.907433  189.723701    198.3288   207.80594  218.237539  229.703444  242.279645   256.03763  271.042075 | 100.038402 |     0.0 | 271.042075 |    3.46967 |
    | 2000-01-03 14:00:00 |  71.565124 | 3.469367  2.969969  2.470763  1.971789  1.473096  0.974746  0.476809  -0.020635  -0.517495  -1.013671  -1.509051  -2.003514  -2.496931  -2.989163  -3.480068  -3.969496    -4.4573   -4.94333  -5.427442  -5.909495   -6.38936  -6.866914  -7.342046  -7.814652  -8.284638  -8.751912  -9.216383   -9.677951  -10.136496  -10.591869  -11.043878  -11.492268  -11.936701   -12.37674   -12.81182  -13.241232  -13.664104  -14.079391  -14.485882   -14.88222  -15.266945  -15.638562   -15.99563  -16.336854  -16.661189  -16.967911  -17.256673  -17.527502  -17.780783   -18.016976 | 100.031322  100.048906  100.072746  100.104678  100.146837  100.201757  100.272438  100.362396  100.475707  100.617041  100.791697  101.005613   101.26537  101.578172   101.95181  102.394605  102.915328  103.523103  104.227295  105.037391  105.962873  107.013105  108.197223  109.524061  111.002101  112.639474  114.444009  116.423346  118.585123  120.937245  123.488243  126.247745  129.227051  132.439839  135.902985  139.637497  143.669526  148.031396   152.76256  157.910361  163.530451  169.686702  176.450537  183.899629  192.116076  201.184263   211.18872  222.212276  234.334689  247.631853   262.17569 | 100.031322 |     0.0 |  262.17569 |   3.469367 |
    | 2000-01-03 14:30:00 |  51.466276 | 3.469113  2.969624  2.470299  1.971173  1.472292  0.973708  0.475484  -0.022309  -0.519589  -1.016265  -1.512236  -2.007392  -2.501611  -2.994766   -3.48672  -3.977334  -4.466461  -4.953958   -5.43968  -5.923485   -6.40524  -6.884815  -7.362092  -7.836957  -8.309307  -8.779041  -9.246058   -9.710252  -10.171501  -10.629657  -11.084536  -11.535896  -11.983424  -12.426713  -12.865241   -13.29835  -13.725227  -14.144895  -14.556209  -14.957875  -15.348483  -15.726569  -16.090692  -16.439529   -16.77197  -17.087204  -17.384775  -17.664601  -17.926961   -18.172247 | 100.025521  100.040232  100.060268  100.087212  100.122917  100.169593  100.229867  100.306835  100.404102  100.525825  100.676743  100.862205  101.088176  101.361235  101.688552  102.077847  102.537328  103.075618  103.701654  104.424583  105.253649  106.198082  107.266992  108.469288  109.813616  111.308338  112.961565  114.781237  116.775278  118.951828  121.319563  123.888124  126.668649  129.674431  132.921701  136.430525  140.225795  144.338258  148.805517  153.672871  158.993875  164.830446   171.25242  178.336502  186.164652  194.822104  204.395255  214.969788  226.629239  239.454089  253.524014 | 100.025521 |     0.0 | 253.524014 |   3.469113 |
    | 2000-01-03 15:00:00 |  31.002007 | 3.468901  2.969333  2.469907  1.970652  1.471609  0.972823   0.47435  -0.023745  -0.521391  -1.018506  -1.514997  -2.010764  -2.505695  -2.999672  -3.492567  -3.984247  -4.474575  -4.963407  -5.450604  -5.936024  -6.419531  -6.900992  -7.380282  -7.857281  -8.331877   -8.80396  -9.273422   -9.740151  -10.204021  -10.664886  -11.122567  -11.576835  -12.027396  -12.473874  -12.915785   -13.35252  -13.783323  -14.207283  -14.623323  -15.030214  -15.426605  -15.811069  -16.182179  -16.538595  -16.879158  -17.202978  -17.509503  -17.798543  -18.070272   -18.325047 | 100.020773  100.033073  100.049901   100.07262  100.102838  100.142479   100.19384   100.25964  100.343062  100.447793  100.578065  100.738676  100.935016  101.173067  101.459398  101.801135  102.205921   102.68185  103.237395  103.881312  104.622539  105.470098  106.432988  107.520105   108.74017    110.1017  111.613009  113.282276  115.117659  117.127497  119.320594  121.706597  124.296486  127.103192  130.142325  133.433034  136.998969  140.869307  145.079768  149.673531  154.701916   160.22467  166.309762  173.032572  180.474544  188.721381   197.86104  207.981868  219.171131  231.514038  245.098526 | 100.020773 |     0.0 | 245.098526 |   3.468901 |
    | 2000-01-03 15:30:00 |  10.205214 | 3.468724   2.96909  2.469577  1.970212  1.471029  0.972069  0.473382  -0.024976  -0.522941  -1.020439  -1.517386  -2.013691  -2.509253  -3.003961  -3.497698  -3.990336  -4.481746  -4.971792  -5.460336  -5.947239  -6.432363  -6.915576  -7.396748  -7.875753  -8.352472   -8.82679  -9.298591   -9.767755  -10.234157  -10.697648  -11.158054  -11.615157  -12.068683   -12.51828    -12.9635   -13.40378  -13.838421  -14.266573  -14.687229  -15.099229  -15.501282   -15.89201  -16.270011  -16.633941  -16.982607  -17.315056  -17.630647   -17.92909  -18.210462   -18.475118 | 100.016892  100.027168  100.041293  100.060439  100.085997   100.11964  100.163373  100.219581  100.291068  100.381099  100.493437  100.632379  100.802775  101.010046  101.260185   101.55974  101.915788  102.335889  102.828023  103.400515  104.061945  104.821059  105.686674  106.667591  107.772526  109.010066  110.388658  111.916652  113.602387  115.454362  117.481481  119.693387  122.100912  124.716637  127.555581  130.636012  133.980371  137.616279  141.577571  145.905257  150.648313  155.864136  161.618555  167.985291  175.044867  182.883046  191.588985  201.253431  211.967271  223.820538  236.909922 | 100.016892 |     0.0 | 236.909922 |   3.468724 |
    | 2000-01-03 16:00:00 | 171.160848 | 3.468576  2.968886  2.469299  1.969839  1.470537  0.971428  0.472555  -0.026031  -0.524273  -1.022105  -1.519452  -2.016231   -2.51235  -3.007707  -3.502193  -3.995691  -4.488076  -4.979219  -5.468988  -5.957248  -6.443862  -6.928696  -7.411619  -7.892503  -8.371222  -8.847656  -9.321683   -9.793179  -10.262014  -10.728041  -11.191087  -11.650946  -12.107358  -12.559994  -13.008442   -13.45218  -13.890561  -14.322797  -14.747949  -15.164927  -15.572506  -15.969363  -16.354133  -16.725482  -17.082199  -17.423282  -17.748015  -18.056022  -18.347288   -18.622094 | 100.013722  100.022309   100.03416  100.050286  100.071892  100.100429  100.137646   100.18563  100.246847  100.324181  100.420973  100.541058  100.688789  100.869061   101.08732  101.349557  101.662297  102.032562  102.467826  102.975954  103.565128  104.243764  105.020431  105.903764  106.902399  108.024916  109.279823  110.675571  112.220627  113.923605  115.793478  117.839873  120.073474  122.506533  125.153508  128.031824  131.162757  134.572413  138.292755  142.362607  146.828525  151.745407  157.176708  163.194161  169.876971  177.310523  185.584758  194.792501  205.028075   216.38637  228.963327 | 100.013722 |     0.0 | 228.963327 |   3.468576 |
    | 2000-01-03 16:30:00 | 151.972425 | 3.468452  2.968714  2.469065  1.969525   1.47012  0.970882   0.47185  -0.026933  -0.525416  -1.023539  -1.521236  -2.018432  -2.515042  -3.010974  -3.506126  -4.000391  -4.493652  -4.985786  -5.476667  -5.966163  -6.454143  -6.940473   -7.42502  -7.907655  -8.388251   -8.86668  -9.342817   -9.816536  -10.287701  -10.756166  -11.221761  -11.684288  -12.143499   -12.59909  -13.050675  -13.497774   -13.93979  -14.375993   -14.80551  -15.227323  -15.640275  -16.043106  -16.434498  -16.813142  -17.177823  -17.527509  -17.861428  -18.179131  -18.480523   -18.765732 | 100.011136  100.018304  100.028237  100.041809   100.06006  100.084247  100.115895  100.156825  100.209202  100.275569  100.358886  100.462566  100.590506   100.74711  100.937312  101.166571  101.440877   101.76672  102.151065  102.601299  103.125175  103.730738  104.426255  105.220133  106.120858  107.136934  108.276856  109.549113  110.962238  112.524913  114.246144  116.135523  118.203581   120.46225  122.925453  125.609808  128.535469  131.727063  135.214709  139.035033  143.232115  147.858219  152.974208  158.649523  164.961683  171.995324  179.840828  188.592867  198.349122  209.209322  221.268631 | 100.011136 |     0.0 | 221.268631 |   3.468452 |
    | 2000-01-03 17:00:00 | 132.606437 | 3.468349  2.968571  2.468868  1.969259  1.469766  0.970418  0.471248  -0.027705  -0.526397  -1.024774  -1.522776  -2.020337  -2.517379  -3.013819  -3.509564  -4.004513  -4.498557  -4.991583  -5.483469  -5.974091  -6.463318  -6.951022   -7.43707  -7.921333   -8.40368  -8.883984  -9.362115   -9.837942   -10.31133   -10.78213  -11.250176  -11.715275  -12.177194  -12.635646  -13.090273  -13.540631   -13.98617  -14.426217  -14.859962  -15.286457  -15.704619  -16.113255  -16.511101  -16.896886  -17.269409  -17.627622  -17.970719  -18.298195  -18.609896   -18.905839 | 100.009028  100.015012  100.023342  100.034765  100.050178  100.070672  100.097572   100.13247   100.17726  100.234182  100.305852    100.3953  100.506006  100.641923  100.807503  101.007705  101.248003  101.534371  101.873266  102.271592  102.736648  103.276078  103.897803  104.609951  105.420794  106.338697  107.372076  108.529395  109.819193  111.250173  112.831341  114.572231  116.483215  118.575926  120.863779   123.36264  126.091597  129.073869  132.337789  135.917829  139.855592  144.200652  149.011136  154.353931  160.304427  166.945817   174.36807  182.666815  191.942357   202.29911  213.841141 | 100.009028 |     0.0 | 213.841141 |   3.468349 |
    | 2000-01-03 17:30:00 | 113.108003 | 3.468263   2.96845  2.468702  1.969035  1.469467  0.970024  0.470736  -0.028365  -0.527237  -1.025835  -1.524104  -2.021984  -2.519406  -3.016294  -3.512563  -4.008121  -4.502867  -4.996693  -5.489486  -5.981127  -6.471491  -6.960453  -7.447883  -7.933652   -8.41763  -8.899688  -9.379693   -9.857515  -10.333013  -10.806041  -11.276435  -11.744007  -12.208535  -12.669748  -13.127315  -13.580825   -14.02977   -14.47353  -14.911359  -15.342377  -15.765575  -16.179829  -16.583942   -16.97669  -17.356899   -17.72353  -18.075757  -18.413048  -18.735211   -19.042199 | 100.007313  100.012305  100.019286  100.028896  100.041907  100.059265  100.082117  100.111853  100.150131  100.198918  100.260521  100.337626  100.433333   100.55118  100.695171  100.869798  101.080044  101.331385  101.629782  101.981651  102.393833  102.873545  103.428329   104.06599  104.794539  105.622139  106.557068  107.607704  108.782537  110.090239  111.539776  113.140597   114.90291  116.838048   118.95895  121.280764  123.821575  126.603256  129.652416  133.001422   136.68941  140.763203  145.278039  150.297977  155.895927  162.153273  169.159204  177.009944  185.808147  195.662721  206.686657 | 100.007313 |     0.0 | 206.686657 |   3.468263 |
    | 2000-01-03 18:00:00 |  93.527914 | 3.468191  2.968349  2.468563  1.968845  1.469214   0.96969  0.470299  -0.028929  -0.527957  -1.026746  -1.525248  -2.023407  -2.521163  -3.018446  -3.515179  -4.011277  -4.506647   -5.00119  -5.494799  -5.987361  -6.478758  -6.968868  -7.457566  -7.944725  -8.430215  -8.913907   -9.39567   -9.875368  -10.352864   -10.82801  -11.300644  -11.770584  -12.237616  -12.701488  -13.161888  -13.618436  -14.070665  -14.518005  -14.959768  -15.395142  -15.823192  -16.242868  -16.653042   -17.05255  -17.440263  -17.815164  -18.176437  -18.523543  -18.856284   -19.174602 | 100.005918  100.010081  100.015928  100.024009  100.034988  100.049682  100.069087  100.094411  100.127103  100.168889  100.221801  100.288214  100.370883  100.472966  100.598058   100.75021  100.933946  101.154266  101.416649  101.727032  102.091793  102.517711  103.011924   103.58188  104.235285  104.980053  105.824269  106.776167  107.844136  109.036766  110.362942  111.832002  113.453972   115.23989  117.202238  119.355489  121.716779  124.306702   127.15022  130.277654  133.725705  137.538431   141.76809  146.475735  151.731509  157.614604  164.212955  171.622881  179.948913  189.304143  199.810496 | 100.005918 |     0.0 | 199.810496 |   3.468191 |
    | 2000-01-03 18:30:00 |  73.921247 | 3.468131  2.968265  2.468446  1.968686  1.468999  0.969406  0.469927   -0.02941  -0.528574  -1.027529  -1.526233  -2.024636  -2.522684  -3.020314  -3.517456  -4.014033  -4.509959  -5.005142  -5.499483  -5.992876  -6.485209  -6.976364  -7.466222  -7.954658  -8.441546  -8.926756  -9.410158   -9.891618  -10.370996  -10.848147   -11.32291  -11.795109  -12.264539   -12.73096  -13.194082  -13.653551  -14.108938  -14.559719  -15.005263  -15.444823  -15.877534  -16.302425  -16.718441  -17.124488  -17.519494  -17.902486  -18.272678  -18.629556  -18.972949   -19.302853 | 100.004785  100.008254  100.013151  100.019943  100.029203  100.041636  100.058106  100.079661  100.107567  100.143334  100.188749  100.245911  100.317259  100.405609  100.514178  100.646611  100.806999  100.999895  101.230313  101.503726   101.82605  102.203622   102.64316  103.151733  103.736705  104.405701  105.166564  106.027335   106.99625   108.08177   109.29266  110.638113  112.127957  113.772935  115.585088  117.578252  119.768667   122.17572  124.822804  127.738272   130.95645  134.518654  138.474115  142.880747  147.805682  153.325544  159.526538  166.504521  174.365361  183.225934  193.217009 | 100.004785 |     0.0 | 193.217009 |   3.468131 |
    | 2000-01-03 19:00:00 |  54.346521 | 3.468081  2.968194  2.468347  1.968551  1.468818  0.969165  0.469611   -0.02982  -0.529101    -1.0282   -1.52708  -2.025696     -2.524  -3.021934  -3.519437  -4.016437  -4.512857  -5.008612  -5.503608  -5.997748  -6.490925  -6.983029  -7.473945  -7.963551  -8.451725  -8.938341  -9.423268   -9.906374  -10.387521  -10.866563  -11.343343   -11.81769  -12.289407  -12.758265  -13.223993  -13.686263  -14.144679   -14.59876  -15.047928  -15.491497  -15.928676  -16.358567  -16.780196  -17.192544  -17.594609   -17.98548  -18.364428   -18.73099  -19.085064    -19.42677 | 100.003865  100.006756  100.010854  100.016561  100.024369  100.034885  100.048856  100.067194  100.091001  100.121598  100.160554  100.209717  100.271248  100.347648   100.44179  100.556945  100.696805  100.865497  101.067596  101.308126  101.592553  101.926771  102.317074   102.77013   103.29294  103.892804  104.577284  105.354184   106.23154  107.217641  108.321088  109.550902  110.916692    112.4289  114.099141  115.940644  117.968808  120.201886  122.661782  125.374972  128.373489  131.695952  135.388565  139.506014  144.112215  149.280892  155.096051  161.652522  169.056912  177.429392  186.909375 | 100.003865 |     0.0 | 186.909375 |   3.468081 |
    | 2000-01-03 19:30:00 |  34.864938 |  3.46804  2.968135  2.468265  1.968437  1.468665  0.968961  0.469342   -0.03017  -0.529552  -1.028776  -1.527808  -2.026609  -2.525137  -3.023339  -3.521159  -4.018533   -4.51539  -5.011653  -5.507235  -6.002044  -6.495983  -6.988946  -7.480823  -7.971499  -8.460854  -8.948766  -9.435107   -9.919746  -10.402547  -10.883367  -11.362052  -11.838433  -12.312322  -12.783503   -13.25172  -13.716668  -14.177982  -14.635219  -15.087852  -15.535255  -15.976703  -16.411376  -16.838378  -17.256775  -17.665645  -18.064157  -18.451658  -18.827774   -19.19251   -19.546197 | 100.003119  100.005527  100.008957   100.01375   100.02033  100.029221  100.041068  100.056661  100.076961  100.103121  100.136515  100.178771  100.231795  100.297809  100.379371  100.479409  100.601244  100.748608  100.925662  101.136999  101.387651  101.683074  102.029138  102.432101  102.898584  103.435535  104.050204  104.750119  105.543074  106.437146  107.440737  108.562661  109.812287  111.199751   112.73625  114.434435  116.308913   118.37687  120.658812  123.179432  125.968566  129.062212  132.503577  136.344074   140.64425  145.474642  150.916622   157.06343  164.021782   171.91457  180.889523 | 100.003119 |     0.0 | 180.889523 |    3.46804 |
    | 2000-01-03 20:00:00 |  15.539549 | 3.468005  2.968086  2.468195  1.968342  1.468535  0.968788  0.469114  -0.030468  -0.529938  -1.029269  -1.528433  -2.027396  -2.526118  -3.024555  -3.522653  -4.020357  -4.517602  -5.014315   -5.51042  -6.005829  -6.500452   -6.99419  -7.486939  -7.978589  -8.469026  -8.958129  -9.445776   -9.931839  -10.416183  -10.898667  -11.379142  -11.857445  -12.333393   -12.80678  -13.277365  -13.744866  -14.208944  -14.669194  -15.125133  -15.576191  -16.021709  -16.460942  -16.893074  -17.317258  -17.732663  -18.138551   -18.53437  -18.919864  -19.295197   -19.660997 | 100.002515  100.004521  100.007389  100.011414  100.016958  100.024472  100.034513  100.047767  100.065068  100.087421  100.116031  100.152326  100.197989  100.254985  100.325589  100.412417  100.518448  100.647048  100.801987  100.987453  101.208059  101.468837  101.775236  102.133103  102.548666  103.028505  103.579531  104.208965  104.924326  105.733442  106.644479  107.666015  108.807158  110.077727  111.488507  113.051595  114.780853  116.692465  118.805627   121.14335  123.733378  126.609186  129.811043  133.387086  137.394394  141.900083    146.9825  152.732759  159.257029  166.680276  175.158094 | 100.002515 |     0.0 | 175.158094 |   3.468005 |
    | 2000-01-03 20:30:00 |  197.76418 | 3.467976  2.968044  2.468137  1.968261  1.468426  0.968641   0.46892  -0.030722  -0.530267  -1.029691   -1.52897  -2.028074  -2.526966  -3.025607   -3.52395  -4.021944  -4.519531  -5.016644  -5.513213  -6.009158  -6.504395  -6.998831  -7.492369  -7.984904  -8.476327  -8.966523  -9.455373   -9.942752   -10.42853  -10.912569  -11.394722  -11.874832  -12.352723  -12.828199  -13.301034  -13.770961   -14.23767  -14.700787  -15.159872  -15.614408  -16.063798  -16.507366  -16.944381  -17.374083  -17.795739  -18.208718  -18.612588  -19.007248  -19.393061   -19.771026 | 100.002026  100.003696  100.006095  100.009474  100.014144  100.020492     100.029  100.040261     100.055  100.074092   100.09859  100.129747  100.169046  100.218221  100.279294  100.354595   100.44679  100.558907  100.694353  100.856934  101.050864  101.280767  101.551681  101.869045  102.238689  102.666814  103.159973  103.725056  104.369279  105.100187   105.92568  106.854074  107.894193  109.055527  110.348453  111.784536  113.376931  115.140885  117.094367  119.258813  121.659988  124.328955  127.303134  130.627425  134.355387  138.550542  143.287885  148.655895  154.759544  161.725168  169.713283 | 100.002026 |     0.0 | 169.713283 |   3.467976 |
    | 2000-01-03 21:00:00 | 181.025881 | 3.467952   2.96801  2.468088  1.968193  1.468333  0.968517  0.468755  -0.030939  -0.530548  -1.030053  -1.529431  -2.028656  -2.527696  -3.026516  -3.525074  -4.023323  -4.521211  -5.018679   -5.51566  -6.012083   -6.50787  -7.002933  -7.497182  -7.990519   -8.48284  -8.974035   -9.46399   -9.952583  -10.439689  -10.925174  -11.408895    -11.8907   -12.37042  -12.847868   -13.32283  -13.795058  -14.264262  -14.730102  -15.192175  -15.650013  -16.103075  -16.550757  -16.992406  -17.427354  -17.854968  -18.274736  -18.686369  -19.089944  -19.486082   -19.876106 | 100.001631  100.003022  100.005027  100.007862  100.011794  100.017155  100.024361  100.033925  100.046475  100.062774   100.08374  100.110471  100.144271  100.186671  100.239461  100.304713  100.384809  100.482466  100.600755  100.743125  100.913412  101.115853  101.355085  101.636149  101.964478  102.345889  102.786567  103.293051  103.872231  104.531343  105.277996  106.120214  107.066519  108.126061  109.308805  110.625803  112.089534  113.714369  115.517126  117.517764  119.740183  122.213153  124.971349  128.056494  131.518637  135.417624  139.824915  144.826023  150.524203  157.046489  164.549695 | 100.001631 |     0.0 | 164.549695 |   3.467952 |
    | 2000-01-03 21:30:00 | 164.600458 | 3.467932  2.967981  2.468047  1.968136  1.468255  0.968411  0.468615  -0.031123  -0.530788  -1.030362  -1.529826  -2.029157  -2.528326  -3.027302  -3.526047  -4.024521  -4.522674  -5.020455  -5.517802  -6.014651  -6.510928  -7.006554  -7.501443  -7.995505  -8.488641  -8.980747  -9.471712   -9.961422  -10.449755  -10.936581  -11.421762  -11.905152  -12.386588  -12.865891   -13.34286  -13.817262   -14.28883  -14.757249  -15.222153  -15.683117  -16.139657  -16.591233   -17.03727  -17.477189  -17.910463  -18.336703  -18.755777  -19.167972  -19.574215    -19.97626 | 100.001312   100.00247  100.004148  100.006527  100.009838  100.014365  100.020468  100.028589  100.039273  100.053183   100.07112  100.094046  100.123103  100.159642  100.205245  100.261752  100.331287  100.416281  100.519497  100.644049  100.793422  100.971482  101.182485  101.431084  101.722323  102.061631   102.45482  102.908069  103.427921  104.021286  104.695457  105.458145  106.317552  107.282474  108.362472  109.568091  110.911177  112.405274  114.066134  115.912341  117.966058    120.2539  122.807937  125.666836   128.87717  132.494971  136.587735  141.237254  146.544065  152.634846  159.672155 | 100.001312 |     0.0 | 159.672155 |   3.467932 |
    | 2000-01-03 22:00:00 | 148.538336 | 3.467916  2.967956  2.468013  1.968088  1.468189  0.968322  0.468496   -0.03128  -0.530992  -1.030627  -1.530165  -2.029587  -2.528868   -3.02798   -3.52689   -4.02556  -4.523947  -5.022004  -5.519675  -6.016901  -6.513616  -7.009745  -7.505211  -7.999926    -8.4938  -8.986733  -9.478622   -9.969356  -10.458818  -10.946884  -11.433421  -11.918287  -12.401328  -12.882372  -13.361228  -13.837681  -14.311479  -14.782335  -15.249915  -15.713834   -16.17366  -16.628914  -17.079095  -17.523711  -17.962342  -18.394729  -18.820909  -19.241402  -19.657491   -20.071473 | 100.001054  100.002019  100.003421  100.005419  100.008206  100.012029  100.017196  100.024089  100.033182  100.045048  100.060387  100.080039  100.105005  100.136474  100.175844  100.224744  100.285064  100.358976  100.448959   100.55782  100.688713  100.845156  101.031044  101.250653   101.50865  101.810083  102.160388  102.565376  103.031236  103.564533  104.172226  104.861697  105.640808  106.517996  107.502407  108.604099  109.834307  111.205798  112.733327  114.434201  116.328962  118.442202  120.803518  123.448626  126.420678  129.771891  133.565724  137.880066  142.812337   148.48818  155.074898 | 100.001054 |     0.0 | 155.074898 |   3.467916 |
    | 2000-01-03 22:30:00 | 132.889031 | 3.467902  2.967936  2.467984  1.968048  1.468133  0.968247  0.468395  -0.031413  -0.531167  -1.030853  -1.530456  -2.029957  -2.529335  -3.028566  -3.527619  -4.026461  -4.525053  -5.023353  -5.521311  -6.018872  -6.515976  -7.012555  -7.508537  -8.003841  -8.498381  -8.992065  -9.484795   -9.976465  -10.466964  -10.956173  -11.443965  -11.930203  -12.414739  -12.897411  -13.378039  -13.856418  -14.332317   -14.80547  -15.275573   -15.74228  -16.205204  -16.663924  -17.118009  -17.567053   -18.01074  -18.448944  -18.881879  -19.310326  -19.735966   -20.161747 | 100.000846  100.001649  100.002822  100.004498  100.006844  100.010071  100.014445  100.020296  100.028031   100.03815  100.051262  100.068099   100.08954  100.116628  100.150595  100.192884  100.245172  100.309397  100.387777  100.482836  100.597423   100.73473  100.898308  101.092079  101.320339  101.587768  101.899428  102.260762  102.677596  103.156143  103.703013  104.325244  105.030348  105.826387  106.722097  107.727052  108.851902  110.108682  111.511215  113.075618  114.820925  116.769839  118.949626  121.393192  124.140384  127.239668  130.750433  134.746458  139.321603   144.59973  150.751816 | 100.000846 |     0.0 | 150.751816 |   3.467902 |
    | 2000-01-03 23:00:00 | 117.698813 |  3.46789  2.967919  2.467959  1.968014  1.468087  0.968183  0.468309  -0.031527  -0.531316  -1.031046  -1.530704  -2.030274  -2.529737  -3.029071  -3.528249  -4.027241  -4.526014  -5.024528  -5.522739  -6.020597  -6.518047  -7.015027   -7.51147  -8.007302  -8.502443  -8.996807  -9.490301   -9.982825  -10.474273  -10.964533  -11.453482  -11.940991  -12.426918  -12.911109  -13.393393  -13.873578   -14.35145  -14.826763  -15.299241  -15.768571  -16.234409  -16.696389  -17.154144  -17.607351  -18.055795  -18.499485  -18.938822  -19.374859  -19.809721   -20.247113 | 100.000679  100.001347  100.002328  100.003734  100.005709  100.008433  100.012134  100.017098  100.023676  100.032303  100.043506  100.057926   100.07633  100.099634  100.128923  100.165471  100.210766  100.266532   100.33475  100.417685  100.517904  100.638295  100.782088  100.952865  101.154572  101.391529  101.668434  101.990364  102.362781  102.791539  103.282891  103.843521  104.480578  105.201746  106.015345  106.930475  107.957223  109.106928  110.392541  111.829074  113.434161  115.228747  117.237917  119.491925  122.027461  124.889339  128.132882  131.827615  136.063435  140.961704  146.695957 | 100.000679 |     0.0 | 146.695957 |    3.46789 |
    | 2000-01-03 23:30:00 | 103.009582 | 3.467881  2.967905  2.467939  1.967985  1.468047  0.968129  0.468237  -0.031623  -0.531442  -1.031211  -1.530917  -2.030547  -2.530083  -3.029506  -3.528793  -4.027917  -4.526848   -5.02555  -5.523984  -6.022104  -6.519861  -7.017198  -7.514054  -8.010359  -8.506041  -9.001018  -9.495204   -9.988505  -10.480821  -10.972044  -11.462059  -11.950742  -12.437958   -12.92356  -13.407389  -13.889263  -14.368983  -14.846323   -15.32103  -15.792824  -16.261398  -16.726437  -17.187635  -17.644746  -18.097655  -18.546505  -18.991883  -19.435134  -19.878859   -20.327623 | 100.000544  100.001101   100.00192    100.0031  100.004762   100.00706  100.010192  100.014402  100.019996  100.027347  100.036917  100.049261  100.065051   100.08509  100.110331  100.141899   100.18111  100.229496  100.288825  100.361123  100.448697  100.554156  100.680427  100.830775  101.008813  101.218514  101.464222  101.750655  102.082916  102.466495  102.907287  103.411613  103.986253  104.638507   105.37628  106.208204  107.143821  108.193809  109.370302  110.687284  112.161091  113.811039  115.660187  117.736298  120.073071  122.711799  125.703799  129.114246  133.028758   137.56553  142.899498 | 100.000544 |     0.0 | 142.899498 |   3.467881 |
    | 2000-01-04 00:00:00 |  88.858113 | 3.467873  2.967893  2.467922  1.967961  1.468013  0.968083  0.468175  -0.031705   -0.53155  -1.031352  -1.531099   -2.03078   -2.53038  -3.029881  -3.529262  -4.028502  -4.527571  -5.026438  -5.525068  -6.023421   -6.52145  -7.019104  -7.516327  -8.013056  -8.509223  -9.004753  -9.499565   -9.993571  -10.486677  -10.978781  -11.469774  -11.959538  -12.447946  -12.934858  -13.420123  -13.903572  -14.385019  -14.864255  -15.341051  -15.815153  -16.286292  -16.754196  -17.218618  -17.679383  -18.136472  -18.590158   -19.04122  -19.491299  -19.943505   -20.403359 | 100.000436  100.000899  100.001584  100.002574  100.003972  100.005912  100.008561  100.012131  100.016886  100.023149   100.03132  100.041883  100.055425  100.072647  100.094389  100.121639  100.155563  100.197518  100.249077  100.312053  100.388513  100.480808  100.591585  100.723811  100.880783  101.066146  101.283902  101.538421   101.83445  102.177121  102.571969  103.024949   103.54247  104.131443  104.799358   105.55439  106.405548  107.362885  108.437764  109.643212  110.994367  112.509036  114.208402  116.117909  118.268436  120.697915  123.453754  126.596759  130.208026  134.401982  139.353812 | 100.000436 |     0.0 | 139.353812 |   3.467873 |
    | 2000-01-04 00:30:00 |  75.275524 | 3.467866  2.967883  2.467908  1.967941  1.467985  0.968045  0.468123  -0.031775  -0.531642  -1.031472  -1.531255   -2.03098  -2.530635  -3.030203  -3.529668  -4.029007  -4.528197   -5.02721  -5.526013   -6.02457  -6.522839  -7.020775  -7.518325  -8.015433  -8.512035  -9.008062  -9.503439   -9.998083  -10.491907  -10.984815  -11.476703  -11.967461  -12.456967   -12.94509  -13.431688  -13.916601  -14.399657  -14.880664  -15.359412  -15.835672   -16.30921  -16.779793  -17.247227  -17.711405    -18.1724  -18.630608  -19.086998  -19.543516  -20.003802    -20.47442 | 100.000349  100.000735  100.001307  100.002137  100.003314   100.00495  100.007191  100.010218  100.014258  100.019593  100.026567  100.035603  100.047212  100.062008  100.080725  100.104236  100.133568  100.169923    100.2147  100.269513  100.336216   100.41692  100.514015  100.630188  100.768441  100.932107  101.124863  101.350741  101.614142  101.919848  102.273035  102.679294   103.14466  103.675658  104.279364    104.9635  105.736565  106.608009  107.588473   108.69009  109.926889  111.315288  112.874734  114.628521  116.604893  118.838597  121.373269  124.265382  127.591349  131.461317   136.04958 | 100.000349 |     0.0 |  136.04958 |   3.467866 |
    | 2000-01-04 01:00:00 |  62.286927 | 3.467861  2.967875  2.467896  1.967924  1.467961  0.968012  0.468079  -0.031834  -0.531721  -1.031575  -1.531388  -2.031152  -2.530854  -3.030481  -3.530017  -4.029444   -4.52874  -5.027879  -5.526834  -6.025571  -6.524053  -7.022238   -7.52008  -8.017525  -8.514516  -9.010989  -9.506874  -10.002096  -10.496571  -10.990211  -11.482917  -11.974585  -12.465101  -12.954341  -13.442172  -13.928444  -14.412997  -14.895652   -15.37622  -15.854495  -16.330271  -16.803355  -17.273598  -17.740958  -18.205592  -18.668018   -19.12939  -19.591956  -20.059909   -20.540929 | 100.000279    100.0006  100.001079  100.001775  100.002765  100.004146  100.006041  100.008607  100.012039  100.016582  100.022533   100.03026  100.040208  100.052913   100.06902  100.089294  100.114641  100.146124  100.184985   100.23266  100.290806  100.361317  100.446344  100.548316   100.66996  100.814315  100.984746  101.184966  101.419042   101.69141  102.006897  102.370732   102.78858  103.266581  103.811403  104.430323  105.131343  105.923339  106.816266  107.821429  108.951825  110.222594  111.651585  113.260107  115.073952  117.124873    119.4529  122.110267  125.168616  128.733394  132.976907 | 100.000279 |     0.0 | 132.976907 |   3.467861 |
    | 2000-01-04 01:30:00 |  49.911263 | 3.467856  2.967868  2.467886  1.967909  1.467941  0.967984  0.468041  -0.031884  -0.531787  -1.031662  -1.531502  -2.031299  -2.531042   -3.03072  -3.530319  -4.029822   -4.52921   -5.02846  -5.527548  -6.026444  -6.525113  -7.023519  -7.521619  -8.019364  -8.516703  -9.013575  -9.509918   -10.00566  -10.500725  -10.995028   -11.48848   -11.98098  -12.472423  -12.962692   -13.45166  -13.939189   -14.42513  -14.909318  -15.391579   -15.87173  -16.349592  -16.825004  -17.297864  -17.768185  -18.236204  -18.702554  -19.168569    -19.6368  -20.111999   -20.603026 | 100.000223   100.00049  100.000891  100.001475  100.002307  100.003472  100.005074  100.007249  100.010165  100.014032  100.019109  100.025715  100.034236  100.045142  100.058996   100.07647  100.098362  100.125611  100.159315  100.200754  100.251404   100.31296  100.387356  100.476783  100.583711  100.710902  100.861432  101.038705  101.246468  101.488827  101.770264  102.095656  102.470303  102.899963  103.390902  103.949964  104.584666  105.303335  106.115279  107.031025   108.06262  109.224023  110.531614  112.004876  113.667332  115.547939  117.683314   120.12158  122.929603  126.207798  130.125454 | 100.000223 |     0.0 | 130.125454 |   3.467856 |
    | 2000-01-04 02:00:00 |  38.161316 | 3.467852  2.967863  2.467877  1.967897  1.467924  0.967961  0.468009  -0.031927  -0.531844  -1.031737    -1.5316  -2.031425  -2.531203  -3.030925  -3.530578  -4.030147  -4.529616  -5.028964  -5.528169  -6.027203  -6.526038  -7.024639  -7.522968   -8.02098  -8.518628  -9.015859  -9.512612  -10.008822  -10.504419  -10.999325  -11.493453  -11.986713  -12.479004  -12.970216  -13.460232  -13.948922  -14.436146  -14.921755  -15.405589  -15.887484  -16.367285  -16.844863  -17.320154  -17.793225  -18.264387  -18.734381  -19.204713  -19.678231  -20.160252   -20.660867 | 100.000178  100.000401  100.000736  100.001226  100.001926  100.002908  100.004263  100.006106  100.008583  100.011874  100.016204  100.021849  100.029146  100.038504  100.050415  100.065469  100.084367  100.107938  100.137153  100.173148  100.217238  100.270936  100.335978  100.414336  100.508241    100.6202  100.753015  100.909799  101.093994  101.309387  101.560129  101.850754  102.186203  102.571861    103.0136  103.517839  104.091628  104.742765  105.479948  106.312976   107.25301  108.312925  109.507753  110.855299  112.376998  114.099211  116.055345  118.289587  120.864075  123.873956  127.484563 | 100.000178 |     0.0 | 127.484563 |   3.467852 |
    | 2000-01-04 02:30:00 |  27.043877 | 3.467849  2.967858   2.46787  1.967887   1.46791  0.967941  0.467982  -0.031963  -0.531893  -1.031801  -1.531683  -2.031532  -2.531342  -3.031102  -3.530802  -4.030429  -4.529967    -5.0294  -5.528707  -6.027864  -6.526845  -7.025618  -7.524149  -8.022399  -8.520323  -9.017872  -9.514993  -10.011624  -10.507701   -11.00315  -11.497894  -11.991845   -12.48491  -12.976986  -13.467964  -13.957723  -14.446133  -14.933055  -15.418346  -15.901859   -16.38346  -16.863047  -17.340595  -17.816217  -18.290291  -18.763662  -19.237998  -19.716437  -20.204859   -20.714621 | 100.000142  100.000328  100.000608  100.001019  100.001608  100.002436  100.003582  100.005144  100.007247  100.010047  100.013739  100.018562  100.024809  100.032835  100.043072  100.056036  100.072342   100.09272   100.11803  100.149277   100.18763   100.23444  100.291261  100.359863  100.442259  100.540718  100.657786  100.796304  100.959424  101.150629  101.373751  101.632991  101.932944  102.278631  102.675538  103.129671   103.64763  104.236705  104.905015  105.661674  106.517026  107.482943  108.573222  109.804132  111.195189   112.77036  114.560053  116.604716   118.96187  121.721244  125.043378 | 100.000142 |     0.0 | 125.043378 |   3.467849 |
    | 2000-01-04 03:00:00 |  17.102875 | 3.467846  2.967854  2.467864  1.967878  1.467898  0.967924  0.467959  -0.031994  -0.531934  -1.031855  -1.531754  -2.031625   -2.53146  -3.031253  -3.530994  -4.030671  -4.530271  -5.029778  -5.529174  -6.028439  -6.527548  -7.026473  -7.525183  -8.023642  -8.521812  -9.019646  -9.517095  -10.014104  -10.510612  -11.006553  -11.501853  -11.996432  -12.490202  -12.983068  -13.474928  -13.965669  -14.455171  -14.943306  -15.429944  -15.914954  -16.398222  -16.879672  -17.359309  -17.837293  -18.314062  -18.790556  -19.268597  -19.751604  -20.246011   -20.764464 | 100.000113  100.000268  100.000502  100.000847  100.001343  100.002042   100.00301  100.004333  100.006119  100.008502  100.011649  100.015768  100.021114  100.027997  100.036791   100.04795  100.062013  100.079623  100.101538  100.128647  100.161989  100.202767  100.252369  100.312384  100.384621  100.471131  100.574222  100.696479  100.840782  101.010326  101.208637  101.439599   101.70747  102.016918  102.373057   102.78149  103.248382  103.780542  104.385533  105.071831  105.849017  106.728039  107.721561  108.844448  110.114461  111.553366  113.188791  115.057657  117.213022  119.739126  122.790894 | 100.000113 |     0.0 | 122.790894 |   3.467846 |
    | 2000-01-04 03:30:00 |   8.845638 | 3.467844   2.96785  2.467859  1.967871  1.467888   0.96791   0.46794   -0.03202  -0.531969  -1.031901  -1.531815  -2.031703  -2.531562  -3.031384   -3.53116   -4.03088  -4.530533  -5.030105   -5.52958  -6.028938  -6.528159  -7.027218  -7.526087  -8.024732  -8.523119  -9.021207  -9.518949  -10.016297  -10.513193  -11.009576  -11.505379  -12.000526  -12.494938  -12.988525   -13.48119  -13.972832  -14.463337  -14.952589   -15.44047  -15.926863  -16.411673  -16.894846  -17.376416  -17.856583  -18.335842  -18.815219  -19.296683  -19.783919  -20.283904   -20.810581 |  100.00009  100.000219  100.000415  100.000704  100.001122  100.001711   100.00253  100.003651  100.005167  100.007194  100.009876  100.013394  100.017968  100.023868  100.031421  100.041022  100.053146  100.068356  100.087321  100.110828  100.139797  100.175297  100.218568  100.271031  100.334312  100.410259   100.50096  100.608762   100.73629  100.886466  101.062528  101.268054  101.506982  101.783638  102.102774   102.46961   102.88989  103.369957  103.916855  104.538455  105.243629  106.042473  106.946612  107.969627  109.127678  110.440516  111.933201  113.639357  115.607759  117.917167  120.716195 |  100.00009 |     0.0 | 120.716195 |   3.467844 |
    | 2000-01-04 04:00:00 |   1.179311 | 3.467842  2.967847  2.467855  1.967865  1.467879  0.967898  0.467924  -0.032043  -0.531998  -1.031941  -1.531866  -2.031771  -2.531649  -3.031496  -3.531302   -4.03106   -4.53076  -5.030388  -5.529931  -6.029372  -6.528692  -7.027868  -7.526876  -8.025686  -8.524266  -9.022579  -9.520583  -10.018233  -10.515477  -11.012258  -11.508515  -12.004177   -12.49917  -12.993412  -13.486814  -13.979279  -14.470705  -14.960984  -15.450008  -15.937677  -16.423909  -16.908673  -17.392029  -17.874211  -18.355766  -18.837801   -19.32242  -19.813563  -20.318733   -20.853165 | 100.000072  100.000179  100.000343  100.000586  100.000937  100.001435  100.002127  100.003077  100.004363  100.006087  100.008373  100.011377  100.015289  100.020345  100.026829  100.035088  100.045535  100.058667  100.075072  100.095444  100.120598  100.151485  100.189208  100.235037   100.29043  100.357049   100.43678  100.531749  100.644342  100.777227  100.933368  101.116052  101.328911  101.575946  101.861561  102.190603  102.568415  103.000898  103.494599  104.056822  104.695776  105.420772   106.24248  107.173309  108.227962  109.424347  110.785169  112.340981  114.136491  116.245056  118.808657 | 100.000072 |     0.0 | 118.808657 |   3.467842 |
    | 2000-01-04 04:30:00 | 220.245269 | 3.467841  2.967845  2.467851   1.96786  1.467872  0.967888   0.46791  -0.032061  -0.532024  -1.031974  -1.531911  -2.031829  -2.531724  -3.031592  -3.531425  -4.031215  -4.530955  -5.030632  -5.530235  -6.029748  -6.529155  -7.028435  -7.527566  -8.026521  -8.525272  -9.023785  -9.522022  -10.019942  -10.517497  -11.014636  -11.511301  -12.007428  -12.502948  -12.997785  -13.491857  -13.985074  -14.477342  -14.968563  -15.458639  -15.947481  -16.435025  -16.921255  -17.406257  -17.890296  -18.373967  -18.858448  -19.345971  -19.840715  -20.350692   -20.892388 | 100.000057  100.000147  100.000284  100.000488  100.000784  100.001203  100.001788  100.002593  100.003685  100.005152  100.007099  100.009663  100.013009  100.017341  100.022907  100.030008  100.039009  100.050342  100.064526  100.082174  100.104005  100.130864   100.16373  100.203739  100.252195   100.31059  100.380623  100.464217  100.563536   100.68101  100.819346  100.981558  101.170985  101.391317  101.646628  101.941408  102.280615  102.669727  103.114822  103.622671  104.200866  104.857988  105.603834  106.449742  107.409091  108.498111  109.737354   111.15454  112.790552  114.713494   117.05698 | 100.000057 |     0.0 |  117.05698 |   3.467841 |
    | 2000-01-04 05:00:00 |  215.15899 | 3.467839  2.967843  2.467848  1.967856  1.467866   0.96788  0.467898  -0.032077  -0.532045  -1.032003  -1.531948  -2.031878  -2.531788  -3.031674   -3.53153  -4.031349  -4.531124  -5.030844  -5.530499  -6.030075  -6.529557  -7.028928  -7.528167  -8.027251  -8.526153  -9.024843  -9.523288  -10.021448  -10.519282  -11.016741  -11.513773  -12.010319  -12.506316  -13.001693  -13.496373  -13.990276  -14.483313  -14.975396  -15.466437  -15.956357  -16.445106  -16.932686  -17.419203  -17.904951  -18.390567  -18.877297  -19.367489  -19.865546  -20.379968   -20.928466 | 100.000046   100.00012  100.000235  100.000406  100.000655  100.001009  100.001503  100.002185  100.003112  100.004359  100.006018  100.008206  100.011066  100.014776  100.019551  100.025655  100.033404   100.04318  100.055437  100.070715  100.089651  100.112991  100.141606  100.176508  100.218861  100.270005  100.331466  100.404979  100.492505  100.596249  100.718681  100.862555  101.030934  101.227214  101.455152    101.7189  102.023046  102.372668  102.773398  103.231504  103.754001  104.348795  105.024878  105.792611  106.664164  107.654249  108.781466  110.070945  111.559992   113.31157  115.452035 | 100.000046 |     0.0 | 115.452035 |   3.467839 |
    | 2000-01-04 05:30:00 | 210.580853 | 3.467838  2.967841  2.467846  1.967852  1.467861  0.967872  0.467888  -0.032091  -0.532063  -1.032027   -1.53198   -2.03192  -2.531843  -3.031745   -3.53162  -4.031464  -4.531269  -5.031026  -5.530727  -6.030358  -6.529907  -7.029357  -7.528692  -8.027889  -8.526924  -9.025772    -9.5244  -10.022775  -10.520857  -11.018603  -11.515965  -12.012888  -12.509315   -13.00518  -13.500413  -13.994939  -14.488678  -14.981548  -15.473472  -15.964381  -16.454237  -16.943057  -17.430966  -17.918285  -18.405688  -18.894482  -19.387122  -19.888221  -20.406741   -20.961592 | 100.000036  100.000098  100.000195  100.000338  100.000548  100.000846  100.001265  100.001842  100.002629  100.003689  100.005103   100.00697  100.009415  100.012592  100.016689  100.021934  100.028605  100.037035  100.047624  100.060847  100.077264  100.097538   100.12244   100.15287  100.189869  100.234633  100.288535  100.353136  100.430209  100.521752  100.630012  100.757502  100.907024  101.081697  101.284976  101.520692  101.793085  102.106851  102.467199  102.879926   103.35151  103.889231  104.501342  105.197315  105.988223  106.887396  107.911636  109.083647  110.437353  112.030816  113.983354 | 100.000036 |     0.0 | 113.983354 |   3.467838 |
    | 2000-01-04 06:00:00 | 206.370107 | 3.467837   2.96784  2.467844  1.967849  1.467856  0.967866   0.46788  -0.032102  -0.532079  -1.032048  -1.532008  -2.031956   -2.53189  -3.031805  -3.531698  -4.031563  -4.531394  -5.031184  -5.530924  -6.030603   -6.53021  -7.029731  -7.529149  -8.028445  -8.527599  -9.026586  -9.525378  -10.023943  -10.522247  -11.020249  -11.517906  -12.015168  -12.511982  -13.008288  -13.504022  -13.999114  -14.493491   -14.98708   -15.47981  -15.971624  -16.462495  -16.952452  -17.441639    -17.9304  -18.419442  -18.910128   -19.40501  -19.908898  -20.431189   -20.991955 | 100.000029  100.000081  100.000161  100.000282  100.000458   100.00071  100.001064  100.001553  100.002221  100.003123  100.004327   100.00592   100.00801   100.01073  100.014243  100.018749   100.02449  100.031758  100.040902   100.05234  100.066568  100.084169  100.105828  100.132344  100.164644  100.203798  100.251036  100.307763  100.375575  100.456283  100.551924  100.664786   100.79743   100.95271    101.1338  101.344225  101.587893   101.86914  102.192779  102.564161  102.989262  103.474781  104.028293  104.658453  105.375337  106.191019  107.120674  108.184814  109.414256  110.862403  112.641421 | 100.000029 |     0.0 | 112.641421 |   3.467837 |
    | 2000-01-04 06:30:00 | 202.591014 | 3.467837  2.967839  2.467842  1.967847  1.467853  0.967861  0.467873  -0.032112  -0.532092  -1.032066  -1.532031  -2.031987   -2.53193  -3.031857  -3.531765  -4.031648  -4.531502   -5.03132  -5.531094  -6.030816  -6.530474  -7.030055  -7.529547  -8.028931  -8.528189  -9.027299  -9.526235   -10.02497  -10.523471  -11.021702  -11.519623   -12.01719  -12.514352  -13.011056  -13.507242  -14.002847  -14.497804  -14.992047  -15.485513  -15.978154  -16.469953  -16.960952   -17.45131  -17.941392  -18.431936  -18.924354  -19.421288  -19.927727  -20.453482   -21.019741 | 100.000023  100.000066  100.000134  100.000235  100.000384  100.000597  100.000896   100.00131  100.001877  100.002643  100.003669  100.005028  100.006814  100.009142  100.012155  100.016025  100.020964  100.027227   100.03512  100.045011  100.057336  100.072609  100.091437  100.114529   100.14271  100.176935  100.218305  100.268079  100.327697  100.398793  100.483211  100.583033   100.70059   100.83849  100.999643  101.187284   101.40501  101.656816  101.947139  102.280914  102.663647    103.1015   103.60142  104.171315  104.820355  105.559479   106.40239   107.36761  108.483098  109.797856  111.416925 | 100.000023 |     0.0 | 111.416925 |   3.467837 |
    | 2000-01-04 07:00:00 |  199.48961 | 3.467836  2.967838  2.467841  1.967844   1.46785  0.967857  0.467867  -0.032121  -0.532103  -1.032081  -1.532051  -2.032013  -2.531964  -3.031902  -3.531822  -4.031722  -4.531595  -5.031438  -5.531242     -6.031  -6.530702  -7.030337  -7.529893  -8.029355  -8.528704  -9.027923  -9.526987  -10.025872  -10.524548  -11.022984  -11.521141   -12.01898  -12.516456  -13.013518  -13.510112  -14.006181  -14.501665  -14.996502  -15.490638  -15.984033   -16.47668  -16.968632  -17.460061  -17.951352   -18.44327  -18.937271   -19.43608  -19.944853  -20.473784    -21.04513 | 100.000018  100.000054  100.000111  100.000196  100.000321  100.000501  100.000754  100.001105  100.001586  100.002238  100.003111  100.004271  100.005797  100.007789  100.010372  100.013695  100.017942  100.023338  100.030149  100.038699   100.04937  100.062617  100.078976  100.099075  100.123648  100.153546  100.189752  100.233395   100.28577  100.348348  100.422799  100.511008  100.615097  100.737444  100.880708  101.047855  101.242189  101.467382  101.727519  102.027144  102.371324  102.765722  103.216705  103.731502  104.318455  104.987477  105.750942  106.625568   107.63669  108.829169  110.301034 | 100.000018 |     0.0 | 110.301034 |   3.467836 |
    | 2000-01-04 07:30:00 | 196.683766 | 3.467835  2.967837  2.467839  1.967843  1.467847  0.967853  0.467861  -0.032128  -0.532113  -1.032094  -1.532068  -2.032036  -2.531994   -3.03194  -3.531871  -4.031785  -4.531675  -5.031539  -5.531369  -6.031159    -6.5309  -7.030582  -7.530195  -8.029724  -8.529154  -9.028468  -9.527646  -10.026664  -10.525496  -11.024113  -11.522482  -12.020565  -12.518321  -13.015705  -13.512668  -14.009156  -14.505115  -15.000492  -15.495237   -15.98932  -16.482739  -16.975561   -17.46797  -17.960366  -18.453539  -18.948987  -19.449507   -19.96041  -20.492248   -21.068293 | 100.000014  100.000045  100.000092  100.000164  100.000269  100.000421  100.000635  100.000932  100.001341  100.001895  100.002639  100.003627  100.004931  100.006636  100.008849  100.011702  100.015354  100.020001  100.025876  100.033263  100.042499  100.053984  100.068191  100.085676  100.107091  100.133193  100.164859  100.203101  100.249079  100.304118  100.369725  100.447607  100.539688  100.648134  100.775371  100.924112  101.097383  101.298556  101.531386   101.80005  102.109204  102.464049  102.870419  103.334916  103.865133  104.470053  105.160853  105.952613  106.868267  107.948805  109.285387 | 100.000014 |     0.0 | 109.285387 |   3.467835 |
    | 2000-01-04 08:00:00 | 194.126645 | 3.467835  2.967836  2.467838  1.967841  1.467845   0.96785  0.467857  -0.032134  -0.532121  -1.032105  -1.532083  -2.032055  -2.532019  -3.031973  -3.531913  -4.031839  -4.531744  -5.031626  -5.531479  -6.031297  -6.531072  -7.030795  -7.530457  -8.030045  -8.529547  -9.028946  -9.528223  -10.027359   -10.52633  -11.025108  -11.523665  -12.021966  -12.519973  -13.017646   -13.51494  -14.011807  -14.508197  -15.004062  -15.499361  -15.994067  -16.488191  -16.981806  -17.475108  -17.968513  -18.462832  -18.959599  -19.461679  -19.974524  -20.509019   -21.089395 | 100.000011  100.000037  100.000077  100.000137  100.000226  100.000354  100.000535  100.000787  100.001134  100.001604  100.002238  100.003081  100.004195  100.005654   100.00755  100.009998  100.013138  100.017138  100.022205  100.028585  100.036576  100.046528   100.05886  100.074064  100.092717  100.115492  100.143171  100.176658  100.216992  100.265365  100.323134  100.391841   100.47323  100.569268  100.682164  100.814395  100.968729  101.148257  101.356421  101.597057   101.87444  102.193339  102.559097  102.977744  103.456188  104.002573  104.626992  105.343047  106.171476  107.149663  108.362106 | 100.000011 |     0.0 | 108.362106 |   3.467835 |
    | 2000-01-04 08:30:00 | 191.825016 | 3.467835  2.967836  2.467837   1.96784  1.467843  0.967847  0.467853  -0.032139  -0.532128  -1.032114  -1.532095  -2.032071   -2.53204  -3.032001   -3.53195  -4.031885  -4.531804  -5.031702  -5.531574  -6.031416   -6.53122   -7.03098  -7.530685  -8.030325  -8.529889  -9.029362  -9.528728  -10.027968  -10.527062  -11.025984  -11.524708  -12.023203  -12.521436  -13.019368   -13.51696  -14.014167  -14.510946  -15.007253  -15.503052  -15.998326   -16.49309  -16.987427  -17.481543  -17.975867  -18.471231    -18.9692  -19.472701  -19.987314  -20.524234   -21.108591 | 100.000009   100.00003  100.000064  100.000114  100.000189  100.000297   100.00045  100.000664  100.000958  100.001359  100.001898  100.002617  100.003569  100.004816  100.006441  100.008542   100.01124  100.014683  100.019051   100.02456   100.03147  100.040091  100.050791  100.064005  100.080243  100.100104  100.124284  100.153589  100.188949  100.231433  100.282263  100.342828  100.414707  100.499682  100.599764  100.717206  100.854539  101.014587  101.200504  101.415806  101.664413  101.950694  102.279538   102.65645  103.087717   103.58072  104.144576  104.791547  105.540374  106.425089  107.523785 | 100.000009 |     0.0 | 107.523785 |   3.467835 |
    | 2000-01-04 09:00:00 | 189.943054 | 3.467834  2.967835  2.467837  1.967839  1.467841  0.967845   0.46785  -0.032143  -0.532134  -1.032122  -1.532106  -2.032085  -2.532059  -3.032025  -3.531981  -4.031925  -4.531855  -5.031767  -5.531656  -6.031519  -6.531349   -7.03114  -7.530883  -8.030569  -8.530188  -9.029726   -9.52917  -10.028502  -10.527704  -11.026754  -11.525627  -12.024295  -12.522729  -13.020894  -13.518752  -14.016266  -14.513395  -15.010101  -15.506354  -16.002142  -16.497487   -16.99248  -17.487337  -17.982498  -18.478813  -18.977876  -19.482671  -19.998892   -20.53802   -21.126031 | 100.000007  100.000025  100.000053  100.000096  100.000159   100.00025   100.00038  100.000561   100.00081  100.001151   100.00161  100.002223  100.003036  100.004103  100.005495  100.007297  100.009615  100.012577  100.016342  100.021097  100.027071  100.034536  100.043816  100.055295  100.069424  100.086735  100.107846  100.133476  100.164456  100.201743  100.246434  100.299782  100.363211  100.438336  100.526979  100.631193  100.753281  100.895823  101.061703   101.25414  101.476721  101.733446  102.028786  102.367764  102.756099  103.200483   103.70915  104.293155  104.969416  105.768867  106.763479 | 100.000007 |     0.0 | 106.763479 |   3.467834 |
    | 2000-01-04 09:30:00 | 188.273318 | 3.467834  2.967835  2.467836  1.967838   1.46784  0.967843  0.467848  -0.032147  -0.532139  -1.032129  -1.532115  -2.032097  -2.532075  -3.032045  -3.532007   -4.03196  -4.531899  -5.031823  -5.531727  -6.031608   -6.53146  -7.031278  -7.531055  -8.030781  -8.530448  -9.030044  -9.529556   -10.02897  -10.528268  -11.027431  -11.526436  -12.025259  -12.523872  -13.022244  -13.520342  -14.018131  -14.515575  -15.012641  -15.509304  -16.005557  -16.501429  -16.997018  -17.492548   -17.98847  -18.485649  -18.985708  -19.491677  -20.009361  -20.550499   -21.141855 | 100.000006   100.00002  100.000044   100.00008  100.000133   100.00021   100.00032  100.000473  100.000686  100.000975  100.001366  100.001889  100.002583  100.003495  100.004687  100.006233  100.008224  100.010773  100.014016  100.018119  100.023282  100.029743  100.037788  100.047756  100.060045  100.075125  100.093547  100.115949  100.143074  100.175779  100.215046  100.262003  100.317933  100.384296  100.462743  100.555136  100.663572  100.790404  100.938262  101.110089  101.309165  101.539149  101.804122  102.108668  102.457987  102.858146  103.316583   103.84326  104.453444  105.175207  106.074699 | 100.000006 |     0.0 | 106.074699 |   3.467834 |
    | 2000-01-04 10:00:00 | 186.763132 | 3.467834  2.967835  2.467836  1.967837  1.467839  0.967842  0.467845   -0.03215  -0.532143  -1.032134  -1.532123  -2.032108  -2.532088  -3.032063   -3.53203  -4.031989  -4.531937  -5.031871  -5.531788  -6.031685  -6.531557  -7.031399  -7.531204  -8.030965  -8.530674  -9.030321  -9.529894  -10.029379  -10.528762  -11.028025  -11.527148  -12.026108  -12.524881  -13.023438  -13.521751  -14.019787  -14.517514  -15.014904  -15.511937  -16.008611   -16.50496  -17.001089  -17.497229  -17.993842  -18.491807  -18.992769  -19.499806  -20.018816   -20.56178   -21.156194 | 100.000004  100.000017  100.000037  100.000067  100.000112  100.000177   100.00027    100.0004   100.00058  100.000826  100.001159  100.001605  100.002197  100.002977  100.003998  100.005324  100.007034  100.009226  100.012019  100.015559  100.020019   100.02561  100.032582  100.041233  100.051916  100.065047  100.081114  100.100684  100.124421  100.153088  100.187567  100.228869  100.278151  100.336728  100.406095  100.487939  100.584166  100.696915  100.828585  100.981861  101.159739  101.365558  101.603047  101.876381   102.19029  102.550271  102.963045  103.437588  103.987665  104.638719  105.451397 | 100.000004 |     0.0 | 105.451397 |   3.467834 |
    | 2000-01-04 10:30:00 | 185.400963 | 3.467834  2.967834  2.467835  1.967836  1.467838   0.96784  0.467843  -0.032152  -0.532147  -1.032139  -1.532129  -2.032116    -2.5321  -3.032078   -3.53205  -4.032014  -4.531969  -5.031912  -5.531841  -6.031751   -6.53164  -7.031503  -7.531333  -8.031125  -8.530871  -9.030562  -9.530188  -10.029737  -10.529195  -11.028546  -11.527773  -12.026855  -12.525771  -13.024494  -13.522998  -14.021255  -14.519237  -15.016919  -15.514285  -16.011338  -16.508118  -17.004736   -17.50143  -17.998669  -18.497348   -18.99913  -19.507134  -20.027347  -20.571969   -21.169171 | 100.000004  100.000014  100.000031  100.000056  100.000094  100.000149  100.000228  100.000338  100.000491    100.0007  100.000983  100.001363  100.001869  100.002536   100.00341  100.004547  100.006015    100.0079  100.010306  100.013358   100.01721  100.022046  100.028086  100.035591  100.044875  100.056303  100.070308  100.087396  100.108155  100.133268  100.163523  100.199828  100.243221  100.294887  100.356177  100.428618  100.513937  100.614079  100.731227  100.867827   101.02661  101.210626  101.423271  101.668349  101.950161  102.273687  102.644998  103.072182  103.567641  104.154406  104.887947 | 100.000004 |     0.0 | 104.887947 |   3.467834 |
    | 2000-01-04 11:00:00 | 184.277259 | 3.467834  2.967834  2.467835  1.967836  1.467837  0.967839  0.467842  -0.032155   -0.53215  -1.032143  -1.532135  -2.032124  -2.532109  -3.032091  -3.532067  -4.032036  -4.531997  -5.031948  -5.531886  -6.031809  -6.531712  -7.031593  -7.531445  -8.031264  -8.531043  -9.030773  -9.530445   -10.03005  -10.529574  -11.029004  -11.528323  -12.027513  -12.526555  -13.025426  -13.524101  -14.022556  -14.520766   -15.01871  -15.516376  -16.013771  -16.510941  -17.008001  -17.505195  -18.003003  -18.502327  -19.004853  -19.513733  -20.035036  -20.581161   -21.180902 | 100.000003  100.000012  100.000026  100.000047  100.000079  100.000125  100.000192  100.000286  100.000415  100.000593  100.000834  100.001159   100.00159  100.002161  100.002909  100.003883  100.005144  100.006764  100.008835  100.011467  100.014793  100.018974  100.024204  100.030714  100.038777  100.048719  100.060922  100.075834   100.09398  100.115966  100.142499   100.17439  100.212571   100.25811  100.312222  100.376289  100.451875  100.540744   100.64488  100.766507  100.908115   101.07248  101.262698  101.482231  101.734985  102.025473  102.359179  102.743387  103.189267  103.717641   104.37913 | 100.000003 |     0.0 |  104.37913 |   3.467834 |
    | 2000-01-04 11:30:00 | 183.313545 | 3.467833  2.967834  2.467834  1.967835  1.467837  0.967838   0.46784  -0.032156  -0.532152  -1.032147   -1.53214   -2.03213  -2.532118  -3.032102  -3.532081  -4.032055  -4.532021  -5.031979  -5.531925  -6.031858  -6.531775  -7.031671  -7.531543  -8.031385  -8.531192  -9.030956   -9.53067  -10.030323  -10.529906  -11.029405  -11.528805  -12.028092  -12.527246  -13.026248  -13.525076  -14.023708  -14.522123  -15.020301  -15.518237   -16.01594  -16.513461   -17.01092  -17.508568  -18.006889  -18.506798  -19.009997  -19.519671   -20.04196  -20.589445   -21.191494 | 100.000002   100.00001  100.000021  100.000039  100.000066  100.000106  100.000162  100.000241  100.000352  100.000503  100.000708  100.000985  100.001353  100.001841  100.002481  100.003316  100.004398  100.005791  100.007574  100.009842  100.012713  100.016327  100.020855  100.026498  100.033499  100.042144  100.052772  100.065779  100.081631   100.10087  100.124125  100.152121  100.185694  100.225804  100.273545  100.330164  100.397074  100.475874  100.568364  100.676565  100.802742   100.94942  101.119418  101.315883  101.542363  101.802947  102.102584  102.447836  102.848758   103.32415  103.920114 | 100.000002 |     0.0 | 103.920114 |   3.467833 |
    | 2000-01-04 12:00:00 |  182.44766 | 3.467833  2.967834  2.467834  1.967835  1.467836  0.967837  0.467839  -0.032158  -0.532155   -1.03215  -1.532144  -2.032136  -2.532125  -3.032111  -3.532093  -4.032071  -4.532042  -5.032005  -5.531959  -6.031901  -6.531829  -7.031739  -7.531627   -8.03149  -8.531321  -9.031116  -9.530865  -10.030562  -10.530196  -11.029756  -11.529229  -12.028601  -12.527855  -13.026974  -13.525938  -14.024728  -14.523325  -15.021714  -15.519892  -16.017872  -16.515709  -17.013529  -17.511585  -18.010371  -18.510809  -19.014616  -19.525007  -20.048188  -20.596903   -21.201047 | 100.000002  100.000008  100.000018  100.000033  100.000056  100.000089  100.000137  100.000204  100.000298  100.000426  100.000601  100.000837  100.001151  100.001568  100.002116  100.002832  100.003761  100.004958  100.006492  100.008446  100.010924  100.014047  100.017965  100.022856  100.028932  100.036446  100.045698  100.057038   100.07088  100.087706  100.108075  100.132638  100.162141  100.197446  100.239536  100.289536  100.348721  100.418537  100.500616  100.596791   100.70912  100.839898  100.991687  101.167349  101.370101  101.603646  101.872454  102.182432  102.542625  102.969993  103.506436 | 100.000002 |     0.0 | 103.506436 |   3.467833 |
    | 2000-01-04 12:30:00 | 181.670569 | 3.467833  2.967834  2.467834  1.967835  1.467835  0.967837  0.467838  -0.032159  -0.532156  -1.032152  -1.532147   -2.03214  -2.532131  -3.032119  -3.532104  -4.032085   -4.53206  -5.032028  -5.531988  -6.031938  -6.531875  -7.031797    -7.5317  -8.031581  -8.531434  -9.031254  -9.531036   -10.03077  -10.530449  -11.030063    -11.5296  -12.029047   -12.52839  -13.027613  -13.526698  -14.025629  -14.524389  -15.022967  -15.521362  -16.019591  -16.517713  -17.015856  -17.514282  -18.013487  -18.514402  -19.018759  -19.529799  -20.053786  -20.603612   -21.209654 | 100.000001  100.000007  100.000015  100.000028  100.000047  100.000075  100.000116  100.000173  100.000252  100.000361   100.00051  100.000711   100.00098  100.001336  100.001805  100.002419  100.003215  100.004244  100.005564  100.007248  100.009385  100.012083  100.015473   100.01971  100.024981   100.03151   100.03956  100.049443  100.061524  100.076231  100.094064  100.115601  100.141512  100.172567  100.209651  100.253774  100.306087  100.367897  100.440678  100.526094   100.62601  100.742511  100.877923  101.034844  101.216192   101.42532  101.666259   101.94433  102.267664  102.651543  103.133978 | 100.000001 |     0.0 | 103.133978 |   3.467833 |
    | 2000-01-04 13:00:00 | 181.021617 | 3.467833  2.967833  2.467834  1.967834  1.467835  0.967836  0.467837  -0.032161  -0.532158  -1.032155   -1.53215  -2.032144  -2.532136  -3.032126  -3.532113  -4.032096  -4.532075  -5.032048  -5.532013   -6.03197  -6.531915  -7.031848  -7.531763   -8.03166  -8.531532  -9.031375  -9.531184  -10.030952  -10.530671  -11.030332  -11.529925  -12.029439  -12.528861  -13.028175  -13.527368  -14.026425  -14.525331  -15.024078  -15.522666  -16.021119  -16.519497  -17.017932   -17.51669  -18.016273   -18.51762  -19.022473  -19.534098  -20.058811  -20.609641     -21.2174 | 100.000001  100.000005  100.000012  100.000023   100.00004  100.000063  100.000098  100.000146  100.000213  100.000306  100.000433  100.000605  100.000834  100.001138   100.00154  100.002065  100.002749  100.003632  100.004768  100.006218  100.008062  100.010392  100.013323  100.016993  100.021565  100.027236  100.034237  100.042846  100.053385  100.066234  100.081838  100.100712  100.123454  100.150754  100.183405  100.222316  100.268523  100.323203  100.387691  100.463491  100.552293  100.655988  100.776687  100.916745  101.078809  101.265909  101.481685  101.730923  102.020931  102.365467  102.798952 | 100.000001 |     0.0 | 102.798952 |   3.467833 |
    | 2000-01-04 13:30:00 | 180.482224 | 3.467833  2.967833  2.467834  1.967834  1.467835  0.967836  0.467837  -0.032162  -0.532159  -1.032157  -1.532153  -2.032148  -2.532141  -3.032132  -3.532121  -4.032106  -4.532088  -5.032064  -5.532035  -6.031997   -6.53195  -7.031891  -7.531818  -8.031728  -8.531617   -9.03148  -9.531313   -10.03111  -10.530864  -11.030567   -11.53021  -12.029783  -12.529274  -13.028671  -13.527959  -14.027128  -14.526164  -15.025061  -15.523824  -16.022477  -16.521084  -17.019782  -17.518838  -18.018762  -18.520497  -19.025798  -19.537951  -20.063319  -20.615054   -21.224365 | 100.000001  100.000004   100.00001   100.00002  100.000033  100.000053  100.000082  100.000124  100.000181   100.00026  100.000368  100.000514   100.00071   100.00097  100.001313  100.001764   100.00235  100.003109  100.004085  100.005335  100.006924  100.008936  100.011471  100.014648  100.018612  100.023535  100.029623  100.037118  100.046307  100.057528  100.071175  100.087706  100.107656  100.131641  100.160371  100.194663  100.235447  100.283786  100.340883  100.408098   100.48696  100.579182  100.686677  100.811579  100.956285  101.123536  101.316614  101.539825  101.799733   102.10871  102.497875 | 100.000001 |     0.0 | 102.497875 |   3.467833 |
    | 2000-01-04 14:00:00 | 180.000368 | 3.467833  2.967833  2.467833  1.967834  1.467834  0.967835  0.467836  -0.032162  -0.532161  -1.032158  -1.532155   -2.03215  -2.532145  -3.032137  -3.532128  -4.032115  -4.532099  -5.032079  -5.532053  -6.032021   -6.53198  -7.031929  -7.531866  -8.031787   -8.53169  -9.031571  -9.531426  -10.031249  -10.531033  -11.030773   -11.53046  -12.030085  -12.529637  -13.029106   -13.52848  -14.027748  -14.526899  -15.025931   -15.52485  -16.023682  -16.522495  -17.021428  -17.520753  -18.020983  -18.523069  -19.028773  -19.541401  -20.067359  -20.619909    -21.23062 | 100.000001  100.000004  100.000009  100.000016  100.000028  100.000045   100.00007  100.000105  100.000153  100.000221  100.000312  100.000437  100.000604  100.000826   100.00112  100.001506  100.002009  100.002661    100.0035  100.004576  100.005946  100.007683  100.009874  100.012624  100.016059  100.020332  100.025623  100.032146  100.040156   100.04995  100.061879  100.076351  100.093841  100.114901  100.140167  100.170368  100.206343  100.249047  100.299564  100.359121  100.429102  100.511055  100.606713  100.718009   100.84711  100.996493  101.169119  101.368858  101.601605  101.878473  102.227552 | 100.000001 |     0.0 | 102.227552 |   3.467833 |
    | 2000-01-04 14:30:00 | 179.570325 | 3.467833  2.967833  2.467833  1.967834  1.467834  0.967835  0.467836  -0.032163  -0.532162  -1.032159  -1.532157  -2.032153  -2.532148  -3.032142  -3.532133  -4.032123  -4.532109  -5.032091  -5.532069  -6.032041  -6.532006  -7.031962  -7.531907  -8.031839  -8.531754  -9.031651  -9.531524  -10.031369  -10.531181  -11.030953  -11.530678  -12.030349  -12.529956  -13.029489  -13.528938  -14.028294  -14.527549  -15.026701  -15.525758  -16.024751  -16.523748  -17.022892  -17.522459  -18.022964  -18.525365  -19.031432  -19.544488  -20.070977   -20.62426   -21.236233 | 100.000001  100.000003  100.000007  100.000014  100.000024  100.000038  100.000059  100.000088   100.00013  100.000187  100.000265  100.000372  100.000514  100.000704  100.000956  100.001286  100.001718  100.002277  100.002999  100.003925  100.005106  100.006605  100.008498  100.010877  100.013854  100.017561  100.022158  100.027833  100.034811  100.043356  100.053779  100.066441  100.081768   100.10025  100.122455  100.149038   100.18075  100.218449  100.263113  100.315848  100.377901  100.450674  100.535732  100.634824   100.74991  100.883227  101.037442  101.216036  101.424295  101.672197  101.985059 | 100.000001 |     0.0 | 101.985059 |   3.467833 |
    | 2000-01-04 15:00:00 | 179.214663 | 3.467833  2.967833  2.467833  1.967834  1.467834  0.967834  0.467835  -0.032164  -0.532162  -1.032161  -1.532158  -2.032155  -2.532151  -3.032145  -3.532138  -4.032129  -4.532117  -5.032102  -5.532083  -6.032059  -6.532028   -7.03199  -7.531942  -8.031883   -8.53181   -9.03172  -9.531609  -10.031474  -10.531309   -11.03111  -11.530869   -12.03058  -12.530235  -13.029825  -13.529341  -14.028776  -14.528122  -15.027381  -15.526562  -16.025698   -16.52486  -17.024194  -17.523977   -18.02473  -18.527413  -19.033806  -19.547248  -20.074214  -20.628155   -21.241265 |      100.0  100.000003  100.000006  100.000012   100.00002  100.000032   100.00005  100.000075   100.00011  100.000159  100.000225  100.000316  100.000438    100.0006  100.000815  100.001098  100.001468  100.001949  100.002569  100.003366  100.004384  100.005677  100.007313  100.009371  100.011949  100.015165  100.019157  100.024093   100.03017  100.037622  100.046724  100.057798  100.071221  100.087431  100.106936   100.13032  100.158257  100.191518   100.23098  100.277641  100.332625  100.397197  100.472773  100.560931  100.663443  100.782327  100.919987  101.079548  101.265753  101.487548   101.76772 |      100.0 |     0.0 |  101.76772 |   3.467833 |
    | 2000-01-04 15:30:00 | 178.921514 | 3.467833  2.967833  2.467833  1.967833  1.467834  0.967834  0.467835  -0.032164  -0.532163  -1.032162   -1.53216  -2.032157  -2.532153  -3.032148  -3.532142  -4.032134  -4.532124  -5.032111  -5.532095  -6.032074  -6.532047  -7.032014  -7.531973  -8.031922  -8.531858  -9.031779  -9.531683  -10.031565  -10.531421  -11.031247  -11.531036  -12.030783   -12.53048   -13.03012  -13.529696    -14.0292  -14.528628  -15.027981  -15.527273  -16.026536  -16.525846  -17.025349  -17.525327  -18.026302  -18.529239  -19.035925  -19.549712  -20.077107  -20.631641   -21.245772 |      100.0  100.000002  100.000005   100.00001  100.000017  100.000027  100.000042  100.000063  100.000093  100.000135  100.000192  100.000269  100.000373  100.000511  100.000696  100.000938  100.001255  100.001668  100.002201  100.002887  100.003764  100.004879  100.006292  100.008072  100.010305  100.013092  100.016558  100.020849   100.02614  100.032636  100.040581  100.050262  100.062012  100.076223  100.093346  100.113904  100.138501  100.167827  100.202671  100.243929  100.292616  100.349871   100.41697  100.495342  100.586583  100.692516  100.815302   100.95775  101.124109  101.322396  101.573092 |      100.0 |     0.0 | 101.573092 |   3.467833 |
    | 2000-01-04 16:00:00 | 178.660939 | 3.467833  2.967833  2.467833  1.967833  1.467834  0.967834  0.467835  -0.032165  -0.532164  -1.032162  -1.532161  -2.032158  -2.532155  -3.032151  -3.532146  -4.032139   -4.53213  -5.032119  -5.532105  -6.032087  -6.532064  -7.032035  -7.531999  -8.031955    -8.5319  -9.031831  -9.531747  -10.031644  -10.531519  -11.031366  -11.531182   -12.03096  -12.530695  -13.030379  -13.530007  -14.029573  -14.529073  -15.028511  -15.527901  -16.027279   -16.52672  -17.026375  -17.526526    -18.0277  -18.530865  -19.037814  -19.551912  -20.079692  -20.634757   -21.249804 |      100.0  100.000002  100.000004  100.000008  100.000014  100.000023  100.000036  100.000054  100.000079  100.000114  100.000163  100.000229  100.000317  100.000436  100.000593  100.000801  100.001073  100.001427  100.001885  100.002475  100.003231  100.004193  100.005413  100.006952  100.008885  100.011301  100.014309  100.018038  100.022642  100.028302  100.035235  100.043694  100.053975  100.066427  100.081451  100.099515  100.121158  100.146999  100.177746  100.214203  100.257284  100.308015  100.367547  100.437168  100.518318  100.612638  100.722075  100.849149  100.997666  101.174804  101.398946 |      100.0 |     0.0 | 101.398946 |   3.467833 |
    | 2000-01-04 16:30:00 | 178.430027 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967834  0.467834  -0.032165  -0.532164  -1.032163  -1.532162   -2.03216  -2.532157  -3.032153  -3.532149  -4.032143  -4.532136  -5.032126  -5.532114  -6.032098  -6.532078  -7.032053  -7.532022  -8.031984  -8.531936  -9.031876  -9.531803  -10.031713  -10.531604  -11.031471   -11.53131  -12.031115  -12.530883  -13.030607  -13.530281  -14.029901  -14.529465  -15.028979  -15.528456  -16.027935  -16.527494  -17.027285  -17.527591  -18.028943  -18.532313  -19.039498  -19.553874  -20.081999   -20.63754    -21.25341 |      100.0  100.000002  100.000004  100.000007  100.000012  100.000019   100.00003  100.000046  100.000067  100.000097  100.000138  100.000194   100.00027  100.000372  100.000506  100.000684  100.000917  100.001221  100.001615  100.002123  100.002773  100.003603  100.004656  100.005986  100.007659  100.009753  100.012363  100.015603  100.019607  100.024538  100.030585  100.037972  100.046964  100.057869  100.071044  100.086908  100.105942  100.128699  100.155813  100.188009  100.226104  100.271024  100.323805  100.385607  100.457729  100.541649  100.639116  100.752391  100.884882   101.04301  101.243256 |      100.0 |     0.0 | 101.243256 |   3.467833 |
    | 2000-01-04 17:00:00 |  178.24588 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967834  0.467834  -0.032165  -0.532165  -1.032164  -1.532162  -2.032161  -2.532158  -3.032155  -3.532152  -4.032147   -4.53214  -5.032132  -5.532121  -6.032108  -6.532091  -7.032069  -7.532042  -8.032009  -8.531967  -9.031915  -9.531852  -10.031773  -10.531678  -11.031562  -11.531421  -12.031251  -12.531048  -13.030806  -13.530521  -14.030189  -14.529811  -15.029391  -15.528946  -16.028516  -16.528179   -17.02809  -17.528536  -18.030047    -18.5336  -19.040996  -19.555622  -20.084056  -20.640023   -21.256631 |      100.0  100.000001  100.000003  100.000006   100.00001  100.000016  100.000026  100.000039  100.000057  100.000083  100.000118  100.000165   100.00023  100.000317  100.000432  100.000584  100.000784  100.001045  100.001383   100.00182   100.00238  100.003096  100.004005  100.005154  100.006601  100.008415  100.010679  100.013493  100.016975  100.021268   100.02654   100.03299   100.04085  100.050396  100.061945   100.07587  100.092599  100.112628  100.136526  100.164939  100.198605  100.238353  100.285117  100.339941  100.403994  100.478606  100.565349  100.666251   100.78436  100.925415  101.104176 |      100.0 |     0.0 | 101.104176 |   3.467833 |
    | 2000-01-04 17:30:00 | 178.090852 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967834  0.467834  -0.032166  -0.532165  -1.032164  -1.532163  -2.032162   -2.53216  -3.032157  -3.532154   -4.03215  -4.532144  -5.032137  -5.532128  -6.032116  -6.532101  -7.032083  -7.532059   -8.03203  -8.531994  -9.031949  -9.531894  -10.031826  -10.531742  -11.031641  -11.531518  -12.031369  -12.531192   -13.03098  -13.530731  -14.030443  -14.530115  -15.029753  -15.529378  -16.029028  -16.528785  -17.028804  -17.529374  -18.031028  -18.534744  -19.042329  -19.557178  -20.085889  -20.642238   -21.259506 |      100.0  100.000001  100.000003  100.000005  100.000009  100.000014  100.000022  100.000033  100.000048   100.00007    100.0001  100.000141  100.000196   100.00027  100.000369  100.000499   100.00067  100.000894  100.001185   100.00156  100.002043  100.002659  100.003444  100.004437  100.005689   100.00726  100.009223  100.011666  100.014693   100.01843  100.023024  100.028653  100.035521  100.043873  100.053992  100.066208  100.080904  100.098524  100.119573  100.144634  100.174367  100.209517  100.250922  100.299523   100.35637   100.42266  100.499805  100.589621  100.694834   100.82057  100.980032 |      100.0 |     0.0 | 100.980032 |   3.467833 |
    | 2000-01-04 18:00:00 | 177.953645 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467834  -0.032166  -0.532165  -1.032165  -1.532164  -2.032162  -2.532161  -3.032159  -3.532156  -4.032152  -4.532147  -5.032141  -5.532133  -6.032123   -6.53211  -7.032094  -7.532074  -8.032049  -8.532018  -9.031979   -9.53193  -10.031871  -10.531798   -11.03171  -11.531603  -12.031473  -12.531318  -13.031133  -13.530916  -14.030665  -14.530382  -15.030073  -15.529759   -16.02948  -16.529321  -17.029436  -17.530116  -18.031897  -18.535759  -19.043514  -19.558563  -20.087522  -20.644212    -21.26207 |      100.0  100.000001  100.000002  100.000004  100.000007  100.000012  100.000018  100.000028  100.000041   100.00006  100.000085   100.00012  100.000167   100.00023  100.000315  100.000426  100.000573  100.000765  100.001015  100.001338  100.001753  100.002284  100.002961  100.003819  100.004902  100.006262  100.007964  100.010084  100.012715  100.015966  100.019969  100.024879  100.030878  100.038182  100.047043  100.057755   100.07066   100.08615  100.104681  100.126773  100.153016   100.18408  100.220717  100.263772  100.314191  100.373047  100.441608  100.521501  100.615161  100.727163  100.869304 |      100.0 |     0.0 | 100.869304 |   3.467833 |
    | 2000-01-04 18:30:00 | 177.835387 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467834  -0.032166  -0.532166  -1.032165  -1.532164  -2.032163  -2.532162   -3.03216  -3.532157  -4.032154   -4.53215  -5.032145  -5.532138  -6.032129  -6.532118  -7.032104  -7.532087  -8.032065  -8.532038  -9.032004  -9.531962   -10.03191  -10.531847   -11.03177  -11.531676  -12.031563  -12.531428  -13.031267  -13.531078  -14.030861  -14.530616  -15.030354  -15.530094   -16.02988  -16.529794  -17.029994  -17.530774  -18.032668  -18.536661  -19.044567  -19.559794  -20.088975   -20.64597   -21.264355 |      100.0  100.000001  100.000002  100.000004  100.000006   100.00001  100.000016  100.000024  100.000035  100.000051  100.000072  100.000102  100.000142  100.000196  100.000268  100.000364   100.00049  100.000655  100.000869  100.001147  100.001504  100.001962  100.002546  100.003287  100.004223  100.005401  100.006875  100.008715     100.011  100.013829  100.017315  100.021596  100.026834  100.033219  100.040976  100.050364  100.061689  100.075301  100.091606  100.111069  100.134219  100.161655  100.194053  100.232172  100.276859   100.32908  100.389972  100.460989  100.544307  100.644006  100.770617 |      100.0 |     0.0 | 100.770617 |   3.467833 |
    | 2000-01-04 19:00:00 | 177.743664 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467833  -0.032166  -0.532166  -1.032165  -1.532165  -2.032164  -2.532163  -3.032161  -3.532159  -4.032156  -4.532153  -5.032148  -5.532142  -6.032135  -6.532125  -7.032113  -7.532098  -8.032079  -8.532055  -9.032026   -9.53199  -10.031945   -10.53189  -11.031822  -11.531741  -12.031642  -12.531524  -13.031384   -13.53122  -14.031032  -14.530823  -15.030601   -15.53039  -16.030231  -16.530211  -17.030488  -17.531355  -18.033351   -18.53746  -19.045502  -19.560888  -20.090266  -20.647534    -21.26639 |      100.0  100.000001  100.000002  100.000003  100.000005  100.000008  100.000013   100.00002   100.00003  100.000043  100.000062  100.000087  100.000121  100.000167  100.000229  100.000311  100.000419   100.00056  100.000744  100.000983  100.001291  100.001685  100.002189  100.002829  100.003638  100.004657  100.005935   100.00753  100.009515  100.011974  100.015009  100.018741  100.023313  100.028892  100.035679  100.043904  100.053838  100.065793  100.080133   100.09727  100.117679  100.141897   100.17053  100.204256  100.243839  100.290143  100.344186  100.407272   100.48134  100.570028  100.682725 |      100.0 |     0.0 | 100.682725 |   3.467833 |
    | 2000-01-04 19:30:00 | 177.663703 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467833  -0.032166  -0.532166  -1.032166  -1.532165  -2.032164  -2.532163  -3.032162   -3.53216  -4.032158  -4.532155  -5.032151  -5.532146  -6.032139  -6.532131  -7.032121  -7.532108  -8.032091  -8.532071  -9.032045  -9.532013  -10.031974  -10.531927  -11.031868  -11.531797  -12.031711  -12.531608  -13.031486  -13.531344  -14.031182  -14.531004  -15.030819   -15.53065  -16.030542   -16.53058  -17.030924  -17.531869  -18.033955  -18.538169  -19.046331   -19.56186  -20.091414  -20.648925     -21.2682 |      100.0  100.000001  100.000001  100.000003  100.000004  100.000007  100.000011  100.000017  100.000025  100.000037  100.000052  100.000074  100.000103  100.000143  100.000196  100.000266  100.000358  100.000479  100.000638  100.000843  100.001107  100.001447  100.001882  100.002434  100.003133  100.004015  100.005122  100.006506  100.008229  100.010367  100.013008   100.01626  100.020248  100.025122  100.031057   100.03826  100.046969  100.057465  100.070069   100.08515  100.103134  100.124499  100.149788  100.179611   100.21465  100.255681  100.303616  100.359619  100.425421  100.504261  100.604505 |      100.0 |     0.0 | 100.604505 |   3.467833 |
    | 2000-01-04 20:00:00 |  177.59328 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467833  -0.032167  -0.532166  -1.032166  -1.532165  -2.032165  -2.532164  -3.032163  -3.532161  -4.032159  -4.532156  -5.032153  -5.532149  -6.032143  -6.532136  -7.032127  -7.532116  -8.032102  -8.532084  -9.032062  -9.532034     -10.032  -10.531959  -11.031908  -11.531846  -12.031771  -12.531681  -13.031576  -13.531453  -14.031314  -14.531162   -15.03101  -15.530879  -16.030815  -16.530904  -17.031309  -17.532324   -18.03449  -18.538796  -19.047065  -19.562722  -20.092434  -20.650161    -21.26981 |      100.0       100.0  100.000001  100.000002  100.000004  100.000006  100.000009  100.000014  100.000021  100.000031  100.000045  100.000063  100.000088  100.000122  100.000167  100.000227  100.000306   100.00041  100.000546  100.000722   100.00095  100.001243  100.001617  100.002094  100.002698  100.003461   100.00442  100.005619  100.007115  100.008973  100.011271  100.014103  100.017581  100.021837  100.027026   100.03333  100.040963  100.050173  100.061245  100.074512  100.090349  100.109187  100.131511  100.157866  100.188864    100.2252   100.26769  100.317374  100.375794  100.445834   100.53494 |      100.0 |     0.0 |  100.53494 |   3.467833 |
    | 2000-01-04 20:30:00 | 177.537584 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467833  -0.032167  -0.532166  -1.032166  -1.532166  -2.032165  -2.532164  -3.032163  -3.532162   -4.03216  -4.532158  -5.032155  -5.532151  -6.032147   -6.53214  -7.032133  -7.532123  -8.032111  -8.532095  -9.032076  -9.532052  -10.032023  -10.531986  -11.031942  -11.531888  -12.031823  -12.531745  -13.031654  -13.531548  -14.031429  -14.531302  -15.031177   -15.53108  -16.031055  -16.531191  -17.031649  -17.532726  -18.034963  -18.539352  -19.047717  -19.563486  -20.093338  -20.651258   -21.271241 |      100.0       100.0  100.000001  100.000002  100.000003  100.000005  100.000008  100.000012  100.000018  100.000026  100.000038  100.000054  100.000075  100.000104  100.000142  100.000194  100.000262  100.000351  100.000468  100.000619  100.000815  100.001067   100.00139  100.001801  100.002324  100.002983  100.003813  100.004853  100.006151  100.007765  100.009764   100.01223  100.015262  100.018977  100.023511  100.029027  100.035714  100.043791  100.053514  100.065177  100.079117  100.095718  100.115413   100.13869  100.166097  100.198256  100.235896  100.279946   100.33178  100.393962  100.473115 |      100.0 |     0.0 | 100.473115 |   3.467833 |
    | 2000-01-04 21:00:00 | 177.491734 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467833  -0.032167  -0.532167  -1.032166  -1.532166  -2.032165  -2.532165  -3.032164  -3.532163  -4.032161  -4.532159  -5.032157  -5.532154  -6.032149  -6.532144  -7.032137  -7.532129  -8.032118  -8.532105  -9.032088  -9.532068  -10.032042  -10.532011  -11.031972  -11.531925  -12.031869  -12.531801  -13.031722  -13.531631   -14.03153  -14.531424  -15.031324  -15.531256  -16.031267  -16.531443  -17.031948   -17.53308  -18.035381  -18.539843  -19.048293  -19.564163  -20.094141  -20.652232   -21.272511 |      100.0       100.0  100.000001  100.000002  100.000003  100.000004  100.000007   100.00001  100.000015  100.000022  100.000032  100.000046  100.000064  100.000089  100.000122  100.000166  100.000224    100.0003    100.0004  100.000531  100.000699  100.000916  100.001195   100.00155  100.002001  100.002571  100.003289  100.004191  100.005317  100.006718  100.008456  100.010603  100.013245  100.016487  100.020448  100.025272  100.031127  100.038208  100.046742   100.05699  100.069254  100.083876  100.101242  100.121789  100.146008  100.174452  100.207776  100.246808   100.29277  100.347941  100.418206 |      100.0 |     0.0 | 100.418206 |   3.467833 |
    | 2000-01-04 21:30:00 | 177.451421 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467833  -0.032167  -0.532167  -1.032166  -1.532166  -2.032166  -2.532165  -3.032164  -3.532163  -4.032162   -4.53216  -5.032158  -5.532155  -6.032152  -6.532147  -7.032142  -7.532134  -8.032125  -8.532113  -9.032099  -9.532081  -10.032059  -10.532032  -11.031998  -11.531957  -12.031908   -12.53185  -13.031782  -13.531704  -14.031618  -14.531531  -15.031454  -15.531412  -16.031453  -16.531665  -17.032212  -17.533393   -18.03575  -18.540277  -19.048803  -19.564763  -20.094852  -20.653096   -21.273639 |      100.0       100.0  100.000001  100.000001  100.000002  100.000004  100.000006  100.000009  100.000013  100.000019  100.000027  100.000039  100.000055  100.000076  100.000104  100.000141  100.000191  100.000257  100.000343  100.000455    100.0006  100.000787  100.001027  100.001333  100.001722  100.002215  100.002837  100.003618  100.004595  100.005811  100.007322   100.00919  100.011493  100.014319  100.017778  100.021996   100.02712  100.033325  100.040812  100.049814  100.060598   100.07347  100.088775  100.106902   100.12829  100.153436  100.182921  100.217485  100.258215  100.307135  100.369471 |      100.0 |     0.0 | 100.369471 |   3.467833 |
    | 2000-01-04 22:00:00 | 177.418068 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467833  -0.032167  -0.532167  -1.032167  -1.532166  -2.032166  -2.532165  -3.032165  -3.532164  -4.032163  -4.532161   -5.03216  -5.532157  -6.032154   -6.53215  -7.032145  -7.532139  -8.032131  -8.532121  -9.032108  -9.532093  -10.032074   -10.53205  -11.032021  -11.531985  -12.031943  -12.531893  -13.031834  -13.531767  -14.031696  -14.531624  -15.031567  -15.531548  -16.031616   -16.53186  -17.032445  -17.533669  -18.036076  -18.540661  -19.049254  -19.565294  -20.095481  -20.653861   -21.274638 |      100.0       100.0  100.000001  100.000001  100.000002  100.000003  100.000005  100.000008  100.000011  100.000016  100.000023  100.000033  100.000046  100.000064  100.000089  100.000121  100.000164   100.00022  100.000294   100.00039  100.000515  100.000676  100.000882  100.001146  100.001483  100.001909  100.002447  100.003123   100.00397  100.005026  100.006339  100.007964  100.009969  100.012434  100.015453  100.019139  100.023623  100.029058  100.035623  100.043526  100.053004   100.06433  100.077811  100.093796  100.112674   100.13489  100.160964  100.191553  100.227625  100.270977  100.326245 |      100.0 |     0.0 | 100.326245 |   3.467833 |
    | 2000-01-04 22:30:00 | 177.392159 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467833  -0.032167  -0.532167  -1.032167  -1.532166  -2.032166  -2.532166  -3.032165  -3.532164  -4.032163  -4.532162  -5.032161  -5.532159  -6.032156  -6.532153  -7.032148  -7.532143  -8.032136  -8.532127  -9.032116  -9.532103  -10.032086  -10.532066   -11.03204   -11.53201  -12.031973   -12.53193  -13.031879  -13.531823  -14.031763  -14.531707  -15.031666  -15.531668   -16.03176  -16.532032  -17.032649  -17.533912  -18.036364     -18.541  -19.049653  -19.565764  -20.096039   -20.65454   -21.275524 |      100.0       100.0       100.0  100.000001  100.000002  100.000003  100.000004  100.000006  100.000009  100.000014   100.00002  100.000028   100.00004  100.000055  100.000076  100.000103   100.00014  100.000188  100.000252  100.000334  100.000441   100.00058  100.000758  100.000986  100.001276  100.001645   100.00211  100.002696   100.00343  100.004346  100.005487    100.0069  100.008646  100.010794  100.013429  100.016649   100.02057  100.025329  100.031084  100.038019  100.046345  100.056306  100.068175  100.082262  100.098916  100.118534  100.141578  100.168634  100.200562  100.238956  100.287927 |      100.0 |     0.0 | 100.287927 |   3.467833 |
    | 2000-01-04 23:00:00 | 177.369451 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467833  -0.032167  -0.532167  -1.032167  -1.532167  -2.032166  -2.532166  -3.032165  -3.532165  -4.032164  -4.532163  -5.032162   -5.53216  -6.032158  -6.532155  -7.032151  -7.532146   -8.03214  -8.532133  -9.032123  -9.532112  -10.032097  -10.532079  -11.032057  -11.532031  -12.031999  -12.531962  -13.031919  -13.531872  -14.031822  -14.531778  -15.031753  -15.531772  -16.031886  -16.532183   -17.03283  -17.534126  -18.036617  -18.541299  -19.050006  -19.566179  -20.096532   -20.65514   -21.276309 |      100.0       100.0       100.0  100.000001  100.000001  100.000002  100.000004  100.000005  100.000008  100.000012  100.000017  100.000024  100.000034  100.000047  100.000065  100.000088   100.00012  100.000161  100.000215  100.000286  100.000379  100.000498  100.000651  100.000848  100.001098  100.001417  100.001819  100.002326  100.002962  100.003758  100.004748  100.005977  100.007497  100.009369  100.011667  100.014479  100.017907  100.022072  100.027114  100.033197  100.040509  100.049265  100.059709  100.072118  100.086803  100.104116  100.124471  100.148389  100.176633  100.210617  100.253982 |      100.0 |     0.0 | 100.253982 |   3.467833 |
    | 2000-01-04 23:30:00 | 177.350285 | 3.467833  2.967833  2.467833  1.967833  1.467833  0.967833  0.467833  -0.032167  -0.532167  -1.032167  -1.532167  -2.032166  -2.532166  -3.032166  -3.532165  -4.032164  -4.532164  -5.032162  -5.532161  -6.032159  -6.532156  -7.032153  -7.532149  -8.032144  -8.532137  -9.032129  -9.532119  -10.032107  -10.532091  -11.032072  -11.532049  -12.032022   -12.53199  -13.031954  -13.531914  -14.031874  -14.531841  -15.031829  -15.531864  -16.031996  -16.532316  -17.032988  -17.534314   -18.03684  -18.541563  -19.050317  -19.566546  -20.096968  -20.655671   -21.277003 |      100.0       100.0       100.0  100.000001  100.000001  100.000002  100.000003  100.000005  100.000007   100.00001  100.000014  100.000021  100.000029   100.00004  100.000055  100.000075  100.000102  100.000138  100.000185  100.000246  100.000325  100.000427   100.00056  100.000729  100.000945   100.00122  100.001568  100.002007  100.002558  100.003248  100.004108  100.005177  100.006499   100.00813  100.010134  100.012588  100.015584  100.019228  100.023644  100.028978  100.035395  100.043089  100.052275    100.0632  100.076141  100.091413  100.109383  100.130515  100.155487  100.185549  100.223928 |      100.0 |     0.0 | 100.223928 |   3.467833 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# ...from HydPy
from hydpy.auxs.anntools import ANN  # pylint: disable=unused-import
from hydpy.auxs.ppolytools import Poly, PPoly  # pylint: disable=unused-import
from hydpy.core import importtools
from hydpy.core import modeltools
from hydpy.core.typingtools import *
from hydpy.exe.modelimports import *
from hydpy.interfaces import routinginterfaces

# ...from musk
from hydpy.models.sw1d import sw1d_control
from hydpy.models.sw1d import sw1d_derived
from hydpy.models.sw1d import sw1d_model

# from hydpy.models import sw1d_network  # actual import at the file's bottom


ADDITIONAL_DERIVEDPARAMETERS = (sw1d_derived.Seconds,)
ADDITIONAL_CONTROLPARAMETERS = (sw1d_control.NmbSegments,)


class Model(modeltools.SubstepModel, routinginterfaces.ChannelModel_V1):
    """|sw1d_channel.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="SW1D-Channel",
        description=(
            '"user model" for preparing single channels that will be combined and '
            "solved by HydPy-SW1D-Network"
        ),
    )

    INLET_METHODS = (sw1d_model.Trigger_Preprocessing_V1,)
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        sw1d_model.Calc_MaxTimeSteps_V1,
        sw1d_model.Calc_TimeStep_V1,
        sw1d_model.Send_TimeStep_V1,
        sw1d_model.Calc_Discharges_V1,
        sw1d_model.Update_Storages_V1,
        sw1d_model.Query_WaterLevels_V1,
    )
    INTERFACE_METHODS = ()
    ADD_METHODS = ()
    OUTLET_METHODS = (
        sw1d_model.Trigger_Postprocessing_V1,
        sw1d_model.Calc_Discharges_V2,
    )
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (
        routinginterfaces.RoutingModel_V1,
        routinginterfaces.RoutingModel_V2,
        routinginterfaces.RoutingModel_V3,
        routinginterfaces.StorageModel_V1,
    )
    SUBMODELS = ()

    storagemodels = modeltools.SubmodelsProperty(routinginterfaces.StorageModel_V1)

    @importtools.prepare_submodel(
        "storagemodels", routinginterfaces.StorageModel_V1, dimensionality=l1
    )
    def add_storagemodel_v1(
        self,
        storagemodel: routinginterfaces.StorageModel_V1,
        *,
        position: int,
        refresh: bool,
    ) -> None:
        """Initialise the given storage model that follows the |StorageModel_V1|
        interface and build the necessary connections to its already available side
        models.

        We construct a channel consisting of two segments:

        >>> from hydpy.models.sw1d_channel import *
        >>> parameterstep()
        >>> nmbsegments(2)

        We first add a routing model between both segments and then add the two
        required storage models:

        >>> with model.add_routingmodel_v2("sw1d_lias", position=1, update=False):
        ...     pass
        >>> with model.add_storagemodel_v1("sw1d_storage", position=0):
        ...     length(2.0)
        >>> with model.add_storagemodel_v1("sw1d_storage", position=1):
        ...     length(3.0)

        Each storage model is at the correct position:

        >>> sm0, sm1 = model.storagemodels
        >>> sm0.parameters.control.length
        length(2.0)
        >>> sm1.parameters.control.length
        length(3.0)

        All side model connections are adequately prepared:

        >>> rm = model.routingmodels[1]

        >>> assert sm0.routingmodelsupstream.number == 0
        >>> assert sm0.routingmodelsdownstream.number == 1
        >>> assert sm0.routingmodelsdownstream[0] is rm

        >>> assert rm.storagemodelupstream is sm0
        >>> assert rm.storagemodeldownstream is sm1

        >>> assert sm1.routingmodelsupstream.number == 1
        >>> assert sm1.routingmodelsupstream[0] is rm
        >>> assert sm1.routingmodelsdownstream.number == 0
        """
        if not refresh:
            s = storagemodel
            p = position
            r1 = routinginterfaces.RoutingModel_V1
            r2 = routinginterfaces.RoutingModel_V2
            r3 = routinginterfaces.RoutingModel_V3
            if (ru := self.routingmodels[p]) is not None:
                if isinstance(ru, (r1, r2)):
                    s.routingmodelsupstream.append_submodel(submodel=ru)
                    ru.storagemodeldownstream = s
                    ru.storagemodeldownstream_typeid = 1
                else:
                    assert not isinstance(ru, r3)
                    assert_never(ru)
            if (rd := self.routingmodels[p + 1]) is not None:
                if isinstance(rd, (r2, r3)):
                    s.routingmodelsdownstream.append_submodel(submodel=rd)
                    rd.storagemodelupstream = s
                    rd.storagemodelupstream_typeid = 1
                else:
                    assert not isinstance(rd, r1)
                    assert_never(rd)

    routingmodels = modeltools.SubmodelsProperty(
        routinginterfaces.RoutingModel_V1,
        routinginterfaces.RoutingModel_V2,
        routinginterfaces.RoutingModel_V3,
    )

    @importtools.prepare_submodel(
        "routingmodels", routinginterfaces.RoutingModel_V1, dimensionality=l1
    )
    def add_routingmodel_v1(
        self,
        routingmodel: routinginterfaces.RoutingModel_V1,
        position: int,
        refresh: bool,
    ) -> None:
        """Initialise the given routing model that follows the |RoutingModel_V1|
        interface and build the necessary connections to its already available side
        models.

        We construct a channel consisting of two segments:

        >>> from hydpy.models.sw1d_channel import *
        >>> parameterstep()
        >>> nmbsegments(2)

        Method |Model.add_routingmodel_v1| checks that the user tries to add the model
        at the first position, as it is the only one valid for "inflow models":

        >>> with model.add_routingmodel_v1("sw1d_q_in", position=1):
        ...     pass
        Traceback (most recent call last):
        ...
        ValueError: While trying to add a submodel to the main model `sw1d_channel`, \
the following error occurred: Submodels of type `sw1d_q_in` can only be added to the \
first channel position 0, but `1` is given.
        >>> assert model.routingmodels[1] is None

        We first add one storage model to each channel segment and one routing model
        between both segments:

        >>> from hydpy.models import sw1d_storage, sw1d_lias, sw1d_q_in
        >>> with model.add_storagemodel_v1(sw1d_storage, position=0) as sm0:
        ...     pass
        >>> with model.add_routingmodel_v2(sw1d_lias, position=1, update=False) as rm1:
        ...     timestepfactor(0.6)
        >>> with model.add_storagemodel_v1(sw1d_storage, position=1) as sm1:
        ...     pass

        Now, we add a routing model to the first position:

        >>> with model.add_routingmodel_v1(sw1d_q_in, position=0, update=False) as rm0:
        ...     timestepfactor(0.7)

        The "inflow model" is at the correct position:

        >>> model.routingmodels[0].parameters.control.timestepfactor
        timestepfactor(0.7)

        All side model connections are adequately prepared:

        >>> assert rm0.routingmodelsdownstream.number == 1
        >>> assert rm1 in rm0.routingmodelsdownstream
        >>> assert rm0.storagemodeldownstream is sm0

        >>> assert sm0.routingmodelsupstream.number == 1
        >>> assert rm0 in sm0.routingmodelsupstream
        >>> assert sm0.routingmodelsdownstream.number == 1
        >>> assert rm0 not in sm0.routingmodelsdownstream

        >>> assert rm1.routingmodelsupstream.number == 1
        >>> assert rm0 in rm1.routingmodelsupstream
        >>> assert rm1.routingmodelsdownstream.number == 0

        >>> assert sm1.routingmodelsupstream.number == 1
        >>> assert rm0 not in sm1.routingmodelsupstream
        >>> assert sm1.routingmodelsdownstream.number == 0
        """
        if not refresh:
            if position != 0:
                self.routingmodels.delete_submodel(position=position)
                raise ValueError(
                    f"Submodels of type `{routingmodel.name}` can only be added to "
                    f"the first channel position 0, but `{position}` is given."
                )
            r = routingmodel
            r1 = routinginterfaces.RoutingModel_V1
            r2 = routinginterfaces.RoutingModel_V2
            r3 = routinginterfaces.RoutingModel_V3
            if (rd := self.routingmodels[1]) is not None:
                if isinstance(rd, (r2, r3)):
                    r.routingmodelsdownstream.append_submodel(submodel=rd)
                    rd.routingmodelsupstream.append_submodel(submodel=r)
                else:
                    assert not isinstance(rd, r1)
                    assert_never(rd)
            if (sd := self.storagemodels[0]) is not None:
                r.storagemodeldownstream = sd
                r.storagemodeldownstream_typeid = 1
                sd.routingmodelsupstream.append_submodel(submodel=r)

    @importtools.prepare_submodel(
        "routingmodels", routinginterfaces.RoutingModel_V2, dimensionality=l1
    )
    def add_routingmodel_v2(
        self,
        routingmodel: routinginterfaces.RoutingModel_V2,
        position: int,
        refresh: bool,
    ) -> None:
        """Initialise the given routing model that follows the |RoutingModel_V2|
        interface and build the necessary connections to its already available side
        models.

        We construct a channel consisting of two segments:

        >>> from hydpy.models.sw1d_channel import *
        >>> parameterstep()
        >>> nmbsegments(2)

        We first one storage model to each channel segment and one routing model above
        the first and below the second segment:

        >>> from hydpy.models import sw1d_storage, sw1d_lias
        >>> with model.add_routingmodel_v2(sw1d_lias, position=0, update=False) as rm0:
        ...     timestepfactor(0.7)
        >>> with model.add_storagemodel_v1(sw1d_storage, position=0) as sm0:
        ...     pass
        >>> with model.add_storagemodel_v1(sw1d_storage, position=1) as sm1:
        ...     pass
        >>> with model.add_routingmodel_v2(sw1d_lias, position=2, update=False) as rm2:
        ...     timestepfactor(0.8)

        Now, we add a routing model to the central location:

        >>> with model.add_routingmodel_v2(sw1d_lias, position=1, update=False) as rm1:
        ...     timestepfactor(0.6)

        The central routing model is actually at the correct position:

        >>> model.routingmodels[1].parameters.control.timestepfactor
        timestepfactor(0.6)

        >>> assert rm0.routingmodelsupstream.number == 0
        >>> assert rm0.routingmodelsdownstream.number == 1
        >>> assert rm1 in rm0.routingmodelsdownstream

        >>> assert sm0.routingmodelsupstream.number == 1
        >>> assert rm1 not in rm0.routingmodelsupstream
        >>> assert sm0.routingmodelsdownstream.number == 1
        >>> assert rm1 in rm0.routingmodelsdownstream

        >>> assert rm1.routingmodelsupstream.number == 1
        >>> assert rm0 in rm1.routingmodelsupstream
        >>> assert sm0 is rm1.storagemodelupstream
        >>> assert sm1 is rm1.storagemodeldownstream
        >>> assert rm1.routingmodelsdownstream.number == 1
        >>> assert rm2 in rm1.routingmodelsdownstream

        >>> assert sm1.routingmodelsupstream.number == 1
        >>> assert rm1 in rm2.routingmodelsupstream
        >>> assert sm1.routingmodelsdownstream.number == 1
        >>> assert rm1 not in rm2.routingmodelsdownstream

        >>> assert rm2.routingmodelsupstream.number == 1
        >>> assert rm1 in rm2.routingmodelsupstream
        >>> assert rm2.routingmodelsdownstream.number == 0
        """
        if not refresh:
            p = position
            r = routingmodel
            r1 = routinginterfaces.RoutingModel_V1
            r2 = routinginterfaces.RoutingModel_V2
            r3 = routinginterfaces.RoutingModel_V3
            nmb = self.parameters.control.nmbsegments.value
            if (p > 0) and ((ru := self.routingmodels[p - 1]) is not None):
                if isinstance(ru, (r1, r2)):
                    r.routingmodelsupstream.append_submodel(submodel=ru)
                    ru.routingmodelsdownstream.append_submodel(submodel=r)
                else:
                    assert not isinstance(ru, r3)
                    assert_never(ru)
            if (p < nmb) and ((rd := self.routingmodels[p + 1]) is not None):
                if isinstance(rd, (r2, r3)):
                    r.routingmodelsdownstream.append_submodel(submodel=rd)
                    rd.routingmodelsupstream.append_submodel(submodel=r)
                else:
                    assert not isinstance(rd, r1)
                    assert_never(rd)
            if (p > 0) and ((su := self.storagemodels[p - 1]) is not None):
                r.storagemodelupstream = su
                r.storagemodelupstream_typeid = 1
                su.routingmodelsdownstream.append_submodel(submodel=r)
            if (p < nmb) and ((sd := self.storagemodels[p]) is not None):
                r.storagemodeldownstream = sd
                r.storagemodeldownstream_typeid = 1
                sd.routingmodelsupstream.append_submodel(submodel=r)

    @importtools.prepare_submodel(
        "routingmodels", routinginterfaces.RoutingModel_V3, dimensionality=l1
    )
    def add_routingmodel_v3(
        self,
        routingmodel: routinginterfaces.RoutingModel_V3,
        position: int,
        refresh: bool,
    ) -> None:
        """"Initialise the given routing model that follows the |RoutingModel_V3|
        interface and build the necessary connections to its already available side
        models.

        We construct a channel consisting of two segments:

        >>> from hydpy.models.sw1d_channel import *
        >>> parameterstep()
        >>> nmbsegments(2)

        Method |Model.add_routingmodel_v3| checks that the user tries to add the model
        at the last position, as it is the only one valid for "outflow models":

        >>> with model.add_routingmodel_v3("sw1d_weir_out", position=1):
        ...     pass
        Traceback (most recent call last):
        ...
        ValueError: While trying to add a submodel to the main model `sw1d_channel`, \
the following error occurred: Submodels of type `sw1d_weir_out` can only be added to \
the last channel position (2), but `1` is given.
        >>> assert model.routingmodels[1] is None

        We first add one storage model to each channel segment and one routing model
        between both segments:

        >>> from hydpy.models import sw1d_storage, sw1d_lias, sw1d_weir_out
        >>> with model.add_storagemodel_v1(sw1d_storage, position=0) as sm0:
        ...     pass
        >>> with model.add_routingmodel_v2(sw1d_lias, position=1, update=False) as rm1:
        ...     timestepfactor(0.6)
        >>> with model.add_storagemodel_v1(sw1d_storage, position=1) as sm1:
        ...     pass

        Now, we add a routing model to the last position:

        >>> with model.add_routingmodel_v3(sw1d_weir_out, position=2, update=False) \
as rm2:
        ...     timestepfactor(0.7)

        The "outflow model" is at the correct position:

        >>> model.routingmodels.submodels[2].parameters.control.timestepfactor
        timestepfactor(0.7)

        >>> assert sm0.routingmodelsupstream.number == 0
        >>> assert sm0.routingmodelsdownstream.number == 1
        >>> assert rm2 not in sm0.routingmodelsdownstream

        >>> assert rm1.routingmodelsupstream.number == 0
        >>> assert rm1.routingmodelsdownstream.number == 1
        >>> assert rm2 in rm1.routingmodelsdownstream

        >>> assert sm1.routingmodelsupstream.number == 1
        >>> assert rm2 not in sm1.routingmodelsupstream
        >>> assert sm1.routingmodelsdownstream.number == 1
        >>> assert rm2 in sm1.routingmodelsdownstream

        >>> assert rm2.storagemodelupstream is sm1
        >>> assert rm2.routingmodelsupstream.number == 1
        >>> assert rm1 in rm2.routingmodelsupstream
        """
        if not refresh:
            nmb = self.parameters.control.nmbsegments.value
            if (p := position) != nmb:
                self.routingmodels.delete_submodel(position=position)
                raise ValueError(
                    f"Submodels of type `{routingmodel.name}` can only be added to "
                    f"the last channel position ({nmb}), but `{position}` is given."
                )
            r = routingmodel
            r1 = routinginterfaces.RoutingModel_V1
            r2 = routinginterfaces.RoutingModel_V2
            r3 = routinginterfaces.RoutingModel_V3
            if (ru := self.routingmodels[p - 1]) is not None:
                if isinstance(ru, (r1, r2)):
                    r.routingmodelsupstream.append_submodel(submodel=ru)
                    ru.routingmodelsdownstream.append_submodel(submodel=r)
                else:
                    assert not isinstance(ru, r3)
                    assert_never(ru)
            if (su := self.storagemodels[p - 1]) is not None:
                r.storagemodelupstream = su
                r.storagemodelupstream_typeid = 1
                su.routingmodelsdownstream.append_submodel(submodel=r)

    def _connect_inlets(self, report_noconnect: bool = False) -> None:
        super()._connect_inlets(report_noconnect)

    def _connect_outlets(self, report_noconnect: bool = False) -> None:
        super()._connect_outlets(report_noconnect)

    @property
    def couple_models(self) -> modeltools.ModelCoupler:
        """The model coupler |sw1d_network.combine_channels|, as defined by the
        composite model |sw1d_network|."""
        return sw1d_network.combine_channels

    def check_waterbalance(self, initial_conditions: ConditionsModel) -> float:
        r"""Determine the water balance error of the previous simulation in m³.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            Error = \Sigma In - \Sigma Out + \Sigma Lat - \Delta Vol \\
            \\
            \Sigma In = \sum_{t=t_0}^{t_1} DischargeVolume_t^1 \\
            \Sigma Out = \sum_{t=t_0}^{t_1} DischargeVolume_t^{N+1} \\
            \Sigma Lat = Seconds \cdot \sum_{t=t_0}^{t_1} LateralFlow_t^1 \\
            \Delta Vol = 1000 \cdot \sum_{i=1}^{N} WaterVolume_{t1}^i - WaterVolume_{t0}^i \\
            \\
            N = NmbSegments

        The returned error should always be in scale with numerical precision so that
        it does not affect the simulation results in any relevant manner.

        Pick the required initial conditions before starting the simulation via
        property |Sequences.conditions|.  See the application model |sw1d_channel|
        integration tests for some examples.

        ToDo: So far, |Model.check_waterbalance| works only with application model
              |sw1d_storage| instances .  We need to implement a more general solution
              as soon as we implement further storage models.
        """
        secs = self.parameters.derived.seconds.value
        volume_old, volume_new, latflow = 0.0, 0.0, 0.0
        for name, model in self.find_submodels().items():
            if isinstance(model, routinginterfaces.StorageModel_V1):
                wv = initial_conditions[name]["states"]["watervolume"]
                assert isinstance(wv, float)
                volume_old += 1000.0 * wv
                volume_new += 1000.0 * model.sequences.states.watervolume
                latflow += secs * numpy.sum(model.sequences.fluxes.lateralflow.series)
        if (r0 := self.routingmodels[0]) is None:
            inflow = 0.0
        else:
            inflow = numpy.sum(r0.sequences.fluxes.dischargevolume.series)
        if (r1 := self.routingmodels[-1]) is None:
            outflow = 0.0
        else:
            outflow = numpy.sum(r1.sequences.fluxes.dischargevolume.series)
        return volume_old + inflow - outflow + latflow - volume_new


tester = Tester()
cythonizer = Cythonizer()

from hydpy.models import sw1d_network  # from pylint: disable=wrong-import-position
