# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""
The |sw1d.DOCNAME.long| model family member |sw1d_pump| is a routing submodel that
calculates water pumping from one channel segment into another.

Integration tests
=================

.. how_to_understand_integration_tests::

We create a test setting identical to the one of the |sw1d_channel| documentation,
except that we replace the |sw1d_lias| submodel at the central location with a
|sw1d_pump| submodel:

>>> from hydpy import pub
>>> pub.timegrids = "2000-01-01 00:00", "2000-01-01 05:00", "5m"

>>> from hydpy.models.sw1d_channel import *
>>> parameterstep()
>>> nmbsegments(8)
>>> lengths = 2.0, 3.0, 2.0, 3.0, 2.0, 3.0, 2.0, 3.0

>>> for i, length_ in enumerate(lengths):
...     with model.add_storagemodel_v1("sw1d_storage", position=i):
...         length(length_)
...         with model.add_crosssection_v2("wq_trapeze"):
...             nmbtrapezes(1)
...             bottomlevels(5.0)
...             bottomwidths(5.0)
...             sideslopes(0.0)

>>> for i in range(1, nmbsegments.value):
...     if i == 4:
...         with model.add_routingmodel_v2("sw1d_pump", position=i) as pump:
...             lengthupstream(2.0 if i % 2 else 3.0)
...             lengthdownstream(3.0 if i % 2 else 2.0)
...             gradient2pumpingrate(PPoly(Poly(x0=0.0, cs=[8.0])))
...             targetwaterlevel1(6.0)
...             targetwaterlevel2(6.0)
...             timestepfactor(0.7)
...             with model.add_crosssection_v2("wq_trapeze"):
...                 nmbtrapezes(1)
...                 bottomlevels(5.0)
...                 bottomwidths(5.0)
...                 sideslopes(0.0)
...     else:
...         with model.add_routingmodel_v2("sw1d_lias", position=i):
...             lengthupstream(2.0 if i % 2 else 3.0)
...             lengthdownstream(3.0 if i % 2 else 2.0)
...             stricklercoefficient(1.0/0.03)
...             timestepfactor(0.7)
...             diffusionfactor(0.2)
...             with model.add_crosssection_v2("wq_trapeze"):
...                 nmbtrapezes(1)
...                 bottomlevels(5.0)
...                 bottomwidths(5.0)
...                 sideslopes(0.0)

>>> from hydpy import Element
>>> channel = Element("channel")
>>> channel.model = model

>>> from hydpy.core.testtools import IntegrationTest
>>> test = IntegrationTest(channel)
>>> test.plotting_options.axis1 = (factors.waterlevels,)

>>> def prepare_inits(hs):
...     if isinstance(hs, float):
...         hs = nmbsegments.value * [hs]
...     inits = []
...     for h, s in zip(hs, model.storagemodels):
...         length = s.parameters.control.length
...         c = s.crosssection.parameters.control
...         v = h * (c.bottomwidths[0] + h * c.sideslopes[0]) * length
...         inits.append((s.sequences.states.watervolume, v))
...     for r in model.routingmodels[1:-1]:
...         inits.append((r.sequences.states.discharge, 0.0))
...     test.inits = inits

Note that we decided on a constant maximum pumping rate for simplicity:

>>> figure = pump.parameters.control.gradient2pumpingrate.plot(0.0, 10.0)
>>> from hydpy.core.testtools import save_autofig
>>> save_autofig("sw1d_pump_gradient2pumpingrate.png", figure=figure)

.. image:: sw1d_pump_gradient2pumpingrate.png
   :width: 400

.. _sw1d_pump_sharp_targets:

Sharp targets
_____________

As in the :ref:`sw1d_channel_zero_inflow_and_outflow` example on |sw1d_lias|, we omit
inflow and outflow but start with uneven water depths:

>>> prepare_inits(hs=[3.0, 3.0, 3.0, 3.0, 1.0, 1.0, 1.0, 1.0])

The pumping station starts lifting all water from the fourth to the fifth channel
segment until the remaining water level is below 6 m,  Then the pumping suddenly stops.
After that, the first three segments still supply water, which leads to an equally
sudden restart of pumping.  Due to the threshold-like configuration of the pump and the
limited precision of the explicit numerical integration scheme, the resulting
unfavourable "on-off" pattern would continue until all water levels of the first four
segments were below 6 m:

.. integration-test::

    >>> conditions = test("sw1d_pump_sharp_targets", get_conditions="2000-01-01 00:00")
    |                date |  timestep |                                                                       waterlevels |                                                                               discharges |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 41.932744 |      8.0       8.0  7.998602  7.840932    6.2392  6.000534       6.0          6.0 | 0.0       0.0       0.0  0.046612       8.0   0.026682        0.0        0.0         0.0 |
    | 2000-01-01 00:05:00 | 41.932744 | 7.999996  7.998089  7.901357  7.747676  6.414353  6.043036  6.001093          6.0 | 0.0  0.000126  0.095686  3.337182       8.0   2.161563   0.036439   0.000008         0.0 |
    | 2000-01-01 00:10:00 | 41.899759 | 7.996272  7.957593  7.837277  7.673375  6.579147  6.081514  6.017422     6.000773 | 0.0  0.124141  2.148931  4.284936       8.0    2.50685   0.582942   0.038629         0.0 |
    | 2000-01-01 00:15:00 |  41.10296 |  7.96331  7.913922  7.797757  7.605366  6.723357  6.128493  6.036042     6.005241 | 0.0   1.09873  3.282265  4.599578       8.0   3.193005   0.844082    0.22342         0.0 |
    | 2000-01-01 00:20:00 |  39.47017 | 7.907889  7.878747  7.759261  7.543152  6.849808  6.183014  6.055596     6.013383 | 0.0  1.847375  3.606121  4.889313       8.0   3.784968   1.058893   0.407116         0.0 |
    | 2000-01-01 00:25:00 | 37.350343 | 7.851107  7.841577  7.720116  7.484274  6.960096  6.244073  6.077405     6.024259 | 0.0  1.892716  3.751213  5.056055       8.0   4.323717   1.270767   0.543799         0.0 |
    | 2000-01-01 00:30:00 |  35.10286 | 7.801383  7.799363  7.678794  7.427186  7.056122  6.310303  6.102456     6.037312 | 0.0  1.657477  3.768209  5.145622       8.0   4.799161    1.48769   0.652636         0.0 |
    | 2000-01-01 00:35:00 | 32.902433 |  7.75764  7.754491  7.634604   7.37068  7.139807  6.380284  6.131132     6.052423 | 0.0  1.458091  3.701694  5.174679       8.0   5.210491   1.711405   0.755537         0.0 |
    | 2000-01-01 00:40:00 | 30.769692 | 7.716413  7.709594  7.588443  7.313836  7.213034  6.452615  6.163528     6.069677 | 0.0   1.37426  3.619079  5.157799       8.0     5.5591   1.942561   0.862694         0.0 |
    | 2000-01-01 00:45:00 | 28.660017 | 7.675345  7.665828  7.541635  7.256186  7.277561  6.525996  6.199604     6.089228 | 0.0  1.368925  3.557261  5.117513       8.0   5.849093   2.180058   0.977539         0.0 |
    | 2000-01-01 00:50:00 | 26.528004 | 7.633781  7.623145  7.495065  7.197625  7.334972  6.599288  6.239226     6.111247 | 0.0  1.385462  3.519595  5.071942       8.0   6.086317   2.421716   1.100966         0.0 |
    | 2000-01-01 00:55:00 | 24.352059 | 7.591997  7.581119  7.449014  7.138207  7.386644  6.671544  6.282179     6.135907 | 0.0  1.392813  3.494122  5.029128       8.0   6.277589   2.664782    1.23302         0.0 |
    | 2000-01-01 01:00:00 | 22.130329 | 7.550407  7.539447  7.403428  7.077996  7.433742   6.74202  6.328167     6.163374 | 0.0  1.386318  3.469925  4.989458       8.0   6.430063   2.906252   1.373323         0.0 |
    | 2000-01-01 01:05:00 | 19.868219 |  7.50924  7.498041  7.358188  7.017007   7.47722  6.810173  6.376827     6.193796 | 0.0  1.372216  3.442513  4.950542       8.0    6.55075   3.143101   1.521119         0.0 |
    | 2000-01-01 01:10:00 | 17.570264 | 7.468546  7.456934  7.313232  6.955213  7.517835  6.875646  6.427744     6.227301 | 0.0  1.356477  3.411802  4.910308       8.0   6.646161   3.372494   1.675253         0.0 |
    | 2000-01-01 01:15:00 | 15.238108 | 7.428298  7.416187   7.26857  6.892568  7.556173  6.938249  6.480477     6.263984 | 0.0  1.341607  3.378963  4.867724       8.0   6.722047   3.591944   1.834159         0.0 |
    | 2000-01-01 01:20:00 | 12.871584 | 7.388469  7.375844  7.224241  6.829016  7.592676  6.997925  6.534583     6.303902 | 0.0  1.327647  3.344801  4.822431       8.0    6.78325   3.799441     1.9959         0.0 |
    | 2000-01-01 01:25:00 | 10.470143 | 7.349053  7.335931  7.180297  6.764502  7.627666  7.054728  6.589642     6.347067 | 0.0  1.313839    3.3095   4.77428       8.0   6.833654     3.9935   2.158231         0.0 |
    | 2000-01-01 01:30:00 | 11.381278 | 7.310011  7.296518  7.136656  6.699038  7.661478  7.108702  6.645353      6.39341 | 0.0  1.301416  3.272056  4.726776       8.0   6.872931   4.174198    2.31716         0.0 |
    | 2000-01-01 01:35:00 | 13.868366 | 7.271379  7.257594  7.093357  6.632581  7.694285  7.160049  6.701402     6.442826 | 0.0  1.287713  3.233919  4.677195       8.0   6.906455   4.339124   2.470805         0.0 |
    | 2000-01-01 01:40:00 |  16.19764 | 7.233219  7.219132  7.050506  6.565051  7.726158  7.209042  6.757516     6.495175 | 0.0  1.272003    3.1951  4.623486       8.0   6.937544   4.487908   2.617444         0.0 |
    | 2000-01-01 01:45:00 | 18.387077 | 7.195557  7.181145  7.008175  6.496367  7.757167  7.255944  6.813512      6.55027 | 0.0  1.255419  3.154769  4.565797       8.0   6.966392   4.621261   2.754738         0.0 |
    | 2000-01-01 01:50:00 | 20.452546 | 7.158403  7.143661  6.966431  6.426449  7.787377  7.301009  6.869282     6.607885 | 0.0  1.238449  3.112641  4.504097       8.0   6.992995   4.739787   2.880773         0.0 |
    | 2000-01-01 01:55:00 | 22.408249 | 7.121771   7.10671  6.925343  6.355214  7.816859  7.344473  6.924783     6.667766 | 0.0  1.221084  3.068657  4.438251       8.0   7.017259   4.844055   2.994019         0.0 |
    | 2000-01-01 02:00:00 | 24.266998 | 7.085676  7.070318  6.884984  6.282576  7.845688  7.386562   6.98002     6.729633 | 0.0  1.203168  3.022773  4.368089       8.0   7.039032   4.934562   3.093355         0.0 |
    | 2000-01-01 02:05:00 | 26.040453 | 7.050139   7.03451   6.84543  6.208444  7.873944   7.42749  7.035029     6.793194 | 0.0  1.184571  2.974934  4.293416       8.0    7.05814    5.01172   3.178058         0.0 |
    | 2000-01-01 02:10:00 | 27.739307 | 7.015182  6.999313  6.806762  6.132725  7.901712  7.467461  7.089872      6.85815 | 0.0  1.165231  2.925087  4.214018       8.0   7.074396   5.075863   3.247785         0.0 |
    | 2000-01-01 02:15:00 | 29.373434 | 6.980828  6.964752  6.769067  6.055318  7.929083  7.506668  7.144613     6.924201 | 0.0  1.145132  2.873191  4.129672       8.0   7.087617    5.12727    3.30255         0.0 |
    | 2000-01-01 02:20:00 | 30.952001 | 6.947099  6.930853  6.732439  5.992629  7.931393  7.545297  7.199319     6.991054 | 0.0  1.124279  2.819217  4.040154  7.174613   7.097626   5.166194   3.342686         0.0 |
    | 2000-01-01 02:25:00 | 31.768887 | 6.914028  6.897715  6.708551  6.046797  7.766885  7.568226  7.253764     7.058444 | 0.0  1.102375  2.759296  3.555578   0.84717   6.330753    5.18431   3.369472         0.0 |
    | 2000-01-01 02:30:00 |  27.33736 | 6.881837  6.867135  6.667454  5.980816  7.832732   7.57013  7.301843     7.126009 | 0.0  1.073048  2.602055  3.971947  7.271004   5.076119   4.980907   3.378278         0.0 |
    | 2000-01-01 02:35:00 | 29.357538 | 6.850924  6.835111  6.649077  6.030042  7.703118  7.578056  7.344218     7.191901 | 0.0  1.030418  2.631622  3.244188  0.782868   5.103351   4.707064   3.294553         0.0 |
    | 2000-01-01 02:40:00 | 25.579017 |  6.82063  6.806236  6.610985   5.95815  7.795912  7.574008  7.382293     7.255059 | 0.0  1.009821  2.453572  3.723285  7.317893   4.224739   4.427099   3.157939         0.0 |
    | 2000-01-01 02:45:00 | 28.332728 | 6.791493  6.775997  6.595016  6.003349  7.680835  7.583606  7.416907     7.314214 | 0.0  0.971233  2.483158  3.015484  0.755539   4.591437   4.111534   2.957732         0.0 |
    | 2000-01-01 02:50:00 | 25.000596 | 6.762911  6.748872  6.558926  5.926921  7.785698  7.583252  7.450508     7.368926 | 0.0  0.952714  2.308955  3.511944  7.333317   3.837891   3.855636   2.735585         0.0 |
    | 2000-01-01 02:55:00 | 28.210657 | 6.735431  6.720354  6.544548  5.983346  7.656114  7.598557  7.482161     7.418908 | 0.0  0.916016   2.34194  2.821215       0.0   4.319464   3.554202     2.4991         0.0 |
    | 2000-01-01 03:00:00 | 24.538705 | 6.708476  6.694891   6.52302  6.028043  7.593045  7.587084  7.513955     7.464318 | 0.0  0.898493  2.171632  2.889222  0.654365   2.756665   3.330319   2.270527         0.0 |
    | 2000-01-01 03:05:00 | 22.292072 | 6.682761  6.669912  6.492942  5.942106  7.742515  7.580514  7.534933     7.505368 | 0.0  0.857156  2.106107  3.108722  7.405545   2.423231   2.751732   2.052482         0.0 |
    | 2000-01-01 03:10:00 | 26.843573 | 6.657926  6.643972  6.480627  5.992813  7.635248   7.60463  7.554538     7.539692 | 0.0  0.827845  2.124865  2.535345       0.0    3.57554   2.369739   1.716228         0.0 |
    | 2000-01-01 03:15:00 | 24.009384 | 6.633508  6.620921  6.462206  6.031618  7.588743  7.602174  7.581733     7.567827 | 0.0  0.813936  1.966467  2.580499   0.64025   2.190443    2.31322   1.406728         0.0 |
    | 2000-01-01 03:20:00 | 22.477049 | 6.610192  6.598385  6.434222  5.940342  7.752928  7.604447   7.59849     7.592938 | 0.0  0.777196  1.903983  2.836777  7.400612   1.927774   1.814117   1.255548         0.0 |
    | 2000-01-01 03:25:00 |  27.64699 |  6.58767   6.57475  6.423255  5.986304  7.655273  7.639203  7.613223     7.613463 | 0.0  0.750737  1.932521  2.298102       0.0   3.255155   1.517387   1.026273         0.0 |
    | 2000-01-01 03:30:00 | 25.359064 | 6.565471  6.553826  6.406946  6.019374  7.619512  7.643088  7.638688     7.629967 | 0.0  0.739943  1.786142  2.329781  0.676242   1.868287   1.674027   0.825195         0.0 |
    | 2000-01-01 03:35:00 | 24.922724 | 6.544248  6.533395  6.380134   5.92512  7.790599  7.649701  7.652231     7.646975 | 0.0   0.70745  1.728963  2.622689  7.335394   1.632474    1.30181   0.850381         0.0 |
    | 2000-01-01 03:40:00 | 29.681844 | 6.523736   6.51172  6.370085   5.96717   7.69808  7.688812  7.663143     7.662269 | 0.0  0.683734  1.767493  2.102473       0.0   3.083984   1.128431   0.764691         0.0 |
    | 2000-01-01 03:45:00 | 27.689307 | 6.503454  6.492584  6.355292  5.994922  7.668827  7.693515    7.6877     7.675464 | 0.0  0.676077  1.632871  2.125978  0.738382   1.713476    1.47835   0.659781         0.0 |
    | 2000-01-01 03:50:00 | 27.469393 | 6.484023  6.473905  6.331256  6.027928  7.647427  7.698292  7.700133     7.691316 | 0.0  0.647674   1.58162  2.382807  0.732517   1.445851    1.20697   0.792562         0.0 |
    | 2000-01-01 03:55:00 |  27.94282 |  6.46526  6.454807  6.311283  5.927753  7.833002  7.700434  7.705781     7.706789 | 0.0  0.625437  1.580376  2.246151  7.254858   1.069033   0.961953    0.77367         0.0 |
    | 2000-01-01 04:00:00 | 31.914626 | 6.446805  6.435378  6.303595  5.964611  7.748044   7.74002   7.71363     7.718608 | 0.0  0.615191  1.586608   1.84286       0.0   2.831906   0.852604   0.590976         0.0 |
    | 2000-01-01 04:05:00 | 30.204043 | 6.428523  6.418336   6.29225  6.001404  7.700135  7.745459      7.74     7.727529 | 0.0  0.609373  1.461487  1.839646       0.0   1.596983   1.325053   0.446047         0.0 |
    | 2000-01-01 04:10:00 | 30.073655 | 6.411042  6.401896  6.280945  5.893073  7.898788   7.73522  7.752683     7.740838 | 0.0  0.582694  1.404693  1.781522  7.198036   0.576275   1.088225   0.665451         0.0 |
    | 2000-01-01 04:15:00 | 34.452257 | 6.394327  6.385074  6.269137  5.928912  7.805187  7.783735  7.751834      7.75529 | 0.0   0.55719  1.398313  1.791917       0.0   3.120013   0.694248   0.722572         0.0 |
    | 2000-01-01 04:20:00 | 32.712986 |  6.37803  6.369568  6.256014   5.96403   7.74949  7.793045  7.777842     7.765772 | 0.0  0.543229  1.318495  1.755925       0.0   1.856566   1.391047   0.524103         0.0 |
    | 2000-01-01 04:25:00 | 32.104819 | 6.362383  6.354432  6.245588  5.996548   7.73103  7.779473  7.795074     7.780164 | 0.0  0.521558  1.278371  1.625914       0.0   0.615363   1.293983    0.71958         0.0 |
    | 2000-01-01 04:30:00 |  32.35402 | 6.347321  6.339887  6.237261  6.009431  7.761487  7.762337  7.791324     7.796749 | 0.0  0.502063  1.229304  1.506881  0.862774  -0.152489   0.704284   0.829274         0.0 |
    | 2000-01-01 04:35:00 | 32.564137 | 6.332848  6.325916  6.214159  5.905819  8.006798  7.738216  7.780946     7.806882 | 0.0  0.482433  1.180988  1.951049  7.131623  -1.045394   0.160702    0.50666         0.0 |
    | 2000-01-01 04:40:00 | 37.492274 | 6.318764  6.310262  6.206364  5.936059   7.92582  7.798315  7.773335     7.805841 | 0.0  0.469472  1.252199   1.51202       0.0   2.699273  -0.305718  -0.052043         0.0 |
    | 2000-01-01 04:45:00 | 36.373929 | 6.304549  6.296668  6.196565  5.965662  7.836525  7.863361  7.783003     7.793881 | 0.0  0.473828  1.153483  1.480115       0.0   2.976477  -0.275781  -0.598033         0.0 |
    | 2000-01-01 04:50:00 | 35.101188 | 6.290792  6.283629  6.188808  5.993044  7.801634  7.875656  7.816664     7.782406 | 0.0   0.45859   1.11055  1.369128       0.0   1.163052   0.548282  -0.573744         0.0 |
    | 2000-01-01 04:55:00 |  34.77501 | 6.277573  6.271165   6.18278  5.999793  7.838988  7.840797   7.85684     7.784125 | 0.0  0.440609  1.063816  1.264746  0.927334  -0.317791   1.425165   0.085948         0.0 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_pump_higher_precision:

Higher precision
________________

One possible solution is increasing numerical precision (like in the
:ref:`sw1d_channel_higher_precision` example), eventually accompanied by reducing
numerical diffusion (like in the :ref:`sw1d_channel_no_additional_diffusion` example):

>>> for routingmodel in model.routingmodels.submodels[1:-1]:
...     routingmodel.parameters.control.timestepfactor(0.1)
...     if routingmodel.name == "sw1d_lias":
...         routingmodel.parameters.control.diffusionfactor(0.0)

Despite the significant increase in computation time, the water level trajectories of
segments four and five are still visibly unsmooth.  The simulated pumping rate does not
oscillate between its maximum and zero anymore but still shows an unsteady decline:

.. integration-test::

    >>> test("sw1d_pump_higher_precision")
    |                date |  timestep |                                                                       waterlevels |                                                                            discharges |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |   5.06557 | 7.999998  7.999785  7.987366  7.848639  6.233013  6.004632  6.000038          6.0 | 0.0  0.000079   0.01082  0.431953       8.0  0.232889  0.001287  0.000005         0.0 |
    | 2000-01-01 00:05:00 |  5.030853 | 7.999682  7.994018  7.924892  7.736266  6.433305  6.030329  6.001181     6.000014 | 0.0  0.010518  0.298892  2.381348       8.0  1.323612   0.03877  0.000673         0.0 |
    | 2000-01-01 00:10:00 |  4.731955 | 7.995652  7.968834  7.845972   7.65675  6.601187  6.074276  6.007086     6.000209 | 0.0  0.134333  1.393515  4.024198       8.0  2.403947  0.206604  0.009775         0.0 |
    | 2000-01-01 00:15:00 |  3.777843 | 7.977735  7.924499  7.792622  7.588596  6.745495  6.127849  6.020821     6.001273 | 0.0  0.597231  2.813988  4.592316       8.0  3.189706  0.511038  0.053201         0.0 |
    | 2000-01-01 00:20:00 |  2.013415 | 7.936819  7.879453  7.756701  7.524867  6.870371  6.187863  6.041054      6.00452 | 0.0  1.363859  3.616171  4.813529       8.0  3.837496  0.836777   0.16235         0.0 |
    | 2000-01-01 00:25:00 | 37.312496 | 7.878269  7.841994  7.720969   7.46518  6.978408  6.253427  6.064681     6.011181 | 0.0  1.951675  3.824599  5.015675       8.0  4.398735  1.120582  0.333035         0.0 |
    | 2000-01-01 00:30:00 | 34.961568 | 7.817139  7.805856  7.681434  7.408428  7.071951  6.323457  6.090537     6.021551 | 0.0  2.037658  3.844551  5.162376       8.0  4.881911  1.380394  0.518507         0.0 |
    | 2000-01-01 00:35:00 | 32.580297 |  7.76414  7.764522  7.639575  7.353002  7.153198  6.396513  6.119276     6.035171 | 0.0  1.766654  3.833386  5.228691       8.0  5.291781  1.638956  0.681011         0.0 |
    | 2000-01-01 00:40:00 | 30.300093 | 7.720945  7.717711  7.595836  7.297768  7.224195  6.471147  6.151757     6.051552 | 0.0  1.439835  3.780367  5.238333       8.0  5.633433  1.901745  0.819043         0.0 |
    | 2000-01-01 00:45:00 | 28.132947 | 7.683236  7.669245  7.549897     7.242  7.286785   6.54608  6.188326     6.070512 | 0.0  1.256954  3.680244  5.211561       8.0  5.913646  2.167008  0.948012         0.0 |
    | 2000-01-01 00:50:00 |  26.01393 | 7.645543  7.622962  7.502268  7.185164  7.342583  6.620242  6.228874      6.09212 | 0.0   1.25643  3.570595  5.158205       8.0  6.140062  2.431971  1.080398         0.0 |
    | 2000-01-01 00:55:00 | 23.861408 |  7.60494  7.580176  7.454354  7.126962  7.392963  6.692776  6.273044     6.116553 | 0.0  1.353445  3.492759  5.089902       8.0  6.320679  2.693971  1.221625         0.0 |
    | 2000-01-01 01:00:00 | 21.629509 |  7.56187  7.539733  7.407446   7.06739  7.439061  6.763037  6.320379     6.144003 | 0.0  1.435669  3.457813  5.021413       8.0   6.46341  2.950367  1.372527         0.0 |
    | 2000-01-01 01:05:00 | 19.321526 | 7.518407  7.499773  7.361956  7.006652  7.481789   6.83058  6.370378     6.174641 | 0.0  1.448764  3.446733  4.963083       8.0  6.575713  3.198549  1.531909         0.0 |
    | 2000-01-01 01:10:00 | 16.966507 | 7.476201  7.459238  7.317484  6.944972   7.52186  6.895144  6.422527     6.208598 | 0.0  1.406871  3.433636  4.916005       8.0  6.664303  3.436116  1.697836         0.0 |
    | 2000-01-01 01:15:00 | 14.589809 | 7.435673   7.41815   7.27341  6.882461  7.559812  6.956622  6.476316     6.245959 | 0.0  1.350934  3.405314  4.874477       8.0  6.734953  3.661044  1.868068         0.0 |
    | 2000-01-01 01:20:00 | 12.199291 | 7.396343  7.377099  7.229361  6.819099  7.596039  7.015034  6.531265     6.286763 | 0.0  1.310989   3.36358  4.831879       8.0  6.792409  3.871797  2.040165         0.0 |
    | 2000-01-01 01:25:00 | 10.710324 | 7.357525   7.33664  7.185329  6.754791  7.630828  7.070495   6.58694     6.330993 | 0.0  1.293954  3.316876  4.784593       8.0  6.840371  4.067326  2.211503         0.0 |
    | 2000-01-01 01:30:00 | 12.887887 | 7.318813  7.297007  7.141522  6.689436  7.664381  7.123185  6.642971      6.37858 | 0.0  1.290392  3.272031  4.732269       8.0  6.881582  4.247084  2.379379         0.0 |
    | 2000-01-01 01:35:00 | 15.249782 | 7.280191  7.258135   7.09817  6.622957  7.696841  7.173322  6.699065     6.429407 | 0.0  1.287386  3.230987  4.676057       8.0  6.917994  4.411137  2.541344         0.0 |
    | 2000-01-01 01:40:00 | 17.464528 | 7.241866  7.219844  7.055419  6.555299  7.728316  7.221147  6.755008     6.483304 | 0.0  1.277493  3.192065  4.617083       8.0  6.950826  4.559602  2.694822         0.0 |
    | 2000-01-01 01:45:00 | 19.548985 | 7.204057     7.182  7.013326  6.486412  7.758892  7.266905  6.810668     6.540054 | 0.0  1.260325  3.152525  4.555645       8.0  6.980797  4.692869  2.837534         0.0 |
    | 2000-01-01 01:50:00 | 21.518021 | 7.166876  7.144577  6.971901  6.416239  7.788645  7.310842  6.865985     6.599405 | 0.0  1.239348  3.110512   4.49135       8.0  7.008257  4.811428  2.967538         0.0 |
    | 2000-01-01 01:55:00 | 23.384915 | 7.130333  7.107628   6.93116   6.34471  7.817645  7.353193   6.92096     6.661071 | 0.0  1.218108  3.065544  4.423551       8.0  7.033309  4.915768  3.083277         0.0 |
    | 2000-01-01 02:00:00 | 25.161669 | 7.094384  7.071233  6.891151  6.271743  7.845968  7.394184  6.975641     6.724743 | 0.0  1.198283   3.01802  4.351677       8.0  7.055899  5.006313  3.183609         0.0 |
    | 2000-01-01 02:05:00 | 26.859258 | 7.058993  7.035457  6.851946   6.19725  7.873692  7.434034  7.030108     6.790099 | 0.0  1.179703  2.968533  4.275349       8.0  7.075882  5.083395  3.267819         0.0 |
    | 2000-01-01 02:10:00 | 28.487831 | 7.024154  7.000334  6.813639  6.121137    7.9009   7.47295  7.084457     6.856811 | 0.0  1.161324   2.91744  4.194338       8.0  7.093062  5.147246  3.335607         0.0 |
    | 2000-01-01 02:15:00 | 30.056854 | 6.989888  6.965882  6.776329  6.043307  7.927684  7.511134  7.138786     6.924553 | 0.0  1.142181   2.86479   4.10847       8.0  7.107218  5.198021   3.38707         0.0 |
    | 2000-01-01 02:20:00 | 31.448318 | 6.956233   6.93211  6.740178   6.00081  7.898981  7.548358   7.19318     6.993006 | 0.0  1.121826  2.810435  4.015466  6.140268  7.097011  5.235805  3.422662         0.0 |
    | 2000-01-01 02:25:00 | 30.243035 | 6.923225  6.899054  6.706586  6.000464  7.820123  7.573823   7.24726     7.061864 | 0.0  1.100277  2.753069  3.872796  3.890123  6.518741  5.245537  3.442872         0.0 |
    | 2000-01-01 02:30:00 |  28.46697 |   6.8909   6.86685  6.675895  5.997389  7.769715   7.58174  7.298451     7.130713 | 0.0  1.077517  2.687745  3.710768  3.864504   5.54475  5.148877  3.442491         0.0 |
    | 2000-01-01 02:35:00 | 27.159147 | 6.859314  6.835624  6.647356  6.006271  7.719535   7.57938  7.343812     7.198713 | 0.0  1.052861  2.614126  3.565435  3.121345  4.794035  4.912015  3.399994         0.0 |
    | 2000-01-01 02:40:00 | 26.138897 | 6.828548   6.80541  6.620474  5.998345  7.705293  7.573793  7.382619     7.264496 | 0.0  1.025534  2.536258  3.432334  3.828615  4.303321  4.582702  3.289139         0.0 |
    | 2000-01-01 02:45:00 | 25.161241 |  6.79868  6.776176  6.595073  6.001534  7.681585  7.568165  7.416122     7.326484 | 0.0  0.995604  2.457287  3.303971  3.144518  3.934785  4.216152  3.099395         0.0 |
    | 2000-01-01 02:50:00 | 24.701273 | 6.769759  6.747884  6.570809  6.002297  7.665439  7.565093  7.446146     7.383291 | 0.0   0.96402   2.37863  3.187432  3.149323  3.687527  3.841173  2.840371         0.0 |
    | 2000-01-01 02:55:00 |  24.22979 | 6.741798  6.720494  6.547561  6.000729  7.655158  7.565536  7.474373     7.433981 | 0.0  0.932035  2.301537  3.076466  3.154859  3.497578   3.47538  2.534474         0.0 |
    | 2000-01-01 03:00:00 | 23.869243 | 6.714776  6.693977  6.525252  5.996954  7.650752  7.569177  7.501789      7.47818 | 0.0  0.900735  2.226592  2.970253  3.158993  3.305849  3.123842  2.209976         0.0 |
    | 2000-01-01 03:05:00 | 23.629646 | 6.688653  6.668315  6.503903  5.991037  7.653787  7.574768  7.528382     7.516064 | 0.0   0.87077  2.153871  2.865493  3.161329  3.060175  2.780594  1.894163         0.0 |
    | 2000-01-01 03:10:00 | 23.656848 | 6.663383  6.643493  6.483331  6.004252  7.631392  7.583075  7.553384     7.548187 | 0.0  0.842314  2.083427  2.769167  2.108427  2.854927  2.439582  1.606178         0.0 |
    | 2000-01-01 03:15:00 | 23.798291 | 6.638927  6.619487  6.463538  5.994561  7.647033  7.593628  7.576099      7.57526 | 0.0  0.815226  2.015518  2.675261  3.159814  2.638438  2.110816  1.353648         0.0 |
    | 2000-01-01 03:20:00 | 24.314658 | 6.615248  6.596266  6.444338  5.991335  7.655103   7.60707  7.596329     7.597983 | 0.0  0.789275   1.95029  2.590321  2.751615  2.482616  1.810484  1.136154         0.0 |
    | 2000-01-01 03:25:00 | 24.642887 |  6.59232  6.573792  6.425877  5.999391  7.649094  7.621872  7.614594     7.617021 | 0.0   0.76429  1.887993  2.503351  2.100534  2.300842  1.560733  0.951911         0.0 |
    | 2000-01-01 03:30:00 | 25.124085 | 6.570113  6.552037  6.408155  5.992422   7.66953  7.636336  7.631354     7.633104 | 0.0  0.740216  1.827985  2.418718  2.767197  2.085989  1.362807  0.804114         0.0 |
    | 2000-01-01 03:35:00 | 25.839577 | 6.548603  6.530976  6.391024  5.997419  7.674271  7.650906  7.646589     7.647041 | 0.0  0.717008  1.770055  2.341072  2.091191  1.933143  1.204651  0.696847         0.0 |
    | 2000-01-01 03:40:00 |  26.60143 | 6.527764  6.510581   6.37442  6.001056  7.681433  7.666104   7.66039     7.659588 | 0.0   0.69462  1.714387  2.267852  2.086014  1.847299  1.087415  0.627355         0.0 |
    | 2000-01-01 03:45:00 | 27.288094 | 6.507573  6.490819  6.358391  6.003342  7.690205  7.681446  7.673404     7.671344 | 0.0  0.673037  1.661102  2.195415  2.081151   1.78873  1.021621  0.587837         0.0 |
    | 2000-01-01 03:50:00 |  27.88735 | 6.488005  6.471669  6.343036  5.989368   7.72424   7.69511  7.686282     7.682811 | 0.0  0.652279  1.609771  2.121603  2.820278  1.685803  1.002603  0.573339         0.0 |
    | 2000-01-01 03:55:00 | 28.661986 | 6.469035  6.453115  6.328105  5.989142  7.736836  7.708051  7.698906     7.694437 | 0.0  0.632327  1.560045  2.057744  2.069055  1.649167   1.00212    0.5813         0.0 |
    | 2000-01-01 04:00:00 | 29.165779 | 6.450642  6.435131  6.313767  5.992726  7.742681  7.720213  7.711074     7.706488 | 0.0  0.613114  1.512311  1.990248  1.811024  1.616218  1.008107  0.602516         0.0 |
    | 2000-01-01 04:05:00 | 29.727684 | 6.432804  6.417703  6.299998  5.990008  7.758049  7.730939  7.722762     7.718941 | 0.0  0.594594  1.465987  1.924939  2.060864  1.548575  1.012261   0.62266         0.0 |
    | 2000-01-01 04:10:00 | 30.264147 | 6.415503  6.400815  6.286667  5.990595  7.767004  7.741581   7.73405     7.731526 | 0.0  0.576694  1.421115  1.865483  1.836103  1.537623  1.005535  0.629263         0.0 |
    | 2000-01-01 04:15:00 | 30.745989 | 6.398723  6.384447  6.273858  5.989748  7.777546  7.751672  7.745271     7.743868 | 0.0  0.559349  1.377766  1.804743  1.847106  1.495706  0.991135  0.617116         0.0 |
    | 2000-01-01 04:20:00 | 31.259475 | 6.382447  6.368582  6.261437  5.987568  7.789109  7.761824  7.756511     7.755691 | 0.0  0.542525  1.335773  1.749805   1.85883  1.473395  0.965809  0.591136         0.0 |
    | 2000-01-01 04:25:00 | 31.797279 | 6.366661  6.353198  6.249311  6.001096  7.775519  7.772691  7.767675     7.766905 | 0.0  0.526204  1.295388  1.699597  1.023183   1.47618  0.932825  0.560688         0.0 |
    | 2000-01-01 04:30:00 | 32.130175 | 6.351348  6.338271   6.23762  5.993149  7.793812  7.783327  7.778722     7.777585 | 0.0  0.510409  1.256739  1.646429  2.043779  1.433999  0.902231  0.534016         0.0 |
    | 2000-01-01 04:35:00 | 32.501314 | 6.336493  6.323787  6.226371  5.987292   7.80998  7.792845  7.789405      7.78791 | 0.0  0.495191  1.219398  1.594355   1.88719  1.348272  0.872355  0.516254         0.0 |
    | 2000-01-01 04:40:00 | 32.935849 | 6.322076  6.309737  6.215579   5.99778  7.803103  7.801141  7.799222     7.798022 | 0.0  0.480554  1.183068  1.542828  1.018415   1.24765  0.832846  0.505604         0.0 |
    | 2000-01-01 04:45:00 | 33.285022 | 6.308083  6.296115  6.205158  5.987002  7.828648  7.809181  7.807886     7.807852 | 0.0  0.466439  1.147544  1.494885  2.033808  1.182312   0.78031  0.491502         0.0 |
    | 2000-01-01 04:50:00 | 33.796488 |   6.2945  6.282906  6.195016   5.99572  7.823544  7.818272  7.815946     7.817097 | 0.0  0.452752  1.113169   1.45124  1.015317  1.185421  0.730884  0.462216         0.0 |
    | 2000-01-01 04:55:00 |  34.08944 | 6.281317  6.270097  6.185363  5.985289  7.847121  7.826981  7.824411     7.825491 | 0.0  0.439434  1.079891  1.401682  1.923228  1.137344  0.701871  0.419718         0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_pump_fuzzy_targets:

Fuzzy targets
_____________

One might reach more satisfactory results by setting two different thresholds and,
thus, a more fuzzy target water level.  Here, we reduce the lower and increase the
higher threshold by 0.1 m:

>>> model.routingmodels[4].parameters.control.targetwaterlevel1(5.9)
>>> model.routingmodels[4].parameters.control.targetwaterlevel2(6.1)

Also, we reset the initial time step and diffusion factors to reduce computation time:

>>> for routingmodel in model.routingmodels.submodels[1:-1]:
...     routingmodel.parameters.control.timestepfactor(0.7)
...     if routingmodel.name == "sw1d_lias":
...         routingmodel.parameters.control.diffusionfactor(0.2)

The fuzziness range of 0.2 m removes all water level trajectory fluctuations:

.. integration-test::

    >>> test("sw1d_pump_fuzzy_targets")
    |                date |  timestep |                                                                       waterlevels |                                                                            discharges |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 41.932744 |      8.0       8.0  7.998602  7.840932    6.2392  6.000534       6.0          6.0 | 0.0       0.0       0.0  0.046612       8.0  0.026682       0.0       0.0         0.0 |
    | 2000-01-01 00:05:00 | 41.932744 | 7.999996  7.998089  7.901357  7.747676  6.414353  6.043036  6.001093          6.0 | 0.0  0.000126  0.095686  3.337182       8.0  2.161563  0.036439  0.000008         0.0 |
    | 2000-01-01 00:10:00 | 41.899759 | 7.996272  7.957593  7.837277  7.673375  6.579147  6.081514  6.017422     6.000773 | 0.0  0.124141  2.148931  4.284936       8.0   2.50685  0.582942  0.038629         0.0 |
    | 2000-01-01 00:15:00 |  41.10296 |  7.96331  7.913922  7.797757  7.605366  6.723357  6.128493  6.036042     6.005241 | 0.0   1.09873  3.282265  4.599578       8.0  3.193005  0.844082   0.22342         0.0 |
    | 2000-01-01 00:20:00 |  39.47017 | 7.907889  7.878747  7.759261  7.543152  6.849808  6.183014  6.055596     6.013383 | 0.0  1.847375  3.606121  4.889313       8.0  3.784968  1.058893  0.407116         0.0 |
    | 2000-01-01 00:25:00 | 37.350343 | 7.851107  7.841577  7.720116  7.484274  6.960096  6.244073  6.077405     6.024259 | 0.0  1.892716  3.751213  5.056055       8.0  4.323717  1.270767  0.543799         0.0 |
    | 2000-01-01 00:30:00 |  35.10286 | 7.801383  7.799363  7.678794  7.427186  7.056122  6.310303  6.102456     6.037312 | 0.0  1.657477  3.768209  5.145622       8.0  4.799161   1.48769  0.652636         0.0 |
    | 2000-01-01 00:35:00 | 32.902433 |  7.75764  7.754491  7.634604   7.37068  7.139807  6.380284  6.131132     6.052423 | 0.0  1.458091  3.701694  5.174679       8.0  5.210491  1.711405  0.755537         0.0 |
    | 2000-01-01 00:40:00 | 30.769692 | 7.716413  7.709594  7.588443  7.313836  7.213034  6.452615  6.163528     6.069677 | 0.0   1.37426  3.619079  5.157799       8.0    5.5591  1.942561  0.862694         0.0 |
    | 2000-01-01 00:45:00 | 28.660017 | 7.675345  7.665828  7.541635  7.256186  7.277561  6.525996  6.199604     6.089228 | 0.0  1.368925  3.557261  5.117513       8.0  5.849093  2.180058  0.977539         0.0 |
    | 2000-01-01 00:50:00 | 26.528004 | 7.633781  7.623145  7.495065  7.197625  7.334972  6.599288  6.239226     6.111247 | 0.0  1.385462  3.519595  5.071942       8.0  6.086317  2.421716  1.100966         0.0 |
    | 2000-01-01 00:55:00 | 24.352059 | 7.591997  7.581119  7.449014  7.138207  7.386644  6.671544  6.282179     6.135907 | 0.0  1.392813  3.494122  5.029128       8.0  6.277589  2.664782   1.23302         0.0 |
    | 2000-01-01 01:00:00 | 22.130329 | 7.550407  7.539447  7.403428  7.077996  7.433742   6.74202  6.328167     6.163374 | 0.0  1.386318  3.469925  4.989458       8.0  6.430063  2.906252  1.373323         0.0 |
    | 2000-01-01 01:05:00 | 19.868219 |  7.50924  7.498041  7.358188  7.017007   7.47722  6.810173  6.376827     6.193796 | 0.0  1.372216  3.442513  4.950542       8.0   6.55075  3.143101  1.521119         0.0 |
    | 2000-01-01 01:10:00 | 17.570264 | 7.468546  7.456934  7.313232  6.955213  7.517835  6.875646  6.427744     6.227301 | 0.0  1.356477  3.411802  4.910308       8.0  6.646161  3.372494  1.675253         0.0 |
    | 2000-01-01 01:15:00 | 15.238108 | 7.428298  7.416187   7.26857  6.892568  7.556173  6.938249  6.480477     6.263984 | 0.0  1.341607  3.378963  4.867724       8.0  6.722047  3.591944  1.834159         0.0 |
    | 2000-01-01 01:20:00 | 12.871584 | 7.388469  7.375844  7.224241  6.829016  7.592676  6.997925  6.534583     6.303902 | 0.0  1.327647  3.344801  4.822431       8.0   6.78325  3.799441    1.9959         0.0 |
    | 2000-01-01 01:25:00 | 10.470143 | 7.349053  7.335931  7.180297  6.764502  7.627666  7.054728  6.589642     6.347067 | 0.0  1.313839    3.3095   4.77428       8.0  6.833654    3.9935  2.158231         0.0 |
    | 2000-01-01 01:30:00 | 11.381278 | 7.310011  7.296518  7.136656  6.699038  7.661478  7.108702  6.645353      6.39341 | 0.0  1.301416  3.272056  4.726776       8.0  6.872931  4.174198   2.31716         0.0 |
    | 2000-01-01 01:35:00 | 13.868366 | 7.271379  7.257594  7.093357  6.632581  7.694285  7.160049  6.701402     6.442826 | 0.0  1.287713  3.233919  4.677195       8.0  6.906455  4.339124  2.470805         0.0 |
    | 2000-01-01 01:40:00 |  16.19764 | 7.233219  7.219132  7.050506  6.565051  7.726158  7.209042  6.757516     6.495175 | 0.0  1.272003    3.1951  4.623486       8.0  6.937544  4.487908  2.617444         0.0 |
    | 2000-01-01 01:45:00 | 18.387077 | 7.195557  7.181145  7.008175  6.496367  7.757167  7.255944  6.813512      6.55027 | 0.0  1.255419  3.154769  4.565797       8.0  6.966392  4.621261  2.754738         0.0 |
    | 2000-01-01 01:50:00 | 20.452546 | 7.158403  7.143661  6.966431  6.426449  7.787377  7.301009  6.869282     6.607885 | 0.0  1.238449  3.112641  4.504097       8.0  6.992995  4.739787  2.880773         0.0 |
    | 2000-01-01 01:55:00 | 22.408249 | 7.121771   7.10671  6.925343  6.355214  7.816859  7.344473  6.924783     6.667766 | 0.0  1.221084  3.068657  4.438251       8.0  7.017259  4.844055  2.994019         0.0 |
    | 2000-01-01 02:00:00 | 24.266998 | 7.085676  7.070318  6.884984  6.282576  7.845688  7.386562   6.98002     6.729633 | 0.0  1.203168  3.022773  4.368089       8.0  7.039032  4.934562  3.093355         0.0 |
    | 2000-01-01 02:05:00 | 26.040453 | 7.050139   7.03451   6.84543  6.208444  7.873944   7.42749  7.035029     6.793194 | 0.0  1.184571  2.974934  4.293416       8.0   7.05814   5.01172  3.178058         0.0 |
    | 2000-01-01 02:10:00 | 27.739307 | 7.015182  6.999313  6.806762  6.132725  7.901712  7.467461  7.089872      6.85815 | 0.0  1.165231  2.925087  4.214018       8.0  7.074396  5.075863  3.247785         0.0 |
    | 2000-01-01 02:15:00 | 29.373434 | 6.980828  6.964752  6.769067  6.058252  7.924683  7.506668  7.144613     6.924201 | 0.0  1.145132  2.873191  4.129672  7.853309  7.087617   5.12727   3.30255         0.0 |
    | 2000-01-01 02:20:00 | 30.823217 | 6.947101  6.930867  6.734694  6.014438  7.900722  7.542447   7.19927     6.991057 | 0.0  1.124223  2.818493  3.964279  6.154952  6.953629  5.164681  3.342805         0.0 |
    | 2000-01-01 02:25:00 | 30.820064 | 6.914069  6.898071  6.702979  5.999991  7.845368  7.566837  7.252594     7.058428 | 0.0   1.10106  2.740847  3.798008  4.520371  6.365516  5.146022  3.368549         0.0 |
    | 2000-01-01 02:30:00 | 29.667782 | 6.881922  6.866369  6.672487  5.993914  7.796743  7.578569  7.302006      7.12571 | 0.0  1.071591  2.656698  3.673077  3.976956  5.597771   5.01117  3.364089         0.0 |
    | 2000-01-01 02:35:00 | 28.449214 | 6.850765   6.83553  6.643836  5.989785  7.760231  7.582288  7.346072     7.191793 | 0.0  1.038538  2.580461  3.535518  3.741926  4.958994  4.773022  3.304153         0.0 |
    | 2000-01-01 02:40:00 | 27.420148 | 6.820572   6.80559   6.61676  5.986313  7.733248  7.582362  7.384993     7.255353 | 0.0  1.006437  2.503463  3.405989  3.579601  4.479061   4.47537  3.178024         0.0 |
    | 2000-01-01 02:45:00 | 26.597358 | 6.791304   6.77657  6.591039   5.98315  7.712998  7.581685   7.41978      7.31518 | 0.0  0.975603   2.42657  3.283944  3.442082  4.117067  4.150915  2.991354         0.0 |
    | 2000-01-01 02:50:00 | 25.959906 | 6.762933   6.74847  6.566495  5.980198  7.697608  7.581907  7.451553     7.370366 | 0.0  0.945718  2.350736  3.168853  3.316489   3.82948   3.81839  2.759266         0.0 |
    | 2000-01-01 02:55:00 | 25.487695 | 6.735436  6.721268  6.542997  5.977414  7.686047   7.58386  7.481151      7.42037 | 0.0  0.916551  2.276648  3.059906  3.199082  3.584469  3.486846  2.500246         0.0 |
    | 2000-01-01 03:00:00 | 25.168451 | 6.708794  6.694936  6.520445  5.974777  7.677818  7.587897  7.509017     7.465008 | 0.0  0.888073  2.204687  2.956441  3.088305  3.362593  3.160726  2.231859         0.0 |
    | 2000-01-01 03:05:00 | 24.995699 | 6.682983  6.669442  6.498758   5.97227  7.672709  7.594096  7.535274     7.504376 | 0.0  0.860363  2.135045  2.857922  2.983275  3.153588  2.843616  1.968402         0.0 |
    | 2000-01-01 03:10:00 |   24.9652 | 6.657978  6.644756  6.477875  5.969881  7.670615  7.602359   7.55987     7.538779 | 0.0  0.833508  2.067808  2.763914   2.88337  2.953173  2.540024  1.720153         0.0 |
    | 2000-01-01 03:15:00 | 25.071806 | 6.633751  6.620848  6.457743    5.9676  7.671434  7.612465  7.582724     7.568654 | 0.0  0.807568  2.002992  2.674058  2.788115  2.760812  2.255542  1.493741         0.0 |
    | 2000-01-01 03:20:00 | 25.307272 | 6.610274  6.597688  6.438319  5.965418     7.675  7.624107  7.603815     7.594515 | 0.0  0.782564  1.940575  2.588051  2.697124  2.578241  1.996111  1.293081         0.0 |
    | 2000-01-01 03:25:00 | 25.659158 | 6.587519  6.575247  6.419565  5.963329  7.681051  7.636935  7.623215     7.616922 | 0.0  0.758487  1.880501  2.505636  2.610076  2.408369  1.767006  1.120342         0.0 |
    | 2000-01-01 03:30:00 |  26.11077 |  6.56546  6.553499  6.401448  5.961327  7.689222  7.650582  7.641078     7.636453 | 0.0  0.735312    1.8227  2.426586  2.526699   2.25433  1.571979  0.976548         0.0 |
    | 2000-01-01 03:35:00 | 26.641977 |  6.54407  6.532418   6.38394  5.959406  7.699065    7.6647  7.657608     7.653687 | 0.0  0.713004  1.767091  2.350702  2.446758  2.118659  1.412728  0.861704         0.0 |
    | 2000-01-01 03:40:00 | 27.230653 | 6.523324  6.511977  6.367013  5.957561  7.710087   7.67898   7.67303     7.669179 | 0.0  0.691528  1.713585  2.277804  2.370044  2.002656  1.288668  0.774606         0.0 |
    | 2000-01-01 03:45:00 | 27.854476 | 6.503199  6.492152  6.350644  5.955788  7.721797  7.693162   7.68756     7.683432 | 0.0  0.670848  1.662093  2.207729  2.296372  1.906046  1.196958  0.712633         0.0 |
    | 2000-01-01 03:50:00 | 28.492756 | 6.483671   6.47292   6.33481  5.954083  7.733754  7.707046   7.70139     7.696868 | 0.0  0.650928  1.612525  2.140326  2.225571  1.826991  1.132784  0.671787         0.0 |
    | 2000-01-01 03:55:00 | 29.128034 | 6.464719  6.454259   6.31949  5.952443  7.745606  7.720495  7.714676     7.709809 | 0.0  0.631736  1.564791  2.075456  2.157486  1.762413  1.089935  0.647052         0.0 |
    | 2000-01-01 04:00:00 |  29.74717 | 6.446321  6.436147  6.304665  5.950863  7.757109  7.733434  7.727534     7.722469 | 0.0  0.613241  1.518806  2.012989  2.091973   1.70854  1.061622  0.633023         0.0 |
    | 2000-01-01 04:05:00 | 30.341767 | 6.428459  6.418566  6.290315  5.949341   7.76813  7.745837  7.740038     7.734961 | 0.0  0.595411  1.474485  1.952803  2.028898   1.66154  1.041386  0.624612         0.0 |
    | 2000-01-01 04:10:00 | 30.907909 | 6.411113  6.401495  6.276424  5.947874  7.778632  7.757721  7.752224     7.747314 | 0.0  0.578218  1.431748  1.894783  1.968135  1.618076  1.023866  0.617646         0.0 |
    | 2000-01-01 04:15:00 | 31.445352 | 6.394264  6.384918  6.262975  5.946459  7.788649  7.769129  7.764106     7.759499 | 0.0   0.56163  1.390517   1.83882  1.909567  1.575674   1.00527  0.609221         0.0 |
    | 2000-01-01 04:20:00 | 31.956387 | 6.377895  6.368816  6.249952  5.945094  7.798256  7.780117  7.775677     7.771454 | 0.0  0.545622  1.350719  1.784813  1.853084  1.532856  0.983475  0.597775         0.0 |
    | 2000-01-01 04:25:00 | 32.444667 |  6.36199  6.353173  6.237341  5.943776  7.807541  7.790741  7.786925     7.783112 | 0.0  0.530164  1.312283  1.732667  1.798581  1.489057  0.957848  0.582915         0.0 |
    | 2000-01-01 04:30:00 | 32.914171 | 6.346533  6.337975  6.225127  5.942502  7.816588  7.801052  7.797837     7.794415 | 0.0   0.51523  1.275144   1.68229  1.745962  1.444407  0.928865  0.565114         0.0 |
    | 2000-01-01 04:35:00 | 33.368463 | 6.331509  6.323206  6.213296  5.941271  7.825458  7.811088  7.808407     7.805322 | 0.0  0.500795  1.239239  1.633598  1.695134  1.399465  0.897672  0.545355         0.0 |
    | 2000-01-01 04:40:00 | 33.810253 | 6.316904  6.308853  6.201836  5.940081  7.834189  7.820873  7.818633     7.815818 | 0.0  0.486834  1.204509  1.586512  1.646013   1.35497   0.86568  0.524806         0.0 |
    | 2000-01-01 04:45:00 | 34.241245 | 6.302704  6.294901  6.190734   5.93893  7.842796  7.830422  7.828523     7.825909 | 0.0  0.473325  1.170899  1.540956  1.598517  1.311641  0.834226   0.50456         0.0 |
    | 2000-01-01 04:50:00 | 34.662203 | 6.288897  6.281339  6.179979  5.937816  7.851271  7.839735   7.83809     7.835618 | 0.0  0.460244  1.138357   1.49686  1.552571   1.27004  0.804356  0.485462         0.0 |
    | 2000-01-01 04:55:00 | 35.073161 |  6.27547  6.268154  6.169559  5.936737  7.859599  7.848811   7.84735     7.844979 | 0.0  0.447573  1.106832  1.454158  1.508102   1.23051  0.776712  0.468024         0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# ...from HydPy
from hydpy.auxs.anntools import ANN  # pylint: disable=unused-import
from hydpy.auxs.ppolytools import Poly, PPoly  # pylint: disable=unused-import
from hydpy.core import modeltools
from hydpy.exe.modelimports import *
from hydpy.interfaces import routinginterfaces

# ...from musk
from hydpy.models.sw1d import sw1d_model


class Model(sw1d_model.Main_CrossSectionModel_V2, routinginterfaces.RoutingModel_V2):
    """|sw1d_pump.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="SW1D-Pump",
        description="submodel for pumping water between two channel segments",
    )

    INLET_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = ()
    INTERFACE_METHODS = (
        sw1d_model.Perform_Preprocessing_V2,
        sw1d_model.Determine_MaxTimeStep_V6,
        sw1d_model.Determine_Discharge_V7,
        sw1d_model.Perform_Postprocessing_V2,
        sw1d_model.Get_MaxTimeStep_V1,
        sw1d_model.Get_Discharge_V1,
        sw1d_model.Get_PartialDischargeUpstream_V1,
        sw1d_model.Get_PartialDischargeDownstream_V1,
        sw1d_model.Get_DischargeVolume_V1,
        sw1d_model.Set_TimeStep_V1,
    )
    ADD_METHODS = (
        sw1d_model.Reset_DischargeVolume_V1,
        sw1d_model.Calc_WaterLevelUpstream_V1,
        sw1d_model.Calc_WaterLevelDownstream_V1,
        sw1d_model.Calc_WaterLevel_V1,
        sw1d_model.Calc_WaterDepth_WettedArea_CrossSectionModel_V2,
        sw1d_model.Calc_WaterDepth_WettedArea_V1,
        sw1d_model.Calc_MaxTimeStep_V6,
        sw1d_model.Calc_WaterLevel_V3,
        sw1d_model.Calc_Discharge_V4,
        sw1d_model.Update_DischargeVolume_V1,
        sw1d_model.Pass_Discharge_V1,
    )
    OUTLET_METHODS = ()
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (
        routinginterfaces.CrossSectionModel_V2,
        routinginterfaces.RoutingModel_V1,
        routinginterfaces.RoutingModel_V2,
        routinginterfaces.RoutingModel_V3,
        routinginterfaces.StorageModel_V1,
    )
    SUBMODELS = ()

    crosssection = modeltools.SubmodelProperty(routinginterfaces.CrossSectionModel_V2)

    storagemodelupstream = modeltools.SubmodelProperty(
        routinginterfaces.StorageModel_V1, sidemodel=True
    )
    storagemodelupstream_is_mainmodel = modeltools.SubmodelIsMainmodelProperty()
    storagemodelupstream_typeid = modeltools.SubmodelTypeIDProperty()

    storagemodeldownstream = modeltools.SubmodelProperty(
        routinginterfaces.StorageModel_V1, sidemodel=True
    )
    storagemodeldownstream_is_mainmodel = modeltools.SubmodelIsMainmodelProperty()
    storagemodeldownstream_typeid = modeltools.SubmodelTypeIDProperty()

    routingmodelsupstream = modeltools.SubmodelsProperty(
        routinginterfaces.RoutingModel_V1,
        routinginterfaces.RoutingModel_V2,
        sidemodels=True,
    )

    routingmodelsdownstream = modeltools.SubmodelsProperty(
        routinginterfaces.RoutingModel_V2,
        routinginterfaces.RoutingModel_V3,
        sidemodels=True,
    )


tester = Tester()
cythonizer = Cythonizer()
