# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""|kinw_williams_ext| is a computationally more efficient alternative to the
application model |kinw_williams|.  The higher efficiency is due to not calculating
discharges based on channel geometries but preprocessed relationships between velocity
and storage, similar to the "V/Q-BEZIEHUNG EXTERN" option of `LARSIM`_.

The "parameter" |VG2FG| describes this relationship.  You can configure it using a
piecewise linear, a spline, or a neural network-based interpolation technique.  See
modules |anntools| and |ppolytools| for further information.  Principally, |VG2FG| can
fit any relationship provided by |kinw_williams| very accurately.  However, high
accuracy might require a considerable number of polynomials or neurons.  At least the
latter can come with a relevant performance cost.

For simple use cases, one can assign a single value to parameter |VG2FG|, either a
constant flow velocity or a constant delay time.  |kinw_williams_ext| then works like
the often-applied linear storage cascade.

.. _`LARSIM`: http://www.larsim.de/en/the-model/

Integration tests
=================

.. how_to_understand_integration_tests::

.. _kinw_williams_ext_main_channel_flow:

main channel flow
_________________

The following integration test repeats the :ref:`kinw_williams_main_channel_flow`
example of the documentation on application model |kinw_williams|.  The spatial and
temporal settings are identical:

>>> from hydpy import ANN, pub, Nodes, Element
>>> pub.timegrids = "2000-01-01", "2000-01-05", "30m"
>>> from hydpy.models.kinw_williams_ext import *
>>> parameterstep("30m")
>>> nodes = Nodes("input1", "input2", "output")
>>> stream = Element("stream",
...                  inlets=["input1", "input2"],
...                  outlets="output")
>>> stream.model = model

We again divide a channel of 100 km into eight subsections:

>>> laen(100.0)
>>> gts(8)
>>> ek(1.0)

Next, we define a relatively small neural network of three neurons in a single hidden
layer.  This network roughly approximates the flow velocity calculated by the
Gauckler-Manning-Strickler equation on the triple trapezoid profile defined in the
:ref:`kinw_williams_main_channel_flow` example:

>>> vg2fg(ANN(nmb_neurons=(3,),
...           weights_input=[[1.239962, 8.434961, 0.116195]],
...           weights_output=[[28.031913],
...                           [22.114402],
...                           [85.802384]],
...           intercepts_hidden=[[4.346762, 4.553889, 4.197525]],
...           intercepts_output=[-134.031566]))

In contrast to application model |kinw_williams|, |kinw_williams_ext| uses the stored
water volume (|VG|) as its state variable instead of the water stage (|H|).  Hence, we
now must set |VG| to a value resulting in an initial outflow of 100 m³/s for the
defined parameterisation of |VG2FG|, which holds for 1.55884384 million m³:

>>> from hydpy.core.testtools import IntegrationTest
>>> IntegrationTest.plotting_options.activated = fluxes.qz, fluxes.qa
>>> test = IntegrationTest(stream, inits=[[states.vg, 1.55884384]])

Finally, we define two identical inflow time series:

>>> import numpy
>>> q_base = 100.0
>>> q_peak = 900.0
>>> t_peak = 24.0
>>> beta = 16.0
>>> ts = pub.timegrids.init.to_timepoints()
>>> nodes.input1.sequences.sim.series = q_base
>>> nodes.input2.sequences.sim.series = (
...     (q_peak-q_base)*((ts/t_peak)*numpy.exp(1.0-ts/t_peak))**beta)

Our approximation of the velocity-storage relationship is far from perfect.  Still, at
least in the range relevant to the selected event, it is sufficient to reproduce the
original results of application model |kinw_williams| with reasonable accuracy (for
example, peak flow is 659.6 m³/s instead of 659.0 m³/s):

.. integration-test::

    >>> test("kinw_williams_ext_main_channel_flow")
    |                date |         qz |        qza |                                                                                             qg |         qa |                                                                             vg | input1 |     input2 |     output |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 100.000003 | 100.000003 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 100.000029 | 100.000029 | 100.000003       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |   0.000029 |      100.0 |
    | 2000-01-01 04:00:00 | 100.000175 | 100.000175 | 100.000018  100.000001       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |   0.000175 |      100.0 |
    | 2000-01-01 04:30:00 | 100.000826 | 100.000826 | 100.000092  100.000005       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558845  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |   0.000826 |      100.0 |
    | 2000-01-01 05:00:00 | 100.003194 | 100.003194 | 100.000385  100.000027  100.000002       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558851  1.558845  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |   0.003194 |      100.0 |
    | 2000-01-01 05:30:00 | 100.010515 | 100.010515 | 100.001368  100.000114   100.00001       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.558867  1.558847  1.558844  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |   0.010515 |      100.0 |
    | 2000-01-01 06:00:00 | 100.030315 | 100.030315 | 100.004253  100.000409  100.000041  100.000003       100.0       100.0       100.0       100.0 |      100.0 | 1.558914  1.558854  1.558845  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |   0.030315 |      100.0 |
    | 2000-01-01 06:30:00 | 100.078179 | 100.078179 | 100.011801  100.001289  100.000142  100.000013  100.000001       100.0       100.0       100.0 |      100.0 | 1.559033  1.558873  1.558847  1.558844  1.558844  1.558844  1.558844  1.558844 |  100.0 |   0.078179 |      100.0 |
    | 2000-01-01 07:00:00 | 100.183351 | 100.183351 | 100.029706  100.003636  100.000438  100.000046  100.000004       100.0       100.0       100.0 |      100.0 |  1.55931   1.55892  1.558853  1.558845  1.558844  1.558844  1.558844  1.558844 |  100.0 |   0.183351 |      100.0 |
    | 2000-01-01 07:30:00 | 100.396214 | 100.396214 | 100.068748  100.009321  100.001225  100.000142  100.000015  100.000001       100.0       100.0 |      100.0 | 1.559899  1.559027  1.558867  1.558847  1.558844  1.558844  1.558844  1.558844 |  100.0 |   0.396214 |      100.0 |
    | 2000-01-01 08:00:00 | 100.797305 | 100.797305 |  100.14783  100.021998  100.003135    100.0004  100.000047  100.000005       100.0       100.0 |      100.0 | 1.561068  1.559253  1.558901  1.558852  1.558845  1.558844  1.558844  1.558844 |  100.0 |   0.797305 |      100.0 |
    | 2000-01-01 08:30:00 | 101.507032 | 101.507032 | 100.292531  100.053067  100.007959   100.00114  100.000147  100.000018  100.000002       100.0 |      100.0 | 1.563255  1.559684  1.558982  1.558864  1.558847  1.558844  1.558844  1.558844 |  100.0 |   1.507032 |      100.0 |
    | 2000-01-01 09:00:00 | 102.694853 | 102.694853 | 100.558297  100.106354  100.017977  100.002789  100.000397  100.000052  100.000006       100.0 |      100.0 |   1.5671  1.560498  1.559141  1.558891  1.558851  1.558845  1.558844  1.558844 |  100.0 |   2.694853 |      100.0 |
    | 2000-01-01 09:30:00 | 104.586356 | 104.586356 | 101.011409  100.203789  100.037257  100.006276  100.000971  100.000139  100.000018  100.000002 | 100.000002 | 1.573535  1.561951  1.559441  1.558947   1.55886  1.558846  1.558844  1.558844 |  100.0 |   4.586356 | 100.000002 |
    | 2000-01-01 10:00:00 | 107.466641 | 107.466641 | 101.749277  100.373255  100.072814  100.013161  100.002196  100.000339  100.000048  100.000006 | 100.000006 | 1.583827  1.564428  1.559982  1.559054   1.55888   1.55885  1.558845  1.558844 |  100.0 |   7.466641 | 100.000006 |
    | 2000-01-01 10:30:00 | 111.678559 | 111.678559 | 102.902216  100.655261  100.135609   100.02608   100.00465   100.00077  100.000118  100.000017 | 100.000017 | 1.599624  1.568473  1.560917  1.559252  1.558919  1.558857  1.558846  1.558844 |  100.0 |  11.678559 | 100.000017 |
    | 2000-01-01 11:00:00 | 117.614836 | 117.614836 | 104.637747  101.106353  100.242075  100.049267  100.009325  100.001644   100.00027  100.000041 | 100.000041 | 1.622983  1.574829  1.562473  1.559599  1.558991   1.55887  1.558848  1.558845 |  100.0 |  17.614836 | 100.000041 |
    | 2000-01-01 11:30:00 | 125.703736 | 125.703736 | 107.163523  101.802679  100.415946  100.089255  100.017844  100.003331  100.000582  100.000095 | 100.000095 | 1.656355  1.584479  1.564969  1.560187  1.559119  1.558897  1.558853  1.558845 |  100.0 |  25.703736 | 100.000095 |
    | 2000-01-01 12:00:00 | 136.388647 | 136.388647 | 110.733575  102.834877  100.693193  100.156318  100.032933  100.006491  100.001199  100.000208 | 100.000208 | 1.702534  1.598696  1.568824  1.561153  1.559341  1.558944  1.558863  1.558847 |  100.0 |  36.388647 | 100.000208 |
    | 2000-01-01 12:30:00 | 150.102726 | 150.102726 | 115.627854  104.344908  101.112372  100.263776  100.058339  100.012083  100.002349   100.00043 |  100.00043 | 1.764589  1.619006  1.574643  1.562681  1.559711  1.559027   1.55888  1.558851 |  100.0 |  50.102726 |  100.00043 |
    | 2000-01-01 13:00:00 | 167.240349 | 167.240349 | 122.175668  106.480358  101.736264  100.431197  100.099904  100.021685  100.004423   100.00085 |  100.00085 | 1.845705  1.647257  1.583182   1.56503  1.560307  1.559168  1.558911  1.558857 |  100.0 |  67.240349 |  100.00085 |
    | 2000-01-01 13:30:00 | 188.127569 | 188.127569 |  130.73097  109.428828  102.641036   100.68524  100.165913   100.03765  100.008034  100.001616 | 100.001616 | 1.949019  1.685601    1.5954   1.56855  1.561242  1.559399  1.558965  1.558869 |  100.0 |  88.127569 | 100.001616 |
    | 2000-01-01 14:00:00 | 212.993962 | 212.993962 |  141.66535  113.411693  103.922771  101.061351   100.26797  100.063436  100.014129  100.002969 | 100.002969 | 2.077411  1.736458   1.61248  1.573701   1.56267  1.559767  1.559053  1.558889 |  100.0 | 112.993962 | 100.002969 |
    | 2000-01-01 14:30:00 | 241.948195 | 241.948195 | 155.352505  118.684435  105.700382  101.605713  100.421953  100.103992  100.024124  100.005282 | 100.005282 | 2.233283   1.80246  1.635851  1.581071  1.564801   1.56034  1.559197  1.558923 |  100.0 | 141.948195 | 100.005282 |
    | 2000-01-01 15:00:00 |  274.95935 |  274.95935 | 172.148372  125.535047  108.118489  102.377432  100.649135  100.166254  100.040094  100.009131 | 100.009131 | 2.418343  1.886364  1.667201  1.591405  1.567912  1.561209  1.559424  1.558978 |  100.0 |  174.95935 | 100.009131 |
    | 2000-01-01 15:30:00 | 311.845554 | 311.845554 | 192.367474  134.279697   111.35004  103.450934  100.977501  100.259738  100.065008  100.015371 | 100.015371 | 2.633403  1.990922  1.708475  1.605623  1.572364  1.562501  1.559775  1.559068 |  100.0 | 211.845554 | 100.015371 |
    | 2000-01-01 16:00:00 | 352.270855 | 352.270855 | 216.256433  145.254837  115.598358   104.91858  101.443272  100.397262  100.103026  100.025252 | 100.025252 | 2.878229  2.118725  1.761856  1.624847   1.57862  1.564384  1.560304  1.559208 |  100.0 | 252.270855 | 100.025252 |
    | 2000-01-01 16:30:00 | 395.750601 | 395.750601 | 243.965962  158.804927  121.098138  106.893428  102.092653  100.595807  100.159871  100.040557 | 100.040557 | 3.151442  2.272015  1.829729  1.650415  1.587261  1.567078  1.561089  1.559422 |  100.0 | 295.750601 | 100.040557 |
    | 2000-01-01 17:00:00 | 441.664962 | 441.664962 | 275.523282  175.265215  128.114703  109.512024  102.983828  100.877543  100.243287  100.063791 | 100.063791 | 3.450497   2.45248  1.914599    1.6839  1.599012  1.570869  1.562231  1.559746 |  100.0 | 341.664962 | 100.063791 |
    | 2000-01-01 17:30:00 | 489.279607 | 489.279607 | 310.807842  194.939384  136.940667  112.937022  104.189189  101.271028  100.363593  100.098408 | 100.098408 | 3.771746  2.661043  2.018997  1.727107  1.614758  1.576122  1.563864  1.560223 |  100.0 | 389.279607 | 100.098408 |
    | 2000-01-01 18:00:00 |  537.77215 |  537.77215 | 349.534319  218.072521  147.889035  117.359285  105.797802  101.812618  100.534351  100.149096 | 100.149096 | 4.110574  2.897674  2.145327   1.78206  1.635569  1.583295  1.566165  1.560916 |  100.0 |  437.77215 | 100.149096 |
    | 2000-01-01 18:30:00 | 586.262607 | 586.262607 | 391.247075  244.820745  161.281779  122.998936  107.918017  102.548098  100.773168  100.222133 | 100.222133 | 4.461602  3.161241  2.295698   1.85097  1.662714  1.592961   1.56936  1.561908 |  100.0 | 486.262607 | 100.222133 |
    | 2000-01-01 19:00:00 | 633.846018 | 633.846018 | 435.329014  275.220078  177.433195  130.104677  110.680117  103.534549  101.102651  100.325816 | 100.325816 | 4.818933  3.449437  2.471714  1.936161  1.697678  1.605823  1.573737  1.563307 |  100.0 | 533.846018 | 100.325816 |
    | 2000-01-01 19:30:00 | 679.625318 | 679.625318 | 481.025177  309.158517  196.627779  138.950491  114.238755  104.842447  101.551533  100.470984 | 100.470984 | 5.176413  3.758797  2.674269   2.03998   1.74216  1.622736  1.579661  1.565252 |  100.0 | 579.625318 | 100.470984 |
    | 2000-01-01 20:00:00 | 722.742733 | 722.742733 | 527.478493  346.356386  219.093192  149.828782  118.774826  106.557971  102.155997  100.671652 | 100.671652 | 5.527888  4.084817  2.903343  2.164656  1.798057  1.644727  1.587585  1.567923 |  100.0 | 622.742733 | 100.671652 |
    | 2000-01-01 20:30:00 | 762.408181 | 762.408181 | 573.773003  386.359912  244.969897  163.039041  124.496246  108.785427  102.961212  100.945765 | 100.945765 | 5.867432  4.422161  3.157845  2.312131  1.867434  1.673006  1.598068  1.571551 |  100.0 | 662.408181 | 100.945765 |
    | 2000-01-01 21:00:00 | 797.923464 | 797.923464 | 618.979214  428.551311  274.280394  178.871433  131.636952  111.649658  104.023081  101.316098 | 101.316098 | 6.189531  4.764931  3.435533  2.483868  1.952456  1.708983  1.611796  1.576424 |  100.0 | 697.923464 | 101.316098 |
    | 2000-01-01 21:30:00 | 828.701418 | 828.701418 | 662.196831  472.175465  306.902285  197.585194  140.453315  115.298182  105.410201  101.811327 | 101.811327 |  6.48924   5.10697  3.733024  2.680638  2.055293  1.754263  1.629594  1.582902 |  100.0 | 728.701418 | 101.811327 |
    | 2000-01-01 22:00:00 |  854.27953 |  854.27953 | 702.591552  516.379915  342.550258  219.382569  151.217076  119.902696  107.205986  102.467267 | 102.467267 | 6.762278   5.44215  4.045918   2.90234  2.177991  1.810629  1.652448  1.591431 |  100.0 |  754.27953 | 102.467267 |
    | 2000-01-01 22:30:00 | 874.327871 | 874.327871 |  739.42409  560.262461  380.771715  244.380047  164.204062  125.659449   109.51086  103.328315 | 103.328315 | 7.005105  5.764641  4.369001  3.147845  2.322308  1.880009  1.681516   1.60256 |  100.0 | 774.327871 | 103.328315 |
    | 2000-01-01 23:00:00 | 888.651531 | 888.651531 | 772.070913  602.919983  420.958879  272.579886  179.678204  132.787834  112.444375  104.449074 | 104.449074 |  7.21495  6.069113  4.696531  3.414927  2.489531  1.964411  1.718134  1.616952 |  100.0 | 788.651531 | 104.449074 |
    | 2000-01-01 23:30:00 | 897.187983 | 897.187983 | 800.037069  643.492973  462.377061  303.846035  197.870929  141.526461  116.146994  105.896152 | 105.896152 | 7.389822  6.350892   5.02254  3.700283  2.680286  2.065832  1.763817  1.635403 |  100.0 | 797.187983 | 105.896152 |
    | 2000-01-02 00:00:00 |      900.0 |      900.0 | 822.961967  681.202101  504.205473  337.889116  218.956796  152.125975  120.781196  107.750083 | 107.750083 |  7.52849   6.60606  5.341134  3.999653  2.894364  2.186127  1.820238  1.658859 |  100.0 |      800.0 | 107.750083 |
    | 2000-01-02 00:30:00 | 897.264896 | 897.264896 | 840.619198  715.375098  545.584719  374.264602  243.027194  164.837986  126.531428  110.107267 | 110.107267 | 7.630452    6.8315  5.646756  4.308029  3.130592  2.326868   1.88919  1.688423 |  100.0 | 797.264896 | 110.107267 |
    | 2000-01-02 01:00:00 | 889.260928 | 889.260928 | 852.911528  745.463856  585.664565  412.386421  270.064965  179.899816  133.602322  113.081773 | 113.081773 | 7.695881  7.024905  5.934395   4.61993   3.38677  2.489165  1.972525   1.72536 |  100.0 | 789.260928 | 113.081773 |
    | 2000-01-02 01:30:00 | 876.351729 | 876.351729 | 859.862096   771.05272  623.646572   451.55528  299.923732  197.515292  142.214541  116.806768 | 116.806768 | 7.725563  7.184762  6.199726  4.929694  3.659707    2.6735  2.072066  1.771094 |  100.0 | 776.351729 | 116.806768 |
    | 2000-01-02 02:00:00 | 858.969595 | 858.969595 | 861.602784  791.859454  658.818091  490.998108  332.316006  217.832503  152.597622  121.435246 | 121.435246 | 7.720823    7.3103  6.439201   5.23177  3.945335   2.87957  2.189489  1.827186 |  100.0 | 758.969595 | 121.435246 |
    | 2000-01-02 02:30:00 |   837.5984 |   837.5984 | 858.360584  807.730499  690.576129  529.913036   366.81351  240.920309  164.979349  127.139606 | 127.139606 | 7.683451  7.401434  6.650078  5.520963  4.238914  3.106178  2.326183  1.895297 |  100.0 |   737.5984 | 127.139606 |
    | 2000-01-02 03:00:00 | 812.756821 | 812.756821 |   850.4427  818.632034  718.441262  567.513969  402.861384  266.746261    179.5715  134.109601 | 134.109601 | 7.615616  7.458694  6.830422  5.792633  4.535289  3.351185  2.483097  1.977129 |  100.0 | 712.756821 | 134.109601 |
    | 2000-01-02 03:30:00 | 784.982427 | 784.982427 | 838.221033   824.63813  742.062791  603.069789  439.805359  295.159229  196.552269  142.548086 | 142.548086 | 7.519787  7.483143  6.979057   6.04282  4.829165  3.611548   2.66059  2.074336 |  100.0 | 684.982427 | 142.548086 |
    | 2000-01-02 04:00:00 | 754.817086 | 754.817086 | 822.116619  825.917051  761.216859  635.935043  476.928498  325.880215  216.046333  152.664069 | 152.664069 | 7.398648  7.476302  7.095518  6.268327  5.115376  3.883435  2.858291  2.188424 |  100.0 | 654.817086 | 152.664069 |
    | 2000-01-02 04:30:00 |  722.79402 |  722.79402 | 802.584484  822.716551  775.799303  665.570835  513.492463  358.504151  238.104218  164.662718 | 164.662718 | 7.255025  7.440064  7.179969  6.466738  5.389117  4.162414  3.075011  2.320619 |  100.0 |  622.79402 | 164.662718 |
    | 2000-01-02 05:00:00 | 689.426711 | 689.426711 | 780.099349  815.348805  785.814867  691.556216  548.778015   392.51387  262.683326  178.732267 | 178.732267 | 7.091814  7.376615   7.23313  6.636404  5.646118   4.44369  3.308706  2.471731 |  100.0 | 589.426711 | 178.732267 |
    | 2000-01-02 05:30:00 | 655.199767 | 655.199767 | 755.142504  804.175519  791.364135   713.59131  582.120338  427.305293  289.633479  195.028194 | 195.028194 | 6.911917  7.288356   7.25619  6.776395  5.882766  4.722357  3.556515   2.64202 |  100.0 | 555.199767 | 195.028194 |
    | 2000-01-02 06:00:00 | 620.561767 | 620.561767 | 728.190113  789.593612   792.62923  731.493904  612.936413   462.22077  318.689856  213.655587 | 213.655587 | 6.718186   7.17783  7.250726  6.886439  6.096169  4.993645  3.814871  2.831082 |  100.0 | 520.561767 | 213.655587 |
    | 2000-01-02 06:30:00 | 585.920022 | 585.920022 | 699.703124  772.021823  789.859105  745.191254  640.743319  496.586173   349.47558  234.651199 | 234.651199 | 6.513377  7.047656  7.218619  6.966841  6.284176  5.253128   4.07967  3.037766 |  100.0 | 485.920022 | 234.651199 |
    | 2000-01-02 07:00:00 |  551.63713 |  551.63713 | 670.118898  751.888476  783.355029  754.708727  665.167782  529.747118  381.514824  257.967254 | 257.967254 | 6.300109  6.900471  7.161979  7.018404  6.445349  5.496885  4.346488  3.260152 |  100.0 |  451.63713 | 257.967254 |
    | 2000-01-02 07:30:00 | 518.029154 | 518.029154 | 639.844569  729.620617  773.456694  760.156583  685.948122  561.100487  414.255526  283.459407 | 283.459407 | 6.080842  6.738874  7.083075  7.042344  6.578925  5.721611  4.610809  3.495585 |  100.0 | 418.029154 | 283.459407 |
    | 2000-01-02 08:00:00 | 485.365238 | 485.365238 |  609.25213  705.634639  760.529283  761.715949   702.93022  590.118789  447.099063  310.881225 | 310.881225 | 5.857845  6.565385  6.984264  7.040208  6.684739  5.924671  4.868244  3.740777 |  100.0 | 385.365238 | 310.881225 |
    | 2000-01-02 08:30:00 |  453.86844 |  453.86844 | 578.675133  680.328486  744.951719  759.624741  716.059181  616.366358  479.433118  339.887012 | 339.887012 | 5.633193  6.382409  6.867942  7.013797  6.763157  6.104118  5.114724   3.99196 |  100.0 |  353.86844 | 339.887012 |
    | 2000-01-02 09:00:00 | 423.717566 | 423.717566 | 548.406885  654.075428  727.106256  754.164088  725.368177  639.507639  510.663785  370.043632 | 370.043632 | 5.408752  6.192206  6.736487  6.965093   6.81499  6.258667  5.346643  4.245076 |  100.0 | 323.717566 | 370.043632 |
    | 2000-01-02 09:30:00 | 395.049792 | 395.049792 | 518.699969  627.219399  707.369506  745.645648  730.965768    659.3086  540.243594  400.850548 | 400.850548 | 5.186182  5.996871  6.592217  6.896196  6.841413   6.38765   5.56096  4.495984 |  100.0 | 295.049792 | 400.850548 |
    | 2000-01-02 10:00:00 | 367.963873 | 367.963873 | 489.766896  600.071791   686.10495  734.400062  733.022639   675.63271  567.693258  431.765823 | 431.765823 | 4.966937  5.798322  6.437357  6.809265  6.843893  6.490952  5.755251  4.740653 |  100.0 | 267.963873 | 431.765823 |
    | 2000-01-02 10:30:00 | 342.523746 | 342.523746 | 461.781702  572.909602   663.65692  720.766718  731.758504  688.433014  592.616225  462.234965 | 462.234965 | 4.752272  5.598292  6.274012  6.706467  6.824108  6.568938  5.927721  4.975339 |  100.0 | 242.523746 | 462.234965 |
    | 2000-01-02 11:00:00 | 318.762374 | 318.762374 |  434.88227  545.974774  640.346006  705.084895  727.429661  697.741696  614.706184  491.719226 | 491.719226 | 4.543256  5.398325  6.104144  6.589937  6.783887  6.622376  6.077185  5.196716 |  100.0 | 218.762374 | 491.719226 |
    | 2000-01-02 11:30:00 |  296.68568 |  296.68568 |  409.17319  519.474552  616.465811  687.686335  720.317551    703.6583  633.748393  519.720506 | 519.720506 | 4.340779  5.199783  5.929559   6.46174  6.725151  6.652363  6.203023  5.401966 |  100.0 |  196.68568 | 519.720506 |
    | 2000-01-02 12:00:00 | 276.276442 | 276.276442 | 384.728983  493.582697  592.280918  668.889199  710.718513  706.337514  649.616085  545.800893 | 545.800893 | 4.145564  5.003846  5.751903  6.323845  6.649858  6.660249  6.305122  5.588833 |  100.0 | 176.276442 | 545.800893 |
    | 2000-01-02 12:30:00 | 257.498082 | 257.498082 | 361.597505  468.441348  568.025952  648.993375  698.934862  705.977203  662.263324  569.595951 | 569.595951 | 3.958185  4.811527   5.57265  6.178104  6.559963  6.647573  6.383807  5.755635 |  100.0 | 157.498082 | 569.595951 |
    | 2000-01-02 13:00:00 | 240.298231 | 240.298231 | 339.803408  444.163374  543.905564  628.277032  685.267309  702.807133  671.715555    590.8218 |   590.8218 | 3.779076  4.623679  5.393114  6.026235  6.457381  6.616001  6.439772  5.901243 |  100.0 | 140.298231 |   590.8218 |
    | 2000-01-02 13:30:00 | 224.612057 | 224.612057 | 319.351526   420.83505  520.095186  606.994308   670.00872   697.07871  678.058935   609.27667 |  609.27667 | 3.608545  4.441009  5.214446  5.869817  6.343955  6.567275  6.474007  6.025051 |  100.0 | 124.612057 |  609.27667 |
    | 2000-01-02 14:00:00 | 210.365288 | 210.365288 | 300.230105  398.518902  496.742394  585.374006  653.439156  689.055889  681.429271  624.837999 | 624.837999 | 3.446788  4.264089  5.037644   5.71028  6.221438  6.503165  6.487735  6.126916 |  100.0 | 110.365288 | 624.837999 |
    | 2000-01-02 14:30:00 |  197.47692 |  197.47692 | 282.413797  377.256607  473.968718  563.619158    635.8221  679.007339   682.00121  637.456265 | 637.456265 | 3.293902  4.093372  4.863562  5.548909  6.091472  6.425431  6.482346  6.207097 |  100.0 |   97.47692 | 637.456265 |
    | 2000-01-02 15:00:00 | 185.861607 | 185.861607 | 265.866382  357.071846   451.87177  541.907306  617.401768  667.199871  679.978087  647.146667 | 647.146667 | 3.149893  3.929202  4.692922  5.386845  5.955582  6.335795  6.459345  6.266193 |  100.0 |  85.861607 | 647.146667 |
    | 2000-01-02 15:30:00 | 175.431719 | 175.431719 | 250.543194   337.97302  430.527549  520.391358  598.401377  653.893091  675.582723   653.97964 |  653.97964 | 3.014693  3.771829  4.526324   5.22509  5.815164   6.23591  6.420304  6.305079 |  100.0 |  75.431719 |  653.97964 |
    | 2000-01-02 16:00:00 | 166.099094 | 166.099094 | 236.393232  319.955782  409.992823  499.200889   579.02225  639.335207  669.049316  658.070968 | 658.070968 | 2.888163  3.621416  4.364257  5.064516  5.671486  6.127346  6.366819   6.32484 |  100.0 |  66.099094 | 658.070968 |
    | 2000-01-02 16:30:00 | 157.776495 | 157.776495 |  223.36099  303.005344   390.30751  478.443755  559.443613  623.759871  660.616492  659.572083 | 659.572083 | 2.770111  3.478056  4.207113   4.90587  5.525686  6.011577  6.300477   6.32672 |  100.0 |  57.776495 | 659.572083 |
    | 2000-01-02 17:00:00 | 150.378786 | 150.378786 | 211.387993  287.098534  371.496959  458.207914  539.822958  607.383977  650.521508  658.660951 | 658.660951 | 2.660295  3.341777  4.055196  4.749791  5.378779  5.889967  6.222829  6.312069 |  100.0 |  50.378786 | 658.660951 |
    | 2000-01-02 17:30:00 | 143.823868 | 143.823868 |  200.41409  272.205611  353.574115  438.563353  520.296859  590.406264  638.995561  655.533798 | 655.533798 | 2.558432  3.212552  3.908733   4.59681  5.231659   5.76377  6.135368    6.2823 |  100.0 |  43.823868 | 655.533798 |
    | 2000-01-02 18:00:00 | 138.033385 | 138.033385 | 190.378515  258.291822   336.54151  419.564054  500.982115  573.006618  626.260122  650.397829 | 650.397829 | 2.464211  3.090309  3.767883  4.447369  5.085106  5.634126  6.039512  6.238852 |  100.0 |  38.033385 | 650.397829 |
    | 2000-01-02 18:30:00 | 132.933235 | 132.933235 | 181.220755   245.31874  320.393064  401.249918  481.977127  555.345948    612.5242  643.464982 | 643.464982 | 2.377293  2.974932   3.63275  4.301827  4.939797  5.502062  5.936591  6.183159 |  100.0 |  32.933235 | 643.464982 |
    | 2000-01-02 19:00:00 | 128.453911 | 128.453911 | 172.881245  233.245383  305.115691  383.648617  463.363427   537.56653  597.982415   634.94673 |  634.94673 | 2.297324  2.866277  3.503383  4.160468  4.796311  5.368497  5.827843  6.116623 |  100.0 |  28.453911 |  634.94673 |
    | 2000-01-02 19:30:00 | 124.530701 | 124.530701 | 165.301919  222.029144  290.690716  366.777327   445.20727  519.792707  582.813793  625.049865 | 625.049865 | 2.223936  2.764168  3.379792  4.023512  4.655137  5.234243  5.714405  6.040598 |  100.0 |  24.530701 | 625.049865 |
    | 2000-01-02 20:00:00 | 121.103761 | 121.103761 | 158.426638  211.626557  277.095087  350.644326  427.561251  502.131851  567.181142  613.973202 | 613.973202 | 2.156755  2.668408  3.261949  3.891123  4.516686  5.100016  5.597316  5.956372 |  100.0 |  21.103761 | 613.973202 |
    | 2000-01-02 20:30:00 | 118.118088 | 118.118088 | 152.201509  201.993919  264.302426  335.250442  410.465881  484.675518  551.230925  601.905101 | 601.905101 | 2.095405  2.578781  3.149794  3.763417  4.381298  4.966439  5.477516  5.865159 |  100.0 |  18.118088 | 601.905101 |
    | 2000-01-02 21:00:00 | 115.523427 | 115.523427 | 146.575121   193.08779  252.283908  320.590349  393.951099  467.500708  535.093525  589.021722 | 589.021722 | 2.039512  2.495059  3.043241  3.640465  4.249249  4.834049  5.355849  5.768088 |  100.0 |  15.523427 | 589.021722 |
    | 2000-01-02 21:30:00 | 113.274094 | 113.274094 | 141.498703  184.865393  241.009002  306.653708   378.03769  450.671194  518.883809  575.485889 | 575.485889 | 1.988707  2.416999  2.942182  3.522305  4.120758  4.703309  5.233066  5.666204 |  100.0 |  13.274094 | 575.485889 |
    | 2000-01-02 22:00:00 | 111.328777 | 111.328777 | 136.931582  177.282448  230.446236   293.42563  362.737602  434.237135  502.699133  561.442327 | 561.442327 | 1.942622  2.344367  2.846487  3.408942  3.995996   4.57461  5.109835  5.560466 |  100.0 |  11.328777 | 561.442327 |
    | 2000-01-02 22:30:00 | 109.650285 | 109.650285 | 132.818341  170.305024  220.562782  280.889849   348.05936  418.243615   486.63201  547.034861 | 547.034861 |  1.90092  2.276891  2.756023  3.300353  3.875091  4.448278  4.986736  5.451741 |  100.0 |   9.650285 | 547.034861 |
    | 2000-01-02 23:00:00 | 108.205287 | 108.205287 | 129.125588   163.88916  211.326986  269.025621  334.002398  402.720414  470.752624  532.379332 | 532.379332 | 1.863263  2.214317  2.670635  3.196496  3.758133  4.324586  4.864278  5.340813 |  100.0 |   8.205287 | 532.379332 |
    | 2000-01-02 23:30:00 | 106.964036 | 106.964036 | 125.815526  157.997352  202.706435  257.811249    320.5628  387.691316   455.12297  517.583018 | 517.583018 | 1.829331  2.156389  2.590159  3.097307   3.64518  4.203754  4.742901  5.228385 |  100.0 |   6.964036 | 517.583018 |
    | 2000-01-03 00:00:00 | 105.900084 | 105.900084 | 122.853046  152.593683  194.669169  247.223711  307.732865  373.173038  439.794513  502.740304 | 502.740304 | 1.798815  2.102856  2.514423  3.002709  3.536264  4.085962  4.622982  5.115083 |  100.0 |   5.900084 | 502.740304 |
    | 2000-01-03 00:30:00 |  104.99001 |  104.99001 | 120.205642  147.643885  187.183874  237.239109   295.50183  359.176231  424.809277  487.933446 | 487.933446 | 1.771427  2.053467  2.443251  2.912609  3.431391  3.971348  4.504843  5.001459 |  100.0 |    4.99001 | 487.933446 |
    | 2000-01-03 01:00:00 | 104.213141 | 104.213141 | 117.843303  143.115384  180.220023  227.833035  283.856497  345.706389  410.200895  473.233432 | 473.233432 | 1.746893  2.007978  2.376463  2.826906  3.330549  3.860018  4.388753  4.888001 |  100.0 |   4.213141 | 473.233432 |
    | 2000-01-03 01:30:00 | 103.551298 | 103.551298 |   115.7384  138.977311  173.747997  218.980862  272.781764  332.764657  395.995608  458.700884 | 458.700884 | 1.724956  1.966148  2.313875  2.745487  3.233707  3.752049  4.274937  4.775131 |  100.0 |   3.551298 | 458.700884 |
    | 2000-01-03 02:00:00 | 102.988541 | 102.988541 | 113.865564  135.200498  167.739162  210.657986   262.26109  320.348553   382.21319  444.386989 | 444.386989 | 1.705377  1.927745  2.255306  2.668233  3.140821  3.647492   4.16358  4.663218 |  100.0 |   2.988541 | 444.386989 |
    | 2000-01-03 02:30:00 |  102.51094 |  102.51094 | 112.201555  131.757456  162.165936  202.840014  252.276877  308.452599  368.867801  430.334414 | 430.334414 | 1.687934  1.892544  2.200571   2.59502  3.051835  3.546375  4.054833  4.552578 |  100.0 |    2.51094 | 430.334414 |
    | 2000-01-03 03:00:00 | 102.106358 | 102.106358 | 110.725137  128.622341  157.001828  195.502914  242.810787   297.06887  355.968776  416.578194 | 416.578194 | 1.672421  1.860329  2.149488  2.525718  2.966681  3.448711  3.948813  4.443481 |  100.0 |   2.106358 | 416.578194 |
    | 2000-01-03 03:30:00 | 101.764251 | 101.764251 | 109.416949  125.770909  152.221466  188.623133  233.844017  286.187477  343.521319  403.146587 | 403.146587 | 1.658646  1.830892  2.101877  2.460195  2.885283  3.354493  3.845612  4.336156 |  100.0 |   1.764251 | 403.146587 |
    | 2000-01-03 04:00:00 |  101.47548 |  101.47548 | 108.259376  123.180465  147.800604  182.177689  225.357513  275.796973  331.527136  390.061867 | 390.061867 | 1.646435  1.804034   2.05756  2.398316   2.80756  3.263702  3.745298  4.230793 |  100.0 |    1.47548 | 390.061867 |
    | 2000-01-03 04:30:00 | 101.232151 | 101.232151 | 107.236426  120.829804  143.716129  176.144239  217.332154  265.884704  319.984996  377.341064 | 377.341064 | 1.635627  1.779566  2.016365  2.339945  2.733421  3.176307  3.647918  4.127552 |  100.0 |   1.232151 | 377.341064 |
    | 2000-01-03 05:00:00 | 101.027458 | 101.027458 | 106.333612   118.69915  139.946049  170.501131  209.748894  256.437108  308.891218  364.996647 | 364.996647 | 1.626076  1.757308   1.97812  2.284946  2.662775  3.092268    3.5535  4.026563 |  100.0 |   1.027458 | 364.996647 |
    | 2000-01-03 05:30:00 | 100.855551 | 100.855551 | 105.537835  116.770091  136.469478   165.22744   202.58888  247.439959  298.240106   353.03714 |  353.03714 | 1.617648   1.73709  1.942662  2.233182  2.595525  3.011536   3.46206  3.927928 |  100.0 |   0.855551 |  353.03714 |
    | 2000-01-03 06:00:00 | 100.711411 | 100.711411 | 104.837276  115.025514  133.266615  160.302993  195.833548  238.878583  288.024317  341.467679 | 341.467679 | 1.610221  1.718751  1.909828  2.184516   2.53157  2.934055  3.373598   3.83173 |  100.0 |   0.711411 | 341.467679 |
    | 2000-01-03 06:30:00 | 100.590745 | 100.590745 | 104.221291  113.449542  130.318719  155.708383  189.464693  230.738029  278.235189  330.290513 | 330.290513 | 1.603686   1.70214  1.879463  2.138815  2.470808  2.859763  3.288103   3.73803 |  100.0 |   0.590745 | 330.290513 |
    | 2000-01-03 07:00:00 | 100.489885 | 100.489885 | 103.680313  112.027462  127.608077  151.424977   183.46453  223.003209  268.863015  319.505446 | 319.505446 | 1.597943  1.687115  1.851418  2.095945  2.413137  2.788594  3.205555  3.646874 |  100.0 |   0.489885 | 319.505446 |
    | 2000-01-03 07:30:00 | 100.405708 | 100.405708 | 103.205759  110.745665  125.117972  147.434915  177.815731  215.659022  259.897279  309.110228 | 309.110228 | 1.592903  1.673544  1.825548  2.055774  2.358452  2.720476  3.125926  3.558291 |  100.0 |   0.405708 | 309.110228 |
    | 2000-01-03 08:00:00 |  100.33556 |  100.33556 | 102.789944  109.591581  122.832648  143.721106  172.501467  208.690444  251.326858  299.100901 | 299.100901 | 1.588485  1.661301  1.801714  2.018175  2.306647  2.655336  3.049181  3.472297 |  100.0 |    0.33556 | 299.100901 |
    | 2000-01-03 08:30:00 | 100.277186 | 100.277186 | 102.425997  108.553618  120.737276  140.267221  167.505421  202.082613  243.140192  289.472097 | 289.472097 | 1.584618  1.650271  1.779783  1.983021  2.257618  2.593097  2.975277    3.3889 |  100.0 |   0.277186 | 289.472097 |
    | 2000-01-03 09:00:00 | 100.228681 | 100.228681 | 102.107789  107.621101  118.817914  137.057675  162.811811   195.82088  235.325421  280.217297 | 280.217297 | 1.581235  1.640347  1.759629  1.950189  2.211261   2.53368  2.904169  3.308095 |  100.0 |   0.228681 | 280.217297 |
    | 2000-01-03 09:30:00 | 100.188431 | 100.188431 |  101.82986  106.784214  117.061471  134.077616    158.4054  189.890869   227.87051  271.329059 | 271.329059 | 1.578281  1.631429   1.74113   1.91956  2.167471  2.477006  2.835805  3.229869 |  100.0 |   0.188431 | 271.329059 |
    | 2000-01-03 10:00:00 | 100.155078 | 100.155078 | 101.587356  106.033945  115.455672  131.312906  154.271494   184.27851  220.763343  262.799209 | 262.799209 | 1.575703  1.623425  1.724171  1.891017  2.126145  2.422994  2.770133  3.154205 |  100.0 |   0.155078 | 262.799209 |
    | 2000-01-03 10:30:00 | 100.127477 | 100.127477 | 101.375968  105.362033  113.989017  128.750102   150.39595  178.970069  213.991805  254.619005 | 254.619005 | 1.573455   1.61625  1.708642  1.864447  2.087183   2.37156  2.707094  3.081076 |  100.0 |   0.127477 | 254.619005 |
    | 2000-01-03 11:00:00 | 100.104666 | 100.104666 | 101.191877  104.760916  112.650745  126.376433   146.76517  173.952172  207.543849  246.779281 | 246.779281 | 1.571498  1.609826  1.694441  1.839741  2.050483  2.322624  2.646629  3.010452 |  100.0 |   0.104666 | 246.779281 |
    | 2000-01-03 11:30:00 | 100.085838 | 100.085838 | 101.031704  104.223682  111.430803  124.179782  143.366096   169.21182  201.407548   239.27056 |  239.27056 | 1.569796   1.60408  1.681468  1.816793  2.015948  2.276101  2.588676  2.942299 |  100.0 |   0.085838 |  239.27056 |
    | 2000-01-03 12:00:00 | 100.070317 | 100.070317 | 100.892464  103.744024  110.319805  122.148665  140.186203  164.736399  195.571143  232.083161 | 232.083161 | 1.568316  1.598948  1.669631  1.795501   1.98348  2.231911  2.533174  2.876577 |  100.0 |   0.070317 | 232.083161 |
    | 2000-01-03 12:30:00 | 100.057539 | 100.057539 | 100.771524  103.316193     109.309  120.272201  137.213488  160.513692  190.023071  225.207278 | 225.207278 | 1.567031  1.594367  1.658844  1.775767  1.952986  2.189971  2.480057  2.813245 |  100.0 |   0.057539 | 225.207278 |
    | 2000-01-03 13:00:00 | 100.047032 | 100.047032 | 100.666565  102.934958  108.390239  118.540099  134.436462  156.531877  184.752001  218.633051 | 218.633051 | 1.565916  1.590284  1.649025  1.757497  1.924372  2.150199  2.429261  2.752259 |  100.0 |   0.047032 | 218.633051 |
    | 2000-01-03 13:30:00 | 100.038402 | 100.038402 | 100.575546  102.595568  107.555943   116.94263  131.844135  152.779532  179.746846  212.350625 | 212.350625 | 1.564949  1.586648  1.640096  1.740601   1.89755  2.112515  2.380719  2.693573 |  100.0 |   0.038402 | 212.350625 |
    | 2000-01-03 14:00:00 | 100.031322 | 100.031322 | 100.496676  102.293712  106.799073  115.470606  129.426006  149.245632  174.996788  206.350197 | 206.350197 | 1.564111  1.583413  1.631986  1.724993   1.87243   2.07684  2.334367  2.637136 |  100.0 |   0.031322 | 206.350197 |
    | 2000-01-03 14:30:00 | 100.025521 | 100.025521 | 100.428384  102.025483  106.113095  114.115357  127.172047  145.919549  170.491285  200.622056 | 200.622056 | 1.563386  1.580539  1.624629  1.710589  1.848928  2.043094  2.290138  2.582901 |  100.0 |   0.025521 | 200.622056 |
    | 2000-01-03 15:00:00 | 100.020773 | 100.020773 |  100.36929  101.787348  105.491957  112.868711  125.072694  142.791042  166.220079  195.156613 | 195.156613 | 1.562759  1.577986   1.61796   1.69731  1.826961  2.011201  2.247966  2.530815 |  100.0 |   0.020773 | 195.156613 |
    | 2000-01-03 15:30:00 | 100.016892 | 100.016892 | 100.318192  101.576115  104.930057  111.722973  123.118825  139.850256  162.173207  189.944431 | 189.944431 | 1.562216  1.575722  1.611923  1.685083  1.806448  1.981085  2.207785  2.480827 |  100.0 |   0.016892 | 189.944431 |
    | 2000-01-03 16:00:00 | 100.013722 | 100.013722 | 100.274034  101.388906  104.422218  110.670903  121.301755  137.087714  158.340999  184.976241 | 184.976241 | 1.561748  1.573715  1.606463  1.673836  1.787313   1.95267  2.169529  2.432884 |  100.0 |   0.013722 | 184.976241 |
    | 2000-01-03 16:30:00 | 100.011136 | 100.011136 | 100.235899  101.223125   103.96366  109.705697  119.613215  134.494309  154.714083  180.242959 | 180.242959 | 1.561343  1.571938   1.60153    1.6635  1.769479  1.925884  2.133133  2.386932 |  100.0 |   0.011136 | 180.242959 |
    | 2000-01-03 17:00:00 | 100.009028 | 100.009028 | 100.202983   101.07644  103.549977  108.820964  118.045341  132.061295   151.28338  175.735701 | 175.735701 | 1.560994  1.570366  1.597078  1.654012  1.752875  1.900655  2.098533  2.342918 |  100.0 |   0.009028 | 175.735701 |
    | 2000-01-03 17:30:00 | 100.007313 | 100.007313 | 100.174589  100.946753  103.177114  108.010712  116.590657  129.780282  148.040111   171.44579 |  171.44579 | 1.560693  1.568976  1.593063  1.645312  1.737431  1.876914  2.065666  2.300787 |  100.0 |   0.007313 |  171.44579 |
    | 2000-01-03 18:00:00 | 100.005918 | 100.005918 | 100.150108  100.832183  102.841339  107.269326  115.242066  127.643222  144.975785  167.364764 | 167.364764 | 1.560433  1.567748  1.589447  1.637341  1.723081  1.854592  2.034467  2.260487 |  100.0 |   0.005918 | 167.364764 |
    | 2000-01-03 18:30:00 | 100.004785 | 100.004785 | 100.129012  100.731044  102.539229  106.591551  113.992831  125.642405  142.082202  163.484382 | 163.484382 |  1.56021  1.566665  1.586192  1.630047  1.709758  1.833623  2.004875  2.221963 |  100.0 |   0.004785 | 163.484382 |
    | 2000-01-03 19:00:00 | 100.003865 | 100.003865 | 100.110842  100.641827  102.267641  105.972471  112.836567  123.770445  139.351446  159.796624 | 159.796624 | 1.560017  1.565709  1.583266  1.623378  1.697403  1.813942   1.97683  2.185162 |  100.0 |   0.003865 | 159.796624 |
    | 2000-01-03 19:30:00 | 100.003119 | 100.003119 | 100.095198  100.563181  102.023699    105.4075  111.767222  122.020274   136.77588  156.293694 | 156.293694 | 1.559851  1.564866  1.580637  1.617287  1.685955  1.795486   1.95027   2.15003 |  100.0 |   0.003119 | 156.293694 |
    | 2000-01-03 20:00:00 | 100.002515 | 100.002515 | 100.081737  100.493903  101.804772  104.892356  110.779068  120.385133  134.348141  152.968024 | 152.968024 | 1.559709  1.564125  1.578277   1.61173  1.675359  1.778195  1.925136  2.116514 |  100.0 |   0.002515 | 152.968024 |
    | 2000-01-03 20:30:00 | 100.002026 | 100.002026 | 100.070158  100.432918  101.608455  104.423051  109.866689  118.858558  132.061137  149.812268 | 149.812268 | 1.559586  1.563472  1.576161  1.606664  1.665561   1.76201  1.901372  2.084562 |  100.0 |   0.002026 | 149.812268 |
    | 2000-01-03 21:00:00 | 100.001631 | 100.001631 | 100.060202  100.379267  101.432555  103.995873  109.024965  117.434376  129.908037  146.819302 | 146.819302 | 1.559481  1.562897  1.574265   1.60205  1.656508  1.746873  1.878919  2.054122 |  100.0 |   0.001631 | 146.819302 |
    | 2000-01-03 21:30:00 | 100.001312 | 100.001312 | 100.051645  100.332099  101.275073  103.607371  108.249065  116.106696  127.882268  143.982224 | 143.982224 |  1.55939  1.562392  1.572568  1.597851  1.648153  1.732729  1.857723  2.025142 |  100.0 |   0.001312 | 143.982224 |
    | 2000-01-03 22:00:00 | 100.001054 | 100.001054 | 100.044292  100.290656  101.134189  103.254342  107.534428  114.869893  125.977506  141.294352 | 141.294352 | 1.559312  1.561949   1.57105  1.594035  1.640449  1.719526  1.837729  1.997571 |  100.0 |   0.001054 | 141.294352 |
    | 2000-01-03 22:30:00 | 100.000846 | 100.000846 | 100.037977  100.254265  101.008251  102.933815  106.876761  113.718609  124.187672  138.749214 | 138.749214 | 1.559245   1.56156  1.569693  1.590569  1.633352   1.70721  1.818885  1.971361 |  100.0 |   0.000846 | 138.749214 |
    | 2000-01-03 23:00:00 | 100.000679 | 100.000679 | 100.032555  100.222328  100.895755  102.643038  106.272021  112.647736  122.506926  136.340555 | 136.340555 | 1.559188  1.561218   1.56848  1.587424   1.62682  1.695734  1.801138   1.94646 |  100.0 |   0.000679 | 136.340555 |
    | 2000-01-03 23:30:00 | 100.000544 | 100.000544 | 100.027901  100.194315  100.795341  102.379465  105.716404  111.652413  120.929659  134.062324 | 134.062324 | 1.559139  1.560919  1.567398  1.584573  1.620813  1.685049  1.784439  1.922821 |  100.0 |   0.000544 | 134.062324 |
    | 2000-01-04 00:00:00 | 100.000436 | 100.000436 | 100.023908  100.169759  100.705775  102.140744  105.206339  110.728017  119.450486  131.908675 | 131.908675 | 1.559097  1.560656  1.566434   1.58199  1.615295   1.67511  1.768739  1.900397 |  100.0 |   0.000436 | 131.908675 |
    | 2000-01-04 00:30:00 | 100.000349 | 100.000349 | 100.020482  100.148244  100.625941  101.924704  104.738473   109.87015  118.064242  129.873963 | 129.873963 |  1.55906  1.560426  1.565574  1.579652   1.61023  1.665873   1.75399  1.879139 |  100.0 |   0.000349 | 129.873963 |
    | 2000-01-04 01:00:00 | 100.000279 | 100.000279 | 100.017544  100.129403  100.554829  101.729343  104.309664  109.074637  116.765976  127.952736 | 127.952736 | 1.559029  1.560225  1.564808  1.577538  1.605586  1.657296  1.740145  1.859003 |  100.0 |   0.000279 | 127.952736 |
    | 2000-01-04 01:30:00 | 100.000223 | 100.000223 | 100.015024  100.112911   100.49153  101.552819  103.916969  108.337517  115.550941  126.139735 | 126.139735 | 1.559003  1.560048  1.564127  1.575627   1.60133  1.649339  1.727161  1.839943 |  100.0 |   0.000223 | 126.139735 |
    | 2000-01-04 02:00:00 | 100.000178 | 100.000178 | 100.012865  100.098484  100.435222  101.393437  103.557636  107.655031  114.414595  124.429887 | 124.429887 |  1.55898  1.559894   1.56352  1.573903  1.597435  1.641964  1.714994  1.821916 |  100.0 |   0.000178 | 124.429887 |
    | 2000-01-04 02:30:00 | 100.000142 | 100.000142 | 100.011014  100.085868  100.385164   101.24964  103.229092  107.023622  113.352587  122.818299 | 122.818299 |  1.55896   1.55976  1.562982  1.572347  1.593872  1.635134  1.703602  1.804877 |  100.0 |   0.000142 | 122.818299 |
    | 2000-01-04 03:00:00 | 100.000113 | 100.000113 | 100.009428  100.074841  100.340691  101.120002  102.928937  106.439919  112.360758  121.300261 | 121.300261 | 1.558943  1.559642  1.562503  1.570944  1.590616  1.628814  1.692944  1.788786 |  100.0 |   0.000113 | 121.300261 |
    | 2000-01-04 03:30:00 |  100.00009 |  100.00009 | 100.008069  100.065207  100.301204  101.003211  102.654933  105.900737  111.435132  119.871231 | 119.871231 | 1.558929  1.559539  1.562078   1.56968  1.587643  1.622971  1.682982  1.773601 |  100.0 |    0.00009 | 119.871231 |
    | 2000-01-04 04:00:00 | 100.000072 | 100.000072 | 100.006906  100.056794  100.266164  100.898072  102.404995  105.403066  110.571911  118.526839 | 118.526839 | 1.558917  1.559449  1.561701  1.568543   1.58493  1.617575  1.673678  1.759282 |  100.0 |   0.000072 | 118.526839 |
    | 2000-01-04 04:30:00 | 100.000057 | 100.000057 | 100.005909  100.049449  100.235089  100.803486  102.177186  104.944066  109.767469   117.26288 |  117.26288 | 1.558906  1.559371  1.561367   1.56752  1.582457  1.612595  1.664996  1.745791 |  100.0 |   0.000057 |  117.26288 |
    | 2000-01-04 05:00:00 | 100.000046 | 100.000046 | 100.005056  100.043041  100.207546  100.718455  101.969704  104.521059   109.01835   116.07531 |  116.07531 | 1.558897  1.559303  1.561071    1.5666  1.580205  1.608002  1.656901  1.733088 |  100.0 |   0.000046 |  116.07531 |
    | 2000-01-04 05:30:00 | 100.000036 | 100.000036 | 100.004325  100.037451  100.183146  100.642064  101.780877  104.131522  108.321257  114.960239 | 114.960239 |  1.55889  1.559243  1.560809  1.565774  1.578155  1.603771  1.649359  1.721138 |  100.0 |   0.000036 | 114.960239 |
    | 2000-01-04 06:00:00 | 100.000029 | 100.000029 |   100.0037  100.032578  100.161544  100.573482  101.609156   103.77308  107.673052  113.913933 | 113.913933 | 1.558883  1.559191  1.560577  1.565033  1.576291  1.599876   1.64234  1.709904 |  100.0 |   0.000029 | 113.913933 |
    | 2000-01-04 06:30:00 | 100.000023 | 100.000023 | 100.003165   100.02833  100.142427   100.51195  101.453105  103.443504  107.070751  112.932804 | 112.932804 | 1.558877  1.559146  1.560371  1.564367  1.574597  1.596293   1.63581  1.699353 |  100.0 |   0.000023 | 112.932804 |
    | 2000-01-04 07:00:00 | 100.000018 | 100.000018 | 100.002706  100.024629   100.12552  100.456778  101.311399  103.140697  106.511512   112.01341 |  112.01341 | 1.558872  1.559106   1.56019  1.563771  1.573059     1.593  1.629743  1.689449 |  100.0 |   0.000018 |  112.01341 |
    | 2000-01-04 07:30:00 | 100.000014 | 100.000014 | 100.002315  100.021405  100.110574  100.407342  101.182809  102.862695  105.992641  111.152449 | 111.152449 | 1.558868  1.559072  1.560029  1.563237  1.571663  1.589977  1.624109  1.680162 |  100.0 |   0.000014 | 111.152449 |
    | 2000-01-04 08:00:00 | 100.000011 | 100.000011 | 100.001979  100.018599  100.097368  100.363071  101.066204  102.607656  105.511577  110.346758 | 110.346758 | 1.558865  1.559042  1.559888  1.562759  1.570397  1.587202  1.618882  1.671458 |  100.0 |   0.000011 | 110.346758 |
    | 2000-01-04 08:30:00 | 100.000009 | 100.000009 | 100.001692  100.016156  100.085705   100.32345   100.96054  102.373857  105.065894  109.593305 | 109.593305 | 1.558862  1.559016  1.559762  1.562331   1.56925  1.584658  1.614036  1.663309 |  100.0 |   0.000009 | 109.593305 |
    | 2000-01-04 09:00:00 | 100.000007 | 100.000007 | 100.001447   100.01403   100.07541  100.288011  100.864857  102.159685  104.653293   108.88919 |  108.88919 | 1.558859  1.558993  1.559652  1.561948  1.568212  1.582327  1.609548  1.655684 |  100.0 |   0.000007 |  108.88919 |
    | 2000-01-04 09:30:00 | 100.000006 | 100.000006 | 100.001237  100.012181  100.066326  100.256333   100.77827  101.963637  104.271598  108.231638 | 108.231638 | 1.558857  1.558973  1.559554  1.561606  1.567273  1.580194  1.605394  1.648556 |  100.0 |   0.000006 | 108.231638 |
    | 2000-01-04 10:00:00 | 100.000004 | 100.000004 | 100.001058  100.010573  100.058315  100.228031  100.699966  101.784307  103.918752  107.617998 | 107.617998 | 1.558855  1.558956  1.559468  1.561301  1.566423  1.578242  1.601552  1.641898 |  100.0 |   0.000004 | 107.617998 |
    | 2000-01-04 10:30:00 | 100.000004 | 100.000004 | 100.000904  100.009175  100.051253  100.202759    100.6292  101.620386  103.592812  107.045735 | 107.045735 | 1.558853  1.558941  1.559393  1.561028  1.565656  1.576458  1.598001  1.635682 |  100.0 |   0.000004 | 107.045735 |
    | 2000-01-04 11:00:00 | 100.000003 | 100.000003 | 100.000773   100.00796  100.045031  100.180207  100.565286  101.470657  103.291944  106.512435 | 106.512435 | 1.558852  1.558929  1.559326  1.560785  1.564962  1.574828  1.594723  1.629885 |  100.0 |   0.000003 | 106.512435 |
    | 2000-01-04 11:30:00 | 100.000002 | 100.000002 | 100.000661  100.006905   100.03955  100.160092  100.507597  101.333985   103.01442  106.015792 | 106.015792 | 1.558851  1.558917  1.559267  1.560568  1.564337  1.573341  1.591698  1.624483 |  100.0 |   0.000002 | 106.015792 |
    | 2000-01-04 12:00:00 | 100.000002 | 100.000002 | 100.000565  100.005988  100.034724   100.14216  100.455561  101.209318  102.758612  105.553611 | 105.553611 |  1.55885  1.558908  1.559216  1.560374  1.563773  1.571984  1.588909  1.619452 |  100.0 |   0.000002 | 105.553611 |
    | 2000-01-04 12:30:00 | 100.000001 | 100.000001 | 100.000483  100.005191  100.030477  100.126182  100.408651   101.09568  102.522989  105.123802 | 105.123802 | 1.558849  1.558899   1.55917  1.560202  1.563264  1.570747   1.58634  1.614771 |  100.0 |   0.000001 | 105.123802 |
    | 2000-01-04 13:00:00 | 100.000001 | 100.000001 | 100.000413    100.0045  100.026741  100.111953  100.366388  100.992163  102.306111  104.724379 | 104.724379 | 1.558848  1.558892   1.55913  1.560049  1.562806  1.569621  1.583975  1.610418 |  100.0 |   0.000001 | 104.724379 |
    | 2000-01-04 13:30:00 | 100.000001 | 100.000001 | 100.000353    100.0039  100.023455  100.099287  100.328334  100.897928  102.106627  104.353453 | 104.353453 | 1.558848  1.558885  1.559095  1.559912  1.562394  1.568595  1.581799  1.606373 |  100.0 |   0.000001 | 104.353453 |
    | 2000-01-04 14:00:00 | 100.000001 | 100.000001 | 100.000301  100.003379  100.020567  100.088019   100.29409  100.812201   101.92327  104.009231 | 104.009231 | 1.558847   1.55888  1.559064  1.559791  1.562023  1.567663    1.5798  1.602619 |  100.0 |   0.000001 | 104.009231 |
    | 2000-01-04 14:30:00 | 100.000001 | 100.000001 | 100.000258  100.002928  100.018028  100.077998  100.263292  100.734262  101.754852  103.690014 | 103.690014 | 1.558847  1.558875  1.559037  1.559683   1.56169  1.566815  1.577962  1.599135 |  100.0 |   0.000001 | 103.690014 |
    | 2000-01-04 15:00:00 |      100.0 |      100.0 |  100.00022  100.002536  100.015799  100.069091  100.235609   100.66345  101.600261   103.39419 |  103.39419 | 1.558846  1.558871  1.559013  1.559587   1.56139  1.566045  1.576276  1.595906 |  100.0 |        0.0 |  103.39419 |
    | 2000-01-04 15:30:00 |      100.0 |      100.0 | 100.000188  100.002196  100.013841  100.061178   100.21074  100.599153  101.458459  103.120235 | 103.120235 | 1.558846  1.558867  1.558992  1.559502  1.561121  1.565346  1.574729  1.592915 |  100.0 |        0.0 | 103.120235 |
    | 2000-01-04 16:00:00 |      100.0 |      100.0 | 100.000161  100.001901  100.012122  100.054151   100.18841  100.540807  101.328474  102.866707 | 102.866707 | 1.558846  1.558864  1.558973  1.559426  1.560879  1.564712  1.573312  1.590146 |  100.0 |        0.0 | 102.866707 |
    | 2000-01-04 16:30:00 |      100.0 |      100.0 | 100.000137  100.001646  100.010614  100.047913  100.168371  100.487895    101.2094  102.632243 | 102.632243 | 1.558845  1.558861  1.558957  1.559359  1.560662  1.564136  1.572013  1.587585 |  100.0 |        0.0 | 102.632243 |
    | 2000-01-04 17:00:00 |      100.0 |      100.0 | 100.000117  100.001425   100.00929  100.042378  100.150397  100.439938  101.100394  102.415557 | 102.415557 | 1.558845  1.558859  1.558943  1.559299  1.560468  1.563615  1.570824  1.585218 |  100.0 |        0.0 | 102.415557 |
    | 2000-01-04 17:30:00 |      100.0 |      100.0 |   100.0001  100.001228  100.008105  100.037372   100.13399  100.395783  100.999192  102.212798 | 102.212798 | 1.558845  1.558857  1.558931  1.559246  1.560294  1.563144  1.569738  1.583033 |  100.0 |        0.0 | 102.212798 |
    | 2000-01-04 18:00:00 |      100.0 |      100.0 | 100.000085  100.001064  100.007093  100.033042  100.119612  100.356587  100.908247  102.028461 | 102.028461 | 1.558845  1.558855   1.55892    1.5592  1.560138  1.562717  1.568745  1.581017 |  100.0 |        0.0 | 102.028461 |
    | 2000-01-04 18:30:00 |      100.0 |      100.0 | 100.000073  100.000921  100.006207  100.029203  100.106732   100.32112  100.825148  101.858441 | 101.858441 | 1.558845  1.558854   1.55891  1.559158  1.559998  1.562332  1.567838  1.579157 |  100.0 |        0.0 | 101.858441 |
    | 2000-01-04 19:00:00 |      100.0 |      100.0 | 100.000062  100.000797   100.00543  100.025803  100.095201  100.289046  100.749263  101.701729 | 101.701729 | 1.558845  1.558852  1.558902  1.559122  1.559874  1.561983  1.567009  1.577443 |  100.0 |        0.0 | 101.701729 |
    | 2000-01-04 19:30:00 |      100.0 |      100.0 | 100.000053   100.00069  100.004749   100.02279  100.084882  100.260056  100.680009  101.557378 | 101.557378 | 1.558844  1.558851  1.558895  1.559089  1.559762  1.561667  1.566253  1.575863 |  100.0 |        0.0 | 101.557378 |
    | 2000-01-04 20:00:00 |      100.0 |      100.0 | 100.000046  100.000597  100.004152  100.020123  100.075652  100.233868  100.616844  101.424498 | 101.424498 | 1.558844   1.55885  1.558888  1.559061  1.559662  1.561383  1.565564   1.57441 |  100.0 |        0.0 | 101.424498 |
    | 2000-01-04 20:30:00 |      100.0 |      100.0 | 100.000039  100.000516   100.00363  100.017763    100.0674  100.210223  100.559268  101.302255 | 101.302255 | 1.558844  1.558849  1.558883  1.559035  1.559572  1.561125  1.564936  1.573072 |  100.0 |        0.0 | 101.302255 |
    | 2000-01-04 21:00:00 |      100.0 |      100.0 | 100.000033  100.000446  100.003172  100.015675  100.060025  100.188885  100.506816   101.18987 |  101.18987 | 1.558844  1.558849  1.558878  1.559013  1.559493  1.560894  1.564364  1.571843 |  100.0 |        0.0 |  101.18987 |
    | 2000-01-04 21:30:00 |      100.0 |      100.0 | 100.000029  100.000386  100.002772  100.013828  100.053437   100.16964  100.459061  101.086612 | 101.086612 | 1.558844  1.558848  1.558874  1.558993  1.559421  1.560684  1.563843  1.570713 |  100.0 |        0.0 | 101.086612 |
    | 2000-01-04 22:00:00 |      100.0 |      100.0 | 100.000024  100.000334  100.002421  100.012195  100.047555  100.152291  100.415606  100.991798 | 100.991798 | 1.558844  1.558847   1.55887  1.558975  1.559358  1.560496  1.563369  1.569676 |  100.0 |        0.0 | 100.991798 |
    | 2000-01-04 22:30:00 |      100.0 |      100.0 | 100.000021  100.000289  100.002115  100.010751  100.042305  100.136658  100.376086   100.90479 |  100.90479 | 1.558844  1.558847  1.558866   1.55896  1.559301  1.560326  1.562938  1.568724 |  100.0 |        0.0 |  100.90479 |
    | 2000-01-04 23:00:00 |      100.0 |      100.0 | 100.000018   100.00025  100.001847  100.009476  100.037622   100.12258  100.340165  100.824996 | 100.824996 | 1.558844  1.558847  1.558864  1.558946   1.55925  1.560173  1.562546  1.567852 |  100.0 |        0.0 | 100.824996 |
    | 2000-01-04 23:30:00 |      100.0 |      100.0 | 100.000015  100.000216  100.001612   100.00835  100.033445  100.109907  100.307534   100.75186 |  100.75186 | 1.558844  1.558846  1.558861  1.558934  1.559205  1.560035   1.56219  1.567052 |  100.0 |        0.0 |  100.75186 |

.. _kinw_williams_ext_linear_storage_cascade:

linear storage cascade
______________________

This example shows using |kinw_williams_ext| like a simple linear storage cascade.
Therefore, you must assign a constant flow velocity to parameter |VG2FG| via the
keyword argument `velocity` in m/s.  Alternatively, you can define the number of hours
it takes for a flood wave to travel through the whole channel via the keyword argument
`timedelay`.

When defining a constant flow velocity, be aware that this is also the wave celerity
due to the assumed linearity.  Hence, we set a velocity of 1.5 m/s, which is higher
than the average flow velocity in the :ref:`kinw_williams_ext_main_channel_flow`
example but leads to comparable results.

Note that, even when used as a linear storage cascade, |kinw_williams_ext| still relies
on the numerical integration of its differential equations.  Hence, it is
computationally less efficient than Unit Hydrograph based approaches like those
implemented in application model |arma_rimorido|, especially for short channels and
high numbers of channel subsections.  On the other hand, |kinw_williams_ext| only
requires the actual water volume stored in the subsections to describe its current
state, which is more concise and understandable than the logged values of
|arma_rimorido| and can simplify the coupling with data assimilation approaches.

.. integration-test::

    >>> vg2fg(velocity=1.5)
    >>> test.inits.vg = 0.83333333
    >>> test("kinw_williams_ext_linear_storage_cascade")
    |                date |         qz |        qza |                                                                                             qg |         qa |                                                                             vg | input1 |     input2 |     output |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 100.000003 | 100.000003 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 100.000029 | 100.000029 | 100.000004       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |   0.000029 |      100.0 |
    | 2000-01-01 04:00:00 | 100.000175 | 100.000175 | 100.000024  100.000001       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 0.833334  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |   0.000175 |      100.0 |
    | 2000-01-01 04:30:00 | 100.000826 | 100.000826 | 100.000124  100.000009  100.000001       100.0       100.0       100.0       100.0       100.0 |      100.0 | 0.833335  0.833334  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |   0.000826 |      100.0 |
    | 2000-01-01 05:00:00 | 100.003194 | 100.003194 | 100.000515  100.000048  100.000005       100.0       100.0       100.0       100.0       100.0 |      100.0 |  0.83334  0.833334  0.833333  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |   0.003194 |      100.0 |
    | 2000-01-01 05:30:00 | 100.010515 | 100.010515 | 100.001822    100.0002  100.000024  100.000002       100.0       100.0       100.0       100.0 |      100.0 | 0.833355  0.833337  0.833334  0.833333  0.833333  0.833333  0.833333  0.833333 |  100.0 |   0.010515 |      100.0 |
    | 2000-01-01 06:00:00 | 100.030315 | 100.030315 | 100.005635  100.000715  100.000096   100.00001  100.000001       100.0       100.0       100.0 |      100.0 |   0.8334  0.833346  0.833335  0.833334  0.833333  0.833333  0.833333  0.833333 |  100.0 |   0.030315 |      100.0 |
    | 2000-01-01 06:30:00 | 100.078179 | 100.078179 |  100.01556  100.002239   100.00033  100.000041  100.000005       100.0       100.0       100.0 |      100.0 | 0.833513   0.83337  0.833338  0.833334  0.833333  0.833333  0.833333  0.833333 |  100.0 |   0.078179 |      100.0 |
    | 2000-01-01 07:00:00 | 100.183351 | 100.183351 | 100.038983  100.006266  100.001009  100.000141  100.000018  100.000002       100.0       100.0 |      100.0 | 0.833772  0.833429  0.833348  0.833336  0.833334  0.833333  0.833333  0.833333 |  100.0 |   0.183351 |      100.0 |
    | 2000-01-01 07:30:00 | 100.396214 | 100.396214 | 100.089788  100.015938  100.002785  100.000431  100.000062  100.000008  100.000001       100.0 |      100.0 | 0.834324  0.833562  0.833371   0.83334  0.833334  0.833333  0.833333  0.833333 |  100.0 |   0.396214 |      100.0 |
    | 2000-01-01 08:00:00 | 100.797305 | 100.797305 | 100.192145  100.037315  100.007042  100.001191  100.000188  100.000027  100.000003       100.0 |      100.0 | 0.835413  0.833841  0.833426   0.83335  0.833336  0.833334  0.833333  0.833333 |  100.0 |   0.797305 |      100.0 |
    | 2000-01-01 08:30:00 | 101.507032 | 101.507032 | 100.375615  100.089574  100.017567  100.003341  100.000575  100.000093  100.000013  100.000002 | 100.000002 |  0.83745  0.834356  0.833556  0.833376  0.833341  0.833335  0.833334  0.833333 |  100.0 |   1.507032 | 100.000002 |
    | 2000-01-01 09:00:00 | 102.694853 | 102.694853 | 100.714556  100.176996  100.039174  100.008024  100.001518  100.000266  100.000043  100.000006 | 100.000006 | 0.841014  0.835323  0.833804  0.833432  0.833353  0.833337  0.833334  0.833333 |  100.0 |   2.694853 | 100.000006 |
    | 2000-01-01 09:30:00 | 104.586356 | 104.586356 | 101.289004  100.335724   100.08002  100.017733  100.003636  100.000691  100.000122   100.00002 |  100.00002 |  0.84695  0.837039  0.834264  0.833544  0.833378  0.833342  0.833335  0.833334 |  100.0 |   4.586356 |  100.00002 |
    | 2000-01-01 10:00:00 | 107.466641 | 107.466641 | 102.218524  100.609237  100.154291  100.036546  100.008049   100.00165  100.000315  100.000056 | 100.000056 | 0.856396  0.839936  0.835083  0.833756  0.833429  0.833354  0.833337  0.833334 |  100.0 |   7.466641 | 100.000056 |
    | 2000-01-01 10:30:00 | 111.678559 | 111.678559 | 103.660639  101.059555  100.283661  100.071218  100.016703   100.00366   100.00075  100.000144 | 100.000144 | 0.870828  0.844618  0.836479  0.834139  0.833528  0.833377  0.833343  0.833335 |  100.0 |  11.678559 | 100.000144 |
    | 2000-01-01 11:00:00 | 117.614836 | 117.614836 | 105.813892  101.771348  100.499864  100.132369  100.032843   100.00764  100.001668  100.000342 | 100.000342 |  0.89207  0.851894  0.838768    0.8348  0.833707  0.833423  0.833353  0.833338 |  100.0 |  17.614836 | 100.000342 |
    | 2000-01-01 11:30:00 | 125.703736 | 125.703736 |  108.92596   102.84193  100.851971  100.236874  100.061977  100.015247  100.003529  100.000769 | 100.000769 |  0.92227  0.862846   0.84235  0.835907  0.834022  0.833507  0.833374  0.833343 |  100.0 |  25.703736 | 100.000769 |
    | 2000-01-01 12:00:00 | 136.388647 | 136.388647 | 113.260597  104.437529   101.38979  100.406727  100.111755  100.028911  100.007053  100.001624 | 100.001624 | 0.963901  0.878727  0.847836  0.837677  0.834553  0.833656  0.833414  0.833352 |  100.0 |  36.388647 | 100.001624 |
    | 2000-01-01 12:30:00 | 150.102726 | 150.102726 | 119.133712  106.714761  102.198745  100.673903  100.194045  100.052635  100.013481  100.003265 | 100.003265 | 1.019645  0.901081  0.855965  0.840421  0.835416   0.83391  0.833484  0.833371 |  100.0 |  50.102726 | 100.003265 |
    | 2000-01-01 13:00:00 | 167.240349 | 167.240349 | 126.874578  109.872987  103.378186  101.081817  100.325609  100.092374  100.024763  100.006284 | 100.006284 | 1.092303  0.931684  0.867655  0.844555  0.836777   0.83433  0.833606  0.833404 |  100.0 |  67.240349 | 100.006284 |
    | 2000-01-01 13:30:00 | 188.127569 | 188.127569 | 136.818241  114.137648  105.049585  101.686963  100.529675   100.15681  100.043891  100.011639 | 100.011639 |  1.18466  0.972509  0.884014  0.850608  0.838861  0.835001  0.833809  0.833462 |  100.0 |  88.127569 | 100.011639 |
    | 2000-01-01 14:00:00 | 212.993962 | 212.993962 | 149.288289  119.754624  107.356776  102.560717  100.837402  100.258225  100.075307   100.02082 |  100.02082 |  1.29933   1.02567   0.90633   0.85924  0.841963  0.836044  0.834138   0.83356 |  100.0 | 112.993962 |  100.02082 |
    | 2000-01-01 14:30:00 | 241.948195 | 241.948195 | 164.577878  126.982151  110.464805  103.790694   101.28938  100.413493  100.125425  100.036079 | 100.036079 | 1.438597  1.093342  0.936061  0.871254  0.846465   0.83762  0.834657  0.833721 |  100.0 | 141.948195 | 100.036079 |
    | 2000-01-01 15:00:00 |  274.95935 |  274.95935 | 182.930279  136.080545  114.557212  105.481472  101.937036  100.645164   100.20324  100.060726 | 100.060726 | 1.604249  1.177672  0.974803   0.88759  0.852845  0.839946  0.835452  0.833977 |  100.0 |  174.95935 | 100.060726 |
    | 2000-01-01 15:30:00 | 311.845554 | 311.845554 | 204.520302  147.300215  119.831655  107.754473  102.843826  100.982593  100.321049  100.099495 | 100.099495 | 1.797435  1.280668  1.024247  0.909329  0.861684  0.843296  0.836643  0.834376 |  100.0 | 211.845554 | 100.099495 |
    | 2000-01-01 16:00:00 | 352.270855 | 352.270855 | 229.437964  160.868596  126.493943  110.746858  104.086078  101.463044  100.495242   100.15899 |  100.15899 | 2.018534  1.404093  1.086121  0.937674  0.873673  0.848017  0.838385  0.834981 |  100.0 | 252.270855 |  100.15899 |
    | 2000-01-01 16:30:00 | 395.750601 | 395.750601 | 257.675565  176.976772  134.750645  114.609327  105.753344  102.132699  100.747144  100.248196 | 100.248196 | 2.267069   1.54935  1.162128  0.973928  0.889614  0.854535  0.840879   0.83588 |  100.0 | 295.750601 | 100.248196 |
    | 2000-01-01 17:00:00 | 441.664962 | 441.664962 | 289.119149  195.766623  144.800567    119.5028  107.948146  103.047481   101.10388  100.379066 | 100.379066 | 2.541651  1.717385  1.253867  1.019464  0.910413  0.863356  0.844378  0.837184 |  100.0 | 341.664962 | 100.379066 |
    | 2000-01-01 17:30:00 | 489.279607 | 489.279607 | 323.544962  217.319294  156.825521  125.593996  110.785015  104.273594  101.599197  100.567155 | 100.567155 | 2.839974  1.908591  1.362756  1.075681  0.937069  0.875076  0.849192  0.839042 |  100.0 | 389.279607 | 100.567155 |
    | 2000-01-01 18:00:00 |  537.77215 |  537.77215 | 360.621205  241.645756  170.980833  133.050053  114.388755  105.887701  102.274201  100.832281 | 100.832281 | 3.158845  2.122747  1.489953  1.143956  0.970659  0.890378  0.855696  0.841637 |  100.0 |  437.77215 | 100.832281 |
    | 2000-01-01 18:30:00 | 586.262607 | 586.262607 | 399.914979  268.680063  187.386136  142.032368  118.891921  107.976642  103.177938  101.199188 | 101.199188 | 3.494271   2.35897  1.636282  1.225593  1.012312  0.910026  0.864334  0.845199 |  100.0 | 486.262607 | 101.199188 |
    | 2000-01-01 19:00:00 | 633.846018 | 633.846018 |  440.90402  298.275763  206.116945  152.689909   124.43153  110.636646  104.367758  101.698174 | 101.698174 | 3.841567  2.615701  1.802168  1.321762  1.063177  0.934857  0.875618  0.850004 |  100.0 | 533.846018 | 101.698174 |
    | 2000-01-01 19:30:00 | 679.625318 | 679.625318 | 482.992517  330.205706  227.197513   165.15232   131.14509  113.971985  105.909398  102.365637 | 102.365637 | 4.195506  2.890717  1.987582  1.433443   1.12439  0.965768   0.89013  0.856383 |  100.0 | 579.625318 | 102.365637 |
    | 2000-01-01 20:00:00 | 722.742733 | 722.742733 |  525.53009  364.165301  250.595386  179.523136  139.166058  118.093057  107.876735  103.244501 | 103.244501 | 4.550489  3.181174  2.192008  1.561373  1.197033    1.0037   0.90852  0.864721 |  100.0 | 622.742733 | 103.244501 |
    | 2000-01-01 20:30:00 | 762.408181 | 762.408181 | 567.832885  399.779055  276.217981  195.873461  148.618912  123.113916  110.351153  104.384471 | 104.384471 | 4.900724   3.48367  2.414418  1.705993  1.282091  1.049609  0.931493  0.875461 |  100.0 | 662.408181 | 104.384471 |
    | 2000-01-01 21:00:00 | 797.923464 | 797.923464 | 609.205657  436.610059  303.911405  214.236431  159.614023  129.149306  113.420501  105.842078 | 105.842078 | 5.240416  3.794343  2.653276  1.867408  1.380411  1.104445  0.959805  0.889102 |  100.0 | 697.923464 | 105.842078 |
    | 2000-01-01 21:30:00 | 828.701418 | 828.701418 | 648.963764  474.171932  333.461599  234.602757  172.242551  136.311269  117.177641  107.680468 | 107.680468 | 5.563944  4.108968  2.906554  2.045354   1.49266  1.169121  0.994245  0.906197 |  100.0 | 728.701418 | 107.680468 |
    | 2000-01-01 22:00:00 |  854.27953 |  854.27953 | 686.454047  511.942614   364.59778   256.91759  186.571597  144.705446  121.718577  109.968909 | 109.968909 |  5.86603  4.423088  3.171775  2.239179  1.619282   1.24448  1.035621  0.927347 |  100.0 |  754.27953 | 109.968909 |
    | 2000-01-01 22:30:00 | 874.327871 | 874.327871 | 721.073722  549.379357   396.99802  281.078879  202.639833  154.427192  127.140214  112.781992 | 112.781992 | 6.141887  4.732138  3.446061  2.447833  1.760473  1.331263  1.084738  0.953191 |  100.0 | 774.327871 | 112.781992 |
    | 2000-01-01 23:00:00 | 888.651531 | 888.651531 | 752.286603   585.93421  430.296724  306.937331  220.453829  165.557677  133.537791   116.19852 |  116.19852 | 6.387344  5.031573  3.726209   2.66988  1.916143  1.430076  1.142374  0.984402 |  100.0 | 788.651531 |  116.19852 |
    | 2000-01-01 23:30:00 | 897.187983 | 897.187983 | 779.636121  621.069333  464.093642  334.297989  239.985245  178.160102  141.002065  120.300084 | 120.300084 | 6.598937  5.316993  4.008765  2.903512  2.085906  1.541361  1.209258  1.021666 |  100.0 | 797.187983 | 120.300084 |
    | 2000-01-02 00:00:00 |      900.0 |      900.0 | 802.754865  654.271519   497.96405  362.923383   261.16903   192.27621  149.616333  125.169355 | 125.169355 | 6.773979  5.584263  4.290119  3.146585  2.269064  1.665369  1.286046   1.06567 |  100.0 |      800.0 | 125.169355 |
    | 2000-01-02 00:30:00 | 897.264896 | 897.264896 | 821.370523  685.065384  531.469639  392.538118  283.902737  207.923238  159.453399  130.888117 | 130.888117 | 6.910589  5.829612  4.566591  3.396662  2.464607  1.802132  1.373292  1.117088 |  100.0 | 797.264896 | 130.888117 |
    | 2000-01-02 01:00:00 | 889.260928 | 889.260928 | 835.308283  713.024796  564.169685  422.834729  308.046991  225.091439  170.572593  137.535102 | 137.535102 | 7.007703  6.049722   4.83453  3.651065  2.671225  1.951452  1.471426  1.176555 |  100.0 | 789.260928 | 137.535102 |
    | 2000-01-02 01:30:00 | 876.351729 | 876.351729 | 844.489941  737.782206  595.632058  453.480553  333.427112  243.742296  183.016951  145.183676 | 145.183676 | 7.065055  6.241796    5.0904  3.906938  2.887322  2.112884  1.580731  1.244655 |  100.0 | 776.351729 | 145.183676 |
    | 2000-01-02 02:00:00 | 858.969595 | 858.969595 | 848.930035  759.035693   625.44368   484.12536   359.83583   263.80751  196.810681  153.899458 | 153.899458 | 7.083126  6.403606  5.330866  4.161311  3.111043  2.285735  1.701326  1.321895 |  100.0 | 758.969595 | 153.899458 |
    | 2000-01-02 02:30:00 |   837.5984 |   837.5984 | 848.729449  776.553625  653.220083  514.409443   387.03702  285.188812     211.957   163.73794 |  163.73794 |  7.06309  6.533523  5.552866   4.41117  3.340313  2.469062  1.833143   1.40869 |  100.0 |   737.5984 |  163.73794 |
    | 2000-01-02 03:00:00 | 812.756821 | 812.756821 | 844.066968  790.176959  678.613773  543.971882  414.770292  307.758631  228.436444    174.7422 |   174.7422 | 7.006732  6.630525   5.75368  4.653525  3.572876  2.661683  1.975923  1.505339 |  100.0 | 712.756821 |   174.7422 |
    | 2000-01-02 03:30:00 | 784.982427 | 784.982427 | 835.189306  799.819288  701.321188  572.458686  442.756294  331.361596  246.205719  186.940782 | 186.940782 | 6.916359  6.694191  5.930977  4.885478   3.80634  2.862194  2.129203  1.612016 |  100.0 | 684.982427 | 186.940782 |
    | 2000-01-02 04:00:00 | 754.817086 | 754.817086 | 822.400114  805.464837  721.088105  599.530551  470.702534  355.816831  265.197149  200.345826 | 200.345826 |  6.79471  6.724674  6.082855  5.104281  4.038231  3.068988  2.292319  1.728749 |  100.0 | 654.817086 | 200.345826 |
    | 2000-01-02 04:30:00 |  722.79402 |  722.79402 | 806.087318  807.083371  737.727009  624.881841  498.322072  380.930257  285.323576  214.952546 | 214.952546 | 6.644782  6.722881  6.207696  5.307403  4.266038  3.280293  2.464411  1.855416 |  100.0 |  622.79402 | 214.952546 |
    | 2000-01-02 05:00:00 | 689.426711 | 689.426711 | 786.509598  805.053778  751.035242   648.18628  525.276742  406.452414  306.455874  230.733725 | 230.733725 | 6.470033  6.689502   6.30493  5.492531  4.487275  3.494177  2.644405  1.991716 |  100.0 | 589.426711 | 230.733725 |
    | 2000-01-02 05:30:00 | 655.199767 | 655.199767 | 764.205454  799.248922  761.002564  669.224289  551.309363  432.172794  328.468727  247.645991 | 247.645991 | 6.273822  6.626423  6.373773  5.657732  4.699522  3.708623  2.831072  2.137197 |  100.0 | 555.199767 | 247.645991 |
    | 2000-01-02 06:00:00 | 620.561767 | 620.561767 | 739.543806   789.99472  767.555054  687.768149  576.123951  457.835722  351.200093  265.624131 | 265.624131 | 6.059655  6.535612  6.414164  5.801348  4.900482  3.921541  3.023016  2.291234 |  100.0 | 520.561767 | 265.624131 |
    | 2000-01-02 06:30:00 | 585.920022 | 585.920022 | 712.943765  777.545498  770.711019  703.640023  599.454019  483.188613  374.473862  284.583212 | 284.583212 | 5.831012  6.419329  6.426466  5.922076  5.088017  4.130819  3.218703  2.453037 |  100.0 | 485.920022 | 284.583212 |
    | 2000-01-02 07:00:00 |  551.63713 |  551.63713 | 684.817653  762.187043  770.537912  716.705182  621.054623  507.978276  398.098928  304.419028 | 304.419028 | 5.591287  6.280064  6.411435  6.018975  5.260188  4.334357  3.416485  2.621661 |  100.0 |  451.63713 | 304.419028 |
    | 2000-01-02 07:30:00 | 518.029154 | 518.029154 | 655.563444  744.228394  767.148075  726.872771  640.706638  531.955654  421.872467  325.009257 | 325.009257 | 5.343725  6.120467   6.37018   6.09147  5.415287  4.530108  3.614635  2.796015 |  100.0 | 418.029154 | 325.009257 |
    | 2000-01-02 08:00:00 | 485.365238 | 485.365238 | 625.558483  723.993883  760.693724  734.095502  658.220256  554.880407  445.583475  346.215056 | 346.215056 | 5.091377  5.943283   6.30412  6.139347  5.551862   4.71612   3.81137  2.974878 |  100.0 | 385.365238 | 346.215056 |
    | 2000-01-02 08:30:00 |  453.86844 |  453.86844 | 595.154454  701.815632  751.361411  738.368396  673.437638   576.52519  469.016494  367.883056 | 367.883056 | 4.837063  5.751293  6.214937  6.162735  5.668738  4.890563  4.004885  3.156918 |  100.0 |  353.86844 | 367.883056 |
    | 2000-01-02 09:00:00 | 423.717566 | 423.717566 | 564.673601   678.02665  739.366165  739.726639  686.234726   596.67954  491.955393  389.847694 | 389.847694 | 4.583342  5.547258  6.104526  6.162086  5.765023  5.051762  4.193389  3.340712 |  100.0 | 323.717566 | 389.847694 |
    | 2000-01-02 09:30:00 | 395.049792 | 395.049792 | 534.406116  652.954665  724.945495  738.242717    696.5222  615.153264  514.187134  411.933839 | 411.933839 |   4.3325   5.33387  5.974943  6.138151   5.84012  5.198226  4.375128  3.524768 |  100.0 | 295.049792 | 411.933839 |
    | 2000-01-02 10:00:00 | 367.963873 | 367.963873 | 504.608596   626.91675  708.353445  734.022931  704.245638  631.779282  535.505401  433.959612 | 433.959612 |  4.08654  5.113715  5.828357  6.091946  5.893719  5.328665  4.548421   3.70755 |  100.0 | 267.963873 | 433.959612 |
    | 2000-01-02 10:30:00 | 342.523746 | 342.523746 | 475.503473  600.214791  689.854843  727.203459  709.384913  646.415864  555.714019  455.739355 | 455.739355 | 3.847176  4.889235  5.667005  6.024718  5.925792   5.44201  4.711684  3.887505 |  100.0 | 242.523746 | 455.739355 |
    | 2000-01-02 11:00:00 | 318.762374 | 318.762374 | 447.279272  573.131818   669.71987  717.946096   711.95291  658.948253  574.630083  477.086655 | 477.086655 | 3.615846  4.662701  5.493146  5.937911   5.93658  5.537418  4.863457  4.063083 |  100.0 | 218.762374 | 477.086655 |
    | 2000-01-02 11:30:00 |  296.68568 |  296.68568 | 420.091577  545.929165  648.219062  706.433803  711.993647  669.289654   592.08673  497.817352 | 497.817352 | 3.393715  4.436193  5.309024  5.833124  5.926572  5.614285  5.002422  4.232768 |  100.0 |  196.68568 | 497.817352 |
    | 2000-01-02 12:00:00 | 276.276442 | 276.276442 | 394.064569  518.844435  625.618805  692.866207  709.579882  677.381619  607.935501  517.752465 | 517.752465 | 3.181697  4.211589   5.11683  5.712079  5.896488  5.672242  5.127425  4.395097 |  100.0 | 176.276442 | 517.752465 |
    | 2000-01-02 12:30:00 | 257.498082 | 257.498082 | 369.293012    492.0902  602.177392  677.455146  704.810318  683.193847  622.048264  536.720957 | 536.720957 | 2.980466  3.990554  4.918674  5.576579  5.847248  5.711152  5.237487  4.548686 |  100.0 | 157.498082 | 536.720957 |
    | 2000-01-02 13:00:00 | 240.298231 | 240.298231 | 345.844566  465.853378  578.141656  660.420381  697.806503  686.723451  634.318667  554.562302 | 554.562302 | 2.790482  3.774538  4.716555  5.428477  5.779953  5.731101  5.331816  4.692248 |  100.0 | 140.298231 | 554.562302 |
    | 2000-01-02 13:30:00 | 224.612057 | 224.612057 | 323.762321  440.295205  553.744205  641.985541  688.709516  687.993734  644.663114  571.128773 | 571.128773 | 2.612012  3.564779  4.512346  5.269643   5.69585   5.73239  5.409811   4.82461 |  100.0 | 124.612057 | 571.128773 |
    | 2000-01-02 14:00:00 | 210.365288 | 210.365288 | 303.067449  415.551714  529.201233  622.374377  677.676541  687.052558   653.02128  586.287441 | 586.287441 | 2.445148  3.362307  4.307777  5.101931  5.596306  5.715513  5.471067  4.944731 |  100.0 | 110.365288 | 586.287441 |
    | 2000-01-02 14:30:00 |  197.47692 |  197.47692 | 283.761907   391.73465  504.710903  601.807381  664.877421   683.97035  659.356159  599.921827 | 599.921827 | 2.289835  3.167957   4.10442  4.927158   5.48278  5.681146  5.515373  5.051712 |  100.0 |   97.47692 | 599.921827 |
    | 2000-01-02 15:00:00 | 185.861607 | 185.861607 | 265.831096  368.932717  480.452246  580.498793  650.491251  678.837836  663.653698  611.933214 | 611.933214 |  2.14589  2.982374  3.903685  4.747074  5.356794  5.630122  5.542704  5.144809 |  100.0 |  85.861607 | 611.933214 |
    | 2000-01-02 15:30:00 | 175.431719 | 175.431719 | 249.246445  347.213106  456.584556  558.654037  634.703106  671.763562  665.922033  622.241576 | 622.241576 | 2.013024  2.806034  3.706816  4.563349  5.219906  5.563413  5.553219  5.223434 |  100.0 |  75.431719 | 622.241576 |
    | 2000-01-02 16:00:00 | 166.099094 | 166.099094 | 233.967857  326.623206  433.247213  536.467568  617.700942   662.87128   666.19038  630.786158 | 630.786158 |  1.89086  2.639254  3.514893  4.377552  5.073686  5.482106  5.547244  5.287162 |  100.0 |  66.099094 | 630.786158 |
    | 2000-01-02 16:30:00 | 157.776495 | 157.776495 |  219.94599  307.192443   410.55989  514.121155  599.672724   652.29726  664.507623  637.525688 | 637.525688 | 1.778955   2.48221  3.328832  4.191142  4.919693  5.387382  5.525266  5.335729 |  100.0 |  57.776495 | 637.525688 |
    | 2000-01-02 17:00:00 | 150.378786 | 150.378786 | 207.124352  288.934191   388.62308  491.782563  580.803823  640.187592  660.940652  642.438245 | 642.438245 | 1.676813  2.334953  3.149392  4.005455  4.759454  5.280491   5.48791  5.369033 |  100.0 |  50.378786 | 642.438245 |
    | 2000-01-02 17:30:00 | 143.823868 | 143.823868 | 195.441194  271.847691    367.5189  469.604622    561.2747  626.695539  655.572509  645.520822 | 645.520822 | 1.583901  2.197421  2.977183  3.821701  4.594448  5.162734  5.435932  5.387126 |  100.0 |  43.823868 | 645.520822 |
    | 2000-01-02 18:00:00 | 138.033385 | 138.033385 | 184.831188  255.919952  347.312106  447.724663  541.258895   611.97898  648.500381  646.788582 | 646.788582 | 1.499665  2.069461  2.812678  3.640958  4.426087  5.035438  5.370193  5.390208 |  100.0 |  38.033385 | 646.788582 |
    | 2000-01-02 18:30:00 | 132.933235 | 132.933235 | 175.226905   241.12758  328.051274  426.264268  520.921328  596.197996  639.833506  646.273869 | 646.273869 | 1.423537   1.95084  2.656215  3.464175  4.255704   4.89994  5.291649  5.378615 |  100.0 |  32.933235 | 646.273869 |
    | 2000-01-02 19:00:00 | 128.453911 | 128.453911 | 166.560081  227.438524  309.770101  405.329318  500.416921   579.51262  629.691028  644.024999 | 644.024999 | 1.354946  1.841259  2.508018  3.292168  4.084546  4.757567  5.201328  5.352814 |  100.0 |  28.453911 | 644.024999 |
    | 2000-01-02 19:30:00 | 124.530701 | 124.530701 | 158.762695  214.813708  292.488778  385.010291   479.88951  562.080793  618.199854  640.104861 | 640.104861 | 1.293328  1.740367  2.368203  3.125629  3.913764  4.609623  5.100314  5.313385 |  100.0 |  24.530701 | 640.104861 |
    | 2000-01-02 20:00:00 | 121.103761 | 121.103761 | 151.767856  203.208532  276.215406  365.382769  459.471058  544.056531  605.492555  634.589385 | 634.589385 | 1.238133  1.647774  2.236791  2.965128  3.744405  4.457369  4.989729  5.261011 |  100.0 |  21.103761 | 634.589385 |
    | 2000-01-02 20:30:00 | 118.118088 | 118.118088 | 145.510531  192.574242  260.947406  346.508128  439.281132  525.588324  591.705335  627.565901 | 627.565901 | 1.188826  1.563059  2.113719  2.811119  3.577413  4.302016  4.870718  5.196462 |  100.0 |  18.118088 | 627.565901 |
    | 2000-01-02 21:00:00 | 115.523427 | 115.523427 | 139.928113  182.859158  246.672917  328.434364  419.426635  506.817774  576.976112  619.131435 | 619.131435 | 1.144898  1.485783  1.998854  2.663948  3.413627  4.144712  4.744433  5.120582 |  100.0 |  15.523427 | 619.131435 |
    | 2000-01-02 21:30:00 | 113.274094 | 113.274094 | 134.960855   174.00976  233.372142  311.197025  400.001757   487.87846  561.442727   609.39098 |  609.39098 | 1.105862  1.415495  1.892002  2.523863  3.253779  3.986534  4.612018  5.034275 |  100.0 |  13.274094 |  609.39098 |
    | 2000-01-02 22:00:00 | 111.328777 | 111.328777 |  130.55218  165.971632  221.018626  294.820217   381.08813  468.895043  545.241312  598.455762 | 598.455762 |  1.07126   1.35174  1.792917   2.39102  3.098497  3.828482  4.474594  4.938489 |  100.0 |  11.328777 | 598.455762 |
    | 2000-01-02 22:30:00 | 109.650285 | 109.650285 | 126.648894  158.690276  209.580464  279.317656  362.755146  449.982584   528.50481  586.441545 | 586.441545 | 1.040662  1.294066  1.701315  2.265493  2.948309  3.671472  4.333254  4.834203 |  100.0 |   9.650285 | 586.441545 |
    | 2000-01-02 23:00:00 | 108.205287 | 108.205287 | 123.201299  152.111799  199.021412  264.693743  345.060427  431.246083  511.361687  573.467002 | 573.467002 | 1.013669  1.242027  1.616878  2.147283  2.803649  3.516338  4.189046  4.722413 |  100.0 |   8.205287 | 573.467002 |
    | 2000-01-02 23:30:00 | 106.964036 | 106.964036 | 120.169398  146.182658  189.303943  250.945089  328.048917  412.776689  493.929957  559.647154 | 559.647154 |   0.9899  1.195203  1.539259  2.036329  2.664862  3.363828   4.04297  4.604122 |  100.0 |   6.964036 | 559.647154 |
    | 2000-01-03 00:00:00 | 105.900084 | 105.900084 | 117.496281  140.854971  180.381619  238.060262  311.760338  394.666507  476.336274   545.11207 |  545.11207 | 0.969026  1.153157  1.468111  1.932508  2.532202  3.214597  3.895965  4.480326 |  100.0 |   5.900084 |  545.11207 |
    | 2000-01-03 00:30:00 |  104.99001 |  104.99001 |  115.15133  136.076533  172.213629  226.021792  296.218448  376.984982  458.684301  529.975288 | 529.975288 | 0.950736  1.115492  1.403065  1.835653  2.405848  3.069217  3.748906  4.352002 |  100.0 |    4.99001 | 529.975288 |
    | 2000-01-03 01:00:00 | 104.213141 | 104.213141 |   113.0986  131.800736  164.755409  214.807329  281.440436  359.795758  441.075195  514.352678 | 514.352678 | 0.934742  1.081828  1.343746   1.74556  2.285908  2.928178  3.602603  4.220103 |  100.0 |   4.213141 | 514.352678 |
    | 2000-01-03 01:30:00 | 103.551298 | 103.551298 | 111.305364  127.983368  157.962232  204.390181  267.435357  343.152625  423.601828  498.356423 | 498.356423 | 0.920785  1.051808  1.289784  1.661989  2.172427  2.791887  3.457795  4.085545 |  100.0 |   3.551298 | 498.356423 |
    | 2000-01-03 02:00:00 | 102.988541 | 102.988541 | 109.741947  124.582756  151.789728   194.74018  254.204963  327.099936  406.348523    482.0941 |   482.0941 | 0.908629  1.025094  1.240812  1.584678   2.06539  2.660676  3.315147  3.949203 |  100.0 |   2.988541 |   482.0941 |
    | 2000-01-03 02:30:00 |  102.51094 |  102.51094 | 108.381545  121.559849  146.194313  185.824474  241.744522  311.673102  389.390931  465.667892 | 465.667892 | 0.898062  1.001373   1.19647  1.513344  1.964734  2.534804  3.175255  3.811904 |  100.0 |    2.51094 | 465.667892 |
    | 2000-01-03 03:00:00 | 102.106358 | 102.106358 | 107.200034   118.87825  141.133551  177.608234  230.043628  296.899151  372.796034  449.173959 | 449.173959 | 0.888893  0.980352   1.15641   1.44769  1.870351  2.414464  3.038641  3.674424 |  100.0 |   2.106358 | 449.173959 |
    | 2000-01-03 03:30:00 | 101.764251 | 101.764251 | 106.175783  116.504195  136.566442  170.055294   219.08699  282.797339  356.622258  432.701927 | 432.701927 | 0.880952  0.961761  1.120298   1.38741  1.782094  2.299786  2.905756   3.53748 |  100.0 |   1.764251 | 432.701927 |
    | 2000-01-03 04:00:00 |  101.47548 |  101.47548 | 105.289453  114.406513  132.453657  163.128716  208.855173  269.379778  340.919694  416.334519 | 416.334519 | 0.874087  0.945351  1.087813  1.332195  1.699786  2.190841  2.776984  3.401734 |  100.0 |    1.47548 | 416.334519 |
    | 2000-01-03 04:30:00 | 101.232151 | 101.232151 | 104.523813  112.556545  128.757703  156.791276  199.325306    256.6521  325.730391  400.147304 | 400.147304 | 0.868162  0.930892  1.058651  1.281734  1.623225  2.087653  2.652643  3.267783 |  100.0 |   1.232151 | 400.147304 |
    | 2000-01-03 05:00:00 | 101.027458 | 101.027458 | 103.863558  110.928049  125.443049  151.005897  190.471741  244.614107  311.088727  384.208561 | 384.208561 | 0.863057  0.918176  1.032524  1.235721  1.552186  1.990197  2.532989  3.136167 |  100.0 |   1.027458 | 384.208561 |
    | 2000-01-03 05:30:00 | 100.855551 | 100.855551 | 103.295128  109.497083  122.476203  145.736004  182.266652  233.260431  297.021834  368.579249 | 368.579249 | 0.858666  0.907012  1.009162  1.193853  1.486431  1.898408  2.418218  3.007364 |  100.0 |   0.855551 | 368.579249 |
    | 2000-01-03 06:00:00 | 100.711411 | 100.711411 | 102.806546  108.241887  119.825747   140.94583  174.680592  222.581168  283.550062  353.313063 | 353.313063 | 0.854895  0.897228  0.988311  1.155837  1.425708  1.812187  2.308474  2.881791 |  100.0 |   0.711411 | 353.313063 |
    | 2000-01-03 06:30:00 | 100.590745 | 100.590745 |  102.38726  107.142744  117.462349  136.600663  167.682988  212.562498  270.687482  338.456583 | 338.456583 | 0.851661  0.888669  0.969735  1.121388   1.36976  1.731404  2.203849  2.759806 |  100.0 |   0.590745 | 338.456583 |
    | 2000-01-03 07:00:00 | 100.489885 | 100.489885 | 102.027996   106.18185  115.358738  132.667043  161.242586  203.187272  258.442402  324.049484 | 324.049484 | 0.848892  0.881192  0.953217  1.090233  1.318324  1.655904   2.10439  2.641714 |  100.0 |   0.489885 | 324.049484 |
    | 2000-01-03 07:30:00 | 100.405708 | 100.405708 | 101.720625  105.343173  113.489665  129.112918  155.327844  194.435565  246.817899  310.124814 | 310.124814 | 0.846526  0.874671  0.938553  1.062112  1.271137   1.58551  2.010102  2.527761 |  100.0 |   0.405708 | 310.124814 |
    | 2000-01-03 08:00:00 |  100.33556 |  100.33556 | 101.458038  104.612322  111.831834  125.907754  149.907271  186.285196  235.812344  296.709316 | 296.709316 | 0.844505  0.868993  0.925558  1.036775  1.227938  1.520029  1.920953  2.418147 |  100.0 |    0.33556 | 296.709316 |
    | 2000-01-03 08:30:00 | 100.277186 | 100.277186 | 101.234032  103.976413  110.363838  123.022613  144.949721  178.712204   225.41993  283.823794 | 283.823794 | 0.842783  0.864057  0.914061  1.013989   1.18847  1.459257  1.836879   2.31302 |  100.0 |   0.277186 | 283.823794 |
    | 2000-01-03 09:00:00 | 100.228681 | 100.228681 | 101.043206  103.423943  109.066063  120.430205  140.424644  171.691292  215.631184  271.483505 | 271.483505 | 0.841317  0.859772  0.903905  0.993534   1.15248  1.402977  1.757787  2.212485 |  100.0 |   0.228681 | 271.483505 |
    | 2000-01-03 09:30:00 | 100.188431 | 100.188431 | 100.880868  102.944665  107.920599    118.1049  136.302288  165.196217  206.433455  259.698574 | 259.698574 |  0.84007  0.856057  0.894948  0.975202  1.119724  1.350968   1.68356  2.116608 |  100.0 |   0.188431 | 259.698574 |
    | 2000-01-03 10:00:00 | 100.155078 | 100.155078 | 100.742949  102.529476  106.911144  116.022735  132.553871   159.20015  197.811391  248.474425 | 248.474425 | 0.839012  0.852841  0.887061  0.958801  1.089968  1.303005   1.61406  2.025415 |  100.0 |   0.155078 | 248.474425 |
    | 2000-01-03 10:30:00 | 100.127477 | 100.127477 | 100.625929  102.170308  106.022897  114.161386  129.151711  153.675992  189.747374  237.812208 | 237.812208 | 0.838115  0.850061  0.880127  0.944152  1.062986  1.258861  1.549131  1.938898 |  100.0 |   0.127477 | 237.812208 |
    | 2000-01-03 11:00:00 | 100.104666 | 100.104666 | 100.526768  101.860024  105.242464  112.500132   126.06933   148.59665  182.221945  227.709237 | 227.709237 | 0.837355  0.847661  0.874038  0.931088  1.038561  1.218312  1.488606  1.857021 |  100.0 |   0.104666 | 227.709237 |
    | 2000-01-03 11:30:00 | 100.085838 | 100.085838 | 100.442844  101.592329  104.557749   111.01981  123.281527  143.935279  175.214181  218.159416 | 218.159416 | 0.836712  0.845592  0.868701  0.919456   1.01649  1.181135  1.432304   1.77972 |  100.0 |   0.085838 | 218.159416 |
    | 2000-01-03 12:00:00 | 100.070317 | 100.070317 | 100.371904  101.361675  103.957861  109.702751  120.764422  139.665484  168.702051  209.153648 | 209.153648 |  0.83617  0.843811  0.864027  0.909115  0.996579  1.147113  1.380038  1.706907 |  100.0 |   0.070317 | 209.153648 |
    | 2000-01-03 12:30:00 | 100.057539 | 100.057539 |  100.31201   101.16319  103.433017  108.532718   118.49549  135.761495  162.662737  200.680244 | 200.680244 | 0.835712  0.842279  0.859942  0.899936  0.978646  1.116034  1.331616  1.638475 |  100.0 |   0.057539 | 200.680244 |
    | 2000-01-03 13:00:00 | 100.047032 | 100.047032 |   100.2615  100.992598  102.974448  107.494827  116.453562  132.198309  157.072922  192.725295 | 192.725295 | 0.835326  0.840963  0.856374  0.891799   0.96252  1.087694  1.286841  1.574301 |  100.0 |   0.047032 | 192.725295 |
    | 2000-01-03 13:30:00 | 100.038402 | 100.038402 | 100.218953  100.846158  102.574313  106.575481  114.618822  128.951804  151.909051  185.273038 | 185.273038 | 0.835001  0.839834  0.853264  0.884597  0.948042  1.061894  1.245518  1.514246 |  100.0 |   0.038402 | 185.273038 |
    | 2000-01-03 14:00:00 | 100.031322 | 100.031322 | 100.183153  100.720599  102.225615  105.762285  112.972785   125.99883  147.147551  178.306192 | 178.306192 | 0.834727  0.838866  0.850555  0.878231  0.935063  1.038447  1.207451   1.45816 |  100.0 |   0.031322 | 178.306192 |
    | 2000-01-03 14:30:00 | 100.025521 | 100.025521 | 100.153063  100.613069  101.922126  105.043972  111.498262  123.317272  142.765038  171.806267 | 171.806267 | 0.834498  0.838038  0.848198  0.872612  0.923446  1.017173  1.172445  1.405886 |  100.0 |   0.025521 | 171.806267 |
    | 2000-01-03 15:00:00 | 100.020773 | 100.020773 | 100.127799  100.521082  101.658309  104.410323  110.179322  120.886101  138.738481  165.753858 | 165.753858 | 0.834305   0.83733  0.846151  0.867658  0.913061  0.997901   1.14031  1.357258 |  100.0 |   0.020773 | 165.753858 |
    | 2000-01-03 15:30:00 | 100.016892 | 100.016892 | 100.106609   100.44248  101.429254  103.852095  109.001243  118.685399  135.045352  160.128902 | 160.128902 | 0.834144  0.836726  0.844375  0.863297  0.903793   0.98047  1.110863  1.312108 |  100.0 |   0.016892 | 160.128902 |
    | 2000-01-03 16:00:00 | 100.013722 | 100.013722 | 100.088855  100.375386  101.230615  103.360943  107.950459  116.696376  131.663739   154.91092 |  154.91092 | 0.834008   0.83621  0.842836  0.859463  0.895532  0.964727  1.083921  1.270263 |  100.0 |   0.013722 |  154.91092 |
    | 2000-01-03 16:30:00 | 100.011136 | 100.011136 | 100.073993  100.318178  101.058552  102.929351  107.014503  114.901368  128.572454  150.079223 | 150.079223 | 0.833895   0.83577  0.841503  0.856095  0.888179  0.950531  1.059313  1.231551 |  100.0 |   0.011136 | 150.079223 |
    | 2000-01-03 17:00:00 | 100.009028 | 100.009028 | 100.061566  100.269449  100.909677  102.550564  106.181946  113.283825  125.751104  145.613106 | 145.613106 | 0.833801  0.835396  0.840351  0.853141  0.881642  0.937747  1.036872  1.195799 |  100.0 |   0.009028 | 145.613106 |
    | 2000-01-03 17:30:00 | 100.007313 | 100.007313 | 100.051183  100.227984  100.781009  102.218521  105.442339  111.828293  123.180153  141.492009 | 141.492009 | 0.833722  0.835078  0.839355  0.850554  0.875839  0.926252  1.016439  1.162838 |  100.0 |   0.007313 | 141.492009 |
    | 2000-01-03 18:00:00 | 100.005918 | 100.005918 | 100.042517  100.192736  100.669923  101.927798  104.786151  110.520382  120.840966  137.695662 | 137.695662 | 0.833656  0.834808  0.838496   0.84829  0.870694  0.915931  0.997862  1.132499 |  100.0 |   0.005918 | 137.695662 |
    | 2000-01-03 18:30:00 | 100.004785 | 100.004785 |  100.03529  100.162801   100.57412  101.673547  104.204706  109.346735  118.715838  134.204204 | 134.204204 | 0.833601  0.834578  0.837756  0.846311  0.866138  0.906675  0.980997   1.10462 |  100.0 |   0.004785 | 134.204204 |
    | 2000-01-03 19:00:00 | 100.003865 | 100.003865 | 100.029269  100.137402  100.491582  101.451444  103.690126  108.294982  116.788006  130.998289 | 130.998289 | 0.833555  0.834383  0.837118  0.844583  0.862108  0.898386   0.96571  1.079042 |  100.0 |   0.003865 | 130.998289 |
    | 2000-01-03 19:30:00 | 100.003119 | 100.003119 | 100.024257  100.115874  100.420545  101.257639  103.235272  107.353705   115.04166  128.059168 | 128.059168 | 0.833517  0.834219   0.83657  0.843076  0.858549  0.890973  0.951872   1.05561 |  100.0 |   0.003119 | 128.059168 |
    | 2000-01-03 20:00:00 | 100.002515 | 100.002515 | 100.020088  100.097641  100.359466   101.08871  102.833688  106.512384  113.461932  125.368758 | 125.368758 | 0.833485  0.834079  0.836099  0.841764  0.855408  0.884352  0.939363  1.034178 |  100.0 |   0.002515 | 125.368758 |
    | 2000-01-03 20:30:00 | 100.002026 | 100.002026 | 100.016624  100.082214  100.307002  100.941622  102.479547  105.761354  112.034886  122.909694 | 122.909694 | 0.833459  0.833961  0.835694  0.840621  0.852639  0.878444   0.92807  1.014603 |  100.0 |   0.002026 | 122.909694 |
    | 2000-01-03 21:00:00 | 100.001631 | 100.001631 | 100.013747  100.069172   100.26198  100.813684  102.167599  105.091753  110.747498  120.665371 | 120.665371 | 0.833437  0.833861  0.835347  0.839628  0.850202  0.873181   0.91789  0.996751 |  100.0 |   0.001631 | 120.665371 |
    | 2000-01-03 21:30:00 | 100.001312 | 100.001312 |  100.01136  100.058155   100.22338  100.702516  101.893123  104.495476  109.587629  118.619966 | 118.619966 | 0.833419  0.833777   0.83505  0.838766  0.848059  0.868497  0.908724  0.980493 |  100.0 |   0.001312 | 118.619966 |
    | 2000-01-03 22:00:00 | 100.001054 | 100.001054 | 100.009382  100.048858  100.190317  100.606017  101.651881  103.965125  108.543995  116.758462 | 116.758462 | 0.833404  0.833706  0.834795  0.838018  0.846177  0.864333  0.900482  0.965707 |  100.0 |   0.001054 | 116.758462 |
    | 2000-01-03 22:30:00 | 100.000846 | 100.000846 | 100.007742  100.041018  100.162022  100.522334  101.440077  103.493959  107.606135  115.066647 | 115.066647 | 0.833392  0.833646  0.834577  0.837369  0.844525  0.860636   0.89308  0.952278 |  100.0 |   0.000846 | 115.066647 |
    | 2000-01-03 23:00:00 | 100.000679 | 100.000679 | 100.006385  100.034412  100.137829  100.449835  101.254314  103.075853  106.764372   113.53112 |  113.53112 | 0.833382  0.833596  0.834391  0.836807  0.843077  0.857357  0.886441  0.940098 |  100.0 |   0.000679 |  113.53112 |
    | 2000-01-03 23:30:00 | 100.000544 | 100.000544 | 100.005263   100.02885   100.11716  100.387084  101.091558  102.705245  106.009777  112.139277 | 112.139277 | 0.833373  0.833553  0.834232  0.836322  0.841809  0.854452  0.880493  0.929065 |  100.0 |   0.000544 | 112.139277 |
    | 2000-01-04 00:00:00 | 100.000436 | 100.000436 | 100.004335  100.024171  100.099518  100.332821  100.949104  102.377102   105.33413  110.879301 | 110.879301 | 0.833366  0.833517  0.834096  0.835902  0.840699  0.851882   0.87517  0.919083 |  100.0 |   0.000436 | 110.879301 |
    | 2000-01-04 00:30:00 | 100.000349 | 100.000349 | 100.003569  100.020237  100.084471   100.28594  100.824543   102.08687  104.729876  109.740141 | 109.740141 |  0.83336  0.833487  0.833981  0.835539   0.83973   0.84961  0.870413  0.910065 |  100.0 |   0.000349 | 109.740141 |
    | 2000-01-04 01:00:00 | 100.000279 | 100.000279 | 100.002936  100.016933  100.071649  100.245473  100.715734  101.830441  104.190092   108.71149 |  108.71149 | 0.833355  0.833462  0.833882  0.835226  0.838883  0.847603  0.866165  0.901926 |  100.0 |   0.000279 |  108.71149 |
    | 2000-01-04 01:30:00 | 100.000223 | 100.000223 | 100.002414  100.014159  100.060731  100.210574  100.620775  101.604113  103.708441  107.783758 | 107.783758 | 0.833352  0.833441  0.833799  0.834956  0.838145  0.845833  0.862377  0.894591 |  100.0 |   0.000223 | 107.783758 |
    | 2000-01-04 02:00:00 | 100.000178 | 100.000178 | 100.001984  100.011832  100.051441  100.180502  100.537981  101.404557  103.279136  106.948041 | 106.948041 | 0.833348  0.833423  0.833727  0.834724  0.837502  0.844273  0.859003  0.887987 |  100.0 |   0.000178 | 106.948041 |
    | 2000-01-04 02:30:00 | 100.000142 | 100.000142 | 100.001629  100.009882  100.043544  100.154612  100.465861  101.228781  102.896904  106.196095 | 106.196095 | 0.833346  0.833408  0.833667  0.834524  0.836941    0.8429     0.856  0.882048 |  100.0 |   0.000142 | 106.196095 |
    | 2000-01-04 03:00:00 | 100.000113 | 100.000113 | 100.001337  100.008248  100.036834  100.132339  100.403095  101.074105  102.556944    105.5203 |   105.5203 | 0.833343  0.833396  0.833615  0.834352  0.836454  0.841692  0.853331  0.876714 |  100.0 |   0.000113 |   105.5203 |
    | 2000-01-04 03:30:00 |  100.00009 |  100.00009 | 100.001097   100.00688  100.031138  100.113195  100.348518  100.938126  102.254898  104.913629 | 104.913629 | 0.833342  0.833386  0.833572  0.834204   0.83603  0.840631  0.850961  0.871929 |  100.0 |    0.00009 | 104.913629 |
    | 2000-01-04 04:00:00 | 100.000072 | 100.000072 |   100.0009  100.005736  100.026306  100.096753  100.301103  100.818698  101.986812  104.369616 | 104.369616 |  0.83334  0.833377  0.833535  0.834078  0.835663  0.839699  0.848859   0.86764 |  100.0 |   0.000072 | 104.369616 |
    | 2000-01-04 04:30:00 | 100.000057 | 100.000057 | 100.000737  100.004779  100.022211  100.082643  100.259946  100.713903  101.749108  103.882319 | 103.882319 | 0.833339   0.83337  0.833503  0.833969  0.835343  0.838882  0.846995    0.8638 |  100.0 |   0.000057 | 103.882319 |
    | 2000-01-04 05:00:00 | 100.000046 | 100.000046 | 100.000604   100.00398  100.018741  100.070543  100.224251  100.622034  101.538552  103.446292 | 103.446292 | 0.833338  0.833364  0.833477  0.833876  0.835067  0.838166  0.845345  0.860366 |  100.0 |   0.000046 | 103.446292 |
    | 2000-01-04 05:30:00 | 100.000036 | 100.000036 | 100.000494  100.003313  100.015804  100.060175  100.193318  100.541568  101.352224   103.05655 |  103.05655 | 0.833337  0.833358  0.833454  0.833796  0.834827  0.837539  0.843886  0.857298 |  100.0 |   0.000036 |  103.05655 |
    | 2000-01-04 06:00:00 | 100.000029 | 100.000029 | 100.000405  100.002756  100.013319  100.051297  100.166535  100.471152  101.187494  102.708538 | 102.708538 | 0.833336  0.833354  0.833435  0.833727   0.83462  0.836991  0.842597   0.85456 |  100.0 |   0.000029 | 102.708538 |
    | 2000-01-04 06:30:00 | 100.000023 | 100.000023 | 100.000331  100.002291  100.011218  100.043701  100.143362  100.409584  101.041995  102.398105 | 102.398105 | 0.833336  0.833351  0.833419  0.833669   0.83444  0.836512  0.841459  0.852119 |  100.0 |   0.000023 | 102.398105 |
    | 2000-01-04 07:00:00 | 100.000018 | 100.000018 | 100.000271  100.001904  100.009444  100.037206  100.123329  100.355799  100.913601  102.121467 | 102.121467 | 0.833335  0.833348  0.833405  0.833619  0.834285  0.836094  0.840455  0.849945 |  100.0 |   0.000018 | 102.121467 |
    | 2000-01-04 07:30:00 | 100.000014 | 100.000014 | 100.000221  100.001582  100.007945  100.031657  100.106024  100.308851  100.800405  101.875187 | 101.875187 | 0.833335  0.833345  0.833394  0.833576  0.834151  0.835728   0.83957   0.84801 |  100.0 |   0.000014 | 101.875187 |
    | 2000-01-04 08:00:00 | 100.000011 | 100.000011 | 100.000181  100.001313  100.006681   100.02692  100.091086  100.267907  100.700695  101.656145 | 101.656145 | 0.833335  0.833343  0.833384   0.83354  0.834036   0.83541  0.838791  0.846291 |  100.0 |   0.000011 | 101.656145 |
    | 2000-01-04 08:30:00 | 100.000009 | 100.000009 | 100.000146  100.001083  100.005585   100.02277  100.077885  100.231407  100.611071  101.457669 | 101.457669 | 0.833334  0.833342  0.833376  0.833509  0.833937  0.835134  0.838107  0.844767 |  100.0 |   0.000009 | 101.457669 |
    | 2000-01-04 09:00:00 | 100.000007 | 100.000007 |  100.00012  100.000899  100.004695  100.019354  100.066865  100.200542  100.534339  101.285696 | 101.285696 | 0.833334   0.83334  0.833369  0.833483  0.833851  0.834893  0.837506  0.843414 |  100.0 |   0.000007 | 101.285696 |
    | 2000-01-04 09:30:00 | 100.000006 | 100.000006 | 100.000098  100.000746  100.003946  100.016441  100.057369  100.173677  100.466907  101.133144 | 101.133144 | 0.833334  0.833339  0.833364   0.83346  0.833778  0.834684  0.836979  0.842215 |  100.0 |   0.000006 | 101.133144 |
    | 2000-01-04 10:00:00 | 100.000004 | 100.000004 |  100.00008  100.000619  100.003314  100.013959  100.049192  100.150313  100.407697  100.997938 | 100.997938 | 0.833334  0.833338  0.833359  0.833441  0.833714  0.834502  0.836515  0.841153 |  100.0 |   0.000004 | 100.997938 |
    | 2000-01-04 10:30:00 | 100.000004 | 100.000004 | 100.000065  100.000514  100.002782  100.011845  100.042155  100.130008  100.355747  100.878211 | 100.878211 | 0.833334  0.833337  0.833355  0.833425   0.83366  0.834344  0.836109  0.840212 |  100.0 |   0.000004 | 100.878211 |
    | 2000-01-04 11:00:00 | 100.000003 | 100.000003 | 100.000053  100.000426  100.002335  100.010046  100.036104  100.112374  100.310204  100.772281 | 100.772281 | 0.833334  0.833337  0.833351  0.833411  0.833613  0.834206  0.835753   0.83938 |  100.0 |   0.000003 | 100.772281 |
    | 2000-01-04 11:30:00 | 100.000002 | 100.000002 | 100.000044  100.000353  100.001958  100.008516  100.030903  100.097071  100.270309  100.678637 | 100.678637 | 0.833334  0.833336  0.833348  0.833399  0.833572  0.834087  0.835441  0.838645 |  100.0 |   0.000002 | 100.678637 |
    | 2000-01-04 12:00:00 | 100.000002 | 100.000002 | 100.000035  100.000292  100.001642  100.007215  100.026437    100.0838  100.235387  100.595922 | 100.595922 | 0.833334  0.833336  0.833346  0.833389  0.833538  0.833984  0.835168  0.837996 |  100.0 |   0.000002 | 100.595922 |
    | 2000-01-04 12:30:00 | 100.000001 | 100.000001 | 100.000029  100.000242  100.001376   100.00611  100.022603  100.072299  100.204842  100.522921 | 100.522921 | 0.833334  0.833335  0.833344   0.83338  0.833508  0.833895   0.83493  0.837424 |  100.0 |   0.000001 | 100.522921 |
    | 2000-01-04 13:00:00 | 100.000001 | 100.000001 | 100.000024    100.0002  100.001152  100.005171  100.019315   100.06234  100.178146  100.458544 | 100.458544 | 0.833334  0.833335  0.833342  0.833373  0.833483  0.833817  0.834721  0.836919 |  100.0 |   0.000001 | 100.458544 |
    | 2000-01-04 13:30:00 | 100.000001 | 100.000001 | 100.000019  100.000166  100.000965  100.004375  100.016496   100.05372  100.154829  100.401818 | 100.401818 | 0.833333  0.833335  0.833341  0.833367  0.833461   0.83375  0.834539  0.836475 |  100.0 |   0.000001 | 100.401818 |
    | 2000-01-04 14:00:00 | 100.000001 | 100.000001 | 100.000016  100.000137  100.000807  100.003699  100.014081  100.046265   100.13448  100.351871 | 100.351871 | 0.833333  0.833334   0.83334  0.833362  0.833442  0.833692   0.83438  0.836083 |  100.0 |   0.000001 | 100.351871 |
    | 2000-01-04 14:30:00 | 100.000001 | 100.000001 | 100.000013  100.000113  100.000675  100.003126  100.012013  100.039822  100.116733  100.307928 | 100.307928 | 0.833333  0.833334  0.833339  0.833357  0.833426  0.833642  0.834242  0.835739 |  100.0 |   0.000001 | 100.307928 |
    | 2000-01-04 15:00:00 |      100.0 |      100.0 |  100.00001  100.000094  100.000565  100.002641  100.010244  100.034257  100.101266  100.269296 | 100.269296 | 0.833333  0.833334  0.833338  0.833354  0.833412  0.833599  0.834121  0.835437 |  100.0 |        0.0 | 100.269296 |
    | 2000-01-04 15:30:00 |      100.0 |      100.0 | 100.000008  100.000077  100.000472   100.00223  100.008731  100.029453  100.087795  100.235358 | 100.235358 | 0.833333  0.833334  0.833337   0.83335  0.833401  0.833562  0.834016  0.835171 |  100.0 |        0.0 | 100.235358 |
    | 2000-01-04 16:00:00 |      100.0 |      100.0 | 100.000007  100.000064  100.000394  100.001882  100.007437  100.025309   100.07607  100.205566 | 100.205566 | 0.833333  0.833334  0.833336  0.833348  0.833391  0.833529  0.833925  0.834938 |  100.0 |        0.0 | 100.205566 |
    | 2000-01-04 16:30:00 |      100.0 |      100.0 | 100.000006  100.000053  100.000329  100.001588  100.006332  100.021736  100.065873  100.179431 | 100.179431 | 0.833333  0.833334  0.833336  0.833346  0.833382  0.833502  0.833846  0.834734 |  100.0 |        0.0 | 100.179431 |
    | 2000-01-04 17:00:00 |      100.0 |      100.0 | 100.000005  100.000044  100.000275  100.001339  100.005389  100.018657   100.05701  100.156522 | 100.156522 | 0.833333  0.833334  0.833335  0.833344  0.833375  0.833478  0.833777  0.834554 |  100.0 |        0.0 | 100.156522 |
    | 2000-01-04 17:30:00 |      100.0 |      100.0 | 100.000004  100.000036   100.00023  100.001128  100.004583  100.016006  100.049311  100.136454 | 100.136454 | 0.833333  0.833334  0.833335  0.833342  0.833369  0.833457  0.833717  0.834398 |  100.0 |        0.0 | 100.136454 |
    | 2000-01-04 18:00:00 |      100.0 |      100.0 | 100.000003   100.00003  100.000191  100.000951  100.003897  100.013725  100.042627  100.118887 | 100.118887 | 0.833333  0.833334  0.833335  0.833341  0.833363   0.83344  0.833665   0.83426 |  100.0 |        0.0 | 100.118887 |
    | 2000-01-04 18:30:00 |      100.0 |      100.0 | 100.000002  100.000024   100.00016  100.000801  100.003311  100.011762  100.036829   100.10352 |  100.10352 | 0.833333  0.833334  0.833335  0.833339  0.833359  0.833424  0.833619   0.83414 |  100.0 |        0.0 |  100.10352 |
    | 2000-01-04 19:00:00 |      100.0 |      100.0 | 100.000002   100.00002  100.000133  100.000674  100.002813  100.010076  100.031803  100.090086 | 100.090086 | 0.833333  0.833333  0.833334  0.833339  0.833355  0.833411   0.83358  0.834035 |  100.0 |        0.0 | 100.090086 |
    | 2000-01-04 19:30:00 |      100.0 |      100.0 | 100.000002  100.000017  100.000111  100.000567  100.002388  100.008626  100.027447  100.078349 | 100.078349 | 0.833333  0.833333  0.833334  0.833338  0.833352    0.8334  0.833546  0.833944 |  100.0 |        0.0 | 100.078349 |
    | 2000-01-04 20:00:00 |      100.0 |      100.0 | 100.000001  100.000014  100.000092  100.000477  100.002027  100.007382  100.023675  100.068103 | 100.068103 | 0.833333  0.833333  0.833334  0.833337  0.833349   0.83339  0.833517  0.833864 |  100.0 |        0.0 | 100.068103 |
    | 2000-01-04 20:30:00 |      100.0 |      100.0 | 100.000001  100.000011  100.000077  100.000401  100.001719  100.006314  100.020411  100.059163 | 100.059163 | 0.833333  0.833333  0.833334  0.833336  0.833347  0.833382  0.833492  0.833794 |  100.0 |        0.0 | 100.059163 |
    | 2000-01-04 21:00:00 |      100.0 |      100.0 | 100.000001  100.000009  100.000064  100.000337  100.001458  100.005398  100.017588  100.051368 | 100.051368 | 0.833333  0.833333  0.833334  0.833336  0.833345  0.833375   0.83347  0.833733 |  100.0 |        0.0 | 100.051368 |
    | 2000-01-04 21:30:00 |      100.0 |      100.0 | 100.000001  100.000008  100.000053  100.000283  100.001235  100.004613  100.015147  100.044575 | 100.044575 | 0.833333  0.833333  0.833334  0.833336  0.833343  0.833369  0.833451   0.83368 |  100.0 |        0.0 | 100.044575 |
    | 2000-01-04 22:00:00 |      100.0 |      100.0 | 100.000001  100.000006  100.000044  100.000238  100.001047   100.00394  100.013039  100.038659 | 100.038659 | 0.833333  0.833333  0.833334  0.833335  0.833341  0.833364  0.833434  0.833634 |  100.0 |        0.0 | 100.038659 |
    | 2000-01-04 22:30:00 |      100.0 |      100.0 |      100.0  100.000005  100.000037    100.0002  100.000886  100.003363  100.011218  100.033511 | 100.033511 | 0.833333  0.833333  0.833334  0.833335   0.83334  0.833359   0.83342  0.833594 |  100.0 |        0.0 | 100.033511 |
    | 2000-01-04 23:00:00 |      100.0 |      100.0 |      100.0  100.000004  100.000031  100.000168   100.00075   100.00287  100.009647  100.029032 | 100.029032 | 0.833333  0.833333  0.833334  0.833335  0.833339  0.833356  0.833408  0.833559 |  100.0 |        0.0 | 100.029032 |
    | 2000-01-04 23:30:00 |      100.0 |      100.0 |      100.0  100.000003  100.000025  100.000141  100.000635  100.002448  100.008292  100.025139 | 100.025139 | 0.833333  0.833333  0.833334  0.833334  0.833338  0.833352  0.833398  0.833529 |  100.0 |        0.0 | 100.025139 |

.. _kinw_williams_ext_directly_forwarded_runoff:

directly forwarded runoff
_________________________

If you need to forward the runoff directly (as if the channel length would be zero),
set the number of subchannels to zero:

.. integration-test::

    >>> laen(0.0)
    >>> gts(0)
    >>> test("kinw_williams_ext_directly_forwarded_runoff")
    |                date |         qz |        qza | qg |         qa | vg | input1 |     input2 |     output |
    -----------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 100.000003 | 100.000003 |  - | 100.000003 |  - |  100.0 |   0.000003 | 100.000003 |
    | 2000-01-01 03:30:00 | 100.000029 | 100.000029 |  - | 100.000029 |  - |  100.0 |   0.000029 | 100.000029 |
    | 2000-01-01 04:00:00 | 100.000175 | 100.000175 |  - | 100.000175 |  - |  100.0 |   0.000175 | 100.000175 |
    | 2000-01-01 04:30:00 | 100.000826 | 100.000826 |  - | 100.000826 |  - |  100.0 |   0.000826 | 100.000826 |
    | 2000-01-01 05:00:00 | 100.003194 | 100.003194 |  - | 100.003194 |  - |  100.0 |   0.003194 | 100.003194 |
    | 2000-01-01 05:30:00 | 100.010515 | 100.010515 |  - | 100.010515 |  - |  100.0 |   0.010515 | 100.010515 |
    | 2000-01-01 06:00:00 | 100.030315 | 100.030315 |  - | 100.030315 |  - |  100.0 |   0.030315 | 100.030315 |
    | 2000-01-01 06:30:00 | 100.078179 | 100.078179 |  - | 100.078179 |  - |  100.0 |   0.078179 | 100.078179 |
    | 2000-01-01 07:00:00 | 100.183351 | 100.183351 |  - | 100.183351 |  - |  100.0 |   0.183351 | 100.183351 |
    | 2000-01-01 07:30:00 | 100.396214 | 100.396214 |  - | 100.396214 |  - |  100.0 |   0.396214 | 100.396214 |
    | 2000-01-01 08:00:00 | 100.797305 | 100.797305 |  - | 100.797305 |  - |  100.0 |   0.797305 | 100.797305 |
    | 2000-01-01 08:30:00 | 101.507032 | 101.507032 |  - | 101.507032 |  - |  100.0 |   1.507032 | 101.507032 |
    | 2000-01-01 09:00:00 | 102.694853 | 102.694853 |  - | 102.694853 |  - |  100.0 |   2.694853 | 102.694853 |
    | 2000-01-01 09:30:00 | 104.586356 | 104.586356 |  - | 104.586356 |  - |  100.0 |   4.586356 | 104.586356 |
    | 2000-01-01 10:00:00 | 107.466641 | 107.466641 |  - | 107.466641 |  - |  100.0 |   7.466641 | 107.466641 |
    | 2000-01-01 10:30:00 | 111.678559 | 111.678559 |  - | 111.678559 |  - |  100.0 |  11.678559 | 111.678559 |
    | 2000-01-01 11:00:00 | 117.614836 | 117.614836 |  - | 117.614836 |  - |  100.0 |  17.614836 | 117.614836 |
    | 2000-01-01 11:30:00 | 125.703736 | 125.703736 |  - | 125.703736 |  - |  100.0 |  25.703736 | 125.703736 |
    | 2000-01-01 12:00:00 | 136.388647 | 136.388647 |  - | 136.388647 |  - |  100.0 |  36.388647 | 136.388647 |
    | 2000-01-01 12:30:00 | 150.102726 | 150.102726 |  - | 150.102726 |  - |  100.0 |  50.102726 | 150.102726 |
    | 2000-01-01 13:00:00 | 167.240349 | 167.240349 |  - | 167.240349 |  - |  100.0 |  67.240349 | 167.240349 |
    | 2000-01-01 13:30:00 | 188.127569 | 188.127569 |  - | 188.127569 |  - |  100.0 |  88.127569 | 188.127569 |
    | 2000-01-01 14:00:00 | 212.993962 | 212.993962 |  - | 212.993962 |  - |  100.0 | 112.993962 | 212.993962 |
    | 2000-01-01 14:30:00 | 241.948195 | 241.948195 |  - | 241.948195 |  - |  100.0 | 141.948195 | 241.948195 |
    | 2000-01-01 15:00:00 |  274.95935 |  274.95935 |  - |  274.95935 |  - |  100.0 |  174.95935 |  274.95935 |
    | 2000-01-01 15:30:00 | 311.845554 | 311.845554 |  - | 311.845554 |  - |  100.0 | 211.845554 | 311.845554 |
    | 2000-01-01 16:00:00 | 352.270855 | 352.270855 |  - | 352.270855 |  - |  100.0 | 252.270855 | 352.270855 |
    | 2000-01-01 16:30:00 | 395.750601 | 395.750601 |  - | 395.750601 |  - |  100.0 | 295.750601 | 395.750601 |
    | 2000-01-01 17:00:00 | 441.664962 | 441.664962 |  - | 441.664962 |  - |  100.0 | 341.664962 | 441.664962 |
    | 2000-01-01 17:30:00 | 489.279607 | 489.279607 |  - | 489.279607 |  - |  100.0 | 389.279607 | 489.279607 |
    | 2000-01-01 18:00:00 |  537.77215 |  537.77215 |  - |  537.77215 |  - |  100.0 |  437.77215 |  537.77215 |
    | 2000-01-01 18:30:00 | 586.262607 | 586.262607 |  - | 586.262607 |  - |  100.0 | 486.262607 | 586.262607 |
    | 2000-01-01 19:00:00 | 633.846018 | 633.846018 |  - | 633.846018 |  - |  100.0 | 533.846018 | 633.846018 |
    | 2000-01-01 19:30:00 | 679.625318 | 679.625318 |  - | 679.625318 |  - |  100.0 | 579.625318 | 679.625318 |
    | 2000-01-01 20:00:00 | 722.742733 | 722.742733 |  - | 722.742733 |  - |  100.0 | 622.742733 | 722.742733 |
    | 2000-01-01 20:30:00 | 762.408181 | 762.408181 |  - | 762.408181 |  - |  100.0 | 662.408181 | 762.408181 |
    | 2000-01-01 21:00:00 | 797.923464 | 797.923464 |  - | 797.923464 |  - |  100.0 | 697.923464 | 797.923464 |
    | 2000-01-01 21:30:00 | 828.701418 | 828.701418 |  - | 828.701418 |  - |  100.0 | 728.701418 | 828.701418 |
    | 2000-01-01 22:00:00 |  854.27953 |  854.27953 |  - |  854.27953 |  - |  100.0 |  754.27953 |  854.27953 |
    | 2000-01-01 22:30:00 | 874.327871 | 874.327871 |  - | 874.327871 |  - |  100.0 | 774.327871 | 874.327871 |
    | 2000-01-01 23:00:00 | 888.651531 | 888.651531 |  - | 888.651531 |  - |  100.0 | 788.651531 | 888.651531 |
    | 2000-01-01 23:30:00 | 897.187983 | 897.187983 |  - | 897.187983 |  - |  100.0 | 797.187983 | 897.187983 |
    | 2000-01-02 00:00:00 |      900.0 |      900.0 |  - |      900.0 |  - |  100.0 |      800.0 |      900.0 |
    | 2000-01-02 00:30:00 | 897.264896 | 897.264896 |  - | 897.264896 |  - |  100.0 | 797.264896 | 897.264896 |
    | 2000-01-02 01:00:00 | 889.260928 | 889.260928 |  - | 889.260928 |  - |  100.0 | 789.260928 | 889.260928 |
    | 2000-01-02 01:30:00 | 876.351729 | 876.351729 |  - | 876.351729 |  - |  100.0 | 776.351729 | 876.351729 |
    | 2000-01-02 02:00:00 | 858.969595 | 858.969595 |  - | 858.969595 |  - |  100.0 | 758.969595 | 858.969595 |
    | 2000-01-02 02:30:00 |   837.5984 |   837.5984 |  - |   837.5984 |  - |  100.0 |   737.5984 |   837.5984 |
    | 2000-01-02 03:00:00 | 812.756821 | 812.756821 |  - | 812.756821 |  - |  100.0 | 712.756821 | 812.756821 |
    | 2000-01-02 03:30:00 | 784.982427 | 784.982427 |  - | 784.982427 |  - |  100.0 | 684.982427 | 784.982427 |
    | 2000-01-02 04:00:00 | 754.817086 | 754.817086 |  - | 754.817086 |  - |  100.0 | 654.817086 | 754.817086 |
    | 2000-01-02 04:30:00 |  722.79402 |  722.79402 |  - |  722.79402 |  - |  100.0 |  622.79402 |  722.79402 |
    | 2000-01-02 05:00:00 | 689.426711 | 689.426711 |  - | 689.426711 |  - |  100.0 | 589.426711 | 689.426711 |
    | 2000-01-02 05:30:00 | 655.199767 | 655.199767 |  - | 655.199767 |  - |  100.0 | 555.199767 | 655.199767 |
    | 2000-01-02 06:00:00 | 620.561767 | 620.561767 |  - | 620.561767 |  - |  100.0 | 520.561767 | 620.561767 |
    | 2000-01-02 06:30:00 | 585.920022 | 585.920022 |  - | 585.920022 |  - |  100.0 | 485.920022 | 585.920022 |
    | 2000-01-02 07:00:00 |  551.63713 |  551.63713 |  - |  551.63713 |  - |  100.0 |  451.63713 |  551.63713 |
    | 2000-01-02 07:30:00 | 518.029154 | 518.029154 |  - | 518.029154 |  - |  100.0 | 418.029154 | 518.029154 |
    | 2000-01-02 08:00:00 | 485.365238 | 485.365238 |  - | 485.365238 |  - |  100.0 | 385.365238 | 485.365238 |
    | 2000-01-02 08:30:00 |  453.86844 |  453.86844 |  - |  453.86844 |  - |  100.0 |  353.86844 |  453.86844 |
    | 2000-01-02 09:00:00 | 423.717566 | 423.717566 |  - | 423.717566 |  - |  100.0 | 323.717566 | 423.717566 |
    | 2000-01-02 09:30:00 | 395.049792 | 395.049792 |  - | 395.049792 |  - |  100.0 | 295.049792 | 395.049792 |
    | 2000-01-02 10:00:00 | 367.963873 | 367.963873 |  - | 367.963873 |  - |  100.0 | 267.963873 | 367.963873 |
    | 2000-01-02 10:30:00 | 342.523746 | 342.523746 |  - | 342.523746 |  - |  100.0 | 242.523746 | 342.523746 |
    | 2000-01-02 11:00:00 | 318.762374 | 318.762374 |  - | 318.762374 |  - |  100.0 | 218.762374 | 318.762374 |
    | 2000-01-02 11:30:00 |  296.68568 |  296.68568 |  - |  296.68568 |  - |  100.0 |  196.68568 |  296.68568 |
    | 2000-01-02 12:00:00 | 276.276442 | 276.276442 |  - | 276.276442 |  - |  100.0 | 176.276442 | 276.276442 |
    | 2000-01-02 12:30:00 | 257.498082 | 257.498082 |  - | 257.498082 |  - |  100.0 | 157.498082 | 257.498082 |
    | 2000-01-02 13:00:00 | 240.298231 | 240.298231 |  - | 240.298231 |  - |  100.0 | 140.298231 | 240.298231 |
    | 2000-01-02 13:30:00 | 224.612057 | 224.612057 |  - | 224.612057 |  - |  100.0 | 124.612057 | 224.612057 |
    | 2000-01-02 14:00:00 | 210.365288 | 210.365288 |  - | 210.365288 |  - |  100.0 | 110.365288 | 210.365288 |
    | 2000-01-02 14:30:00 |  197.47692 |  197.47692 |  - |  197.47692 |  - |  100.0 |   97.47692 |  197.47692 |
    | 2000-01-02 15:00:00 | 185.861607 | 185.861607 |  - | 185.861607 |  - |  100.0 |  85.861607 | 185.861607 |
    | 2000-01-02 15:30:00 | 175.431719 | 175.431719 |  - | 175.431719 |  - |  100.0 |  75.431719 | 175.431719 |
    | 2000-01-02 16:00:00 | 166.099094 | 166.099094 |  - | 166.099094 |  - |  100.0 |  66.099094 | 166.099094 |
    | 2000-01-02 16:30:00 | 157.776495 | 157.776495 |  - | 157.776495 |  - |  100.0 |  57.776495 | 157.776495 |
    | 2000-01-02 17:00:00 | 150.378786 | 150.378786 |  - | 150.378786 |  - |  100.0 |  50.378786 | 150.378786 |
    | 2000-01-02 17:30:00 | 143.823868 | 143.823868 |  - | 143.823868 |  - |  100.0 |  43.823868 | 143.823868 |
    | 2000-01-02 18:00:00 | 138.033385 | 138.033385 |  - | 138.033385 |  - |  100.0 |  38.033385 | 138.033385 |
    | 2000-01-02 18:30:00 | 132.933235 | 132.933235 |  - | 132.933235 |  - |  100.0 |  32.933235 | 132.933235 |
    | 2000-01-02 19:00:00 | 128.453911 | 128.453911 |  - | 128.453911 |  - |  100.0 |  28.453911 | 128.453911 |
    | 2000-01-02 19:30:00 | 124.530701 | 124.530701 |  - | 124.530701 |  - |  100.0 |  24.530701 | 124.530701 |
    | 2000-01-02 20:00:00 | 121.103761 | 121.103761 |  - | 121.103761 |  - |  100.0 |  21.103761 | 121.103761 |
    | 2000-01-02 20:30:00 | 118.118088 | 118.118088 |  - | 118.118088 |  - |  100.0 |  18.118088 | 118.118088 |
    | 2000-01-02 21:00:00 | 115.523427 | 115.523427 |  - | 115.523427 |  - |  100.0 |  15.523427 | 115.523427 |
    | 2000-01-02 21:30:00 | 113.274094 | 113.274094 |  - | 113.274094 |  - |  100.0 |  13.274094 | 113.274094 |
    | 2000-01-02 22:00:00 | 111.328777 | 111.328777 |  - | 111.328777 |  - |  100.0 |  11.328777 | 111.328777 |
    | 2000-01-02 22:30:00 | 109.650285 | 109.650285 |  - | 109.650285 |  - |  100.0 |   9.650285 | 109.650285 |
    | 2000-01-02 23:00:00 | 108.205287 | 108.205287 |  - | 108.205287 |  - |  100.0 |   8.205287 | 108.205287 |
    | 2000-01-02 23:30:00 | 106.964036 | 106.964036 |  - | 106.964036 |  - |  100.0 |   6.964036 | 106.964036 |
    | 2000-01-03 00:00:00 | 105.900084 | 105.900084 |  - | 105.900084 |  - |  100.0 |   5.900084 | 105.900084 |
    | 2000-01-03 00:30:00 |  104.99001 |  104.99001 |  - |  104.99001 |  - |  100.0 |    4.99001 |  104.99001 |
    | 2000-01-03 01:00:00 | 104.213141 | 104.213141 |  - | 104.213141 |  - |  100.0 |   4.213141 | 104.213141 |
    | 2000-01-03 01:30:00 | 103.551298 | 103.551298 |  - | 103.551298 |  - |  100.0 |   3.551298 | 103.551298 |
    | 2000-01-03 02:00:00 | 102.988541 | 102.988541 |  - | 102.988541 |  - |  100.0 |   2.988541 | 102.988541 |
    | 2000-01-03 02:30:00 |  102.51094 |  102.51094 |  - |  102.51094 |  - |  100.0 |    2.51094 |  102.51094 |
    | 2000-01-03 03:00:00 | 102.106358 | 102.106358 |  - | 102.106358 |  - |  100.0 |   2.106358 | 102.106358 |
    | 2000-01-03 03:30:00 | 101.764251 | 101.764251 |  - | 101.764251 |  - |  100.0 |   1.764251 | 101.764251 |
    | 2000-01-03 04:00:00 |  101.47548 |  101.47548 |  - |  101.47548 |  - |  100.0 |    1.47548 |  101.47548 |
    | 2000-01-03 04:30:00 | 101.232151 | 101.232151 |  - | 101.232151 |  - |  100.0 |   1.232151 | 101.232151 |
    | 2000-01-03 05:00:00 | 101.027458 | 101.027458 |  - | 101.027458 |  - |  100.0 |   1.027458 | 101.027458 |
    | 2000-01-03 05:30:00 | 100.855551 | 100.855551 |  - | 100.855551 |  - |  100.0 |   0.855551 | 100.855551 |
    | 2000-01-03 06:00:00 | 100.711411 | 100.711411 |  - | 100.711411 |  - |  100.0 |   0.711411 | 100.711411 |
    | 2000-01-03 06:30:00 | 100.590745 | 100.590745 |  - | 100.590745 |  - |  100.0 |   0.590745 | 100.590745 |
    | 2000-01-03 07:00:00 | 100.489885 | 100.489885 |  - | 100.489885 |  - |  100.0 |   0.489885 | 100.489885 |
    | 2000-01-03 07:30:00 | 100.405708 | 100.405708 |  - | 100.405708 |  - |  100.0 |   0.405708 | 100.405708 |
    | 2000-01-03 08:00:00 |  100.33556 |  100.33556 |  - |  100.33556 |  - |  100.0 |    0.33556 |  100.33556 |
    | 2000-01-03 08:30:00 | 100.277186 | 100.277186 |  - | 100.277186 |  - |  100.0 |   0.277186 | 100.277186 |
    | 2000-01-03 09:00:00 | 100.228681 | 100.228681 |  - | 100.228681 |  - |  100.0 |   0.228681 | 100.228681 |
    | 2000-01-03 09:30:00 | 100.188431 | 100.188431 |  - | 100.188431 |  - |  100.0 |   0.188431 | 100.188431 |
    | 2000-01-03 10:00:00 | 100.155078 | 100.155078 |  - | 100.155078 |  - |  100.0 |   0.155078 | 100.155078 |
    | 2000-01-03 10:30:00 | 100.127477 | 100.127477 |  - | 100.127477 |  - |  100.0 |   0.127477 | 100.127477 |
    | 2000-01-03 11:00:00 | 100.104666 | 100.104666 |  - | 100.104666 |  - |  100.0 |   0.104666 | 100.104666 |
    | 2000-01-03 11:30:00 | 100.085838 | 100.085838 |  - | 100.085838 |  - |  100.0 |   0.085838 | 100.085838 |
    | 2000-01-03 12:00:00 | 100.070317 | 100.070317 |  - | 100.070317 |  - |  100.0 |   0.070317 | 100.070317 |
    | 2000-01-03 12:30:00 | 100.057539 | 100.057539 |  - | 100.057539 |  - |  100.0 |   0.057539 | 100.057539 |
    | 2000-01-03 13:00:00 | 100.047032 | 100.047032 |  - | 100.047032 |  - |  100.0 |   0.047032 | 100.047032 |
    | 2000-01-03 13:30:00 | 100.038402 | 100.038402 |  - | 100.038402 |  - |  100.0 |   0.038402 | 100.038402 |
    | 2000-01-03 14:00:00 | 100.031322 | 100.031322 |  - | 100.031322 |  - |  100.0 |   0.031322 | 100.031322 |
    | 2000-01-03 14:30:00 | 100.025521 | 100.025521 |  - | 100.025521 |  - |  100.0 |   0.025521 | 100.025521 |
    | 2000-01-03 15:00:00 | 100.020773 | 100.020773 |  - | 100.020773 |  - |  100.0 |   0.020773 | 100.020773 |
    | 2000-01-03 15:30:00 | 100.016892 | 100.016892 |  - | 100.016892 |  - |  100.0 |   0.016892 | 100.016892 |
    | 2000-01-03 16:00:00 | 100.013722 | 100.013722 |  - | 100.013722 |  - |  100.0 |   0.013722 | 100.013722 |
    | 2000-01-03 16:30:00 | 100.011136 | 100.011136 |  - | 100.011136 |  - |  100.0 |   0.011136 | 100.011136 |
    | 2000-01-03 17:00:00 | 100.009028 | 100.009028 |  - | 100.009028 |  - |  100.0 |   0.009028 | 100.009028 |
    | 2000-01-03 17:30:00 | 100.007313 | 100.007313 |  - | 100.007313 |  - |  100.0 |   0.007313 | 100.007313 |
    | 2000-01-03 18:00:00 | 100.005918 | 100.005918 |  - | 100.005918 |  - |  100.0 |   0.005918 | 100.005918 |
    | 2000-01-03 18:30:00 | 100.004785 | 100.004785 |  - | 100.004785 |  - |  100.0 |   0.004785 | 100.004785 |
    | 2000-01-03 19:00:00 | 100.003865 | 100.003865 |  - | 100.003865 |  - |  100.0 |   0.003865 | 100.003865 |
    | 2000-01-03 19:30:00 | 100.003119 | 100.003119 |  - | 100.003119 |  - |  100.0 |   0.003119 | 100.003119 |
    | 2000-01-03 20:00:00 | 100.002515 | 100.002515 |  - | 100.002515 |  - |  100.0 |   0.002515 | 100.002515 |
    | 2000-01-03 20:30:00 | 100.002026 | 100.002026 |  - | 100.002026 |  - |  100.0 |   0.002026 | 100.002026 |
    | 2000-01-03 21:00:00 | 100.001631 | 100.001631 |  - | 100.001631 |  - |  100.0 |   0.001631 | 100.001631 |
    | 2000-01-03 21:30:00 | 100.001312 | 100.001312 |  - | 100.001312 |  - |  100.0 |   0.001312 | 100.001312 |
    | 2000-01-03 22:00:00 | 100.001054 | 100.001054 |  - | 100.001054 |  - |  100.0 |   0.001054 | 100.001054 |
    | 2000-01-03 22:30:00 | 100.000846 | 100.000846 |  - | 100.000846 |  - |  100.0 |   0.000846 | 100.000846 |
    | 2000-01-03 23:00:00 | 100.000679 | 100.000679 |  - | 100.000679 |  - |  100.0 |   0.000679 | 100.000679 |
    | 2000-01-03 23:30:00 | 100.000544 | 100.000544 |  - | 100.000544 |  - |  100.0 |   0.000544 | 100.000544 |
    | 2000-01-04 00:00:00 | 100.000436 | 100.000436 |  - | 100.000436 |  - |  100.0 |   0.000436 | 100.000436 |
    | 2000-01-04 00:30:00 | 100.000349 | 100.000349 |  - | 100.000349 |  - |  100.0 |   0.000349 | 100.000349 |
    | 2000-01-04 01:00:00 | 100.000279 | 100.000279 |  - | 100.000279 |  - |  100.0 |   0.000279 | 100.000279 |
    | 2000-01-04 01:30:00 | 100.000223 | 100.000223 |  - | 100.000223 |  - |  100.0 |   0.000223 | 100.000223 |
    | 2000-01-04 02:00:00 | 100.000178 | 100.000178 |  - | 100.000178 |  - |  100.0 |   0.000178 | 100.000178 |
    | 2000-01-04 02:30:00 | 100.000142 | 100.000142 |  - | 100.000142 |  - |  100.0 |   0.000142 | 100.000142 |
    | 2000-01-04 03:00:00 | 100.000113 | 100.000113 |  - | 100.000113 |  - |  100.0 |   0.000113 | 100.000113 |
    | 2000-01-04 03:30:00 |  100.00009 |  100.00009 |  - |  100.00009 |  - |  100.0 |    0.00009 |  100.00009 |
    | 2000-01-04 04:00:00 | 100.000072 | 100.000072 |  - | 100.000072 |  - |  100.0 |   0.000072 | 100.000072 |
    | 2000-01-04 04:30:00 | 100.000057 | 100.000057 |  - | 100.000057 |  - |  100.0 |   0.000057 | 100.000057 |
    | 2000-01-04 05:00:00 | 100.000046 | 100.000046 |  - | 100.000046 |  - |  100.0 |   0.000046 | 100.000046 |
    | 2000-01-04 05:30:00 | 100.000036 | 100.000036 |  - | 100.000036 |  - |  100.0 |   0.000036 | 100.000036 |
    | 2000-01-04 06:00:00 | 100.000029 | 100.000029 |  - | 100.000029 |  - |  100.0 |   0.000029 | 100.000029 |
    | 2000-01-04 06:30:00 | 100.000023 | 100.000023 |  - | 100.000023 |  - |  100.0 |   0.000023 | 100.000023 |
    | 2000-01-04 07:00:00 | 100.000018 | 100.000018 |  - | 100.000018 |  - |  100.0 |   0.000018 | 100.000018 |
    | 2000-01-04 07:30:00 | 100.000014 | 100.000014 |  - | 100.000014 |  - |  100.0 |   0.000014 | 100.000014 |
    | 2000-01-04 08:00:00 | 100.000011 | 100.000011 |  - | 100.000011 |  - |  100.0 |   0.000011 | 100.000011 |
    | 2000-01-04 08:30:00 | 100.000009 | 100.000009 |  - | 100.000009 |  - |  100.0 |   0.000009 | 100.000009 |
    | 2000-01-04 09:00:00 | 100.000007 | 100.000007 |  - | 100.000007 |  - |  100.0 |   0.000007 | 100.000007 |
    | 2000-01-04 09:30:00 | 100.000006 | 100.000006 |  - | 100.000006 |  - |  100.0 |   0.000006 | 100.000006 |
    | 2000-01-04 10:00:00 | 100.000004 | 100.000004 |  - | 100.000004 |  - |  100.0 |   0.000004 | 100.000004 |
    | 2000-01-04 10:30:00 | 100.000004 | 100.000004 |  - | 100.000004 |  - |  100.0 |   0.000004 | 100.000004 |
    | 2000-01-04 11:00:00 | 100.000003 | 100.000003 |  - | 100.000003 |  - |  100.0 |   0.000003 | 100.000003 |
    | 2000-01-04 11:30:00 | 100.000002 | 100.000002 |  - | 100.000002 |  - |  100.0 |   0.000002 | 100.000002 |
    | 2000-01-04 12:00:00 | 100.000002 | 100.000002 |  - | 100.000002 |  - |  100.0 |   0.000002 | 100.000002 |
    | 2000-01-04 12:30:00 | 100.000001 | 100.000001 |  - | 100.000001 |  - |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 13:00:00 | 100.000001 | 100.000001 |  - | 100.000001 |  - |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 13:30:00 | 100.000001 | 100.000001 |  - | 100.000001 |  - |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 14:00:00 | 100.000001 | 100.000001 |  - | 100.000001 |  - |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 14:30:00 | 100.000001 | 100.000001 |  - | 100.000001 |  - |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 15:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 15:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 16:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 16:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 17:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 17:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 18:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 18:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 19:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 19:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 20:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 20:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 21:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 21:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 22:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 22:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 23:00:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 23:30:00 |      100.0 |      100.0 |  - |      100.0 |  - |  100.0 |        0.0 |      100.0 |
"""
# import...
# ...from HydPy
from hydpy.core import modeltools
from hydpy.auxs.anntools import ANN  # pylint: disable=unused-import
from hydpy.auxs.ppolytools import Poly, PPoly  # pylint: disable=unused-import
from hydpy.exe.modelimports import *

# ...from kinw
from hydpy.models.kinw import kinw_fluxes
from hydpy.models.kinw import kinw_model
from hydpy.models.kinw import kinw_solver


class Model(modeltools.ELSModel):
    """|kinw_williams_ext.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="KinW-Williams-Ext",
        description=(
            "Williams routing based on an externally calculated volume-discharge "
            "relationship"
        ),
    )

    SOLVERPARAMETERS = (
        kinw_solver.AbsErrorMax,
        kinw_solver.RelErrorMax,
        kinw_solver.RelDTMin,
        kinw_solver.RelDTMax,
    )
    SOLVERSEQUENCES = (kinw_fluxes.QG,)
    INLET_METHODS = (kinw_model.Pick_Q_V1,)
    RECEIVER_METHODS = ()
    ADD_METHODS = ()
    PART_ODE_METHODS = (
        kinw_model.Calc_QZA_V1,
        kinw_model.Calc_QG_V2,
        kinw_model.Calc_QA_V1,
    )
    FULL_ODE_METHODS = (kinw_model.Update_VG_V1,)
    OUTLET_METHODS = (kinw_model.Pass_Q_V1,)
    SENDER_METHODS = ()
    SUBMODELINTERFACES = ()
    SUBMODELS = ()


tester = Tester()
cythonizer = Cythonizer()
