#! /usr/bin/env python

"""
Convert PWM outputs from a ULog file into an RCbencemark script

This script depends on https://github.com/PX4/pyulog (which depends on numpy)
"""

from __future__ import print_function
import argparse
import csv


js_template = '''/* //////////////// generated by ulog2rcbm ////////////////
- filePrefix
- data -- [[timestamp_in_ms, pwm_output], [..., ...], ...]
- samplesAvg
///////////// User defined variables //////////// */

const filePrefix = {};
const data = {};
const samplesAvg = {};

let lastTimestamp = data[0][0];
let position = 0; // index of data array
let timeDifference;

///////////////// Beginning of the script //////////////////

if (!data) {{
    rcb.console.print("Error: data is not an array of length > 0");
    rcb.endScript();
}}

rcb.console.setVerbose(false);

rcb.files.newLogFile({{prefix: filePrefix}});

readSensor();

function readSensor() {{
    rcb.sensors.read(readDone, samplesAvg);
}}
function readDone(result) {{
    rcb.files.newLogEntry(result, readSensor);
}}

rcb.console.print("Starting...");

rcb.wait(callback, lastTimestamp > 5 ? 5 : lastTimestamp); // wait 5s or first timestamp, whichever smaller

function callback() {{
    rcb.output.set("esc", data[position][1]);
    position++;
    if (data.length > position) {{
        timeDifference = data[position][0] - lastTimestamp;
        lastTimestamp = data[position][0];
        rcb.wait(callback, timeDifference);
    }} else {{
        rcb.wait(function() {{
            rcb.endScript();
        }}, 1);
    }}
}}
'''

def main():
    parser = argparse.ArgumentParser(description='Convert CSV to RCbenchmark script')
    parser.add_argument('filename', metavar='file.csv', help='CSV input file')
    parser.add_argument('-c', '--col', dest='column', action='store',
                        help="column name (which column to use)")
    parser.add_argument('-t', '--time', dest='time', action='store',
                        help="time column name (which column to use as time, default is 'timestamp')", default='timestamp')
    parser.add_argument('--avg', dest='average', action='store',
                        help="sample average (default is '1')", default='1')
    args = parser.parse_args()
    convert_ulog2rcbm(args.filename, args.time, args.column, args.average)

def convert_ulog2rcbm(csv_file_name, time_col, column, sample_average):
    sample_average = int(sample_average)
    data = []
    with open(csv_file_name) as csv_file:
        csv_reader = csv.reader(csv_file, delimiter=',')
        line_count = 0
        col_index = 0
        time_col_index = 0
        for row in csv_reader:
            if line_count == 0:
                line_count += 1
                col_index = row.index(column)
                time_col_index = row.index(time_col)
            else:
                data.append({
                    'timestamp': int(row[time_col_index]),
                    'output': int(float(row[col_index]))
                })
    output_file_prefix = csv_file_name
    if output_file_prefix.lower().endswith('.csv'):
        output_file_prefix = output_file_prefix[:-4]
    output_file_prefix = '{0}_{1}_{2}'.format(output_file_prefix, column, sample_average)
    with open(output_file_prefix + '.js', 'w') as js_file:
        timestamp_list = map(lambda x: x['timestamp'], data)
        pwm_list = map(lambda x: x['output'], data)
        data_string = '[{}]'.format(
            ','.join(
                [ '[{},{}]'.format(row[0] / 1000000, int(row[1])) for row in list(zip(timestamp_list, pwm_list)) ]
            )
        )

        if output_file_prefix[:2] == './' or output_file_prefix[:2] == '.\\':
            output_file_prefix = output_file_prefix[2:]


        js_file.write(js_template.format('"{}"'.format(output_file_prefix), data_string, sample_average))

if __name__ == '__main__':
    main()
