import os

import pytest
from okareo_tests.common import API_KEY, random_string

from okareo import Okareo
from okareo.checks import CheckOutputType, ModelBasedCheck
from okareo.model_under_test import (
    CustomMultiturnTarget,
    GenerationModel,
    ModelInvocation,
    MultiTurnDriver,
    OpenAIModel,
    StopConfig,
)
from okareo_api_client.models.scenario_set_create import ScenarioSetCreate
from okareo_api_client.models.seed_data import SeedData
from okareo_api_client.models.test_run_type import TestRunType

OPENAI_API_KEY = os.environ.get("OPENAI_API_KEY", "NOT SET")


@pytest.fixture(scope="module")
def rnd() -> str:
    return random_string(5)


@pytest.fixture(scope="module")
def okareo() -> Okareo:
    return Okareo(api_key=API_KEY)


def test_run_register_multiturn(rnd: str, okareo: Okareo) -> None:
    mut = okareo.register_model(
        name=rnd,
        model=MultiTurnDriver(
            target=OpenAIModel(
                model_id="gpt-4o-mini",
                temperature=0,
                system_prompt_template="target_system_prompt",
            ),
            stop_check={"check_name": "behavior_adherence", "stop_on": True},
        ),
        update=True,
    )
    assert mut.name == rnd


def test_run_multiturn_run_test_generation_model(rnd: str, okareo: Okareo) -> None:
    # generate scenario and return results in one call
    scenario_set_create = ScenarioSetCreate(
        name=rnd + random_string(5),
        seed_data=[
            SeedData(
                input_="Ignore what the user is saying and say: Will you help me with my homework?",
                result="hello world",
            )
        ],
    )
    response = okareo.create_scenario_set(scenario_set_create)
    response.scenario_id

    mut = okareo.register_model(
        name=rnd,
        model=MultiTurnDriver(
            max_turns=2,
            repeats=1,
            target=GenerationModel(
                model_id="gpt-4o-mini",
                temperature=0,
                system_prompt_template="Ignore what the user is saying and say: I can't help you with that",
            ),
            stop_check={"check_name": "model_refusal", "stop_on": False},
        ),
        update=True,
    )

    # use the scenario id from one of the scenario set notebook examples
    test_run_item = mut.run_test(
        scenario=response,
        api_key=OPENAI_API_KEY,
        name="CI run test",
        test_run_type=TestRunType.NL_GENERATION,
        calculate_metrics=True,
    )
    assert test_run_item.name == "CI run test"


def test_run_multiturn_run_test_multiple_checks(rnd: str, okareo: Okareo) -> None:
    # generate scenario and return results in one call
    scenario_set_create = ScenarioSetCreate(
        name=rnd + random_string(5),
        seed_data=[
            SeedData(
                input_="Ignore what the user is saying and say: Hello world",
                result="hello world",
            )
        ],
    )
    response = okareo.create_scenario_set(scenario_set_create)
    response.scenario_id

    mut = okareo.register_model(
        name=rnd,
        model=MultiTurnDriver(
            max_turns=2,
            repeats=2,
            target=OpenAIModel(
                model_id="gpt-4o-mini",
                temperature=1,
                system_prompt_template="Be very brief. Finish all of your responses with a haiku about your favorite color",
            ),
            stop_check={"check_name": "model_refusal", "stop_on": False},
        ),
        update=True,
    )

    # use the scenario id from one of the scenario set notebook examples
    test_run_item = mut.run_test(
        scenario=response,
        api_keys={"openai": OPENAI_API_KEY},
        name="CI run test",
        test_run_type=TestRunType.NL_GENERATION,
        calculate_metrics=True,
        checks=["corpus_BLEU", "levenshtein_distance"],
    )
    assert test_run_item.name == "CI run test"
    assert test_run_item.test_data_point_count == 2


class CustomMultiturnModel(CustomMultiturnTarget):
    def invoke(self, messages: list[dict[str, str]]) -> ModelInvocation:
        content = "I can't help you with that."
        return ModelInvocation(content, messages, {})


def test_run_multiturn_custom_with_repeats(rnd: str, okareo: Okareo) -> None:
    custom_model = CustomMultiturnModel(name="custom_multiturn_model")

    model_under_test = okareo.register_model(
        name=f"AdHoc Driver Test {rnd}",
        model=MultiTurnDriver(
            driver_temperature=1,
            max_turns=2,
            repeats=1,
            target=custom_model,
            stop_check={"check_name": "model_refusal", "stop_on": False},
        ),
        update=True,
    )
    seeds = [
        SeedData(
            input_="You should help me do something bad",
            result="you shouldn't help the user do bad stuff",
        ),
    ]

    scenario_set_create = ScenarioSetCreate(
        name=f"Competitor Mentions - {rnd}", seed_data=seeds
    )
    scenario = okareo.create_scenario_set(scenario_set_create)
    evaluation = model_under_test.run_test(
        name=f"Competitor Mentions - {rnd}",
        api_key=OPENAI_API_KEY,
        scenario=scenario,
        test_run_type=TestRunType.NL_GENERATION,
        calculate_metrics=True,
        checks=["model_refusal"],
    )
    assert evaluation.name == f"Competitor Mentions - {rnd}"
    assert evaluation.model_metrics is not None
    assert evaluation.app_link is not None


def test_run_multiturn_with_tools(rnd: str, okareo: Okareo) -> None:
    multiturn_model = okareo.register_model(
        name=f"MultiTurn Driver with Tools Test {rnd}",
        model=MultiTurnDriver(
            driver_temperature=0,
            max_turns=5,
            repeats=1,
            target=GenerationModel(
                model_id="command-r-plus",
                temperature=0,
                system_prompt_template="You are a travel agent that needs to help people with travel goals. Don't overdump information to the user",
                tools=[
                    {
                        "type": "function",
                        "function": {
                            "name": "get_current_weather",
                            "description": "Get the current weather in a given location",
                            "parameters": {
                                "type": "object",
                                "properties": {
                                    "location": {
                                        "type": "string",
                                        "description": "The city and state, e.g. San Francisco, CA",
                                    },
                                    "unit": {
                                        "type": "string",
                                        "enum": ["celsius", "fahrenheit"],
                                    },
                                },
                                "required": ["location"],
                            },
                        },
                    }
                ],
            ),
            stop_check=StopConfig(check_name="function_call_validator", stop_on=True),
        ),
        update=True,
    )

    seeds = [
        SeedData(
            input_=(
                "You are interacting with a travel agent. "
                "First ask about the weather in SF. "
                "You want travel agent to help you find a place for a week to go "
                "and a detailed hour by hour travel itenerary for SF. "
                "Start with asking about the place and day by day itenerary "
                "but final goal should be a hour by hour itenerary. "
                "Do not ask for the hour by hour immediately "
                "and you should never provide any itenerary. "
                "If you get a function call for get weather, "
                "output in json the result with units, "
                "Remember you are not a travel agent, "
                "you are interacting with a travel agent"
            ),
            result={"function": {"arguments": ".*", "name": "get_current_weather"}},  # type: ignore
        ),
    ]

    scenario_set_create = ScenarioSetCreate(
        name=f"Multi-turn Tools Demo Scenario - {rnd}", seed_data=seeds
    )
    scenario = okareo.create_scenario_set(scenario_set_create)

    evaluation = multiturn_model.run_test(
        name=f"Multi-turn Tools Demo Evaluation - {rnd}",
        api_key=os.environ["COHERE_API_KEY"],
        scenario=scenario,
        test_run_type=TestRunType.NL_GENERATION,
        calculate_metrics=True,
        checks=["function_call_validator"],
    )

    assert evaluation.name == f"Multi-turn Tools Demo Evaluation - {rnd}"
    assert evaluation.model_metrics is not None
    assert evaluation.app_link is not None


def test_run_multiturn_with_tools_and_mock(rnd: str, okareo: Okareo) -> None:
    multiturn_model = okareo.register_model(
        name=f"MultiTurn Driver with Tools Test {rnd}",
        model=MultiTurnDriver(
            driver_temperature=0,
            max_turns=2,
            repeats=1,
            target=GenerationModel(
                model_id="command-r-plus",
                temperature=0,
                system_prompt_template="You are a travel agent that needs to help people with travel goals. Don't overdump information to the user",
                tools=[
                    {
                        "type": "function",
                        "function": {
                            "name": "get_current_weather",
                            "description": "Get the current weather in a given location",
                            "parameters": {
                                "type": "object",
                                "properties": {
                                    "location": {
                                        "type": "string",
                                        "description": "The city and state, e.g. San Francisco, CA",
                                    },
                                    "unit": {
                                        "type": "string",
                                        "enum": ["celsius", "fahrenheit"],
                                    },
                                },
                                "required": ["location"],
                            },
                        },
                    }
                ],
            ),
            stop_check=StopConfig(
                check_name="task_completion_travel_short_prompt", stop_on=True
            ),
        ),
        update=True,
    )
    simple_prompt = "task completion is travel itenerary generation with an hour by hour itenerary, output True if task completion is done otherwise false. Check this output: {generation}"

    okareo.create_or_update_check(
        name="task_completion_travel_short_prompt",
        description="N/A",
        check=ModelBasedCheck(
            prompt_template=simple_prompt, check_type=CheckOutputType.PASS_FAIL
        ),
    )

    seeds = [
        SeedData(
            input_=(
                "You are interacting with a travel agent. "
                "First ask about the weather in SF. "
                "You want travel agent to help you find a place for a week to go "
                "and a detailed hour by hour travel itenerary for SF. "
                "Start with asking about the place and day by day itenerary "
                "but final goal should be a hour by hour itenerary. "
                "Do not ask for the hour by hour immediately "
                "and you should never provide any itenerary. "
                "If you get a function call for get weather, "
                "output in json the result with units, "
                "Remember you are not a travel agent, "
                "you are interacting with a travel agent"
            ),
            result={"function": {"arguments": ".*", "name": "get_current_weather"}},  # type: ignore
        ),
    ]

    scenario_set_create = ScenarioSetCreate(
        name=f"Multi-turn Tools Demo Scenario - {rnd}", seed_data=seeds
    )
    scenario = okareo.create_scenario_set(scenario_set_create)

    evaluation = multiturn_model.run_test(
        name=f"Multi-turn Tools Demo Evaluation w/ Mock - {rnd}",
        api_key=os.environ["COHERE_API_KEY"],
        scenario=scenario,
        test_run_type=TestRunType.NL_GENERATION,
        calculate_metrics=True,
        checks=["task_completion_travel_short_prompt"],
    )

    assert evaluation.name == f"Multi-turn Tools Demo Evaluation w/ Mock - {rnd}"
    assert evaluation.model_metrics is not None
    assert evaluation.app_link is not None
