"use strict";
const assert_1 = require("@aws-cdk/assert");
const kinesis = require("@aws-cdk/aws-kinesis");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const sources = require("../lib");
const test_function_1 = require("./test-function");
module.exports = {
    'sufficiently complex example'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        fn.addEventSource(new sources.KinesisEventSource(stream, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "kinesis:DescribeStream",
                            "kinesis:GetRecords",
                            "kinesis:GetShardIterator"
                        ],
                        "Effect": "Allow",
                        "Resource": {
                            "Fn::GetAtt": [
                                "S509448A1",
                                "Arn"
                            ]
                        }
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "FnServiceRoleDefaultPolicyC6A839BF",
            "Roles": [{
                    "Ref": "FnServiceRoleB9001A96"
                }]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "S509448A1",
                    "Arn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "BatchSize": 100,
            "StartingPosition": "TRIM_HORIZON"
        }));
        test.done();
    },
    'specific batch size'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        fn.addEventSource(new sources.KinesisEventSource(stream, {
            batchSize: 50,
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "S509448A1",
                    "Arn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "BatchSize": 50,
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
    'fails if batch size < 1'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        test.throws(() => fn.addEventSource(new sources.KinesisEventSource(stream, {
            batchSize: 0,
            startingPosition: lambda.StartingPosition.LATEST
        })), /Maximum batch size must be between 1 and 10000 inclusive \(given 0\)/);
        test.done();
    },
    'fails if batch size > 10000'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        test.throws(() => fn.addEventSource(new sources.KinesisEventSource(stream, {
            batchSize: 10001,
            startingPosition: lambda.StartingPosition.LATEST
        })), /Maximum batch size must be between 1 and 10000 inclusive \(given 10001\)/);
        test.done();
    },
    'specific maxBatchingWindow'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        fn.addEventSource(new sources.KinesisEventSource(stream, {
            maxBatchingWindow: cdk.Duration.minutes(2),
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "S509448A1",
                    "Arn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "MaximumBatchingWindowInSeconds": 120,
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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