# https://mathematica.stackexchange.com/a/104696
import zlib
import base64
import struct
import io


def _decomp(b):
    """
    _decomp(b) decompresses a Mathematica-encoded (zlib and b64)
    string and returns a file pointer
    """
    fp = io.BytesIO(
        zlib.decompress(base64.b64decode(b[2:]))
    )
    if fp.read(4) != b'!boR':
        raise KeyError('Header is not matching')
    return fp


def _parse(fp):
    """
    _parse(fp) iterates over a file pointer and returns a nested array
    containing the data
    """
    ans = []
    cur = []
    while True:
        cmd = fp.read(1).decode()
        if cmd == '':
            ans.append(cur)
            return ans
        elif cmd == 'i':
            cur.append(struct.unpack('<i', fp.read(4))[0])
        elif cmd == 'S' or cmd == 's':
            le = struct.unpack('<i', fp.read(4))[0]
            cur.append(fp.read(le).decode())
        elif cmd == 'r':
            cur.append(struct.unpack('<d', fp.read(8))[0])
        elif cmd == 'f':
            le = struct.unpack('<i', fp.read(4))[0]
            ans.append(cur[:])
            cur = []
        else:
            raise KeyError('Unknown cmd %s' % cmd)


def _flatten(l):
    """
    _flatten(l) tries to flatten the nested list obtained from _parse
    """
    if len(l) == 0:
        return None
    if len(l) == 1 and len(l[0]) == 1:
        return l[0][0]
    if l[0] == 'List' and len(l) == 1:
        return list
    if l[0] == 'List' and len(l) > 1:
        return l[1:]
    else:
        def iflat(ll):
            if type(ll) != list:
                return ll
            if len(ll) == 0:
                return None
            if ll[0] == list:
                return ll[1:]
            if type(ll[0]) == list:
                return ll[0]
            else:
                return ll
        ans = [_flatten(i) for i in l]
        ans = iflat([i for i in ans if i])
        return ans


def uncompress(b):
    """
    uncompress("string") recovers an expression from a compressed
    string representation generated by Mathematica's Compress. Only
    lists, numbers, and strings are supported. Lists can be nested.
    """
    return _flatten(_parse(_decomp(b)))


def _tests():
    tests = [
        ('1:eJxTTMoPymRkYGAAAAtIAa8=', 1),
        ('1:eJxTTMoPSmNkYGAoZgESPpnFJZkgHgA3AQQp', [1]),
        ('1:eJxTTMoPSmNiYGAoZgESPpnFJZmMQEYmSAgATfoElQ==', [1,2]),
        (
          '1:eJxTTMoPSmNiYGAoZgESPpnFJai8TEYgIxMkhCbODBIH8QB/Mwpk',
          [[1,2],[3,4]]
        ),
        ('1:eJxTTMoPKmIAgw/2ABRsAuY=', 1.),
        ('1:eJxTTMoPSmNiYGAoZgESPpnFJUUMYPDBHspgcAAAfs4GEw==', [1.,2.]),
        ('1:eJxTTMoPCmZmYGBITEoGABH9AsE=', 'abc')
    ]

    return all(
        uncompress(c) == v
        for c,v in tests
    )
