import numpy as np

pi = np.pi

zeta2 = 1.6449340668482264364724151666460251892189499012068
zeta3 = 1.2020569031595942853997381615114499907649862923405

__coeff_li2 = [
    +0.42996693560813697203703367869938799,
    +0.40975987533077105846826371092525528,
    -0.01858843665014591964764164914021227,
    +0.00145751084062267855367392841645949,
    -0.00014304184442340048774883012009088,
    +0.00001588415541879553236190550471677,
    -0.00000190784959386582722719632114209,
    +0.00000024195180854164749499461464343,
    -0.00000003193341274251783460496014143,
    +0.00000000434545062676912298795717848,
    -0.00000000060578480118407444429705331,
    +0.00000000008612097799359498244283685,
    -0.00000000001244331659938867989642421,
    +0.00000000000182255696235736330065548,
    -0.00000000000027006766049114651808572,
    +0.00000000000004042209263152664648833,
    -0.00000000000000610325145269187950378,
    +0.00000000000000092862975330195758613,
    -0.00000000000000014226020855112446840,
    +0.00000000000000002192631718153957354,
    -0.00000000000000000339797324215897863,
    +0.00000000000000000052919542448331471,
    -0.00000000000000000008278580814278998,
    +0.00000000000000000001300371734545560,
    -0.00000000000000000000205022224255282,
    +0.00000000000000000000032435785491489,
    -0.00000000000000000000005147799903343,
    +0.00000000000000000000000819387747717,
    -0.00000000000000000000000130778354057,
    +0.00000000000000000000000020925629306,
    -0.00000000000000000000000003356166151,
    +0.00000000000000000000000000539465777,
    -0.00000000000000000000000000086891932,
    +0.00000000000000000000000000014022817,
    -0.00000000000000000000000000002267156,
    +0.00000000000000000000000000000367174,
    -0.00000000000000000000000000000059562,
    +0.00000000000000000000000000000009677,
    -0.00000000000000000000000000000001574,
    +0.00000000000000000000000000000000257,
    -0.00000000000000000000000000000000042,
    +0.00000000000000000000000000000000007,
    -0.00000000000000000000000000000000001
]

__coeff_li3a = [
    +0.46172939286012092817954516381760016,
    +0.45017399588550288560580364647352070,
    -0.01091284195229295374494914320402658,
    +0.00059324547127243642952756961712713,
    -0.00004479593219280756178998757870776,
    +0.00000405154578580684540984293800468,
    -0.00000041095398606214457668547736075,
    +0.00000004513178770934181970313262557,
    -0.00000000525466158515342604029419927,
    +0.00000000063982547910449452549291936,
    -0.00000000008071938839872532971820424,
    +0.00000000001048073680087126094928657,
    -0.00000000000139365085138335067524094,
    +0.00000000000018907205037339730044704,
    -0.00000000000002609371657183250621931,
    +0.00000000000000365481859219879483309,
    -0.00000000000000051855842492271228151,
    +0.00000000000000007441491722173878908,
    -0.00000000000000001078686838424874221,
    +0.00000000000000000157774237809543778,
    -0.00000000000000000023264073800573828,
    +0.00000000000000000003455457587964154,
    -0.00000000000000000000516658458392580,
    +0.00000000000000000000077718849383139,
    -0.00000000000000000000011755815708807,
    +0.00000000000000000000001787262690583,
    -0.00000000000000000000000272999302683,
    +0.00000000000000000000000041881267359,
    -0.00000000000000000000000006451004176,
    +0.00000000000000000000000000997383916,
    -0.00000000000000000000000000154744603,
    +0.00000000000000000000000000024087296,
    -0.00000000000000000000000000003760889,
    +0.00000000000000000000000000000588900,
    -0.00000000000000000000000000000092463,
    +0.00000000000000000000000000000014555,
    -0.00000000000000000000000000000002297,
    +0.00000000000000000000000000000000363,
    -0.00000000000000000000000000000000058,
    +0.00000000000000000000000000000000009,
    -0.00000000000000000000000000000000001
]
__coeff_li3b = [
    -0.01601618044919582873670691984756338,
    -0.50364244007530129181209541016960792,
    -0.01615099243050025888745446951929454,
    -0.00124402421042449361265610524413112,
    -0.00013757218124461673921971996409271,
    -0.00001856381852603773316486795183129,
    -0.00000284173534515440415934505790039,
    -0.00000047459967905789937221638951390,
    -0.00000008448038543781200676091819474,
    -0.00000001578767124043400543246870475,
    -0.00000000306576207139903798128889004,
    -0.00000000061407921728125845808062189,
    -0.00000000012618830243156719690872484,
    -0.00000000002649314179819609957126783,
    -0.00000000000566470854636425926158812,
    -0.00000000000123041115779581117517467,
    -0.00000000000027093457836786768143960,
    -0.00000000000006038026463383701279197,
    -0.00000000000001360008993995749682352,
    -0.00000000000000309244740631856875855,
    -0.00000000000000070917249609207158220,
    -0.00000000000000016388083639226002471,
    -0.00000000000000003813464350168994613,
    -0.00000000000000000893010739611811656,
    -0.00000000000000000210331341599359416,
    -0.00000000000000000049802988416537866,
    -0.00000000000000000011850292695597351,
    -0.00000000000000000002832460494402074,
    -0.00000000000000000000679854955943073,
    -0.00000000000000000000163816629435900,
    -0.00000000000000000000039616291258646,
    -0.00000000000000000000009613022139972,
    -0.00000000000000000000002340035706102,
    -0.00000000000000000000000571315840877,
    -0.00000000000000000000000139876183805,
    -0.00000000000000000000000034336361321,
    -0.00000000000000000000000008449733573,
    -0.00000000000000000000000002084253881,
    -0.00000000000000000000000000515255292,
    -0.00000000000000000000000000127646290,
    -0.00000000000000000000000000031685555,
    -0.00000000000000000000000000007880228,
    -0.00000000000000000000000000001963363,
    -0.00000000000000000000000000000490016,
    -0.00000000000000000000000000000122499,
    -0.00000000000000000000000000000030671,
    -0.00000000000000000000000000000007691,
    -0.00000000000000000000000000000001931,
    -0.00000000000000000000000000000000486,
    -0.00000000000000000000000000000000122,
    -0.00000000000000000000000000000000031,
    -0.00000000000000000000000000000000008,
    -0.00000000000000000000000000000000002
]


def Li2(x):
    """
    Li2(x) returns PolyLog[2, x] for x as a number, a list, or an
    np.ndarray.
    """
    if type(x) == list:
        x = np.array(x)
    if type(x) != np.ndarray:
        x = np.array([x])

    y = np.zeros(x.shape)
    s = np.zeros(x.shape)
    a = np.zeros(x.shape)

    mask = -1. > x
    s[mask] = -1.
    y[mask] = -1./x[mask]
    a[mask] = zeta2 + 0.5*np.log(-x[mask])**2

    mask = (0. > x) & (x > -1.)
    s[mask] = +1.
    y[mask] = -x[mask]
    a[mask] = 0.

    mask = (0.5 > x) & (x > 0.)
    s[mask] = -1.
    y[mask] = x[mask]/(1-x[mask])
    a[mask] = 0.5 * np.log(1.-x[mask])**2

    mask = (1. > x) & (x > 0.5)
    s[mask] = +1.
    y[mask] = (1.-x[mask])/x[mask]
    a[mask] = -zeta2 + np.log(x[mask]) * (
        -0.5*np.log(x[mask]) + np.log(1.-x[mask])
    )

    mask = (2. > x) & (x > 1.)
    s[mask] = -1.
    y[mask] = x[mask]-1.
    a[mask] = -zeta2 + np.log(x[mask])*(np.log(x[mask])+np.log(1.-1./x[mask]))

    mask = x > 2.
    s[mask] = +1.
    y[mask] = -1. / (1.-x[mask])
    a[mask] = -2.*zeta2 + 0.5*(np.log(x[mask])**2-np.log(1.-1./x[mask])**2)

    h = -1.+2.*y
    ans = - s*np.polynomial.chebyshev.chebval(h, __coeff_li2) - a
    if len(ans) == 1:
        return ans[0]
    else:
        return ans


def Li3(x):
    """
    Li3(x) returns PolyLog[3, x] for x as a number, a list, or an
    np.ndarray.
    """
    if type(x) == list:
        x = np.array(x)
    if type(x) != np.ndarray:
        x = np.array([x])

    ya = np.zeros(x.shape)
    yb = np.zeros(x.shape)
    s = np.zeros(x.shape)
    a = np.zeros(x.shape)

    mask = x > 2.
    ya[mask] = 0.
    yb[mask] = 1./x[mask]
    s[mask] = -1.
    a[mask] = 2.*zeta2*np.log(x[mask]) - np.log(x[mask])**3/6.

    mask = (1. < x) & (x < 2.)
    ya[mask] = 1.-x[mask]
    yb[mask] = (x[mask]-1.)/x[mask]
    s[mask] = 1.
    a[mask] = (
        zeta3 + zeta2*np.log(x[mask])
        - (np.log(x[mask] - 1.)*np.log(x[mask])**2)/2.
        + np.log(x[mask])**3/6.
    )

    mask = (0.5 < x) & (x < 1.)
    ya[mask] = (x[mask]-1)/x[mask]
    yb[mask] = 1.-x[mask]
    s[mask] = 1.
    a[mask] = (
        zeta3 + zeta2*np.log(x[mask])
        - (np.log(1-x[mask])*np.log(x[mask])**2)/2.
        + np.log(x[mask])**3/6.
    )

    mask = (0. < x) & (x < 0.5)
    ya[mask] = 0.
    yb[mask] = x[mask]
    s[mask] = -1.
    a[mask] = 0.

    mask = (-1. < x) & (x < 0.)
    ya[mask] = x[mask]
    yb[mask] = 0.
    s[mask] = -1.
    a[mask] = 0.

    mask = x < -1.
    ya[mask] = 1./x[mask]
    yb[mask] = 0.
    s[mask] = -1.
    a[mask] = -np.log(-x[mask])*(zeta2 + np.log(-x[mask])**2/6.)

    ans = s * (
        + np.polynomial.chebyshev.chebval(-1. - 2. * ya, __coeff_li3a)
        + np.polynomial.chebyshev.chebval(-0. + 2. * yb, __coeff_li3b)
    ) + a
    if len(ans) == 1:
        return ans[0]
    else:
        return ans


def _test():
    delta = Li2(np.linspace(-5, 5)) - np.array([
        -2.74927912606080, -2.675333579811486, -2.599703922819041,
        -2.52229547292464, -2.443004659690393, -2.361717822189668,
        -2.27830978980379, -2.192642196132418, -2.104561461953607,
        -2.01389636413251, -1.920455081483496, -1.824021572894432,
        -1.72435109308091, -1.621164580327744, -1.514141547786766,
        -1.40291095813099, -1.287039332257734, -1.166014988157043,
        -1.03922674160052, -0.905934472259533, -0.765227375086993,
        -0.61596289350954, -0.456674030636577, -0.285422140984183,
        -0.09954942518892, +0.104769204444255, +0.333423619531027,
        +0.59645084173761, +0.914188314378351, +1.348206971080113,
        +2.00640965880388, +2.264346142313667, +2.387794539331424,
        +2.44606913754080, +2.466424217946511, +2.462745206298323,
        +2.44303283877077, +2.412260698399273, +2.373687785652424,
        +2.32953467134883, +2.281360767067206, +2.230287768609120,
        +2.17713831620581, +2.122525350789271, +2.066911526456334,
        +2.01064977016234, +1.954011604468827, +1.897207317023113,
        +1.84040057268513, +1.783719161266630
    ])
    print(np.sum(np.abs(delta)))

    delta = Li3(np.linspace(-5, 5)) - np.array([
        -3.53751143761860, -3.42448732647973, -3.30980017697430,
        -3.19337745427346, -3.07514097491575, -2.95500625033939,
        -2.83288172636787, -2.70866789732938, -2.58225626803600,
        -2.45352812969421, -2.32235310632768, -2.18858741555158,
        -2.05207177020411, -1.91262882341454, -1.77006002613222,
        -1.62414171823916, -1.47462020462693, -1.32120546380476,
        -1.16356297816002, -1.00130292616127, -0.83396557295181,
        -0.66100101473036, -0.48174023044546, -0.29535215421824,
        -0.10077701608696, +0.10338349665284, +0.31906225852367,
        +0.54911887068965, +0.79828073240277, +1.07639965208425,
        +1.41634617733515, +1.77529684006970, +2.10898409178295,
        +2.41191896746043, +2.68536905447592, +2.93219733785331,
        +3.15545429356693, +3.35795710877776, +3.54218390358867,
        +3.71027441009754, +3.86406461226205, +4.00512891436202,
        +4.13482034864287, +4.25430583634250, +4.36459602118885,
        +4.46657010627095, +4.56099639428021, +4.64854924688435,
        +4.72982310377487, +4.80534410296558
    ])
    print(np.sum(np.abs(delta)))
