"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnOutput = void 0;
const cfn_element_1 = require("./cfn-element");
/**
 * @stability stable
 */
class CfnOutput extends cfn_element_1.CfnElement {
    /**
     * Creates an CfnOutput value for this stack.
     *
     * @param scope The parent construct.
     * @param props CfnOutput properties.
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (props.value === undefined) {
            throw new Error(`Missing value for CloudFormation output at path "${this.node.path}"`);
        }
        this._description = props.description;
        this._value = props.value;
        this._condition = props.condition;
        this._exportName = props.exportName;
    }
    /**
     * A String type that describes the output value.
     *
     * The description can be a maximum of 4 K in length.
     *
     * @default - No description.
     * @stability stable
     */
    get description() {
        return this._description;
    }
    /**
     * A String type that describes the output value.
     *
     * The description can be a maximum of 4 K in length.
     *
     * @default - No description.
     * @stability stable
     */
    set description(description) {
        this._description = description;
    }
    /**
     * The value of the property returned by the aws cloudformation describe-stacks command.
     *
     * The value of an output can include literals, parameter references, pseudo-parameters,
     * a mapping value, or intrinsic functions.
     *
     * @stability stable
     */
    get value() {
        return this._value;
    }
    /**
     * The value of the property returned by the aws cloudformation describe-stacks command.
     *
     * The value of an output can include literals, parameter references, pseudo-parameters,
     * a mapping value, or intrinsic functions.
     *
     * @stability stable
     */
    set value(value) {
        this._value = value;
    }
    /**
     * A condition to associate with this output value.
     *
     * If the condition evaluates
     * to `false`, this output value will not be included in the stack.
     *
     * @default - No condition is associated with the output.
     * @stability stable
     */
    get condition() {
        return this._condition;
    }
    /**
     * A condition to associate with this output value.
     *
     * If the condition evaluates
     * to `false`, this output value will not be included in the stack.
     *
     * @default - No condition is associated with the output.
     * @stability stable
     */
    set condition(condition) {
        this._condition = condition;
    }
    /**
     * The name used to export the value of this output across stacks.
     *
     * To use the value in another stack, pass the value of
     * `output.importValue` to it.
     *
     * @default - the output is not exported
     * @stability stable
     */
    get exportName() {
        return this._exportName;
    }
    /**
     * The name used to export the value of this output across stacks.
     *
     * To use the value in another stack, pass the value of
     * `output.importValue` to it.
     *
     * @default - the output is not exported
     * @stability stable
     */
    set exportName(exportName) {
        this._exportName = exportName;
    }
    /**
     * Return the `Fn.importValue` expression to import this value into another stack.
     *
     * The returned value should not be used in the same stack, but in a
     * different one. It must be deployed to the same environment, as
     * CloudFormation exports can only be imported in the same Region and
     * account.
     *
     * The is no automatic registration of dependencies between stacks when using
     * this mechanism, so you should make sure to deploy them in the right order
     * yourself.
     *
     * You can use this mechanism to share values across Stacks in different
     * Stages. If you intend to share the value to another Stack inside the same
     * Stage, the automatic cross-stack referencing mechanism is more convenient.
     *
     * @stability stable
     */
    get importValue() {
        // We made _exportName mutable so this will have to be lazy.
        return cfn_fn_1.Fn.importValue(lazy_1.Lazy.stringValue({
            produce: (ctx) => {
                if (stack_1.Stack.of(ctx.scope) === this.stack) {
                    throw new Error(`'importValue' property of '${this.node.path}' should only be used in a different Stack`);
                }
                if (!this._exportName) {
                    throw new Error(`Add an exportName to the CfnOutput at '${this.node.path}' in order to use 'output.importValue'`);
                }
                return this._exportName;
            },
        }));
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Outputs: {
                [this.logicalId]: {
                    Description: this._description,
                    Value: this._value,
                    Export: this._exportName != null ? { Name: this._exportName } : undefined,
                    Condition: this._condition ? this._condition.logicalId : undefined,
                },
            },
        };
    }
}
exports.CfnOutput = CfnOutput;
const cfn_fn_1 = require("./cfn-fn");
const lazy_1 = require("./lazy");
const stack_1 = require("./stack");
//# sourceMappingURL=data:application/json;base64,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