"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenIdConnectProvider = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const RESOURCE_TYPE = 'Custom::AWSCDKOpenIdConnectProvider';
/**
 * (experimental) IAM OIDC identity providers are entities in IAM that describe an external identity provider (IdP) service that supports the OpenID Connect (OIDC) standard, such as Google or Salesforce.
 *
 * You use an IAM OIDC identity provider
 * when you want to establish trust between an OIDC-compatible IdP and your AWS
 * account. This is useful when creating a mobile app or web application that
 * requires access to AWS resources, but you don't want to create custom sign-in
 * code or manage your own user identities.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc.html
 * @experimental
 * @resource AWS::CloudFormation::CustomResource
 */
class OpenIdConnectProvider extends core_1.Resource {
    /**
     * (experimental) Defines an OpenID Connect provider.
     *
     * @param scope The definition scope.
     * @param id Construct ID.
     * @param props Initialization properties.
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: RESOURCE_TYPE,
            serviceToken: this.getOrCreateProvider(),
            properties: {
                ClientIDList: props.clientIds,
                ThumbprintList: props.thumbprints,
                Url: props.url,
            },
        });
        this.openIdConnectProviderArn = core_1.Token.asString(resource.ref);
        this.openIdConnectProviderIssuer = core_1.Fn.select(1, core_1.Fn.split('oidc-provider/', this.openIdConnectProviderArn));
    }
    /**
     * (experimental) Imports an Open ID connect provider from an ARN.
     *
     * @param scope The definition scope.
     * @param id ID of the construct.
     * @param openIdConnectProviderArn the ARN to import.
     * @experimental
     */
    static fromOpenIdConnectProviderArn(scope, id, openIdConnectProviderArn) {
        const parsedResourceName = core_1.Stack.of(scope).parseArn(openIdConnectProviderArn).resourceName;
        if (!parsedResourceName) {
            throw new Error(`Invalid arn: ${openIdConnectProviderArn}. Unable to extract issuer url`);
        }
        // this needed because TS don't understand that prev. condition
        // actually does mutate the type from "string | undefined" to "string"
        // inside class definition,
        const resourceName = parsedResourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.openIdConnectProviderArn = openIdConnectProviderArn;
                this.openIdConnectProviderIssuer = resourceName;
            }
        }
        return new Import(scope, id);
    }
    getOrCreateProvider() {
        return core_1.CustomResourceProvider.getOrCreate(this, RESOURCE_TYPE, {
            codeDirectory: path.join(__dirname, 'oidc-provider'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12,
            policyStatements: [
                {
                    Effect: 'Allow',
                    Resource: '*',
                    Action: [
                        'iam:CreateOpenIDConnectProvider',
                        'iam:DeleteOpenIDConnectProvider',
                        'iam:UpdateOpenIDConnectProviderThumbprint',
                        'iam:AddClientIDToOpenIDConnectProvider',
                        'iam:RemoveClientIDFromOpenIDConnectProvider',
                    ],
                },
            ],
        });
    }
}
exports.OpenIdConnectProvider = OpenIdConnectProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib2lkYy1wcm92aWRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm9pZGMtcHJvdmlkZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBQzdCLHdDQVN1QjtBQUd2QixNQUFNLGFBQWEsR0FBRyxxQ0FBcUMsQ0FBQzs7Ozs7Ozs7Ozs7Ozs7QUE4RjVELE1BQWEscUJBQXNCLFNBQVEsZUFBUTs7Ozs7Ozs7O0lBdUNqRCxZQUFtQixLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFpQztRQUNoRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sUUFBUSxHQUFHLElBQUkscUJBQWMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3BELFlBQVksRUFBRSxhQUFhO1lBQzNCLFlBQVksRUFBRSxJQUFJLENBQUMsbUJBQW1CLEVBQUU7WUFDeEMsVUFBVSxFQUFFO2dCQUNWLFlBQVksRUFBRSxLQUFLLENBQUMsU0FBUztnQkFDN0IsY0FBYyxFQUFFLEtBQUssQ0FBQyxXQUFXO2dCQUNqQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7YUFDZjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyx3QkFBd0IsR0FBRyxZQUFLLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUM3RCxJQUFJLENBQUMsMkJBQTJCLEdBQUcsU0FBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsU0FBRSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsd0JBQXdCLENBQUMsQ0FBQyxDQUFDO0lBQzdHLENBQUM7Ozs7Ozs7OztJQS9DTSxNQUFNLENBQUMsNEJBQTRCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsd0JBQWdDO1FBQ3ZHLE1BQU0sa0JBQWtCLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsd0JBQXdCLENBQUMsQ0FBQyxZQUFZLENBQUM7UUFDM0YsSUFBSSxDQUFDLGtCQUFrQixFQUFFO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0JBQWdCLHdCQUF3QixnQ0FBZ0MsQ0FBQyxDQUFDO1NBQzNGO1FBRUQsK0RBQStEO1FBQy9ELHNFQUFzRTtRQUN0RSwyQkFBMkI7UUFDM0IsTUFBTSxZQUFZLEdBQUcsa0JBQWtCLENBQUM7UUFFeEMsTUFBTSxNQUFPLFNBQVEsZUFBUTtZQUE3Qjs7Z0JBQ2tCLDZCQUF3QixHQUFHLHdCQUF3QixDQUFDO2dCQUNwRCxnQ0FBMkIsR0FBRyxZQUFZLENBQUM7WUFDN0QsQ0FBQztTQUFBO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQWdDTyxtQkFBbUI7UUFDekIsT0FBTyw2QkFBc0IsQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUM3RCxhQUFhLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsZUFBZSxDQUFDO1lBQ3BELE9BQU8sRUFBRSxvQ0FBNkIsQ0FBQyxTQUFTO1lBQ2hELGdCQUFnQixFQUFFO2dCQUNoQjtvQkFDRSxNQUFNLEVBQUUsT0FBTztvQkFDZixRQUFRLEVBQUUsR0FBRztvQkFDYixNQUFNLEVBQUU7d0JBQ04saUNBQWlDO3dCQUNqQyxpQ0FBaUM7d0JBQ2pDLDJDQUEyQzt3QkFDM0Msd0NBQXdDO3dCQUN4Qyw2Q0FBNkM7cUJBQzlDO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUEzRUQsc0RBMkVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7XG4gIEN1c3RvbVJlc291cmNlLFxuICBDdXN0b21SZXNvdXJjZVByb3ZpZGVyLFxuICBDdXN0b21SZXNvdXJjZVByb3ZpZGVyUnVudGltZSxcbiAgSVJlc291cmNlLFxuICBSZXNvdXJjZSxcbiAgU3RhY2ssXG4gIFRva2VuLFxuICBGbixcbn0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuY29uc3QgUkVTT1VSQ0VfVFlQRSA9ICdDdXN0b206OkFXU0NES09wZW5JZENvbm5lY3RQcm92aWRlcic7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSU9wZW5JZENvbm5lY3RQcm92aWRlciBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBvcGVuSWRDb25uZWN0UHJvdmlkZXJBcm46IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG9wZW5JZENvbm5lY3RQcm92aWRlcklzc3Vlcjogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIE9wZW5JZENvbm5lY3RQcm92aWRlclByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHVybDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNsaWVudElkcz86IHN0cmluZ1tdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdGh1bWJwcmludHM/OiBzdHJpbmdbXTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIE9wZW5JZENvbm5lY3RQcm92aWRlciBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSU9wZW5JZENvbm5lY3RQcm92aWRlciB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21PcGVuSWRDb25uZWN0UHJvdmlkZXJBcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgb3BlbklkQ29ubmVjdFByb3ZpZGVyQXJuOiBzdHJpbmcpOiBJT3BlbklkQ29ubmVjdFByb3ZpZGVyIHtcbiAgICBjb25zdCBwYXJzZWRSZXNvdXJjZU5hbWUgPSBTdGFjay5vZihzY29wZSkucGFyc2VBcm4ob3BlbklkQ29ubmVjdFByb3ZpZGVyQXJuKS5yZXNvdXJjZU5hbWU7XG4gICAgaWYgKCFwYXJzZWRSZXNvdXJjZU5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBhcm46ICR7b3BlbklkQ29ubmVjdFByb3ZpZGVyQXJufS4gVW5hYmxlIHRvIGV4dHJhY3QgaXNzdWVyIHVybGApO1xuICAgIH1cblxuICAgIC8vIHRoaXMgbmVlZGVkIGJlY2F1c2UgVFMgZG9uJ3QgdW5kZXJzdGFuZCB0aGF0IHByZXYuIGNvbmRpdGlvblxuICAgIC8vIGFjdHVhbGx5IGRvZXMgbXV0YXRlIHRoZSB0eXBlIGZyb20gXCJzdHJpbmcgfCB1bmRlZmluZWRcIiB0byBcInN0cmluZ1wiXG4gICAgLy8gaW5zaWRlIGNsYXNzIGRlZmluaXRpb24sXG4gICAgY29uc3QgcmVzb3VyY2VOYW1lID0gcGFyc2VkUmVzb3VyY2VOYW1lO1xuXG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJT3BlbklkQ29ubmVjdFByb3ZpZGVyIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBvcGVuSWRDb25uZWN0UHJvdmlkZXJBcm4gPSBvcGVuSWRDb25uZWN0UHJvdmlkZXJBcm47XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgb3BlbklkQ29ubmVjdFByb3ZpZGVySXNzdWVyID0gcmVzb3VyY2VOYW1lO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IG9wZW5JZENvbm5lY3RQcm92aWRlckFybjogc3RyaW5nO1xuXG4gIHB1YmxpYyByZWFkb25seSBvcGVuSWRDb25uZWN0UHJvdmlkZXJJc3N1ZXI6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IE9wZW5JZENvbm5lY3RQcm92aWRlclByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IEN1c3RvbVJlc291cmNlKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIHJlc291cmNlVHlwZTogUkVTT1VSQ0VfVFlQRSxcbiAgICAgIHNlcnZpY2VUb2tlbjogdGhpcy5nZXRPckNyZWF0ZVByb3ZpZGVyKCksXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIENsaWVudElETGlzdDogcHJvcHMuY2xpZW50SWRzLFxuICAgICAgICBUaHVtYnByaW50TGlzdDogcHJvcHMudGh1bWJwcmludHMsXG4gICAgICAgIFVybDogcHJvcHMudXJsLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMub3BlbklkQ29ubmVjdFByb3ZpZGVyQXJuID0gVG9rZW4uYXNTdHJpbmcocmVzb3VyY2UucmVmKTtcbiAgICB0aGlzLm9wZW5JZENvbm5lY3RQcm92aWRlcklzc3VlciA9IEZuLnNlbGVjdCgxLCBGbi5zcGxpdCgnb2lkYy1wcm92aWRlci8nLCB0aGlzLm9wZW5JZENvbm5lY3RQcm92aWRlckFybikpO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXRPckNyZWF0ZVByb3ZpZGVyKCkge1xuICAgIHJldHVybiBDdXN0b21SZXNvdXJjZVByb3ZpZGVyLmdldE9yQ3JlYXRlKHRoaXMsIFJFU09VUkNFX1RZUEUsIHtcbiAgICAgIGNvZGVEaXJlY3Rvcnk6IHBhdGguam9pbihfX2Rpcm5hbWUsICdvaWRjLXByb3ZpZGVyJyksXG4gICAgICBydW50aW1lOiBDdXN0b21SZXNvdXJjZVByb3ZpZGVyUnVudGltZS5OT0RFSlNfMTIsXG4gICAgICBwb2xpY3lTdGF0ZW1lbnRzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgUmVzb3VyY2U6ICcqJyxcbiAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICdpYW06Q3JlYXRlT3BlbklEQ29ubmVjdFByb3ZpZGVyJyxcbiAgICAgICAgICAgICdpYW06RGVsZXRlT3BlbklEQ29ubmVjdFByb3ZpZGVyJyxcbiAgICAgICAgICAgICdpYW06VXBkYXRlT3BlbklEQ29ubmVjdFByb3ZpZGVyVGh1bWJwcmludCcsXG4gICAgICAgICAgICAnaWFtOkFkZENsaWVudElEVG9PcGVuSURDb25uZWN0UHJvdmlkZXInLFxuICAgICAgICAgICAgJ2lhbTpSZW1vdmVDbGllbnRJREZyb21PcGVuSURDb25uZWN0UHJvdmlkZXInLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pO1xuICB9XG59XG4iXX0=