"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerImageAsset = void 0;
const fs = require("fs");
const path = require("path");
const assets = require("@aws-cdk/assets");
const ecr = require("@aws-cdk/aws-ecr");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
/**
 * (experimental) An asset that represents a Docker image.
 *
 * The image will be created in build time and uploaded to an ECR repository.
 *
 * @experimental
 */
class DockerImageAsset extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // none of the properties use tokens
        validateProps(props);
        // resolve full path
        const dir = path.resolve(props.directory);
        if (!fs.existsSync(dir)) {
            throw new Error(`Cannot find image directory at ${dir}`);
        }
        // validate the docker file exists
        const file = path.join(dir, props.file || 'Dockerfile');
        if (!fs.existsSync(file)) {
            throw new Error(`Cannot find file at ${file}`);
        }
        const defaultIgnoreMode = core_1.FeatureFlags.of(this).isEnabled(cxapi.DOCKER_IGNORE_SUPPORT)
            ? core_1.IgnoreMode.DOCKER : core_1.IgnoreMode.GLOB;
        let ignoreMode = (_a = props.ignoreMode) !== null && _a !== void 0 ? _a : defaultIgnoreMode;
        let exclude = props.exclude || [];
        const ignore = path.join(dir, '.dockerignore');
        if (fs.existsSync(ignore)) {
            const dockerIgnorePatterns = fs.readFileSync(ignore).toString().split('\n').filter(e => !!e);
            exclude = [
                ...dockerIgnorePatterns,
                ...exclude,
                // Ensure .dockerignore is whitelisted no matter what.
                '!.dockerignore',
            ];
        }
        // Ensure the Dockerfile is whitelisted no matter what.
        exclude.push('!' + path.basename(file));
        if (props.repositoryName) {
            core_1.Annotations.of(this).addWarning('DockerImageAsset.repositoryName is deprecated. Override "core.Stack.addDockerImageAsset" to control asset locations');
        }
        // include build context in "extra" so it will impact the hash
        const extraHash = {};
        if (props.extraHash) {
            extraHash.user = props.extraHash;
        }
        if (props.buildArgs) {
            extraHash.buildArgs = props.buildArgs;
        }
        if (props.target) {
            extraHash.target = props.target;
        }
        if (props.file) {
            extraHash.file = props.file;
        }
        if (props.repositoryName) {
            extraHash.repositoryName = props.repositoryName;
        }
        // add "salt" to the hash in order to invalidate the image in the upgrade to
        // 1.21.0 which removes the AdoptedRepository resource (and will cause the
        // deletion of the ECR repository the app used).
        extraHash.version = '1.21.0';
        const staging = new assets.Staging(this, 'Staging', {
            ...props,
            exclude,
            ignoreMode,
            sourcePath: dir,
            extraHash: Object.keys(extraHash).length === 0
                ? undefined
                : JSON.stringify(extraHash),
        });
        this.sourceHash = staging.sourceHash;
        const stack = core_1.Stack.of(this);
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: staging.relativeStagedPath(stack),
            dockerBuildArgs: props.buildArgs,
            dockerBuildTarget: props.target,
            dockerFile: props.file,
            repositoryName: props.repositoryName,
            sourceHash: staging.sourceHash,
        });
        this.repository = ecr.Repository.fromRepositoryName(this, 'Repository', location.repositoryName);
        this.imageUri = location.imageUri;
    }
}
exports.DockerImageAsset = DockerImageAsset;
function validateProps(props) {
    for (const [key, value] of Object.entries(props)) {
        if (core_1.Token.isUnresolved(value)) {
            throw new Error(`Cannot use Token as value of '${key}': this value is used before deployment starts`);
        }
    }
    validateBuildArgs(props.buildArgs);
}
function validateBuildArgs(buildArgs) {
    for (const [key, value] of Object.entries(buildArgs || {})) {
        if (core_1.Token.isUnresolved(key) || core_1.Token.isUnresolved(value)) {
            throw new Error('Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW1hZ2UtYXNzZXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbWFnZS1hc3NldC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLDBDQUEwQztBQUMxQyx3Q0FBd0M7QUFDeEMsd0NBQWdIO0FBQ2hILHlDQUF5Qzs7Ozs7Ozs7QUE4RHpDLE1BQWEsZ0JBQWlCLFNBQVEsZ0JBQWE7Ozs7SUFjakQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE0Qjs7UUFDcEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixvQ0FBb0M7UUFDcEMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXJCLG9CQUFvQjtRQUNwQixNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUMxQyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUN2QixNQUFNLElBQUksS0FBSyxDQUFDLGtDQUFrQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1NBQzFEO1FBRUQsa0NBQWtDO1FBQ2xDLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLElBQUksWUFBWSxDQUFDLENBQUM7UUFDeEQsSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsSUFBSSxFQUFFLENBQUMsQ0FBQztTQUNoRDtRQUVELE1BQU0saUJBQWlCLEdBQUcsbUJBQVksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQztZQUNwRixDQUFDLENBQUMsaUJBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLGlCQUFVLENBQUMsSUFBSSxDQUFDO1FBQ3hDLElBQUksVUFBVSxTQUFHLEtBQUssQ0FBQyxVQUFVLG1DQUFJLGlCQUFpQixDQUFDO1FBRXZELElBQUksT0FBTyxHQUFhLEtBQUssQ0FBQyxPQUFPLElBQUksRUFBRSxDQUFDO1FBRTVDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBRS9DLElBQUksRUFBRSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUN6QixNQUFNLG9CQUFvQixHQUFHLEVBQUUsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUU3RixPQUFPLEdBQUc7Z0JBQ1IsR0FBRyxvQkFBb0I7Z0JBQ3ZCLEdBQUcsT0FBTztnQkFFVixzREFBc0Q7Z0JBQ3RELGdCQUFnQjthQUNqQixDQUFDO1NBQ0g7UUFFRCx1REFBdUQ7UUFDdkQsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBRXhDLElBQUksS0FBSyxDQUFDLGNBQWMsRUFBRTtZQUN4QixrQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMscUhBQXFILENBQUMsQ0FBQztTQUN4SjtRQUVELDhEQUE4RDtRQUM5RCxNQUFNLFNBQVMsR0FBNkIsRUFBRyxDQUFDO1FBQ2hELElBQUksS0FBSyxDQUFDLFNBQVMsRUFBRTtZQUFFLFNBQVMsQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztTQUFFO1FBQzFELElBQUksS0FBSyxDQUFDLFNBQVMsRUFBRTtZQUFFLFNBQVMsQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztTQUFFO1FBQy9ELElBQUksS0FBSyxDQUFDLE1BQU0sRUFBRTtZQUFFLFNBQVMsQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQztTQUFFO1FBQ3RELElBQUksS0FBSyxDQUFDLElBQUksRUFBRTtZQUFFLFNBQVMsQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztTQUFFO1FBQ2hELElBQUksS0FBSyxDQUFDLGNBQWMsRUFBRTtZQUFFLFNBQVMsQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztTQUFFO1FBRTlFLDRFQUE0RTtRQUM1RSwwRUFBMEU7UUFDMUUsZ0RBQWdEO1FBQ2hELFNBQVMsQ0FBQyxPQUFPLEdBQUcsUUFBUSxDQUFDO1FBRTdCLE1BQU0sT0FBTyxHQUFHLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ2xELEdBQUcsS0FBSztZQUNSLE9BQU87WUFDUCxVQUFVO1lBQ1YsVUFBVSxFQUFFLEdBQUc7WUFDZixTQUFTLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQztnQkFDNUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQ1gsQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDO1NBQzlCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztRQUVyQyxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdCLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUMsbUJBQW1CLENBQUM7WUFDckQsYUFBYSxFQUFFLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUM7WUFDaEQsZUFBZSxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQ2hDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxNQUFNO1lBQy9CLFVBQVUsRUFBRSxLQUFLLENBQUMsSUFBSTtZQUN0QixjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsVUFBVSxFQUFFLE9BQU8sQ0FBQyxVQUFVO1NBQy9CLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxVQUFVLEdBQUcsR0FBRyxDQUFDLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFLFFBQVEsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUNqRyxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUM7SUFDcEMsQ0FBQztDQUNGO0FBakdELDRDQWlHQztBQUVELFNBQVMsYUFBYSxDQUFDLEtBQTRCO0lBQ2pELEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFO1FBQ2hELElBQUksWUFBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxHQUFHLGdEQUFnRCxDQUFDLENBQUM7U0FDdkc7S0FDRjtJQUVELGlCQUFpQixDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztBQUNyQyxDQUFDO0FBRUQsU0FBUyxpQkFBaUIsQ0FBQyxTQUFxQztJQUM5RCxLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxTQUFTLElBQUksRUFBRSxDQUFDLEVBQUU7UUFDMUQsSUFBSSxZQUFLLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDeEQsTUFBTSxJQUFJLEtBQUssQ0FBQyw0RkFBNEYsQ0FBQyxDQUFDO1NBQy9HO0tBQ0Y7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGFzc2V0cyBmcm9tICdAYXdzLWNkay9hc3NldHMnO1xuaW1wb3J0ICogYXMgZWNyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3InO1xuaW1wb3J0IHsgQW5ub3RhdGlvbnMsIENvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0LCBGZWF0dXJlRmxhZ3MsIElnbm9yZU1vZGUsIFN0YWNrLCBUb2tlbiB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgY3hhcGkgZnJvbSAnQGF3cy1jZGsvY3gtYXBpJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgRG9ja2VySW1hZ2VBc3NldE9wdGlvbnMgZXh0ZW5kcyBhc3NldHMuRmluZ2VycHJpbnRPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVwb3NpdG9yeU5hbWU/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYnVpbGRBcmdzPzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0YXJnZXQ/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZmlsZT86IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgRG9ja2VySW1hZ2VBc3NldFByb3BzIGV4dGVuZHMgRG9ja2VySW1hZ2VBc3NldE9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkaXJlY3Rvcnk6IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgRG9ja2VySW1hZ2VBc3NldCBleHRlbmRzIENvcmVDb25zdHJ1Y3QgaW1wbGVtZW50cyBhc3NldHMuSUFzc2V0IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBpbWFnZVVyaTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVwb3NpdG9yeTogZWNyLklSZXBvc2l0b3J5O1xuXG4gIHB1YmxpYyByZWFkb25seSBzb3VyY2VIYXNoOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IERvY2tlckltYWdlQXNzZXRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICAvLyBub25lIG9mIHRoZSBwcm9wZXJ0aWVzIHVzZSB0b2tlbnNcbiAgICB2YWxpZGF0ZVByb3BzKHByb3BzKTtcblxuICAgIC8vIHJlc29sdmUgZnVsbCBwYXRoXG4gICAgY29uc3QgZGlyID0gcGF0aC5yZXNvbHZlKHByb3BzLmRpcmVjdG9yeSk7XG4gICAgaWYgKCFmcy5leGlzdHNTeW5jKGRpcikpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ2Fubm90IGZpbmQgaW1hZ2UgZGlyZWN0b3J5IGF0ICR7ZGlyfWApO1xuICAgIH1cblxuICAgIC8vIHZhbGlkYXRlIHRoZSBkb2NrZXIgZmlsZSBleGlzdHNcbiAgICBjb25zdCBmaWxlID0gcGF0aC5qb2luKGRpciwgcHJvcHMuZmlsZSB8fCAnRG9ja2VyZmlsZScpO1xuICAgIGlmICghZnMuZXhpc3RzU3luYyhmaWxlKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3QgZmluZCBmaWxlIGF0ICR7ZmlsZX1gKTtcbiAgICB9XG5cbiAgICBjb25zdCBkZWZhdWx0SWdub3JlTW9kZSA9IEZlYXR1cmVGbGFncy5vZih0aGlzKS5pc0VuYWJsZWQoY3hhcGkuRE9DS0VSX0lHTk9SRV9TVVBQT1JUKVxuICAgICAgPyBJZ25vcmVNb2RlLkRPQ0tFUiA6IElnbm9yZU1vZGUuR0xPQjtcbiAgICBsZXQgaWdub3JlTW9kZSA9IHByb3BzLmlnbm9yZU1vZGUgPz8gZGVmYXVsdElnbm9yZU1vZGU7XG5cbiAgICBsZXQgZXhjbHVkZTogc3RyaW5nW10gPSBwcm9wcy5leGNsdWRlIHx8IFtdO1xuXG4gICAgY29uc3QgaWdub3JlID0gcGF0aC5qb2luKGRpciwgJy5kb2NrZXJpZ25vcmUnKTtcblxuICAgIGlmIChmcy5leGlzdHNTeW5jKGlnbm9yZSkpIHtcbiAgICAgIGNvbnN0IGRvY2tlcklnbm9yZVBhdHRlcm5zID0gZnMucmVhZEZpbGVTeW5jKGlnbm9yZSkudG9TdHJpbmcoKS5zcGxpdCgnXFxuJykuZmlsdGVyKGUgPT4gISFlKTtcblxuICAgICAgZXhjbHVkZSA9IFtcbiAgICAgICAgLi4uZG9ja2VySWdub3JlUGF0dGVybnMsXG4gICAgICAgIC4uLmV4Y2x1ZGUsXG5cbiAgICAgICAgLy8gRW5zdXJlIC5kb2NrZXJpZ25vcmUgaXMgd2hpdGVsaXN0ZWQgbm8gbWF0dGVyIHdoYXQuXG4gICAgICAgICchLmRvY2tlcmlnbm9yZScsXG4gICAgICBdO1xuICAgIH1cblxuICAgIC8vIEVuc3VyZSB0aGUgRG9ja2VyZmlsZSBpcyB3aGl0ZWxpc3RlZCBubyBtYXR0ZXIgd2hhdC5cbiAgICBleGNsdWRlLnB1c2goJyEnICsgcGF0aC5iYXNlbmFtZShmaWxlKSk7XG5cbiAgICBpZiAocHJvcHMucmVwb3NpdG9yeU5hbWUpIHtcbiAgICAgIEFubm90YXRpb25zLm9mKHRoaXMpLmFkZFdhcm5pbmcoJ0RvY2tlckltYWdlQXNzZXQucmVwb3NpdG9yeU5hbWUgaXMgZGVwcmVjYXRlZC4gT3ZlcnJpZGUgXCJjb3JlLlN0YWNrLmFkZERvY2tlckltYWdlQXNzZXRcIiB0byBjb250cm9sIGFzc2V0IGxvY2F0aW9ucycpO1xuICAgIH1cblxuICAgIC8vIGluY2x1ZGUgYnVpbGQgY29udGV4dCBpbiBcImV4dHJhXCIgc28gaXQgd2lsbCBpbXBhY3QgdGhlIGhhc2hcbiAgICBjb25zdCBleHRyYUhhc2g6IHsgW2ZpZWxkOiBzdHJpbmddOiBhbnkgfSA9IHsgfTtcbiAgICBpZiAocHJvcHMuZXh0cmFIYXNoKSB7IGV4dHJhSGFzaC51c2VyID0gcHJvcHMuZXh0cmFIYXNoOyB9XG4gICAgaWYgKHByb3BzLmJ1aWxkQXJncykgeyBleHRyYUhhc2guYnVpbGRBcmdzID0gcHJvcHMuYnVpbGRBcmdzOyB9XG4gICAgaWYgKHByb3BzLnRhcmdldCkgeyBleHRyYUhhc2gudGFyZ2V0ID0gcHJvcHMudGFyZ2V0OyB9XG4gICAgaWYgKHByb3BzLmZpbGUpIHsgZXh0cmFIYXNoLmZpbGUgPSBwcm9wcy5maWxlOyB9XG4gICAgaWYgKHByb3BzLnJlcG9zaXRvcnlOYW1lKSB7IGV4dHJhSGFzaC5yZXBvc2l0b3J5TmFtZSA9IHByb3BzLnJlcG9zaXRvcnlOYW1lOyB9XG5cbiAgICAvLyBhZGQgXCJzYWx0XCIgdG8gdGhlIGhhc2ggaW4gb3JkZXIgdG8gaW52YWxpZGF0ZSB0aGUgaW1hZ2UgaW4gdGhlIHVwZ3JhZGUgdG9cbiAgICAvLyAxLjIxLjAgd2hpY2ggcmVtb3ZlcyB0aGUgQWRvcHRlZFJlcG9zaXRvcnkgcmVzb3VyY2UgKGFuZCB3aWxsIGNhdXNlIHRoZVxuICAgIC8vIGRlbGV0aW9uIG9mIHRoZSBFQ1IgcmVwb3NpdG9yeSB0aGUgYXBwIHVzZWQpLlxuICAgIGV4dHJhSGFzaC52ZXJzaW9uID0gJzEuMjEuMCc7XG5cbiAgICBjb25zdCBzdGFnaW5nID0gbmV3IGFzc2V0cy5TdGFnaW5nKHRoaXMsICdTdGFnaW5nJywge1xuICAgICAgLi4ucHJvcHMsXG4gICAgICBleGNsdWRlLFxuICAgICAgaWdub3JlTW9kZSxcbiAgICAgIHNvdXJjZVBhdGg6IGRpcixcbiAgICAgIGV4dHJhSGFzaDogT2JqZWN0LmtleXMoZXh0cmFIYXNoKS5sZW5ndGggPT09IDBcbiAgICAgICAgPyB1bmRlZmluZWRcbiAgICAgICAgOiBKU09OLnN0cmluZ2lmeShleHRyYUhhc2gpLFxuICAgIH0pO1xuXG4gICAgdGhpcy5zb3VyY2VIYXNoID0gc3RhZ2luZy5zb3VyY2VIYXNoO1xuXG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcbiAgICBjb25zdCBsb2NhdGlvbiA9IHN0YWNrLnN5bnRoZXNpemVyLmFkZERvY2tlckltYWdlQXNzZXQoe1xuICAgICAgZGlyZWN0b3J5TmFtZTogc3RhZ2luZy5yZWxhdGl2ZVN0YWdlZFBhdGgoc3RhY2spLFxuICAgICAgZG9ja2VyQnVpbGRBcmdzOiBwcm9wcy5idWlsZEFyZ3MsXG4gICAgICBkb2NrZXJCdWlsZFRhcmdldDogcHJvcHMudGFyZ2V0LFxuICAgICAgZG9ja2VyRmlsZTogcHJvcHMuZmlsZSxcbiAgICAgIHJlcG9zaXRvcnlOYW1lOiBwcm9wcy5yZXBvc2l0b3J5TmFtZSxcbiAgICAgIHNvdXJjZUhhc2g6IHN0YWdpbmcuc291cmNlSGFzaCxcbiAgICB9KTtcblxuICAgIHRoaXMucmVwb3NpdG9yeSA9IGVjci5SZXBvc2l0b3J5LmZyb21SZXBvc2l0b3J5TmFtZSh0aGlzLCAnUmVwb3NpdG9yeScsIGxvY2F0aW9uLnJlcG9zaXRvcnlOYW1lKTtcbiAgICB0aGlzLmltYWdlVXJpID0gbG9jYXRpb24uaW1hZ2VVcmk7XG4gIH1cbn1cblxuZnVuY3Rpb24gdmFsaWRhdGVQcm9wcyhwcm9wczogRG9ja2VySW1hZ2VBc3NldFByb3BzKSB7XG4gIGZvciAoY29uc3QgW2tleSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKHByb3BzKSkge1xuICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQodmFsdWUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCB1c2UgVG9rZW4gYXMgdmFsdWUgb2YgJyR7a2V5fSc6IHRoaXMgdmFsdWUgaXMgdXNlZCBiZWZvcmUgZGVwbG95bWVudCBzdGFydHNgKTtcbiAgICB9XG4gIH1cblxuICB2YWxpZGF0ZUJ1aWxkQXJncyhwcm9wcy5idWlsZEFyZ3MpO1xufVxuXG5mdW5jdGlvbiB2YWxpZGF0ZUJ1aWxkQXJncyhidWlsZEFyZ3M/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9KSB7XG4gIGZvciAoY29uc3QgW2tleSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKGJ1aWxkQXJncyB8fCB7fSkpIHtcbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKGtleSkgfHwgVG9rZW4uaXNVbnJlc29sdmVkKHZhbHVlKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgdXNlIHRva2VucyBpbiBrZXlzIG9yIHZhbHVlcyBvZiBcImJ1aWxkQXJnc1wiIHNpbmNlIHRoZXkgYXJlIG5lZWRlZCBiZWZvcmUgZGVwbG95bWVudCcpO1xuICAgIH1cbiAgfVxufVxuIl19