import { IResource, Resource } from 'aws-cdk-lib';
import { CfnDomain } from 'aws-cdk-lib/aws-codeartifact';
import { AddToResourcePolicyResult, Grant, IGrantable, PolicyDocument, PolicyStatement } from 'aws-cdk-lib/aws-iam';
import { IKey } from 'aws-cdk-lib/aws-kms';
import { Construct } from 'constructs';
/**
 * Represents a CodeArtifact Domain
 */
export interface IDomain extends IResource {
    /**
     * The ARN of the Domain
     *
     * @attribute
     */
    readonly domainArn: string;
    /**
     * The name of the Domain
     *
     * @attribute
     */
    readonly domainName: string;
    /**
     * The ARN of the key used to encrypt the Domain
     *
     * @attribute
     */
    readonly domainEncryptionKey?: string;
    /**
     * 12-digit account number of the AWS account that owns the domain that contains the Domain.
     *
     * @attribute
     */
    readonly domainOwner: string;
    /**
     * The KMS key used to encrypt the Domain
     */
    readonly encryptionKey?: IKey;
    /**
     * Adds a statement to the Codeartifact domain resource policy.
     * @param statement The policy statement to add
     */
    addToResourcePolicy(statement: PolicyStatement): AddToResourcePolicyResult;
    /**
     * Grants permissions to the specified grantee on this CodeArtifact domain.
     *
     * It handles both same-environment and cross-environment scenarios:
     * - For same-environment grants, it adds the permissions to the principal or resource.
     * - For cross-environment grants, it adds the permissions to both the principal and the resource.
     *
     * @param grantee - The principal to grant permissions to.
     * @param actions - The actions to grant. These should be valid CodeArtifact actions.
     */
    grant(grantee: IGrantable, ...actions: string[]): Grant;
    /**
     * Grants contribute permissions to the specified grantee on this CodeArtifact domain.
     *
     * @param grantee - The principal to grant contribute permissions to.
     */
    grantContribute(grantee: IGrantable): Grant;
}
/**
 * A new or imported CodeArtifact Domain.
 */
declare abstract class DomainBase extends Resource implements IDomain {
    /**
     * The ARN (Amazon Resource Name) of the CodeArtifact domain.
     */
    abstract readonly domainArn: string;
    /**
     * The name of the CodeArtifact domain.
     */
    abstract readonly domainName: string;
    /**
     * The AWS KMS encryption key associated with the domain, if any.
     */
    abstract readonly encryptionKey?: IKey;
    /**
     * The ARN of the key used to encrypt the Domain
     */
    abstract readonly domainEncryptionKey?: string;
    /**
     * The AWS account ID that owns the domain.
     */
    abstract readonly domainOwner: string;
    protected abstract policy?: PolicyDocument;
    /**
     * Adds a statement to the Codeartifact domain resource policy.
     * @param statement The policy statement to add
     */
    addToResourcePolicy(statement: PolicyStatement): AddToResourcePolicyResult;
    private isCrossEnvironmentGrantee;
    /**
     * Grants permissions to the specified grantee on this CodeArtifact domain.
     *
     * It handles both same-environment and cross-environment scenarios:
     * - For same-environment grants, it adds the permissions to the principal or resource.
     * - For cross-environment grants, it adds the permissions to both the principal and the resource.
     *
     * @param grantee - The principal to grant permissions to.
     * @param actions - The actions to grant. These should be valid CodeArtifact actions.
     */
    grant(grantee: IGrantable, ...actions: string[]): Grant;
    /**
     * Grants contribute permissions to the specified grantee on this CodeArtifact domain.
     *
     * @param grantee - The principal to grant contribute permissions to.
     */
    grantContribute(grantee: IGrantable): Grant;
}
/**
 * Interface representing the attributes of a CodeArtifact domain.
 */
export interface DomainAttributes {
    /**
     * The ARN (Amazon Resource Name) of the CodeArtifact domain.
     */
    readonly domainArn: string;
    /**
     * The name of the CodeArtifact domain.
     */
    readonly domainName: string;
    /**
     * The AWS KMS encryption key associated with the domain, if any.
     */
    readonly encryptionKey?: IKey;
    /**
     * The AWS account ID that owns the domain.
     */
    readonly domainOwner: string;
}
/**
 * Construction properties for `Domain`.
 */
export interface DomainProps {
    /**
     * The name of the Domain
     */
    readonly domainName: string;
    /**
     * The key used to encrypt the Domain
     *
     * @default - An AWS managed KMS key is used
     */
    readonly encryptionKey?: IKey;
}
/**
 * Deploys a CodeArtifact domain.
 */
export declare class Domain extends DomainBase implements IDomain {
    /**
     * Creates a Domain object from existing domain attributes.
     *
     * @param scope The parent construct.
     * @param id The construct id.
     * @param attrs The attributes of the domain to import.
     */
    static fromDomainAttributes(scope: Construct, id: string, attrs: DomainAttributes): IDomain;
    /**
     * Creates an IDomain object from an existing CodeArtifact domain ARN.
     *
     * @param scope The parent construct.
     * @param id The construct id.
     * @param domainArn - The ARN (Amazon Resource Name) of the existing CodeArtifact domain.
     */
    static fromDomainArn(scope: Construct, id: string, domainArn: string): IDomain;
    /**
     * (internal) The CloudFormation resource representing this CodeArtifact domain.
     */
    protected cfnResource: CfnDomain;
    /**
     * The properties used to create the CloudFormation resource for this domain.
     */
    private cfnResourceProps;
    /**
     * The ARN (Amazon Resource Name) of this CodeArtifact domain.
     */
    readonly domainArn: string;
    /**
     * The name of this CodeArtifact domain.
     */
    readonly domainName: string;
    /**
     * The AWS KMS encryption key associated with this domain, if any.
     */
    readonly encryptionKey?: IKey;
    /**
     * The AWS account ID that owns this domain.
     */
    readonly domainOwner: string;
    /**
     * The ARN of the key used to encrypt the Domain
     */
    readonly domainEncryptionKey?: string;
    protected policy?: PolicyDocument;
    constructor(scope: Construct, id: string, props: DomainProps);
    protected createResource(scope: Construct, id: string): CfnDomain;
}
export {};
