import { IResource, Resource } from 'aws-cdk-lib';
import { CfnRepository, CfnRepositoryProps } from 'aws-cdk-lib/aws-codeartifact';
import { AddToResourcePolicyResult, Grant, IGrantable, PolicyDocument, PolicyStatement } from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import { IDomain } from './domain';
/**
 * Represents the supported external connections for CodeArtifact repositories.
 */
export declare enum RepositoryConnection {
    /** Python Package Index (PyPI) */
    PYTHON = "public:pypi",
    /** Node Package Manager (npm) */
    NPM = "public:npmjs",
    /** NuGet Gallery */
    NUGET = "public:nuget-org",
    /** RubyGems */
    RUBY = "public:ruby-gems-org",
    /** Crates.io (Rust) */
    RUST = "public:crates-io",
    /** Maven Central Repository */
    MAVEN_CENTRAL = "public:maven-central",
    /** Gradle Plugins */
    GRADLE_PLUGINS = "public:gradle-plugins",
    /** Maven Google */
    MAVEN_GOOGLE = "public:maven-google",
    /** Maven Apache */
    MAVEN_APACHE = "public:maven-apache",
    /** Maven Atlassian */
    MAVEN_ATLASSIAN = "public:maven-atlassian",
    /** Maven Eclipse */
    MAVEN_ECLIPSE = "public:maven-eclipse",
    /** Maven JBoss */
    MAVEN_JBOSS = "public:maven-jboss",
    /** Maven Spring */
    MAVEN_SPRING = "public:maven-spring",
    /** Maven Spring Plugins */
    MAVEN_SPRING_PLUGINS = "public:maven-spring-plugins"
}
/**
 * Represents an CodeArtifact Repository
 */
export interface IRepository extends IResource {
    /**
     * The ARN of the repository
     *
     * @attribute
     */
    readonly repositoryArn: string;
    /**
     * The name of the repository
     *
     * @attribute
     */
    readonly repositoryName: string;
    /**
     * The domain that contains the repository
     *
     * @attribute
     */
    readonly repositoryDomainName: string;
    /**
     * The domain owner of the repository
     *
     * @attribute
     */
    readonly repositoryDomainOwner: string;
    /**
     * The domain that contains the repository
     *
     */
    readonly domain: IDomain;
    /**
     * Adds a statement to the CodeArtifact repository resource policy.
     *
     * @param statement The policy statement to add
     */
    addToResourcePolicy(statement: PolicyStatement): AddToResourcePolicyResult;
    /**
     * Grants the given principal identity permissions to perform the actions on the repository.
     *
     * @param grantee The principal to grant permissions to
     * @param actions The actions to grant
     */
    grant(grantee: IGrantable, ...actions: string[]): Grant;
    /**
     * Grants the given principal identity permissions to perform the actions on the repository.
     *
     * @param grantee The principal to grant permissions to
     */
    grantReadAndPublish(grantee: IGrantable): Grant;
    /**
     * Grants the given principal identity permissions to perform the actions on the repository.
     *
     * @param grantee The principal to grant permissions to
     */
    grantRead(grantee: IGrantable): Grant;
}
/**
 * Properties for creating a new CodeArtifact repository.
 *
 */
export interface RepositoryProps {
    /**
     * The name of the repository
     *
     * @default - A name is automatically generated
     */
    readonly repositoryName?: string;
    /**
     * The domain that contains the repository
     */
    readonly domain: IDomain;
    /**
     * The description of the repository
     *
     * @default - No description
     */
    readonly description?: string;
    /**
     * The connections to external repositories (like npmjs, pypi, etc.)
     *
     * You can use the AWS CLI to connect your CodeArtifact repository to an external repository by adding an external connection directly to the repository.
     * This will allow users connected to the CodeArtifact repository, or any of its downstream repositories, to fetch packages from the configured external repository.
     * Each CodeArtifact repository can only have one external connection.
     *
     * @default - No external connections
     */
    readonly externalConnection?: RepositoryConnection;
    /**
     * A list of upstream Codeartifact repositories to associate with the repository.
     * The order of the upstream repositories in the list determines their priority order when CodeArtifact looks for a requested package version.
     * see https://docs.aws.amazon.com/codeartifact/latest/ug/repo-upstream-behavior.html#package-retention-intermediate-repositories
     *
     * @default - No upstream repositories
     */
    readonly upstreams?: IRepository[];
}
/**
 * A new or imported Codeartifact Repository.
 */
declare abstract class RepositoryBase extends Resource implements IRepository {
    /**
     * The ARN of the repository
     */
    abstract readonly repositoryArn: string;
    /**
     * The name of the repository
     */
    abstract readonly repositoryName: string;
    /**
     * The domain that contains the repository
     */
    abstract readonly repositoryDomainName: string;
    /**
     * The domain owner of the repository
     */
    abstract readonly repositoryDomainOwner: string;
    /**
     * The domain that contains the repository
     */
    abstract readonly domain: IDomain;
    protected abstract policy?: PolicyDocument;
    private isCrossEnvironmentGrantee;
    /**
     * Adds a statement to the CodeArtifact repository resource policy.
     * @param statement The policy statement to add
     */
    addToResourcePolicy(statement: PolicyStatement): AddToResourcePolicyResult;
    /**
     * Grants permissions to the specified grantee on this CodeArtifact repository.
     *
     * @param grantee The principal to grant permissions to
     * @param actions The actions to grant
     */
    grant(grantee: IGrantable, ...actions: string[]): Grant;
    /**
     * Grants read permissions to the specified grantee on this CodeArtifact repository.
     *
     * @param grantee The principal to grant read permissions to
     */
    grantRead(grantee: IGrantable): Grant;
    /**
     * Grants read and publish permissions to the specified grantee on this CodeArtifact repository.
     *
     * @param grantee The principal to grant read and publish permissions to
     */
    grantReadAndPublish(grantee: IGrantable): Grant;
}
/**
 * Represents the attributes of an existing CodeArtifact repository.
 */
export interface RepositoryAttributes {
    /**
     * The ARN (Amazon Resource Name) of the CodeArtifact repository.
     */
    readonly repositoryArn: string;
    /**
     * The name of the CodeArtifact repository.
     */
    readonly repositoryName: string;
    /**
     * The CodeArtifact domain associated with this repository.
     */
    readonly domain: IDomain;
}
/**
 * Deploys a CodeArtifact repository
 */
export declare class Repository extends RepositoryBase implements IRepository {
    /**
     * Creates an IRepository object from existing repository attributes.
     *
     * @param scope - The parent construct in which to create this repository reference.
     * @param id - The identifier of the construct.
     * @param attrs - The attributes of the repository to import.
     */
    static fromRepositoryAttributes(scope: Construct, id: string, attrs: RepositoryAttributes): IRepository;
    /**
     * Creates an IRepository object from an existing repository ARN.
     *
     * @param scope - The parent construct in which to create this repository reference.
     * @param id - The identifier of the construct.
     * @param repositoryArn - The ARN of the repository to import.
     */
    static fromRepositoryArn(scope: Construct, id: string, repositoryArn: string): IRepository;
    /**
     * (internal) The CloudFormation resource representing this CodeArtifact repository.
     */
    protected cfnResource: CfnRepository;
    /**
     * The properties used to create the CloudFormation resource for this repository.
     */
    protected cfnResourceProps: CfnRepositoryProps;
    /**
     * The ARN (Amazon Resource Name) of this CodeArtifact repository.
     */
    readonly repositoryArn: string;
    /**
     * The name of this CodeArtifact repository.
     */
    readonly repositoryName: string;
    /**
     * The domain that contains this repository.
     */
    readonly repositoryDomainName: string;
    /**
     * The domain owner of this repository.
     */
    readonly repositoryDomainOwner: string;
    /**
     * The domain that contains this repository.
     */
    readonly domain: IDomain;
    protected readonly policy?: PolicyDocument;
    protected readonly upstreams: IRepository[];
    constructor(scope: Construct, id: string, props: RepositoryProps);
    protected createResource(scope: Construct, id: string): CfnRepository;
    protected renderUpstreams(): string[];
}
export {};
