"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = exports.RepositoryConnection = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codeartifact_1 = require("aws-cdk-lib/aws-codeartifact");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const domain_1 = require("./domain");
const READ_ACTIONS = [
    'codeartifact:DescribePackageVersion',
    'codeartifact:DescribeRepository',
    'codeartifact:GetPackageVersionReadme',
    'codeartifact:GetRepositoryEndpoint',
    'codeartifact:ListPackageVersionAssets',
    'codeartifact:ListPackageVersionDependencies',
    'codeartifact:ListPackageVersions',
    'codeartifact:ListPackages',
    'codeartifact:ReadFromRepository',
];
const PUBLISH_ACTIONS = ['codeartifact:PublishPackageVersion', 'codeartifact:PutPackageMetadata'];
/**
 * Represents the supported external connections for CodeArtifact repositories.
 */
var RepositoryConnection;
(function (RepositoryConnection) {
    /** Python Package Index (PyPI) */
    RepositoryConnection["PYTHON"] = "public:pypi";
    /** Node Package Manager (npm) */
    RepositoryConnection["NPM"] = "public:npmjs";
    /** NuGet Gallery */
    RepositoryConnection["NUGET"] = "public:nuget-org";
    /** RubyGems */
    RepositoryConnection["RUBY"] = "public:ruby-gems-org";
    /** Crates.io (Rust) */
    RepositoryConnection["RUST"] = "public:crates-io";
    /** Maven Central Repository */
    RepositoryConnection["MAVEN_CENTRAL"] = "public:maven-central";
    /** Gradle Plugins */
    RepositoryConnection["GRADLE_PLUGINS"] = "public:gradle-plugins";
    /** Maven Google */
    RepositoryConnection["MAVEN_GOOGLE"] = "public:maven-google";
    /** Maven Apache */
    RepositoryConnection["MAVEN_APACHE"] = "public:maven-apache";
    /** Maven Atlassian */
    RepositoryConnection["MAVEN_ATLASSIAN"] = "public:maven-atlassian";
    /** Maven Eclipse */
    RepositoryConnection["MAVEN_ECLIPSE"] = "public:maven-eclipse";
    /** Maven JBoss */
    RepositoryConnection["MAVEN_JBOSS"] = "public:maven-jboss";
    /** Maven Spring */
    RepositoryConnection["MAVEN_SPRING"] = "public:maven-spring";
    /** Maven Spring Plugins */
    RepositoryConnection["MAVEN_SPRING_PLUGINS"] = "public:maven-spring-plugins";
})(RepositoryConnection || (exports.RepositoryConnection = RepositoryConnection = {}));
/**
 * A new or imported Codeartifact Repository.
 */
class RepositoryBase extends aws_cdk_lib_1.Resource {
    isCrossEnvironmentGrantee(grantee) {
        if (!(0, aws_iam_1.principalIsOwnedResource)(grantee.grantPrincipal)) {
            return false;
        }
        const thisStack = aws_cdk_lib_1.Stack.of(this);
        const identityStack = aws_cdk_lib_1.Stack.of(grantee.grantPrincipal);
        return (aws_cdk_lib_1.Token.compareStrings(thisStack.region, identityStack.region) === aws_cdk_lib_1.TokenComparison.DIFFERENT ||
            aws_cdk_lib_1.Token.compareStrings(thisStack.account, identityStack.account) === aws_cdk_lib_1.TokenComparison.DIFFERENT);
    }
    /**
     * Adds a statement to the CodeArtifact repository resource policy.
     * @param statement The policy statement to add
     */
    addToResourcePolicy(statement) {
        if (!aws_cdk_lib_1.Resource.isOwnedResource(this)) {
            aws_cdk_lib_1.Annotations.of(this).addWarningV2('NoResourcePolicyStatementAdded', `No statements added to imported resource ${this.repositoryArn}.`);
            return { statementAdded: false };
        }
        if (!this.policy) {
            this.policy = new aws_iam_1.PolicyDocument();
        }
        this.policy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.policy };
    }
    /**
     * Grants permissions to the specified grantee on this CodeArtifact repository.
     *
     * @param grantee The principal to grant permissions to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        const crossEnvironment = this.isCrossEnvironmentGrantee(grantee);
        const grantOptions = {
            grantee,
            actions,
            resource: this,
            resourceArns: [this.repositoryArn],
            resourceSelfArns: crossEnvironment ? undefined : ['*'],
        };
        if (!crossEnvironment) {
            return aws_iam_1.Grant.addToPrincipalOrResource(grantOptions);
        }
        else {
            return aws_iam_1.Grant.addToPrincipalAndResource({
                ...grantOptions,
                resourceArns: [this.repositoryArn],
                resourcePolicyPrincipal: grantee.grantPrincipal,
            });
        }
    }
    /**
     * Grants read permissions to the specified grantee on this CodeArtifact repository.
     *
     * @param grantee The principal to grant read permissions to
     */
    grantRead(grantee) {
        return this.grant(grantee, ...READ_ACTIONS);
    }
    /**
     * Grants read and publish permissions to the specified grantee on this CodeArtifact repository.
     *
     * @param grantee The principal to grant read and publish permissions to
     */
    grantReadAndPublish(grantee) {
        return this.grant(grantee, ...READ_ACTIONS, ...PUBLISH_ACTIONS);
    }
}
/**
 * Deploys a CodeArtifact repository
 */
class Repository extends RepositoryBase {
    /**
     * Creates an IRepository object from existing repository attributes.
     *
     * @param scope - The parent construct in which to create this repository reference.
     * @param id - The identifier of the construct.
     * @param attrs - The attributes of the repository to import.
     */
    static fromRepositoryAttributes(scope, id, attrs) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryArn = attrs.repositoryArn;
                this.repositoryName = attrs.repositoryName;
                this.repositoryDomainName = attrs.domain.domainName;
                this.repositoryDomainOwner = attrs.domain.domainOwner;
                this.domain = attrs.domain;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates an IRepository object from an existing repository ARN.
     *
     * @param scope - The parent construct in which to create this repository reference.
     * @param id - The identifier of the construct.
     * @param repositoryArn - The ARN of the repository to import.
     */
    static fromRepositoryArn(scope, id, repositoryArn) {
        const repositoryResourceArnParts = aws_cdk_lib_1.Stack.of(scope).splitArn(repositoryArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (repositoryResourceArnParts.resource !== 'repository' ||
            repositoryResourceArnParts.account === '' ||
            repositoryResourceArnParts.resourceName === undefined) {
            throw new Error(`Expected a repository ARN, but got ${repositoryArn}`);
        }
        const repositoryNameParts = repositoryResourceArnParts.resourceName.split('/');
        if (repositoryNameParts.length !== 2) {
            throw new Error(`Expected a repository ARN with a domain and repository name (arn:aws:codeartifact:region-ID:account-ID:repository/my_domain/my_repo), but got ${repositoryArn}`);
        }
        const domainName = repositoryNameParts[0];
        const repositoryName = repositoryNameParts[1];
        const domain = domain_1.Domain.fromDomainArn(scope, 'Domain', aws_cdk_lib_1.Stack.of(scope).formatArn({
            resource: 'domain',
            service: 'codeartifact',
            resourceName: domainName,
        }));
        return Repository.fromRepositoryAttributes(scope, id, {
            repositoryArn: repositoryArn,
            repositoryName: repositoryName,
            domain,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.upstreams = props.upstreams ?? [];
        this.cfnResourceProps = {
            domainName: props.domain.domainName,
            repositoryName: props.repositoryName ??
                aws_cdk_lib_1.Lazy.string({
                    produce: () => aws_cdk_lib_1.Names.uniqueResourceName(this, { maxLength: 100 }),
                }),
            description: props.description,
            domainOwner: props.domain.domainOwner,
            externalConnections: props.externalConnection !== undefined ? [props.externalConnection] : undefined, // only 1 allowed
            permissionsPolicyDocument: aws_cdk_lib_1.Lazy.any({ produce: () => this.policy?.toJSON() }),
            upstreams: aws_cdk_lib_1.Lazy.list({ produce: () => this.renderUpstreams() }, { omitEmpty: true }),
        };
        this.cfnResource = this.createResource(this, 'Resource');
        this.repositoryArn = this.cfnResource.attrArn;
        this.repositoryName = this.cfnResource.attrName;
        this.repositoryDomainName = this.cfnResource.attrDomainName;
        this.repositoryDomainOwner = this.cfnResource.attrDomainOwner;
        this.domain = domain_1.Domain.fromDomainArn(this, 'Domain', aws_cdk_lib_1.Stack.of(this).formatArn({
            resource: 'domain',
            service: 'codeartifact',
            account: this.repositoryDomainOwner,
            resourceName: this.repositoryDomainName,
        }));
    }
    createResource(scope, id) {
        return new aws_codeartifact_1.CfnRepository(scope, id, this.cfnResourceProps);
    }
    renderUpstreams() {
        return this.upstreams.map(repo => repo.repositoryName);
    }
}
exports.Repository = Repository;
_a = JSII_RTTI_SYMBOL_1;
Repository[_a] = { fqn: "@open-constructs/aws-cdk.aws_codeartifact.Repository", version: "0.1.2" };
//# sourceMappingURL=data:application/json;base64,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