"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CostReport = exports.CurFormat = exports.ReportGranularity = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * Enum for the possible granularities of a cost report
 */
class ReportGranularity {
    /**
     * Returns a ReportGranularity instance for the given granularity string value.
     *
     * @param granularity - The granularity string value to create an instance for.
     * @returns A ReportGranularity instance.
     */
    static for(granularity) {
        return new ReportGranularity(granularity);
    }
    constructor(value) {
        this.value = value;
    }
}
exports.ReportGranularity = ReportGranularity;
_a = JSII_RTTI_SYMBOL_1;
ReportGranularity[_a] = { fqn: "@open-constructs/aws-cdk.aws_cur.ReportGranularity", version: "0.1.2" };
/** Hourly granularity */
ReportGranularity.HOURLY = new ReportGranularity('HOURLY');
/** Daily granularity */
ReportGranularity.DAILY = new ReportGranularity('DAILY');
/** Weekly granularity */
ReportGranularity.MONTHLY = new ReportGranularity('MONTHLY');
/**
 * Enum for the possible formats of a cost report
 */
class CurFormat {
    /**
     * Returns a CurFormat instance for the given compression and format string values.
     *
     * @param compression - The compression string value
     * @param format - The format string value
     * @returns A CurFormat instance.
     */
    static for(compression, format) {
        return new CurFormat(compression, format);
    }
    constructor(compression, format) {
        this.compression = compression;
        this.format = format;
    }
}
exports.CurFormat = CurFormat;
_b = JSII_RTTI_SYMBOL_1;
CurFormat[_b] = { fqn: "@open-constructs/aws-cdk.aws_cur.CurFormat", version: "0.1.2" };
/** GZIP compressed text or CSV format */
CurFormat.TEXT_OR_CSV = new CurFormat('GZIP', 'textORcsv');
/** Parquet format */
CurFormat.PARQUET = new CurFormat('Parquet', 'Parquet');
/**
 * Represents a Cost Report construct in AWS CDK.
 * This class creates an AWS Cost and Usage Report, stored in an S3 bucket, and configures the necessary permissions.
 *
 * @example
 * const report = new CostReport(stack, 'MyReport', {
 *   costReportName: 'business-report',
 *   reportGranularity: ReportGranularity.MONTHLY,
 *   format: CurFormat.PARQUET
 * });
 */
class CostReport extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const currentStack = aws_cdk_lib_1.Stack.of(this);
        if (!aws_cdk_lib_1.Token.isUnresolved(currentStack.region) && currentStack.region !== 'us-east-1') {
            throw new Error(`The \`CostReport\` construct is only available in the us-east-1 region, got: ${currentStack.region} region`);
        }
        this.reportBucket =
            props.bucket ??
                this.createReportBucket(this, 'Bucket', {
                    blockPublicAccess: aws_cdk_lib_1.aws_s3.BlockPublicAccess.BLOCK_ALL,
                    enforceSSL: true,
                });
        const billingPrincipal = new aws_cdk_lib_1.aws_iam.ServicePrincipal('billingreports.amazonaws.com').withConditions({
            StringEquals: {
                'aws:SourceArn': `arn:aws:cur:${currentStack.region}:${currentStack.account}:definition/*`,
                'aws:SourceAccount': currentStack.account,
            },
        });
        // Grant the global CUR Account write access to the bucket
        this.reportBucket.grantPut(billingPrincipal);
        this.reportBucket.addToResourcePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            principals: [billingPrincipal],
            actions: ['s3:GetBucketAcl', 's3:GetBucketPolicy'],
            resources: [this.reportBucket.bucketArn],
        }));
        const format = props.format ?? CurFormat.TEXT_OR_CSV;
        if (props.costReportName !== undefined && !aws_cdk_lib_1.Token.isUnresolved(props.costReportName)) {
            if (props.costReportName.length < 1 || props.costReportName.length > 256) {
                throw new Error(`'costReportName' must be between 1 and 256 characters long, got: ${props.costReportName.length}`);
            }
            if (!/^[0-9A-Za-z!\-_.*'()]+$/.test(props.costReportName)) {
                throw new Error(`'costReportName' must only contain alphanumeric characters and the following special characters: !-_.*'(), got: '${props.costReportName}'`);
            }
        }
        const reportName = props.costReportName ??
            (props.enableDefaultUniqueReportName
                ? aws_cdk_lib_1.Names.uniqueResourceName(this, {
                    maxLength: 256,
                    allowedSpecialCharacters: "!-_.*'()",
                })
                : 'default-cur');
        const reportDefinition = this.createReportDefinition(this, 'Resource', {
            compression: format.compression,
            format: format.format,
            refreshClosedReports: false,
            reportName,
            reportVersioning: 'CREATE_NEW_REPORT',
            s3Bucket: this.reportBucket.bucketName,
            s3Prefix: 'reports',
            s3Region: currentStack.region,
            timeUnit: props.reportGranularity?.value ?? 'HOURLY',
            additionalSchemaElements: ['RESOURCES'],
        });
        reportDefinition.node.addDependency(this.reportBucket.policy);
        this.costReportName = reportDefinition.ref;
    }
    createReportBucket(scope, id, props) {
        return new aws_cdk_lib_1.aws_s3.Bucket(scope, id, props);
    }
    createReportDefinition(scope, id, props) {
        return new aws_cdk_lib_1.aws_cur.CfnReportDefinition(scope, id, props);
    }
}
exports.CostReport = CostReport;
_c = JSII_RTTI_SYMBOL_1;
CostReport[_c] = { fqn: "@open-constructs/aws-cdk.aws_cur.CostReport", version: "0.1.2" };
//# sourceMappingURL=data:application/json;base64,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