"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessCache = exports.MajorVersion = exports.ECPUPerSecond = exports.DataStorage = exports.StorageUnit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const util_1 = require("./util");
/**
 * Storage unit for data storage in ElastiCache Serverless.
 */
var StorageUnit;
(function (StorageUnit) {
    /**
     * Gigabytes
     */
    StorageUnit["GB"] = "GB";
})(StorageUnit || (exports.StorageUnit = StorageUnit = {}));
/**
 * The data storage limit.
 */
class DataStorage {
    /**
     * Creates data storage settings with gigabytes as the unit.
     * @param options The configuration options containing min and/or max values.
     */
    static gb(options) {
        return new DataStorage(StorageUnit.GB, options.minimum, options.maximum);
    }
    constructor(unit, minimum, maximum) {
        this.validate(minimum, maximum);
        this.minimum = minimum;
        this.maximum = maximum;
        this.unit = unit;
    }
    /**
     * Render dataStorage property.
     *
     * @internal
     */
    _render() {
        return {
            unit: this.unit,
            maximum: this.maximum,
            minimum: this.minimum,
        };
    }
    validate(minimum, maximum) {
        const validMinimum = minimum !== undefined && !aws_cdk_lib_1.Token.isUnresolved(minimum);
        const validMaximum = maximum !== undefined && !aws_cdk_lib_1.Token.isUnresolved(maximum);
        if (validMinimum && (minimum < 1 || minimum > 5000)) {
            throw new Error(`\`dataStorage.minimum\` must be between 1 and 5000, got: ${minimum}.`);
        }
        if (validMaximum && (maximum < 1 || maximum > 5000)) {
            throw new Error(`\`dataStorage.maximum\` must be between 1 and 5000, got: ${maximum}.`);
        }
        if (validMinimum && validMaximum && maximum < minimum) {
            throw new Error(`\`dataStorage.maximum\` must be greater than or equal to \`dataStorage.minimum\`, got: maximum ${maximum}, minimum ${minimum}.`);
        }
    }
}
exports.DataStorage = DataStorage;
_a = JSII_RTTI_SYMBOL_1;
DataStorage[_a] = { fqn: "@open-constructs/aws-cdk.aws_elasticache.DataStorage", version: "0.1.2" };
/**
 * The configuration for the number of ElastiCache Processing Units (ECPU) the cache can consume per second.
 */
class ECPUPerSecond {
    /**
     * Creates ECPU per second settings.
     *
     * @param options The configuration options containing min and/or max values.
     */
    static of(options) {
        return new ECPUPerSecond(options.minimum, options.maximum);
    }
    constructor(minimum, maximum) {
        this.validate(minimum, maximum);
        this.minimum = minimum;
        this.maximum = maximum;
    }
    /**
     * Render ecpuPerSecond property.
     *
     * @internal
     */
    _render() {
        return {
            maximum: this.maximum,
            minimum: this.minimum,
        };
    }
    validate(minimum, maximum) {
        const validMinimum = minimum !== undefined && !aws_cdk_lib_1.Token.isUnresolved(minimum);
        const validMaximum = maximum !== undefined && !aws_cdk_lib_1.Token.isUnresolved(maximum);
        if (validMinimum && (minimum < 1000 || minimum > 15000000)) {
            throw new Error(`\`ecpuPerSecond.minimum\` must be between 1000 and 15000000, got: ${minimum}.`);
        }
        if (validMaximum && (maximum < 1000 || maximum > 15000000)) {
            throw new Error(`\`ecpuPerSecond.maximum\` must be between 1000 and 15000000, got: ${maximum}.`);
        }
        if (validMinimum && validMaximum && maximum < minimum) {
            throw new Error(`\`ecpuPerSecond.maximum\` must be greater than or equal to \`ecpuPerSecond.minimum\`, got: maximum ${maximum}, minimum ${minimum}.`);
        }
    }
}
exports.ECPUPerSecond = ECPUPerSecond;
_b = JSII_RTTI_SYMBOL_1;
ECPUPerSecond[_b] = { fqn: "@open-constructs/aws-cdk.aws_elasticache.ECPUPerSecond", version: "0.1.2" };
/**
 * The version number of the engine the serverless cache is compatible with.
 */
var MajorVersion;
(function (MajorVersion) {
    /**
     * Version 7
     */
    MajorVersion["VER_7"] = "7";
    /**
     * Version 8
     */
    MajorVersion["VER_8"] = "8";
})(MajorVersion || (exports.MajorVersion = MajorVersion = {}));
/**
 * A new or imported serverless cache.
 */
class ServerlessCacheBase extends aws_cdk_lib_1.Resource {
    /**
     * Imports an existing ServerlessCache from attributes
     */
    static fromServerlessCacheAttributes(scope, id, attrs) {
        class Import extends ServerlessCacheBase {
            constructor() {
                super(...arguments);
                this.serverlessCacheName = attrs.serverlessCacheName;
                this.endpointAddress = attrs.endpointAddress;
                this.endpointPort = attrs.endpointPort;
                this.connections = new aws_cdk_lib_1.aws_ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: aws_cdk_lib_1.aws_ec2.Port.tcp(attrs.endpointPort),
                });
                this.serverlessCacheArn = aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'elasticache',
                    resource: 'serverlesscache',
                    resourceName: attrs.serverlessCacheName,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Grant the given identity the specified actions.
     * @param grantee the identity to be granted the actions
     * @param actions the data-access actions
     *
     * @see https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticache.html
     */
    grant(grantee, ...actions) {
        return aws_cdk_lib_1.aws_iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'elasticache',
                    resource: 'serverlesscache',
                    resourceName: this.serverlessCacheName,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                }),
            ],
        });
    }
    /**
     * Permits an IAM principal to perform connect to the serverless cache.
     *
     * Actions: Connect
     *
     * @param grantee The principal to grant access to.
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/auth-iam.html
     */
    grantConnect(grantee) {
        return this.grant(grantee, 'elasticache:Connect');
    }
    /**
     * Create a CloudWatch metric for serverless cache.
     *
     * @param metricName name of the metric.
     * @param props metric options.
     *
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/serverless-metrics-events-redis.html
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/serverless-metrics-events.memcached.html
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ElastiCache',
            dimensionsMap: {
                clusterId: this.serverlessCacheName,
            },
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the total number of bytes used by the data stored in your cache.
     *
     * @default - average over 5 minutes
     */
    metricBytesUsedForCache(props) {
        return this.metric('BytesUsedForCache', { statistic: 'Average', ...props });
    }
    /**
     * Metric for the total number of ElastiCacheProcessingUnits (ECPUs) consumed by the requests executed on your cache.
     *
     * @default - average over 5 minutes
     */
    metricElastiCacheProcessingUnits(props) {
        return this.metric('ElastiCacheProcessingUnits', { statistic: 'Average', ...props });
    }
}
/**
 * Represents an ElastiCache Serverless Cache construct in AWS CDK.
 *
 * @example
 * declare const vpc: aws_ec2.IVpc;
 *
 * const serverlessCache = new ServerlessCache(
 *   stack,
 *   'ServerlessCache',
 *   {
 *     serverlessCacheName: 'my-serverlessCache',
 *     engine: Engine.VALKEY,
 *     vpc,
 *   },
 * );
 */
class ServerlessCache extends ServerlessCacheBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.serverlessCacheName ??
                aws_cdk_lib_1.Lazy.string({
                    produce: () => aws_cdk_lib_1.Names.uniqueResourceName(this, { separator: '-', maxLength: 40 }).toLowerCase(),
                }),
        });
        this.props = props;
        this.securityGroups = props.securityGroups ?? [
            this.createSecurityGroup(this, 'SecurityGroup', {
                vpc: this.props.vpc,
                description: 'Automatic generated security group for ElastiCache Serverless Security Group',
            }),
        ];
        this.vpcSubnets = props.vpcSubnets ?? {
            subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_EGRESS,
        };
        this.validateServerlessCacheName();
        this.validateDescription();
        this.validateAutomaticBackupSettings();
        this.validateFinalSnapshotName();
        this.validateUserGroup();
        const serverlessCache = this.createResource(this, 'Resource', {
            engine: this.props.engine,
            serverlessCacheName: this.physicalName,
            cacheUsageLimits: this.renderCacheUsageLimits(),
            dailySnapshotTime: props.dailySnapshotTime?.toTimestamp(),
            description: this.props.description,
            finalSnapshotName: this.props.finalSnapshotName,
            kmsKeyId: this.props.kmsKey?.keyArn,
            majorEngineVersion: this.props.majorEngineVersion,
            securityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
            subnetIds: this.props.vpc.selectSubnets(this.vpcSubnets).subnetIds,
            snapshotArnsToRestore: this.props.snapshotArnsToRestore,
            snapshotRetentionLimit: this.props.snapshotRetentionLimit,
            userGroupId: this.props.userGroup?.userGroupId,
        });
        this.serverlessCacheArn = serverlessCache.attrArn;
        this.serverlessCacheName = serverlessCache.ref;
        this.endpointAddress = serverlessCache.attrEndpointAddress;
        this.endpointPort = aws_cdk_lib_1.Token.asNumber(serverlessCache.attrEndpointPort);
        this.connections = new aws_cdk_lib_1.aws_ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: aws_cdk_lib_1.aws_ec2.Port.tcp(this.endpointPort),
        });
    }
    createResource(scope, id, props) {
        return new aws_cdk_lib_1.aws_elasticache.CfnServerlessCache(scope, id, props);
    }
    createSecurityGroup(scope, id, props) {
        return new aws_cdk_lib_1.aws_ec2.SecurityGroup(scope, id, props);
    }
    renderCacheUsageLimits() {
        const usageLimits = this.props.cacheUsageLimits;
        if (usageLimits === undefined) {
            return undefined;
        }
        const dataStorage = usageLimits.dataStorage;
        const ecpuPerSecond = usageLimits.ecpuPerSecond;
        if (dataStorage === undefined && ecpuPerSecond === undefined) {
            return undefined;
        }
        return {
            dataStorage: dataStorage ? dataStorage._render() : undefined,
            ecpuPerSecond: ecpuPerSecond ? ecpuPerSecond._render() : undefined,
        };
    }
    /**
     * Validates a serverless cache name.
     */
    validateServerlessCacheName() {
        const serverlessCacheName = this.props.serverlessCacheName;
        if (aws_cdk_lib_1.Token.isUnresolved(serverlessCacheName) || serverlessCacheName === undefined) {
            return;
        }
        if (/\s/.test(serverlessCacheName)) {
            throw new Error(`\`serverlessCacheName\` must not contain spaces, got: ${serverlessCacheName}.`);
        }
        if (serverlessCacheName.length < 1 || serverlessCacheName.length > 40) {
            throw new Error(`\`serverlessCacheName\` must be between 1 and 40 characters, got: ${serverlessCacheName.length} characters.`);
        }
    }
    /**
     * Validates a description.
     */
    validateDescription() {
        const description = this.props.description;
        if (aws_cdk_lib_1.Token.isUnresolved(description) || description === undefined) {
            return;
        }
        if (/<|>/.test(description)) {
            throw new Error(`\`description\` must not contain < and >, got: ${description}`);
        }
        if (description.length > 255) {
            throw new Error(`\`description\` must not exceed 255 characters, got: ${description.length} characters.`);
        }
    }
    /**
     * Validates an automatic backup settings.
     */
    validateAutomaticBackupSettings() {
        const dailySnapshotTime = this.props.dailySnapshotTime;
        const snapshotRetentionLimit = this.props.snapshotRetentionLimit;
        if (aws_cdk_lib_1.Token.isUnresolved(snapshotRetentionLimit)) {
            return;
        }
        if (dailySnapshotTime !== undefined && snapshotRetentionLimit === undefined) {
            throw new Error('`snapshotRetentionLimit` must be specified when `dailySnapshotTime` is set.');
        }
        if (snapshotRetentionLimit !== undefined && (snapshotRetentionLimit < 1 || snapshotRetentionLimit > 35)) {
            throw new Error(`\`snapshotRetentionLimit\` must be between 1 and 35, got: ${snapshotRetentionLimit}.`);
        }
    }
    /**
     * Validates final snapshot name.
     */
    validateFinalSnapshotName() {
        const finalSnapshotName = this.props.finalSnapshotName;
        if (aws_cdk_lib_1.Token.isUnresolved(finalSnapshotName) || finalSnapshotName === undefined)
            return;
        if (!/^[a-z][a-z0-9]*(-[a-z0-9]+)*$/.test(finalSnapshotName)) {
            throw new Error(`\`finalSnapshotName\` must consist only of lowercase alphanumeric characters or hyphens, with the first character as a letter, and it can't end with a hyphen or contain two consecutive hyphens, got: ${finalSnapshotName}.`);
        }
        if (finalSnapshotName.length > 255) {
            throw new Error(`\`finalSnapshotName\` must not exceed 255 characters, got: ${finalSnapshotName.length} characters.`);
        }
    }
    /**
     * Validates an engine and a user group combination.
     */
    validateUserGroup() {
        if (this.props.userGroup === undefined)
            return;
        if (![util_1.Engine.REDIS, util_1.Engine.VALKEY].includes(this.props.engine)) {
            throw new Error(`\`userGroup\` is available for Valkey and Redis OSS only, got engine: ${this.props.engine}.`);
        }
    }
}
exports.ServerlessCache = ServerlessCache;
_c = JSII_RTTI_SYMBOL_1;
ServerlessCache[_c] = { fqn: "@open-constructs/aws-cdk.aws_elasticache.ServerlessCache", version: "0.1.2" };
//# sourceMappingURL=data:application/json;base64,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