import { IResource, Resource, aws_elasticache, SecretValue, aws_iam, aws_secretsmanager } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * Interface for a User
 */
export interface IUser extends IResource {
    /**
     * The ARN of the user.
     *
     * @attribute
     */
    readonly userArn: string;
    /**
     * The ID of the user.
     */
    readonly userId: string;
    /**
     * The name of the user.
     */
    readonly userName: string;
}
/**
 * Properties for all user types
 */
export interface BaseUserProps {
    /**
     * The ID of the user.
     * Must consist only of alphanumeric characters or hyphens, with the first character as a letter.
     * Cannot end with a hyphen or contain two consecutive hyphens.
     * @default - auto generated
     */
    readonly userId?: string;
    /**
     * Access permissions string used for this user.
     * @default - 'off -@all'
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/Clusters.RBAC.html#Access-string
     */
    readonly accessString?: string;
}
/**
 * Abstract base class for creating users
 */
declare abstract class BaseUser extends Resource implements IUser {
    /**
     * The ARN of the user.
     */
    readonly userArn: string;
    /**
     * The ID of the user.
     */
    readonly userId: string;
    /**
     * The name of the user.
     */
    readonly userName: string;
    protected readonly props: BaseUserProps;
    protected constructor(scope: Construct, id: string, props?: BaseUserProps);
    protected createResource(scope: Construct, id: string, props: aws_elasticache.CfnUserProps): aws_elasticache.CfnUser;
    /**
     * Render userName property
     */
    protected abstract renderUserName(): string;
    /**
     * Render authenticationMode property
     */
    protected abstract renderAuthenticationMode(): any;
    /**
     * Validates user id
     */
    protected validateUserId(userId?: string): void;
    /**
     * Validates username
     */
    protected validateUserName(userName?: string): void;
}
/**
 * Interface for IAM-enabled users
 */
export interface IIamUser extends IUser {
    /**
     * Grant permissions to this user
     */
    grant(grantee: aws_iam.IGrantable, ...actions: string[]): aws_iam.Grant;
    /**
     * Grant connect permissions to this user
     */
    grantConnect(grantee: aws_iam.IGrantable): aws_iam.Grant;
}
/**
 * Properties for IAM-enabled users
 */
export interface IamUserProps extends BaseUserProps {
}
/**
 * Represents an IAM-enabled user construct in AWS CDK.
 *
 * @example
 *
 * const user = new IamUser(
 *   stack,
 *   'User',
 *   {
 *     accessString: 'on ~* +@all',
 *   },
 * );
 */
export declare class IamUser extends BaseUser implements IIamUser {
    /**
     * Imports an existing IAM-enabled user from userId
     */
    static fromUserId(scope: Construct, id: string, userId: string): IIamUser;
    constructor(scope: Construct, id: string, props?: IamUserProps);
    /**
     * For IAM-enabled ElastiCache users the username and user id properties must be identical
     *
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/auth-iam.html
     */
    protected renderUserName(): string;
    protected renderAuthenticationMode(): any;
    /**
     * Grant the given identity the specified actions
     * @param grantee the identity to be granted the actions
     * @param actions the data-access actions
     *
     * @see https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticache.html
     */
    grant(grantee: aws_iam.IGrantable, ...actions: string[]): aws_iam.Grant;
    /**
     * Permits an IAM principal to perform connect to the user.
     *
     * Actions: Connect
     *
     * @param grantee The principal to grant access to.
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/auth-iam.html
     */
    grantConnect(grantee: aws_iam.IGrantable): aws_iam.Grant;
}
/**
 * Interface for password-authenticated users
 */
export interface IPasswordUser extends IUser {
}
/**
 * Attributes for importing a password-authenticated user
 */
export interface PasswordUserAttributes {
    /**
     * The ID of the user.
     */
    readonly userId: string;
    /**
     * The name of the user.
     */
    readonly userName: string;
}
/**
 * Properties for password-authenticated users
 */
export interface PasswordUserProps extends BaseUserProps {
    /**
     * The username of the user.
     * @default - same as userId
     */
    readonly userName?: string;
    /**
     * Passwords used for this user account.
     * You can create up to two passwords for each user.
     *
     * @default - automatically generate a password for the user
     */
    readonly passwords?: SecretValue[];
}
/**
 * Represents a password authentication user construct in AWS CDK.
 *
 * @example
 *
 * const user = new PasswordUser(
 *   stack,
 *   'User',
 *   {
 *    passwords: [
 *      cdk.SecretValue.unsafePlainText('exampleUserPassword123'),
 *      cdk.SecretValue.unsafePlainText('anotherUserPassword123'),
 *    ],
 *   },
 * );
 */
export declare class PasswordUser extends BaseUser implements IPasswordUser {
    /**
     * Imports an existing password authentication user from attributes
     */
    static fromUserAttributes(scope: Construct, id: string, attrs: PasswordUserAttributes): IPasswordUser;
    private _generatedSecret?;
    constructor(scope: Construct, id: string, props: PasswordUserProps);
    protected renderUserName(): string;
    protected renderAuthenticationMode(): any;
    /**
     * The secret containing the generated password
     *
     * Throws an exception if `passwords` is provided in the props
     */
    get generatedSecret(): aws_secretsmanager.ISecret;
}
/**
 * Interface for no password required users
 */
export interface INoPasswordRequiredUser extends IUser {
}
/**
 * Attributes for importing a no password required user
 */
export interface NoPasswordUserAttributes {
    /**
     * The ID of the user.
     */
    readonly userId: string;
    /**
     * The name of the user.
     */
    readonly userName: string;
}
/**
 * Properties for no password required users
 */
export interface NoPasswordRequiredUserProps extends BaseUserProps {
    /**
     * The username of the user.
     * @default - same as userId
     */
    readonly userName?: string;
}
/**
 * Represents a no password required user construct in AWS CDK
 *
 * @example
 *
 * const user = new NoPasswordRequiredUser(
 *   stack,
 *   'User',
 *   {
 *     userName: 'my-user',
 *     accessString: 'on ~* +@all',
 *   },
 * );
 */
export declare class NoPasswordRequiredUser extends BaseUser implements INoPasswordRequiredUser {
    /**
     * Imports an existing no password required user from attributes
     */
    static fromUserAttributes(scope: Construct, id: string, attrs: NoPasswordUserAttributes): INoPasswordRequiredUser;
    constructor(scope: Construct, id: string, props?: NoPasswordRequiredUserProps);
    protected renderUserName(): string;
    protected renderAuthenticationMode(): any;
}
export {};
