"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NoPasswordRequiredUser = exports.PasswordUser = exports.IamUser = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const util_1 = require("./util");
/**
 * Abstract base class for creating users
 */
class BaseUser extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.userId ??
                aws_cdk_lib_1.Lazy.string({
                    produce: () => aws_cdk_lib_1.Names.uniqueResourceName(this, { separator: '-', maxLength: 40 }).toLowerCase(),
                }),
        });
        this.props = props;
        this.validateUserId(props.userId);
        const user = this.createResource(this, 'Resource', {
            engine: util_1.Engine.REDIS,
            userId: this.physicalName,
            userName: this.renderUserName(),
            accessString: this.props.accessString ?? 'off -@all',
            authenticationMode: this.renderAuthenticationMode(),
        });
        this.userArn = user.attrArn;
        this.userId = user.ref;
        this.userName = user.userName;
    }
    createResource(scope, id, props) {
        return new aws_cdk_lib_1.aws_elasticache.CfnUser(scope, id, props);
    }
    /**
     * Validates user id
     */
    validateUserId(userId) {
        if (aws_cdk_lib_1.Token.isUnresolved(userId) || userId === undefined) {
            return;
        }
        if (userId.length < 1 || userId.length > 40) {
            throw new Error(`\`userId\` must be between 1 and 40 characters, got ${userId.length} characters.`);
        }
        if (userId === 'default') {
            throw new Error('`userId` cannot be `default` because ElastiCache automatically configures a default user with user ID `default`.');
        }
        if (!/^[A-Za-z][A-Za-z0-9]*(-[A-Za-z0-9]+)*$/.test(userId)) {
            throw new Error(`\`userId\` must consist only of alphanumeric characters or hyphens, with the first character as a letter, and it can't end with a hyphen or contain two consecutive hyphens, got: ${userId}.`);
        }
    }
    /**
     * Validates username
     */
    validateUserName(userName) {
        if (aws_cdk_lib_1.Token.isUnresolved(userName) || userName === undefined) {
            return;
        }
        if (userName.length < 1 || userName.length > 120) {
            throw new Error(`\`userName\` must be between 1 and 120 characters, got ${userName.length} characters.`);
        }
        if (/\s/.test(userName)) {
            throw new Error(`\`userName\` must not contain spaces, got: ${userName}.`);
        }
    }
}
/**
 * Represents an IAM-enabled user construct in AWS CDK.
 *
 * @example
 *
 * const user = new IamUser(
 *   stack,
 *   'User',
 *   {
 *     accessString: 'on ~* +@all',
 *   },
 * );
 */
class IamUser extends BaseUser {
    /**
     * Imports an existing IAM-enabled user from userId
     */
    static fromUserId(scope, id, userId) {
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.userId = userId;
                this.userName = userId;
                this.userArn = aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'elasticache',
                    resource: 'user',
                    resourceName: userId,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
            grant(grantee, ...actions) {
                return aws_cdk_lib_1.aws_iam.Grant.addToPrincipal({
                    grantee,
                    actions,
                    resourceArns: [this.userArn],
                });
            }
            grantConnect(grantee) {
                return this.grant(grantee, 'elasticache:Connect');
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props = {}) {
        super(scope, id, props);
    }
    /**
     * For IAM-enabled ElastiCache users the username and user id properties must be identical
     *
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/auth-iam.html
     */
    renderUserName() {
        return this.physicalName;
    }
    renderAuthenticationMode() {
        return {
            Type: 'iam',
        };
    }
    /**
     * Grant the given identity the specified actions
     * @param grantee the identity to be granted the actions
     * @param actions the data-access actions
     *
     * @see https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticache.html
     */
    grant(grantee, ...actions) {
        return aws_cdk_lib_1.aws_iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'elasticache',
                    resource: 'user',
                    resourceName: this.userId,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                }),
            ],
        });
    }
    /**
     * Permits an IAM principal to perform connect to the user.
     *
     * Actions: Connect
     *
     * @param grantee The principal to grant access to.
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/auth-iam.html
     */
    grantConnect(grantee) {
        return this.grant(grantee, 'elasticache:Connect');
    }
}
exports.IamUser = IamUser;
_a = JSII_RTTI_SYMBOL_1;
IamUser[_a] = { fqn: "@open-constructs/aws-cdk.aws_elasticache.IamUser", version: "0.1.2" };
/**
 * Represents a password authentication user construct in AWS CDK.
 *
 * @example
 *
 * const user = new PasswordUser(
 *   stack,
 *   'User',
 *   {
 *    passwords: [
 *      cdk.SecretValue.unsafePlainText('exampleUserPassword123'),
 *      cdk.SecretValue.unsafePlainText('anotherUserPassword123'),
 *    ],
 *   },
 * );
 */
class PasswordUser extends BaseUser {
    /**
     * Imports an existing password authentication user from attributes
     */
    static fromUserAttributes(scope, id, attrs) {
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.userId = attrs.userId;
                this.userName = attrs.userName;
                this.userArn = aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'elasticache',
                    resource: 'user',
                    resourceName: attrs.userId,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.passwords && props.passwords.length > 2) {
            throw new Error(`Up to 2 passwords can be set, got ${props.passwords.length} passwords.`);
        }
        this.validateUserName(props.userName);
    }
    renderUserName() {
        return this.props.userName ?? this.physicalName;
    }
    renderAuthenticationMode() {
        const props = this.props;
        if (props.passwords?.length) {
            return {
                Type: 'password',
                Passwords: props.passwords?.map(password => password.unsafeUnwrap()),
            };
        }
        this._generatedSecret = new aws_cdk_lib_1.aws_secretsmanager.Secret(this, 'Secret', {
            generateSecretString: {
                // https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/auth.html#auth-overview
                secretStringTemplate: JSON.stringify({
                    username: this.renderUserName(),
                }),
                generateStringKey: 'password',
                passwordLength: 32,
                // SecretsManager uses the following non-alphanumeric characters to generate a password.
                // ( !\"#$%&'()*+,-./:;<=>?@[\\]^_`{|}~ )
                // https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_GetRandomPassword.html
                // Redis OSS and Valkey are only support the following non-alphanumeric characters in a password.
                // ( !&#$^<>- )
                // Based on the above, exclude any characters that are not supported by Redis OSS and Valkey.
                excludeCharacters: '"%\'()*+,./:;=?@[\\]_`{|}~',
            },
            description: `Generated by the CDK for stack: ${aws_cdk_lib_1.Stack.of(this).stackName}`,
        });
        return {
            Type: 'password',
            Passwords: [this._generatedSecret.secretValueFromJson('password').unsafeUnwrap()],
        };
    }
    /**
     * The secret containing the generated password
     *
     * Throws an exception if `passwords` is provided in the props
     */
    get generatedSecret() {
        if (!this._generatedSecret) {
            throw new Error(`The generated secret is only available when 'passwords' is not specified.`);
        }
        return this._generatedSecret;
    }
}
exports.PasswordUser = PasswordUser;
_b = JSII_RTTI_SYMBOL_1;
PasswordUser[_b] = { fqn: "@open-constructs/aws-cdk.aws_elasticache.PasswordUser", version: "0.1.2" };
/**
 * Represents a no password required user construct in AWS CDK
 *
 * @example
 *
 * const user = new NoPasswordRequiredUser(
 *   stack,
 *   'User',
 *   {
 *     userName: 'my-user',
 *     accessString: 'on ~* +@all',
 *   },
 * );
 */
class NoPasswordRequiredUser extends BaseUser {
    /**
     * Imports an existing no password required user from attributes
     */
    static fromUserAttributes(scope, id, attrs) {
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.userId = attrs.userId;
                this.userName = attrs.userName;
                this.userArn = aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'elasticache',
                    resource: 'user',
                    resourceName: attrs.userId,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.validateUserName(props.userName);
    }
    renderUserName() {
        return this.props.userName ?? this.physicalName;
    }
    renderAuthenticationMode() {
        return {
            Type: 'no-password-required',
        };
    }
}
exports.NoPasswordRequiredUser = NoPasswordRequiredUser;
_c = JSII_RTTI_SYMBOL_1;
NoPasswordRequiredUser[_c] = { fqn: "@open-constructs/aws-cdk.aws_elasticache.NoPasswordRequiredUser", version: "0.1.2" };
//# sourceMappingURL=data:application/json;base64,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