"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OntapFileSystem = exports.MultiAz2ThroughputCapacityPerHaPair = exports.SingleAz2ThroughputCapacityPerHaPair = exports.MultiAz1ThroughputCapacityPerHaPair = exports.SingleAz1ThroughputCapacityPerHaPair = exports.ThroughputCapacityPerHaPair = exports.OntapDeploymentType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * The different kinds of file system deployments used by NetApp ONTAP.
 */
var OntapDeploymentType;
(function (OntapDeploymentType) {
    /**
     * A high availability file system configured for Multi-AZ redundancy to tolerate temporary Availability Zone (AZ) unavailability.
     * This is a first-generation FSx for ONTAP file system.
     */
    OntapDeploymentType["MULTI_AZ_1"] = "MULTI_AZ_1";
    /**
     * A high availability file system configured for Multi-AZ redundancy to tolerate temporary AZ unavailability.
     * This is a second-generation FSx for ONTAP file system.
     */
    OntapDeploymentType["MULTI_AZ_2"] = "MULTI_AZ_2";
    /**
     * A file system configured for Single-AZ redundancy.
     * This is a first-generation FSx for ONTAP file system.
     */
    OntapDeploymentType["SINGLE_AZ_1"] = "SINGLE_AZ_1";
    /**
     * A file system configured with multiple high-availability (HA) pairs for Single-AZ redundancy.
     * This is a second-generation FSx for ONTAP file system.
     */
    OntapDeploymentType["SINGLE_AZ_2"] = "SINGLE_AZ_2";
})(OntapDeploymentType || (exports.OntapDeploymentType = OntapDeploymentType = {}));
/**
 * The throughput capacity per HA pair for an Amazon FSx for NetApp ONTAP file system.
 */
class ThroughputCapacityPerHaPair {
    constructor(capacity) {
        this.capacity = capacity;
    }
}
exports.ThroughputCapacityPerHaPair = ThroughputCapacityPerHaPair;
_a = JSII_RTTI_SYMBOL_1;
ThroughputCapacityPerHaPair[_a] = { fqn: "@open-constructs/aws-cdk.aws_fsx.ThroughputCapacityPerHaPair", version: "0.1.2" };
/**
 * The throughput capacity for the Single-AZ 1 deployment type.
 */
class SingleAz1ThroughputCapacityPerHaPair extends ThroughputCapacityPerHaPair {
    constructor() {
        super(...arguments);
        /**
         * The deployment type of the throughput capacity.
         */
        this.deploymentType = OntapDeploymentType.SINGLE_AZ_1;
    }
}
exports.SingleAz1ThroughputCapacityPerHaPair = SingleAz1ThroughputCapacityPerHaPair;
_b = JSII_RTTI_SYMBOL_1;
SingleAz1ThroughputCapacityPerHaPair[_b] = { fqn: "@open-constructs/aws-cdk.aws_fsx.SingleAz1ThroughputCapacityPerHaPair", version: "0.1.2" };
/**
 * The throughput capacity of 128 MBps per HA pair.
 */
SingleAz1ThroughputCapacityPerHaPair.MB_PER_SEC_128 = new SingleAz1ThroughputCapacityPerHaPair(128);
/**
 * The throughput capacity of 256 MBps per HA pair.
 */
SingleAz1ThroughputCapacityPerHaPair.MB_PER_SEC_256 = new SingleAz1ThroughputCapacityPerHaPair(256);
/**
 * The throughput capacity of 512 MBps per HA pair.
 */
SingleAz1ThroughputCapacityPerHaPair.MB_PER_SEC_512 = new SingleAz1ThroughputCapacityPerHaPair(512);
/**
 * The throughput capacity of 1024 MBps per HA pair.
 */
SingleAz1ThroughputCapacityPerHaPair.MB_PER_SEC_1024 = new SingleAz1ThroughputCapacityPerHaPair(1024);
/**
 * The throughput capacity of 2048 MBps per HA pair.
 */
SingleAz1ThroughputCapacityPerHaPair.MB_PER_SEC_2048 = new SingleAz1ThroughputCapacityPerHaPair(2048);
/**
 * The throughput capacity of 4096 MBps per HA pair.
 */
SingleAz1ThroughputCapacityPerHaPair.MB_PER_SEC_4096 = new SingleAz1ThroughputCapacityPerHaPair(4096);
/**
 * The throughput capacity for the Multi-AZ 1 deployment type.
 */
class MultiAz1ThroughputCapacityPerHaPair extends ThroughputCapacityPerHaPair {
    constructor() {
        super(...arguments);
        /**
         * The deployment type of the throughput capacity.
         */
        this.deploymentType = OntapDeploymentType.MULTI_AZ_1;
    }
}
exports.MultiAz1ThroughputCapacityPerHaPair = MultiAz1ThroughputCapacityPerHaPair;
_c = JSII_RTTI_SYMBOL_1;
MultiAz1ThroughputCapacityPerHaPair[_c] = { fqn: "@open-constructs/aws-cdk.aws_fsx.MultiAz1ThroughputCapacityPerHaPair", version: "0.1.2" };
/**
 * The throughput capacity of 128 MBps per HA pair.
 */
MultiAz1ThroughputCapacityPerHaPair.MB_PER_SEC_128 = new MultiAz1ThroughputCapacityPerHaPair(128);
/**
 * The throughput capacity of 256 MBps per HA pair.
 */
MultiAz1ThroughputCapacityPerHaPair.MB_PER_SEC_256 = new MultiAz1ThroughputCapacityPerHaPair(256);
/**
 * The throughput capacity of 512 MBps per HA pair.
 */
MultiAz1ThroughputCapacityPerHaPair.MB_PER_SEC_512 = new MultiAz1ThroughputCapacityPerHaPair(512);
/**
 * The throughput capacity of 1024 MBps per HA pair.
 */
MultiAz1ThroughputCapacityPerHaPair.MB_PER_SEC_1024 = new MultiAz1ThroughputCapacityPerHaPair(1024);
/**
 * The throughput capacity of 2048 MBps per HA pair.
 */
MultiAz1ThroughputCapacityPerHaPair.MB_PER_SEC_2048 = new MultiAz1ThroughputCapacityPerHaPair(2048);
/**
 * The throughput capacity of 4096 MBps per HA pair.
 */
MultiAz1ThroughputCapacityPerHaPair.MB_PER_SEC_4096 = new MultiAz1ThroughputCapacityPerHaPair(4096);
/**
 * The throughput capacity for the Single-AZ 2 deployment type.
 */
class SingleAz2ThroughputCapacityPerHaPair extends ThroughputCapacityPerHaPair {
    constructor() {
        super(...arguments);
        /**
         * The deployment type of the throughput capacity.
         */
        this.deploymentType = OntapDeploymentType.SINGLE_AZ_2;
    }
}
exports.SingleAz2ThroughputCapacityPerHaPair = SingleAz2ThroughputCapacityPerHaPair;
_d = JSII_RTTI_SYMBOL_1;
SingleAz2ThroughputCapacityPerHaPair[_d] = { fqn: "@open-constructs/aws-cdk.aws_fsx.SingleAz2ThroughputCapacityPerHaPair", version: "0.1.2" };
/**
 * The throughput capacity of 1536 MBps per HA pair.
 */
SingleAz2ThroughputCapacityPerHaPair.MB_PER_SEC_1536 = new SingleAz2ThroughputCapacityPerHaPair(1536);
/**
 * The throughput capacity of 3072 MBps per HA pair.
 */
SingleAz2ThroughputCapacityPerHaPair.MB_PER_SEC_3072 = new SingleAz2ThroughputCapacityPerHaPair(3072);
/**
 * The throughput capacity of 6144 MBps per HA pair.
 */
SingleAz2ThroughputCapacityPerHaPair.MB_PER_SEC_6144 = new SingleAz2ThroughputCapacityPerHaPair(6144);
/**
 * The throughput capacity for the Multi-AZ 2 deployment type.
 */
class MultiAz2ThroughputCapacityPerHaPair extends ThroughputCapacityPerHaPair {
    constructor() {
        super(...arguments);
        /**
         * The deployment type of the throughput capacity.
         */
        this.deploymentType = OntapDeploymentType.MULTI_AZ_2;
    }
}
exports.MultiAz2ThroughputCapacityPerHaPair = MultiAz2ThroughputCapacityPerHaPair;
_e = JSII_RTTI_SYMBOL_1;
MultiAz2ThroughputCapacityPerHaPair[_e] = { fqn: "@open-constructs/aws-cdk.aws_fsx.MultiAz2ThroughputCapacityPerHaPair", version: "0.1.2" };
/**
 * The throughput capacity of 384 MBps per HA pair.
 */
MultiAz2ThroughputCapacityPerHaPair.MB_PER_SEC_384 = new MultiAz2ThroughputCapacityPerHaPair(384);
/**
 * The throughput capacity of 768 MBps per HA pair.
 */
MultiAz2ThroughputCapacityPerHaPair.MB_PER_SEC_768 = new MultiAz2ThroughputCapacityPerHaPair(768);
/**
 * The throughput capacity of 1536 MBps per HA pair.
 */
MultiAz2ThroughputCapacityPerHaPair.MB_PER_SEC_1536 = new MultiAz2ThroughputCapacityPerHaPair(1536);
/**
 * The throughput capacity of 3072 MBps per HA pair.
 */
MultiAz2ThroughputCapacityPerHaPair.MB_PER_SEC_3072 = new MultiAz2ThroughputCapacityPerHaPair(3072);
/**
 * The throughput capacity of 6144 MBps per HA pair.
 */
MultiAz2ThroughputCapacityPerHaPair.MB_PER_SEC_6144 = new MultiAz2ThroughputCapacityPerHaPair(6144);
/**
 * The FSx for NetApp ONTAP File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 *
 * @resource AWS::FSx::FileSystem
 */
class OntapFileSystem extends aws_cdk_lib_1.aws_fsx.FileSystemBase {
    /**
     * Import an existing FSx for NetApp ONTAP file system from the given properties.
     */
    static fromOntapFileSystemAttributes(scope, id, attrs) {
        class Import extends aws_cdk_lib_1.aws_fsx.FileSystemBase {
            constructor() {
                super(...arguments);
                this.dnsName = attrs.dnsName;
                this.fileSystemId = attrs.fileSystemId;
                this.connections = OntapFileSystem.configureConnections(attrs.securityGroup);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Configures a Connections object.
     */
    static configureConnections(securityGroup) {
        const connections = new aws_cdk_lib_1.aws_ec2.Connections({
            securityGroups: [securityGroup],
        });
        return connections;
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.deploymentType = props.ontapConfiguration.deploymentType ?? OntapDeploymentType.MULTI_AZ_2;
        this.validateProps(props);
        const securityGroup = props.securityGroup || this.createSecurityGroup(props.vpc);
        this.connections = OntapFileSystem.configureConnections(securityGroup);
        this.fileSystem = this.createOntapFileSystem(securityGroup, props);
        this.fileSystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = this.fileSystem.ref;
        const baseDnsName = `${this.fileSystemId}.fsx.${this.env.region}.${aws_cdk_lib_1.Aws.URL_SUFFIX}`;
        this.dnsName = `management.${baseDnsName}`;
        this.interClusterDnsName = `intercluster.${baseDnsName}`;
    }
    createSecurityGroup(vpc) {
        return new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'FsxOntapSecurityGroup', {
            vpc,
        });
    }
    createOntapFileSystem(fileSystemSecurityGroup, props) {
        const ontapConfiguration = props.ontapConfiguration;
        const fileSystem = new aws_cdk_lib_1.aws_fsx.CfnFileSystem(this, 'Resource', {
            fileSystemType: OntapFileSystem.DEFAULT_FILE_SYSTEM_TYPE,
            subnetIds: props.vpcSubnets.map(subnet => subnet.subnetId),
            backupId: props.backupId,
            kmsKeyId: props.kmsKey?.keyId,
            ontapConfiguration: {
                automaticBackupRetentionDays: ontapConfiguration.automaticBackupRetention?.toDays() ?? 30,
                dailyAutomaticBackupStartTime: ontapConfiguration.dailyAutomaticBackupStartTime?.toTimestamp(),
                deploymentType: this.deploymentType,
                diskIopsConfiguration: {
                    mode: ontapConfiguration.diskIops ? 'USER_PROVISIONED' : 'AUTOMATIC',
                    iops: ontapConfiguration.diskIops,
                },
                endpointIpAddressRange: ontapConfiguration.endpointIpAddressRange,
                fsxAdminPassword: ontapConfiguration.fsxAdminPassword?.unsafeUnwrap(),
                haPairs: ontapConfiguration.haPairs,
                preferredSubnetId: ontapConfiguration.preferredSubnet?.subnetId,
                routeTableIds: ontapConfiguration.routeTables?.map(routeTable => routeTable.routeTableId),
                throughputCapacityPerHaPair: ontapConfiguration.throughputCapacityPerHaPair?.capacity,
                weeklyMaintenanceStartTime: ontapConfiguration.weeklyMaintenanceStartTime?.toTimestamp(),
            },
            securityGroupIds: [fileSystemSecurityGroup.securityGroupId],
            storageCapacity: props.storageCapacityGiB,
        });
        return fileSystem;
    }
    /**
     * Validates the props provided for a new FSx for Ontap file system.
     */
    validateProps(props) {
        const ontapConfiguration = props.ontapConfiguration;
        this.validateHaPairs(ontapConfiguration.haPairs);
        this.validateAutomaticBackupRetention(ontapConfiguration.automaticBackupRetention);
        this.validateDailyAutomaticBackupStartTime(ontapConfiguration.automaticBackupRetention, ontapConfiguration.dailyAutomaticBackupStartTime);
        this.validateDiskIops(props.storageCapacityGiB, ontapConfiguration.diskIops, ontapConfiguration.haPairs);
        this.validateEndpointIpAddressRange(ontapConfiguration.endpointIpAddressRange);
        this.validateSubnets(props.vpcSubnets, ontapConfiguration.preferredSubnet);
        this.validateRouteTables(ontapConfiguration.routeTables);
        this.validateThroughputCapacity(ontapConfiguration.throughputCapacityPerHaPair);
        this.validateStorageCapacity(ontapConfiguration.haPairs, props.storageCapacityGiB);
    }
    validateAutomaticBackupRetention(automaticBackupRetention) {
        if (automaticBackupRetention == null ||
            automaticBackupRetention.isUnresolved() ||
            automaticBackupRetention.toMilliseconds() === 0) {
            return;
        }
        if (automaticBackupRetention.toMilliseconds() < aws_cdk_lib_1.Duration.days(1).toMilliseconds()) {
            throw new Error('automaticBackupRetention must be between 1 and 90 days or 0 for disabled');
        }
        if (automaticBackupRetention.toDays() > 90) {
            throw new Error(`automaticBackupRetention must be between 1 and 90 days or 0 for disabled. got: ${automaticBackupRetention.toDays()} days`);
        }
    }
    validateDailyAutomaticBackupStartTime(automaticBackupRetention, dailyAutomaticBackupStartTime) {
        if (!dailyAutomaticBackupStartTime) {
            return;
        }
        const automaticBackupDisabled = !automaticBackupRetention || automaticBackupRetention.toDays() === 0;
        if (dailyAutomaticBackupStartTime && automaticBackupDisabled) {
            throw new Error("'automaticBackupRetention' period must be set a non-zero day when 'dailyAutomaticBackupStartTime' is set");
        }
    }
    validateDiskIops(storageCapacityGiB, diskIops, haPairs = 1) {
        if (diskIops == null ||
            aws_cdk_lib_1.Token.isUnresolved(diskIops) ||
            aws_cdk_lib_1.Token.isUnresolved(storageCapacityGiB) ||
            aws_cdk_lib_1.Token.isUnresolved(haPairs)) {
            return;
        }
        const minDiskIops = storageCapacityGiB * 3 * haPairs;
        const maxDiskIops = 200000 * haPairs;
        if (!Number.isInteger(diskIops) || diskIops < minDiskIops || diskIops > maxDiskIops) {
            throw new Error(`\'diskIops\' must be an integer between ${minDiskIops} and ${maxDiskIops}, got ${diskIops}`);
        }
    }
    validateEndpointIpAddressRange(endpointIpAddressRange) {
        if (endpointIpAddressRange == null || aws_cdk_lib_1.Token.isUnresolved(endpointIpAddressRange)) {
            return;
        }
        if (this.deploymentType !== OntapDeploymentType.MULTI_AZ_1 &&
            this.deploymentType !== OntapDeploymentType.MULTI_AZ_2) {
            throw new Error("'endpointIpAddressRange' can only be specified for multi-AZ file systems");
        }
        if (!/^[^\u0000\u0085\u2028\u2029\r\n]{9,17}$/.test(endpointIpAddressRange)) {
            throw new Error("'endpointIpAddressRange' must be between 9 and 17 characters long and not contain any of the following characters: \\u0000, \\u0085, \\u2028, \\u2029, \\r, or \\n");
        }
    }
    validateHaPairs(haPairs) {
        if (haPairs == null || aws_cdk_lib_1.Token.isUnresolved(haPairs)) {
            return;
        }
        if (!Number.isInteger(haPairs) || haPairs < 1 || haPairs > 12) {
            throw new Error(`\'haPairs\' must be an integer between 1 and 12, got ${haPairs}`);
        }
        if (haPairs > 1 &&
            [OntapDeploymentType.SINGLE_AZ_1, OntapDeploymentType.MULTI_AZ_1, OntapDeploymentType.MULTI_AZ_2].includes(this.deploymentType)) {
            throw new Error(`\'haPairs\' must be 1 for deployment type ${this.deploymentType}, got ${haPairs}`);
        }
    }
    validateSubnets(vpcSubnets, preferredSubnet) {
        if ((this.deploymentType === OntapDeploymentType.MULTI_AZ_1 ||
            this.deploymentType === OntapDeploymentType.MULTI_AZ_2) &&
            !preferredSubnet) {
            throw new Error("'preferredSubnet' must be specified for multi-AZ file systems");
        }
        if ((this.deploymentType === OntapDeploymentType.SINGLE_AZ_1 ||
            this.deploymentType === OntapDeploymentType.SINGLE_AZ_2) &&
            preferredSubnet) {
            throw new Error("'preferredSubnet' can only be specified for multi-AZ file systems");
        }
        if (preferredSubnet && !vpcSubnets.includes(preferredSubnet)) {
            throw new Error("'preferredSubnet' must be one of the specified 'vpcSubnets'");
        }
    }
    validateRouteTables(routeTables) {
        if (routeTables == null || routeTables.length === 0) {
            return;
        }
        if (this.deploymentType !== OntapDeploymentType.MULTI_AZ_1 &&
            this.deploymentType !== OntapDeploymentType.MULTI_AZ_2) {
            throw new Error("'routeTables' can only be specified for multi-AZ file systems");
        }
    }
    validateThroughputCapacity(throughputCapacityPerHaPair) {
        if (throughputCapacityPerHaPair == null) {
            return;
        }
        if (throughputCapacityPerHaPair.deploymentType !== this.deploymentType) {
            throw new Error(`'throughputCapacityPerHaPair' must be compatible with the deployment type, deployment type: ${this.deploymentType}, deployment type from throughput capacity: ${throughputCapacityPerHaPair.deploymentType}`);
        }
    }
    validateStorageCapacity(haPairs = 1, storageCapacityGiB) {
        if (aws_cdk_lib_1.Token.isUnresolved(storageCapacityGiB) || aws_cdk_lib_1.Token.isUnresolved(haPairs)) {
            return;
        }
        if (!Number.isInteger(storageCapacityGiB) ||
            storageCapacityGiB < 1024 * haPairs ||
            storageCapacityGiB > Math.min(1048576, 524288 * haPairs)) {
            throw new Error(`storageCapacityGiB must be an integer between ${1024 * haPairs} and ${Math.min(1048576, 524288 * haPairs)}, got ${storageCapacityGiB}`);
        }
    }
}
exports.OntapFileSystem = OntapFileSystem;
_f = JSII_RTTI_SYMBOL_1;
OntapFileSystem[_f] = { fqn: "@open-constructs/aws-cdk.aws_fsx.OntapFileSystem", version: "0.1.2" };
/**
 * The default FSx file system type used by FSx for NetApp ONTAP.
 */
OntapFileSystem.DEFAULT_FILE_SYSTEM_TYPE = 'ONTAP';
//# sourceMappingURL=data:application/json;base64,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