"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Namespace = exports.LogExport = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * The types of logs the namespace can export.
 */
var LogExport;
(function (LogExport) {
    /**
     * User log
     */
    LogExport["USER_LOG"] = "userlog";
    /**
     * Connection log
     */
    LogExport["CONNECTION_LOG"] = "connectionlog";
    /**
     * User activity log
     */
    LogExport["USER_ACTIVITY_LOG"] = "useractivitylog";
})(LogExport || (exports.LogExport = LogExport = {}));
/**
 * Represents a Redshift Serverless Namespace construct in AWS CDK.
 *
 * @example
 *
 * const nameSpace = new Namespace(
 *   stack,
 *   'Namespace',
 *   {
 *     namespaceName: 'my-namespace',
 *   },
 * );
 */
class Namespace extends aws_cdk_lib_1.Resource {
    /**
     * Imports an existing Namespace from attributes
     */
    static fromNamespaceAttributes(scope, id, attrs) {
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.namespaceName = attrs.namespaceName;
                this.namespaceId = attrs.namespaceId;
                this.namespaceArn = aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'redshift-serverless',
                    resource: 'namespace',
                    resourceName: attrs.namespaceId,
                });
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.namespaceName ??
                aws_cdk_lib_1.Lazy.string({
                    produce: () => aws_cdk_lib_1.Names.uniqueResourceName(this, { maxLength: 64, allowedSpecialCharacters: '-' }).toLowerCase(),
                }),
        });
        this.props = props;
        this.iamRoles = props.iamRoles ?? [];
        this.validateAdmin();
        this.validateDbName();
        this.validateFinalSnapshot();
        this.validateDefaultIamRole();
        this.validateNamespaceName();
        const namespace = this.createResource(this, 'Resource', {
            adminUsername: this.props.adminUsername,
            adminUserPassword: this.props.adminUserPassword?.unsafeUnwrap(),
            dbName: this.props.dbName,
            defaultIamRoleArn: this.props.defaultIamRole?.roleArn,
            finalSnapshotName: this.props.finalSnapshotName,
            finalSnapshotRetentionPeriod: this.props.finalSnapshotRetentionPeriod,
            iamRoles: aws_cdk_lib_1.Lazy.list({ produce: () => this.iamRoles.map(role => role.roleArn) }, { omitEmpty: true }),
            kmsKeyId: this.props.kmsKey?.keyId,
            logExports: this.props.logExports,
            namespaceName: this.physicalName,
        });
        this.namespaceArn = namespace.attrNamespaceNamespaceArn;
        this.namespaceName = namespace.attrNamespaceNamespaceName;
        this.namespaceId = namespace.attrNamespaceNamespaceId;
    }
    createResource(scope, id, props) {
        return new aws_cdk_lib_1.aws_redshiftserverless.CfnNamespace(scope, id, props);
    }
    /**
     * Validates admin settings.
     */
    validateAdmin() {
        const adminUsername = this.props.adminUsername;
        const adminUserPassword = this.props.adminUserPassword;
        if (aws_cdk_lib_1.Token.isUnresolved(adminUsername)) {
            return;
        }
        if ((adminUsername !== undefined && adminUserPassword === undefined) ||
            (adminUsername === undefined && adminUserPassword !== undefined)) {
            throw new Error('You must specify both `adminUsername` and `adminUserPassword`, or neither.');
        }
        if (adminUsername && !/^[a-zA-Z][a-zA-Z_0-9+.@-]*$/.test(adminUsername)) {
            throw new Error(`\`adminUsername\` must start with a letter and can only contain letters, numbers, and the special characters: _, +, ., @, -, got: ${adminUsername}.`);
        }
    }
    /**
     * Validates a database name.
     */
    validateDbName() {
        const dbName = this.props.dbName;
        if (aws_cdk_lib_1.Token.isUnresolved(dbName) || dbName === undefined) {
            return;
        }
        if (!/^[a-zA-Z][a-zA-Z_0-9+.@-]*$/.test(dbName)) {
            throw new Error(`\`dbName\` must start with a letter, can only contain letters, numbers, and the special characters: _, +, ., @, -, got: ${dbName}.`);
        }
        if (dbName.length > 127) {
            throw new Error(`\`dbName\` must not exceed 127 characters, got: ${dbName.length} characters.`);
        }
    }
    /**
     * Validates final snapshot settings.
     */
    validateFinalSnapshot() {
        const finalSnapshotName = this.props.finalSnapshotName;
        const finalSnapshotRetentionPeriod = this.props.finalSnapshotRetentionPeriod;
        if (aws_cdk_lib_1.Token.isUnresolved(finalSnapshotName))
            return;
        if (finalSnapshotName !== undefined) {
            if (!/^[a-z][a-z0-9]*(-[a-z0-9]+)*$/.test(finalSnapshotName)) {
                throw new Error(`\`finalSnapshotName\` must consist only of lowercase alphanumeric characters or hyphens, with the first character as a letter, and it can't end with a hyphen or contain two consecutive hyphens, got: ${finalSnapshotName}.`);
            }
            if (finalSnapshotName.length > 255) {
                throw new Error(`\`finalSnapshotName\` must not exceed 255 characters, got: ${finalSnapshotName.length} characters.`);
            }
        }
        if (aws_cdk_lib_1.Token.isUnresolved(finalSnapshotRetentionPeriod) || finalSnapshotRetentionPeriod === undefined) {
            return;
        }
        if (!finalSnapshotName) {
            throw new Error('You must set `finalSnapshotName` when you specify `finalSnapshotRetentionPeriod`.');
        }
        if (finalSnapshotRetentionPeriod < 1 || finalSnapshotRetentionPeriod > 3653) {
            {
                throw new Error(`\`finalSnapshotRetentionPeriod\` must be between 1 and 3653, got: ${finalSnapshotRetentionPeriod}.`);
            }
        }
    }
    /**
     * Validates role settings.
     */
    validateDefaultIamRole() {
        if (!this.props.defaultIamRole) {
            return;
        }
        if (!this.props.iamRoles || !this.props.iamRoles.includes(this.props.defaultIamRole)) {
            throw new Error('`defaultIamRole` must be included in `iamRoles`.');
        }
    }
    /**
     * Validates a namespace name.
     */
    validateNamespaceName() {
        const namespaceName = this.props.namespaceName;
        if (aws_cdk_lib_1.Token.isUnresolved(namespaceName) || namespaceName === undefined) {
            return;
        }
        if (!/^[a-z0-9-]+$/.test(namespaceName)) {
            throw new Error(`\`namespaceName\` must consist only of lowercase alphanumeric characters or hyphens, got: ${namespaceName}.`);
        }
        if (namespaceName.length < 3 || namespaceName.length > 64) {
            throw new Error(`\`namespaceName\` must be between 3 and 64 characters, got: ${namespaceName.length} characters.`);
        }
    }
    /**
     * Adds a role to the namespace
     *
     * @param role the role to add
     */
    addIamRole(role) {
        if (this.iamRoles.includes(role)) {
            throw new Error(`An adding IAM Role is already attached to the namespace, ARN: ${role.roleArn}.`);
        }
        this.iamRoles.push(role);
    }
}
exports.Namespace = Namespace;
_a = JSII_RTTI_SYMBOL_1;
Namespace[_a] = { fqn: "@open-constructs/aws-cdk.aws_redshiftserverless.Namespace", version: "0.1.2" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmFtZXNwYWNlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2F3cy1yZWRzaGlmdHNlcnZlcmxlc3MvbmFtZXNwYWNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsNkNBQWtIO0FBb0hsSDs7R0FFRztBQUNILElBQVksU0FlWDtBQWZELFdBQVksU0FBUztJQUNuQjs7T0FFRztJQUNILGlDQUFvQixDQUFBO0lBRXBCOztPQUVHO0lBQ0gsNkNBQWdDLENBQUE7SUFFaEM7O09BRUc7SUFDSCxrREFBcUMsQ0FBQTtBQUN2QyxDQUFDLEVBZlcsU0FBUyx5QkFBVCxTQUFTLFFBZXBCO0FBZ0JEOzs7Ozs7Ozs7Ozs7R0FZRztBQUNILE1BQWEsU0FBVSxTQUFRLHNCQUFRO0lBQ3JDOztPQUVHO0lBQ0ksTUFBTSxDQUFDLHVCQUF1QixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTBCO1FBQzVGLE1BQU0sTUFBTyxTQUFRLHNCQUFRO1lBQTdCOztnQkFDa0Isa0JBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO2dCQUNwQyxnQkFBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7Z0JBQ2hDLGlCQUFZLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO29CQUN0RCxPQUFPLEVBQUUscUJBQXFCO29CQUM5QixRQUFRLEVBQUUsV0FBVztvQkFDckIsWUFBWSxFQUFFLEtBQUssQ0FBQyxXQUFXO2lCQUNoQyxDQUFDLENBQUM7WUFDTCxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBbUJELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBcUI7UUFDN0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQ1YsS0FBSyxDQUFDLGFBQWE7Z0JBQ25CLGtCQUFJLENBQUMsTUFBTSxDQUFDO29CQUNWLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxtQkFBSyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxFQUFFLFNBQVMsRUFBRSxFQUFFLEVBQUUsd0JBQXdCLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxXQUFXLEVBQUU7aUJBQzlHLENBQUM7U0FDTCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQztRQUNuQixJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksRUFBRSxDQUFDO1FBRXJDLElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUNyQixJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7UUFDdEIsSUFBSSxDQUFDLHFCQUFxQixFQUFFLENBQUM7UUFDN0IsSUFBSSxDQUFDLHNCQUFzQixFQUFFLENBQUM7UUFDOUIsSUFBSSxDQUFDLHFCQUFxQixFQUFFLENBQUM7UUFFN0IsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3RELGFBQWEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWE7WUFDdkMsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsRUFBRSxZQUFZLEVBQUU7WUFDL0QsTUFBTSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTTtZQUN6QixpQkFBaUIsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsRUFBRSxPQUFPO1lBQ3JELGlCQUFpQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCO1lBQy9DLDRCQUE0QixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsNEJBQTRCO1lBQ3JFLFFBQVEsRUFBRSxrQkFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsRUFBRSxFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDO1lBQ3BHLFFBQVEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxLQUFLO1lBQ2xDLFVBQVUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVU7WUFDakMsYUFBYSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ2pDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxZQUFZLEdBQUcsU0FBUyxDQUFDLHlCQUF5QixDQUFDO1FBQ3hELElBQUksQ0FBQyxhQUFhLEdBQUcsU0FBUyxDQUFDLDBCQUEwQixDQUFDO1FBQzFELElBQUksQ0FBQyxXQUFXLEdBQUcsU0FBUyxDQUFDLHdCQUF3QixDQUFDO0lBQ3hELENBQUM7SUFFUyxjQUFjLENBQ3RCLEtBQWdCLEVBQ2hCLEVBQVUsRUFDVixLQUErQztRQUUvQyxPQUFPLElBQUksb0NBQXNCLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDbkUsQ0FBQztJQUVEOztPQUVHO0lBQ0ssYUFBYTtRQUNuQixNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUMvQyxNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUM7UUFFdkQsSUFBSSxtQkFBSyxDQUFDLFlBQVksQ0FBQyxhQUFhLENBQUMsRUFBRSxDQUFDO1lBQ3RDLE9BQU87UUFDVCxDQUFDO1FBRUQsSUFDRSxDQUFDLGFBQWEsS0FBSyxTQUFTLElBQUksaUJBQWlCLEtBQUssU0FBUyxDQUFDO1lBQ2hFLENBQUMsYUFBYSxLQUFLLFNBQVMsSUFBSSxpQkFBaUIsS0FBSyxTQUFTLENBQUMsRUFDaEUsQ0FBQztZQUNELE1BQU0sSUFBSSxLQUFLLENBQUMsNEVBQTRFLENBQUMsQ0FBQztRQUNoRyxDQUFDO1FBRUQsSUFBSSxhQUFhLElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEVBQUUsQ0FBQztZQUN4RSxNQUFNLElBQUksS0FBSyxDQUNiLHFJQUFxSSxhQUFhLEdBQUcsQ0FDdEosQ0FBQztRQUNKLENBQUM7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxjQUFjO1FBQ3BCLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDO1FBRWpDLElBQUksbUJBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLElBQUksTUFBTSxLQUFLLFNBQVMsRUFBRSxDQUFDO1lBQ3ZELE9BQU87UUFDVCxDQUFDO1FBRUQsSUFBSSxDQUFDLDZCQUE2QixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQ2IsMkhBQTJILE1BQU0sR0FBRyxDQUNySSxDQUFDO1FBQ0osQ0FBQztRQUVELElBQUksTUFBTSxDQUFDLE1BQU0sR0FBRyxHQUFHLEVBQUUsQ0FBQztZQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLG1EQUFtRCxNQUFNLENBQUMsTUFBTSxjQUFjLENBQUMsQ0FBQztRQUNsRyxDQUFDO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0sscUJBQXFCO1FBQzNCLE1BQU0saUJBQWlCLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUN2RCxNQUFNLDRCQUE0QixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsNEJBQTRCLENBQUM7UUFFN0UsSUFBSSxtQkFBSyxDQUFDLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQztZQUFFLE9BQU87UUFFbEQsSUFBSSxpQkFBaUIsS0FBSyxTQUFTLEVBQUUsQ0FBQztZQUNwQyxJQUFJLENBQUMsK0JBQStCLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsQ0FBQztnQkFDN0QsTUFBTSxJQUFJLEtBQUssQ0FDYiwwTUFBME0saUJBQWlCLEdBQUcsQ0FDL04sQ0FBQztZQUNKLENBQUM7WUFFRCxJQUFJLGlCQUFpQixDQUFDLE1BQU0sR0FBRyxHQUFHLEVBQUUsQ0FBQztnQkFDbkMsTUFBTSxJQUFJLEtBQUssQ0FDYiw4REFBOEQsaUJBQWlCLENBQUMsTUFBTSxjQUFjLENBQ3JHLENBQUM7WUFDSixDQUFDO1FBQ0gsQ0FBQztRQUVELElBQUksbUJBQUssQ0FBQyxZQUFZLENBQUMsNEJBQTRCLENBQUMsSUFBSSw0QkFBNEIsS0FBSyxTQUFTLEVBQUUsQ0FBQztZQUNuRyxPQUFPO1FBQ1QsQ0FBQztRQUVELElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsbUZBQW1GLENBQUMsQ0FBQztRQUN2RyxDQUFDO1FBRUQsSUFBSSw0QkFBNEIsR0FBRyxDQUFDLElBQUksNEJBQTRCLEdBQUcsSUFBSSxFQUFFLENBQUM7WUFDNUUsQ0FBQztnQkFDQyxNQUFNLElBQUksS0FBSyxDQUNiLHFFQUFxRSw0QkFBNEIsR0FBRyxDQUNyRyxDQUFDO1lBQ0osQ0FBQztRQUNILENBQUM7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxzQkFBc0I7UUFDNUIsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxFQUFFLENBQUM7WUFDL0IsT0FBTztRQUNULENBQUM7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsRUFBRSxDQUFDO1lBQ3JGLE1BQU0sSUFBSSxLQUFLLENBQUMsa0RBQWtELENBQUMsQ0FBQztRQUN0RSxDQUFDO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0sscUJBQXFCO1FBQzNCLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBRS9DLElBQUksbUJBQUssQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDLElBQUksYUFBYSxLQUFLLFNBQVMsRUFBRSxDQUFDO1lBQ3JFLE9BQU87UUFDVCxDQUFDO1FBRUQsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEVBQUUsQ0FBQztZQUN4QyxNQUFNLElBQUksS0FBSyxDQUNiLDZGQUE2RixhQUFhLEdBQUcsQ0FDOUcsQ0FBQztRQUNKLENBQUM7UUFFRCxJQUFJLGFBQWEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxJQUFJLGFBQWEsQ0FBQyxNQUFNLEdBQUcsRUFBRSxFQUFFLENBQUM7WUFDMUQsTUFBTSxJQUFJLEtBQUssQ0FDYiwrREFBK0QsYUFBYSxDQUFDLE1BQU0sY0FBYyxDQUNsRyxDQUFDO1FBQ0osQ0FBQztJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksVUFBVSxDQUFDLElBQVc7UUFDM0IsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO1lBQ2pDLE1BQU0sSUFBSSxLQUFLLENBQUMsaUVBQWlFLElBQUksQ0FBQyxPQUFPLEdBQUcsQ0FBQyxDQUFDO1FBQ3BHLENBQUM7UUFFRCxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMzQixDQUFDOztBQW5OSCw4QkFvTkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJUmVzb3VyY2UsIExhenksIE5hbWVzLCBSZXNvdXJjZSwgU2VjcmV0VmFsdWUsIFN0YWNrLCBUb2tlbiwgYXdzX3JlZHNoaWZ0c2VydmVybGVzcyB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IElSb2xlIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgeyBJS2V5IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWttcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuLyoqXG4gKiBBIFJlZHNoaWZ0IFNlcnZlcmxlc3MgTmFtZXNwYWNlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSU5hbWVzcGFjZSBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZXNwYWNlIEFSTlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBuYW1lc3BhY2VBcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWVzcGFjZSBuYW1lXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IG5hbWVzcGFjZU5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWVzcGFjZSBpZFxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBuYW1lc3BhY2VJZDogc3RyaW5nO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGRlZmluaW5nIGEgUmVkc2hpZnQgU2VydmVybGVzcyBOYW1lc3BhY2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTmFtZXNwYWNlUHJvcHMge1xuICAvKipcbiAgICogVGhlIHVzZXJuYW1lIG9mIHRoZSBhZG1pbmlzdHJhdG9yIGZvciB0aGUgcHJpbWFyeSBkYXRhYmFzZSBjcmVhdGVkIGluIHRoZSBuYW1lc3BhY2UuXG4gICAqXG4gICAqIFlvdSBtdXN0IHNwZWNpZnkgYm90aCBgYWRtaW5Vc2VybmFtZWAgYW5kIGBhZG1pblVzZXJQYXNzd29yZGAsIG9yIG5laXRoZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gYWRtaW4gdXNlclxuICAgKi9cbiAgcmVhZG9ubHkgYWRtaW5Vc2VybmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHBhc3N3b3JkIG9mIHRoZSBhZG1pbmlzdHJhdG9yIGZvciB0aGUgcHJpbWFyeSBkYXRhYmFzZSBjcmVhdGVkIGluIHRoZSBuYW1lc3BhY2UuXG4gICAqXG4gICAqIFlvdSBtdXN0IHNwZWNpZnkgYm90aCBgYWRtaW5Vc2VybmFtZWAgYW5kIGBhZG1pblVzZXJQYXNzd29yZGAsIG9yIG5laXRoZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gYWRtaW4gdXNlclxuICAgKi9cbiAgcmVhZG9ubHkgYWRtaW5Vc2VyUGFzc3dvcmQ/OiBTZWNyZXRWYWx1ZTtcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHByaW1hcnkgZGF0YWJhc2UgY3JlYXRlZCBpbiB0aGUgbmFtZXNwYWNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtICdkZXYnXG4gICAqL1xuICByZWFkb25seSBkYk5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBJQU0gcm9sZSB0byBzZXQgYXMgYSBkZWZhdWx0IGluIHRoZSBuYW1lc3BhY2UuXG4gICAqXG4gICAqIGBkZWZhdWx0SWFtUm9sZWAgbXVzdCBiZSBpbmNsdWRlZCBpbiBgaWFtUm9sZXNgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGRlZmF1bHQgSUFNIHJvbGVcbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRJYW1Sb2xlPzogSVJvbGU7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBzbmFwc2hvdCB0byBiZSBjcmVhdGVkIGJlZm9yZSB0aGUgbmFtZXNwYWNlIGlzIGRlbGV0ZWQuXG4gICAqXG4gICAqIElmIG5vdCBzcGVjaWZpZWQsIHRoZSBmaW5hbCBzbmFwc2hvdCB3aWxsIG5vdCBiZSB0YWtlbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBmaW5hbCBzbmFwc2hvdFxuICAgKi9cbiAgcmVhZG9ubHkgZmluYWxTbmFwc2hvdE5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEhvdyBsb25nIGRheXMgdG8gcmV0YWluIHRoZSBmaW5hbCBzbmFwc2hvdC5cbiAgICpcbiAgICogWW91IG11c3Qgc2V0IGBmaW5hbFNuYXBzaG90TmFtZWAgd2hlbiB5b3Ugc3BlY2lmeSBgZmluYWxTbmFwc2hvdFJldGVudGlvblBlcmlvZGAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gUmV0YWluZWQgaW5kZWZpbml0ZWx5IGlmIGBmaW5hbFNuYXBzaG90TmFtZWAgaXMgc3BlY2lmaWVkLCBvdGhlcndpc2Ugbm8gZmluYWwgc25hcHNob3RcbiAgICovXG4gIHJlYWRvbmx5IGZpbmFsU25hcHNob3RSZXRlbnRpb25QZXJpb2Q/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBJQU0gcm9sZXMgdG8gYXNzb2NpYXRlIHdpdGggdGhlIG5hbWVzcGFjZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBJQU0gcm9sZSBhc3NvY2lhdGVkXG4gICAqL1xuICByZWFkb25seSBpYW1Sb2xlcz86IElSb2xlW107XG5cbiAgLyoqXG4gICAqIEEgQ3VzdG9tZXIgTWFuYWdlZCBLZXkgdXNlZCB0byBlbmNyeXB0IHlvdXIgZGF0YS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB1c2UgQVdTIG1hbmFnZWQga2V5XG4gICAqL1xuICByZWFkb25seSBrbXNLZXk/OiBJS2V5O1xuXG4gIC8qKlxuICAgKiBUaGUgdHlwZXMgb2YgbG9ncyB0aGUgbmFtZXNwYWNlIGNhbiBleHBvcnQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gbG9ncyBleHBvcnRcbiAgICovXG4gIHJlYWRvbmx5IGxvZ0V4cG9ydHM/OiBMb2dFeHBvcnRbXTtcblxuICAvKipcbiAgICogVGhlIG5hbWVzcGFjZSBuYW1lLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGF1dG8gZ2VuZXJhdGVcbiAgICovXG4gIHJlYWRvbmx5IG5hbWVzcGFjZU5hbWU/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogVGhlIHR5cGVzIG9mIGxvZ3MgdGhlIG5hbWVzcGFjZSBjYW4gZXhwb3J0LlxuICovXG5leHBvcnQgZW51bSBMb2dFeHBvcnQge1xuICAvKipcbiAgICogVXNlciBsb2dcbiAgICovXG4gIFVTRVJfTE9HID0gJ3VzZXJsb2cnLFxuXG4gIC8qKlxuICAgKiBDb25uZWN0aW9uIGxvZ1xuICAgKi9cbiAgQ09OTkVDVElPTl9MT0cgPSAnY29ubmVjdGlvbmxvZycsXG5cbiAgLyoqXG4gICAqIFVzZXIgYWN0aXZpdHkgbG9nXG4gICAqL1xuICBVU0VSX0FDVElWSVRZX0xPRyA9ICd1c2VyYWN0aXZpdHlsb2cnLFxufVxuXG4vKipcbiAqIEF0dHJpYnV0ZXMgZm9yIGltcG9ydGluZyBhIFJlZHNoaWZ0IFNlcnZlcmxlc3MgTmFtZXNwYWNlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE5hbWVzcGFjZUF0dHJpYnV0ZXMge1xuICAvKipcbiAgICogVGhlIG5hbWVzcGFjZSBuYW1lXG4gICAqL1xuICByZWFkb25seSBuYW1lc3BhY2VOYW1lOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgbmFtZXNwYWNlIGlkXG4gICAqL1xuICByZWFkb25seSBuYW1lc3BhY2VJZDogc3RyaW5nO1xufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBSZWRzaGlmdCBTZXJ2ZXJsZXNzIE5hbWVzcGFjZSBjb25zdHJ1Y3QgaW4gQVdTIENESy5cbiAqXG4gKiBAZXhhbXBsZVxuICpcbiAqIGNvbnN0IG5hbWVTcGFjZSA9IG5ldyBOYW1lc3BhY2UoXG4gKiAgIHN0YWNrLFxuICogICAnTmFtZXNwYWNlJyxcbiAqICAge1xuICogICAgIG5hbWVzcGFjZU5hbWU6ICdteS1uYW1lc3BhY2UnLFxuICogICB9LFxuICogKTtcbiAqL1xuZXhwb3J0IGNsYXNzIE5hbWVzcGFjZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSU5hbWVzcGFjZSB7XG4gIC8qKlxuICAgKiBJbXBvcnRzIGFuIGV4aXN0aW5nIE5hbWVzcGFjZSBmcm9tIGF0dHJpYnV0ZXNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbU5hbWVzcGFjZUF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IE5hbWVzcGFjZUF0dHJpYnV0ZXMpOiBJTmFtZXNwYWNlIHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElOYW1lc3BhY2Uge1xuICAgICAgcHVibGljIHJlYWRvbmx5IG5hbWVzcGFjZU5hbWUgPSBhdHRycy5uYW1lc3BhY2VOYW1lO1xuICAgICAgcHVibGljIHJlYWRvbmx5IG5hbWVzcGFjZUlkID0gYXR0cnMubmFtZXNwYWNlSWQ7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgbmFtZXNwYWNlQXJuID0gU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHtcbiAgICAgICAgc2VydmljZTogJ3JlZHNoaWZ0LXNlcnZlcmxlc3MnLFxuICAgICAgICByZXNvdXJjZTogJ25hbWVzcGFjZScsXG4gICAgICAgIHJlc291cmNlTmFtZTogYXR0cnMubmFtZXNwYWNlSWQsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lc3BhY2UgQXJuXG4gICAqL1xuICByZWFkb25seSBuYW1lc3BhY2VBcm46IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBuYW1lc3BhY2UgbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZXNwYWNlTmFtZTogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIG5hbWVzcGFjZSBpZFxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZXNwYWNlSWQ6IHN0cmluZztcblxuICBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBOYW1lc3BhY2VQcm9wcztcblxuICBwcml2YXRlIHJlYWRvbmx5IGlhbVJvbGVzOiBJUm9sZVtdO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBOYW1lc3BhY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOlxuICAgICAgICBwcm9wcy5uYW1lc3BhY2VOYW1lID8/XG4gICAgICAgIExhenkuc3RyaW5nKHtcbiAgICAgICAgICBwcm9kdWNlOiAoKSA9PiBOYW1lcy51bmlxdWVSZXNvdXJjZU5hbWUodGhpcywgeyBtYXhMZW5ndGg6IDY0LCBhbGxvd2VkU3BlY2lhbENoYXJhY3RlcnM6ICctJyB9KS50b0xvd2VyQ2FzZSgpLFxuICAgICAgICB9KSxcbiAgICB9KTtcbiAgICB0aGlzLnByb3BzID0gcHJvcHM7XG4gICAgdGhpcy5pYW1Sb2xlcyA9IHByb3BzLmlhbVJvbGVzID8/IFtdO1xuXG4gICAgdGhpcy52YWxpZGF0ZUFkbWluKCk7XG4gICAgdGhpcy52YWxpZGF0ZURiTmFtZSgpO1xuICAgIHRoaXMudmFsaWRhdGVGaW5hbFNuYXBzaG90KCk7XG4gICAgdGhpcy52YWxpZGF0ZURlZmF1bHRJYW1Sb2xlKCk7XG4gICAgdGhpcy52YWxpZGF0ZU5hbWVzcGFjZU5hbWUoKTtcblxuICAgIGNvbnN0IG5hbWVzcGFjZSA9IHRoaXMuY3JlYXRlUmVzb3VyY2UodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgYWRtaW5Vc2VybmFtZTogdGhpcy5wcm9wcy5hZG1pblVzZXJuYW1lLFxuICAgICAgYWRtaW5Vc2VyUGFzc3dvcmQ6IHRoaXMucHJvcHMuYWRtaW5Vc2VyUGFzc3dvcmQ/LnVuc2FmZVVud3JhcCgpLFxuICAgICAgZGJOYW1lOiB0aGlzLnByb3BzLmRiTmFtZSxcbiAgICAgIGRlZmF1bHRJYW1Sb2xlQXJuOiB0aGlzLnByb3BzLmRlZmF1bHRJYW1Sb2xlPy5yb2xlQXJuLFxuICAgICAgZmluYWxTbmFwc2hvdE5hbWU6IHRoaXMucHJvcHMuZmluYWxTbmFwc2hvdE5hbWUsXG4gICAgICBmaW5hbFNuYXBzaG90UmV0ZW50aW9uUGVyaW9kOiB0aGlzLnByb3BzLmZpbmFsU25hcHNob3RSZXRlbnRpb25QZXJpb2QsXG4gICAgICBpYW1Sb2xlczogTGF6eS5saXN0KHsgcHJvZHVjZTogKCkgPT4gdGhpcy5pYW1Sb2xlcy5tYXAocm9sZSA9PiByb2xlLnJvbGVBcm4pIH0sIHsgb21pdEVtcHR5OiB0cnVlIH0pLFxuICAgICAga21zS2V5SWQ6IHRoaXMucHJvcHMua21zS2V5Py5rZXlJZCxcbiAgICAgIGxvZ0V4cG9ydHM6IHRoaXMucHJvcHMubG9nRXhwb3J0cyxcbiAgICAgIG5hbWVzcGFjZU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgIH0pO1xuXG4gICAgdGhpcy5uYW1lc3BhY2VBcm4gPSBuYW1lc3BhY2UuYXR0ck5hbWVzcGFjZU5hbWVzcGFjZUFybjtcbiAgICB0aGlzLm5hbWVzcGFjZU5hbWUgPSBuYW1lc3BhY2UuYXR0ck5hbWVzcGFjZU5hbWVzcGFjZU5hbWU7XG4gICAgdGhpcy5uYW1lc3BhY2VJZCA9IG5hbWVzcGFjZS5hdHRyTmFtZXNwYWNlTmFtZXNwYWNlSWQ7XG4gIH1cblxuICBwcm90ZWN0ZWQgY3JlYXRlUmVzb3VyY2UoXG4gICAgc2NvcGU6IENvbnN0cnVjdCxcbiAgICBpZDogc3RyaW5nLFxuICAgIHByb3BzOiBhd3NfcmVkc2hpZnRzZXJ2ZXJsZXNzLkNmbk5hbWVzcGFjZVByb3BzLFxuICApOiBhd3NfcmVkc2hpZnRzZXJ2ZXJsZXNzLkNmbk5hbWVzcGFjZSB7XG4gICAgcmV0dXJuIG5ldyBhd3NfcmVkc2hpZnRzZXJ2ZXJsZXNzLkNmbk5hbWVzcGFjZShzY29wZSwgaWQsIHByb3BzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZXMgYWRtaW4gc2V0dGluZ3MuXG4gICAqL1xuICBwcml2YXRlIHZhbGlkYXRlQWRtaW4oKTogdm9pZCB7XG4gICAgY29uc3QgYWRtaW5Vc2VybmFtZSA9IHRoaXMucHJvcHMuYWRtaW5Vc2VybmFtZTtcbiAgICBjb25zdCBhZG1pblVzZXJQYXNzd29yZCA9IHRoaXMucHJvcHMuYWRtaW5Vc2VyUGFzc3dvcmQ7XG5cbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKGFkbWluVXNlcm5hbWUpKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgaWYgKFxuICAgICAgKGFkbWluVXNlcm5hbWUgIT09IHVuZGVmaW5lZCAmJiBhZG1pblVzZXJQYXNzd29yZCA9PT0gdW5kZWZpbmVkKSB8fFxuICAgICAgKGFkbWluVXNlcm5hbWUgPT09IHVuZGVmaW5lZCAmJiBhZG1pblVzZXJQYXNzd29yZCAhPT0gdW5kZWZpbmVkKVxuICAgICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgbXVzdCBzcGVjaWZ5IGJvdGggYGFkbWluVXNlcm5hbWVgIGFuZCBgYWRtaW5Vc2VyUGFzc3dvcmRgLCBvciBuZWl0aGVyLicpO1xuICAgIH1cblxuICAgIGlmIChhZG1pblVzZXJuYW1lICYmICEvXlthLXpBLVpdW2EtekEtWl8wLTkrLkAtXSokLy50ZXN0KGFkbWluVXNlcm5hbWUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIGBcXGBhZG1pblVzZXJuYW1lXFxgIG11c3Qgc3RhcnQgd2l0aCBhIGxldHRlciBhbmQgY2FuIG9ubHkgY29udGFpbiBsZXR0ZXJzLCBudW1iZXJzLCBhbmQgdGhlIHNwZWNpYWwgY2hhcmFjdGVyczogXywgKywgLiwgQCwgLSwgZ290OiAke2FkbWluVXNlcm5hbWV9LmAsXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZXMgYSBkYXRhYmFzZSBuYW1lLlxuICAgKi9cbiAgcHJpdmF0ZSB2YWxpZGF0ZURiTmFtZSgpOiB2b2lkIHtcbiAgICBjb25zdCBkYk5hbWUgPSB0aGlzLnByb3BzLmRiTmFtZTtcblxuICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQoZGJOYW1lKSB8fCBkYk5hbWUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGlmICghL15bYS16QS1aXVthLXpBLVpfMC05Ky5ALV0qJC8udGVzdChkYk5hbWUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIGBcXGBkYk5hbWVcXGAgbXVzdCBzdGFydCB3aXRoIGEgbGV0dGVyLCBjYW4gb25seSBjb250YWluIGxldHRlcnMsIG51bWJlcnMsIGFuZCB0aGUgc3BlY2lhbCBjaGFyYWN0ZXJzOiBfLCArLCAuLCBALCAtLCBnb3Q6ICR7ZGJOYW1lfS5gLFxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAoZGJOYW1lLmxlbmd0aCA+IDEyNykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBcXGBkYk5hbWVcXGAgbXVzdCBub3QgZXhjZWVkIDEyNyBjaGFyYWN0ZXJzLCBnb3Q6ICR7ZGJOYW1lLmxlbmd0aH0gY2hhcmFjdGVycy5gKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogVmFsaWRhdGVzIGZpbmFsIHNuYXBzaG90IHNldHRpbmdzLlxuICAgKi9cbiAgcHJpdmF0ZSB2YWxpZGF0ZUZpbmFsU25hcHNob3QoKTogdm9pZCB7XG4gICAgY29uc3QgZmluYWxTbmFwc2hvdE5hbWUgPSB0aGlzLnByb3BzLmZpbmFsU25hcHNob3ROYW1lO1xuICAgIGNvbnN0IGZpbmFsU25hcHNob3RSZXRlbnRpb25QZXJpb2QgPSB0aGlzLnByb3BzLmZpbmFsU25hcHNob3RSZXRlbnRpb25QZXJpb2Q7XG5cbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKGZpbmFsU25hcHNob3ROYW1lKSkgcmV0dXJuO1xuXG4gICAgaWYgKGZpbmFsU25hcHNob3ROYW1lICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIGlmICghL15bYS16XVthLXowLTldKigtW2EtejAtOV0rKSokLy50ZXN0KGZpbmFsU25hcHNob3ROYW1lKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgYFxcYGZpbmFsU25hcHNob3ROYW1lXFxgIG11c3QgY29uc2lzdCBvbmx5IG9mIGxvd2VyY2FzZSBhbHBoYW51bWVyaWMgY2hhcmFjdGVycyBvciBoeXBoZW5zLCB3aXRoIHRoZSBmaXJzdCBjaGFyYWN0ZXIgYXMgYSBsZXR0ZXIsIGFuZCBpdCBjYW4ndCBlbmQgd2l0aCBhIGh5cGhlbiBvciBjb250YWluIHR3byBjb25zZWN1dGl2ZSBoeXBoZW5zLCBnb3Q6ICR7ZmluYWxTbmFwc2hvdE5hbWV9LmAsXG4gICAgICAgICk7XG4gICAgICB9XG5cbiAgICAgIGlmIChmaW5hbFNuYXBzaG90TmFtZS5sZW5ndGggPiAyNTUpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgIGBcXGBmaW5hbFNuYXBzaG90TmFtZVxcYCBtdXN0IG5vdCBleGNlZWQgMjU1IGNoYXJhY3RlcnMsIGdvdDogJHtmaW5hbFNuYXBzaG90TmFtZS5sZW5ndGh9IGNoYXJhY3RlcnMuYCxcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKGZpbmFsU25hcHNob3RSZXRlbnRpb25QZXJpb2QpIHx8IGZpbmFsU25hcHNob3RSZXRlbnRpb25QZXJpb2QgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGlmICghZmluYWxTbmFwc2hvdE5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignWW91IG11c3Qgc2V0IGBmaW5hbFNuYXBzaG90TmFtZWAgd2hlbiB5b3Ugc3BlY2lmeSBgZmluYWxTbmFwc2hvdFJldGVudGlvblBlcmlvZGAuJyk7XG4gICAgfVxuXG4gICAgaWYgKGZpbmFsU25hcHNob3RSZXRlbnRpb25QZXJpb2QgPCAxIHx8IGZpbmFsU25hcHNob3RSZXRlbnRpb25QZXJpb2QgPiAzNjUzKSB7XG4gICAgICB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICBgXFxgZmluYWxTbmFwc2hvdFJldGVudGlvblBlcmlvZFxcYCBtdXN0IGJlIGJldHdlZW4gMSBhbmQgMzY1MywgZ290OiAke2ZpbmFsU25hcHNob3RSZXRlbnRpb25QZXJpb2R9LmAsXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlcyByb2xlIHNldHRpbmdzLlxuICAgKi9cbiAgcHJpdmF0ZSB2YWxpZGF0ZURlZmF1bHRJYW1Sb2xlKCk6IHZvaWQge1xuICAgIGlmICghdGhpcy5wcm9wcy5kZWZhdWx0SWFtUm9sZSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGlmICghdGhpcy5wcm9wcy5pYW1Sb2xlcyB8fCAhdGhpcy5wcm9wcy5pYW1Sb2xlcy5pbmNsdWRlcyh0aGlzLnByb3BzLmRlZmF1bHRJYW1Sb2xlKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdgZGVmYXVsdElhbVJvbGVgIG11c3QgYmUgaW5jbHVkZWQgaW4gYGlhbVJvbGVzYC4nKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogVmFsaWRhdGVzIGEgbmFtZXNwYWNlIG5hbWUuXG4gICAqL1xuICBwcml2YXRlIHZhbGlkYXRlTmFtZXNwYWNlTmFtZSgpOiB2b2lkIHtcbiAgICBjb25zdCBuYW1lc3BhY2VOYW1lID0gdGhpcy5wcm9wcy5uYW1lc3BhY2VOYW1lO1xuXG4gICAgaWYgKFRva2VuLmlzVW5yZXNvbHZlZChuYW1lc3BhY2VOYW1lKSB8fCBuYW1lc3BhY2VOYW1lID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBpZiAoIS9eW2EtejAtOS1dKyQvLnRlc3QobmFtZXNwYWNlTmFtZSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgYFxcYG5hbWVzcGFjZU5hbWVcXGAgbXVzdCBjb25zaXN0IG9ubHkgb2YgbG93ZXJjYXNlIGFscGhhbnVtZXJpYyBjaGFyYWN0ZXJzIG9yIGh5cGhlbnMsIGdvdDogJHtuYW1lc3BhY2VOYW1lfS5gLFxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAobmFtZXNwYWNlTmFtZS5sZW5ndGggPCAzIHx8IG5hbWVzcGFjZU5hbWUubGVuZ3RoID4gNjQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgYFxcYG5hbWVzcGFjZU5hbWVcXGAgbXVzdCBiZSBiZXR3ZWVuIDMgYW5kIDY0IGNoYXJhY3RlcnMsIGdvdDogJHtuYW1lc3BhY2VOYW1lLmxlbmd0aH0gY2hhcmFjdGVycy5gLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJvbGUgdG8gdGhlIG5hbWVzcGFjZVxuICAgKlxuICAgKiBAcGFyYW0gcm9sZSB0aGUgcm9sZSB0byBhZGRcbiAgICovXG4gIHB1YmxpYyBhZGRJYW1Sb2xlKHJvbGU6IElSb2xlKTogdm9pZCB7XG4gICAgaWYgKHRoaXMuaWFtUm9sZXMuaW5jbHVkZXMocm9sZSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQW4gYWRkaW5nIElBTSBSb2xlIGlzIGFscmVhZHkgYXR0YWNoZWQgdG8gdGhlIG5hbWVzcGFjZSwgQVJOOiAke3JvbGUucm9sZUFybn0uYCk7XG4gICAgfVxuXG4gICAgdGhpcy5pYW1Sb2xlcy5wdXNoKHJvbGUpO1xuICB9XG59XG4iXX0=