"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workgroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * Represents a Redshift Serverless Workgroup construct in AWS CDK.
 *
 * @example
 * declare const namespace: Namespace;
 * declare const vpc: aws_ec2.IVpc;
 *
 * const nameSpace = new Workgroup(
 *   stack,
 *   'Workgroup',
 *   {
 *     workgroupName: 'my-workgroup',
 *     namespace: namespace,
 *     vpc,
 *   },
 * );
 */
class Workgroup extends aws_cdk_lib_1.Resource {
    /**
     * Import an existing workgroup to the stack from its attributes.
     */
    static fromWorkgroupAttributes(scope, id, attrs) {
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.workgroupName = attrs.workgroupName;
                this.workgroupId = attrs.workgroupId;
                this.endpointAddress = attrs.endpointAddress;
                this.port = attrs.port;
                this.connections = new aws_cdk_lib_1.aws_ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: aws_cdk_lib_1.aws_ec2.Port.tcp(attrs.port),
                });
                this.workgroupArn = aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'redshift-serverless',
                    resource: 'workgroup',
                    resourceName: attrs.workgroupId,
                });
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.workgroupName ??
                aws_cdk_lib_1.Lazy.string({
                    produce: () => aws_cdk_lib_1.Names.uniqueResourceName(this, { maxLength: 64, allowedSpecialCharacters: '-' }).toLowerCase(),
                }),
        });
        this.props = props;
        this.securityGroups = props.securityGroups ?? [
            this.createSecurityGroup(this, 'SecurityGroup', {
                vpc: this.props.vpc,
                description: 'Automatic generated security group for Redshift Serverless Security Group',
            }),
        ];
        this.connections = new aws_cdk_lib_1.aws_ec2.Connections({
            securityGroups: this.securityGroups,
        });
        this.vpcSubnets = props.vpcSubnets ?? {
            subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_EGRESS,
        };
        this.validateCapacity();
        this.validateWorkgroupName();
        this.validatePort();
        this.validateSubnet();
        const workgroup = this.createResource(this, 'Resource', {
            baseCapacity: this.props.baseCapacity,
            configParameters: this.props.configParameters
                ? Object.entries(this.props.configParameters).map(([key, value]) => ({
                    parameterKey: key,
                    parameterValue: value,
                }))
                : undefined,
            enhancedVpcRouting: this.props.enhancedVpcRouting,
            namespaceName: this.props.namespace?.namespaceName,
            publiclyAccessible: this.props.publiclyAccessible,
            port: this.props.port,
            securityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
            subnetIds: this.props.vpc.selectSubnets(this.vpcSubnets).subnetIds,
            workgroupName: this.physicalName,
        });
        this.workgroupArn = workgroup.attrWorkgroupWorkgroupArn;
        this.workgroupName = workgroup.attrWorkgroupWorkgroupName;
        this.workgroupId = workgroup.attrWorkgroupWorkgroupId;
        this.endpointAddress = workgroup.attrWorkgroupEndpointAddress;
        this.port = workgroup.attrWorkgroupEndpointPort;
    }
    createResource(scope, id, props) {
        return new aws_cdk_lib_1.aws_redshiftserverless.CfnWorkgroup(scope, id, props);
    }
    createSecurityGroup(scope, id, props) {
        return new aws_cdk_lib_1.aws_ec2.SecurityGroup(scope, id, props);
    }
    /**
     * Validates capacity settings.
     */
    validateCapacity() {
        const baseCapacity = this.props.baseCapacity;
        if (!aws_cdk_lib_1.Token.isUnresolved(baseCapacity) && baseCapacity !== undefined) {
            if (baseCapacity < 8 || baseCapacity > 1024) {
                throw new Error(`\`baseCapacity\` must be between 8 and 1024, got: ${baseCapacity}.`);
            }
            if (8 <= baseCapacity && baseCapacity <= 512 && baseCapacity % 8 !== 0) {
                throw new Error(`\`baseCapacity\` must be units of 8 between 8 and 512, got: ${baseCapacity}.`);
            }
            if (512 <= baseCapacity && baseCapacity <= 1024 && baseCapacity % 32 !== 0) {
                throw new Error(`\`baseCapacity\` must be units of 32 between 512 and 1024, got: ${baseCapacity}.`);
            }
        }
    }
    /**
     * Validates a workgroup name.
     */
    validateWorkgroupName() {
        const workgroupName = this.props.workgroupName;
        if (aws_cdk_lib_1.Token.isUnresolved(workgroupName) || workgroupName === undefined) {
            return;
        }
        if (!/^[a-z0-9-]+$/.test(workgroupName)) {
            throw new Error(`\`workgroupName\` must contain only lowercase letters, numbers, and hyphens, got: ${workgroupName}.`);
        }
        if (workgroupName.length < 3 || workgroupName.length > 64) {
            throw new Error(`\`workgroupName\` must be between 3 and 64 characters, got: ${workgroupName.length} characters.`);
        }
    }
    /**
     * Validates a port number.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-connecting.html
     */
    validatePort() {
        const port = this.props.port;
        if (!aws_cdk_lib_1.Token.isUnresolved(port) && port !== undefined) {
            const isValidPort = (port >= 5431 && port <= 5455) || (port >= 8191 && port <= 8215);
            if (!isValidPort) {
                throw new Error(`\`port\` must be in the range of 5431-5455 or 8191-8215 for Amazon Redshift Serverless, got: ${port}.`);
            }
        }
    }
    /**
     * Validates subnets.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-usage-considerations.html
     */
    validateSubnet() {
        const azLength = this.props.vpc.availabilityZones.length;
        if (azLength < 3) {
            throw new Error(`\`vpc\` must have at least 3 subnets, and they must span across 3 Availability Zones, got: ${azLength} AZs.`);
        }
    }
}
exports.Workgroup = Workgroup;
_a = JSII_RTTI_SYMBOL_1;
Workgroup[_a] = { fqn: "@open-constructs/aws-cdk.aws_redshiftserverless.Workgroup", version: "0.1.2" };
//# sourceMappingURL=data:application/json;base64,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