import instaloader
import os
import logging
import time
from typing import Dict, Any, List
from .base import VideoProvider, StreamQuality
from ..exceptions import VideoNotFoundError, ProviderAuthenticationError, StreamUnavailableError, InvalidURLError, UnsupportedQualityError
from tubefetcherlib.config import Config

logger = logging.getLogger(__name__)

class InstagramProvider(VideoProvider):
    def __init__(self, username: str = None, password: str = None, config: Config = Config()):
        self.config = config
        self.L = instaloader.Instaloader(compress_json=False, 
                                         dirname_pattern="{target}", 
                                         filename_pattern="{shortcode}",
                                         download_videos=False, # We only want metadata/urls
                                         download_geotags=False,
                                         download_comments=False,
                                         download_pictures=False,
                                         download_video_thumbnails=False,
                                         save_metadata=False,
                                         post_metadata_txt_pattern="",
                                         quiet=True)
        
        # Apply proxy if configured
        if self.config.proxies:
            self.L.context.proxies = self.config.proxies

        if username and password:
            session_file_path = os.path.join(self.config.instagram_session_dir, f"{username}.session")
            try:
                self.L.load_session_from_file(username, filename=session_file_path)
                logger.info(f"Instaloader session loaded for {username}")
            except FileNotFoundError:
                logger.info(f"No Instaloader session found for {username}. Attempting login...")
                try:
                    self.L.login(username, password)
                    self.L.save_session_to_file(filename=session_file_path)
                    logger.info(f"Instaloader login successful for {username}")
                except Exception as e:
                    logger.warning(f"Could not log in to Instagram with provided credentials: {e}")
                    raise ProviderAuthenticationError(f"Could not log in to Instagram: {e}")
        else:
            logger.info("Instaloader will operate in public mode. Some content may not be accessible.")

    def _get_post_from_url(self, video_url: str) -> instaloader.Post:
        parts = video_url.split("/")
        shortcode = None
        for i, part in enumerate(parts):
            if part == "p" and i + 1 < len(parts):
                shortcode = parts[i+1]
                break
            elif part == "reel" and i + 1 < len(parts):
                shortcode = parts[i+1]
                break
            elif part == "tv" and i + 1 < len(parts):
                shortcode = parts[i+1]
                break
        
        if not shortcode:
            raise InvalidURLError("Could not extract shortcode from Instagram URL.")

        for attempt in range(self.config.retries):
            try:
                post = instaloader.Post.from_shortcode(self.L.context, shortcode)
                return post
            except instaloader.exceptions.BadResponseException as e:
                logger.warning(f"Attempt {attempt + 1}/{self.config.retries}: Bad response from Instagram for {video_url}: {e}")
                if attempt < self.config.retries - 1:
                    time.sleep(self.config.backoff_factor * (2 ** attempt))
                else:
                    raise VideoNotFoundError(f"Instagram post not found or inaccessible after multiple retries: {e}")
            except Exception as e:
                logger.error(f"Error fetching Instagram post for {video_url}: {e}")
                raise VideoNotFoundError(f"Error fetching Instagram post: {e}")

    def get_stream_url(self, video_url: str, quality: StreamQuality = StreamQuality.HIGHEST) -> str:
        post = self._get_post_from_url(video_url)
        if not post.is_video:
            raise StreamUnavailableError("The provided URL is not for a video.")
        
        if quality != StreamQuality.HIGHEST:
            raise UnsupportedQualityError("InstagramProvider currently only supports 'highest' quality for video streams.")

        return post.video_url

    def get_video_info(self, video_url: str) -> Dict[str, Any]:
        post = self._get_post_from_url(video_url)
        return {
            "title": post.title if post.title else f"Instagram Video by {post.owner_username}",
            "author": post.owner_username,
            "likes": post.likes,
            "comments": post.comments,
            "date_utc": post.date_utc.isoformat(),
            "is_video": post.is_video,
            "video_url": post.video_url if post.is_video else None,
            "thumbnail_url": post.url, # A URL do post é a thumbnail para imagens/videos
        }

    def get_available_qualities(self, video_url: str) -> List[StreamQuality]:
        post = self._get_post_from_url(video_url)
        if post.is_video:
            return [StreamQuality.HIGHEST]
        return []