# MIT License
#
# Copyright (c) 2018 Evgeny Medvedev, evge.medvedev@gmail.com
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.


import pytest

from celoetl.service.eth_contract_service import EthContractService


@pytest.mark.parametrize("bytecode,expected_sighashes,is_erc20,is_erc721", [
    (
            '0x0x6060604052361561011b576000357c0100000000000000000000000000000000000000000000000000000000900463ffffffff16806306fdde031461011d57806307da68f51461014b578063095ea7b31461015d57806313af4035146101b457806318160ddd146101ea57806323b872dd14610210578063313ce567146102865780633452f51d146102ac5780635ac801fe1461031557806369d3e20e1461033957806370a082311461036b57806375f12b21146103b55780637a9e5e4b146103df5780638402181f146104155780638da5cb5b1461047e57806390bc1693146104d057806395d89b4114610502578063a9059cbb14610530578063be9a655514610587578063bf7e214f14610599578063dd62ed3e146105eb575bfe5b341561012557fe5b61012d610654565b60405180826000191660001916815260200191505060405180910390f35b341561015357fe5b61015b61065a565b005b341561016557fe5b61019a600480803573ffffffffffffffffffffffffffffffffffffffff1690602001909190803590602001909190505061075e565b604051808215151515815260200191505060405180910390f35b34156101bc57fe5b6101e8600480803573ffffffffffffffffffffffffffffffffffffffff1690602001909190505061083c565b005b34156101f257fe5b6101fa610920565b6040518082815260200191505060405180910390f35b341561021857fe5b61026c600480803573ffffffffffffffffffffffffffffffffffffffff1690602001909190803573ffffffffffffffffffffffffffffffffffffffff1690602001909190803590602001909190505061092b565b604051808215151515815260200191505060405180910390f35b341561028e57fe5b610296610a0b565b6040518082815260200191505060405180910390f35b34156102b457fe5b6102fb600480803573ffffffffffffffffffffffffffffffffffffffff169060200190919080356fffffffffffffffffffffffffffffffff16906020019091905050610a11565b604051808215151515815260200191505060405180910390f35b341561031d57fe5b610337600480803560001916906020019091905050610a38565b005b341561034157fe5b61036960048080356fffffffffffffffffffffffffffffffff16906020019091905050610a7e565b005b341561037357fe5b61039f600480803573ffffffffffffffffffffffffffffffffffffffff16906020019091905050610c44565b6040518082815260200191505060405180910390f35b34156103bd57fe5b6103c5610c8e565b604051808215151515815260200191505060405180910390f35b34156103e757fe5b610413600480803573ffffffffffffffffffffffffffffffffffffffff16906020019091905050610ca1565b005b341561041d57fe5b610464600480803573ffffffffffffffffffffffffffffffffffffffff169060200190919080356fffffffffffffffffffffffffffffffff16906020019091905050610d85565b604051808215151515815260200191505060405180910390f35b341561048657fe5b61048e610dad565b604051808273ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200191505060405180910390f35b34156104d857fe5b61050060048080356fffffffffffffffffffffffffffffffff16906020019091905050610dd3565b005b341561050a57fe5b610512610f99565b60405180826000191660001916815260200191505060405180910390f35b341561053857fe5b61056d600480803573ffffffffffffffffffffffffffffffffffffffff16906020019091908035906020019091905050610f9f565b604051808215151515815260200191505060405180910390f35b341561058f57fe5b61059761107d565b005b34156105a157fe5b6105a9611181565b604051808273ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200191505060405180910390f35b34156105f357fe5b61063e600480803573ffffffffffffffffffffffffffffffffffffffff1690602001909190803573ffffffffffffffffffffffffffffffffffffffff169060200190919050506111a7565b6040518082815260200191505060405180910390f35b60075481565b61069061068b336000357fffffffff000000000000000000000000000000000000000000000000000000001661122f565b611491565b6000600060043591506024359050806000191682600019163373ffffffffffffffffffffffffffffffffffffffff166000357fffffffff00000000000000000000000000000000000000000000000000000000167bffffffffffffffffffffffffffffffffffffffffffffffffffffffff19163460003660405180848152602001806020018281038252848482818152602001925080828437820191505094505050505060405180910390a46001600460146101000a81548160ff0219169083151502179055505b5b50505b565b6000610779600460149054906101000a900460ff1615611491565b6000600060043591506024359050806000191682600019163373ffffffffffffffffffffffffffffffffffffffff166000357fffffffff00000000000000000000000000000000000000000000000000000000167bffffffffffffffffffffffffffffffffffffffffffffffffffffffff19163460003660405180848152602001806020018281038252848482818152602001925080828437820191505094505050505060405180910390a461082f85856114a2565b92505b5b50505b92915050565b61087261086d336000357fffffffff000000000000000000000000000000000000000000000000000000001661122f565b611491565b80600460006101000a81548173ffffffffffffffffffffffffffffffffffffffff021916908373ffffffffffffffffffffffffffffffffffffffff160217905550600460009054906101000a900473ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff167fce241d7ca1f669fee44b6fc00b8eba2df3bb514eed0f6f668f8f89096e81ed9460405180905060405180910390a25b5b50565b600060005490505b90565b6000610946600460149054906101000a900460ff1615611491565b6000600060043591506024359050806000191682600019163373ffffffffffffffffffffffffffffffffffffffff166000357fffffffff00000000000000000000000000000000000000000000000000000000167bffffffffffffffffffffffffffffffffffffffffffffffffffffffff19163460003660405180848152602001806020018281038252848482818152602001925080828437820191505094505050505060405180910390a46109fd868686611595565b92505b5b50505b9392505050565b60065481565b6000610a2f83836fffffffffffffffffffffffffffffffff16610f9f565b90505b92915050565b610a6e610a69336000357fffffffff000000000000000000000000000000000000000000000000000000001661122f565b611491565b80600781600019169055505b5b50565b610ab4610aaf336000357fffffffff000000000000000000000000000000000000000000000000000000001661122f565b611491565b610acd600460149054906101000a900460ff1615611491565b6000600060043591506024359050806000191682600019163373ffffffffffffffffffffffffffffffffffffffff166000357fffffffff00000000000000000000000000000000000000000000000000000000167bffffffffffffffffffffffffffffffffffffffffffffffffffffffff19163460003660405180848152602001806020018281038252848482818152602001925080828437820191505094505050505060405180910390a4610bd4600160003373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002054846fffffffffffffffffffffffffffffffff166118f9565b600160003373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002081905550610c35600054846fffffffffffffffffffffffffffffffff166118f9565b6000819055505b5b50505b5b50565b6000600160008373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff1681526020019081526020016000205490505b919050565b600460149054906101000a900460ff1681565b610cd7610cd2336000357fffffffff000000000000000000000000000000000000000000000000000000001661122f565b611491565b80600360006101000a81548173ffffffffffffffffffffffffffffffffffffffff021916908373ffffffffffffffffffffffffffffffffffffffff160217905550600360009054906101000a900473ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff167f1abebea81bfa2637f28358c371278fb15ede7ea8dd28d2e03b112ff6d936ada460405180905060405180910390a25b5b50565b6000610da48333846fffffffffffffffffffffffffffffffff1661092b565b90505b92915050565b600460009054906101000a900473ffffffffffffffffffffffffffffffffffffffff1681565b610e09610e04336000357fffffffff000000000000000000000000000000000000000000000000000000001661122f565b611491565b610e22600460149054906101000a900460ff1615611491565b6000600060043591506024359050806000191682600019163373ffffffffffffffffffffffffffffffffffffffff166000357fffffffff00000000000000000000000000000000000000000000000000000000167bffffffffffffffffffffffffffffffffffffffffffffffffffffffff19163460003660405180848152602001806020018281038252848482818152602001925080828437820191505094505050505060405180910390a4610f29600160003373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002054846fffffffffffffffffffffffffffffffff16611913565b600160003373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002081905550610f8a600054846fffffffffffffffffffffffffffffffff16611913565b6000819055505b5b50505b5b50565b60055481565b6000610fba600460149054906101000a900460ff1615611491565b6000600060043591506024359050806000191682600019163373ffffffffffffffffffffffffffffffffffffffff166000357fffffffff00000000000000000000000000000000000000000000000000000000167bffffffffffffffffffffffffffffffffffffffffffffffffffffffff19163460003660405180848152602001806020018281038252848482818152602001925080828437820191505094505050505060405180910390a4611070858561192d565b92505b5b50505b92915050565b6110b36110ae336000357fffffffff000000000000000000000000000000000000000000000000000000001661122f565b611491565b6000600060043591506024359050806000191682600019163373ffffffffffffffffffffffffffffffffffffffff166000357fffffffff00000000000000000000000000000000000000000000000000000000167bffffffffffffffffffffffffffffffffffffffffffffffffffffffff19163460003660405180848152602001806020018281038252848482818152602001925080828437820191505094505050505060405180910390a46000600460146101000a81548160ff0219169083151502179055505b5b50505b565b600360009054906101000a900473ffffffffffffffffffffffffffffffffffffffff1681565b6000600260008473ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002060008373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff1681526020019081526020016000205490505b92915050565b60003073ffffffffffffffffffffffffffffffffffffffff168373ffffffffffffffffffffffffffffffffffffffff16141561126e576001905061148b565b600460009054906101000a900473ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff168373ffffffffffffffffffffffffffffffffffffffff1614156112cd576001905061148b565b600073ffffffffffffffffffffffffffffffffffffffff16600360009054906101000a900473ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16141561132d576000905061148b565b600360009054906101000a900473ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff1663b70096138430856000604051602001526040518463ffffffff167c0100000000000000000000000000000000000000000000000000000000028152600401808473ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff1681526020018373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff168152602001827bffffffffffffffffffffffffffffffffffffffffffffffffffffffff19167bffffffffffffffffffffffffffffffffffffffffffffffffffffffff191681526020019350505050602060405180830381600087803b151561146957fe5b6102c65a03f1151561147757fe5b50505060405180519050905061148b565b5b5b5b92915050565b80151561149e5760006000fd5b5b50565b600081600260003373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002060008573ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff168152602001908152602001600020819055508273ffffffffffffffffffffffffffffffffffffffff163373ffffffffffffffffffffffffffffffffffffffff167f8c5be1e5ebec7d5bd14f71427d1e84f3dd0314c0f7b2291e5b200ac8c7c3b925846040518082815260200191505060405180910390a3600190505b92915050565b600081600160008673ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002054101515156115e257fe5b81600260008673ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002060003373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff168152602001908152602001600020541015151561166a57fe5b6116f0600260008673ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002060003373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff1681526020019081526020016000205483611913565b600260008673ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002060003373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff168152602001908152602001600020819055506117b9600160008673ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff1681526020019081526020016000205483611913565b600160008673ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002081905550611845600160008573ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002054836118f9565b600160008573ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff168152602001908152602001600020819055508273ffffffffffffffffffffffffffffffffffffffff168473ffffffffffffffffffffffffffffffffffffffff167fddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef846040518082815260200191505060405180910390a3600190505b9392505050565b6000828284019150811015151561190c57fe5b5b92915050565b6000828284039150811115151561192657fe5b5b92915050565b600081600160003373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff168152602001908152602001600020541015151561197a57fe5b6119c3600160003373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff1681526020019081526020016000205483611913565b600160003373ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002081905550611a4f600160008573ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200190815260200160002054836118f9565b600160008573ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff168152602001908152602001600020819055508273ffffffffffffffffffffffffffffffffffffffff163373ffffffffffffffffffffffffffffffffffffffff167fddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef846040518082815260200191505060405180910390a3600190505b929150505600a165627a7a723058204432a84cfe06a995bd95935559d84003b39a006720c2eabd96115f376347f9b80029',
            ['0x06fdde03', '0x07da68f5', '0x095ea7b3', '0x13af4035', '0x18160ddd', '0x23b872dd', '0x313ce567',
             '0x3452f51d', '0x5ac801fe', '0x69d3e20e', '0x70a08231', '0x75f12b21', '0x7a9e5e4b', '0x8402181f',
             '0x8da5cb5b', '0x90bc1693', '0x95d89b41', '0xa9059cbb', '0xbe9a6555', '0xbf7e214f', '0xdd62ed3e',
             '0xffffffff'],
            True, False),
    (
            '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',
            ['0x01ffc9a7', '0x0519ce79', '0x0560ff44', '0x05e45546', '0x06fdde03', '0x095ea7b3', '0x0a0f8168',
             '0x0e583df0', '0x14001f4c', '0x18160ddd', '0x183a7947', '0x1940a936', '0x19c2f201', '0x21717ebf',
             '0x23b872dd', '0x24e7a38a', '0x27d7874c', '0x2ba73c15', '0x3d7d3f5a', '0x3f4ba83a', '0x46116e6f',
             '0x46d22c70', '0x481af3d3', '0x4ad8c938', '0x4b85fd55', '0x4dfff04f', '0x4e0a3379', '0x56129134',
             '0x5663896e', '0x5c975abb', '0x5fd8c710', '0x6352211e', '0x680eba27', '0x6af04a57', '0x6fbde40d',
             '0x70a08231', '0x71587988', '0x7a7d4937', '0x8456cb59', '0x8462151c', '0x88c2a0bf', '0x91876e57',
             '0x95d89b41', '0x9d6fac6f', '0xa45f4bfc', '0xa9059cbb', '0xb047fb50', '0xb0c35c05', '0xbc4006f5',
             '0xc3bea9af', '0xd3e6f49f', '0xdefb9584', '0xe17b25af', '0xe6cbe351', '0xe98b7f4d',
             '0xed60ade6', '0xf1ca9410', '0xf2b47d52', '0xf7d8c883', '0xffffffff'],
            False, True),
    ('0x44919b8026f38d70437a8eb3be47b06ab1c3e4bf', [], False, False),
    (
            '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',
            ['0x06fdde03', '0x07da68f5', '0x095ea7b3', '0x18160ddd', '0x23b872dd', '0x313ce567', '0x42966c68',
             '0x70a08231', '0x75f12b21', '0x95d89b41', '0xa9059cbb', '0xbe9a6555', '0xc47f0027', '0xdd62ed3e',
             '0xffffffff'], True, False),
    (
    '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',
    ['0x01ffc9a7', '0x06fdde03', '0x081812fc', '0x095ea7b3', '0x12c3f754', '0x17ffc320', '0x18160ddd', '0x23b872dd',
     '0x2f745c59', '0x34c05ca8', '0x392e53cd', '0x3f4ba83a', '0x42842e0e', '0x46904840', '0x4f558e79', '0x4f6ccce7',
     '0x509484d5', '0x5c975abb', '0x6352211e', '0x70a08231', '0x7bdc60d9', '0x7bf9a7c4', '0x8456cb59', '0x8da5cb5b',
     '0x8ed3fa7c', '0x90e7a074', '0x95d89b41', '0x99e0dd7c', '0xa22cb465', '0xacb2ad6f', '0xb88d4fde', '0xc0ac9983',
     '0xc87b56dd', '0xc9a6964a', '0xdce0b4e4', '0xe985e9c5', '0xee5301d5', '0xf2fde38b', '0xffffffff'], False, True)
])
def test_get_function_sighashes(bytecode, expected_sighashes, is_erc20, is_erc721):
    eth_contract_service = EthContractService()
    sighashes = eth_contract_service.get_function_sighashes(bytecode)
    assert expected_sighashes == sighashes
    assert eth_contract_service.is_erc20_contract(sighashes) == is_erc20
    assert eth_contract_service.is_erc721_contract(sighashes) == is_erc721
