"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assets = require("@aws-cdk/assets");
const ecr = require("@aws-cdk/aws-ecr");
const core_1 = require("@aws-cdk/core");
const fs = require("fs");
const minimatch = require("minimatch");
const path = require("path");
/**
 * An asset that represents a Docker image.
 *
 * The image will be created in build time and uploaded to an ECR repository.
 */
class DockerImageAsset extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // none of the properties use tokens
        validateProps(props);
        // resolve full path
        const dir = path.resolve(props.directory);
        if (!fs.existsSync(dir)) {
            throw new Error(`Cannot find image directory at ${dir}`);
        }
        // validate the docker file exists
        const file = path.join(dir, props.file || 'Dockerfile');
        if (!fs.existsSync(file)) {
            throw new Error(`Cannot find file at ${file}`);
        }
        let exclude = props.exclude || [];
        const ignore = path.join(dir, '.dockerignore');
        if (fs.existsSync(ignore)) {
            exclude = [...exclude, ...fs.readFileSync(ignore).toString().split('\n').filter(e => !!e)];
        }
        // make sure the docker file and the dockerignore file end up in the staging area
        // see https://github.com/aws/aws-cdk/issues/6004
        exclude = exclude.filter(ignoreExpression => {
            return !(minimatch(file, ignoreExpression, { matchBase: true }) ||
                minimatch(ignore, ignoreExpression, { matchBase: true }));
        });
        if (props.repositoryName) {
            this.node.addWarning('DockerImageAsset.repositoryName is deprecated. Override "core.Stack.addDockerImageAsset" to control asset locations');
        }
        // include build context in "extra" so it will impact the hash
        const extraHash = {};
        if (props.extraHash) {
            extraHash.user = props.extraHash;
        }
        if (props.buildArgs) {
            extraHash.buildArgs = props.buildArgs;
        }
        if (props.target) {
            extraHash.target = props.target;
        }
        if (props.file) {
            extraHash.file = props.file;
        }
        if (props.repositoryName) {
            extraHash.repositoryName = props.repositoryName;
        }
        // add "salt" to the hash in order to invalidate the image in the upgrade to
        // 1.21.0 which removes the AdoptedRepository resource (and will cause the
        // deletion of the ECR repository the app used).
        extraHash.version = '1.21.0';
        const staging = new assets.Staging(this, 'Staging', {
            ...props,
            exclude,
            sourcePath: dir,
            extraHash: Object.keys(extraHash).length === 0
                ? undefined
                : JSON.stringify(extraHash),
        });
        this.sourceHash = staging.sourceHash;
        const stack = core_1.Stack.of(this);
        const location = stack.addDockerImageAsset({
            directoryName: staging.stagedPath,
            dockerBuildArgs: props.buildArgs,
            dockerBuildTarget: props.target,
            dockerFile: props.file,
            repositoryName: props.repositoryName,
            sourceHash: staging.sourceHash,
        });
        this.repository = ecr.Repository.fromRepositoryName(this, 'Repository', location.repositoryName);
        this.imageUri = location.imageUri;
    }
}
exports.DockerImageAsset = DockerImageAsset;
function validateProps(props) {
    for (const [key, value] of Object.entries(props)) {
        if (core_1.Token.isUnresolved(value)) {
            throw new Error(`Cannot use Token as value of '${key}': this value is used before deployment starts`);
        }
    }
    validateBuildArgs(props.buildArgs);
}
function validateBuildArgs(buildArgs) {
    for (const [key, value] of Object.entries(buildArgs || {})) {
        if (core_1.Token.isUnresolved(key) || core_1.Token.isUnresolved(value)) {
            throw new Error('Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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