include(${CMAKE_CURRENT_LIST_DIR}/BuildEnv.cmake)
cmake_minimum_required(VERSION 3.26)
set(script                  ${CMAKE_CURRENT_LIST_DIR}/../devel.py)
set(LexYacc_DIR             ${CMAKE_CURRENT_LIST_DIR})
set(EmbeddedResource_DIR    ${CMAKE_CURRENT_LIST_DIR})
set(QuickCMake_DIR          ${CMAKE_CURRENT_LIST_DIR})
file(CONFIGURE OUTPUT "${CMAKE_CURRENT_SOURCE_DIR}/.clangd"
     CONTENT
"CompileFlags:
  CompilationDatabase: \"${CMAKE_BINARY_DIR}\"
"
)
function(buildenv_scriptexec outvar)
    if (NOT EXISTS ${script})
        message(FATAL_ERROR "Cannot find ${script}")
    endif()

    if (NOT EXISTS ${Python3_EXECUTABLE})
        find_package (Python3 REQUIRED COMPONENTS Interpreter)
    endif()

    set (CMD ${Python3_EXECUTABLE} ${script} --reporoot "${CMAKE_CURRENT_SOURCE_DIR}" --nostdin ${ARGN})

    if (outvar STREQUAL NONE)
        execute_process(COMMAND_ERROR_IS_FATAL ANY COMMAND_ECHO STDOUT COMMAND ${CMD})
    else()
        execute_process(COMMAND ${CMD} OUTPUT_VARIABLE output COMMAND_ERROR_IS_FATAL ANY COMMAND_ECHO STDOUT)
        string(STRIP ${output} output)
        set(${outvar} ${output} PARENT_SCOPE)
    endif()
endfunction()

macro (init_build_platform)
    # List of supported platforms : windows, uwp ,android ,linux
    if (NOT DEFINED SUPPORTED_PLATFORMS)
        if (ANDROID)
            list(APPEND SUPPORTED_PLATFORMS "android")
        endif()
        if (EMSCRIPTEN)
            list(APPEND SUPPORTED_PLATFORMS "emscripten")
        endif()
        if (CMAKE_SYSTEM_NAME STREQUAL "WindowsStore")
            list(APPEND SUPPORTED_PLATFORMS "uwp")
        endif()
        if (CMAKE_SYSTEM_NAME STREQUAL "Linux")
            list(APPEND SUPPORTED_PLATFORMS "linux")
        endif()
        if (WIN32)
            list(APPEND SUPPORTED_PLATFORMS "windows")
        endif()
        if (MSVC)
            list(APPEND SUPPORTED_PLATFORMS "winrt")
        endif()
        if (MINGW)
            list(APPEND SUPPORTED_PLATFORMS "mingw")
        endif()
        set(SUPPORTED_PLATFORMS "${SUPPORTED_PLATFORMS}" CACHE STRING "Build Platform")
    endif()
endmacro()

function (is_platform_applicable outvar)
    set(p ${ARGN})
    set(q ${ARGN})
    list(REMOVE_ITEM p ${SUPPORTED_PLATFORMS})
    if ("${p}" STREQUAL "${q}")
        set(${outvar} FALSE PARENT_SCOPE)
    else()
        set(${outvar} TRUE PARENT_SCOPE)
    endif()
endfunction()

macro (init_build_arch)
    if (NOT BUILD_ARCH)
        if (CMAKE_SYSTEM_PROCESSOR STREQUAL "x86_64")
            set(BUILD_ARCH x64)
        elseif (CMAKE_SYSTEM_PROCESSOR STREQUAL "AMD64")
            set(BUILD_ARCH x64)
        elseif (CMAKE_SYSTEM_PROCESSOR STREQUAL "i686")
            set(BUILD_ARCH x86)
        elseif (CMAKE_SYSTEM_PROCESSOR STREQUAL "armv7l")
            set(BUILD_ARCH arm)
        elseif (CMAKE_SYSTEM_PROCESSOR STREQUAL "armv6")
            set(BUILD_ARCH arm)
        elseif (CMAKE_SYSTEM_PROCESSOR STREQUAL "armv7-a")
            set(BUILD_ARCH arm)
        elseif (CMAKE_SYSTEM_PROCESSOR MATCHES "[aA](arch|rm|RM)64")
            set(BUILD_ARCH arm64)
        elseif (CMAKE_SYSTEM_NAME STREQUAL Emscripten)
            set(BUILD_ARCH wasm32)
        else()
            message(FATAL_ERROR ${CMAKE_SYSTEM_PROCESSOR} :: ${CMAKE_SYSTEM_NAME})
        endif()
    endif()
endmacro()

macro (init_host_arch)
    if (NOT HOST_ARCH)
        if (CMAKE_HOST_SYSTEM_PROCESSOR STREQUAL "x86_64")
            set(HOST_ARCH x64)
        elseif (CMAKE_HOST_SYSTEM_PROCESSOR STREQUAL "AMD64")
            set(HOST_ARCH x64)
        elseif (CMAKE_HOST_SYSTEM_PROCESSOR STREQUAL "i686")
            set(HOST_ARCH x86)
        elseif (CMAKE_HOST_SYSTEM_PROCESSOR STREQUAL "armv7l")
            set(HOST_ARCH arm)
        elseif (CMAKE_HOST_SYSTEM_PROCESSOR STREQUAL "armv6")
            set(HOST_ARCH arm)
        elseif (CMAKE_HOST_SYSTEM_PROCESSOR STREQUAL "armv7-a")
            set(HOST_ARCH arm)
        elseif (CMAKE_HOST_SYSTEM_PROCESSOR MATCHES "[aA](arch|rm|RM)64")
            set(HOST_ARCH arm64)
        else()
            message(FATAL_ERROR ${CMAKE_HOST_SYSTEM_PROCESSOR} :: ${CMAKE_SYSTEM_NAME})
        endif()
    endif()
endmacro()

macro(QuickInit)
    set(CMAKE_CXX_STANDARD 20)
    find_package(Threads) # Detect this before enabling strict compilation
    EnableStrictCompilation()
    init_build_platform()
    init_build_arch()
    init_host_arch()
    set(VCPKG_USE_HOST_TOOLS ON)
    set(VCPKG_SETUP_CMAKE_PROGRAM_PATH ON)
    if (CMAKE_HOST_SYSTEM_NAME STREQUAL "Windows")
        # Could be mingw or msvc
        find_program(MINGW_MAKE NAMES mingw32-make)
        if (MINGW OR MINGW_MAKE_FOUND)
            set(VCPKG_HOST_TRIPLET "${HOST_ARCH}-mingw-static")
        else()
            set(VCPKG_HOST_TRIPLET "${HOST_ARCH}-windows-static-md")
        endif()
    elseif (CMAKE_HOST_SYSTEM_NAME STREQUAL "Linux")
        set(VCPKG_HOST_TRIPLET "${HOST_ARCH}-linux")
    elseif (CMAKE_HOST_SYSTEM_NAME STREQUAL "MSYS")
        set(VCPKG_HOST_TRIPLET "${HOST_ARCH}-mingw-static")
    else()
        message(FATAL_ERROR "Unknown Host: ${CMAKE_HOST_SYSTEM_NAME}")
    endif()
    if (ANDROID)
        set(VCPKG_TARGET_TRIPLET "${BUILD_ARCH}-android")
    elseif (EMSCRIPTEN)
        set(VCPKG_TARGET_TRIPLET "${BUILD_ARCH}-emscripten")
    elseif (CMAKE_SYSTEM_NAME STREQUAL "WindowsStore")
        set(VCPKG_TARGET_TRIPLET ${BUILD_ARCH}-uwp-static-md)
    elseif (WIN32)
        if (MINGW OR CMAKE_SYSTEM_NAME STREQUAL "MSYS")
            # set(VCPKG_HOST_TRIPLET x64-mingw-static)
            set(VCPKG_TARGET_TRIPLET ${BUILD_ARCH}-mingw-static)
        elseif (MSVC)
            set(VCPKG_TARGET_TRIPLET ${BUILD_ARCH}-windows-static-md)
        else()
            message(FATAL_ERROR ${CMAKE_SYSTEM_NAME})
        endif()
    elseif (LINUX)
        set(VCPKG_TARGET_TRIPLET "${BUILD_ARCH}-linux")
    endif()

    if (VCPKG_TARGET_TRIPLET STREQUAL "arm-android")
        set(VCPKG_TARGET_TRIPLET "arm-neon-android")
    endif()
                if (BUILD_ARCH AND CMAKE_GENERATOR MATCHES "Visual")
                    set(CMAKE_GENERATOR_PLATFORM ${BUILD_ARCH})
                    if (WIN32 AND CMAKE_GENERATOR MATCHES "Visual" AND BUILD_ARCH AND BUILD_ARCH MATCHES "x86")
                        set(CMAKE_GENERATOR_PLATFORM Win32)
                    endif()
                endif()

                if (WIN32 AND CMAKE_GENERATOR MATCHES "Visual" AND BUILD_CONFIG)
                    set(CMAKE_CONFIGURATION_TYPES ${BUILD_CONFIG})
                endif()

                if (CMAKE_GENERATOR_PLATFORM)
                    set(CMAKE_VS_PLATFORM_NAME_DEFAULT ${CMAKE_GENERATOR_PLATFORM})
                endif()

    if (NOT DEFINED VCPKG_TARGET_TRIPLET)
        message(FATAL_ERROR "Cannot detect the target triplet")
    endif()
    if (WIN32 AND DEFINED ENV{CPPWINRT_INCLUDE_PATH})
        set(CPPWINRT_INCLUDE_PATH "$ENV{CPPWINRT_INCLUDE_PATH}" CACHE PATH "CppWinRT include path")
    endif()
    if (EXISTS "${CPPWINRT_INCLUDE_PATH}")
        include_directories("${CPPWINRT_INCLUDE_PATH}")
    endif()
    vcpkg_init()
    fetchcontent_init()
endmacro()

macro(fetchcontent_init)
    buildenv_scriptexec(FETCHCONTENT_BASE_DIR config --queryparam fetchcontent_base_dir)
    set(FETCHCONTENT_BASE_DIR "${FETCHCONTENT_BASE_DIR}" CACHE PATH "Location of FETCHCONTENT sources")
endmacro()

macro(vcpkg_init)
    if (NOT NO_VCPKG)
        set(VCPKG_AVAILABLE_PORT_LIST_FILE "${PROJECT_BINARY_DIR}/vcpkg-available-port-list.txt")
        set(VCPKG_EXTRA_ARGS --list ${VCPKG_AVAILABLE_PORT_LIST_FILE})

        if (DEFINED VCPKG_BUILD_CACHE_VERSION)
            list(APPEND VCPKG_EXTRA_ARGS --buildcache ${PROJECT_NAME}.${VCPKG_BUILD_CACHE_VERSION})
        endif()
        if (DEFINED VCPKG_COMMIT)
            list(APPEND VCPKG_EXTRA_ARGS --commit ${VCPKG_COMMIT})
        endif()
        set(vcpkg_init_cmd vcpkg --triplet ${VCPKG_TARGET_TRIPLET} --apply-patches ${VCPKG_EXTRA_ARGS})
        if (NOT "${VCPKG_HOST_TRIPLET}" STREQUAL "")
            list(APPEND vcpkg_init_cmd --host-triplet "${VCPKG_HOST_TRIPLET}")
        endif()
        buildenv_scriptexec(VCPKG_ROOT_init ${vcpkg_init_cmd})
        if (NOT IS_DIRECTORY "${VCPKG_ROOT_init}")
            # Try again
            buildenv_scriptexec(VCPKG_ROOT_init ${vcpkg_init_cmd})
        endif()
        if (NOT IS_DIRECTORY "${VCPKG_ROOT_init}")
            message(FATAL_ERROR "Invalid VCPKG_ROOT : ${VCPKG_ROOT_init}")
        endif()
        set(VCPKG_ROOT "${VCPKG_ROOT_init}" CACHE PATH "VCPKG Root")
        if (NOT "${VCPKG_ROOT_init}" STREQUAL "${VCPKG_ROOT}")
            set(VCPKG_ROOT "${VCPKG_ROOT_init}" CACHE PATH "VCPKG Root" FORCE)
        endif()
        file(STRINGS "${VCPKG_AVAILABLE_PORT_LIST_FILE}" available_ports)
        if (NOT EXISTS ${VCPKG_ROOT}/scripts/buildsystems/vcpkg.cmake)
            message(FATAL_ERROR "Cannot find ${VCPKG_ROOT}/scripts/buildsystems/vcpkg.cmake")
        endif()
        include(${VCPKG_ROOT}/scripts/buildsystems/vcpkg.cmake)
        message(STATUS "VCPKG_ROOT:: ${VCPKG_ROOT} VCPKG_TARGET_TRIPLET::${VCPKG_TARGET_TRIPLET}")
        set(VCPKG_REQUESTED_PORTS "")
        set_property(GLOBAL PROPERTY VCPKG_AVAILABLE_PORTS "${available_ports}")
    endif()
endmacro()

function(vcpkg_export)
    return()
    list(JOIN VCPKG_REQUESTED_PORTS , exported_ports)
    set(vcpkg_export_cmd vcpkg  --triplet ${VCPKG_TARGET_TRIPLET} --export ${exported_ports} ${VCPKG_EXTRA_ARGS})
    if (NOT "${VCPKG_HOST_TRIPLET}" STREQUAL "")
        list(APPEND vcpkg_export_cmd --host-triplet "${VCPKG_HOST_TRIPLET}")
    endif()
    if (NOT "${exported_ports}" STREQUAL "")
        buildenv_scriptexec(NONE )
    endif()
endfunction()

function(vcpkg_install)
    vcpkg_download(${ARGN})
endfunction()

function(is_packages_applicable outvar)
    get_property(available_ports GLOBAL PROPERTY VCPKG_AVAILABLE_PORTS)
    foreach(l ${ARGN})
        if (NOT "${l}" MATCHES ".*:.*")
            set(l "${l}:${VCPKG_TARGET_TRIPLET}")
        endif()
        string(FIND ";${available_ports};" ";${l};" index)
        if (index STREQUAL "-1")
            list(APPEND failed "${l}")
        endif()
    endforeach()
    if ("${failed}" STREQUAL "")
        set(${outvar} TRUE PARENT_SCOPE)
    else()
        set(${outvar}_FAILED "${failed}" PARENT_SCOPE)
        set(${outvar} FALSE PARENT_SCOPE)
    endif()
endfunction()

function(vcpkg_download)
    is_packages_applicable(notneeded ${ARGN})
    if (notneeded)
        return()
    endif()
    list(JOIN ARGN "," libs)
    set(vcpkg_download_cmd vcpkg --triplet ${VCPKG_TARGET_TRIPLET} --download ${libs} ${VCPKG_EXTRA_ARGS})
    if (NOT "${VCPKG_HOST_TRIPLET}" STREQUAL "")
        list(APPEND vcpkg_download_cmd --host-triplet "${VCPKG_HOST_TRIPLET}")
    endif()
    buildenv_scriptexec(NONE ${vcpkg_download_cmd})
    file(STRINGS "${VCPKG_AVAILABLE_PORT_LIST_FILE}" available_ports)
    set_property(GLOBAL PROPERTY VCPKG_AVAILABLE_PORTS "${available_ports}")
endfunction()

function(use_qt5 target)
    #vcpkg_download(qt5)
    find_package(Qt5OpenGL REQUIRED QUIET)
    target_link_libraries(${target} PRIVATE Qt5OpenGL)
    qt5_use_modules(${target} Widgets Quick )
endfunction()

function(find_vcpkg_library varname name)
    cmake_parse_arguments("" "" "" "SUBDIRECTORY" ${ARGN})

    set(VCPKG_DEBUG_SEARCH_PATHS "debug/lib" "debug/lib64")
    set(VCPKG_OPTIMIZED_SEARCH_PATHS "lib" "lib64")

    set(SEARCH_PATHS_OPTIMIZED "")
    set(SEARCH_PATHS_DEBUG "")
    foreach (subdir ${_SUBDIRECTORY})
        foreach(vcpkg_search_path ${VCPKG_DEBUG_SEARCH_PATHS})
            list(APPEND SEARCH_PATHS_DEBUG ${vcpkg_search_path})
            list(APPEND SEARCH_PATHS_DEBUG ${vcpkg_search_path}/${subdir})
        endforeach()
        foreach(vcpkg_search_path ${VCPKG_OPTIMIZED_SEARCH_PATHS})
            list(APPEND SEARCH_PATHS_OPTIMIZED ${vcpkg_search_path})
            list(APPEND SEARCH_PATHS_OPTIMIZED ${vcpkg_search_path}/${subdir})
        endforeach()
    endforeach()

    set(dvar ${varname}_LIBRARY_DEBUG)
    set(rvar ${varname}_LIBRARY_RELEASE)

    if ((EXISTS ${${dvar}}) AND (EXISTS ${${rvar}}))
        set(${varname} debug ${${dvar}} optimized ${${rvar}} PARENT_SCOPE)
        return()
    endif()

    find_library(${rvar} ${name} QUIET PATH_SUFFIXES  ${_SUBDIRECTORY})
    if (NOT EXISTS ${${rvar}})
        if ((WIN32) AND (NOT "${name}" MATCHES "^[lL][iI][bB].*" ))
            find_vcpkg_library(${varname} lib${name} ${ARGN})
            return()
        endif()
        message(FATAL_ERROR "Cannot find library ${name} ${${rvar}}")
    endif()
    get_filename_component(installdir ${${rvar}} DIRECTORY)

    find_library(${dvar} NAMES ${name}d ${name}_d ${name} PATHS ${installdir}/debug/lib ${installdir}/debug/lib64 QUIET PATH_SUFFIXES  ${_SUBDIRECTORY})
    if (NOT EXISTS ${${dvar}})
        message(FATAL_ERROR "Cannot find debug library ${name}")
    endif()
    set(${varname} debug ${${dvar}} optimized ${${rvar}} PARENT_SCOPE)
    message(STATUS "VCPKG Libs ::${varname}:: debug ${${dvar}} optimized ${${rvar}}::")
endfunction()

set(OpenGL_GL_PREFERENCE GLVND)
function(use_nanogui target)
    vcpkg_download(glad)
    vcpkg_download(nanogui)
    find_package(OpenGL REQUIRED QUIET)
    find_package(glfw3 REQUIRED QUIET)
    find_package(nanovg)
    find_path(NANOGUI_INCLUDE include/nanogui)
    find_vcpkg_library(NANOGUI_LIB nanogui)
    find_vcpkg_library(GLFW3_LIB glfw3)
    target_compile_definitions(${target} PRIVATE GLAD_GLAPI_EXPORT)
    target_include_directories(${target} PRIVATE ${NANOGUI_INCLUDE}/include ${OPENGL_INCLUDE_DIRS})
    target_link_libraries(${target} PRIVATE ${NANOGUI_LIB} nanovg::nanovg GLU dl glfw)
endfunction()

function(use_glfw3 target)
    if (CMAKE_SYSTEM_NAME STREQUAL "WindowsStore")
        use_angle(${target})
    else()
        if (WIN32)
            #target_link_libraries(${target} PRIVATE OpenGL::GL)
        else()
            target_link_libraries(${target} PRIVATE GLU dl)
        endif()
    endif()
    vcpkg_download(glfw3)
    find_package(glfw3 REQUIRED QUIET)
    target_link_libraries(${target} PRIVATE glfw)
endfunction()

function(use_gles target)
    use_angle(${target})
endfunction()

function(use_angle target)
    vcpkg_download(angle)
    find_package(unofficial-angle REQUIRED QUIET)
    target_link_libraries(${target} PRIVATE unofficial::angle::libANGLE unofficial::angle::libEGL unofficial::angle::libGLESv2)
endfunction()

function(use_gattlib target)
    vcpkg_download(gattlib)
    find_vcpkg_library(gattlib gattlib)
    find_file(gattlibh gattlib.h)
    get_filename_component(incpath ${gattlibh} DIRECTORY)
    target_link_libraries(${target} PRIVATE ${gattlib})
    target_include_directories(${target} PRIVATE ${incpath})
    set(THREADS_PREFER_PTHREAD_FLAG ON)
    find_package(Threads REQUIRED QUIET)
    target_link_libraries(${target} PUBLIC Threads::Threads)
endfunction()

function(use_catch2 target)
    vcpkg_download(catch2)
    find_package(Catch2 REQUIRED QUIET)
    target_link_libraries(${target} PRIVATE Catch2::Catch2 Catch2::Catch2WithMain)
    if (MSVC)
        target_compile_options(Catch2::Catch2WithMain INTERFACE "/wd4868") #not enforce evaluation order in braced initializer list
    endif()
endfunction()

function(use_rapidjson target)
    vcpkg_download(rapidjson)
    find_package(RapidJSON REQUIRED)
    target_compile_definitions(${target} PRIVATE USE_RAPIDJSON)
    target_include_directories(${target} PRIVATE ${RapidJSON_INCLUDE_DIR})
endfunction()

function(use_dtl target)
    vcpkg_download(dtl)
    find_path(DTL_INCLUDE_DIRS "dtl/Diff.hpp")
    target_include_directories(${target} PRIVATE ${DTL_INCLUDE_DIRS})
endfunction()

macro(use_vcpkg target)
    foreach (use ${ARGN})
        string(REPLACE "-" "_" cmake_fname "use_${use}.cmake")
        if (EXISTS ${QuickCMake_DIR}/${cmake_fname})
            include(${QuickCMake_DIR}/${cmake_fname})
            link_to_target(${target})
        else()
            use_external(${target} ${use})
        endif()
    endforeach()
endmacro()

function(use_cmake_vcpkg_package target use)
    string(TOUPPER "USE_${use}" use_define)
    target_compile_definitions(${target} PRIVATE ${use_define})
    if (TARGET ${use})
        target_link_libraries(${target} PRIVATE ${use})
    elseif (TARGET ${use}::${use})
        target_link_libraries(${target} PRIVATE ${use}::${use})
    else()
        message(FATAL_ERROR "vcpkg package ${use} doesnt define target ${use} or ${use}::${use}. Cannot link with ${target}")
    endif()
endfunction()

macro(use_external target use)
    #if (CMAKE_SYSTEM_NAME STREQUAL "Android")
    #    return()
    #endif()

    if (${use} STREQUAL nanogui)
        use_nanogui(${target})
    elseif (${use} STREQUAL angle)
        use_angle(${target})
    elseif (${use} STREQUAL gles)
        use_gles(${target})
    elseif (${use} STREQUAL glfw3)
        use_glfw3(${target})
    elseif (${use} STREQUAL gattlib)
        use_gattlib(${target})
    elseif (${use} STREQUAL catch2)
        use_catch2(${target})
    elseif (${use} STREQUAL openscenegraph)
        use_openscenegraph(${target})
    elseif(${use} STREQUAL libxml2)
        vcpkg_download(libxml2)
        find_package(LibXml2 CONFIG REQUIRED)
        target_compile_definitions(${target} PRIVATE HAVE_LIBXML2=1)
        target_link_libraries(${target} PRIVATE LibXml2::LibXml2)
    elseif(${use} STREQUAL qtbase)
        vcpkg_download(qtbase)
        set(CMAKE_AUTOMOC ON)
        set(HAVE_EGL 1)
        if (EMSCRIPTEN)
            set(HAVE_GETTIME 1)
        endif()
        if (ANDROID)
            set(HAVE_GLESv2 1)
        endif()
        find_package(Qt6 REQUIRED COMPONENTS OpenGLWidgets Widgets OpenGL Gui Sql Network)
        
        target_link_libraries(${target} PRIVATE
            Qt6::OpenGLWidgets Qt6::Widgets Qt6::OpenGL Qt6::Gui Qt6::Sql Qt6::Network
            #Qt::QSQLiteDriverPlugin Qt::QTlsBackendOpenSSLPlugin
        )
        qt_import_plugins(${target} INCLUDE)
        #if (LINUX)
        #    target_link_libraries(${target} PRIVATE Qt::QXcbIntegrationPlugin)
        #    qt_import_plugins(${target} INCLUDE Qt::QXcbIntegrationPlugin)
        #endif()
        #qt_import_plugins(${target}
        #    INCLUDE Qt::QSQLiteDriverPlugin Qt::QTlsBackendOpenSSLPlugin Qt::QXcbIntegrationPlugin
        #)
    elseif(${use} STREQUAL qtlocation)
        vcpkg_download(qtlocation)
        find_package(Qt6 COMPONENTS Core Location REQUIRED)
        target_link_libraries(${target} PRIVATE Qt::Location)
    elseif(${use} STREQUAL qtpositioning)
        vcpkg_download(qtpositioning)
        find_package(Qt6 COMPONENTS Core Positioning REQUIRED)
        target_link_libraries(${target} PRIVATE Qt::Positioning)
    elseif(${use} STREQUAL qtsensors)
        vcpkg_download(qtsensors)
        find_package(Qt6 COMPONENTS Core Sensors REQUIRED)
        target_link_libraries(${target} PRIVATE Qt::Sensors)
    elseif(${use} STREQUAL yaml-cpp)
        vcpkg_download(yaml-cpp)
        find_package(yaml-cpp CONFIG REQUIRED)
        target_link_libraries(${target} PRIVATE yaml-cpp)
    elseif(${use} STREQUAL curl)
        find_package(CURL CONFIG REQUIRED)
        target_link_libraries(${target} PRIVATE CURL::libcurl)
    else()
        vcpkg_download(${use})
        string(REPLACE "-" "_" __use_pkg_name "${use}")
        find_package(${__use_pkg_name} CONFIG)
        if (${__use_pkg_name}_FOUND)
            use_cmake_vcpkg_package(${target} ${__use_pkg_name})
        else()
            message(FATAL_ERROR "Unknown external target ${use}")
        endif()
    endif()
endmacro()

macro(use_embed_resource)
    find_package(EmbedResource QUIET)
    if (NOT EmbedResource_FOUND AND NOT TARGET embedresource)
        vcpkg_download(embedresource)
        find_package(EmbedResource REQUIRED)
    endif()
endmacro()

macro(use_stencil)
    if(NOT TARGET stencil)
        if(EXISTS "${STENCIL_DIR}")
            add_subdirectory("${STENCIL_DIR}" stencil)
        else()
            find_package(stencil QUIET)
            if (NOT stencil_FOUND)
                vcpkg_download(stencil)
                find_program(LEXYACC_EXECUTABLE lexyacc REQUIRED NO_DEFAULT_PATH HINTS  ${VCPKG_ROOT}/installed/${VCPKG_HOST_TRIPLET}/tools)
                find_program(STENCIL_EXECUTABLE stencil REQUIRED NO_DEFAULT_PATH HINTS ${VCPKG_ROOT}/installed/${VCPKG_HOST_TRIPLET}/tools)
                find_path(stencil_INCLUDE_PATH "stencil/stencil.h" REQUIRED)
                set(stencil_INCLUDE_PATH "${stencil_INCLUDE_PATH}" CACHE PATH "Stencil include path")
                find_package(stencil REQUIRED)
            endif()
        endif()
    endif()
endmacro()

macro(quickload_config)
    QuickInit()
    set(manifests)
    foreach (item ${ARGN})
        get_filename_component(manifestfullpath ${item} ABSOLUTE)
        list(APPEND manifests ${manifestfullpath})
    endforeach()
    buildenv_scriptexec(NONE quick --outtype cmake ${manifests} ${CMAKE_CURRENT_BINARY_DIR})
    file(TO_CMAKE_PATH ${CMAKE_CURRENT_BINARY_DIR}/Quick.cmake includepath)
    include(${includepath})
    cpack_init()
endmacro()

function (git_submodule_init dir)
    if (EXISTS "${dir}/.git")
        return()
    endif()
    find_package(Git QUIET)
    message(STATUS "Submodule update ${dir}")
    get_filename_component(dirname "${dir}" DIRECTORY)
    get_filename_component(basename "${dir}" NAME)

    execute_process(COMMAND ${GIT_EXECUTABLE} -C ${dirname} submodule update --init --recursive ${basename}
                    WORKING_DIRECTORY ${dirname}
                    COMMAND_ERROR_IS_FATAL ANY)
endfunction()

function(_add_npm_build_embedded_resource_helper srcdir targetName)
    use_embed_resource()
    set(embedded_spec_file "${CMAKE_CURRENT_BINARY_DIR}/dist-artifacts.txt")

    set (COMPILE_CMD "${Python3_EXECUTABLE}" "${script}"
        --nostdin
        --reporoot "${CMAKE_CURRENT_SOURCE_DIR}"
        npm --generate
        --subdir "${srcdir}"
        --buildroot "${CMAKE_CURRENT_BINARY_DIR}"
        --out-file-list "${embedded_spec_file}"
    )

    file(GLOB srcs "${srcdir}/*.*")

    add_custom_command(
        OUTPUT "${embedded_spec_file}"
        COMMAND ${COMPILE_CMD}
        WORKING_DIRECTORY "${CMAKE_CURRENT_BINARY_DIR}"
        DEPENDS ${srcs} "${script}"
    )

    add_resource_library(TARGET ${targetName} GENERATOR_SPECFILE "${embedded_spec_file}")
endfunction()

macro(add_npm_build_embedded_resource targetName linktype srcdir)
    use_embed_resource()
    if(NOT EXISTS ${Python3_EXECUTABLE})
        find_package (Python3 REQUIRED COMPONENTS Interpreter)
    endif()
    _add_npm_build_embedded_resource_helper("${srcdir}" "${targetName}")
endmacro()

function(create_app_package targetName manifest)
    file(CONFIGURE OUTPUT "${CMAKE_CURRENT_BINARY_DIR}/${targetName}.manifestpath.txt" CONTENT "${manifest}")
    install(FILES "${CMAKE_CURRENT_BINARY_DIR}/${targetName}.manifestpath.txt" DESTINATION . COMPONENT ${targetName} RENAME manifestpath.txt)
endfunction()

function(add_test_executable targetName)
    add_executable(${targetName} ${ARGN})
    use_catch2(${targetName})
    use_dtl(${targetName})
endfunction()

macro(add_qt_executable)
    find_package(Qt6 REQUIRED COMPONENTS Core)
    qt_add_executable(${ARGN})
endmacro()
#
# CPACK
#
set (CPACK_PACKAGE_NAME                ${PROJECT_NAME}_${CMAKE_BUILD_TYPE})
set (CPACK_PACKAGE_DESCRIPTION_SUMMARY ${PROJECT_DESCRIPTION})
set (CPACK_PACKAGE_VERSION             ${PROJECT_VERSION})
set (CPACK_PACKAGE_VENDOR              "Ankur Verma")
set (CPACK_GENERATOR                   ZIP)
set (CPACK_ARCHIVE_COMPONENT_INSTALL   ON)
set (CPACK_NUGET_COMPONENT_INSTALL     ON)
set (CPACK_COMPONENTS_GROUPING         IGNORE)
set (CPACK_EXTERNAL_ENABLE_STAGING      ON)
include(CPack)


macro (cpack_init)

cpack_add_component_group(allbinaries
    DISPLAY_NAME "Binaries"
    DESCRIPTION "Binaries"
)

cpack_add_component_group(alltests
    DISPLAY_NAME "Tests"
    DESCRIPTION "Tests"
)

cpack_add_component(tests GROUP alltests)
cpack_add_component(binaries GROUP allbinaries)

endmacro()
