"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxDockerManifestProject = exports.LinuxDockerBuildProject = exports.BaseDockerProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
/**
 * The base for Linux-based Docker build projects.
 */
class BaseDockerProject extends constructs_1.Construct {
    /**
     * Creates a new BaseDockerProject.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { repository, removalPolicy, logRetention } = props;
        this.buildSpec = props.buildSpec;
        this.logGroup = new aws_logs_1.LogGroup(this, "Logs", { retention: logRetention });
        if (props.buildEnvironment.buildImage) {
            if (/\b(windows|macos)\b/.test(props.buildEnvironment.buildImage.imageId)) {
                throw new aws_cdk_lib_1.ValidationError("You must use a Linux-based build image", this);
            }
        }
        const repoEnv = repository.env;
        const projectVariables = {
            AWS_DEFAULT_REGION: { value: repoEnv.region },
            AWS_ECR_HOSTNAME: {
                value: `${repoEnv.account}.dkr.ecr.${repoEnv.region}.amazonaws.com`,
            },
            AWS_ECR_URL: { value: repository.repositoryUri },
        };
        this.project = new aws_codebuild_1.PipelineProject(this, "Project", {
            description: props.description,
            environment: props.buildEnvironment,
            environmentVariables: projectVariables,
            buildSpec: this.buildSpec,
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
            securityGroups: props.securityGroups,
            logging: { cloudWatch: { logGroup: this.logGroup } },
        });
        repository.grantPullPush(this.project);
        if (removalPolicy) {
            this.applyRemovalPolicy(removalPolicy);
        }
    }
    get stack() {
        return this.project.stack;
    }
    get env() {
        return this.project.env;
    }
    applyRemovalPolicy(policy) {
        this.project.applyRemovalPolicy(policy);
        this.logGroup.applyRemovalPolicy(policy);
    }
}
exports.BaseDockerProject = BaseDockerProject;
_a = JSII_RTTI_SYMBOL_1;
BaseDockerProject[_a] = { fqn: "shady-island.automation.BaseDockerProject", version: "0.1.60" };
/**
 * Sets up a standardized Docker build project.
 *
 * This project accepts the following optional environment variables:
 *
 * - IMAGE_LABELS: JSON-formatted object of container labels and their values.
 * - BUILD_ARGS: JSON-formatted object of build arguments and their values.
 * - IMAGE_TAG: Optional. The image tag (e.g. Git commit ID) (default: build
 *   number).
 */
class LinuxDockerBuildProject extends BaseDockerProject {
    /**
     * Creates a new LinuxDockerBuildProject.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope, id, props) {
        const { dockerfile = "Dockerfile", buildDirectory = ".", enableCache = false, testCommands = [], pushLatest = true, } = props;
        const buildCommands = [
            'echo "Building the Docker image..."',
            'if [ -z "$IMAGE_TAG" ]; then IMAGE_TAG="$CODEBUILD_BUILD_NUMBER"; fi',
            [
                "build_args=()",
                'if [ -n "$BUILD_ARGS" ]; then for row in $(echo $BUILD_ARGS | jq -r .[]); do build_args+=("$row"); done; fi',
                'build_args_args=""',
                'for build_arg in "${build_args[@]}"; do build_args_args="--build-arg $build_arg $build_args_args"; done',
            ].join("\n"),
            [
                'labels=("org.opencontainers.image.created=$(date --iso-8601=seconds)")',
                'if [ -n "$IMAGE_LABELS" ]; then for row in $(echo $IMAGE_LABELS | jq -r .[]); do labels+=("$row"); done; fi',
                'label_args=""',
                'for label in "${labels[@]}"; do label_args="--label $label $label_args"; done',
            ].join("\n"),
        ];
        const buildArguments = [
            "$label_args",
            "$build_args_args",
            '-t "$AWS_ECR_URL:$IMAGE_TAG"',
            `-f ${dockerfile}`,
        ];
        if (enableCache) {
            buildArguments.push("--cache-from $AWS_ECR_URL:latest");
            buildCommands.push("docker pull $AWS_ECR_URL:latest || true");
        }
        const buildCommand = [
            "docker",
            "build",
            ...buildArguments,
            `"${buildDirectory}"`,
        ].join(" ");
        buildCommands.push(`echo Build command: ${buildCommand}`, buildCommand, 'docker inspect "$AWS_ECR_URL:$IMAGE_TAG"');
        if (testCommands.length > 0) {
            buildCommands.push(...testCommands);
        }
        const postBuildCommands = [
            'if [ "$CODEBUILD_BUILD_SUCCEEDING" == "0" ]; then exit 1; fi',
            'echo "Pushing the Docker image..."',
            'docker push "$AWS_ECR_URL:$IMAGE_TAG"',
        ];
        if (pushLatest) {
            postBuildCommands.push('docker tag "$AWS_ECR_URL:$IMAGE_TAG" $AWS_ECR_URL:latest', "docker push $AWS_ECR_URL:latest");
        }
        const buildSpec = aws_codebuild_1.BuildSpec.fromObjectToYaml({
            version: "0.2",
            env: {
                shell: "bash",
            },
            phases: {
                pre_build: {
                    commands: [
                        'echo "Logging in to Amazon ECR..."',
                        [
                            'aws ecr get-login-password --region "$AWS_DEFAULT_REGION"',
                            'docker login --username AWS --password-stdin "$AWS_ECR_HOSTNAME"',
                        ].join(" | "),
                    ],
                },
                build: {
                    commands: buildCommands,
                },
                post_build: {
                    commands: postBuildCommands,
                },
            },
        });
        super(scope, id, {
            repository: props.repository,
            buildSpec,
            description: props.description ?? "Builds and pushes a Linux Docker container",
            buildEnvironment: {
                buildImage: props.buildImage,
                computeType: props.computeType,
                privileged: true,
            },
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
            securityGroups: props.securityGroups,
            removalPolicy: props.removalPolicy,
        });
    }
}
exports.LinuxDockerBuildProject = LinuxDockerBuildProject;
_b = JSII_RTTI_SYMBOL_1;
LinuxDockerBuildProject[_b] = { fqn: "shady-island.automation.LinuxDockerBuildProject", version: "0.1.60" };
/**
 * Sets up a standardized Docker manifest build project.
 *
 * This project accepts the following variables:
 *
 * - LATEST_TAG: Optional. The tag to push (default: "latest").
 * - MANIFEST_CUSTOM_TAG: Optional. The tag to push, in addition to $LATEST_TAG.
 */
class LinuxDockerManifestProject extends BaseDockerProject {
    /**
     * Creates a new LinuxDockerManifestProject.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope, id, props) {
        const { tagVariableNames } = props;
        const archTags = tagVariableNames.map((x) => `"$AWS_ECR_URL:$${x}"`);
        const buildCommands = [
            ...tagVariableNames.map((x) => `echo "${x}: $${x}"`),
            [
                'latest_tag="latest"',
                'if [ -n "$LATEST_TAG" ]; then',
                '  latest_tag="$LATEST_TAG"',
                "fi",
            ].join("\n"),
            [
                'if [ -n "$MANIFEST_CUSTOM_TAG" ]; then',
                [
                    '  docker manifest create "$AWS_ECR_URL:$MANIFEST_CUSTOM_TAG"',
                    ...archTags,
                ].join(" "),
                '  docker manifest inspect "$AWS_ECR_URL:$MANIFEST_CUSTOM_TAG"',
                '  docker manifest push "$AWS_ECR_URL:$MANIFEST_CUSTOM_TAG"',
                "fi",
            ].join("\n"),
            ['docker manifest create "$AWS_ECR_URL:$latest_tag"', ...archTags].join(" "),
            'docker manifest inspect "$AWS_ECR_URL:$latest_tag"',
            'docker manifest push "$AWS_ECR_URL:$latest_tag"',
        ];
        const buildSpec = aws_codebuild_1.BuildSpec.fromObjectToYaml({
            version: "0.2",
            env: { shell: "bash" },
            phases: {
                pre_build: {
                    commands: [
                        'echo "Logging in to Amazon ECR..."',
                        [
                            'aws ecr get-login-password --region "$AWS_DEFAULT_REGION"',
                            'docker login --username AWS --password-stdin "$AWS_ECR_HOSTNAME"',
                        ].join(" | "),
                    ],
                },
                build: {
                    commands: buildCommands,
                },
            },
        });
        super(scope, id, {
            repository: props.repository,
            buildSpec,
            description: props.description ?? "Pushes Docker manifest files",
            buildEnvironment: {
                buildImage: props.buildImage,
                computeType: props.computeType,
                privileged: true,
            },
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
            securityGroups: props.securityGroups,
            removalPolicy: props.removalPolicy,
        });
    }
}
exports.LinuxDockerManifestProject = LinuxDockerManifestProject;
_c = JSII_RTTI_SYMBOL_1;
LinuxDockerManifestProject[_c] = { fqn: "shady-island.automation.LinuxDockerManifestProject", version: "0.1.60" };
//# sourceMappingURL=data:application/json;base64,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