import unittest
import os, os.path
from collections import OrderedDict
from importlib.util import find_spec

from estnltk import Text
from estnltk_core import Span
from estnltk.converters import dict_to_text, text_to_dict
from estnltk.converters import layer_to_dict, dict_to_layer
from estnltk.downloader import get_resource_paths

from estnltk.taggers.standard.syntax.phrase_extraction.time_loc_tagger import TimeLocTagger

# Try to get the resources path for wordnet. If missing, do nothing. It's up for the user to download the missing resources
WORDNET_PATH = get_resource_paths("wordnet", only_latest=True, download_missing=False)

# Try to get the resources path for stanzasyntaxtagger. If missing, do nothing. It's up for the user to download the missing resources
STANZA_SYNTAX_MODELS_PATH = get_resource_paths("stanzasyntaxtagger", only_latest=True, download_missing=False)

def check_if_estnltk_neural_is_available():
    return find_spec("estnltk_neural") is not None

# Example inputs
example_input_1_dict = \
    {'layers': [{'ambiguous': True,
                 'attributes': ('normalized_form',),
                 'enveloping': None,
                 'meta': {},
                 'name': 'words',
                 'parent': None,
                 'secondary_attributes': (),
                 'serialisation_module': None,
                 'spans': [{'annotations': [{'normalized_form': None}],
                            'base_span': (0, 7)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (8, 15)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (16, 22)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (23, 28)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (29, 35)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (36, 37)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (38, 41)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (42, 50)},
                           {'annotations': [{'normalized_form': '70.'}],
                            'base_span': (51, 54)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (55, 62)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (63, 69)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (70, 78)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (79, 82)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (83, 98)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (99, 100)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (101, 103)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (104, 105)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (106, 114)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (115, 120)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (121, 128)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (129, 132)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (133, 146)},
                           {'annotations': [{'normalized_form': None}],
                            'base_span': (147, 148)}]},
                {'ambiguous': True,
                 'attributes': ('normalized_text',
                                'lemma',
                                'root',
                                'root_tokens',
                                'ending',
                                'clitic',
                                'form',
                                'partofspeech'),
                 'enveloping': None,
                 'meta': {},
                 'name': 'morph_analysis',
                 'parent': 'words',
                 'secondary_attributes': (),
                 'serialisation_module': None,
                 'spans': [{'annotations': [{'clitic': '',
                                             'ending': 'st',
                                             'form': 'sg el',
                                             'lemma': 'valge',
                                             'normalized_text': 'Valgest',
                                             'partofspeech': 'A',
                                             'root': 'valge',
                                             'root_tokens': ['valge']}],
                            'base_span': (0, 7)},
                           {'annotations': [{'clitic': '',
                                             'ending': 'st',
                                             'form': 'sg el',
                                             'lemma': 'torn',
                                             'normalized_text': 'tornist',
                                             'partofspeech': 'S',
                                             'root': 'torn',
                                             'root_tokens': ['torn']}],
                            'base_span': (8, 15)},
                           {'annotations': [{'clitic': '',
                                             'ending': 'b',
                                             'form': 'b',
                                             'lemma': 'tõusma',
                                             'normalized_text': 'tõuseb',
                                             'partofspeech': 'V',
                                             'root': 'tõus',
                                             'root_tokens': ['tõus']}],
                            'base_span': (16, 22)},
                           {'annotations': [{'clitic': '',
                                             'ending': '0',
                                             'form': 'sg p',
                                             'lemma': 'paks',
                                             'normalized_text': 'paksu',
                                             'partofspeech': 'A',
                                             'root': 'paks',
                                             'root_tokens': ['paks']}],
                            'base_span': (23, 28)},
                           {'annotations': [{'clitic': '',
                                             'ending': '0',
                                             'form': 'sg p',
                                             'lemma': 'suits',
                                             'normalized_text': 'suitsu',
                                             'partofspeech': 'S',
                                             'root': 'suits',
                                             'root_tokens': ['suits']}],
                            'base_span': (29, 35)},
                           {'annotations': [{'clitic': '',
                                             'ending': '',
                                             'form': '',
                                             'lemma': '.',
                                             'normalized_text': '.',
                                             'partofspeech': 'Z',
                                             'root': '.',
                                             'root_tokens': ['.']}],
                            'base_span': (36, 37)},
                           {'annotations': [{'clitic': '',
                                             'ending': '0',
                                             'form': '',
                                             'lemma': 'nii',
                                             'normalized_text': 'Nii',
                                             'partofspeech': 'D',
                                             'root': 'nii',
                                             'root_tokens': ['nii']}],
                            'base_span': (38, 41)},
                           {'annotations': [{'clitic': 'gi',
                                             'ending': 'ti',
                                             'form': 'ti',
                                             'lemma': 'hakkama',
                                             'normalized_text': 'hakatigi',
                                             'partofspeech': 'V',
                                             'root': 'hakka',
                                             'root_tokens': ['hakka']}],
                            'base_span': (42, 50)},
                           {'annotations': [{'clitic': '',
                                             'ending': '0',
                                             'form': '?',
                                             'lemma': '70.',
                                             'normalized_text': '70.',
                                             'partofspeech': 'O',
                                             'root': '70.',
                                             'root_tokens': ['70.']}],
                            'base_span': (51, 54)},
                           {'annotations': [{'clitic': '',
                                             'ending': 'te',
                                             'form': 'pl g',
                                             'lemma': 'aasta',
                                             'normalized_text': 'aastate',
                                             'partofspeech': 'S',
                                             'root': 'aasta',
                                             'root_tokens': ['aasta']}],
                            'base_span': (55, 62)},
                           {'annotations': [{'clitic': '',
                                             'ending': 'st',
                                             'form': 'sg el',
                                             'lemma': 'lõpp',
                                             'normalized_text': 'lõpust',
                                             'partofspeech': 'S',
                                             'root': 'lõpp',
                                             'root_tokens': ['lõpp']}],
                            'base_span': (63, 69)},
                           {'annotations': [{'clitic': '',
                                             'ending': 'ma',
                                             'form': 'ma',
                                             'lemma': 'kasutama',
                                             'normalized_text': 'kasutama',
                                             'partofspeech': 'V',
                                             'root': 'kasuta',
                                             'root_tokens': ['kasuta']}],
                            'base_span': (70, 78)},
                           {'annotations': [{'clitic': '',
                                             'ending': '0',
                                             'form': '?',
                                             'lemma': 'USA',
                                             'normalized_text': 'USA',
                                             'partofspeech': 'Y',
                                             'root': 'USA',
                                             'root_tokens': ['USA']}],
                            'base_span': (79, 82)},
                           {'annotations': [{'clitic': '',
                                             'ending': '0',
                                             'form': 'sg p',
                                             'lemma': 'aretusmaterjal',
                                             'normalized_text': 'aretusmaterjali',
                                             'partofspeech': 'S',
                                             'root': 'aretus_materjal',
                                             'root_tokens': ['aretus',
                                                             'materjal']}],
                            'base_span': (83, 98)},
                           {'annotations': [{'clitic': '',
                                             'ending': '',
                                             'form': '',
                                             'lemma': '.',
                                             'normalized_text': '.',
                                             'partofspeech': 'Z',
                                             'root': '.',
                                             'root_tokens': ['.']}],
                            'base_span': (99, 100)},
                           {'annotations': [{'clitic': '',
                                             'ending': '0',
                                             'form': '?',
                                             'lemma': 'tv',
                                             'normalized_text': 'TV',
                                             'partofspeech': 'Y',
                                             'root': 'tv',
                                             'root_tokens': ['tv']}],
                            'base_span': (101, 103)},
                           {'annotations': [{'clitic': '',
                                             'ending': '0',
                                             'form': '?',
                                             'lemma': '3',
                                             'normalized_text': '3',
                                             'partofspeech': 'N',
                                             'root': '3',
                                             'root_tokens': ['3']}],
                            'base_span': (104, 105)},
                           {'annotations': [{'clitic': '',
                                             'ending': 'sse',
                                             'form': 'sg ill',
                                             'lemma': 'eeter',
                                             'normalized_text': 'eetrisse',
                                             'partofspeech': 'S',
                                             'root': 'eeter',
                                             'root_tokens': ['eeter']}],
                            'base_span': (106, 114)},
                           {'annotations': [{'clitic': '',
                                             'ending': 'b',
                                             'form': 'b',
                                             'lemma': 'jõudma',
                                             'normalized_text': 'jõuab',
                                             'partofspeech': 'V',
                                             'root': 'jõud',
                                             'root_tokens': ['jõud']}],
                            'base_span': (115, 120)},
                           {'annotations': [{'clitic': '',
                                             'ending': 'l',
                                             'form': 'sg ad',
                                             'lemma': 'sügis',
                                             'normalized_text': 'sügisel',
                                             'partofspeech': 'S',
                                             'root': 'sügis',
                                             'root_tokens': ['sügis']}],
                            'base_span': (121, 128)},
                           {'annotations': [{'clitic': '',
                                             'ending': '0',
                                             'form': 'sg n',
                                             'lemma': 'uus',
                                             'normalized_text': 'uus',
                                             'partofspeech': 'A',
                                             'root': 'uus',
                                             'root_tokens': ['uus']}],
                            'base_span': (129, 132)},
                           {'annotations': [{'clitic': '',
                                             'ending': '0',
                                             'form': 'sg n',
                                             'lemma': 'mälumängusari',
                                             'normalized_text': 'mälumängusari',
                                             'partofspeech': 'S',
                                             'root': 'mälu_mängu_sari',
                                             'root_tokens': ['mälu',
                                                             'mängu',
                                                             'sari']}],
                            'base_span': (133, 146)},
                           {'annotations': [{'clitic': '',
                                             'ending': '',
                                             'form': '',
                                             'lemma': '.',
                                             'normalized_text': '.',
                                             'partofspeech': 'Z',
                                             'root': '.',
                                             'root_tokens': ['.']}],
                            'base_span': (147, 148)}]},
                {'ambiguous': False,
                 'attributes': (),
                 'enveloping': 'words',
                 'meta': {},
                 'name': 'sentences',
                 'parent': None,
                 'secondary_attributes': (),
                 'serialisation_module': None,
                 'spans': [{'annotations': [{}],
                            'base_span': ((0, 7),
                                          (8, 15),
                                          (16, 22),
                                          (23, 28),
                                          (29, 35),
                                          (36, 37))},
                           {'annotations': [{}],
                            'base_span': ((38, 41),
                                          (42, 50),
                                          (51, 54),
                                          (55, 62),
                                          (63, 69),
                                          (70, 78),
                                          (79, 82),
                                          (83, 98),
                                          (99, 100))},
                           {'annotations': [{}],
                            'base_span': ((101, 103),
                                          (104, 105),
                                          (106, 114),
                                          (115, 120),
                                          (121, 128),
                                          (129, 132),
                                          (133, 146),
                                          (147, 148))}]},
                {'ambiguous': False,
                 'attributes': ('id',
                                'lemma',
                                'upostag',
                                'xpostag',
                                'feats',
                                'head',
                                'deprel',
                                'deps',
                                'misc',
                                'parent_span',
                                'children'),
                 'enveloping': None,
                 'meta': {},
                 'name': 'stanza_syntax',
                 'parent': 'morph_analysis',
                 'secondary_attributes': ('parent_span', 'children'),
                 'serialisation_module': 'syntax_v0',
                 'spans': [{'annotations': [{'deprel': 'amod',
                                             'deps': '_',
                                             'feats': OrderedDict([('pos', 'pos'),
                                                                   ('sg', 'sg'),
                                                                   ('el', 'el')]),
                                             'head': 2,
                                             'id': 1,
                                             'lemma': 'valge',
                                             'misc': '_',
                                             'upostag': 'A',
                                             'xpostag': 'A'}],
                            'base_span': (0, 7)},
                           {'annotations': [{'deprel': 'obl',
                                             'deps': '_',
                                             'feats': OrderedDict([('com', 'com'),
                                                                   ('sg', 'sg'),
                                                                   ('el', 'el')]),
                                             'head': 3,
                                             'id': 2,
                                             'lemma': 'torn',
                                             'misc': '_',
                                             'upostag': 'S',
                                             'xpostag': 'S'}],
                            'base_span': (8, 15)},
                           {'annotations': [{'deprel': 'root',
                                             'deps': '_',
                                             'feats': OrderedDict([('mod', 'mod'),
                                                                   ('indic',
                                                                    'indic'),
                                                                   ('pres', 'pres'),
                                                                   ('ps3', 'ps3'),
                                                                   ('sg', 'sg'),
                                                                   ('ps', 'ps'),
                                                                   ('af', 'af')]),
                                             'head': 0,
                                             'id': 3,
                                             'lemma': 'tõusma',
                                             'misc': '_',
                                             'upostag': 'V',
                                             'xpostag': 'V'}],
                            'base_span': (16, 22)},
                           {'annotations': [{'deprel': 'amod',
                                             'deps': '_',
                                             'feats': OrderedDict([('pos', 'pos'),
                                                                   ('sg', 'sg'),
                                                                   ('part',
                                                                    'part')]),
                                             'head': 5,
                                             'id': 4,
                                             'lemma': 'paks',
                                             'misc': '_',
                                             'upostag': 'A',
                                             'xpostag': 'A'}],
                            'base_span': (23, 28)},
                           {'annotations': [{'deprel': 'nsubj',
                                             'deps': '_',
                                             'feats': OrderedDict([('com', 'com'),
                                                                   ('sg', 'sg'),
                                                                   ('part',
                                                                    'part')]),
                                             'head': 3,
                                             'id': 5,
                                             'lemma': 'suits',
                                             'misc': '_',
                                             'upostag': 'S',
                                             'xpostag': 'S'}],
                            'base_span': (29, 35)},
                           {'annotations': [{'deprel': 'punct',
                                             'deps': '_',
                                             'feats': OrderedDict(),
                                             'head': 3,
                                             'id': 6,
                                             'lemma': '.',
                                             'misc': '_',
                                             'upostag': 'Z',
                                             'xpostag': 'Z'}],
                            'base_span': (36, 37)},
                           {'annotations': [{'deprel': 'advmod',
                                             'deps': '_',
                                             'feats': OrderedDict(),
                                             'head': 2,
                                             'id': 1,
                                             'lemma': 'nii',
                                             'misc': '_',
                                             'upostag': 'D',
                                             'xpostag': 'D'}],
                            'base_span': (38, 41)},
                           {'annotations': [{'deprel': 'root',
                                             'deps': '_',
                                             'feats': OrderedDict([('main', 'main'),
                                                                   ('indic',
                                                                    'indic'),
                                                                   ('impf', 'impf'),
                                                                   ('imps', 'imps'),
                                                                   ('af', 'af')]),
                                             'head': 0,
                                             'id': 2,
                                             'lemma': 'hakkama',
                                             'misc': '_',
                                             'upostag': 'V',
                                             'xpostag': 'V'}],
                            'base_span': (42, 50)},
                           {'annotations': [{'deprel': 'amod',
                                             'deps': '_',
                                             'feats': OrderedDict([('ord', 'ord'),
                                                                   ('<?>', '<?>'),
                                                                   ('roman',
                                                                    'roman')]),
                                             'head': 4,
                                             'id': 3,
                                             'lemma': '70.',
                                             'misc': '_',
                                             'upostag': 'N',
                                             'xpostag': 'N'}],
                            'base_span': (51, 54)},
                           {'annotations': [{'deprel': 'nmod',
                                             'deps': '_',
                                             'feats': OrderedDict([('com', 'com'),
                                                                   ('pl', 'pl'),
                                                                   ('gen', 'gen')]),
                                             'head': 5,
                                             'id': 4,
                                             'lemma': 'aasta',
                                             'misc': '_',
                                             'upostag': 'S',
                                             'xpostag': 'S'}],
                            'base_span': (55, 62)},
                           {'annotations': [{'deprel': 'obl',
                                             'deps': '_',
                                             'feats': OrderedDict([('com', 'com'),
                                                                   ('sg', 'sg'),
                                                                   ('el', 'el')]),
                                             'head': 6,
                                             'id': 5,
                                             'lemma': 'lõpp',
                                             'misc': '_',
                                             'upostag': 'S',
                                             'xpostag': 'S'}],
                            'base_span': (63, 69)},
                           {'annotations': [{'deprel': 'xcomp',
                                             'deps': '_',
                                             'feats': OrderedDict([('aux', 'aux'),
                                                                   ('sup', 'sup'),
                                                                   ('ps', 'ps'),
                                                                   ('ill', 'ill')]),
                                             'head': 2,
                                             'id': 6,
                                             'lemma': 'kasutama',
                                             'misc': '_',
                                             'upostag': 'V',
                                             'xpostag': 'V'}],
                            'base_span': (70, 78)},
                           {'annotations': [{'deprel': 'nmod',
                                             'deps': '_',
                                             'feats': OrderedDict([('nominal',
                                                                    'nominal')]),
                                             'head': 8,
                                             'id': 7,
                                             'lemma': 'USA',
                                             'misc': '_',
                                             'upostag': 'Y',
                                             'xpostag': 'Y'}],
                            'base_span': (79, 82)},
                           {'annotations': [{'deprel': 'obj',
                                             'deps': '_',
                                             'feats': OrderedDict([('com', 'com'),
                                                                   ('sg', 'sg'),
                                                                   ('part',
                                                                    'part')]),
                                             'head': 6,
                                             'id': 8,
                                             'lemma': 'aretusmaterjal',
                                             'misc': '_',
                                             'upostag': 'S',
                                             'xpostag': 'S'}],
                            'base_span': (83, 98)},
                           {'annotations': [{'deprel': 'punct',
                                             'deps': '_',
                                             'feats': OrderedDict(),
                                             'head': 2,
                                             'id': 9,
                                             'lemma': '.',
                                             'misc': '_',
                                             'upostag': 'Z',
                                             'xpostag': 'Z'}],
                            'base_span': (99, 100)},
                           {'annotations': [{'deprel': 'nmod',
                                             'deps': '_',
                                             'feats': OrderedDict([('nominal',
                                                                    'nominal')]),
                                             'head': 3,
                                             'id': 1,
                                             'lemma': 'tv',
                                             'misc': '_',
                                             'upostag': 'Y',
                                             'xpostag': 'Y'}],
                            'base_span': (101, 103)},
                           {'annotations': [{'deprel': 'nummod',
                                             'deps': '_',
                                             'feats': OrderedDict([('card', 'card'),
                                                                   ('<?>', '<?>'),
                                                                   ('digit',
                                                                    'digit')]),
                                             'head': 1,
                                             'id': 2,
                                             'lemma': '3',
                                             'misc': '_',
                                             'upostag': 'N',
                                             'xpostag': 'N'}],
                            'base_span': (104, 105)},
                           {'annotations': [{'deprel': 'obl',
                                             'deps': '_',
                                             'feats': OrderedDict([('com', 'com'),
                                                                   ('sg', 'sg'),
                                                                   ('ill', 'ill')]),
                                             'head': 4,
                                             'id': 3,
                                             'lemma': 'eeter',
                                             'misc': '_',
                                             'upostag': 'S',
                                             'xpostag': 'S'}],
                            'base_span': (106, 114)},
                           {'annotations': [{'deprel': 'root',
                                             'deps': '_',
                                             'feats': OrderedDict([('main', 'main'),
                                                                   ('indic',
                                                                    'indic'),
                                                                   ('pres', 'pres'),
                                                                   ('ps3', 'ps3'),
                                                                   ('sg', 'sg'),
                                                                   ('ps', 'ps'),
                                                                   ('af', 'af')]),
                                             'head': 0,
                                             'id': 4,
                                             'lemma': 'jõudma',
                                             'misc': '_',
                                             'upostag': 'V',
                                             'xpostag': 'V'}],
                            'base_span': (115, 120)},
                           {'annotations': [{'deprel': 'obl',
                                             'deps': '_',
                                             'feats': OrderedDict([('com', 'com'),
                                                                   ('sg', 'sg'),
                                                                   ('ad', 'ad')]),
                                             'head': 4,
                                             'id': 5,
                                             'lemma': 'sügis',
                                             'misc': '_',
                                             'upostag': 'S',
                                             'xpostag': 'S'}],
                            'base_span': (121, 128)},
                           {'annotations': [{'deprel': 'amod',
                                             'deps': '_',
                                             'feats': OrderedDict([('pos', 'pos'),
                                                                   ('sg', 'sg'),
                                                                   ('nom', 'nom')]),
                                             'head': 7,
                                             'id': 6,
                                             'lemma': 'uus',
                                             'misc': '_',
                                             'upostag': 'A',
                                             'xpostag': 'A'}],
                            'base_span': (129, 132)},
                           {'annotations': [{'deprel': 'nsubj',
                                             'deps': '_',
                                             'feats': OrderedDict([('com', 'com'),
                                                                   ('sg', 'sg'),
                                                                   ('nom', 'nom')]),
                                             'head': 4,
                                             'id': 7,
                                             'lemma': 'mälumängusari',
                                             'misc': '_',
                                             'upostag': 'S',
                                             'xpostag': 'S'}],
                            'base_span': (133, 146)},
                           {'annotations': [{'deprel': 'punct',
                                             'deps': '_',
                                             'feats': OrderedDict(),
                                             'head': 4,
                                             'id': 8,
                                             'lemma': '.',
                                             'misc': '_',
                                             'upostag': 'Z',
                                             'xpostag': 'Z'}],
                            'base_span': (147, 148)}]}],
     'meta': {},
     'relation_layers': [],
     'text': 'Valgest tornist tõuseb paksu suitsu . Nii hakatigi 70. aastate '
             'lõpust kasutama USA aretusmaterjali . TV 3 eetrisse jõuab sügisel '
             'uus mälumängusari .'}


@unittest.skipIf(WORDNET_PATH is None,
                 reason="Wordnet's database is required by this test. Use estnltk.download('wordnet') to fetch the missing resource.")
def test_syntax_time_loc_tagger_smoke():
    # Smoke test that TimeLocTagger works on stanza_syntax layer
    time_loc_tagger = TimeLocTagger(syntax_layer='stanza_syntax', 
                                    sentences_layer='sentences')
    # Case 1
    # 'text': 'Valgest tornist tõuseb paksu suitsu . Nii hakatigi 70. aastate '
    #         'lõpust kasutama USA aretusmaterjali . TV 3 eetrisse jõuab sügisel '
    #         'uus mälumängusari .'
    three_sentences = dict_to_text( example_input_1_dict )
    assert len(three_sentences['sentences']) == 3
    time_loc_tagger.tag( three_sentences )
    # Validate output_layer
    assert time_loc_tagger.output_layer in three_sentences.layers
    assert layer_to_dict(three_sentences[time_loc_tagger.output_layer]) == \
        {'ambiguous': False,
         'attributes': ('phrase_type', 'root_id', 'root'),
         'enveloping': 'stanza_syntax',
         'meta': {},
         'name': 'time_loc_phrases',
         'parent': None,
         'secondary_attributes': (),
         'serialisation_module': 'syntax_phrases_v0',
         'spans': [{'annotations': [{'phrase_type': 'LOC', 'root_id': 2}],
                    'base_span': ((0, 7), (8, 15))},
                   {'annotations': [{'phrase_type': 'TIME', 'root_id': 5}],
                    'base_span': ((51, 54), (55, 62), (63, 69))},
                   {'annotations': [{'phrase_type': 'LOC', 'root_id': 3}],
                    'base_span': ((101, 103), (104, 105), (106, 114))},
                   {'annotations': [{'phrase_type': 'TIME', 'root_id': 5}],
                    'base_span': ((121, 128),)}]}
    phrases = []
    for phrase in three_sentences[time_loc_tagger.output_layer]:
        phrases.append( (phrase.enclosing_text, phrase.phrase_type) )
    assert phrases == \
        [('Valgest tornist', 'LOC'), 
         ('70. aastate lõpust', 'TIME'), 
         ('TV 3 eetrisse', 'LOC'), 
         ('sügisel', 'TIME')]

   