import logging
import inspect
from enum import Enum
from tornado.web import RequestHandler
from etornado.error_code_manager import HTTP_OK, ErrorCode, error_code_manager


class AggregateHttMethodMeta(type):

    def __init__(cls, *args, **kwargs):
        super().__init__(*args, **kwargs)
        cls.head = cls.process
        cls.get = cls.process
        cls.post = cls.process
        cls.delete = cls.process
        cls.patch = cls.process
        cls.put = cls.process
        cls.options = cls.process


class BaseHandler(RequestHandler, metaclass=AggregateHttMethodMeta):

    def initialize(self, **kwargs):
        self.logger = logging.getLogger(self.__class__.__name__)
        for k, v in kwargs.items():
            if hasattr(self, k):
                self.logger.exception("member [%s] has exists", k)
                raise Exception("member [%s] has exists" % k)
            setattr(self, k, v)
        self.url_args = []

    def write_error(self, status_code, **kwargs):
        error_code = kwargs.get("ec")
        error_info = error_code_manager.format_error_info(error_code, **kwargs)
        self.logger.error(
            "write error for url[%s], method[%s], error_info[%s]",
            self.request.uri, self.request.method, error_info)
        self.write(error_info)
        self.finish()

    async def process(self, *args, **kwargs):
        method = getattr(self, "do_" + self.request.method.lower(), None)
        self.url_args = args
        if method is None:
            self.send_error(HTTP_OK,
                            ec=ErrorCode.UNSUPPORTED_METHOD,
                            method=self.request.method,
                            url=self.request.uri)
            return
        try:
            r = method()
            if inspect.isawaitable(r):
                r = await r
            error_code, result, error_args = self.__parse_result(r)
        except Exception as e:
            self.logger.exception("exception [%s] catched when call [%s]", e, method)
            self.send_error(HTTP_OK, ec=ErrorCode.UNKNOWN)
            return
        response = error_code_manager.format_error_info(error_code, **error_args)
        if error_code == ErrorCode.NONE.value:
            response["result"] = result
        self.write(response)
        self.finish()

    def __parse_result(self, r):
        if isinstance(r, tuple):
            if len(r) != 2:
                raise Exception("tuple result [%s]'s length must be 2" % r)
            if isinstance(r[0], Enum):
                r = (r[0].value, r[1])
            if r[0] != ErrorCode.NONE.value:
                if not isinstance(r[1], dict):
                    raise Exception("error info [%s] args must be dict" % r[1])
                return r[0], {}, r[1]
            else:
                return r[0], r[1], {}
        return ErrorCode.NONE.value, r, {}
