"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const iam = require("../lib");
let stack;
let resource;
beforeEach(() => {
    stack = new core_1.Stack();
    resource = new core_1.CfnResource(stack, 'SomeResource', {
        type: 'CDK::Test::SomeResource',
    });
});
describe('IAM grant', () => {
    test('Grant.drop() returns a no-op grant', () => {
        const user = new iam.User(stack, 'poo');
        const grant = iam.Grant.drop(user, 'dropping me');
        expect(grant.success).toBeFalsy();
        expect(grant.principalStatement).toBeUndefined();
        expect(grant.resourceStatement).toBeUndefined();
    });
});
describe('Grant dependencies', () => {
    test('can depend on grant added to role', () => {
        // GIVEN
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('bla.amazonaws.com'),
        });
        // WHEN
        applyGrantWithDependencyTo(role);
        // THEN
        expectDependencyOn('RoleDefaultPolicy5FFB7DAB');
    });
    test('can depend on grant added to role wrapped with conditions', () => {
        // GIVEN
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('bla.amazonaws.com'),
        });
        // WHEN
        applyGrantWithDependencyTo(new iam.PrincipalWithConditions(role, {
            StringEquals: { 'aws:something': 'happy' },
        }));
        // THEN
        expectDependencyOn('RoleDefaultPolicy5FFB7DAB');
    });
    test('can depend on grant added to lazy role', () => {
        // GIVEN
        const role = new iam.LazyRole(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('bla.amazonaws.com'),
        });
        // WHEN
        applyGrantWithDependencyTo(role);
        Array.isArray(role.roleArn); // Force instantiation
        // THEN
        expectDependencyOn('RoleDefaultPolicy5FFB7DAB');
    });
    test('can depend on grant added to resource', () => {
        // WHEN
        iam.Grant.addToPrincipalOrResource({
            actions: ['service:DoAThing'],
            grantee: new iam.ServicePrincipal('bla.amazonaws.com'),
            resourceArns: ['*'],
            resource: new FakeResourceWithPolicy(stack, 'Buckaroo'),
        }).applyBefore(resource);
        // THEN
        expectDependencyOn('BuckarooPolicy74174DA4');
    });
    test('can depend on grant added to principal AND resource', () => {
        // GIVEN
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('bla.amazonaws.com'),
        });
        // WHEN
        iam.Grant.addToPrincipalAndResource({
            actions: ['service:DoAThing'],
            grantee: role,
            resourceArns: ['*'],
            resource: new FakeResourceWithPolicy(stack, 'Buckaroo'),
        }).applyBefore(resource);
        // THEN
        expectDependencyOn('RoleDefaultPolicy5FFB7DAB');
        expectDependencyOn('BuckarooPolicy74174DA4');
    });
});
function applyGrantWithDependencyTo(principal) {
    iam.Grant.addToPrincipal({
        actions: ['service:DoAThing'],
        grantee: principal,
        resourceArns: ['*'],
    }).applyBefore(resource);
}
function expectDependencyOn(id) {
    expect(stack).toHaveResource('CDK::Test::SomeResource', (props) => {
        var _a;
        return ((_a = props === null || props === void 0 ? void 0 : props.DependsOn) !== null && _a !== void 0 ? _a : []).includes(id);
    }, assert_1.ResourcePart.CompleteDefinition);
}
class FakeResourceWithPolicy extends core_1.CfnResource {
    constructor(scope, id) {
        super(scope, id, {
            type: 'CDK::Test::Buckaroo',
        });
        this.policy = new core_1.CfnResource(this, 'Policy', {
            type: 'CDK::Test::BuckarooPolicy',
        });
    }
    addToResourcePolicy(_statement) {
        return { statementAdded: true, policyDependable: this.policy };
    }
}
//# sourceMappingURL=data:application/json;base64,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