"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
const roleAccount = '123456789012';
const notRoleAccount = '012345678901';
describe('IAM Role.fromRoleArn', () => {
    let app;
    beforeEach(() => {
        app = new core_1.App();
    });
    let roleStack;
    let importedRole;
    describe('imported with a static ARN', () => {
        const roleName = 'MyRole';
        describe('into an env-agnostic stack', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack');
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`);
            });
            test('correctly parses the imported role ARN', () => {
                expect(importedRole.roleArn).toBe(`arn:aws:iam::${roleAccount}:role/${roleName}`);
            });
            test('correctly parses the imported role name', () => {
                expect(importedRole.roleName).toBe(roleName);
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('returns true', () => {
                    expect(addToPolicyResult).toBe(true);
                });
                test("generates a default Policy resource pointing at the imported role's physical name", () => {
                    assertRoleHasDefaultPolicy(roleStack, roleName);
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to the same stack as the imported role', () => {
                    beforeEach(() => {
                        importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a different env-agnostic stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack');
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a targeted stack, with account set to', () => {
                    describe('the same account as in the ARN of the imported role', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('correctly attaches the Policy to the imported role', () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('a different account than in the ARN of the imported role', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('does NOT attach the Policy to the imported role', () => {
                            assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                        });
                    });
                });
            });
        });
        describe('into a targeted stack with account set to', () => {
            describe('the same account as in the ARN the role was imported with', () => {
                beforeEach(() => {
                    roleStack = new core_1.Stack(app, 'RoleStack', { env: { account: roleAccount } });
                    importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`);
                });
                describe('then adding a PolicyStatement to it', () => {
                    let addToPolicyResult;
                    beforeEach(() => {
                        addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                    });
                    test('returns true', () => {
                        expect(addToPolicyResult).toBe(true);
                    });
                    test("generates a default Policy resource pointing at the imported role's physical name", () => {
                        assertRoleHasDefaultPolicy(roleStack, roleName);
                    });
                });
                describe('then attaching a Policy to it', () => {
                    describe('that belongs to the same stack as the imported role', () => {
                        beforeEach(() => {
                            importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                        });
                        test('correctly attaches the Policy to the imported role', () => {
                            assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('that belongs to an env-agnostic stack', () => {
                        let policyStack;
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack');
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('correctly attaches the Policy to the imported role', () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('that belongs to a targeted stack, with account set to', () => {
                        let policyStack;
                        describe('the same account as in the imported role ARN and in the stack the imported role belongs to', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test('correctly attaches the Policy to the imported role', () => {
                                assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                            });
                        });
                        describe('a different account than in the imported role ARN and in the stack the imported role belongs to', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test('does NOT attach the Policy to the imported role', () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                    });
                });
            });
            describe('a different account than in the ARN the role was imported with', () => {
                beforeEach(() => {
                    roleStack = new core_1.Stack(app, 'RoleStack', { env: { account: notRoleAccount } });
                    importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`);
                });
                describe('then adding a PolicyStatement to it', () => {
                    let addToPolicyResult;
                    beforeEach(() => {
                        addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                    });
                    test('pretends to succeed', () => {
                        expect(addToPolicyResult).toBe(true);
                    });
                    test("does NOT generate a default Policy resource pointing at the imported role's physical name", () => {
                        expect(roleStack).not.toHaveResourceLike('AWS::IAM::Policy');
                    });
                });
                describe('then attaching a Policy to it', () => {
                    describe('that belongs to the same stack as the imported role', () => {
                        beforeEach(() => {
                            importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                        });
                        test('does NOT attach the Policy to the imported role', () => {
                            assertPolicyDidNotAttachToRole(roleStack, 'MyPolicy');
                        });
                    });
                    describe('that belongs to an env-agnostic stack', () => {
                        let policyStack;
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack');
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('does NOT attach the Policy to the imported role', () => {
                            assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                        });
                    });
                    describe('that belongs to a different targeted stack, with account set to', () => {
                        let policyStack;
                        describe('the same account as in the ARN of the imported role', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test('does NOT attach the Policy to the imported role', () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                        describe('the same account as in the stack the imported role belongs to', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test('does NOT attach the Policy to the imported role', () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                        describe('a third account, different from both the role and scope stack accounts', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: 'some-random-account' } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test('does NOT attach the Policy to the imported role', () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                    });
                });
            });
        });
        describe('and with mutable=false', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack');
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`, { mutable: false });
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('pretends to succeed', () => {
                    expect(addToPolicyResult).toBe(true);
                });
                test("does NOT generate a default Policy resource pointing at the imported role's physical name", () => {
                    expect(roleStack).not.toHaveResourceLike('AWS::IAM::Policy');
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to a stack with account equal to the account in the imported role ARN', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test('does NOT attach the Policy to the imported role', () => {
                        assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                    });
                });
            });
        });
    });
    describe('imported with a dynamic ARN', () => {
        const dynamicValue = core_1.Lazy.stringValue({ produce: () => 'role-arn' });
        const roleName = {
            'Fn::Select': [1,
                {
                    'Fn::Split': ['/',
                        {
                            'Fn::Select': [5,
                                { 'Fn::Split': [':', 'role-arn'] },
                            ],
                        },
                    ],
                },
            ],
        };
        describe('into an env-agnostic stack', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack');
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', dynamicValue);
            });
            test('correctly parses the imported role ARN', () => {
                expect(importedRole.roleArn).toBe(dynamicValue);
            });
            test('correctly parses the imported role name', () => {
                new lib_1.Role(roleStack, 'AnyRole', {
                    roleName: 'AnyRole',
                    assumedBy: new lib_1.ArnPrincipal(importedRole.roleName),
                });
                expect(roleStack).toHaveResourceLike('AWS::IAM::Role', {
                    'AssumeRolePolicyDocument': {
                        'Statement': [
                            {
                                'Action': 'sts:AssumeRole',
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': roleName,
                                },
                            },
                        ],
                    },
                });
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('returns true', () => {
                    expect(addToPolicyResult).toBe(true);
                });
                test("generates a default Policy resource pointing at the imported role's physical name", () => {
                    assertRoleHasDefaultPolicy(roleStack, roleName);
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to the same stack as the imported role', () => {
                    beforeEach(() => {
                        importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a different env-agnostic stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack');
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a targeted stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
            });
        });
        describe('into a targeted stack with account set', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack', { env: { account: roleAccount } });
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', dynamicValue);
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('returns true', () => {
                    expect(addToPolicyResult).toBe(true);
                });
                test("generates a default Policy resource pointing at the imported role's physical name", () => {
                    assertRoleHasDefaultPolicy(roleStack, roleName);
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to the same stack as the imported role', () => {
                    beforeEach(() => {
                        importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to an env-agnostic stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack');
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a different targeted stack, with account set to', () => {
                    describe('the same account as the stack the role was imported into', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('correctly attaches the Policy to the imported role', () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('a different account than the stack the role was imported into', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('correctly attaches the Policy to the imported role', () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                });
            });
        });
    });
    describe('imported with the ARN of a service role', () => {
        beforeEach(() => {
            roleStack = new core_1.Stack();
        });
        describe('without a service principal in the role name', () => {
            beforeEach(() => {
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'Role', `arn:aws:iam::${roleAccount}:role/service-role/codebuild-role`);
            });
            it("correctly strips the 'service-role' prefix from the role name", () => {
                new lib_1.Policy(roleStack, 'Policy', {
                    statements: [somePolicyStatement()],
                    roles: [importedRole],
                });
                expect(roleStack).toHaveResourceLike('AWS::IAM::Policy', {
                    'Roles': [
                        'codebuild-role',
                    ],
                });
            });
        });
        describe('with a service principal in the role name', () => {
            beforeEach(() => {
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'Role', `arn:aws:iam::${roleAccount}:role/aws-service-role/anyservice.amazonaws.com/codebuild-role`);
            });
            it("correctly strips both the 'aws-service-role' prefix and the service principal from the role name", () => {
                new lib_1.Policy(roleStack, 'Policy', {
                    statements: [somePolicyStatement()],
                    roles: [importedRole],
                });
                expect(roleStack).toHaveResourceLike('AWS::IAM::Policy', {
                    'Roles': [
                        'codebuild-role',
                    ],
                });
            });
        });
    });
});
function somePolicyStatement() {
    return new lib_1.PolicyStatement({
        actions: ['s3:*'],
        resources: ['xyz'],
    });
}
function somePolicy(policyStack, policyName) {
    const someRole = new lib_1.Role(policyStack, 'SomeExampleRole', {
        assumedBy: new lib_1.AnyPrincipal(),
    });
    const roleResource = someRole.node.defaultChild;
    roleResource.overrideLogicalId('SomeRole'); // force a particular logical ID in the Ref expression
    return new lib_1.Policy(policyStack, 'MyPolicy', {
        policyName,
        statements: [somePolicyStatement()],
        // need at least one of user/group/role, otherwise validation fails
        roles: [someRole],
    });
}
function assertRoleHasDefaultPolicy(stack, roleName) {
    _assertStackContainsPolicyResource(stack, [roleName], undefined);
}
function assertRoleHasAttachedPolicy(stack, roleName, attachedPolicyName) {
    _assertStackContainsPolicyResource(stack, [{ Ref: 'SomeRole' }, roleName], attachedPolicyName);
}
function assertPolicyDidNotAttachToRole(stack, policyName) {
    _assertStackContainsPolicyResource(stack, [{ Ref: 'SomeRole' }], policyName);
}
function _assertStackContainsPolicyResource(stack, roleNames, nameOfPolicy) {
    const expected = {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:*',
                    Effect: 'Allow',
                    Resource: 'xyz',
                },
            ],
        },
        Roles: roleNames,
    };
    if (nameOfPolicy) {
        expected.PolicyName = nameOfPolicy;
    }
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', expected);
}
//# sourceMappingURL=data:application/json;base64,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