import cdk = require('@aws-cdk/core');
import { PolicyStatement } from "./policy-statement";
import { IGrantable } from "./principals";
/**
 * Basic options for a grant operation
 */
export interface CommonGrantOptions {
    /**
     * The principal to grant to
     *
     * @default if principal is undefined, no work is done.
     */
    readonly grantee: IGrantable;
    /**
     * The actions to grant
     */
    readonly actions: string[];
    /**
     * The resource ARNs to grant to
     */
    readonly resourceArns: string[];
}
/**
 * Options for a grant operation
 */
export interface GrantWithResourceOptions extends CommonGrantOptions {
    /**
     * The resource with a resource policy
     *
     * The statement will be added to the resource policy if it couldn't be
     * added to the principal policy.
     */
    readonly resource: IResourceWithPolicy;
    /**
     * When referring to the resource in a resource policy, use this as ARN.
     *
     * (Depending on the resource type, this needs to be '*' in a resource policy).
     *
     * @default Same as regular resource ARNs
     */
    readonly resourceSelfArns?: string[];
}
/**
 * Options for a grant operation that only applies to principals
 */
export interface GrantOnPrincipalOptions extends CommonGrantOptions {
    /**
     * Construct to report warnings on in case grant could not be registered
     */
    readonly scope?: cdk.IConstruct;
}
/**
 * Options for a grant operation to both identity and resource
 */
export interface GrantOnPrincipalAndResourceOptions extends CommonGrantOptions {
    /**
     * The resource with a resource policy
     *
     * The statement will always be added to the resource policy.
     */
    readonly resource: IResourceWithPolicy;
    /**
     * When referring to the resource in a resource policy, use this as ARN.
     *
     * (Depending on the resource type, this needs to be '*' in a resource policy).
     *
     * @default Same as regular resource ARNs
     */
    readonly resourceSelfArns?: string[];
}
/**
 * Result of a grant() operation
 *
 * This class is not instantiable by consumers on purpose, so that they will be
 * required to call the Grant factory functions.
 */
export declare class Grant {
    /**
     * Grant the given permissions to the principal
     *
     * The permissions will be added to the principal policy primarily, falling
     * back to the resource policy if necessary. The permissions must be granted
     * somewhere.
     *
     * - Trying to grant permissions to a principal that does not admit adding to
     *   the principal policy while not providing a resource with a resource policy
     *   is an error.
     * - Trying to grant permissions to an absent principal (possible in the
     *   case of imported resources) leads to a warning being added to the
     *   resource construct.
     */
    static addToPrincipalOrResource(options: GrantWithResourceOptions): Grant;
    /**
     * Try to grant the given permissions to the given principal
     *
     * Absence of a principal leads to a warning, but failing to add
     * the permissions to a present principal is not an error.
     */
    static addToPrincipal(options: GrantOnPrincipalOptions): Grant;
    /**
     * Add a grant both on the principal and on the resource
     *
     * As long as any principal is given, granting on the pricipal may fail (in
     * case of a non-identity principal), but granting on the resource will
     * never fail.
     *
     * Statement will be the resource statement.
     */
    static addToPrincipalAndResource(options: GrantOnPrincipalAndResourceOptions): Grant;
    /**
     * Returns a "no-op" `Grant` object which represents a "dropped grant".
     *
     * This can be used for e.g. imported resources where you may not be able to modify
     * the resource's policy or some underlying policy which you don't know about.
     *
     * @param grantee The intended grantee
     * @param _intent The user's intent (will be ignored at the moment)
     */
    static drop(grantee: IGrantable, _intent: string): Grant;
    /**
     * The statement that was added to the principal's policy
     *
     * Can be accessed to (e.g.) add additional conditions to the statement.
     */
    readonly principalStatement?: PolicyStatement;
    /**
     * The statement that was added to the resource policy
     *
     * Can be accessed to (e.g.) add additional conditions to the statement.
     */
    readonly resourceStatement?: PolicyStatement;
    /**
     * The options originally used to set this result
     *
     * Private member doubles as a way to make it impossible for an object literal to
     * be structurally the same as this class.
     */
    private readonly options;
    private constructor();
    /**
     * Whether the grant operation was successful
     */
    readonly success: boolean;
    /**
     * Throw an error if this grant wasn't successful
     */
    assertSuccess(): void;
}
/**
 * A resource with a resource policy that can be added to
 */
export interface IResourceWithPolicy extends cdk.IConstruct {
    /**
     * Add a statement to the resource's resource policy
     */
    addToResourcePolicy(statement: PolicyStatement): void;
}
