"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const role_1 = require("./role");
/**
 * An IAM role that only gets attached to the construct tree once it gets used, not before
 *
 * This construct can be used to simplify logic in other constructs
 * which need to create a role but only if certain configurations occur
 * (such as when AutoScaling is configured). The role can be configured in one
 * place, but if it never gets used it doesn't get instantiated and will
 * not be synthesized or deployed.
 *
 * @resource AWS::IAM::Role
 */
class LazyRole extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.grantPrincipal = this;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.statements = new Array();
        this.policies = new Array();
        this.managedPolicies = new Array();
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param statement The permission statement to add to the policy document
     */
    addToPolicy(statement) {
        if (this.role) {
            return this.role.addToPolicy(statement);
        }
        else {
            this.statements.push(statement);
            return true;
        }
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        if (this.role) {
            this.role.attachInlinePolicy(policy);
        }
        else {
            this.policies.push(policy);
        }
    }
    /**
     * Attaches a managed policy to this role.
     * @param policy The managed policy to attach.
     */
    addManagedPolicy(policy) {
        if (this.role) {
            this.role.addManagedPolicy(policy);
        }
        else {
            this.managedPolicies.push(policy);
        }
    }
    /**
     * Returns the ARN of this role.
     */
    get roleArn() {
        return this.instantiate().roleArn;
    }
    /** @attribute */
    get roleId() {
        return this.instantiate().roleId;
    }
    get roleName() {
        return this.instantiate().roleName;
    }
    get policyFragment() {
        return this.instantiate().policyFragment;
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(identity, ...actions) {
        return this.instantiate().grant(identity, ...actions);
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        return this.instantiate().grantPassRole(identity);
    }
    instantiate() {
        if (!this.role) {
            const role = new role_1.Role(this, 'Default', this.props);
            this.statements.forEach(role.addToPolicy.bind(role));
            this.policies.forEach(role.attachInlinePolicy.bind(role));
            this.managedPolicies.forEach(role.addManagedPolicy.bind(role));
            this.role = role;
        }
        return this.role;
    }
}
exports.LazyRole = LazyRole;
//# sourceMappingURL=data:application/json;base64,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