import { Construct, Duration, Resource } from '@aws-cdk/core';
import { Grant } from './grant';
import { IIdentity } from './identity-base';
import { IManagedPolicy } from './managed-policy';
import { Policy } from './policy';
import { PolicyDocument } from './policy-document';
import { PolicyStatement } from './policy-statement';
import { IPrincipal, PrincipalPolicyFragment } from './principals';
export interface RoleProps {
    /**
     * The IAM principal (i.e. `new ServicePrincipal('sns.amazonaws.com')`)
     * which can assume this role.
     *
     * You can later modify the assume role policy document by accessing it via
     * the `assumeRolePolicy` property.
     */
    readonly assumedBy: IPrincipal;
    /**
     * ID that the role assumer needs to provide when assuming this role
     *
     * If the configured and provided external IDs do not match, the
     * AssumeRole operation will fail.
     *
     * @default No external ID required
     */
    readonly externalId?: string;
    /**
     * A list of ARNs for managed policies associated with this role.
     * You can add managed policies later using `attachManagedPolicy(arn)`.
     *
     * @default - No managed policies.
     */
    readonly managedPolicies?: IManagedPolicy[];
    /**
     * A list of named policies to inline into this role. These policies will be
     * created with the role, whereas those added by ``addToPolicy`` are added
     * using a separate CloudFormation resource (allowing a way around circular
     * dependencies that could otherwise be introduced).
     *
     * @default - No policy is inlined in the Role resource.
     */
    readonly inlinePolicies?: {
        [name: string]: PolicyDocument;
    };
    /**
     * The path associated with this role. For information about IAM paths, see
     * Friendly Names and Paths in IAM User Guide.
     *
     * @default /
     */
    readonly path?: string;
    /**
     * A name for the IAM role. For valid values, see the RoleName parameter for
     * the CreateRole action in the IAM API Reference.
     *
     * IMPORTANT: If you specify a name, you cannot perform updates that require
     * replacement of this resource. You can perform updates that require no or
     * some interruption. If you must replace the resource, specify a new name.
     *
     * If you specify a name, you must specify the CAPABILITY_NAMED_IAM value to
     * acknowledge your template's capabilities. For more information, see
     * Acknowledging IAM Resources in AWS CloudFormation Templates.
     *
     * @default - AWS CloudFormation generates a unique physical ID and uses that ID
     * for the group name.
     */
    readonly roleName?: string;
    /**
     * The maximum session duration that you want to set for the specified role.
     * This setting can have a value from 1 hour (3600sec) to 12 (43200sec) hours.
     *
     * Anyone who assumes the role from the AWS CLI or API can use the
     * DurationSeconds API parameter or the duration-seconds CLI parameter to
     * request a longer session. The MaxSessionDuration setting determines the
     * maximum duration that can be requested using the DurationSeconds
     * parameter.
     *
     * If users don't specify a value for the DurationSeconds parameter, their
     * security credentials are valid for one hour by default. This applies when
     * you use the AssumeRole* API operations or the assume-role* CLI operations
     * but does not apply when you use those operations to create a console URL.
     *
     * @link https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use.html
     *
     * @default Duration.hours(1)
     */
    readonly maxSessionDuration?: Duration;
}
/**
 * IAM Role
 *
 * Defines an IAM role. The role is created with an assume policy document associated with
 * the specified AWS service principal defined in `serviceAssumeRole`.
 */
export declare class Role extends Resource implements IRole {
    /**
     * Imports an external role by ARN
     * @param scope construct scope
     * @param id construct id
     * @param roleArn the ARN of the role to import
     */
    static fromRoleArn(scope: Construct, id: string, roleArn: string): IRole;
    readonly grantPrincipal: IPrincipal;
    readonly assumeRoleAction: string;
    /**
     * The assume role policy document associated with this role.
     */
    readonly assumeRolePolicy?: PolicyDocument;
    /**
     * Returns the ARN of this role.
     */
    readonly roleArn: string;
    /**
     * Returns the stable and unique string identifying the role. For example,
     * AIDAJQABLZS4A3QDU576Q.
     *
     * @attribute
     */
    readonly roleId: string;
    /**
     * Returns the name of the role.
     */
    readonly roleName: string;
    /**
     * Returns the role.
     */
    readonly policyFragment: PrincipalPolicyFragment;
    private defaultPolicy?;
    private readonly managedPolicies;
    private readonly attachedPolicies;
    constructor(scope: Construct, id: string, props: RoleProps);
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param statement The permission statement to add to the policy document
     */
    addToPolicy(statement: PolicyStatement): boolean;
    /**
     * Attaches a managed policy to this role.
     * @param policy The the managed policy to attach.
     */
    addManagedPolicy(policy: IManagedPolicy): void;
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy: Policy): void;
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(grantee: IPrincipal, ...actions: string[]): Grant;
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity: IPrincipal): Grant;
}
/**
 * A Role object
 */
export interface IRole extends IIdentity {
    /**
     * Returns the ARN of this role.
     *
     * @attribute
     */
    readonly roleArn: string;
    /**
     * Returns the name of this role.
     *
     * @attribute
     */
    readonly roleName: string;
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(grantee: IPrincipal, ...actions: string[]): Grant;
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(grantee: IPrincipal): Grant;
}
