"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const iam_generated_1 = require("./iam.generated");
const policy_1 = require("./policy");
const principals_1 = require("./principals");
const util_1 = require("./util");
class User extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.userName,
        });
        this.grantPrincipal = this;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.groups = new Array();
        this.managedPolicies = new Array();
        this.attachedPolicies = new util_1.AttachedPolicies();
        const user = new iam_generated_1.CfnUser(this, 'Resource', {
            userName: this.physicalName,
            groups: util_1.undefinedIfEmpty(() => this.groups),
            managedPolicyArns: core_1.Lazy.listValue({ produce: () => this.managedPolicies.map(p => p.managedPolicyArn) }, { omitEmpty: true }),
            path: props.path,
            loginProfile: this.parseLoginProfile(props)
        });
        this.userName = this.getResourceNameAttribute(user.ref);
        this.userArn = this.getResourceArnAttribute(user.attrArn, {
            region: '',
            service: 'iam',
            resource: 'user',
            resourceName: this.physicalName,
        });
        this.policyFragment = new principals_1.ArnPrincipal(this.userArn).policyFragment;
        if (props.groups) {
            props.groups.forEach(g => this.addToGroup(g));
        }
    }
    /**
     * Adds this user to a group.
     */
    addToGroup(group) {
        this.groups.push(group.groupName);
    }
    /**
     * Attaches a managed policy to the user.
     * @param policy The managed policy to attach.
     */
    addManagedPolicy(policy) {
        this.managedPolicies.push(policy);
    }
    /**
     * Attaches a policy to this user.
     */
    attachInlinePolicy(policy) {
        this.attachedPolicies.attach(policy);
        policy.attachToUser(this);
    }
    /**
     * Adds an IAM statement to the default policy.
     *
     * @returns true
     */
    addToPolicy(statement) {
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.defaultPolicy.attachToUser(this);
        }
        this.defaultPolicy.addStatements(statement);
        return true;
    }
    parseLoginProfile(props) {
        if (props.password) {
            return {
                password: props.password.toString(),
                passwordResetRequired: props.passwordResetRequired
            };
        }
        if (props.passwordResetRequired) {
            throw new Error('Cannot set "passwordResetRequired" without specifying "initialPassword"');
        }
        return undefined; // no console access
    }
}
exports.User = User;
//# sourceMappingURL=data:application/json;base64,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