"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const MAX_POLICY_NAME_LEN = 128;
function undefinedIfEmpty(f) {
    return core_1.Lazy.listValue({ produce: () => {
            const array = f();
            return (array && array.length > 0) ? array : undefined;
        } });
}
exports.undefinedIfEmpty = undefinedIfEmpty;
/**
 * Used to generate a unique policy name based on the policy resource construct.
 * The logical ID of the resource is a great candidate as long as it doesn't exceed
 * 128 characters, so we take the last 128 characters (in order to make sure the hash
 * is there).
 */
function generatePolicyName(logicalId) {
    return logicalId.substring(Math.max(logicalId.length - MAX_POLICY_NAME_LEN, 0), logicalId.length);
}
exports.generatePolicyName = generatePolicyName;
/**
 * Helper class that maintains the set of attached policies for a principal.
 */
class AttachedPolicies {
    constructor() {
        this.policies = new Array();
    }
    /**
     * Adds a policy to the list of attached policies.
     *
     * If this policy is already, attached, returns false.
     * If there is another policy attached with the same name, throws an exception.
     */
    attach(policy) {
        if (this.policies.find(p => p === policy)) {
            return; // already attached
        }
        if (this.policies.find(p => p.policyName === policy.policyName)) {
            throw new Error(`A policy named "${policy.policyName}" is already attached`);
        }
        this.policies.push(policy);
    }
}
exports.AttachedPolicies = AttachedPolicies;
/**
 * Merge two dictionaries that represent IAM principals
 */
function mergePrincipal(target, source) {
    for (const key of Object.keys(source)) {
        target[key] = target[key] || [];
        const value = source[key];
        if (!Array.isArray(value)) {
            throw new Error(`Principal value must be an array (it will be normalized later): ${value}`);
        }
        target[key].push(...value);
    }
    return target;
}
exports.mergePrincipal = mergePrincipal;
//# sourceMappingURL=data:application/json;base64,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