# This file is part of RADKit / Lazy Maestro <radkit@cisco.com>
# Copyright (c) 2018-2025 by Cisco Systems, Inc.
# All rights reserved.

#
# DO NOT EDIT THIS FILE IN THE PACKAGE DIRECTORY
# PLEASE EDIT IT IN ./shared/ INSTEAD (from the source tree root)
# AND RUN ./ci/copy_shared.sh TO REPLICATE IT
#


from __future__ import annotations

import dataclasses
import json
import re
import textwrap
from datetime import datetime, timezone
from pathlib import Path
from typing import Any

from .version import version_str

# We can't import DEFAULT_FILE_ENCODING from radkit_common as some modules that
# share licensing.py may not depend on radkit_common. Define it locally instead.
DEFAULT_FILE_ENCODING = "utf-8"

# The license file, in the LEGAL sense (this is just handled in this module
# because there was no better place to put it; sorry if it is confusing)
_PATH_PREFIX = Path(__file__).parent
LICENSE_TEXT = _PATH_PREFIX / "LICENSE.md"
LICENSE_HTML = _PATH_PREFIX / "LICENSE.html"
LICENSE_JSON = _PATH_PREFIX / "licensing.meta.json"

SOURCE_BANNER = "CISCO CONFIDENTIAL -- FOR INTERNAL USE ONLY -- DO NOT DISTRIBUTE"
RELEASE_BANNER = "This RADKit release does not expire"

EXPIRED_ERROR = "This software is too old and has expired"
INVALID_ERROR = "The license for this software is invalid"
ERROR_MESSAGE = "; please download and install the latest version"


class LicenseError(Exception):
    pass


def _datetime_hook(raw_json: dict[Any, Any]) -> Any:
    pattern = re.compile(
        r"^(-?(?:[1-9][0-9]*)?[0-9]{4})-(1[0-2]|0[1-9])-(3[01]|0[1-9]|[12][0-9])T(2[0-3]|[01][0-9]):([0-5][0-9]):([0-5][0-9])(\.[0-9]+)?(Z|[+-](?:2[0-3]|[01][0-9]):[0-5][0-9])?$"
    )
    for k, v in raw_json.items():
        if isinstance(v, str) and pattern.match(v):
            raw_json[k] = datetime.fromisoformat(v)
    return raw_json


@dataclasses.dataclass
class _LicensingJSON:
    # None means: no expiration date
    expiration_date: datetime | None
    code: str

    @classmethod
    def read(cls) -> _LicensingJSON:
        with LICENSE_JSON.open("r") as json_in:
            raw_json = json.load(json_in, object_hook=_datetime_hook)
        return cls(**raw_json)


PYARMOR_RUNTIME_ERROR_MESSAGES: list[str] = [
    "this license key is expired",
    "this license key is not for this machine",
    "missing license key to run the script",
    "unauthorized use of script",
    "this Python version is not supported",
    "the script doesn’t work in this system",
    "the format of obfuscated script is incorrect",
    "the format of obfuscated function is incorrect",
]


@dataclasses.dataclass
class _PyarmorInfo:
    expiration_date: datetime | None


# This value is set only if the package is is obfuscated
_pyarmor_info: _PyarmorInfo | None = None

try:
    # If this import succeeds it means that the package is obfuscated
    # and the expiration date can be extracted from Pyarmor's runtime
    from .pyarmor_runtime_006005.pyarmor_runtime import __pyarmor__  # type:ignore

    # If the value returned by 'keyinfo' is -1, the package does not expire
    expiration_timestamp = __pyarmor__(1, None, b"keyinfo", 1)
    if expiration_timestamp < 0:
        expiration_date = None
    else:
        expiration_date = datetime.fromtimestamp(expiration_timestamp, tz=timezone.utc)
    _pyarmor_info = _PyarmorInfo(expiration_date=expiration_date)

except RuntimeError as exc:
    # Handling of RuntimeErrors raised by Pyarmor
    if PYARMOR_RUNTIME_ERROR_MESSAGES[0] in str(exc):
        error_text = f"{EXPIRED_ERROR}{ERROR_MESSAGE} (current: {version_str}, expired: {_LicensingJSON.read().expiration_date})"
        raise LicenseError(error_text)
    elif any(msg in str(exc) for msg in PYARMOR_RUNTIME_ERROR_MESSAGES[1:]):
        error_text = f"{INVALID_ERROR}{ERROR_MESSAGE} (current: {version_str})"
        raise LicenseError(error_text)
    # Other RuntimeErrors should be raised without changes
    else:
        raise
except ModuleNotFoundError:
    # The ModuleNotFoundError will be raised if the package is not-obfuscated
    # Do nothing and proceed with default values for the API.
    pass


def get_license_text() -> str:
    try:
        with LICENSE_TEXT.open("r", encoding=DEFAULT_FILE_ENCODING) as f:
            return "".join(f.readlines())
    except Exception:
        raise LicenseError(f"Error reading license text (legal): {LICENSE_TEXT}")


def get_license_html() -> str:
    try:
        with LICENSE_HTML.open("r", encoding=DEFAULT_FILE_ENCODING) as f:
            return "".join(f.readlines())
    except Exception:
        raise LicenseError(f"Error reading license HTML (legal): {LICENSE_HTML}")


def get_license_text_wrapped(width: int) -> str:
    """
    Return license text, but wrapped according to the given width.
    """
    # Split license text into paragraphs.
    paragraphs = re.split(r"\n\n+", get_license_text())

    # Wrap each paragraph individually.
    wrapped_paragraphs = [
        "\n".join(textwrap.wrap(paragraph, width=width, break_on_hyphens=False))
        for paragraph in paragraphs
    ]

    # Join paragraphs.
    return "\n\n".join(wrapped_paragraphs)


def get_days_left() -> int:
    if _pyarmor_info and _pyarmor_info.expiration_date is not None:
        return (_pyarmor_info.expiration_date - datetime.now(timezone.utc)).days
    return -1


def get_expiration() -> str:
    if _pyarmor_info:
        days_left = get_days_left()
        if days_left >= 0:
            return f"{_pyarmor_info.expiration_date} ({days_left} days left)"
    return "(unlimited)"


def get_banner() -> str:
    if _pyarmor_info:
        days_left = get_days_left()
        if days_left >= 0:
            return (
                f"This software will expire in {days_left} "
                f"day{'s' if days_left > 1 else ''}, after which it must be updated"
            )
        return RELEASE_BANNER
    return SOURCE_BANNER


def get_code() -> str:
    if _pyarmor_info:
        return _LicensingJSON.read().code
    return "(no license)"
