"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const diff_1 = require("./diff");
const external_1 = require("./external");
async function handler(event) {
    if (event.RequestType === 'Create') {
        return onCreate(event);
    }
    if (event.RequestType === 'Update') {
        return onUpdate(event);
    }
    if (event.RequestType === 'Delete') {
        return onDelete(event);
    }
    throw new Error('invalid request type');
}
exports.handler = handler;
async function onCreate(event) {
    var _a, _b;
    const issuerUrl = event.ResourceProperties.Url;
    const thumbprints = ((_a = event.ResourceProperties.ThumbprintList) !== null && _a !== void 0 ? _a : []).sort(); // keep sorted for UPDATE
    const clients = ((_b = event.ResourceProperties.ClientIDList) !== null && _b !== void 0 ? _b : []).sort();
    if (thumbprints.length === 0) {
        thumbprints.push(await external_1.external.downloadThumbprint(issuerUrl));
    }
    const resp = await external_1.external.createOpenIDConnectProvider({
        Url: issuerUrl,
        ClientIDList: clients,
        ThumbprintList: thumbprints,
    });
    return {
        PhysicalResourceId: resp.OpenIDConnectProviderArn,
    };
}
async function onUpdate(event) {
    var _a, _b;
    const issuerUrl = event.ResourceProperties.Url;
    const thumbprints = ((_a = event.ResourceProperties.ThumbprintList) !== null && _a !== void 0 ? _a : []).sort(); // keep sorted for UPDATE
    const clients = ((_b = event.ResourceProperties.ClientIDList) !== null && _b !== void 0 ? _b : []).sort();
    // determine which update we are talking about.
    const oldIssuerUrl = event.OldResourceProperties.Url;
    // if this is a URL update, then we basically create a new resource and cfn will delete the old one
    // since the physical resource ID will change.
    if (oldIssuerUrl !== issuerUrl) {
        return onCreate({ ...event, RequestType: 'Create' });
    }
    const providerArn = event.PhysicalResourceId;
    // if thumbprints changed, we can update in-place, but bear in mind that if the new thumbprint list
    // is empty, we will grab it from the server like we do in CREATE
    const oldThumbprints = (event.OldResourceProperties.ThumbprintList || []).sort();
    if (JSON.stringify(oldThumbprints) !== JSON.stringify(thumbprints)) {
        const thumbprintList = thumbprints.length > 0 ? thumbprints : [await external_1.external.downloadThumbprint(issuerUrl)];
        external_1.external.log('updating thumbprint list from', oldThumbprints, 'to', thumbprints);
        await external_1.external.updateOpenIDConnectProviderThumbprint({
            OpenIDConnectProviderArn: providerArn,
            ThumbprintList: thumbprintList,
        });
        // don't return, we might have more updates...
    }
    // if client ID list has changed, determine "diff" because the API is add/remove
    const oldClients = (event.OldResourceProperties.ClientIDList || []).sort();
    const diff = diff_1.arrayDiff(oldClients, clients);
    external_1.external.log(`client ID diff: ${JSON.stringify(diff)}`);
    for (const addClient of diff.adds) {
        external_1.external.log(`adding client id "${addClient}" to provider ${providerArn}`);
        await external_1.external.addClientIDToOpenIDConnectProvider({
            OpenIDConnectProviderArn: providerArn,
            ClientID: addClient,
        });
    }
    for (const deleteClient of diff.deletes) {
        external_1.external.log(`removing client id "${deleteClient}" from provider ${providerArn}`);
        await external_1.external.removeClientIDFromOpenIDConnectProvider({
            OpenIDConnectProviderArn: providerArn,
            ClientID: deleteClient,
        });
    }
    return;
}
async function onDelete(deleteEvent) {
    await external_1.external.deleteOpenIDConnectProvider({
        OpenIDConnectProviderArn: deleteEvent.PhysicalResourceId,
    });
}
//# sourceMappingURL=data:application/json;base64,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