"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const app_1 = require("../lib/app");
function withApp(props, block) {
    const app = new app_1.App(Object.assign({ runtimeInfo: false, stackTraces: false }, props));
    block(app);
    return app.synth();
}
function synth(context) {
    return withApp({ context }, app => {
        const stack1 = new lib_1.Stack(app, 'stack1', { env: { account: '12345', region: 'us-east-1' } });
        new lib_1.CfnResource(stack1, 's1c1', { type: 'DummyResource', properties: { Prop1: 'Prop1' } });
        const r2 = new lib_1.CfnResource(stack1, 's1c2', { type: 'DummyResource', properties: { Foo: 123 } });
        const stack2 = new lib_1.Stack(app, 'stack2');
        new lib_1.CfnResource(stack2, 's2c1', { type: 'DummyResource', properties: { Prog2: 'Prog2' } });
        const c1 = new MyConstruct(stack2, 's1c2');
        // add some metadata
        stack1.node.addMetadata('meta', 111);
        r2.node.addWarning('warning1');
        r2.node.addWarning('warning2');
        c1.node.addMetadata('meta', { key: 'value' });
        app.node.addMetadata('applevel', 123); // apps can also have metadata
    });
}
function synthStack(name, includeMetadata = false, context) {
    const response = synth(context);
    const stack = response.getStackByName(name);
    if (!includeMetadata) {
        delete stack.metadata;
    }
    return stack;
}
class MyConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        new lib_1.CfnResource(this, 'r1', { type: 'ResourceType1' });
        new lib_1.CfnResource(this, 'r2', { type: 'ResourceType2', properties: { FromContext: this.node.tryGetContext('ctx1') } });
    }
}
module.exports = {
    'synthesizes all stacks and returns synthesis result'(test) {
        const response = synth();
        delete response.dir;
        test.deepEqual(response.stacks.length, 2);
        const stack1 = response.stacks[0];
        test.deepEqual(stack1.stackName, 'stack1');
        test.deepEqual(stack1.id, 'stack1');
        test.deepEqual(stack1.environment.account, 12345);
        test.deepEqual(stack1.environment.region, 'us-east-1');
        test.deepEqual(stack1.environment.name, 'aws://12345/us-east-1');
        test.deepEqual(stack1.template, { Resources: { s1c1: { Type: 'DummyResource', Properties: { Prop1: 'Prop1' } },
                s1c2: { Type: 'DummyResource', Properties: { Foo: 123 } } } });
        test.deepEqual(stack1.manifest.metadata, {
            '/stack1': [{ type: 'meta', data: 111 }],
            '/stack1/s1c1': [{ type: 'aws:cdk:logicalId', data: 's1c1' }],
            '/stack1/s1c2': [{ type: 'aws:cdk:logicalId', data: 's1c2' },
                { type: 'aws:cdk:warning', data: 'warning1' },
                { type: 'aws:cdk:warning', data: 'warning2' }],
        });
        const stack2 = response.stacks[1];
        test.deepEqual(stack2.stackName, 'stack2');
        test.deepEqual(stack2.id, 'stack2');
        test.deepEqual(stack2.environment.name, 'aws://unknown-account/unknown-region');
        test.deepEqual(stack2.template, { Resources: { s2c1: { Type: 'DummyResource', Properties: { Prog2: 'Prog2' } },
                s1c2r1D1791C01: { Type: 'ResourceType1' },
                s1c2r25F685FFF: { Type: 'ResourceType2' } } });
        test.deepEqual(stack2.manifest.metadata, {
            '/stack2/s2c1': [{ type: 'aws:cdk:logicalId', data: 's2c1' }],
            '/stack2/s1c2': [{ type: 'meta', data: { key: 'value' } }],
            '/stack2/s1c2/r1': [{ type: 'aws:cdk:logicalId', data: 's1c2r1D1791C01' }],
            '/stack2/s1c2/r2': [{ type: 'aws:cdk:logicalId', data: 's1c2r25F685FFF' }],
        });
        test.done();
    },
    'context can be passed through CDK_CONTEXT'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2'
        });
        const prog = new app_1.App();
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context passed through CDK_CONTEXT has precedence'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2'
        });
        const prog = new app_1.App({
            context: {
                key1: 'val3',
                key2: 'val4'
            }
        });
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context from the command line can be used when creating the stack'(test) {
        const output = synthStack('stack2', false, { ctx1: 'HELLO' });
        test.deepEqual(output.template, {
            Resources: {
                s2c1: {
                    Type: "DummyResource",
                    Properties: {
                        Prog2: "Prog2"
                    }
                },
                s1c2r1D1791C01: {
                    Type: "ResourceType1"
                },
                s1c2r25F685FFF: {
                    Type: "ResourceType2",
                    Properties: {
                        FromContext: "HELLO"
                    }
                }
            }
        });
        test.done();
    },
    'setContext(k,v) can be used to set context programmatically'(test) {
        const prog = new app_1.App({
            context: {
                foo: 'bar'
            }
        });
        test.deepEqual(prog.node.tryGetContext('foo'), 'bar');
        test.done();
    },
    'setContext(k,v) cannot be called after stacks have been added because stacks may use the context'(test) {
        const prog = new app_1.App();
        new lib_1.Stack(prog, 's1');
        test.throws(() => prog.node.setContext('foo', 'bar'));
        test.done();
    },
    'app.synth() performs validation first and if there are errors, it returns the errors'(test) {
        class Child extends lib_1.Construct {
            validate() {
                return [`Error from ${this.node.id}`];
            }
        }
        class Parent extends lib_1.Stack {
        }
        const app = new app_1.App();
        const parent = new Parent(app, 'Parent');
        new Child(parent, 'C1');
        new Child(parent, 'C2');
        test.throws(() => app.synth(), /Validation failed with the following errors/);
        test.done();
    },
    'app.synthesizeStack(stack) will return a list of missing contextual information'(test) {
        class MyStack extends lib_1.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.reportMissingContext({
                    key: 'missing-context-key',
                    provider: 'fake',
                    props: {
                        account: '12345689012',
                        region: 'ab-north-1',
                    },
                });
                this.reportMissingContext({
                    key: 'missing-context-key-2',
                    provider: 'fake2',
                    props: {
                        foo: 'bar',
                        account: '12345689012',
                        region: 'ab-south-1',
                    },
                });
            }
        }
        const assembly = withApp({}, app => {
            new MyStack(app, 'MyStack');
        });
        test.deepEqual(assembly.manifest.missing, [
            {
                key: "missing-context-key",
                provider: 'fake',
                props: {
                    account: '12345689012',
                    region: 'ab-north-1',
                },
            },
            {
                key: "missing-context-key-2",
                provider: 'fake2',
                props: {
                    account: '12345689012',
                    region: 'ab-south-1',
                    foo: 'bar',
                },
            }
        ]);
        test.done();
    },
    'runtime library versions disabled'(test) {
        const context = {};
        context[cxapi.DISABLE_VERSION_REPORTING] = true;
        const assembly = withApp(context, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        test.deepEqual(assembly.runtime, { libraries: {} });
        test.done();
    },
    'runtime library versions'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        const version = require('../package.json').version;
        test.deepEqual(libs['@aws-cdk/core'], version);
        test.deepEqual(libs['@aws-cdk/cx-api'], version);
        test.deepEqual(libs['jsii-runtime'], `node.js/${process.version}`);
        test.done();
    },
    'jsii-runtime version loaded from JSII_AGENT'(test) {
        process.env.JSII_AGENT = 'Java/1.2.3.4';
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        test.deepEqual(libs['jsii-runtime'], `Java/1.2.3.4`);
        delete process.env.JSII_AGENT;
        test.done();
    },
    'version reporting includes only @aws-cdk, aws-cdk and jsii libraries'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        const version = require('../package.json').version;
        test.deepEqual(libs, {
            '@aws-cdk/core': version,
            '@aws-cdk/cx-api': version,
            'jsii-runtime': `node.js/${process.version}`
        });
        test.done();
    },
    'deep stack is shown and synthesized properly'(test) {
        // WHEN
        const response = withApp({}, (app) => {
            const topStack = new lib_1.Stack(app, 'Stack');
            const topResource = new lib_1.CfnResource(topStack, 'Res', { type: 'CDK::TopStack::Resource' });
            const bottomStack = new lib_1.Stack(topResource, 'Stack');
            new lib_1.CfnResource(bottomStack, 'Res', { type: 'CDK::BottomStack::Resource' });
        });
        // THEN
        test.deepEqual(response.stacks.map(s => ({ name: s.stackName, template: s.template })), [
            {
                name: 'Stack',
                template: { Resources: { Res: { Type: 'CDK::TopStack::Resource' } } },
            },
            {
                name: 'StackResStack7E4AFA86',
                template: { Resources: { Res: { Type: 'CDK::BottomStack::Resource' } } },
            },
        ]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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