"use strict";
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const os = require("os");
const path = require("path");
const cdk = require("../lib");
const lib_1 = require("../lib");
function createModernApp() {
    return new cdk.App({
        context: {
            [cxapi.DISABLE_VERSION_REPORTING]: 'true',
        }
    });
}
function list(outdir) {
    return fs.readdirSync(outdir).sort();
}
function readJson(outdir, file) {
    return JSON.parse(fs.readFileSync(path.join(outdir, file), 'utf-8'));
}
function writeJson(outdir, file, data) {
    fs.writeFileSync(path.join(outdir, file), JSON.stringify(data, undefined, 2));
}
module.exports = {
    'synthesis with an empty app'(test) {
        // GIVEN
        const app = createModernApp();
        // WHEN
        const session = app.synth();
        // THEN
        test.same(app.synth(), session); // same session if we synth() again
        test.deepEqual(list(session.directory), ['cdk.out', 'manifest.json', 'tree.json']);
        test.deepEqual(readJson(session.directory, 'manifest.json').artifacts, {
            Tree: {
                type: 'cdk:tree',
                properties: { file: 'tree.json' }
            }
        });
        test.deepEqual(readJson(session.directory, 'tree.json'), {
            version: 'tree-0.1',
            tree: {
                id: 'App',
                path: '',
                children: {
                    Tree: { id: 'Tree', path: 'Tree' }
                }
            }
        });
        test.done();
    },
    'synthesis respects disabling tree metadata'(test) {
        const app = new cdk.App({
            treeMetadata: false,
        });
        const assembly = app.synth();
        test.deepEqual(list(assembly.directory), ['cdk.out', 'manifest.json']);
        test.done();
    },
    'single empty stack'(test) {
        // GIVEN
        const app = createModernApp();
        new cdk.Stack(app, 'one-stack');
        // WHEN
        const session = app.synth();
        // THEN
        test.ok(list(session.directory).includes('one-stack.template.json'));
        test.done();
    },
    'some random construct implements "synthesize"'(test) {
        // GIVEN
        const app = createModernApp();
        const stack = new cdk.Stack(app, 'one-stack');
        class MyConstruct extends cdk.Construct {
            synthesize(s) {
                writeJson(s.assembly.outdir, 'foo.json', { bar: 123 });
                s.assembly.addArtifact('my-random-construct', {
                    type: cxapi.ArtifactType.AWS_CLOUDFORMATION_STACK,
                    environment: 'aws://12345/bar',
                    properties: {
                        templateFile: 'foo.json'
                    }
                });
            }
        }
        new MyConstruct(stack, 'MyConstruct');
        // WHEN
        const session = app.synth();
        // THEN
        test.ok(list(session.directory).includes('one-stack.template.json'));
        test.ok(list(session.directory).includes('foo.json'));
        test.deepEqual(readJson(session.directory, 'foo.json'), { bar: 123 });
        test.deepEqual(session.manifest, {
            version: cxapi.CLOUD_ASSEMBLY_VERSION,
            artifacts: {
                'Tree': {
                    type: 'cdk:tree',
                    properties: { file: 'tree.json' }
                },
                'my-random-construct': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://12345/bar',
                    properties: { templateFile: 'foo.json' }
                },
                'one-stack': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://unknown-account/unknown-region',
                    properties: { templateFile: 'one-stack.template.json' },
                }
            },
        });
        test.done();
    },
    'it should be possible to synthesize without an app'(test) {
        const calls = new Array();
        class SynthesizeMe extends lib_1.Construct {
            constructor() {
                super(undefined, 'id');
            }
            synthesize(session) {
                calls.push('synthesize');
                session.assembly.addArtifact('art', {
                    type: cxapi.ArtifactType.AWS_CLOUDFORMATION_STACK,
                    properties: {
                        templateFile: 'hey.json',
                        parameters: {
                            paramId: 'paramValue',
                            paramId2: 'paramValue2'
                        }
                    },
                    environment: 'aws://unknown-account/us-east-1'
                });
                writeJson(session.assembly.outdir, 'hey.json', { hello: 123 });
            }
            validate() {
                calls.push('validate');
                return [];
            }
            prepare() {
                calls.push('prepare');
            }
        }
        const root = new SynthesizeMe();
        const assembly = lib_1.ConstructNode.synth(root.node, { outdir: fs.mkdtempSync(path.join(os.tmpdir(), 'outdir')) });
        test.deepEqual(calls, ['prepare', 'validate', 'synthesize']);
        const stack = assembly.getStackByName('art');
        test.deepEqual(stack.template, { hello: 123 });
        test.deepEqual(stack.templateFile, 'hey.json');
        test.deepEqual(stack.parameters, { paramId: 'paramValue', paramId2: 'paramValue2' });
        test.deepEqual(stack.environment, { region: 'us-east-1', account: 'unknown-account', name: 'aws://unknown-account/us-east-1' });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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