"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToDynamoDB = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
class LambdaToDynamoDB extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToDynamoDBProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        // Other permissions for constructs are accepted as arrays, turning tablePermissions into
        // an array to use the same validation function.
        if (props.tablePermissions) {
            defaults.CheckListValues(['All', 'Read', 'ReadWrite', 'Write'], [props.tablePermissions], 'table permission');
        }
        if (props.deployVpc || props.existingVpc) {
            if (props.deployVpc && props.existingVpc) {
                throw new Error("More than 1 VPC specified in the properties");
            }
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.DYNAMODB);
        }
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc
        });
        // Since we are only invoking this function with an existing Table or tableProps,
        // (not a table interface), we know that the implementation will always return
        // a Table object and we can safely cast away the optional aspect of the type.
        const buildDynamoDBTableResponse = defaults.buildDynamoDBTable(this, {
            dynamoTableProps: props.dynamoTableProps,
            existingTableObj: props.existingTableObj
        });
        this.dynamoTable = buildDynamoDBTableResponse.tableObject;
        // Configure environment variables
        const tableEnvironmentVariableName = props.tableEnvironmentVariableName || 'DDB_TABLE_NAME';
        this.lambdaFunction.addEnvironment(tableEnvironmentVariableName, this.dynamoTable.tableName);
        // Add the requested or default table permissions
        if (props.tablePermissions) {
            const _permissions = props.tablePermissions.toUpperCase();
            if (_permissions === 'ALL') {
                this.dynamoTable.grantFullAccess(this.lambdaFunction.grantPrincipal);
            }
            else if (_permissions === 'READ') {
                this.dynamoTable.grantReadData(this.lambdaFunction.grantPrincipal);
            }
            else if (_permissions === 'READWRITE') {
                this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
            }
            else if (_permissions === 'WRITE') {
                this.dynamoTable.grantWriteData(this.lambdaFunction.grantPrincipal);
            }
        }
        else {
            this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
        }
        // Conditional metadata for cfn_nag
        if (props.dynamoTableProps?.billingMode === dynamodb.BillingMode.PROVISIONED) {
            defaults.addCfnSuppressRules(this.dynamoTable, [
                {
                    id: "W73",
                    reason: `PROVISIONED billing mode is a default and is not explicitly applied as a setting.`,
                },
            ]);
        }
    }
}
exports.LambdaToDynamoDB = LambdaToDynamoDB;
_a = JSII_RTTI_SYMBOL_1;
LambdaToDynamoDB[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-dynamodb.LambdaToDynamoDB", version: "2.40.0" };
//# sourceMappingURL=data:application/json;base64,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