"""
This package parse data generated by PWscf code stored in xml file
"""

from __future__ import annotations

__author__ = "Viet-Anh Ha"
__copyright__ = "Copyright 2024, EPWpy project"
__version__ = "1.0"
__maintainer__ = "Viet-Anh Ha"
__maintainer_email__="haviet.anh52@gmail.com"
__status__ = "Production"
__date__ = "April 24, 2024"


import os
import re
import warnings
import numpy as np
import numpy.linalg as nplin
from numpy.typing import ArrayLike
import xml.etree.ElementTree as ET
from monty.json import MSONable
from os.path import isdir, isfile, join

from EPWpy.utilities.constants import Consts, Units
from EPWpy.structure.structure import Structure

units = Units()
Bohr2Ang = units.Bohr2Ang
Ry2eV = units.Ry2eV
Ha2Ry = units.Ha2Ry
Ha2eV = units.Ha2eV
temp = re.compile("([a-zA-Z]+)([0-9]+)")


class PWscf_xml(MSONable):
    """
    This class parses the data stored in xml file generated in PWscf run
    **PWscf results**
    .. attribute:: ionic_steps
    .. attribute:: nionic_steps
    .. attribute:: force_constants
    .. attribute:: final_structure
    .. attribute:: final_energy
    .. attribute:: all_kpoints
    .. attribute:: efermi
    .. attribute:: eigenvalues
    .. attribute:: occupations

    **PWscf inputs**
    .. attribute:: control
    .. attribute:: system
    .. attribute:: electrons
    .. attribute:: ions
    .. attribute:: cell
    .. attribute:: kpoints
    .. attribute:: init_structure
    """

    def __init__(self, 
                 filename: str, 
                 get_ionic_steps: bool = False, 
                 occupation_tol: float =1e-6, 
                 bad_xml_warning: bool ="True") -> None:
        """
        Args: 
            filename: xml filename generated by PWscf code
            get_ionic_steps: get data for all ionic steps in relaxation
            occupation_tol: tolerance for occupation number in definition of Ef
            bad_xml_warning: print warning message in reading xml file
        """
        self.filename = filename
        self.get_ionic_steps = get_ionic_steps
        self.occupation_tol = occupation_tol
        self.bad_xml_warning = bad_xml_warning
        ionic_steps = []

        try:
            contents = ET.iterparse(self.filename)
            for event, elem in contents:
                tag = elem.tag
                if tag == "input":
                    self._parse_inputs(elem)
                elif tag == "output":
                    self._parse_outputs(elem)
                elif tag == "step" and self.get_ionic_steps:
                    ionic_steps.append(elem)
                elif tag == "exit_status":
                    self.exit_status = int( text_realtype(elem.text) )
                else:
                    pass
            del contents

        except ET.ParseError as ex:
            if self.bad_xml_warning:
                raise ex
            else:
                warnings.warn("XML is malformed. Parsing has stopped.", UserWarning)

        if len(ionic_steps) > 0:
            self._parse_ionic_step(ionic_steps)
        else:
            self.all_ionic_data = []     
        del ionic_steps

    def _parse_inputs(self, elem) -> None:
        """
        get input data in input section
        """
        self.control = {}
        control_vars = elem.find("control_variables")
        for selem in control_vars:
            self.control[selem.tag] = text_realtype(selem.text)

        self.electrons = {}
        electrons = elem.find("electron_control")
        for selem in electrons:
            self.electrons[selem.tag] = text_realtype(selem.text)

        self.ions = {}
        ions = elem.find("ion_control")
        for selem in ions:
            if selem.tag != "bfgs": self.ions[selem.tag] = text_realtype(selem.text)
         
        self.cell = {}
        cell = elem.find("cell_control")
        for selem in cell:
            self.cell[selem.tag] = text_realtype(selem.text)

        self.system = {}
        dft = elem.find("dft")
        spin = elem.find("spin")
        basis = elem.find("basis")
        bands = elem.find("bands")
        sym_flags = elem.find("symmetry_flags")
        self.system.update( dft_info(dft) )

        for selem in spin:
            self.system[selem.tag] = text_realtype(selem.text)

        for selem in basis:
            if selem.tag in ['ecutwfc', 'ecutrho']:
                self.system[selem.tag] = text_realtype(selem.text) * Ha2Ry
            else:
                self.system[selem.tag] = text_realtype(selem.text)

        for selem in bands:
            self.system[selem.tag] = text_realtype(selem.text)        
            if selem.tag == "smearing":
                attrib = selem.attrib
                for key in attrib.keys():
                    self.system[key] = attrib[key]

        for selem in sym_flags:
            self.system[selem.tag] = text_realtype(selem.text)

        self.atomic_species = {}
        atomic_species = elem.find("atomic_species")
        self.atomic_species["ntype"] = int(float(atomic_species.attrib["ntyp"]))
        all_species = atomic_species.findall("species")
        all_species_info = get_species_info(all_species)
        self.atomic_species.update(all_species_info)

        self.init_structure = get_structure(elem.find("atomic_structure"))
        self.init_structure_dict = get_structure(elem.find("atomic_structure"), struct_dict=True)

    def _parse_outputs(self, elem) -> None:
        """
        get output data in output section
        """
        # convergence information
        self.parse_convergence( elem.find("convergence_info") )
        # algorithmic info
        self.algorithm = self.parse_algorithm( elem.find("algorithmic_info") )
        # final_structure
        self.final_structure = get_structure( elem.find("atomic_structure") )
        self.final_structure_dict = get_structure( elem.find("atomic_structure"), struct_dict=True)
        # symmetries
        self.parse_symmetries( elem.find("symmetries") )
        # basis set fft & npwx
        self.parse_basis_set( elem.find("basis_set") )
        # final_magnetization
        self.parse_magnetization( elem.find("magnetization") )
        # total_energy and all components
        self.parse_energies( elem.find("total_energy") )
        # band_structure
        band_structure = elem.find("band_structure")
        if self.system["lsda"]:
            nbands = band_structure.find("nbnd_up")
        else:
            nbands = band_structure.find("nbnd")

        self.nbands = int(text_realtype(nbands.text))
        nelects = band_structure.find("nelec")
        self.nelects = text_realtype(nelects.text)

        try:
            fermi_energy = band_structure.find("fermi_energy")
            self.fermi_energy = text_realtype(fermi_energy.text) * Ha2eV
        except:
            try:
                highest_oc = band_structure.find("highestOccupiedLevel")
                self.fermi_energy = text_realtype(highest_oc.text) * Ha2eV + 1.e-3
            except:
                highest_oc = band_structure.find("two_fermi_energies")
                two_fermi_eng = text_realtype(highest_oc.text)
                self.fermi_energy = two_fermi_eng[0] * Ha2eV + 1.e-3

        nks = band_structure.find("nks")
        self.nkpoints = int(text_realtype(nks.text))

        inv_r_lttvec_trans = nplin.inv( self.rep_latt_vec.transpose() )
        all_ks_energies = band_structure.findall("ks_energies")
        self.band_energies, self.occupations, self.kpoints, self.kpt_weights = \
        self.parse_band_energy(all_ks_energies, self.nbands, inv_r_lttvec_trans, self.system["lsda"]) 
  
        if self.control["forces"]:
            self.force_atoms = self.parse_array( elem.find("forces") )
        else:
            self.force_atoms = None

        if self.control["stress"]:
            self.stress_matrix = self.parse_array( elem.find("stress") )
        else:
            self.stress_matrix = None

    def parse_convergence(self, xml_elem) -> None:
        """
        get info about convergence
        """
        self.opt_conv = {}
        if "relax" not in self.control["calculation"]:
            self.opt_conv["ion_move"] = True
        else:
            for selem in xml_elem.find("opt_conv"):
                tag = selem.tag
                self.opt_conv[tag] = text_realtype(selem.text)

        self.scf_conv = {}
        if self.control["calculation"] != "nscf":
            for selem in xml_elem.find("scf_conv"):
                tag = selem.tag
                self.scf_conv[tag] = text_realtype(selem.text)
        else:
            self.scf_conv["convergence_achieved"] = True

    def parse_array(self, xml_elem) -> ArrayLike:
        """
        get data in array format 
        """
        attrib = xml_elem.attrib
        dims = attrib["dims"]
        col = int( float(dims.split()[0]) )
        row = int( float(dims.split()[-1]) )
        array = np.array( text_realtype(xml_elem.text) )
        array = array.reshape( (row,col) )

        return array

    def parse_energies(self, xml_elem) -> None:
        """
        get total energy and components
        """
        self.energy_comps = {}
        for selem in xml_elem:
             if selem.tag == "etot":
                 self.total_energy = text_realtype(selem.text) * Ha2eV
             else:
                 self.energy_comps[selem.tag] = text_realtype(selem.text) * Ha2eV

    def parse_symmetries(self, xml_elem) -> None:
        """
        get all symmetry matrices
        """
        self.symmetries = {}
        self.symmetries["all_syms"] = []
        for selem in xml_elem:
            if selem.tag != "symmetry":
                self.symmetries[selem.tag] = text_realtype(selem.text)
            else:
                info = selem.find("info")
                info_attrib = info.attrib
                rotation = selem.find("rotation")
                rot_attrib = rotation.attrib
                matrix = self.parse_array( rotation )
                frac_trans = selem.find("fractional_translation")
                if frac_trans:
                    frac_trans = np.array( text_realtype(frac_trans.text) )
                else:
                    frac_trans = None
                save = {}
                save["info"] = info_attrib
                save["rotation"] = rot_attrib
                save["rot_matrix"] = matrix
                save["frac_trans"] = frac_trans
                self.symmetries["all_syms"].append( save )

    def parse_magnetization(self, xml_elem) -> None:
        """
        get magnetization if spin polarization is implemented
        """ 
        self.do_magnetization = None
        self.total_mag = None
        self.abs_mag = None
        self.site_mag = None
        for selem in xml_elem:
            if selem.tag == "lsda":
                self.lsda = text_realtype(selem.text)
            elif selem.tag == "noncolin":
                self.noncolin = text_realtype(selem.text)
            elif selem.tag == "spinorbit":
                self.spinorbit = text_realtype(selem.text)
            elif selem.tag == "total_vec":
                self.total_mag_vec = np.array(text_realtype(selem.text))
                self.total_mag = np.linalg.norm( self.total_mag_vec )
            elif selem.tag == "total":
                self.total_mag = text_realtype(selem.text)
            elif selem.tag == "absolute":
                self.abs_mag = text_realtype(selem.text)
            elif selem.tag == "Site_Magnetizations":
                self.site_mag = []
                for sselem in selem:
                    self.site_mag.append( np.array(text_realtype(sselem.text)) )
            elif selem.tag == "Scalar_Site_Magnetic_Moments":
                self.do_magnetization = text_realtype(selem.text)
                self.site_mag = []
                for sselem in selem:
                    self.site_mag.append( text_realtype(sselem.text) )
            elif selem.tag == "do_magnetization":
                self.do_magnetization = text_realtype(selem.text)
            else:
                pass

    def parse_algorithm(self, xml_elem) -> dict:
        """
        get algorithm info
        """
        algorithm = {}
        for selem in xml_elem:
            algorithm[selem.tag] = text_realtype(selem.text)

        return algorithm

    def parse_basis_set(self, xml_elem) -> None:
        """
        get basic parameters for Fourier transform
        """
        self.basis_set = {}
        for selem in xml_elem:
            if selem.tag != "reciprocal_lattice":
                if len(selem.attrib) == 0:
                    self.basis_set[selem.tag] = text_realtype(selem.text)
                else:
                    self.basis_set[selem.tag] = {}
                    for key in selem.attrib.keys():
                        self.basis_set[selem.tag][key] = text_realtype(selem.attrib[key])
            else:
                rep_latt_vec = self.parse_recp_lattice_vector(xml_elem.find("reciprocal_lattice"))

        self.rep_latt_vec = rep_latt_vec

    def parse_recp_lattice_vector(self, xml_elem) -> ArrayLike:
        """
        get repciprocal lattice vectors
        """
        b1 = xml_elem.find("b1")
        b2 = xml_elem.find("b2")
        b3 = xml_elem.find("b3")
        r_vec1 = np.array(text_realtype(b1.text))
        r_vec2 = np.array(text_realtype(b2.text))
        r_vec3 = np.array(text_realtype(b3.text))
        r_lattice_vec = np.array([r_vec1, r_vec2, r_vec3])

        return r_lattice_vec

    def parse_band_energy(self, xml_elem, nbands, inv_r_lttvec_trans, spin_polar=False):
        """
        get electronic band structures for all kpoints and bands
        """
        if spin_polar:
            band_energies = {"band_up": [], "band_dw": []}
            _occupations = {"band_up": [], "band_dw": []}
        else:
            band_energies = {"band_up": []}
            _occupations = {"band_up": []}

        kpoints = []
        kpt_weights = []
        for elem in xml_elem:
            k_point = elem.find("k_point")
            eigenvalues = elem.find("eigenvalues")
            occupations = elem.find("occupations")

            weight = float(k_point.attrib["weight"])
            kpt_weights.append(weight)
            Cart_coords = np.array(text_realtype(k_point.text))
            kpt_frac_coord = np.matmul(inv_r_lttvec_trans, Cart_coords)
            kpoints.append(kpt_frac_coord)

            energies = text_realtype(eigenvalues.text)
            occupations_ = text_realtype(occupations.text)
            if spin_polar:
                band_energies["band_up"].append(np.array(energies[:nbands]) * Ha2eV)
                band_energies["band_dw"].append(np.array(energies[nbands:]) * Ha2eV)
                _occupations["band_up"].append(np.array(occupations_[:nbands]))
                _occupations["band_dw"].append(np.array(occupations_[nbands:]))
            else:
                band_energies["band_up"].append(np.array(energies) * Ha2eV)
                _occupations["band_up"].append(np.array(occupations_))
       
        return band_energies, _occupations, kpoints, kpt_weights

    def _parse_ionic_step(self, list_ionic_steps) -> None:
        """
        get all ionic steps in relaxation
        """
        self.nionic_steps = len(list_ionic_steps)
        self.all_ionic_data = []
        for elem in list_ionic_steps:
            struct = get_structure(elem.find("atomic_structure"))

            scf_conv = {}
            for selem in elem.find("scf_conv"):
                tag = selem.tag
                scf_conv[tag] = text_realtype(selem.text)

            energy = {}
            for selem in elem.find("total_energy"):
                tag = selem.tag
                val = text_realtype(selem.text) * Ha2eV
                energy[tag] = val
              
            force_atom = self.parse_array( elem.find("forces") )
            self.all_ionic_data.append({"scf_conv": scf_conv, "structure": struct, \
                                        "energy": energy, "forces": force_atom})


def text_realtype(text):
    """Convert text to data in proper type
    """
    regex = re.compile("[0-9]")
    relet = re.compile("[a-zA-Z]|^&+\-%*/=!>.")

    if text is not None:
        sptext = text.split()
        data = []
        for elem in sptext:
            if "true" in elem.lower():
                data.append(True)
            elif "false" in elem.lower():
                data.append(False)
            elif len(relet.findall(elem)) == 1:
                if 'd' in elem and ('2d' not in elem.lower()):
                    elem = float(elem.replace("d", "e"))
                elif 'D' in elem and ('2d' not in elem.lower()):
                    elem = float(elem.replace("D", "e"))
                elif 'e' in elem or 'E' in elem:
                    elem = float(elem)
                else:
                    pass
                data.append(elem)
            elif len(relet.findall(elem)) == 0 and len(regex.findall(elem)) >= 1:
                data.append(float(elem))
            else:
                data.append(elem)

        if len(data) == 1:
            return data[0]
        else:
            return data
    else:
        return None
   

def get_structure(xml_elem, struct_dict=False) -> Structure | dict:
    """Return Structure object
    """
    attrib = xml_elem.attrib
    natom = int(float(attrib["nat"]))
    alat = float(attrib["alat"])

    cell = xml_elem.find("cell")
    a1 = cell.find("a1")
    a2 = cell.find("a2")
    a3 = cell.find("a3")
    vec1 = np.array( text_realtype(a1.text) )
    vec2 = np.array( text_realtype(a2.text) )
    vec3 = np.array( text_realtype(a3.text) )
    lattice_vec = np.array([vec1, vec2, vec3])

    atomic_pos = xml_elem.find("atomic_positions")
    all_atoms = atomic_pos.findall("atom")
    species = []
    if struct_dict: _species = []
    coordinates = []
    for atom in all_atoms:
        attrib = atom.attrib
        coords = np.array( text_realtype(atom.text) )
        species_ = attrib["name"]
        species__ = temp.match(species_)
        if struct_dict: _species.append(species_)
        if species__ is not None:
            species.append(species__.groups()[0])
        else:
            species.append(species_)
        coordinates.append(coords)

    t_lattice_vec = lattice_vec.transpose()
    invt_lattice_vec = np.linalg.inv(t_lattice_vec)
    frac_coords = []
    for cart_coord in coordinates:
        frac_coord =  np.matmul(invt_lattice_vec, cart_coord)
        frac_coords.append(frac_coord)

    if not struct_dict:
        struct = Structure(lattice_vec * Bohr2Ang, species, frac_coords)
    else:
        struct = {"lattice_vec": lattice_vec * Bohr2Ang, \
                  "species": _species, \
                  "frac_coords": frac_coords}

    return struct
    

def dft_info(dft) -> dict:
    """Return info in dft setting
    """
    dict_params = {}
    for selem in dft:
        stag = selem.tag
        if stag == "dftU":
            dict_params["lda_plus_u"] = True
            lda_plus_u_kind = selem.find("lda_plus_u_kind")
            U_projection_type = selem.find("U_projection_type")
            Hubbard_U = selem.findall("Hubbard_U")
            dict_params["lda_plus_u_kind"] = text_realtype(lda_plus_u_kind.text)
            dict_params["U_projection_type"] = text_realtype(U_projection_type.text)
            Hubbards = []
            for Hubbard in Hubbard_U:
                attrib = Hubbard.attrib
                U_value = text_realtype(Hubbard.text) * Ry2eV
                attrib["U_value"] = U_value
                Hubbards.append(attrib)
            dict_params["Hubbard_U"] = Hubbards
        else:
            dict_params[selem.tag] = text_realtype(selem.text)
     
    return dict_params


def get_species_info(species_xml) -> dict:
    """Return info about atoms
    """
    species = {"species": []}
    for elem in species_xml:
        name = elem.attrib["name"]
        mass = text_realtype(elem.find("mass").text)
        psp = elem.find("pseudo_file").text
        try:
            starting_mag = text_realtype(elem.find("starting_magnetization").text)
        except:
            starting_mag = 0.0
        setting = {"name": name, "mass": mass, "starting_mag": starting_mag, "pseudo_file": psp}
        species["species"].append(setting)

    return species

