import argparse
import collections
import contextlib
import csv
import functools
import itertools
import io
import os
import pathlib
import tempfile
import subprocess
import typing
import multiprocessing.pool

import anndata
import gb_io
import rich
import numpy
import pandas
import scipy.sparse
from scipy.cluster.hierarchy import fcluster

try:
    import argcomplete
except ImportError as err:
    argcomplete = err

try:
    from rich_argparse import RichHelpFormatter as HelpFormatter
except ImportError:
    from argparse import HelpFormatter

from . import __version__
from .seqio import extract_proteins, extract_sequences
from .mmseqs import MMSeqs, Database, Clustering
from .hca import manhattan, linkage


_PARAMS_NUC1 = dict(
    e_value=0.001,
    sequence_identity=0.85,
    coverage=1,
    cluster_mode=0,
    coverage_mode=1,
    spaced_kmer_mode=0,
)

_PARAMS_NUC2 = dict(
    e_value=0.001,
    sequence_identity=0.6,
    coverage=0.5,
    cluster_mode=0,
    coverage_mode=0,
    spaced_kmer_mode=0,
)

_PARAMS_PROT = dict(
    e_value=0.001,
    coverage=0.9,
    coverage_mode=1,
    sequence_identity=0.5,
)


def build_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(
        prog="igua",
        formatter_class=HelpFormatter,
        add_help=False,
        description=(
            "A method for content-agnostic high-throughput identification of "
            "Gene Cluster Families (GCFs) from gene clusters of genomic and "
            "metagenomic origin. "
        ),
    )
    parser.add_argument(
        "-h",
        "--help",
        help="Display this help message and exit.",
        action="help",
    )
    parser.add_argument(
        "-V",
        "--version",
        help="Display the program version and exit.",
        action="version",
        version=f"igua v{__version__}",
    )
    parser.add_argument(
        "-j",
        "--jobs",
        help="The number of threads to use in parallel sections.",
        type=int,
        default=os.cpu_count(),
        metavar="N",
    )

    group_input = parser.add_argument_group(
        'Input',
        'Mandatory input files required by the pipeline.'
    )
    group_input.add_argument(
        "-i",
        "--input",
        help="The GenBank files containing the clusters to process.",
        action="append",
        type=pathlib.Path,
        default=[],
        required=True,
    )

    group_output = parser.add_argument_group(
        'Output',
        'Output files generated by the pipeline.'
    )
    group_output.add_argument(
        "-o",
        "--output",
        help="The name of the ouput file to generate.",
        default=pathlib.Path("gcfs.tsv"),
        type=pathlib.Path,
    )
    group_output.add_argument(
        "-C",
        "--compositions",
        help="A file where to write compositional data for GCF representatives.",
        type=pathlib.Path,
    )
    group_output.add_argument(
        "-F",
        "--features",
        help="A file where to write protein cluster representatives in FASTA format.",
        type=pathlib.Path,
    )

    group_parameters = parser.add_argument_group(
        'Parameters',
        'General purpose parameters.'
    )
    group_parameters.add_argument(
        "-w", 
        "--workdir", 
        help="A folder to use for temporary data produced by MMseqs2.", 
    )
    
    group_parameters.add_argument(
        "--prefix", 
        help="The prefix for GCF identifiers generated by the pipeline.", 
        default="GCF",
    )

    group_clustering = parser.add_argument_group(
        'Clustering',
        'Parameters to control the hierarchical clustering.'
    )
    group_clustering.add_argument(
        "--no-clustering",
        help="Disable the protein-level clustering.",
        action="store_false",
        dest="clustering",
    )
    group_clustering.add_argument(
        "--clustering-method",
        help="The hierarchical method to use for protein-level clustering.",
        default="average",
        choices={
            "average",
            "single",
            "complete",
            "weighted",
            "centroid",
            "median",
            "ward"
        }
    )
    group_clustering.add_argument(
        "--clustering-distance",
        help="The distance threshold after which to stop merging clusters.",
        type=float,
        default=0.8,
    )
    group_clustering.add_argument(
        "--precision",
        help="The numerical precision to use for computing distances for hierarchical clustering.",
        default="double",
        choices={
            "half",
            "single",
            "double"
        }
    )
    return parser


def get_protein_representative(
    mmseqs: MMSeqs,
    input_path: pathlib.Path,
    output_prefix: pathlib.Path,
    fasta_path: pathlib.Path
) -> None:
    db = Database(mmseqs, input_path.with_suffix(".db"))
    result = Clustering(mmseqs, output_prefix.with_suffix(".db"), db)
    subdb = result.to_subdb(output_prefix.with_name(f"{output_prefix.name}_rep_seq.db"))
    subdb.to_fasta(fasta_path)


def make_compositions(
    progress: rich.progress.Progress,
    protein_clusters: pandas.DataFrame,
    representatives: typing.Dict[str, int],
    protein_representatives: typing.Dict[str, int],
    protein_sizes: typing.Dict[str, int],
) -> anndata.AnnData: #scipy.sparse.csr_matrix:
    compositions = scipy.sparse.dok_matrix(
        (len(representatives), len(protein_representatives)), dtype=numpy.int32
    )

    task = progress.add_task(description=f"[bold blue]{'Working':>9}[/]", total=len(protein_clusters))
    for row in progress.track(protein_clusters.itertuples(), task_id=task):
        cluster_index = representatives[row.cluster_id]
        prot_index = protein_representatives[row.protein_representative]
        compositions[cluster_index, prot_index] += protein_sizes[
            row.protein_representative
        ]
    progress.remove_task(task)

    sorted_representatives = sorted(representatives, key=representatives.__getitem__)
    sorted_protein_representatives = sorted(protein_representatives, key=protein_representatives.__getitem__)
    return anndata.AnnData(
        X=compositions.tocsr(),
        obs=pandas.DataFrame(index=pandas.Index(sorted_representatives, name="cluster_id")),
        var=pandas.DataFrame(
            index=pandas.Index(sorted_protein_representatives, name="protein_id"),
            data=dict(size=[protein_sizes[x] for x in sorted_protein_representatives]),
        )
    )


def compute_distances(
    progress: rich.progress.Progress,
    compositions: scipy.sparse.csr_matrix,
    jobs: typing.Optional[int],
    precision: str,
) -> numpy.ndarray:
    n = 0
    r = compositions.shape[0]
    # compute the number of amino acids per cluster
    clusters_aa = numpy.zeros(r, dtype=numpy.int32)
    clusters_aa[:] = compositions.sum(axis=1).A1
    # make sure the sparse matrix has sorted indices (necessary for 
    # the distance algorithm to work efficiently)
    if not compositions.has_sorted_indices:
        compositions.sort_indices()
    # compute manhattan distance on sparse matrix
    distance_vector = numpy.zeros(r*(r-1) // 2, dtype=precision)
    manhattan(
        compositions.data,
        compositions.indices,
        compositions.indptr,
        distance_vector,
        threads=jobs,
    )
    # ponderate by sum of amino-acid distance
    for i in range(r-1):
        l = r - (i+1)
        distance_vector[n:n+l] /= (clusters_aa[i+1:] + clusters_aa[i]).clip(min=1)
        n += l
    # check distances are in [0, 1]
    return numpy.clip(distance_vector, 0.0, 1.0, out=distance_vector)


def main(argv: typing.Optional[typing.List[str]] = None) -> int:
    # build parser and get arguments
    parser = build_parser()
    if not isinstance(argcomplete, ImportError):
        argcomplete.autocomplete(parser)
    args = parser.parse_args()

    with contextlib.ExitStack() as ctx:
        # open temporary folder
        if args.workdir is None:
            workdir = pathlib.Path(ctx.enter_context(tempfile.TemporaryDirectory()))
        else:
            workdir = pathlib.Path(args.workdir)
            workdir.mkdir(parents=True, exist_ok=True)

        # prepare progress bar
        progress = ctx.enter_context(
            rich.progress.Progress(
                "",
                rich.progress.SpinnerColumn(),
                *rich.progress.Progress.get_default_columns(),
            )
        )
        mmseqs = MMSeqs(progress=progress, threads=args.jobs, tempdir=workdir)

        # extract raw sequences
        clusters_fna = workdir.joinpath("clusters.fna")
        progress.console.print(f"[bold blue]{'Loading':>12}[/] input clusters")
        input_sequences = extract_sequences(progress, args.input, clusters_fna)
        progress.console.print(
            f"[bold green]{'Loaded':>12}[/] {len(input_sequences)} clusters to process"
        )

        # create initial sequence database
        progress.console.print(
            f"[bold blue]{'Starting':>12}[/] nucleotide deduplication step with [purple]mmseqs[/]"
        )
        db = Database.create(mmseqs, clusters_fna)
        step1 = db.cluster(workdir / "step1.db", **_PARAMS_NUC1)
        gcfs1 = step1.to_dataframe(columns=["fragment_representative", "cluster_id"]).sort_values("cluster_id")
        progress.console.print(
            f"[bold green]{'Reduced':>12}[/] {len(gcfs1)} clusters to {len(gcfs1.fragment_representative.unique())} complete representatives"
        )

        # cluster sequences
        progress.console.print(
            f"[bold blue]{'Starting':>12}[/] nucleotide clustering step with [purple]mmseqs[/]"
        )
        repdb = step1.to_subdb(workdir / "step1.rep_seq.db")
        step2 = repdb.cluster(workdir / "step2.db", **_PARAMS_NUC2)
        gcfs2 = step2.to_dataframe(columns=["nucleotide_representative", "fragment_representative"]).sort_values("fragment_representative")
        progress.console.print(
            f"[bold green]{'Reduced':>12}[/] {len(gcfs2)} clusters to {len(gcfs2.nucleotide_representative.unique())} nucleotide representatives"
        )

        # load representatives
        progress.console.print(
            f"[bold blue]{'Extracting':>12}[/] representative clusters"
        )
        representatives = {
            x: i
            for i, x in enumerate(sorted(gcfs2["nucleotide_representative"].unique()))
        }
        progress.console.print(
            f"[bold green]{'Found':>12}[/] {len(representatives)} nucleotide representative clusters"
        )

        if args.clustering and len(representatives) > 1:
            # extract proteins and record sizes
            proteins_faa = workdir.joinpath("proteins.faa")
            progress.console.print(
                f"[bold blue]{'Extracting':>12}[/] protein sequences from clusters"
            )
            protein_sizes = extract_proteins(
                progress, args.input, proteins_faa, representatives
            )

            # cluster proteins
            prot_db = Database.create(mmseqs, proteins_faa)
            prot_result = prot_db.cluster(workdir / "step3.db", **_PARAMS_PROT)
            prot_clusters = prot_result.to_dataframe(columns=["protein_representative", "protein_id"])

            # extract protein representatives
            prot_clusters["cluster_id"] = (
                prot_clusters["protein_id"].str.rsplit("_", n=1).str[0]
            )
            protein_representatives = {
                x: i
                for i, x in enumerate(
                    sorted(prot_clusters["protein_representative"].unique())
                )
            }
            progress.console.print(
                f"[bold green]{'Found':>12}[/] {len(protein_representatives)} protein representatives for {len(prot_clusters)} proteins"
            )

            # build weighted compositional array
            progress.console.print(
                f"[bold blue]{'Building':>12}[/] weighted compositional array"
            )
            compositions = make_compositions(
                progress, prot_clusters, representatives, protein_representatives, protein_sizes
            )

            # compute and ponderate distances
            progress.console.print(
                f"[bold blue]{'Computing':>12}[/] pairwise distance based on protein composition"
            )
            distance_vector = compute_distances(progress, compositions.X, args.jobs, args.precision)

            # run hierarchical clustering
            progress.console.print(
                f"[bold blue]{'Clustering':>12}[/] gene clusters using {args.clustering_method} linkage"
            )
            Z = linkage(distance_vector, method=args.clustering_method)
            flat = fcluster(Z, criterion="distance", t=args.clustering_distance)

            # build GCFs based on flat clustering
            gcfs3 = pandas.DataFrame(
                {
                    "gcf_id": [f"{args.prefix}{i:07}" for i in flat],
                    "nucleotide_representative": compositions.obs_names,
                }
            )
        else:
            sorted_representatives = sorted(representatives, key=representatives.__getitem__)
            gcfs3 = pandas.DataFrame(
                {
                    "gcf_id": [f"{args.prefix}{i+1:07}" for i in range(len(sorted_representatives))],
                    "nucleotide_representative": sorted_representatives,
                }
            )

        progress.console.print(
            f"[bold green]{'Built':>12}[/] {len(gcfs3.gcf_id.unique())} GCFs from {len(input_sequences)} initial clusters"
        )

        # extract protein representative using the largest cluster of each GCF
        gcf3_representatives = (
            pandas.merge(
                gcfs3,
                input_sequences["cluster_length"],
                left_on="nucleotide_representative",
                right_index=True
            )
            .sort_values("cluster_length")
            .drop_duplicates("gcf_id", keep="last")
            .set_index("gcf_id")
        )
        gcfs3 = pandas.merge(
            gcfs3,
            gcf3_representatives["nucleotide_representative"].rename("gcf_representative"),
            left_on="gcf_id",
            right_index=True,
        )

        # build final GCF table
        gcfs = pandas.merge(
            pandas.merge(
                pandas.merge(gcfs1, gcfs2, on="fragment_representative"),
                gcfs3,
                on="nucleotide_representative",
            ),
            input_sequences,
            left_on="cluster_id",
            right_index=True,
        )

        # save results
        gcfs.sort_values(["gcf_id", "cluster_length"], inplace=True)
        gcfs = gcfs[
            [
                "cluster_id",
                "cluster_length",
                "gcf_id",
                "gcf_representative",
                "nucleotide_representative",
                "fragment_representative",
                "filename"
            ]
        ]
        gcfs.to_csv(args.output, sep="\t", index=False)
        progress.console.print(
            f"[bold green]{'Saved':>12}[/] final GCFs table to {str(args.output)!r}"
        )

        # save compositions
        if args.compositions is not None:
            gcf_representatives = gcfs["gcf_representative"].unique()
            representatives_compositions = anndata.AnnData(
                X=compositions[gcf_representatives].X,
                var=compositions.var,
                obs=(
                    gcfs.set_index("cluster_id")
                        .loc[gcf_representatives, ["gcf_id", "gcf_representative"]]
                        .set_index("gcf_id")
                )
            )
            representatives_compositions.write(args.compositions)
            progress.console.print(
                f"[bold green]{'Saved':>12}[/] GCF compositions to {str(args.compositions)!r}"
            )

        # save protein features
        if args.features is not None:
            get_protein_representative(
                mmseqs,
                proteins_faa,
                workdir.joinpath("step3"),
                args.features,
            )
            progress.console.print(
                f"[bold green]{'Saved':>12}[/] protein features to {str(args.features)!r}"
            )


    return 0
