"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'default stream'(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 1
                    }
                }
            }
        });
        test.done();
    },
    'stream from attributes'(test) {
        const stack = new core_1.Stack();
        const s = lib_1.Stream.fromStreamAttributes(stack, 'MyStream', {
            streamArn: 'arn:aws:kinesis:region:account-id:stream/stream-name'
        });
        test.equals(s.streamArn, 'arn:aws:kinesis:region:account-id:stream/stream-name');
        test.done();
    },
    "uses explicit shard count"(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream', {
            shardCount: 2
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 2
                    }
                }
            }
        });
        test.done();
    },
    "uses explicit retention period"(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream', {
            retentionPeriodHours: 168
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 168,
                        "ShardCount": 1
                    }
                }
            }
        });
        test.done();
    },
    "retention period must be between 24 and 168 hours"(test) {
        test.throws({
            block: () => {
                new lib_1.Stream(new core_1.Stack(), 'MyStream', {
                    retentionPeriodHours: 169
                });
            },
            message: "retentionPeriodHours must be between 24 and 168 hours"
        });
        test.throws({
            block: () => {
                new lib_1.Stream(new core_1.Stack(), 'MyStream', {
                    retentionPeriodHours: 23
                });
            },
            message: "retentionPeriodHours must be between 24 and 168 hours"
        });
        test.done();
    },
    "auto-creates KMS key if encryption type is KMS but no key is provided"(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream', {
            encryption: lib_1.StreamEncryption.KMS
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStreamKey76F3300E": {
                    "Type": "AWS::KMS::Key",
                    "Properties": {
                        "Description": "Created by MyStream",
                        "KeyPolicy": {
                            "Statement": [
                                {
                                    "Action": [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        "Ref": "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        "Ref": "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain"
                },
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 1,
                        "StreamEncryption": {
                            "EncryptionType": "KMS",
                            "KeyId": {
                                "Fn::GetAtt": [
                                    "MyStreamKey76F3300E",
                                    "Arn"
                                ]
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    "uses explicit KMS key if encryption type is KMS and a key is provided"(test) {
        const stack = new core_1.Stack();
        const explicitKey = new kms.Key(stack, 'ExplicitKey', {
            description: `Explicit Key`
        });
        new lib_1.Stream(stack, 'MyStream', {
            encryption: lib_1.StreamEncryption.KMS,
            encryptionKey: explicitKey
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "ExplicitKey7DF42F37": {
                    "Type": "AWS::KMS::Key",
                    "Properties": {
                        "Description": "Explicit Key",
                        "KeyPolicy": {
                            "Statement": [
                                {
                                    "Action": [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        "Ref": "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        "Ref": "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain"
                },
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 1,
                        "StreamEncryption": {
                            "EncryptionType": "KMS",
                            "KeyId": {
                                "Fn::GetAtt": [
                                    "ExplicitKey7DF42F37",
                                    "Arn"
                                ]
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    "permissions": {
        "with encryption": {
            "grantRead creates and attaches a policy with read only access to Stream and EncryptionKey"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream', {
                    encryption: lib_1.StreamEncryption.KMS
                });
                const user = new iam.User(stack, "MyUser");
                stream.grantRead(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStreamKey76F3300E": {
                            "Type": "AWS::KMS::Key",
                            "Properties": {
                                "Description": "Created by MyStream",
                                "KeyPolicy": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kms:Create*",
                                                "kms:Describe*",
                                                "kms:Enable*",
                                                "kms:List*",
                                                "kms:Put*",
                                                "kms:Update*",
                                                "kms:Revoke*",
                                                "kms:Disable*",
                                                "kms:Get*",
                                                "kms:Delete*",
                                                "kms:ScheduleKeyDeletion",
                                                "kms:CancelKeyDeletion"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::Join": [
                                                        "",
                                                        [
                                                            "arn:",
                                                            {
                                                                "Ref": "AWS::Partition"
                                                            },
                                                            ":iam::",
                                                            {
                                                                "Ref": "AWS::AccountId"
                                                            },
                                                            ":root"
                                                        ]
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        },
                                        {
                                            "Action": "kms:Decrypt",
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::GetAtt": [
                                                        "MyUserDC45028B",
                                                        "Arn"
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                }
                            },
                            "DeletionPolicy": "Retain",
                            "UpdateReplacePolicy": "Retain"
                        },
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1,
                                "StreamEncryption": {
                                    "EncryptionType": "KMS",
                                    "KeyId": {
                                        "Fn::GetAtt": [
                                            "MyStreamKey76F3300E",
                                            "Arn"
                                        ]
                                    }
                                }
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        {
                                            "Action": "kms:Decrypt",
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStreamKey76F3300E",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "grantWrite creates and attaches a policy with write only access to Stream and EncryptionKey"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream', {
                    encryption: lib_1.StreamEncryption.KMS
                });
                const user = new iam.User(stack, "MyUser");
                stream.grantWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStreamKey76F3300E": {
                            "Type": "AWS::KMS::Key",
                            "Properties": {
                                "Description": "Created by MyStream",
                                "KeyPolicy": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kms:Create*",
                                                "kms:Describe*",
                                                "kms:Enable*",
                                                "kms:List*",
                                                "kms:Put*",
                                                "kms:Update*",
                                                "kms:Revoke*",
                                                "kms:Disable*",
                                                "kms:Get*",
                                                "kms:Delete*",
                                                "kms:ScheduleKeyDeletion",
                                                "kms:CancelKeyDeletion"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::Join": [
                                                        "",
                                                        [
                                                            "arn:",
                                                            {
                                                                "Ref": "AWS::Partition"
                                                            },
                                                            ":iam::",
                                                            {
                                                                "Ref": "AWS::AccountId"
                                                            },
                                                            ":root"
                                                        ]
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        },
                                        {
                                            "Action": [
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::GetAtt": [
                                                        "MyUserDC45028B",
                                                        "Arn"
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                }
                            },
                            "DeletionPolicy": "Retain",
                            "UpdateReplacePolicy": "Retain"
                        },
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1,
                                "StreamEncryption": {
                                    "EncryptionType": "KMS",
                                    "KeyId": {
                                        "Fn::GetAtt": [
                                            "MyStreamKey76F3300E",
                                            "Arn"
                                        ]
                                    }
                                }
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        {
                                            "Action": [
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*",
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStreamKey76F3300E",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "grantReadWrite creates and attaches a policy with access to Stream and EncryptionKey"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream', {
                    encryption: lib_1.StreamEncryption.KMS
                });
                const user = new iam.User(stack, "MyUser");
                stream.grantReadWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStreamKey76F3300E": {
                            "Type": "AWS::KMS::Key",
                            "Properties": {
                                "Description": "Created by MyStream",
                                "KeyPolicy": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kms:Create*",
                                                "kms:Describe*",
                                                "kms:Enable*",
                                                "kms:List*",
                                                "kms:Put*",
                                                "kms:Update*",
                                                "kms:Revoke*",
                                                "kms:Disable*",
                                                "kms:Get*",
                                                "kms:Delete*",
                                                "kms:ScheduleKeyDeletion",
                                                "kms:CancelKeyDeletion"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::Join": [
                                                        "",
                                                        [
                                                            "arn:",
                                                            {
                                                                "Ref": "AWS::Partition"
                                                            },
                                                            ":iam::",
                                                            {
                                                                "Ref": "AWS::AccountId"
                                                            },
                                                            ":root"
                                                        ]
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        },
                                        {
                                            "Action": [
                                                "kms:Decrypt",
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::GetAtt": [
                                                        "MyUserDC45028B",
                                                        "Arn"
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                }
                            },
                            "DeletionPolicy": "Retain",
                            "UpdateReplacePolicy": "Retain"
                        },
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1,
                                "StreamEncryption": {
                                    "EncryptionType": "KMS",
                                    "KeyId": {
                                        "Fn::GetAtt": [
                                            "MyStreamKey76F3300E",
                                            "Arn"
                                        ]
                                    }
                                }
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        {
                                            "Action": [
                                                "kms:Decrypt",
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStreamKey76F3300E",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            }
        },
        "with no encryption": {
            "grantRead creates and associates a policy with read only access to Stream"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream');
                const user = new iam.User(stack, "MyUser");
                stream.grantRead(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "grantWrite creates and attaches a policy with write only access to Stream"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream');
                const user = new iam.User(stack, "MyUser");
                stream.grantWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "greatReadWrite creates and attaches a policy with write only access to Stream"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream');
                const user = new iam.User(stack, "MyUser");
                stream.grantReadWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            }
        }
    },
    "cross-stack permissions": {
        "no encryption"(test) {
            const app = new core_1.App();
            const stackA = new core_1.Stack(app, 'stackA');
            const streamFromStackA = new lib_1.Stream(stackA, 'MyStream');
            const stackB = new core_1.Stack(app, 'stackB');
            const user = new iam.User(stackB, 'UserWhoNeedsAccess');
            streamFromStackA.grantRead(user);
            assert_1.expect(stackA).toMatch({
                "Resources": {
                    "MyStream5C050E93": {
                        "Type": "AWS::Kinesis::Stream",
                        "Properties": {
                            "RetentionPeriodHours": 24,
                            "ShardCount": 1
                        }
                    }
                },
                "Outputs": {
                    "ExportsOutputFnGetAttMyStream5C050E93Arn4ABF30CD": {
                        "Value": {
                            "Fn::GetAtt": [
                                "MyStream5C050E93",
                                "Arn"
                            ]
                        },
                        "Export": {
                            "Name": "stackA:ExportsOutputFnGetAttMyStream5C050E93Arn4ABF30CD"
                        }
                    }
                }
            });
            assert_1.expect(stackB).toMatch({
                "Resources": {
                    "UserWhoNeedsAccessF8959C3D": {
                        "Type": "AWS::IAM::User"
                    },
                    "UserWhoNeedsAccessDefaultPolicy6A9EB530": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "kinesis:DescribeStream",
                                            "kinesis:GetRecords",
                                            "kinesis:GetShardIterator"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": {
                                            "Fn::ImportValue": "stackA:ExportsOutputFnGetAttMyStream5C050E93Arn4ABF30CD"
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "UserWhoNeedsAccessDefaultPolicy6A9EB530",
                            "Users": [
                                {
                                    "Ref": "UserWhoNeedsAccessF8959C3D"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
        "fails with encryption due to cyclic dependency"(test) {
            const app = new core_1.App();
            const stackA = new core_1.Stack(app, 'stackA');
            const streamFromStackA = new lib_1.Stream(stackA, 'MyStream', {
                encryption: lib_1.StreamEncryption.KMS
            });
            const stackB = new core_1.Stack(app, 'stackB');
            const user = new iam.User(stackB, 'UserWhoNeedsAccess');
            streamFromStackA.grantRead(user);
            test.throws(() => app.synth(), /'stackB' depends on 'stackA'/);
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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