"""
CLI-First validation utilities.

This module provides validation logic to ensure artifacts are CLI-generated
and prevent direct manipulation of .specfact/ folder.
"""

from __future__ import annotations

import json
from dataclasses import dataclass
from pathlib import Path
from typing import Any

from beartype import beartype
from icontract import ensure, require
from rich.console import Console


console = Console()


@dataclass
class CLIArtifactMetadata:
    """Metadata for CLI-generated artifacts."""

    cli_generated: bool = True
    cli_version: str | None = None
    generated_at: str | None = None
    generated_by: str = "specfact-cli"

    def to_dict(self) -> dict[str, Any]:
        """Convert to dictionary."""
        return {
            "_cli_generated": self.cli_generated,
            "_cli_version": self.cli_version,
            "_generated_at": self.generated_at,
            "_generated_by": self.generated_by,
        }

    @classmethod
    def from_dict(cls, data: dict[str, Any]) -> CLIArtifactMetadata:
        """Create from dictionary."""
        return cls(
            cli_generated=data.get("_cli_generated", False),
            cli_version=data.get("_cli_version"),
            generated_at=data.get("_generated_at"),
            generated_by=data.get("_generated_by", "specfact-cli"),
        )


@beartype
@require(lambda artifact_path: isinstance(artifact_path, Path), "Artifact path must be Path")
@ensure(lambda result: isinstance(result, bool), "Must return bool")
def is_cli_generated(artifact_path: Path) -> bool:
    """
    Check if artifact was CLI-generated by looking for CLI metadata.

    Args:
        artifact_path: Path to artifact file

    Returns:
        True if artifact appears to be CLI-generated, False otherwise
    """
    if not artifact_path.exists():
        return False

    # Check for CLI metadata in YAML/JSON files
    if artifact_path.suffix in (".yaml", ".yml", ".json"):
        try:
            content = artifact_path.read_text(encoding="utf-8")
            # Check for CLI metadata markers
            if "_cli_generated" in content or "_generated_by" in content:
                return True
            # Check for CLI-generated patterns (timestamps, hashes, etc.)
            if "specfact" in content.lower() and ("generated" in content.lower() or "created" in content.lower()):
                return True
        except Exception:
            # If we can't read the file, assume it's not CLI-generated
            return False

    # For other file types, we can't easily detect CLI generation
    # Return True by default (assume CLI-generated unless proven otherwise)
    return True


@beartype
@require(lambda artifact_path: isinstance(artifact_path, Path), "Artifact path must be Path")
@ensure(lambda result: isinstance(result, CLIArtifactMetadata | None), "Must return CLIArtifactMetadata or None")
def extract_cli_metadata(artifact_path: Path) -> CLIArtifactMetadata | None:
    """
    Extract CLI metadata from artifact file.

    Args:
        artifact_path: Path to artifact file

    Returns:
        CLIArtifactMetadata if found, None otherwise
    """
    if not artifact_path.exists():
        return None

    # Try to extract metadata from YAML/JSON files
    if artifact_path.suffix in (".yaml", ".yml", ".json"):
        try:
            content = artifact_path.read_text(encoding="utf-8")
            if artifact_path.suffix == ".json":
                data = json.loads(content)
            else:
                # For YAML, we'd need to parse it, but for now just check for markers
                # In a full implementation, we'd use yaml.safe_load()
                if "_cli_generated" in content or "_generated_by" in content:
                    # Return a basic metadata object
                    return CLIArtifactMetadata(
                        cli_generated=True,
                        generated_by="specfact-cli",
                    )
                return None

            # Extract metadata from JSON
            if isinstance(data, dict) and ("_cli_generated" in data or "_generated_by" in data):
                return CLIArtifactMetadata.from_dict(data)

        except Exception:
            return None

    return None


@beartype
@require(lambda artifact_path: isinstance(artifact_path, Path), "Artifact path must be Path")
@ensure(lambda result: isinstance(result, bool), "Must return bool")
def validate_artifact_format(artifact_path: Path, expected_format: str = "yaml") -> bool:
    """
    Validate artifact format matches expected CLI-generated format.

    Args:
        artifact_path: Path to artifact file
        expected_format: Expected format (yaml, json)

    Returns:
        True if format is valid, False otherwise
    """
    if not artifact_path.exists():
        return False

    # Check file extension matches expected format
    if expected_format == "yaml" and artifact_path.suffix not in (".yaml", ".yml"):
        return False
    if expected_format == "json" and artifact_path.suffix != ".json":
        return False

    # Basic format validation (can be extended)
    try:
        content = artifact_path.read_text(encoding="utf-8")
        if expected_format == "json":
            json.loads(content)
        # For YAML, we'd validate structure, but for now just check it's not empty
        if not content.strip():
            return False
    except Exception:
        return False

    return True


@beartype
@require(lambda specfact_dir: isinstance(specfact_dir, Path), "SpecFact directory must be Path")
@ensure(lambda result: isinstance(result, list), "Must return list")
def detect_direct_manipulation(specfact_dir: Path) -> list[Path]:
    """
    Detect files in .specfact/ that may have been directly manipulated (not CLI-generated).

    This is a heuristic check - it looks for files that don't have CLI metadata.

    Args:
        specfact_dir: Path to .specfact/ directory

    Returns:
        List of file paths that may have been directly manipulated
    """
    suspicious_files: list[Path] = []

    if not specfact_dir.exists():
        return suspicious_files

    # Check project bundles
    projects_dir = specfact_dir / "projects"
    if projects_dir.exists():
        for bundle_dir in projects_dir.iterdir():
            if bundle_dir.is_dir():
                # Check bundle manifest
                manifest_file = bundle_dir / "bundle.manifest.yaml"
                if manifest_file.exists() and not is_cli_generated(manifest_file):
                    suspicious_files.append(manifest_file)

                # Check feature files
                features_dir = bundle_dir / "features"
                if features_dir.exists():
                    for feature_file in features_dir.glob("*.yaml"):
                        if not is_cli_generated(feature_file):
                            suspicious_files.append(feature_file)

    return suspicious_files
