import { Resource } from 'aws-cdk-lib';
import { IVpc, SubnetSelection, InstanceType, IMachineImage } from 'aws-cdk-lib/aws-ec2';
import { IRole } from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
/**
 * Properties for NAT instances.
 *
 * @stability stable
 */
export interface SimpleNATProps {
    /**
     * The VPC the NAT instances will reside.
     *
     * @stability stable
     */
    readonly vpc: IVpc;
    /**
     * The subnet selection for NAT instances, one NAT instance will be placed in the selected subnets.
     *
     * NOTE: must select the public subnet
     *
     * @default - subnetType is SubnetType.PUBLIC and onePerAZ is true.
     * @stability stable
     */
    readonly natSubnetsSelection?: SubnetSelection;
    /**
     * The subnet selection for updating route tables for selected subnets.
     *
     * @default - subnetType is SubnetType.PRIVATE_WITH_NAT.
     * @stability stable
     */
    readonly privateSubnetsSelection?: SubnetSelection;
    /**
     * The instance type of NAT instances.
     *
     * @default - t3.MICRO.
     * @stability stable
     */
    readonly instanceType?: InstanceType;
    /**
     * The AMI of NAT instances.
     *
     * @default - Amazon Linux 2 for x86_64.
     * @stability stable
     */
    readonly machineImage?: IMachineImage;
    /**
     * The key name of ssh key of NAT instances.
     *
     * @default - No SSH access will be possible.
     * @stability stable
     */
    readonly keyName?: string;
    /**
     * The custom script when provisioning the NAT instances.
     *
     * @default - no custom script.
     * @stability stable
     */
    readonly customScripts?: string;
    /**
     * The IAM role attached to NAT instances.
     *
     * @default - an IAM role is created.
     * @stability stable
     */
    readonly role?: IRole;
}
/**
 * Properties for how adding IPs to route.
 *
 * @stability stable
 */
export interface RouteProps {
    /**
     * If excluding IPv6 when creating route.
     *
     * @default - false
     * @stability stable
     */
    readonly excludeIPv6?: boolean;
}
/**
 * Simple NAT instaces construct.
 *
 * @stability stable
 */
export declare class SimpleNAT extends Resource {
    /**
     * @stability stable
     */
    static readonly Ipv6Regex = "^s*((([0-9A-Fa-f]{1,4}:){7}([0-9A-Fa-f]{1,4}|:))|(([0-9A-Fa-f]{1,4}:){6}(:[0-9A-Fa-f]{1,4}|((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3})|:))|(([0-9A-Fa-f]{1,4}:){5}(((:[0-9A-Fa-f]{1,4}){1,2})|:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3})|:))|(([0-9A-Fa-f]{1,4}:){4}(((:[0-9A-Fa-f]{1,4}){1,3})|((:[0-9A-Fa-f]{1,4})?:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){3}(((:[0-9A-Fa-f]{1,4}){1,4})|((:[0-9A-Fa-f]{1,4}){0,2}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){2}(((:[0-9A-Fa-f]{1,4}){1,5})|((:[0-9A-Fa-f]{1,4}){0,3}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){1}(((:[0-9A-Fa-f]{1,4}){1,6})|((:[0-9A-Fa-f]{1,4}){0,4}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(:(((:[0-9A-Fa-f]{1,4}){1,7})|((:[0-9A-Fa-f]{1,4}){0,5}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:)))(%.+)?s*(/([0-9]|[1-9][0-9]|1[0-1][0-9]|12[0-8]))?$";
    private gateways;
    private _securityGroup;
    private _routeMappingSubnets;
    private _routeTablesLimit;
    private readonly _defaultRoutesPerTable;
    private readonly _ipV6Regex;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: SimpleNATProps);
    /**
     * @stability stable
     */
    addV4Route(v4CIDR: string): SimpleNAT;
    /**
     * @stability stable
     */
    addV6Route(v6CIDR: string): SimpleNAT;
    /**
     * Add Github IPs to route table.
     *
     * @stability stable
     */
    withGithubRoute(props?: RouteProps): SimpleNAT;
    /**
     * Add Google IPs to route table.
     *
     * @stability stable
     */
    withGoogleRoute(props?: RouteProps): SimpleNAT;
    /**
     * Add Cloudflare IPs to route table.
     *
     * See https://www.cloudflare.com/ips/ for details
     *
     * @stability stable
     */
    withCloudflareRoute(props?: RouteProps): SimpleNAT;
    private _configureSubnet;
    private _addRoute;
}
