class Action:
    '''A collection of F-Curves for animation '''

    fcurves = None
    '''The individual F-Curves that make up the action 

    :type:  ActionFCurves bpy_prop_collection of FCurve, (readonly) 
    '''

    frame_range = None
    '''The final frame range of all F-Curves within this action 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0), (readonly) 
    '''

    groups = None
    '''Convenient groupings of F-Curves 

    :type:  ActionGroups bpy_prop_collection of ActionGroup, (readonly) 
    '''

    id_root = None
    '''Type of ID block that action can be used on - DO NOT CHANGE UNLESS YOU KNOW WHAT YOU ARE DOING 

    :type:  enum in [‘ACTION’, ‘ARMATURE’, ‘BRUSH’, ‘CAMERA’, ‘CACHEFILE’, ‘CURVE’, ‘FONT’, ‘GREASEPENCIL’, ‘GROUP’, ‘IMAGE’, ‘KEY’, ‘LAMP’, ‘LIBRARY’, ‘LINESTYLE’, ‘LATTICE’, ‘MASK’, ‘MATERIAL’, ‘META’, ‘MESH’, ‘MOVIECLIP’, ‘NODETREE’, ‘OBJECT’, ‘PAINTCURVE’, ‘PALETTE’, ‘PARTICLE’, ‘SCENE’, ‘SCREEN’, ‘SOUND’, ‘SPEAKER’, ‘TEXT’, ‘TEXTURE’, ‘WINDOWMANAGER’, ‘WORLD’], default ‘ACTION’ 
    '''

    pose_markers = None
    '''Markers specific to this action, for labeling poses 

    :type:  ActionPoseMarkers bpy_prop_collection of TimelineMarker, (readonly) 
    '''


class ActionActuator:
    '''Actuator to control the object movement '''

    action = None
    '''

    :type:  Action 
    '''

    apply_to_children = None
    '''Update Action on all children Objects as well 

    :type:  boolean, default False 
    '''

    blend_mode = None
    '''How this layer is blended with previous layers 

    :type:  enum in [‘BLEND’, ‘ADD’], default ‘BLEND’ 
    '''

    frame_blend_in = None
    '''Number of frames of motion blending 

    :type:  int in [0, 32767], default 0 
    '''

    frame_end = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    frame_property = None
    '''Assign the action’s current frame number to this property 

    :type:  string, default “”, (never None) 
    '''

    frame_start = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    layer = None
    '''The animation layer to play the action on 

    :type:  int in [0, 32766], default 0 
    '''

    layer_weight = None
    '''How much of the previous layer to blend into this one 

    :type:  float in [0, 1], default 0.0 
    '''

    play_mode = None
    '''Action playback type 

    :type:  enum in [‘PLAY’, ‘PINGPONG’, ‘FLIPPER’, ‘LOOPSTOP’, ‘LOOPEND’, ‘PROPERTY’], default ‘PLAY’ 
    '''

    priority = None
    '''Execution priority - lower numbers will override actions with higher numbers (with 2 or more actions at once, the overriding channels must be lower in the stack) 

    :type:  int in [0, 100], default 0 
    '''

    property = None
    '''Use this property to define the Action position 

    :type:  string, default “”, (never None) 
    '''

    use_additive = None
    '''Action is added to the current loc/rot/scale in global or local coordinate according to Local flag 

    :type:  boolean, default False 
    '''

    use_continue_last_frame = None
    '''Restore last frame when switching on/off, otherwise play from the start each time 

    :type:  boolean, default False 
    '''

    use_force = None
    '''Apply Action as a global or local force depending on the local option (dynamic objects only) 

    :type:  boolean, default False 
    '''

    use_local = None
    '''Let the Action act in local coordinates, used in Force and Add mode 

    :type:  boolean, default False 
    '''


class ActionConstraint:
    '''Map an action to the transform axes of a bone '''

    action = None
    '''The constraining action 

    :type:  Action 
    '''

    frame_end = None
    '''Last frame of the Action to use 

    :type:  int in [-500000, 500000], default 0 
    '''

    frame_start = None
    '''First frame of the Action to use 

    :type:  int in [-500000, 500000], default 0 
    '''

    max = None
    '''Maximum value for target channel range 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    min = None
    '''Minimum value for target channel range 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    transform_channel = None
    '''Transformation channel from the target that is used to key the Action 

    :type:  enum in [‘LOCATION_X’, ‘LOCATION_Y’, ‘LOCATION_Z’, ‘ROTATION_X’, ‘ROTATION_Y’, ‘ROTATION_Z’, ‘SCALE_X’, ‘SCALE_Y’, ‘SCALE_Z’], default ‘ROTATION_X’ 
    '''

    use_bone_object_action = None
    '''Bones only: apply the object’s transformation channels of the action to the constrained bone, instead of bone’s channels 

    :type:  boolean, default False 
    '''


class ActionFCurves:
    '''Collection of action F-Curves '''

    def new(self, data_path, index=0, action_group=""):
        '''Add an F-Curve to the action 

        :param data_path: Data Path, F-Curve data path to use 
        :type data_path: string, (never None)
        :param index: Index, Array index 
        :type index: int in [0, inf], (optional)
        :param action_group: Action Group, Acton group to add this F-Curve into 
        :type action_group: string, (optional, never None)
        :rtype: FCurve 
        :return:  Newly created F-Curve 
        '''
        pass

    def find(self, data_path, index=0):
        '''Find an F-Curve. Note that this function performs a linear scan of all F-Curves in the action. 

        :param data_path: Data Path, F-Curve data path 
        :type data_path: string, (never None)
        :param index: Index, Array index 
        :type index: int in [0, inf], (optional)
        :rtype: FCurve 
        :return:  The found F-Curve, or None if it doesn’t exist 
        '''
        pass

    def remove(self, fcurve):
        '''Remove action group 

        :param fcurve: F-Curve to remove 
        :type fcurve: FCurve, (never None)
        '''
        pass


class ActionGroup:
    '''Groups of F-Curves '''

    channels = None
    '''F-Curves in this group 

    :type:  bpy_prop_collection of FCurve, (readonly) 
    '''

    color_set = None
    '''Custom color set to use 

    :type:  enum in [‘DEFAULT’, ‘THEME01’, ‘THEME02’, ‘THEME03’, ‘THEME04’, ‘THEME05’, ‘THEME06’, ‘THEME07’, ‘THEME08’, ‘THEME09’, ‘THEME10’, ‘THEME11’, ‘THEME12’, ‘THEME13’, ‘THEME14’, ‘THEME15’, ‘THEME16’, ‘THEME17’, ‘THEME18’, ‘THEME19’, ‘THEME20’, ‘CUSTOM’], default ‘DEFAULT’ 
    '''

    colors = None
    '''Copy of the colors associated with the group’s color set 

    :type:  ThemeBoneColorSet, (readonly, never None) 
    '''

    is_custom_color_set = None
    '''Color set is user-defined instead of a fixed theme color set 

    :type:  boolean, default False, (readonly) 
    '''

    lock = None
    '''Action group is locked 

    :type:  boolean, default False 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''

    select = None
    '''Action group is selected 

    :type:  boolean, default False 
    '''

    show_expanded = None
    '''Action group is expanded 

    :type:  boolean, default False 
    '''


class ActionGroups:
    '''Collection of action groups '''

    def new(self, name):
        '''Create a new action group and add it to the action 

        :param name: New name for the action group 
        :type name: string, (never None)
        :rtype: ActionGroup 
        :return:  Newly created action group 
        '''
        pass

    def remove(self, action_group):
        '''Remove action group 

        :param action_group: Action group to remove 
        :type action_group: ActionGroup, (never None)
        '''
        pass


class ActionPoseMarkers:
    '''Collection of timeline markers '''

    active = None
    '''Active pose marker for this action 

    :type:  TimelineMarker 
    '''

    active_index = None
    '''Index of active pose marker 

    :type:  int in [0, inf], default 0 
    '''

    def new(self, name):
        '''Add a pose marker to the action 

        :param name: New name for the marker (not unique) 
        :type name: string, (never None)
        :rtype: TimelineMarker 
        :return:  Newly created marker 
        '''
        pass

    def remove(self, marker):
        '''Remove a timeline marker 

        :param marker: Timeline marker to remove 
        :type marker: TimelineMarker, (never None)
        '''
        pass


class Actuator:
    '''Actuator to apply actions in the game engine '''

    active = None
    '''Set the active state of the actuator 

    :type:  boolean, default False 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''

    pin = None
    '''Display when not linked to a visible states controller 

    :type:  boolean, default False 
    '''

    show_expanded = None
    '''Set actuator expanded in the user interface 

    :type:  boolean, default False 
    '''

    type = None
    '''

    :type:  enum in [‘ACTION’, ‘ARMATURE’, ‘CAMERA’, ‘CONSTRAINT’, ‘EDIT_OBJECT’, ‘FILTER_2D’, ‘GAME’, ‘MESSAGE’, ‘MOTION’, ‘MOUSE’, ‘PARENT’, ‘PROPERTY’, ‘RANDOM’, ‘SCENE’, ‘SOUND’, ‘STATE’, ‘STEERING’, ‘VISIBILITY’], default ‘MOTION’ 
    '''

    def link(self, controller):
        '''Link the actuator to a controller 

        :param controller: Controller to link to 
        :type controller: Controller
        '''
        pass

    def unlink(self, controller):
        '''Unlink the actuator from a controller 

        :param controller: Controller to unlink from 
        :type controller: Controller
        '''
        pass


class ActuatorSensor:
    '''Sensor to detect state modifications of actuators '''

    actuator = None
    '''Actuator name, actuator active state modifications will be detected 

    :type:  string, default “”, (never None) 
    '''


class AddSequence:
    '''Add Sequence '''

    input_1 = None
    '''First input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_2 = None
    '''Second input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''


class Addon:
    '''Python add-ons to be loaded automatically '''

    module = None
    '''Module name 

    :type:  string, default “”, (never None) 
    '''

    preferences = None
    '''

    :type:  AddonPreferences, (readonly) 
    '''


class AddonPreferences:
    bl_idname = None
    '''

    :type:  string, default “”, (never None) 
    '''


class Addons:
    '''Collection of add-ons '''

    pass


class AdjustmentSequence:
    '''Sequence strip to perform filter adjustments to layers below '''

    animation_offset_end = None
    '''Animation end offset (trim end) 

    :type:  int in [0, inf], default 0 
    '''

    animation_offset_start = None
    '''Animation start offset (trim start) 

    :type:  int in [0, inf], default 0 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''


class AlembicObjectPath:
    '''Path of an object inside of an Alembic archive '''

    path = None
    '''Object path 

    :type:  string, default “”, (never None) 
    '''


class AlembicObjectPaths:
    '''Collection of object paths '''

    pass


class AlphaOverSequence:
    '''Alpha Over Sequence '''

    input_1 = None
    '''First input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_2 = None
    '''Second input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''


class AlphaUnderSequence:
    '''Alpha Under Sequence '''

    input_1 = None
    '''First input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_2 = None
    '''Second input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''


class AlwaysSensor:
    '''Sensor to generate continuous pulses '''

    pass


class AndController:
    '''Controller passing on events based on a logical AND operation '''

    pass


class AnimData:
    '''Animation data for data-block '''

    action = None
    '''Active Action for this data-block 

    :type:  Action 
    '''

    action_blend_type = None
    '''Method used for combining Active Action’s result with result of NLA stack 

    :type:  enum in [‘REPLACE’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’], default ‘REPLACE’ 
    '''

    action_extrapolation = None
    '''Action to take for gaps past the Active Action’s range (when evaluating with NLA) 

    :type:  enum in [‘NOTHING’, ‘HOLD’, ‘HOLD_FORWARD’], default ‘HOLD’ 
    '''

    action_influence = None
    '''Amount the Active Action contributes to the result of the NLA stack 

    :type:  float in [0, 1], default 1.0 
    '''

    drivers = None
    '''The Drivers/Expressions for this data-block 

    :type:  AnimDataDrivers bpy_prop_collection of FCurve, (readonly) 
    '''

    nla_tracks = None
    '''NLA Tracks (i.e. Animation Layers) 

    :type:  NlaTracks bpy_prop_collection of NlaTrack, (readonly) 
    '''

    use_nla = None
    '''NLA stack is evaluated when evaluating this block 

    :type:  boolean, default False 
    '''

    use_tweak_mode = None
    '''Whether to enable or disable tweak mode in NLA 

    :type:  boolean, default False 
    '''


class AnimDataDrivers:
    '''Collection of Driver F-Curves '''

    def from_existing(self, src_driver=None):
        '''Add a new driver given an existing one 

        :param src_driver: Existing Driver F-Curve to use as template for a new one 
        :type src_driver: FCurve, (optional)
        :rtype: FCurve 
        :return:  New Driver F-Curve 
        '''
        pass

    def find(self, data_path, index=0):
        '''Find a driver F-Curve. Note that this function performs a linear scan of all driver F-Curves. 

        :param data_path: Data Path, F-Curve data path 
        :type data_path: string, (never None)
        :param index: Index, Array index 
        :type index: int in [0, inf], (optional)
        :rtype: FCurve 
        :return:  The found F-Curve, or None if it doesn’t exist 
        '''
        pass


class AnimViz:
    '''Settings for the visualization of motion '''

    motion_path = None
    '''Motion Path settings for visualization 

    :type:  AnimVizMotionPaths, (readonly, never None) 
    '''

    onion_skin_frames = None
    '''Onion Skinning (ghosting) settings for visualization 

    :type:  AnimVizOnionSkinning, (readonly, never None) 
    '''


class AnimVizMotionPaths:
    '''Motion Path settings for animation visualization '''

    bake_location = None
    '''When calculating Bone Paths, use Head or Tips 

    :type:  enum in [‘HEADS’, ‘TAILS’], default ‘TAILS’ 
    '''

    frame_after = None
    '''Number of frames to show after the current frame (only for ‘Around Current Frame’ Onion-skinning method) 

    :type:  int in [1, 250000], default 0 
    '''

    frame_before = None
    '''Number of frames to show before the current frame (only for ‘Around Current Frame’ Onion-skinning method) 

    :type:  int in [1, 250000], default 0 
    '''

    frame_end = None
    '''End frame of range of paths to display/calculate (not for ‘Around Current Frame’ Onion-skinning method) 

    :type:  int in [-inf, inf], default 0 
    '''

    frame_start = None
    '''Starting frame of range of paths to display/calculate (not for ‘Around Current Frame’ Onion-skinning method) 

    :type:  int in [-inf, inf], default 0 
    '''

    frame_step = None
    '''Number of frames between paths shown (not for ‘On Keyframes’ Onion-skinning method) 

    :type:  int in [1, 100], default 0 
    '''

    show_frame_numbers = None
    '''Show frame numbers on Motion Paths 

    :type:  boolean, default False 
    '''

    show_keyframe_action_all = None
    '''For bone motion paths, search whole Action for keyframes instead of in group with matching name only (is slower) 

    :type:  boolean, default False 
    '''

    show_keyframe_highlight = None
    '''Emphasize position of keyframes on Motion Paths 

    :type:  boolean, default False 
    '''

    show_keyframe_numbers = None
    '''Show frame numbers of Keyframes on Motion Paths 

    :type:  boolean, default False 
    '''

    type = None
    '''Type of range to show for Motion Paths 

    :type:  enum in [‘CURRENT_FRAME’, ‘RANGE’], default ‘RANGE’ 
    '''


class AnimVizOnionSkinning:
    '''Onion Skinning settings for animation visualization '''

    frame_after = None
    '''Number of frames to show after the current frame (only for ‘Around Current Frame’ Onion-skinning method) 

    :type:  int in [0, 30], default 0 
    '''

    frame_before = None
    '''Number of frames to show before the current frame (only for ‘Around Current Frame’ Onion-skinning method) 

    :type:  int in [0, 30], default 0 
    '''

    frame_end = None
    '''End frame of range of Ghosts to display (not for ‘Around Current Frame’ Onion-skinning method) 

    :type:  int in [-inf, inf], default 0 
    '''

    frame_start = None
    '''Starting frame of range of Ghosts to display (not for ‘Around Current Frame’ Onion-skinning method) 

    :type:  int in [-inf, inf], default 0 
    '''

    frame_step = None
    '''Number of frames between ghosts shown (not for ‘On Keyframes’ Onion-skinning method) 

    :type:  int in [1, 20], default 0 
    '''

    show_only_selected = None
    '''For Pose-Mode drawing, only draw ghosts for selected bones 

    :type:  boolean, default False 
    '''

    type = None
    '''Method used for determining what ghosts get drawn 

    :type:  enum in [‘NONE’, ‘CURRENT_FRAME’, ‘RANGE’, ‘KEYS’], default ‘NONE’ 
    '''


class AnyType:
    '''RNA type used for pointers to any possible data '''

    pass


class Area:
    '''Area in a subdivided screen, containing an editor '''

    height = None
    '''Area height 

    :type:  int in [0, 32767], default 0, (readonly) 
    '''

    regions = None
    '''Regions this area is subdivided in 

    :type:  bpy_prop_collection of Region, (readonly) 
    '''

    show_menus = None
    '''Show menus in the header 

    :type:  boolean, default False 
    '''

    spaces = None
    '''Spaces contained in this area, the first being the active space (NOTE: Useful for example to restore a previously used 3D view space in a certain area to get the old view orientation) 

    :type:  AreaSpaces bpy_prop_collection of Space, (readonly) 
    '''

    type = None
    '''Current editor type for this area 

    :type:  enum in [‘EMPTY’, ‘VIEW_3D’, ‘TIMELINE’, ‘GRAPH_EDITOR’, ‘DOPESHEET_EDITOR’, ‘NLA_EDITOR’, ‘IMAGE_EDITOR’, ‘SEQUENCE_EDITOR’, ‘CLIP_EDITOR’, ‘TEXT_EDITOR’, ‘NODE_EDITOR’, ‘LOGIC_EDITOR’, ‘PROPERTIES’, ‘OUTLINER’, ‘USER_PREFERENCES’, ‘INFO’, ‘FILE_BROWSER’, ‘CONSOLE’], default ‘VIEW_3D’ 
    '''

    width = None
    '''Area width 

    :type:  int in [0, 32767], default 0, (readonly) 
    '''

    x = None
    '''The window relative vertical location of the area 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    y = None
    '''The window relative horizontal location of the area 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    def tag_redraw(self):
        '''tag_redraw 

        '''
        pass

    def header_text_set(self, text=""):
        '''Set the header text 

        :param text: Text, New string for the header, no argument clears the text 
        :type text: string, (optional, never None)
        '''
        pass


class AreaLamp:
    '''Directional area lamp '''

    compression_threshold = None
    '''Deep shadow map compression threshold 

    :type:  float in [0, 1], default 0.0 
    '''

    gamma = None
    '''Light gamma correction value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    ge_shadow_buffer_type = None
    '''The shadow mapping algorithm used 

    :type:  enum in [‘SIMPLE’, ‘VARIANCE’], default ‘SIMPLE’ 
    '''

    shadow_adaptive_threshold = None
    '''Threshold for Adaptive Sampling (Raytraced shadows) 

    :type:  float in [0, 1], default 0.0 
    '''

    shadow_buffer_bias = None
    '''Shadow buffer sampling bias 

    :type:  float in [0.001, 5], default 0.0 
    '''

    shadow_buffer_bleed_bias = None
    '''Bias for reducing light-bleed on variance shadow maps 

    :type:  float in [0, 1], default 0.0 
    '''

    shadow_buffer_clip_end = None
    '''Shadow map clip end, beyond which objects will not generate shadows 

    :type:  float in [0, 9999], default 0.0 
    '''

    shadow_buffer_clip_start = None
    '''Shadow map clip start, below which objects will not generate shadows 

    :type:  float in [0, 9999], default 0.0 
    '''

    shadow_buffer_samples = None
    '''Number of shadow buffer samples 

    :type:  int in [1, 16], default 0 
    '''

    shadow_buffer_size = None
    '''Resolution of the shadow buffer, higher values give crisper shadows but use more memory 

    :type:  int in [128, 10240], default 0 
    '''

    shadow_buffer_soft = None
    '''Size of shadow buffer sampling area 

    :type:  float in [0, 100], default 0.0 
    '''

    shadow_buffer_type = None
    '''Type of shadow buffer 

    :type:  enum in [‘REGULAR’, ‘HALFWAY’, ‘IRREGULAR’, ‘DEEP’], default ‘REGULAR’ 
    '''

    shadow_color = None
    '''Color of shadows cast by the lamp 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    shadow_filter_type = None
    '''Type of shadow filter (Buffer Shadows) 

    :type:  enum in [‘BOX’, ‘TENT’, ‘GAUSS’], default ‘BOX’ 
    '''

    shadow_method = None
    '''

    :type:  enum in [‘NOSHADOW’, ‘RAY_SHADOW’], default ‘NOSHADOW’ 
    '''

    shadow_ray_sample_method = None
    '''Method for generating shadow samples: Adaptive QMC is fastest, Constant QMC is less noisy but slower 

    :type:  enum in [‘ADAPTIVE_QMC’, ‘CONSTANT_QMC’, ‘CONSTANT_JITTERED’], default ‘CONSTANT_JITTERED’ 
    '''

    shadow_ray_samples_x = None
    '''Number of samples taken extra (samples x samples) 

    :type:  int in [1, 64], default 0 
    '''

    shadow_ray_samples_y = None
    '''Number of samples taken extra (samples x samples) 

    :type:  int in [1, 64], default 0 
    '''

    shadow_sample_buffers = None
    '''Number of shadow buffers to render for better AA, this increases memory usage 

    :type:  enum in [‘BUFFERS_1’, ‘BUFFERS_4’, ‘BUFFERS_9’], default ‘BUFFERS_1’ 
    '''

    shadow_soft_size = None
    '''Light size for ray shadow sampling (Raytraced shadows) 

    :type:  float in [0, inf], default 0.0 
    '''

    shape = None
    '''Shape of the area lamp 

    :type:  enum in [‘SQUARE’, ‘RECTANGLE’], default ‘SQUARE’ 
    '''

    size = None
    '''Size of the area of the area Lamp, X direction size for Rectangle shapes 

    :type:  float in [0, inf], default 0.0 
    '''

    size_y = None
    '''Size of the area of the area Lamp in the Y direction for Rectangle shapes 

    :type:  float in [0, inf], default 0.0 
    '''

    use_auto_clip_end = None
    '''Automatic calculation of clipping-end, based on visible vertices 

    :type:  boolean, default False 
    '''

    use_auto_clip_start = None
    '''Automatic calculation of clipping-start, based on visible vertices 

    :type:  boolean, default False 
    '''

    use_dither = None
    '''Use 2x2 dithering for sampling (Constant Jittered sampling) 

    :type:  boolean, default False 
    '''

    use_jitter = None
    '''Use noise for sampling (Constant Jittered sampling) 

    :type:  boolean, default False 
    '''

    use_only_shadow = None
    '''Cast shadows only, without illuminating objects 

    :type:  boolean, default False 
    '''

    use_shadow = None
    '''

    :type:  boolean, default False 
    '''

    use_shadow_layer = None
    '''Objects on the same layers only cast shadows 

    :type:  boolean, default False 
    '''

    use_umbra = None
    '''Emphasize parts that are fully shadowed (Constant Jittered sampling) 

    :type:  boolean, default False 
    '''


class AreaSpaces:
    '''Collection of spaces '''

    active = None
    '''Space currently being displayed in this area 

    :type:  Space, (readonly) 
    '''


class Armature:
    '''Armature data-block containing a hierarchy of bones, usually used for rigging characters '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    bones = None
    '''

    :type:  ArmatureBones bpy_prop_collection of Bone, (readonly) 
    '''

    deform_method = None
    '''Vertex Deformer Method (Game Engine only) 

    :type:  enum in [‘BLENDER’, ‘BGE_CPU’], default ‘BLENDER’ 
    '''

    draw_type = None
    '''

    :type:  enum in [‘OCTAHEDRAL’, ‘STICK’, ‘BBONE’, ‘ENVELOPE’, ‘WIRE’], default ‘OCTAHEDRAL’ 
    '''

    edit_bones = None
    '''

    :type:  ArmatureEditBones bpy_prop_collection of EditBone, (readonly) 
    '''

    ghost_frame_end = None
    '''End frame of range of Ghosts to display (not for ‘Around Current Frame’ Onion-skinning method) 

    :type:  int in [-inf, inf], default 0 
    '''

    ghost_frame_start = None
    '''Starting frame of range of Ghosts to display (not for ‘Around Current Frame’ Onion-skinning method) 

    :type:  int in [-inf, inf], default 0 
    '''

    ghost_size = None
    '''Frame step for Ghosts (not for ‘On Keyframes’ Onion-skinning method) 

    :type:  int in [1, 20], default 0 
    '''

    ghost_step = None
    '''Number of frame steps on either side of current frame to show as ghosts (only for ‘Around Current Frame’ Onion-skinning method) 

    :type:  int in [0, 30], default 0 
    '''

    ghost_type = None
    '''Method of Onion-skinning for active Action 

    :type:  enum in [‘CURRENT_FRAME’, ‘RANGE’, ‘KEYS’], default ‘CURRENT_FRAME’ 
    '''

    is_editmode = None
    '''True when used in editmode 

    :type:  boolean, default False, (readonly) 
    '''

    layers = None
    '''Armature layer visibility 

    :type:  boolean array of 32 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    layers_protected = None
    '''Protected layers in Proxy Instances are restored to Proxy settings on file reload and undo 

    :type:  boolean array of 32 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    pose_position = None
    '''Show armature in binding pose or final posed state 

    :type:  enum in [‘POSE’, ‘REST’], default ‘POSE’ 
    '''

    show_axes = None
    '''Draw bone axes 

    :type:  boolean, default False 
    '''

    show_bone_custom_shapes = None
    '''Draw bones with their custom shapes 

    :type:  boolean, default False 
    '''

    show_group_colors = None
    '''Draw bone group colors 

    :type:  boolean, default False 
    '''

    show_names = None
    '''Draw bone names 

    :type:  boolean, default False 
    '''

    show_only_ghost_selected = None
    '''

    :type:  boolean, default False 
    '''

    use_auto_ik = None
    '''Add temporary IK constraints while grabbing bones in Pose Mode 

    :type:  boolean, default False 
    '''

    use_deform_delay = None
    '''Don’t deform children when manipulating bones in Pose Mode 

    :type:  boolean, default False 
    '''

    use_mirror_x = None
    '''Apply changes to matching bone on opposite side of X-Axis 

    :type:  boolean, default False 
    '''

    def transform(self, matrix):
        '''Transform armature bones by a matrix 

        :param matrix: Matrix 
        :type matrix: float array of 16 items in [-inf, inf]
        '''
        pass


class ArmatureActuator:
    bone = None
    '''Bone on which the constraint is defined 

    :type:  string, default “”, (never None) 
    '''

    constraint = None
    '''Name of the constraint to control 

    :type:  string, default “”, (never None) 
    '''

    influence = None
    '''Influence of this constraint 

    :type:  float in [0, 1], default 0.0 
    '''

    mode = None
    '''

    :type:  enum in [‘RUN’, ‘ENABLE’, ‘DISABLE’, ‘SETTARGET’, ‘SETWEIGHT’, ‘SETINFLUENCE’], default ‘RUN’ 
    '''

    secondary_target = None
    '''Set this object as the secondary target of the constraint (only IK polar target at the moment) 

    :type:  Object 
    '''

    target = None
    '''Set this object as the target of the constraint 

    :type:  Object 
    '''

    weight = None
    '''Weight of this constraint 

    :type:  float in [0, 1], default 0.0 
    '''


class ArmatureBones:
    '''Collection of armature bones '''

    active = None
    '''Armature’s active bone 

    :type:  Bone 
    '''


class ArmatureEditBones:
    '''Collection of armature edit bones '''

    active = None
    '''Armatures active edit bone 

    :type:  EditBone 
    '''

    def new(self, name):
        '''Add a new bone 

        :param name: New name for the bone 
        :type name: string, (never None)
        :rtype: EditBone 
        :return:  Newly created edit bone 
        '''
        pass

    def remove(self, bone):
        '''Remove an existing bone from the armature 

        :param bone: EditBone to remove 
        :type bone: EditBone, (never None)
        '''
        pass


class ArmatureModifier:
    '''Armature deformation modifier '''

    invert_vertex_group = None
    '''Invert vertex group influence 

    :type:  boolean, default False 
    '''

    object = None
    '''Armature object to deform with 

    :type:  Object 
    '''

    use_bone_envelopes = None
    '''Bind Bone envelopes to armature modifier 

    :type:  boolean, default False 
    '''

    use_deform_preserve_volume = None
    '''Deform rotation interpolation with quaternions 

    :type:  boolean, default False 
    '''

    use_multi_modifier = None
    '''Use same input as previous modifier, and mix results using overall vgroup 

    :type:  boolean, default False 
    '''

    use_vertex_groups = None
    '''Bind vertex groups to armature modifier 

    :type:  boolean, default False 
    '''

    vertex_group = None
    '''Name of Vertex Group which determines influence of modifier per point 

    :type:  string, default “”, (never None) 
    '''


class ArmatureSensor:
    '''Sensor to detect values and changes in values of IK solver '''

    bone = None
    '''Identify the bone to check value from 

    :type:  string, default “”, (never None) 
    '''

    constraint = None
    '''Identify the bone constraint to check value from 

    :type:  string, default “”, (never None) 
    '''

    test_type = None
    '''Type of value and test 

    :type:  enum in [‘STATECHG’, ‘LINERRORBELOW’, ‘LINERRORABOVE’, ‘ROTERRORBELOW’, ‘ROTERRORABOVE’], default ‘STATECHG’ 
    '''

    value = None
    '''Value to be used in comparison 

    :type:  float in [-inf, inf], default 0.0 
    '''


class ArrayModifier:
    '''Array duplication modifier '''

    constant_offset_displace = None
    '''Value for the distance between arrayed items 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    count = None
    '''Number of duplicates to make 

    :type:  int in [1, inf], default 0 
    '''

    curve = None
    '''Curve object to fit array length to 

    :type:  Object 
    '''

    end_cap = None
    '''Mesh object to use as an end cap 

    :type:  Object 
    '''

    fit_length = None
    '''Length to fit array within 

    :type:  float in [0, inf], default 0.0 
    '''

    fit_type = None
    '''Array length calculation method 

    :type:  enum in [‘FIXED_COUNT’, ‘FIT_LENGTH’, ‘FIT_CURVE’], default ‘FIXED_COUNT’ 
    '''

    merge_threshold = None
    '''Limit below which to merge vertices 

    :type:  float in [0, inf], default 0.0 
    '''

    offset_object = None
    '''Use the location and rotation of another object to determine the distance and rotational change between arrayed items 

    :type:  Object 
    '''

    relative_offset_displace = None
    '''The size of the geometry will determine the distance between arrayed items 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    start_cap = None
    '''Mesh object to use as a start cap 

    :type:  Object 
    '''

    use_constant_offset = None
    '''Add a constant offset 

    :type:  boolean, default False 
    '''

    use_merge_vertices = None
    '''Merge vertices in adjacent duplicates 

    :type:  boolean, default False 
    '''

    use_merge_vertices_cap = None
    '''Merge vertices in first and last duplicates 

    :type:  boolean, default False 
    '''

    use_object_offset = None
    '''Add another object’s transformation to the total offset 

    :type:  boolean, default False 
    '''

    use_relative_offset = None
    '''Add an offset relative to the object’s bounding box 

    :type:  boolean, default False 
    '''


class Available:
    pass


class BackgroundImage:
    '''Image and settings for display in the 3D View background '''

    clip = None
    '''Movie clip displayed and edited in this space 

    :type:  MovieClip 
    '''

    clip_user = None
    '''Parameters defining which frame of the movie clip is displayed 

    :type:  MovieClipUser, (readonly, never None) 
    '''

    draw_depth = None
    '''Draw under or over everything 

    :type:  enum in [‘BACK’, ‘FRONT’], default ‘BACK’ 
    '''

    frame_method = None
    '''How the image fits in the camera frame 

    :type:  enum in [‘STRETCH’, ‘FIT’, ‘CROP’], default ‘STRETCH’ 
    '''

    image = None
    '''Image displayed and edited in this space 

    :type:  Image 
    '''

    image_user = None
    '''Parameters defining which layer, pass and frame of the image is displayed 

    :type:  ImageUser, (readonly, never None) 
    '''

    offset_x = None
    '''Offset image horizontally from the world origin 

    :type:  float in [-inf, inf], default 0.0 
    '''

    offset_y = None
    '''Offset image vertically from the world origin 

    :type:  float in [-inf, inf], default 0.0 
    '''

    opacity = None
    '''Image opacity to blend the image against the background color 

    :type:  float in [0, 1], default 0.0 
    '''

    rotation = None
    '''Rotation for the background image (ortho view only) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    show_background_image = None
    '''Show this image as background 

    :type:  boolean, default False 
    '''

    show_expanded = None
    '''Show the expanded in the user interface 

    :type:  boolean, default False 
    '''

    show_on_foreground = None
    '''Show this image in front of objects in viewport 

    :type:  boolean, default False 
    '''

    size = None
    '''Size of the background image (ortho view only) 

    :type:  float in [0, inf], default 0.0 
    '''

    source = None
    '''Data source used for background 

    :type:  enum in [‘IMAGE’, ‘MOVIE_CLIP’], default ‘IMAGE’ 
    '''

    use_camera_clip = None
    '''Use movie clip from active scene camera 

    :type:  boolean, default False 
    '''

    use_flip_x = None
    '''Flip the background image horizontally 

    :type:  boolean, default False 
    '''

    use_flip_y = None
    '''Flip the background image vertically 

    :type:  boolean, default False 
    '''

    view_axis = None
    '''The axis to display the image on 

    :type:  enum in [‘LEFT’, ‘RIGHT’, ‘BACK’, ‘FRONT’, ‘BOTTOM’, ‘TOP’, ‘ALL’, ‘CAMERA’], default ‘ALL’ 
    '''


class BackgroundImages:
    '''Collection of background images '''

    def new(self):
        '''Add new background image 

        :rtype: BackgroundImage 
        :return:  Image displayed as viewport background 
        '''
        pass

    def remove(self, image):
        '''Remove background image 

        :param image: Image displayed as viewport background 
        :type image: BackgroundImage, (never None)
        '''
        pass

    def clear(self):
        '''Remove all background images 

        '''
        pass


class BakePixel:
    du_dx = None
    '''

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    du_dy = None
    '''

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    dv_dx = None
    '''

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    dv_dy = None
    '''

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    next = None
    '''

    :type:  BakePixel, (readonly) 
    '''

    object_id = None
    '''

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    primitive_id = None
    '''

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    uv = None
    '''

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0), (readonly) 
    '''


class BakeSettings:
    '''Bake data for a Scene data-block '''

    cage_extrusion = None
    '''Distance to use for the inward ray cast when using selected to active 

    :type:  float in [0, inf], default 0.0 
    '''

    cage_object = None
    '''Object to use as cage instead of calculating the cage from the active object with cage extrusion 

    :type:  string, default “”, (never None) 
    '''

    filepath = None
    '''Image filepath to use when saving externally 

    :type:  string, default “”, (never None) 
    '''

    height = None
    '''Vertical dimension of the baking map 

    :type:  int in [4, 10000], default 0 
    '''

    image_settings = None
    '''

    :type:  ImageFormatSettings, (readonly, never None) 
    '''

    margin = None
    '''Extends the baked result as a post process filter 

    :type:  int in [0, 32767], default 0 
    '''

    normal_b = None
    '''Axis to bake in blue channel 

    :type:  enum in [‘POS_X’, ‘POS_Y’, ‘POS_Z’, ‘NEG_X’, ‘NEG_Y’, ‘NEG_Z’], default ‘POS_X’ 
    '''

    normal_g = None
    '''Axis to bake in green channel 

    :type:  enum in [‘POS_X’, ‘POS_Y’, ‘POS_Z’, ‘NEG_X’, ‘NEG_Y’, ‘NEG_Z’], default ‘POS_X’ 
    '''

    normal_r = None
    '''Axis to bake in red channel 

    :type:  enum in [‘POS_X’, ‘POS_Y’, ‘POS_Z’, ‘NEG_X’, ‘NEG_Y’, ‘NEG_Z’], default ‘POS_X’ 
    '''

    normal_space = None
    '''Choose normal space for baking 

    :type:  enum in [‘OBJECT’, ‘TANGENT’], default ‘OBJECT’ 
    '''

    pass_filter = None
    '''Passes to include in the active baking pass 

    :type:  enum set in {‘NONE’, ‘AO’, ‘EMIT’, ‘DIRECT’, ‘INDIRECT’, ‘COLOR’, ‘DIFFUSE’, ‘GLOSSY’, ‘TRANSMISSION’, ‘SUBSURFACE’}, default {}, (readonly) 
    '''

    save_mode = None
    '''Choose how to save the baking map 

    :type:  enum in [‘INTERNAL’, ‘EXTERNAL’], default ‘INTERNAL’ 
    '''

    use_automatic_name = None
    '''Automatically name the output file with the pass type (external only) 

    :type:  boolean, default False 
    '''

    use_cage = None
    '''Cast rays to active object from a cage 

    :type:  boolean, default False 
    '''

    use_clear = None
    '''Clear Images before baking (internal only) 

    :type:  boolean, default False 
    '''

    use_pass_ambient_occlusion = None
    '''Add ambient occlusion contribution 

    :type:  boolean, default False 
    '''

    use_pass_color = None
    '''Color the pass 

    :type:  boolean, default False 
    '''

    use_pass_diffuse = None
    '''Add diffuse contribution 

    :type:  boolean, default False 
    '''

    use_pass_direct = None
    '''Add direct lighting contribution 

    :type:  boolean, default False 
    '''

    use_pass_emit = None
    '''Add emission contribution 

    :type:  boolean, default False 
    '''

    use_pass_glossy = None
    '''Add glossy contribution 

    :type:  boolean, default False 
    '''

    use_pass_indirect = None
    '''Add indirect lighting contribution 

    :type:  boolean, default False 
    '''

    use_pass_subsurface = None
    '''Add subsurface contribution 

    :type:  boolean, default False 
    '''

    use_pass_transmission = None
    '''Add transmission contribution 

    :type:  boolean, default False 
    '''

    use_selected_to_active = None
    '''Bake shading on the surface of selected objects to the active object 

    :type:  boolean, default False 
    '''

    use_split_materials = None
    '''Split external images per material (external only) 

    :type:  boolean, default False 
    '''

    width = None
    '''Horizontal dimension of the baking map 

    :type:  int in [4, 10000], default 0 
    '''


class BevelModifier:
    '''Bevel modifier to make edges and vertices more rounded '''

    angle_limit = None
    '''Angle above which to bevel edges 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    edge_weight_method = None
    '''What edge weight to use for weighting a vertex 

    :type:  enum in [‘AVERAGE’, ‘SHARPEST’, ‘LARGEST’], default ‘AVERAGE’ 
    '''

    limit_method = None
    '''

    :type:  enum in [‘NONE’, ‘ANGLE’, ‘WEIGHT’, ‘VGROUP’], default ‘NONE’ 
    '''

    loop_slide = None
    '''Prefer sliding along edges to having even widths 

    :type:  boolean, default False 
    '''

    material = None
    '''Material index of generated faces, -1 for automatic 

    :type:  int in [-1, 32767], default 0 
    '''

    offset_type = None
    '''What distance Width measures 

    :type:  enum in [‘OFFSET’, ‘WIDTH’, ‘DEPTH’, ‘PERCENT’], default ‘OFFSET’ 
    '''

    profile = None
    '''The profile shape (0.5 = round) 

    :type:  float in [0, 1], default 0.0 
    '''

    segments = None
    '''Number of segments for round edges/verts 

    :type:  int in [1, 100], default 0 
    '''

    use_clamp_overlap = None
    '''Clamp the width to avoid overlap 

    :type:  boolean, default False 
    '''

    use_only_vertices = None
    '''Bevel verts/corners, not edges 

    :type:  boolean, default False 
    '''

    vertex_group = None
    '''Vertex group name 

    :type:  string, default “”, (never None) 
    '''

    width = None
    '''Bevel value/amount 

    :type:  float in [0, inf], default 0.0 
    '''


class BezierSplinePoint:
    '''Bezier curve point with two handles '''

    co = None
    '''Coordinates of the control point 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    handle_left = None
    '''Coordinates of the first handle 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    handle_left_type = None
    '''Handle types 

    :type:  enum in [‘FREE’, ‘VECTOR’, ‘ALIGNED’, ‘AUTO’], default ‘FREE’ 
    '''

    handle_right = None
    '''Coordinates of the second handle 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    handle_right_type = None
    '''Handle types 

    :type:  enum in [‘FREE’, ‘VECTOR’, ‘ALIGNED’, ‘AUTO’], default ‘FREE’ 
    '''

    hide = None
    '''Visibility status 

    :type:  boolean, default False 
    '''

    radius = None
    '''Radius for beveling 

    :type:  float in [0, inf], default 0.0 
    '''

    select_control_point = None
    '''Control point selection status 

    :type:  boolean, default False 
    '''

    select_left_handle = None
    '''Handle 1 selection status 

    :type:  boolean, default False 
    '''

    select_right_handle = None
    '''Handle 2 selection status 

    :type:  boolean, default False 
    '''

    tilt = None
    '''Tilt in 3D View 

    :type:  float in [-376.991, 376.991], default 0.0 
    '''

    weight_softbody = None
    '''Softbody goal weight 

    :type:  float in [0.01, 100], default 0.0 
    '''


class BlendData:
    '''Main data structure representing a .blend file and all its datablocks '''

    actions = None
    '''Action datablocks 

    :type:  BlendDataActions bpy_prop_collection of Action, (readonly) 
    '''

    armatures = None
    '''Armature datablocks 

    :type:  BlendDataArmatures bpy_prop_collection of Armature, (readonly) 
    '''

    brushes = None
    '''Brush datablocks 

    :type:  BlendDataBrushes bpy_prop_collection of Brush, (readonly) 
    '''

    cache_files = None
    '''Cache Files datablocks 

    :type:  BlendDataCacheFiles bpy_prop_collection of CacheFile, (readonly) 
    '''

    cameras = None
    '''Camera datablocks 

    :type:  BlendDataCameras bpy_prop_collection of Camera, (readonly) 
    '''

    curves = None
    '''Curve datablocks 

    :type:  BlendDataCurves bpy_prop_collection of Curve, (readonly) 
    '''

    filepath = None
    '''Path to the .blend file 

    :type:  string, default “”, (readonly, never None) 
    '''

    fonts = None
    '''Vector font datablocks 

    :type:  BlendDataFonts bpy_prop_collection of VectorFont, (readonly) 
    '''

    grease_pencil = None
    '''Grease Pencil datablocks 

    :type:  BlendDataGreasePencils bpy_prop_collection of GreasePencil, (readonly) 
    '''

    groups = None
    '''Group datablocks 

    :type:  BlendDataGroups bpy_prop_collection of Group, (readonly) 
    '''

    images = None
    '''Image datablocks 

    :type:  BlendDataImages bpy_prop_collection of Image, (readonly) 
    '''

    is_dirty = None
    '''Have recent edits been saved to disk 

    :type:  boolean, default False, (readonly) 
    '''

    is_saved = None
    '''Has the current session been saved to disk as a .blend file 

    :type:  boolean, default False, (readonly) 
    '''

    lamps = None
    '''Lamp datablocks 

    :type:  BlendDataLamps bpy_prop_collection of Lamp, (readonly) 
    '''

    lattices = None
    '''Lattice datablocks 

    :type:  BlendDataLattices bpy_prop_collection of Lattice, (readonly) 
    '''

    libraries = None
    '''Library datablocks 

    :type:  BlendDataLibraries bpy_prop_collection of Library, (readonly) 
    '''

    linestyles = None
    '''Line Style datablocks 

    :type:  BlendDataLineStyles bpy_prop_collection of FreestyleLineStyle, (readonly) 
    '''

    masks = None
    '''Masks datablocks 

    :type:  BlendDataMasks bpy_prop_collection of Mask, (readonly) 
    '''

    materials = None
    '''Material datablocks 

    :type:  BlendDataMaterials bpy_prop_collection of Material, (readonly) 
    '''

    meshes = None
    '''Mesh datablocks 

    :type:  BlendDataMeshes bpy_prop_collection of Mesh, (readonly) 
    '''

    metaballs = None
    '''Metaball datablocks 

    :type:  BlendDataMetaBalls bpy_prop_collection of MetaBall, (readonly) 
    '''

    movieclips = None
    '''Movie Clip datablocks 

    :type:  BlendDataMovieClips bpy_prop_collection of MovieClip, (readonly) 
    '''

    node_groups = None
    '''Node group datablocks 

    :type:  BlendDataNodeTrees bpy_prop_collection of NodeTree, (readonly) 
    '''

    objects = None
    '''Object datablocks 

    :type:  BlendDataObjects bpy_prop_collection of Object, (readonly) 
    '''

    paint_curves = None
    '''Paint Curves datablocks 

    :type:  BlendDataPaintCurves bpy_prop_collection of PaintCurve, (readonly) 
    '''

    palettes = None
    '''Palette datablocks 

    :type:  BlendDataPalettes bpy_prop_collection of Palette, (readonly) 
    '''

    particles = None
    '''Particle datablocks 

    :type:  BlendDataParticles bpy_prop_collection of ParticleSettings, (readonly) 
    '''

    scenes = None
    '''Scene datablocks 

    :type:  BlendDataScenes bpy_prop_collection of Scene, (readonly) 
    '''

    screens = None
    '''Screen datablocks 

    :type:  BlendDataScreens bpy_prop_collection of Screen, (readonly) 
    '''

    shape_keys = None
    '''Shape Key datablocks 

    :type:  bpy_prop_collection of Key, (readonly) 
    '''

    sounds = None
    '''Sound datablocks 

    :type:  BlendDataSounds bpy_prop_collection of Sound, (readonly) 
    '''

    speakers = None
    '''Speaker datablocks 

    :type:  BlendDataSpeakers bpy_prop_collection of Speaker, (readonly) 
    '''

    texts = None
    '''Text datablocks 

    :type:  BlendDataTexts bpy_prop_collection of Text, (readonly) 
    '''

    textures = None
    '''Texture datablocks 

    :type:  BlendDataTextures bpy_prop_collection of Texture, (readonly) 
    '''

    use_autopack = None
    '''Automatically pack all external data into .blend file 

    :type:  boolean, default False 
    '''

    version = None
    '''Version of the blender the .blend was saved with 

    :type:  int array of 3 items in [0, inf], default (0, 0, 0), (readonly) 
    '''

    window_managers = None
    '''Window manager datablocks 

    :type:  BlendDataWindowManagers bpy_prop_collection of WindowManager, (readonly) 
    '''

    worlds = None
    '''World datablocks 

    :type:  BlendDataWorlds bpy_prop_collection of World, (readonly) 
    '''

    def user_map(self, key_types, value_types):
        '''For list of valid set members for key_types & value_types, see: bpy.types.KeyingSetPath.id_type. 

        :param subset: When passed, only these data-blocks and their users will be included as keys/values in the map. 
        :type subset: sequence
        :param key_types: Filter the keys mapped by ID types. 
        :type key_types: set of strings
        :param value_types: Filter the values in the set by ID types. 
        :type value_types: set of strings
        :rtype: dict 
        :return:  dictionary of bpy.types.ID instances, with sets of ID’s as their values. 
        '''
        pass


class BlendDataActions:
    '''Collection of actions '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new action to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: Action 
        :return:  New action data-block 
        '''
        pass

    def remove(self, action, do_unlink=False):
        '''Remove a action from the current blendfile 

        :param action: Action to remove 
        :type action: Action, (never None)
        :param do_unlink: Unlink all usages of this action before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataArmatures:
    '''Collection of armatures '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new armature to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: Armature 
        :return:  New armature data-block 
        '''
        pass

    def remove(self, armature, do_unlink=False):
        '''Remove a armature from the current blendfile 

        :param armature: Armature to remove 
        :type armature: Armature, (never None)
        :param do_unlink: Unlink all usages of this armature before deleting it (WARNING: will also delete objects instancing that armature data) 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataBrushes:
    '''Collection of brushes '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name, mode='TEXTURE_PAINT'):
        '''Add a new brush to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :param mode: Paint Mode for the new brushOBJECT Object Mode.EDIT Edit Mode.POSE Pose Mode.SCULPT Sculpt Mode.VERTEX_PAINT Vertex Paint.WEIGHT_PAINT Weight Paint.TEXTURE_PAINT Texture Paint.PARTICLE_EDIT Particle Edit.GPENCIL_EDIT Edit Strokes, Edit Grease Pencil Strokes. 
        :type mode: enum in ['OBJECT', 'EDIT', 'POSE', 'SCULPT', 'VERTEX_PAINT', 'WEIGHT_PAINT', 'TEXTURE_PAINT', 'PARTICLE_EDIT', 'GPENCIL_EDIT'], (optional)
        :rtype: Brush 
        :return:  New brush data-block 
        '''
        pass

    def remove(self, brush, do_unlink=False):
        '''Remove a brush from the current blendfile 

        :param brush: Brush to remove 
        :type brush: Brush, (never None)
        :param do_unlink: Unlink all usages of this brush before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataCacheFiles:
    '''Collection of cache files '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataCameras:
    '''Collection of cameras '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new camera to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: Camera 
        :return:  New camera data-block 
        '''
        pass

    def remove(self, camera, do_unlink=False):
        '''Remove a camera from the current blendfile 

        :param camera: Camera to remove 
        :type camera: Camera, (never None)
        :param do_unlink: Unlink all usages of this camera before deleting it (WARNING: will also delete objects instancing that camera data) 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataCurves:
    '''Collection of curves '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name, type):
        '''Add a new curve to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :param type: Type, The type of curve to add 
        :type type: enum in ['CURVE', 'SURFACE', 'FONT']
        :rtype: Curve 
        :return:  New curve data-block 
        '''
        pass

    def remove(self, curve, do_unlink=False):
        '''Remove a curve from the current blendfile 

        :param curve: Curve to remove 
        :type curve: Curve, (never None)
        :param do_unlink: Unlink all usages of this curve before deleting it (WARNING: will also delete objects instancing that curve data) 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataFonts:
    '''Collection of fonts '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def load(self, filepath, check_existing=False):
        '''Load a new font into the main database 

        :param filepath: path of the font to load 
        :type filepath: string, (never None)
        :param check_existing: Using existing data-block if this file is already loaded 
        :type check_existing: boolean, (optional)
        :rtype: VectorFont 
        :return:  New font data-block 
        '''
        pass

    def remove(self, vfont, do_unlink=False):
        '''Remove a font from the current blendfile 

        :param vfont: Font to remove 
        :type vfont: VectorFont, (never None)
        :param do_unlink: Unlink all usages of this font before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataGreasePencils:
    '''Collection of grease pencils '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass

    def remove(self, grease_pencil, do_unlink=False):
        '''Remove a grease pencil instance from the current blendfile 

        :param grease_pencil: Grease Pencil to remove 
        :type grease_pencil: GreasePencil, (never None)
        :param do_unlink: Unlink all usages of this grease pencil before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass


class BlendDataGroups:
    '''Collection of groups '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new group to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: Group 
        :return:  New group data-block 
        '''
        pass

    def remove(self, group, do_unlink=False):
        '''Remove a group from the current blendfile 

        :param group: Group to remove 
        :type group: Group, (never None)
        :param do_unlink: Unlink all usages of this group before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataImages:
    '''Collection of images '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self,
            name,
            width,
            height,
            alpha=False,
            float_buffer=False,
            stereo3d=False):
        '''Add a new image to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :param width: Width of the image 
        :type width: int in [1, inf]
        :param height: Height of the image 
        :type height: int in [1, inf]
        :param alpha: Alpha, Use alpha channel 
        :type alpha: boolean, (optional)
        :param float_buffer: Float Buffer, Create an image with floating point color 
        :type float_buffer: boolean, (optional)
        :param stereo3d: Stereo 3D, Create left and right views 
        :type stereo3d: boolean, (optional)
        :rtype: Image 
        :return:  New image data-block 
        '''
        pass

    def load(self, filepath, check_existing=False):
        '''Load a new image into the main database 

        :param filepath: path of the file to load 
        :type filepath: string, (never None)
        :param check_existing: Using existing data-block if this file is already loaded 
        :type check_existing: boolean, (optional)
        :rtype: Image 
        :return:  New image data-block 
        '''
        pass

    def remove(self, image, do_unlink=False):
        '''Remove an image from the current blendfile 

        :param image: Image to remove 
        :type image: Image, (never None)
        :param do_unlink: Unlink all usages of this image before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataLamps:
    '''Collection of lamps '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name, type):
        '''Add a new lamp to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :param type: Type, The type of texture to addPOINT Point, Omnidirectional point light source.SUN Sun, Constant direction parallel ray light source.SPOT Spot, Directional cone light source.HEMI Hemi, 180 degree constant light source.AREA Area, Directional area light source. 
        :type type: enum in ['POINT', 'SUN', 'SPOT', 'HEMI', 'AREA']
        :rtype: Lamp 
        :return:  New lamp data-block 
        '''
        pass

    def remove(self, lamp, do_unlink=False):
        '''Remove a lamp from the current blendfile 

        :param lamp: Lamp to remove 
        :type lamp: Lamp, (never None)
        :param do_unlink: Unlink all usages of this lamp before deleting it (WARNING: will also delete objects instancing that lamp data) 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataLattices:
    '''Collection of lattices '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new lattice to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: Lattice 
        :return:  New lattices data-block 
        '''
        pass

    def remove(self, lattice, do_unlink=False):
        '''Remove a lattice from the current blendfile 

        :param lattice: Lattice to remove 
        :type lattice: Lattice, (never None)
        :param do_unlink: Unlink all usages of this lattice before deleting it (WARNING: will also delete objects instancing that lattice data) 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataLibraries:
    '''Collection of libraries '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass

    def load(self, filepath, link=False, relative=False):
        '''Returns a context manager which exposes 2 library objects on entering. Each object has attributes matching bpy.data which are lists of strings to be linked. 

        :param filepath: The path to a blend file. 
        :type filepath: string
        :param link: When False reference to the original file is lost. 
        :type link: bool
        :param relative: When True the path is stored relative to the open blend file. 
        :type relative: bool
        '''
        pass

    def write(self,
              filepath,
              datablocks,
              relative_remap=False,
              fake_user=False,
              compress=False):
        '''Write data-blocks into a blend file. 

        :param filepath: The path to write the blend-file. 
        :type filepath: string
        :param datablocks: set of data-blocks (bpy.types.ID instances). 
        :type datablocks: set
        :param relative_remap: When True, remap the paths relative to the current blend-file. 
        :type relative_remap: bool
        :param fake_user: When True, data-blocks will be written with fake-user flag enabled. 
        :type fake_user: bool
        :param compress: When True, write a compressed blend file. 
        :type compress: bool
        '''
        pass


class BlendDataLineStyles:
    '''Collection of line styles '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass

    def new(self, name):
        '''Add a new line style instance to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: FreestyleLineStyle 
        :return:  New line style data-block 
        '''
        pass

    def remove(self, linestyle, do_unlink=False):
        '''Remove a line style instance from the current blendfile 

        :param linestyle: Line style to remove 
        :type linestyle: FreestyleLineStyle, (never None)
        :param do_unlink: Unlink all usages of this line style before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass


class BlendDataMasks:
    '''Collection of masks '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass

    def new(self, name=""):
        '''Add a new mask with a given name to the main database 

        :param name: Mask, Name of new mask data-block 
        :type name: string, (optional, never None)
        :rtype: Mask 
        :return:  New mask data-block 
        '''
        pass

    def remove(self, mask, do_unlink=False):
        '''Remove a masks from the current blendfile. 

        :param mask: Mask to remove 
        :type mask: Mask, (never None)
        :param do_unlink: Unlink all usages of this mask before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass


class BlendDataMaterials:
    '''Collection of materials '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new material to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: Material 
        :return:  New material data-block 
        '''
        pass

    def remove(self, material, do_unlink=False):
        '''Remove a material from the current blendfile 

        :param material: Material to remove 
        :type material: Material, (never None)
        :param do_unlink: Unlink all usages of this material before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataMeshes:
    '''Collection of meshes '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new mesh to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: Mesh 
        :return:  New mesh data-block 
        '''
        pass

    def new_from_object(self,
                        scene,
                        object,
                        apply_modifiers,
                        settings,
                        calc_tessface=True,
                        calc_undeformed=False):
        '''Add a new mesh created from object with modifiers applied 

        :param scene: Scene within which to evaluate modifiers 
        :type scene: Scene, (never None)
        :param object: Object to create mesh from 
        :type object: Object, (never None)
        :param apply_modifiers: Apply modifiers 
        :type apply_modifiers: boolean
        :param settings: Modifier settings to applyPREVIEW Preview, Apply modifier preview settings.RENDER Render, Apply modifier render settings. 
        :type settings: enum in ['PREVIEW', 'RENDER']
        :param calc_tessface: Calculate Tessellation, Calculate tessellation faces 
        :type calc_tessface: boolean, (optional)
        :param calc_undeformed: Calculate Undeformed, Calculate undeformed vertex coordinates 
        :type calc_undeformed: boolean, (optional)
        :rtype: Mesh 
        :return:  Mesh created from object, remove it if it is only used for export 
        '''
        pass

    def remove(self, mesh, do_unlink=False):
        '''Remove a mesh from the current blendfile 

        :param mesh: Mesh to remove 
        :type mesh: Mesh, (never None)
        :param do_unlink: Unlink all usages of this mesh before deleting it (WARNING: will also delete objects instancing that mesh data) 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataMetaBalls:
    '''Collection of metaballs '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new metaball to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: MetaBall 
        :return:  New metaball data-block 
        '''
        pass

    def remove(self, metaball, do_unlink=False):
        '''Remove a metaball from the current blendfile 

        :param metaball: Metaball to remove 
        :type metaball: MetaBall, (never None)
        :param do_unlink: Unlink all usages of this metaball before deleting it (WARNING: will also delete objects instancing that metaball data) 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataMovieClips:
    '''Collection of movie clips '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass

    def remove(self, clip, do_unlink=False):
        '''Remove a movie clip from the current blendfile. 

        :param clip: Movie clip to remove 
        :type clip: MovieClip, (never None)
        :param do_unlink: Unlink all usages of this movie clip before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def load(self, filepath, check_existing=False):
        '''Add a new movie clip to the main database from a file (while check_existing is disabled for consistency with other load functions, behavior with multiple movie-clips using the same file may incorrectly generate proxies) 

        :param filepath: path for the data-block 
        :type filepath: string, (never None)
        :param check_existing: Using existing data-block if this file is already loaded 
        :type check_existing: boolean, (optional)
        :rtype: MovieClip 
        :return:  New movie clip data-block 
        '''
        pass


class BlendDataNodeTrees:
    '''Collection of node trees '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name, type):
        '''Add a new node tree to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :param type: Type, The type of node_group to add 
        :type type: enum in ['DUMMY']
        :rtype: NodeTree 
        :return:  New node tree data-block 
        '''
        pass

    def remove(self, tree, do_unlink=False):
        '''Remove a node tree from the current blendfile 

        :param tree: Node tree to remove 
        :type tree: NodeTree, (never None)
        :param do_unlink: Unlink all usages of this node tree before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataObjects:
    '''Collection of objects '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name, object_data):
        '''Add a new object to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :param object_data: Object data or None for an empty object 
        :type object_data: ID
        :rtype: Object 
        :return:  New object data-block 
        '''
        pass

    def remove(self, object, do_unlink=False):
        '''Remove a object from the current blendfile 

        :param object: Object to remove 
        :type object: Object, (never None)
        :param do_unlink: Unlink all usages of this object before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataPaintCurves:
    '''Collection of paint curves '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataPalettes:
    '''Collection of palettes '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new palette to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: Palette 
        :return:  New palette data-block 
        '''
        pass

    def remove(self, palette, do_unlink=False):
        '''Remove a palette from the current blendfile 

        :param palette: Palette to remove 
        :type palette: Palette, (never None)
        :param do_unlink: Unlink all usages of this palette before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataParticles:
    '''Collection of particle settings '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new particle settings instance to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: ParticleSettings 
        :return:  New particle settings data-block 
        '''
        pass

    def remove(self, particle, do_unlink=False):
        '''Remove a particle settings instance from the current blendfile 

        :param particle: Particle Settings to remove 
        :type particle: ParticleSettings, (never None)
        :param do_unlink: Unlink all usages of those particle settings before deleting them 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataScenes:
    '''Collection of scenes '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new scene to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: Scene 
        :return:  New scene data-block 
        '''
        pass

    def remove(self, scene, do_unlink=False):
        '''Remove a scene from the current blendfile 

        :param scene: Scene to remove 
        :type scene: Scene, (never None)
        :param do_unlink: Unlink all usages of this scene before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataScreens:
    '''Collection of screens '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataSounds:
    '''Collection of sounds '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def load(self, filepath, check_existing=False):
        '''Add a new sound to the main database from a file 

        :param filepath: path for the data-block 
        :type filepath: string, (never None)
        :param check_existing: Using existing data-block if this file is already loaded 
        :type check_existing: boolean, (optional)
        :rtype: Sound 
        :return:  New text data-block 
        '''
        pass

    def remove(self, sound, do_unlink=False):
        '''Remove a sound from the current blendfile 

        :param sound: Sound to remove 
        :type sound: Sound, (never None)
        :param do_unlink: Unlink all usages of this sound before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataSpeakers:
    '''Collection of speakers '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new speaker to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: Speaker 
        :return:  New speaker data-block 
        '''
        pass

    def remove(self, speaker, do_unlink=False):
        '''Remove a speaker from the current blendfile 

        :param speaker: Speaker to remove 
        :type speaker: Speaker, (never None)
        :param do_unlink: Unlink all usages of this speaker before deleting it (WARNING: will also delete objects instancing that speaker data) 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataTexts:
    '''Collection of texts '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new text to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: Text 
        :return:  New text data-block 
        '''
        pass

    def remove(self, text, do_unlink=False):
        '''Remove a text from the current blendfile 

        :param text: Text to remove 
        :type text: Text, (never None)
        :param do_unlink: Unlink all usages of this text before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def load(self, filepath, internal=False):
        '''Add a new text to the main database from a file 

        :param filepath: path for the data-block 
        :type filepath: string, (never None)
        :param internal: Make internal, Make text file internal after loading 
        :type internal: boolean, (optional)
        :rtype: Text 
        :return:  New text data-block 
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataTextures:
    '''Collection of groups '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name, type):
        '''Add a new texture to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :param type: Type, The type of texture to addNONE None.BLEND Blend, Procedural - create a ramp texture.CLOUDS Clouds, Procedural - create a cloud-like fractal noise texture.DISTORTED_NOISE Distorted Noise, Procedural - noise texture distorted by two noise algorithms.ENVIRONMENT_MAP Environment Map, Create a render of the environment mapped to a texture.IMAGE Image or Movie, Allow for images or movies to be used as textures.MAGIC Magic, Procedural - color texture based on trigonometric functions.MARBLE Marble, Procedural - marble-like noise texture with wave generated bands.MUSGRAVE Musgrave, Procedural - highly flexible fractal noise texture.NOISE Noise, Procedural - random noise, gives a different result every time, for every frame, for every pixel.OCEAN Ocean, Use a texture generated by an Ocean modifier.POINT_DENSITY Point Density.STUCCI Stucci, Procedural - create a fractal noise texture.VORONOI Voronoi, Procedural - create cell-like patterns based on Worley noise.VOXEL_DATA Voxel Data, Create a 3D texture based on volumetric data.WOOD Wood, Procedural - wave generated bands or rings, with optional noise. 
        :type type: enum in ['NONE', 'BLEND', 'CLOUDS', 'DISTORTED_NOISE', 'ENVIRONMENT_MAP', 'IMAGE', 'MAGIC', 'MARBLE', 'MUSGRAVE', 'NOISE', 'OCEAN', 'POINT_DENSITY', 'STUCCI', 'VORONOI', 'VOXEL_DATA', 'WOOD']
        :rtype: Texture 
        :return:  New texture data-block 
        '''
        pass

    def remove(self, texture, do_unlink=False):
        '''Remove a texture from the current blendfile 

        :param texture: Texture to remove 
        :type texture: Texture, (never None)
        :param do_unlink: Unlink all usages of this texture before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataWindowManagers:
    '''Collection of window managers '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendDataWorlds:
    '''Collection of worlds '''

    is_updated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    def new(self, name):
        '''Add a new world to the main database 

        :param name: New name for the data-block 
        :type name: string, (never None)
        :rtype: World 
        :return:  New world data-block 
        '''
        pass

    def remove(self, world, do_unlink=False):
        '''Remove a world from the current blendfile 

        :param world: World to remove 
        :type world: World, (never None)
        :param do_unlink: Unlink all usages of this world before deleting it 
        :type do_unlink: boolean, (optional)
        '''
        pass

    def tag(self, value):
        '''tag 

        :param value: Value 
        :type value: boolean
        '''
        pass


class BlendTexture:
    '''Procedural color blending texture '''

    progression = None
    '''Style of the color blending 

    :type:  enum in [‘LINEAR’, ‘QUADRATIC’, ‘EASING’, ‘DIAGONAL’, ‘SPHERICAL’, ‘QUADRATIC_SPHERE’, ‘RADIAL’], default ‘LINEAR’ 
    '''

    use_flip_axis = None
    '''Flip the texture’s X and Y axis 

    :type:  enum in [‘HORIZONTAL’, ‘VERTICAL’], default ‘HORIZONTAL’ 
    '''

    users_material = None
    '''Materials that use this texture (readonly) '''

    users_object_modifier = None
    '''Object modifiers that use this texture (readonly) '''


class BlenderRNA:
    '''Blender RNA structure definitions '''

    structs = None
    '''

    :type:  bpy_prop_collection of Struct, (readonly) 
    '''


class BoidRule:
    name = None
    '''Boid rule name 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''

    :type:  enum in [‘GOAL’, ‘AVOID’, ‘AVOID_COLLISION’, ‘SEPARATE’, ‘FLOCK’, ‘FOLLOW_LEADER’, ‘AVERAGE_SPEED’, ‘FIGHT’], default ‘GOAL’, (readonly) 
    '''

    use_in_air = None
    '''Use rule when boid is flying 

    :type:  boolean, default False 
    '''

    use_on_land = None
    '''Use rule when boid is on land 

    :type:  boolean, default False 
    '''


class BoidRuleAverageSpeed:
    level = None
    '''How much velocity’s z-component is kept constant 

    :type:  float in [0, 1], default 0.0 
    '''

    speed = None
    '''Percentage of maximum speed 

    :type:  float in [0, 1], default 0.0 
    '''

    wander = None
    '''How fast velocity’s direction is randomized 

    :type:  float in [0, 1], default 0.0 
    '''


class BoidRuleAvoid:
    fear_factor = None
    '''Avoid object if danger from it is above this threshold 

    :type:  float in [0, 100], default 0.0 
    '''

    object = None
    '''Object to avoid 

    :type:  Object 
    '''

    use_predict = None
    '''Predict target movement 

    :type:  boolean, default False 
    '''


class BoidRuleAvoidCollision:
    look_ahead = None
    '''Time to look ahead in seconds 

    :type:  float in [0, 100], default 0.0 
    '''

    use_avoid = None
    '''Avoid collision with other boids 

    :type:  boolean, default False 
    '''

    use_avoid_collision = None
    '''Avoid collision with deflector objects 

    :type:  boolean, default False 
    '''


class BoidRuleFight:
    distance = None
    '''Attack boids at max this distance 

    :type:  float in [0, 100], default 0.0 
    '''

    flee_distance = None
    '''Flee to this distance 

    :type:  float in [0, 100], default 0.0 
    '''


class BoidRuleFollowLeader:
    distance = None
    '''Distance behind leader to follow 

    :type:  float in [0, 100], default 0.0 
    '''

    object = None
    '''Follow this object instead of a boid 

    :type:  Object 
    '''

    queue_count = None
    '''How many boids in a line 

    :type:  int in [0, 100], default 0 
    '''

    use_line = None
    '''Follow leader in a line 

    :type:  boolean, default False 
    '''


class BoidRuleGoal:
    object = None
    '''Goal object 

    :type:  Object 
    '''

    use_predict = None
    '''Predict target movement 

    :type:  boolean, default False 
    '''


class BoidSettings:
    '''Settings for boid physics '''

    accuracy = None
    '''Accuracy of attack 

    :type:  float in [0, 1], default 0.0 
    '''

    active_boid_state = None
    '''

    :type:  BoidRule, (readonly) 
    '''

    active_boid_state_index = None
    '''

    :type:  int in [0, inf], default 0 
    '''

    aggression = None
    '''Boid will fight this times stronger enemy 

    :type:  float in [0, 100], default 0.0 
    '''

    air_acc_max = None
    '''Maximum acceleration in air (relative to maximum speed) 

    :type:  float in [0, 1], default 0.0 
    '''

    air_ave_max = None
    '''Maximum angular velocity in air (relative to 180 degrees) 

    :type:  float in [0, 1], default 0.0 
    '''

    air_personal_space = None
    '''Radius of boids personal space in air (% of particle size) 

    :type:  float in [0, 10], default 0.0 
    '''

    air_speed_max = None
    '''Maximum speed in air 

    :type:  float in [0, 100], default 0.0 
    '''

    air_speed_min = None
    '''Minimum speed in air (relative to maximum speed) 

    :type:  float in [0, 1], default 0.0 
    '''

    bank = None
    '''Amount of rotation around velocity vector on turns 

    :type:  float in [0, 2], default 0.0 
    '''

    health = None
    '''Initial boid health when born 

    :type:  float in [0, 100], default 0.0 
    '''

    height = None
    '''Boid height relative to particle size 

    :type:  float in [0, 2], default 0.0 
    '''

    land_acc_max = None
    '''Maximum acceleration on land (relative to maximum speed) 

    :type:  float in [0, 1], default 0.0 
    '''

    land_ave_max = None
    '''Maximum angular velocity on land (relative to 180 degrees) 

    :type:  float in [0, 1], default 0.0 
    '''

    land_jump_speed = None
    '''Maximum speed for jumping 

    :type:  float in [0, 100], default 0.0 
    '''

    land_personal_space = None
    '''Radius of boids personal space on land (% of particle size) 

    :type:  float in [0, 10], default 0.0 
    '''

    land_smooth = None
    '''How smoothly the boids land 

    :type:  float in [0, 10], default 0.0 
    '''

    land_speed_max = None
    '''Maximum speed on land 

    :type:  float in [0, 100], default 0.0 
    '''

    land_stick_force = None
    '''How strong a force must be to start effecting a boid on land 

    :type:  float in [0, 1000], default 0.0 
    '''

    pitch = None
    '''Amount of rotation around side vector 

    :type:  float in [0, 2], default 0.0 
    '''

    range = None
    '''Maximum distance from which a boid can attack 

    :type:  float in [0, 100], default 0.0 
    '''

    states = None
    '''

    :type:  bpy_prop_collection of BoidState, (readonly) 
    '''

    strength = None
    '''Maximum caused damage on attack per second 

    :type:  float in [0, 100], default 0.0 
    '''

    use_climb = None
    '''Allow boids to climb goal objects 

    :type:  boolean, default False 
    '''

    use_flight = None
    '''Allow boids to move in air 

    :type:  boolean, default False 
    '''

    use_land = None
    '''Allow boids to move on land 

    :type:  boolean, default False 
    '''


class BoidState:
    '''Boid state for boid physics '''

    active_boid_rule = None
    '''

    :type:  BoidRule, (readonly) 
    '''

    active_boid_rule_index = None
    '''

    :type:  int in [0, inf], default 0 
    '''

    falloff = None
    '''

    :type:  float in [0, 10], default 0.0 
    '''

    name = None
    '''Boid state name 

    :type:  string, default “”, (never None) 
    '''

    rule_fuzzy = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''

    rules = None
    '''

    :type:  bpy_prop_collection of BoidRule, (readonly) 
    '''

    ruleset_type = None
    '''How the rules in the list are evaluated 

    :type:  enum in [‘FUZZY’, ‘RANDOM’, ‘AVERAGE’], default ‘FUZZY’ 
    '''

    volume = None
    '''

    :type:  float in [0, 100], default 0.0 
    '''


class Bone:
    '''Bone in an Armature data-block '''

    bbone_curveinx = None
    '''X-axis handle offset for start of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_curveiny = None
    '''Y-axis handle offset for start of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_curveoutx = None
    '''X-axis handle offset for end of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_curveouty = None
    '''Y-axis handle offset for end of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_in = None
    '''Length of first Bezier Handle (for B-Bones only) 

    :type:  float in [0, 2], default 0.0 
    '''

    bbone_out = None
    '''Length of second Bezier Handle (for B-Bones only) 

    :type:  float in [0, 2], default 0.0 
    '''

    bbone_rollin = None
    '''Roll offset for the start of the B-Bone, adjusts twist 

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    bbone_rollout = None
    '''Roll offset for the end of the B-Bone, adjusts twist 

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    bbone_scalein = None
    '''Scale factor for start of the B-Bone, adjusts thickness (for tapering effects) 

    :type:  float in [0, 5], default 1.0 
    '''

    bbone_scaleout = None
    '''Scale factor for end of the B-Bone, adjusts thickness (for tapering effects) 

    :type:  float in [0, 5], default 1.0 
    '''

    bbone_segments = None
    '''Number of subdivisions of bone (for B-Bones only) 

    :type:  int in [1, 32], default 0 
    '''

    bbone_x = None
    '''B-Bone X size 

    :type:  float in [0, 1000], default 0.0 
    '''

    bbone_z = None
    '''B-Bone Z size 

    :type:  float in [0, 1000], default 0.0 
    '''

    children = None
    '''Bones which are children of this bone 

    :type:  bpy_prop_collection of Bone, (readonly) 
    '''

    envelope_distance = None
    '''Bone deformation distance (for Envelope deform only) 

    :type:  float in [0, 1000], default 0.0 
    '''

    envelope_weight = None
    '''Bone deformation weight (for Envelope deform only) 

    :type:  float in [0, 1000], default 0.0 
    '''

    head = None
    '''Location of head end of the bone relative to its parent 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    head_local = None
    '''Location of head end of the bone relative to armature 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    head_radius = None
    '''Radius of head of bone (for Envelope deform only) 

    :type:  float in [0, inf], default 0.0 
    '''

    hide = None
    '''Bone is not visible when it is not in Edit Mode (i.e. in Object or Pose Modes) 

    :type:  boolean, default False 
    '''

    hide_select = None
    '''Bone is able to be selected 

    :type:  boolean, default False 
    '''

    layers = None
    '''Layers bone exists in 

    :type:  boolean array of 32 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    matrix = None
    '''3x3 bone matrix 

    :type:  float array of 9 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    matrix_local = None
    '''4x4 bone matrix relative to armature 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''

    parent = None
    '''Parent bone (in same Armature) 

    :type:  Bone, (readonly) 
    '''

    select = None
    '''

    :type:  boolean, default False 
    '''

    select_head = None
    '''

    :type:  boolean, default False 
    '''

    select_tail = None
    '''

    :type:  boolean, default False 
    '''

    show_wire = None
    '''Bone is always drawn as Wireframe regardless of viewport draw mode (useful for non-obstructive custom bone shapes) 

    :type:  boolean, default False 
    '''

    tail = None
    '''Location of tail end of the bone 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    tail_local = None
    '''Location of tail end of the bone relative to armature 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    tail_radius = None
    '''Radius of tail of bone (for Envelope deform only) 

    :type:  float in [0, inf], default 0.0 
    '''

    use_connect = None
    '''When bone has a parent, bone’s head is stuck to the parent’s tail 

    :type:  boolean, default False, (readonly) 
    '''

    use_cyclic_offset = None
    '''When bone doesn’t have a parent, it receives cyclic offset effects (Deprecated) 

    :type:  boolean, default False 
    '''

    use_deform = None
    '''Enable Bone to deform geometry 

    :type:  boolean, default False 
    '''

    use_endroll_as_inroll = None
    '''Use Roll Out of parent bone as Roll In of its children 

    :type:  boolean, default False 
    '''

    use_envelope_multiply = None
    '''When deforming bone, multiply effects of Vertex Group weights with Envelope influence 

    :type:  boolean, default False 
    '''

    use_inherit_rotation = None
    '''Bone inherits rotation or scale from parent bone 

    :type:  boolean, default False 
    '''

    use_inherit_scale = None
    '''Bone inherits scaling from parent bone 

    :type:  boolean, default False 
    '''

    use_local_location = None
    '''Bone location is set in local space 

    :type:  boolean, default False 
    '''

    use_relative_parent = None
    '''Object children will use relative transform, like deform 

    :type:  boolean, default False 
    '''

    basename = None
    '''The name of this bone before any ‘.’ character (readonly) '''

    center = None
    '''The midpoint between the head and the tail. (readonly) '''

    children = None
    '''A list of all the bones children. (readonly) '''

    children_recursive = None
    '''A list of all children from this bone. (readonly) '''

    children_recursive_basename = None
    '''Returns a chain of children with the same base name as this bone. Only direct chains are supported, forks caused by multiple children with matching base names will terminate the function and not be returned. (readonly) '''

    length = None
    '''The distance from head to tail, when set the head is moved to fit the length. '''

    parent_recursive = None
    '''A list of parents, starting with the immediate parent (readonly) '''

    vector = None
    '''The direction this bone is pointing. Utility function for (tail - head) (readonly) '''

    x_axis = None
    '''Vector pointing down the x-axis of the bone. (readonly) '''

    y_axis = None
    '''Vector pointing down the y-axis of the bone. (readonly) '''

    z_axis = None
    '''Vector pointing down the z-axis of the bone. (readonly) '''

    def evaluate_envelope(self, point):
        '''Calculate bone envelope at given point 

        :param point: Point, Position in 3d space to evaluate 
        :type point: float array of 3 items in [-inf, inf]
        :rtype: float in [-inf, inf] 
        :return:  Factor, Envelope factor 
        '''
        pass

    def parent_index(self, parent_test):
        '''The same as ‘bone in other_bone.parent_recursive’ but saved generating a list. 

        '''
        pass

    def translate(self, vec):
        '''Utility function to add vec to the head and tail of this bone 

        '''
        pass


class BoneGroup:
    '''Groups of Pose Channels (Bones) '''

    color_set = None
    '''Custom color set to use 

    :type:  enum in [‘DEFAULT’, ‘THEME01’, ‘THEME02’, ‘THEME03’, ‘THEME04’, ‘THEME05’, ‘THEME06’, ‘THEME07’, ‘THEME08’, ‘THEME09’, ‘THEME10’, ‘THEME11’, ‘THEME12’, ‘THEME13’, ‘THEME14’, ‘THEME15’, ‘THEME16’, ‘THEME17’, ‘THEME18’, ‘THEME19’, ‘THEME20’, ‘CUSTOM’], default ‘DEFAULT’ 
    '''

    colors = None
    '''Copy of the colors associated with the group’s color set 

    :type:  ThemeBoneColorSet, (readonly, never None) 
    '''

    is_custom_color_set = None
    '''Color set is user-defined instead of a fixed theme color set 

    :type:  boolean, default False, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''


class BoneGroups:
    '''Collection of bone groups '''

    active = None
    '''Active bone group for this pose 

    :type:  BoneGroup 
    '''

    active_index = None
    '''Active index in bone groups array 

    :type:  int in [0, inf], default 0 
    '''

    def new(self, name="Group"):
        '''Add a new bone group to the object 

        :param name: Name of the new group 
        :type name: string, (optional, never None)
        :rtype: BoneGroup 
        :return:  New bone group 
        '''
        pass

    def remove(self, group):
        '''Remove a bone group from this object 

        :param group: Removed bone group 
        :type group: BoneGroup, (never None)
        '''
        pass


class BoolProperty:
    '''RNA boolean property definition '''

    array_length = None
    '''Maximum length of the array, 0 means unlimited 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    default = None
    '''Default value for this number 

    :type:  boolean, default False, (readonly) 
    '''

    default_array = None
    '''Default value for this array 

    :type:  boolean array of 3 items, default (False, False, False), (readonly) 
    '''

    is_array = None
    '''

    :type:  boolean, default False, (readonly) 
    '''


class BooleanModifier:
    '''Boolean operations modifier '''

    double_threshold = None
    '''Threshold for checking overlapping geometry 

    :type:  float in [0, 1], default 0.0 
    '''

    object = None
    '''Mesh object to use for Boolean operation 

    :type:  Object 
    '''

    operation = None
    '''

    :type:  enum in [‘INTERSECT’, ‘UNION’, ‘DIFFERENCE’], default ‘INTERSECT’ 
    '''

    solver = None
    '''

    :type:  enum in [‘BMESH’, ‘CARVE’], default ‘CARVE’ 
    '''


class BrightContrastModifier:
    '''Bright/contrast modifier data for sequence strip '''

    bright = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    contrast = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''


class Brush:
    '''Brush data-block for storing brush settings for painting and sculpting '''

    auto_smooth_factor = None
    '''Amount of smoothing to automatically apply to each stroke 

    :type:  float in [0, 1], default 0.0 
    '''

    blend = None
    '''Brush blending mode 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUB’, ‘MUL’, ‘LIGHTEN’, ‘DARKEN’, ‘ERASE_ALPHA’, ‘ADD_ALPHA’, ‘OVERLAY’, ‘HARDLIGHT’, ‘COLORBURN’, ‘LINEARBURN’, ‘COLORDODGE’, ‘SCREEN’, ‘SOFTLIGHT’, ‘PINLIGHT’, ‘VIVIDLIGHT’, ‘LINEARLIGHT’, ‘DIFFERENCE’, ‘EXCLUSION’, ‘HUE’, ‘SATURATION’, ‘LUMINOSITY’, ‘COLOR’], default ‘MIX’ 
    '''

    blur_kernel_radius = None
    '''Radius of kernel used for soften and sharpen in pixels 

    :type:  int in [1, 10000], default 0 
    '''

    blur_mode = None
    '''

    :type:  enum in [‘BOX’, ‘GAUSSIAN’], default ‘GAUSSIAN’ 
    '''

    brush_capabilities = None
    '''Brush’s capabilities 

    :type:  BrushCapabilities, (readonly, never None) 
    '''

    clone_alpha = None
    '''Opacity of clone image display 

    :type:  float in [0, 1], default 0.0 
    '''

    clone_image = None
    '''Image for clone tool 

    :type:  Image 
    '''

    clone_offset = None
    '''

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    color = None
    '''

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    crease_pinch_factor = None
    '''How much the crease brush pinches 

    :type:  float in [0, 1], default 0.666667 
    '''

    cursor_color_add = None
    '''Color of cursor when adding 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    cursor_color_subtract = None
    '''Color of cursor when subtracting 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    cursor_overlay_alpha = None
    '''

    :type:  int in [0, 100], default 0 
    '''

    curve = None
    '''Editable falloff curve 

    :type:  CurveMapping, (readonly, never None) 
    '''

    direction = None
    '''

    :type:  enum in [‘ADD’, ‘SUBTRACT’], default ‘ADD’ 
    '''

    fill_threshold = None
    '''Threshold above which filling is not propagated 

    :type:  float in [0, 100], default 0.0 
    '''

    grad_spacing = None
    '''Spacing before brush gradient goes full circle 

    :type:  int in [1, 10000], default 0 
    '''

    gradient = None
    '''

    :type:  ColorRamp, (readonly) 
    '''

    gradient_fill_mode = None
    '''

    :type:  enum in [‘LINEAR’, ‘RADIAL’], default ‘LINEAR’ 
    '''

    gradient_stroke_mode = None
    '''

    :type:  enum in [‘PRESSURE’, ‘SPACING_REPEAT’, ‘SPACING_CLAMP’], default ‘PRESSURE’ 
    '''

    height = None
    '''Affectable height of brush (layer height for layer tool, i.e.) 

    :type:  float in [0, 1], default 0.5 
    '''

    icon_filepath = None
    '''File path to brush icon 

    :type:  string, default “”, (never None) 
    '''

    image_paint_capabilities = None
    '''Brush’s capabilities in image paint mode 

    :type:  ImapaintToolCapabilities, (readonly, never None) 
    '''

    image_tool = None
    '''

    :type:  enum in [‘DRAW’, ‘SOFTEN’, ‘SMEAR’, ‘CLONE’, ‘FILL’, ‘MASK’], default ‘DRAW’ 
    '''

    jitter = None
    '''Jitter the position of the brush while painting 

    :type:  float in [0, 1000], default 0.0 
    '''

    jitter_absolute = None
    '''Jitter the position of the brush in pixels while painting 

    :type:  int in [0, 1000000], default 0 
    '''

    mask_overlay_alpha = None
    '''

    :type:  int in [0, 100], default 0 
    '''

    mask_stencil_dimension = None
    '''Dimensions of mask stencil in viewport 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    mask_stencil_pos = None
    '''Position of mask stencil in viewport 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    mask_texture = None
    '''

    :type:  Texture 
    '''

    mask_texture_slot = None
    '''

    :type:  BrushTextureSlot, (readonly) 
    '''

    mask_tool = None
    '''

    :type:  enum in [‘DRAW’, ‘SMOOTH’], default ‘DRAW’ 
    '''

    normal_weight = None
    '''How much grab will pull vertexes out of surface during a grab 

    :type:  float in [0, 1], default 0.0 
    '''

    paint_curve = None
    '''Active Paint Curve 

    :type:  PaintCurve 
    '''

    plane_offset = None
    '''Adjust plane on which the brush acts towards or away from the object surface 

    :type:  float in [-2, 2], default 0.0 
    '''

    plane_trim = None
    '''If a vertex is further away from offset plane than this, then it is not affected 

    :type:  float in [0, 1], default 0.5 
    '''

    rake_factor = None
    '''How much grab will follow cursor rotation 

    :type:  float in [0, 10], default 0.0 
    '''

    rate = None
    '''Interval between paints for Airbrush 

    :type:  float in [0.0001, 10000], default 0.0 
    '''

    sculpt_capabilities = None
    '''Brush’s capabilities in sculpt mode 

    :type:  SculptToolCapabilities, (readonly, never None) 
    '''

    sculpt_plane = None
    '''

    :type:  enum in [‘AREA’, ‘VIEW’, ‘X’, ‘Y’, ‘Z’], default ‘AREA’ 
    '''

    sculpt_tool = None
    '''

    :type:  enum in [‘BLOB’, ‘CLAY’, ‘CLAY_STRIPS’, ‘CREASE’, ‘DRAW’, ‘FILL’, ‘FLATTEN’, ‘GRAB’, ‘INFLATE’, ‘LAYER’, ‘MASK’, ‘NUDGE’, ‘PINCH’, ‘ROTATE’, ‘SCRAPE’, ‘SIMPLIFY’, ‘SMOOTH’, ‘SNAKE_HOOK’, ‘THUMB’], default ‘BLOB’ 
    '''

    secondary_color = None
    '''

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    sharp_threshold = None
    '''Threshold below which, no sharpening is done 

    :type:  float in [0, 100], default 0.0 
    '''

    size = None
    '''Radius of the brush in pixels 

    :type:  int in [1, 5000], default 0 
    '''

    smooth_stroke_factor = None
    '''Higher values give a smoother stroke 

    :type:  float in [0.5, 0.99], default 0.0 
    '''

    smooth_stroke_radius = None
    '''Minimum distance from last point before stroke continues 

    :type:  int in [10, 200], default 0 
    '''

    spacing = None
    '''Spacing between brush daubs as a percentage of brush diameter 

    :type:  int in [1, 1000], default 0 
    '''

    stencil_dimension = None
    '''Dimensions of stencil in viewport 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    stencil_pos = None
    '''Position of stencil in viewport 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    strength = None
    '''How powerful the effect of the brush is when applied 

    :type:  float in [0, 10], default 0.5 
    '''

    stroke_method = None
    '''

    :type:  enum in [‘DOTS’, ‘DRAG_DOT’, ‘SPACE’, ‘AIRBRUSH’, ‘ANCHORED’, ‘LINE’, ‘CURVE’], default ‘DOTS’ 
    '''

    texture = None
    '''

    :type:  Texture 
    '''

    texture_overlay_alpha = None
    '''

    :type:  int in [0, 100], default 0 
    '''

    texture_sample_bias = None
    '''Value added to texture samples 

    :type:  float in [-1, 1], default 0.0 
    '''

    texture_slot = None
    '''

    :type:  BrushTextureSlot, (readonly) 
    '''

    unprojected_radius = None
    '''Radius of brush in Blender units 

    :type:  float in [0.001, inf], default 0.0 
    '''

    use_accumulate = None
    '''Accumulate stroke daubs on top of each other 

    :type:  boolean, default False 
    '''

    use_adaptive_space = None
    '''Space daubs according to surface orientation instead of screen space 

    :type:  boolean, default False 
    '''

    use_airbrush = None
    '''Keep applying paint effect while holding mouse (spray) 

    :type:  boolean, default False 
    '''

    use_alpha = None
    '''When this is disabled, lock alpha while painting 

    :type:  boolean, default False 
    '''

    use_anchor = None
    '''Keep the brush anchored to the initial location 

    :type:  boolean, default False 
    '''

    use_cursor_overlay = None
    '''Show cursor in viewport 

    :type:  boolean, default False 
    '''

    use_cursor_overlay_override = None
    '''Don’t show overlay during a stroke 

    :type:  boolean, default False 
    '''

    use_curve = None
    '''Define the stroke curve with a bezier curve. Dabs are separated according to spacing 

    :type:  boolean, default False 
    '''

    use_custom_icon = None
    '''Set the brush icon from an image file 

    :type:  boolean, default False 
    '''

    use_edge_to_edge = None
    '''Drag anchor brush from edge-to-edge 

    :type:  boolean, default False 
    '''

    use_frontface = None
    '''Brush only affects vertexes that face the viewer 

    :type:  boolean, default False 
    '''

    use_gradient = None
    '''Use Gradient by utilizing a sampling method 

    :type:  boolean, default False 
    '''

    use_inverse_smooth_pressure = None
    '''Lighter pressure causes more smoothing to be applied 

    :type:  boolean, default False 
    '''

    use_line = None
    '''Draw a line with dabs separated according to spacing 

    :type:  boolean, default False 
    '''

    use_locked_size = None
    '''When locked brush stays same size relative to object; when unlocked brush size is given in pixels 

    :type:  boolean, default False 
    '''

    use_offset_pressure = None
    '''Enable tablet pressure sensitivity for offset 

    :type:  boolean, default False 
    '''

    use_original_normal = None
    '''When locked keep using normal of surface where stroke was initiated 

    :type:  boolean, default False 
    '''

    use_paint_image = None
    '''Use this brush in texture paint mode 

    :type:  boolean, default False 
    '''

    use_paint_sculpt = None
    '''Use this brush in sculpt mode 

    :type:  boolean, default False 
    '''

    use_paint_vertex = None
    '''Use this brush in vertex paint mode 

    :type:  boolean, default False 
    '''

    use_paint_weight = None
    '''Use this brush in weight paint mode 

    :type:  boolean, default False 
    '''

    use_persistent = None
    '''Sculpt on a persistent layer of the mesh 

    :type:  boolean, default False 
    '''

    use_plane_trim = None
    '''Enable Plane Trim 

    :type:  boolean, default False 
    '''

    use_pressure_jitter = None
    '''Enable tablet pressure sensitivity for jitter 

    :type:  boolean, default False 
    '''

    use_pressure_masking = None
    '''Pen pressure makes texture influence smaller 

    :type:  enum in [‘NONE’, ‘RAMP’, ‘CUTOFF’], default ‘NONE’ 
    '''

    use_pressure_size = None
    '''Enable tablet pressure sensitivity for size 

    :type:  boolean, default False 
    '''

    use_pressure_spacing = None
    '''Enable tablet pressure sensitivity for spacing 

    :type:  boolean, default False 
    '''

    use_pressure_strength = None
    '''Enable tablet pressure sensitivity for strength 

    :type:  boolean, default False 
    '''

    use_primary_overlay = None
    '''Show texture in viewport 

    :type:  boolean, default False 
    '''

    use_primary_overlay_override = None
    '''Don’t show overlay during a stroke 

    :type:  boolean, default False 
    '''

    use_relative_jitter = None
    '''Jittering happens in screen space, not relative to brush size 

    :type:  boolean, default False 
    '''

    use_restore_mesh = None
    '''Allow a single dot to be carefully positioned 

    :type:  boolean, default False 
    '''

    use_secondary_overlay = None
    '''Show texture in viewport 

    :type:  boolean, default False 
    '''

    use_secondary_overlay_override = None
    '''Don’t show overlay during a stroke 

    :type:  boolean, default False 
    '''

    use_smooth_stroke = None
    '''Brush lags behind mouse and follows a smoother path 

    :type:  boolean, default False 
    '''

    use_space = None
    '''Limit brush application to the distance specified by spacing 

    :type:  boolean, default False 
    '''

    use_space_attenuation = None
    '''Automatically adjust strength to give consistent results for different spacings 

    :type:  boolean, default False 
    '''

    vertex_tool = None
    '''Brush blending mode 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUB’, ‘MUL’, ‘BLUR’, ‘LIGHTEN’, ‘DARKEN’], default ‘MIX’ 
    '''

    weight = None
    '''Vertex weight when brush is applied 

    :type:  float in [0, 1], default 1.0 
    '''


class BrushCapabilities:
    '''Read-only indications of which brush operations are supported by the current brush '''

    has_overlay = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    has_random_texture_angle = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    has_smooth_stroke = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    has_spacing = None
    '''

    :type:  boolean, default False, (readonly) 
    '''


class BrushTextureSlot:
    '''Texture slot for textures in a Brush data-block '''

    angle = None
    '''Brush texture rotation 

    :type:  float in [0, 6.28319], default 0.0 
    '''

    has_random_texture_angle = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    has_texture_angle = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    has_texture_angle_source = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    map_mode = None
    '''

    :type:  enum in [‘VIEW_PLANE’, ‘AREA_PLANE’, ‘TILED’, ‘3D’, ‘RANDOM’, ‘STENCIL’], default ‘VIEW_PLANE’ 
    '''

    mask_map_mode = None
    '''

    :type:  enum in [‘VIEW_PLANE’, ‘TILED’, ‘RANDOM’, ‘STENCIL’], default ‘VIEW_PLANE’ 
    '''

    random_angle = None
    '''Brush texture random angle 

    :type:  float in [0, 6.28319], default 0.0 
    '''

    tex_paint_map_mode = None
    '''

    :type:  enum in [‘VIEW_PLANE’, ‘TILED’, ‘3D’, ‘RANDOM’, ‘STENCIL’], default ‘VIEW_PLANE’ 
    '''

    use_rake = None
    '''

    :type:  boolean, default False 
    '''

    use_random = None
    '''

    :type:  boolean, default False 
    '''


class BuildModifier:
    '''Build effect modifier '''

    frame_duration = None
    '''Total time the build effect requires 

    :type:  float in [1, 500000], default 0.0 
    '''

    frame_start = None
    '''Start frame of the effect 

    :type:  float in [-500000, 500000], default 0.0 
    '''

    seed = None
    '''Seed for random if used 

    :type:  int in [1, 500000], default 0 
    '''

    use_random_order = None
    '''Randomize the faces or edges during build 

    :type:  boolean, default False 
    '''

    use_reverse = None
    '''Deconstruct the mesh instead of building it 

    :type:  boolean, default False 
    '''


class CLIP_UL_tracking_objects:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class CYCLES:
    def bake(self, scene, obj, pass_type, pass_filter, object_id, pixel_array,
             num_pixels, depth, result):
        '''

        '''
        pass

    def render(self, scene):
        '''

        '''
        pass

    def update(self, data, scene):
        '''

        '''
        pass

    def update_script_node(self, node):
        '''

        '''
        pass

    def view_draw(self, context):
        '''

        '''
        pass

    def view_update(self, context):
        '''

        '''
        pass


class CacheFile:
    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    filepath = None
    '''Path to external displacements file 

    :type:  string, default “”, (never None) 
    '''

    forward_axis = None
    '''

    :type:  enum in [‘POS_X’, ‘POS_Y’, ‘POS_Z’, ‘NEG_X’, ‘NEG_Y’, ‘NEG_Z’], default ‘POS_X’ 
    '''

    frame = None
    '''The time to use for looking up the data in the cache file, or to determine which file to use in a file sequence 

    :type:  float in [-500000, 500000], default 0.0 
    '''

    is_sequence = None
    '''Whether the cache is separated in a series of files 

    :type:  boolean, default False 
    '''

    object_paths = None
    '''Paths of the objects inside the Alembic archive 

    :type:  AlembicObjectPaths bpy_prop_collection of AlembicObjectPath, (readonly) 
    '''

    override_frame = None
    '''Whether to use a custom frame for looking up data in the cache file, instead of using the current scene frame 

    :type:  boolean, default False 
    '''

    scale = None
    '''Value by which to enlarge or shrink the object with respect to the world’s origin (only applicable through a Transform Cache constraint) 

    :type:  float in [0.0001, 1000], default 0.0 
    '''

    up_axis = None
    '''

    :type:  enum in [‘POS_X’, ‘POS_Y’, ‘POS_Z’, ‘NEG_X’, ‘NEG_Y’, ‘NEG_Z’], default ‘POS_X’ 
    '''


class Camera:
    '''Camera data-block for storing camera settings '''

    angle = None
    '''Camera lens field of view 

    :type:  float in [0.00640536, 3.01675], default 0.0 
    '''

    angle_x = None
    '''Camera lens horizontal field of view 

    :type:  float in [0.00640536, 3.01675], default 0.0 
    '''

    angle_y = None
    '''Camera lens vertical field of view 

    :type:  float in [0.00640536, 3.01675], default 0.0 
    '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    clip_end = None
    '''Camera far clipping distance 

    :type:  float in [1e-06, inf], default 0.0 
    '''

    clip_start = None
    '''Camera near clipping distance 

    :type:  float in [1e-06, inf], default 0.0 
    '''

    cycles = None
    '''Cycles camera settings 

    :type:  CyclesCameraSettings, (readonly) 
    '''

    dof_distance = None
    '''Distance to the focus point for depth of field 

    :type:  float in [0, inf], default 0.0 
    '''

    dof_object = None
    '''Use this object to define the depth of field focal point 

    :type:  Object 
    '''

    draw_size = None
    '''Apparent size of the Camera object in the 3D View 

    :type:  float in [0.01, 1000], default 0.0 
    '''

    gpu_dof = None
    '''

    :type:  GPUDOFSettings, (readonly) 
    '''

    lens = None
    '''Perspective Camera lens value in millimeters 

    :type:  float in [1, inf], default 0.0 
    '''

    lens_unit = None
    '''Unit to edit lens in for the user interface 

    :type:  enum in [‘MILLIMETERS’, ‘FOV’], default ‘MILLIMETERS’ 
    '''

    ortho_scale = None
    '''Orthographic Camera scale (similar to zoom) 

    :type:  float in [0, inf], default 0.0 
    '''

    passepartout_alpha = None
    '''Opacity (alpha) of the darkened overlay in Camera view 

    :type:  float in [0, 1], default 0.0 
    '''

    sensor_fit = None
    '''Method to fit image and field of view angle inside the sensor 

    :type:  enum in [‘AUTO’, ‘HORIZONTAL’, ‘VERTICAL’], default ‘AUTO’ 
    '''

    sensor_height = None
    '''Vertical size of the image sensor area in millimeters 

    :type:  float in [1, inf], default 0.0 
    '''

    sensor_width = None
    '''Horizontal size of the image sensor area in millimeters 

    :type:  float in [1, inf], default 0.0 
    '''

    shift_x = None
    '''Camera horizontal shift 

    :type:  float in [-10, 10], default 0.0 
    '''

    shift_y = None
    '''Camera vertical shift 

    :type:  float in [-10, 10], default 0.0 
    '''

    show_guide = None
    '''Draw overlay 

    :type:  enum set in {‘CENTER’, ‘CENTER_DIAGONAL’, ‘THIRDS’, ‘GOLDEN’, ‘GOLDEN_TRIANGLE_A’, ‘GOLDEN_TRIANGLE_B’, ‘HARMONY_TRIANGLE_A’, ‘HARMONY_TRIANGLE_B’}, default {‘CENTER’} 
    '''

    show_limits = None
    '''Draw the clipping range and focus point on the camera 

    :type:  boolean, default False 
    '''

    show_mist = None
    '''Draw a line from the Camera to indicate the mist area 

    :type:  boolean, default False 
    '''

    show_name = None
    '''Show the active Camera’s name in Camera view 

    :type:  boolean, default False 
    '''

    show_passepartout = None
    '''Show a darkened overlay outside the image area in Camera view 

    :type:  boolean, default False 
    '''

    show_safe_areas = None
    '''Show TV title safe and action safe areas in Camera view 

    :type:  boolean, default False 
    '''

    show_safe_center = None
    '''Show safe areas to fit content in a different aspect ratio 

    :type:  boolean, default False 
    '''

    show_sensor = None
    '''Show sensor size (film gate) in Camera view 

    :type:  boolean, default False 
    '''

    stereo = None
    '''

    :type:  CameraStereoData, (readonly, never None) 
    '''

    type = None
    '''Camera types 

    :type:  enum in [‘PERSP’, ‘ORTHO’, ‘PANO’], default ‘PERSP’ 
    '''

    def view_frame(self, scene=None):
        '''Return 4 points for the cameras frame (before object transformation) 

        :param scene: Scene to use for aspect calculation, when omitted 1:1 aspect is used 
        :type scene: Scene, (optional)
        '''
        pass


class CameraActuator:
    axis = None
    '''Axis the Camera will try to get behind 

    :type:  enum in [‘POS_X’, ‘POS_Y’, ‘NEG_X’, ‘NEG_Y’], default ‘POS_X’ 
    '''

    damping = None
    '''Strength of the constraint that drives the camera behind the target 

    :type:  float in [0, 10], default 0.0 
    '''

    height = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    max = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    min = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    object = None
    '''Look at this Object 

    :type:  Object 
    '''


class CameraSolverConstraint:
    '''Lock motion to the reconstructed camera movement '''

    clip = None
    '''Movie Clip to get tracking data from 

    :type:  MovieClip 
    '''

    use_active_clip = None
    '''Use active clip defined in scene 

    :type:  boolean, default False 
    '''


class CameraStereoData:
    '''Stereoscopy settings for a Camera data-block '''

    convergence_distance = None
    '''The converge point for the stereo cameras (often the distance between a projector and the projection screen) 

    :type:  float in [1e-05, inf], default 0.0 
    '''

    convergence_mode = None
    '''

    :type:  enum in [‘OFFAXIS’, ‘PARALLEL’, ‘TOE’], default ‘OFFAXIS’ 
    '''

    interocular_distance = None
    '''Set the distance between the eyes - the stereo plane distance / 30 should be fine 

    :type:  float in [0, inf], default 0.0 
    '''

    pivot = None
    '''

    :type:  enum in [‘LEFT’, ‘RIGHT’, ‘CENTER’], default ‘LEFT’ 
    '''

    pole_merge_angle_from = None
    '''Angle at which interocular distance starts to fade to 0 

    :type:  float in [0, 1.5708], default 0.0 
    '''

    pole_merge_angle_to = None
    '''Angle at which interocular distance is 0 

    :type:  float in [0, 1.5708], default 0.0 
    '''

    use_pole_merge = None
    '''Fade interocular distance to 0 after the given cutoff angle 

    :type:  boolean, default False 
    '''

    use_spherical_stereo = None
    '''Render every pixel rotating the camera around the middle of the interocular distance 

    :type:  boolean, default False 
    '''


class CastModifier:
    '''Modifier to cast to other shapes '''

    cast_type = None
    '''Target object shape 

    :type:  enum in [‘SPHERE’, ‘CYLINDER’, ‘CUBOID’], default ‘SPHERE’ 
    '''

    factor = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    object = None
    '''Control object: if available, its location determines the center of the effect 

    :type:  Object 
    '''

    radius = None
    '''Only deform vertices within this distance from the center of the effect (leave as 0 for infinite.) 

    :type:  float in [0, inf], default 0.0 
    '''

    size = None
    '''Size of projection shape (leave as 0 for auto) 

    :type:  float in [0, inf], default 0.0 
    '''

    use_radius_as_size = None
    '''Use radius as size of projection shape (0 = auto) 

    :type:  boolean, default False 
    '''

    use_transform = None
    '''Use object transform to control projection shape 

    :type:  boolean, default False 
    '''

    use_x = None
    '''

    :type:  boolean, default False 
    '''

    use_y = None
    '''

    :type:  boolean, default False 
    '''

    use_z = None
    '''

    :type:  boolean, default False 
    '''

    vertex_group = None
    '''Vertex group name 

    :type:  string, default “”, (never None) 
    '''


class ChannelDriverVariables:
    '''Collection of channel driver Variables '''

    def new(self):
        '''Add a new variable for the driver 

        :rtype: DriverVariable 
        :return:  Newly created Driver Variable 
        '''
        pass

    def remove(self, variable):
        '''Remove an existing variable from the driver 

        :param variable: Variable to remove from the driver 
        :type variable: DriverVariable, (never None)
        '''
        pass


class ChildOfConstraint:
    '''Create constraint-based parent-child relationship '''

    inverse_matrix = None
    '''Transformation matrix to apply before 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    use_location_x = None
    '''Use X Location of Parent 

    :type:  boolean, default False 
    '''

    use_location_y = None
    '''Use Y Location of Parent 

    :type:  boolean, default False 
    '''

    use_location_z = None
    '''Use Z Location of Parent 

    :type:  boolean, default False 
    '''

    use_rotation_x = None
    '''Use X Rotation of Parent 

    :type:  boolean, default False 
    '''

    use_rotation_y = None
    '''Use Y Rotation of Parent 

    :type:  boolean, default False 
    '''

    use_rotation_z = None
    '''Use Z Rotation of Parent 

    :type:  boolean, default False 
    '''

    use_scale_x = None
    '''Use X Scale of Parent 

    :type:  boolean, default False 
    '''

    use_scale_y = None
    '''Use Y Scale of Parent 

    :type:  boolean, default False 
    '''

    use_scale_z = None
    '''Use Z Scale of Parent 

    :type:  boolean, default False 
    '''


class ChildParticle:
    '''Child particle interpolated from simulated or edited particles '''

    pass


class ClampToConstraint:
    '''Constrain an object’s location to the nearest point along the target path '''

    main_axis = None
    '''Main axis of movement 

    :type:  enum in [‘CLAMPTO_AUTO’, ‘CLAMPTO_X’, ‘CLAMPTO_Y’, ‘CLAMPTO_Z’], default ‘CLAMPTO_AUTO’ 
    '''

    target = None
    '''Target Object (Curves only) 

    :type:  Object 
    '''

    use_cyclic = None
    '''Treat curve as cyclic curve (no clamping to curve bounding box) 

    :type:  boolean, default False 
    '''


class ClothCollisionSettings:
    '''Cloth simulation settings for self collision and collision with other objects '''

    collision_quality = None
    '''How many collision iterations should be done. (higher is better quality but slower) 

    :type:  int in [1, 32767], default 0 
    '''

    damping = None
    '''Amount of velocity lost on collision 

    :type:  float in [0, 1], default 1.0 
    '''

    distance_min = None
    '''Minimum distance between collision objects before collision response takes in 

    :type:  float in [0.001, 1], default 0.0 
    '''

    distance_repel = None
    '''Maximum distance to apply repulsion force, must be greater than minimum distance 

    :type:  float in [0.001, 10], default 0.005 
    '''

    friction = None
    '''Friction force if a collision happened (higher = less movement) 

    :type:  float in [0, 80], default 0.0 
    '''

    group = None
    '''Limit colliders to this Group 

    :type:  Group 
    '''

    repel_force = None
    '''Repulsion force to apply on cloth when close to colliding 

    :type:  float in [0, 20], default 1.0 
    '''

    self_collision_quality = None
    '''How many self collision iterations should be done (higher is better quality but slower) 

    :type:  int in [1, 32767], default 0 
    '''

    self_distance_min = None
    '''0.5 means no distance at all, 1.0 is maximum distance 

    :type:  float in [0.5, 1], default 0.0 
    '''

    self_friction = None
    '''Friction/damping with self contact 

    :type:  float in [0, 80], default 0.0 
    '''

    use_collision = None
    '''Enable collisions with other objects 

    :type:  boolean, default False 
    '''

    use_self_collision = None
    '''Enable self collisions 

    :type:  boolean, default False 
    '''

    vertex_group_self_collisions = None
    '''Vertex group to define vertices which are not used during self collisions 

    :type:  string, default “”, (never None) 
    '''


class ClothModifier:
    '''Cloth simulation modifier '''

    collision_settings = None
    '''

    :type:  ClothCollisionSettings, (readonly, never None) 
    '''

    hair_grid_max = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0), (readonly) 
    '''

    hair_grid_min = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0), (readonly) 
    '''

    hair_grid_resolution = None
    '''

    :type:  int array of 3 items in [-inf, inf], default (0, 0, 0), (readonly) 
    '''

    point_cache = None
    '''

    :type:  PointCache, (readonly, never None) 
    '''

    settings = None
    '''

    :type:  ClothSettings, (readonly, never None) 
    '''

    solver_result = None
    '''

    :type:  ClothSolverResult, (readonly) 
    '''


class ClothSettings:
    '''Cloth simulation settings for an object '''

    air_damping = None
    '''Air has normally some thickness which slows falling things down 

    :type:  float in [0, 10], default 0.0 
    '''

    bending_damping = None
    '''Damping of bending motion 

    :type:  float in [0, 1000], default 0.0 
    '''

    bending_stiffness = None
    '''Wrinkle coefficient (higher = less smaller but more big wrinkles) 

    :type:  float in [0, 10000], default 0.0 
    '''

    bending_stiffness_max = None
    '''Maximum bending stiffness value 

    :type:  float in [0, 10000], default 0.0 
    '''

    collider_friction = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''

    density_strength = None
    '''Influence of target density on the simulation 

    :type:  float in [0, 1], default 0.0 
    '''

    density_target = None
    '''Maximum density of hair 

    :type:  float in [0, 10000], default 0.0 
    '''

    effector_weights = None
    '''

    :type:  EffectorWeights, (readonly) 
    '''

    goal_default = None
    '''Default Goal (vertex target position) value, when no Vertex Group used 

    :type:  float in [0, 1], default 0.0 
    '''

    goal_friction = None
    '''Goal (vertex target position) friction 

    :type:  float in [0, 50], default 0.0 
    '''

    goal_max = None
    '''Goal maximum, vertex group weights are scaled to match this range 

    :type:  float in [0, 1], default 0.0 
    '''

    goal_min = None
    '''Goal minimum, vertex group weights are scaled to match this range 

    :type:  float in [0, 1], default 0.0 
    '''

    goal_spring = None
    '''Goal (vertex target position) spring stiffness 

    :type:  float in [0, 0.999], default 0.0 
    '''

    gravity = None
    '''Gravity or external force vector 

    :type:  float array of 3 items in [-100, 100], default (0.0, 0.0, 0.0) 
    '''

    internal_friction = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''

    mass = None
    '''Mass of cloth material 

    :type:  float in [0, 10], default 0.0 
    '''

    pin_stiffness = None
    '''Pin (vertex target position) spring stiffness 

    :type:  float in [0, 50], default 0.0 
    '''

    quality = None
    '''Quality of the simulation in steps per frame (higher is better quality but slower) 

    :type:  int in [1, inf], default 0 
    '''

    rest_shape_key = None
    '''Shape key to use the rest spring lengths from 

    :type:  ShapeKey 
    '''

    sewing_force_max = None
    '''Maximum sewing force 

    :type:  float in [0, 10000], default 0.0 
    '''

    shrink_max = None
    '''Max amount to shrink cloth by 

    :type:  float in [0, 1], default 0.0 
    '''

    shrink_min = None
    '''Min amount to shrink cloth by 

    :type:  float in [0, 1], default 0.0 
    '''

    spring_damping = None
    '''Damping of cloth velocity (higher = more smooth, less jiggling) 

    :type:  float in [0, 50], default 0.0 
    '''

    structural_stiffness = None
    '''Overall stiffness of structure 

    :type:  float in [0, 10000], default 0.0 
    '''

    structural_stiffness_max = None
    '''Maximum structural stiffness value 

    :type:  float in [0, 10000], default 0.0 
    '''

    time_scale = None
    '''Cloth speed is multiplied by this value 

    :type:  float in [0, inf], default 0.0 
    '''

    use_dynamic_mesh = None
    '''Make simulation respect deformations in the base mesh 

    :type:  boolean, default False 
    '''

    use_pin_cloth = None
    '''Enable pinning of cloth vertices to other objects/positions 

    :type:  boolean, default False 
    '''

    use_sewing_springs = None
    '''Pulls loose edges together 

    :type:  boolean, default False 
    '''

    use_stiffness_scale = None
    '''If enabled, stiffness can be scaled along a weight painted vertex group 

    :type:  boolean, default False 
    '''

    vel_damping = None
    '''Damp velocity to help cloth reach the resting position faster (1.0 = no damping, 0.0 = fully dampened) 

    :type:  float in [0, 1], default 0.0 
    '''

    vertex_group_bending = None
    '''Vertex group for fine control over bending stiffness 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_mass = None
    '''Vertex Group for pinning of vertices 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_shrink = None
    '''Vertex Group for shrinking cloth 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_structural_stiffness = None
    '''Vertex group for fine control over structural stiffness 

    :type:  string, default “”, (never None) 
    '''

    voxel_cell_size = None
    '''Size of the voxel grid cells for interaction effects 

    :type:  float in [0.0001, 10000], default 0.1 
    '''


class ClothSolverResult:
    '''Result of cloth solver iteration '''

    avg_error = None
    '''Average error during substeps 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    avg_iterations = None
    '''Average iterations during substeps 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    max_error = None
    '''Maximum error during substeps 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    max_iterations = None
    '''Maximum iterations during substeps 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    min_error = None
    '''Minimum error during substeps 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    min_iterations = None
    '''Minimum iterations during substeps 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    status = None
    '''Status of the solver iteration 

    :type:  enum set in {‘SUCCESS’, ‘NUMERICAL_ISSUE’, ‘NO_CONVERGENCE’, ‘INVALID_INPUT’}, default {‘SUCCESS’}, (readonly) 
    '''


class CloudsTexture:
    '''Procedural noise texture '''

    cloud_type = None
    '''Determine whether Noise returns grayscale or RGB values 

    :type:  enum in [‘GRAYSCALE’, ‘COLOR’], default ‘GRAYSCALE’ 
    '''

    nabla = None
    '''Size of derivative offset used for calculating normal 

    :type:  float in [0.001, 0.1], default 0.0 
    '''

    noise_basis = None
    '''Noise basis used for turbulence 

    :type:  enum in [‘BLENDER_ORIGINAL’, ‘ORIGINAL_PERLIN’, ‘IMPROVED_PERLIN’, ‘VORONOI_F1’, ‘VORONOI_F2’, ‘VORONOI_F3’, ‘VORONOI_F4’, ‘VORONOI_F2_F1’, ‘VORONOI_CRACKLE’, ‘CELL_NOISE’], default ‘BLENDER_ORIGINAL’ 
    '''

    noise_depth = None
    '''Depth of the cloud calculation 

    :type:  int in [0, 30], default 0 
    '''

    noise_scale = None
    '''Scaling for noise input 

    :type:  float in [0.0001, inf], default 0.0 
    '''

    noise_type = None
    '''

    :type:  enum in [‘SOFT_NOISE’, ‘HARD_NOISE’], default ‘SOFT_NOISE’ 
    '''

    users_material = None
    '''Materials that use this texture (readonly) '''

    users_object_modifier = None
    '''Object modifiers that use this texture (readonly) '''


class CollectionProperty:
    '''RNA collection property to define lists, arrays and mappings '''

    fixed_type = None
    '''Fixed pointer type, empty if variable type 

    :type:  Struct, (readonly) 
    '''


class CollisionModifier:
    '''Collision modifier defining modifier stack position used for collision '''

    settings = None
    '''

    :type:  CollisionSettings, (readonly, never None) 
    '''


class CollisionSensor:
    '''Sensor to detect objects colliding with the current object, with more settings than the Touch sensor '''

    material = None
    '''Only look for objects with this material (blank = all objects) 

    :type:  string, default “”, (never None) 
    '''

    property = None
    '''Only look for objects with this property (blank = all objects) 

    :type:  string, default “”, (never None) 
    '''

    use_material = None
    '''Toggle collision on material or property 

    :type:  boolean, default False 
    '''

    use_pulse = None
    '''Change to the set of colliding objects generates pulse 

    :type:  boolean, default False 
    '''


class CollisionSettings:
    '''Collision settings for object in physics simulation '''

    absorption = None
    '''How much of effector force gets lost during collision with this object (in percent) 

    :type:  float in [0, 1], default 0.0 
    '''

    damping = None
    '''Amount of damping during collision 

    :type:  float in [0, 1], default 0.0 
    '''

    damping_factor = None
    '''Amount of damping during particle collision 

    :type:  float in [0, 1], default 0.0 
    '''

    damping_random = None
    '''Random variation of damping 

    :type:  float in [0, 1], default 0.0 
    '''

    friction_factor = None
    '''Amount of friction during particle collision 

    :type:  float in [0, 1], default 0.0 
    '''

    friction_random = None
    '''Random variation of friction 

    :type:  float in [0, 1], default 0.0 
    '''

    permeability = None
    '''Chance that the particle will pass through the mesh 

    :type:  float in [0, 1], default 0.0 
    '''

    stickiness = None
    '''Amount of stickiness to surface collision 

    :type:  float in [0, 10], default 0.0 
    '''

    thickness_inner = None
    '''Inner face thickness (only used by softbodies) 

    :type:  float in [0.001, 1], default 0.0 
    '''

    thickness_outer = None
    '''Outer face thickness 

    :type:  float in [0.001, 1], default 0.0 
    '''

    use = None
    '''Enable this objects as a collider for physics systems 

    :type:  boolean, default False 
    '''

    use_particle_kill = None
    '''Kill collided particles 

    :type:  boolean, default False 
    '''


class ColorBalanceModifier:
    '''Color balance modifier for sequence strip '''

    color_balance = None
    '''

    :type:  SequenceColorBalanceData, (readonly) 
    '''

    color_multiply = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''


class ColorManagedDisplaySettings:
    '''Color management specific to display device '''

    display_device = None
    '''Display device name 

    :type:  enum in [‘DEFAULT’], default ‘DEFAULT’ 
    '''


class ColorManagedInputColorspaceSettings:
    '''Input color space settings '''

    name = None
    '''Color space of the image or movie on disk 

    :type:  enum in [‘Linear’, ‘Linear ACES’, ‘Non-Color’, ‘Raw’, ‘sRGB’, ‘VD16’, ‘XYZ’], default ‘NONE’ 
    '''


class ColorManagedSequencerColorspaceSettings:
    '''Input color space settings '''

    name = None
    '''Color space that the sequencer operates in 

    :type:  enum in [‘Linear’, ‘Linear ACES’, ‘Non-Color’, ‘Raw’, ‘sRGB’, ‘VD16’, ‘XYZ’], default ‘NONE’ 
    '''


class ColorManagedViewSettings:
    '''Color management settings used for displaying images on the display '''

    curve_mapping = None
    '''Color curve mapping applied before display transform 

    :type:  CurveMapping, (readonly) 
    '''

    exposure = None
    '''Exposure (stops) applied before display transform 

    :type:  float in [-10, 10], default 0.0 
    '''

    gamma = None
    '''Amount of gamma modification applied after display transform 

    :type:  float in [0, 5], default 1.0 
    '''

    look = None
    '''Additional transform applied before view transform for an artistic needs 

    :type:  enum in [‘NONE’], default ‘NONE’ 
    '''

    use_curve_mapping = None
    '''Use RGB curved for pre-display transformation 

    :type:  boolean, default False 
    '''

    view_transform = None
    '''View used when converting image to a display space 

    :type:  enum in [‘NONE’], default ‘NONE’ 
    '''


class ColorMapping:
    '''Color mapping settings '''

    blend_color = None
    '''Blend color to mix with texture output color 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    blend_factor = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    blend_type = None
    '''Mode used to mix with texture output color 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘SCREEN’, ‘OVERLAY’, ‘DIFFERENCE’, ‘DIVIDE’, ‘DARKEN’, ‘LIGHTEN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    brightness = None
    '''Adjust the brightness of the texture 

    :type:  float in [0, 2], default 0.0 
    '''

    color_ramp = None
    '''

    :type:  ColorRamp, (readonly) 
    '''

    contrast = None
    '''Adjust the contrast of the texture 

    :type:  float in [0, 5], default 0.0 
    '''

    saturation = None
    '''Adjust the saturation of colors in the texture 

    :type:  float in [0, 2], default 0.0 
    '''

    use_color_ramp = None
    '''Toggle color ramp operations 

    :type:  boolean, default False 
    '''


class ColorRamp:
    '''Color ramp mapping a scalar value to a color '''

    color_mode = None
    '''Set color mode to use for interpolation 

    :type:  enum in [‘RGB’, ‘HSV’, ‘HSL’], default ‘RGB’ 
    '''

    elements = None
    '''

    :type:  ColorRampElements bpy_prop_collection of ColorRampElement, (readonly) 
    '''

    hue_interpolation = None
    '''Set color interpolation 

    :type:  enum in [‘NEAR’, ‘FAR’, ‘CW’, ‘CCW’], default ‘NEAR’ 
    '''

    interpolation = None
    '''Set interpolation between color stops 

    :type:  enum in [‘EASE’, ‘CARDINAL’, ‘LINEAR’, ‘B_SPLINE’, ‘CONSTANT’], default ‘LINEAR’ 
    '''

    def evaluate(self, position):
        '''Evaluate ColorRamp 

        :param position: Position, Evaluate ColorRamp at position 
        :type position: float in [0, 1]
        :rtype: float array of 4 items in [-inf, inf] 
        :return:  Color, Color at given position 
        '''
        pass


class ColorRampElement:
    '''Element defining a color at a position in the color ramp '''

    alpha = None
    '''Set alpha of selected color stop 

    :type:  float in [0, inf], default 0.0 
    '''

    color = None
    '''Set color of selected color stop 

    :type:  float array of 4 items in [0, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''

    position = None
    '''Set position of selected color stop 

    :type:  float in [0, 1], default 0.0 
    '''


class ColorRampElements:
    '''Collection of Color Ramp Elements '''

    def new(self, position):
        '''Add element to ColorRamp 

        :param position: Position, Position to add element 
        :type position: float in [0, 1]
        :rtype: ColorRampElement 
        :return:  New element 
        '''
        pass

    def remove(self, element):
        '''Delete element from ColorRamp 

        :param element: Element to remove 
        :type element: ColorRampElement, (never None)
        '''
        pass


class ColorSequence:
    '''Sequence strip creating an image filled with a single color '''

    color = None
    '''

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''


class CompositorNode:
    def tag_need_exec(self):
        '''Tag the node for compositor update 

        '''
        pass

    def update(self):
        '''

        '''
        pass


class CompositorNodeAlphaOver:
    premul = None
    '''Mix Factor 

    :type:  float in [0, 1], default 0.0 
    '''

    use_premultiply = None
    '''

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeBilateralblur:
    iterations = None
    '''

    :type:  int in [1, 128], default 0 
    '''

    sigma_color = None
    '''

    :type:  float in [0.01, 3], default 0.0 
    '''

    sigma_space = None
    '''

    :type:  float in [0.01, 30], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeBlur:
    aspect_correction = None
    '''Type of aspect correction to use 

    :type:  enum in [‘NONE’, ‘Y’, ‘X’], default ‘NONE’ 
    '''

    factor = None
    '''

    :type:  float in [0, 2], default 0.0 
    '''

    factor_x = None
    '''

    :type:  float in [0, 100], default 0.0 
    '''

    factor_y = None
    '''

    :type:  float in [0, 100], default 0.0 
    '''

    filter_type = None
    '''

    :type:  enum in [‘FLAT’, ‘TENT’, ‘QUAD’, ‘CUBIC’, ‘GAUSS’, ‘FAST_GAUSS’, ‘CATROM’, ‘MITCH’], default ‘FLAT’ 
    '''

    size_x = None
    '''

    :type:  int in [0, 2048], default 0 
    '''

    size_y = None
    '''

    :type:  int in [0, 2048], default 0 
    '''

    use_bokeh = None
    '''Use circular filter (slower) 

    :type:  boolean, default False 
    '''

    use_extended_bounds = None
    '''Extend bounds of the input image to fully fit blurred image 

    :type:  boolean, default False 
    '''

    use_gamma_correction = None
    '''Apply filter on gamma corrected values 

    :type:  boolean, default False 
    '''

    use_relative = None
    '''Use relative (percent) values to define blur radius 

    :type:  boolean, default False 
    '''

    use_variable_size = None
    '''Support variable blur per-pixel when using an image for size input 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeBokehBlur:
    blur_max = None
    '''Blur limit, maximum CoC radius 

    :type:  float in [0, 10000], default 0.0 
    '''

    use_extended_bounds = None
    '''Extend bounds of the input image to fully fit blurred image 

    :type:  boolean, default False 
    '''

    use_variable_size = None
    '''Support variable blur per-pixel when using an image for size input 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeBokehImage:
    angle = None
    '''Angle of the bokeh 

    :type:  float in [-12.5664, 12.5664], default 0.0 
    '''

    catadioptric = None
    '''Level of catadioptric of the bokeh 

    :type:  float in [-0, 1], default 0.0 
    '''

    flaps = None
    '''Number of flaps 

    :type:  int in [3, 24], default 5 
    '''

    rounding = None
    '''Level of rounding of the bokeh 

    :type:  float in [-0, 1], default 0.0 
    '''

    shift = None
    '''Shift of the lens components 

    :type:  float in [-1, 1], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeBoxMask:
    height = None
    '''Height of the box 

    :type:  float in [0, 2], default 0.2 
    '''

    mask_type = None
    '''

    :type:  enum in [‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘NOT’], default ‘ADD’ 
    '''

    rotation = None
    '''Rotation angle of the box 

    :type:  float in [-31.4159, 31.4159], default 0.0 
    '''

    width = None
    '''Width of the box 

    :type:  float in [0, 2], default 0.3 
    '''

    x = None
    '''X position of the middle of the box 

    :type:  float in [-1, 2], default 0.5 
    '''

    y = None
    '''Y position of the middle of the box 

    :type:  float in [-1, 2], default 0.5 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeBrightContrast:
    def update(self):
        '''

        '''
        pass


class CompositorNodeChannelMatte:
    color_space = None
    '''

    :type:  enum in [‘RGB’, ‘HSV’, ‘YUV’, ‘YCC’], default ‘RGB’ 
    '''

    limit_channel = None
    '''Limit by this channel’s value 

    :type:  enum in [‘R’, ‘G’, ‘B’], default ‘R’ 
    '''

    limit_max = None
    '''Values higher than this setting are 100% opaque 

    :type:  float in [0, 1], default 0.0 
    '''

    limit_method = None
    '''Algorithm to use to limit channel 

    :type:  enum in [‘SINGLE’, ‘MAX’], default ‘SINGLE’ 
    '''

    limit_min = None
    '''Values lower than this setting are 100% keyed 

    :type:  float in [0, 1], default 0.0 
    '''

    matte_channel = None
    '''Channel used to determine matte 

    :type:  enum in [‘R’, ‘G’, ‘B’], default ‘R’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeChromaMatte:
    gain = None
    '''Alpha falloff 

    :type:  float in [0, 1], default 0.0 
    '''

    lift = None
    '''Alpha lift 

    :type:  float in [0, 1], default 0.0 
    '''

    shadow_adjust = None
    '''Adjusts the brightness of any shadows captured 

    :type:  float in [0, 1], default 0.0 
    '''

    threshold = None
    '''Tolerance below which colors will be considered as exact matches 

    :type:  float in [0, 0.523599], default 0.0 
    '''

    tolerance = None
    '''Tolerance for a color to be considered a keying color 

    :type:  float in [0.0174533, 1.39626], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeColorBalance:
    correction_method = None
    '''

    :type:  enum in [‘LIFT_GAMMA_GAIN’, ‘OFFSET_POWER_SLOPE’], default ‘LIFT_GAMMA_GAIN’ 
    '''

    gain = None
    '''Correction for Highlights 

    :type:  float array of 3 items in [0, inf], default (1.0, 1.0, 1.0) 
    '''

    gamma = None
    '''Correction for Midtones 

    :type:  float array of 3 items in [0, inf], default (1.0, 1.0, 1.0) 
    '''

    lift = None
    '''Correction for Shadows 

    :type:  float array of 3 items in [0, inf], default (1.0, 1.0, 1.0) 
    '''

    offset = None
    '''Correction for Shadows 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    offset_basis = None
    '''Support negative color by using this as the RGB basis 

    :type:  float in [-inf, inf], default 0.0 
    '''

    power = None
    '''Correction for Midtones 

    :type:  float array of 3 items in [0, inf], default (1.0, 1.0, 1.0) 
    '''

    slope = None
    '''Correction for Highlights 

    :type:  float array of 3 items in [0, inf], default (1.0, 1.0, 1.0) 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeColorCorrection:
    blue = None
    '''Blue channel active 

    :type:  boolean, default True 
    '''

    green = None
    '''Green channel active 

    :type:  boolean, default True 
    '''

    highlights_contrast = None
    '''Highlights contrast 

    :type:  float in [0, 4], default 1.0 
    '''

    highlights_gain = None
    '''Highlights gain 

    :type:  float in [0, 4], default 1.0 
    '''

    highlights_gamma = None
    '''Highlights gamma 

    :type:  float in [0, 4], default 1.0 
    '''

    highlights_lift = None
    '''Highlights lift 

    :type:  float in [-1, 1], default 0.0 
    '''

    highlights_saturation = None
    '''Highlights saturation 

    :type:  float in [0, 4], default 1.0 
    '''

    master_contrast = None
    '''Master contrast 

    :type:  float in [0, 4], default 1.0 
    '''

    master_gain = None
    '''Master gain 

    :type:  float in [0, 4], default 1.0 
    '''

    master_gamma = None
    '''Master gamma 

    :type:  float in [0, 4], default 1.0 
    '''

    master_lift = None
    '''Master lift 

    :type:  float in [-1, 1], default 0.0 
    '''

    master_saturation = None
    '''Master saturation 

    :type:  float in [0, 4], default 1.0 
    '''

    midtones_contrast = None
    '''Midtones contrast 

    :type:  float in [0, 4], default 1.0 
    '''

    midtones_end = None
    '''End of midtones 

    :type:  float in [0, 1], default 0.7 
    '''

    midtones_gain = None
    '''Midtones gain 

    :type:  float in [0, 4], default 1.0 
    '''

    midtones_gamma = None
    '''Midtones gamma 

    :type:  float in [0, 4], default 1.0 
    '''

    midtones_lift = None
    '''Midtones lift 

    :type:  float in [-1, 1], default 0.0 
    '''

    midtones_saturation = None
    '''Midtones saturation 

    :type:  float in [0, 4], default 1.0 
    '''

    midtones_start = None
    '''Start of midtones 

    :type:  float in [0, 1], default 0.2 
    '''

    red = None
    '''Red channel active 

    :type:  boolean, default True 
    '''

    shadows_contrast = None
    '''Shadows contrast 

    :type:  float in [0, 4], default 1.0 
    '''

    shadows_gain = None
    '''Shadows gain 

    :type:  float in [0, 4], default 1.0 
    '''

    shadows_gamma = None
    '''Shadows gamma 

    :type:  float in [0, 4], default 1.0 
    '''

    shadows_lift = None
    '''Shadows lift 

    :type:  float in [-1, 1], default 0.0 
    '''

    shadows_saturation = None
    '''Shadows saturation 

    :type:  float in [0, 4], default 1.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeColorMatte:
    color_hue = None
    '''Hue tolerance for colors to be considered a keying color 

    :type:  float in [0, 1], default 0.0 
    '''

    color_saturation = None
    '''Saturation Tolerance for the color 

    :type:  float in [0, 1], default 0.0 
    '''

    color_value = None
    '''Value Tolerance for the color 

    :type:  float in [0, 1], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeColorSpill:
    channel = None
    '''

    :type:  enum in [‘R’, ‘G’, ‘B’], default ‘R’ 
    '''

    limit_channel = None
    '''

    :type:  enum in [‘R’, ‘G’, ‘B’], default ‘R’ 
    '''

    limit_method = None
    '''

    :type:  enum in [‘SIMPLE’, ‘AVERAGE’], default ‘SIMPLE’ 
    '''

    ratio = None
    '''Scale limit by value 

    :type:  float in [0.5, 1.5], default 0.0 
    '''

    unspill_blue = None
    '''Blue spillmap scale 

    :type:  float in [0, 1.5], default 0.0 
    '''

    unspill_green = None
    '''Green spillmap scale 

    :type:  float in [0, 1.5], default 0.0 
    '''

    unspill_red = None
    '''Red spillmap scale 

    :type:  float in [0, 1.5], default 0.0 
    '''

    use_unspill = None
    '''Compensate all channels (differently) by hand 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeCombHSVA:
    def update(self):
        '''

        '''
        pass


class CompositorNodeCombRGBA:
    def update(self):
        '''

        '''
        pass


class CompositorNodeCombYCCA:
    mode = None
    '''

    :type:  enum in [‘ITUBT601’, ‘ITUBT709’, ‘JFIF’], default ‘ITUBT601’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeCombYUVA:
    def update(self):
        '''

        '''
        pass


class CompositorNodeComposite:
    use_alpha = None
    '''Colors are treated alpha premultiplied, or colors output straight (alpha gets set to 1) 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeCornerPin:
    def update(self):
        '''

        '''
        pass


class CompositorNodeCrop:
    max_x = None
    '''

    :type:  int in [0, 10000], default 0 
    '''

    max_y = None
    '''

    :type:  int in [0, 10000], default 0 
    '''

    min_x = None
    '''

    :type:  int in [0, 10000], default 0 
    '''

    min_y = None
    '''

    :type:  int in [0, 10000], default 0 
    '''

    rel_max_x = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''

    rel_max_y = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''

    rel_min_x = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''

    rel_min_y = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''

    relative = None
    '''Use relative values to crop image 

    :type:  boolean, default False 
    '''

    use_crop_size = None
    '''Whether to crop the size of the input image 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeCurveRGB:
    mapping = None
    '''

    :type:  CurveMapping, (readonly) 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeCurveVec:
    mapping = None
    '''

    :type:  CurveMapping, (readonly) 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeDBlur:
    angle = None
    '''

    :type:  float in [0, 6.28319], default 0.0 
    '''

    center_x = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''

    center_y = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''

    distance = None
    '''

    :type:  float in [-1, 1], default 0.0 
    '''

    iterations = None
    '''

    :type:  int in [1, 32], default 0 
    '''

    spin = None
    '''

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    use_wrap = None
    '''

    :type:  boolean, default False 
    '''

    zoom = None
    '''

    :type:  float in [0, 100], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeDefocus:
    angle = None
    '''Bokeh shape rotation offset 

    :type:  float in [0, 1.5708], default 0.0 
    '''

    blur_max = None
    '''Blur limit, maximum CoC radius 

    :type:  float in [0, 10000], default 0.0 
    '''

    bokeh = None
    '''

    :type:  enum in [‘OCTAGON’, ‘HEPTAGON’, ‘HEXAGON’, ‘PENTAGON’, ‘SQUARE’, ‘TRIANGLE’, ‘CIRCLE’], default ‘CIRCLE’ 
    '''

    f_stop = None
    '''Amount of focal blur, 128=infinity=perfect focus, half the value doubles the blur radius 

    :type:  float in [0, 128], default 0.0 
    '''

    scene = None
    '''Scene from which to select the active camera (render scene if undefined) 

    :type:  Scene 
    '''

    threshold = None
    '''CoC radius threshold, prevents background bleed on in-focus midground, 0=off 

    :type:  float in [0, 100], default 0.0 
    '''

    use_gamma_correction = None
    '''Enable gamma correction before and after main process 

    :type:  boolean, default False 
    '''

    use_preview = None
    '''Enable low quality mode, useful for preview 

    :type:  boolean, default False 
    '''

    use_zbuffer = None
    '''Disable when using an image as input instead of actual z-buffer (auto enabled if node not image based, eg. time node) 

    :type:  boolean, default False 
    '''

    z_scale = None
    '''Scale the Z input when not using a z-buffer, controls maximum blur designated by the color white or input value 1 

    :type:  float in [0, 1000], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeDespeckle:
    threshold = None
    '''Threshold for detecting pixels to despeckle 

    :type:  float in [0, 1], default 0.0 
    '''

    threshold_neighbor = None
    '''Threshold for the number of neighbor pixels that must match 

    :type:  float in [0, 1], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeDiffMatte:
    falloff = None
    '''Color distances below this additional threshold are partially keyed 

    :type:  float in [0, 1], default 0.0 
    '''

    tolerance = None
    '''Color distances below this threshold are keyed 

    :type:  float in [0, 1], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeDilateErode:
    distance = None
    '''Distance to grow/shrink (number of iterations) 

    :type:  int in [-5000, 5000], default 0 
    '''

    edge = None
    '''Edge to inset 

    :type:  float in [-100, 100], default 0.0 
    '''

    falloff = None
    '''Falloff type the feather 

    :type:  enum in [‘SMOOTH’, ‘SPHERE’, ‘ROOT’, ‘INVERSE_SQUARE’, ‘SHARP’, ‘LINEAR’], default ‘SMOOTH’ 
    '''

    mode = None
    '''Growing/shrinking mode 

    :type:  enum in [‘STEP’, ‘THRESHOLD’, ‘DISTANCE’, ‘FEATHER’], default ‘STEP’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeDisplace:
    def update(self):
        '''

        '''
        pass


class CompositorNodeDistanceMatte:
    channel = None
    '''

    :type:  enum in [‘RGB’, ‘YCC’], default ‘RGB’ 
    '''

    falloff = None
    '''Color distances below this additional threshold are partially keyed 

    :type:  float in [0, 1], default 0.0 
    '''

    tolerance = None
    '''Color distances below this threshold are keyed 

    :type:  float in [0, 1], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeDoubleEdgeMask:
    edge_mode = None
    '''

    :type:  enum in [‘BLEED_OUT’, ‘KEEP_IN’], default ‘BLEED_OUT’ 
    '''

    inner_mode = None
    '''

    :type:  enum in [‘ALL’, ‘ADJACENT_ONLY’], default ‘ALL’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeEllipseMask:
    height = None
    '''Height of the ellipse 

    :type:  float in [0, 2], default 0.2 
    '''

    mask_type = None
    '''

    :type:  enum in [‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘NOT’], default ‘ADD’ 
    '''

    rotation = None
    '''Rotation angle of the ellipse 

    :type:  float in [-31.4159, 31.4159], default 0.0 
    '''

    width = None
    '''Width of the ellipse 

    :type:  float in [0, 2], default 0.3 
    '''

    x = None
    '''X position of the middle of the ellipse 

    :type:  float in [-1, 2], default 0.5 
    '''

    y = None
    '''Y position of the middle of the ellipse 

    :type:  float in [-1, 2], default 0.5 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeFilter:
    filter_type = None
    '''

    :type:  enum in [‘SOFTEN’, ‘SHARPEN’, ‘LAPLACE’, ‘SOBEL’, ‘PREWITT’, ‘KIRSCH’, ‘SHADOW’], default ‘SOFTEN’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeFlip:
    axis = None
    '''

    :type:  enum in [‘X’, ‘Y’, ‘XY’], default ‘X’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeGamma:
    def update(self):
        '''

        '''
        pass


class CompositorNodeGlare:
    angle_offset = None
    '''Streak angle offset 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    color_modulation = None
    '''Amount of Color Modulation, modulates colors of streaks and ghosts for a spectral dispersion effect 

    :type:  float in [0, 1], default 0.0 
    '''

    fade = None
    '''Streak fade-out factor 

    :type:  float in [0.75, 1], default 0.0 
    '''

    glare_type = None
    '''

    :type:  enum in [‘GHOSTS’, ‘STREAKS’, ‘FOG_GLOW’, ‘SIMPLE_STAR’], default ‘SIMPLE_STAR’ 
    '''

    iterations = None
    '''

    :type:  int in [2, 5], default 0 
    '''

    mix = None
    '''-1 is original image only, 0 is exact 50/50 mix, 1 is processed image only 

    :type:  float in [-1, 1], default 0.0 
    '''

    quality = None
    '''If not set to high quality, the effect will be applied to a low-res copy of the source image 

    :type:  enum in [‘HIGH’, ‘MEDIUM’, ‘LOW’], default ‘HIGH’ 
    '''

    size = None
    '''Glow/glare size (not actual size; relative to initial size of bright area of pixels) 

    :type:  int in [6, 9], default 0 
    '''

    streaks = None
    '''Total number of streaks 

    :type:  int in [2, 16], default 0 
    '''

    threshold = None
    '''The glare filter will only be applied to pixels brighter than this value 

    :type:  float in [0, 1000], default 0.0 
    '''

    use_rotate_45 = None
    '''Simple star filter: add 45 degree rotation offset 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeGroup:
    interface = None
    '''Interface socket data 

    :type:  PropertyGroup, (readonly) 
    '''

    node_tree = None
    '''

    :type:  NodeTree 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeHueCorrect:
    mapping = None
    '''

    :type:  CurveMapping, (readonly) 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeHueSat:
    color_hue = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''

    color_saturation = None
    '''

    :type:  float in [0, 2], default 0.0 
    '''

    color_value = None
    '''

    :type:  float in [0, 2], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeIDMask:
    index = None
    '''Pass index number to convert to alpha 

    :type:  int in [0, 32767], default 0 
    '''

    use_antialiasing = None
    '''Apply an anti-aliasing filter to the mask 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeImage:
    frame_duration = None
    '''Number of images of a movie to use 

    :type:  int in [0, 500000], default 0 
    '''

    frame_offset = None
    '''Offset the number of the frame to use in the animation 

    :type:  int in [-500000, 500000], default 0 
    '''

    frame_start = None
    '''Global starting frame of the movie/sequence, assuming first picture has a #1 

    :type:  int in [-500000, 500000], default 0 
    '''

    has_layers = None
    '''True if this image has any named layer 

    :type:  boolean, default False, (readonly) 
    '''

    has_views = None
    '''True if this image has multiple views 

    :type:  boolean, default False, (readonly) 
    '''

    image = None
    '''

    :type:  Image 
    '''

    layer = None
    '''

    :type:  enum in [‘PLACEHOLDER’], default ‘PLACEHOLDER’ 
    '''

    use_auto_refresh = None
    '''Always refresh image on frame changes 

    :type:  boolean, default False 
    '''

    use_cyclic = None
    '''Cycle the images in the movie 

    :type:  boolean, default False 
    '''

    use_straight_alpha_output = None
    '''Put Node output buffer to straight alpha instead of premultiplied 

    :type:  boolean, default False 
    '''

    view = None
    '''

    :type:  enum in [‘ALL’], default ‘ALL’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeInpaint:
    distance = None
    '''Distance to inpaint (number of iterations) 

    :type:  int in [1, 10000], default 0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeInvert:
    invert_alpha = None
    '''

    :type:  boolean, default False 
    '''

    invert_rgb = None
    '''

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeKeying:
    blur_post = None
    '''Matte blur size which applies after clipping and dilate/eroding 

    :type:  int in [0, 2048], default 0 
    '''

    blur_pre = None
    '''Chroma pre-blur size which applies before running keyer 

    :type:  int in [0, 2048], default 0 
    '''

    clip_black = None
    '''Value of non-scaled matte pixel which considers as fully background pixel 

    :type:  float in [0, 1], default 0.0 
    '''

    clip_white = None
    '''Value of non-scaled matte pixel which considers as fully foreground pixel 

    :type:  float in [0, 1], default 0.0 
    '''

    despill_balance = None
    '''Balance between non-key colors used to detect amount of key color to be removed 

    :type:  float in [0, 1], default 0.0 
    '''

    despill_factor = None
    '''Factor of despilling screen color from image 

    :type:  float in [0, 1], default 0.0 
    '''

    dilate_distance = None
    '''Matte dilate/erode side 

    :type:  int in [-100, 100], default 0 
    '''

    edge_kernel_radius = None
    '''Radius of kernel used to detect whether pixel belongs to edge 

    :type:  int in [0, 100], default 0 
    '''

    edge_kernel_tolerance = None
    '''Tolerance to pixels inside kernel which are treating as belonging to the same plane 

    :type:  float in [0, 1], default 0.0 
    '''

    feather_distance = None
    '''Distance to grow/shrink the feather 

    :type:  int in [-100, 100], default 0 
    '''

    feather_falloff = None
    '''Falloff type the feather 

    :type:  enum in [‘SMOOTH’, ‘SPHERE’, ‘ROOT’, ‘INVERSE_SQUARE’, ‘SHARP’, ‘LINEAR’], default ‘SMOOTH’ 
    '''

    screen_balance = None
    '''Balance between two non-primary channels primary channel is comparing against 

    :type:  float in [0, 1], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeKeyingScreen:
    clip = None
    '''

    :type:  MovieClip 
    '''

    tracking_object = None
    '''

    :type:  string, default “”, (never None) 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeLensdist:
    use_fit = None
    '''For positive distortion factor only: scale image such that black areas are not visible 

    :type:  boolean, default False 
    '''

    use_jitter = None
    '''Enable/disable jittering (faster, but also noisier) 

    :type:  boolean, default False 
    '''

    use_projector = None
    '''Enable/disable projector mode (the effect is applied in horizontal direction only) 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeLevels:
    channel = None
    '''

    :type:  enum in [‘COMBINED_RGB’, ‘RED’, ‘GREEN’, ‘BLUE’, ‘LUMINANCE’], default ‘COMBINED_RGB’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeLumaMatte:
    limit_max = None
    '''Values higher than this setting are 100% opaque 

    :type:  float in [0, 1], default 0.0 
    '''

    limit_min = None
    '''Values lower than this setting are 100% keyed 

    :type:  float in [0, 1], default 0.0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeMapRange:
    use_clamp = None
    '''Clamp result of the node to 0..1 range 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeMapUV:
    alpha = None
    '''

    :type:  int in [0, 100], default 0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeMapValue:
    max = None
    '''

    :type:  float array of 1 items in [-1000, 1000], default (0.0) 
    '''

    min = None
    '''

    :type:  float array of 1 items in [-1000, 1000], default (0.0) 
    '''

    offset = None
    '''

    :type:  float array of 1 items in [-1000, 1000], default (0.0) 
    '''

    size = None
    '''

    :type:  float array of 1 items in [-1000, 1000], default (0.0) 
    '''

    use_max = None
    '''

    :type:  boolean, default False 
    '''

    use_min = None
    '''

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeMask:
    mask = None
    '''

    :type:  Mask 
    '''

    motion_blur_samples = None
    '''Number of motion blur samples 

    :type:  int in [1, 64], default 0 
    '''

    motion_blur_shutter = None
    '''Exposure for motion blur as a factor of FPS 

    :type:  float in [0, 1], default 0.0 
    '''

    size_source = None
    '''Where to get the mask size from for aspect/size information 

    :type:  enum in [‘SCENE’, ‘FIXED’, ‘FIXED_SCENE’], default ‘SCENE’ 
    '''

    size_x = None
    '''

    :type:  int in [1, 10000], default 0 
    '''

    size_y = None
    '''

    :type:  int in [1, 10000], default 0 
    '''

    use_antialiasing = None
    '''Apply an anti-aliasing filter to the mask 

    :type:  boolean, default False 
    '''

    use_feather = None
    '''Use feather information from the mask 

    :type:  boolean, default False 
    '''

    use_motion_blur = None
    '''Use multi-sampled motion blur of the mask 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeMath:
    operation = None
    '''

    :type:  enum in [‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘SINE’, ‘COSINE’, ‘TANGENT’, ‘ARCSINE’, ‘ARCCOSINE’, ‘ARCTANGENT’, ‘POWER’, ‘LOGARITHM’, ‘MINIMUM’, ‘MAXIMUM’, ‘ROUND’, ‘LESS_THAN’, ‘GREATER_THAN’, ‘MODULO’, ‘ABSOLUTE’], default ‘ADD’ 
    '''

    use_clamp = None
    '''Clamp result of the node to 0..1 range 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeMixRGB:
    blend_type = None
    '''

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    use_alpha = None
    '''Include alpha of second input in this operation 

    :type:  boolean, default False 
    '''

    use_clamp = None
    '''Clamp result of the node to 0..1 range 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeMovieClip:
    clip = None
    '''

    :type:  MovieClip 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeMovieDistortion:
    clip = None
    '''

    :type:  MovieClip 
    '''

    distortion_type = None
    '''Distortion to use to filter image 

    :type:  enum in [‘UNDISTORT’, ‘DISTORT’], default ‘UNDISTORT’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeNormal:
    def update(self):
        '''

        '''
        pass


class CompositorNodeNormalize:
    def update(self):
        '''

        '''
        pass


class CompositorNodeOutputFile:
    active_input_index = None
    '''Active input index in details view list 

    :type:  int in [-inf, inf], default 0 
    '''

    base_path = None
    '''Base output path for the image 

    :type:  string, default “”, (never None) 
    '''

    file_slots = None
    '''

    :type:  CompositorNodeOutputFileFileSlots bpy_prop_collection of NodeOutputFileSlotFile, (readonly) 
    '''

    format = None
    '''

    :type:  ImageFormatSettings, (readonly) 
    '''

    layer_slots = None
    '''

    :type:  CompositorNodeOutputFileLayerSlots bpy_prop_collection of NodeOutputFileSlotLayer, (readonly) 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeOutputFileFileSlots:
    '''Collection of File Output node slots '''

    def new(self, name):
        '''Add a file slot to this node 

        :param name: Name 
        :type name: string, (never None)
        :rtype: NodeSocket 
        :return:  New socket 
        '''
        pass

    def remove(self, socket):
        '''Remove a file slot from this node 

        :param socket: The socket to remove 
        :type socket: NodeSocket
        '''
        pass

    def clear(self):
        '''Remove all file slots from this node 

        '''
        pass

    def move(self, from_index, to_index):
        '''Move a file slot to another position 

        :param from_index: From Index, Index of the socket to move 
        :type from_index: int in [0, inf]
        :param to_index: To Index, Target index for the socket 
        :type to_index: int in [0, inf]
        '''
        pass


class CompositorNodeOutputFileLayerSlots:
    '''Collection of File Output node slots '''

    def new(self, name):
        '''Add a file slot to this node 

        :param name: Name 
        :type name: string, (never None)
        :rtype: NodeSocket 
        :return:  New socket 
        '''
        pass

    def remove(self, socket):
        '''Remove a file slot from this node 

        :param socket: The socket to remove 
        :type socket: NodeSocket
        '''
        pass

    def clear(self):
        '''Remove all file slots from this node 

        '''
        pass

    def move(self, from_index, to_index):
        '''Move a file slot to another position 

        :param from_index: From Index, Index of the socket to move 
        :type from_index: int in [0, inf]
        :param to_index: To Index, Target index for the socket 
        :type to_index: int in [0, inf]
        '''
        pass


class CompositorNodePixelate:
    def update(self):
        '''

        '''
        pass


class CompositorNodePlaneTrackDeform:
    clip = None
    '''

    :type:  MovieClip 
    '''

    motion_blur_samples = None
    '''Number of motion blur samples 

    :type:  int in [1, 64], default 0 
    '''

    motion_blur_shutter = None
    '''Exposure for motion blur as a factor of FPS 

    :type:  float in [0, 1], default 0.0 
    '''

    plane_track_name = None
    '''

    :type:  string, default “”, (never None) 
    '''

    tracking_object = None
    '''

    :type:  string, default “”, (never None) 
    '''

    use_motion_blur = None
    '''Use multi-sampled motion blur of the mask 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodePremulKey:
    mapping = None
    '''Conversion between premultiplied alpha and key alpha 

    :type:  enum in [‘STRAIGHT_TO_PREMUL’, ‘PREMUL_TO_STRAIGHT’], default ‘STRAIGHT_TO_PREMUL’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeRGB:
    def update(self):
        '''

        '''
        pass


class CompositorNodeRGBToBW:
    def update(self):
        '''

        '''
        pass


class CompositorNodeRLayers:
    layer = None
    '''

    :type:  enum in [‘PLACEHOLDER’], default ‘PLACEHOLDER’ 
    '''

    scene = None
    '''

    :type:  Scene 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeRotate:
    filter_type = None
    '''Method to use to filter rotation 

    :type:  enum in [‘NEAREST’, ‘BILINEAR’, ‘BICUBIC’], default ‘NEAREST’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeScale:
    frame_method = None
    '''How the image fits in the camera frame 

    :type:  enum in [‘STRETCH’, ‘FIT’, ‘CROP’], default ‘STRETCH’ 
    '''

    offset_x = None
    '''Offset image horizontally (factor of image size) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    offset_y = None
    '''Offset image vertically (factor of image size) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    space = None
    '''Coordinate space to scale relative to 

    :type:  enum in [‘RELATIVE’, ‘ABSOLUTE’, ‘SCENE_SIZE’, ‘RENDER_SIZE’], default ‘RELATIVE’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeSepHSVA:
    def update(self):
        '''

        '''
        pass


class CompositorNodeSepRGBA:
    def update(self):
        '''

        '''
        pass


class CompositorNodeSepYCCA:
    mode = None
    '''

    :type:  enum in [‘ITUBT601’, ‘ITUBT709’, ‘JFIF’], default ‘ITUBT601’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeSepYUVA:
    def update(self):
        '''

        '''
        pass


class CompositorNodeSetAlpha:
    def update(self):
        '''

        '''
        pass


class CompositorNodeSplitViewer:
    axis = None
    '''

    :type:  enum in [‘X’, ‘Y’], default ‘X’ 
    '''

    factor = None
    '''

    :type:  int in [0, 100], default 0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeStabilize:
    clip = None
    '''

    :type:  MovieClip 
    '''

    filter_type = None
    '''Method to use to filter stabilization 

    :type:  enum in [‘NEAREST’, ‘BILINEAR’, ‘BICUBIC’], default ‘NEAREST’ 
    '''

    invert = None
    '''Invert stabilization to re-introduce motion to the frame 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeSunBeams:
    ray_length = None
    '''Length of rays as a factor of the image size 

    :type:  float in [0, 100], default 0.0 
    '''

    source = None
    '''Source point of rays as a factor of the image width & height 

    :type:  float array of 2 items in [-100, 100], default (0.0, 0.0) 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeSwitch:
    check = None
    '''Off: first socket, On: second socket 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeSwitchView:
    check = None
    '''Off: first socket, On: second socket 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeTexture:
    node_output = None
    '''For node-based textures, which output node to use 

    :type:  int in [-32768, 32767], default 0 
    '''

    texture = None
    '''

    :type:  Texture 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeTime:
    curve = None
    '''

    :type:  CurveMapping, (readonly) 
    '''

    frame_end = None
    '''

    :type:  int in [-32768, 32767], default 0 
    '''

    frame_start = None
    '''

    :type:  int in [-32768, 32767], default 0 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeTonemap:
    adaptation = None
    '''If 0, global; if 1, based on pixel intensity 

    :type:  float in [0, 1], default 0.0 
    '''

    contrast = None
    '''Set to 0 to use estimate from input image 

    :type:  float in [0, 1], default 0.0 
    '''

    correction = None
    '''If 0, same for all channels; if 1, each independent 

    :type:  float in [0, 1], default 0.0 
    '''

    gamma = None
    '''If not used, set to 1 

    :type:  float in [0.001, 3], default 0.0 
    '''

    intensity = None
    '''If less than zero, darkens image; otherwise, makes it brighter 

    :type:  float in [-8, 8], default 0.0 
    '''

    key = None
    '''The value the average luminance is mapped to 

    :type:  float in [0, 1], default 0.0 
    '''

    offset = None
    '''Normally always 1, but can be used as an extra control to alter the brightness curve 

    :type:  float in [0.001, 10], default 0.0 
    '''

    tonemap_type = None
    '''

    :type:  enum in [‘RD_PHOTORECEPTOR’, ‘RH_SIMPLE’], default ‘RH_SIMPLE’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeTrackPos:
    clip = None
    '''

    :type:  MovieClip 
    '''

    frame_relative = None
    '''Frame to be used for relative position 

    :type:  int in [-32768, 32767], default 0 
    '''

    position = None
    '''Which marker position to use for output 

    :type:  enum in [‘ABSOLUTE’, ‘RELATIVE_START’, ‘RELATIVE_FRAME’, ‘ABSOLUTE_FRAME’], default ‘ABSOLUTE’ 
    '''

    track_name = None
    '''

    :type:  string, default “”, (never None) 
    '''

    tracking_object = None
    '''

    :type:  string, default “”, (never None) 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeTransform:
    filter_type = None
    '''Method to use to filter transform 

    :type:  enum in [‘NEAREST’, ‘BILINEAR’, ‘BICUBIC’], default ‘NEAREST’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeTranslate:
    use_relative = None
    '''Use relative (fraction of input image size) values to define translation 

    :type:  boolean, default False 
    '''

    wrap_axis = None
    '''Wrap image on a specific axis 

    :type:  enum in [‘NONE’, ‘XAXIS’, ‘YAXIS’, ‘BOTH’], default ‘NONE’ 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeTree:
    '''Node tree consisting of linked nodes used for compositing '''

    chunk_size = None
    '''Max size of a tile (smaller values gives better distribution of multiple threads, but more overhead) 

    :type:  enum in [‘32’, ‘64’, ‘128’, ‘256’, ‘512’, ‘1024’], default ‘32’ 
    '''

    edit_quality = None
    '''Quality when editing 

    :type:  enum in [‘HIGH’, ‘MEDIUM’, ‘LOW’], default ‘HIGH’ 
    '''

    render_quality = None
    '''Quality when rendering 

    :type:  enum in [‘HIGH’, ‘MEDIUM’, ‘LOW’], default ‘HIGH’ 
    '''

    use_groupnode_buffer = None
    '''Enable buffering of group nodes 

    :type:  boolean, default False 
    '''

    use_opencl = None
    '''Enable GPU calculations 

    :type:  boolean, default False 
    '''

    use_two_pass = None
    '''Use two pass execution during editing: first calculate fast nodes, second pass calculate all nodes 

    :type:  boolean, default False 
    '''

    use_viewer_border = None
    '''Use boundaries for viewer nodes and composite backdrop 

    :type:  boolean, default False 
    '''


class CompositorNodeValToRGB:
    color_ramp = None
    '''

    :type:  ColorRamp, (readonly) 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeValue:
    def update(self):
        '''

        '''
        pass


class CompositorNodeVecBlur:
    factor = None
    '''Scaling factor for motion vectors (actually, ‘shutter speed’, in frames) 

    :type:  float in [0, 20], default 0.0 
    '''

    samples = None
    '''

    :type:  int in [1, 256], default 0 
    '''

    speed_max = None
    '''Maximum speed, or zero for none 

    :type:  int in [0, 1024], default 0 
    '''

    speed_min = None
    '''Minimum speed for a pixel to be blurred (used to separate background from foreground) 

    :type:  int in [0, 1024], default 0 
    '''

    use_curved = None
    '''Interpolate between frames in a Bezier curve, rather than linearly 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeViewer:
    center_x = None
    '''

    :type:  float in [0, 1], default 0.5 
    '''

    center_y = None
    '''

    :type:  float in [0, 1], default 0.5 
    '''

    tile_order = None
    '''Tile order 

    :type:  enum in [‘CENTEROUT’, ‘RANDOM’, ‘BOTTOMUP’, ‘RULE_OF_THIRDS’], default ‘CENTEROUT’ 
    '''

    use_alpha = None
    '''Colors are treated alpha premultiplied, or colors output straight (alpha gets set to 1) 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class CompositorNodeZcombine:
    use_alpha = None
    '''Take Alpha channel into account when doing the Z operation 

    :type:  boolean, default False 
    '''

    use_antialias_z = None
    '''Anti-alias the z-buffer to try to avoid artifacts, mostly useful for Blender renders 

    :type:  boolean, default False 
    '''

    def update(self):
        '''

        '''
        pass


class ConsoleLine:
    '''Input line for the interactive console '''

    body = None
    '''Text in the line 

    :type:  string, default “”, (never None) 
    '''

    current_character = None
    '''

    :type:  int in [-inf, inf], default 0 
    '''

    type = None
    '''Console line type when used in scrollback 

    :type:  enum in [‘OUTPUT’, ‘INPUT’, ‘INFO’, ‘ERROR’], default ‘OUTPUT’ 
    '''


class Constraint:
    '''Constraint modifying the transformation of objects and bones '''

    active = None
    '''Constraint is the one being edited 

    :type:  boolean, default False 
    '''

    error_location = None
    '''Amount of residual error in Blender space unit for constraints that work on position 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    error_rotation = None
    '''Amount of residual error in radians for constraints that work on orientation 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    influence = None
    '''Amount of influence constraint will have on the final solution 

    :type:  float in [0, 1], default 0.0 
    '''

    is_proxy_local = None
    '''Constraint was added in this proxy instance (i.e. did not belong to source Armature) 

    :type:  boolean, default False 
    '''

    is_valid = None
    '''Constraint has valid settings and can be evaluated 

    :type:  boolean, default False, (readonly) 
    '''

    mute = None
    '''Enable/Disable Constraint 

    :type:  boolean, default False 
    '''

    name = None
    '''Constraint name 

    :type:  string, default “”, (never None) 
    '''

    owner_space = None
    '''Space that owner is evaluated in 

    :type:  enum in [‘WORLD’, ‘POSE’, ‘LOCAL_WITH_PARENT’, ‘LOCAL’], default ‘WORLD’ 
    '''

    show_expanded = None
    '''Constraint’s panel is expanded in UI 

    :type:  boolean, default False 
    '''

    target_space = None
    '''Space that target is evaluated in 

    :type:  enum in [‘WORLD’, ‘POSE’, ‘LOCAL_WITH_PARENT’, ‘LOCAL’], default ‘WORLD’ 
    '''

    type = None
    '''

    :type:  enum in [‘CAMERA_SOLVER’, ‘FOLLOW_TRACK’, ‘OBJECT_SOLVER’, ‘COPY_LOCATION’, ‘COPY_ROTATION’, ‘COPY_SCALE’, ‘COPY_TRANSFORMS’, ‘LIMIT_DISTANCE’, ‘LIMIT_LOCATION’, ‘LIMIT_ROTATION’, ‘LIMIT_SCALE’, ‘MAINTAIN_VOLUME’, ‘TRANSFORM’, ‘TRANSFORM_CACHE’, ‘CLAMP_TO’, ‘DAMPED_TRACK’, ‘IK’, ‘LOCKED_TRACK’, ‘SPLINE_IK’, ‘STRETCH_TO’, ‘TRACK_TO’, ‘ACTION’, ‘CHILD_OF’, ‘FLOOR’, ‘FOLLOW_PATH’, ‘PIVOT’, ‘RIGID_BODY_JOINT’, ‘SHRINKWRAP’], default ‘CAMERA_SOLVER’, (readonly) 
    '''


class ConstraintActuator:
    '''Actuator to handle Constraints '''

    angle_max = None
    '''Maximum angle allowed with target direction (no correction is done if angle with target direction is between min and max) 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    angle_min = None
    '''Minimum angle to maintain with target direction (no correction is done if angle with target direction is between min and max) 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    damping = None
    '''Damping factor: time constant (in frame) of low pass filter 

    :type:  int in [-32768, 32767], default 0 
    '''

    damping_rotation = None
    '''Use a different damping for orientation 

    :type:  int in [-32768, 32767], default 0 
    '''

    direction = None
    '''Direction of the ray 

    :type:  enum in [‘NONE’, ‘DIRPX’, ‘DIRPY’, ‘DIRPZ’, ‘DIRNX’, ‘DIRNY’, ‘DIRNZ’], default ‘NONE’ 
    '''

    direction_axis = None
    '''Select the axis to be aligned along the reference direction 

    :type:  enum in [‘NONE’, ‘DIRPX’, ‘DIRPY’, ‘DIRPZ’, ‘DIRNX’, ‘DIRNY’, ‘DIRNZ’], default ‘NONE’ 
    '''

    direction_axis_pos = None
    '''Select the axis to be aligned along the reference direction 

    :type:  enum in [‘NONE’, ‘DIRPX’, ‘DIRPY’, ‘DIRPZ’], default ‘NONE’ 
    '''

    distance = None
    '''Keep this distance to target 

    :type:  float in [-inf, inf], default 0.0 
    '''

    fh_damping = None
    '''Damping factor of the force field spring 

    :type:  float in [-inf, inf], default 0.0 
    '''

    fh_force = None
    '''Spring force within the force field area 

    :type:  float in [-inf, inf], default 0.0 
    '''

    fh_height = None
    '''Height of the force field area 

    :type:  float in [-inf, inf], default 0.0 
    '''

    limit = None
    '''

    :type:  enum in [‘NONE’, ‘LOCX’, ‘LOCY’, ‘LOCZ’], default ‘NONE’ 
    '''

    limit_max = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    limit_min = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    material = None
    '''Ray detects only Objects with this material 

    :type:  string, default “”, (never None) 
    '''

    mode = None
    '''The type of the constraint 

    :type:  enum in [‘LOC’, ‘DIST’, ‘ORI’, ‘FH’], default ‘LOC’ 
    '''

    property = None
    '''Ray detects only Objects with this property 

    :type:  string, default “”, (never None) 
    '''

    range = None
    '''Maximum length of ray 

    :type:  float in [-inf, inf], default 0.0 
    '''

    rotation_max = None
    '''Reference Direction 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    time = None
    '''Maximum activation time in frame, 0 for unlimited 

    :type:  int in [-32768, 32767], default 0 
    '''

    use_fh_normal = None
    '''Add a horizontal spring force on slopes 

    :type:  boolean, default False 
    '''

    use_fh_paralel_axis = None
    '''Keep object axis parallel to normal 

    :type:  boolean, default False 
    '''

    use_force_distance = None
    '''Force distance of object to point of impact of ray 

    :type:  boolean, default False 
    '''

    use_local = None
    '''Set ray along object’s axis or global axis 

    :type:  boolean, default False 
    '''

    use_material_detect = None
    '''Detect material instead of property 

    :type:  boolean, default False 
    '''

    use_normal = None
    '''Set object axis along (local axis) or parallel (global axis) to the normal at hit position 

    :type:  boolean, default False 
    '''

    use_persistent = None
    '''Persistent actuator: stays active even if ray does not reach target 

    :type:  boolean, default False 
    '''


class ConstraintTarget:
    '''Target object for multi-target constraints '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''


class Context:
    '''Current windowmanager and data context '''

    area = None
    '''

    :type:  Area, (readonly) 
    '''

    blend_data = None
    '''

    :type:  BlendData, (readonly) 
    '''

    mode = None
    '''

    :type:  enum in [‘EDIT_MESH’, ‘EDIT_CURVE’, ‘EDIT_SURFACE’, ‘EDIT_TEXT’, ‘EDIT_ARMATURE’, ‘EDIT_METABALL’, ‘EDIT_LATTICE’, ‘POSE’, ‘SCULPT’, ‘PAINT_WEIGHT’, ‘PAINT_VERTEX’, ‘PAINT_TEXTURE’, ‘PARTICLE’, ‘OBJECT’], default ‘EDIT_MESH’, (readonly) 
    '''

    region = None
    '''

    :type:  Region, (readonly) 
    '''

    region_data = None
    '''

    :type:  RegionView3D, (readonly) 
    '''

    scene = None
    '''

    :type:  Scene, (readonly) 
    '''

    screen = None
    '''

    :type:  Screen, (readonly) 
    '''

    space_data = None
    '''

    :type:  Space, (readonly) 
    '''

    tool_settings = None
    '''

    :type:  ToolSettings, (readonly) 
    '''

    user_preferences = None
    '''

    :type:  UserPreferences, (readonly) 
    '''

    window = None
    '''

    :type:  Window, (readonly) 
    '''

    window_manager = None
    '''

    :type:  WindowManager, (readonly) 
    '''

    def copy(self):
        '''

        '''
        pass


class ControlFluidSettings:
    '''Fluid simulation settings for objects controlling the motion of fluid in the simulation '''

    attraction_radius = None
    '''Force field radius around the control object 

    :type:  float in [0, 10], default 0.0 
    '''

    attraction_strength = None
    '''Force strength for directional attraction towards the control object 

    :type:  float in [-10, 10], default 0.0 
    '''

    end_time = None
    '''Time when the control particles are deactivated 

    :type:  float in [0, inf], default 0.0 
    '''

    quality = None
    '''Quality which is used for object sampling (higher = better but slower) 

    :type:  float in [5, 100], default 0.0 
    '''

    start_time = None
    '''Time when the control particles are activated 

    :type:  float in [0, inf], default 0.0 
    '''

    use = None
    '''Object contributes to the fluid simulation 

    :type:  boolean, default False 
    '''

    use_reverse_frames = None
    '''Reverse control object movement 

    :type:  boolean, default False 
    '''

    velocity_radius = None
    '''Force field radius around the control object 

    :type:  float in [0, 10], default 0.0 
    '''

    velocity_strength = None
    '''Force strength of how much of the control object’s velocity is influencing the fluid velocity 

    :type:  float in [0, 10], default 0.0 
    '''


class Controller:
    '''Game engine logic brick to process events, connecting sensors to actuators '''

    active = None
    '''Set the active state of the controller 

    :type:  boolean, default False 
    '''

    actuators = None
    '''The list containing the actuators connected to the controller 

    :type:  bpy_prop_collection of Actuator, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''

    show_expanded = None
    '''Set controller expanded in the user interface 

    :type:  boolean, default False 
    '''

    states = None
    '''Set Controller state index (1 to 30) 

    :type:  int in [1, 30], default 0 
    '''

    type = None
    '''

    :type:  enum in [‘LOGIC_AND’, ‘LOGIC_OR’, ‘LOGIC_NAND’, ‘LOGIC_NOR’, ‘LOGIC_XOR’, ‘LOGIC_XNOR’, ‘EXPRESSION’, ‘PYTHON’], default ‘LOGIC_AND’ 
    '''

    use_priority = None
    '''Mark controller for execution before all non-marked controllers (good for startup scripts) 

    :type:  boolean, default False 
    '''

    def link(self, sensor=None, actuator=None):
        '''Link the controller with a sensor/actuator 

        :param sensor: Sensor to link the controller to 
        :type sensor: Sensor, (optional)
        :param actuator: Actuator to link the controller to 
        :type actuator: Actuator, (optional)
        '''
        pass

    def unlink(self, sensor=None, actuator=None):
        '''Unlink the controller from a sensor/actuator 

        :param sensor: Sensor to unlink the controller from 
        :type sensor: Sensor, (optional)
        :param actuator: Actuator to unlink the controller from 
        :type actuator: Actuator, (optional)
        '''
        pass


class CopyLocationConstraint:
    '''Copy the location of the target '''

    head_tail = None
    '''Target along length of bone: Head=0, Tail=1 

    :type:  float in [0, 1], default 0.0 
    '''

    invert_x = None
    '''Invert the X location 

    :type:  boolean, default False 
    '''

    invert_y = None
    '''Invert the Y location 

    :type:  boolean, default False 
    '''

    invert_z = None
    '''Invert the Z location 

    :type:  boolean, default False 
    '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    use_bbone_shape = None
    '''Follow shape of B-Bone segments when calculating Head/Tail position 

    :type:  boolean, default False 
    '''

    use_offset = None
    '''Add original location into copied location 

    :type:  boolean, default False 
    '''

    use_x = None
    '''Copy the target’s X location 

    :type:  boolean, default False 
    '''

    use_y = None
    '''Copy the target’s Y location 

    :type:  boolean, default False 
    '''

    use_z = None
    '''Copy the target’s Z location 

    :type:  boolean, default False 
    '''


class CopyRotationConstraint:
    '''Copy the rotation of the target '''

    invert_x = None
    '''Invert the X rotation 

    :type:  boolean, default False 
    '''

    invert_y = None
    '''Invert the Y rotation 

    :type:  boolean, default False 
    '''

    invert_z = None
    '''Invert the Z rotation 

    :type:  boolean, default False 
    '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    use_offset = None
    '''Add original rotation into copied rotation 

    :type:  boolean, default False 
    '''

    use_x = None
    '''Copy the target’s X rotation 

    :type:  boolean, default False 
    '''

    use_y = None
    '''Copy the target’s Y rotation 

    :type:  boolean, default False 
    '''

    use_z = None
    '''Copy the target’s Z rotation 

    :type:  boolean, default False 
    '''


class CopyScaleConstraint:
    '''Copy the scale of the target '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    use_offset = None
    '''Add original scale into copied scale 

    :type:  boolean, default False 
    '''

    use_x = None
    '''Copy the target’s X scale 

    :type:  boolean, default False 
    '''

    use_y = None
    '''Copy the target’s Y scale 

    :type:  boolean, default False 
    '''

    use_z = None
    '''Copy the target’s Z scale 

    :type:  boolean, default False 
    '''


class CopyTransformsConstraint:
    '''Copy all the transforms of the target '''

    head_tail = None
    '''Target along length of bone: Head=0, Tail=1 

    :type:  float in [0, 1], default 0.0 
    '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    use_bbone_shape = None
    '''Follow shape of B-Bone segments when calculating Head/Tail position 

    :type:  boolean, default False 
    '''


class CorrectiveSmoothModifier:
    '''Correct distortion caused by deformation '''

    factor = None
    '''Smooth factor effect 

    :type:  float in [-inf, inf], default 0.0 
    '''

    invert_vertex_group = None
    '''Invert vertex group influence 

    :type:  boolean, default False 
    '''

    is_bind = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    iterations = None
    '''

    :type:  int in [-32768, 32767], default 0 
    '''

    rest_source = None
    '''Select the source of rest positions 

    :type:  enum in [‘ORCO’, ‘BIND’], default ‘ORCO’ 
    '''

    smooth_type = None
    '''Method used for smoothing 

    :type:  enum in [‘SIMPLE’, ‘LENGTH_WEIGHTED’], default ‘SIMPLE’ 
    '''

    use_only_smooth = None
    '''Apply smoothing without reconstructing the surface 

    :type:  boolean, default False 
    '''

    use_pin_boundary = None
    '''Excludes boundary vertices from being smoothed 

    :type:  boolean, default False 
    '''

    vertex_group = None
    '''Name of Vertex Group which determines influence of modifier per point 

    :type:  string, default “”, (never None) 
    '''


class CrossSequence:
    '''Cross Sequence '''

    input_1 = None
    '''First input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_2 = None
    '''Second input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''


class Curve:
    '''Curve data-block storing curves, splines and NURBS '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    bevel_depth = None
    '''Bevel depth when not using a bevel object 

    :type:  float in [-inf, inf], default 0.0 
    '''

    bevel_factor_end = None
    '''Factor that defines to where beveling of spline happens (0=to the very beginning, 1=to the very end) 

    :type:  float in [0, 1], default 0.0 
    '''

    bevel_factor_mapping_end = None
    '''Determines how the end bevel factor is mapped to a spline 

    :type:  enum in [‘RESOLUTION’, ‘SEGMENTS’, ‘SPLINE’], default ‘RESOLUTION’ 
    '''

    bevel_factor_mapping_start = None
    '''Determines how the start bevel factor is mapped to a spline 

    :type:  enum in [‘RESOLUTION’, ‘SEGMENTS’, ‘SPLINE’], default ‘RESOLUTION’ 
    '''

    bevel_factor_start = None
    '''Factor that defines from where beveling of spline happens (0=from the very beginning, 1=from the very end) 

    :type:  float in [0, 1], default 0.0 
    '''

    bevel_object = None
    '''Curve object name that defines the bevel shape 

    :type:  Object 
    '''

    bevel_resolution = None
    '''Bevel resolution when depth is non-zero and no specific bevel object has been defined 

    :type:  int in [0, 32], default 0 
    '''

    cycles = None
    '''Cycles mesh settings 

    :type:  CyclesMeshSettings, (readonly) 
    '''

    dimensions = None
    '''Select 2D or 3D curve type 

    :type:  enum in [‘2D’, ‘3D’], default ‘2D’ 
    '''

    eval_time = None
    '''Parametric position along the length of the curve that Objects ‘following’ it should be at (position is evaluated by dividing by the ‘Path Length’ value) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    extrude = None
    '''Amount of curve extrusion when not using a bevel object 

    :type:  float in [0, inf], default 0.0 
    '''

    fill_mode = None
    '''Mode of filling curve 

    :type:  enum in [‘FULL’, ‘BACK’, ‘FRONT’, ‘HALF’], default ‘FULL’ 
    '''

    is_editmode = None
    '''True when used in editmode 

    :type:  boolean, default False, (readonly) 
    '''

    materials = None
    '''

    :type:  IDMaterials bpy_prop_collection of Material, (readonly) 
    '''

    offset = None
    '''Offset the curve to adjust the width of a text 

    :type:  float in [-inf, inf], default 0.0 
    '''

    path_duration = None
    '''The number of frames that are needed to traverse the path, defining the maximum value for the ‘Evaluation Time’ setting 

    :type:  int in [1, 500000], default 0 
    '''

    render_resolution_u = None
    '''Surface resolution in U direction used while rendering (zero uses preview resolution) 

    :type:  int in [0, 1024], default 0 
    '''

    render_resolution_v = None
    '''Surface resolution in V direction used while rendering (zero uses preview resolution) 

    :type:  int in [0, 1024], default 0 
    '''

    resolution_u = None
    '''Surface resolution in U direction 

    :type:  int in [1, 1024], default 0 
    '''

    resolution_v = None
    '''Surface resolution in V direction 

    :type:  int in [1, 1024], default 0 
    '''

    shape_keys = None
    '''

    :type:  Key, (readonly) 
    '''

    show_handles = None
    '''Display Bezier handles in editmode 

    :type:  boolean, default False 
    '''

    show_normal_face = None
    '''Display 3D curve normals in editmode 

    :type:  boolean, default False 
    '''

    splines = None
    '''Collection of splines in this curve data object 

    :type:  CurveSplines bpy_prop_collection of Spline, (readonly) 
    '''

    taper_object = None
    '''Curve object name that defines the taper (width) 

    :type:  Object 
    '''

    texspace_location = None
    '''Texture space location 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    texspace_size = None
    '''Texture space size 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    twist_mode = None
    '''The type of tilt calculation for 3D Curves 

    :type:  enum in [‘Z_UP’, ‘MINIMUM’, ‘TANGENT’], default ‘Z_UP’ 
    '''

    twist_smooth = None
    '''Smoothing iteration for tangents 

    :type:  float in [-inf, inf], default 0.0 
    '''

    use_auto_texspace = None
    '''Adjust active object’s texture space automatically when transforming object 

    :type:  boolean, default False 
    '''

    use_deform_bounds = None
    '''Option for curve-deform: Use the mesh bounds to clamp the deformation 

    :type:  boolean, default False 
    '''

    use_fill_caps = None
    '''Fill caps for beveled curves 

    :type:  boolean, default False 
    '''

    use_fill_deform = None
    '''Fill curve after applying shape keys and all modifiers 

    :type:  boolean, default False 
    '''

    use_map_taper = None
    '''Map effect of taper object on actually beveled curve 

    :type:  boolean, default False 
    '''

    use_path = None
    '''Enable the curve to become a translation path 

    :type:  boolean, default False 
    '''

    use_path_follow = None
    '''Make curve path children to rotate along the path 

    :type:  boolean, default False 
    '''

    use_radius = None
    '''Option for paths and curve-deform: apply the curve radius with path following it and deforming 

    :type:  boolean, default False 
    '''

    use_stretch = None
    '''Option for curve-deform: make deformed child to stretch along entire path 

    :type:  boolean, default False 
    '''

    use_uv_as_generated = None
    '''Uses the UV values as Generated textured coordinates 

    :type:  boolean, default False 
    '''

    def transform(self, matrix, shape_keys=False):
        '''Transform curve by a matrix 

        :param matrix: Matrix 
        :type matrix: float array of 16 items in [-inf, inf]
        :param shape_keys: Transform Shape Keys 
        :type shape_keys: boolean, (optional)
        '''
        pass

    def validate_material_indices(self):
        '''Validate material indices of splines or letters, return True when the curve has had invalid indices corrected (to default 0) 

        :rtype: boolean 
        :return:  Result 
        '''
        pass


class CurveMap:
    '''Curve in a curve mapping '''

    extend = None
    '''Extrapolate the curve or extend it horizontally 

    :type:  enum in [‘HORIZONTAL’, ‘EXTRAPOLATED’], default ‘HORIZONTAL’ 
    '''

    points = None
    '''

    :type:  CurveMapPoints bpy_prop_collection of CurveMapPoint, (readonly) 
    '''

    def evaluate(self, position):
        '''Evaluate curve at given location 

        :param position: Position, Position to evaluate curve at 
        :type position: float in [-inf, inf]
        :rtype: float in [-inf, inf] 
        :return:  Value, Value of curve at given location 
        '''
        pass


class CurveMapPoint:
    '''Point of a curve used for a curve mapping '''

    handle_type = None
    '''Curve interpolation at this point: Bezier or vector 

    :type:  enum in [‘AUTO’, ‘AUTO_CLAMPED’, ‘VECTOR’], default ‘AUTO’ 
    '''

    location = None
    '''X/Y coordinates of the curve point 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    select = None
    '''Selection state of the curve point 

    :type:  boolean, default False 
    '''


class CurveMapPoints:
    '''Collection of Curve Map Points '''

    def new(self, position, value):
        '''Add point to CurveMap 

        :param position: Position, Position to add point 
        :type position: float in [-inf, inf]
        :param value: Value, Value of point 
        :type value: float in [-inf, inf]
        :rtype: CurveMapPoint 
        :return:  New point 
        '''
        pass

    def remove(self, point):
        '''Delete point from CurveMap 

        :param point: PointElement to remove 
        :type point: CurveMapPoint, (never None)
        '''
        pass


class CurveMapping:
    '''Curve mapping to map color, vector and scalar values to other values using a user defined curve '''

    black_level = None
    '''For RGB curves, the color that black is mapped to 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    clip_max_x = None
    '''

    :type:  float in [-100, 100], default 0.0 
    '''

    clip_max_y = None
    '''

    :type:  float in [-100, 100], default 0.0 
    '''

    clip_min_x = None
    '''

    :type:  float in [-100, 100], default 0.0 
    '''

    clip_min_y = None
    '''

    :type:  float in [-100, 100], default 0.0 
    '''

    curves = None
    '''

    :type:  bpy_prop_collection of CurveMap, (readonly) 
    '''

    use_clip = None
    '''Force the curve view to fit a defined boundary 

    :type:  boolean, default False 
    '''

    white_level = None
    '''For RGB curves, the color that white is mapped to 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    def update(self):
        '''Update curve mapping after making changes 

        '''
        pass

    def initialize(self):
        '''Initialize curve 

        '''
        pass


class CurveModifier:
    '''Curve deformation modifier '''

    deform_axis = None
    '''The axis that the curve deforms along 

    :type:  enum in [‘POS_X’, ‘POS_Y’, ‘POS_Z’, ‘NEG_X’, ‘NEG_Y’, ‘NEG_Z’], default ‘POS_X’ 
    '''

    object = None
    '''Curve object to deform with 

    :type:  Object 
    '''

    vertex_group = None
    '''Name of Vertex Group which determines influence of modifier per point 

    :type:  string, default “”, (never None) 
    '''


class CurvePaintSettings:
    corner_angle = None
    '''Angles above this are considered corners 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    curve_type = None
    '''Type of curve to use for new strokes 

    :type:  enum in [‘POLY’, ‘BEZIER’], default ‘POLY’ 
    '''

    depth_mode = None
    '''Method of projecting depth 

    :type:  enum in [‘CURSOR’, ‘SURFACE’], default ‘CURSOR’ 
    '''

    error_threshold = None
    '''Allow deviation for a smoother, less precise line 

    :type:  int in [1, 100], default 0 
    '''

    fit_method = None
    '''Curve fitting method 

    :type:  enum in [‘REFIT’, ‘SPLIT’], default ‘REFIT’ 
    '''

    radius_max = None
    '''Radius to use when the maximum pressure is applied (or when a tablet isn’t used) 

    :type:  float in [0, 100], default 0.0 
    '''

    radius_min = None
    '''Minimum radius when the minimum pressure is applied (also the minimum when tapering) 

    :type:  float in [0, 100], default 0.0 
    '''

    radius_taper_end = None
    '''Taper factor for the radius of each point along the curve 

    :type:  float in [0, 10], default 0.0 
    '''

    radius_taper_start = None
    '''Taper factor for the radius of each point along the curve 

    :type:  float in [0, 1], default 0.0 
    '''

    surface_offset = None
    '''Offset the stroke from the surface 

    :type:  float in [-10, 10], default 0.0 
    '''

    surface_plane = None
    '''Plane for projected stroke 

    :type:  enum in [‘NORMAL_VIEW’, ‘NORMAL_SURFACE’, ‘VIEW’], default ‘NORMAL_VIEW’ 
    '''

    use_corners_detect = None
    '''Detect corners and use non-aligned handles 

    :type:  boolean, default False 
    '''

    use_offset_absolute = None
    '''Apply a fixed offset (don’t scale by the radius) 

    :type:  boolean, default False 
    '''

    use_pressure_radius = None
    '''Map tablet pressure to curve radius 

    :type:  boolean, default False 
    '''

    use_stroke_endpoints = None
    '''Use the start of the stroke for the depth 

    :type:  boolean, default False 
    '''


class CurveSplines:
    '''Collection of curve splines '''

    active = None
    '''Active curve spline 

    :type:  Spline 
    '''

    def new(self, type):
        '''Add a new spline to the curve 

        :param type: type for the new spline 
        :type type: enum in ['POLY', 'BEZIER', 'BSPLINE', 'CARDINAL', 'NURBS']
        :rtype: Spline 
        :return:  The newly created spline 
        '''
        pass

    def remove(self, spline):
        '''Remove a spline from a curve 

        :param spline: The spline to remove 
        :type spline: Spline, (never None)
        '''
        pass

    def clear(self):
        '''Remove all splines from a curve 

        '''
        pass


class CurvesModifier:
    '''RGB curves modifier for sequence strip '''

    curve_mapping = None
    '''

    :type:  CurveMapping, (readonly) 
    '''


class CyclesCameraSettings:
    aperture_blades = None
    '''Number of blades in aperture for polygonal bokeh (at least 3) 

    :type:  int in [0, 100], default 0 
    '''

    aperture_fstop = None
    '''F-stop ratio (lower numbers give more defocus, higher numbers give a sharper image) 

    :type:  float in [0, inf], default 5.6 
    '''

    aperture_ratio = None
    '''Distortion to simulate anamorphic lens bokeh 

    :type:  float in [0.01, inf], default 1.0 
    '''

    aperture_rotation = None
    '''Rotation of blades in aperture 

    :type:  float in [-inf, inf], default 0.0 
    '''

    aperture_size = None
    '''Radius of the aperture for depth of field (higher values give more defocus) 

    :type:  float in [0, inf], default 0.0 
    '''

    aperture_type = None
    '''Use f-stop number or aperture radius 

    :type:  enum in [‘RADIUS’, ‘FSTOP’], default ‘RADIUS’ 
    '''

    fisheye_fov = None
    '''Field of view for the fisheye lens 

    :type:  float in [0.1745, 31.4159], default 3.14159 
    '''

    fisheye_lens = None
    '''Lens focal length (mm) 

    :type:  float in [0.01, 100], default 10.5 
    '''

    latitude_max = None
    '''Maximum latitude (vertical angle) for the equirectangular lens 

    :type:  float in [-1.5708, 1.5708], default 1.5708 
    '''

    latitude_min = None
    '''Minimum latitude (vertical angle) for the equirectangular lens 

    :type:  float in [-1.5708, 1.5708], default -1.5708 
    '''

    longitude_max = None
    '''Maximum longitude (horizontal angle) for the equirectangular lens 

    :type:  float in [-3.14159, 3.14159], default 3.14159 
    '''

    longitude_min = None
    '''Minimum longitude (horizontal angle) for the equirectangular lens 

    :type:  float in [-3.14159, 3.14159], default -3.14159 
    '''

    panorama_type = None
    '''Distortion to use for the calculation 

    :type:  enum in [‘EQUIRECTANGULAR’, ‘FISHEYE_EQUIDISTANT’, ‘FISHEYE_EQUISOLID’, ‘MIRRORBALL’], default ‘FISHEYE_EQUISOLID’ 
    '''


class CyclesCurveRenderSettings:
    cull_backfacing = None
    '''Do not test the back-face of each strand 

    :type:  boolean, default True 
    '''

    maximum_width = None
    '''Maximum extension that strand radius can be increased by 

    :type:  float in [0, 100], default 0.1 
    '''

    minimum_width = None
    '''Minimal pixel width for strands (0 - deactivated) 

    :type:  float in [0, 100], default 0.0 
    '''

    primitive = None
    '''Type of primitive used for hair rendering 

    :type:  enum in [‘TRIANGLES’, ‘LINE_SEGMENTS’, ‘CURVE_SEGMENTS’], default ‘LINE_SEGMENTS’ 
    '''

    resolution = None
    '''Resolution of generated mesh 

    :type:  int in [3, 64], default 3 
    '''

    shape = None
    '''Form of hair 

    :type:  enum in [‘RIBBONS’, ‘THICK’], default ‘THICK’ 
    '''

    subdivisions = None
    '''Number of subdivisions used in Cardinal curve intersection (power of 2) 

    :type:  int in [0, 24], default 4 
    '''

    use_curves = None
    '''Activate Cycles hair rendering for particle system 

    :type:  boolean, default True 
    '''


class CyclesCurveSettings:
    radius_scale = None
    '''Multiplier of width properties 

    :type:  float in [0, 1000], default 0.01 
    '''

    root_width = None
    '''Strand’s width at root 

    :type:  float in [0, 1000], default 1.0 
    '''

    shape = None
    '''Strand shape parameter 

    :type:  float in [-1, 1], default 0.0 
    '''

    tip_width = None
    '''Strand’s width at tip 

    :type:  float in [0, 1000], default 0.0 
    '''

    use_closetip = None
    '''Set tip radius to zero 

    :type:  boolean, default True 
    '''


class CyclesLampSettings:
    cast_shadow = None
    '''Lamp casts shadows 

    :type:  boolean, default True 
    '''

    is_portal = None
    '''Use this area lamp to guide sampling of the background, note that this will make the lamp invisible 

    :type:  boolean, default False 
    '''

    max_bounces = None
    '''Maximum number of bounces the light will contribute to the render 

    :type:  int in [0, 1024], default 1024 
    '''

    samples = None
    '''Number of light samples to render for each AA sample 

    :type:  int in [1, 10000], default 1 
    '''

    use_multiple_importance_sampling = None
    '''Use multiple importance sampling for the lamp, reduces noise for area lamps and sharp glossy materials 

    :type:  boolean, default True 
    '''


class CyclesMaterialSettings:
    displacement_method = None
    '''Method to use for the displacement 

    :type:  enum in [‘BUMP’, ‘TRUE’, ‘BOTH’], default ‘BUMP’ 
    '''

    homogeneous_volume = None
    '''When using volume rendering, assume volume has the same density everywhere (not using any textures), for faster rendering 

    :type:  boolean, default False 
    '''

    sample_as_light = None
    '''Use multiple importance sampling for this material, disabling may reduce overall noise for large objects that emit little light compared to other light sources 

    :type:  boolean, default True 
    '''

    use_transparent_shadow = None
    '''Use transparent shadows for this material if it contains a Transparent BSDF, disabling will render faster but not give accurate shadows 

    :type:  boolean, default True 
    '''

    volume_interpolation = None
    '''Interpolation method to use for smoke/fire volumes 

    :type:  enum in [‘LINEAR’, ‘CUBIC’], default ‘LINEAR’ 
    '''

    volume_sampling = None
    '''Sampling method to use for volumes 

    :type:  enum in [‘DISTANCE’, ‘EQUIANGULAR’, ‘MULTIPLE_IMPORTANCE’], default ‘MULTIPLE_IMPORTANCE’ 
    '''


class CyclesMeshSettings:
    pass


class CyclesObjectSettings:
    dicing_rate = None
    '''Multiplier for scene dicing rate (located in the Geometry Panel) 

    :type:  float in [0.1, 1000], default 1.0 
    '''

    motion_steps = None
    '''Control accuracy of deformation motion blur, more steps gives more memory usage (actual number of steps is 2^(steps - 1)) 

    :type:  int in [1, inf], default 1 
    '''

    use_adaptive_subdivision = None
    '''Use adaptive render time subdivision 

    :type:  boolean, default False 
    '''

    use_camera_cull = None
    '''Allow this object and its duplicators to be culled by camera space culling 

    :type:  boolean, default False 
    '''

    use_deform_motion = None
    '''Use deformation motion blur for this object 

    :type:  boolean, default True 
    '''

    use_motion_blur = None
    '''Use motion blur for this object 

    :type:  boolean, default True 
    '''


class CyclesRenderSettings:
    aa_samples = None
    '''Number of antialiasing samples to render for each pixel 

    :type:  int in [1, 2097151], default 4 
    '''

    ao_samples = None
    '''Number of ambient occlusion samples to render for each AA sample 

    :type:  int in [1, 1024], default 1 
    '''

    bake_type = None
    '''Type of pass to bake 

    :type:  enum in [‘COMBINED’, ‘AO’, ‘SHADOW’, ‘NORMAL’, ‘UV’, ‘EMIT’, ‘ENVIRONMENT’, ‘DIFFUSE’, ‘GLOSSY’, ‘TRANSMISSION’, ‘SUBSURFACE’], default ‘COMBINED’ 
    '''

    blur_glossy = None
    '''Adaptively blur glossy shaders after blurry bounces, to reduce noise at the cost of accuracy 

    :type:  float in [0, 10], default 0.0 
    '''

    camera_cull_margin = None
    '''Margin for the camera space culling 

    :type:  float in [0, 5], default 0.1 
    '''

    caustics_reflective = None
    '''Use reflective caustics, resulting in a brighter image (more noise but added realism) 

    :type:  boolean, default True 
    '''

    caustics_refractive = None
    '''Use refractive caustics, resulting in a brighter image (more noise but added realism) 

    :type:  boolean, default True 
    '''

    debug_bvh_type = None
    '''Choose between faster updates, or faster render 

    :type:  enum in [‘DYNAMIC_BVH’, ‘STATIC_BVH’], default ‘DYNAMIC_BVH’ 
    '''

    debug_cancel_timeout = None
    '''

    :type:  float in [0.01, 10], default 0.1 
    '''

    debug_opencl_device_type = None
    '''

    :type:  enum in [‘NONE’, ‘ALL’, ‘DEFAULT’, ‘CPU’, ‘GPU’, ‘ACCELERATOR’], default ‘ALL’ 
    '''

    debug_opencl_kernel_type = None
    '''

    :type:  enum in [‘DEFAULT’, ‘MEGA’, ‘SPLIT’], default ‘DEFAULT’ 
    '''

    debug_reset_timeout = None
    '''

    :type:  float in [0.01, 10], default 0.1 
    '''

    debug_text_timeout = None
    '''

    :type:  float in [0.01, 10], default 1.0 
    '''

    debug_tile_size = None
    '''

    :type:  int in [1, 4096], default 1024 
    '''

    debug_use_cpu_avx = None
    '''

    :type:  boolean, default True 
    '''

    debug_use_cpu_avx2 = None
    '''

    :type:  boolean, default True 
    '''

    debug_use_cpu_sse2 = None
    '''

    :type:  boolean, default True 
    '''

    debug_use_cpu_sse3 = None
    '''

    :type:  boolean, default True 
    '''

    debug_use_cpu_sse41 = None
    '''

    :type:  boolean, default True 
    '''

    debug_use_cuda_adaptive_compile = None
    '''

    :type:  boolean, default False 
    '''

    debug_use_hair_bvh = None
    '''Use special type BVH optimized for hair (uses more ram but renders faster) 

    :type:  boolean, default True 
    '''

    debug_use_opencl_debug = None
    '''

    :type:  boolean, default False 
    '''

    debug_use_qbvh = None
    '''

    :type:  boolean, default True 
    '''

    debug_use_spatial_splits = None
    '''Use BVH spatial splits: longer builder time, faster render 

    :type:  boolean, default False 
    '''

    device = None
    '''Device to use for rendering 

    :type:  enum in [‘CPU’, ‘GPU’], default ‘CPU’ 
    '''

    dicing_rate = None
    '''Size of a micropolygon in pixels 

    :type:  float in [0.1, 1000], default 1.0 
    '''

    diffuse_bounces = None
    '''Maximum number of diffuse reflection bounces, bounded by total maximum 

    :type:  int in [0, 1024], default 4 
    '''

    diffuse_samples = None
    '''Number of diffuse bounce samples to render for each AA sample 

    :type:  int in [1, 1024], default 1 
    '''

    feature_set = None
    '''Feature set to use for rendering 

    :type:  enum in [‘SUPPORTED’, ‘EXPERIMENTAL’], default ‘SUPPORTED’ 
    '''

    film_exposure = None
    '''Image brightness scale 

    :type:  float in [0, 10], default 1.0 
    '''

    film_transparent = None
    '''World background is transparent with premultiplied alpha 

    :type:  boolean, default False 
    '''

    filter_type = None
    '''Pixel filter type 

    :type:  enum in [‘BOX’, ‘GAUSSIAN’, ‘BLACKMAN_HARRIS’], default ‘BLACKMAN_HARRIS’ 
    '''

    filter_width = None
    '''Pixel filter width 

    :type:  float in [0.01, 10], default 1.5 
    '''

    glossy_bounces = None
    '''Maximum number of glossy reflection bounces, bounded by total maximum 

    :type:  int in [0, 1024], default 4 
    '''

    glossy_samples = None
    '''Number of glossy bounce samples to render for each AA sample 

    :type:  int in [1, 1024], default 1 
    '''

    max_bounces = None
    '''Total maximum number of bounces 

    :type:  int in [0, 1024], default 12 
    '''

    max_subdivisions = None
    '''Stop subdividing when this level is reached even if the dice rate would produce finer tessellation 

    :type:  int in [0, 16], default 12 
    '''

    mesh_light_samples = None
    '''Number of mesh emission light samples to render for each AA sample 

    :type:  int in [1, 1024], default 1 
    '''

    min_bounces = None
    '''Minimum number of bounces, setting this lower than the maximum enables probabilistic path termination (faster but noisier) 

    :type:  int in [0, 1024], default 3 
    '''

    motion_blur_position = None
    '''Offset for the shutter’s time interval, allows to change the motion blur trails 

    :type:  enum in [‘START’, ‘CENTER’, ‘END’], default ‘CENTER’ 
    '''

    pixel_filter_type = None
    '''Pixel filter type 

    :type:  enum in [‘BOX’, ‘GAUSSIAN’, ‘BLACKMAN_HARRIS’], default ‘BLACKMAN_HARRIS’ 
    '''

    preview_aa_samples = None
    '''Number of antialiasing samples to render in the viewport, unlimited if 0 

    :type:  int in [0, 2097151], default 4 
    '''

    preview_active_layer = None
    '''Preview active render layer in viewport 

    :type:  boolean, default False 
    '''

    preview_dicing_rate = None
    '''Size of a micropolygon in pixels during preview render 

    :type:  float in [0.1, 1000], default 8.0 
    '''

    preview_pause = None
    '''Pause all viewport preview renders 

    :type:  boolean, default False 
    '''

    preview_samples = None
    '''Number of samples to render in the viewport, unlimited if 0 

    :type:  int in [0, inf], default 32 
    '''

    preview_start_resolution = None
    '''Resolution to start rendering preview at, progressively increasing it to the full viewport size 

    :type:  int in [8, 16384], default 64 
    '''

    progressive = None
    '''Method to sample lights and materials 

    :type:  enum in [‘BRANCHED_PATH’, ‘PATH’], default ‘PATH’ 
    '''

    rolling_shutter_duration = None
    '''Scanline “exposure” time for the rolling shutter effect 

    :type:  float in [0, 1], default 0.1 
    '''

    rolling_shutter_type = None
    '''Type of rolling shutter effect matching CMOS-based cameras 

    :type:  enum in [‘NONE’, ‘TOP’], default ‘NONE’ 
    '''

    sample_all_lights_direct = None
    '''Sample all lights (for direct samples), rather than randomly picking one 

    :type:  boolean, default True 
    '''

    sample_all_lights_indirect = None
    '''Sample all lights (for indirect samples), rather than randomly picking one 

    :type:  boolean, default True 
    '''

    sample_clamp_direct = None
    '''If non-zero, the maximum value for a direct sample, higher values will be scaled down to avoid too much noise and slow convergence at the cost of accuracy 

    :type:  float in [0, inf], default 0.0 
    '''

    sample_clamp_indirect = None
    '''If non-zero, the maximum value for an indirect sample, higher values will be scaled down to avoid too much noise and slow convergence at the cost of accuracy 

    :type:  float in [0, inf], default 0.0 
    '''

    samples = None
    '''Number of samples to render for each pixel 

    :type:  int in [1, inf], default 128 
    '''

    sampling_pattern = None
    '''Random sampling pattern used by the integrator 

    :type:  enum in [‘SOBOL’, ‘CORRELATED_MUTI_JITTER’], default ‘SOBOL’ 
    '''

    seed = None
    '''Seed value for integrator to get different noise patterns 

    :type:  int in [0, inf], default 0 
    '''

    shading_system = None
    '''Use Open Shading Language (CPU rendering only) 

    :type:  boolean, default False 
    '''

    subsurface_samples = None
    '''Number of subsurface scattering samples to render for each AA sample 

    :type:  int in [1, 1024], default 1 
    '''

    tile_order = None
    '''Tile order for rendering 

    :type:  enum in [‘CENTER’, ‘RIGHT_TO_LEFT’, ‘LEFT_TO_RIGHT’, ‘TOP_TO_BOTTOM’, ‘BOTTOM_TO_TOP’, ‘HILBERT_SPIRAL’], default ‘HILBERT_SPIRAL’ 
    '''

    transmission_bounces = None
    '''Maximum number of transmission bounces, bounded by total maximum 

    :type:  int in [0, 1024], default 12 
    '''

    transmission_samples = None
    '''Number of transmission bounce samples to render for each AA sample 

    :type:  int in [1, 1024], default 1 
    '''

    transparent_max_bounces = None
    '''Maximum number of transparent bounces 

    :type:  int in [0, 1024], default 8 
    '''

    transparent_min_bounces = None
    '''Minimum number of transparent bounces, setting this lower than the maximum enables probabilistic path termination (faster but noisier) 

    :type:  int in [0, 1024], default 8 
    '''

    use_animated_seed = None
    '''Use different seed values (and hence noise patterns) at different frames 

    :type:  boolean, default False 
    '''

    use_camera_cull = None
    '''Allow objects to be culled based on the camera frustum 

    :type:  boolean, default False 
    '''

    use_layer_samples = None
    '''How to use per render layer sample settings 

    :type:  enum in [‘USE’, ‘BOUNDED’, ‘IGNORE’], default ‘USE’ 
    '''

    use_progressive_refine = None
    '''Instead of rendering each tile until it is finished, refine the whole image progressively (this renders somewhat slower, but time can be saved by manually stopping the render when the noise is low enough) 

    :type:  boolean, default False 
    '''

    use_square_samples = None
    '''Square sampling values for easier artist control 

    :type:  boolean, default False 
    '''

    use_transparent_shadows = None
    '''Use transparency of surfaces for rendering shadows 

    :type:  boolean, default True 
    '''

    volume_bounces = None
    '''Maximum number of volumetric scattering events 

    :type:  int in [0, 1024], default 0 
    '''

    volume_max_steps = None
    '''Maximum number of steps through the volume before giving up, to avoid extremely long render times with big objects or small step sizes 

    :type:  int in [2, 65536], default 1024 
    '''

    volume_samples = None
    '''Number of volume scattering samples to render for each AA sample 

    :type:  int in [1, 1024], default 1 
    '''

    volume_step_size = None
    '''Distance between volume shader samples when rendering the volume (lower values give more accurate and detailed results, but also increased render time) 

    :type:  float in [1e-07, 100000], default 0.1 
    '''


class CyclesVisibilitySettings:
    camera = None
    '''Object visibility for camera rays 

    :type:  boolean, default True 
    '''

    diffuse = None
    '''Object visibility for diffuse reflection rays 

    :type:  boolean, default True 
    '''

    glossy = None
    '''Object visibility for glossy reflection rays 

    :type:  boolean, default True 
    '''

    scatter = None
    '''Object visibility for volume scatter rays 

    :type:  boolean, default True 
    '''

    shadow = None
    '''Object visibility for shadow rays 

    :type:  boolean, default True 
    '''

    transmission = None
    '''Object visibility for transmission rays 

    :type:  boolean, default True 
    '''


class CyclesWorldSettings:
    homogeneous_volume = None
    '''When using volume rendering, assume volume has the same density everywhere(not using any textures), for faster rendering 

    :type:  boolean, default False 
    '''

    max_bounces = None
    '''Maximum number of bounces the background light will contribute to the render 

    :type:  int in [0, 1024], default 1024 
    '''

    sample_as_light = None
    '''Use multiple importance sampling for the environment, enabling for non-solid colors is recommended 

    :type:  boolean, default True 
    '''

    sample_map_resolution = None
    '''Importance map size is resolution x resolution; higher values potentially produce less noise, at the cost of memory and speed 

    :type:  int in [4, 8192], default 1024 
    '''

    samples = None
    '''Number of light samples to render for each AA sample 

    :type:  int in [1, 10000], default 1 
    '''

    volume_interpolation = None
    '''Interpolation method to use for volumes 

    :type:  enum in [‘LINEAR’, ‘CUBIC’], default ‘LINEAR’ 
    '''

    volume_sampling = None
    '''Sampling method to use for volumes 

    :type:  enum in [‘DISTANCE’, ‘EQUIANGULAR’, ‘MULTIPLE_IMPORTANCE’], default ‘EQUIANGULAR’ 
    '''


class DampedTrackConstraint:
    '''Point toward target by taking the shortest rotation path '''

    head_tail = None
    '''Target along length of bone: Head=0, Tail=1 

    :type:  float in [0, 1], default 0.0 
    '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    track_axis = None
    '''Axis that points to the target object 

    :type:  enum in [‘TRACK_X’, ‘TRACK_Y’, ‘TRACK_Z’, ‘TRACK_NEGATIVE_X’, ‘TRACK_NEGATIVE_Y’, ‘TRACK_NEGATIVE_Z’], default ‘TRACK_X’ 
    '''

    use_bbone_shape = None
    '''Follow shape of B-Bone segments when calculating Head/Tail position 

    :type:  boolean, default False 
    '''


class DataTransferModifier:
    '''Modifier transferring some data from a source mesh '''

    data_types_edges = None
    '''Which edge data layers to transfer 

    :type:  enum set in {‘SHARP_EDGE’, ‘SEAM’, ‘CREASE’, ‘BEVEL_WEIGHT_EDGE’, ‘FREESTYLE_EDGE’}, default {‘SHARP_EDGE’} 
    '''

    data_types_loops = None
    '''Which face corner data layers to transfer 

    :type:  enum set in {‘CUSTOM_NORMAL’, ‘VCOL’, ‘UV’}, default {‘CUSTOM_NORMAL’} 
    '''

    data_types_polys = None
    '''Which poly data layers to transfer 

    :type:  enum set in {‘SMOOTH’, ‘FREESTYLE_FACE’}, default {‘SMOOTH’} 
    '''

    data_types_verts = None
    '''Which vertex data layers to transfer 

    :type:  enum set in {‘VGROUP_WEIGHTS’, ‘BEVEL_WEIGHT_VERT’}, default {‘VGROUP_WEIGHTS’} 
    '''

    edge_mapping = None
    '''Method used to map source edges to destination ones 

    :type:  enum in [‘TOPOLOGY’, ‘VERT_NEAREST’, ‘NEAREST’, ‘POLY_NEAREST’, ‘EDGEINTERP_VNORPROJ’], default ‘NEAREST’ 
    '''

    invert_vertex_group = None
    '''Invert vertex group influence 

    :type:  boolean, default False 
    '''

    islands_precision = None
    '''Factor controlling precision of islands handling (typically, 0.1 should be enough, higher values can make things really slow) 

    :type:  float in [0, 1], default 0.0 
    '''

    layers_uv_select_dst = None
    '''How to match source and destination layers 

    :type:  enum in [‘ACTIVE’, ‘NAME’, ‘INDEX’], default ‘NAME’ 
    '''

    layers_uv_select_src = None
    '''Which layers to transfer, in case of multi-layers types 

    :type:  enum in [‘ACTIVE’, ‘ALL’, ‘BONE_SELECT’, ‘BONE_DEFORM’], default ‘ALL’ 
    '''

    layers_vcol_select_dst = None
    '''How to match source and destination layers 

    :type:  enum in [‘ACTIVE’, ‘NAME’, ‘INDEX’], default ‘NAME’ 
    '''

    layers_vcol_select_src = None
    '''Which layers to transfer, in case of multi-layers types 

    :type:  enum in [‘ACTIVE’, ‘ALL’, ‘BONE_SELECT’, ‘BONE_DEFORM’], default ‘ALL’ 
    '''

    layers_vgroup_select_dst = None
    '''How to match source and destination layers 

    :type:  enum in [‘ACTIVE’, ‘NAME’, ‘INDEX’], default ‘NAME’ 
    '''

    layers_vgroup_select_src = None
    '''Which layers to transfer, in case of multi-layers types 

    :type:  enum in [‘ACTIVE’, ‘ALL’, ‘BONE_SELECT’, ‘BONE_DEFORM’], default ‘ALL’ 
    '''

    loop_mapping = None
    '''Method used to map source faces’ corners to destination ones 

    :type:  enum in [‘TOPOLOGY’, ‘NEAREST_NORMAL’, ‘NEAREST_POLYNOR’, ‘NEAREST_POLY’, ‘POLYINTERP_NEAREST’, ‘POLYINTERP_LNORPROJ’], default ‘NEAREST_POLYNOR’ 
    '''

    max_distance = None
    '''Maximum allowed distance between source and destination element, for non-topology mappings 

    :type:  float in [0, inf], default 1.0 
    '''

    mix_factor = None
    '''Factor to use when applying data to destination (exact behavior depends on mix mode) 

    :type:  float in [0, 1], default 1.0 
    '''

    mix_mode = None
    '''How to affect destination elements with source values 

    :type:  enum in [‘REPLACE’, ‘ABOVE_THRESHOLD’, ‘BELOW_THRESHOLD’, ‘MIX’, ‘ADD’, ‘SUB’, ‘MUL’], default ‘REPLACE’ 
    '''

    object = None
    '''Object to transfer data from 

    :type:  Object 
    '''

    poly_mapping = None
    '''Method used to map source faces to destination ones 

    :type:  enum in [‘TOPOLOGY’, ‘NEAREST’, ‘NORMAL’, ‘POLYINTERP_PNORPROJ’], default ‘NEAREST’ 
    '''

    ray_radius = None
    '''‘Width’ of rays (especially useful when raycasting against vertices or edges) 

    :type:  float in [0, inf], default 0.0 
    '''

    use_edge_data = None
    '''Enable edge data transfer 

    :type:  boolean, default False 
    '''

    use_loop_data = None
    '''Enable face corner data transfer 

    :type:  boolean, default False 
    '''

    use_max_distance = None
    '''Source elements must be closer than given distance from destination one 

    :type:  boolean, default False 
    '''

    use_object_transform = None
    '''Evaluate source and destination meshes in global space 

    :type:  boolean, default True 
    '''

    use_poly_data = None
    '''Enable face data transfer 

    :type:  boolean, default False 
    '''

    use_vert_data = None
    '''Enable vertex data transfer 

    :type:  boolean, default False 
    '''

    vert_mapping = None
    '''Method used to map source vertices to destination ones 

    :type:  enum in [‘TOPOLOGY’, ‘NEAREST’, ‘EDGE_NEAREST’, ‘EDGEINTERP_NEAREST’, ‘POLY_NEAREST’, ‘POLYINTERP_NEAREST’, ‘POLYINTERP_VNORPROJ’], default ‘NEAREST’ 
    '''

    vertex_group = None
    '''Vertex group name for selecting the affected areas 

    :type:  string, default “”, (never None) 
    '''


class DecimateModifier:
    '''Decimation modifier '''

    angle_limit = None
    '''Only dissolve angles below this (planar only) 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    decimate_type = None
    '''

    :type:  enum in [‘COLLAPSE’, ‘UNSUBDIV’, ‘DISSOLVE’], default ‘COLLAPSE’ 
    '''

    delimit = None
    '''Limit merging geometry 

    :type:  enum set in {‘NORMAL’, ‘MATERIAL’, ‘SEAM’, ‘SHARP’, ‘UV’}, default {‘NORMAL’} 
    '''

    face_count = None
    '''The current number of faces in the decimated mesh 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    invert_vertex_group = None
    '''Invert vertex group influence (collapse only) 

    :type:  boolean, default False 
    '''

    iterations = None
    '''Number of times reduce the geometry (unsubdivide only) 

    :type:  int in [0, 32767], default 0 
    '''

    ratio = None
    '''Ratio of triangles to reduce to (collapse only) 

    :type:  float in [0, 1], default 0.0 
    '''

    symmetry_axis = None
    '''Axis of symmetry 

    :type:  enum in [‘X’, ‘Y’, ‘Z’], default ‘X’ 
    '''

    use_collapse_triangulate = None
    '''Keep triangulated faces resulting from decimation (collapse only) 

    :type:  boolean, default False 
    '''

    use_dissolve_boundaries = None
    '''Dissolve all vertices inbetween face boundaries (planar only) 

    :type:  boolean, default False 
    '''

    use_symmetry = None
    '''Maintain symmetry on an axis 

    :type:  boolean, default False 
    '''

    vertex_group = None
    '''Vertex group name (collapse only) 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_factor = None
    '''Vertex group strength 

    :type:  float in [0, 1000], default 0.0 
    '''


class DelaySensor:
    '''Sensor to send delayed events '''

    delay = None
    '''Delay in number of logic tics before the positive trigger (default 60 per second) 

    :type:  int in [0, 5000], default 0 
    '''

    duration = None
    '''If >0, delay in number of logic tics before the negative trigger following the positive trigger 

    :type:  int in [0, 5000], default 0 
    '''

    use_repeat = None
    '''Toggle repeat option (if selected, the sensor restarts after Delay+Duration logic tics) 

    :type:  boolean, default False 
    '''


class Depsgraph:
    def debug_graphviz(self, filename):
        '''debug_graphviz 

        :param filename: File Name, File in which to store graphviz debug output 
        :type filename: string, (never None)
        '''
        pass

    def debug_rebuild(self):
        '''debug_rebuild 

        '''
        pass

    def debug_stats(self):
        '''Report the number of elements in the Dependency Graph 

        '''
        pass


class DisplaceModifier:
    '''Displacement modifier '''

    direction = None
    '''

    :type:  enum in [‘X’, ‘Y’, ‘Z’, ‘NORMAL’, ‘CUSTOM_NORMAL’, ‘RGB_TO_XYZ’], default ‘X’ 
    '''

    mid_level = None
    '''Material value that gives no displacement 

    :type:  float in [-inf, inf], default 0.0 
    '''

    strength = None
    '''Amount to displace geometry 

    :type:  float in [-inf, inf], default 0.0 
    '''

    texture = None
    '''

    :type:  Texture 
    '''

    texture_coords = None
    '''

    :type:  enum in [‘LOCAL’, ‘GLOBAL’, ‘OBJECT’, ‘UV’], default ‘LOCAL’ 
    '''

    texture_coords_object = None
    '''Object to set the texture coordinates 

    :type:  Object 
    '''

    uv_layer = None
    '''UV map name 

    :type:  string, default “”, (never None) 
    '''

    vertex_group = None
    '''Name of Vertex Group which determines influence of modifier per point 

    :type:  string, default “”, (never None) 
    '''


class DisplaySafeAreas:
    '''Safe Areas used in 3D view and the VSE '''

    action = None
    '''Safe area for general elements 

    :type:  float array of 2 items in [0, 1], default (0.1, 0.05) 
    '''

    action_center = None
    '''Safe area for general elements in a different aspect ratio 

    :type:  float array of 2 items in [0, 1], default (0.15, 0.05) 
    '''

    title = None
    '''Safe area for text and graphics 

    :type:  float array of 2 items in [0, 1], default (0.035, 0.035) 
    '''

    title_center = None
    '''Safe area for text and graphics in a different aspect ratio 

    :type:  float array of 2 items in [0, 1], default (0.175, 0.05) 
    '''


class DistortedNoiseTexture:
    '''Procedural distorted noise texture '''

    distortion = None
    '''Amount of distortion 

    :type:  float in [0, 10], default 0.0 
    '''

    nabla = None
    '''Size of derivative offset used for calculating normal 

    :type:  float in [0.001, 0.1], default 0.0 
    '''

    noise_basis = None
    '''Noise basis used for turbulence 

    :type:  enum in [‘BLENDER_ORIGINAL’, ‘ORIGINAL_PERLIN’, ‘IMPROVED_PERLIN’, ‘VORONOI_F1’, ‘VORONOI_F2’, ‘VORONOI_F3’, ‘VORONOI_F4’, ‘VORONOI_F2_F1’, ‘VORONOI_CRACKLE’, ‘CELL_NOISE’], default ‘BLENDER_ORIGINAL’ 
    '''

    noise_distortion = None
    '''Noise basis for the distortion 

    :type:  enum in [‘BLENDER_ORIGINAL’, ‘ORIGINAL_PERLIN’, ‘IMPROVED_PERLIN’, ‘VORONOI_F1’, ‘VORONOI_F2’, ‘VORONOI_F3’, ‘VORONOI_F4’, ‘VORONOI_F2_F1’, ‘VORONOI_CRACKLE’, ‘CELL_NOISE’], default ‘BLENDER_ORIGINAL’ 
    '''

    noise_scale = None
    '''Scaling for noise input 

    :type:  float in [0.0001, inf], default 0.0 
    '''

    users_material = None
    '''Materials that use this texture (readonly) '''

    users_object_modifier = None
    '''Object modifiers that use this texture (readonly) '''


class DomainFluidSettings:
    '''Fluid simulation settings for the domain of a fluid simulation '''

    compressibility = None
    '''Allowed compressibility due to gravitational force for standing fluid (directly affects simulation step size) 

    :type:  float in [0.001, 0.1], default 0.0 
    '''

    end_time = None
    '''Simulation time of the last blender frame (in seconds) 

    :type:  float in [0, inf], default 0.0 
    '''

    filepath = None
    '''Directory (and/or filename prefix) to store baked fluid simulation files in 

    :type:  string, default “”, (never None) 
    '''

    fluid_mesh_vertices = None
    '''Vertices of the fluid mesh generated by simulation 

    :type:  bpy_prop_collection of FluidVertexVelocity, (readonly) 
    '''

    frame_offset = None
    '''Offset when reading baked cache 

    :type:  int in [-inf, inf], default 0 
    '''

    generate_particles = None
    '''Amount of particles to generate (0=off, 1=normal, >1=more) 

    :type:  float in [0, 10], default 0.0 
    '''

    gravity = None
    '''Gravity in X, Y and Z direction 

    :type:  float array of 3 items in [-1000.1, 1000.1], default (0.0, 0.0, 0.0) 
    '''

    grid_levels = None
    '''Number of coarsened grids to use (-1 for automatic) 

    :type:  int in [-1, 4], default 0 
    '''

    memory_estimate = None
    '''Estimated amount of memory needed for baking the domain 

    :type:  string, default “”, (readonly, never None) 
    '''

    partial_slip_factor = None
    '''Amount of mixing between no- and free-slip, 0 is no slip and 1 is free slip 

    :type:  float in [0, 1], default 0.0 
    '''

    preview_resolution = None
    '''Preview resolution in X,Y and Z direction 

    :type:  int in [1, 100], default 0 
    '''

    render_display_mode = None
    '''How to display the mesh for rendering 

    :type:  enum in [‘GEOMETRY’, ‘PREVIEW’, ‘FINAL’], default ‘GEOMETRY’ 
    '''

    resolution = None
    '''Domain resolution in X,Y and Z direction 

    :type:  int in [1, 1024], default 0 
    '''

    simulation_rate = None
    '''Fluid motion rate (0 = stationary, 1 = normal speed) 

    :type:  float in [0, 100], default 0.0 
    '''

    simulation_scale = None
    '''Size of the simulation domain in meters 

    :type:  float in [0.001, 10], default 0.0 
    '''

    slip_type = None
    '''

    :type:  enum in [‘NOSLIP’, ‘PARTIALSLIP’, ‘FREESLIP’], default ‘NOSLIP’ 
    '''

    start_time = None
    '''Simulation time of the first blender frame (in seconds) 

    :type:  float in [0, inf], default 0.0 
    '''

    surface_smooth = None
    '''Amount of surface smoothing (a value of 0 is off, 1 is normal smoothing and more than 1 is extra smoothing) 

    :type:  float in [0, 5], default 0.0 
    '''

    surface_subdivisions = None
    '''Number of isosurface subdivisions (this is necessary for the inclusion of particles into the surface generation - WARNING: can lead to longer computation times !) 

    :type:  int in [0, 5], default 0 
    '''

    threads = None
    '''Override number of threads for the simulation, 0 is automatic 

    :type:  int in [0, 1024], default 0 
    '''

    tracer_particles = None
    '''Number of tracer particles to generate 

    :type:  int in [0, 10000], default 0 
    '''

    use_reverse_frames = None
    '''Reverse fluid frames 

    :type:  boolean, default False 
    '''

    use_speed_vectors = None
    '''Generate speed vectors for vector blur 

    :type:  boolean, default False 
    '''

    use_surface_noobs = None
    '''Removes the air gap between fluid surface and obstacles - WARNING: Can result in a dissolving surface in other areas 

    :type:  boolean, default False 
    '''

    use_time_override = None
    '''Use a custom start and end time (in seconds) instead of the scene’s timeline 

    :type:  boolean, default False 
    '''

    viewport_display_mode = None
    '''How to display the mesh in the viewport 

    :type:  enum in [‘GEOMETRY’, ‘PREVIEW’, ‘FINAL’], default ‘GEOMETRY’ 
    '''

    viscosity_base = None
    '''Viscosity setting: value that is multiplied by 10 to the power of (exponent*-1) 

    :type:  float in [0, 10], default 0.0 
    '''

    viscosity_exponent = None
    '''Negative exponent for the viscosity value (to simplify entering small values e.g. 5*10^-6) 

    :type:  int in [0, 10], default 0 
    '''


class DopeSheet:
    '''Settings for filtering the channels shown in animation editors '''

    filter_fcurve_name = None
    '''F-Curve live filtering string 

    :type:  string, default “”, (never None) 
    '''

    filter_group = None
    '''Group that included object should be a member of 

    :type:  Group 
    '''

    filter_text = None
    '''Live filtering string 

    :type:  string, default “”, (never None) 
    '''

    show_armatures = None
    '''Include visualization of armature related animation data 

    :type:  boolean, default False 
    '''

    show_cameras = None
    '''Include visualization of camera related animation data 

    :type:  boolean, default False 
    '''

    show_curves = None
    '''Include visualization of curve related animation data 

    :type:  boolean, default False 
    '''

    show_datablock_filters = None
    '''Show options for whether channels related to certain types of data are included 

    :type:  boolean, default False 
    '''

    show_expanded_summary = None
    '''Collapse summary when shown, so all other channels get hidden (Dope Sheet editors only) 

    :type:  boolean, default False 
    '''

    show_gpencil = None
    '''Include visualization of Grease Pencil related animation data and frames 

    :type:  boolean, default False 
    '''

    show_gpencil_3d_only = None
    '''Only show Grease Pencil datablocks used as part of the active scene 

    :type:  boolean, default False 
    '''

    show_hidden = None
    '''Include channels from objects/bone that are not visible 

    :type:  boolean, default False 
    '''

    show_lamps = None
    '''Include visualization of lamp related animation data 

    :type:  boolean, default False 
    '''

    show_lattices = None
    '''Include visualization of lattice related animation data 

    :type:  boolean, default False 
    '''

    show_linestyles = None
    '''Include visualization of Line Style related Animation data 

    :type:  boolean, default False 
    '''

    show_materials = None
    '''Include visualization of material related animation data 

    :type:  boolean, default False 
    '''

    show_meshes = None
    '''Include visualization of mesh related animation data 

    :type:  boolean, default False 
    '''

    show_metaballs = None
    '''Include visualization of metaball related animation data 

    :type:  boolean, default False 
    '''

    show_missing_nla = None
    '''Include animation data blocks with no NLA data (NLA editor only) 

    :type:  boolean, default False 
    '''

    show_modifiers = None
    '''Include visualization of animation data related to data-blocks linked to modifiers 

    :type:  boolean, default False 
    '''

    show_nodes = None
    '''Include visualization of node related animation data 

    :type:  boolean, default False 
    '''

    show_only_errors = None
    '''Only include F-Curves and drivers that are disabled or have errors 

    :type:  boolean, default False 
    '''

    show_only_group_objects = None
    '''Only include channels from objects in the specified group 

    :type:  boolean, default False 
    '''

    show_only_matching_fcurves = None
    '''Only include F-Curves with names containing search text 

    :type:  boolean, default False 
    '''

    show_only_selected = None
    '''Only include channels relating to selected objects and data 

    :type:  boolean, default False 
    '''

    show_particles = None
    '''Include visualization of particle related animation data 

    :type:  boolean, default False 
    '''

    show_scenes = None
    '''Include visualization of scene related animation data 

    :type:  boolean, default False 
    '''

    show_shapekeys = None
    '''Include visualization of shape key related animation data 

    :type:  boolean, default False 
    '''

    show_speakers = None
    '''Include visualization of speaker related animation data 

    :type:  boolean, default False 
    '''

    show_summary = None
    '''Display an additional ‘summary’ line (Dope Sheet editors only) 

    :type:  boolean, default False 
    '''

    show_textures = None
    '''Include visualization of texture related animation data 

    :type:  boolean, default False 
    '''

    show_transforms = None
    '''Include visualization of object-level animation data (mostly transforms) 

    :type:  boolean, default False 
    '''

    show_worlds = None
    '''Include visualization of world related animation data 

    :type:  boolean, default False 
    '''

    source = None
    '''ID-Block representing source data, usually ID_SCE (i.e. Scene) 

    :type:  ID, (readonly) 
    '''

    use_datablock_sort = None
    '''Alphabetically sorts datablocks - mainly objects in the scene (disable to increase viewport speed) 

    :type:  boolean, default False 
    '''

    use_filter_text = None
    '''Only include channels with names containing search text 

    :type:  boolean, default False 
    '''

    use_multi_word_filter = None
    '''Perform fuzzy/multi-word matching (WARNING: May be slow) 

    :type:  boolean, default False 
    '''


class Driver:
    '''Driver for the value of a setting based on an external value '''

    expression = None
    '''Expression to use for Scripted Expression 

    :type:  string, default “”, (never None) 
    '''

    is_valid = None
    '''Driver could not be evaluated in past, so should be skipped 

    :type:  boolean, default False 
    '''

    show_debug_info = None
    '''Show intermediate values for the driver calculations to allow debugging of drivers 

    :type:  boolean, default False 
    '''

    type = None
    '''Driver type 

    :type:  enum in [‘AVERAGE’, ‘SUM’, ‘SCRIPTED’, ‘MIN’, ‘MAX’], default ‘AVERAGE’ 
    '''

    use_self = None
    '''Include a ‘self’ variable in the name-space, so drivers can easily reference the data being modified (object, bone, etc…) 

    :type:  boolean, default False 
    '''

    variables = None
    '''Properties acting as inputs for this driver 

    :type:  ChannelDriverVariables bpy_prop_collection of DriverVariable, (readonly) 
    '''


class DriverTarget:
    '''Source of input values for driver variables '''

    bone_target = None
    '''Name of PoseBone to use as target 

    :type:  string, default “”, (never None) 
    '''

    data_path = None
    '''RNA Path (from ID-block) to property used 

    :type:  string, default “”, (never None) 
    '''

    id = None
    '''ID-block that the specific property used can be found from (id_type property must be set first) 

    :type:  ID 
    '''

    id_type = None
    '''Type of ID-block that can be used 

    :type:  enum in [‘ACTION’, ‘ARMATURE’, ‘BRUSH’, ‘CAMERA’, ‘CACHEFILE’, ‘CURVE’, ‘FONT’, ‘GREASEPENCIL’, ‘GROUP’, ‘IMAGE’, ‘KEY’, ‘LAMP’, ‘LIBRARY’, ‘LINESTYLE’, ‘LATTICE’, ‘MASK’, ‘MATERIAL’, ‘META’, ‘MESH’, ‘MOVIECLIP’, ‘NODETREE’, ‘OBJECT’, ‘PAINTCURVE’, ‘PALETTE’, ‘PARTICLE’, ‘SCENE’, ‘SCREEN’, ‘SOUND’, ‘SPEAKER’, ‘TEXT’, ‘TEXTURE’, ‘WINDOWMANAGER’, ‘WORLD’], default ‘OBJECT’ 
    '''

    transform_space = None
    '''Space in which transforms are used 

    :type:  enum in [‘WORLD_SPACE’, ‘TRANSFORM_SPACE’, ‘LOCAL_SPACE’], default ‘WORLD_SPACE’ 
    '''

    transform_type = None
    '''Driver variable type 

    :type:  enum in [‘LOC_X’, ‘LOC_Y’, ‘LOC_Z’, ‘ROT_X’, ‘ROT_Y’, ‘ROT_Z’, ‘SCALE_X’, ‘SCALE_Y’, ‘SCALE_Z’], default ‘LOC_X’ 
    '''


class DriverVariable:
    '''Variable from some source/target for driver relationship '''

    is_name_valid = None
    '''Is this a valid name for a driver variable 

    :type:  boolean, default False, (readonly) 
    '''

    name = None
    '''Name to use in scripted expressions/functions (no spaces or dots are allowed, and must start with a letter) 

    :type:  string, default “”, (never None) 
    '''

    targets = None
    '''Sources of input data for evaluating this variable 

    :type:  bpy_prop_collection of DriverTarget, (readonly) 
    '''

    type = None
    '''Driver variable type 

    :type:  enum in [‘SINGLE_PROP’, ‘TRANSFORMS’, ‘ROTATION_DIFF’, ‘LOC_DIFF’], default ‘SINGLE_PROP’ 
    '''


class DupliObject:
    '''An object duplicate '''

    hide = None
    '''Don’t show dupli object in viewport or render 

    :type:  boolean, default False, (readonly) 
    '''

    index = None
    '''Index in the lowest-level dupli list 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    matrix = None
    '''Object duplicate transformation matrix 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0), (readonly) 
    '''

    object = None
    '''Object being duplicated 

    :type:  Object, (readonly) 
    '''

    orco = None
    '''Generated coordinates in parent object space 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0), (readonly) 
    '''

    particle_system = None
    '''Particle system that this dupli object was instanced from 

    :type:  ParticleSystem, (readonly) 
    '''

    persistent_id = None
    '''Persistent identifier for inter-frame matching of objects with motion blur 

    :type:  int array of 16 items in [-inf, inf], default (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), (readonly) 
    '''

    type = None
    '''Duplicator type that generated this dupli object 

    :type:  enum in [‘NONE’, ‘FRAMES’, ‘VERTS’, ‘FACES’, ‘GROUP’], default ‘NONE’, (readonly) 
    '''

    uv = None
    '''UV coordinates in parent object space 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0), (readonly) 
    '''


class DynamicPaintBrushSettings:
    '''Brush settings '''

    invert_proximity = None
    '''Proximity falloff is applied inside the volume 

    :type:  boolean, default False 
    '''

    material = None
    '''Material to use (if not defined, material linked to the mesh is used) 

    :type:  Material 
    '''

    paint_alpha = None
    '''Paint alpha 

    :type:  float in [0, 1], default 0.0 
    '''

    paint_color = None
    '''Color of the paint 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    paint_distance = None
    '''Maximum distance from brush to mesh surface to affect paint 

    :type:  float in [0, 500], default 0.0 
    '''

    paint_ramp = None
    '''Color ramp used to define proximity falloff 

    :type:  ColorRamp, (readonly) 
    '''

    paint_source = None
    '''

    :type:  enum in [‘PARTICLE_SYSTEM’, ‘POINT’, ‘DISTANCE’, ‘VOLUME_DISTANCE’, ‘VOLUME’], default ‘VOLUME’ 
    '''

    paint_wetness = None
    '''Paint wetness, visible in wetmap (some effects only affect wet paint) 

    :type:  float in [0, 1], default 0.0 
    '''

    particle_system = None
    '''The particle system to paint with 

    :type:  ParticleSystem 
    '''

    proximity_falloff = None
    '''Proximity falloff type 

    :type:  enum in [‘SMOOTH’, ‘CONSTANT’, ‘RAMP’], default ‘CONSTANT’ 
    '''

    ray_direction = None
    '''Ray direction to use for projection (if brush object is located in that direction it’s painted) 

    :type:  enum in [‘CANVAS’, ‘BRUSH’, ‘Z_AXIS’], default ‘CANVAS’ 
    '''

    smooth_radius = None
    '''Smooth falloff added after solid radius 

    :type:  float in [0, 10], default 0.0 
    '''

    smudge_strength = None
    '''Smudge effect strength 

    :type:  float in [0, 1], default 0.0 
    '''

    solid_radius = None
    '''Radius that will be painted solid 

    :type:  float in [0.01, 10], default 0.0 
    '''

    use_absolute_alpha = None
    '''Only increase alpha value if paint alpha is higher than existing 

    :type:  boolean, default False 
    '''

    use_material = None
    '''Use object material to define color and influence 

    :type:  boolean, default False 
    '''

    use_negative_volume = None
    '''Negate influence inside the volume 

    :type:  boolean, default False 
    '''

    use_paint_erase = None
    '''Erase / remove paint instead of adding it 

    :type:  boolean, default False 
    '''

    use_particle_radius = None
    '''Use radius from particle settings 

    :type:  boolean, default False 
    '''

    use_proximity_project = None
    '''Brush is projected to canvas from defined direction within brush proximity 

    :type:  boolean, default False 
    '''

    use_proximity_ramp_alpha = None
    '''Only read color ramp alpha 

    :type:  boolean, default False 
    '''

    use_smudge = None
    '''Make this brush to smudge existing paint as it moves 

    :type:  boolean, default False 
    '''

    use_velocity_alpha = None
    '''Multiply brush influence by velocity color ramp alpha 

    :type:  boolean, default False 
    '''

    use_velocity_color = None
    '''Replace brush color by velocity color ramp 

    :type:  boolean, default False 
    '''

    use_velocity_depth = None
    '''Multiply brush intersection depth (displace, waves) by velocity ramp alpha 

    :type:  boolean, default False 
    '''

    velocity_max = None
    '''Velocity considered as maximum influence (Blender units per frame) 

    :type:  float in [0.0001, 10], default 0.0 
    '''

    velocity_ramp = None
    '''Color ramp used to define brush velocity effect 

    :type:  ColorRamp, (readonly) 
    '''

    wave_clamp = None
    '''Maximum level of surface intersection used to influence waves (use 0.0 to disable) 

    :type:  float in [0, 50], default 0.0 
    '''

    wave_factor = None
    '''Multiplier for wave influence of this brush 

    :type:  float in [-2, 2], default 0.0 
    '''

    wave_type = None
    '''

    :type:  enum in [‘CHANGE’, ‘DEPTH’, ‘FORCE’, ‘REFLECT’], default ‘DEPTH’ 
    '''


class DynamicPaintCanvasSettings:
    '''Dynamic Paint canvas settings '''

    canvas_surfaces = None
    '''Paint surface list 

    :type:  DynamicPaintSurfaces bpy_prop_collection of DynamicPaintSurface, (readonly) 
    '''


class DynamicPaintModifier:
    '''Dynamic Paint modifier '''

    brush_settings = None
    '''

    :type:  DynamicPaintBrushSettings, (readonly) 
    '''

    canvas_settings = None
    '''

    :type:  DynamicPaintCanvasSettings, (readonly) 
    '''

    ui_type = None
    '''

    :type:  enum in [‘CANVAS’, ‘BRUSH’], default ‘CANVAS’ 
    '''


class DynamicPaintSurface:
    '''A canvas surface layer '''

    brush_group = None
    '''Only use brush objects from this group 

    :type:  Group 
    '''

    brush_influence_scale = None
    '''Adjust influence brush objects have on this surface 

    :type:  float in [0, 1], default 0.0 
    '''

    brush_radius_scale = None
    '''Adjust radius of proximity brushes or particles for this surface 

    :type:  float in [0, 10], default 0.0 
    '''

    color_dry_threshold = None
    '''The wetness level when colors start to shift to the background 

    :type:  float in [0, 1], default 0.0 
    '''

    color_spread_speed = None
    '''How fast colors get mixed within wet paint 

    :type:  float in [0, 2], default 0.0 
    '''

    depth_clamp = None
    '''Maximum level of depth intersection in object space (use 0.0 to disable) 

    :type:  float in [0, 50], default 0.0 
    '''

    displace_factor = None
    '''Strength of displace when applied to the mesh 

    :type:  float in [-50, 50], default 0.0 
    '''

    displace_type = None
    '''

    :type:  enum in [‘DISPLACE’, ‘DEPTH’], default ‘DISPLACE’ 
    '''

    dissolve_speed = None
    '''Approximately in how many frames should dissolve happen 

    :type:  int in [1, 10000], default 0 
    '''

    drip_acceleration = None
    '''How much surface acceleration affects dripping 

    :type:  float in [-200, 200], default 0.0 
    '''

    drip_velocity = None
    '''How much surface velocity affects dripping 

    :type:  float in [-200, 200], default 0.0 
    '''

    dry_speed = None
    '''Approximately in how many frames should drying happen 

    :type:  int in [1, 10000], default 0 
    '''

    effect_ui = None
    '''

    :type:  enum in [‘SPREAD’, ‘DRIP’, ‘SHRINK’], default ‘SPREAD’ 
    '''

    effector_weights = None
    '''

    :type:  EffectorWeights, (readonly) 
    '''

    frame_end = None
    '''Simulation end frame 

    :type:  int in [1, 500000], default 0 
    '''

    frame_start = None
    '''Simulation start frame 

    :type:  int in [1, 500000], default 0 
    '''

    frame_substeps = None
    '''Do extra frames between scene frames to ensure smooth motion 

    :type:  int in [0, 20], default 0 
    '''

    image_fileformat = None
    '''

    :type:  enum in [‘PNG’, ‘OPENEXR’], default ‘PNG’ 
    '''

    image_output_path = None
    '''Directory to save the textures 

    :type:  string, default “”, (never None) 
    '''

    image_resolution = None
    '''Output image resolution 

    :type:  int in [16, 4096], default 0 
    '''

    init_color = None
    '''Initial color of the surface 

    :type:  float array of 4 items in [0, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''

    init_color_type = None
    '''

    :type:  enum in [‘NONE’, ‘COLOR’, ‘TEXTURE’, ‘VERTEX_COLOR’], default ‘NONE’ 
    '''

    init_layername = None
    '''

    :type:  string, default “”, (never None) 
    '''

    init_texture = None
    '''

    :type:  Texture 
    '''

    is_active = None
    '''Toggle whether surface is processed or ignored 

    :type:  boolean, default False 
    '''

    is_cache_user = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    name = None
    '''Surface name 

    :type:  string, default “”, (never None) 
    '''

    output_name_a = None
    '''Name used to save output from this surface 

    :type:  string, default “”, (never None) 
    '''

    output_name_b = None
    '''Name used to save output from this surface 

    :type:  string, default “”, (never None) 
    '''

    point_cache = None
    '''

    :type:  PointCache, (readonly, never None) 
    '''

    preview_id = None
    '''

    :type:  enum in [‘PAINT’, ‘WETMAP’], default ‘PAINT’ 
    '''

    show_preview = None
    '''Display surface preview in 3D-views 

    :type:  boolean, default False 
    '''

    shrink_speed = None
    '''How fast shrink effect moves on the canvas surface 

    :type:  float in [0.001, 10], default 0.0 
    '''

    spread_speed = None
    '''How fast spread effect moves on the canvas surface 

    :type:  float in [0.001, 10], default 0.0 
    '''

    surface_format = None
    '''Surface Format 

    :type:  enum in [‘VERTEX’, ‘IMAGE’], default ‘VERTEX’ 
    '''

    surface_type = None
    '''Surface Type 

    :type:  enum in [‘PAINT’], default ‘PAINT’ 
    '''

    use_antialiasing = None
    '''Use 5x multisampling to smooth paint edges 

    :type:  boolean, default False 
    '''

    use_color_preview = None
    '''Whether this surface has some color preview for 3D view 

    :type:  boolean, default False, (readonly) 
    '''

    use_dissolve = None
    '''Enable to make surface changes disappear over time 

    :type:  boolean, default False 
    '''

    use_dissolve_log = None
    '''Use logarithmic dissolve (makes high values to fade faster than low values) 

    :type:  boolean, default False 
    '''

    use_drip = None
    '''Process drip effect (drip wet paint to gravity direction) 

    :type:  boolean, default False 
    '''

    use_dry_log = None
    '''Use logarithmic drying (makes high values to dry faster than low values) 

    :type:  boolean, default False 
    '''

    use_drying = None
    '''Enable to make surface wetness dry over time 

    :type:  boolean, default False 
    '''

    use_incremental_displace = None
    '''New displace is added cumulatively on top of existing 

    :type:  boolean, default False 
    '''

    use_output_a = None
    '''Save this output layer 

    :type:  boolean, default False 
    '''

    use_output_b = None
    '''Save this output layer 

    :type:  boolean, default False 
    '''

    use_premultiply = None
    '''Multiply color by alpha (recommended for Blender input) 

    :type:  boolean, default False 
    '''

    use_shrink = None
    '''Process shrink effect (shrink paint areas) 

    :type:  boolean, default False 
    '''

    use_spread = None
    '''Process spread effect (spread wet paint around surface) 

    :type:  boolean, default False 
    '''

    use_wave_open_border = None
    '''Pass waves through mesh edges 

    :type:  boolean, default False 
    '''

    uv_layer = None
    '''UV map name 

    :type:  string, default “”, (never None) 
    '''

    wave_damping = None
    '''Wave damping factor 

    :type:  float in [0, 1], default 0.0 
    '''

    wave_smoothness = None
    '''Limit maximum steepness of wave slope between simulation points (use higher values for smoother waves at expense of reduced detail) 

    :type:  float in [0, 10], default 0.0 
    '''

    wave_speed = None
    '''Wave propagation speed 

    :type:  float in [0.01, 5], default 0.0 
    '''

    wave_spring = None
    '''Spring force that pulls water level back to zero 

    :type:  float in [0, 1], default 0.0 
    '''

    wave_timescale = None
    '''Wave time scaling factor 

    :type:  float in [0.01, 3], default 0.0 
    '''

    def output_exists(self, object, index):
        '''Checks if surface output layer of given name exists 

        :param index: Index 
        :type index: int in [0, 1]
        :rtype: boolean 
        '''
        pass


class DynamicPaintSurfaces:
    '''Collection of Dynamic Paint Canvas surfaces '''

    active = None
    '''Active Dynamic Paint surface being displayed 

    :type:  DynamicPaintSurface, (readonly) 
    '''

    active_index = None
    '''

    :type:  int in [0, inf], default 0 
    '''


class EdgeSplitModifier:
    '''Edge splitting modifier to create sharp edges '''

    split_angle = None
    '''Angle above which to split edges 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    use_edge_angle = None
    '''Split edges with high angle between faces 

    :type:  boolean, default False 
    '''

    use_edge_sharp = None
    '''Split edges that are marked as sharp 

    :type:  boolean, default False 
    '''


class EditBone:
    '''Editmode bone in an Armature data-block '''

    bbone_curveinx = None
    '''X-axis handle offset for start of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_curveiny = None
    '''Y-axis handle offset for start of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_curveoutx = None
    '''X-axis handle offset for end of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_curveouty = None
    '''Y-axis handle offset for end of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_in = None
    '''Length of first Bezier Handle (for B-Bones only) 

    :type:  float in [0, 2], default 0.0 
    '''

    bbone_out = None
    '''Length of second Bezier Handle (for B-Bones only) 

    :type:  float in [0, 2], default 0.0 
    '''

    bbone_rollin = None
    '''Roll offset for the start of the B-Bone, adjusts twist 

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    bbone_rollout = None
    '''Roll offset for the end of the B-Bone, adjusts twist 

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    bbone_scalein = None
    '''Scale factor for start of the B-Bone, adjusts thickness (for tapering effects) 

    :type:  float in [0, 5], default 1.0 
    '''

    bbone_scaleout = None
    '''Scale factor for end of the B-Bone, adjusts thickness (for tapering effects) 

    :type:  float in [0, 5], default 1.0 
    '''

    bbone_segments = None
    '''Number of subdivisions of bone (for B-Bones only) 

    :type:  int in [1, 32], default 0 
    '''

    bbone_x = None
    '''B-Bone X size 

    :type:  float in [0, 1000], default 0.0 
    '''

    bbone_z = None
    '''B-Bone Z size 

    :type:  float in [0, 1000], default 0.0 
    '''

    envelope_distance = None
    '''Bone deformation distance (for Envelope deform only) 

    :type:  float in [0, 1000], default 0.0 
    '''

    envelope_weight = None
    '''Bone deformation weight (for Envelope deform only) 

    :type:  float in [0, 1000], default 0.0 
    '''

    head = None
    '''Location of head end of the bone 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    head_radius = None
    '''Radius of head of bone (for Envelope deform only) 

    :type:  float in [0, inf], default 0.0 
    '''

    hide = None
    '''Bone is not visible when in Edit Mode 

    :type:  boolean, default False 
    '''

    hide_select = None
    '''Bone is able to be selected 

    :type:  boolean, default False 
    '''

    layers = None
    '''Layers bone exists in 

    :type:  boolean array of 32 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    lock = None
    '''Bone is not able to be transformed when in Edit Mode 

    :type:  boolean, default False 
    '''

    matrix = None
    '''Matrix combining loc/rot of the bone (head position, direction and roll), in armature space (WARNING: does not include/support bone’s length/size) 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''

    parent = None
    '''Parent edit bone (in same Armature) 

    :type:  EditBone 
    '''

    roll = None
    '''Bone rotation around head-tail axis 

    :type:  float in [-inf, inf], default 0.0 
    '''

    select = None
    '''

    :type:  boolean, default False 
    '''

    select_head = None
    '''

    :type:  boolean, default False 
    '''

    select_tail = None
    '''

    :type:  boolean, default False 
    '''

    show_wire = None
    '''Bone is always drawn as Wireframe regardless of viewport draw mode (useful for non-obstructive custom bone shapes) 

    :type:  boolean, default False 
    '''

    tail = None
    '''Location of tail end of the bone 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    tail_radius = None
    '''Radius of tail of bone (for Envelope deform only) 

    :type:  float in [0, inf], default 0.0 
    '''

    use_connect = None
    '''When bone has a parent, bone’s head is stuck to the parent’s tail 

    :type:  boolean, default False 
    '''

    use_cyclic_offset = None
    '''When bone doesn’t have a parent, it receives cyclic offset effects (Deprecated) 

    :type:  boolean, default False 
    '''

    use_deform = None
    '''Enable Bone to deform geometry 

    :type:  boolean, default False 
    '''

    use_endroll_as_inroll = None
    '''Use Roll Out of parent bone as Roll In of its children 

    :type:  boolean, default False 
    '''

    use_envelope_multiply = None
    '''When deforming bone, multiply effects of Vertex Group weights with Envelope influence 

    :type:  boolean, default False 
    '''

    use_inherit_rotation = None
    '''Bone inherits rotation or scale from parent bone 

    :type:  boolean, default False 
    '''

    use_inherit_scale = None
    '''Bone inherits scaling from parent bone 

    :type:  boolean, default False 
    '''

    use_local_location = None
    '''Bone location is set in local space 

    :type:  boolean, default False 
    '''

    use_relative_parent = None
    '''Object children will use relative transform, like deform 

    :type:  boolean, default False 
    '''

    basename = None
    '''The name of this bone before any ‘.’ character (readonly) '''

    center = None
    '''The midpoint between the head and the tail. (readonly) '''

    children = None
    '''A list of all the bones children. (readonly) '''

    children_recursive = None
    '''A list of all children from this bone. (readonly) '''

    children_recursive_basename = None
    '''Returns a chain of children with the same base name as this bone. Only direct chains are supported, forks caused by multiple children with matching base names will terminate the function and not be returned. (readonly) '''

    length = None
    '''The distance from head to tail, when set the head is moved to fit the length. '''

    parent_recursive = None
    '''A list of parents, starting with the immediate parent (readonly) '''

    vector = None
    '''The direction this bone is pointing. Utility function for (tail - head) (readonly) '''

    x_axis = None
    '''Vector pointing down the x-axis of the bone. (readonly) '''

    y_axis = None
    '''Vector pointing down the y-axis of the bone. (readonly) '''

    z_axis = None
    '''Vector pointing down the z-axis of the bone. (readonly) '''

    def align_roll(self, vector):
        '''Align the bone to a localspace roll so the Z axis points in the direction of the vector given 

        :param vector: Vector 
        :type vector: float array of 3 items in [-inf, inf]
        '''
        pass

    def align_orientation(self, other):
        '''Align this bone to another by moving its tail and settings its roll the length of the other bone is not used. 

        '''
        pass

    def parent_index(self, parent_test):
        '''The same as ‘bone in other_bone.parent_recursive’ but saved generating a list. 

        '''
        pass

    def transform(self, matrix, scale=True, roll=True):
        '''Transform the the bones head, tail, roll and envelope (when the matrix has a scale component). 

        :param matrix: 3x3 or 4x4 transformation matrix. 
        :type matrix: mathutils.Matrix
        :param scale: Scale the bone envelope by the matrix. 
        :type scale: bool
        :param roll: Correct the roll to point in the same relative direction to the head and tail. 
        :type roll: bool
        '''
        pass

    def translate(self, vec):
        '''Utility function to add vec to the head and tail of this bone 

        '''
        pass


class EditObjectActuator:
    '''Actuator used to edit objects '''

    angular_velocity = None
    '''Angular velocity upon creation 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    dynamic_operation = None
    '''

    :type:  enum in [‘RESTOREDYN’, ‘SUSPENDDYN’, ‘ENABLERIGIDBODY’, ‘DISABLERIGIDBODY’, ‘SETMASS’], default ‘RESTOREDYN’ 
    '''

    linear_velocity = None
    '''Velocity upon creation 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    mass = None
    '''The mass of the object 

    :type:  float in [-inf, inf], default 0.0 
    '''

    mesh = None
    '''Replace the existing, when left blank ‘Phys’ will remake the existing physics mesh 

    :type:  Mesh 
    '''

    mode = None
    '''The mode of the actuator 

    :type:  enum in [‘ADDOBJECT’, ‘ENDOBJECT’, ‘REPLACEMESH’, ‘TRACKTO’, ‘DYNAMICS’], default ‘ADDOBJECT’ 
    '''

    object = None
    '''Add this Object and all its children (can’t be on a visible layer) 

    :type:  Object 
    '''

    time = None
    '''Duration the new Object lives or the track takes 

    :type:  int in [-inf, inf], default 0 
    '''

    track_axis = None
    '''The axis that points to the target object 

    :type:  enum in [‘TRACKAXISX’, ‘TRACKAXISY’, ‘TRACKAXISZ’, ‘TRACKAXISNEGX’, ‘TRACKAXISNEGY’, ‘TRACKAXISNEGZ’], default ‘TRACKAXISX’ 
    '''

    track_object = None
    '''Track to this Object 

    :type:  Object 
    '''

    up_axis = None
    '''The axis that points upward 

    :type:  enum in [‘UPAXISX’, ‘UPAXISY’, ‘UPAXISZ’], default ‘UPAXISX’ 
    '''

    use_3d_tracking = None
    '''Enable 3D tracking 

    :type:  boolean, default False 
    '''

    use_local_angular_velocity = None
    '''Apply the rotation locally 

    :type:  boolean, default False 
    '''

    use_local_linear_velocity = None
    '''Apply the transformation locally 

    :type:  boolean, default False 
    '''

    use_replace_display_mesh = None
    '''Replace the display mesh 

    :type:  boolean, default False 
    '''

    use_replace_physics_mesh = None
    '''Replace the physics mesh (triangle bounds only - compound shapes not supported) 

    :type:  boolean, default False 
    '''


class EffectSequence:
    '''Sequence strip applying an effect on the images created by other strips '''

    alpha_mode = None
    '''Representation of alpha information in the RGBA pixels 

    :type:  enum in [‘STRAIGHT’, ‘PREMUL’], default ‘STRAIGHT’ 
    '''

    color_multiply = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    color_saturation = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    crop = None
    '''

    :type:  SequenceCrop, (readonly) 
    '''

    proxy = None
    '''

    :type:  SequenceProxy, (readonly) 
    '''

    strobe = None
    '''Only display every nth frame 

    :type:  float in [1, 30], default 0.0 
    '''

    transform = None
    '''

    :type:  SequenceTransform, (readonly) 
    '''

    use_crop = None
    '''Crop image before processing 

    :type:  boolean, default False 
    '''

    use_deinterlace = None
    '''For video movies to remove fields 

    :type:  boolean, default False 
    '''

    use_flip_x = None
    '''Flip on the X axis 

    :type:  boolean, default False 
    '''

    use_flip_y = None
    '''Flip on the Y axis 

    :type:  boolean, default False 
    '''

    use_float = None
    '''Convert input to float data 

    :type:  boolean, default False 
    '''

    use_proxy = None
    '''Use a preview proxy and/or timecode index for this strip 

    :type:  boolean, default False 
    '''

    use_reverse_frames = None
    '''Reverse frame order 

    :type:  boolean, default False 
    '''

    use_translation = None
    '''Translate image before processing 

    :type:  boolean, default False 
    '''


class EffectorWeights:
    '''Effector weights for physics simulation '''

    all = None
    '''All effector’s weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    apply_to_hair_growing = None
    '''Use force fields when growing hair 

    :type:  boolean, default False 
    '''

    boid = None
    '''Boid effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    charge = None
    '''Charge effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    curve_guide = None
    '''Curve guide effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    drag = None
    '''Drag effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    force = None
    '''Force effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    gravity = None
    '''Global gravity weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    group = None
    '''Limit effectors to this Group 

    :type:  Group 
    '''

    harmonic = None
    '''Harmonic effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    lennardjones = None
    '''Lennard-Jones effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    magnetic = None
    '''Magnetic effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    smokeflow = None
    '''Smoke Flow effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    texture = None
    '''Texture effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    turbulence = None
    '''Turbulence effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    vortex = None
    '''Vortex effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''

    wind = None
    '''Wind effector weight 

    :type:  float in [-200, 200], default 0.0 
    '''


class EnumProperty:
    '''RNA enumeration property definition, to choose from a number of predefined options '''

    default = None
    '''Default value for this enum 

    :type:  enum in [‘DUMMY’], default ‘DUMMY’, (readonly) 
    '''

    default_flag = None
    '''Default value for this enum 

    :type:  enum set in {‘DUMMY’}, default {}, (readonly) 
    '''

    enum_items = None
    '''Possible values for the property 

    :type:  bpy_prop_collection of EnumPropertyItem, (readonly) 
    '''

    enum_items_static = None
    '''Possible values for the property (never calls optional dynamic generation of those) 

    :type:  bpy_prop_collection of EnumPropertyItem, (readonly) 
    '''


class EnumPropertyItem:
    '''Definition of a choice in an RNA enum property '''

    description = None
    '''Description of the item’s purpose 

    :type:  string, default “”, (readonly, never None) 
    '''

    icon = None
    '''Icon of the item 

    :type:  enum in [‘NONE’, ‘QUESTION’, ‘ERROR’, ‘CANCEL’, ‘TRIA_RIGHT’, ‘TRIA_DOWN’, ‘TRIA_LEFT’, ‘TRIA_UP’, ‘ARROW_LEFTRIGHT’, ‘PLUS’, ‘DISCLOSURE_TRI_DOWN’, ‘DISCLOSURE_TRI_RIGHT’, ‘RADIOBUT_OFF’, ‘RADIOBUT_ON’, ‘MENU_PANEL’, ‘BLENDER’, ‘GRIP’, ‘DOT’, ‘COLLAPSEMENU’, ‘X’, ‘GO_LEFT’, ‘PLUG’, ‘UI’, ‘NODE’, ‘NODE_SEL’, ‘FULLSCREEN’, ‘SPLITSCREEN’, ‘RIGHTARROW_THIN’, ‘BORDERMOVE’, ‘VIEWZOOM’, ‘ZOOMIN’, ‘ZOOMOUT’, ‘PANEL_CLOSE’, ‘COPY_ID’, ‘EYEDROPPER’, ‘LINK_AREA’, ‘AUTO’, ‘CHECKBOX_DEHLT’, ‘CHECKBOX_HLT’, ‘UNLOCKED’, ‘LOCKED’, ‘UNPINNED’, ‘PINNED’, ‘SCREEN_BACK’, ‘RIGHTARROW’, ‘DOWNARROW_HLT’, ‘DOTSUP’, ‘DOTSDOWN’, ‘LINK’, ‘INLINK’, ‘PLUGIN’, ‘HELP’, ‘GHOST_ENABLED’, ‘COLOR’, ‘LINKED’, ‘UNLINKED’, ‘HAND’, ‘ZOOM_ALL’, ‘ZOOM_SELECTED’, ‘ZOOM_PREVIOUS’, ‘ZOOM_IN’, ‘ZOOM_OUT’, ‘RENDER_REGION’, ‘BORDER_RECT’, ‘BORDER_LASSO’, ‘FREEZE’, ‘STYLUS_PRESSURE’, ‘GHOST_DISABLED’, ‘NEW’, ‘FILE_TICK’, ‘QUIT’, ‘URL’, ‘RECOVER_LAST’, ‘FULLSCREEN_ENTER’, ‘FULLSCREEN_EXIT’, ‘BLANK1’, ‘LAMP’, ‘MATERIAL’, ‘TEXTURE’, ‘ANIM’, ‘WORLD’, ‘SCENE’, ‘EDIT’, ‘GAME’, ‘RADIO’, ‘SCRIPT’, ‘PARTICLES’, ‘PHYSICS’, ‘SPEAKER’, ‘TEXTURE_SHADED’, ‘VIEW3D’, ‘IPO’, ‘OOPS’, ‘BUTS’, ‘FILESEL’, ‘IMAGE_COL’, ‘INFO’, ‘SEQUENCE’, ‘TEXT’, ‘IMASEL’, ‘SOUND’, ‘ACTION’, ‘NLA’, ‘SCRIPTWIN’, ‘TIME’, ‘NODETREE’, ‘LOGIC’, ‘CONSOLE’, ‘PREFERENCES’, ‘CLIP’, ‘ASSET_MANAGER’, ‘OBJECT_DATAMODE’, ‘EDITMODE_HLT’, ‘FACESEL_HLT’, ‘VPAINT_HLT’, ‘TPAINT_HLT’, ‘WPAINT_HLT’, ‘SCULPTMODE_HLT’, ‘POSE_HLT’, ‘PARTICLEMODE’, ‘LIGHTPAINT’, ‘SCENE_DATA’, ‘RENDERLAYERS’, ‘WORLD_DATA’, ‘OBJECT_DATA’, ‘MESH_DATA’, ‘CURVE_DATA’, ‘META_DATA’, ‘LATTICE_DATA’, ‘LAMP_DATA’, ‘MATERIAL_DATA’, ‘TEXTURE_DATA’, ‘ANIM_DATA’, ‘CAMERA_DATA’, ‘PARTICLE_DATA’, ‘LIBRARY_DATA_DIRECT’, ‘GROUP’, ‘ARMATURE_DATA’, ‘POSE_DATA’, ‘BONE_DATA’, ‘CONSTRAINT’, ‘SHAPEKEY_DATA’, ‘CONSTRAINT_BONE’, ‘CAMERA_STEREO’, ‘PACKAGE’, ‘UGLYPACKAGE’, ‘BRUSH_DATA’, ‘IMAGE_DATA’, ‘FILE’, ‘FCURVE’, ‘FONT_DATA’, ‘RENDER_RESULT’, ‘SURFACE_DATA’, ‘EMPTY_DATA’, ‘SETTINGS’, ‘RENDER_ANIMATION’, ‘RENDER_STILL’, ‘LIBRARY_DATA_BROKEN’, ‘BOIDS’, ‘STRANDS’, ‘LIBRARY_DATA_INDIRECT’, ‘GREASEPENCIL’, ‘LINE_DATA’, ‘GROUP_BONE’, ‘GROUP_VERTEX’, ‘GROUP_VCOL’, ‘GROUP_UVS’, ‘RNA’, ‘RNA_ADD’, ‘OUTLINER_OB_EMPTY’, ‘OUTLINER_OB_MESH’, ‘OUTLINER_OB_CURVE’, ‘OUTLINER_OB_LATTICE’, ‘OUTLINER_OB_META’, ‘OUTLINER_OB_LAMP’, ‘OUTLINER_OB_CAMERA’, ‘OUTLINER_OB_ARMATURE’, ‘OUTLINER_OB_FONT’, ‘OUTLINER_OB_SURFACE’, ‘OUTLINER_OB_SPEAKER’, ‘RESTRICT_COLOR_OFF’, ‘RESTRICT_COLOR_ON’, ‘RESTRICT_VIEW_OFF’, ‘RESTRICT_VIEW_ON’, ‘RESTRICT_SELECT_OFF’, ‘RESTRICT_SELECT_ON’, ‘RESTRICT_RENDER_OFF’, ‘RESTRICT_RENDER_ON’, ‘OUTLINER_DATA_EMPTY’, ‘OUTLINER_DATA_MESH’, ‘OUTLINER_DATA_CURVE’, ‘OUTLINER_DATA_LATTICE’, ‘OUTLINER_DATA_META’, ‘OUTLINER_DATA_LAMP’, ‘OUTLINER_DATA_CAMERA’, ‘OUTLINER_DATA_ARMATURE’, ‘OUTLINER_DATA_FONT’, ‘OUTLINER_DATA_SURFACE’, ‘OUTLINER_DATA_SPEAKER’, ‘OUTLINER_DATA_POSE’, ‘MESH_PLANE’, ‘MESH_CUBE’, ‘MESH_CIRCLE’, ‘MESH_UVSPHERE’, ‘MESH_ICOSPHERE’, ‘MESH_GRID’, ‘MESH_MONKEY’, ‘MESH_CYLINDER’, ‘MESH_TORUS’, ‘MESH_CONE’, ‘MESH_CAPSULE’, ‘LAMP_POINT’, ‘LAMP_SUN’, ‘LAMP_SPOT’, ‘LAMP_HEMI’, ‘LAMP_AREA’, ‘META_EMPTY’, ‘META_PLANE’, ‘META_CUBE’, ‘META_BALL’, ‘META_ELLIPSOID’, ‘META_CAPSULE’, ‘SURFACE_NCURVE’, ‘SURFACE_NCIRCLE’, ‘SURFACE_NSURFACE’, ‘SURFACE_NCYLINDER’, ‘SURFACE_NSPHERE’, ‘SURFACE_NTORUS’, ‘CURVE_BEZCURVE’, ‘CURVE_BEZCIRCLE’, ‘CURVE_NCURVE’, ‘CURVE_NCIRCLE’, ‘CURVE_PATH’, ‘COLOR_RED’, ‘COLOR_GREEN’, ‘COLOR_BLUE’, ‘TRIA_RIGHT_BAR’, ‘TRIA_DOWN_BAR’, ‘TRIA_LEFT_BAR’, ‘TRIA_UP_BAR’, ‘FORCE_FORCE’, ‘FORCE_WIND’, ‘FORCE_VORTEX’, ‘FORCE_MAGNETIC’, ‘FORCE_HARMONIC’, ‘FORCE_CHARGE’, ‘FORCE_LENNARDJONES’, ‘FORCE_TEXTURE’, ‘FORCE_CURVE’, ‘FORCE_BOID’, ‘FORCE_TURBULENCE’, ‘FORCE_DRAG’, ‘FORCE_SMOKEFLOW’, ‘NODE_INSERT_ON’, ‘NODE_INSERT_OFF’, ‘MODIFIER’, ‘MOD_WAVE’, ‘MOD_BUILD’, ‘MOD_DECIM’, ‘MOD_MIRROR’, ‘MOD_SOFT’, ‘MOD_SUBSURF’, ‘HOOK’, ‘MOD_PHYSICS’, ‘MOD_PARTICLES’, ‘MOD_BOOLEAN’, ‘MOD_EDGESPLIT’, ‘MOD_ARRAY’, ‘MOD_UVPROJECT’, ‘MOD_DISPLACE’, ‘MOD_CURVE’, ‘MOD_LATTICE’, ‘CONSTRAINT_DATA’, ‘MOD_ARMATURE’, ‘MOD_SHRINKWRAP’, ‘MOD_CAST’, ‘MOD_MESHDEFORM’, ‘MOD_BEVEL’, ‘MOD_SMOOTH’, ‘MOD_SIMPLEDEFORM’, ‘MOD_MASK’, ‘MOD_CLOTH’, ‘MOD_EXPLODE’, ‘MOD_FLUIDSIM’, ‘MOD_MULTIRES’, ‘MOD_SMOKE’, ‘MOD_SOLIDIFY’, ‘MOD_SCREW’, ‘MOD_VERTEX_WEIGHT’, ‘MOD_DYNAMICPAINT’, ‘MOD_REMESH’, ‘MOD_OCEAN’, ‘MOD_WARP’, ‘MOD_SKIN’, ‘MOD_TRIANGULATE’, ‘MOD_WIREFRAME’, ‘MOD_DATA_TRANSFER’, ‘MOD_NORMALEDIT’, ‘REC’, ‘PLAY’, ‘FF’, ‘REW’, ‘PAUSE’, ‘PREV_KEYFRAME’, ‘NEXT_KEYFRAME’, ‘PLAY_AUDIO’, ‘PLAY_REVERSE’, ‘PREVIEW_RANGE’, ‘ACTION_TWEAK’, ‘PMARKER_ACT’, ‘PMARKER_SEL’, ‘PMARKER’, ‘MARKER_HLT’, ‘MARKER’, ‘SPACE2’, ‘SPACE3’, ‘KEYINGSET’, ‘KEY_DEHLT’, ‘KEY_HLT’, ‘MUTE_IPO_OFF’, ‘MUTE_IPO_ON’, ‘VISIBLE_IPO_OFF’, ‘VISIBLE_IPO_ON’, ‘DRIVER’, ‘SOLO_OFF’, ‘SOLO_ON’, ‘FRAME_PREV’, ‘FRAME_NEXT’, ‘NLA_PUSHDOWN’, ‘IPO_CONSTANT’, ‘IPO_LINEAR’, ‘IPO_BEZIER’, ‘IPO_SINE’, ‘IPO_QUAD’, ‘IPO_CUBIC’, ‘IPO_QUART’, ‘IPO_QUINT’, ‘IPO_EXPO’, ‘IPO_CIRC’, ‘IPO_BOUNCE’, ‘IPO_ELASTIC’, ‘IPO_BACK’, ‘IPO_EASE_IN’, ‘IPO_EASE_OUT’, ‘IPO_EASE_IN_OUT’, ‘VERTEXSEL’, ‘EDGESEL’, ‘FACESEL’, ‘LOOPSEL’, ‘ROTATE’, ‘CURSOR’, ‘ROTATECOLLECTION’, ‘ROTATECENTER’, ‘ROTACTIVE’, ‘ALIGN’, ‘SMOOTHCURVE’, ‘SPHERECURVE’, ‘ROOTCURVE’, ‘SHARPCURVE’, ‘LINCURVE’, ‘NOCURVE’, ‘RNDCURVE’, ‘PROP_OFF’, ‘PROP_ON’, ‘PROP_CON’, ‘SCULPT_DYNTOPO’, ‘PARTICLE_POINT’, ‘PARTICLE_TIP’, ‘PARTICLE_PATH’, ‘MAN_TRANS’, ‘MAN_ROT’, ‘MAN_SCALE’, ‘MANIPUL’, ‘SNAP_OFF’, ‘SNAP_ON’, ‘SNAP_NORMAL’, ‘SNAP_GRID’, ‘SNAP_VERTEX’, ‘SNAP_EDGE’, ‘SNAP_FACE’, ‘SNAP_VOLUME’, ‘SNAP_INCREMENT’, ‘STICKY_UVS_LOC’, ‘STICKY_UVS_DISABLE’, ‘STICKY_UVS_VERT’, ‘CLIPUV_DEHLT’, ‘CLIPUV_HLT’, ‘SNAP_PEEL_OBJECT’, ‘GRID’, ‘PASTEDOWN’, ‘COPYDOWN’, ‘PASTEFLIPUP’, ‘PASTEFLIPDOWN’, ‘SNAP_SURFACE’, ‘AUTOMERGE_ON’, ‘AUTOMERGE_OFF’, ‘RETOPO’, ‘UV_VERTEXSEL’, ‘UV_EDGESEL’, ‘UV_FACESEL’, ‘UV_ISLANDSEL’, ‘UV_SYNC_SELECT’, ‘BBOX’, ‘WIRE’, ‘SOLID’, ‘SMOOTH’, ‘POTATO’, ‘ORTHO’, ‘LOCKVIEW_OFF’, ‘LOCKVIEW_ON’, ‘AXIS_SIDE’, ‘AXIS_FRONT’, ‘AXIS_TOP’, ‘NDOF_DOM’, ‘NDOF_TURN’, ‘NDOF_FLY’, ‘NDOF_TRANS’, ‘LAYER_USED’, ‘LAYER_ACTIVE’, ‘SORTALPHA’, ‘SORTBYEXT’, ‘SORTTIME’, ‘SORTSIZE’, ‘LONGDISPLAY’, ‘SHORTDISPLAY’, ‘GHOST’, ‘IMGDISPLAY’, ‘SAVE_AS’, ‘SAVE_COPY’, ‘BOOKMARKS’, ‘FONTPREVIEW’, ‘FILTER’, ‘NEWFOLDER’, ‘OPEN_RECENT’, ‘FILE_PARENT’, ‘FILE_REFRESH’, ‘FILE_FOLDER’, ‘FILE_BLANK’, ‘FILE_BLEND’, ‘FILE_IMAGE’, ‘FILE_MOVIE’, ‘FILE_SCRIPT’, ‘FILE_SOUND’, ‘FILE_FONT’, ‘FILE_TEXT’, ‘RECOVER_AUTO’, ‘SAVE_PREFS’, ‘LINK_BLEND’, ‘APPEND_BLEND’, ‘IMPORT’, ‘EXPORT’, ‘EXTERNAL_DATA’, ‘LOAD_FACTORY’, ‘LOOP_BACK’, ‘LOOP_FORWARDS’, ‘BACK’, ‘FORWARD’, ‘FILE_HIDDEN’, ‘FILE_BACKUP’, ‘DISK_DRIVE’, ‘MATPLANE’, ‘MATSPHERE’, ‘MATCUBE’, ‘MONKEY’, ‘HAIR’, ‘ALIASED’, ‘ANTIALIASED’, ‘MAT_SPHERE_SKY’, ‘WORDWRAP_OFF’, ‘WORDWRAP_ON’, ‘SYNTAX_OFF’, ‘SYNTAX_ON’, ‘LINENUMBERS_OFF’, ‘LINENUMBERS_ON’, ‘SCRIPTPLUGINS’, ‘SEQ_SEQUENCER’, ‘SEQ_PREVIEW’, ‘SEQ_LUMA_WAVEFORM’, ‘SEQ_CHROMA_SCOPE’, ‘SEQ_HISTOGRAM’, ‘SEQ_SPLITVIEW’, ‘IMAGE_RGB’, ‘IMAGE_RGB_ALPHA’, ‘IMAGE_ALPHA’, ‘IMAGE_ZDEPTH’, ‘IMAGEFILE’, ‘BRUSH_ADD’, ‘BRUSH_BLOB’, ‘BRUSH_BLUR’, ‘BRUSH_CLAY’, ‘BRUSH_CLAY_STRIPS’, ‘BRUSH_CLONE’, ‘BRUSH_CREASE’, ‘BRUSH_DARKEN’, ‘BRUSH_FILL’, ‘BRUSH_FLATTEN’, ‘BRUSH_GRAB’, ‘BRUSH_INFLATE’, ‘BRUSH_LAYER’, ‘BRUSH_LIGHTEN’, ‘BRUSH_MASK’, ‘BRUSH_MIX’, ‘BRUSH_MULTIPLY’, ‘BRUSH_NUDGE’, ‘BRUSH_PINCH’, ‘BRUSH_SCRAPE’, ‘BRUSH_SCULPT_DRAW’, ‘BRUSH_SMEAR’, ‘BRUSH_SMOOTH’, ‘BRUSH_SNAKE_HOOK’, ‘BRUSH_SOFTEN’, ‘BRUSH_SUBTRACT’, ‘BRUSH_TEXDRAW’, ‘BRUSH_TEXFILL’, ‘BRUSH_TEXMASK’, ‘BRUSH_THUMB’, ‘BRUSH_ROTATE’, ‘BRUSH_VERTEXDRAW’, ‘MATCAP_01’, ‘MATCAP_02’, ‘MATCAP_03’, ‘MATCAP_04’, ‘MATCAP_05’, ‘MATCAP_06’, ‘MATCAP_07’, ‘MATCAP_08’, ‘MATCAP_09’, ‘MATCAP_10’, ‘MATCAP_11’, ‘MATCAP_12’, ‘MATCAP_13’, ‘MATCAP_14’, ‘MATCAP_15’, ‘MATCAP_16’, ‘MATCAP_17’, ‘MATCAP_18’, ‘MATCAP_19’, ‘MATCAP_20’, ‘MATCAP_21’, ‘MATCAP_22’, ‘MATCAP_23’, ‘MATCAP_24’, ‘VIEW3D_VEC’, ‘EDIT_VEC’, ‘EDITMODE_VEC_DEHLT’, ‘EDITMODE_VEC_HLT’, ‘DISCLOSURE_TRI_RIGHT_VEC’, ‘DISCLOSURE_TRI_DOWN_VEC’, ‘MOVE_UP_VEC’, ‘MOVE_DOWN_VEC’, ‘X_VEC’, ‘SMALL_TRI_RIGHT_VEC’, ‘KEYTYPE_KEYFRAME_VEC’, ‘KEYTYPE_BREAKDOWN_VEC’, ‘KEYTYPE_EXTREME_VEC’, ‘KEYTYPE_JITTER_VEC’, ‘KEYTYPE_MOVING_HOLD_VEC’, ‘COLORSET_01_VEC’, ‘COLORSET_02_VEC’, ‘COLORSET_03_VEC’, ‘COLORSET_04_VEC’, ‘COLORSET_05_VEC’, ‘COLORSET_06_VEC’, ‘COLORSET_07_VEC’, ‘COLORSET_08_VEC’, ‘COLORSET_09_VEC’, ‘COLORSET_10_VEC’, ‘COLORSET_11_VEC’, ‘COLORSET_12_VEC’, ‘COLORSET_13_VEC’, ‘COLORSET_14_VEC’, ‘COLORSET_15_VEC’, ‘COLORSET_16_VEC’, ‘COLORSET_17_VEC’, ‘COLORSET_18_VEC’, ‘COLORSET_19_VEC’, ‘COLORSET_20_VEC’], default ‘NONE’, (readonly) 
    '''

    identifier = None
    '''Unique name used in the code and scripting 

    :type:  string, default “”, (readonly, never None) 
    '''

    name = None
    '''Human readable name 

    :type:  string, default “”, (readonly, never None) 
    '''

    value = None
    '''Value of the item 

    :type:  int in [0, inf], default 0, (readonly) 
    '''


class EnvironmentMap:
    '''Environment map created by the renderer and cached for subsequent renders '''

    clip_end = None
    '''Objects further than this are not visible to map 

    :type:  float in [0.01, inf], default 0.0 
    '''

    clip_start = None
    '''Objects nearer than this are not visible to map 

    :type:  float in [0.001, inf], default 0.0 
    '''

    depth = None
    '''Number of times a map will be rendered recursively (mirror effects) 

    :type:  int in [0, 5], default 0 
    '''

    is_valid = None
    '''True if this map is ready for use, False if it needs rendering 

    :type:  boolean, default False, (readonly) 
    '''

    layers_ignore = None
    '''Hide objects on these layers when generating the Environment Map 

    :type:  boolean array of 20 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    mapping = None
    '''

    :type:  enum in [‘CUBE’, ‘PLANE’], default ‘CUBE’ 
    '''

    resolution = None
    '''Pixel resolution of the rendered environment map 

    :type:  int in [50, 4096], default 0 
    '''

    source = None
    '''

    :type:  enum in [‘STATIC’, ‘ANIMATED’, ‘IMAGE_FILE’], default ‘STATIC’ 
    '''

    viewpoint_object = None
    '''Object to use as the environment map’s viewpoint location 

    :type:  Object 
    '''

    zoom = None
    '''

    :type:  float in [0.1, 5], default 0.0 
    '''

    def clear(self):
        '''Discard the environment map and free it from memory 

        '''
        pass

    def save(self,
             filepath,
             scene=None,
             layout=(0.0, 0.0, 1.0, 0.0, 2.0, 0.0, 0.0, 1.0, 1.0, 1.0, 2.0,
                     1.0)):
        '''Save the environment map to disc using the scene render settings 

        :param filepath: File path, Location of the output file 
        :type filepath: string, (never None)
        :param scene: Overrides the scene from which image parameters are taken 
        :type scene: Scene, (optional)
        :param layout: File layout, Flat array describing the X,Y position of each cube face in the output image, where 1 is the size of a face - order is [+Z -Z +Y -X -Y +X] (use -1 to skip a face) 
        :type layout: float array of 12 items in [0, 1000], (optional)
        '''
        pass


class EnvironmentMapTexture:
    '''Environment map texture '''

    environment_map = None
    '''Get the environment map associated with this texture 

    :type:  EnvironmentMap, (readonly) 
    '''

    filter_eccentricity = None
    '''Maximum eccentricity (higher gives less blur at distant/oblique angles, but is also slower) 

    :type:  int in [1, 256], default 0 
    '''

    filter_probes = None
    '''Maximum number of samples (higher gives less blur at distant/oblique angles, but is also slower) 

    :type:  int in [1, 256], default 0 
    '''

    filter_size = None
    '''Multiply the filter size used by MIP Map and Interpolation 

    :type:  float in [0.1, 50], default 0.0 
    '''

    filter_type = None
    '''Texture filter to use for sampling image 

    :type:  enum in [‘BOX’, ‘EWA’, ‘FELINE’, ‘AREA’], default ‘BOX’ 
    '''

    image = None
    '''Source image file to read the environment map from 

    :type:  Image 
    '''

    image_user = None
    '''Parameters defining which layer, pass and frame of the image is displayed 

    :type:  ImageUser, (readonly) 
    '''

    use_filter_size_min = None
    '''Use Filter Size as a minimal filter value in pixels 

    :type:  boolean, default False 
    '''

    use_mipmap = None
    '''Use auto-generated MIP maps for the image 

    :type:  boolean, default False 
    '''

    use_mipmap_gauss = None
    '''Use Gauss filter to sample down MIP maps 

    :type:  boolean, default False 
    '''

    users_material = None
    '''Materials that use this texture (readonly) '''

    users_object_modifier = None
    '''Object modifiers that use this texture (readonly) '''


class Event:
    '''Window Manager Event '''

    alt = None
    '''True when the Alt/Option key is held 

    :type:  boolean, default False, (readonly) 
    '''

    ascii = None
    '''Single ASCII character for this event 

    :type:  string, default “”, (readonly, never None) 
    '''

    ctrl = None
    '''True when the Ctrl key is held 

    :type:  boolean, default False, (readonly) 
    '''

    is_tablet = None
    '''The pressure of the tablet or 1.0 if no tablet present 

    :type:  boolean, default False, (readonly) 
    '''

    mouse_prev_x = None
    '''The window relative horizontal location of the mouse 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    mouse_prev_y = None
    '''The window relative vertical location of the mouse 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    mouse_region_x = None
    '''The region relative horizontal location of the mouse 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    mouse_region_y = None
    '''The region relative vertical location of the mouse 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    mouse_x = None
    '''The window relative horizontal location of the mouse 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    mouse_y = None
    '''The window relative vertical location of the mouse 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    oskey = None
    '''True when the Cmd key is held 

    :type:  boolean, default False, (readonly) 
    '''

    pressure = None
    '''The pressure of the tablet or 1.0 if no tablet present 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    shift = None
    '''True when the Shift key is held 

    :type:  boolean, default False, (readonly) 
    '''

    tilt = None
    '''The pressure of the tablet or zeroes if no tablet present 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0), (readonly) 
    '''

    type = None
    '''

    :type:  enum in [‘NONE’, ‘LEFTMOUSE’, ‘MIDDLEMOUSE’, ‘RIGHTMOUSE’, ‘BUTTON4MOUSE’, ‘BUTTON5MOUSE’, ‘BUTTON6MOUSE’, ‘BUTTON7MOUSE’, ‘ACTIONMOUSE’, ‘SELECTMOUSE’, ‘PEN’, ‘ERASER’, ‘MOUSEMOVE’, ‘INBETWEEN_MOUSEMOVE’, ‘TRACKPADPAN’, ‘TRACKPADZOOM’, ‘MOUSEROTATE’, ‘WHEELUPMOUSE’, ‘WHEELDOWNMOUSE’, ‘WHEELINMOUSE’, ‘WHEELOUTMOUSE’, ‘EVT_TWEAK_L’, ‘EVT_TWEAK_M’, ‘EVT_TWEAK_R’, ‘EVT_TWEAK_A’, ‘EVT_TWEAK_S’, ‘A’, ‘B’, ‘C’, ‘D’, ‘E’, ‘F’, ‘G’, ‘H’, ‘I’, ‘J’, ‘K’, ‘L’, ‘M’, ‘N’, ‘O’, ‘P’, ‘Q’, ‘R’, ‘S’, ‘T’, ‘U’, ‘V’, ‘W’, ‘X’, ‘Y’, ‘Z’, ‘ZERO’, ‘ONE’, ‘TWO’, ‘THREE’, ‘FOUR’, ‘FIVE’, ‘SIX’, ‘SEVEN’, ‘EIGHT’, ‘NINE’, ‘LEFT_CTRL’, ‘LEFT_ALT’, ‘LEFT_SHIFT’, ‘RIGHT_ALT’, ‘RIGHT_CTRL’, ‘RIGHT_SHIFT’, ‘OSKEY’, ‘GRLESS’, ‘ESC’, ‘TAB’, ‘RET’, ‘SPACE’, ‘LINE_FEED’, ‘BACK_SPACE’, ‘DEL’, ‘SEMI_COLON’, ‘PERIOD’, ‘COMMA’, ‘QUOTE’, ‘ACCENT_GRAVE’, ‘MINUS’, ‘PLUS’, ‘SLASH’, ‘BACK_SLASH’, ‘EQUAL’, ‘LEFT_BRACKET’, ‘RIGHT_BRACKET’, ‘LEFT_ARROW’, ‘DOWN_ARROW’, ‘RIGHT_ARROW’, ‘UP_ARROW’, ‘NUMPAD_2’, ‘NUMPAD_4’, ‘NUMPAD_6’, ‘NUMPAD_8’, ‘NUMPAD_1’, ‘NUMPAD_3’, ‘NUMPAD_5’, ‘NUMPAD_7’, ‘NUMPAD_9’, ‘NUMPAD_PERIOD’, ‘NUMPAD_SLASH’, ‘NUMPAD_ASTERIX’, ‘NUMPAD_0’, ‘NUMPAD_MINUS’, ‘NUMPAD_ENTER’, ‘NUMPAD_PLUS’, ‘F1’, ‘F2’, ‘F3’, ‘F4’, ‘F5’, ‘F6’, ‘F7’, ‘F8’, ‘F9’, ‘F10’, ‘F11’, ‘F12’, ‘F13’, ‘F14’, ‘F15’, ‘F16’, ‘F17’, ‘F18’, ‘F19’, ‘PAUSE’, ‘INSERT’, ‘HOME’, ‘PAGE_UP’, ‘PAGE_DOWN’, ‘END’, ‘MEDIA_PLAY’, ‘MEDIA_STOP’, ‘MEDIA_FIRST’, ‘MEDIA_LAST’, ‘TEXTINPUT’, ‘WINDOW_DEACTIVATE’, ‘TIMER’, ‘TIMER0’, ‘TIMER1’, ‘TIMER2’, ‘TIMER_JOBS’, ‘TIMER_AUTOSAVE’, ‘TIMER_REPORT’, ‘TIMERREGION’, ‘NDOF_MOTION’, ‘NDOF_BUTTON_MENU’, ‘NDOF_BUTTON_FIT’, ‘NDOF_BUTTON_TOP’, ‘NDOF_BUTTON_BOTTOM’, ‘NDOF_BUTTON_LEFT’, ‘NDOF_BUTTON_RIGHT’, ‘NDOF_BUTTON_FRONT’, ‘NDOF_BUTTON_BACK’, ‘NDOF_BUTTON_ISO1’, ‘NDOF_BUTTON_ISO2’, ‘NDOF_BUTTON_ROLL_CW’, ‘NDOF_BUTTON_ROLL_CCW’, ‘NDOF_BUTTON_SPIN_CW’, ‘NDOF_BUTTON_SPIN_CCW’, ‘NDOF_BUTTON_TILT_CW’, ‘NDOF_BUTTON_TILT_CCW’, ‘NDOF_BUTTON_ROTATE’, ‘NDOF_BUTTON_PANZOOM’, ‘NDOF_BUTTON_DOMINANT’, ‘NDOF_BUTTON_PLUS’, ‘NDOF_BUTTON_MINUS’, ‘NDOF_BUTTON_ESC’, ‘NDOF_BUTTON_ALT’, ‘NDOF_BUTTON_SHIFT’, ‘NDOF_BUTTON_CTRL’, ‘NDOF_BUTTON_1’, ‘NDOF_BUTTON_2’, ‘NDOF_BUTTON_3’, ‘NDOF_BUTTON_4’, ‘NDOF_BUTTON_5’, ‘NDOF_BUTTON_6’, ‘NDOF_BUTTON_7’, ‘NDOF_BUTTON_8’, ‘NDOF_BUTTON_9’, ‘NDOF_BUTTON_10’, ‘NDOF_BUTTON_A’, ‘NDOF_BUTTON_B’, ‘NDOF_BUTTON_C’], default ‘NONE’, (readonly) 
    '''

    unicode = None
    '''Single unicode character for this event 

    :type:  string, default “”, (readonly, never None) 
    '''

    value = None
    '''The type of event, only applies to some 

    :type:  enum in [‘ANY’, ‘NOTHING’, ‘PRESS’, ‘RELEASE’, ‘CLICK’, ‘DOUBLE_CLICK’, ‘NORTH’, ‘NORTH_EAST’, ‘EAST’, ‘SOUTH_EAST’, ‘SOUTH’, ‘SOUTH_WEST’, ‘WEST’, ‘NORTH_WEST’], default ‘NOTHING’, (readonly) 
    '''


class ExplodeModifier:
    '''Explosion effect modifier based on a particle system '''

    particle_uv = None
    '''UV map to change with particle age 

    :type:  string, default “”, (never None) 
    '''

    protect = None
    '''Clean vertex group edges 

    :type:  float in [0, 1], default 0.0 
    '''

    show_alive = None
    '''Show mesh when particles are alive 

    :type:  boolean, default False 
    '''

    show_dead = None
    '''Show mesh when particles are dead 

    :type:  boolean, default False 
    '''

    show_unborn = None
    '''Show mesh when particles are unborn 

    :type:  boolean, default False 
    '''

    use_edge_cut = None
    '''Cut face edges for nicer shrapnel 

    :type:  boolean, default False 
    '''

    use_size = None
    '''Use particle size for the shrapnel 

    :type:  boolean, default False 
    '''

    vertex_group = None
    '''

    :type:  string, default “”, (never None) 
    '''


class ExpressionController:
    '''Controller passing on events based on the evaluation of an expression '''

    expression = None
    '''

    :type:  string, default “”, (never None) 
    '''


class FCurve:
    '''F-Curve defining values of a period of time '''

    array_index = None
    '''Index to the specific property affected by F-Curve if applicable 

    :type:  int in [-inf, inf], default 0 
    '''

    color = None
    '''Color of the F-Curve in the Graph Editor 

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    color_mode = None
    '''Method used to determine color of F-Curve in Graph Editor 

    :type:  enum in [‘AUTO_RAINBOW’, ‘AUTO_RGB’, ‘AUTO_YRGB’, ‘CUSTOM’], default ‘AUTO_RAINBOW’ 
    '''

    data_path = None
    '''RNA Path to property affected by F-Curve 

    :type:  string, default “”, (never None) 
    '''

    driver = None
    '''Channel Driver (only set for Driver F-Curves) 

    :type:  Driver, (readonly) 
    '''

    extrapolation = None
    '''Method used for evaluating value of F-Curve outside first and last keyframes 

    :type:  enum in [‘CONSTANT’, ‘LINEAR’], default ‘CONSTANT’ 
    '''

    group = None
    '''Action Group that this F-Curve belongs to 

    :type:  ActionGroup 
    '''

    hide = None
    '''F-Curve and its keyframes are hidden in the Graph Editor graphs 

    :type:  boolean, default False 
    '''

    is_valid = None
    '''False when F-Curve could not be evaluated in past, so should be skipped when evaluating 

    :type:  boolean, default False 
    '''

    keyframe_points = None
    '''User-editable keyframes 

    :type:  FCurveKeyframePoints bpy_prop_collection of Keyframe, (readonly) 
    '''

    lock = None
    '''F-Curve’s settings cannot be edited 

    :type:  boolean, default False 
    '''

    modifiers = None
    '''Modifiers affecting the shape of the F-Curve 

    :type:  FCurveModifiers bpy_prop_collection of FModifier, (readonly) 
    '''

    mute = None
    '''F-Curve is not evaluated 

    :type:  boolean, default False 
    '''

    sampled_points = None
    '''Sampled animation data 

    :type:  bpy_prop_collection of FCurveSample, (readonly) 
    '''

    select = None
    '''F-Curve is selected for editing 

    :type:  boolean, default False 
    '''

    def evaluate(self, frame):
        '''Evaluate F-Curve 

        :param frame: Frame, Evaluate F-Curve at given frame 
        :type frame: float in [-inf, inf]
        :rtype: float in [-inf, inf] 
        :return:  Value, Value of F-Curve specific frame 
        '''
        pass

    def update(self):
        '''Ensure keyframes are sorted in chronological order and handles are set correctly 

        '''
        pass

    def range(self):
        '''Get the time extents for F-Curve 

        :rtype: float array of 2 items in [-inf, inf] 
        :return:  Range, Min/Max values 
        '''
        pass

    def update_autoflags(self, data):
        '''Update FCurve flags set automatically from affected property (currently, integer/discrete flags set when the property is not a float) 

        :param data: Data, Data containing the property controlled by given FCurve 
        :type data: AnyType, (never None)
        '''
        pass

    def convert_to_samples(self, start, end):
        '''Convert current FCurve from keyframes to sample points, if necessary 

        :param start: Start Frame 
        :type start: int in [-500000, 500000]
        :param end: End Frame 
        :type end: int in [-500000, 500000]
        '''
        pass

    def convert_to_keyframes(self, start, end):
        '''Convert current FCurve from sample points to keyframes (linear interpolation), if necessary 

        :param start: Start Frame 
        :type start: int in [-500000, 500000]
        :param end: End Frame 
        :type end: int in [-500000, 500000]
        '''
        pass


class FCurveKeyframePoints:
    '''Collection of keyframe points '''

    def insert(self, frame, value, options={}, keyframe_type='KEYFRAME'):
        '''Add a keyframe point to a F-Curve 

        :param frame: X Value of this keyframe point 
        :type frame: float in [-inf, inf]
        :param value: Y Value of this keyframe point 
        :type value: float in [-inf, inf]
        :param options: Keyframe optionsREPLACE Replace, Don’t add any new keyframes, but just replace existing ones.NEEDED Needed, Only adds keyframes that are needed.FAST Fast, Fast keyframe insertion to avoid recalculating the curve each time. 
        :type options: enum set in {'REPLACE', 'NEEDED', 'FAST'}, (optional)
        :param keyframe_type: Type of keyframe to insertKEYFRAME Keyframe, Normal keyframe - e.g. for key poses.BREAKDOWN Breakdown, A breakdown pose - e.g. for transitions between key poses.MOVING_HOLD Moving Hold, A keyframe that is part of a moving hold.EXTREME Extreme, An ‘extreme’ pose, or some other purpose as needed.JITTER Jitter, A filler or baked keyframe for keying on ones, or some other purpose as needed. 
        :type keyframe_type: enum in ['KEYFRAME', 'BREAKDOWN', 'MOVING_HOLD', 'EXTREME', 'JITTER'], (optional)
        :rtype: Keyframe 
        :return:  Newly created keyframe 
        '''
        pass

    def add(self, count=1):
        '''Add a keyframe point to a F-Curve 

        :param count: Number, Number of points to add to the spline 
        :type count: int in [0, inf], (optional)
        '''
        pass

    def remove(self, keyframe, fast=False):
        '''Remove keyframe from an F-Curve 

        :param keyframe: Keyframe to remove 
        :type keyframe: Keyframe, (never None)
        :param fast: Fast, Fast keyframe removal to avoid recalculating the curve each time 
        :type fast: boolean, (optional)
        '''
        pass


class FCurveModifiers:
    '''Collection of F-Curve Modifiers '''

    active = None
    '''Active F-Curve Modifier 

    :type:  FModifier 
    '''

    def new(self, type):
        '''Add a constraint to this object 

        :param type: Constraint type to addNULL Invalid.GENERATOR Generator, Generate a curve using a factorized or expanded polynomial.FNGENERATOR Built-In Function, Generate a curve using standard math functions such as sin and cos.ENVELOPE Envelope, Reshape F-Curve values - e.g. change amplitude of movements.CYCLES Cycles, Cyclic extend/repeat keyframe sequence.NOISE Noise, Add pseudo-random noise on top of F-Curves.LIMITS Limits, Restrict maximum and minimum values of F-Curve.STEPPED Stepped Interpolation, Snap values to nearest grid-step - e.g. for a stop-motion look. 
        :type type: enum in ['NULL', 'GENERATOR', 'FNGENERATOR', 'ENVELOPE', 'CYCLES', 'NOISE', 'LIMITS', 'STEPPED']
        :rtype: FModifier 
        :return:  New fmodifier 
        '''
        pass

    def remove(self, modifier):
        '''Remove a modifier from this F-Curve 

        :param modifier: Removed modifier 
        :type modifier: FModifier, (never None)
        '''
        pass


class FCurveSample:
    '''Sample point for F-Curve '''

    co = None
    '''Point coordinates 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    select = None
    '''Selection status 

    :type:  boolean, default False 
    '''


class FFmpegSettings:
    '''FFmpeg related settings for the scene '''

    audio_bitrate = None
    '''Audio bitrate (kb/s) 

    :type:  int in [32, 384], default 0 
    '''

    audio_channels = None
    '''Audio channel count 

    :type:  enum in [‘MONO’, ‘STEREO’, ‘SURROUND4’, ‘SURROUND51’, ‘SURROUND71’], default ‘MONO’ 
    '''

    audio_codec = None
    '''FFmpeg audio codec to use 

    :type:  enum in [‘NONE’, ‘MP2’, ‘MP3’, ‘AC3’, ‘AAC’, ‘VORBIS’, ‘FLAC’, ‘PCM’], default ‘NONE’ 
    '''

    audio_mixrate = None
    '''Audio samplerate(samples/s) 

    :type:  int in [8000, 192000], default 0 
    '''

    audio_volume = None
    '''Audio volume 

    :type:  float in [0, 1], default 0.0 
    '''

    buffersize = None
    '''Rate control: buffer size (kb) 

    :type:  int in [0, 2000], default 0 
    '''

    codec = None
    '''FFmpeg codec to use 

    :type:  enum in [‘NONE’, ‘MPEG1’, ‘MPEG2’, ‘MPEG4’, ‘HUFFYUV’, ‘DV’, ‘H264’, ‘THEORA’, ‘FLASH’, ‘FFV1’, ‘QTRLE’, ‘DNXHD’, ‘PNG’], default ‘NONE’ 
    '''

    format = None
    '''Output file format 

    :type:  enum in [‘MPEG1’, ‘MPEG2’, ‘MPEG4’, ‘AVI’, ‘QUICKTIME’, ‘DV’, ‘H264’, ‘XVID’, ‘OGG’, ‘MKV’, ‘FLASH’], default ‘MPEG1’ 
    '''

    gopsize = None
    '''Distance between key frames 

    :type:  int in [0, 100], default 0 
    '''

    maxrate = None
    '''Rate control: max rate (kb/s) 

    :type:  int in [-inf, inf], default 0 
    '''

    minrate = None
    '''Rate control: min rate (kb/s) 

    :type:  int in [-inf, inf], default 0 
    '''

    muxrate = None
    '''Mux rate (bits/s(!)) 

    :type:  int in [0, inf], default 0 
    '''

    packetsize = None
    '''Mux packet size (byte) 

    :type:  int in [0, 16384], default 0 
    '''

    use_autosplit = None
    '''Autosplit output at 2GB boundary 

    :type:  boolean, default False 
    '''

    use_lossless_output = None
    '''Use lossless output for video streams 

    :type:  boolean, default False 
    '''

    video_bitrate = None
    '''Video bitrate (kb/s) 

    :type:  int in [-inf, inf], default 0 
    '''


class FILEBROWSER_UL_dir:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class FModifier:
    '''Modifier for values of F-Curve '''

    active = None
    '''F-Curve Modifier is the one being edited 

    :type:  boolean, default False 
    '''

    blend_in = None
    '''Number of frames from start frame for influence to take effect 

    :type:  float in [-inf, inf], default 0.0 
    '''

    blend_out = None
    '''Number of frames from end frame for influence to fade out 

    :type:  float in [-inf, inf], default 0.0 
    '''

    frame_end = None
    '''Frame that modifier’s influence ends (if Restrict Frame Range is in use) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    frame_start = None
    '''Frame that modifier’s influence starts (if Restrict Frame Range is in use) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    influence = None
    '''Amount of influence F-Curve Modifier will have when not fading in/out 

    :type:  float in [0, 1], default 1.0 
    '''

    is_valid = None
    '''F-Curve Modifier has invalid settings and will not be evaluated 

    :type:  boolean, default False, (readonly) 
    '''

    mute = None
    '''F-Curve Modifier will not be evaluated 

    :type:  boolean, default False 
    '''

    show_expanded = None
    '''F-Curve Modifier’s panel is expanded in UI 

    :type:  boolean, default False 
    '''

    type = None
    '''F-Curve Modifier Type 

    :type:  enum in [‘NULL’, ‘GENERATOR’, ‘FNGENERATOR’, ‘ENVELOPE’, ‘CYCLES’, ‘NOISE’, ‘LIMITS’, ‘STEPPED’], default ‘NULL’, (readonly) 
    '''

    use_influence = None
    '''F-Curve Modifier’s effects will be tempered by a default factor 

    :type:  boolean, default False 
    '''

    use_restricted_range = None
    '''F-Curve Modifier is only applied for the specified frame range to help mask off effects in order to chain them 

    :type:  boolean, default False 
    '''


class FModifierCycles:
    '''Repeat the values of the modified F-Curve '''

    cycles_after = None
    '''Maximum number of cycles to allow after last keyframe (0 = infinite) 

    :type:  int in [-32768, 32767], default 0 
    '''

    cycles_before = None
    '''Maximum number of cycles to allow before first keyframe (0 = infinite) 

    :type:  int in [-32768, 32767], default 0 
    '''

    mode_after = None
    '''Cycling mode to use after last keyframe 

    :type:  enum in [‘NONE’, ‘REPEAT’, ‘REPEAT_OFFSET’, ‘MIRROR’], default ‘NONE’ 
    '''

    mode_before = None
    '''Cycling mode to use before first keyframe 

    :type:  enum in [‘NONE’, ‘REPEAT’, ‘REPEAT_OFFSET’, ‘MIRROR’], default ‘NONE’ 
    '''


class FModifierEnvelope:
    '''Scale the values of the modified F-Curve '''

    control_points = None
    '''Control points defining the shape of the envelope 

    :type:  FModifierEnvelopeControlPoints bpy_prop_collection of FModifierEnvelopeControlPoint, (readonly) 
    '''

    default_max = None
    '''Upper distance from Reference Value for 1:1 default influence 

    :type:  float in [-inf, inf], default 0.0 
    '''

    default_min = None
    '''Lower distance from Reference Value for 1:1 default influence 

    :type:  float in [-inf, inf], default 0.0 
    '''

    reference_value = None
    '''Value that envelope’s influence is centered around / based on 

    :type:  float in [-inf, inf], default 0.0 
    '''


class FModifierEnvelopeControlPoint:
    '''Control point for envelope F-Modifier '''

    frame = None
    '''Frame this control-point occurs on 

    :type:  float in [-inf, inf], default 0.0 
    '''

    max = None
    '''Upper bound of envelope at this control-point 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min = None
    '''Lower bound of envelope at this control-point 

    :type:  float in [-inf, inf], default 0.0 
    '''


class FModifierEnvelopeControlPoints:
    '''Control points defining the shape of the envelope '''

    def add(self, frame):
        '''Add a control point to a FModifierEnvelope 

        :param frame: Frame to add this control-point 
        :type frame: float in [-inf, inf]
        :rtype: FModifierEnvelopeControlPoint 
        :return:  Newly created control-point 
        '''
        pass

    def remove(self, point):
        '''Remove a control-point from an FModifierEnvelope 

        :param point: Control-point to remove 
        :type point: FModifierEnvelopeControlPoint, (never None)
        '''
        pass


class FModifierFunctionGenerator:
    '''Generate values using a Built-In Function '''

    amplitude = None
    '''Scale factor determining the maximum/minimum values 

    :type:  float in [-inf, inf], default 0.0 
    '''

    function_type = None
    '''Type of built-in function to use 

    :type:  enum in [‘SIN’, ‘COS’, ‘TAN’, ‘SQRT’, ‘LN’, ‘SINC’], default ‘SIN’ 
    '''

    phase_multiplier = None
    '''Scale factor determining the ‘speed’ of the function 

    :type:  float in [-inf, inf], default 0.0 
    '''

    phase_offset = None
    '''Constant factor to offset time by for function 

    :type:  float in [-inf, inf], default 0.0 
    '''

    use_additive = None
    '''Values generated by this modifier are applied on top of the existing values instead of overwriting them 

    :type:  boolean, default False 
    '''

    value_offset = None
    '''Constant factor to offset values by 

    :type:  float in [-inf, inf], default 0.0 
    '''


class FModifierGenerator:
    '''Deterministically generate values for the modified F-Curve '''

    coefficients = None
    '''Coefficients for ‘x’ (starting from lowest power of x^0) 

    :type:  float array of 32 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    mode = None
    '''Type of generator to use 

    :type:  enum in [‘POLYNOMIAL’, ‘POLYNOMIAL_FACTORISED’], default ‘POLYNOMIAL’ 
    '''

    poly_order = None
    '''The highest power of ‘x’ for this polynomial (number of coefficients - 1) 

    :type:  int in [1, 100], default 0 
    '''

    use_additive = None
    '''Values generated by this modifier are applied on top of the existing values instead of overwriting them 

    :type:  boolean, default False 
    '''


class FModifierLimits:
    '''Limit the time/value ranges of the modified F-Curve '''

    max_x = None
    '''Highest X value to allow 

    :type:  float in [-inf, inf], default 0.0 
    '''

    max_y = None
    '''Highest Y value to allow 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_x = None
    '''Lowest X value to allow 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_y = None
    '''Lowest Y value to allow 

    :type:  float in [-inf, inf], default 0.0 
    '''

    use_max_x = None
    '''Use the maximum X value 

    :type:  boolean, default False 
    '''

    use_max_y = None
    '''Use the maximum Y value 

    :type:  boolean, default False 
    '''

    use_min_x = None
    '''Use the minimum X value 

    :type:  boolean, default False 
    '''

    use_min_y = None
    '''Use the minimum Y value 

    :type:  boolean, default False 
    '''


class FModifierNoise:
    '''Give randomness to the modified F-Curve '''

    blend_type = None
    '''Method of modifying the existing F-Curve 

    :type:  enum in [‘REPLACE’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’], default ‘REPLACE’ 
    '''

    depth = None
    '''Amount of fine level detail present in the noise 

    :type:  int in [0, 32767], default 0 
    '''

    offset = None
    '''Time offset for the noise effect 

    :type:  float in [-inf, inf], default 0.0 
    '''

    phase = None
    '''A random seed for the noise effect 

    :type:  float in [-inf, inf], default 0.0 
    '''

    scale = None
    '''Scaling (in time) of the noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    strength = None
    '''Amplitude of the noise - the amount that it modifies the underlying curve 

    :type:  float in [-inf, inf], default 0.0 
    '''


class FModifierPython:
    '''Perform user-defined operation on the modified F-Curve '''

    pass


class FModifierStepped:
    '''Hold each interpolated value from the F-Curve for several frames without changing the timing '''

    frame_end = None
    '''Frame that modifier’s influence ends (if applicable) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    frame_offset = None
    '''Reference number of frames before frames get held (use to get hold for ‘1-3’ vs ‘5-7’ holding patterns) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    frame_start = None
    '''Frame that modifier’s influence starts (if applicable) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    frame_step = None
    '''Number of frames to hold each value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    use_frame_end = None
    '''Restrict modifier to only act before its ‘end’ frame 

    :type:  boolean, default False 
    '''

    use_frame_start = None
    '''Restrict modifier to only act after its ‘start’ frame 

    :type:  boolean, default False 
    '''


class FieldSettings:
    '''Field settings for an object in physics simulation '''

    apply_to_location = None
    '''Effect particles’ location 

    :type:  boolean, default False 
    '''

    apply_to_rotation = None
    '''Effect particles’ dynamic rotation 

    :type:  boolean, default False 
    '''

    distance_max = None
    '''Maximum distance for the field to work 

    :type:  float in [0, 1000], default 0.0 
    '''

    distance_min = None
    '''Minimum distance for the field’s fall-off 

    :type:  float in [0, 1000], default 0.0 
    '''

    falloff_power = None
    '''Falloff power (real gravitational falloff = 2) 

    :type:  float in [0, 10], default 0.0 
    '''

    falloff_type = None
    '''

    :type:  enum in [‘SPHERE’, ‘TUBE’, ‘CONE’], default ‘SPHERE’ 
    '''

    flow = None
    '''Convert effector force into air flow velocity 

    :type:  float in [0, 10], default 0.0 
    '''

    guide_clump_amount = None
    '''Amount of clumping 

    :type:  float in [-1, 1], default 0.0 
    '''

    guide_clump_shape = None
    '''Shape of clumping 

    :type:  float in [-0.999, 0.999], default 0.0 
    '''

    guide_free = None
    '''Guide-free time from particle life’s end 

    :type:  float in [0, 0.99], default 0.0 
    '''

    guide_kink_amplitude = None
    '''The amplitude of the offset 

    :type:  float in [0, 10], default 0.0 
    '''

    guide_kink_axis = None
    '''Which axis to use for offset 

    :type:  enum in [‘X’, ‘Y’, ‘Z’], default ‘X’ 
    '''

    guide_kink_frequency = None
    '''The frequency of the offset (1/total length) 

    :type:  float in [0, 10], default 0.0 
    '''

    guide_kink_shape = None
    '''Adjust the offset to the beginning/end 

    :type:  float in [-0.999, 0.999], default 0.0 
    '''

    guide_kink_type = None
    '''Type of periodic offset on the curve 

    :type:  enum in [‘NONE’, ‘CURL’, ‘RADIAL’, ‘WAVE’, ‘BRAID’, ‘ROTATION’, ‘ROLL’], default ‘NONE’ 
    '''

    guide_minimum = None
    '''The distance from which particles are affected fully 

    :type:  float in [0, 1000], default 0.0 
    '''

    harmonic_damping = None
    '''Damping of the harmonic force 

    :type:  float in [0, 10], default 0.0 
    '''

    inflow = None
    '''Inwards component of the vortex force 

    :type:  float in [-10, 10], default 0.0 
    '''

    linear_drag = None
    '''Drag component proportional to velocity 

    :type:  float in [-2, 2], default 0.0 
    '''

    noise = None
    '''Amount of noise for the force strength 

    :type:  float in [0, 10], default 0.0 
    '''

    quadratic_drag = None
    '''Drag component proportional to the square of velocity 

    :type:  float in [-2, 2], default 0.0 
    '''

    radial_falloff = None
    '''Radial falloff power (real gravitational falloff = 2) 

    :type:  float in [0, 10], default 0.0 
    '''

    radial_max = None
    '''Maximum radial distance for the field to work 

    :type:  float in [0, 1000], default 0.0 
    '''

    radial_min = None
    '''Minimum radial distance for the field’s fall-off 

    :type:  float in [0, 1000], default 0.0 
    '''

    rest_length = None
    '''Rest length of the harmonic force 

    :type:  float in [0, 1000], default 0.0 
    '''

    seed = None
    '''Seed of the noise 

    :type:  int in [1, 128], default 0 
    '''

    shape = None
    '''Which direction is used to calculate the effector force 

    :type:  enum in [‘POINT’, ‘PLANE’, ‘SURFACE’, ‘POINTS’], default ‘POINT’ 
    '''

    size = None
    '''Size of the turbulence 

    :type:  float in [0, 10], default 0.0 
    '''

    source_object = None
    '''Select domain object of the smoke simulation 

    :type:  Object 
    '''

    strength = None
    '''Strength of force field 

    :type:  float in [-inf, inf], default 0.0 
    '''

    texture = None
    '''Texture to use as force 

    :type:  Texture 
    '''

    texture_mode = None
    '''How the texture effect is calculated (RGB & Curl need a RGB texture, else Gradient will be used instead) 

    :type:  enum in [‘RGB’, ‘GRADIENT’, ‘CURL’], default ‘RGB’ 
    '''

    texture_nabla = None
    '''Defines size of derivative offset used for calculating gradient and curl 

    :type:  float in [0.0001, 1], default 0.0 
    '''

    type = None
    '''Type of field 

    :type:  enum in [‘NONE’, ‘FORCE’, ‘WIND’, ‘VORTEX’, ‘MAGNET’, ‘HARMONIC’, ‘CHARGE’, ‘LENNARDJ’, ‘TEXTURE’, ‘GUIDE’, ‘BOID’, ‘TURBULENCE’, ‘DRAG’, ‘SMOKE_FLOW’], default ‘NONE’ 
    '''

    use_2d_force = None
    '''Apply force only in 2D 

    :type:  boolean, default False 
    '''

    use_absorption = None
    '''Force gets absorbed by collision objects 

    :type:  boolean, default False 
    '''

    use_global_coords = None
    '''Use effector/global coordinates for turbulence 

    :type:  boolean, default False 
    '''

    use_guide_path_add = None
    '''Based on distance/falloff it adds a portion of the entire path 

    :type:  boolean, default False 
    '''

    use_guide_path_weight = None
    '''Use curve weights to influence the particle influence along the curve 

    :type:  boolean, default False 
    '''

    use_max_distance = None
    '''Use a maximum distance for the field to work 

    :type:  boolean, default False 
    '''

    use_min_distance = None
    '''Use a minimum distance for the field’s fall-off 

    :type:  boolean, default False 
    '''

    use_multiple_springs = None
    '''Every point is effected by multiple springs 

    :type:  boolean, default False 
    '''

    use_object_coords = None
    '''Use object/global coordinates for texture 

    :type:  boolean, default False 
    '''

    use_radial_max = None
    '''Use a maximum radial distance for the field to work 

    :type:  boolean, default False 
    '''

    use_radial_min = None
    '''Use a minimum radial distance for the field’s fall-off 

    :type:  boolean, default False 
    '''

    use_root_coords = None
    '''Texture coordinates from root particle locations 

    :type:  boolean, default False 
    '''

    use_smoke_density = None
    '''Adjust force strength based on smoke density 

    :type:  boolean, default False 
    '''

    z_direction = None
    '''Effect in full or only positive/negative Z direction 

    :type:  enum in [‘BOTH’, ‘POSITIVE’, ‘NEGATIVE’], default ‘BOTH’ 
    '''


class FileBrowserFSMenuEntry:
    '''File Select Parameters '''

    is_valid = None
    '''Whether this path is currently reachable 

    :type:  boolean, default False, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''

    path = None
    '''

    :type:  string, default “”, (never None) 
    '''

    use_save = None
    '''Whether this path is saved in bookmarks, or generated from OS 

    :type:  boolean, default False, (readonly) 
    '''


class FileSelectParams:
    '''File Select Parameters '''

    directory = None
    '''Directory displayed in the file browser 

    :type:  string, default “”, (never None) 
    '''

    display_size = None
    '''Change the size of the display (width of columns or thumbnails size) 

    :type:  enum in [‘TINY’, ‘SMALL’, ‘NORMAL’, ‘LARGE’], default ‘TINY’ 
    '''

    display_type = None
    '''Display mode for the file list 

    :type:  enum in [‘LIST_SHORT’, ‘LIST_LONG’, ‘THUMBNAIL’], default ‘LIST_SHORT’ 
    '''

    filename = None
    '''Active file in the file browser 

    :type:  string, default “”, (never None) 
    '''

    filter_glob = None
    '''

    :type:  string, default “”, (never None) 
    '''

    filter_id = None
    '''Which ID types to show/hide, when browsing a library 

    :type:  enum set in {‘ACTION’, ‘ARMATURE’, ‘BRUSH’, ‘CAMERA’, ‘CACHEFILE’, ‘CURVE’, ‘GREASE_PENCIL’, ‘GROUP’, ‘IMAGE’, ‘LAMP’, ‘LINESTYLE’, ‘LATTICE’, ‘MATERIAL’, ‘METABALL’, ‘MOVIE_CLIP’, ‘MESH’, ‘MASK’, ‘NODE_TREE’, ‘OBJECT’, ‘PARTICLE_SETTINGS’, ‘PALETTE’, ‘PAINT_CURVE’, ‘SCENE’, ‘SPEAKER’, ‘SOUND’, ‘TEXTURE’, ‘TEXT’, ‘FONT’, ‘WORLD’}, default {‘ACTION’} 
    '''

    filter_id_category = None
    '''Which ID categories to show/hide, when browsing a library 

    :type:  enum set in {‘SCENE’, ‘ANIMATION’, ‘OBJECT’, ‘GEOMETRY’, ‘SHADING’, ‘IMAGE’, ‘ENVIRONMENT’, ‘MISC’}, default {‘SCENE’} 
    '''

    filter_search = None
    '''Filter by name, supports ‘*’ wildcard 

    :type:  string, default “”, (never None) 
    '''

    recursion_level = None
    '''Numbers of dirtree levels to show simultaneously 

    :type:  enum in [‘NONE’, ‘BLEND’, ‘ALL_1’, ‘ALL_2’, ‘ALL_3’], default ‘NONE’ 
    '''

    show_hidden = None
    '''Show hidden dot files 

    :type:  boolean, default False 
    '''

    sort_method = None
    '''

    :type:  enum in [‘FILE_SORT_ALPHA’, ‘FILE_SORT_EXTENSION’, ‘FILE_SORT_TIME’, ‘FILE_SORT_SIZE’], default ‘FILE_SORT_ALPHA’ 
    '''

    title = None
    '''Title for the file browser 

    :type:  string, default “”, (readonly, never None) 
    '''

    use_filter = None
    '''Enable filtering of files 

    :type:  boolean, default False 
    '''

    use_filter_backup = None
    '''Show .blend1, .blend2, etc. files 

    :type:  boolean, default False 
    '''

    use_filter_blender = None
    '''Show .blend files 

    :type:  boolean, default False 
    '''

    use_filter_blendid = None
    '''Show .blend files items (objects, materials, etc.) 

    :type:  boolean, default False 
    '''

    use_filter_folder = None
    '''Show folders 

    :type:  boolean, default False 
    '''

    use_filter_font = None
    '''Show font files 

    :type:  boolean, default False 
    '''

    use_filter_image = None
    '''Show image files 

    :type:  boolean, default False 
    '''

    use_filter_movie = None
    '''Show movie files 

    :type:  boolean, default False 
    '''

    use_filter_script = None
    '''Show script files 

    :type:  boolean, default False 
    '''

    use_filter_sound = None
    '''Show sound files 

    :type:  boolean, default False 
    '''

    use_filter_text = None
    '''Show text files 

    :type:  boolean, default False 
    '''

    use_library_browsing = None
    '''Whether we may browse blender files’ content or not 

    :type:  boolean, default False, (readonly) 
    '''


class Filter2DActuator:
    '''Actuator to apply screen graphic effects '''

    filter_pass = None
    '''Set filter order 

    :type:  int in [0, 99], default 0 
    '''

    glsl_shader = None
    '''

    :type:  Text 
    '''

    mode = None
    '''

    :type:  enum in [‘ENABLE’, ‘DISABLE’, ‘REMOVE’, ‘MOTIONBLUR’, ‘BLUR’, ‘SHARPEN’, ‘DILATION’, ‘EROSION’, ‘LAPLACIAN’, ‘SOBEL’, ‘PREWITT’, ‘GRAYSCALE’, ‘SEPIA’, ‘INVERT’, ‘CUSTOMFILTER’], default ‘REMOVE’ 
    '''

    motion_blur_factor = None
    '''Motion blur factor 

    :type:  float in [0, 1], default 0.0 
    '''

    use_motion_blur = None
    '''Enable/Disable Motion Blur 

    :type:  boolean, default False 
    '''


class FloatProperty:
    '''RNA floating pointer number property definition '''

    array_length = None
    '''Maximum length of the array, 0 means unlimited 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    default = None
    '''Default value for this number 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    default_array = None
    '''Default value for this array 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0), (readonly) 
    '''

    hard_max = None
    '''Maximum value used by buttons 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    hard_min = None
    '''Minimum value used by buttons 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    is_array = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    precision = None
    '''Number of digits after the dot used by buttons 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    soft_max = None
    '''Maximum value used by buttons 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    soft_min = None
    '''Minimum value used by buttons 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    step = None
    '''Step size used by number buttons, for floats 1/100th of the step size 

    :type:  float in [0, inf], default 0.0, (readonly) 
    '''


class FloorConstraint:
    '''Use the target object for location limitation '''

    floor_location = None
    '''Location of target that object will not pass through 

    :type:  enum in [‘FLOOR_X’, ‘FLOOR_Y’, ‘FLOOR_Z’, ‘FLOOR_NEGATIVE_X’, ‘FLOOR_NEGATIVE_Y’, ‘FLOOR_NEGATIVE_Z’], default ‘FLOOR_X’ 
    '''

    offset = None
    '''Offset of floor from object origin 

    :type:  float in [-inf, inf], default 0.0 
    '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    use_rotation = None
    '''Use the target’s rotation to determine floor 

    :type:  boolean, default False 
    '''

    use_sticky = None
    '''Immobilize object while constrained 

    :type:  boolean, default False 
    '''


class FluidFluidSettings:
    '''Fluid simulation settings for the fluid in the simulation '''

    initial_velocity = None
    '''Initial velocity of fluid 

    :type:  float array of 3 items in [-1000.1, 1000.1], default (0.0, 0.0, 0.0) 
    '''

    use = None
    '''Object contributes to the fluid simulation 

    :type:  boolean, default False 
    '''

    use_animated_mesh = None
    '''Export this mesh as an animated one (slower and enforces No Slip, only use if really necessary [e.g. armatures or parented objects], animated pos/rot/scale F-Curves do not require it) 

    :type:  boolean, default False 
    '''

    volume_initialization = None
    '''Volume initialization type (WARNING: complex volumes might require too much memory and break simulation) 

    :type:  enum in [‘VOLUME’, ‘SHELL’, ‘BOTH’], default ‘VOLUME’ 
    '''


class FluidSettings:
    '''Fluid simulation settings for an object taking part in the simulation '''

    type = None
    '''Type of participation in the fluid simulation 

    :type:  enum in [‘NONE’, ‘DOMAIN’, ‘FLUID’, ‘OBSTACLE’, ‘INFLOW’, ‘OUTFLOW’, ‘PARTICLE’, ‘CONTROL’], default ‘NONE’ 
    '''


class FluidSimulationModifier:
    '''Fluid simulation modifier '''

    settings = None
    '''Settings for how this object is used in the fluid simulation 

    :type:  FluidSettings, (readonly, never None) 
    '''


class FluidVertexVelocity:
    '''Velocity of a simulated fluid mesh '''

    velocity = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0), (readonly) 
    '''


class FollowPathConstraint:
    '''Lock motion to the target path '''

    forward_axis = None
    '''Axis that points forward along the path 

    :type:  enum in [‘FORWARD_X’, ‘FORWARD_Y’, ‘FORWARD_Z’, ‘TRACK_NEGATIVE_X’, ‘TRACK_NEGATIVE_Y’, ‘TRACK_NEGATIVE_Z’], default ‘FORWARD_X’ 
    '''

    offset = None
    '''Offset from the position corresponding to the time frame 

    :type:  float in [-500000, 500000], default 0.0 
    '''

    offset_factor = None
    '''Percentage value defining target position along length of curve 

    :type:  float in [0, 1], default 0.0 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    up_axis = None
    '''Axis that points upward 

    :type:  enum in [‘UP_X’, ‘UP_Y’, ‘UP_Z’], default ‘UP_X’ 
    '''

    use_curve_follow = None
    '''Object will follow the heading and banking of the curve 

    :type:  boolean, default False 
    '''

    use_curve_radius = None
    '''Object is scaled by the curve radius 

    :type:  boolean, default False 
    '''

    use_fixed_location = None
    '''Object will stay locked to a single point somewhere along the length of the curve regardless of time 

    :type:  boolean, default False 
    '''


class FollowTrackConstraint:
    '''Lock motion to the target motion track '''

    camera = None
    '''Camera to which motion is parented (if empty active scene camera is used) 

    :type:  Object 
    '''

    clip = None
    '''Movie Clip to get tracking data from 

    :type:  MovieClip 
    '''

    depth_object = None
    '''Object used to define depth in camera space by projecting onto surface of this object 

    :type:  Object 
    '''

    frame_method = None
    '''How the footage fits in the camera frame 

    :type:  enum in [‘STRETCH’, ‘FIT’, ‘CROP’], default ‘STRETCH’ 
    '''

    object = None
    '''Movie tracking object to follow (if empty, camera object is used) 

    :type:  string, default “”, (never None) 
    '''

    track = None
    '''Movie tracking track to follow 

    :type:  string, default “”, (never None) 
    '''

    use_3d_position = None
    '''Use 3D position of track to parent to 

    :type:  boolean, default False 
    '''

    use_active_clip = None
    '''Use active clip defined in scene 

    :type:  boolean, default False 
    '''

    use_undistorted_position = None
    '''Parent to undistorted position of 2D track 

    :type:  boolean, default False 
    '''


class FreestyleLineSet:
    '''Line set for associating lines and style parameters '''

    edge_type_combination = None
    '''Specify a logical combination of selection conditions on feature edge types 

    :type:  enum in [‘OR’, ‘AND’], default ‘OR’ 
    '''

    edge_type_negation = None
    '''Specify either inclusion or exclusion of feature edges selected by edge types 

    :type:  enum in [‘INCLUSIVE’, ‘EXCLUSIVE’], default ‘INCLUSIVE’ 
    '''

    exclude_border = None
    '''Exclude border edges 

    :type:  boolean, default False 
    '''

    exclude_contour = None
    '''Exclude contours 

    :type:  boolean, default False 
    '''

    exclude_crease = None
    '''Exclude crease edges 

    :type:  boolean, default False 
    '''

    exclude_edge_mark = None
    '''Exclude edge marks 

    :type:  boolean, default False 
    '''

    exclude_external_contour = None
    '''Exclude external contours 

    :type:  boolean, default False 
    '''

    exclude_material_boundary = None
    '''Exclude edges at material boundaries 

    :type:  boolean, default False 
    '''

    exclude_ridge_valley = None
    '''Exclude ridges and valleys 

    :type:  boolean, default False 
    '''

    exclude_silhouette = None
    '''Exclude silhouette edges 

    :type:  boolean, default False 
    '''

    exclude_suggestive_contour = None
    '''Exclude suggestive contours 

    :type:  boolean, default False 
    '''

    face_mark_condition = None
    '''Specify a feature edge selection condition based on face marks 

    :type:  enum in [‘ONE’, ‘BOTH’], default ‘ONE’ 
    '''

    face_mark_negation = None
    '''Specify either inclusion or exclusion of feature edges selected by face marks 

    :type:  enum in [‘INCLUSIVE’, ‘EXCLUSIVE’], default ‘INCLUSIVE’ 
    '''

    group = None
    '''A group of objects based on which feature edges are selected 

    :type:  Group 
    '''

    group_negation = None
    '''Specify either inclusion or exclusion of feature edges belonging to a group of objects 

    :type:  enum in [‘INCLUSIVE’, ‘EXCLUSIVE’], default ‘INCLUSIVE’ 
    '''

    linestyle = None
    '''Line style settings 

    :type:  FreestyleLineStyle, (never None) 
    '''

    name = None
    '''Line set name 

    :type:  string, default “”, (never None) 
    '''

    qi_end = None
    '''Last QI value of the QI range 

    :type:  int in [0, inf], default 0 
    '''

    qi_start = None
    '''First QI value of the QI range 

    :type:  int in [0, inf], default 0 
    '''

    select_border = None
    '''Select border edges (open mesh edges) 

    :type:  boolean, default False 
    '''

    select_by_edge_types = None
    '''Select feature edges based on edge types 

    :type:  boolean, default False 
    '''

    select_by_face_marks = None
    '''Select feature edges by face marks 

    :type:  boolean, default False 
    '''

    select_by_group = None
    '''Select feature edges based on a group of objects 

    :type:  boolean, default False 
    '''

    select_by_image_border = None
    '''Select feature edges by image border (less memory consumption) 

    :type:  boolean, default False 
    '''

    select_by_visibility = None
    '''Select feature edges based on visibility 

    :type:  boolean, default False 
    '''

    select_contour = None
    '''Select contours (outer silhouettes of each object) 

    :type:  boolean, default False 
    '''

    select_crease = None
    '''Select crease edges (those between two faces making an angle smaller than the Crease Angle) 

    :type:  boolean, default False 
    '''

    select_edge_mark = None
    '''Select edge marks (edges annotated by Freestyle edge marks) 

    :type:  boolean, default False 
    '''

    select_external_contour = None
    '''Select external contours (outer silhouettes of occluding and occluded objects) 

    :type:  boolean, default False 
    '''

    select_material_boundary = None
    '''Select edges at material boundaries 

    :type:  boolean, default False 
    '''

    select_ridge_valley = None
    '''Select ridges and valleys (boundary lines between convex and concave areas of surface) 

    :type:  boolean, default False 
    '''

    select_silhouette = None
    '''Select silhouettes (edges at the boundary of visible and hidden faces) 

    :type:  boolean, default False 
    '''

    select_suggestive_contour = None
    '''Select suggestive contours (almost silhouette/contour edges) 

    :type:  boolean, default False 
    '''

    show_render = None
    '''Enable or disable this line set during stroke rendering 

    :type:  boolean, default False 
    '''

    visibility = None
    '''Determine how to use visibility for feature edge selection 

    :type:  enum in [‘VISIBLE’, ‘HIDDEN’, ‘RANGE’], default ‘VISIBLE’ 
    '''


class FreestyleLineStyle:
    '''Freestyle line style, reusable by multiple line sets '''

    active_texture = None
    '''Active texture slot being displayed 

    :type:  Texture 
    '''

    active_texture_index = None
    '''Index of active texture slot 

    :type:  int in [0, 17], default 0 
    '''

    alpha = None
    '''Base alpha transparency, possibly modified by alpha transparency modifiers 

    :type:  float in [0, 1], default 0.0 
    '''

    alpha_modifiers = None
    '''List of alpha transparency modifiers 

    :type:  LineStyleAlphaModifiers bpy_prop_collection of LineStyleAlphaModifier, (readonly) 
    '''

    angle_max = None
    '''Maximum 2D angle for splitting chains 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    angle_min = None
    '''Minimum 2D angle for splitting chains 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    caps = None
    '''Select the shape of both ends of strokes 

    :type:  enum in [‘BUTT’, ‘ROUND’, ‘SQUARE’], default ‘BUTT’ 
    '''

    chain_count = None
    '''Chain count for the selection of first N chains 

    :type:  int in [0, inf], default 0 
    '''

    chaining = None
    '''Select the way how feature edges are jointed to form chains 

    :type:  enum in [‘PLAIN’, ‘SKETCHY’], default ‘PLAIN’ 
    '''

    color = None
    '''Base line color, possibly modified by line color modifiers 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    color_modifiers = None
    '''List of line color modifiers 

    :type:  LineStyleColorModifiers bpy_prop_collection of LineStyleColorModifier, (readonly) 
    '''

    dash1 = None
    '''Length of the 1st dash for dashed lines 

    :type:  int in [0, 65535], default 0 
    '''

    dash2 = None
    '''Length of the 2nd dash for dashed lines 

    :type:  int in [0, 65535], default 0 
    '''

    dash3 = None
    '''Length of the 3rd dash for dashed lines 

    :type:  int in [0, 65535], default 0 
    '''

    gap1 = None
    '''Length of the 1st gap for dashed lines 

    :type:  int in [0, 65535], default 0 
    '''

    gap2 = None
    '''Length of the 2nd gap for dashed lines 

    :type:  int in [0, 65535], default 0 
    '''

    gap3 = None
    '''Length of the 3rd gap for dashed lines 

    :type:  int in [0, 65535], default 0 
    '''

    geometry_modifiers = None
    '''List of stroke geometry modifiers 

    :type:  LineStyleGeometryModifiers bpy_prop_collection of LineStyleGeometryModifier, (readonly) 
    '''

    integration_type = None
    '''Select the way how the sort key is computed for each chain 

    :type:  enum in [‘MEAN’, ‘MIN’, ‘MAX’, ‘FIRST’, ‘LAST’], default ‘MEAN’ 
    '''

    length_max = None
    '''Maximum curvilinear 2D length for the selection of chains 

    :type:  float in [0, 10000], default 0.0 
    '''

    length_min = None
    '''Minimum curvilinear 2D length for the selection of chains 

    :type:  float in [0, 10000], default 0.0 
    '''

    material_boundary = None
    '''If true, chains of feature edges are split at material boundaries 

    :type:  boolean, default False 
    '''

    node_tree = None
    '''Node tree for node-based shaders 

    :type:  NodeTree, (readonly) 
    '''

    panel = None
    '''Select the property panel to be shown 

    :type:  enum in [‘STROKES’, ‘COLOR’, ‘ALPHA’, ‘THICKNESS’, ‘GEOMETRY’, ‘TEXTURE’], default ‘STROKES’ 
    '''

    rounds = None
    '''Number of rounds in a sketchy multiple touch 

    :type:  int in [1, 1000], default 0 
    '''

    sort_key = None
    '''Select the sort key to determine the stacking order of chains 

    :type:  enum in [‘DISTANCE_FROM_CAMERA’, ‘2D_LENGTH’, ‘PROJECTED_X’, ‘PROJECTED_Y’], default ‘DISTANCE_FROM_CAMERA’ 
    '''

    sort_order = None
    '''Select the sort order 

    :type:  enum in [‘DEFAULT’, ‘REVERSE’], default ‘DEFAULT’ 
    '''

    split_dash1 = None
    '''Length of the 1st dash for splitting 

    :type:  int in [0, 65535], default 0 
    '''

    split_dash2 = None
    '''Length of the 2nd dash for splitting 

    :type:  int in [0, 65535], default 0 
    '''

    split_dash3 = None
    '''Length of the 3rd dash for splitting 

    :type:  int in [0, 65535], default 0 
    '''

    split_gap1 = None
    '''Length of the 1st gap for splitting 

    :type:  int in [0, 65535], default 0 
    '''

    split_gap2 = None
    '''Length of the 2nd gap for splitting 

    :type:  int in [0, 65535], default 0 
    '''

    split_gap3 = None
    '''Length of the 3rd gap for splitting 

    :type:  int in [0, 65535], default 0 
    '''

    split_length = None
    '''Curvilinear 2D length for chain splitting 

    :type:  float in [0, 10000], default 0.0 
    '''

    texture_slots = None
    '''Texture slots defining the mapping and influence of textures 

    :type:  LineStyleTextureSlots bpy_prop_collection of LineStyleTextureSlot, (readonly) 
    '''

    texture_spacing = None
    '''Spacing for textures along stroke length 

    :type:  float in [0.01, 100], default 0.0 
    '''

    thickness = None
    '''Base line thickness, possibly modified by line thickness modifiers 

    :type:  float in [0, 10000], default 0.0 
    '''

    thickness_modifiers = None
    '''List of line thickness modifiers 

    :type:  LineStyleThicknessModifiers bpy_prop_collection of LineStyleThicknessModifier, (readonly) 
    '''

    thickness_position = None
    '''Thickness position of silhouettes and border edges (applicable when plain chaining is used with the Same Object option) 

    :type:  enum in [‘CENTER’, ‘INSIDE’, ‘OUTSIDE’, ‘RELATIVE’], default ‘CENTER’ 
    '''

    thickness_ratio = None
    '''A number between 0 (inside) and 1 (outside) specifying the relative position of stroke thickness 

    :type:  float in [0, 1], default 0.0 
    '''

    use_angle_max = None
    '''Split chains at points with angles larger than the maximum 2D angle 

    :type:  boolean, default False 
    '''

    use_angle_min = None
    '''Split chains at points with angles smaller than the minimum 2D angle 

    :type:  boolean, default False 
    '''

    use_chain_count = None
    '''Enable the selection of first N chains 

    :type:  boolean, default False 
    '''

    use_chaining = None
    '''Enable chaining of feature edges 

    :type:  boolean, default False 
    '''

    use_dashed_line = None
    '''Enable or disable dashed line 

    :type:  boolean, default False 
    '''

    use_length_max = None
    '''Enable the selection of chains by a maximum 2D length 

    :type:  boolean, default False 
    '''

    use_length_min = None
    '''Enable the selection of chains by a minimum 2D length 

    :type:  boolean, default False 
    '''

    use_nodes = None
    '''Use shader nodes for the line style 

    :type:  boolean, default False 
    '''

    use_same_object = None
    '''If true, only feature edges of the same object are joined 

    :type:  boolean, default False 
    '''

    use_sorting = None
    '''Arrange the stacking order of strokes 

    :type:  boolean, default False 
    '''

    use_split_length = None
    '''Enable chain splitting by curvilinear 2D length 

    :type:  boolean, default False 
    '''

    use_split_pattern = None
    '''Enable chain splitting by dashed line patterns 

    :type:  boolean, default False 
    '''

    use_texture = None
    '''Enable or disable textured strokes 

    :type:  boolean, default False 
    '''


class FreestyleModuleSettings:
    '''Style module configuration for specifying a style module '''

    script = None
    '''Python script to define a style module 

    :type:  Text 
    '''

    use = None
    '''Enable or disable this style module during stroke rendering 

    :type:  boolean, default False 
    '''


class FreestyleModules:
    '''A list of style modules (to be applied from top to bottom) '''

    def new(self):
        '''Add a style module to scene render layer Freestyle settings 

        :rtype: FreestyleModuleSettings 
        :return:  Newly created style module 
        '''
        pass

    def remove(self, module):
        '''Remove a style module from scene render layer Freestyle settings 

        :param module: Style module to remove 
        :type module: FreestyleModuleSettings, (never None)
        '''
        pass


class FreestyleSettings:
    '''Freestyle settings for a SceneRenderLayer data-block '''

    crease_angle = None
    '''Angular threshold for detecting crease edges 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    kr_derivative_epsilon = None
    '''Kr derivative epsilon for computing suggestive contours 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    linesets = None
    '''

    :type:  Linesets bpy_prop_collection of FreestyleLineSet, (readonly) 
    '''

    mode = None
    '''Select the Freestyle control mode 

    :type:  enum in [‘SCRIPT’, ‘EDITOR’], default ‘SCRIPT’ 
    '''

    modules = None
    '''A list of style modules (to be applied from top to bottom) 

    :type:  FreestyleModules bpy_prop_collection of FreestyleModuleSettings, (readonly) 
    '''

    sphere_radius = None
    '''Sphere radius for computing curvatures 

    :type:  float in [0, 1000], default 0.0 
    '''

    use_advanced_options = None
    '''Enable advanced edge detection options (sphere radius and Kr derivative epsilon) 

    :type:  boolean, default False 
    '''

    use_culling = None
    '''If enabled, out-of-view edges are ignored 

    :type:  boolean, default False 
    '''

    use_material_boundaries = None
    '''Enable material boundaries 

    :type:  boolean, default False 
    '''

    use_ridges_and_valleys = None
    '''Enable ridges and valleys 

    :type:  boolean, default False 
    '''

    use_smoothness = None
    '''Take face smoothness into account in view map calculation 

    :type:  boolean, default False 
    '''

    use_suggestive_contours = None
    '''Enable suggestive contours 

    :type:  boolean, default False 
    '''

    use_view_map_cache = None
    '''Keep the computed view map and avoid re-calculating it if mesh geometry is unchanged 

    :type:  boolean, default False 
    '''


class Function:
    '''RNA function definition '''

    description = None
    '''Description of the Function’s purpose 

    :type:  string, default “”, (readonly, never None) 
    '''

    identifier = None
    '''Unique name used in the code and scripting 

    :type:  string, default “”, (readonly, never None) 
    '''

    is_registered = None
    '''Function is registered as callback as part of type registration 

    :type:  boolean, default False, (readonly) 
    '''

    is_registered_optional = None
    '''Function is optionally registered as callback part of type registration 

    :type:  boolean, default False, (readonly) 
    '''

    parameters = None
    '''Parameters for the function 

    :type:  bpy_prop_collection of Property, (readonly) 
    '''

    use_self = None
    '''Function does not pass its self as an argument (becomes a static method in python) 

    :type:  boolean, default False, (readonly) 
    '''

    use_self_type = None
    '''Function passes its self type as an argument (becomes a class method in python if use_self is false) 

    :type:  boolean, default False, (readonly) 
    '''


class GPENCIL_PIE_sculpt:
    def draw(self, context):
        '''

        '''
        pass

    def draw_preset(self, context):
        '''Optionally: - preset_extensions (set of strings) - preset_operator_defaults (dict of keyword args) 

        '''
        pass

    def path_menu(self,
                  searchpaths,
                  operator,
                  props_default=None,
                  filter_ext=None):
        '''

        '''
        pass


class GPENCIL_PIE_settings_palette:
    def draw(self, context):
        '''

        '''
        pass

    def draw_preset(self, context):
        '''Optionally: - preset_extensions (set of strings) - preset_operator_defaults (dict of keyword args) 

        '''
        pass

    def path_menu(self,
                  searchpaths,
                  operator,
                  props_default=None,
                  filter_ext=None):
        '''

        '''
        pass


class GPENCIL_PIE_tool_palette:
    def draw(self, context):
        '''

        '''
        pass

    def draw_preset(self, context):
        '''Optionally: - preset_extensions (set of strings) - preset_operator_defaults (dict of keyword args) 

        '''
        pass

    def path_menu(self,
                  searchpaths,
                  operator,
                  props_default=None,
                  filter_ext=None):
        '''

        '''
        pass


class GPENCIL_PIE_tools_more:
    def draw(self, context):
        '''

        '''
        pass

    def draw_preset(self, context):
        '''Optionally: - preset_extensions (set of strings) - preset_operator_defaults (dict of keyword args) 

        '''
        pass

    def path_menu(self,
                  searchpaths,
                  operator,
                  props_default=None,
                  filter_ext=None):
        '''

        '''
        pass


class GPENCIL_UL_brush:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class GPENCIL_UL_layer:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class GPENCIL_UL_palettecolor:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class GPUDOFSettings:
    '''Settings for GPU based depth of field '''

    blades = None
    '''Blades for dof effect 

    :type:  int in [0, 16], default 0 
    '''

    focal_length = None
    '''Focal length for dof effect 

    :type:  float in [1, inf], default 0.0 
    '''

    focus_distance = None
    '''Viewport depth of field focus distance 

    :type:  float in [0, inf], default 0.0 
    '''

    fstop = None
    '''F-stop for dof effect 

    :type:  float in [0, inf], default 0.0 
    '''

    is_hq_supported = None
    '''Use high quality depth of field 

    :type:  boolean, default False, (readonly) 
    '''

    sensor = None
    '''Size of sensor 

    :type:  float in [1, inf], default 0.0 
    '''

    use_high_quality = None
    '''Use high quality depth of field 

    :type:  boolean, default False 
    '''


class GPUFXSettings:
    '''Settings for GPU based compositing '''

    dof = None
    '''

    :type:  GPUDOFSettings, (readonly, never None) 
    '''

    ssao = None
    '''

    :type:  GPUSSAOSettings, (readonly, never None) 
    '''

    use_dof = None
    '''Use depth of field on viewport using the values from active camera 

    :type:  boolean, default False 
    '''

    use_ssao = None
    '''Use screen space ambient occlusion of field on viewport 

    :type:  boolean, default False 
    '''


class GPUSSAOSettings:
    '''Settings for GPU based screen space ambient occlusion '''

    attenuation = None
    '''Attenuation constant 

    :type:  float in [1, 100000], default 0.0 
    '''

    color = None
    '''Color for screen space ambient occlusion effect 

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    distance_max = None
    '''Distance of object that contribute to the SSAO effect 

    :type:  float in [0, 100000], default 0.0 
    '''

    factor = None
    '''Strength of the SSAO effect 

    :type:  float in [0, 250], default 0.0 
    '''

    samples = None
    '''Number of samples 

    :type:  int in [1, 500], default 0 
    '''


class GPencilBrush:
    '''Collection of brushes being used to control the line style of new strokes '''

    angle = None
    '''Direction of the stroke at which brush gives maximal thickness (0° for horizontal) 

    :type:  float in [-1.5708, 1.5708], default 0.0 
    '''

    angle_factor = None
    '''Reduce brush thickness by this factor when stroke is perpendicular to ‘Angle’ direction 

    :type:  float in [0, 1], default 0.0 
    '''

    curve_jitter = None
    '''Curve used for the jitter effect 

    :type:  CurveMapping, (readonly) 
    '''

    curve_sensitivity = None
    '''Curve used for the sensitivity 

    :type:  CurveMapping, (readonly) 
    '''

    curve_strength = None
    '''Curve used for the strength 

    :type:  CurveMapping, (readonly) 
    '''

    jitter = None
    '''Jitter factor for new strokes 

    :type:  float in [0, 1], default 0.0 
    '''

    line_width = None
    '''Thickness of strokes (in pixels) 

    :type:  int in [1, 300], default 0 
    '''

    name = None
    '''Brush name 

    :type:  string, default “”, (never None) 
    '''

    pen_sensitivity_factor = None
    '''Pressure sensitivity factor for new strokes 

    :type:  float in [0.1, 3], default 0.0 
    '''

    pen_smooth_factor = None
    '''Amount of smoothing to apply to newly created strokes, to reduce jitter/noise 

    :type:  float in [0, 2], default 0.0 
    '''

    pen_smooth_steps = None
    '''Number of times to smooth newly created strokes 

    :type:  int in [1, 3], default 0 
    '''

    pen_subdivision_steps = None
    '''Number of times to subdivide newly created strokes, for less jagged strokes 

    :type:  int in [0, 3], default 0 
    '''

    random_press = None
    '''Randomness factor for pressure and strength in new strokes 

    :type:  float in [0, 1], default 0.0 
    '''

    random_subdiv = None
    '''Randomness factor for new strokes after subdivision 

    :type:  float in [0, 1], default 0.0 
    '''

    strength = None
    '''Color strength for new strokes (affect alpha factor of color) 

    :type:  float in [0, 1], default 0.0 
    '''

    use_jitter_pressure = None
    '''Use tablet pressure for jitter 

    :type:  boolean, default False 
    '''

    use_pressure = None
    '''Use tablet pressure 

    :type:  boolean, default False 
    '''

    use_random_pressure = None
    '''Use random value for pressure 

    :type:  boolean, default False 
    '''

    use_random_strength = None
    '''Use random value for strength 

    :type:  boolean, default False 
    '''

    use_strength_pressure = None
    '''Use tablet pressure for color strength 

    :type:  boolean, default False 
    '''


class GPencilFrame:
    '''Collection of related sketches on a particular frame '''

    frame_number = None
    '''The frame on which this sketch appears 

    :type:  int in [-500000, 500000], default 0 
    '''

    is_edited = None
    '''Frame is being edited (painted on) 

    :type:  boolean, default False 
    '''

    select = None
    '''Frame is selected for editing in the Dope Sheet 

    :type:  boolean, default False 
    '''

    strokes = None
    '''Freehand curves defining the sketch on this frame 

    :type:  GPencilStrokes bpy_prop_collection of GPencilStroke, (readonly) 
    '''

    def clear(self):
        '''Remove all the grease pencil frame data 

        '''
        pass


class GPencilFrames:
    '''Collection of grease pencil frames '''

    def new(self, frame_number):
        '''Add a new grease pencil frame 

        :param frame_number: Frame Number, The frame on which this sketch appears 
        :type frame_number: int in [-500000, 500000]
        :rtype: GPencilFrame 
        :return:  The newly created frame 
        '''
        pass

    def remove(self, frame):
        '''Remove a grease pencil frame 

        :param frame: Frame, The frame to remove 
        :type frame: GPencilFrame, (never None)
        '''
        pass

    def copy(self, source):
        '''Copy a grease pencil frame 

        :param source: Source, The source frame 
        :type source: GPencilFrame, (never None)
        :rtype: GPencilFrame 
        :return:  The newly copied frame 
        '''
        pass


class GPencilLayer:
    '''Collection of related sketches '''

    active_frame = None
    '''Frame currently being displayed for this layer 

    :type:  GPencilFrame, (readonly) 
    '''

    after_color = None
    '''Base color for ghosts after the active frame 

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    before_color = None
    '''Base color for ghosts before the active frame 

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    frames = None
    '''Sketches for this layer on different frames 

    :type:  GPencilFrames bpy_prop_collection of GPencilFrame, (readonly) 
    '''

    ghost_after_range = None
    '''Maximum number of frames to show after current frame (0 = show only the next sketch, -1 = don’t show any frames after current) 

    :type:  int in [-1, 120], default 0 
    '''

    ghost_before_range = None
    '''Maximum number of frames to show before current frame (0 = show only the previous sketch, -1 = don’t show any frames before current) 

    :type:  int in [-1, 120], default 0 
    '''

    hide = None
    '''Set layer Visibility 

    :type:  boolean, default False 
    '''

    info = None
    '''Layer name 

    :type:  string, default “”, (never None) 
    '''

    is_parented = None
    '''True when the layer parent object is set 

    :type:  boolean, default False, (readonly) 
    '''

    line_change = None
    '''Thickness change to apply to current strokes (in pixels) 

    :type:  int in [-32768, 32767], default 0 
    '''

    lock = None
    '''Protect layer from further editing and/or frame changes 

    :type:  boolean, default False 
    '''

    lock_frame = None
    '''Lock current frame displayed by layer 

    :type:  boolean, default False 
    '''

    matrix_inverse = None
    '''Parent inverse transformation matrix 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    opacity = None
    '''Layer Opacity 

    :type:  float in [0, 1], default 0.0 
    '''

    parent = None
    '''Parent Object 

    :type:  Object 
    '''

    parent_bone = None
    '''Name of parent bone in case of a bone parenting relation 

    :type:  string, default “”, (never None) 
    '''

    parent_type = None
    '''Type of parent relation 

    :type:  enum in [‘OBJECT’, ‘ARMATURE’, ‘BONE’], default ‘OBJECT’ 
    '''

    select = None
    '''Layer is selected for editing in the Dope Sheet 

    :type:  boolean, default False 
    '''

    show_points = None
    '''Draw the points which make up the strokes (for debugging purposes) 

    :type:  boolean, default False 
    '''

    show_x_ray = None
    '''Make the layer draw in front of objects 

    :type:  boolean, default False 
    '''

    tint_color = None
    '''Color for tinting stroke colors 

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    tint_factor = None
    '''Factor of tinting color 

    :type:  float in [0, 1], default 0.0 
    '''

    unlock_color = None
    '''Unprotect selected colors from further editing and/or frame changes 

    :type:  boolean, default False 
    '''

    use_ghost_custom_colors = None
    '''Use custom colors for ghost frames 

    :type:  boolean, default False 
    '''

    use_onion_skinning = None
    '''Ghost frames on either side of frame 

    :type:  boolean, default False 
    '''

    use_volumetric_strokes = None
    '''Draw strokes as a series of circular blobs, resulting in a volumetric effect 

    :type:  boolean, default False 
    '''

    def clear(self):
        '''Remove all the grease pencil layer data 

        '''
        pass


class GPencilPalette:
    '''Collection of related palettes '''

    colors = None
    '''Colors of the palette 

    :type:  GPencilPaletteColors bpy_prop_collection of GPencilPaletteColor, (readonly) 
    '''

    name = None
    '''Palette name 

    :type:  string, default “”, (never None) 
    '''


class GPencilPaletteColor:
    '''Collection of related colors '''

    alpha = None
    '''Color Opacity 

    :type:  float in [0, 1], default 0.0 
    '''

    color = None
    '''Color for strokes 

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    fill_alpha = None
    '''Opacity for filling region bounded by each stroke 

    :type:  float in [0, 1], default 0.0 
    '''

    fill_color = None
    '''Color for filling region bounded by each stroke 

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    ghost = None
    '''Display the color in onion skinning 

    :type:  boolean, default False 
    '''

    hide = None
    '''Set color Visibility 

    :type:  boolean, default False 
    '''

    is_fill_visible = None
    '''True when opacity of fill is set high enough to be visible 

    :type:  boolean, default False, (readonly) 
    '''

    is_stroke_visible = None
    '''True when opacity of stroke is set high enough to be visible 

    :type:  boolean, default False, (readonly) 
    '''

    lock = None
    '''Protect color from further editing and/or frame changes 

    :type:  boolean, default False 
    '''

    name = None
    '''Color name 

    :type:  string, default “”, (never None) 
    '''

    use_hq_fill = None
    '''Fill strokes using high quality to avoid glitches (slower fps during animation play) 

    :type:  boolean, default False 
    '''

    use_volumetric_strokes = None
    '''Draw strokes as a series of circular blobs, resulting in a volumetric effect 

    :type:  boolean, default False 
    '''


class GPencilPaletteColors:
    '''Collection of palette colors '''

    active = None
    '''Current active color 

    :type:  GPencilPaletteColor 
    '''

    active_index = None
    '''Index of active palette color 

    :type:  int in [0, inf], default 0 
    '''

    def new(self):
        '''Add a new color to the palette 

        :rtype: GPencilPaletteColor 
        :return:  The newly created color 
        '''
        pass

    def remove(self, color):
        '''Remove a color from the palette 

        :param color: The color to remove 
        :type color: GPencilPaletteColor, (never None)
        '''
        pass


class GPencilSculptBrush:
    '''Stroke editing brush '''

    affect_pressure = None
    '''Affect pressure values as well when smoothing strokes 

    :type:  boolean, default False 
    '''

    direction = None
    '''

    :type:  enum in [‘ADD’, ‘SUBTRACT’], default ‘ADD’ 
    '''

    size = None
    '''Radius of the brush in pixels 

    :type:  int in [1, 500], default 0 
    '''

    strength = None
    '''Brush strength 

    :type:  float in [0.001, 1], default 0.0 
    '''

    use_falloff = None
    '''Strength of brush decays with distance from cursor 

    :type:  boolean, default False 
    '''

    use_pressure_strength = None
    '''Enable tablet pressure sensitivity for strength 

    :type:  boolean, default False 
    '''


class GPencilSculptSettings:
    '''Properties for Grease Pencil stroke sculpting tool '''

    affect_position = None
    '''The brush affects the position of the point 

    :type:  boolean, default False 
    '''

    affect_strength = None
    '''The brush affects the color strength of the point 

    :type:  boolean, default False 
    '''

    affect_thickness = None
    '''The brush affects the thickness of the point 

    :type:  boolean, default False 
    '''

    brush = None
    '''

    :type:  GPencilSculptBrush, (readonly) 
    '''

    selection_alpha = None
    '''Alpha value for selected vertices 

    :type:  float in [0, 1], default 0.0 
    '''

    tool = None
    '''

    :type:  enum in [‘SMOOTH’, ‘THICKNESS’, ‘STRENGTH’, ‘GRAB’, ‘PUSH’, ‘TWIST’, ‘PINCH’, ‘RANDOMIZE’, ‘CLONE’], default ‘SMOOTH’ 
    '''

    use_select_mask = None
    '''Only sculpt selected stroke points 

    :type:  boolean, default False 
    '''


class GPencilStroke:
    '''Freehand curve defining part of a sketch '''

    color = None
    '''Color from palette used in Stroke 

    :type:  GPencilPaletteColor, (readonly) 
    '''

    colorname = None
    '''Palette color name 

    :type:  string, default “”, (never None) 
    '''

    draw_cyclic = None
    '''Enable cyclic drawing, closing the stroke 

    :type:  boolean, default False 
    '''

    draw_mode = None
    '''

    :type:  enum in [‘SCREEN’, ‘3DSPACE’, ‘2DSPACE’, ‘2DIMAGE’], default ‘SCREEN’ 
    '''

    line_width = None
    '''Thickness of stroke (in pixels) 

    :type:  int in [1, 300], default 0 
    '''

    points = None
    '''Stroke data points 

    :type:  GPencilStrokePoints bpy_prop_collection of GPencilStrokePoint, (readonly) 
    '''

    select = None
    '''Stroke is selected for viewport editing 

    :type:  boolean, default False 
    '''

    triangles = None
    '''Triangulation data for HQ fill 

    :type:  bpy_prop_collection of GPencilTriangle, (readonly) 
    '''


class GPencilStrokePoint:
    '''Data point for freehand stroke curve '''

    co = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    pressure = None
    '''Pressure of tablet at point when drawing it 

    :type:  float in [0, 1], default 0.0 
    '''

    select = None
    '''Point is selected for viewport editing 

    :type:  boolean, default False 
    '''

    strength = None
    '''Color intensity (alpha factor) 

    :type:  float in [0, 1], default 0.0 
    '''


class GPencilStrokePoints:
    '''Collection of grease pencil stroke points '''

    def add(self, count=1, pressure=1.0, strength=1.0):
        '''Add a new grease pencil stroke point 

        :param count: Number, Number of points to add to the stroke 
        :type count: int in [0, inf], (optional)
        :param pressure: Pressure, Pressure for newly created points 
        :type pressure: float in [0, 1], (optional)
        :param strength: Strength, Color intensity (alpha factor) for newly created points 
        :type strength: float in [0, 1], (optional)
        '''
        pass

    def pop(self, index=-1):
        '''Remove a grease pencil stroke point 

        :param index: Index, point index 
        :type index: int in [-inf, inf], (optional)
        '''
        pass


class GPencilStrokes:
    '''Collection of grease pencil stroke '''

    def new(self, colorname=""):
        '''Add a new grease pencil stroke 

        :param colorname: Color, Name of the color 
        :type colorname: string, (optional, never None)
        :rtype: GPencilStroke 
        :return:  The newly created stroke 
        '''
        pass

    def remove(self, stroke):
        '''Remove a grease pencil stroke 

        :param stroke: Stroke, The stroke to remove 
        :type stroke: GPencilStroke, (never None)
        '''
        pass


class GPencilTriangle:
    '''Triangulation data for HQ fill '''

    v1 = None
    '''First triangle vertex index 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    v2 = None
    '''Second triangle vertex index 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    v3 = None
    '''Third triangle vertex index 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''


class GameActuator:
    filename = None
    '''The file to use, depending on the mode (e.g. the blend file to load or a destination for saving a screenshot) - use the “//” prefix for a relative path 

    :type:  string, default “”, (never None) 
    '''

    mode = None
    '''

    :type:  enum in [‘START’, ‘RESTART’, ‘QUIT’, ‘SAVECFG’, ‘LOADCFG’, ‘SCREENSHOT’], default ‘START’ 
    '''


class GameBooleanProperty:
    '''Game engine user defined Boolean property '''

    value = None
    '''Property value 

    :type:  boolean, default False 
    '''


class GameFloatProperty:
    '''Game engine user defined floating point number property '''

    value = None
    '''Property value 

    :type:  float in [-10000, 10000], default 0.0 
    '''


class GameIntProperty:
    '''Game engine user defined integer number property '''

    value = None
    '''Property value 

    :type:  int in [-10000, 10000], default 0 
    '''


class GameObjectSettings:
    '''Game engine related settings for the object '''

    actuators = None
    '''Game engine actuators to act on events 

    :type:  bpy_prop_collection of Actuator, (readonly) 
    '''

    angular_velocity_max = None
    '''Clamp angular velocity to this maximum speed, in angle per second 

    :type:  float in [0, 1000], default 0.0 
    '''

    angular_velocity_min = None
    '''Clamp angular velocity to this minimum speed (except when totally still), in angle per second 

    :type:  float in [0, 1000], default 0.0 
    '''

    collision_bounds_type = None
    '''Select the collision shape that better fits the object 

    :type:  enum in [‘BOX’, ‘SPHERE’, ‘CYLINDER’, ‘CONE’, ‘CONVEX_HULL’, ‘TRIANGLE_MESH’, ‘CAPSULE’], default ‘BOX’ 
    '''

    collision_group = None
    '''The collision group of the object 

    :type:  boolean array of 16 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    collision_margin = None
    '''Extra margin around object for collision detection, small amount required for stability 

    :type:  float in [0, 1], default 0.04 
    '''

    collision_mask = None
    '''The groups this object can collide with 

    :type:  boolean array of 16 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    controllers = None
    '''Game engine controllers to process events, connecting sensors to actuators 

    :type:  bpy_prop_collection of Controller, (readonly) 
    '''

    damping = None
    '''General movement damping 

    :type:  float in [0, 1], default 0.04 
    '''

    fall_speed = None
    '''Maximum speed at which the character will fall 

    :type:  float in [0, 1000], default 55.0 
    '''

    form_factor = None
    '''Form factor scales the inertia tensor 

    :type:  float in [0, 1], default 0.4 
    '''

    friction_coefficients = None
    '''Relative friction coefficients in the in the X, Y and Z directions, when anisotropic friction is enabled 

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    jump_max = None
    '''The maximum number of jumps the character can make before it hits the ground 

    :type:  int in [1, 255], default 1 
    '''

    jump_speed = None
    '''Upward velocity applied to the character when jumping 

    :type:  float in [0, 1000], default 10.0 
    '''

    lock_location_x = None
    '''Disable simulation of linear motion along the X axis 

    :type:  boolean, default False 
    '''

    lock_location_y = None
    '''Disable simulation of linear motion along the Y axis 

    :type:  boolean, default False 
    '''

    lock_location_z = None
    '''Disable simulation of linear motion along the Z axis 

    :type:  boolean, default False 
    '''

    lock_rotation_x = None
    '''Disable simulation of angular motion along the X axis 

    :type:  boolean, default False 
    '''

    lock_rotation_y = None
    '''Disable simulation of angular motion along the Y axis 

    :type:  boolean, default False 
    '''

    lock_rotation_z = None
    '''Disable simulation of angular motion along the Z axis 

    :type:  boolean, default False 
    '''

    mass = None
    '''Mass of the object 

    :type:  float in [0.01, 10000], default 1.0 
    '''

    obstacle_radius = None
    '''Radius of object representation in obstacle simulation 

    :type:  float in [0, 1000], default 1.0 
    '''

    physics_type = None
    '''Select the type of physical representation 

    :type:  enum in [‘NO_COLLISION’, ‘STATIC’, ‘DYNAMIC’, ‘RIGID_BODY’, ‘SOFT_BODY’, ‘OCCLUDER’, ‘SENSOR’, ‘NAVMESH’, ‘CHARACTER’], default ‘STATIC’ 
    '''

    properties = None
    '''Game engine properties 

    :type:  bpy_prop_collection of GameProperty, (readonly) 
    '''

    radius = None
    '''Radius of bounding sphere and material physics 

    :type:  float in [0.01, inf], default 1.0 
    '''

    rotation_damping = None
    '''General rotation damping 

    :type:  float in [0, 1], default 0.1 
    '''

    sensors = None
    '''Game engine sensor to detect events 

    :type:  bpy_prop_collection of Sensor, (readonly) 
    '''

    show_actuators = None
    '''Shows actuators for this object in the user interface 

    :type:  boolean, default False 
    '''

    show_controllers = None
    '''Shows controllers for this object in the user interface 

    :type:  boolean, default False 
    '''

    show_debug_state = None
    '''Print state debug info in the game engine 

    :type:  boolean, default False 
    '''

    show_sensors = None
    '''Shows sensors for this object in the user interface 

    :type:  boolean, default False 
    '''

    show_state_panel = None
    '''Show state panel 

    :type:  boolean, default False 
    '''

    soft_body = None
    '''Settings for Bullet soft body simulation 

    :type:  GameSoftBodySettings, (readonly) 
    '''

    states_initial = None
    '''Initial state when the game starts 

    :type:  boolean array of 30 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    states_visible = None
    '''State determining which controllers are displayed 

    :type:  boolean array of 30 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    step_height = None
    '''Maximum height of steps the character can run over 

    :type:  float in [0.01, 1], default 0.15 
    '''

    use_activity_culling = None
    '''Disable simulation of angular motion along the Z axis 

    :type:  boolean, default False 
    '''

    use_actor = None
    '''Object is detected by the Near and Radar sensor 

    :type:  boolean, default False 
    '''

    use_all_states = None
    '''Set all state bits 

    :type:  boolean, default False 
    '''

    use_anisotropic_friction = None
    '''Enable anisotropic friction 

    :type:  boolean, default False 
    '''

    use_collision_bounds = None
    '''Specify a collision bounds type other than the default 

    :type:  boolean, default False 
    '''

    use_collision_compound = None
    '''Add children to form a compound collision object 

    :type:  boolean, default False 
    '''

    use_ghost = None
    '''Object does not react to collisions, like a ghost 

    :type:  boolean, default False 
    '''

    use_material_physics_fh = None
    '''React to force field physics settings in materials 

    :type:  boolean, default False 
    '''

    use_obstacle_create = None
    '''Create representation for obstacle simulation 

    :type:  boolean, default False 
    '''

    use_record_animation = None
    '''Record animation objects without physics 

    :type:  boolean, default False 
    '''

    use_rotate_from_normal = None
    '''Use face normal to rotate object, so that it points away from the surface 

    :type:  boolean, default False 
    '''

    use_sleep = None
    '''Disable auto (de)activation in physics simulation 

    :type:  boolean, default False 
    '''

    used_states = None
    '''States which are being used by controllers 

    :type:  boolean array of 30 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False), (readonly) 
    '''

    velocity_max = None
    '''Clamp velocity to this maximum speed, in distance per second 

    :type:  float in [0, 1000], default 0.0 
    '''

    velocity_min = None
    '''Clamp velocity to this minimum speed (except when totally still), in distance per second 

    :type:  float in [0, 1000], default 0.0 
    '''


class GameProperty:
    '''Game engine user defined object property '''

    name = None
    '''Available as GameObject attributes in the game engine’s python API 

    :type:  string, default “”, (never None) 
    '''

    show_debug = None
    '''Print debug information for this property 

    :type:  boolean, default False 
    '''

    type = None
    '''

    :type:  enum in [‘BOOL’, ‘INT’, ‘FLOAT’, ‘STRING’, ‘TIMER’], default ‘BOOL’ 
    '''


class GameSoftBodySettings:
    '''Soft body simulation settings for an object in the game engine '''

    cluster_iterations = None
    '''Number of cluster iterations 

    :type:  int in [1, 128], default 0 
    '''

    collision_margin = None
    '''Collision margin for soft body. Small value makes the algorithm unstable 

    :type:  float in [0.01, 1], default 0.0 
    '''

    dynamic_friction = None
    '''Dynamic Friction 

    :type:  float in [0, 1], default 0.0 
    '''

    linear_stiffness = None
    '''Linear stiffness of the soft body links 

    :type:  float in [0, 1], default 0.0 
    '''

    location_iterations = None
    '''Position solver iterations 

    :type:  int in [0, 10], default 0 
    '''

    shape_threshold = None
    '''Shape matching threshold 

    :type:  float in [0, 1], default 0.0 
    '''

    use_bending_constraints = None
    '''Enable bending constraints 

    :type:  boolean, default False 
    '''

    use_cluster_rigid_to_softbody = None
    '''Enable cluster collision between soft and rigid body 

    :type:  boolean, default False 
    '''

    use_cluster_soft_to_softbody = None
    '''Enable cluster collision between soft and soft body 

    :type:  boolean, default False 
    '''

    use_shape_match = None
    '''Enable soft body shape matching goal 

    :type:  boolean, default False 
    '''

    weld_threshold = None
    '''Welding threshold: distance between nearby vertices to be considered equal => set to 0.0 to disable welding test and speed up scene loading (ok if the mesh has no duplicates) 

    :type:  float in [0, 0.01], default 0.0 
    '''


class GameStringProperty:
    '''Game engine user defined text string property '''

    value = None
    '''Property value 

    :type:  string, default “”, (never None) 
    '''


class GameTimerProperty:
    '''Game engine user defined timer property '''

    value = None
    '''Property value 

    :type:  float in [-10000, 10000], default 0.0 
    '''


class GammaCrossSequence:
    '''Gamma Cross Sequence '''

    input_1 = None
    '''First input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_2 = None
    '''Second input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''


class GaussianBlurSequence:
    '''Sequence strip creating a gaussian blur '''

    input_1 = None
    '''First input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    size_x = None
    '''Size of the blur along X axis 

    :type:  float in [0, inf], default 0.0 
    '''

    size_y = None
    '''Size of the blur along Y axis 

    :type:  float in [0, inf], default 0.0 
    '''


class GlowSequence:
    '''Sequence strip creating a glow effect '''

    blur_radius = None
    '''Radius of glow effect 

    :type:  float in [0.5, 20], default 0.0 
    '''

    boost_factor = None
    '''Brightness multiplier 

    :type:  float in [0, 10], default 0.0 
    '''

    clamp = None
    '''Brightness limit of intensity 

    :type:  float in [0, 1], default 0.0 
    '''

    input_1 = None
    '''First input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    quality = None
    '''Accuracy of the blur effect 

    :type:  int in [1, 5], default 0 
    '''

    threshold = None
    '''Minimum intensity to trigger a glow 

    :type:  float in [0, 1], default 0.0 
    '''

    use_only_boost = None
    '''Show the glow buffer only 

    :type:  boolean, default False 
    '''


class GreasePencil:
    '''Freehand annotation sketchbook '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    layers = None
    '''

    :type:  GreasePencilLayers bpy_prop_collection of GPencilLayer, (readonly) 
    '''

    palettes = None
    '''

    :type:  GreasePencilPalettes bpy_prop_collection of GPencilPalette, (readonly) 
    '''

    show_stroke_direction = None
    '''Show stroke drawing direction with a bigger green dot (start) and smaller red dot (end) points 

    :type:  boolean, default False 
    '''

    use_onion_skinning = None
    '''Show ghosts of the frames before and after the current frame, toggle to enable on active layer or disable all 

    :type:  boolean, default False 
    '''

    use_stroke_edit_mode = None
    '''Edit Grease Pencil strokes instead of viewport data 

    :type:  boolean, default False 
    '''

    def clear(self):
        '''Remove all the grease pencil data 

        '''
        pass


class GreasePencilBrushes:
    '''Collection of grease pencil brushes '''

    active = None
    '''Current active brush 

    :type:  GPencilBrush 
    '''

    active_index = None
    '''Index of active brush 

    :type:  int in [0, inf], default 0 
    '''

    def new(self, name, set_active=False):
        '''Add a new grease pencil brush 

        :param name: Name, Name of the brush 
        :type name: string, (never None)
        :param set_active: Set Active, Set the newly created brush to the active brush 
        :type set_active: boolean, (optional)
        :rtype: GPencilBrush 
        :return:  The newly created brush 
        '''
        pass

    def remove(self, brush):
        '''Remove a grease pencil brush 

        :param brush: The brush to remove 
        :type brush: GPencilBrush, (never None)
        '''
        pass


class GreasePencilLayers:
    '''Collection of grease pencil layers '''

    active = None
    '''Active grease pencil layer 

    :type:  GPencilLayer 
    '''

    active_index = None
    '''Index of active grease pencil layer 

    :type:  int in [0, inf], default 0 
    '''

    def new(self, name, set_active=False):
        '''Add a new grease pencil layer 

        :param name: Name, Name of the layer 
        :type name: string, (never None)
        :param set_active: Set Active, Set the newly created layer to the active layer 
        :type set_active: boolean, (optional)
        :rtype: GPencilLayer 
        :return:  The newly created layer 
        '''
        pass

    def remove(self, layer):
        '''Remove a grease pencil layer 

        :param layer: The layer to remove 
        :type layer: GPencilLayer, (never None)
        '''
        pass


class GreasePencilPalettes:
    '''Collection of grease pencil palettes '''

    active = None
    '''Current active palette 

    :type:  GPencilPalette 
    '''

    active_index = None
    '''Index of active palette 

    :type:  int in [0, inf], default 0 
    '''

    def new(self, name, set_active=False):
        '''Add a new grease pencil palette 

        :param name: Name, Name of the palette 
        :type name: string, (never None)
        :param set_active: Set Active, Activate the newly created palette 
        :type set_active: boolean, (optional)
        :rtype: GPencilPalette 
        :return:  The newly created palette 
        '''
        pass

    def remove(self, palette):
        '''Remove a grease pencil palette 

        :param palette: The palette to remove 
        :type palette: GPencilPalette, (never None)
        '''
        pass


class Group:
    '''Group of Object data-blocks '''

    dupli_offset = None
    '''Offset from the origin to use when instancing as DupliGroup 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    layers = None
    '''Layers visible when this group is instanced as a dupli 

    :type:  boolean array of 20 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    objects = None
    '''A collection of this groups objects 

    :type:  GroupObjects bpy_prop_collection of Object, (readonly) 
    '''

    users_dupli_group = None
    '''The dupli group this group is used in (readonly) '''


class GroupObjects:
    '''Collection of group objects '''

    def link(self, object):
        '''Add this object to a group 

        :param object: Object to add 
        :type object: Object, (never None)
        '''
        pass

    def unlink(self, object):
        '''Remove this object to a group 

        :param object: Object to remove 
        :type object: Object
        '''
        pass


class Header:
    '''Editor header containing UI elements '''

    bl_idname = None
    '''If this is set, the header gets a custom ID, otherwise it takes the name of the class used to define the panel; for example, if the class name is “OBJECT_HT_hello”, and bl_idname is not set by the script, then bl_idname = “OBJECT_HT_hello” 

    :type:  string, default “”, (never None) 
    '''

    bl_space_type = None
    '''The space where the header is going to be used in 

    :type:  enum in [‘EMPTY’, ‘VIEW_3D’, ‘TIMELINE’, ‘GRAPH_EDITOR’, ‘DOPESHEET_EDITOR’, ‘NLA_EDITOR’, ‘IMAGE_EDITOR’, ‘SEQUENCE_EDITOR’, ‘CLIP_EDITOR’, ‘TEXT_EDITOR’, ‘NODE_EDITOR’, ‘LOGIC_EDITOR’, ‘PROPERTIES’, ‘OUTLINER’, ‘USER_PREFERENCES’, ‘INFO’, ‘FILE_BROWSER’, ‘CONSOLE’], default ‘EMPTY’ 
    '''

    layout = None
    '''Structure of the header in the UI 

    :type:  UILayout, (readonly) 
    '''

    def draw(self, context):
        '''Draw UI elements into the header UI layout 

        '''
        pass


class HemiLamp:
    '''180 degree constant lamp '''

    pass


class Histogram:
    '''Statistical view of the levels of color in an image '''

    mode = None
    '''Channels to display when drawing the histogram 

    :type:  enum in [‘LUMA’, ‘RGB’, ‘R’, ‘G’, ‘B’, ‘A’], default ‘LUMA’ 
    '''

    show_line = None
    '''Display lines rather than filled shapes 

    :type:  boolean, default False 
    '''


class HookModifier:
    '''Hook modifier to modify the location of vertices '''

    center = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    falloff_curve = None
    '''Custom Lamp Falloff Curve 

    :type:  CurveMapping, (readonly) 
    '''

    falloff_radius = None
    '''If not zero, the distance from the hook where influence ends 

    :type:  float in [0, inf], default 0.0 
    '''

    falloff_type = None
    '''

    :type:  enum in [‘NONE’, ‘CURVE’, ‘SMOOTH’, ‘SPHERE’, ‘ROOT’, ‘INVERSE_SQUARE’, ‘SHARP’, ‘LINEAR’, ‘CONSTANT’], default ‘NONE’ 
    '''

    matrix_inverse = None
    '''Reverse the transformation between this object and its target 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    object = None
    '''Parent Object for hook, also recalculates and clears offset 

    :type:  Object 
    '''

    strength = None
    '''Relative force of the hook 

    :type:  float in [0, 1], default 0.0 
    '''

    subtarget = None
    '''Name of Parent Bone for hook (if applicable), also recalculates and clears offset 

    :type:  string, default “”, (never None) 
    '''

    use_falloff_uniform = None
    '''Compensate for non-uniform object scale 

    :type:  boolean, default False 
    '''

    vertex_group = None
    '''Name of Vertex Group which determines influence of modifier per point 

    :type:  string, default “”, (never None) 
    '''


class HueCorrectModifier:
    '''Hue correction modifier for sequence strip '''

    curve_mapping = None
    '''

    :type:  CurveMapping, (readonly) 
    '''


class ID:
    '''Base type for data-blocks, defining a unique name, linking from other libraries and garbage collection '''

    is_library_indirect = None
    '''Is this ID block linked indirectly 

    :type:  boolean, default False, (readonly) 
    '''

    is_updated = None
    '''Datablock is tagged for recalculation 

    :type:  boolean, default False, (readonly) 
    '''

    is_updated_data = None
    '''Datablock data is tagged for recalculation 

    :type:  boolean, default False, (readonly) 
    '''

    library = None
    '''Library file the data-block is linked from 

    :type:  Library, (readonly) 
    '''

    name = None
    '''Unique data-block ID name 

    :type:  string, default “”, (never None) 
    '''

    preview = None
    '''Preview image and icon of this data-block (None if not supported for this type of data) 

    :type:  ImagePreview, (readonly) 
    '''

    tag = None
    '''Tools can use this to tag data for their own purposes (initial state is undefined) 

    :type:  boolean, default False 
    '''

    use_fake_user = None
    '''Save this data-block even if it has no users 

    :type:  boolean, default False 
    '''

    users = None
    '''Number of times this data-block is referenced 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    def copy(self):
        '''Create a copy of this data-block (not supported for all data-blocks) 

        :rtype: ID 
        :return:  New copy of the ID 
        '''
        pass

    def user_clear(self):
        '''This function is for advanced use only, misuse can crash blender since the user count is used to prevent data being removed when it is used. 

        '''
        pass

    def user_remap(self, new_id):
        '''Replace all usage in the .blend file of this ID by new given one 

        :param new_id: New ID to use 
        :type new_id: ID, (never None)
        '''
        pass

    def user_of_id(self, id):
        '''Count the number of times that ID uses/references given one 

        :param id: ID to count usages 
        :type id: ID, (never None)
        :rtype: int in [0, inf] 
        :return:  Number of usages/references of given id by current datablock 
        '''
        pass

    def animation_data_create(self):
        '''Create animation data to this ID, note that not all ID types support this 

        :rtype: AnimData 
        :return:  New animation data or NULL 
        '''
        pass

    def animation_data_clear(self):
        '''Clear animation on this this ID 

        '''
        pass

    def update_tag(self, refresh={}):
        '''Tag the ID to update its display data, e.g. when calling bpy.types.Scene.update 

        :param refresh: Type of updates to perform 
        :type refresh: enum set in {'OBJECT', 'DATA', 'TIME'}, (optional)
        '''
        pass


class IDMaterials:
    '''Collection of materials '''

    def append(self, material):
        '''Add a new material to the data block 

        :param material: Material to add 
        :type material: Material
        '''
        pass

    def pop(self, index=-1, update_data=False):
        '''Remove a material from the data block 

        :param index: Index of material to remove 
        :type index: int in [-32766, 32766], (optional)
        :param update_data: Update data by re-adjusting the material slots assigned 
        :type update_data: boolean, (optional)
        :rtype: Material 
        :return:  Material to remove 
        '''
        pass

    def clear(self, update_data=False):
        '''Remove all materials from the data block 

        :param update_data: Update data by re-adjusting the material slots assigned 
        :type update_data: boolean, (optional)
        '''
        pass


class IKParam:
    '''Base type for IK solver parameters '''

    ik_solver = None
    '''IK solver for which these parameters are defined 

    :type:  enum in [‘LEGACY’, ‘ITASC’], default ‘LEGACY’, (readonly) 
    '''


class IMAGE_UV_sculpt:
    def draw(self, context):
        '''

        '''
        pass


class IMAGE_UV_sculpt_curve:
    def draw(self, context):
        '''

        '''
        pass


class Image:
    '''Image data-block referencing an external or packed image '''

    alpha_mode = None
    '''Representation of alpha information in the RGBA pixels 

    :type:  enum in [‘STRAIGHT’, ‘PREMUL’], default ‘STRAIGHT’ 
    '''

    bindcode = None
    '''OpenGL bindcode 

    :type:  int array of 2 items in [0, inf], default (0, 0), (readonly) 
    '''

    channels = None
    '''Number of channels in pixels buffer 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    colorspace_settings = None
    '''Input color space settings 

    :type:  ColorManagedInputColorspaceSettings, (readonly) 
    '''

    depth = None
    '''Image bit depth 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    display_aspect = None
    '''Display Aspect for this image, does not affect rendering 

    :type:  float array of 2 items in [0.1, inf], default (0.0, 0.0) 
    '''

    field_order = None
    '''Order of video fields (select which lines are displayed first) 

    :type:  enum in [‘EVEN’, ‘ODD’], default ‘EVEN’ 
    '''

    file_format = None
    '''Format used for re-saving this file 

    :type:  enum in [‘BMP’, ‘IRIS’, ‘PNG’, ‘JPEG’, ‘JPEG2000’, ‘TARGA’, ‘TARGA_RAW’, ‘CINEON’, ‘DPX’, ‘OPEN_EXR_MULTILAYER’, ‘OPEN_EXR’, ‘HDR’, ‘TIFF’, ‘AVI_JPEG’, ‘AVI_RAW’, ‘FRAMESERVER’, ‘H264’, ‘FFMPEG’, ‘THEORA’, ‘XVID’], default ‘TARGA’ 
    '''

    filepath = None
    '''Image/Movie file name 

    :type:  string, default “”, (never None) 
    '''

    filepath_raw = None
    '''Image/Movie file name (without data refreshing) 

    :type:  string, default “”, (never None) 
    '''

    fps = None
    '''Speed of the animation in frames per second 

    :type:  int in [1, 100], default 0 
    '''

    frame_duration = None
    '''Duration (in frames) of the image (1 when not a video/sequence) 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    frame_end = None
    '''End frame of an animated texture 

    :type:  int in [0, 255], default 0 
    '''

    frame_start = None
    '''Start frame of an animated texture 

    :type:  int in [0, 255], default 0 
    '''

    generated_color = None
    '''Fill color for the generated image 

    :type:  float array of 4 items in [0, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''

    generated_height = None
    '''Generated image height 

    :type:  int in [1, 65536], default 0 
    '''

    generated_type = None
    '''Generated image type 

    :type:  enum in [‘BLANK’, ‘UV_GRID’, ‘COLOR_GRID’], default ‘BLANK’ 
    '''

    generated_width = None
    '''Generated image width 

    :type:  int in [1, 65536], default 0 
    '''

    has_data = None
    '''True if this image has data 

    :type:  boolean, default False, (readonly) 
    '''

    is_dirty = None
    '''Image has changed and is not saved 

    :type:  boolean, default False, (readonly) 
    '''

    is_float = None
    '''True if this image is stored in float buffer 

    :type:  boolean, default False, (readonly) 
    '''

    is_multiview = None
    '''Image has more than one view 

    :type:  boolean, default False, (readonly) 
    '''

    is_stereo_3d = None
    '''Image has left and right views 

    :type:  boolean, default False, (readonly) 
    '''

    mapping = None
    '''Mapping type to use for this image in the game engine 

    :type:  enum in [‘UV’, ‘REFLECTION’], default ‘UV’ 
    '''

    packed_file = None
    '''First packed file of the image 

    :type:  PackedFile, (readonly) 
    '''

    packed_files = None
    '''Collection of packed images 

    :type:  bpy_prop_collection of ImagePackedFile, (readonly) 
    '''

    pixels = None
    '''Image pixels in floating point values 

    :type:  float in [-inf, inf], default 0.0 
    '''

    render_slots = None
    '''Render slots of the image 

    :type:  RenderSlots bpy_prop_collection of RenderSlot, (readonly) 
    '''

    resolution = None
    '''X/Y pixels per meter 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    size = None
    '''Width and height in pixels, zero when image data cant be loaded 

    :type:  int array of 2 items in [-inf, inf], default (0, 0), (readonly) 
    '''

    source = None
    '''Where the image comes from 

    :type:  enum in [‘FILE’, ‘SEQUENCE’, ‘MOVIE’, ‘GENERATED’, ‘VIEWER’], default ‘FILE’ 
    '''

    stereo_3d_format = None
    '''Settings for stereo 3d 

    :type:  Stereo3dFormat, (readonly, never None) 
    '''

    tiles_x = None
    '''Degree of repetition in the X direction 

    :type:  int in [1, 16], default 0 
    '''

    tiles_y = None
    '''Degree of repetition in the Y direction 

    :type:  int in [1, 16], default 0 
    '''

    type = None
    '''How to generate the image 

    :type:  enum in [‘IMAGE’, ‘MULTILAYER’, ‘UV_TEST’, ‘RENDER_RESULT’, ‘COMPOSITING’], default ‘IMAGE’, (readonly) 
    '''

    use_alpha = None
    '''Use the alpha channel information from the image or make image fully opaque 

    :type:  boolean, default False 
    '''

    use_animation = None
    '''Use as animated texture in the game engine 

    :type:  boolean, default False 
    '''

    use_clamp_x = None
    '''Disable texture repeating horizontally 

    :type:  boolean, default False 
    '''

    use_clamp_y = None
    '''Disable texture repeating vertically 

    :type:  boolean, default False 
    '''

    use_deinterlace = None
    '''Deinterlace movie file on load 

    :type:  boolean, default False 
    '''

    use_fields = None
    '''Use fields of the image 

    :type:  boolean, default False 
    '''

    use_generated_float = None
    '''Generate floating point buffer 

    :type:  boolean, default False 
    '''

    use_multiview = None
    '''Use Multiple Views (when available) 

    :type:  boolean, default False 
    '''

    use_tiles = None
    '''Use of tilemode for faces (default shift-LMB to pick the tile for selected faces) 

    :type:  boolean, default False 
    '''

    use_view_as_render = None
    '''Apply render part of display transformation when displaying this image on the screen 

    :type:  boolean, default False 
    '''

    views_format = None
    '''Mode to load image views 

    :type:  enum in [‘INDIVIDUAL’, ‘STEREO_3D’], default ‘INDIVIDUAL’ 
    '''

    def save_render(self, filepath, scene=None):
        '''Save image to a specific path using a scenes render settings 

        :param filepath: Save path 
        :type filepath: string, (never None)
        :param scene: Scene to take image parameters from 
        :type scene: Scene, (optional)
        '''
        pass

    def save(self):
        '''Save image to its source path 

        '''
        pass

    def pack(self, as_png=False, data="", data_len=0):
        '''Pack an image as embedded data into the .blend file 

        :param as_png: as_png, Pack the image as PNG (needed for generated/dirty images) 
        :type as_png: boolean, (optional)
        :param data: data, Raw data (bytes, exact content of the embedded file) 
        :type data: string, (optional, never None)
        :param data_len: data_len, length of given data (mandatory if data is provided) 
        :type data_len: int in [0, inf], (optional)
        '''
        pass

    def unpack(self, method='USE_LOCAL'):
        '''Save an image packed in the .blend file to disk 

        :param method: method, How to unpack 
        :type method: enum in ['USE_LOCAL', 'WRITE_LOCAL', 'USE_ORIGINAL', 'WRITE_ORIGINAL'], (optional)
        '''
        pass

    def reload(self):
        '''Reload the image from its source path 

        '''
        pass

    def update(self):
        '''Update the display image from the floating point buffer 

        '''
        pass

    def scale(self, width, height):
        '''Scale the image in pixels 

        :param width: Width 
        :type width: int in [1, 10000]
        :param height: Height 
        :type height: int in [1, 10000]
        '''
        pass

    def gl_touch(self, frame=0, filter=9985, mag=9729):
        '''Delay the image from being cleaned from the cache due inactivity 

        :param frame: Frame, Frame of image sequence or movie 
        :type frame: int in [0, inf], (optional)
        :param filter: Filter, The texture minifying function to use if the image wasn’t loaded 
        :type filter: int in [-inf, inf], (optional)
        :param mag: Magnification, The texture magnification function to use if the image wasn’t loaded 
        :type mag: int in [-inf, inf], (optional)
        :rtype: int in [-inf, inf] 
        :return:  Error, OpenGL error value 
        '''
        pass

    def gl_load(self, frame=0, filter=9985, mag=9729):
        '''Load the image into OpenGL graphics memory 

        :param frame: Frame, Frame of image sequence or movie 
        :type frame: int in [0, inf], (optional)
        :param filter: Filter, The texture minifying function 
        :type filter: int in [-inf, inf], (optional)
        :param mag: Magnification, The texture magnification function 
        :type mag: int in [-inf, inf], (optional)
        :rtype: int in [-inf, inf] 
        :return:  Error, OpenGL error value 
        '''
        pass

    def gl_free(self):
        '''Free the image from OpenGL graphics memory 

        '''
        pass

    def filepath_from_user(self, image_user=None):
        '''Return the absolute path to the filepath of an image frame specified by the image user 

        :param image_user: Image user of the image to get filepath for 
        :type image_user: ImageUser, (optional)
        :rtype: string, (never None) 
        :return:  File Path, The resulting filepath from the image and it’s user 
        '''
        pass

    def buffers_free(self):
        '''Free the image buffers from memory 

        '''
        pass


class ImageFormatSettings:
    '''Settings for image formats '''

    cineon_black = None
    '''Log conversion reference blackpoint 

    :type:  int in [0, 1024], default 0 
    '''

    cineon_gamma = None
    '''Log conversion gamma 

    :type:  float in [0, 10], default 0.0 
    '''

    cineon_white = None
    '''Log conversion reference whitepoint 

    :type:  int in [0, 1024], default 0 
    '''

    color_depth = None
    '''Bit depth per channel 

    :type:  enum in [‘8’, ‘10’, ‘12’, ‘16’, ‘32’], default ‘8’ 
    '''

    color_mode = None
    '''Choose BW for saving grayscale images, RGB for saving red, green and blue channels, and RGBA for saving red, green, blue and alpha channels 

    :type:  enum in [‘BW’, ‘RGB’, ‘RGBA’], default ‘BW’ 
    '''

    compression = None
    '''Amount of time to determine best compression: 0 = no compression with fast file output, 100 = maximum lossless compression with slow file output 

    :type:  int in [0, 100], default 0 
    '''

    display_settings = None
    '''Settings of device saved image would be displayed on 

    :type:  ColorManagedDisplaySettings, (readonly) 
    '''

    exr_codec = None
    '''Codec settings for OpenEXR 

    :type:  enum in [‘NONE’, ‘PXR24’, ‘ZIP’, ‘PIZ’, ‘RLE’, ‘ZIPS’, ‘B44’, ‘B44A’, ‘DWAA’, ‘DWAB’], default ‘NONE’ 
    '''

    file_format = None
    '''File format to save the rendered images as 

    :type:  enum in [‘BMP’, ‘IRIS’, ‘PNG’, ‘JPEG’, ‘JPEG2000’, ‘TARGA’, ‘TARGA_RAW’, ‘CINEON’, ‘DPX’, ‘OPEN_EXR_MULTILAYER’, ‘OPEN_EXR’, ‘HDR’, ‘TIFF’, ‘AVI_JPEG’, ‘AVI_RAW’, ‘FRAMESERVER’, ‘H264’, ‘FFMPEG’, ‘THEORA’, ‘XVID’], default ‘TARGA’ 
    '''

    jpeg2k_codec = None
    '''Codec settings for Jpek2000 

    :type:  enum in [‘JP2’, ‘J2K’], default ‘JP2’ 
    '''

    quality = None
    '''Quality for image formats that support lossy compression 

    :type:  int in [0, 100], default 0 
    '''

    stereo_3d_format = None
    '''Settings for stereo 3d 

    :type:  Stereo3dFormat, (readonly, never None) 
    '''

    tiff_codec = None
    '''Compression mode for TIFF 

    :type:  enum in [‘NONE’, ‘DEFLATE’, ‘LZW’, ‘PACKBITS’], default ‘DEFLATE’ 
    '''

    use_cineon_log = None
    '''Convert to logarithmic color space 

    :type:  boolean, default False 
    '''

    use_jpeg2k_cinema_48 = None
    '''Use Openjpeg Cinema Preset (48fps) 

    :type:  boolean, default False 
    '''

    use_jpeg2k_cinema_preset = None
    '''Use Openjpeg Cinema Preset 

    :type:  boolean, default False 
    '''

    use_jpeg2k_ycc = None
    '''Save luminance-chrominance-chrominance channels instead of RGB colors 

    :type:  boolean, default False 
    '''

    use_preview = None
    '''When rendering animations, save JPG preview images in same directory 

    :type:  boolean, default False 
    '''

    use_zbuffer = None
    '''Save the z-depth per pixel (32 bit unsigned int z-buffer) 

    :type:  boolean, default False 
    '''

    view_settings = None
    '''Color management settings applied on image before saving 

    :type:  ColorManagedViewSettings, (readonly) 
    '''

    views_format = None
    '''Format of multiview media 

    :type:  enum in [‘INDIVIDUAL’, ‘STEREO_3D’], default ‘INDIVIDUAL’ 
    '''


class ImagePackedFile:
    filepath = None
    '''

    :type:  string, default “”, (never None) 
    '''

    packed_file = None
    '''

    :type:  PackedFile, (readonly) 
    '''

    def save(self):
        '''Save the packed file to its filepath 

        '''
        pass


class ImagePaint:
    '''Properties of image and texture painting mode '''

    canvas = None
    '''Image used as canvas 

    :type:  Image 
    '''

    clone_image = None
    '''Image used as clone source 

    :type:  Image 
    '''

    dither = None
    '''Amount of dithering when painting on byte images 

    :type:  float in [0, 2], default 0.0 
    '''

    invert_stencil = None
    '''Invert the stencil layer 

    :type:  boolean, default False 
    '''

    missing_materials = None
    '''The mesh is missing materials 

    :type:  boolean, default False, (readonly) 
    '''

    missing_stencil = None
    '''Image Painting does not have a stencil 

    :type:  boolean, default False, (readonly) 
    '''

    missing_texture = None
    '''Image Painting does not have a texture to paint on 

    :type:  boolean, default False, (readonly) 
    '''

    missing_uvs = None
    '''A UV layer is missing on the mesh 

    :type:  boolean, default False, (readonly) 
    '''

    mode = None
    '''Mode of operation for projection painting 

    :type:  enum in [‘MATERIAL’, ‘IMAGE’], default ‘MATERIAL’ 
    '''

    normal_angle = None
    '''Paint most on faces pointing towards the view according to this angle 

    :type:  int in [0, 90], default 0 
    '''

    screen_grab_size = None
    '''Size to capture the image for re-projecting 

    :type:  int array of 2 items in [512, 16384], default (0, 0) 
    '''

    seam_bleed = None
    '''Extend paint beyond the faces UVs to reduce seams (in pixels, slower) 

    :type:  int in [-32768, 32767], default 0 
    '''

    stencil_color = None
    '''Stencil color in the viewport 

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    stencil_image = None
    '''Image used as stencil 

    :type:  Image 
    '''

    use_backface_culling = None
    '''Ignore faces pointing away from the view (faster) 

    :type:  boolean, default False 
    '''

    use_clone_layer = None
    '''Use another UV map as clone source, otherwise use the 3D cursor as the source 

    :type:  boolean, default False 
    '''

    use_normal_falloff = None
    '''Paint most on faces pointing towards the view 

    :type:  boolean, default False 
    '''

    use_occlude = None
    '''Only paint onto the faces directly under the brush (slower) 

    :type:  boolean, default False 
    '''

    use_stencil_layer = None
    '''Set the mask layer from the UV map buttons 

    :type:  boolean, default False 
    '''

    def detect_data(self):
        '''Check if required texpaint data exist 

        :rtype: boolean 
        '''
        pass


class ImagePreview:
    '''Preview image and icon '''

    icon_id = None
    '''Unique integer identifying this preview as an icon (zero means invalid) 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    icon_pixels = None
    '''Icon pixels, as bytes (always RGBA 32bits) 

    :type:  int in [-inf, inf], default 0 
    '''

    icon_pixels_float = None
    '''Icon pixels components, as floats (RGBA concatenated values) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    icon_size = None
    '''Width and height in pixels 

    :type:  int array of 2 items in [-inf, inf], default (0, 0) 
    '''

    image_pixels = None
    '''Image pixels, as bytes (always RGBA 32bits) 

    :type:  int in [-inf, inf], default 0 
    '''

    image_pixels_float = None
    '''Image pixels components, as floats (RGBA concatenated values) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    image_size = None
    '''Width and height in pixels 

    :type:  int array of 2 items in [-inf, inf], default (0, 0) 
    '''

    is_icon_custom = None
    '''True if this preview icon has been modified by py script,and is no more auto-generated by Blender 

    :type:  boolean, default False 
    '''

    is_image_custom = None
    '''True if this preview image has been modified by py script,and is no more auto-generated by Blender 

    :type:  boolean, default False 
    '''

    def reload(self):
        '''Reload the preview from its source path 

        '''
        pass


class ImageSequence:
    '''Sequence strip to load one or more images '''

    alpha_mode = None
    '''Representation of alpha information in the RGBA pixels 

    :type:  enum in [‘STRAIGHT’, ‘PREMUL’], default ‘STRAIGHT’ 
    '''

    animation_offset_end = None
    '''Animation end offset (trim end) 

    :type:  int in [0, inf], default 0 
    '''

    animation_offset_start = None
    '''Animation start offset (trim start) 

    :type:  int in [0, inf], default 0 
    '''

    color_multiply = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    color_saturation = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    colorspace_settings = None
    '''Input color space settings 

    :type:  ColorManagedInputColorspaceSettings, (readonly) 
    '''

    crop = None
    '''

    :type:  SequenceCrop, (readonly) 
    '''

    directory = None
    '''

    :type:  string, default “”, (never None) 
    '''

    elements = None
    '''

    :type:  SequenceElements bpy_prop_collection of SequenceElement, (readonly) 
    '''

    proxy = None
    '''

    :type:  SequenceProxy, (readonly) 
    '''

    stereo_3d_format = None
    '''Settings for stereo 3d 

    :type:  Stereo3dFormat, (readonly, never None) 
    '''

    strobe = None
    '''Only display every nth frame 

    :type:  float in [1, 30], default 0.0 
    '''

    transform = None
    '''

    :type:  SequenceTransform, (readonly) 
    '''

    use_crop = None
    '''Crop image before processing 

    :type:  boolean, default False 
    '''

    use_deinterlace = None
    '''For video movies to remove fields 

    :type:  boolean, default False 
    '''

    use_flip_x = None
    '''Flip on the X axis 

    :type:  boolean, default False 
    '''

    use_flip_y = None
    '''Flip on the Y axis 

    :type:  boolean, default False 
    '''

    use_float = None
    '''Convert input to float data 

    :type:  boolean, default False 
    '''

    use_multiview = None
    '''Use Multiple Views (when available) 

    :type:  boolean, default False 
    '''

    use_proxy = None
    '''Use a preview proxy and/or timecode index for this strip 

    :type:  boolean, default False 
    '''

    use_reverse_frames = None
    '''Reverse frame order 

    :type:  boolean, default False 
    '''

    use_translation = None
    '''Translate image before processing 

    :type:  boolean, default False 
    '''

    views_format = None
    '''Mode to load image views 

    :type:  enum in [‘INDIVIDUAL’, ‘STEREO_3D’], default ‘INDIVIDUAL’ 
    '''


class ImageTexture:
    checker_distance = None
    '''Distance between checker tiles 

    :type:  float in [0, 0.99], default 0.0 
    '''

    crop_max_x = None
    '''Maximum X value to crop the image 

    :type:  float in [-10, 10], default 0.0 
    '''

    crop_max_y = None
    '''Maximum Y value to crop the image 

    :type:  float in [-10, 10], default 0.0 
    '''

    crop_min_x = None
    '''Minimum X value to crop the image 

    :type:  float in [-10, 10], default 0.0 
    '''

    crop_min_y = None
    '''Minimum Y value to crop the image 

    :type:  float in [-10, 10], default 0.0 
    '''

    extension = None
    '''How the image is extrapolated past its original bounds 

    :type:  enum in [‘EXTEND’, ‘CLIP’, ‘CLIP_CUBE’, ‘REPEAT’, ‘CHECKER’], default ‘EXTEND’ 
    '''

    filter_eccentricity = None
    '''Maximum eccentricity (higher gives less blur at distant/oblique angles, but is also slower) 

    :type:  int in [1, 256], default 0 
    '''

    filter_probes = None
    '''Maximum number of samples (higher gives less blur at distant/oblique angles, but is also slower) 

    :type:  int in [1, 256], default 0 
    '''

    filter_size = None
    '''Multiply the filter size used by MIP Map and Interpolation 

    :type:  float in [0.1, 50], default 0.0 
    '''

    filter_type = None
    '''Texture filter to use for sampling image 

    :type:  enum in [‘BOX’, ‘EWA’, ‘FELINE’, ‘AREA’], default ‘BOX’ 
    '''

    image = None
    '''

    :type:  Image 
    '''

    image_user = None
    '''Parameters defining which layer, pass and frame of the image is displayed 

    :type:  ImageUser, (readonly) 
    '''

    invert_alpha = None
    '''Invert all the alpha values in the image 

    :type:  boolean, default False 
    '''

    repeat_x = None
    '''Repetition multiplier in the X direction 

    :type:  int in [1, 512], default 0 
    '''

    repeat_y = None
    '''Repetition multiplier in the Y direction 

    :type:  int in [1, 512], default 0 
    '''

    use_alpha = None
    '''Use the alpha channel information in the image 

    :type:  boolean, default False 
    '''

    use_calculate_alpha = None
    '''Calculate an alpha channel based on RGB values in the image 

    :type:  boolean, default False 
    '''

    use_checker_even = None
    '''Even checker tiles 

    :type:  boolean, default False 
    '''

    use_checker_odd = None
    '''Odd checker tiles 

    :type:  boolean, default False 
    '''

    use_derivative_map = None
    '''Use red and green as derivative values 

    :type:  boolean, default False 
    '''

    use_filter_size_min = None
    '''Use Filter Size as a minimal filter value in pixels 

    :type:  boolean, default False 
    '''

    use_flip_axis = None
    '''Flip the texture’s X and Y axis 

    :type:  boolean, default False 
    '''

    use_interpolation = None
    '''Interpolate pixels using selected filter 

    :type:  boolean, default False 
    '''

    use_mipmap = None
    '''Use auto-generated MIP maps for the image 

    :type:  boolean, default False 
    '''

    use_mipmap_gauss = None
    '''Use Gauss filter to sample down MIP maps 

    :type:  boolean, default False 
    '''

    use_mirror_x = None
    '''Mirror the image repetition on the X direction 

    :type:  boolean, default False 
    '''

    use_mirror_y = None
    '''Mirror the image repetition on the Y direction 

    :type:  boolean, default False 
    '''

    use_normal_map = None
    '''Use image RGB values for normal mapping 

    :type:  boolean, default False 
    '''

    users_material = None
    '''Materials that use this texture (readonly) '''

    users_object_modifier = None
    '''Object modifiers that use this texture (readonly) '''


class ImageUser:
    '''Parameters defining how an Image data-block is used by another data-block '''

    fields_per_frame = None
    '''Number of fields per rendered frame (2 fields is 1 image) 

    :type:  int in [1, 200], default 0 
    '''

    frame_current = None
    '''Current frame number in image sequence or movie 

    :type:  int in [-500000, 500000], default 0 
    '''

    frame_duration = None
    '''Number of images of a movie to use 

    :type:  int in [0, 500000], default 0 
    '''

    frame_offset = None
    '''Offset the number of the frame to use in the animation 

    :type:  int in [-inf, inf], default 0 
    '''

    frame_start = None
    '''Global starting frame of the movie/sequence, assuming first picture has a #1 

    :type:  int in [-500000, 500000], default 0 
    '''

    multilayer_layer = None
    '''Layer in multilayer image 

    :type:  int in [0, 32767], default 0, (readonly) 
    '''

    multilayer_pass = None
    '''Pass in multilayer image 

    :type:  int in [0, 32767], default 0, (readonly) 
    '''

    multilayer_view = None
    '''View in multilayer image 

    :type:  int in [0, 32767], default 0, (readonly) 
    '''

    use_auto_refresh = None
    '''Always refresh image on frame changes 

    :type:  boolean, default False 
    '''

    use_cyclic = None
    '''Cycle the images in the movie 

    :type:  boolean, default False 
    '''


class ImapaintToolCapabilities:
    '''Read-only indications of which brush operations are supported by the current image paint brush '''

    has_accumulate = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    has_radius = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    has_space_attenuation = None
    '''

    :type:  boolean, default False, (readonly) 
    '''


class InflowFluidSettings:
    '''Fluid simulation settings for objects adding fluids in the simulation '''

    inflow_velocity = None
    '''Initial velocity of fluid 

    :type:  float array of 3 items in [-1000.1, 1000.1], default (0.0, 0.0, 0.0) 
    '''

    use = None
    '''Object contributes to the fluid simulation 

    :type:  boolean, default False 
    '''

    use_animated_mesh = None
    '''Export this mesh as an animated one (slower and enforces No Slip, only use if really necessary [e.g. armatures or parented objects], animated pos/rot/scale F-Curves do not require it) 

    :type:  boolean, default False 
    '''

    use_local_coords = None
    '''Use local coordinates for inflow (e.g. for rotating objects) 

    :type:  boolean, default False 
    '''

    volume_initialization = None
    '''Volume initialization type (WARNING: complex volumes might require too much memory and break simulation) 

    :type:  enum in [‘VOLUME’, ‘SHELL’, ‘BOTH’], default ‘VOLUME’ 
    '''


class IntProperty:
    '''RNA integer number property definition '''

    array_length = None
    '''Maximum length of the array, 0 means unlimited 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    default = None
    '''Default value for this number 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    default_array = None
    '''Default value for this array 

    :type:  int array of 3 items in [-inf, inf], default (0, 0, 0), (readonly) 
    '''

    hard_max = None
    '''Maximum value used by buttons 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    hard_min = None
    '''Minimum value used by buttons 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    is_array = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    soft_max = None
    '''Maximum value used by buttons 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    soft_min = None
    '''Minimum value used by buttons 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    step = None
    '''Step size used by number buttons, for floats 1/100th of the step size 

    :type:  int in [0, inf], default 0, (readonly) 
    '''


class Itasc:
    '''Parameters for the iTaSC IK solver '''

    damping_epsilon = None
    '''Singular value under which damping is progressively applied (higher values=more stability, less reactivity - default=0.1) 

    :type:  float in [0, 1], default 0.0 
    '''

    damping_max = None
    '''Maximum damping coefficient when singular value is nearly 0 (higher values=more stability, less reactivity - default=0.5) 

    :type:  float in [0, 1], default 0.0 
    '''

    feedback = None
    '''Feedback coefficient for error correction, average response time is 1/feedback (default=20) 

    :type:  float in [0, 100], default 0.0 
    '''

    iterations = None
    '''Maximum number of iterations for convergence in case of reiteration 

    :type:  int in [0, 1000], default 0 
    '''

    mode = None
    '''

    :type:  enum in [‘ANIMATION’, ‘SIMULATION’], default ‘ANIMATION’ 
    '''

    precision = None
    '''Precision of convergence in case of reiteration 

    :type:  float in [0, 0.1], default 0.0 
    '''

    reiteration_method = None
    '''Defines if the solver is allowed to reiterate (converge until precision is met) on none, first or all frames 

    :type:  enum in [‘NEVER’, ‘INITIAL’, ‘ALWAYS’], default ‘NEVER’ 
    '''

    solver = None
    '''Solving method selection: automatic damping or manual damping 

    :type:  enum in [‘SDLS’, ‘DLS’], default ‘SDLS’ 
    '''

    step_count = None
    '''Divide the frame interval into this many steps 

    :type:  int in [1, 50], default 0 
    '''

    step_max = None
    '''Higher bound for timestep in second in case of automatic substeps 

    :type:  float in [0, 1], default 0.0 
    '''

    step_min = None
    '''Lower bound for timestep in second in case of automatic substeps 

    :type:  float in [0, 0.1], default 0.0 
    '''

    use_auto_step = None
    '''Automatically determine the optimal number of steps for best performance/accuracy trade off 

    :type:  boolean, default False 
    '''

    velocity_max = None
    '''Maximum joint velocity in rad/s (default=50) 

    :type:  float in [0, 100], default 0.0 
    '''


class JoystickSensor:
    '''Sensor to detect joystick events '''

    axis_direction = None
    '''The direction of the axis 

    :type:  enum in [‘RIGHTAXIS’, ‘UPAXIS’, ‘LEFTAXIS’, ‘DOWNAXIS’], default ‘RIGHTAXIS’ 
    '''

    axis_number = None
    '''Which axis pair to use, 1 is usually the main direction input 

    :type:  int in [1, 8], default 0 
    '''

    axis_threshold = None
    '''Precision of the axis 

    :type:  int in [0, 32768], default 0 
    '''

    button_number = None
    '''Which button to use 

    :type:  int in [0, 18], default 0 
    '''

    event_type = None
    '''The type of event this joystick sensor is triggered on 

    :type:  enum in [‘BUTTON’, ‘AXIS’, ‘HAT’, ‘AXIS_SINGLE’], default ‘BUTTON’ 
    '''

    hat_direction = None
    '''Hat direction 

    :type:  enum in [‘UP’, ‘DOWN’, ‘LEFT’, ‘RIGHT’, ‘UPRIGHT’, ‘DOWNLEFT’, ‘UPLEFT’, ‘DOWNRIGHT’], default ‘UP’ 
    '''

    hat_number = None
    '''Which hat to use 

    :type:  int in [1, 2], default 0 
    '''

    joystick_index = None
    '''Which joystick to use 

    :type:  int in [0, 7], default 0 
    '''

    single_axis_number = None
    '''Single axis (vertical/horizontal/other) to detect 

    :type:  int in [1, 16], default 0 
    '''

    use_all_events = None
    '''Triggered by all events on this joystick’s current type (axis/button/hat) 

    :type:  boolean, default False 
    '''


class Key:
    '''Shape keys data-block containing different shapes of geometric data-blocks '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    eval_time = None
    '''Evaluation time for absolute shape keys 

    :type:  float in [0, 500000], default 0.0 
    '''

    key_blocks = None
    '''Shape keys 

    :type:  bpy_prop_collection of ShapeKey, (readonly) 
    '''

    reference_key = None
    '''

    :type:  ShapeKey, (readonly, never None) 
    '''

    use_relative = None
    '''Make shape keys relative, otherwise play through shapes as a sequence using the evaluation time 

    :type:  boolean, default False 
    '''

    user = None
    '''Datablock using these shape keys 

    :type:  ID, (readonly, never None) 
    '''


class KeyConfig:
    '''Input configuration, including keymaps '''

    is_user_defined = None
    '''Indicates that a keyconfig was defined by the user 

    :type:  boolean, default False, (readonly) 
    '''

    keymaps = None
    '''Key maps configured as part of this configuration 

    :type:  KeyMaps bpy_prop_collection of KeyMap, (readonly) 
    '''

    name = None
    '''Name of the key configuration 

    :type:  string, default “”, (never None) 
    '''


class KeyConfigurations:
    '''Collection of KeyConfigs '''

    active = None
    '''Active key configuration (preset) 

    :type:  KeyConfig 
    '''

    addon = None
    '''Key configuration that can be extended by add-ons, and is added to the active configuration when handling events 

    :type:  KeyConfig, (readonly) 
    '''

    default = None
    '''Default builtin key configuration 

    :type:  KeyConfig, (readonly) 
    '''

    user = None
    '''Final key configuration that combines keymaps from the active and add-on configurations, and can be edited by the user 

    :type:  KeyConfig, (readonly) 
    '''

    def new(self, name):
        '''new 

        :param name: Name 
        :type name: string, (never None)
        :rtype: KeyConfig 
        :return:  Key Configuration, Added key configuration 
        '''
        pass

    def remove(self, keyconfig):
        '''remove 

        :param keyconfig: Key Configuration, Removed key configuration 
        :type keyconfig: KeyConfig, (never None)
        '''
        pass


class KeyMap:
    '''Input configuration, including keymaps '''

    is_modal = None
    '''Indicates that a keymap is used for translate modal events for an operator 

    :type:  boolean, default False, (readonly) 
    '''

    is_user_modified = None
    '''Keymap is defined by the user 

    :type:  boolean, default False 
    '''

    keymap_items = None
    '''Items in the keymap, linking an operator to an input event 

    :type:  KeyMapItems bpy_prop_collection of KeyMapItem, (readonly) 
    '''

    name = None
    '''Name of the key map 

    :type:  string, default “”, (readonly, never None) 
    '''

    region_type = None
    '''Optional region type keymap is associated with 

    :type:  enum in [‘WINDOW’, ‘HEADER’, ‘CHANNELS’, ‘TEMPORARY’, ‘UI’, ‘TOOLS’, ‘TOOL_PROPS’, ‘PREVIEW’], default ‘WINDOW’, (readonly) 
    '''

    show_expanded_children = None
    '''Children expanded in the user interface 

    :type:  boolean, default False 
    '''

    show_expanded_items = None
    '''Expanded in the user interface 

    :type:  boolean, default False 
    '''

    space_type = None
    '''Optional space type keymap is associated with 

    :type:  enum in [‘EMPTY’, ‘VIEW_3D’, ‘TIMELINE’, ‘GRAPH_EDITOR’, ‘DOPESHEET_EDITOR’, ‘NLA_EDITOR’, ‘IMAGE_EDITOR’, ‘SEQUENCE_EDITOR’, ‘CLIP_EDITOR’, ‘TEXT_EDITOR’, ‘NODE_EDITOR’, ‘LOGIC_EDITOR’, ‘PROPERTIES’, ‘OUTLINER’, ‘USER_PREFERENCES’, ‘INFO’, ‘FILE_BROWSER’, ‘CONSOLE’], default ‘EMPTY’, (readonly) 
    '''

    def active(self):
        '''active 

        :rtype: KeyMap 
        :return:  Key Map, Active key map 
        '''
        pass

    def restore_to_default(self):
        '''restore_to_default 

        '''
        pass

    def restore_item_to_default(self, item):
        '''restore_item_to_default 

        :param item: Item 
        :type item: KeyMapItem, (never None)
        '''
        pass


class KeyMapItem:
    '''Item in a Key Map '''

    active = None
    '''Activate or deactivate item 

    :type:  boolean, default False 
    '''

    alt = None
    '''Alt key pressed 

    :type:  boolean, default False 
    '''

    any = None
    '''Any modifier keys pressed 

    :type:  boolean, default False 
    '''

    ctrl = None
    '''Control key pressed 

    :type:  boolean, default False 
    '''

    id = None
    '''ID of the item 

    :type:  int in [-32768, 32767], default 0, (readonly) 
    '''

    idname = None
    '''Identifier of operator to call on input event 

    :type:  string, default “”, (never None) 
    '''

    is_user_defined = None
    '''Is this keymap item user defined (doesn’t just replace a builtin item) 

    :type:  boolean, default False, (readonly) 
    '''

    is_user_modified = None
    '''Is this keymap item modified by the user 

    :type:  boolean, default False, (readonly) 
    '''

    key_modifier = None
    '''Regular key pressed as a modifier 

    :type:  enum in [‘NONE’, ‘LEFTMOUSE’, ‘MIDDLEMOUSE’, ‘RIGHTMOUSE’, ‘BUTTON4MOUSE’, ‘BUTTON5MOUSE’, ‘BUTTON6MOUSE’, ‘BUTTON7MOUSE’, ‘ACTIONMOUSE’, ‘SELECTMOUSE’, ‘PEN’, ‘ERASER’, ‘MOUSEMOVE’, ‘INBETWEEN_MOUSEMOVE’, ‘TRACKPADPAN’, ‘TRACKPADZOOM’, ‘MOUSEROTATE’, ‘WHEELUPMOUSE’, ‘WHEELDOWNMOUSE’, ‘WHEELINMOUSE’, ‘WHEELOUTMOUSE’, ‘EVT_TWEAK_L’, ‘EVT_TWEAK_M’, ‘EVT_TWEAK_R’, ‘EVT_TWEAK_A’, ‘EVT_TWEAK_S’, ‘A’, ‘B’, ‘C’, ‘D’, ‘E’, ‘F’, ‘G’, ‘H’, ‘I’, ‘J’, ‘K’, ‘L’, ‘M’, ‘N’, ‘O’, ‘P’, ‘Q’, ‘R’, ‘S’, ‘T’, ‘U’, ‘V’, ‘W’, ‘X’, ‘Y’, ‘Z’, ‘ZERO’, ‘ONE’, ‘TWO’, ‘THREE’, ‘FOUR’, ‘FIVE’, ‘SIX’, ‘SEVEN’, ‘EIGHT’, ‘NINE’, ‘LEFT_CTRL’, ‘LEFT_ALT’, ‘LEFT_SHIFT’, ‘RIGHT_ALT’, ‘RIGHT_CTRL’, ‘RIGHT_SHIFT’, ‘OSKEY’, ‘GRLESS’, ‘ESC’, ‘TAB’, ‘RET’, ‘SPACE’, ‘LINE_FEED’, ‘BACK_SPACE’, ‘DEL’, ‘SEMI_COLON’, ‘PERIOD’, ‘COMMA’, ‘QUOTE’, ‘ACCENT_GRAVE’, ‘MINUS’, ‘PLUS’, ‘SLASH’, ‘BACK_SLASH’, ‘EQUAL’, ‘LEFT_BRACKET’, ‘RIGHT_BRACKET’, ‘LEFT_ARROW’, ‘DOWN_ARROW’, ‘RIGHT_ARROW’, ‘UP_ARROW’, ‘NUMPAD_2’, ‘NUMPAD_4’, ‘NUMPAD_6’, ‘NUMPAD_8’, ‘NUMPAD_1’, ‘NUMPAD_3’, ‘NUMPAD_5’, ‘NUMPAD_7’, ‘NUMPAD_9’, ‘NUMPAD_PERIOD’, ‘NUMPAD_SLASH’, ‘NUMPAD_ASTERIX’, ‘NUMPAD_0’, ‘NUMPAD_MINUS’, ‘NUMPAD_ENTER’, ‘NUMPAD_PLUS’, ‘F1’, ‘F2’, ‘F3’, ‘F4’, ‘F5’, ‘F6’, ‘F7’, ‘F8’, ‘F9’, ‘F10’, ‘F11’, ‘F12’, ‘F13’, ‘F14’, ‘F15’, ‘F16’, ‘F17’, ‘F18’, ‘F19’, ‘PAUSE’, ‘INSERT’, ‘HOME’, ‘PAGE_UP’, ‘PAGE_DOWN’, ‘END’, ‘MEDIA_PLAY’, ‘MEDIA_STOP’, ‘MEDIA_FIRST’, ‘MEDIA_LAST’, ‘TEXTINPUT’, ‘WINDOW_DEACTIVATE’, ‘TIMER’, ‘TIMER0’, ‘TIMER1’, ‘TIMER2’, ‘TIMER_JOBS’, ‘TIMER_AUTOSAVE’, ‘TIMER_REPORT’, ‘TIMERREGION’, ‘NDOF_MOTION’, ‘NDOF_BUTTON_MENU’, ‘NDOF_BUTTON_FIT’, ‘NDOF_BUTTON_TOP’, ‘NDOF_BUTTON_BOTTOM’, ‘NDOF_BUTTON_LEFT’, ‘NDOF_BUTTON_RIGHT’, ‘NDOF_BUTTON_FRONT’, ‘NDOF_BUTTON_BACK’, ‘NDOF_BUTTON_ISO1’, ‘NDOF_BUTTON_ISO2’, ‘NDOF_BUTTON_ROLL_CW’, ‘NDOF_BUTTON_ROLL_CCW’, ‘NDOF_BUTTON_SPIN_CW’, ‘NDOF_BUTTON_SPIN_CCW’, ‘NDOF_BUTTON_TILT_CW’, ‘NDOF_BUTTON_TILT_CCW’, ‘NDOF_BUTTON_ROTATE’, ‘NDOF_BUTTON_PANZOOM’, ‘NDOF_BUTTON_DOMINANT’, ‘NDOF_BUTTON_PLUS’, ‘NDOF_BUTTON_MINUS’, ‘NDOF_BUTTON_ESC’, ‘NDOF_BUTTON_ALT’, ‘NDOF_BUTTON_SHIFT’, ‘NDOF_BUTTON_CTRL’, ‘NDOF_BUTTON_1’, ‘NDOF_BUTTON_2’, ‘NDOF_BUTTON_3’, ‘NDOF_BUTTON_4’, ‘NDOF_BUTTON_5’, ‘NDOF_BUTTON_6’, ‘NDOF_BUTTON_7’, ‘NDOF_BUTTON_8’, ‘NDOF_BUTTON_9’, ‘NDOF_BUTTON_10’, ‘NDOF_BUTTON_A’, ‘NDOF_BUTTON_B’, ‘NDOF_BUTTON_C’], default ‘NONE’ 
    '''

    map_type = None
    '''Type of event mapping 

    :type:  enum in [‘KEYBOARD’, ‘TWEAK’, ‘MOUSE’, ‘NDOF’, ‘TEXTINPUT’, ‘TIMER’], default ‘KEYBOARD’ 
    '''

    name = None
    '''Name of operator (translated) to call on input event 

    :type:  string, default “”, (readonly, never None) 
    '''

    oskey = None
    '''Operating system key pressed 

    :type:  boolean, default False 
    '''

    properties = None
    '''Properties to set when the operator is called 

    :type:  OperatorProperties, (readonly) 
    '''

    propvalue = None
    '''The value this event translates to in a modal keymap 

    :type:  enum in [‘NONE’], default ‘NONE’ 
    '''

    shift = None
    '''Shift key pressed 

    :type:  boolean, default False 
    '''

    show_expanded = None
    '''Show key map event and property details in the user interface 

    :type:  boolean, default False 
    '''

    type = None
    '''Type of event 

    :type:  enum in [‘NONE’, ‘LEFTMOUSE’, ‘MIDDLEMOUSE’, ‘RIGHTMOUSE’, ‘BUTTON4MOUSE’, ‘BUTTON5MOUSE’, ‘BUTTON6MOUSE’, ‘BUTTON7MOUSE’, ‘ACTIONMOUSE’, ‘SELECTMOUSE’, ‘PEN’, ‘ERASER’, ‘MOUSEMOVE’, ‘INBETWEEN_MOUSEMOVE’, ‘TRACKPADPAN’, ‘TRACKPADZOOM’, ‘MOUSEROTATE’, ‘WHEELUPMOUSE’, ‘WHEELDOWNMOUSE’, ‘WHEELINMOUSE’, ‘WHEELOUTMOUSE’, ‘EVT_TWEAK_L’, ‘EVT_TWEAK_M’, ‘EVT_TWEAK_R’, ‘EVT_TWEAK_A’, ‘EVT_TWEAK_S’, ‘A’, ‘B’, ‘C’, ‘D’, ‘E’, ‘F’, ‘G’, ‘H’, ‘I’, ‘J’, ‘K’, ‘L’, ‘M’, ‘N’, ‘O’, ‘P’, ‘Q’, ‘R’, ‘S’, ‘T’, ‘U’, ‘V’, ‘W’, ‘X’, ‘Y’, ‘Z’, ‘ZERO’, ‘ONE’, ‘TWO’, ‘THREE’, ‘FOUR’, ‘FIVE’, ‘SIX’, ‘SEVEN’, ‘EIGHT’, ‘NINE’, ‘LEFT_CTRL’, ‘LEFT_ALT’, ‘LEFT_SHIFT’, ‘RIGHT_ALT’, ‘RIGHT_CTRL’, ‘RIGHT_SHIFT’, ‘OSKEY’, ‘GRLESS’, ‘ESC’, ‘TAB’, ‘RET’, ‘SPACE’, ‘LINE_FEED’, ‘BACK_SPACE’, ‘DEL’, ‘SEMI_COLON’, ‘PERIOD’, ‘COMMA’, ‘QUOTE’, ‘ACCENT_GRAVE’, ‘MINUS’, ‘PLUS’, ‘SLASH’, ‘BACK_SLASH’, ‘EQUAL’, ‘LEFT_BRACKET’, ‘RIGHT_BRACKET’, ‘LEFT_ARROW’, ‘DOWN_ARROW’, ‘RIGHT_ARROW’, ‘UP_ARROW’, ‘NUMPAD_2’, ‘NUMPAD_4’, ‘NUMPAD_6’, ‘NUMPAD_8’, ‘NUMPAD_1’, ‘NUMPAD_3’, ‘NUMPAD_5’, ‘NUMPAD_7’, ‘NUMPAD_9’, ‘NUMPAD_PERIOD’, ‘NUMPAD_SLASH’, ‘NUMPAD_ASTERIX’, ‘NUMPAD_0’, ‘NUMPAD_MINUS’, ‘NUMPAD_ENTER’, ‘NUMPAD_PLUS’, ‘F1’, ‘F2’, ‘F3’, ‘F4’, ‘F5’, ‘F6’, ‘F7’, ‘F8’, ‘F9’, ‘F10’, ‘F11’, ‘F12’, ‘F13’, ‘F14’, ‘F15’, ‘F16’, ‘F17’, ‘F18’, ‘F19’, ‘PAUSE’, ‘INSERT’, ‘HOME’, ‘PAGE_UP’, ‘PAGE_DOWN’, ‘END’, ‘MEDIA_PLAY’, ‘MEDIA_STOP’, ‘MEDIA_FIRST’, ‘MEDIA_LAST’, ‘TEXTINPUT’, ‘WINDOW_DEACTIVATE’, ‘TIMER’, ‘TIMER0’, ‘TIMER1’, ‘TIMER2’, ‘TIMER_JOBS’, ‘TIMER_AUTOSAVE’, ‘TIMER_REPORT’, ‘TIMERREGION’, ‘NDOF_MOTION’, ‘NDOF_BUTTON_MENU’, ‘NDOF_BUTTON_FIT’, ‘NDOF_BUTTON_TOP’, ‘NDOF_BUTTON_BOTTOM’, ‘NDOF_BUTTON_LEFT’, ‘NDOF_BUTTON_RIGHT’, ‘NDOF_BUTTON_FRONT’, ‘NDOF_BUTTON_BACK’, ‘NDOF_BUTTON_ISO1’, ‘NDOF_BUTTON_ISO2’, ‘NDOF_BUTTON_ROLL_CW’, ‘NDOF_BUTTON_ROLL_CCW’, ‘NDOF_BUTTON_SPIN_CW’, ‘NDOF_BUTTON_SPIN_CCW’, ‘NDOF_BUTTON_TILT_CW’, ‘NDOF_BUTTON_TILT_CCW’, ‘NDOF_BUTTON_ROTATE’, ‘NDOF_BUTTON_PANZOOM’, ‘NDOF_BUTTON_DOMINANT’, ‘NDOF_BUTTON_PLUS’, ‘NDOF_BUTTON_MINUS’, ‘NDOF_BUTTON_ESC’, ‘NDOF_BUTTON_ALT’, ‘NDOF_BUTTON_SHIFT’, ‘NDOF_BUTTON_CTRL’, ‘NDOF_BUTTON_1’, ‘NDOF_BUTTON_2’, ‘NDOF_BUTTON_3’, ‘NDOF_BUTTON_4’, ‘NDOF_BUTTON_5’, ‘NDOF_BUTTON_6’, ‘NDOF_BUTTON_7’, ‘NDOF_BUTTON_8’, ‘NDOF_BUTTON_9’, ‘NDOF_BUTTON_10’, ‘NDOF_BUTTON_A’, ‘NDOF_BUTTON_B’, ‘NDOF_BUTTON_C’], default ‘NONE’ 
    '''

    value = None
    '''

    :type:  enum in [‘ANY’, ‘NOTHING’, ‘PRESS’, ‘RELEASE’, ‘CLICK’, ‘DOUBLE_CLICK’, ‘NORTH’, ‘NORTH_EAST’, ‘EAST’, ‘SOUTH_EAST’, ‘SOUTH’, ‘SOUTH_WEST’, ‘WEST’, ‘NORTH_WEST’], default ‘NOTHING’ 
    '''

    def compare(self, item):
        '''compare 

        :param item: Item 
        :type item: KeyMapItem
        :rtype: boolean 
        :return:  Comparison result 
        '''
        pass


class KeyMapItems:
    '''Collection of keymap items '''

    def new(self,
            idname,
            type,
            value,
            any=False,
            shift=False,
            ctrl=False,
            alt=False,
            oskey=False,
            key_modifier='NONE',
            head=False):
        '''new 

        :param idname: Operator Identifier 
        :type idname: string, (never None)
        :param type: TypeNONE .LEFTMOUSE Left Mouse, LMB.MIDDLEMOUSE Middle Mouse, MMB.RIGHTMOUSE Right Mouse, RMB.BUTTON4MOUSE Button4 Mouse, MB4.BUTTON5MOUSE Button5 Mouse, MB5.BUTTON6MOUSE Button6 Mouse, MB6.BUTTON7MOUSE Button7 Mouse, MB7.ACTIONMOUSE Action Mouse, MBA.SELECTMOUSE Select Mouse, MBS.PEN Pen.ERASER Eraser.MOUSEMOVE Mouse Move, MsMov.INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.TRACKPADPAN Mouse/Trackpad Pan, MsPan.TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.MOUSEROTATE Mouse/Trackpad Rotate, MsRot.WHEELUPMOUSE Wheel Up, WhUp.WHEELDOWNMOUSE Wheel Down, WhDown.WHEELINMOUSE Wheel In, WhIn.WHEELOUTMOUSE Wheel Out, WhOut.EVT_TWEAK_L Tweak Left, TwkL.EVT_TWEAK_M Tweak Middle, TwkM.EVT_TWEAK_R Tweak Right, TwkR.EVT_TWEAK_A Tweak Action, TwkA.EVT_TWEAK_S Tweak Select, TwkS.A A.B B.C C.D D.E E.F F.G G.H H.I I.J J.K K.L L.M M.N N.O O.P P.Q Q.R R.S S.T T.U U.V V.W W.X X.Y Y.Z Z.ZERO 0.ONE 1.TWO 2.THREE 3.FOUR 4.FIVE 5.SIX 6.SEVEN 7.EIGHT 8.NINE 9.LEFT_CTRL Left Ctrl, CtrlL.LEFT_ALT Left Alt, AltL.LEFT_SHIFT Left Shift, ShiftL.RIGHT_ALT Right Alt, AltR.RIGHT_CTRL Right Ctrl, CtrlR.RIGHT_SHIFT Right Shift, ShiftR.OSKEY OS Key, Cmd.GRLESS Grless.ESC Esc.TAB Tab.RET Return, Enter.SPACE Spacebar, Space.LINE_FEED Line Feed.BACK_SPACE Back Space, BkSpace.DEL Delete, Del.SEMI_COLON ;.PERIOD ..COMMA ,.QUOTE “.ACCENT_GRAVE `.MINUS -.PLUS +.SLASH /.BACK_SLASH .EQUAL =.LEFT_BRACKET [.RIGHT_BRACKET ].LEFT_ARROW Left Arrow, ←.DOWN_ARROW Down Arrow, ↓.RIGHT_ARROW Right Arrow, →.UP_ARROW Up Arrow, ↑.NUMPAD_2 Numpad 2, Pad2.NUMPAD_4 Numpad 4, Pad4.NUMPAD_6 Numpad 6, Pad6.NUMPAD_8 Numpad 8, Pad8.NUMPAD_1 Numpad 1, Pad1.NUMPAD_3 Numpad 3, Pad3.NUMPAD_5 Numpad 5, Pad5.NUMPAD_7 Numpad 7, Pad7.NUMPAD_9 Numpad 9, Pad9.NUMPAD_PERIOD Numpad ., Pad..NUMPAD_SLASH Numpad /, Pad/.NUMPAD_ASTERIX Numpad *, Pad*.NUMPAD_0 Numpad 0, Pad0.NUMPAD_MINUS Numpad -, Pad-.NUMPAD_ENTER Numpad Enter, PadEnter.NUMPAD_PLUS Numpad +, Pad+.F1 F1.F2 F2.F3 F3.F4 F4.F5 F5.F6 F6.F7 F7.F8 F8.F9 F9.F10 F10.F11 F11.F12 F12.F13 F13.F14 F14.F15 F15.F16 F16.F17 F17.F18 F18.F19 F19.PAUSE Pause.INSERT Insert, Ins.HOME Home.PAGE_UP Page Up, PgUp.PAGE_DOWN Page Down, PgDown.END End.MEDIA_PLAY Media Play/Pause, >/||.MEDIA_STOP Media Stop, Stop.MEDIA_FIRST Media First, |<<.MEDIA_LAST Media Last, >>|.TEXTINPUT Text Input, TxtIn.WINDOW_DEACTIVATE Window Deactivate.TIMER Timer, Tmr.TIMER0 Timer 0, Tmr0.TIMER1 Timer 1, Tmr1.TIMER2 Timer 2, Tmr2.TIMER_JOBS Timer Jobs, TmrJob.TIMER_AUTOSAVE Timer Autosave, TmrSave.TIMER_REPORT Timer Report, TmrReport.TIMERREGION Timer Region, TmrReg.NDOF_MOTION NDOF Motion, NdofMov.NDOF_BUTTON_MENU NDOF Menu, NdofMenu.NDOF_BUTTON_FIT NDOF Fit, NdofFit.NDOF_BUTTON_TOP NDOF Top, Ndof↑.NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.NDOF_BUTTON_LEFT NDOF Left, Ndof←.NDOF_BUTTON_RIGHT NDOF Right, Ndof→.NDOF_BUTTON_FRONT NDOF Front, NdofFront.NDOF_BUTTON_BACK NDOF Back, NdofBack.NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.NDOF_BUTTON_PLUS NDOF Plus, Ndof+.NDOF_BUTTON_MINUS NDOF Minus, Ndof-.NDOF_BUTTON_ESC NDOF Esc, NdofEsc.NDOF_BUTTON_ALT NDOF Alt, NdofAlt.NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.NDOF_BUTTON_1 NDOF Button 1, NdofB1.NDOF_BUTTON_2 NDOF Button 2, NdofB2.NDOF_BUTTON_3 NDOF Button 3, NdofB3.NDOF_BUTTON_4 NDOF Button 4, NdofB4.NDOF_BUTTON_5 NDOF Button 5, NdofB5.NDOF_BUTTON_6 NDOF Button 6, NdofB6.NDOF_BUTTON_7 NDOF Button 7, NdofB7.NDOF_BUTTON_8 NDOF Button 8, NdofB8.NDOF_BUTTON_9 NDOF Button 9, NdofB9.NDOF_BUTTON_10 NDOF Button 10, NdofB10.NDOF_BUTTON_A NDOF Button A, NdofBA.NDOF_BUTTON_B NDOF Button B, NdofBB.NDOF_BUTTON_C NDOF Button C, NdofBC. 
        :type type: enum in ['NONE', 'LEFTMOUSE', 'MIDDLEMOUSE', 'RIGHTMOUSE', 'BUTTON4MOUSE', 'BUTTON5MOUSE', 'BUTTON6MOUSE', 'BUTTON7MOUSE', 'ACTIONMOUSE', 'SELECTMOUSE', 'PEN', 'ERASER', 'MOUSEMOVE', 'INBETWEEN_MOUSEMOVE', 'TRACKPADPAN', 'TRACKPADZOOM', 'MOUSEROTATE', 'WHEELUPMOUSE', 'WHEELDOWNMOUSE', 'WHEELINMOUSE', 'WHEELOUTMOUSE', 'EVT_TWEAK_L', 'EVT_TWEAK_M', 'EVT_TWEAK_R', 'EVT_TWEAK_A', 'EVT_TWEAK_S', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'ZERO', 'ONE', 'TWO', 'THREE', 'FOUR', 'FIVE', 'SIX', 'SEVEN', 'EIGHT', 'NINE', 'LEFT_CTRL', 'LEFT_ALT', 'LEFT_SHIFT', 'RIGHT_ALT', 'RIGHT_CTRL', 'RIGHT_SHIFT', 'OSKEY', 'GRLESS', 'ESC', 'TAB', 'RET', 'SPACE', 'LINE_FEED', 'BACK_SPACE', 'DEL', 'SEMI_COLON', 'PERIOD', 'COMMA', 'QUOTE', 'ACCENT_GRAVE', 'MINUS', 'PLUS', 'SLASH', 'BACK_SLASH', 'EQUAL', 'LEFT_BRACKET', 'RIGHT_BRACKET', 'LEFT_ARROW', 'DOWN_ARROW', 'RIGHT_ARROW', 'UP_ARROW', 'NUMPAD_2', 'NUMPAD_4', 'NUMPAD_6', 'NUMPAD_8', 'NUMPAD_1', 'NUMPAD_3', 'NUMPAD_5', 'NUMPAD_7', 'NUMPAD_9', 'NUMPAD_PERIOD', 'NUMPAD_SLASH', 'NUMPAD_ASTERIX', 'NUMPAD_0', 'NUMPAD_MINUS', 'NUMPAD_ENTER', 'NUMPAD_PLUS', 'F1', 'F2', 'F3', 'F4', 'F5', 'F6', 'F7', 'F8', 'F9', 'F10', 'F11', 'F12', 'F13', 'F14', 'F15', 'F16', 'F17', 'F18', 'F19', 'PAUSE', 'INSERT', 'HOME', 'PAGE_UP', 'PAGE_DOWN', 'END', 'MEDIA_PLAY', 'MEDIA_STOP', 'MEDIA_FIRST', 'MEDIA_LAST', 'TEXTINPUT', 'WINDOW_DEACTIVATE', 'TIMER', 'TIMER0', 'TIMER1', 'TIMER2', 'TIMER_JOBS', 'TIMER_AUTOSAVE', 'TIMER_REPORT', 'TIMERREGION', 'NDOF_MOTION', 'NDOF_BUTTON_MENU', 'NDOF_BUTTON_FIT', 'NDOF_BUTTON_TOP', 'NDOF_BUTTON_BOTTOM', 'NDOF_BUTTON_LEFT', 'NDOF_BUTTON_RIGHT', 'NDOF_BUTTON_FRONT', 'NDOF_BUTTON_BACK', 'NDOF_BUTTON_ISO1', 'NDOF_BUTTON_ISO2', 'NDOF_BUTTON_ROLL_CW', 'NDOF_BUTTON_ROLL_CCW', 'NDOF_BUTTON_SPIN_CW', 'NDOF_BUTTON_SPIN_CCW', 'NDOF_BUTTON_TILT_CW', 'NDOF_BUTTON_TILT_CCW', 'NDOF_BUTTON_ROTATE', 'NDOF_BUTTON_PANZOOM', 'NDOF_BUTTON_DOMINANT', 'NDOF_BUTTON_PLUS', 'NDOF_BUTTON_MINUS', 'NDOF_BUTTON_ESC', 'NDOF_BUTTON_ALT', 'NDOF_BUTTON_SHIFT', 'NDOF_BUTTON_CTRL', 'NDOF_BUTTON_1', 'NDOF_BUTTON_2', 'NDOF_BUTTON_3', 'NDOF_BUTTON_4', 'NDOF_BUTTON_5', 'NDOF_BUTTON_6', 'NDOF_BUTTON_7', 'NDOF_BUTTON_8', 'NDOF_BUTTON_9', 'NDOF_BUTTON_10', 'NDOF_BUTTON_A', 'NDOF_BUTTON_B', 'NDOF_BUTTON_C']
        :param value: Value 
        :type value: enum in ['ANY', 'NOTHING', 'PRESS', 'RELEASE', 'CLICK', 'DOUBLE_CLICK', 'NORTH', 'NORTH_EAST', 'EAST', 'SOUTH_EAST', 'SOUTH', 'SOUTH_WEST', 'WEST', 'NORTH_WEST']
        :param any: Any 
        :type any: boolean, (optional)
        :param shift: Shift 
        :type shift: boolean, (optional)
        :param ctrl: Ctrl 
        :type ctrl: boolean, (optional)
        :param alt: Alt 
        :type alt: boolean, (optional)
        :param oskey: OS Key 
        :type oskey: boolean, (optional)
        :param key_modifier: Key ModifierNONE .LEFTMOUSE Left Mouse, LMB.MIDDLEMOUSE Middle Mouse, MMB.RIGHTMOUSE Right Mouse, RMB.BUTTON4MOUSE Button4 Mouse, MB4.BUTTON5MOUSE Button5 Mouse, MB5.BUTTON6MOUSE Button6 Mouse, MB6.BUTTON7MOUSE Button7 Mouse, MB7.ACTIONMOUSE Action Mouse, MBA.SELECTMOUSE Select Mouse, MBS.PEN Pen.ERASER Eraser.MOUSEMOVE Mouse Move, MsMov.INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.TRACKPADPAN Mouse/Trackpad Pan, MsPan.TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.MOUSEROTATE Mouse/Trackpad Rotate, MsRot.WHEELUPMOUSE Wheel Up, WhUp.WHEELDOWNMOUSE Wheel Down, WhDown.WHEELINMOUSE Wheel In, WhIn.WHEELOUTMOUSE Wheel Out, WhOut.EVT_TWEAK_L Tweak Left, TwkL.EVT_TWEAK_M Tweak Middle, TwkM.EVT_TWEAK_R Tweak Right, TwkR.EVT_TWEAK_A Tweak Action, TwkA.EVT_TWEAK_S Tweak Select, TwkS.A A.B B.C C.D D.E E.F F.G G.H H.I I.J J.K K.L L.M M.N N.O O.P P.Q Q.R R.S S.T T.U U.V V.W W.X X.Y Y.Z Z.ZERO 0.ONE 1.TWO 2.THREE 3.FOUR 4.FIVE 5.SIX 6.SEVEN 7.EIGHT 8.NINE 9.LEFT_CTRL Left Ctrl, CtrlL.LEFT_ALT Left Alt, AltL.LEFT_SHIFT Left Shift, ShiftL.RIGHT_ALT Right Alt, AltR.RIGHT_CTRL Right Ctrl, CtrlR.RIGHT_SHIFT Right Shift, ShiftR.OSKEY OS Key, Cmd.GRLESS Grless.ESC Esc.TAB Tab.RET Return, Enter.SPACE Spacebar, Space.LINE_FEED Line Feed.BACK_SPACE Back Space, BkSpace.DEL Delete, Del.SEMI_COLON ;.PERIOD ..COMMA ,.QUOTE “.ACCENT_GRAVE `.MINUS -.PLUS +.SLASH /.BACK_SLASH .EQUAL =.LEFT_BRACKET [.RIGHT_BRACKET ].LEFT_ARROW Left Arrow, ←.DOWN_ARROW Down Arrow, ↓.RIGHT_ARROW Right Arrow, →.UP_ARROW Up Arrow, ↑.NUMPAD_2 Numpad 2, Pad2.NUMPAD_4 Numpad 4, Pad4.NUMPAD_6 Numpad 6, Pad6.NUMPAD_8 Numpad 8, Pad8.NUMPAD_1 Numpad 1, Pad1.NUMPAD_3 Numpad 3, Pad3.NUMPAD_5 Numpad 5, Pad5.NUMPAD_7 Numpad 7, Pad7.NUMPAD_9 Numpad 9, Pad9.NUMPAD_PERIOD Numpad ., Pad..NUMPAD_SLASH Numpad /, Pad/.NUMPAD_ASTERIX Numpad *, Pad*.NUMPAD_0 Numpad 0, Pad0.NUMPAD_MINUS Numpad -, Pad-.NUMPAD_ENTER Numpad Enter, PadEnter.NUMPAD_PLUS Numpad +, Pad+.F1 F1.F2 F2.F3 F3.F4 F4.F5 F5.F6 F6.F7 F7.F8 F8.F9 F9.F10 F10.F11 F11.F12 F12.F13 F13.F14 F14.F15 F15.F16 F16.F17 F17.F18 F18.F19 F19.PAUSE Pause.INSERT Insert, Ins.HOME Home.PAGE_UP Page Up, PgUp.PAGE_DOWN Page Down, PgDown.END End.MEDIA_PLAY Media Play/Pause, >/||.MEDIA_STOP Media Stop, Stop.MEDIA_FIRST Media First, |<<.MEDIA_LAST Media Last, >>|.TEXTINPUT Text Input, TxtIn.WINDOW_DEACTIVATE Window Deactivate.TIMER Timer, Tmr.TIMER0 Timer 0, Tmr0.TIMER1 Timer 1, Tmr1.TIMER2 Timer 2, Tmr2.TIMER_JOBS Timer Jobs, TmrJob.TIMER_AUTOSAVE Timer Autosave, TmrSave.TIMER_REPORT Timer Report, TmrReport.TIMERREGION Timer Region, TmrReg.NDOF_MOTION NDOF Motion, NdofMov.NDOF_BUTTON_MENU NDOF Menu, NdofMenu.NDOF_BUTTON_FIT NDOF Fit, NdofFit.NDOF_BUTTON_TOP NDOF Top, Ndof↑.NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.NDOF_BUTTON_LEFT NDOF Left, Ndof←.NDOF_BUTTON_RIGHT NDOF Right, Ndof→.NDOF_BUTTON_FRONT NDOF Front, NdofFront.NDOF_BUTTON_BACK NDOF Back, NdofBack.NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.NDOF_BUTTON_PLUS NDOF Plus, Ndof+.NDOF_BUTTON_MINUS NDOF Minus, Ndof-.NDOF_BUTTON_ESC NDOF Esc, NdofEsc.NDOF_BUTTON_ALT NDOF Alt, NdofAlt.NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.NDOF_BUTTON_1 NDOF Button 1, NdofB1.NDOF_BUTTON_2 NDOF Button 2, NdofB2.NDOF_BUTTON_3 NDOF Button 3, NdofB3.NDOF_BUTTON_4 NDOF Button 4, NdofB4.NDOF_BUTTON_5 NDOF Button 5, NdofB5.NDOF_BUTTON_6 NDOF Button 6, NdofB6.NDOF_BUTTON_7 NDOF Button 7, NdofB7.NDOF_BUTTON_8 NDOF Button 8, NdofB8.NDOF_BUTTON_9 NDOF Button 9, NdofB9.NDOF_BUTTON_10 NDOF Button 10, NdofB10.NDOF_BUTTON_A NDOF Button A, NdofBA.NDOF_BUTTON_B NDOF Button B, NdofBB.NDOF_BUTTON_C NDOF Button C, NdofBC. 
        :type key_modifier: enum in ['NONE', 'LEFTMOUSE', 'MIDDLEMOUSE', 'RIGHTMOUSE', 'BUTTON4MOUSE', 'BUTTON5MOUSE', 'BUTTON6MOUSE', 'BUTTON7MOUSE', 'ACTIONMOUSE', 'SELECTMOUSE', 'PEN', 'ERASER', 'MOUSEMOVE', 'INBETWEEN_MOUSEMOVE', 'TRACKPADPAN', 'TRACKPADZOOM', 'MOUSEROTATE', 'WHEELUPMOUSE', 'WHEELDOWNMOUSE', 'WHEELINMOUSE', 'WHEELOUTMOUSE', 'EVT_TWEAK_L', 'EVT_TWEAK_M', 'EVT_TWEAK_R', 'EVT_TWEAK_A', 'EVT_TWEAK_S', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'ZERO', 'ONE', 'TWO', 'THREE', 'FOUR', 'FIVE', 'SIX', 'SEVEN', 'EIGHT', 'NINE', 'LEFT_CTRL', 'LEFT_ALT', 'LEFT_SHIFT', 'RIGHT_ALT', 'RIGHT_CTRL', 'RIGHT_SHIFT', 'OSKEY', 'GRLESS', 'ESC', 'TAB', 'RET', 'SPACE', 'LINE_FEED', 'BACK_SPACE', 'DEL', 'SEMI_COLON', 'PERIOD', 'COMMA', 'QUOTE', 'ACCENT_GRAVE', 'MINUS', 'PLUS', 'SLASH', 'BACK_SLASH', 'EQUAL', 'LEFT_BRACKET', 'RIGHT_BRACKET', 'LEFT_ARROW', 'DOWN_ARROW', 'RIGHT_ARROW', 'UP_ARROW', 'NUMPAD_2', 'NUMPAD_4', 'NUMPAD_6', 'NUMPAD_8', 'NUMPAD_1', 'NUMPAD_3', 'NUMPAD_5', 'NUMPAD_7', 'NUMPAD_9', 'NUMPAD_PERIOD', 'NUMPAD_SLASH', 'NUMPAD_ASTERIX', 'NUMPAD_0', 'NUMPAD_MINUS', 'NUMPAD_ENTER', 'NUMPAD_PLUS', 'F1', 'F2', 'F3', 'F4', 'F5', 'F6', 'F7', 'F8', 'F9', 'F10', 'F11', 'F12', 'F13', 'F14', 'F15', 'F16', 'F17', 'F18', 'F19', 'PAUSE', 'INSERT', 'HOME', 'PAGE_UP', 'PAGE_DOWN', 'END', 'MEDIA_PLAY', 'MEDIA_STOP', 'MEDIA_FIRST', 'MEDIA_LAST', 'TEXTINPUT', 'WINDOW_DEACTIVATE', 'TIMER', 'TIMER0', 'TIMER1', 'TIMER2', 'TIMER_JOBS', 'TIMER_AUTOSAVE', 'TIMER_REPORT', 'TIMERREGION', 'NDOF_MOTION', 'NDOF_BUTTON_MENU', 'NDOF_BUTTON_FIT', 'NDOF_BUTTON_TOP', 'NDOF_BUTTON_BOTTOM', 'NDOF_BUTTON_LEFT', 'NDOF_BUTTON_RIGHT', 'NDOF_BUTTON_FRONT', 'NDOF_BUTTON_BACK', 'NDOF_BUTTON_ISO1', 'NDOF_BUTTON_ISO2', 'NDOF_BUTTON_ROLL_CW', 'NDOF_BUTTON_ROLL_CCW', 'NDOF_BUTTON_SPIN_CW', 'NDOF_BUTTON_SPIN_CCW', 'NDOF_BUTTON_TILT_CW', 'NDOF_BUTTON_TILT_CCW', 'NDOF_BUTTON_ROTATE', 'NDOF_BUTTON_PANZOOM', 'NDOF_BUTTON_DOMINANT', 'NDOF_BUTTON_PLUS', 'NDOF_BUTTON_MINUS', 'NDOF_BUTTON_ESC', 'NDOF_BUTTON_ALT', 'NDOF_BUTTON_SHIFT', 'NDOF_BUTTON_CTRL', 'NDOF_BUTTON_1', 'NDOF_BUTTON_2', 'NDOF_BUTTON_3', 'NDOF_BUTTON_4', 'NDOF_BUTTON_5', 'NDOF_BUTTON_6', 'NDOF_BUTTON_7', 'NDOF_BUTTON_8', 'NDOF_BUTTON_9', 'NDOF_BUTTON_10', 'NDOF_BUTTON_A', 'NDOF_BUTTON_B', 'NDOF_BUTTON_C'], (optional)
        :param head: At Head, Force item to be added at start (not end) of key map so that it doesn’t get blocked by an existing key map item 
        :type head: boolean, (optional)
        :rtype: KeyMapItem 
        :return:  Item, Added key map item 
        '''
        pass

    def new_modal(self,
                  propvalue,
                  type,
                  value,
                  any=False,
                  shift=False,
                  ctrl=False,
                  alt=False,
                  oskey=False,
                  key_modifier='NONE'):
        '''new_modal 

        :param propvalue: Property Value 
        :type propvalue: string, (never None)
        :param type: TypeNONE .LEFTMOUSE Left Mouse, LMB.MIDDLEMOUSE Middle Mouse, MMB.RIGHTMOUSE Right Mouse, RMB.BUTTON4MOUSE Button4 Mouse, MB4.BUTTON5MOUSE Button5 Mouse, MB5.BUTTON6MOUSE Button6 Mouse, MB6.BUTTON7MOUSE Button7 Mouse, MB7.ACTIONMOUSE Action Mouse, MBA.SELECTMOUSE Select Mouse, MBS.PEN Pen.ERASER Eraser.MOUSEMOVE Mouse Move, MsMov.INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.TRACKPADPAN Mouse/Trackpad Pan, MsPan.TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.MOUSEROTATE Mouse/Trackpad Rotate, MsRot.WHEELUPMOUSE Wheel Up, WhUp.WHEELDOWNMOUSE Wheel Down, WhDown.WHEELINMOUSE Wheel In, WhIn.WHEELOUTMOUSE Wheel Out, WhOut.EVT_TWEAK_L Tweak Left, TwkL.EVT_TWEAK_M Tweak Middle, TwkM.EVT_TWEAK_R Tweak Right, TwkR.EVT_TWEAK_A Tweak Action, TwkA.EVT_TWEAK_S Tweak Select, TwkS.A A.B B.C C.D D.E E.F F.G G.H H.I I.J J.K K.L L.M M.N N.O O.P P.Q Q.R R.S S.T T.U U.V V.W W.X X.Y Y.Z Z.ZERO 0.ONE 1.TWO 2.THREE 3.FOUR 4.FIVE 5.SIX 6.SEVEN 7.EIGHT 8.NINE 9.LEFT_CTRL Left Ctrl, CtrlL.LEFT_ALT Left Alt, AltL.LEFT_SHIFT Left Shift, ShiftL.RIGHT_ALT Right Alt, AltR.RIGHT_CTRL Right Ctrl, CtrlR.RIGHT_SHIFT Right Shift, ShiftR.OSKEY OS Key, Cmd.GRLESS Grless.ESC Esc.TAB Tab.RET Return, Enter.SPACE Spacebar, Space.LINE_FEED Line Feed.BACK_SPACE Back Space, BkSpace.DEL Delete, Del.SEMI_COLON ;.PERIOD ..COMMA ,.QUOTE “.ACCENT_GRAVE `.MINUS -.PLUS +.SLASH /.BACK_SLASH .EQUAL =.LEFT_BRACKET [.RIGHT_BRACKET ].LEFT_ARROW Left Arrow, ←.DOWN_ARROW Down Arrow, ↓.RIGHT_ARROW Right Arrow, →.UP_ARROW Up Arrow, ↑.NUMPAD_2 Numpad 2, Pad2.NUMPAD_4 Numpad 4, Pad4.NUMPAD_6 Numpad 6, Pad6.NUMPAD_8 Numpad 8, Pad8.NUMPAD_1 Numpad 1, Pad1.NUMPAD_3 Numpad 3, Pad3.NUMPAD_5 Numpad 5, Pad5.NUMPAD_7 Numpad 7, Pad7.NUMPAD_9 Numpad 9, Pad9.NUMPAD_PERIOD Numpad ., Pad..NUMPAD_SLASH Numpad /, Pad/.NUMPAD_ASTERIX Numpad *, Pad*.NUMPAD_0 Numpad 0, Pad0.NUMPAD_MINUS Numpad -, Pad-.NUMPAD_ENTER Numpad Enter, PadEnter.NUMPAD_PLUS Numpad +, Pad+.F1 F1.F2 F2.F3 F3.F4 F4.F5 F5.F6 F6.F7 F7.F8 F8.F9 F9.F10 F10.F11 F11.F12 F12.F13 F13.F14 F14.F15 F15.F16 F16.F17 F17.F18 F18.F19 F19.PAUSE Pause.INSERT Insert, Ins.HOME Home.PAGE_UP Page Up, PgUp.PAGE_DOWN Page Down, PgDown.END End.MEDIA_PLAY Media Play/Pause, >/||.MEDIA_STOP Media Stop, Stop.MEDIA_FIRST Media First, |<<.MEDIA_LAST Media Last, >>|.TEXTINPUT Text Input, TxtIn.WINDOW_DEACTIVATE Window Deactivate.TIMER Timer, Tmr.TIMER0 Timer 0, Tmr0.TIMER1 Timer 1, Tmr1.TIMER2 Timer 2, Tmr2.TIMER_JOBS Timer Jobs, TmrJob.TIMER_AUTOSAVE Timer Autosave, TmrSave.TIMER_REPORT Timer Report, TmrReport.TIMERREGION Timer Region, TmrReg.NDOF_MOTION NDOF Motion, NdofMov.NDOF_BUTTON_MENU NDOF Menu, NdofMenu.NDOF_BUTTON_FIT NDOF Fit, NdofFit.NDOF_BUTTON_TOP NDOF Top, Ndof↑.NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.NDOF_BUTTON_LEFT NDOF Left, Ndof←.NDOF_BUTTON_RIGHT NDOF Right, Ndof→.NDOF_BUTTON_FRONT NDOF Front, NdofFront.NDOF_BUTTON_BACK NDOF Back, NdofBack.NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.NDOF_BUTTON_PLUS NDOF Plus, Ndof+.NDOF_BUTTON_MINUS NDOF Minus, Ndof-.NDOF_BUTTON_ESC NDOF Esc, NdofEsc.NDOF_BUTTON_ALT NDOF Alt, NdofAlt.NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.NDOF_BUTTON_1 NDOF Button 1, NdofB1.NDOF_BUTTON_2 NDOF Button 2, NdofB2.NDOF_BUTTON_3 NDOF Button 3, NdofB3.NDOF_BUTTON_4 NDOF Button 4, NdofB4.NDOF_BUTTON_5 NDOF Button 5, NdofB5.NDOF_BUTTON_6 NDOF Button 6, NdofB6.NDOF_BUTTON_7 NDOF Button 7, NdofB7.NDOF_BUTTON_8 NDOF Button 8, NdofB8.NDOF_BUTTON_9 NDOF Button 9, NdofB9.NDOF_BUTTON_10 NDOF Button 10, NdofB10.NDOF_BUTTON_A NDOF Button A, NdofBA.NDOF_BUTTON_B NDOF Button B, NdofBB.NDOF_BUTTON_C NDOF Button C, NdofBC. 
        :type type: enum in ['NONE', 'LEFTMOUSE', 'MIDDLEMOUSE', 'RIGHTMOUSE', 'BUTTON4MOUSE', 'BUTTON5MOUSE', 'BUTTON6MOUSE', 'BUTTON7MOUSE', 'ACTIONMOUSE', 'SELECTMOUSE', 'PEN', 'ERASER', 'MOUSEMOVE', 'INBETWEEN_MOUSEMOVE', 'TRACKPADPAN', 'TRACKPADZOOM', 'MOUSEROTATE', 'WHEELUPMOUSE', 'WHEELDOWNMOUSE', 'WHEELINMOUSE', 'WHEELOUTMOUSE', 'EVT_TWEAK_L', 'EVT_TWEAK_M', 'EVT_TWEAK_R', 'EVT_TWEAK_A', 'EVT_TWEAK_S', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'ZERO', 'ONE', 'TWO', 'THREE', 'FOUR', 'FIVE', 'SIX', 'SEVEN', 'EIGHT', 'NINE', 'LEFT_CTRL', 'LEFT_ALT', 'LEFT_SHIFT', 'RIGHT_ALT', 'RIGHT_CTRL', 'RIGHT_SHIFT', 'OSKEY', 'GRLESS', 'ESC', 'TAB', 'RET', 'SPACE', 'LINE_FEED', 'BACK_SPACE', 'DEL', 'SEMI_COLON', 'PERIOD', 'COMMA', 'QUOTE', 'ACCENT_GRAVE', 'MINUS', 'PLUS', 'SLASH', 'BACK_SLASH', 'EQUAL', 'LEFT_BRACKET', 'RIGHT_BRACKET', 'LEFT_ARROW', 'DOWN_ARROW', 'RIGHT_ARROW', 'UP_ARROW', 'NUMPAD_2', 'NUMPAD_4', 'NUMPAD_6', 'NUMPAD_8', 'NUMPAD_1', 'NUMPAD_3', 'NUMPAD_5', 'NUMPAD_7', 'NUMPAD_9', 'NUMPAD_PERIOD', 'NUMPAD_SLASH', 'NUMPAD_ASTERIX', 'NUMPAD_0', 'NUMPAD_MINUS', 'NUMPAD_ENTER', 'NUMPAD_PLUS', 'F1', 'F2', 'F3', 'F4', 'F5', 'F6', 'F7', 'F8', 'F9', 'F10', 'F11', 'F12', 'F13', 'F14', 'F15', 'F16', 'F17', 'F18', 'F19', 'PAUSE', 'INSERT', 'HOME', 'PAGE_UP', 'PAGE_DOWN', 'END', 'MEDIA_PLAY', 'MEDIA_STOP', 'MEDIA_FIRST', 'MEDIA_LAST', 'TEXTINPUT', 'WINDOW_DEACTIVATE', 'TIMER', 'TIMER0', 'TIMER1', 'TIMER2', 'TIMER_JOBS', 'TIMER_AUTOSAVE', 'TIMER_REPORT', 'TIMERREGION', 'NDOF_MOTION', 'NDOF_BUTTON_MENU', 'NDOF_BUTTON_FIT', 'NDOF_BUTTON_TOP', 'NDOF_BUTTON_BOTTOM', 'NDOF_BUTTON_LEFT', 'NDOF_BUTTON_RIGHT', 'NDOF_BUTTON_FRONT', 'NDOF_BUTTON_BACK', 'NDOF_BUTTON_ISO1', 'NDOF_BUTTON_ISO2', 'NDOF_BUTTON_ROLL_CW', 'NDOF_BUTTON_ROLL_CCW', 'NDOF_BUTTON_SPIN_CW', 'NDOF_BUTTON_SPIN_CCW', 'NDOF_BUTTON_TILT_CW', 'NDOF_BUTTON_TILT_CCW', 'NDOF_BUTTON_ROTATE', 'NDOF_BUTTON_PANZOOM', 'NDOF_BUTTON_DOMINANT', 'NDOF_BUTTON_PLUS', 'NDOF_BUTTON_MINUS', 'NDOF_BUTTON_ESC', 'NDOF_BUTTON_ALT', 'NDOF_BUTTON_SHIFT', 'NDOF_BUTTON_CTRL', 'NDOF_BUTTON_1', 'NDOF_BUTTON_2', 'NDOF_BUTTON_3', 'NDOF_BUTTON_4', 'NDOF_BUTTON_5', 'NDOF_BUTTON_6', 'NDOF_BUTTON_7', 'NDOF_BUTTON_8', 'NDOF_BUTTON_9', 'NDOF_BUTTON_10', 'NDOF_BUTTON_A', 'NDOF_BUTTON_B', 'NDOF_BUTTON_C']
        :param value: Value 
        :type value: enum in ['ANY', 'NOTHING', 'PRESS', 'RELEASE', 'CLICK', 'DOUBLE_CLICK', 'NORTH', 'NORTH_EAST', 'EAST', 'SOUTH_EAST', 'SOUTH', 'SOUTH_WEST', 'WEST', 'NORTH_WEST']
        :param any: Any 
        :type any: boolean, (optional)
        :param shift: Shift 
        :type shift: boolean, (optional)
        :param ctrl: Ctrl 
        :type ctrl: boolean, (optional)
        :param alt: Alt 
        :type alt: boolean, (optional)
        :param oskey: OS Key 
        :type oskey: boolean, (optional)
        :param key_modifier: Key ModifierNONE .LEFTMOUSE Left Mouse, LMB.MIDDLEMOUSE Middle Mouse, MMB.RIGHTMOUSE Right Mouse, RMB.BUTTON4MOUSE Button4 Mouse, MB4.BUTTON5MOUSE Button5 Mouse, MB5.BUTTON6MOUSE Button6 Mouse, MB6.BUTTON7MOUSE Button7 Mouse, MB7.ACTIONMOUSE Action Mouse, MBA.SELECTMOUSE Select Mouse, MBS.PEN Pen.ERASER Eraser.MOUSEMOVE Mouse Move, MsMov.INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.TRACKPADPAN Mouse/Trackpad Pan, MsPan.TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.MOUSEROTATE Mouse/Trackpad Rotate, MsRot.WHEELUPMOUSE Wheel Up, WhUp.WHEELDOWNMOUSE Wheel Down, WhDown.WHEELINMOUSE Wheel In, WhIn.WHEELOUTMOUSE Wheel Out, WhOut.EVT_TWEAK_L Tweak Left, TwkL.EVT_TWEAK_M Tweak Middle, TwkM.EVT_TWEAK_R Tweak Right, TwkR.EVT_TWEAK_A Tweak Action, TwkA.EVT_TWEAK_S Tweak Select, TwkS.A A.B B.C C.D D.E E.F F.G G.H H.I I.J J.K K.L L.M M.N N.O O.P P.Q Q.R R.S S.T T.U U.V V.W W.X X.Y Y.Z Z.ZERO 0.ONE 1.TWO 2.THREE 3.FOUR 4.FIVE 5.SIX 6.SEVEN 7.EIGHT 8.NINE 9.LEFT_CTRL Left Ctrl, CtrlL.LEFT_ALT Left Alt, AltL.LEFT_SHIFT Left Shift, ShiftL.RIGHT_ALT Right Alt, AltR.RIGHT_CTRL Right Ctrl, CtrlR.RIGHT_SHIFT Right Shift, ShiftR.OSKEY OS Key, Cmd.GRLESS Grless.ESC Esc.TAB Tab.RET Return, Enter.SPACE Spacebar, Space.LINE_FEED Line Feed.BACK_SPACE Back Space, BkSpace.DEL Delete, Del.SEMI_COLON ;.PERIOD ..COMMA ,.QUOTE “.ACCENT_GRAVE `.MINUS -.PLUS +.SLASH /.BACK_SLASH .EQUAL =.LEFT_BRACKET [.RIGHT_BRACKET ].LEFT_ARROW Left Arrow, ←.DOWN_ARROW Down Arrow, ↓.RIGHT_ARROW Right Arrow, →.UP_ARROW Up Arrow, ↑.NUMPAD_2 Numpad 2, Pad2.NUMPAD_4 Numpad 4, Pad4.NUMPAD_6 Numpad 6, Pad6.NUMPAD_8 Numpad 8, Pad8.NUMPAD_1 Numpad 1, Pad1.NUMPAD_3 Numpad 3, Pad3.NUMPAD_5 Numpad 5, Pad5.NUMPAD_7 Numpad 7, Pad7.NUMPAD_9 Numpad 9, Pad9.NUMPAD_PERIOD Numpad ., Pad..NUMPAD_SLASH Numpad /, Pad/.NUMPAD_ASTERIX Numpad *, Pad*.NUMPAD_0 Numpad 0, Pad0.NUMPAD_MINUS Numpad -, Pad-.NUMPAD_ENTER Numpad Enter, PadEnter.NUMPAD_PLUS Numpad +, Pad+.F1 F1.F2 F2.F3 F3.F4 F4.F5 F5.F6 F6.F7 F7.F8 F8.F9 F9.F10 F10.F11 F11.F12 F12.F13 F13.F14 F14.F15 F15.F16 F16.F17 F17.F18 F18.F19 F19.PAUSE Pause.INSERT Insert, Ins.HOME Home.PAGE_UP Page Up, PgUp.PAGE_DOWN Page Down, PgDown.END End.MEDIA_PLAY Media Play/Pause, >/||.MEDIA_STOP Media Stop, Stop.MEDIA_FIRST Media First, |<<.MEDIA_LAST Media Last, >>|.TEXTINPUT Text Input, TxtIn.WINDOW_DEACTIVATE Window Deactivate.TIMER Timer, Tmr.TIMER0 Timer 0, Tmr0.TIMER1 Timer 1, Tmr1.TIMER2 Timer 2, Tmr2.TIMER_JOBS Timer Jobs, TmrJob.TIMER_AUTOSAVE Timer Autosave, TmrSave.TIMER_REPORT Timer Report, TmrReport.TIMERREGION Timer Region, TmrReg.NDOF_MOTION NDOF Motion, NdofMov.NDOF_BUTTON_MENU NDOF Menu, NdofMenu.NDOF_BUTTON_FIT NDOF Fit, NdofFit.NDOF_BUTTON_TOP NDOF Top, Ndof↑.NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.NDOF_BUTTON_LEFT NDOF Left, Ndof←.NDOF_BUTTON_RIGHT NDOF Right, Ndof→.NDOF_BUTTON_FRONT NDOF Front, NdofFront.NDOF_BUTTON_BACK NDOF Back, NdofBack.NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.NDOF_BUTTON_PLUS NDOF Plus, Ndof+.NDOF_BUTTON_MINUS NDOF Minus, Ndof-.NDOF_BUTTON_ESC NDOF Esc, NdofEsc.NDOF_BUTTON_ALT NDOF Alt, NdofAlt.NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.NDOF_BUTTON_1 NDOF Button 1, NdofB1.NDOF_BUTTON_2 NDOF Button 2, NdofB2.NDOF_BUTTON_3 NDOF Button 3, NdofB3.NDOF_BUTTON_4 NDOF Button 4, NdofB4.NDOF_BUTTON_5 NDOF Button 5, NdofB5.NDOF_BUTTON_6 NDOF Button 6, NdofB6.NDOF_BUTTON_7 NDOF Button 7, NdofB7.NDOF_BUTTON_8 NDOF Button 8, NdofB8.NDOF_BUTTON_9 NDOF Button 9, NdofB9.NDOF_BUTTON_10 NDOF Button 10, NdofB10.NDOF_BUTTON_A NDOF Button A, NdofBA.NDOF_BUTTON_B NDOF Button B, NdofBB.NDOF_BUTTON_C NDOF Button C, NdofBC. 
        :type key_modifier: enum in ['NONE', 'LEFTMOUSE', 'MIDDLEMOUSE', 'RIGHTMOUSE', 'BUTTON4MOUSE', 'BUTTON5MOUSE', 'BUTTON6MOUSE', 'BUTTON7MOUSE', 'ACTIONMOUSE', 'SELECTMOUSE', 'PEN', 'ERASER', 'MOUSEMOVE', 'INBETWEEN_MOUSEMOVE', 'TRACKPADPAN', 'TRACKPADZOOM', 'MOUSEROTATE', 'WHEELUPMOUSE', 'WHEELDOWNMOUSE', 'WHEELINMOUSE', 'WHEELOUTMOUSE', 'EVT_TWEAK_L', 'EVT_TWEAK_M', 'EVT_TWEAK_R', 'EVT_TWEAK_A', 'EVT_TWEAK_S', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'ZERO', 'ONE', 'TWO', 'THREE', 'FOUR', 'FIVE', 'SIX', 'SEVEN', 'EIGHT', 'NINE', 'LEFT_CTRL', 'LEFT_ALT', 'LEFT_SHIFT', 'RIGHT_ALT', 'RIGHT_CTRL', 'RIGHT_SHIFT', 'OSKEY', 'GRLESS', 'ESC', 'TAB', 'RET', 'SPACE', 'LINE_FEED', 'BACK_SPACE', 'DEL', 'SEMI_COLON', 'PERIOD', 'COMMA', 'QUOTE', 'ACCENT_GRAVE', 'MINUS', 'PLUS', 'SLASH', 'BACK_SLASH', 'EQUAL', 'LEFT_BRACKET', 'RIGHT_BRACKET', 'LEFT_ARROW', 'DOWN_ARROW', 'RIGHT_ARROW', 'UP_ARROW', 'NUMPAD_2', 'NUMPAD_4', 'NUMPAD_6', 'NUMPAD_8', 'NUMPAD_1', 'NUMPAD_3', 'NUMPAD_5', 'NUMPAD_7', 'NUMPAD_9', 'NUMPAD_PERIOD', 'NUMPAD_SLASH', 'NUMPAD_ASTERIX', 'NUMPAD_0', 'NUMPAD_MINUS', 'NUMPAD_ENTER', 'NUMPAD_PLUS', 'F1', 'F2', 'F3', 'F4', 'F5', 'F6', 'F7', 'F8', 'F9', 'F10', 'F11', 'F12', 'F13', 'F14', 'F15', 'F16', 'F17', 'F18', 'F19', 'PAUSE', 'INSERT', 'HOME', 'PAGE_UP', 'PAGE_DOWN', 'END', 'MEDIA_PLAY', 'MEDIA_STOP', 'MEDIA_FIRST', 'MEDIA_LAST', 'TEXTINPUT', 'WINDOW_DEACTIVATE', 'TIMER', 'TIMER0', 'TIMER1', 'TIMER2', 'TIMER_JOBS', 'TIMER_AUTOSAVE', 'TIMER_REPORT', 'TIMERREGION', 'NDOF_MOTION', 'NDOF_BUTTON_MENU', 'NDOF_BUTTON_FIT', 'NDOF_BUTTON_TOP', 'NDOF_BUTTON_BOTTOM', 'NDOF_BUTTON_LEFT', 'NDOF_BUTTON_RIGHT', 'NDOF_BUTTON_FRONT', 'NDOF_BUTTON_BACK', 'NDOF_BUTTON_ISO1', 'NDOF_BUTTON_ISO2', 'NDOF_BUTTON_ROLL_CW', 'NDOF_BUTTON_ROLL_CCW', 'NDOF_BUTTON_SPIN_CW', 'NDOF_BUTTON_SPIN_CCW', 'NDOF_BUTTON_TILT_CW', 'NDOF_BUTTON_TILT_CCW', 'NDOF_BUTTON_ROTATE', 'NDOF_BUTTON_PANZOOM', 'NDOF_BUTTON_DOMINANT', 'NDOF_BUTTON_PLUS', 'NDOF_BUTTON_MINUS', 'NDOF_BUTTON_ESC', 'NDOF_BUTTON_ALT', 'NDOF_BUTTON_SHIFT', 'NDOF_BUTTON_CTRL', 'NDOF_BUTTON_1', 'NDOF_BUTTON_2', 'NDOF_BUTTON_3', 'NDOF_BUTTON_4', 'NDOF_BUTTON_5', 'NDOF_BUTTON_6', 'NDOF_BUTTON_7', 'NDOF_BUTTON_8', 'NDOF_BUTTON_9', 'NDOF_BUTTON_10', 'NDOF_BUTTON_A', 'NDOF_BUTTON_B', 'NDOF_BUTTON_C'], (optional)
        :rtype: KeyMapItem 
        :return:  Item, Added key map item 
        '''
        pass

    def remove(self, item):
        '''remove 

        :param item: Item 
        :type item: KeyMapItem, (never None)
        '''
        pass

    def from_id(self, id):
        '''from_id 

        :param id: id, ID of the item 
        :type id: int in [-inf, inf]
        :rtype: KeyMapItem 
        :return:  Item 
        '''
        pass


class KeyMaps:
    '''Collection of keymaps '''

    def new(self, name, space_type='EMPTY', region_type='WINDOW', modal=False):
        '''new 

        :param name: Name 
        :type name: string, (never None)
        :param space_type: Space TypeEMPTY Empty.VIEW_3D 3D View, 3D viewport.TIMELINE Timeline, Timeline and playback controls.GRAPH_EDITOR Graph Editor, Edit drivers and keyframe interpolation.DOPESHEET_EDITOR Dope Sheet, Adjust timing of keyframes.NLA_EDITOR NLA Editor, Combine and layer Actions.IMAGE_EDITOR UV/Image Editor, View and edit images and UV Maps.SEQUENCE_EDITOR Video Sequence Editor, Video editing tools.CLIP_EDITOR Movie Clip Editor, Motion tracking tools.TEXT_EDITOR Text Editor, Edit scripts and in-file documentation.NODE_EDITOR Node Editor, Editor for node-based shading and compositing tools.LOGIC_EDITOR Logic Editor, Game logic editing.PROPERTIES Properties, Edit properties of active object and related data-blocks.OUTLINER Outliner, Overview of scene graph and all available data-blocks.USER_PREFERENCES User Preferences, Edit persistent configuration settings.INFO Info, Main menu bar and list of error messages (drag down to expand and display).FILE_BROWSER File Browser, Browse for files and assets.CONSOLE Python Console, Interactive programmatic console for advanced editing and script development. 
        :type space_type: enum in ['EMPTY', 'VIEW_3D', 'TIMELINE', 'GRAPH_EDITOR', 'DOPESHEET_EDITOR', 'NLA_EDITOR', 'IMAGE_EDITOR', 'SEQUENCE_EDITOR', 'CLIP_EDITOR', 'TEXT_EDITOR', 'NODE_EDITOR', 'LOGIC_EDITOR', 'PROPERTIES', 'OUTLINER', 'USER_PREFERENCES', 'INFO', 'FILE_BROWSER', 'CONSOLE'], (optional)
        :param region_type: Region Type 
        :type region_type: enum in ['WINDOW', 'HEADER', 'CHANNELS', 'TEMPORARY', 'UI', 'TOOLS', 'TOOL_PROPS', 'PREVIEW'], (optional)
        :param modal: Modal 
        :type modal: boolean, (optional)
        :rtype: KeyMap 
        :return:  Key Map, Added key map 
        '''
        pass

    def remove(self, keymap):
        '''remove 

        :param keymap: Key Map, Removed key map 
        :type keymap: KeyMap, (never None)
        '''
        pass

    def find(self, name, space_type='EMPTY', region_type='WINDOW'):
        '''find 

        :param name: Name 
        :type name: string, (never None)
        :param space_type: Space TypeEMPTY Empty.VIEW_3D 3D View, 3D viewport.TIMELINE Timeline, Timeline and playback controls.GRAPH_EDITOR Graph Editor, Edit drivers and keyframe interpolation.DOPESHEET_EDITOR Dope Sheet, Adjust timing of keyframes.NLA_EDITOR NLA Editor, Combine and layer Actions.IMAGE_EDITOR UV/Image Editor, View and edit images and UV Maps.SEQUENCE_EDITOR Video Sequence Editor, Video editing tools.CLIP_EDITOR Movie Clip Editor, Motion tracking tools.TEXT_EDITOR Text Editor, Edit scripts and in-file documentation.NODE_EDITOR Node Editor, Editor for node-based shading and compositing tools.LOGIC_EDITOR Logic Editor, Game logic editing.PROPERTIES Properties, Edit properties of active object and related data-blocks.OUTLINER Outliner, Overview of scene graph and all available data-blocks.USER_PREFERENCES User Preferences, Edit persistent configuration settings.INFO Info, Main menu bar and list of error messages (drag down to expand and display).FILE_BROWSER File Browser, Browse for files and assets.CONSOLE Python Console, Interactive programmatic console for advanced editing and script development. 
        :type space_type: enum in ['EMPTY', 'VIEW_3D', 'TIMELINE', 'GRAPH_EDITOR', 'DOPESHEET_EDITOR', 'NLA_EDITOR', 'IMAGE_EDITOR', 'SEQUENCE_EDITOR', 'CLIP_EDITOR', 'TEXT_EDITOR', 'NODE_EDITOR', 'LOGIC_EDITOR', 'PROPERTIES', 'OUTLINER', 'USER_PREFERENCES', 'INFO', 'FILE_BROWSER', 'CONSOLE'], (optional)
        :param region_type: Region Type 
        :type region_type: enum in ['WINDOW', 'HEADER', 'CHANNELS', 'TEMPORARY', 'UI', 'TOOLS', 'TOOL_PROPS', 'PREVIEW'], (optional)
        :rtype: KeyMap 
        :return:  Key Map, Corresponding key map 
        '''
        pass

    def find_modal(self, name):
        '''find_modal 

        :param name: Operator Name 
        :type name: string, (never None)
        :rtype: KeyMap 
        :return:  Key Map, Corresponding key map 
        '''
        pass


class KeyboardSensor:
    '''Sensor to detect keyboard events '''

    key = None
    '''

    :type:  enum in [‘NONE’, ‘LEFTMOUSE’, ‘MIDDLEMOUSE’, ‘RIGHTMOUSE’, ‘BUTTON4MOUSE’, ‘BUTTON5MOUSE’, ‘BUTTON6MOUSE’, ‘BUTTON7MOUSE’, ‘ACTIONMOUSE’, ‘SELECTMOUSE’, ‘PEN’, ‘ERASER’, ‘MOUSEMOVE’, ‘INBETWEEN_MOUSEMOVE’, ‘TRACKPADPAN’, ‘TRACKPADZOOM’, ‘MOUSEROTATE’, ‘WHEELUPMOUSE’, ‘WHEELDOWNMOUSE’, ‘WHEELINMOUSE’, ‘WHEELOUTMOUSE’, ‘EVT_TWEAK_L’, ‘EVT_TWEAK_M’, ‘EVT_TWEAK_R’, ‘EVT_TWEAK_A’, ‘EVT_TWEAK_S’, ‘A’, ‘B’, ‘C’, ‘D’, ‘E’, ‘F’, ‘G’, ‘H’, ‘I’, ‘J’, ‘K’, ‘L’, ‘M’, ‘N’, ‘O’, ‘P’, ‘Q’, ‘R’, ‘S’, ‘T’, ‘U’, ‘V’, ‘W’, ‘X’, ‘Y’, ‘Z’, ‘ZERO’, ‘ONE’, ‘TWO’, ‘THREE’, ‘FOUR’, ‘FIVE’, ‘SIX’, ‘SEVEN’, ‘EIGHT’, ‘NINE’, ‘LEFT_CTRL’, ‘LEFT_ALT’, ‘LEFT_SHIFT’, ‘RIGHT_ALT’, ‘RIGHT_CTRL’, ‘RIGHT_SHIFT’, ‘OSKEY’, ‘GRLESS’, ‘ESC’, ‘TAB’, ‘RET’, ‘SPACE’, ‘LINE_FEED’, ‘BACK_SPACE’, ‘DEL’, ‘SEMI_COLON’, ‘PERIOD’, ‘COMMA’, ‘QUOTE’, ‘ACCENT_GRAVE’, ‘MINUS’, ‘PLUS’, ‘SLASH’, ‘BACK_SLASH’, ‘EQUAL’, ‘LEFT_BRACKET’, ‘RIGHT_BRACKET’, ‘LEFT_ARROW’, ‘DOWN_ARROW’, ‘RIGHT_ARROW’, ‘UP_ARROW’, ‘NUMPAD_2’, ‘NUMPAD_4’, ‘NUMPAD_6’, ‘NUMPAD_8’, ‘NUMPAD_1’, ‘NUMPAD_3’, ‘NUMPAD_5’, ‘NUMPAD_7’, ‘NUMPAD_9’, ‘NUMPAD_PERIOD’, ‘NUMPAD_SLASH’, ‘NUMPAD_ASTERIX’, ‘NUMPAD_0’, ‘NUMPAD_MINUS’, ‘NUMPAD_ENTER’, ‘NUMPAD_PLUS’, ‘F1’, ‘F2’, ‘F3’, ‘F4’, ‘F5’, ‘F6’, ‘F7’, ‘F8’, ‘F9’, ‘F10’, ‘F11’, ‘F12’, ‘F13’, ‘F14’, ‘F15’, ‘F16’, ‘F17’, ‘F18’, ‘F19’, ‘PAUSE’, ‘INSERT’, ‘HOME’, ‘PAGE_UP’, ‘PAGE_DOWN’, ‘END’, ‘MEDIA_PLAY’, ‘MEDIA_STOP’, ‘MEDIA_FIRST’, ‘MEDIA_LAST’, ‘TEXTINPUT’, ‘WINDOW_DEACTIVATE’, ‘TIMER’, ‘TIMER0’, ‘TIMER1’, ‘TIMER2’, ‘TIMER_JOBS’, ‘TIMER_AUTOSAVE’, ‘TIMER_REPORT’, ‘TIMERREGION’, ‘NDOF_MOTION’, ‘NDOF_BUTTON_MENU’, ‘NDOF_BUTTON_FIT’, ‘NDOF_BUTTON_TOP’, ‘NDOF_BUTTON_BOTTOM’, ‘NDOF_BUTTON_LEFT’, ‘NDOF_BUTTON_RIGHT’, ‘NDOF_BUTTON_FRONT’, ‘NDOF_BUTTON_BACK’, ‘NDOF_BUTTON_ISO1’, ‘NDOF_BUTTON_ISO2’, ‘NDOF_BUTTON_ROLL_CW’, ‘NDOF_BUTTON_ROLL_CCW’, ‘NDOF_BUTTON_SPIN_CW’, ‘NDOF_BUTTON_SPIN_CCW’, ‘NDOF_BUTTON_TILT_CW’, ‘NDOF_BUTTON_TILT_CCW’, ‘NDOF_BUTTON_ROTATE’, ‘NDOF_BUTTON_PANZOOM’, ‘NDOF_BUTTON_DOMINANT’, ‘NDOF_BUTTON_PLUS’, ‘NDOF_BUTTON_MINUS’, ‘NDOF_BUTTON_ESC’, ‘NDOF_BUTTON_ALT’, ‘NDOF_BUTTON_SHIFT’, ‘NDOF_BUTTON_CTRL’, ‘NDOF_BUTTON_1’, ‘NDOF_BUTTON_2’, ‘NDOF_BUTTON_3’, ‘NDOF_BUTTON_4’, ‘NDOF_BUTTON_5’, ‘NDOF_BUTTON_6’, ‘NDOF_BUTTON_7’, ‘NDOF_BUTTON_8’, ‘NDOF_BUTTON_9’, ‘NDOF_BUTTON_10’, ‘NDOF_BUTTON_A’, ‘NDOF_BUTTON_B’, ‘NDOF_BUTTON_C’], default ‘NONE’ 
    '''

    log = None
    '''Property that indicates whether to log keystrokes as a string 

    :type:  string, default “”, (never None) 
    '''

    modifier_key_1 = None
    '''Modifier key code 

    :type:  enum in [‘NONE’, ‘LEFTMOUSE’, ‘MIDDLEMOUSE’, ‘RIGHTMOUSE’, ‘BUTTON4MOUSE’, ‘BUTTON5MOUSE’, ‘BUTTON6MOUSE’, ‘BUTTON7MOUSE’, ‘ACTIONMOUSE’, ‘SELECTMOUSE’, ‘PEN’, ‘ERASER’, ‘MOUSEMOVE’, ‘INBETWEEN_MOUSEMOVE’, ‘TRACKPADPAN’, ‘TRACKPADZOOM’, ‘MOUSEROTATE’, ‘WHEELUPMOUSE’, ‘WHEELDOWNMOUSE’, ‘WHEELINMOUSE’, ‘WHEELOUTMOUSE’, ‘EVT_TWEAK_L’, ‘EVT_TWEAK_M’, ‘EVT_TWEAK_R’, ‘EVT_TWEAK_A’, ‘EVT_TWEAK_S’, ‘A’, ‘B’, ‘C’, ‘D’, ‘E’, ‘F’, ‘G’, ‘H’, ‘I’, ‘J’, ‘K’, ‘L’, ‘M’, ‘N’, ‘O’, ‘P’, ‘Q’, ‘R’, ‘S’, ‘T’, ‘U’, ‘V’, ‘W’, ‘X’, ‘Y’, ‘Z’, ‘ZERO’, ‘ONE’, ‘TWO’, ‘THREE’, ‘FOUR’, ‘FIVE’, ‘SIX’, ‘SEVEN’, ‘EIGHT’, ‘NINE’, ‘LEFT_CTRL’, ‘LEFT_ALT’, ‘LEFT_SHIFT’, ‘RIGHT_ALT’, ‘RIGHT_CTRL’, ‘RIGHT_SHIFT’, ‘OSKEY’, ‘GRLESS’, ‘ESC’, ‘TAB’, ‘RET’, ‘SPACE’, ‘LINE_FEED’, ‘BACK_SPACE’, ‘DEL’, ‘SEMI_COLON’, ‘PERIOD’, ‘COMMA’, ‘QUOTE’, ‘ACCENT_GRAVE’, ‘MINUS’, ‘PLUS’, ‘SLASH’, ‘BACK_SLASH’, ‘EQUAL’, ‘LEFT_BRACKET’, ‘RIGHT_BRACKET’, ‘LEFT_ARROW’, ‘DOWN_ARROW’, ‘RIGHT_ARROW’, ‘UP_ARROW’, ‘NUMPAD_2’, ‘NUMPAD_4’, ‘NUMPAD_6’, ‘NUMPAD_8’, ‘NUMPAD_1’, ‘NUMPAD_3’, ‘NUMPAD_5’, ‘NUMPAD_7’, ‘NUMPAD_9’, ‘NUMPAD_PERIOD’, ‘NUMPAD_SLASH’, ‘NUMPAD_ASTERIX’, ‘NUMPAD_0’, ‘NUMPAD_MINUS’, ‘NUMPAD_ENTER’, ‘NUMPAD_PLUS’, ‘F1’, ‘F2’, ‘F3’, ‘F4’, ‘F5’, ‘F6’, ‘F7’, ‘F8’, ‘F9’, ‘F10’, ‘F11’, ‘F12’, ‘F13’, ‘F14’, ‘F15’, ‘F16’, ‘F17’, ‘F18’, ‘F19’, ‘PAUSE’, ‘INSERT’, ‘HOME’, ‘PAGE_UP’, ‘PAGE_DOWN’, ‘END’, ‘MEDIA_PLAY’, ‘MEDIA_STOP’, ‘MEDIA_FIRST’, ‘MEDIA_LAST’, ‘TEXTINPUT’, ‘WINDOW_DEACTIVATE’, ‘TIMER’, ‘TIMER0’, ‘TIMER1’, ‘TIMER2’, ‘TIMER_JOBS’, ‘TIMER_AUTOSAVE’, ‘TIMER_REPORT’, ‘TIMERREGION’, ‘NDOF_MOTION’, ‘NDOF_BUTTON_MENU’, ‘NDOF_BUTTON_FIT’, ‘NDOF_BUTTON_TOP’, ‘NDOF_BUTTON_BOTTOM’, ‘NDOF_BUTTON_LEFT’, ‘NDOF_BUTTON_RIGHT’, ‘NDOF_BUTTON_FRONT’, ‘NDOF_BUTTON_BACK’, ‘NDOF_BUTTON_ISO1’, ‘NDOF_BUTTON_ISO2’, ‘NDOF_BUTTON_ROLL_CW’, ‘NDOF_BUTTON_ROLL_CCW’, ‘NDOF_BUTTON_SPIN_CW’, ‘NDOF_BUTTON_SPIN_CCW’, ‘NDOF_BUTTON_TILT_CW’, ‘NDOF_BUTTON_TILT_CCW’, ‘NDOF_BUTTON_ROTATE’, ‘NDOF_BUTTON_PANZOOM’, ‘NDOF_BUTTON_DOMINANT’, ‘NDOF_BUTTON_PLUS’, ‘NDOF_BUTTON_MINUS’, ‘NDOF_BUTTON_ESC’, ‘NDOF_BUTTON_ALT’, ‘NDOF_BUTTON_SHIFT’, ‘NDOF_BUTTON_CTRL’, ‘NDOF_BUTTON_1’, ‘NDOF_BUTTON_2’, ‘NDOF_BUTTON_3’, ‘NDOF_BUTTON_4’, ‘NDOF_BUTTON_5’, ‘NDOF_BUTTON_6’, ‘NDOF_BUTTON_7’, ‘NDOF_BUTTON_8’, ‘NDOF_BUTTON_9’, ‘NDOF_BUTTON_10’, ‘NDOF_BUTTON_A’, ‘NDOF_BUTTON_B’, ‘NDOF_BUTTON_C’], default ‘NONE’ 
    '''

    modifier_key_2 = None
    '''Modifier key code 

    :type:  enum in [‘NONE’, ‘LEFTMOUSE’, ‘MIDDLEMOUSE’, ‘RIGHTMOUSE’, ‘BUTTON4MOUSE’, ‘BUTTON5MOUSE’, ‘BUTTON6MOUSE’, ‘BUTTON7MOUSE’, ‘ACTIONMOUSE’, ‘SELECTMOUSE’, ‘PEN’, ‘ERASER’, ‘MOUSEMOVE’, ‘INBETWEEN_MOUSEMOVE’, ‘TRACKPADPAN’, ‘TRACKPADZOOM’, ‘MOUSEROTATE’, ‘WHEELUPMOUSE’, ‘WHEELDOWNMOUSE’, ‘WHEELINMOUSE’, ‘WHEELOUTMOUSE’, ‘EVT_TWEAK_L’, ‘EVT_TWEAK_M’, ‘EVT_TWEAK_R’, ‘EVT_TWEAK_A’, ‘EVT_TWEAK_S’, ‘A’, ‘B’, ‘C’, ‘D’, ‘E’, ‘F’, ‘G’, ‘H’, ‘I’, ‘J’, ‘K’, ‘L’, ‘M’, ‘N’, ‘O’, ‘P’, ‘Q’, ‘R’, ‘S’, ‘T’, ‘U’, ‘V’, ‘W’, ‘X’, ‘Y’, ‘Z’, ‘ZERO’, ‘ONE’, ‘TWO’, ‘THREE’, ‘FOUR’, ‘FIVE’, ‘SIX’, ‘SEVEN’, ‘EIGHT’, ‘NINE’, ‘LEFT_CTRL’, ‘LEFT_ALT’, ‘LEFT_SHIFT’, ‘RIGHT_ALT’, ‘RIGHT_CTRL’, ‘RIGHT_SHIFT’, ‘OSKEY’, ‘GRLESS’, ‘ESC’, ‘TAB’, ‘RET’, ‘SPACE’, ‘LINE_FEED’, ‘BACK_SPACE’, ‘DEL’, ‘SEMI_COLON’, ‘PERIOD’, ‘COMMA’, ‘QUOTE’, ‘ACCENT_GRAVE’, ‘MINUS’, ‘PLUS’, ‘SLASH’, ‘BACK_SLASH’, ‘EQUAL’, ‘LEFT_BRACKET’, ‘RIGHT_BRACKET’, ‘LEFT_ARROW’, ‘DOWN_ARROW’, ‘RIGHT_ARROW’, ‘UP_ARROW’, ‘NUMPAD_2’, ‘NUMPAD_4’, ‘NUMPAD_6’, ‘NUMPAD_8’, ‘NUMPAD_1’, ‘NUMPAD_3’, ‘NUMPAD_5’, ‘NUMPAD_7’, ‘NUMPAD_9’, ‘NUMPAD_PERIOD’, ‘NUMPAD_SLASH’, ‘NUMPAD_ASTERIX’, ‘NUMPAD_0’, ‘NUMPAD_MINUS’, ‘NUMPAD_ENTER’, ‘NUMPAD_PLUS’, ‘F1’, ‘F2’, ‘F3’, ‘F4’, ‘F5’, ‘F6’, ‘F7’, ‘F8’, ‘F9’, ‘F10’, ‘F11’, ‘F12’, ‘F13’, ‘F14’, ‘F15’, ‘F16’, ‘F17’, ‘F18’, ‘F19’, ‘PAUSE’, ‘INSERT’, ‘HOME’, ‘PAGE_UP’, ‘PAGE_DOWN’, ‘END’, ‘MEDIA_PLAY’, ‘MEDIA_STOP’, ‘MEDIA_FIRST’, ‘MEDIA_LAST’, ‘TEXTINPUT’, ‘WINDOW_DEACTIVATE’, ‘TIMER’, ‘TIMER0’, ‘TIMER1’, ‘TIMER2’, ‘TIMER_JOBS’, ‘TIMER_AUTOSAVE’, ‘TIMER_REPORT’, ‘TIMERREGION’, ‘NDOF_MOTION’, ‘NDOF_BUTTON_MENU’, ‘NDOF_BUTTON_FIT’, ‘NDOF_BUTTON_TOP’, ‘NDOF_BUTTON_BOTTOM’, ‘NDOF_BUTTON_LEFT’, ‘NDOF_BUTTON_RIGHT’, ‘NDOF_BUTTON_FRONT’, ‘NDOF_BUTTON_BACK’, ‘NDOF_BUTTON_ISO1’, ‘NDOF_BUTTON_ISO2’, ‘NDOF_BUTTON_ROLL_CW’, ‘NDOF_BUTTON_ROLL_CCW’, ‘NDOF_BUTTON_SPIN_CW’, ‘NDOF_BUTTON_SPIN_CCW’, ‘NDOF_BUTTON_TILT_CW’, ‘NDOF_BUTTON_TILT_CCW’, ‘NDOF_BUTTON_ROTATE’, ‘NDOF_BUTTON_PANZOOM’, ‘NDOF_BUTTON_DOMINANT’, ‘NDOF_BUTTON_PLUS’, ‘NDOF_BUTTON_MINUS’, ‘NDOF_BUTTON_ESC’, ‘NDOF_BUTTON_ALT’, ‘NDOF_BUTTON_SHIFT’, ‘NDOF_BUTTON_CTRL’, ‘NDOF_BUTTON_1’, ‘NDOF_BUTTON_2’, ‘NDOF_BUTTON_3’, ‘NDOF_BUTTON_4’, ‘NDOF_BUTTON_5’, ‘NDOF_BUTTON_6’, ‘NDOF_BUTTON_7’, ‘NDOF_BUTTON_8’, ‘NDOF_BUTTON_9’, ‘NDOF_BUTTON_10’, ‘NDOF_BUTTON_A’, ‘NDOF_BUTTON_B’, ‘NDOF_BUTTON_C’], default ‘NONE’ 
    '''

    target = None
    '''Property that receives the keystrokes in case a string is logged 

    :type:  string, default “”, (never None) 
    '''

    use_all_keys = None
    '''Trigger this sensor on any keystroke 

    :type:  boolean, default False 
    '''


class Keyframe:
    '''Bezier curve point with two handles defining a Keyframe on an F-Curve '''

    amplitude = None
    '''Amount to boost elastic bounces for ‘elastic’ easing 

    :type:  float in [0, inf], default 0.0 
    '''

    back = None
    '''Amount of overshoot for ‘back’ easing 

    :type:  float in [-inf, inf], default 0.0 
    '''

    co = None
    '''Coordinates of the control point 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    easing = None
    '''Which ends of the segment between this and the next keyframe easing interpolation is applied to 

    :type:  enum in [‘AUTO’, ‘EASE_IN’, ‘EASE_OUT’, ‘EASE_IN_OUT’], default ‘AUTO’ 
    '''

    handle_left = None
    '''Coordinates of the left handle (before the control point) 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    handle_left_type = None
    '''Handle types 

    :type:  enum in [‘FREE’, ‘VECTOR’, ‘ALIGNED’, ‘AUTO’, ‘AUTO_CLAMPED’], default ‘FREE’ 
    '''

    handle_right = None
    '''Coordinates of the right handle (after the control point) 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    handle_right_type = None
    '''Handle types 

    :type:  enum in [‘FREE’, ‘VECTOR’, ‘ALIGNED’, ‘AUTO’, ‘AUTO_CLAMPED’], default ‘FREE’ 
    '''

    interpolation = None
    '''Interpolation method to use for segment of the F-Curve from this Keyframe until the next Keyframe 

    :type:  enum in [‘CONSTANT’, ‘LINEAR’, ‘BEZIER’, ‘SINE’, ‘QUAD’, ‘CUBIC’, ‘QUART’, ‘QUINT’, ‘EXPO’, ‘CIRC’, ‘BACK’, ‘BOUNCE’, ‘ELASTIC’], default ‘CONSTANT’ 
    '''

    period = None
    '''Time between bounces for elastic easing 

    :type:  float in [-inf, inf], default 0.0 
    '''

    select_control_point = None
    '''Control point selection status 

    :type:  boolean, default False 
    '''

    select_left_handle = None
    '''Left handle selection status 

    :type:  boolean, default False 
    '''

    select_right_handle = None
    '''Right handle selection status 

    :type:  boolean, default False 
    '''

    type = None
    '''Type of keyframe (for visual purposes only) 

    :type:  enum in [‘KEYFRAME’, ‘BREAKDOWN’, ‘MOVING_HOLD’, ‘EXTREME’, ‘JITTER’], default ‘KEYFRAME’ 
    '''


class KeyingSet:
    '''Settings that should be keyframed together '''

    bl_description = None
    '''A short description of the keying set 

    :type:  string, default “”, (never None) 
    '''

    bl_idname = None
    '''If this is set, the Keying Set gets a custom ID, otherwise it takes the name of the class used to define the Keying Set (for example, if the class name is “BUILTIN_KSI_location”, and bl_idname is not set by the script, then bl_idname = “BUILTIN_KSI_location”) 

    :type:  string, default “”, (never None) 
    '''

    bl_label = None
    '''

    :type:  string, default “”, (never None) 
    '''

    is_path_absolute = None
    '''Keying Set defines specific paths/settings to be keyframed (i.e. is not reliant on context info) 

    :type:  boolean, default False, (readonly) 
    '''

    paths = None
    '''Keying Set Paths to define settings that get keyframed together 

    :type:  KeyingSetPaths bpy_prop_collection of KeyingSetPath, (readonly) 
    '''

    type_info = None
    '''Callback function defines for built-in Keying Sets 

    :type:  KeyingSetInfo, (readonly) 
    '''

    use_insertkey_needed = None
    '''Only insert keyframes where they’re needed in the relevant F-Curves 

    :type:  boolean, default False 
    '''

    use_insertkey_override_needed = None
    '''Override default setting to only insert keyframes where they’re needed in the relevant F-Curves 

    :type:  boolean, default False 
    '''

    use_insertkey_override_visual = None
    '''Override default setting to insert keyframes based on ‘visual transforms’ 

    :type:  boolean, default False 
    '''

    use_insertkey_override_xyz_to_rgb = None
    '''Override default setting to set color for newly added transformation F-Curves (Location, Rotation, Scale) to be based on the transform axis 

    :type:  boolean, default False 
    '''

    use_insertkey_visual = None
    '''Insert keyframes based on ‘visual transforms’ 

    :type:  boolean, default False 
    '''

    use_insertkey_xyz_to_rgb = None
    '''Color for newly added transformation F-Curves (Location, Rotation, Scale) is based on the transform axis 

    :type:  boolean, default False 
    '''

    def refresh(self):
        '''Refresh Keying Set to ensure that it is valid for the current context (call before each use of one) 

        '''
        pass


class KeyingSetInfo:
    '''Callback function defines for builtin Keying Sets '''

    bl_description = None
    '''A short description of the keying set 

    :type:  string, default “”, (never None) 
    '''

    bl_idname = None
    '''If this is set, the Keying Set gets a custom ID, otherwise it takes the name of the class used to define the Keying Set (for example, if the class name is “BUILTIN_KSI_location”, and bl_idname is not set by the script, then bl_idname = “BUILTIN_KSI_location”) 

    :type:  string, default “”, (never None) 
    '''

    bl_label = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_options = None
    '''Keying Set options to use when inserting keyframes 

    :type:  enum set in {‘INSERTKEY_NEEDED’, ‘INSERTKEY_VISUAL’, ‘INSERTKEY_XYZ_TO_RGB’}, default {‘INSERTKEY_NEEDED’} 
    '''

    def poll(self, context):
        '''Test if Keying Set can be used or not 

        :rtype: boolean 
        '''
        pass

    def iterator(self, context, ks):
        '''Call generate() on the structs which have properties to be keyframed 

        '''
        pass

    def generate(self, context, ks, data):
        '''Add Paths to the Keying Set to keyframe the properties of the given data 

        '''
        pass


class KeyingSetPath:
    '''Path to a setting for use in a Keying Set '''

    array_index = None
    '''Index to the specific setting if applicable 

    :type:  int in [-inf, inf], default 0 
    '''

    data_path = None
    '''Path to property setting 

    :type:  string, default “”, (never None) 
    '''

    group = None
    '''Name of Action Group to assign setting(s) for this path to 

    :type:  string, default “”, (never None) 
    '''

    group_method = None
    '''Method used to define which Group-name to use 

    :type:  enum in [‘NAMED’, ‘NONE’, ‘KEYINGSET’], default ‘NAMED’ 
    '''

    id = None
    '''ID-Block that keyframes for Keying Set should be added to (for Absolute Keying Sets only) 

    :type:  ID 
    '''

    id_type = None
    '''Type of ID-block that can be used 

    :type:  enum in [‘ACTION’, ‘ARMATURE’, ‘BRUSH’, ‘CAMERA’, ‘CACHEFILE’, ‘CURVE’, ‘FONT’, ‘GREASEPENCIL’, ‘GROUP’, ‘IMAGE’, ‘KEY’, ‘LAMP’, ‘LIBRARY’, ‘LINESTYLE’, ‘LATTICE’, ‘MASK’, ‘MATERIAL’, ‘META’, ‘MESH’, ‘MOVIECLIP’, ‘NODETREE’, ‘OBJECT’, ‘PAINTCURVE’, ‘PALETTE’, ‘PARTICLE’, ‘SCENE’, ‘SCREEN’, ‘SOUND’, ‘SPEAKER’, ‘TEXT’, ‘TEXTURE’, ‘WINDOWMANAGER’, ‘WORLD’], default ‘OBJECT’ 
    '''

    use_entire_array = None
    '''When an ‘array/vector’ type is chosen (Location, Rotation, Color, etc.), entire array is to be used 

    :type:  boolean, default False 
    '''

    use_insertkey_needed = None
    '''Only insert keyframes where they’re needed in the relevant F-Curves 

    :type:  boolean, default False 
    '''

    use_insertkey_override_needed = None
    '''Override default setting to only insert keyframes where they’re needed in the relevant F-Curves 

    :type:  boolean, default False 
    '''

    use_insertkey_override_visual = None
    '''Override default setting to insert keyframes based on ‘visual transforms’ 

    :type:  boolean, default False 
    '''

    use_insertkey_override_xyz_to_rgb = None
    '''Override default setting to set color for newly added transformation F-Curves (Location, Rotation, Scale) to be based on the transform axis 

    :type:  boolean, default False 
    '''

    use_insertkey_visual = None
    '''Insert keyframes based on ‘visual transforms’ 

    :type:  boolean, default False 
    '''

    use_insertkey_xyz_to_rgb = None
    '''Color for newly added transformation F-Curves (Location, Rotation, Scale) is based on the transform axis 

    :type:  boolean, default False 
    '''


class KeyingSetPaths:
    '''Collection of keying set paths '''

    active = None
    '''Active Keying Set used to insert/delete keyframes 

    :type:  KeyingSetPath 
    '''

    active_index = None
    '''Current Keying Set index 

    :type:  int in [-inf, inf], default 0 
    '''

    def add(self,
            target_id,
            data_path,
            index=-1,
            group_method='KEYINGSET',
            group_name=""):
        '''Add a new path for the Keying Set 

        :param target_id: Target ID, ID-Datablock for the destination 
        :type target_id: ID
        :param data_path: Data-Path, RNA-Path to destination property 
        :type data_path: string, (never None)
        :param index: Index, The index of the destination property (i.e. axis of Location/Rotation/etc.), or -1 for the entire array 
        :type index: int in [-1, inf], (optional)
        :param group_method: Grouping Method, Method used to define which Group-name to use 
        :type group_method: enum in ['NAMED', 'NONE', 'KEYINGSET'], (optional)
        :param group_name: Group Name, Name of Action Group to assign destination to (only if grouping mode is to use this name) 
        :type group_name: string, (optional, never None)
        :rtype: KeyingSetPath 
        :return:  New Path, Path created and added to the Keying Set 
        '''
        pass

    def remove(self, path):
        '''Remove the given path from the Keying Set 

        :param path: Path 
        :type path: KeyingSetPath, (never None)
        '''
        pass

    def clear(self):
        '''Remove all the paths from the Keying Set 

        '''
        pass


class KeyingSets:
    '''Scene keying sets '''

    active = None
    '''Active Keying Set used to insert/delete keyframes 

    :type:  KeyingSet 
    '''

    active_index = None
    '''Current Keying Set index (negative for ‘builtin’ and positive for ‘absolute’) 

    :type:  int in [-inf, inf], default 0 
    '''

    def new(self, idname="KeyingSet", name="KeyingSet"):
        '''Add a new Keying Set to Scene 

        :param idname: IDName, Internal identifier of Keying Set 
        :type idname: string, (optional, never None)
        :param name: Name, User visible name of Keying Set 
        :type name: string, (optional, never None)
        :rtype: KeyingSet 
        :return:  Newly created Keying Set 
        '''
        pass


class KeyingSetsAll:
    '''All available keying sets '''

    active = None
    '''Active Keying Set used to insert/delete keyframes 

    :type:  KeyingSet 
    '''

    active_index = None
    '''Current Keying Set index (negative for ‘builtin’ and positive for ‘absolute’) 

    :type:  int in [-inf, inf], default 0 
    '''


class KinematicConstraint:
    '''Inverse Kinematics '''

    chain_count = None
    '''How many bones are included in the IK effect - 0 uses all bones 

    :type:  int in [0, 255], default 0 
    '''

    distance = None
    '''Radius of limiting sphere 

    :type:  float in [0, 100], default 0.0 
    '''

    ik_type = None
    '''

    :type:  enum in [‘COPY_POSE’, ‘DISTANCE’], default ‘COPY_POSE’ 
    '''

    iterations = None
    '''Maximum number of solving iterations 

    :type:  int in [0, 10000], default 0 
    '''

    limit_mode = None
    '''Distances in relation to sphere of influence to allow 

    :type:  enum in [‘LIMITDIST_INSIDE’, ‘LIMITDIST_OUTSIDE’, ‘LIMITDIST_ONSURFACE’], default ‘LIMITDIST_INSIDE’ 
    '''

    lock_location_x = None
    '''Constraint position along X axis 

    :type:  boolean, default False 
    '''

    lock_location_y = None
    '''Constraint position along Y axis 

    :type:  boolean, default False 
    '''

    lock_location_z = None
    '''Constraint position along Z axis 

    :type:  boolean, default False 
    '''

    lock_rotation_x = None
    '''Constraint rotation along X axis 

    :type:  boolean, default False 
    '''

    lock_rotation_y = None
    '''Constraint rotation along Y axis 

    :type:  boolean, default False 
    '''

    lock_rotation_z = None
    '''Constraint rotation along Z axis 

    :type:  boolean, default False 
    '''

    orient_weight = None
    '''For Tree-IK: Weight of orientation control for this target 

    :type:  float in [0.01, 1], default 0.0 
    '''

    pole_angle = None
    '''Pole rotation offset 

    :type:  float in [-3.14159, 3.14159], default 0.0 
    '''

    pole_subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    pole_target = None
    '''Object for pole rotation 

    :type:  Object 
    '''

    reference_axis = None
    '''Constraint axis Lock options relative to Bone or Target reference 

    :type:  enum in [‘BONE’, ‘TARGET’], default ‘BONE’ 
    '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    use_location = None
    '''Chain follows position of target 

    :type:  boolean, default False 
    '''

    use_rotation = None
    '''Chain follows rotation of target 

    :type:  boolean, default False 
    '''

    use_stretch = None
    '''Enable IK Stretching 

    :type:  boolean, default False 
    '''

    use_tail = None
    '''Include bone’s tail as last element in chain 

    :type:  boolean, default False 
    '''

    weight = None
    '''For Tree-IK: Weight of position control for this target 

    :type:  float in [0.01, 1], default 0.0 
    '''


class Lamp:
    '''Lamp data-block for lighting a scene '''

    active_texture = None
    '''Active texture slot being displayed 

    :type:  Texture 
    '''

    active_texture_index = None
    '''Index of active texture slot 

    :type:  int in [0, 17], default 0 
    '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    color = None
    '''Light color 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    cycles = None
    '''Cycles lamp settings 

    :type:  CyclesLampSettings, (readonly) 
    '''

    distance = None
    '''Falloff distance - the light is at half the original intensity at this point 

    :type:  float in [0, inf], default 0.0 
    '''

    energy = None
    '''Amount of light that the lamp emits 

    :type:  float in [-inf, inf], default 0.0 
    '''

    node_tree = None
    '''Node tree for node based lamps 

    :type:  NodeTree, (readonly) 
    '''

    texture_slots = None
    '''Texture slots defining the mapping and influence of textures 

    :type:  LampTextureSlots bpy_prop_collection of LampTextureSlot, (readonly) 
    '''

    type = None
    '''Type of Lamp 

    :type:  enum in [‘POINT’, ‘SUN’, ‘SPOT’, ‘HEMI’, ‘AREA’], default ‘POINT’ 
    '''

    use_diffuse = None
    '''Do diffuse shading 

    :type:  boolean, default False 
    '''

    use_negative = None
    '''Cast negative light 

    :type:  boolean, default False 
    '''

    use_nodes = None
    '''Use shader nodes to render the lamp 

    :type:  boolean, default False 
    '''

    use_own_layer = None
    '''Illuminate objects only on the same layers the lamp is on 

    :type:  boolean, default False 
    '''

    use_specular = None
    '''Create specular highlights 

    :type:  boolean, default False 
    '''


class LampSkySettings:
    '''Sky related settings for a sun lamp '''

    atmosphere_distance_factor = None
    '''Multiplier to convert blender units to physical distance 

    :type:  float in [0, 500], default 0.0 
    '''

    atmosphere_extinction = None
    '''Extinction scattering contribution factor 

    :type:  float in [0, 1], default 0.0 
    '''

    atmosphere_inscattering = None
    '''Scatter contribution factor 

    :type:  float in [0, 1], default 0.0 
    '''

    atmosphere_turbidity = None
    '''Sky turbidity 

    :type:  float in [1, 30], default 0.0 
    '''

    backscattered_light = None
    '''Backscattered light 

    :type:  float in [-1, 1], default 0.0 
    '''

    horizon_brightness = None
    '''Horizon brightness 

    :type:  float in [0, 20], default 0.0 
    '''

    sky_blend = None
    '''Blend factor with sky 

    :type:  float in [0, 2], default 0.0 
    '''

    sky_blend_type = None
    '''Blend mode for combining sun sky with world sky 

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    sky_color_space = None
    '''Color space to use for internal XYZ->RGB color conversion 

    :type:  enum in [‘SMPTE’, ‘REC709’, ‘CIE’], default ‘SMPTE’ 
    '''

    sky_exposure = None
    '''Strength of sky shading exponential exposure correction 

    :type:  float in [0, 20], default 0.0 
    '''

    spread = None
    '''Horizon Spread 

    :type:  float in [0, 10], default 0.0 
    '''

    sun_brightness = None
    '''Sun brightness 

    :type:  float in [0, 10], default 0.0 
    '''

    sun_intensity = None
    '''Sun intensity 

    :type:  float in [0, 10], default 0.0 
    '''

    sun_size = None
    '''Sun size 

    :type:  float in [0, 10], default 0.0 
    '''

    use_atmosphere = None
    '''Apply sun effect on atmosphere 

    :type:  boolean, default False 
    '''

    use_sky = None
    '''Apply sun effect on sky 

    :type:  boolean, default False 
    '''


class LampTextureSlot:
    '''Texture slot for textures in a Lamp data-block '''

    color_factor = None
    '''Amount texture affects color values 

    :type:  float in [-inf, inf], default 0.0 
    '''

    object = None
    '''Object to use for mapping with Object texture coordinates 

    :type:  Object 
    '''

    shadow_factor = None
    '''Amount texture affects shadow 

    :type:  float in [-inf, inf], default 0.0 
    '''

    texture_coords = None
    '''

    :type:  enum in [‘GLOBAL’, ‘VIEW’, ‘OBJECT’], default ‘GLOBAL’ 
    '''

    use_map_color = None
    '''Let the texture affect the basic color of the lamp 

    :type:  boolean, default False 
    '''

    use_map_shadow = None
    '''Let the texture affect the shadow color of the lamp 

    :type:  boolean, default False 
    '''


class LampTextureSlots:
    '''Collection of texture slots '''

    pass


class LaplacianDeformModifier:
    '''Mesh deform modifier '''

    is_bind = None
    '''Whether geometry has been bound to anchors 

    :type:  boolean, default False, (readonly) 
    '''

    iterations = None
    '''

    :type:  int in [-inf, inf], default 0 
    '''

    vertex_group = None
    '''Name of Vertex Group which determines Anchors 

    :type:  string, default “”, (never None) 
    '''


class LaplacianSmoothModifier:
    '''Smoothing effect modifier '''

    iterations = None
    '''

    :type:  int in [-32768, 32767], default 0 
    '''

    lambda_border = None
    '''Lambda factor in border 

    :type:  float in [-inf, inf], default 0.0 
    '''

    lambda_factor = None
    '''Smooth factor effect 

    :type:  float in [-inf, inf], default 0.0 
    '''

    use_normalized = None
    '''Improve and stabilize the enhanced shape 

    :type:  boolean, default False 
    '''

    use_volume_preserve = None
    '''Apply volume preservation after smooth 

    :type:  boolean, default False 
    '''

    use_x = None
    '''Smooth object along X axis 

    :type:  boolean, default False 
    '''

    use_y = None
    '''Smooth object along Y axis 

    :type:  boolean, default False 
    '''

    use_z = None
    '''Smooth object along Z axis 

    :type:  boolean, default False 
    '''

    vertex_group = None
    '''Name of Vertex Group which determines influence of modifier per point 

    :type:  string, default “”, (never None) 
    '''


class Lattice:
    '''Lattice data-block defining a grid for deforming other objects '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    interpolation_type_u = None
    '''

    :type:  enum in [‘KEY_LINEAR’, ‘KEY_CARDINAL’, ‘KEY_CATMULL_ROM’, ‘KEY_BSPLINE’], default ‘KEY_LINEAR’ 
    '''

    interpolation_type_v = None
    '''

    :type:  enum in [‘KEY_LINEAR’, ‘KEY_CARDINAL’, ‘KEY_CATMULL_ROM’, ‘KEY_BSPLINE’], default ‘KEY_LINEAR’ 
    '''

    interpolation_type_w = None
    '''

    :type:  enum in [‘KEY_LINEAR’, ‘KEY_CARDINAL’, ‘KEY_CATMULL_ROM’, ‘KEY_BSPLINE’], default ‘KEY_LINEAR’ 
    '''

    is_editmode = None
    '''True when used in editmode 

    :type:  boolean, default False, (readonly) 
    '''

    points = None
    '''Points of the lattice 

    :type:  bpy_prop_collection of LatticePoint, (readonly) 
    '''

    points_u = None
    '''Point in U direction (can’t be changed when there are shape keys) 

    :type:  int in [1, 64], default 0 
    '''

    points_v = None
    '''Point in V direction (can’t be changed when there are shape keys) 

    :type:  int in [1, 64], default 0 
    '''

    points_w = None
    '''Point in W direction (can’t be changed when there are shape keys) 

    :type:  int in [1, 64], default 0 
    '''

    shape_keys = None
    '''

    :type:  Key, (readonly) 
    '''

    use_outside = None
    '''Only draw, and take into account, the outer vertices 

    :type:  boolean, default False 
    '''

    vertex_group = None
    '''Vertex group to apply the influence of the lattice 

    :type:  string, default “”, (never None) 
    '''

    def transform(self, matrix, shape_keys=False):
        '''Transform lattice by a matrix 

        :param matrix: Matrix 
        :type matrix: float array of 16 items in [-inf, inf]
        :param shape_keys: Transform Shape Keys 
        :type shape_keys: boolean, (optional)
        '''
        pass


class LatticeModifier:
    '''Lattice deformation modifier '''

    object = None
    '''Lattice object to deform with 

    :type:  Object 
    '''

    strength = None
    '''Strength of modifier effect 

    :type:  float in [-inf, inf], default 0.0 
    '''

    vertex_group = None
    '''Name of Vertex Group which determines influence of modifier per point 

    :type:  string, default “”, (never None) 
    '''


class LatticePoint:
    '''Point in the lattice grid '''

    co = None
    '''Original undeformed location used to calculate the strength of the deform effect (edit/animate the Deformed Location instead) 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0), (readonly) 
    '''

    co_deform = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    groups = None
    '''Weights for the vertex groups this point is member of 

    :type:  bpy_prop_collection of VertexGroupElement, (readonly) 
    '''

    select = None
    '''Selection status 

    :type:  boolean, default False 
    '''

    weight_softbody = None
    '''Softbody goal weight 

    :type:  float in [0.01, 100], default 0.0 
    '''


class Library:
    '''External .blend file from which data is linked '''

    filepath = None
    '''Path to the library .blend file 

    :type:  string, default “”, (never None) 
    '''

    packed_file = None
    '''

    :type:  PackedFile, (readonly) 
    '''

    parent = None
    '''

    :type:  Library, (readonly) 
    '''

    users_id = None
    '''ID data blocks which use this library (readonly) '''

    def reload(self):
        '''Reload this library and all its linked datablocks 

        '''
        pass


class LimitDistanceConstraint:
    '''Limit the distance from target object '''

    distance = None
    '''Radius of limiting sphere 

    :type:  float in [-inf, inf], default 0.0 
    '''

    head_tail = None
    '''Target along length of bone: Head=0, Tail=1 

    :type:  float in [0, 1], default 0.0 
    '''

    limit_mode = None
    '''Distances in relation to sphere of influence to allow 

    :type:  enum in [‘LIMITDIST_INSIDE’, ‘LIMITDIST_OUTSIDE’, ‘LIMITDIST_ONSURFACE’], default ‘LIMITDIST_INSIDE’ 
    '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    use_bbone_shape = None
    '''Follow shape of B-Bone segments when calculating Head/Tail position 

    :type:  boolean, default False 
    '''

    use_transform_limit = None
    '''Transforms are affected by this constraint as well 

    :type:  boolean, default False 
    '''


class LimitLocationConstraint:
    '''Limit the location of the constrained object '''

    max_x = None
    '''Highest X value to allow 

    :type:  float in [-inf, inf], default 0.0 
    '''

    max_y = None
    '''Highest Y value to allow 

    :type:  float in [-inf, inf], default 0.0 
    '''

    max_z = None
    '''Highest Z value to allow 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_x = None
    '''Lowest X value to allow 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_y = None
    '''Lowest Y value to allow 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_z = None
    '''Lowest Z value to allow 

    :type:  float in [-inf, inf], default 0.0 
    '''

    use_max_x = None
    '''Use the maximum X value 

    :type:  boolean, default False 
    '''

    use_max_y = None
    '''Use the maximum Y value 

    :type:  boolean, default False 
    '''

    use_max_z = None
    '''Use the maximum Z value 

    :type:  boolean, default False 
    '''

    use_min_x = None
    '''Use the minimum X value 

    :type:  boolean, default False 
    '''

    use_min_y = None
    '''Use the minimum Y value 

    :type:  boolean, default False 
    '''

    use_min_z = None
    '''Use the minimum Z value 

    :type:  boolean, default False 
    '''

    use_transform_limit = None
    '''Transforms are affected by this constraint as well 

    :type:  boolean, default False 
    '''


class LimitRotationConstraint:
    '''Limit the rotation of the constrained object '''

    max_x = None
    '''Highest X value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    max_y = None
    '''Highest Y value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    max_z = None
    '''Highest Z value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    min_x = None
    '''Lowest X value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    min_y = None
    '''Lowest Y value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    min_z = None
    '''Lowest Z value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    use_limit_x = None
    '''Use the minimum X value 

    :type:  boolean, default False 
    '''

    use_limit_y = None
    '''Use the minimum Y value 

    :type:  boolean, default False 
    '''

    use_limit_z = None
    '''Use the minimum Z value 

    :type:  boolean, default False 
    '''

    use_transform_limit = None
    '''Transforms are affected by this constraint as well 

    :type:  boolean, default False 
    '''


class LimitScaleConstraint:
    '''Limit the scaling of the constrained object '''

    max_x = None
    '''Highest X value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    max_y = None
    '''Highest Y value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    max_z = None
    '''Highest Z value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    min_x = None
    '''Lowest X value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    min_y = None
    '''Lowest Y value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    min_z = None
    '''Lowest Z value to allow 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    use_max_x = None
    '''Use the maximum X value 

    :type:  boolean, default False 
    '''

    use_max_y = None
    '''Use the maximum Y value 

    :type:  boolean, default False 
    '''

    use_max_z = None
    '''Use the maximum Z value 

    :type:  boolean, default False 
    '''

    use_min_x = None
    '''Use the minimum X value 

    :type:  boolean, default False 
    '''

    use_min_y = None
    '''Use the minimum Y value 

    :type:  boolean, default False 
    '''

    use_min_z = None
    '''Use the minimum Z value 

    :type:  boolean, default False 
    '''

    use_transform_limit = None
    '''Transforms are affected by this constraint as well 

    :type:  boolean, default False 
    '''


class LineStyleAlphaModifier:
    '''Base type to define alpha transparency modifiers '''

    pass


class LineStyleAlphaModifier_AlongStroke:
    '''Change alpha transparency along stroke '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleAlphaModifier_CreaseAngle:
    '''Alpha transparency based on the angle between two adjacent faces '''

    angle_max = None
    '''Maximum angle to modify thickness 

    :type:  float in [-inf, inf], default 0.0 
    '''

    angle_min = None
    '''Minimum angle to modify thickness 

    :type:  float in [-inf, inf], default 0.0 
    '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleAlphaModifier_Curvature_3D:
    '''Alpha transparency based on the radial curvature of 3D mesh surfaces '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curvature_max = None
    '''Maximum Curvature 

    :type:  float in [0, 10000], default 0.0 
    '''

    curvature_min = None
    '''Minimum Curvature 

    :type:  float in [0, 10000], default 0.0 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleAlphaModifier_DistanceFromCamera:
    '''Change alpha transparency based on the distance from the camera '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    range_max = None
    '''Upper bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    range_min = None
    '''Lower bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleAlphaModifier_DistanceFromObject:
    '''Change alpha transparency based on the distance from an object '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    range_max = None
    '''Upper bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    range_min = None
    '''Lower bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    target = None
    '''Target object from which the distance is measured 

    :type:  Object 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleAlphaModifier_Material:
    '''Change alpha transparency based on a material attribute '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    material_attribute = None
    '''Specify which material attribute is used 

    :type:  enum in [‘LINE’, ‘LINE_R’, ‘LINE_G’, ‘LINE_B’, ‘LINE_A’, ‘DIFF’, ‘DIFF_R’, ‘DIFF_G’, ‘DIFF_B’, ‘SPEC’, ‘SPEC_R’, ‘SPEC_G’, ‘SPEC_B’, ‘SPEC_HARD’, ‘ALPHA’], default ‘LINE’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleAlphaModifier_Noise:
    '''Alpha transparency based on random noise '''

    amplitude = None
    '''Amplitude of the noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    period = None
    '''Period of the noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    seed = None
    '''Seed for the noise generation 

    :type:  int in [-inf, inf], default 0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleAlphaModifier_Tangent:
    '''Alpha transparency based on the direction of the stroke '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleAlphaModifiers:
    '''Alpha modifiers for changing line alphas '''

    def new(self, name, type):
        '''Add a alpha modifier to line style 

        :param name: New name for the alpha modifier (not unique) 
        :type name: string, (never None)
        :param type: Alpha modifier type to add 
        :type type: enum in ['ALONG_STROKE', 'CREASE_ANGLE', 'CURVATURE_3D', 'DISTANCE_FROM_CAMERA', 'DISTANCE_FROM_OBJECT', 'MATERIAL', 'NOISE', 'TANGENT']
        :rtype: LineStyleAlphaModifier 
        :return:  Newly added alpha modifier 
        '''
        pass

    def remove(self, modifier):
        '''Remove a alpha modifier from line style 

        :param modifier: Alpha modifier to remove 
        :type modifier: LineStyleAlphaModifier, (never None)
        '''
        pass


class LineStyleColorModifier:
    '''Base type to define line color modifiers '''

    pass


class LineStyleColorModifier_AlongStroke:
    '''Change line color along stroke '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    color_ramp = None
    '''Color ramp used to change line color 

    :type:  ColorRamp, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleColorModifier_CreaseAngle:
    '''Change line color based on the underlying crease angle '''

    angle_max = None
    '''Maximum angle to modify thickness 

    :type:  float in [-inf, inf], default 0.0 
    '''

    angle_min = None
    '''Minimum angle to modify thickness 

    :type:  float in [-inf, inf], default 0.0 
    '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    color_ramp = None
    '''Color ramp used to change line color 

    :type:  ColorRamp, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleColorModifier_Curvature_3D:
    '''Change line color based on the radial curvature of 3D mesh surfaces '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    color_ramp = None
    '''Color ramp used to change line color 

    :type:  ColorRamp, (readonly) 
    '''

    curvature_max = None
    '''Maximum Curvature 

    :type:  float in [-inf, inf], default 0.0 
    '''

    curvature_min = None
    '''Minimum Curvature 

    :type:  float in [-inf, inf], default 0.0 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleColorModifier_DistanceFromCamera:
    '''Change line color based on the distance from the camera '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    color_ramp = None
    '''Color ramp used to change line color 

    :type:  ColorRamp, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    range_max = None
    '''Upper bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    range_min = None
    '''Lower bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleColorModifier_DistanceFromObject:
    '''Change line color based on the distance from an object '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    color_ramp = None
    '''Color ramp used to change line color 

    :type:  ColorRamp, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    range_max = None
    '''Upper bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    range_min = None
    '''Lower bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    target = None
    '''Target object from which the distance is measured 

    :type:  Object 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleColorModifier_Material:
    '''Change line color based on a material attribute '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    color_ramp = None
    '''Color ramp used to change line color 

    :type:  ColorRamp, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    material_attribute = None
    '''Specify which material attribute is used 

    :type:  enum in [‘LINE’, ‘LINE_R’, ‘LINE_G’, ‘LINE_B’, ‘LINE_A’, ‘DIFF’, ‘DIFF_R’, ‘DIFF_G’, ‘DIFF_B’, ‘SPEC’, ‘SPEC_R’, ‘SPEC_G’, ‘SPEC_B’, ‘SPEC_HARD’, ‘ALPHA’], default ‘LINE’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''

    use_ramp = None
    '''Use color ramp to map the BW average into an RGB color 

    :type:  boolean, default False 
    '''


class LineStyleColorModifier_Noise:
    '''Change line color based on random noise '''

    amplitude = None
    '''Amplitude of the noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    color_ramp = None
    '''Color ramp used to change line color 

    :type:  ColorRamp, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    period = None
    '''Period of the noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    seed = None
    '''Seed for the noise generation 

    :type:  int in [-inf, inf], default 0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleColorModifier_Tangent:
    '''Change line color based on the direction of a stroke '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    color_ramp = None
    '''Color ramp used to change line color 

    :type:  ColorRamp, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleColorModifiers:
    '''Color modifiers for changing line colors '''

    def new(self, name, type):
        '''Add a color modifier to line style 

        :param name: New name for the color modifier (not unique) 
        :type name: string, (never None)
        :param type: Color modifier type to add 
        :type type: enum in ['ALONG_STROKE', 'CREASE_ANGLE', 'CURVATURE_3D', 'DISTANCE_FROM_CAMERA', 'DISTANCE_FROM_OBJECT', 'MATERIAL', 'NOISE', 'TANGENT']
        :rtype: LineStyleColorModifier 
        :return:  Newly added color modifier 
        '''
        pass

    def remove(self, modifier):
        '''Remove a color modifier from line style 

        :param modifier: Color modifier to remove 
        :type modifier: LineStyleColorModifier, (never None)
        '''
        pass


class LineStyleGeometryModifier:
    '''Base type to define stroke geometry modifiers '''

    pass


class LineStyleGeometryModifier_2DOffset:
    '''Add two-dimensional offsets to stroke backbone geometry '''

    end = None
    '''Displacement that is applied from the end of the stroke 

    :type:  float in [-inf, inf], default 0.0 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    start = None
    '''Displacement that is applied from the beginning of the stroke 

    :type:  float in [-inf, inf], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''

    x = None
    '''Displacement that is applied to the X coordinates of stroke vertices 

    :type:  float in [-inf, inf], default 0.0 
    '''

    y = None
    '''Displacement that is applied to the Y coordinates of stroke vertices 

    :type:  float in [-inf, inf], default 0.0 
    '''


class LineStyleGeometryModifier_2DTransform:
    '''Apply two-dimensional scaling and rotation to stroke backbone geometry '''

    angle = None
    '''Rotation angle 

    :type:  float in [-inf, inf], default 0.0 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    pivot = None
    '''Pivot of scaling and rotation operations 

    :type:  enum in [‘CENTER’, ‘START’, ‘END’, ‘PARAM’, ‘ABSOLUTE’], default ‘CENTER’ 
    '''

    pivot_u = None
    '''Pivot in terms of the stroke point parameter u (0 <= u <= 1) 

    :type:  float in [0, 1], default 0.0 
    '''

    pivot_x = None
    '''2D X coordinate of the absolute pivot 

    :type:  float in [-inf, inf], default 0.0 
    '''

    pivot_y = None
    '''2D Y coordinate of the absolute pivot 

    :type:  float in [-inf, inf], default 0.0 
    '''

    scale_x = None
    '''Scaling factor that is applied along the X axis 

    :type:  float in [-inf, inf], default 0.0 
    '''

    scale_y = None
    '''Scaling factor that is applied along the Y axis 

    :type:  float in [-inf, inf], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifier_BackboneStretcher:
    '''Stretch the beginning and the end of stroke backbone '''

    backbone_length = None
    '''Amount of backbone stretching 

    :type:  float in [-inf, inf], default 0.0 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifier_BezierCurve:
    '''Replace stroke backbone geometry by a Bezier curve approximation of the original backbone geometry '''

    error = None
    '''Maximum distance allowed between the new Bezier curve and the original backbone geometry 

    :type:  float in [-inf, inf], default 0.0 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifier_Blueprint:
    '''Produce a blueprint using circular, elliptic, and square contour strokes '''

    backbone_length = None
    '''Amount of backbone stretching 

    :type:  float in [-inf, inf], default 0.0 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    random_backbone = None
    '''Randomness of the backbone stretching 

    :type:  int in [0, inf], default 0 
    '''

    random_center = None
    '''Randomness of the center 

    :type:  int in [0, inf], default 0 
    '''

    random_radius = None
    '''Randomness of the radius 

    :type:  int in [0, inf], default 0 
    '''

    rounds = None
    '''Number of rounds in contour strokes 

    :type:  int in [1, 1000], default 0 
    '''

    shape = None
    '''Select the shape of blueprint contour strokes 

    :type:  enum in [‘CIRCLES’, ‘ELLIPSES’, ‘SQUARES’], default ‘CIRCLES’ 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifier_GuidingLines:
    '''Modify the stroke geometry so that it corresponds to its main direction line '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    offset = None
    '''Displacement that is applied to the main direction line along its normal 

    :type:  float in [-inf, inf], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifier_PerlinNoise1D:
    '''Add one-dimensional Perlin noise to stroke backbone geometry '''

    amplitude = None
    '''Amplitude of the Perlin noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    angle = None
    '''Displacement direction 

    :type:  float in [-inf, inf], default 0.0 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    frequency = None
    '''Frequency of the Perlin noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    octaves = None
    '''Number of octaves (i.e., the amount of detail of the Perlin noise) 

    :type:  int in [0, inf], default 0 
    '''

    seed = None
    '''Seed for random number generation (if negative, time is used as a seed instead) 

    :type:  int in [-inf, inf], default 0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifier_PerlinNoise2D:
    '''Add two-dimensional Perlin noise to stroke backbone geometry '''

    amplitude = None
    '''Amplitude of the Perlin noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    angle = None
    '''Displacement direction 

    :type:  float in [-inf, inf], default 0.0 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    frequency = None
    '''Frequency of the Perlin noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    octaves = None
    '''Number of octaves (i.e., the amount of detail of the Perlin noise) 

    :type:  int in [0, inf], default 0 
    '''

    seed = None
    '''Seed for random number generation (if negative, time is used as a seed instead) 

    :type:  int in [-inf, inf], default 0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifier_Polygonalization:
    '''Modify the stroke geometry so that it looks more ‘polygonal’ '''

    error = None
    '''Maximum distance between the original stroke and its polygonal approximation 

    :type:  float in [-inf, inf], default 0.0 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifier_Sampling:
    '''Specify a new sampling value that determines the resolution of stroke polylines '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    sampling = None
    '''New sampling value to be used for subsequent modifiers 

    :type:  float in [0, 10000], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifier_Simplification:
    '''Simplify the stroke set '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    tolerance = None
    '''Distance below which segments will be merged 

    :type:  float in [-inf, inf], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifier_SinusDisplacement:
    '''Add sinus displacement to stroke backbone geometry '''

    amplitude = None
    '''Amplitude of the sinus displacement 

    :type:  float in [-inf, inf], default 0.0 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    phase = None
    '''Phase of the sinus displacement 

    :type:  float in [-inf, inf], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''

    wavelength = None
    '''Wavelength of the sinus displacement 

    :type:  float in [-inf, inf], default 0.0 
    '''


class LineStyleGeometryModifier_SpatialNoise:
    '''Add spatial noise to stroke backbone geometry '''

    amplitude = None
    '''Amplitude of the spatial noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    octaves = None
    '''Number of octaves (i.e., the amount of detail of the spatial noise) 

    :type:  int in [0, inf], default 0 
    '''

    scale = None
    '''Scale of the spatial noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    smooth = None
    '''If true, the spatial noise is smooth 

    :type:  boolean, default False 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''

    use_pure_random = None
    '''If true, the spatial noise does not show any coherence 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifier_TipRemover:
    '''Remove a piece of stroke at the beginning and the end of stroke backbone '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    tip_length = None
    '''Length of tips to be removed 

    :type:  float in [-inf, inf], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘2D_OFFSET’, ‘2D_TRANSFORM’, ‘BACKBONE_STRETCHER’, ‘BEZIER_CURVE’, ‘BLUEPRINT’, ‘GUIDING_LINES’, ‘PERLIN_NOISE_1D’, ‘PERLIN_NOISE_2D’, ‘POLYGONIZATION’, ‘SAMPLING’, ‘SIMPLIFICATION’, ‘SINUS_DISPLACEMENT’, ‘SPATIAL_NOISE’, ‘TIP_REMOVER’], default ‘2D_OFFSET’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleGeometryModifiers:
    '''Geometry modifiers for changing line geometries '''

    def new(self, name, type):
        '''Add a geometry modifier to line style 

        :param name: New name for the geometry modifier (not unique) 
        :type name: string, (never None)
        :param type: Geometry modifier type to add 
        :type type: enum in ['2D_OFFSET', '2D_TRANSFORM', 'BACKBONE_STRETCHER', 'BEZIER_CURVE', 'BLUEPRINT', 'GUIDING_LINES', 'PERLIN_NOISE_1D', 'PERLIN_NOISE_2D', 'POLYGONIZATION', 'SAMPLING', 'SIMPLIFICATION', 'SINUS_DISPLACEMENT', 'SPATIAL_NOISE', 'TIP_REMOVER']
        :rtype: LineStyleGeometryModifier 
        :return:  Newly added geometry modifier 
        '''
        pass

    def remove(self, modifier):
        '''Remove a geometry modifier from line style 

        :param modifier: Geometry modifier to remove 
        :type modifier: LineStyleGeometryModifier, (never None)
        '''
        pass


class LineStyleModifier:
    '''Base type to define modifiers '''

    pass


class LineStyleTextureSlot:
    '''Texture slot for textures in a LineStyle data-block '''

    alpha_factor = None
    '''Amount texture affects alpha 

    :type:  float in [-inf, inf], default 0.0 
    '''

    diffuse_color_factor = None
    '''Amount texture affects diffuse color 

    :type:  float in [-inf, inf], default 0.0 
    '''

    mapping = None
    '''

    :type:  enum in [‘FLAT’, ‘CUBE’, ‘TUBE’, ‘SPHERE’], default ‘FLAT’ 
    '''

    mapping_x = None
    '''

    :type:  enum in [‘NONE’, ‘X’, ‘Y’, ‘Z’], default ‘NONE’ 
    '''

    mapping_y = None
    '''

    :type:  enum in [‘NONE’, ‘X’, ‘Y’, ‘Z’], default ‘NONE’ 
    '''

    mapping_z = None
    '''

    :type:  enum in [‘NONE’, ‘X’, ‘Y’, ‘Z’], default ‘NONE’ 
    '''

    texture_coords = None
    '''Texture coordinates used to map the texture onto the background 

    :type:  enum in [‘WINDOW’, ‘GLOBAL’, ‘ALONG_STROKE’, ‘ORCO’], default ‘WINDOW’ 
    '''

    use_map_alpha = None
    '''The texture affects the alpha value 

    :type:  boolean, default False 
    '''

    use_map_color_diffuse = None
    '''The texture affects basic color of the stroke 

    :type:  boolean, default False 
    '''

    use_tips = None
    '''Lower half of the texture is for tips of the stroke 

    :type:  boolean, default False 
    '''


class LineStyleTextureSlots:
    '''Collection of texture slots '''

    pass


class LineStyleThicknessModifier:
    '''Base type to define line thickness modifiers '''

    pass


class LineStyleThicknessModifier_AlongStroke:
    '''Change line thickness along stroke '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CALLIGRAPHY’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''

    value_max = None
    '''Maximum output value of the mapping 

    :type:  float in [-inf, inf], default 0.0 
    '''

    value_min = None
    '''Minimum output value of the mapping 

    :type:  float in [-inf, inf], default 0.0 
    '''


class LineStyleThicknessModifier_Calligraphy:
    '''Change line thickness so that stroke looks like made with a calligraphic pen '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    orientation = None
    '''Angle of the main direction 

    :type:  float in [-inf, inf], default 0.0 
    '''

    thickness_max = None
    '''Maximum thickness in the main direction 

    :type:  float in [0, 10000], default 0.0 
    '''

    thickness_min = None
    '''Minimum thickness in the direction perpendicular to the main direction 

    :type:  float in [0, 10000], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CALLIGRAPHY’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleThicknessModifier_CreaseAngle:
    '''Line thickness based on the angle between two adjacent faces '''

    angle_max = None
    '''Maximum angle to modify thickness 

    :type:  float in [-inf, inf], default 0.0 
    '''

    angle_min = None
    '''Minimum angle to modify thickness 

    :type:  float in [-inf, inf], default 0.0 
    '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    thickness_max = None
    '''Maximum thickness 

    :type:  float in [0, 10000], default 0.0 
    '''

    thickness_min = None
    '''Minimum thickness 

    :type:  float in [0, 10000], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CALLIGRAPHY’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleThicknessModifier_Curvature_3D:
    '''Line thickness based on the radial curvature of 3D mesh surfaces '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curvature_max = None
    '''Maximum Curvature 

    :type:  float in [0, 10000], default 0.0 
    '''

    curvature_min = None
    '''Minimum Curvature 

    :type:  float in [0, 10000], default 0.0 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    thickness_max = None
    '''Maximum thickness 

    :type:  float in [0, 10000], default 0.0 
    '''

    thickness_min = None
    '''Minimum thickness 

    :type:  float in [0, 10000], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CALLIGRAPHY’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleThicknessModifier_DistanceFromCamera:
    '''Change line thickness based on the distance from the camera '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    range_max = None
    '''Upper bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    range_min = None
    '''Lower bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CALLIGRAPHY’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''

    value_max = None
    '''Maximum output value of the mapping 

    :type:  float in [-inf, inf], default 0.0 
    '''

    value_min = None
    '''Minimum output value of the mapping 

    :type:  float in [-inf, inf], default 0.0 
    '''


class LineStyleThicknessModifier_DistanceFromObject:
    '''Change line thickness based on the distance from an object '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    range_max = None
    '''Upper bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    range_min = None
    '''Lower bound of the input range the mapping is applied 

    :type:  float in [-inf, inf], default 0.0 
    '''

    target = None
    '''Target object from which the distance is measured 

    :type:  Object 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CALLIGRAPHY’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''

    value_max = None
    '''Maximum output value of the mapping 

    :type:  float in [-inf, inf], default 0.0 
    '''

    value_min = None
    '''Minimum output value of the mapping 

    :type:  float in [-inf, inf], default 0.0 
    '''


class LineStyleThicknessModifier_Material:
    '''Change line thickness based on a material attribute '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    material_attribute = None
    '''Specify which material attribute is used 

    :type:  enum in [‘LINE’, ‘LINE_R’, ‘LINE_G’, ‘LINE_B’, ‘LINE_A’, ‘DIFF’, ‘DIFF_R’, ‘DIFF_G’, ‘DIFF_B’, ‘SPEC’, ‘SPEC_R’, ‘SPEC_G’, ‘SPEC_B’, ‘SPEC_HARD’, ‘ALPHA’], default ‘LINE’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CALLIGRAPHY’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''

    value_max = None
    '''Maximum output value of the mapping 

    :type:  float in [-inf, inf], default 0.0 
    '''

    value_min = None
    '''Minimum output value of the mapping 

    :type:  float in [-inf, inf], default 0.0 
    '''


class LineStyleThicknessModifier_Noise:
    '''Line thickness based on random noise '''

    amplitude = None
    '''Amplitude of the noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    period = None
    '''Period of the noise 

    :type:  float in [-inf, inf], default 0.0 
    '''

    seed = None
    '''Seed for the noise generation 

    :type:  int in [-inf, inf], default 0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CALLIGRAPHY’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''

    use_asymmetric = None
    '''Allow thickness to be assigned asymmetrically 

    :type:  boolean, default False 
    '''


class LineStyleThicknessModifier_Tangent:
    '''Thickness based on the direction of the stroke '''

    blend = None
    '''Specify how the modifier value is blended into the base value 

    :type:  enum in [‘MIX’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’, ‘DIVIDE’, ‘DIFFERENCE’, ‘MININUM’, ‘MAXIMUM’], default ‘MIX’ 
    '''

    curve = None
    '''Curve used for the curve mapping 

    :type:  CurveMapping, (readonly) 
    '''

    expanded = None
    '''True if the modifier tab is expanded 

    :type:  boolean, default False 
    '''

    influence = None
    '''Influence factor by which the modifier changes the property 

    :type:  float in [0, 1], default 0.0 
    '''

    invert = None
    '''Invert the fade-out direction of the linear mapping 

    :type:  boolean, default False 
    '''

    mapping = None
    '''Select the mapping type 

    :type:  enum in [‘LINEAR’, ‘CURVE’], default ‘LINEAR’ 
    '''

    name = None
    '''Name of the modifier 

    :type:  string, default “”, (never None) 
    '''

    thickness_max = None
    '''Maximum thickness 

    :type:  float in [0, 10000], default 0.0 
    '''

    thickness_min = None
    '''Minimum thickness 

    :type:  float in [0, 10000], default 0.0 
    '''

    type = None
    '''Type of the modifier 

    :type:  enum in [‘ALONG_STROKE’, ‘CALLIGRAPHY’, ‘CREASE_ANGLE’, ‘CURVATURE_3D’, ‘DISTANCE_FROM_CAMERA’, ‘DISTANCE_FROM_OBJECT’, ‘MATERIAL’, ‘NOISE’, ‘TANGENT’], default ‘ALONG_STROKE’, (readonly) 
    '''

    use = None
    '''Enable or disable this modifier during stroke rendering 

    :type:  boolean, default False 
    '''


class LineStyleThicknessModifiers:
    '''Thickness modifiers for changing line thickness '''

    def new(self, name, type):
        '''Add a thickness modifier to line style 

        :param name: New name for the thickness modifier (not unique) 
        :type name: string, (never None)
        :param type: Thickness modifier type to add 
        :type type: enum in ['ALONG_STROKE', 'CALLIGRAPHY', 'CREASE_ANGLE', 'CURVATURE_3D', 'DISTANCE_FROM_CAMERA', 'DISTANCE_FROM_OBJECT', 'MATERIAL', 'NOISE', 'TANGENT']
        :rtype: LineStyleThicknessModifier 
        :return:  Newly added thickness modifier 
        '''
        pass

    def remove(self, modifier):
        '''Remove a thickness modifier from line style 

        :param modifier: Thickness modifier to remove 
        :type modifier: LineStyleThicknessModifier, (never None)
        '''
        pass


class Linesets:
    '''Line sets for associating lines and style parameters '''

    active = None
    '''Active line set being displayed 

    :type:  FreestyleLineSet, (readonly) 
    '''

    active_index = None
    '''Index of active line set slot 

    :type:  int in [0, inf], default 0 
    '''

    def new(self, name):
        '''Add a line set to scene render layer Freestyle settings 

        :param name: New name for the line set (not unique) 
        :type name: string, (never None)
        :rtype: FreestyleLineSet 
        :return:  Newly created line set 
        '''
        pass

    def remove(self, lineset):
        '''Remove a line set from scene render layer Freestyle settings 

        :param lineset: Line set to remove 
        :type lineset: FreestyleLineSet, (never None)
        '''
        pass


class LocRotScale:
    def generate(self, context, ks, data):
        '''

        '''
        pass


class Location:
    pass


class LockedTrackConstraint:
    '''Point toward the target along the track axis, while locking the other axis '''

    head_tail = None
    '''Target along length of bone: Head=0, Tail=1 

    :type:  float in [0, 1], default 0.0 
    '''

    lock_axis = None
    '''Axis that points upward 

    :type:  enum in [‘LOCK_X’, ‘LOCK_Y’, ‘LOCK_Z’], default ‘LOCK_X’ 
    '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    track_axis = None
    '''Axis that points to the target object 

    :type:  enum in [‘TRACK_X’, ‘TRACK_Y’, ‘TRACK_Z’, ‘TRACK_NEGATIVE_X’, ‘TRACK_NEGATIVE_Y’, ‘TRACK_NEGATIVE_Z’], default ‘TRACK_X’ 
    '''

    use_bbone_shape = None
    '''Follow shape of B-Bone segments when calculating Head/Tail position 

    :type:  boolean, default False 
    '''


class LodLevel:
    distance = None
    '''Distance to begin using this level of detail 

    :type:  float in [0, inf], default 0.0 
    '''

    object = None
    '''Object to use for this level of detail 

    :type:  Object 
    '''

    object_hysteresis_percentage = None
    '''Minimum distance change required to transition to the previous level of detail 

    :type:  int in [0, 100], default 0 
    '''

    use_material = None
    '''Use the material from this object at this level of detail 

    :type:  boolean, default False 
    '''

    use_mesh = None
    '''Use the mesh from this object at this level of detail 

    :type:  boolean, default False 
    '''

    use_object_hysteresis = None
    '''Override LoD Hysteresis scene setting for this LoD level 

    :type:  boolean, default False 
    '''


class LoopColors:
    '''Collection of vertex colors '''

    active = None
    '''Active vertex color layer 

    :type:  MeshLoopColorLayer 
    '''

    active_index = None
    '''Active vertex color index 

    :type:  int in [0, inf], default 0 
    '''

    def new(self, name="Col"):
        '''Add a vertex color layer to Mesh 

        :param name: Vertex color name 
        :type name: string, (optional, never None)
        :rtype: MeshLoopColorLayer 
        :return:  The newly created layer 
        '''
        pass

    def remove(self, layer):
        '''Remove a vertex color layer 

        :param layer: The layer to remove 
        :type layer: MeshLoopColorLayer, (never None)
        '''
        pass


class MASK_UL_layers:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class MATERIAL_UL_matslots:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class MESH_UL_shape_keys:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class MESH_UL_uvmaps_vcols:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class MESH_UL_vgroups:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class Macro:
    '''Storage of a macro operator being executed, or registered after execution '''

    bl_description = None
    '''

    :type:  string, default “” 
    '''

    bl_idname = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_label = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_options = None
    '''Options for this operator type 

    :type:  enum set in {‘REGISTER’, ‘UNDO’, ‘BLOCKING’, ‘MACRO’, ‘GRAB_CURSOR’, ‘PRESET’, ‘INTERNAL’}, default {‘REGISTER’} 
    '''

    bl_translation_context = None
    '''

    :type:  string, default “Operator” 
    '''

    name = None
    '''

    :type:  string, default “”, (readonly, never None) 
    '''

    properties = None
    '''

    :type:  OperatorProperties, (readonly, never None) 
    '''

    def report(self, type, message):
        '''report 

        :param type: Type 
        :type type: enum set in {'DEBUG', 'INFO', 'OPERATOR', 'PROPERTY', 'WARNING', 'ERROR', 'ERROR_INVALID_INPUT', 'ERROR_INVALID_CONTEXT', 'ERROR_OUT_OF_MEMORY'}
        :param message: Report Message 
        :type message: string, (never None)
        '''
        pass

    def draw(self, context):
        '''Draw function for the operator 

        '''
        pass


class MagicTexture:
    '''Procedural noise texture '''

    noise_depth = None
    '''Depth of the noise 

    :type:  int in [0, 30], default 0 
    '''

    turbulence = None
    '''Turbulence of the noise 

    :type:  float in [0.0001, inf], default 0.0 
    '''

    users_material = None
    '''Materials that use this texture (readonly) '''

    users_object_modifier = None
    '''Object modifiers that use this texture (readonly) '''


class MaintainVolumeConstraint:
    '''Maintain a constant volume along a single scaling axis '''

    free_axis = None
    '''The free scaling axis of the object 

    :type:  enum in [‘SAMEVOL_X’, ‘SAMEVOL_Y’, ‘SAMEVOL_Z’], default ‘SAMEVOL_X’ 
    '''

    volume = None
    '''Volume of the bone at rest 

    :type:  float in [0.001, 100], default 0.0 
    '''


class MarbleTexture:
    '''Procedural noise texture '''

    marble_type = None
    '''

    :type:  enum in [‘SOFT’, ‘SHARP’, ‘SHARPER’], default ‘SOFT’ 
    '''

    nabla = None
    '''Size of derivative offset used for calculating normal 

    :type:  float in [0.001, 0.1], default 0.0 
    '''

    noise_basis = None
    '''Noise basis used for turbulence 

    :type:  enum in [‘BLENDER_ORIGINAL’, ‘ORIGINAL_PERLIN’, ‘IMPROVED_PERLIN’, ‘VORONOI_F1’, ‘VORONOI_F2’, ‘VORONOI_F3’, ‘VORONOI_F4’, ‘VORONOI_F2_F1’, ‘VORONOI_CRACKLE’, ‘CELL_NOISE’], default ‘BLENDER_ORIGINAL’ 
    '''

    noise_basis_2 = None
    '''

    :type:  enum in [‘SIN’, ‘SAW’, ‘TRI’], default ‘SIN’ 
    '''

    noise_depth = None
    '''Depth of the cloud calculation 

    :type:  int in [0, 30], default 0 
    '''

    noise_scale = None
    '''Scaling for noise input 

    :type:  float in [0.0001, inf], default 0.0 
    '''

    noise_type = None
    '''

    :type:  enum in [‘SOFT_NOISE’, ‘HARD_NOISE’], default ‘SOFT_NOISE’ 
    '''

    turbulence = None
    '''Turbulence of the bandnoise and ringnoise types 

    :type:  float in [0.0001, inf], default 0.0 
    '''

    users_material = None
    '''Materials that use this texture (readonly) '''

    users_object_modifier = None
    '''Object modifiers that use this texture (readonly) '''


class Mask:
    '''Mask data-block defining mask for compositing '''

    active_layer_index = None
    '''Index of active layer in list of all mask’s layers 

    :type:  int in [-inf, inf], default 0 
    '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    frame_end = None
    '''Final frame of the mask (used for sequencer) 

    :type:  int in [0, 500000], default 0 
    '''

    frame_start = None
    '''First frame of the mask (used for sequencer) 

    :type:  int in [0, 500000], default 0 
    '''

    layers = None
    '''Collection of layers which defines this mask 

    :type:  MaskLayers bpy_prop_collection of MaskLayer, (readonly) 
    '''


class MaskLayer:
    '''Single layer used for masking pixels '''

    alpha = None
    '''Render Opacity 

    :type:  float in [-inf, inf], default 0.0 
    '''

    blend = None
    '''Method of blending mask layers 

    :type:  enum in [‘MERGE_ADD’, ‘MERGE_SUBTRACT’, ‘ADD’, ‘SUBTRACT’, ‘LIGHTEN’, ‘DARKEN’, ‘MUL’, ‘REPLACE’, ‘DIFFERENCE’], default ‘ADD’ 
    '''

    falloff = None
    '''Falloff type the feather 

    :type:  enum in [‘SMOOTH’, ‘SPHERE’, ‘ROOT’, ‘INVERSE_SQUARE’, ‘SHARP’, ‘LINEAR’], default ‘SMOOTH’ 
    '''

    hide = None
    '''Restrict visibility in the viewport 

    :type:  boolean, default False 
    '''

    hide_render = None
    '''Restrict renderability 

    :type:  boolean, default False 
    '''

    hide_select = None
    '''Restrict selection in the viewport 

    :type:  boolean, default False 
    '''

    invert = None
    '''Invert the mask black/white 

    :type:  boolean, default False 
    '''

    name = None
    '''Unique name of layer 

    :type:  string, default “”, (never None) 
    '''

    select = None
    '''Layer is selected for editing in the Dope Sheet 

    :type:  boolean, default False 
    '''

    splines = None
    '''Collection of splines which defines this layer 

    :type:  MaskSplines bpy_prop_collection of MaskSpline, (readonly) 
    '''

    use_fill_holes = None
    '''Calculate holes when filling overlapping curves 

    :type:  boolean, default False 
    '''

    use_fill_overlap = None
    '''Calculate self intersections and overlap before filling 

    :type:  boolean, default False 
    '''


class MaskLayers:
    '''Collection of layers used by mask '''

    active = None
    '''Active layer in this mask 

    :type:  MaskLayer 
    '''

    def new(self, name=""):
        '''Add layer to this mask 

        :param name: Name, Name of new layer 
        :type name: string, (optional, never None)
        :rtype: MaskLayer 
        :return:  New mask layer 
        '''
        pass

    def remove(self, layer):
        '''Remove layer from this mask 

        :param layer: Shape to be removed 
        :type layer: MaskLayer, (never None)
        '''
        pass

    def clear(self):
        '''Remove all mask layers 

        '''
        pass


class MaskModifier:
    '''Mask modifier to hide parts of the mesh '''

    armature = None
    '''Armature to use as source of bones to mask 

    :type:  Object 
    '''

    invert_vertex_group = None
    '''Use vertices that are not part of region defined 

    :type:  boolean, default False 
    '''

    mode = None
    '''

    :type:  enum in [‘VERTEX_GROUP’, ‘ARMATURE’], default ‘VERTEX_GROUP’ 
    '''

    vertex_group = None
    '''Vertex group name 

    :type:  string, default “”, (never None) 
    '''


class MaskParent:
    '''Parenting settings for masking element '''

    id = None
    '''ID-block to which masking element would be parented to or to it’s property 

    :type:  ID 
    '''

    id_type = None
    '''Type of ID-block that can be used 

    :type:  enum in [‘MOVIECLIP’], default ‘MOVIECLIP’ 
    '''

    parent = None
    '''Name of parent object in specified data block to which parenting happens 

    :type:  string, default “”, (never None) 
    '''

    sub_parent = None
    '''Name of parent sub-object in specified data block to which parenting happens 

    :type:  string, default “”, (never None) 
    '''

    type = None
    '''Parent Type 

    :type:  enum in [‘POINT_TRACK’, ‘PLANE_TRACK’], default ‘POINT_TRACK’ 
    '''


class MaskSequence:
    '''Sequence strip to load a video from a mask '''

    alpha_mode = None
    '''Representation of alpha information in the RGBA pixels 

    :type:  enum in [‘STRAIGHT’, ‘PREMUL’], default ‘STRAIGHT’ 
    '''

    animation_offset_end = None
    '''Animation end offset (trim end) 

    :type:  int in [0, inf], default 0 
    '''

    animation_offset_start = None
    '''Animation start offset (trim start) 

    :type:  int in [0, inf], default 0 
    '''

    color_multiply = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    color_saturation = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    crop = None
    '''

    :type:  SequenceCrop, (readonly) 
    '''

    mask = None
    '''Mask that this sequence uses 

    :type:  Mask 
    '''

    strobe = None
    '''Only display every nth frame 

    :type:  float in [1, 30], default 0.0 
    '''

    transform = None
    '''

    :type:  SequenceTransform, (readonly) 
    '''

    use_crop = None
    '''Crop image before processing 

    :type:  boolean, default False 
    '''

    use_deinterlace = None
    '''For video movies to remove fields 

    :type:  boolean, default False 
    '''

    use_flip_x = None
    '''Flip on the X axis 

    :type:  boolean, default False 
    '''

    use_flip_y = None
    '''Flip on the Y axis 

    :type:  boolean, default False 
    '''

    use_float = None
    '''Convert input to float data 

    :type:  boolean, default False 
    '''

    use_reverse_frames = None
    '''Reverse frame order 

    :type:  boolean, default False 
    '''

    use_translation = None
    '''Translate image before processing 

    :type:  boolean, default False 
    '''


class MaskSpline:
    '''Single spline used for defining mask shape '''

    offset_mode = None
    '''The method used for calculating the feather offset 

    :type:  enum in [‘EVEN’, ‘SMOOTH’], default ‘EVEN’ 
    '''

    points = None
    '''Collection of points 

    :type:  MaskSplinePoints bpy_prop_collection of MaskSplinePoint, (readonly) 
    '''

    use_cyclic = None
    '''Make this spline a closed loop 

    :type:  boolean, default False 
    '''

    use_fill = None
    '''Make this spline filled 

    :type:  boolean, default False 
    '''

    use_self_intersection_check = None
    '''Prevent feather from self-intersections 

    :type:  boolean, default False 
    '''

    weight_interpolation = None
    '''The type of weight interpolation for spline 

    :type:  enum in [‘LINEAR’, ‘EASE’], default ‘LINEAR’ 
    '''


class MaskSplinePoint:
    '''Single point in spline used for defining mask '''

    co = None
    '''Coordinates of the control point 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    feather_points = None
    '''Points defining feather 

    :type:  bpy_prop_collection of MaskSplinePointUW, (readonly) 
    '''

    handle_left = None
    '''Coordinates of the first handle 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    handle_left_type = None
    '''Handle type 

    :type:  enum in [‘AUTO’, ‘VECTOR’, ‘ALIGNED’, ‘ALIGNED_DOUBLESIDE’, ‘FREE’], default ‘FREE’ 
    '''

    handle_right = None
    '''Coordinates of the second handle 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    handle_right_type = None
    '''Handle type 

    :type:  enum in [‘AUTO’, ‘VECTOR’, ‘ALIGNED’, ‘ALIGNED_DOUBLESIDE’, ‘FREE’], default ‘FREE’ 
    '''

    handle_type = None
    '''Handle type 

    :type:  enum in [‘AUTO’, ‘VECTOR’, ‘ALIGNED’, ‘ALIGNED_DOUBLESIDE’, ‘FREE’], default ‘FREE’ 
    '''

    parent = None
    '''

    :type:  MaskParent, (readonly) 
    '''

    select = None
    '''Selection status 

    :type:  boolean, default False 
    '''

    weight = None
    '''Weight of the point 

    :type:  float in [0, 1], default 0.0 
    '''


class MaskSplinePointUW:
    '''Single point in spline segment defining feather '''

    select = None
    '''Selection status 

    :type:  boolean, default False 
    '''

    u = None
    '''U coordinate of point along spline segment 

    :type:  float in [0, 1], default 0.0 
    '''

    weight = None
    '''Weight of feather point 

    :type:  float in [0, 1], default 0.0 
    '''


class MaskSplinePoints:
    '''Collection of masking spline points '''

    def add(self, count=1):
        '''Add a number of point to this spline 

        :param count: Number, Number of points to add to the spline 
        :type count: int in [0, inf], (optional)
        '''
        pass

    def remove(self, point):
        '''Remove a point from a spline 

        :param point: The point to remove 
        :type point: MaskSplinePoint, (never None)
        '''
        pass


class MaskSplines:
    '''Collection of masking splines '''

    active = None
    '''Active spline of masking layer 

    :type:  MaskSpline 
    '''

    active_point = None
    '''Active spline of masking layer 

    :type:  MaskSplinePoint 
    '''

    def new(self):
        '''Add a new spline to the layer 

        :rtype: MaskSpline 
        :return:  The newly created spline 
        '''
        pass

    def remove(self, spline):
        '''Remove a spline from a layer 

        :param spline: The spline to remove 
        :type spline: MaskSpline, (never None)
        '''
        pass


class Material:
    '''Material data-block to define the appearance of geometric objects for rendering '''

    active_node_material = None
    '''Active node material 

    :type:  Material 
    '''

    active_texture = None
    '''Active texture slot being displayed 

    :type:  Texture 
    '''

    active_texture_index = None
    '''Index of active texture slot 

    :type:  int in [0, 17], default 0 
    '''

    alpha = None
    '''Alpha transparency of the material 

    :type:  float in [0, 1], default 0.0 
    '''

    ambient = None
    '''Amount of global ambient color the material receives 

    :type:  float in [0, 1], default 0.0 
    '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    cycles = None
    '''Cycles material settings 

    :type:  CyclesMaterialSettings, (readonly) 
    '''

    darkness = None
    '''Minnaert darkness 

    :type:  float in [0, 2], default 0.0 
    '''

    diffuse_color = None
    '''Diffuse color of the material 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    diffuse_fresnel = None
    '''Power of Fresnel 

    :type:  float in [0, 5], default 0.0 
    '''

    diffuse_fresnel_factor = None
    '''Blending factor of Fresnel 

    :type:  float in [0, 5], default 0.0 
    '''

    diffuse_intensity = None
    '''Amount of diffuse reflection 

    :type:  float in [0, 1], default 0.0 
    '''

    diffuse_ramp = None
    '''Color ramp used to affect diffuse shading 

    :type:  ColorRamp, (readonly) 
    '''

    diffuse_ramp_blend = None
    '''Blending method of the ramp and the diffuse color 

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    diffuse_ramp_factor = None
    '''Blending factor (also uses alpha in Colorband) 

    :type:  float in [0, 1], default 0.0 
    '''

    diffuse_ramp_input = None
    '''How the ramp maps on the surface 

    :type:  enum in [‘SHADER’, ‘ENERGY’, ‘NORMAL’, ‘RESULT’], default ‘SHADER’ 
    '''

    diffuse_shader = None
    '''

    :type:  enum in [‘LAMBERT’, ‘OREN_NAYAR’, ‘TOON’, ‘MINNAERT’, ‘FRESNEL’], default ‘LAMBERT’ 
    '''

    diffuse_toon_size = None
    '''Size of diffuse toon area 

    :type:  float in [0, 3.14], default 0.0 
    '''

    diffuse_toon_smooth = None
    '''Smoothness of diffuse toon area 

    :type:  float in [0, 1], default 0.0 
    '''

    emit = None
    '''Amount of light to emit 

    :type:  float in [0, inf], default 0.0 
    '''

    game_settings = None
    '''Game material settings 

    :type:  MaterialGameSettings, (readonly, never None) 
    '''

    halo = None
    '''Halo settings for the material 

    :type:  MaterialHalo, (readonly, never None) 
    '''

    invert_z = None
    '''Render material’s faces with an inverted Z buffer (scanline only) 

    :type:  boolean, default False 
    '''

    light_group = None
    '''Limit lighting to lamps in this Group 

    :type:  Group 
    '''

    line_color = None
    '''Line color used for Freestyle line rendering 

    :type:  float array of 4 items in [0, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''

    line_priority = None
    '''The line color of a higher priority is used at material boundaries 

    :type:  int in [0, 32767], default 0 
    '''

    mirror_color = None
    '''Mirror color of the material 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    node_tree = None
    '''Node tree for node based materials 

    :type:  NodeTree, (readonly) 
    '''

    offset_z = None
    '''Give faces an artificial offset in the Z buffer for Z transparency 

    :type:  float in [-inf, inf], default 0.0 
    '''

    paint_active_slot = None
    '''Index of active texture paint slot 

    :type:  int in [0, 32767], default 0 
    '''

    paint_clone_slot = None
    '''Index of clone texture paint slot 

    :type:  int in [0, 32767], default 0 
    '''

    pass_index = None
    '''Index number for the “Material Index” render pass 

    :type:  int in [0, 32767], default 0 
    '''

    physics = None
    '''Game physics settings 

    :type:  MaterialPhysics, (readonly, never None) 
    '''

    preview_render_type = None
    '''Type of preview render 

    :type:  enum in [‘FLAT’, ‘SPHERE’, ‘CUBE’, ‘MONKEY’, ‘HAIR’, ‘SPHERE_A’], default ‘FLAT’ 
    '''

    raytrace_mirror = None
    '''Raytraced reflection settings for the material 

    :type:  MaterialRaytraceMirror, (readonly, never None) 
    '''

    raytrace_transparency = None
    '''Raytraced transparency settings for the material 

    :type:  MaterialRaytraceTransparency, (readonly, never None) 
    '''

    roughness = None
    '''Oren-Nayar Roughness 

    :type:  float in [0, 3.14], default 0.0 
    '''

    shadow_buffer_bias = None
    '''Factor to multiply shadow buffer bias with (0 is ignore) 

    :type:  float in [0, 10], default 0.0 
    '''

    shadow_cast_alpha = None
    '''Shadow casting alpha, in use for Irregular and Deep shadow buffer 

    :type:  float in [0.001, 1], default 0.0 
    '''

    shadow_only_type = None
    '''How to draw shadows 

    :type:  enum in [‘SHADOW_ONLY_OLD’, ‘SHADOW_ONLY’, ‘SHADOW_ONLY_SHADED’], default ‘SHADOW_ONLY_OLD’ 
    '''

    shadow_ray_bias = None
    '''Shadow raytracing bias to prevent terminator problems on shadow boundary 

    :type:  float in [0, 0.25], default 0.0 
    '''

    specular_alpha = None
    '''Alpha transparency for specular areas 

    :type:  float in [0, 1], default 0.0 
    '''

    specular_color = None
    '''Specular color of the material 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    specular_hardness = None
    '''How hard (sharp) the specular reflection is 

    :type:  int in [1, 511], default 0 
    '''

    specular_intensity = None
    '''How intense (bright) the specular reflection is 

    :type:  float in [0, 1], default 0.0 
    '''

    specular_ior = None
    '''Specular index of refraction 

    :type:  float in [1, 10], default 0.0 
    '''

    specular_ramp = None
    '''Color ramp used to affect specular shading 

    :type:  ColorRamp, (readonly) 
    '''

    specular_ramp_blend = None
    '''Blending method of the ramp and the specular color 

    :type:  enum in [‘MIX’, ‘ADD’, ‘MULTIPLY’, ‘SUBTRACT’, ‘SCREEN’, ‘DIVIDE’, ‘DIFFERENCE’, ‘DARKEN’, ‘LIGHTEN’, ‘OVERLAY’, ‘DODGE’, ‘BURN’, ‘HUE’, ‘SATURATION’, ‘VALUE’, ‘COLOR’, ‘SOFT_LIGHT’, ‘LINEAR_LIGHT’], default ‘MIX’ 
    '''

    specular_ramp_factor = None
    '''Blending factor (also uses alpha in Colorband) 

    :type:  float in [0, 1], default 0.0 
    '''

    specular_ramp_input = None
    '''How the ramp maps on the surface 

    :type:  enum in [‘SHADER’, ‘ENERGY’, ‘NORMAL’, ‘RESULT’], default ‘SHADER’ 
    '''

    specular_shader = None
    '''

    :type:  enum in [‘COOKTORR’, ‘PHONG’, ‘BLINN’, ‘TOON’, ‘WARDISO’], default ‘COOKTORR’ 
    '''

    specular_slope = None
    '''The standard deviation of surface slope 

    :type:  float in [0, 0.4], default 0.0 
    '''

    specular_toon_size = None
    '''Size of specular toon area 

    :type:  float in [0, 1.53], default 0.0 
    '''

    specular_toon_smooth = None
    '''Smoothness of specular toon area 

    :type:  float in [0, 1], default 0.0 
    '''

    strand = None
    '''Strand settings for the material 

    :type:  MaterialStrand, (readonly, never None) 
    '''

    subsurface_scattering = None
    '''Subsurface scattering settings for the material 

    :type:  MaterialSubsurfaceScattering, (readonly, never None) 
    '''

    texture_paint_images = None
    '''Texture images used for texture painting 

    :type:  bpy_prop_collection of Image, (readonly) 
    '''

    texture_paint_slots = None
    '''Texture slots defining the mapping and influence of textures 

    :type:  bpy_prop_collection of TexPaintSlot, (readonly) 
    '''

    texture_slots = None
    '''Texture slots defining the mapping and influence of textures 

    :type:  MaterialTextureSlots bpy_prop_collection of MaterialTextureSlot, (readonly) 
    '''

    translucency = None
    '''Amount of diffuse shading on the back side 

    :type:  float in [0, 1], default 0.0 
    '''

    transparency_method = None
    '''Method to use for rendering transparency 

    :type:  enum in [‘MASK’, ‘Z_TRANSPARENCY’, ‘RAYTRACE’], default ‘MASK’ 
    '''

    type = None
    '''Material type defining how the object is rendered 

    :type:  enum in [‘SURFACE’, ‘WIRE’, ‘VOLUME’, ‘HALO’], default ‘SURFACE’ 
    '''

    use_cast_approximate = None
    '''Allow this material to cast shadows when using approximate ambient occlusion 

    :type:  boolean, default False 
    '''

    use_cast_buffer_shadows = None
    '''Allow this material to cast shadows from shadow buffer lamps 

    :type:  boolean, default False 
    '''

    use_cast_shadows = None
    '''Allow this material to cast shadows 

    :type:  boolean, default False 
    '''

    use_cast_shadows_only = None
    '''Make objects with this material appear invisible (not rendered), only casting shadows 

    :type:  boolean, default False 
    '''

    use_cubic = None
    '''Use cubic interpolation for diffuse values, for smoother transitions 

    :type:  boolean, default False 
    '''

    use_diffuse_ramp = None
    '''Toggle diffuse ramp operations 

    :type:  boolean, default False 
    '''

    use_face_texture = None
    '''Replace the object’s base color with color from UV map image textures 

    :type:  boolean, default False 
    '''

    use_face_texture_alpha = None
    '''Replace the object’s base alpha value with alpha from UV map image textures 

    :type:  boolean, default False 
    '''

    use_full_oversampling = None
    '''Force this material to render full shading/textures for all anti-aliasing samples 

    :type:  boolean, default False 
    '''

    use_light_group_exclusive = None
    '''Material uses the light group exclusively - these lamps are excluded from other scene lighting 

    :type:  boolean, default False 
    '''

    use_light_group_local = None
    '''When linked in, material uses local light group with the same name 

    :type:  boolean, default False 
    '''

    use_mist = None
    '''Use mist with this material (in world settings) 

    :type:  boolean, default False 
    '''

    use_nodes = None
    '''Use shader nodes to render the material 

    :type:  boolean, default False 
    '''

    use_object_color = None
    '''Modulate the result with a per-object color 

    :type:  boolean, default False 
    '''

    use_only_shadow = None
    '''Render shadows as the material’s alpha value, making the material transparent except for shadowed areas 

    :type:  boolean, default False 
    '''

    use_ray_shadow_bias = None
    '''Prevent raytraced shadow errors on surfaces with smooth shaded normals (terminator problem) 

    :type:  boolean, default False 
    '''

    use_raytrace = None
    '''Include this material and geometry that uses it in raytracing calculations 

    :type:  boolean, default False 
    '''

    use_shadeless = None
    '''Make this material insensitive to light or shadow 

    :type:  boolean, default False 
    '''

    use_shadows = None
    '''Allow this material to receive shadows 

    :type:  boolean, default False 
    '''

    use_sky = None
    '''Render this material with zero alpha, with sky background in place (scanline only) 

    :type:  boolean, default False 
    '''

    use_specular_ramp = None
    '''Toggle specular ramp operations 

    :type:  boolean, default False 
    '''

    use_tangent_shading = None
    '''Use the material’s tangent vector instead of the normal for shading - for anisotropic shading effects 

    :type:  boolean, default False 
    '''

    use_textures = None
    '''Enable/Disable each texture 

    :type:  boolean array of 18 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    use_transparency = None
    '''Render material as transparent 

    :type:  boolean, default False 
    '''

    use_transparent_shadows = None
    '''Allow this object to receive transparent shadows cast through other objects 

    :type:  boolean, default False 
    '''

    use_uv_project = None
    '''Use to ensure UV interpolation is correct for camera projections (use with UV project modifier) 

    :type:  boolean, default False 
    '''

    use_vertex_color_light = None
    '''Add vertex colors as additional lighting 

    :type:  boolean, default False 
    '''

    use_vertex_color_paint = None
    '''Replace object base color with vertex colors (multiply with ‘texture face’ face assigned textures) 

    :type:  boolean, default False 
    '''

    volume = None
    '''Volume settings for the material 

    :type:  MaterialVolume, (readonly, never None) 
    '''


class MaterialGameSettings:
    '''Game Engine settings for a Material data-block '''

    alpha_blend = None
    '''Blend Mode for Transparent Faces 

    :type:  enum in [‘OPAQUE’, ‘ADD’, ‘CLIP’, ‘ALPHA’, ‘ALPHA_SORT’, ‘ALPHA_ANTIALIASING’], default ‘OPAQUE’ 
    '''

    face_orientation = None
    '''Especial face orientation options 

    :type:  enum in [‘NORMAL’, ‘HALO’, ‘BILLBOARD’, ‘SHADOW’], default ‘NORMAL’ 
    '''

    invisible = None
    '''Make face invisible 

    :type:  boolean, default False 
    '''

    physics = None
    '''Use physics properties of materials 

    :type:  boolean, default False 
    '''

    text = None
    '''Use material as text in Game Engine 

    :type:  boolean, default False 
    '''

    use_backface_culling = None
    '''Hide Back of the face in Game Engine 

    :type:  boolean, default False 
    '''


class MaterialHalo:
    '''Halo particle effect settings for a Material data-block '''

    add = None
    '''Strength of the add effect 

    :type:  float in [0, 1], default 0.0 
    '''

    flare_boost = None
    '''Give the flare extra strength 

    :type:  float in [0.1, 10], default 0.0 
    '''

    flare_seed = None
    '''Offset in the flare seed table 

    :type:  int in [0, 255], default 0 
    '''

    flare_size = None
    '''Factor by which the flare is larger than the halo 

    :type:  float in [0.1, 25], default 0.0 
    '''

    flare_subflare_count = None
    '''Number of sub-flares 

    :type:  int in [1, 32], default 0 
    '''

    flare_subflare_size = None
    '''Dimension of the sub-flares, dots and circles 

    :type:  float in [0.1, 25], default 0.0 
    '''

    hardness = None
    '''Hardness of the halo 

    :type:  int in [0, 127], default 0 
    '''

    line_count = None
    '''Number of star shaped lines rendered over the halo 

    :type:  int in [0, 250], default 0 
    '''

    ring_count = None
    '''Number of rings rendered over the halo 

    :type:  int in [0, 24], default 0 
    '''

    seed = None
    '''Randomize ring dimension and line location 

    :type:  int in [0, 255], default 0 
    '''

    size = None
    '''Dimension of the halo 

    :type:  float in [0, 100], default 0.0 
    '''

    star_tip_count = None
    '''Number of points on the star shaped halo 

    :type:  int in [3, 50], default 0 
    '''

    use_extreme_alpha = None
    '''Use extreme alpha 

    :type:  boolean, default False 
    '''

    use_flare_mode = None
    '''Render halo as a lens flare 

    :type:  boolean, default False 
    '''

    use_lines = None
    '''Render star shaped lines over halo 

    :type:  boolean, default False 
    '''

    use_ring = None
    '''Render rings over halo 

    :type:  boolean, default False 
    '''

    use_shaded = None
    '''Let halo receive light and shadows from external objects 

    :type:  boolean, default False 
    '''

    use_soft = None
    '''Soften the edges of halos at intersections with other geometry 

    :type:  boolean, default False 
    '''

    use_star = None
    '''Render halo as a star 

    :type:  boolean, default False 
    '''

    use_texture = None
    '''Give halo a texture 

    :type:  boolean, default False 
    '''

    use_vertex_normal = None
    '''Use the vertex normal to specify the dimension of the halo 

    :type:  boolean, default False 
    '''


class MaterialPhysics:
    '''Physics settings for a Material data-block '''

    elasticity = None
    '''Elasticity of collisions 

    :type:  float in [0, 1], default 0.0 
    '''

    fh_damping = None
    '''Damping of the spring force, when inside the physics distance area 

    :type:  float in [0, 1], default 0.0 
    '''

    fh_distance = None
    '''Distance of the physics area 

    :type:  float in [0, 20], default 0.0 
    '''

    fh_force = None
    '''Upward spring force, when inside the physics distance area 

    :type:  float in [0, 1], default 0.0 
    '''

    friction = None
    '''Coulomb friction coefficient, when inside the physics distance area 

    :type:  float in [0, 100], default 0.0 
    '''

    use_fh_normal = None
    '''Align dynamic game objects along the surface normal, when inside the physics distance area 

    :type:  boolean, default False 
    '''


class MaterialRaytraceMirror:
    '''Raytraced reflection settings for a Material data-block '''

    depth = None
    '''Maximum allowed number of light inter-reflections 

    :type:  int in [0, 32767], default 0 
    '''

    distance = None
    '''Maximum distance of reflected rays (reflections further than this range fade to sky color or material color) 

    :type:  float in [0, 10000], default 0.0 
    '''

    fade_to = None
    '''The color that rays with no intersection within the Max Distance take (material color can be best for indoor scenes, sky color for outdoor) 

    :type:  enum in [‘FADE_TO_SKY’, ‘FADE_TO_MATERIAL’], default ‘FADE_TO_SKY’ 
    '''

    fresnel = None
    '''Power of Fresnel for mirror reflection 

    :type:  float in [0, 5], default 0.0 
    '''

    fresnel_factor = None
    '''Blending factor for Fresnel 

    :type:  float in [0, 5], default 0.0 
    '''

    gloss_anisotropic = None
    '''The shape of the reflection, from 0.0 (circular) to 1.0 (fully stretched along the tangent 

    :type:  float in [0, 1], default 0.0 
    '''

    gloss_factor = None
    '''The shininess of the reflection (values < 1.0 give diffuse, blurry reflections) 

    :type:  float in [0, 1], default 0.0 
    '''

    gloss_samples = None
    '''Number of cone samples averaged for blurry reflections 

    :type:  int in [0, 1024], default 0 
    '''

    gloss_threshold = None
    '''Threshold for adaptive sampling (if a sample contributes less than this amount [as a percentage], sampling is stopped) 

    :type:  float in [0, 1], default 0.0 
    '''

    reflect_factor = None
    '''Amount of mirror reflection for raytrace 

    :type:  float in [0, 1], default 0.0 
    '''

    use = None
    '''Enable raytraced reflections 

    :type:  boolean, default False 
    '''


class MaterialRaytraceTransparency:
    '''Raytraced refraction settings for a Material data-block '''

    depth = None
    '''Maximum allowed number of light inter-refractions 

    :type:  int in [0, 32767], default 0 
    '''

    depth_max = None
    '''Maximum depth for light to travel through the transparent material before becoming fully filtered (0.0 is disabled) 

    :type:  float in [0, 100], default 0.0 
    '''

    falloff = None
    '''Falloff power for transmissivity filter effect (1.0 is linear) 

    :type:  float in [0.1, 10], default 0.0 
    '''

    filter = None
    '''Amount to blend in the material’s diffuse color in raytraced transparency (simulating absorption) 

    :type:  float in [0, 1], default 0.0 
    '''

    fresnel = None
    '''Power of Fresnel for transparency (Ray or ZTransp) 

    :type:  float in [0, 5], default 0.0 
    '''

    fresnel_factor = None
    '''Blending factor for Fresnel 

    :type:  float in [1, 5], default 0.0 
    '''

    gloss_factor = None
    '''The clarity of the refraction. Values < 1.0 give diffuse, blurry refractions 

    :type:  float in [0, 1], default 0.0 
    '''

    gloss_samples = None
    '''Number of cone samples averaged for blurry refractions 

    :type:  int in [0, 1024], default 0 
    '''

    gloss_threshold = None
    '''Threshold for adaptive sampling. If a sample contributes less than this amount (as a percentage), sampling is stopped 

    :type:  float in [0, 1], default 0.0 
    '''

    ior = None
    '''Angular index of refraction for raytraced refraction 

    :type:  float in [0.25, 4], default 0.0 
    '''


class MaterialSlot:
    '''Material slot in an object '''

    link = None
    '''Link material to object or the object’s data 

    :type:  enum in [‘OBJECT’, ‘DATA’], default ‘DATA’ 
    '''

    material = None
    '''Material data-block used by this material slot 

    :type:  Material 
    '''

    name = None
    '''Material slot name 

    :type:  string, default “”, (readonly, never None) 
    '''


class MaterialStrand:
    '''Strand settings for a Material data-block '''

    blend_distance = None
    '''Worldspace distance over which to blend in the surface normal 

    :type:  float in [0, 10], default 0.0 
    '''

    root_size = None
    '''Start size of strands in pixels or Blender units 

    :type:  float in [0, inf], default 0.0 
    '''

    shape = None
    '''Positive values make strands rounder, negative ones make strands spiky 

    :type:  float in [-0.9, 0.9], default 0.0 
    '''

    size_min = None
    '''Minimum size of strands in pixels 

    :type:  float in [0.001, 10], default 0.0 
    '''

    tip_size = None
    '''End size of strands in pixels or Blender units 

    :type:  float in [0, inf], default 0.0 
    '''

    use_blender_units = None
    '''Use Blender units for widths instead of pixels 

    :type:  boolean, default False 
    '''

    use_surface_diffuse = None
    '''Make diffuse shading more similar to shading the surface 

    :type:  boolean, default False, (readonly) 
    '''

    use_tangent_shading = None
    '''Use direction of strands as normal for tangent-shading 

    :type:  boolean, default False 
    '''

    uv_layer = None
    '''Name of UV map to override 

    :type:  string, default “”, (never None) 
    '''

    width_fade = None
    '''Transparency along the width of the strand 

    :type:  float in [0, 2], default 0.0 
    '''


class MaterialSubsurfaceScattering:
    '''Diffuse subsurface scattering settings for a Material data-block '''

    back = None
    '''Back scattering weight 

    :type:  float in [0, 10], default 0.0 
    '''

    color = None
    '''Scattering color 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    color_factor = None
    '''Blend factor for SSS colors 

    :type:  float in [0, 1], default 0.0 
    '''

    error_threshold = None
    '''Error tolerance (low values are slower and higher quality) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    front = None
    '''Front scattering weight 

    :type:  float in [0, 2], default 0.0 
    '''

    ior = None
    '''Index of refraction (higher values are denser) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    radius = None
    '''Mean red/green/blue scattering path length 

    :type:  float array of 3 items in [0.001, inf], default (0.0, 0.0, 0.0) 
    '''

    scale = None
    '''Object scale factor 

    :type:  float in [-inf, inf], default 0.0 
    '''

    texture_factor = None
    '''Texture scattering blend factor 

    :type:  float in [0, 1], default 0.0 
    '''

    use = None
    '''Enable diffuse subsurface scattering effects in a material 

    :type:  boolean, default False 
    '''


class MaterialTextureSlot:
    '''Texture slot for textures in a Material data-block '''

    alpha_factor = None
    '''Amount texture affects alpha 

    :type:  float in [-inf, inf], default 0.0 
    '''

    ambient_factor = None
    '''Amount texture affects ambient 

    :type:  float in [-inf, inf], default 0.0 
    '''

    bump_method = None
    '''Method to use for bump mapping 

    :type:  enum in [‘BUMP_ORIGINAL’, ‘BUMP_COMPATIBLE’, ‘BUMP_LOW_QUALITY’, ‘BUMP_MEDIUM_QUALITY’, ‘BUMP_BEST_QUALITY’], default ‘BUMP_ORIGINAL’ 
    '''

    bump_objectspace = None
    '''Space to apply bump mapping in 

    :type:  enum in [‘BUMP_VIEWSPACE’, ‘BUMP_OBJECTSPACE’, ‘BUMP_TEXTURESPACE’], default ‘BUMP_VIEWSPACE’ 
    '''

    density_factor = None
    '''Amount texture affects density 

    :type:  float in [-inf, inf], default 0.0 
    '''

    diffuse_color_factor = None
    '''Amount texture affects diffuse color 

    :type:  float in [-inf, inf], default 0.0 
    '''

    diffuse_factor = None
    '''Amount texture affects diffuse reflectivity 

    :type:  float in [-inf, inf], default 0.0 
    '''

    displacement_factor = None
    '''Amount texture displaces the surface 

    :type:  float in [-inf, inf], default 0.0 
    '''

    emission_color_factor = None
    '''Amount texture affects emission color 

    :type:  float in [-inf, inf], default 0.0 
    '''

    emission_factor = None
    '''Amount texture affects emission 

    :type:  float in [-inf, inf], default 0.0 
    '''

    emit_factor = None
    '''Amount texture affects emission 

    :type:  float in [-inf, inf], default 0.0 
    '''

    hardness_factor = None
    '''Amount texture affects hardness 

    :type:  float in [-inf, inf], default 0.0 
    '''

    mapping = None
    '''

    :type:  enum in [‘FLAT’, ‘CUBE’, ‘TUBE’, ‘SPHERE’], default ‘FLAT’ 
    '''

    mapping_x = None
    '''

    :type:  enum in [‘NONE’, ‘X’, ‘Y’, ‘Z’], default ‘NONE’ 
    '''

    mapping_y = None
    '''

    :type:  enum in [‘NONE’, ‘X’, ‘Y’, ‘Z’], default ‘NONE’ 
    '''

    mapping_z = None
    '''

    :type:  enum in [‘NONE’, ‘X’, ‘Y’, ‘Z’], default ‘NONE’ 
    '''

    mirror_factor = None
    '''Amount texture affects mirror color 

    :type:  float in [-inf, inf], default 0.0 
    '''

    normal_factor = None
    '''Amount texture affects normal values 

    :type:  float in [-inf, inf], default 0.0 
    '''

    normal_map_space = None
    '''Set space of normal map image 

    :type:  enum in [‘CAMERA’, ‘WORLD’, ‘OBJECT’, ‘TANGENT’], default ‘CAMERA’ 
    '''

    object = None
    '''Object to use for mapping with Object texture coordinates 

    :type:  Object 
    '''

    raymir_factor = None
    '''Amount texture affects ray mirror 

    :type:  float in [-inf, inf], default 0.0 
    '''

    reflection_color_factor = None
    '''Amount texture affects color of out-scattered light 

    :type:  float in [-inf, inf], default 0.0 
    '''

    reflection_factor = None
    '''Amount texture affects brightness of out-scattered light 

    :type:  float in [-inf, inf], default 0.0 
    '''

    scattering_factor = None
    '''Amount texture affects scattering 

    :type:  float in [-inf, inf], default 0.0 
    '''

    specular_color_factor = None
    '''Amount texture affects specular color 

    :type:  float in [-inf, inf], default 0.0 
    '''

    specular_factor = None
    '''Amount texture affects specular reflectivity 

    :type:  float in [-inf, inf], default 0.0 
    '''

    texture_coords = None
    '''

    :type:  enum in [‘GLOBAL’, ‘OBJECT’, ‘UV’, ‘ORCO’, ‘STRAND’, ‘WINDOW’, ‘NORMAL’, ‘REFLECTION’, ‘STRESS’, ‘TANGENT’], default ‘GLOBAL’ 
    '''

    translucency_factor = None
    '''Amount texture affects translucency 

    :type:  float in [-inf, inf], default 0.0 
    '''

    transmission_color_factor = None
    '''Amount texture affects result color after light has been scattered/absorbed 

    :type:  float in [-inf, inf], default 0.0 
    '''

    use = None
    '''Enable this material texture slot 

    :type:  boolean, default False 
    '''

    use_from_dupli = None
    '''Dupli’s instanced from verts, faces or particles, inherit texture coordinate from their parent 

    :type:  boolean, default False 
    '''

    use_from_original = None
    '''Dupli’s derive their object coordinates from the original object’s transformation 

    :type:  boolean, default False 
    '''

    use_map_alpha = None
    '''The texture affects the alpha value 

    :type:  boolean, default False 
    '''

    use_map_ambient = None
    '''The texture affects the value of ambient 

    :type:  boolean, default False 
    '''

    use_map_color_diffuse = None
    '''The texture affects basic color of the material 

    :type:  boolean, default False 
    '''

    use_map_color_emission = None
    '''The texture affects the color of emission 

    :type:  boolean, default False 
    '''

    use_map_color_reflection = None
    '''The texture affects the color of scattered light 

    :type:  boolean, default False 
    '''

    use_map_color_spec = None
    '''The texture affects the specularity color 

    :type:  boolean, default False 
    '''

    use_map_color_transmission = None
    '''The texture affects the result color after other light has been scattered/absorbed 

    :type:  boolean, default False 
    '''

    use_map_density = None
    '''The texture affects the volume’s density 

    :type:  boolean, default False 
    '''

    use_map_diffuse = None
    '''The texture affects the value of diffuse reflectivity 

    :type:  boolean, default False 
    '''

    use_map_displacement = None
    '''Let the texture displace the surface 

    :type:  boolean, default False 
    '''

    use_map_emission = None
    '''The texture affects the volume’s emission 

    :type:  boolean, default False 
    '''

    use_map_emit = None
    '''The texture affects the emit value 

    :type:  boolean, default False 
    '''

    use_map_hardness = None
    '''The texture affects the hardness value 

    :type:  boolean, default False 
    '''

    use_map_mirror = None
    '''The texture affects the mirror color 

    :type:  boolean, default False 
    '''

    use_map_normal = None
    '''The texture affects the rendered normal 

    :type:  boolean, default False 
    '''

    use_map_raymir = None
    '''The texture affects the ray-mirror value 

    :type:  boolean, default False 
    '''

    use_map_reflect = None
    '''The texture affects the reflected light’s brightness 

    :type:  boolean, default False 
    '''

    use_map_scatter = None
    '''The texture affects the volume’s scattering 

    :type:  boolean, default False 
    '''

    use_map_specular = None
    '''The texture affects the value of specular reflectivity 

    :type:  boolean, default False 
    '''

    use_map_to_bounds = None
    '''Map coordinates in object bounds 

    :type:  boolean, default False 
    '''

    use_map_translucency = None
    '''The texture affects the translucency value 

    :type:  boolean, default False 
    '''

    use_map_warp = None
    '''Let the texture warp texture coordinates of next channels 

    :type:  boolean, default False 
    '''

    uv_layer = None
    '''UV map to use for mapping with UV texture coordinates 

    :type:  string, default “”, (never None) 
    '''

    warp_factor = None
    '''Amount texture affects texture coordinates of next channels 

    :type:  float in [-inf, inf], default 0.0 
    '''


class MaterialTextureSlots:
    '''Collection of texture slots '''

    pass


class MaterialVolume:
    '''Volume rendering settings for a Material data-block '''

    asymmetry = None
    '''Back scattering (-1.0) to Forward scattering (1.0) and the range in between 

    :type:  float in [-1, 1], default 0.0 
    '''

    cache_resolution = None
    '''Resolution of the voxel grid, low resolutions are faster, high resolutions use more memory 

    :type:  int in [1, 1024], default 0 
    '''

    density = None
    '''The base density of the volume 

    :type:  float in [0, 1], default 0.0 
    '''

    density_scale = None
    '''Multiplier for the material’s density 

    :type:  float in [0, inf], default 0.0 
    '''

    depth_threshold = None
    '''Stop ray marching early if transmission drops below this luminance - higher values give speedups in dense volumes at the expense of accuracy 

    :type:  float in [0, 1], default 0.0 
    '''

    emission = None
    '''Amount of light that gets emitted by the volume 

    :type:  float in [0, inf], default 0.0 
    '''

    emission_color = None
    '''Color of emitted light 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    light_method = None
    '''Method of shading, attenuating, and scattering light through the volume 

    :type:  enum in [‘SHADELESS’, ‘SHADOWED’, ‘SHADED’, ‘MULTIPLE_SCATTERING’, ‘SHADED_PLUS_MULTIPLE_SCATTERING’], default ‘SHADELESS’ 
    '''

    ms_diffusion = None
    '''Diffusion factor, the strength of the blurring effect 

    :type:  float in [0, inf], default 0.0 
    '''

    ms_intensity = None
    '''Multiplier for multiple scattered light energy 

    :type:  float in [0, inf], default 0.0 
    '''

    ms_spread = None
    '''Proportional distance over which the light is diffused 

    :type:  float in [0, inf], default 0.0 
    '''

    reflection = None
    '''Multiplier to make out-scattered light brighter or darker (non-physically correct) 

    :type:  float in [0, inf], default 0.0 
    '''

    reflection_color = None
    '''Color of light scattered out of the volume (does not affect transmission) 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    scattering = None
    '''Amount of light that gets scattered out by the volume - the more out-scattering, the shallower the light will penetrate 

    :type:  float in [0, inf], default 0.0 
    '''

    step_method = None
    '''Method of calculating the steps through the volume 

    :type:  enum in [‘RANDOMIZED’, ‘CONSTANT’], default ‘RANDOMIZED’ 
    '''

    step_size = None
    '''Distance between subsequent volume depth samples 

    :type:  float in [0, inf], default 0.0 
    '''

    transmission_color = None
    '''Result color of the volume, after other light has been scattered/absorbed 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    use_external_shadows = None
    '''Receive shadows from sources outside the volume (temporary) 

    :type:  boolean, default False 
    '''

    use_light_cache = None
    '''Pre-calculate the shading information into a voxel grid, speeds up shading at slightly less accuracy 

    :type:  boolean, default False 
    '''


class Menu:
    '''Editor menu containing buttons '''

    bl_description = None
    '''

    :type:  string, default “” 
    '''

    bl_idname = None
    '''If this is set, the menu gets a custom ID, otherwise it takes the name of the class used to define the menu (for example, if the class name is “OBJECT_MT_hello”, and bl_idname is not set by the script, then bl_idname = “OBJECT_MT_hello”) 

    :type:  string, default “”, (never None) 
    '''

    bl_label = None
    '''The menu label 

    :type:  string, default “”, (never None) 
    '''

    bl_translation_context = None
    '''

    :type:  string, default “*”, (never None) 
    '''

    layout = None
    '''Defines the structure of the menu in the UI 

    :type:  UILayout, (readonly) 
    '''

    def draw(self, context):
        '''Draw UI elements into the menu UI layout 

        '''
        pass

    def draw_preset(self, context):
        '''Optionally: - preset_extensions (set of strings) - preset_operator_defaults (dict of keyword args) 

        '''
        pass

    def path_menu(self,
                  searchpaths,
                  operator,
                  props_default=None,
                  filter_ext=None):
        '''

        '''
        pass


class Mesh:
    '''Mesh data-block defining geometric surfaces '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    auto_smooth_angle = None
    '''Maximum angle between face normals that will be considered as smooth (unused if custom split normals data are available) 

    :type:  float in [0, 3.14159], default 3.14159 
    '''

    auto_texspace = None
    '''Adjust active object’s texture space automatically when transforming object 

    :type:  boolean, default False 
    '''

    cycles = None
    '''Cycles mesh settings 

    :type:  CyclesMeshSettings, (readonly) 
    '''

    edges = None
    '''Edges of the mesh 

    :type:  MeshEdges bpy_prop_collection of MeshEdge, (readonly) 
    '''

    has_custom_normals = None
    '''True if there are custom split normals data in this mesh 

    :type:  boolean, default False, (readonly) 
    '''

    is_editmode = None
    '''True when used in editmode 

    :type:  boolean, default False, (readonly) 
    '''

    loops = None
    '''Loops of the mesh (polygon corners) 

    :type:  MeshLoops bpy_prop_collection of MeshLoop, (readonly) 
    '''

    materials = None
    '''

    :type:  IDMaterials bpy_prop_collection of Material, (readonly) 
    '''

    polygon_layers_float = None
    '''

    :type:  PolygonFloatProperties bpy_prop_collection of MeshPolygonFloatPropertyLayer, (readonly) 
    '''

    polygon_layers_int = None
    '''

    :type:  PolygonIntProperties bpy_prop_collection of MeshPolygonIntPropertyLayer, (readonly) 
    '''

    polygon_layers_string = None
    '''

    :type:  PolygonStringProperties bpy_prop_collection of MeshPolygonStringPropertyLayer, (readonly) 
    '''

    polygons = None
    '''Polygons of the mesh 

    :type:  MeshPolygons bpy_prop_collection of MeshPolygon, (readonly) 
    '''

    shape_keys = None
    '''

    :type:  Key, (readonly) 
    '''

    show_double_sided = None
    '''Display the mesh with double or single sided lighting (OpenGL only) 

    :type:  boolean, default False 
    '''

    show_edge_bevel_weight = None
    '''Display weights created for the Bevel modifier 

    :type:  boolean, default False 
    '''

    show_edge_crease = None
    '''Display creases created for subsurf weighting 

    :type:  boolean, default False 
    '''

    show_edge_seams = None
    '''Display UV unwrapping seams 

    :type:  boolean, default False 
    '''

    show_edge_sharp = None
    '''Display sharp edges, used with the EdgeSplit modifier 

    :type:  boolean, default False 
    '''

    show_edges = None
    '''Display selected edges using highlights in the 3D view and UV editor 

    :type:  boolean, default False 
    '''

    show_extra_edge_angle = None
    '''Display selected edge angle, using global values when set in the transform panel 

    :type:  boolean, default False 
    '''

    show_extra_edge_length = None
    '''Display selected edge lengths, using global values when set in the transform panel 

    :type:  boolean, default False 
    '''

    show_extra_face_angle = None
    '''Display the angles in the selected edges, using global values when set in the transform panel 

    :type:  boolean, default False 
    '''

    show_extra_face_area = None
    '''Display the area of selected faces, using global values when set in the transform panel 

    :type:  boolean, default False 
    '''

    show_extra_indices = None
    '''Display the index numbers of selected vertices, edges, and faces 

    :type:  boolean, default False 
    '''

    show_faces = None
    '''Display all faces as shades in the 3D view and UV editor 

    :type:  boolean, default False 
    '''

    show_freestyle_edge_marks = None
    '''Display Freestyle edge marks, used with the Freestyle renderer 

    :type:  boolean, default False 
    '''

    show_freestyle_face_marks = None
    '''Display Freestyle face marks, used with the Freestyle renderer 

    :type:  boolean, default False 
    '''

    show_normal_face = None
    '''Display face normals as lines 

    :type:  boolean, default False 
    '''

    show_normal_loop = None
    '''Display vertex-per-face normals as lines 

    :type:  boolean, default False 
    '''

    show_normal_vertex = None
    '''Display vertex normals as lines 

    :type:  boolean, default False 
    '''

    show_statvis = None
    '''Display statistical information about the mesh 

    :type:  boolean, default False 
    '''

    show_weight = None
    '''Draw weights in editmode 

    :type:  boolean, default False 
    '''

    skin_vertices = None
    '''All skin vertices 

    :type:  bpy_prop_collection of MeshSkinVertexLayer, (readonly) 
    '''

    tessface_uv_textures = None
    '''All UV maps for tessellated faces (read-only, for use by renderers) 

    :type:  TessfaceUVTextures bpy_prop_collection of MeshTextureFaceLayer, (readonly) 
    '''

    tessface_vertex_colors = None
    '''All tessellated face colors (read-only, for use by renderers) 

    :type:  VertexColors bpy_prop_collection of MeshColorLayer, (readonly) 
    '''

    tessfaces = None
    '''Tessellated faces of the mesh (derived from polygons) 

    :type:  MeshTessFaces bpy_prop_collection of MeshTessFace, (readonly) 
    '''

    texco_mesh = None
    '''Derive texture coordinates from another mesh 

    :type:  Mesh 
    '''

    texspace_location = None
    '''Texture space location 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    texspace_size = None
    '''Texture space size 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    texture_mesh = None
    '''Use another mesh for texture indices (vertex indices must be aligned) 

    :type:  Mesh 
    '''

    total_edge_sel = None
    '''Selected edge count in editmode 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    total_face_sel = None
    '''Selected face count in editmode 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    total_vert_sel = None
    '''Selected vertex count in editmode 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    use_auto_smooth = None
    '''Auto smooth (based on smooth/sharp faces/edges and angle between faces), or use custom split normals data if available 

    :type:  boolean, default False 
    '''

    use_auto_texspace = None
    '''Adjust active object’s texture space automatically when transforming object 

    :type:  boolean, default False 
    '''

    use_customdata_edge_bevel = None
    '''

    :type:  boolean, default False 
    '''

    use_customdata_edge_crease = None
    '''

    :type:  boolean, default False 
    '''

    use_customdata_vertex_bevel = None
    '''

    :type:  boolean, default False 
    '''

    use_mirror_topology = None
    '''Use topology based mirroring (for when both sides of mesh have matching, unique topology) 

    :type:  boolean, default False 
    '''

    use_mirror_x = None
    '''X Axis mirror editing 

    :type:  boolean, default False 
    '''

    use_paint_mask = None
    '''Face selection masking for painting 

    :type:  boolean, default False 
    '''

    use_paint_mask_vertex = None
    '''Vertex selection masking for painting (weight paint only) 

    :type:  boolean, default False 
    '''

    uv_layer_clone = None
    '''UV loop layer to be used as cloning source 

    :type:  MeshUVLoopLayer 
    '''

    uv_layer_clone_index = None
    '''Clone UV loop layer index 

    :type:  int in [0, inf], default 0 
    '''

    uv_layer_stencil = None
    '''UV loop layer to mask the painted area 

    :type:  MeshUVLoopLayer 
    '''

    uv_layer_stencil_index = None
    '''Mask UV loop layer index 

    :type:  int in [0, inf], default 0 
    '''

    uv_layers = None
    '''All UV loop layers 

    :type:  UVLoopLayers bpy_prop_collection of MeshUVLoopLayer, (readonly) 
    '''

    uv_texture_clone = None
    '''UV map to be used as cloning source 

    :type:  MeshTexturePolyLayer 
    '''

    uv_texture_clone_index = None
    '''Clone UV map index 

    :type:  int in [0, inf], default 0 
    '''

    uv_texture_stencil = None
    '''UV map to mask the painted area 

    :type:  MeshTexturePolyLayer 
    '''

    uv_texture_stencil_index = None
    '''Mask UV map index 

    :type:  int in [0, inf], default 0 
    '''

    uv_textures = None
    '''All UV maps 

    :type:  UVTextures bpy_prop_collection of MeshTexturePolyLayer, (readonly) 
    '''

    vertex_colors = None
    '''All vertex colors 

    :type:  LoopColors bpy_prop_collection of MeshLoopColorLayer, (readonly) 
    '''

    vertex_layers_float = None
    '''

    :type:  VertexFloatProperties bpy_prop_collection of MeshVertexFloatPropertyLayer, (readonly) 
    '''

    vertex_layers_int = None
    '''

    :type:  VertexIntProperties bpy_prop_collection of MeshVertexIntPropertyLayer, (readonly) 
    '''

    vertex_layers_string = None
    '''

    :type:  VertexStringProperties bpy_prop_collection of MeshVertexStringPropertyLayer, (readonly) 
    '''

    vertex_paint_masks = None
    '''Vertex paint mask 

    :type:  bpy_prop_collection of MeshPaintMaskLayer, (readonly) 
    '''

    vertices = None
    '''Vertices of the mesh 

    :type:  MeshVertices bpy_prop_collection of MeshVertex, (readonly) 
    '''

    edge_keys = None
    '''(readonly) '''

    def transform(self, matrix, shape_keys=False):
        '''Transform mesh vertices by a matrix (Warning: inverts normals if matrix is negative) 

        :param matrix: Matrix 
        :type matrix: float array of 16 items in [-inf, inf]
        :param shape_keys: Transform Shape Keys 
        :type shape_keys: boolean, (optional)
        '''
        pass

    def flip_normals(self):
        '''Invert winding of all polygons (clears tessellation, does not handle custom normals) 

        '''
        pass

    def calc_normals(self):
        '''Calculate vertex normals 

        '''
        pass

    def create_normals_split(self):
        '''Empty split vertex normals 

        '''
        pass

    def calc_normals_split(self):
        '''Calculate split vertex normals, which preserve sharp edges 

        '''
        pass

    def free_normals_split(self):
        '''Free split vertex normals 

        '''
        pass

    def calc_tangents(self, uvmap=""):
        '''Compute tangents and bitangent signs, to be used together with the split normals to get a complete tangent space for normal mapping (split normals are also computed if not yet present) 

        :param uvmap: Name of the UV map to use for tangent space computation 
        :type uvmap: string, (optional, never None)
        '''
        pass

    def free_tangents(self):
        '''Free tangents 

        '''
        pass

    def calc_tessface(self, free_mpoly=False):
        '''Calculate face tessellation (supports editmode too) 

        :param free_mpoly: Free MPoly, Free data used by polygons and loops. WARNING: This destructive operation removes regular faces, only used on temporary mesh data-blocks to reduce memory footprint of render engines and export scripts 
        :type free_mpoly: boolean, (optional)
        '''
        pass

    def calc_smooth_groups(self, use_bitflags=False):
        '''Calculate smooth groups from sharp edges 

        :param use_bitflags: Produce bitflags groups instead of simple numeric values 
        :type use_bitflags: boolean, (optional)
        '''
        pass

    def normals_split_custom_set(self, normals):
        '''Define custom split normals of this mesh (use zero-vectors to keep auto ones) 

        :param normals: Normals 
        :type normals: float array of 3 items in [-1, 1]
        '''
        pass

    def normals_split_custom_set_from_vertices(self, normals):
        '''Define custom split normals of this mesh, from vertices’ normals (use zero-vectors to keep auto ones) 

        :param normals: Normals 
        :type normals: float array of 3 items in [-1, 1]
        '''
        pass

    def update(self, calc_edges=False, calc_tessface=False):
        '''update 

        :param calc_edges: Calculate Edges, Force recalculation of edges 
        :type calc_edges: boolean, (optional)
        :param calc_tessface: Calculate Tessellation, Force recalculation of tessellation faces 
        :type calc_tessface: boolean, (optional)
        '''
        pass

    def unit_test_compare(self, mesh=None):
        '''unit_test_compare 

        :param mesh: Mesh to compare to 
        :type mesh: Mesh, (optional)
        :rtype: string, (never None) 
        :return:  Return value, String description of result of comparison 
        '''
        pass

    def validate(self, verbose=False, clean_customdata=True):
        '''Validate geometry, return True when the mesh has had invalid geometry corrected/removed 

        :param verbose: Verbose, Output information about the errors found 
        :type verbose: boolean, (optional)
        :param clean_customdata: Clean Custom Data, Remove temp/cached custom-data layers, like e.g. normals… 
        :type clean_customdata: boolean, (optional)
        :rtype: boolean 
        :return:  Result 
        '''
        pass

    def validate_material_indices(self):
        '''Validate material indices of polygons, return True when the mesh has had invalid indices corrected (to default 0) 

        :rtype: boolean 
        :return:  Result 
        '''
        pass

    def from_pydata(self, vertices, edges, faces):
        '''Make a mesh from a list of vertices/edges/faces Until we have a nicer way to make geometry, use this. 

        :param vertices: float triplets each representing (X, Y, Z) eg: [(0.0, 1.0, 0.5), …]. 
        :type vertices: iterable object
        :param edges: int pairs, each pair contains two indices to the vertices argument. eg: [(1, 2), …] 
        :type edges: iterable object
        :param faces: iterator of faces, each faces contains three or more indices to the vertices argument. eg: [(5, 6, 8, 9), (1, 2, 3), …] 
        :type faces: iterable object
        '''
        pass


class MeshCacheModifier:
    '''Cache Mesh '''

    cache_format = None
    '''

    :type:  enum in [‘MDD’, ‘PC2’], default ‘MDD’ 
    '''

    deform_mode = None
    '''

    :type:  enum in [‘OVERWRITE’, ‘INTEGRATE’], default ‘OVERWRITE’ 
    '''

    eval_factor = None
    '''Evaluation time in seconds 

    :type:  float in [0, 1], default 0.0 
    '''

    eval_frame = None
    '''The frame to evaluate (starting at 0) 

    :type:  float in [0, 500000], default 0.0 
    '''

    eval_time = None
    '''Evaluation time in seconds 

    :type:  float in [0, inf], default 0.0 
    '''

    factor = None
    '''Influence of the deformation 

    :type:  float in [0, 1], default 0.0 
    '''

    filepath = None
    '''Path to external displacements file 

    :type:  string, default “”, (never None) 
    '''

    flip_axis = None
    '''

    :type:  enum set in {‘X’, ‘Y’, ‘Z’}, default {‘X’} 
    '''

    forward_axis = None
    '''

    :type:  enum in [‘POS_X’, ‘POS_Y’, ‘POS_Z’, ‘NEG_X’, ‘NEG_Y’, ‘NEG_Z’], default ‘POS_X’ 
    '''

    frame_scale = None
    '''Evaluation time in seconds 

    :type:  float in [0, 100], default 0.0 
    '''

    frame_start = None
    '''Add this to the start frame 

    :type:  float in [-500000, 500000], default 0.0 
    '''

    interpolation = None
    '''

    :type:  enum in [‘NONE’, ‘LINEAR’], default ‘NONE’ 
    '''

    play_mode = None
    '''

    :type:  enum in [‘SCENE’, ‘CUSTOM’], default ‘SCENE’ 
    '''

    time_mode = None
    '''Method to control playback time 

    :type:  enum in [‘FRAME’, ‘TIME’, ‘FACTOR’], default ‘FRAME’ 
    '''

    up_axis = None
    '''

    :type:  enum in [‘POS_X’, ‘POS_Y’, ‘POS_Z’, ‘NEG_X’, ‘NEG_Y’, ‘NEG_Z’], default ‘POS_X’ 
    '''


class MeshColor:
    '''Vertex colors for a face in a Mesh '''

    color1 = None
    '''

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    color2 = None
    '''

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    color3 = None
    '''

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    color4 = None
    '''

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''


class MeshColorLayer:
    '''Layer of vertex colors in a Mesh data-block '''

    active = None
    '''Sets the layer as active for display and editing 

    :type:  boolean, default False 
    '''

    active_render = None
    '''Sets the layer as active for rendering 

    :type:  boolean, default False 
    '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshColor, (readonly) 
    '''

    name = None
    '''Name of Vertex color layer 

    :type:  string, default “”, (never None) 
    '''


class MeshDeformModifier:
    '''Mesh deformation modifier to deform with other meshes '''

    invert_vertex_group = None
    '''Invert vertex group influence 

    :type:  boolean, default False 
    '''

    is_bound = None
    '''Whether geometry has been bound to control cage 

    :type:  boolean, default False, (readonly) 
    '''

    object = None
    '''Mesh object to deform with 

    :type:  Object 
    '''

    precision = None
    '''The grid size for binding 

    :type:  int in [2, 10], default 0 
    '''

    use_dynamic_bind = None
    '''Recompute binding dynamically on top of other deformers (slower and more memory consuming) 

    :type:  boolean, default False 
    '''

    vertex_group = None
    '''Vertex group name 

    :type:  string, default “”, (never None) 
    '''


class MeshEdge:
    '''Edge in a Mesh data-block '''

    bevel_weight = None
    '''Weight used by the Bevel modifier 

    :type:  float in [-inf, inf], default 0.0 
    '''

    crease = None
    '''Weight used by the Subsurf modifier for creasing 

    :type:  float in [0, 1], default 0.0 
    '''

    hide = None
    '''

    :type:  boolean, default False 
    '''

    index = None
    '''Index of this edge 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    is_loose = None
    '''Loose edge 

    :type:  boolean, default False 
    '''

    select = None
    '''

    :type:  boolean, default False 
    '''

    use_edge_sharp = None
    '''Sharp edge for the EdgeSplit modifier 

    :type:  boolean, default False 
    '''

    use_freestyle_mark = None
    '''Edge mark for Freestyle line rendering 

    :type:  boolean, default False 
    '''

    use_seam = None
    '''Seam edge for UV unwrapping 

    :type:  boolean, default False 
    '''

    vertices = None
    '''Vertex indices 

    :type:  int array of 2 items in [0, inf], default (0, 0) 
    '''

    key = None
    '''(readonly) '''


class MeshEdges:
    '''Collection of mesh edges '''

    def add(self, count=0):
        '''add 

        :param count: Count, Number of edges to add 
        :type count: int in [0, inf], (optional)
        '''
        pass


class MeshLoop:
    '''Loop in a Mesh data-block '''

    bitangent = None
    '''Bitangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents, use it only if really needed, slower access than bitangent_sign) 

    :type:  float array of 3 items in [-1, 1], default (0.0, 0.0, 0.0), (readonly) 
    '''

    bitangent_sign = None
    '''Sign of the bitangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents, bitangent = bitangent_sign * cross(normal, tangent)) 

    :type:  float in [-1, 1], default 0.0, (readonly) 
    '''

    edge_index = None
    '''Edge index 

    :type:  int in [0, inf], default 0 
    '''

    index = None
    '''Index of this loop 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    normal = None
    '''Local space unit length split normal vector of this vertex for this polygon (must be computed beforehand using calc_normals_split or calc_tangents) 

    :type:  float array of 3 items in [-1, 1], default (0.0, 0.0, 0.0) 
    '''

    tangent = None
    '''Local space unit length tangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents) 

    :type:  float array of 3 items in [-1, 1], default (0.0, 0.0, 0.0), (readonly) 
    '''

    vertex_index = None
    '''Vertex index 

    :type:  int in [0, inf], default 0 
    '''


class MeshLoopColor:
    '''Vertex loop colors in a Mesh '''

    color = None
    '''

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''


class MeshLoopColorLayer:
    '''Layer of vertex colors in a Mesh data-block '''

    active = None
    '''Sets the layer as active for display and editing 

    :type:  boolean, default False 
    '''

    active_render = None
    '''Sets the layer as active for rendering 

    :type:  boolean, default False 
    '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshLoopColor, (readonly) 
    '''

    name = None
    '''Name of Vertex color layer 

    :type:  string, default “”, (never None) 
    '''


class MeshLoops:
    '''Collection of mesh loops '''

    def add(self, count=0):
        '''add 

        :param count: Count, Number of loops to add 
        :type count: int in [0, inf], (optional)
        '''
        pass


class MeshPaintMaskLayer:
    '''Per-vertex paint mask data '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshPaintMaskProperty, (readonly) 
    '''


class MeshPaintMaskProperty:
    '''Floating point paint mask value '''

    value = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''


class MeshPolygon:
    '''Polygon in a Mesh data-block '''

    area = None
    '''Read only area of this polygon 

    :type:  float in [0, inf], default 0.0, (readonly) 
    '''

    center = None
    '''Center of this polygon 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0), (readonly) 
    '''

    hide = None
    '''

    :type:  boolean, default False 
    '''

    index = None
    '''Index of this polygon 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    loop_start = None
    '''Index of the first loop of this polygon 

    :type:  int in [0, inf], default 0 
    '''

    loop_total = None
    '''Number of loops used by this polygon 

    :type:  int in [0, inf], default 0 
    '''

    material_index = None
    '''

    :type:  int in [0, 32767], default 0 
    '''

    normal = None
    '''Local space unit length normal vector for this polygon 

    :type:  float array of 3 items in [-1, 1], default (0.0, 0.0, 0.0), (readonly) 
    '''

    select = None
    '''

    :type:  boolean, default False 
    '''

    use_freestyle_mark = None
    '''Face mark for Freestyle line rendering 

    :type:  boolean, default False 
    '''

    use_smooth = None
    '''

    :type:  boolean, default False 
    '''

    vertices = None
    '''Vertex indices 

    :type:  int array of 3 items in [0, inf], default (0, 0, 0) 
    '''

    edge_keys = None
    '''(readonly) '''

    loop_indices = None
    '''(readonly) '''

    def flip(self):
        '''Invert winding of this polygon (flip its normal) 

        '''
        pass


class MeshPolygonFloatProperty:
    '''User defined floating point number value in a float properties layer '''

    value = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''


class MeshPolygonFloatPropertyLayer:
    '''User defined layer of floating point number values '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshPolygonFloatProperty, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''


class MeshPolygonIntProperty:
    '''User defined integer number value in an integer properties layer '''

    value = None
    '''

    :type:  int in [-inf, inf], default 0 
    '''


class MeshPolygonIntPropertyLayer:
    '''User defined layer of integer number values '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshPolygonIntProperty, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''


class MeshPolygonStringProperty:
    '''User defined string text value in a string properties layer '''

    value = None
    '''

    :type:  string, default “”, (never None) 
    '''


class MeshPolygonStringPropertyLayer:
    '''User defined layer of string text values '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshPolygonStringProperty, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''


class MeshPolygons:
    '''Collection of mesh polygons '''

    active = None
    '''The active polygon for this mesh 

    :type:  int in [-inf, inf], default 0 
    '''

    def add(self, count=0):
        '''add 

        :param count: Count, Number of polygons to add 
        :type count: int in [0, inf], (optional)
        '''
        pass


class MeshSequenceCacheModifier:
    '''Cache Mesh '''

    cache_file = None
    '''

    :type:  CacheFile 
    '''

    object_path = None
    '''Path to the object in the Alembic archive used to lookup geometric data 

    :type:  string, default “”, (never None) 
    '''

    read_data = None
    '''

    :type:  enum set in {‘VERT’, ‘POLY’, ‘UV’, ‘COLOR’}, default {‘VERT’} 
    '''


class MeshSkinVertex:
    '''Per-vertex skin data for use with the Skin modifier '''

    radius = None
    '''Radius of the skin 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    use_loose = None
    '''If vertex has multiple adjacent edges, it is hulled to them directly 

    :type:  boolean, default False 
    '''

    use_root = None
    '''Vertex is a root for rotation calculations and armature generation 

    :type:  boolean, default False 
    '''


class MeshSkinVertexLayer:
    '''Per-vertex skin data for use with the Skin modifier '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshSkinVertex, (readonly) 
    '''

    name = None
    '''Name of skin layer 

    :type:  string, default “”, (never None) 
    '''


class MeshStatVis:
    distort_max = None
    '''Maximum angle to display 

    :type:  float in [0, 3.14159], default 0.5 
    '''

    distort_min = None
    '''Minimum angle to display 

    :type:  float in [0, 3.14159], default 0.5 
    '''

    overhang_axis = None
    '''

    :type:  enum in [‘POS_X’, ‘POS_Y’, ‘POS_Z’, ‘NEG_X’, ‘NEG_Y’, ‘NEG_Z’], default ‘POS_X’ 
    '''

    overhang_max = None
    '''Maximum angle to display 

    :type:  float in [0, 3.14159], default 0.5 
    '''

    overhang_min = None
    '''Minimum angle to display 

    :type:  float in [0, 3.14159], default 0.5 
    '''

    sharp_max = None
    '''Maximum angle to display 

    :type:  float in [-3.14159, 3.14159], default 0.5 
    '''

    sharp_min = None
    '''Minimum angle to display 

    :type:  float in [-3.14159, 3.14159], default 0.5 
    '''

    thickness_max = None
    '''Maximum for measuring thickness 

    :type:  float in [0, 1000], default 0.5 
    '''

    thickness_min = None
    '''Minimum for measuring thickness 

    :type:  float in [0, 1000], default 0.5 
    '''

    thickness_samples = None
    '''Number of samples to test per face 

    :type:  int in [1, 32], default 0 
    '''

    type = None
    '''Type of data to visualize/check 

    :type:  enum in [‘OVERHANG’, ‘THICKNESS’, ‘INTERSECT’, ‘DISTORT’, ‘SHARP’], default ‘OVERHANG’ 
    '''


class MeshTessFace:
    '''TessFace in a Mesh data-block '''

    area = None
    '''Read only area of this face 

    :type:  float in [0, inf], default 0.0, (readonly) 
    '''

    hide = None
    '''

    :type:  boolean, default False 
    '''

    index = None
    '''Index of this face 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    material_index = None
    '''

    :type:  int in [0, 32767], default 0 
    '''

    normal = None
    '''Local space unit length normal vector for this face 

    :type:  float array of 3 items in [-1, 1], default (0.0, 0.0, 0.0), (readonly) 
    '''

    select = None
    '''

    :type:  boolean, default False 
    '''

    split_normals = None
    '''Local space unit length split normals vectors of the vertices of this face (must be computed beforehand using calc_normals_split or calc_tangents, and then calc_tessface) 

    :type:  float array of 12 items in [-1, 1], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0), (readonly) 
    '''

    use_smooth = None
    '''

    :type:  boolean, default False 
    '''

    vertices = None
    '''Vertex indices 

    :type:  int array of 4 items in [0, inf], default (0, 0, 0, 0) 
    '''

    vertices_raw = None
    '''Fixed size vertex indices array 

    :type:  int array of 4 items in [0, inf], default (0, 0, 0, 0) 
    '''

    center = None
    '''The midpoint of the face. (readonly) '''

    edge_keys = None
    '''(readonly) '''


class MeshTessFaces:
    '''Collection of mesh faces '''

    active = None
    '''The active face for this mesh 

    :type:  int in [-inf, inf], default 0 
    '''

    def add(self, count=0):
        '''add 

        :param count: Count, Number of faces to add 
        :type count: int in [0, inf], (optional)
        '''
        pass


class MeshTextureFace:
    '''UV map and image texture for a face '''

    image = None
    '''

    :type:  Image 
    '''

    uv = None
    '''

    :type:  float array of 8 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    uv1 = None
    '''

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    uv2 = None
    '''

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    uv3 = None
    '''

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    uv4 = None
    '''

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    uv_raw = None
    '''Fixed size UV coordinates array 

    :type:  float array of 8 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''


class MeshTextureFaceLayer:
    '''UV map with assigned image textures in a Mesh data-block '''

    active = None
    '''Set the map as active for display and editing 

    :type:  boolean, default False 
    '''

    active_clone = None
    '''Set the map as active for cloning 

    :type:  boolean, default False 
    '''

    active_render = None
    '''Set the map as active for rendering 

    :type:  boolean, default False 
    '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshTextureFace, (readonly) 
    '''

    name = None
    '''Name of UV map 

    :type:  string, default “”, (never None) 
    '''


class MeshTexturePoly:
    '''UV map and image texture for a face '''

    image = None
    '''

    :type:  Image 
    '''


class MeshTexturePolyLayer:
    '''UV map with assigned image textures in a Mesh data-block '''

    active = None
    '''Set the map as active for display and editing 

    :type:  boolean, default False 
    '''

    active_clone = None
    '''Set the map as active for cloning 

    :type:  boolean, default False 
    '''

    active_render = None
    '''Set the map as active for rendering 

    :type:  boolean, default False 
    '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshTexturePoly, (readonly) 
    '''

    name = None
    '''Name of UV map 

    :type:  string, default “”, (never None) 
    '''


class MeshUVLoop:
    pin_uv = None
    '''

    :type:  boolean, default False 
    '''

    select = None
    '''

    :type:  boolean, default False 
    '''

    select_edge = None
    '''

    :type:  boolean, default False 
    '''

    uv = None
    '''

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''


class MeshUVLoopLayer:
    data = None
    '''

    :type:  bpy_prop_collection of MeshUVLoop, (readonly) 
    '''

    name = None
    '''Name of UV map 

    :type:  string, default “”, (never None) 
    '''


class MeshVertex:
    '''Vertex in a Mesh data-block '''

    bevel_weight = None
    '''Weight used by the Bevel modifier ‘Only Vertices’ option 

    :type:  float in [-inf, inf], default 0.0 
    '''

    co = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    groups = None
    '''Weights for the vertex groups this vertex is member of 

    :type:  bpy_prop_collection of VertexGroupElement, (readonly) 
    '''

    hide = None
    '''

    :type:  boolean, default False 
    '''

    index = None
    '''Index of this vertex 

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    normal = None
    '''Vertex Normal 

    :type:  float array of 3 items in [-1, 1], default (0.0, 0.0, 0.0) 
    '''

    select = None
    '''

    :type:  boolean, default False 
    '''

    undeformed_co = None
    '''For meshes with modifiers applied, the coordinate of the vertex with no deforming modifiers applied, as used for generated texture coordinates 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0), (readonly) 
    '''


class MeshVertexFloatProperty:
    '''User defined floating point number value in a float properties layer '''

    value = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''


class MeshVertexFloatPropertyLayer:
    '''User defined layer of floating point number values '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshVertexFloatProperty, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''


class MeshVertexIntProperty:
    '''User defined integer number value in an integer properties layer '''

    value = None
    '''

    :type:  int in [-inf, inf], default 0 
    '''


class MeshVertexIntPropertyLayer:
    '''User defined layer of integer number values '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshVertexIntProperty, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''


class MeshVertexStringProperty:
    '''User defined string text value in a string properties layer '''

    value = None
    '''

    :type:  string, default “”, (never None) 
    '''


class MeshVertexStringPropertyLayer:
    '''User defined layer of string text values '''

    data = None
    '''

    :type:  bpy_prop_collection of MeshVertexStringProperty, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''


class MeshVertices:
    '''Collection of mesh vertices '''

    def add(self, count=0):
        '''add 

        :param count: Count, Number of vertices to add 
        :type count: int in [0, inf], (optional)
        '''
        pass


class MessageActuator:
    body_message = None
    '''Optional, message body Text 

    :type:  string, default “”, (never None) 
    '''

    body_property = None
    '''The message body will be set by the Property Value 

    :type:  string, default “”, (never None) 
    '''

    body_type = None
    '''Toggle message type: either Text or a PropertyName 

    :type:  enum in [‘TEXT’, ‘PROPERTY’], default ‘TEXT’ 
    '''

    subject = None
    '''Optional, message subject (this is what can be filtered on) 

    :type:  string, default “”, (never None) 
    '''

    to_property = None
    '''Optional, send message to objects with this name only, or empty to broadcast 

    :type:  string, default “”, (never None) 
    '''


class MessageSensor:
    '''Sensor to detect incoming messages '''

    subject = None
    '''Optional subject filter: only accept messages with this subject, or empty to accept all 

    :type:  string, default “”, (never None) 
    '''


class MetaBall:
    '''Metaball data-block to defined blobby surfaces '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    cycles = None
    '''Cycles mesh settings 

    :type:  CyclesMeshSettings, (readonly) 
    '''

    elements = None
    '''Meta elements 

    :type:  MetaBallElements bpy_prop_collection of MetaElement, (readonly) 
    '''

    is_editmode = None
    '''True when used in editmode 

    :type:  boolean, default False, (readonly) 
    '''

    materials = None
    '''

    :type:  IDMaterials bpy_prop_collection of Material, (readonly) 
    '''

    render_resolution = None
    '''Polygonization resolution in rendering 

    :type:  float in [0.005, 10000], default 0.0 
    '''

    resolution = None
    '''Polygonization resolution in the 3D viewport 

    :type:  float in [0.005, 10000], default 0.0 
    '''

    texspace_location = None
    '''Texture space location 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    texspace_size = None
    '''Texture space size 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    threshold = None
    '''Influence of meta elements 

    :type:  float in [0, 5], default 0.0 
    '''

    update_method = None
    '''Metaball edit update behavior 

    :type:  enum in [‘UPDATE_ALWAYS’, ‘HALFRES’, ‘FAST’, ‘NEVER’], default ‘UPDATE_ALWAYS’ 
    '''

    use_auto_texspace = None
    '''Adjust active object’s texture space automatically when transforming object 

    :type:  boolean, default False 
    '''

    def transform(self, matrix):
        '''Transform meta elements by a matrix 

        :param matrix: Matrix 
        :type matrix: float array of 16 items in [-inf, inf]
        '''
        pass


class MetaBallElements:
    '''Collection of metaball elements '''

    active = None
    '''Last selected element 

    :type:  MetaElement, (readonly) 
    '''

    def new(self, type='BALL'):
        '''Add a new element to the metaball 

        :param type: type for the new meta-element 
        :type type: enum in ['BALL', 'CAPSULE', 'PLANE', 'ELLIPSOID', 'CUBE'], (optional)
        :rtype: MetaElement 
        :return:  The newly created meta-element 
        '''
        pass

    def remove(self, element):
        '''Remove an element from the metaball 

        :param element: The element to remove 
        :type element: MetaElement, (never None)
        '''
        pass

    def clear(self):
        '''Remove all elements from the metaball 

        '''
        pass


class MetaElement:
    '''Blobby element in a Metaball data-block '''

    co = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    hide = None
    '''Hide element 

    :type:  boolean, default False 
    '''

    radius = None
    '''

    :type:  float in [0, inf], default 0.0 
    '''

    rotation = None
    '''Normalized quaternion rotation 

    :type:  float array of 4 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''

    size_x = None
    '''Size of element, use of components depends on element type 

    :type:  float in [0, 20], default 0.0 
    '''

    size_y = None
    '''Size of element, use of components depends on element type 

    :type:  float in [0, 20], default 0.0 
    '''

    size_z = None
    '''Size of element, use of components depends on element type 

    :type:  float in [0, 20], default 0.0 
    '''

    stiffness = None
    '''Stiffness defines how much of the element to fill 

    :type:  float in [0, 10], default 0.0 
    '''

    type = None
    '''Metaball types 

    :type:  enum in [‘BALL’, ‘CAPSULE’, ‘PLANE’, ‘ELLIPSOID’, ‘CUBE’], default ‘BALL’ 
    '''

    use_negative = None
    '''Set metaball as negative one 

    :type:  boolean, default False 
    '''


class MetaSequence:
    '''Sequence strip to group other strips as a single sequence strip '''

    alpha_mode = None
    '''Representation of alpha information in the RGBA pixels 

    :type:  enum in [‘STRAIGHT’, ‘PREMUL’], default ‘STRAIGHT’ 
    '''

    animation_offset_end = None
    '''Animation end offset (trim end) 

    :type:  int in [0, inf], default 0 
    '''

    animation_offset_start = None
    '''Animation start offset (trim start) 

    :type:  int in [0, inf], default 0 
    '''

    color_multiply = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    color_saturation = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    crop = None
    '''

    :type:  SequenceCrop, (readonly) 
    '''

    proxy = None
    '''

    :type:  SequenceProxy, (readonly) 
    '''

    sequences = None
    '''

    :type:  bpy_prop_collection of Sequence, (readonly) 
    '''

    strobe = None
    '''Only display every nth frame 

    :type:  float in [1, 30], default 0.0 
    '''

    transform = None
    '''

    :type:  SequenceTransform, (readonly) 
    '''

    use_crop = None
    '''Crop image before processing 

    :type:  boolean, default False 
    '''

    use_deinterlace = None
    '''For video movies to remove fields 

    :type:  boolean, default False 
    '''

    use_flip_x = None
    '''Flip on the X axis 

    :type:  boolean, default False 
    '''

    use_flip_y = None
    '''Flip on the Y axis 

    :type:  boolean, default False 
    '''

    use_float = None
    '''Convert input to float data 

    :type:  boolean, default False 
    '''

    use_proxy = None
    '''Use a preview proxy and/or timecode index for this strip 

    :type:  boolean, default False 
    '''

    use_reverse_frames = None
    '''Reverse frame order 

    :type:  boolean, default False 
    '''

    use_translation = None
    '''Translate image before processing 

    :type:  boolean, default False 
    '''


class MirrorModifier:
    '''Mirroring modifier '''

    merge_threshold = None
    '''Distance within which mirrored vertices are merged 

    :type:  float in [0, inf], default 0.0 
    '''

    mirror_object = None
    '''Object to use as mirror 

    :type:  Object 
    '''

    use_clip = None
    '''Prevent vertices from going through the mirror during transform 

    :type:  boolean, default False 
    '''

    use_mirror_merge = None
    '''Merge vertices within the merge threshold 

    :type:  boolean, default False 
    '''

    use_mirror_u = None
    '''Mirror the U texture coordinate around the 0.5 point 

    :type:  boolean, default False 
    '''

    use_mirror_v = None
    '''Mirror the V texture coordinate around the 0.5 point 

    :type:  boolean, default False 
    '''

    use_mirror_vertex_groups = None
    '''Mirror vertex groups (e.g. .R->.L) 

    :type:  boolean, default False 
    '''

    use_x = None
    '''Enable X axis mirror 

    :type:  boolean, default False 
    '''

    use_y = None
    '''Enable Y axis mirror 

    :type:  boolean, default False 
    '''

    use_z = None
    '''Enable Z axis mirror 

    :type:  boolean, default False 
    '''


class Modifier:
    '''Modifier affecting the geometry data of an object '''

    name = None
    '''Modifier name 

    :type:  string, default “”, (never None) 
    '''

    show_expanded = None
    '''Set modifier expanded in the user interface 

    :type:  boolean, default False 
    '''

    show_in_editmode = None
    '''Display modifier in Edit mode 

    :type:  boolean, default False 
    '''

    show_on_cage = None
    '''Adjust edit cage to modifier result 

    :type:  boolean, default False 
    '''

    show_render = None
    '''Use modifier during render 

    :type:  boolean, default False 
    '''

    show_viewport = None
    '''Display modifier in viewport 

    :type:  boolean, default False 
    '''

    type = None
    '''

    :type:  enum in [‘DATA_TRANSFER’, ‘MESH_CACHE’, ‘MESH_SEQUENCE_CACHE’, ‘NORMAL_EDIT’, ‘UV_PROJECT’, ‘UV_WARP’, ‘VERTEX_WEIGHT_EDIT’, ‘VERTEX_WEIGHT_MIX’, ‘VERTEX_WEIGHT_PROXIMITY’, ‘ARRAY’, ‘BEVEL’, ‘BOOLEAN’, ‘BUILD’, ‘DECIMATE’, ‘EDGE_SPLIT’, ‘MASK’, ‘MIRROR’, ‘MULTIRES’, ‘REMESH’, ‘SCREW’, ‘SKIN’, ‘SOLIDIFY’, ‘SUBSURF’, ‘TRIANGULATE’, ‘WIREFRAME’, ‘ARMATURE’, ‘CAST’, ‘CORRECTIVE_SMOOTH’, ‘CURVE’, ‘DISPLACE’, ‘HOOK’, ‘LAPLACIANSMOOTH’, ‘LAPLACIANDEFORM’, ‘LATTICE’, ‘MESH_DEFORM’, ‘SHRINKWRAP’, ‘SIMPLE_DEFORM’, ‘SMOOTH’, ‘WARP’, ‘WAVE’, ‘CLOTH’, ‘COLLISION’, ‘DYNAMIC_PAINT’, ‘EXPLODE’, ‘FLUID_SIMULATION’, ‘OCEAN’, ‘PARTICLE_INSTANCE’, ‘PARTICLE_SYSTEM’, ‘SMOKE’, ‘SOFT_BODY’, ‘SURFACE’], default ‘DATA_TRANSFER’, (readonly) 
    '''

    use_apply_on_spline = None
    '''Apply this and all preceding deformation modifiers on splines’ points rather than on filled curve/surface 

    :type:  boolean, default False 
    '''


class MotionPath:
    '''Cache of the worldspace positions of an element over a frame range '''

    frame_end = None
    '''End frame of the stored range 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    frame_start = None
    '''Starting frame of the stored range 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    is_modified = None
    '''Path is being edited 

    :type:  boolean, default False 
    '''

    length = None
    '''Number of frames cached 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    points = None
    '''Cached positions per frame 

    :type:  bpy_prop_collection of MotionPathVert, (readonly) 
    '''

    use_bone_head = None
    '''For PoseBone paths, use the bone head location when calculating this path 

    :type:  boolean, default False, (readonly) 
    '''


class MotionPathVert:
    '''Cached location on path '''

    co = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    select = None
    '''Path point is selected for editing 

    :type:  boolean, default False 
    '''


class MouseActuator:
    local_x = None
    '''Apply rotation locally 

    :type:  boolean, default False 
    '''

    local_y = None
    '''Apply rotation locally 

    :type:  boolean, default False 
    '''

    max_x = None
    '''Maximum positive rotation allowed by X mouse movement (0 for infinite) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    max_y = None
    '''Maximum positive rotation allowed by Y mouse movement (0 for infinite) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_x = None
    '''Maximum negative rotation allowed by X mouse movement (0 for infinite) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_y = None
    '''Maximum negative rotation allowed by Y mouse movement (0 for infinite) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    mode = None
    '''

    :type:  enum in [‘VISIBILITY’, ‘LOOK’], default ‘VISIBILITY’ 
    '''

    object_axis_x = None
    '''Local object axis mouse movement in the X direction will apply to 

    :type:  enum in [‘OBJECT_AXIS_X’, ‘OBJECT_AXIS_Y’, ‘OBJECT_AXIS_Z’], default ‘OBJECT_AXIS_X’ 
    '''

    object_axis_y = None
    '''Local object axis mouse movement in the Y direction will apply to 

    :type:  enum in [‘OBJECT_AXIS_X’, ‘OBJECT_AXIS_Y’, ‘OBJECT_AXIS_Z’], default ‘OBJECT_AXIS_X’ 
    '''

    reset_x = None
    '''Reset the cursor’s X position to the center of the screen space after calculating 

    :type:  boolean, default False 
    '''

    reset_y = None
    '''Reset the cursor’s Y position to the center of the screen space after calculating 

    :type:  boolean, default False 
    '''

    sensitivity_x = None
    '''Sensitivity of the X axis 

    :type:  float in [-inf, inf], default 0.0 
    '''

    sensitivity_y = None
    '''Sensitivity of the Y axis 

    :type:  float in [-inf, inf], default 0.0 
    '''

    threshold_x = None
    '''Amount of X motion before mouse movement will register 

    :type:  float in [-inf, inf], default 0.0 
    '''

    threshold_y = None
    '''Amount of Y motion before mouse movement will register 

    :type:  float in [-inf, inf], default 0.0 
    '''

    use_axis_x = None
    '''Calculate mouse movement on the X axis 

    :type:  boolean, default False 
    '''

    use_axis_y = None
    '''Calculate mouse movement on the Y axis 

    :type:  boolean, default False 
    '''

    visible = None
    '''Make mouse cursor visible 

    :type:  boolean, default False 
    '''


class MouseSensor:
    '''Sensor to detect mouse events '''

    material = None
    '''Only look for objects with this material (blank = all objects) 

    :type:  string, default “”, (never None) 
    '''

    mouse_event = None
    '''Type of event this mouse sensor should trigger on 

    :type:  enum in [‘LEFTCLICK’, ‘MIDDLECLICK’, ‘RIGHTCLICK’, ‘WHEELUP’, ‘WHEELDOWN’, ‘MOVEMENT’, ‘MOUSEOVER’, ‘MOUSEOVERANY’], default ‘LEFTCLICK’ 
    '''

    property = None
    '''Only look for objects with this property (blank = all objects) 

    :type:  string, default “”, (never None) 
    '''

    use_material = None
    '''Toggle collision on material or property 

    :type:  enum in [‘PROPERTY’, ‘MATERIAL’], default ‘PROPERTY’ 
    '''

    use_pulse = None
    '''Moving the mouse over a different object generates a pulse 

    :type:  boolean, default False 
    '''

    use_x_ray = None
    '''Toggle X-Ray option (see through objects that don’t have the property) 

    :type:  boolean, default False 
    '''


class MovieClip:
    '''MovieClip data-block referencing an external movie file '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    colorspace_settings = None
    '''Input color space settings 

    :type:  ColorManagedInputColorspaceSettings, (readonly) 
    '''

    display_aspect = None
    '''Display Aspect for this clip, does not affect rendering 

    :type:  float array of 2 items in [0.1, inf], default (0.0, 0.0) 
    '''

    filepath = None
    '''Filename of the movie or sequence file 

    :type:  string, default “”, (never None) 
    '''

    frame_duration = None
    '''Detected duration of movie clip in frames 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    frame_offset = None
    '''Offset of footage first frame relative to it’s file name (affects only how footage is loading, does not change data associated with a clip) 

    :type:  int in [-inf, inf], default 0 
    '''

    frame_start = None
    '''Global scene frame number at which this movie starts playing (affects all data associated with a clip) 

    :type:  int in [-inf, inf], default 0 
    '''

    grease_pencil = None
    '''Grease pencil data for this movie clip 

    :type:  GreasePencil 
    '''

    proxy = None
    '''

    :type:  MovieClipProxy, (readonly) 
    '''

    size = None
    '''Width and height in pixels, zero when image data cant be loaded 

    :type:  int array of 2 items in [-inf, inf], default (0, 0), (readonly) 
    '''

    source = None
    '''Where the clip comes from 

    :type:  enum in [‘SEQUENCE’, ‘MOVIE’], default ‘SEQUENCE’, (readonly) 
    '''

    tracking = None
    '''

    :type:  MovieTracking, (readonly) 
    '''

    use_proxy = None
    '''Use a preview proxy and/or timecode index for this clip 

    :type:  boolean, default False 
    '''

    use_proxy_custom_directory = None
    '''Create proxy images in a custom directory (default is movie location) 

    :type:  boolean, default False 
    '''


class MovieClipProxy:
    '''Proxy parameters for a movie clip '''

    build_100 = None
    '''Build proxy resolution 100% of the original footage dimension 

    :type:  boolean, default False 
    '''

    build_25 = None
    '''Build proxy resolution 25% of the original footage dimension 

    :type:  boolean, default False 
    '''

    build_50 = None
    '''Build proxy resolution 50% of the original footage dimension 

    :type:  boolean, default False 
    '''

    build_75 = None
    '''Build proxy resolution 75% of the original footage dimension 

    :type:  boolean, default False 
    '''

    build_free_run = None
    '''Build free run time code index 

    :type:  boolean, default False 
    '''

    build_free_run_rec_date = None
    '''Build free run time code index using Record Date/Time 

    :type:  boolean, default False 
    '''

    build_record_run = None
    '''Build record run time code index 

    :type:  boolean, default False 
    '''

    build_undistorted_100 = None
    '''Build proxy resolution 100% of the original undistorted footage dimension 

    :type:  boolean, default False 
    '''

    build_undistorted_25 = None
    '''Build proxy resolution 25% of the original undistorted footage dimension 

    :type:  boolean, default False 
    '''

    build_undistorted_50 = None
    '''Build proxy resolution 50% of the original undistorted footage dimension 

    :type:  boolean, default False 
    '''

    build_undistorted_75 = None
    '''Build proxy resolution 75% of the original undistorted footage dimension 

    :type:  boolean, default False 
    '''

    directory = None
    '''Location to store the proxy files 

    :type:  string, default “”, (never None) 
    '''

    quality = None
    '''JPEG quality of proxy images 

    :type:  int in [0, 32767], default 0 
    '''

    timecode = None
    '''

    :type:  enum in [‘NONE’, ‘RECORD_RUN’, ‘FREE_RUN’, ‘FREE_RUN_REC_DATE’, ‘FREE_RUN_NO_GAPS’], default ‘NONE’ 
    '''


class MovieClipScopes:
    '''Scopes for statistical view of a movie clip '''

    pass


class MovieClipSequence:
    '''Sequence strip to load a video from the clip editor '''

    alpha_mode = None
    '''Representation of alpha information in the RGBA pixels 

    :type:  enum in [‘STRAIGHT’, ‘PREMUL’], default ‘STRAIGHT’ 
    '''

    animation_offset_end = None
    '''Animation end offset (trim end) 

    :type:  int in [0, inf], default 0 
    '''

    animation_offset_start = None
    '''Animation start offset (trim start) 

    :type:  int in [0, inf], default 0 
    '''

    color_multiply = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    color_saturation = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    crop = None
    '''

    :type:  SequenceCrop, (readonly) 
    '''

    stabilize2d = None
    '''Use the 2D stabilized version of the clip 

    :type:  boolean, default False 
    '''

    strobe = None
    '''Only display every nth frame 

    :type:  float in [1, 30], default 0.0 
    '''

    transform = None
    '''

    :type:  SequenceTransform, (readonly) 
    '''

    undistort = None
    '''Use the undistorted version of the clip 

    :type:  boolean, default False 
    '''

    use_crop = None
    '''Crop image before processing 

    :type:  boolean, default False 
    '''

    use_deinterlace = None
    '''For video movies to remove fields 

    :type:  boolean, default False 
    '''

    use_flip_x = None
    '''Flip on the X axis 

    :type:  boolean, default False 
    '''

    use_flip_y = None
    '''Flip on the Y axis 

    :type:  boolean, default False 
    '''

    use_float = None
    '''Convert input to float data 

    :type:  boolean, default False 
    '''

    use_reverse_frames = None
    '''Reverse frame order 

    :type:  boolean, default False 
    '''

    use_translation = None
    '''Translate image before processing 

    :type:  boolean, default False 
    '''


class MovieClipUser:
    '''Parameters defining how a MovieClip data-block is used by another data-block '''

    frame_current = None
    '''Current frame number in movie or image sequence 

    :type:  int in [-500000, 500000], default 0 
    '''

    proxy_render_size = None
    '''Draw preview using full resolution or different proxy resolutions 

    :type:  enum in [‘PROXY_25’, ‘PROXY_50’, ‘PROXY_75’, ‘PROXY_100’, ‘FULL’], default ‘FULL’ 
    '''

    use_render_undistorted = None
    '''Render preview using undistorted proxy 

    :type:  boolean, default False 
    '''


class MovieReconstructedCamera:
    '''Match-moving reconstructed camera data from tracker '''

    average_error = None
    '''Average error of reconstruction 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    frame = None
    '''Frame number marker is keyframed on 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    matrix = None
    '''Worldspace transformation matrix 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0), (readonly) 
    '''


class MovieSequence:
    '''Sequence strip to load a video '''

    alpha_mode = None
    '''Representation of alpha information in the RGBA pixels 

    :type:  enum in [‘STRAIGHT’, ‘PREMUL’], default ‘STRAIGHT’ 
    '''

    animation_offset_end = None
    '''Animation end offset (trim end) 

    :type:  int in [0, inf], default 0 
    '''

    animation_offset_start = None
    '''Animation start offset (trim start) 

    :type:  int in [0, inf], default 0 
    '''

    color_multiply = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    color_saturation = None
    '''

    :type:  float in [0, 20], default 1.0 
    '''

    colorspace_settings = None
    '''Input color space settings 

    :type:  ColorManagedInputColorspaceSettings, (readonly) 
    '''

    crop = None
    '''

    :type:  SequenceCrop, (readonly) 
    '''

    elements = None
    '''

    :type:  bpy_prop_collection of SequenceElement, (readonly) 
    '''

    filepath = None
    '''

    :type:  string, default “”, (never None) 
    '''

    mpeg_preseek = None
    '''For MPEG movies, preseek this many frames 

    :type:  int in [0, 50], default 0 
    '''

    proxy = None
    '''

    :type:  SequenceProxy, (readonly) 
    '''

    stereo_3d_format = None
    '''Settings for stereo 3d 

    :type:  Stereo3dFormat, (readonly, never None) 
    '''

    stream_index = None
    '''For files with several movie streams, use the stream with the given index 

    :type:  int in [0, 20], default 0 
    '''

    strobe = None
    '''Only display every nth frame 

    :type:  float in [1, 30], default 0.0 
    '''

    transform = None
    '''

    :type:  SequenceTransform, (readonly) 
    '''

    use_crop = None
    '''Crop image before processing 

    :type:  boolean, default False 
    '''

    use_deinterlace = None
    '''For video movies to remove fields 

    :type:  boolean, default False 
    '''

    use_flip_x = None
    '''Flip on the X axis 

    :type:  boolean, default False 
    '''

    use_flip_y = None
    '''Flip on the Y axis 

    :type:  boolean, default False 
    '''

    use_float = None
    '''Convert input to float data 

    :type:  boolean, default False 
    '''

    use_multiview = None
    '''Use Multiple Views (when available) 

    :type:  boolean, default False 
    '''

    use_proxy = None
    '''Use a preview proxy and/or timecode index for this strip 

    :type:  boolean, default False 
    '''

    use_reverse_frames = None
    '''Reverse frame order 

    :type:  boolean, default False 
    '''

    use_translation = None
    '''Translate image before processing 

    :type:  boolean, default False 
    '''

    views_format = None
    '''Mode to load movie views 

    :type:  enum in [‘INDIVIDUAL’, ‘STEREO_3D’], default ‘INDIVIDUAL’ 
    '''


class MovieTracking:
    '''Match-moving data for tracking '''

    active_object_index = None
    '''Index of active object 

    :type:  int in [-inf, inf], default 0 
    '''

    camera = None
    '''

    :type:  MovieTrackingCamera, (readonly) 
    '''

    dopesheet = None
    '''

    :type:  MovieTrackingDopesheet, (readonly) 
    '''

    objects = None
    '''Collection of objects in this tracking data object 

    :type:  MovieTrackingObjects bpy_prop_collection of MovieTrackingObject, (readonly) 
    '''

    plane_tracks = None
    '''Collection of plane tracks in this tracking data object 

    :type:  MovieTrackingPlaneTracks bpy_prop_collection of MovieTrackingPlaneTrack, (readonly) 
    '''

    reconstruction = None
    '''

    :type:  MovieTrackingReconstruction, (readonly) 
    '''

    settings = None
    '''

    :type:  MovieTrackingSettings, (readonly) 
    '''

    stabilization = None
    '''

    :type:  MovieTrackingStabilization, (readonly) 
    '''

    tracks = None
    '''Collection of tracks in this tracking data object 

    :type:  MovieTrackingTracks bpy_prop_collection of MovieTrackingTrack, (readonly) 
    '''


class MovieTrackingCamera:
    '''Match-moving camera data for tracking '''

    distortion_model = None
    '''Distortion model used for camera lenses 

    :type:  enum in [‘POLYNOMIAL’, ‘DIVISION’], default ‘POLYNOMIAL’ 
    '''

    division_k1 = None
    '''First coefficient of second order division distortion 

    :type:  float in [-inf, inf], default 0.0 
    '''

    division_k2 = None
    '''First coefficient of second order division distortion 

    :type:  float in [-inf, inf], default 0.0 
    '''

    focal_length = None
    '''Camera’s focal length 

    :type:  float in [0.0001, inf], default 0.0 
    '''

    focal_length_pixels = None
    '''Camera’s focal length 

    :type:  float in [0, inf], default 0.0 
    '''

    k1 = None
    '''First coefficient of third order polynomial radial distortion 

    :type:  float in [-inf, inf], default 0.0 
    '''

    k2 = None
    '''Second coefficient of third order polynomial radial distortion 

    :type:  float in [-inf, inf], default 0.0 
    '''

    k3 = None
    '''Third coefficient of third order polynomial radial distortion 

    :type:  float in [-inf, inf], default 0.0 
    '''

    pixel_aspect = None
    '''Pixel aspect ratio 

    :type:  float in [0.1, inf], default 1.0 
    '''

    principal = None
    '''Optical center of lens 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    sensor_width = None
    '''Width of CCD sensor in millimeters 

    :type:  float in [0, 500], default 0.0 
    '''

    units = None
    '''Units used for camera focal length 

    :type:  enum in [‘PIXELS’, ‘MILLIMETERS’], default ‘PIXELS’ 
    '''


class MovieTrackingDopesheet:
    '''Match-moving dopesheet data '''

    show_hidden = None
    '''Include channels from objects/bone that aren’t visible 

    :type:  boolean, default False 
    '''

    show_only_selected = None
    '''Only include channels relating to selected objects and data 

    :type:  boolean, default False 
    '''

    sort_method = None
    '''Method to be used to sort channels in dopesheet view 

    :type:  enum in [‘NAME’, ‘LONGEST’, ‘TOTAL’, ‘AVERAGE_ERROR’], default ‘NAME’ 
    '''

    use_invert_sort = None
    '''Invert sort order of dopesheet channels 

    :type:  boolean, default False 
    '''


class MovieTrackingMarker:
    '''Match-moving marker data for tracking '''

    co = None
    '''Marker position at frame in normalized coordinates 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    frame = None
    '''Frame number marker is keyframed on 

    :type:  int in [-inf, inf], default 0 
    '''

    is_keyed = None
    '''Whether the position of the marker is keyframed or tracked 

    :type:  boolean, default False 
    '''

    mute = None
    '''Is marker muted for current frame 

    :type:  boolean, default False 
    '''

    pattern_bound_box = None
    '''Pattern area bounding box in normalized coordinates 

    :type:  float array of 4 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0), (readonly) 
    '''

    pattern_corners = None
    '''Array of coordinates which represents pattern’s corners in normalized coordinates relative to marker position 

    :type:  float array of 8 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    search_max = None
    '''Right-bottom corner of search area in normalized coordinates relative to marker position 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    search_min = None
    '''Left-bottom corner of search area in normalized coordinates relative to marker position 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''


class MovieTrackingMarkers:
    '''Collection of markers for movie tracking track '''

    def find_frame(self, frame, exact=True):
        '''Get marker for specified frame 

        :param frame: Frame, Frame number to find marker for 
        :type frame: int in [0, 500000]
        :param exact: Exact, Get marker at exact frame number rather than get estimated marker 
        :type exact: boolean, (optional)
        :rtype: MovieTrackingMarker 
        :return:  Marker for specified frame 
        '''
        pass

    def insert_frame(self, frame, co=(0.0, 0.0)):
        '''Insert a new marker at the specified frame 

        :param frame: Frame, Frame number to insert marker to 
        :type frame: int in [0, 500000]
        :param co: Coordinate, Place new marker at the given frame using specified in normalized space coordinates 
        :type co: float array of 2 items in [-1, 1], (optional)
        :rtype: MovieTrackingMarker 
        :return:  Newly created marker 
        '''
        pass

    def delete_frame(self, frame):
        '''Delete marker at specified frame 

        :param frame: Frame, Frame number to delete marker from 
        :type frame: int in [0, 500000]
        '''
        pass


class MovieTrackingObject:
    '''Match-moving object tracking and reconstruction data '''

    is_camera = None
    '''Object is used for camera tracking 

    :type:  boolean, default False, (readonly) 
    '''

    keyframe_a = None
    '''First keyframe used for reconstruction initialization 

    :type:  int in [-inf, inf], default 0 
    '''

    keyframe_b = None
    '''Second keyframe used for reconstruction initialization 

    :type:  int in [-inf, inf], default 0 
    '''

    name = None
    '''Unique name of object 

    :type:  string, default “”, (never None) 
    '''

    plane_tracks = None
    '''Collection of plane tracks in this tracking data object 

    :type:  MovieTrackingObjectPlaneTracks bpy_prop_collection of MovieTrackingPlaneTrack, (readonly) 
    '''

    reconstruction = None
    '''

    :type:  MovieTrackingReconstruction, (readonly) 
    '''

    scale = None
    '''Scale of object solution in camera space 

    :type:  float in [0.0001, 10000], default 1.0 
    '''

    tracks = None
    '''Collection of tracks in this tracking data object 

    :type:  MovieTrackingObjectTracks bpy_prop_collection of MovieTrackingTrack, (readonly) 
    '''


class MovieTrackingObjectPlaneTracks:
    '''Collection of tracking plane tracks '''

    active = None
    '''Active track in this tracking data object 

    :type:  MovieTrackingTrack 
    '''


class MovieTrackingObjectTracks:
    '''Collection of movie tracking tracks '''

    active = None
    '''Active track in this tracking data object 

    :type:  MovieTrackingTrack 
    '''

    def new(self, name="", frame=1):
        '''create new motion track in this movie clip 

        :param name: Name of new track 
        :type name: string, (optional, never None)
        :param frame: Frame, Frame number to add tracks on 
        :type frame: int in [0, 500000], (optional)
        :rtype: MovieTrackingTrack 
        :return:  Newly created track 
        '''
        pass


class MovieTrackingObjects:
    '''Collection of movie tracking objects '''

    active = None
    '''Active object in this tracking data object 

    :type:  MovieTrackingObject 
    '''

    def new(self, name):
        '''Add tracking object to this movie clip 

        :param name: Name of new object 
        :type name: string, (never None)
        :rtype: MovieTrackingObject 
        :return:  New motion tracking object 
        '''
        pass

    def remove(self, object):
        '''Remove tracking object from this movie clip 

        :param object: Motion tracking object to be removed 
        :type object: MovieTrackingObject, (never None)
        '''
        pass


class MovieTrackingPlaneMarker:
    '''Match-moving plane marker data for tracking '''

    corners = None
    '''Array of coordinates which represents UI rectangle corners in frame normalized coordinates 

    :type:  float array of 8 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    frame = None
    '''Frame number marker is keyframed on 

    :type:  int in [-inf, inf], default 0 
    '''

    mute = None
    '''Is marker muted for current frame 

    :type:  boolean, default False 
    '''


class MovieTrackingPlaneMarkers:
    '''Collection of markers for movie tracking plane track '''

    def find_frame(self, frame, exact=True):
        '''Get plane marker for specified frame 

        :param frame: Frame, Frame number to find marker for 
        :type frame: int in [0, 500000]
        :param exact: Exact, Get plane marker at exact frame number rather than get estimated marker 
        :type exact: boolean, (optional)
        :rtype: MovieTrackingPlaneMarker 
        :return:  Plane marker for specified frame 
        '''
        pass

    def insert_frame(self, frame):
        '''Insert a new plane marker at the specified frame 

        :param frame: Frame, Frame number to insert marker to 
        :type frame: int in [0, 500000]
        :rtype: MovieTrackingPlaneMarker 
        :return:  Newly created plane marker 
        '''
        pass

    def delete_frame(self, frame):
        '''Delete plane marker at specified frame 

        :param frame: Frame, Frame number to delete plane marker from 
        :type frame: int in [0, 500000]
        '''
        pass


class MovieTrackingPlaneTrack:
    '''Match-moving plane track data for tracking '''

    image = None
    '''Image displayed in the track during editing in clip editor 

    :type:  Image 
    '''

    image_opacity = None
    '''Opacity of the image 

    :type:  float in [0, 1], default 0.0 
    '''

    markers = None
    '''Collection of markers in track 

    :type:  MovieTrackingPlaneMarkers bpy_prop_collection of MovieTrackingPlaneMarker, (readonly) 
    '''

    name = None
    '''Unique name of track 

    :type:  string, default “”, (never None) 
    '''

    select = None
    '''Plane track is selected 

    :type:  boolean, default False 
    '''

    use_auto_keying = None
    '''Automatic keyframe insertion when moving plane corners 

    :type:  boolean, default False 
    '''


class MovieTrackingPlaneTracks:
    '''Collection of movie tracking plane tracks '''

    active = None
    '''Active plane track in this tracking data object 

    :type:  MovieTrackingPlaneTrack 
    '''


class MovieTrackingReconstructedCameras:
    '''Collection of solved cameras '''

    def find_frame(self, frame=1):
        '''Find a reconstructed camera for a give frame number 

        :param frame: Frame, Frame number to find camera for 
        :type frame: int in [0, 500000], (optional)
        :rtype: MovieReconstructedCamera 
        :return:  Camera for a given frame 
        '''
        pass

    def matrix_from_frame(self, frame=1):
        '''Return interpolated camera matrix for a given frame 

        :param frame: Frame, Frame number to find camera for 
        :type frame: int in [0, 500000], (optional)
        :rtype: float array of 16 items in [-inf, inf] 
        :return:  Matrix, Interpolated camera matrix for a given frame 
        '''
        pass


class MovieTrackingReconstruction:
    '''Match-moving reconstruction data from tracker '''

    average_error = None
    '''Average error of reconstruction 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    cameras = None
    '''Collection of solved cameras 

    :type:  MovieTrackingReconstructedCameras bpy_prop_collection of MovieReconstructedCamera, (readonly) 
    '''

    is_valid = None
    '''Is tracking data contains valid reconstruction information 

    :type:  boolean, default False, (readonly) 
    '''


class MovieTrackingSettings:
    '''Match moving settings '''

    clean_action = None
    '''Cleanup action to execute 

    :type:  enum in [‘SELECT’, ‘DELETE_TRACK’, ‘DELETE_SEGMENTS’], default ‘SELECT’ 
    '''

    clean_error = None
    '''Effect on tracks which have a larger re-projection error 

    :type:  float in [0, inf], default 0.0 
    '''

    clean_frames = None
    '''Effect on tracks which are tracked less than the specified amount of frames 

    :type:  int in [0, inf], default 0 
    '''

    default_correlation_min = None
    '''Default minimum value of correlation between matched pattern and reference that is still treated as successful tracking 

    :type:  float in [0, 1], default 0.0 
    '''

    default_frames_limit = None
    '''Every tracking cycle, this number of frames are tracked 

    :type:  int in [0, 32767], default 0 
    '''

    default_margin = None
    '''Default distance from image boundary at which marker stops tracking 

    :type:  int in [0, 300], default 0 
    '''

    default_motion_model = None
    '''Default motion model to use for tracking 

    :type:  enum in [‘Perspective’, ‘Affine’, ‘LocRotScale’, ‘LocScale’, ‘LocRot’, ‘Loc’], default ‘Loc’ 
    '''

    default_pattern_match = None
    '''Track pattern from given frame when tracking marker to next frame 

    :type:  enum in [‘KEYFRAME’, ‘PREV_FRAME’], default ‘KEYFRAME’ 
    '''

    default_pattern_size = None
    '''Size of pattern area for newly created tracks 

    :type:  int in [5, 1000], default 0 
    '''

    default_search_size = None
    '''Size of search area for newly created tracks 

    :type:  int in [5, 1000], default 0 
    '''

    default_weight = None
    '''Influence of newly created track on a final solution 

    :type:  float in [0, 1], default 0.0 
    '''

    distance = None
    '''Distance between two bundles used for scene scaling 

    :type:  float in [-inf, inf], default 1.0 
    '''

    object_distance = None
    '''Distance between two bundles used for object scaling 

    :type:  float in [0.001, 10000], default 1.0 
    '''

    refine_intrinsics = None
    '''Refine intrinsics during camera solving 

    :type:  enum in [‘NONE’, ‘FOCAL_LENGTH’, ‘FOCAL_LENGTH_RADIAL_K1’, ‘FOCAL_LENGTH_RADIAL_K1_K2’, ‘FOCAL_LENGTH_PRINCIPAL_POINT_RADIAL_K1_K2’, ‘FOCAL_LENGTH_PRINCIPAL_POINT’, ‘RADIAL_K1_K2’], default ‘NONE’ 
    '''

    show_default_expanded = None
    '''Show default options expanded in the user interface 

    :type:  boolean, default False 
    '''

    show_extra_expanded = None
    '''Show extra options expanded in the user interface 

    :type:  boolean, default False 
    '''

    speed = None
    '''Limit speed of tracking to make visual feedback easier (this does not affect the tracking quality) 

    :type:  enum in [‘FASTEST’, ‘DOUBLE’, ‘REALTIME’, ‘HALF’, ‘QUARTER’], default ‘FASTEST’ 
    '''

    use_default_blue_channel = None
    '''Use blue channel from footage for tracking 

    :type:  boolean, default False 
    '''

    use_default_brute = None
    '''Use a brute-force translation-only initialization when tracking 

    :type:  boolean, default False 
    '''

    use_default_green_channel = None
    '''Use green channel from footage for tracking 

    :type:  boolean, default False 
    '''

    use_default_mask = None
    '''Use a grease pencil data-block as a mask to use only specified areas of pattern when tracking 

    :type:  boolean, default False 
    '''

    use_default_normalization = None
    '''Normalize light intensities while tracking (slower) 

    :type:  boolean, default False 
    '''

    use_default_red_channel = None
    '''Use red channel from footage for tracking 

    :type:  boolean, default False 
    '''

    use_keyframe_selection = None
    '''Automatically select keyframes when solving camera/object motion 

    :type:  boolean, default False 
    '''

    use_tripod_solver = None
    '''Use special solver to track a stable camera position, such as a tripod 

    :type:  boolean, default False 
    '''


class MovieTrackingStabilization:
    '''2D stabilization based on tracking markers '''

    active_rotation_track_index = None
    '''Index of active track in rotation stabilization tracks list 

    :type:  int in [-inf, inf], default 0 
    '''

    active_track_index = None
    '''Index of active track in translation stabilization tracks list 

    :type:  int in [-inf, inf], default 0 
    '''

    anchor_frame = None
    '''Reference point to anchor stabilization (other frames will be adjusted relative to this frame’s position) 

    :type:  int in [0, 500000], default 0 
    '''

    filter_type = None
    '''Interpolation to use for sub-pixel shifts and rotations due to stabilization 

    :type:  enum in [‘NEAREST’, ‘BILINEAR’, ‘BICUBIC’], default ‘NEAREST’ 
    '''

    influence_location = None
    '''Influence of stabilization algorithm on footage location 

    :type:  float in [0, 1], default 0.0 
    '''

    influence_rotation = None
    '''Influence of stabilization algorithm on footage rotation 

    :type:  float in [0, 1], default 0.0 
    '''

    influence_scale = None
    '''Influence of stabilization algorithm on footage scale 

    :type:  float in [0, 1], default 0.0 
    '''

    rotation_tracks = None
    '''Collection of tracks used for 2D stabilization (translation) 

    :type:  bpy_prop_collection of MovieTrackingTrack, (readonly) 
    '''

    scale_max = None
    '''Limit the amount of automatic scaling 

    :type:  float in [0, 10], default 0.0 
    '''

    show_tracks_expanded = None
    '''Show UI list of tracks participating in stabilization 

    :type:  boolean, default False 
    '''

    target_position = None
    '''Known relative offset of original shot, will be subtracted (e.g. for panning shot, can be animated) 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    target_rotation = None
    '''Rotation present on original shot, will be compensated (e.g. for deliberate tilting) 

    :type:  float in [-inf, inf], default 0.0 
    '''

    target_scale = None
    '''Explicitly scale resulting frame to compensate zoom of original shot 

    :type:  float in [1.192e-07, inf], default 0.0 
    '''

    tracks = None
    '''Collection of tracks used for 2D stabilization (translation) 

    :type:  bpy_prop_collection of MovieTrackingTrack, (readonly) 
    '''

    use_2d_stabilization = None
    '''Use 2D stabilization for footage 

    :type:  boolean, default False 
    '''

    use_autoscale = None
    '''Automatically scale footage to cover unfilled areas when stabilizing 

    :type:  boolean, default False 
    '''

    use_stabilize_rotation = None
    '''Stabilize detected rotation around center of frame 

    :type:  boolean, default False 
    '''

    use_stabilize_scale = None
    '''Compensate any scale changes relative to center of rotation 

    :type:  boolean, default False 
    '''


class MovieTrackingTrack:
    '''Match-moving track data for tracking '''

    average_error = None
    '''Average error of re-projection 

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    bundle = None
    '''Position of bundle reconstructed from this track 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0), (readonly) 
    '''

    color = None
    '''Color of the track in the Movie Clip Editor and the 3D viewport after a solve 

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    correlation_min = None
    '''Minimal value of correlation between matched pattern and reference that is still treated as successful tracking 

    :type:  float in [0, 1], default 0.0 
    '''

    frames_limit = None
    '''Every tracking cycle, this number of frames are tracked 

    :type:  int in [0, 32767], default 0 
    '''

    grease_pencil = None
    '''Grease pencil data for this track 

    :type:  GreasePencil 
    '''

    has_bundle = None
    '''True if track has a valid bundle 

    :type:  boolean, default False, (readonly) 
    '''

    hide = None
    '''Track is hidden 

    :type:  boolean, default False 
    '''

    lock = None
    '''Track is locked and all changes to it are disabled 

    :type:  boolean, default False 
    '''

    margin = None
    '''Distance from image boundary at which marker stops tracking 

    :type:  int in [0, 300], default 0 
    '''

    markers = None
    '''Collection of markers in track 

    :type:  MovieTrackingMarkers bpy_prop_collection of MovieTrackingMarker, (readonly) 
    '''

    motion_model = None
    '''Default motion model to use for tracking 

    :type:  enum in [‘Perspective’, ‘Affine’, ‘LocRotScale’, ‘LocScale’, ‘LocRot’, ‘Loc’], default ‘Loc’ 
    '''

    name = None
    '''Unique name of track 

    :type:  string, default “”, (never None) 
    '''

    offset = None
    '''Offset of track from the parenting point 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    pattern_match = None
    '''Track pattern from given frame when tracking marker to next frame 

    :type:  enum in [‘KEYFRAME’, ‘PREV_FRAME’], default ‘KEYFRAME’ 
    '''

    select = None
    '''Track is selected 

    :type:  boolean, default False 
    '''

    select_anchor = None
    '''Track’s anchor point is selected 

    :type:  boolean, default False 
    '''

    select_pattern = None
    '''Track’s pattern area is selected 

    :type:  boolean, default False 
    '''

    select_search = None
    '''Track’s search area is selected 

    :type:  boolean, default False 
    '''

    use_alpha_preview = None
    '''Apply track’s mask on displaying preview 

    :type:  boolean, default False 
    '''

    use_blue_channel = None
    '''Use blue channel from footage for tracking 

    :type:  boolean, default False 
    '''

    use_brute = None
    '''Use a brute-force translation only pre-track before refinement 

    :type:  boolean, default False 
    '''

    use_custom_color = None
    '''Use custom color instead of theme-defined 

    :type:  boolean, default False 
    '''

    use_grayscale_preview = None
    '''Display what the tracking algorithm sees in the preview 

    :type:  boolean, default False 
    '''

    use_green_channel = None
    '''Use green channel from footage for tracking 

    :type:  boolean, default False 
    '''

    use_mask = None
    '''Use a grease pencil data-block as a mask to use only specified areas of pattern when tracking 

    :type:  boolean, default False 
    '''

    use_normalization = None
    '''Normalize light intensities while tracking. Slower 

    :type:  boolean, default False 
    '''

    use_red_channel = None
    '''Use red channel from footage for tracking 

    :type:  boolean, default False 
    '''

    weight = None
    '''Influence of this track on a final solution 

    :type:  float in [0, 1], default 0.0 
    '''

    weight_stab = None
    '''Influence of this track on 2D stabilization 

    :type:  float in [0, 1], default 0.0 
    '''


class MovieTrackingTracks:
    '''Collection of movie tracking tracks '''

    active = None
    '''Active track in this tracking data object 

    :type:  MovieTrackingTrack 
    '''

    def new(self, name="", frame=1):
        '''Create new motion track in this movie clip 

        :param name: Name of new track 
        :type name: string, (optional, never None)
        :param frame: Frame, Frame number to add track on 
        :type frame: int in [0, 500000], (optional)
        :rtype: MovieTrackingTrack 
        :return:  Newly created track 
        '''
        pass


class MulticamSequence:
    '''Sequence strip to perform multicam editing '''

    animation_offset_end = None
    '''Animation end offset (trim end) 

    :type:  int in [0, inf], default 0 
    '''

    animation_offset_start = None
    '''Animation start offset (trim start) 

    :type:  int in [0, inf], default 0 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''

    multicam_source = None
    '''

    :type:  int in [0, 31], default 0 
    '''


class MultiplySequence:
    '''Multiply Sequence '''

    input_1 = None
    '''First input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_2 = None
    '''Second input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''


class MultiresModifier:
    '''Multiresolution mesh modifier '''

    filepath = None
    '''Path to external displacements file 

    :type:  string, default “”, (never None) 
    '''

    is_external = None
    '''Store multires displacements outside the .blend file, to save memory 

    :type:  boolean, default False, (readonly) 
    '''

    levels = None
    '''Number of subdivisions to use in the viewport 

    :type:  int in [0, 255], default 0 
    '''

    render_levels = None
    '''The subdivision level visible at render time 

    :type:  int in [0, 255], default 0 
    '''

    sculpt_levels = None
    '''Number of subdivisions to use in sculpt mode 

    :type:  int in [0, 255], default 0 
    '''

    show_only_control_edges = None
    '''Skip drawing/rendering of interior subdivided edges 

    :type:  boolean, default False 
    '''

    subdivision_type = None
    '''Select type of subdivision algorithm 

    :type:  enum in [‘CATMULL_CLARK’, ‘SIMPLE’], default ‘CATMULL_CLARK’ 
    '''

    total_levels = None
    '''Number of subdivisions for which displacements are stored 

    :type:  int in [0, 255], default 0, (readonly) 
    '''

    use_subsurf_uv = None
    '''Use subsurf to subdivide UVs 

    :type:  boolean, default False 
    '''


class MusgraveTexture:
    '''Procedural musgrave texture '''

    dimension_max = None
    '''Highest fractal dimension 

    :type:  float in [0.0001, 2], default 0.0 
    '''

    gain = None
    '''The gain multiplier 

    :type:  float in [0, 6], default 0.0 
    '''

    lacunarity = None
    '''Gap between successive frequencies 

    :type:  float in [0, 6], default 0.0 
    '''

    musgrave_type = None
    '''Fractal noise algorithm 

    :type:  enum in [‘MULTIFRACTAL’, ‘RIDGED_MULTIFRACTAL’, ‘HYBRID_MULTIFRACTAL’, ‘FBM’, ‘HETERO_TERRAIN’], default ‘MULTIFRACTAL’ 
    '''

    nabla = None
    '''Size of derivative offset used for calculating normal 

    :type:  float in [0.001, 0.1], default 0.0 
    '''

    noise_basis = None
    '''Noise basis used for turbulence 

    :type:  enum in [‘BLENDER_ORIGINAL’, ‘ORIGINAL_PERLIN’, ‘IMPROVED_PERLIN’, ‘VORONOI_F1’, ‘VORONOI_F2’, ‘VORONOI_F3’, ‘VORONOI_F4’, ‘VORONOI_F2_F1’, ‘VORONOI_CRACKLE’, ‘CELL_NOISE’], default ‘BLENDER_ORIGINAL’ 
    '''

    noise_intensity = None
    '''Intensity of the noise 

    :type:  float in [0, 10], default 0.0 
    '''

    noise_scale = None
    '''Scaling for noise input 

    :type:  float in [0.0001, inf], default 0.0 
    '''

    octaves = None
    '''Number of frequencies used 

    :type:  float in [0, 8], default 0.0 
    '''

    offset = None
    '''The fractal offset 

    :type:  float in [0, 6], default 0.0 
    '''

    users_material = None
    '''Materials that use this texture (readonly) '''

    users_object_modifier = None
    '''Object modifiers that use this texture (readonly) '''


class NODE_UL_interface_sockets:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class NandController:
    '''Controller passing on events based on a logical NAND operation '''

    pass


class NearSensor:
    '''Sensor to detect nearby objects '''

    distance = None
    '''Trigger distance 

    :type:  float in [0, 10000], default 0.0 
    '''

    property = None
    '''Only look for objects with this property (blank = all objects) 

    :type:  string, default “”, (never None) 
    '''

    reset_distance = None
    '''The distance where the sensor forgets the actor 

    :type:  float in [0, 10000], default 0.0 
    '''


class NlaStrip:
    '''A container referencing an existing Action '''

    action = None
    '''Action referenced by this strip 

    :type:  Action 
    '''

    action_frame_end = None
    '''Last frame from action to use 

    :type:  float in [-inf, inf], default 0.0 
    '''

    action_frame_start = None
    '''First frame from action to use 

    :type:  float in [-inf, inf], default 0.0 
    '''

    active = None
    '''NLA Strip is active 

    :type:  boolean, default False, (readonly) 
    '''

    blend_in = None
    '''Number of frames at start of strip to fade in influence 

    :type:  float in [-inf, inf], default 0.0 
    '''

    blend_out = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    blend_type = None
    '''Method used for combining strip’s result with accumulated result 

    :type:  enum in [‘REPLACE’, ‘ADD’, ‘SUBTRACT’, ‘MULTIPLY’], default ‘REPLACE’ 
    '''

    extrapolation = None
    '''Action to take for gaps past the strip extents 

    :type:  enum in [‘NOTHING’, ‘HOLD’, ‘HOLD_FORWARD’], default ‘HOLD’ 
    '''

    fcurves = None
    '''F-Curves for controlling the strip’s influence and timing 

    :type:  NlaStripFCurves bpy_prop_collection of FCurve, (readonly) 
    '''

    frame_end = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    frame_start = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    influence = None
    '''Amount the strip contributes to the current result 

    :type:  float in [0, 1], default 0.0 
    '''

    modifiers = None
    '''Modifiers affecting all the F-Curves in the referenced Action 

    :type:  bpy_prop_collection of FModifier, (readonly) 
    '''

    mute = None
    '''NLA Strip is not evaluated 

    :type:  boolean, default False 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''

    repeat = None
    '''Number of times to repeat the action range 

    :type:  float in [0.1, 1000], default 0.0 
    '''

    scale = None
    '''Scaling factor for action 

    :type:  float in [0.0001, 1000], default 0.0 
    '''

    select = None
    '''NLA Strip is selected 

    :type:  boolean, default False 
    '''

    strip_time = None
    '''Frame of referenced Action to evaluate 

    :type:  float in [-inf, inf], default 0.0 
    '''

    strips = None
    '''NLA Strips that this strip acts as a container for (if it is of type Meta) 

    :type:  bpy_prop_collection of NlaStrip, (readonly) 
    '''

    type = None
    '''Type of NLA Strip 

    :type:  enum in [‘CLIP’, ‘TRANSITION’, ‘META’, ‘SOUND’], default ‘CLIP’, (readonly) 
    '''

    use_animated_influence = None
    '''Influence setting is controlled by an F-Curve rather than automatically determined 

    :type:  boolean, default False 
    '''

    use_animated_time = None
    '''Strip time is controlled by an F-Curve rather than automatically determined 

    :type:  boolean, default False 
    '''

    use_animated_time_cyclic = None
    '''Cycle the animated time within the action start & end 

    :type:  boolean, default False 
    '''

    use_auto_blend = None
    '''Number of frames for Blending In/Out is automatically determined from overlapping strips 

    :type:  boolean, default False 
    '''

    use_reverse = None
    '''NLA Strip is played back in reverse order (only when timing is automatically determined) 

    :type:  boolean, default False 
    '''

    use_sync_length = None
    '''Update range of frames referenced from action after tweaking strip and its keyframes 

    :type:  boolean, default False 
    '''


class NlaStripFCurves:
    '''Collection of NLA strip F-Curves '''

    def find(self, data_path, index=0):
        '''Find an F-Curve. Note that this function performs a linear scan of all F-Curves in the NLA strip. 

        :param data_path: Data Path, F-Curve data path 
        :type data_path: string, (never None)
        :param index: Index, Array index 
        :type index: int in [0, inf], (optional)
        :rtype: FCurve 
        :return:  The found F-Curve, or None if it doesn’t exist 
        '''
        pass


class NlaStrips:
    '''Collection of Nla Strips '''

    def new(self, name, start, action):
        '''Add a new Action-Clip strip to the track 

        :param name: Name for the NLA Strips 
        :type name: string, (never None)
        :param start: Start Frame, Start frame for this strip 
        :type start: int in [-inf, inf]
        :param action: Action to assign to this strip 
        :type action: Action, (never None)
        :rtype: NlaStrip 
        :return:  New NLA Strip 
        '''
        pass

    def remove(self, strip):
        '''Remove a NLA Strip 

        :param strip: NLA Strip to remove 
        :type strip: NlaStrip, (never None)
        '''
        pass


class NlaTrack:
    '''A animation layer containing Actions referenced as NLA strips '''

    active = None
    '''NLA Track is active 

    :type:  boolean, default False, (readonly) 
    '''

    is_solo = None
    '''NLA Track is evaluated itself (i.e. active Action and all other NLA Tracks in the same AnimData block are disabled) 

    :type:  boolean, default False 
    '''

    lock = None
    '''NLA Track is locked 

    :type:  boolean, default False 
    '''

    mute = None
    '''NLA Track is not evaluated 

    :type:  boolean, default False 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''

    select = None
    '''NLA Track is selected 

    :type:  boolean, default False 
    '''

    strips = None
    '''NLA Strips on this NLA-track 

    :type:  NlaStrips bpy_prop_collection of NlaStrip, (readonly) 
    '''


class NlaTracks:
    '''Collection of NLA Tracks '''

    active = None
    '''Active Object constraint 

    :type:  NlaTrack 
    '''

    def new(self, prev=None):
        '''Add a new NLA Track 

        :param prev: NLA Track to add the new one after 
        :type prev: NlaTrack, (optional)
        :rtype: NlaTrack 
        :return:  New NLA Track 
        '''
        pass

    def remove(self, track):
        '''Remove a NLA Track 

        :param track: NLA Track to remove 
        :type track: NlaTrack, (never None)
        '''
        pass


class Node:
    '''Node in a node tree '''

    bl_description = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_height_default = None
    '''

    :type:  float in [0, inf], default 0.0 
    '''

    bl_height_max = None
    '''

    :type:  float in [0, inf], default 0.0 
    '''

    bl_height_min = None
    '''

    :type:  float in [0, inf], default 0.0 
    '''

    bl_icon = None
    '''The node icon 

    :type:  enum in [‘NONE’, ‘QUESTION’, ‘ERROR’, ‘CANCEL’, ‘TRIA_RIGHT’, ‘TRIA_DOWN’, ‘TRIA_LEFT’, ‘TRIA_UP’, ‘ARROW_LEFTRIGHT’, ‘PLUS’, ‘DISCLOSURE_TRI_DOWN’, ‘DISCLOSURE_TRI_RIGHT’, ‘RADIOBUT_OFF’, ‘RADIOBUT_ON’, ‘MENU_PANEL’, ‘BLENDER’, ‘GRIP’, ‘DOT’, ‘COLLAPSEMENU’, ‘X’, ‘GO_LEFT’, ‘PLUG’, ‘UI’, ‘NODE’, ‘NODE_SEL’, ‘FULLSCREEN’, ‘SPLITSCREEN’, ‘RIGHTARROW_THIN’, ‘BORDERMOVE’, ‘VIEWZOOM’, ‘ZOOMIN’, ‘ZOOMOUT’, ‘PANEL_CLOSE’, ‘COPY_ID’, ‘EYEDROPPER’, ‘LINK_AREA’, ‘AUTO’, ‘CHECKBOX_DEHLT’, ‘CHECKBOX_HLT’, ‘UNLOCKED’, ‘LOCKED’, ‘UNPINNED’, ‘PINNED’, ‘SCREEN_BACK’, ‘RIGHTARROW’, ‘DOWNARROW_HLT’, ‘DOTSUP’, ‘DOTSDOWN’, ‘LINK’, ‘INLINK’, ‘PLUGIN’, ‘HELP’, ‘GHOST_ENABLED’, ‘COLOR’, ‘LINKED’, ‘UNLINKED’, ‘HAND’, ‘ZOOM_ALL’, ‘ZOOM_SELECTED’, ‘ZOOM_PREVIOUS’, ‘ZOOM_IN’, ‘ZOOM_OUT’, ‘RENDER_REGION’, ‘BORDER_RECT’, ‘BORDER_LASSO’, ‘FREEZE’, ‘STYLUS_PRESSURE’, ‘GHOST_DISABLED’, ‘NEW’, ‘FILE_TICK’, ‘QUIT’, ‘URL’, ‘RECOVER_LAST’, ‘FULLSCREEN_ENTER’, ‘FULLSCREEN_EXIT’, ‘BLANK1’, ‘LAMP’, ‘MATERIAL’, ‘TEXTURE’, ‘ANIM’, ‘WORLD’, ‘SCENE’, ‘EDIT’, ‘GAME’, ‘RADIO’, ‘SCRIPT’, ‘PARTICLES’, ‘PHYSICS’, ‘SPEAKER’, ‘TEXTURE_SHADED’, ‘VIEW3D’, ‘IPO’, ‘OOPS’, ‘BUTS’, ‘FILESEL’, ‘IMAGE_COL’, ‘INFO’, ‘SEQUENCE’, ‘TEXT’, ‘IMASEL’, ‘SOUND’, ‘ACTION’, ‘NLA’, ‘SCRIPTWIN’, ‘TIME’, ‘NODETREE’, ‘LOGIC’, ‘CONSOLE’, ‘PREFERENCES’, ‘CLIP’, ‘ASSET_MANAGER’, ‘OBJECT_DATAMODE’, ‘EDITMODE_HLT’, ‘FACESEL_HLT’, ‘VPAINT_HLT’, ‘TPAINT_HLT’, ‘WPAINT_HLT’, ‘SCULPTMODE_HLT’, ‘POSE_HLT’, ‘PARTICLEMODE’, ‘LIGHTPAINT’, ‘SCENE_DATA’, ‘RENDERLAYERS’, ‘WORLD_DATA’, ‘OBJECT_DATA’, ‘MESH_DATA’, ‘CURVE_DATA’, ‘META_DATA’, ‘LATTICE_DATA’, ‘LAMP_DATA’, ‘MATERIAL_DATA’, ‘TEXTURE_DATA’, ‘ANIM_DATA’, ‘CAMERA_DATA’, ‘PARTICLE_DATA’, ‘LIBRARY_DATA_DIRECT’, ‘GROUP’, ‘ARMATURE_DATA’, ‘POSE_DATA’, ‘BONE_DATA’, ‘CONSTRAINT’, ‘SHAPEKEY_DATA’, ‘CONSTRAINT_BONE’, ‘CAMERA_STEREO’, ‘PACKAGE’, ‘UGLYPACKAGE’, ‘BRUSH_DATA’, ‘IMAGE_DATA’, ‘FILE’, ‘FCURVE’, ‘FONT_DATA’, ‘RENDER_RESULT’, ‘SURFACE_DATA’, ‘EMPTY_DATA’, ‘SETTINGS’, ‘RENDER_ANIMATION’, ‘RENDER_STILL’, ‘LIBRARY_DATA_BROKEN’, ‘BOIDS’, ‘STRANDS’, ‘LIBRARY_DATA_INDIRECT’, ‘GREASEPENCIL’, ‘LINE_DATA’, ‘GROUP_BONE’, ‘GROUP_VERTEX’, ‘GROUP_VCOL’, ‘GROUP_UVS’, ‘RNA’, ‘RNA_ADD’, ‘OUTLINER_OB_EMPTY’, ‘OUTLINER_OB_MESH’, ‘OUTLINER_OB_CURVE’, ‘OUTLINER_OB_LATTICE’, ‘OUTLINER_OB_META’, ‘OUTLINER_OB_LAMP’, ‘OUTLINER_OB_CAMERA’, ‘OUTLINER_OB_ARMATURE’, ‘OUTLINER_OB_FONT’, ‘OUTLINER_OB_SURFACE’, ‘OUTLINER_OB_SPEAKER’, ‘RESTRICT_COLOR_OFF’, ‘RESTRICT_COLOR_ON’, ‘RESTRICT_VIEW_OFF’, ‘RESTRICT_VIEW_ON’, ‘RESTRICT_SELECT_OFF’, ‘RESTRICT_SELECT_ON’, ‘RESTRICT_RENDER_OFF’, ‘RESTRICT_RENDER_ON’, ‘OUTLINER_DATA_EMPTY’, ‘OUTLINER_DATA_MESH’, ‘OUTLINER_DATA_CURVE’, ‘OUTLINER_DATA_LATTICE’, ‘OUTLINER_DATA_META’, ‘OUTLINER_DATA_LAMP’, ‘OUTLINER_DATA_CAMERA’, ‘OUTLINER_DATA_ARMATURE’, ‘OUTLINER_DATA_FONT’, ‘OUTLINER_DATA_SURFACE’, ‘OUTLINER_DATA_SPEAKER’, ‘OUTLINER_DATA_POSE’, ‘MESH_PLANE’, ‘MESH_CUBE’, ‘MESH_CIRCLE’, ‘MESH_UVSPHERE’, ‘MESH_ICOSPHERE’, ‘MESH_GRID’, ‘MESH_MONKEY’, ‘MESH_CYLINDER’, ‘MESH_TORUS’, ‘MESH_CONE’, ‘MESH_CAPSULE’, ‘LAMP_POINT’, ‘LAMP_SUN’, ‘LAMP_SPOT’, ‘LAMP_HEMI’, ‘LAMP_AREA’, ‘META_EMPTY’, ‘META_PLANE’, ‘META_CUBE’, ‘META_BALL’, ‘META_ELLIPSOID’, ‘META_CAPSULE’, ‘SURFACE_NCURVE’, ‘SURFACE_NCIRCLE’, ‘SURFACE_NSURFACE’, ‘SURFACE_NCYLINDER’, ‘SURFACE_NSPHERE’, ‘SURFACE_NTORUS’, ‘CURVE_BEZCURVE’, ‘CURVE_BEZCIRCLE’, ‘CURVE_NCURVE’, ‘CURVE_NCIRCLE’, ‘CURVE_PATH’, ‘COLOR_RED’, ‘COLOR_GREEN’, ‘COLOR_BLUE’, ‘TRIA_RIGHT_BAR’, ‘TRIA_DOWN_BAR’, ‘TRIA_LEFT_BAR’, ‘TRIA_UP_BAR’, ‘FORCE_FORCE’, ‘FORCE_WIND’, ‘FORCE_VORTEX’, ‘FORCE_MAGNETIC’, ‘FORCE_HARMONIC’, ‘FORCE_CHARGE’, ‘FORCE_LENNARDJONES’, ‘FORCE_TEXTURE’, ‘FORCE_CURVE’, ‘FORCE_BOID’, ‘FORCE_TURBULENCE’, ‘FORCE_DRAG’, ‘FORCE_SMOKEFLOW’, ‘NODE_INSERT_ON’, ‘NODE_INSERT_OFF’, ‘MODIFIER’, ‘MOD_WAVE’, ‘MOD_BUILD’, ‘MOD_DECIM’, ‘MOD_MIRROR’, ‘MOD_SOFT’, ‘MOD_SUBSURF’, ‘HOOK’, ‘MOD_PHYSICS’, ‘MOD_PARTICLES’, ‘MOD_BOOLEAN’, ‘MOD_EDGESPLIT’, ‘MOD_ARRAY’, ‘MOD_UVPROJECT’, ‘MOD_DISPLACE’, ‘MOD_CURVE’, ‘MOD_LATTICE’, ‘CONSTRAINT_DATA’, ‘MOD_ARMATURE’, ‘MOD_SHRINKWRAP’, ‘MOD_CAST’, ‘MOD_MESHDEFORM’, ‘MOD_BEVEL’, ‘MOD_SMOOTH’, ‘MOD_SIMPLEDEFORM’, ‘MOD_MASK’, ‘MOD_CLOTH’, ‘MOD_EXPLODE’, ‘MOD_FLUIDSIM’, ‘MOD_MULTIRES’, ‘MOD_SMOKE’, ‘MOD_SOLIDIFY’, ‘MOD_SCREW’, ‘MOD_VERTEX_WEIGHT’, ‘MOD_DYNAMICPAINT’, ‘MOD_REMESH’, ‘MOD_OCEAN’, ‘MOD_WARP’, ‘MOD_SKIN’, ‘MOD_TRIANGULATE’, ‘MOD_WIREFRAME’, ‘MOD_DATA_TRANSFER’, ‘MOD_NORMALEDIT’, ‘REC’, ‘PLAY’, ‘FF’, ‘REW’, ‘PAUSE’, ‘PREV_KEYFRAME’, ‘NEXT_KEYFRAME’, ‘PLAY_AUDIO’, ‘PLAY_REVERSE’, ‘PREVIEW_RANGE’, ‘ACTION_TWEAK’, ‘PMARKER_ACT’, ‘PMARKER_SEL’, ‘PMARKER’, ‘MARKER_HLT’, ‘MARKER’, ‘SPACE2’, ‘SPACE3’, ‘KEYINGSET’, ‘KEY_DEHLT’, ‘KEY_HLT’, ‘MUTE_IPO_OFF’, ‘MUTE_IPO_ON’, ‘VISIBLE_IPO_OFF’, ‘VISIBLE_IPO_ON’, ‘DRIVER’, ‘SOLO_OFF’, ‘SOLO_ON’, ‘FRAME_PREV’, ‘FRAME_NEXT’, ‘NLA_PUSHDOWN’, ‘IPO_CONSTANT’, ‘IPO_LINEAR’, ‘IPO_BEZIER’, ‘IPO_SINE’, ‘IPO_QUAD’, ‘IPO_CUBIC’, ‘IPO_QUART’, ‘IPO_QUINT’, ‘IPO_EXPO’, ‘IPO_CIRC’, ‘IPO_BOUNCE’, ‘IPO_ELASTIC’, ‘IPO_BACK’, ‘IPO_EASE_IN’, ‘IPO_EASE_OUT’, ‘IPO_EASE_IN_OUT’, ‘VERTEXSEL’, ‘EDGESEL’, ‘FACESEL’, ‘LOOPSEL’, ‘ROTATE’, ‘CURSOR’, ‘ROTATECOLLECTION’, ‘ROTATECENTER’, ‘ROTACTIVE’, ‘ALIGN’, ‘SMOOTHCURVE’, ‘SPHERECURVE’, ‘ROOTCURVE’, ‘SHARPCURVE’, ‘LINCURVE’, ‘NOCURVE’, ‘RNDCURVE’, ‘PROP_OFF’, ‘PROP_ON’, ‘PROP_CON’, ‘SCULPT_DYNTOPO’, ‘PARTICLE_POINT’, ‘PARTICLE_TIP’, ‘PARTICLE_PATH’, ‘MAN_TRANS’, ‘MAN_ROT’, ‘MAN_SCALE’, ‘MANIPUL’, ‘SNAP_OFF’, ‘SNAP_ON’, ‘SNAP_NORMAL’, ‘SNAP_GRID’, ‘SNAP_VERTEX’, ‘SNAP_EDGE’, ‘SNAP_FACE’, ‘SNAP_VOLUME’, ‘SNAP_INCREMENT’, ‘STICKY_UVS_LOC’, ‘STICKY_UVS_DISABLE’, ‘STICKY_UVS_VERT’, ‘CLIPUV_DEHLT’, ‘CLIPUV_HLT’, ‘SNAP_PEEL_OBJECT’, ‘GRID’, ‘PASTEDOWN’, ‘COPYDOWN’, ‘PASTEFLIPUP’, ‘PASTEFLIPDOWN’, ‘SNAP_SURFACE’, ‘AUTOMERGE_ON’, ‘AUTOMERGE_OFF’, ‘RETOPO’, ‘UV_VERTEXSEL’, ‘UV_EDGESEL’, ‘UV_FACESEL’, ‘UV_ISLANDSEL’, ‘UV_SYNC_SELECT’, ‘BBOX’, ‘WIRE’, ‘SOLID’, ‘SMOOTH’, ‘POTATO’, ‘ORTHO’, ‘LOCKVIEW_OFF’, ‘LOCKVIEW_ON’, ‘AXIS_SIDE’, ‘AXIS_FRONT’, ‘AXIS_TOP’, ‘NDOF_DOM’, ‘NDOF_TURN’, ‘NDOF_FLY’, ‘NDOF_TRANS’, ‘LAYER_USED’, ‘LAYER_ACTIVE’, ‘SORTALPHA’, ‘SORTBYEXT’, ‘SORTTIME’, ‘SORTSIZE’, ‘LONGDISPLAY’, ‘SHORTDISPLAY’, ‘GHOST’, ‘IMGDISPLAY’, ‘SAVE_AS’, ‘SAVE_COPY’, ‘BOOKMARKS’, ‘FONTPREVIEW’, ‘FILTER’, ‘NEWFOLDER’, ‘OPEN_RECENT’, ‘FILE_PARENT’, ‘FILE_REFRESH’, ‘FILE_FOLDER’, ‘FILE_BLANK’, ‘FILE_BLEND’, ‘FILE_IMAGE’, ‘FILE_MOVIE’, ‘FILE_SCRIPT’, ‘FILE_SOUND’, ‘FILE_FONT’, ‘FILE_TEXT’, ‘RECOVER_AUTO’, ‘SAVE_PREFS’, ‘LINK_BLEND’, ‘APPEND_BLEND’, ‘IMPORT’, ‘EXPORT’, ‘EXTERNAL_DATA’, ‘LOAD_FACTORY’, ‘LOOP_BACK’, ‘LOOP_FORWARDS’, ‘BACK’, ‘FORWARD’, ‘FILE_HIDDEN’, ‘FILE_BACKUP’, ‘DISK_DRIVE’, ‘MATPLANE’, ‘MATSPHERE’, ‘MATCUBE’, ‘MONKEY’, ‘HAIR’, ‘ALIASED’, ‘ANTIALIASED’, ‘MAT_SPHERE_SKY’, ‘WORDWRAP_OFF’, ‘WORDWRAP_ON’, ‘SYNTAX_OFF’, ‘SYNTAX_ON’, ‘LINENUMBERS_OFF’, ‘LINENUMBERS_ON’, ‘SCRIPTPLUGINS’, ‘SEQ_SEQUENCER’, ‘SEQ_PREVIEW’, ‘SEQ_LUMA_WAVEFORM’, ‘SEQ_CHROMA_SCOPE’, ‘SEQ_HISTOGRAM’, ‘SEQ_SPLITVIEW’, ‘IMAGE_RGB’, ‘IMAGE_RGB_ALPHA’, ‘IMAGE_ALPHA’, ‘IMAGE_ZDEPTH’, ‘IMAGEFILE’, ‘BRUSH_ADD’, ‘BRUSH_BLOB’, ‘BRUSH_BLUR’, ‘BRUSH_CLAY’, ‘BRUSH_CLAY_STRIPS’, ‘BRUSH_CLONE’, ‘BRUSH_CREASE’, ‘BRUSH_DARKEN’, ‘BRUSH_FILL’, ‘BRUSH_FLATTEN’, ‘BRUSH_GRAB’, ‘BRUSH_INFLATE’, ‘BRUSH_LAYER’, ‘BRUSH_LIGHTEN’, ‘BRUSH_MASK’, ‘BRUSH_MIX’, ‘BRUSH_MULTIPLY’, ‘BRUSH_NUDGE’, ‘BRUSH_PINCH’, ‘BRUSH_SCRAPE’, ‘BRUSH_SCULPT_DRAW’, ‘BRUSH_SMEAR’, ‘BRUSH_SMOOTH’, ‘BRUSH_SNAKE_HOOK’, ‘BRUSH_SOFTEN’, ‘BRUSH_SUBTRACT’, ‘BRUSH_TEXDRAW’, ‘BRUSH_TEXFILL’, ‘BRUSH_TEXMASK’, ‘BRUSH_THUMB’, ‘BRUSH_ROTATE’, ‘BRUSH_VERTEXDRAW’, ‘MATCAP_01’, ‘MATCAP_02’, ‘MATCAP_03’, ‘MATCAP_04’, ‘MATCAP_05’, ‘MATCAP_06’, ‘MATCAP_07’, ‘MATCAP_08’, ‘MATCAP_09’, ‘MATCAP_10’, ‘MATCAP_11’, ‘MATCAP_12’, ‘MATCAP_13’, ‘MATCAP_14’, ‘MATCAP_15’, ‘MATCAP_16’, ‘MATCAP_17’, ‘MATCAP_18’, ‘MATCAP_19’, ‘MATCAP_20’, ‘MATCAP_21’, ‘MATCAP_22’, ‘MATCAP_23’, ‘MATCAP_24’], default ‘NODE’ 
    '''

    bl_idname = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_label = None
    '''The node label 

    :type:  string, default “”, (never None) 
    '''

    bl_static_type = None
    '''Node type (deprecated, use with care) 

    :type:  enum in [‘CUSTOM’], default ‘CUSTOM’ 
    '''

    bl_width_default = None
    '''

    :type:  float in [0, inf], default 0.0 
    '''

    bl_width_max = None
    '''

    :type:  float in [0, inf], default 0.0 
    '''

    bl_width_min = None
    '''

    :type:  float in [0, inf], default 0.0 
    '''

    color = None
    '''Custom color of the node body 

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    dimensions = None
    '''Absolute bounding box dimensions of the node 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0), (readonly) 
    '''

    height = None
    '''Height of the node 

    :type:  float in [-inf, inf], default 0.0 
    '''

    hide = None
    '''

    :type:  boolean, default False 
    '''

    inputs = None
    '''

    :type:  NodeInputs bpy_prop_collection of NodeSocket, (readonly) 
    '''

    internal_links = None
    '''Internal input-to-output connections for muting 

    :type:  bpy_prop_collection of NodeLink, (readonly) 
    '''

    label = None
    '''Optional custom node label 

    :type:  string, default “”, (never None) 
    '''

    location = None
    '''

    :type:  float array of 2 items in [-100000, 100000], default (0.0, 0.0) 
    '''

    mute = None
    '''

    :type:  boolean, default False 
    '''

    name = None
    '''Unique node identifier 

    :type:  string, default “”, (never None) 
    '''

    outputs = None
    '''

    :type:  NodeOutputs bpy_prop_collection of NodeSocket, (readonly) 
    '''

    parent = None
    '''Parent this node is attached to 

    :type:  Node 
    '''

    select = None
    '''Node selection state 

    :type:  boolean, default False 
    '''

    shading_compatibility = None
    '''

    :type:  enum set in {‘OLD_SHADING’, ‘NEW_SHADING’}, default {‘OLD_SHADING’}, (readonly) 
    '''

    show_options = None
    '''

    :type:  boolean, default False 
    '''

    show_preview = None
    '''

    :type:  boolean, default False 
    '''

    show_texture = None
    '''Draw node in viewport textured draw mode 

    :type:  boolean, default False 
    '''

    type = None
    '''Node type (deprecated, use bl_static_type or bl_idname for the actual identifier string) 

    :type:  enum in [‘CUSTOM’], default ‘CUSTOM’, (readonly) 
    '''

    use_custom_color = None
    '''Use custom color for the node 

    :type:  boolean, default False 
    '''

    width = None
    '''Width of the node 

    :type:  float in [-inf, inf], default 0.0 
    '''

    width_hidden = None
    '''Width of the node in hidden state 

    :type:  float in [-inf, inf], default 0.0 
    '''

    def socket_value_update(self, context):
        '''Update after property changes 

        '''
        pass

    def poll_instance(self, node_tree):
        '''If non-null output is returned, the node can be added to the tree 

        :param node_tree: Node Tree 
        :type node_tree: NodeTree
        :rtype: boolean 
        '''
        pass

    def update(self):
        '''Update on editor changes 

        '''
        pass

    def insert_link(self, link):
        '''Handle creation of a link to or from the node 

        :param link: Link, Node link that will be inserted 
        :type link: NodeLink, (never None)
        '''
        pass

    def init(self, context):
        '''Initialize a new instance of this node 

        '''
        pass

    def copy(self, node):
        '''Initialize a new instance of this node from an existing node 

        :param node: Node, Existing node to copy 
        :type node: Node, (never None)
        '''
        pass

    def free(self):
        '''Clean up node on removal 

        '''
        pass

    def draw_buttons(self, context, layout):
        '''Draw node buttons 

        :param layout: Layout, Layout in the UI 
        :type layout: UILayout, (never None)
        '''
        pass

    def draw_buttons_ext(self, context, layout):
        '''Draw node buttons in the sidebar 

        :param layout: Layout, Layout in the UI 
        :type layout: UILayout, (never None)
        '''
        pass

    def draw_label(self):
        '''Returns a dynamic label string 

        :rtype: string, (never None) 
        :return:  Label 
        '''
        pass


class NodeCustomGroup:
    '''Base node type for custom registered node group types '''

    interface = None
    '''Interface socket data 

    :type:  PropertyGroup, (readonly) 
    '''

    node_tree = None
    '''

    :type:  NodeTree 
    '''


class NodeFrame:
    label_size = None
    '''Font size to use for displaying the label 

    :type:  int in [8, 64], default 0 
    '''

    shrink = None
    '''Shrink the frame to minimal bounding box 

    :type:  boolean, default False 
    '''

    text = None
    '''

    :type:  Text 
    '''


class NodeGroup:
    interface = None
    '''Interface socket data 

    :type:  PropertyGroup, (readonly) 
    '''

    node_tree = None
    '''

    :type:  NodeTree 
    '''


class NodeGroupInput:
    interface = None
    '''Interface socket data 

    :type:  PropertyGroup, (readonly) 
    '''


class NodeGroupOutput:
    interface = None
    '''Interface socket data 

    :type:  PropertyGroup, (readonly) 
    '''

    is_active_output = None
    '''True if this node is used as the active group output 

    :type:  boolean, default False 
    '''


class NodeInputs:
    '''Collection of Node Sockets '''

    def new(self, type, name, identifier=""):
        '''Add a socket to this node 

        :param type: Type, Data type 
        :type type: string, (never None)
        :param name: Name 
        :type name: string, (never None)
        :param identifier: Identifier, Unique socket identifier 
        :type identifier: string, (optional, never None)
        :rtype: NodeSocket 
        :return:  New socket 
        '''
        pass

    def remove(self, socket):
        '''Remove a socket from this node 

        :param socket: The socket to remove 
        :type socket: NodeSocket
        '''
        pass

    def clear(self):
        '''Remove all sockets from this node 

        '''
        pass

    def move(self, from_index, to_index):
        '''Move a socket to another position 

        :param from_index: From Index, Index of the socket to move 
        :type from_index: int in [0, inf]
        :param to_index: To Index, Target index for the socket 
        :type to_index: int in [0, inf]
        '''
        pass


class NodeInstanceHash:
    '''Hash table containing node instance data '''

    pass


class NodeInternal:
    def poll_instance(self, node_tree):
        '''If non-null output is returned, the node can be added to the tree 

        :param node_tree: Node Tree 
        :type node_tree: NodeTree
        :rtype: boolean 
        '''
        pass

    def update(self):
        '''Update on editor changes 

        '''
        pass

    def draw_buttons(self, context, layout):
        '''Draw node buttons 

        :param layout: Layout, Layout in the UI 
        :type layout: UILayout, (never None)
        '''
        pass

    def draw_buttons_ext(self, context, layout):
        '''Draw node buttons in the sidebar 

        :param layout: Layout, Layout in the UI 
        :type layout: UILayout, (never None)
        '''
        pass


class NodeInternalSocketTemplate:
    '''Type and default value of a node socket '''

    identifier = None
    '''Identifier of the socket 

    :type:  string, default “”, (readonly, never None) 
    '''

    name = None
    '''Name of the socket 

    :type:  string, default “”, (readonly, never None) 
    '''

    type = None
    '''Data type of the socket 

    :type:  enum in [‘CUSTOM’, ‘VALUE’, ‘INT’, ‘BOOLEAN’, ‘VECTOR’, ‘STRING’, ‘RGBA’, ‘SHADER’], default ‘VALUE’, (readonly) 
    '''


class NodeLink:
    '''Link is valid '''

    from_node = None
    '''

    :type:  Node, (readonly) 
    '''

    from_socket = None
    '''

    :type:  NodeSocket, (readonly) 
    '''

    is_hidden = None
    '''Link is hidden due to invisible sockets 

    :type:  boolean, default False, (readonly) 
    '''

    is_valid = None
    '''

    :type:  boolean, default False 
    '''

    to_node = None
    '''

    :type:  Node, (readonly) 
    '''

    to_socket = None
    '''

    :type:  NodeSocket, (readonly) 
    '''


class NodeLinks:
    '''Collection of Node Links '''

    def new(self, input, output, verify_limits=True):
        '''Add a node link to this node tree 

        :param input: The input socket 
        :type input: NodeSocket, (never None)
        :param output: The output socket 
        :type output: NodeSocket, (never None)
        :param verify_limits: Verify Limits, Remove existing links if connection limit is exceeded 
        :type verify_limits: boolean, (optional)
        :rtype: NodeLink 
        :return:  New node link 
        '''
        pass

    def remove(self, link):
        '''remove a node link from the node tree 

        :param link: The node link to remove 
        :type link: NodeLink, (never None)
        '''
        pass

    def clear(self):
        '''remove all node links from the node tree 

        '''
        pass


class NodeOutputFileSlotFile:
    '''Single layer file slot of the file output node '''

    format = None
    '''

    :type:  ImageFormatSettings, (readonly) 
    '''

    path = None
    '''Subpath used for this slot 

    :type:  string, default “”, (never None) 
    '''

    use_node_format = None
    '''

    :type:  boolean, default False 
    '''


class NodeOutputFileSlotLayer:
    '''Multilayer slot of the file output node '''

    name = None
    '''OpenEXR layer name used for this slot 

    :type:  string, default “”, (never None) 
    '''


class NodeOutputs:
    '''Collection of Node Sockets '''

    def new(self, type, name, identifier=""):
        '''Add a socket to this node 

        :param type: Type, Data type 
        :type type: string, (never None)
        :param name: Name 
        :type name: string, (never None)
        :param identifier: Identifier, Unique socket identifier 
        :type identifier: string, (optional, never None)
        :rtype: NodeSocket 
        :return:  New socket 
        '''
        pass

    def remove(self, socket):
        '''Remove a socket from this node 

        :param socket: The socket to remove 
        :type socket: NodeSocket
        '''
        pass

    def clear(self):
        '''Remove all sockets from this node 

        '''
        pass

    def move(self, from_index, to_index):
        '''Move a socket to another position 

        :param from_index: From Index, Index of the socket to move 
        :type from_index: int in [0, inf]
        :param to_index: To Index, Target index for the socket 
        :type to_index: int in [0, inf]
        '''
        pass


class NodeReroute:
    pass


class NodeSetting:
    value = None
    '''Python expression to be evaluated as the initial node setting 

    :type:  string, default “”, (never None) 
    '''


class NodeSocket:
    '''Input or output socket of a node '''

    bl_idname = None
    '''

    :type:  string, default “”, (never None) 
    '''

    enabled = None
    '''Enable the socket 

    :type:  boolean, default False 
    '''

    hide = None
    '''Hide the socket 

    :type:  boolean, default False 
    '''

    hide_value = None
    '''Hide the socket input value 

    :type:  boolean, default False 
    '''

    identifier = None
    '''Unique identifier for mapping sockets 

    :type:  string, default “”, (readonly, never None) 
    '''

    is_linked = None
    '''True if the socket is connected 

    :type:  boolean, default False, (readonly) 
    '''

    is_output = None
    '''True if the socket is an output, otherwise input 

    :type:  boolean, default False, (readonly) 
    '''

    link_limit = None
    '''Max number of links allowed for this socket 

    :type:  int in [1, 4095], default 0 
    '''

    name = None
    '''Socket name 

    :type:  string, default “”, (never None) 
    '''

    node = None
    '''Node owning this socket 

    :type:  Node, (readonly) 
    '''

    show_expanded = None
    '''Socket links are expanded in the user interface 

    :type:  boolean, default False 
    '''

    type = None
    '''Data type 

    :type:  enum in [‘CUSTOM’, ‘VALUE’, ‘INT’, ‘BOOLEAN’, ‘VECTOR’, ‘STRING’, ‘RGBA’, ‘SHADER’], default ‘VALUE’ 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''

    def draw(self, context, layout, node, text):
        '''Draw socket 

        :param layout: Layout, Layout in the UI 
        :type layout: UILayout, (never None)
        :param node: Node, Node the socket belongs to 
        :type node: Node, (never None)
        :param text: Text, Text label to draw alongside properties 
        :type text: string, (never None)
        '''
        pass

    def draw_color(self, context, node):
        '''Color of the socket icon 

        :param node: Node, Node the socket belongs to 
        :type node: Node, (never None)
        :rtype: float array of 4 items in [0, 1] 
        :return:  Color 
        '''
        pass


class NodeSocketBool:
    '''Boolean value socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  boolean, default False 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketColor:
    '''RGBA color socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 4 items in [0, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketFloat:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [-inf, inf], default 0.0 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketFloatAngle:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [-inf, inf], default 0.0 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketFloatFactor:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [0, 1], default 0.0 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketFloatPercentage:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [-inf, inf], default 0.0 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketFloatTime:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [-inf, inf], default 0.0 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketFloatUnsigned:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [0, inf], default 0.0 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketInt:
    '''Integer number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  int in [-inf, inf], default 0 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketIntFactor:
    '''Integer number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  int in [0, inf], default 1 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketIntPercentage:
    '''Integer number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  int in [0, inf], default 100 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketIntUnsigned:
    '''Integer number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  int in [0, inf], default 0 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketInterface:
    '''Parameters to define node sockets '''

    bl_socket_idname = None
    '''

    :type:  string, default “”, (never None) 
    '''

    identifier = None
    '''Unique identifier for mapping sockets 

    :type:  string, default “”, (readonly, never None) 
    '''

    is_output = None
    '''True if the socket is an output, otherwise input 

    :type:  boolean, default False, (readonly) 
    '''

    name = None
    '''Socket name 

    :type:  string, default “”, (never None) 
    '''

    def draw(self, context, layout):
        '''Draw template settings 

        :param layout: Layout, Layout in the UI 
        :type layout: UILayout, (never None)
        '''
        pass

    def draw_color(self, context):
        '''Color of the socket icon 

        :rtype: float array of 4 items in [0, 1] 
        :return:  Color 
        '''
        pass

    def register_properties(self, data_rna_type):
        '''Define RNA properties of a socket 

        :param data_rna_type: Data RNA Type, RNA type for special socket properties 
        :type data_rna_type: Struct
        '''
        pass

    def init_socket(self, node, socket, data_path):
        '''Initialize a node socket instance 

        :param node: Node, Node of the socket to initialize 
        :type node: Node, (never None)
        :param socket: Socket, Socket to initialize 
        :type socket: NodeSocket, (never None)
        :param data_path: Data Path, Path to specialized socket data 
        :type data_path: string, (never None)
        '''
        pass

    def from_socket(self, node, socket):
        '''Setup template parameters from an existing socket 

        :param node: Node, Node of the original socket 
        :type node: Node, (never None)
        :param socket: Socket, Original socket 
        :type socket: NodeSocket, (never None)
        '''
        pass


class NodeSocketInterfaceBool:
    '''Boolean value socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  boolean, default False 
    '''


class NodeSocketInterfaceColor:
    '''RGBA color socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 4 items in [0, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''


class NodeSocketInterfaceFloat:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [-inf, inf], default 0.0 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceFloatAngle:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [-inf, inf], default 0.0 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceFloatFactor:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [0, 1], default 1.0 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceFloatPercentage:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [-inf, inf], default 100.0 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceFloatTime:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [-inf, inf], default 0.0 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceFloatUnsigned:
    '''Floating point number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float in [0, inf], default 0.0 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceInt:
    '''Integer number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  int in [-inf, inf], default 0 
    '''

    max_value = None
    '''Maximum value 

    :type:  int in [-inf, inf], default 0 
    '''

    min_value = None
    '''Minimum value 

    :type:  int in [-inf, inf], default 0 
    '''


class NodeSocketInterfaceIntFactor:
    '''Integer number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  int in [0, inf], default 0 
    '''

    max_value = None
    '''Maximum value 

    :type:  int in [-inf, inf], default 0 
    '''

    min_value = None
    '''Minimum value 

    :type:  int in [-inf, inf], default 0 
    '''


class NodeSocketInterfaceIntPercentage:
    '''Integer number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  int in [0, inf], default 0 
    '''

    max_value = None
    '''Maximum value 

    :type:  int in [-inf, inf], default 0 
    '''

    min_value = None
    '''Minimum value 

    :type:  int in [-inf, inf], default 0 
    '''


class NodeSocketInterfaceIntUnsigned:
    '''Integer number socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  int in [0, inf], default 0 
    '''

    max_value = None
    '''Maximum value 

    :type:  int in [-inf, inf], default 0 
    '''

    min_value = None
    '''Minimum value 

    :type:  int in [-inf, inf], default 0 
    '''


class NodeSocketInterfaceShader:
    '''Shader socket of a node '''

    pass


class NodeSocketInterfaceStandard:
    type = None
    '''Data type 

    :type:  enum in [‘CUSTOM’, ‘VALUE’, ‘INT’, ‘BOOLEAN’, ‘VECTOR’, ‘STRING’, ‘RGBA’, ‘SHADER’], default ‘VALUE’, (readonly) 
    '''

    def draw(self, context, layout):
        '''Draw template settings 

        :param layout: Layout, Layout in the UI 
        :type layout: UILayout, (never None)
        '''
        pass

    def draw_color(self, context):
        '''Color of the socket icon 

        :rtype: float array of 4 items in [0, 1] 
        :return:  Color 
        '''
        pass


class NodeSocketInterfaceString:
    '''String socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  string, default “”, (never None) 
    '''


class NodeSocketInterfaceVector:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceVectorAcceleration:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceVectorDirection:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceVectorEuler:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceVectorTranslation:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceVectorVelocity:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketInterfaceVectorXYZ:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    max_value = None
    '''Maximum value 

    :type:  float in [-inf, inf], default 0.0 
    '''

    min_value = None
    '''Minimum value 

    :type:  float in [-inf, inf], default 0.0 
    '''


class NodeSocketShader:
    '''Shader socket of a node '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketStandard:
    links = None
    '''List of node links from or to this socket (readonly) '''

    def draw(self, context, layout, node, text):
        '''Draw socket 

        :param layout: Layout, Layout in the UI 
        :type layout: UILayout, (never None)
        :param node: Node, Node the socket belongs to 
        :type node: Node, (never None)
        :param text: Text, Text label to draw alongside properties 
        :type text: string, (never None)
        '''
        pass

    def draw_color(self, context, node):
        '''Color of the socket icon 

        :param node: Node, Node the socket belongs to 
        :type node: Node, (never None)
        :rtype: float array of 4 items in [0, 1] 
        :return:  Color 
        '''
        pass


class NodeSocketString:
    '''String socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  string, default “”, (never None) 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketVector:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketVectorAcceleration:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketVectorDirection:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 1.0) 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketVectorEuler:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketVectorTranslation:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketVectorVelocity:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketVectorXYZ:
    '''3D vector socket of a node '''

    default_value = None
    '''Input value used for unconnected socket 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeSocketVirtual:
    '''Virtual socket of a node '''

    links = None
    '''List of node links from or to this socket (readonly) '''


class NodeTree:
    '''Node tree consisting of linked nodes used for shading, textures and compositing '''

    active_input = None
    '''Index of the active input 

    :type:  int in [0, inf], default 0 
    '''

    active_output = None
    '''Index of the active output 

    :type:  int in [0, inf], default 0 
    '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    bl_description = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_icon = None
    '''The node tree icon 

    :type:  enum in [‘NONE’, ‘QUESTION’, ‘ERROR’, ‘CANCEL’, ‘TRIA_RIGHT’, ‘TRIA_DOWN’, ‘TRIA_LEFT’, ‘TRIA_UP’, ‘ARROW_LEFTRIGHT’, ‘PLUS’, ‘DISCLOSURE_TRI_DOWN’, ‘DISCLOSURE_TRI_RIGHT’, ‘RADIOBUT_OFF’, ‘RADIOBUT_ON’, ‘MENU_PANEL’, ‘BLENDER’, ‘GRIP’, ‘DOT’, ‘COLLAPSEMENU’, ‘X’, ‘GO_LEFT’, ‘PLUG’, ‘UI’, ‘NODE’, ‘NODE_SEL’, ‘FULLSCREEN’, ‘SPLITSCREEN’, ‘RIGHTARROW_THIN’, ‘BORDERMOVE’, ‘VIEWZOOM’, ‘ZOOMIN’, ‘ZOOMOUT’, ‘PANEL_CLOSE’, ‘COPY_ID’, ‘EYEDROPPER’, ‘LINK_AREA’, ‘AUTO’, ‘CHECKBOX_DEHLT’, ‘CHECKBOX_HLT’, ‘UNLOCKED’, ‘LOCKED’, ‘UNPINNED’, ‘PINNED’, ‘SCREEN_BACK’, ‘RIGHTARROW’, ‘DOWNARROW_HLT’, ‘DOTSUP’, ‘DOTSDOWN’, ‘LINK’, ‘INLINK’, ‘PLUGIN’, ‘HELP’, ‘GHOST_ENABLED’, ‘COLOR’, ‘LINKED’, ‘UNLINKED’, ‘HAND’, ‘ZOOM_ALL’, ‘ZOOM_SELECTED’, ‘ZOOM_PREVIOUS’, ‘ZOOM_IN’, ‘ZOOM_OUT’, ‘RENDER_REGION’, ‘BORDER_RECT’, ‘BORDER_LASSO’, ‘FREEZE’, ‘STYLUS_PRESSURE’, ‘GHOST_DISABLED’, ‘NEW’, ‘FILE_TICK’, ‘QUIT’, ‘URL’, ‘RECOVER_LAST’, ‘FULLSCREEN_ENTER’, ‘FULLSCREEN_EXIT’, ‘BLANK1’, ‘LAMP’, ‘MATERIAL’, ‘TEXTURE’, ‘ANIM’, ‘WORLD’, ‘SCENE’, ‘EDIT’, ‘GAME’, ‘RADIO’, ‘SCRIPT’, ‘PARTICLES’, ‘PHYSICS’, ‘SPEAKER’, ‘TEXTURE_SHADED’, ‘VIEW3D’, ‘IPO’, ‘OOPS’, ‘BUTS’, ‘FILESEL’, ‘IMAGE_COL’, ‘INFO’, ‘SEQUENCE’, ‘TEXT’, ‘IMASEL’, ‘SOUND’, ‘ACTION’, ‘NLA’, ‘SCRIPTWIN’, ‘TIME’, ‘NODETREE’, ‘LOGIC’, ‘CONSOLE’, ‘PREFERENCES’, ‘CLIP’, ‘ASSET_MANAGER’, ‘OBJECT_DATAMODE’, ‘EDITMODE_HLT’, ‘FACESEL_HLT’, ‘VPAINT_HLT’, ‘TPAINT_HLT’, ‘WPAINT_HLT’, ‘SCULPTMODE_HLT’, ‘POSE_HLT’, ‘PARTICLEMODE’, ‘LIGHTPAINT’, ‘SCENE_DATA’, ‘RENDERLAYERS’, ‘WORLD_DATA’, ‘OBJECT_DATA’, ‘MESH_DATA’, ‘CURVE_DATA’, ‘META_DATA’, ‘LATTICE_DATA’, ‘LAMP_DATA’, ‘MATERIAL_DATA’, ‘TEXTURE_DATA’, ‘ANIM_DATA’, ‘CAMERA_DATA’, ‘PARTICLE_DATA’, ‘LIBRARY_DATA_DIRECT’, ‘GROUP’, ‘ARMATURE_DATA’, ‘POSE_DATA’, ‘BONE_DATA’, ‘CONSTRAINT’, ‘SHAPEKEY_DATA’, ‘CONSTRAINT_BONE’, ‘CAMERA_STEREO’, ‘PACKAGE’, ‘UGLYPACKAGE’, ‘BRUSH_DATA’, ‘IMAGE_DATA’, ‘FILE’, ‘FCURVE’, ‘FONT_DATA’, ‘RENDER_RESULT’, ‘SURFACE_DATA’, ‘EMPTY_DATA’, ‘SETTINGS’, ‘RENDER_ANIMATION’, ‘RENDER_STILL’, ‘LIBRARY_DATA_BROKEN’, ‘BOIDS’, ‘STRANDS’, ‘LIBRARY_DATA_INDIRECT’, ‘GREASEPENCIL’, ‘LINE_DATA’, ‘GROUP_BONE’, ‘GROUP_VERTEX’, ‘GROUP_VCOL’, ‘GROUP_UVS’, ‘RNA’, ‘RNA_ADD’, ‘OUTLINER_OB_EMPTY’, ‘OUTLINER_OB_MESH’, ‘OUTLINER_OB_CURVE’, ‘OUTLINER_OB_LATTICE’, ‘OUTLINER_OB_META’, ‘OUTLINER_OB_LAMP’, ‘OUTLINER_OB_CAMERA’, ‘OUTLINER_OB_ARMATURE’, ‘OUTLINER_OB_FONT’, ‘OUTLINER_OB_SURFACE’, ‘OUTLINER_OB_SPEAKER’, ‘RESTRICT_COLOR_OFF’, ‘RESTRICT_COLOR_ON’, ‘RESTRICT_VIEW_OFF’, ‘RESTRICT_VIEW_ON’, ‘RESTRICT_SELECT_OFF’, ‘RESTRICT_SELECT_ON’, ‘RESTRICT_RENDER_OFF’, ‘RESTRICT_RENDER_ON’, ‘OUTLINER_DATA_EMPTY’, ‘OUTLINER_DATA_MESH’, ‘OUTLINER_DATA_CURVE’, ‘OUTLINER_DATA_LATTICE’, ‘OUTLINER_DATA_META’, ‘OUTLINER_DATA_LAMP’, ‘OUTLINER_DATA_CAMERA’, ‘OUTLINER_DATA_ARMATURE’, ‘OUTLINER_DATA_FONT’, ‘OUTLINER_DATA_SURFACE’, ‘OUTLINER_DATA_SPEAKER’, ‘OUTLINER_DATA_POSE’, ‘MESH_PLANE’, ‘MESH_CUBE’, ‘MESH_CIRCLE’, ‘MESH_UVSPHERE’, ‘MESH_ICOSPHERE’, ‘MESH_GRID’, ‘MESH_MONKEY’, ‘MESH_CYLINDER’, ‘MESH_TORUS’, ‘MESH_CONE’, ‘MESH_CAPSULE’, ‘LAMP_POINT’, ‘LAMP_SUN’, ‘LAMP_SPOT’, ‘LAMP_HEMI’, ‘LAMP_AREA’, ‘META_EMPTY’, ‘META_PLANE’, ‘META_CUBE’, ‘META_BALL’, ‘META_ELLIPSOID’, ‘META_CAPSULE’, ‘SURFACE_NCURVE’, ‘SURFACE_NCIRCLE’, ‘SURFACE_NSURFACE’, ‘SURFACE_NCYLINDER’, ‘SURFACE_NSPHERE’, ‘SURFACE_NTORUS’, ‘CURVE_BEZCURVE’, ‘CURVE_BEZCIRCLE’, ‘CURVE_NCURVE’, ‘CURVE_NCIRCLE’, ‘CURVE_PATH’, ‘COLOR_RED’, ‘COLOR_GREEN’, ‘COLOR_BLUE’, ‘TRIA_RIGHT_BAR’, ‘TRIA_DOWN_BAR’, ‘TRIA_LEFT_BAR’, ‘TRIA_UP_BAR’, ‘FORCE_FORCE’, ‘FORCE_WIND’, ‘FORCE_VORTEX’, ‘FORCE_MAGNETIC’, ‘FORCE_HARMONIC’, ‘FORCE_CHARGE’, ‘FORCE_LENNARDJONES’, ‘FORCE_TEXTURE’, ‘FORCE_CURVE’, ‘FORCE_BOID’, ‘FORCE_TURBULENCE’, ‘FORCE_DRAG’, ‘FORCE_SMOKEFLOW’, ‘NODE_INSERT_ON’, ‘NODE_INSERT_OFF’, ‘MODIFIER’, ‘MOD_WAVE’, ‘MOD_BUILD’, ‘MOD_DECIM’, ‘MOD_MIRROR’, ‘MOD_SOFT’, ‘MOD_SUBSURF’, ‘HOOK’, ‘MOD_PHYSICS’, ‘MOD_PARTICLES’, ‘MOD_BOOLEAN’, ‘MOD_EDGESPLIT’, ‘MOD_ARRAY’, ‘MOD_UVPROJECT’, ‘MOD_DISPLACE’, ‘MOD_CURVE’, ‘MOD_LATTICE’, ‘CONSTRAINT_DATA’, ‘MOD_ARMATURE’, ‘MOD_SHRINKWRAP’, ‘MOD_CAST’, ‘MOD_MESHDEFORM’, ‘MOD_BEVEL’, ‘MOD_SMOOTH’, ‘MOD_SIMPLEDEFORM’, ‘MOD_MASK’, ‘MOD_CLOTH’, ‘MOD_EXPLODE’, ‘MOD_FLUIDSIM’, ‘MOD_MULTIRES’, ‘MOD_SMOKE’, ‘MOD_SOLIDIFY’, ‘MOD_SCREW’, ‘MOD_VERTEX_WEIGHT’, ‘MOD_DYNAMICPAINT’, ‘MOD_REMESH’, ‘MOD_OCEAN’, ‘MOD_WARP’, ‘MOD_SKIN’, ‘MOD_TRIANGULATE’, ‘MOD_WIREFRAME’, ‘MOD_DATA_TRANSFER’, ‘MOD_NORMALEDIT’, ‘REC’, ‘PLAY’, ‘FF’, ‘REW’, ‘PAUSE’, ‘PREV_KEYFRAME’, ‘NEXT_KEYFRAME’, ‘PLAY_AUDIO’, ‘PLAY_REVERSE’, ‘PREVIEW_RANGE’, ‘ACTION_TWEAK’, ‘PMARKER_ACT’, ‘PMARKER_SEL’, ‘PMARKER’, ‘MARKER_HLT’, ‘MARKER’, ‘SPACE2’, ‘SPACE3’, ‘KEYINGSET’, ‘KEY_DEHLT’, ‘KEY_HLT’, ‘MUTE_IPO_OFF’, ‘MUTE_IPO_ON’, ‘VISIBLE_IPO_OFF’, ‘VISIBLE_IPO_ON’, ‘DRIVER’, ‘SOLO_OFF’, ‘SOLO_ON’, ‘FRAME_PREV’, ‘FRAME_NEXT’, ‘NLA_PUSHDOWN’, ‘IPO_CONSTANT’, ‘IPO_LINEAR’, ‘IPO_BEZIER’, ‘IPO_SINE’, ‘IPO_QUAD’, ‘IPO_CUBIC’, ‘IPO_QUART’, ‘IPO_QUINT’, ‘IPO_EXPO’, ‘IPO_CIRC’, ‘IPO_BOUNCE’, ‘IPO_ELASTIC’, ‘IPO_BACK’, ‘IPO_EASE_IN’, ‘IPO_EASE_OUT’, ‘IPO_EASE_IN_OUT’, ‘VERTEXSEL’, ‘EDGESEL’, ‘FACESEL’, ‘LOOPSEL’, ‘ROTATE’, ‘CURSOR’, ‘ROTATECOLLECTION’, ‘ROTATECENTER’, ‘ROTACTIVE’, ‘ALIGN’, ‘SMOOTHCURVE’, ‘SPHERECURVE’, ‘ROOTCURVE’, ‘SHARPCURVE’, ‘LINCURVE’, ‘NOCURVE’, ‘RNDCURVE’, ‘PROP_OFF’, ‘PROP_ON’, ‘PROP_CON’, ‘SCULPT_DYNTOPO’, ‘PARTICLE_POINT’, ‘PARTICLE_TIP’, ‘PARTICLE_PATH’, ‘MAN_TRANS’, ‘MAN_ROT’, ‘MAN_SCALE’, ‘MANIPUL’, ‘SNAP_OFF’, ‘SNAP_ON’, ‘SNAP_NORMAL’, ‘SNAP_GRID’, ‘SNAP_VERTEX’, ‘SNAP_EDGE’, ‘SNAP_FACE’, ‘SNAP_VOLUME’, ‘SNAP_INCREMENT’, ‘STICKY_UVS_LOC’, ‘STICKY_UVS_DISABLE’, ‘STICKY_UVS_VERT’, ‘CLIPUV_DEHLT’, ‘CLIPUV_HLT’, ‘SNAP_PEEL_OBJECT’, ‘GRID’, ‘PASTEDOWN’, ‘COPYDOWN’, ‘PASTEFLIPUP’, ‘PASTEFLIPDOWN’, ‘SNAP_SURFACE’, ‘AUTOMERGE_ON’, ‘AUTOMERGE_OFF’, ‘RETOPO’, ‘UV_VERTEXSEL’, ‘UV_EDGESEL’, ‘UV_FACESEL’, ‘UV_ISLANDSEL’, ‘UV_SYNC_SELECT’, ‘BBOX’, ‘WIRE’, ‘SOLID’, ‘SMOOTH’, ‘POTATO’, ‘ORTHO’, ‘LOCKVIEW_OFF’, ‘LOCKVIEW_ON’, ‘AXIS_SIDE’, ‘AXIS_FRONT’, ‘AXIS_TOP’, ‘NDOF_DOM’, ‘NDOF_TURN’, ‘NDOF_FLY’, ‘NDOF_TRANS’, ‘LAYER_USED’, ‘LAYER_ACTIVE’, ‘SORTALPHA’, ‘SORTBYEXT’, ‘SORTTIME’, ‘SORTSIZE’, ‘LONGDISPLAY’, ‘SHORTDISPLAY’, ‘GHOST’, ‘IMGDISPLAY’, ‘SAVE_AS’, ‘SAVE_COPY’, ‘BOOKMARKS’, ‘FONTPREVIEW’, ‘FILTER’, ‘NEWFOLDER’, ‘OPEN_RECENT’, ‘FILE_PARENT’, ‘FILE_REFRESH’, ‘FILE_FOLDER’, ‘FILE_BLANK’, ‘FILE_BLEND’, ‘FILE_IMAGE’, ‘FILE_MOVIE’, ‘FILE_SCRIPT’, ‘FILE_SOUND’, ‘FILE_FONT’, ‘FILE_TEXT’, ‘RECOVER_AUTO’, ‘SAVE_PREFS’, ‘LINK_BLEND’, ‘APPEND_BLEND’, ‘IMPORT’, ‘EXPORT’, ‘EXTERNAL_DATA’, ‘LOAD_FACTORY’, ‘LOOP_BACK’, ‘LOOP_FORWARDS’, ‘BACK’, ‘FORWARD’, ‘FILE_HIDDEN’, ‘FILE_BACKUP’, ‘DISK_DRIVE’, ‘MATPLANE’, ‘MATSPHERE’, ‘MATCUBE’, ‘MONKEY’, ‘HAIR’, ‘ALIASED’, ‘ANTIALIASED’, ‘MAT_SPHERE_SKY’, ‘WORDWRAP_OFF’, ‘WORDWRAP_ON’, ‘SYNTAX_OFF’, ‘SYNTAX_ON’, ‘LINENUMBERS_OFF’, ‘LINENUMBERS_ON’, ‘SCRIPTPLUGINS’, ‘SEQ_SEQUENCER’, ‘SEQ_PREVIEW’, ‘SEQ_LUMA_WAVEFORM’, ‘SEQ_CHROMA_SCOPE’, ‘SEQ_HISTOGRAM’, ‘SEQ_SPLITVIEW’, ‘IMAGE_RGB’, ‘IMAGE_RGB_ALPHA’, ‘IMAGE_ALPHA’, ‘IMAGE_ZDEPTH’, ‘IMAGEFILE’, ‘BRUSH_ADD’, ‘BRUSH_BLOB’, ‘BRUSH_BLUR’, ‘BRUSH_CLAY’, ‘BRUSH_CLAY_STRIPS’, ‘BRUSH_CLONE’, ‘BRUSH_CREASE’, ‘BRUSH_DARKEN’, ‘BRUSH_FILL’, ‘BRUSH_FLATTEN’, ‘BRUSH_GRAB’, ‘BRUSH_INFLATE’, ‘BRUSH_LAYER’, ‘BRUSH_LIGHTEN’, ‘BRUSH_MASK’, ‘BRUSH_MIX’, ‘BRUSH_MULTIPLY’, ‘BRUSH_NUDGE’, ‘BRUSH_PINCH’, ‘BRUSH_SCRAPE’, ‘BRUSH_SCULPT_DRAW’, ‘BRUSH_SMEAR’, ‘BRUSH_SMOOTH’, ‘BRUSH_SNAKE_HOOK’, ‘BRUSH_SOFTEN’, ‘BRUSH_SUBTRACT’, ‘BRUSH_TEXDRAW’, ‘BRUSH_TEXFILL’, ‘BRUSH_TEXMASK’, ‘BRUSH_THUMB’, ‘BRUSH_ROTATE’, ‘BRUSH_VERTEXDRAW’, ‘MATCAP_01’, ‘MATCAP_02’, ‘MATCAP_03’, ‘MATCAP_04’, ‘MATCAP_05’, ‘MATCAP_06’, ‘MATCAP_07’, ‘MATCAP_08’, ‘MATCAP_09’, ‘MATCAP_10’, ‘MATCAP_11’, ‘MATCAP_12’, ‘MATCAP_13’, ‘MATCAP_14’, ‘MATCAP_15’, ‘MATCAP_16’, ‘MATCAP_17’, ‘MATCAP_18’, ‘MATCAP_19’, ‘MATCAP_20’, ‘MATCAP_21’, ‘MATCAP_22’, ‘MATCAP_23’, ‘MATCAP_24’], default ‘NODETREE’ 
    '''

    bl_idname = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_label = None
    '''The node tree label 

    :type:  string, default “”, (never None) 
    '''

    grease_pencil = None
    '''Grease Pencil data-block 

    :type:  GreasePencil 
    '''

    inputs = None
    '''Node tree inputs 

    :type:  NodeTreeInputs bpy_prop_collection of NodeSocketInterface, (readonly) 
    '''

    links = None
    '''

    :type:  NodeLinks bpy_prop_collection of NodeLink, (readonly) 
    '''

    nodes = None
    '''

    :type:  Nodes bpy_prop_collection of Node, (readonly) 
    '''

    outputs = None
    '''Node tree outputs 

    :type:  NodeTreeOutputs bpy_prop_collection of NodeSocketInterface, (readonly) 
    '''

    type = None
    '''Node Tree type (deprecated, bl_idname is the actual node tree type identifier) 

    :type:  enum in [‘SHADER’, ‘TEXTURE’, ‘COMPOSITING’], default ‘SHADER’, (readonly) 
    '''

    view_center = None
    '''

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0), (readonly) 
    '''

    def interface_update(self, context):
        '''Updated node group interface 

        '''
        pass

    def update(self):
        '''Update on editor changes 

        '''
        pass


class NodeTreeInputs:
    '''Collection of Node Tree Sockets '''

    def new(self, type, name):
        '''Add a socket to this node tree 

        :param type: Type, Data type 
        :type type: string, (never None)
        :param name: Name 
        :type name: string, (never None)
        :rtype: NodeSocketInterface 
        :return:  New socket 
        '''
        pass

    def remove(self, socket):
        '''Remove a socket from this node tree 

        :param socket: The socket to remove 
        :type socket: NodeSocketInterface
        '''
        pass

    def clear(self):
        '''Remove all sockets from this node tree 

        '''
        pass

    def move(self, from_index, to_index):
        '''Move a socket to another position 

        :param from_index: From Index, Index of the socket to move 
        :type from_index: int in [0, inf]
        :param to_index: To Index, Target index for the socket 
        :type to_index: int in [0, inf]
        '''
        pass


class NodeTreeOutputs:
    '''Collection of Node Tree Sockets '''

    def new(self, type, name):
        '''Add a socket to this node tree 

        :param type: Type, Data type 
        :type type: string, (never None)
        :param name: Name 
        :type name: string, (never None)
        :rtype: NodeSocketInterface 
        :return:  New socket 
        '''
        pass

    def remove(self, socket):
        '''Remove a socket from this node tree 

        :param socket: The socket to remove 
        :type socket: NodeSocketInterface
        '''
        pass

    def clear(self):
        '''Remove all sockets from this node tree 

        '''
        pass

    def move(self, from_index, to_index):
        '''Move a socket to another position 

        :param from_index: From Index, Index of the socket to move 
        :type from_index: int in [0, inf]
        :param to_index: To Index, Target index for the socket 
        :type to_index: int in [0, inf]
        '''
        pass


class NodeTreePath:
    '''Element of the node space tree path '''

    node_tree = None
    '''Base node tree from context 

    :type:  NodeTree, (readonly) 
    '''


class Nodes:
    '''Collection of Nodes '''

    active = None
    '''Active node in this tree 

    :type:  Node 
    '''

    def new(self, type):
        '''Add a node to this node tree 

        :param type: Type, Type of node to add (Warning: should be same as node.bl_idname, not node.type!) 
        :type type: string, (never None)
        :rtype: Node 
        :return:  New node 
        '''
        pass

    def remove(self, node):
        '''Remove a node from this node tree 

        :param node: The node to remove 
        :type node: Node, (never None)
        '''
        pass

    def clear(self):
        '''Remove all nodes from this node tree 

        '''
        pass


class NoiseTexture:
    '''Procedural noise texture '''

    users_material = None
    '''Materials that use this texture (readonly) '''

    users_object_modifier = None
    '''Object modifiers that use this texture (readonly) '''


class NorController:
    '''Controller passing on events based on a logical NOR operation '''

    pass


class NormalEditModifier:
    '''Modifier affecting/generating custom normals '''

    invert_vertex_group = None
    '''Invert vertex group influence 

    :type:  boolean, default False 
    '''

    mix_factor = None
    '''How much of generated normals to mix with exiting ones 

    :type:  float in [0, 1], default 1.0 
    '''

    mix_limit = None
    '''Maximum angle between old and new normals 

    :type:  float in [0, 3.14159], default 1.0 
    '''

    mix_mode = None
    '''How to mix generated normals with existing ones 

    :type:  enum in [‘COPY’, ‘ADD’, ‘SUB’, ‘MUL’], default ‘COPY’ 
    '''

    mode = None
    '''How to affect (generate) normals 

    :type:  enum in [‘RADIAL’, ‘DIRECTIONAL’], default ‘RADIAL’ 
    '''

    offset = None
    '''Offset from object’s center 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    target = None
    '''Target object used to affect normals 

    :type:  Object 
    '''

    use_direction_parallel = None
    '''Use same direction for all normals, from origin to target’s center (Directional mode only) 

    :type:  boolean, default True 
    '''

    vertex_group = None
    '''Vertex group name for selecting/weighting the affected areas 

    :type:  string, default “”, (never None) 
    '''


class Object:
    '''Object data-block defining an object in a scene '''

    active_material = None
    '''Active material being displayed 

    :type:  Material 
    '''

    active_material_index = None
    '''Index of active material slot 

    :type:  int in [0, inf], default 0 
    '''

    active_shape_key = None
    '''Current shape key 

    :type:  ShapeKey, (readonly) 
    '''

    active_shape_key_index = None
    '''Current shape key index 

    :type:  int in [-32768, 32767], default 0 
    '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    animation_visualization = None
    '''Animation data for this data-block 

    :type:  AnimViz, (readonly, never None) 
    '''

    bound_box = None
    '''Object’s bounding box in object-space coordinates, all values are -1.0 when not available 

    :type:  float array of 24 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0), (readonly) 
    '''

    collision = None
    '''Settings for using the object as a collider in physics simulation 

    :type:  CollisionSettings, (readonly) 
    '''

    color = None
    '''Object color and alpha, used when faces have the ObColor mode enabled 

    :type:  float array of 4 items in [0, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''

    constraints = None
    '''Constraints affecting the transformation of the object 

    :type:  ObjectConstraints bpy_prop_collection of Constraint, (readonly) 
    '''

    cycles = None
    '''Cycles object settings 

    :type:  CyclesObjectSettings, (readonly) 
    '''

    cycles_visibility = None
    '''Cycles visibility settings 

    :type:  CyclesVisibilitySettings, (readonly) 
    '''

    data = None
    '''Object data 

    :type:  ID 
    '''

    delta_location = None
    '''Extra translation added to the location of the object 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    delta_rotation_euler = None
    '''Extra rotation added to the rotation of the object (when using Euler rotations) 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    delta_rotation_quaternion = None
    '''Extra rotation added to the rotation of the object (when using Quaternion rotations) 

    :type:  float array of 4 items in [-inf, inf], default (1.0, 0.0, 0.0, 0.0) 
    '''

    delta_scale = None
    '''Extra scaling added to the scale of the object 

    :type:  float array of 3 items in [-inf, inf], default (1.0, 1.0, 1.0) 
    '''

    dimensions = None
    '''Absolute bounding box dimensions of the object 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    draw_bounds_type = None
    '''Object boundary display type 

    :type:  enum in [‘BOX’, ‘SPHERE’, ‘CYLINDER’, ‘CONE’, ‘CAPSULE’], default ‘BOX’ 
    '''

    draw_type = None
    '''Maximum draw type to display object with in viewport 

    :type:  enum in [‘BOUNDS’, ‘WIRE’, ‘SOLID’, ‘TEXTURED’], default ‘BOUNDS’ 
    '''

    dupli_faces_scale = None
    '''Scale the DupliFace objects 

    :type:  float in [0.001, 10000], default 0.0 
    '''

    dupli_frames_end = None
    '''End frame for DupliFrames 

    :type:  int in [-500000, 500000], default 0 
    '''

    dupli_frames_off = None
    '''Recurring frames to exclude from the Dupliframes 

    :type:  int in [0, 500000], default 0 
    '''

    dupli_frames_on = None
    '''Number of frames to use between DupOff frames 

    :type:  int in [0, 500000], default 0 
    '''

    dupli_frames_start = None
    '''Start frame for DupliFrames 

    :type:  int in [-500000, 500000], default 0 
    '''

    dupli_group = None
    '''Instance an existing group 

    :type:  Group 
    '''

    dupli_list = None
    '''Object duplis 

    :type:  bpy_prop_collection of DupliObject, (readonly) 
    '''

    dupli_type = None
    '''If not None, object duplication method to use 

    :type:  enum in [‘NONE’, ‘FRAMES’, ‘VERTS’, ‘FACES’, ‘GROUP’], default ‘NONE’ 
    '''

    empty_draw_size = None
    '''Size of display for empties in the viewport 

    :type:  float in [0.0001, 1000], default 0.0 
    '''

    empty_draw_type = None
    '''Viewport display style for empties 

    :type:  enum in [‘PLAIN_AXES’, ‘ARROWS’, ‘SINGLE_ARROW’, ‘CIRCLE’, ‘CUBE’, ‘SPHERE’, ‘CONE’, ‘IMAGE’], default ‘PLAIN_AXES’ 
    '''

    empty_image_offset = None
    '''Origin offset distance 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    field = None
    '''Settings for using the object as a field in physics simulation 

    :type:  FieldSettings, (readonly) 
    '''

    game = None
    '''Game engine related settings for the object 

    :type:  GameObjectSettings, (readonly, never None) 
    '''

    grease_pencil = None
    '''Grease Pencil data-block 

    :type:  GreasePencil 
    '''

    hide = None
    '''Restrict visibility in the viewport 

    :type:  boolean, default False 
    '''

    hide_render = None
    '''Restrict renderability 

    :type:  boolean, default False 
    '''

    hide_select = None
    '''Restrict selection in the viewport 

    :type:  boolean, default False 
    '''

    image_user = None
    '''Parameters defining which layer, pass and frame of the image is displayed 

    :type:  ImageUser, (readonly, never None) 
    '''

    is_duplicator = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    layers = None
    '''Layers the object is on 

    :type:  boolean array of 20 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    layers_local_view = None
    '''3D local view layers the object is on 

    :type:  boolean array of 8 items, default (False, False, False, False, False, False, False, False), (readonly) 
    '''

    location = None
    '''Location of the object 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    lock_location = None
    '''Lock editing of location in the interface 

    :type:  boolean array of 3 items, default (False, False, False) 
    '''

    lock_rotation = None
    '''Lock editing of rotation in the interface 

    :type:  boolean array of 3 items, default (False, False, False) 
    '''

    lock_rotation_w = None
    '''Lock editing of ‘angle’ component of four-component rotations in the interface 

    :type:  boolean, default False 
    '''

    lock_rotations_4d = None
    '''Lock editing of four component rotations by components (instead of as Eulers) 

    :type:  boolean, default False 
    '''

    lock_scale = None
    '''Lock editing of scale in the interface 

    :type:  boolean array of 3 items, default (False, False, False) 
    '''

    lod_levels = None
    '''A collection of detail levels to automatically switch between 

    :type:  bpy_prop_collection of LodLevel, (readonly) 
    '''

    material_slots = None
    '''Material slots in the object 

    :type:  bpy_prop_collection of MaterialSlot, (readonly) 
    '''

    matrix_basis = None
    '''Matrix access to location, rotation and scale (including deltas), before constraints and parenting are applied 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    matrix_local = None
    '''Parent relative transformation matrix - WARNING: Only takes into account ‘Object’ parenting, so e.g. in case of bone parenting you get a matrix relative to the Armature object, not to the actual parent bone 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    matrix_parent_inverse = None
    '''Inverse of object’s parent matrix at time of parenting 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    matrix_world = None
    '''Worldspace transformation matrix 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    mode = None
    '''Object interaction mode 

    :type:  enum in [‘OBJECT’, ‘EDIT’, ‘POSE’, ‘SCULPT’, ‘VERTEX_PAINT’, ‘WEIGHT_PAINT’, ‘TEXTURE_PAINT’, ‘PARTICLE_EDIT’, ‘GPENCIL_EDIT’], default ‘OBJECT’, (readonly) 
    '''

    modifiers = None
    '''Modifiers affecting the geometric data of the object 

    :type:  ObjectModifiers bpy_prop_collection of Modifier, (readonly) 
    '''

    motion_path = None
    '''Motion Path for this element 

    :type:  MotionPath, (readonly) 
    '''

    parent = None
    '''Parent Object 

    :type:  Object 
    '''

    parent_bone = None
    '''Name of parent bone in case of a bone parenting relation 

    :type:  string, default “”, (never None) 
    '''

    parent_type = None
    '''Type of parent relation 

    :type:  enum in [‘OBJECT’, ‘ARMATURE’, ‘LATTICE’, ‘VERTEX’, ‘VERTEX_3’, ‘BONE’], default ‘OBJECT’ 
    '''

    parent_vertices = None
    '''Indices of vertices in case of a vertex parenting relation 

    :type:  int array of 3 items in [0, inf], default (0, 0, 0) 
    '''

    particle_systems = None
    '''Particle systems emitted from the object 

    :type:  ParticleSystems bpy_prop_collection of ParticleSystem, (readonly) 
    '''

    pass_index = None
    '''Index number for the “Object Index” render pass 

    :type:  int in [0, 32767], default 0 
    '''

    pose = None
    '''Current pose for armatures 

    :type:  Pose, (readonly) 
    '''

    pose_library = None
    '''Action used as a pose library for armatures 

    :type:  Action 
    '''

    proxy = None
    '''Library object this proxy object controls 

    :type:  Object, (readonly) 
    '''

    proxy_group = None
    '''Library group duplicator object this proxy object controls 

    :type:  Object, (readonly) 
    '''

    rigid_body = None
    '''Settings for rigid body simulation 

    :type:  RigidBodyObject, (readonly) 
    '''

    rigid_body_constraint = None
    '''Constraint constraining rigid bodies 

    :type:  RigidBodyConstraint, (readonly) 
    '''

    rotation_axis_angle = None
    '''Angle of Rotation for Axis-Angle rotation representation 

    :type:  float array of 4 items in [-inf, inf], default (0.0, 0.0, 1.0, 0.0) 
    '''

    rotation_euler = None
    '''Rotation in Eulers 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    rotation_mode = None
    '''

    :type:  enum in [‘QUATERNION’, ‘XYZ’, ‘XZY’, ‘YXZ’, ‘YZX’, ‘ZXY’, ‘ZYX’, ‘AXIS_ANGLE’], default ‘QUATERNION’ 
    '''

    rotation_quaternion = None
    '''Rotation in Quaternions 

    :type:  float array of 4 items in [-inf, inf], default (1.0, 0.0, 0.0, 0.0) 
    '''

    scale = None
    '''Scaling of the object 

    :type:  float array of 3 items in [-inf, inf], default (1.0, 1.0, 1.0) 
    '''

    select = None
    '''Object selection state 

    :type:  boolean, default False 
    '''

    show_all_edges = None
    '''Display all edges for mesh objects 

    :type:  boolean, default False 
    '''

    show_axis = None
    '''Display the object’s origin and axes 

    :type:  boolean, default False 
    '''

    show_bounds = None
    '''Display the object’s bounds 

    :type:  boolean, default False 
    '''

    show_name = None
    '''Display the object’s name 

    :type:  boolean, default False 
    '''

    show_only_shape_key = None
    '''Always show the current Shape for this Object 

    :type:  boolean, default False 
    '''

    show_texture_space = None
    '''Display the object’s texture space 

    :type:  boolean, default False 
    '''

    show_transparent = None
    '''Display material transparency in the object (unsupported for duplicator drawing) 

    :type:  boolean, default False 
    '''

    show_wire = None
    '''Add the object’s wireframe over solid drawing 

    :type:  boolean, default False 
    '''

    show_x_ray = None
    '''Make the object draw in front of others (unsupported for duplicator drawing) 

    :type:  boolean, default False 
    '''

    slow_parent_offset = None
    '''Delay in the parent relationship 

    :type:  float in [-500000, 500000], default 0.0 
    '''

    soft_body = None
    '''Settings for soft body simulation 

    :type:  SoftBodySettings, (readonly) 
    '''

    track_axis = None
    '''Axis that points in ‘forward’ direction (applies to DupliFrame when parent ‘Follow’ is enabled) 

    :type:  enum in [‘POS_X’, ‘POS_Y’, ‘POS_Z’, ‘NEG_X’, ‘NEG_Y’, ‘NEG_Z’], default ‘POS_X’ 
    '''

    type = None
    '''Type of Object 

    :type:  enum in [‘MESH’, ‘CURVE’, ‘SURFACE’, ‘META’, ‘FONT’, ‘ARMATURE’, ‘LATTICE’, ‘EMPTY’, ‘CAMERA’, ‘LAMP’, ‘SPEAKER’], default ‘EMPTY’, (readonly) 
    '''

    up_axis = None
    '''Axis that points in the upward direction (applies to DupliFrame when parent ‘Follow’ is enabled) 

    :type:  enum in [‘X’, ‘Y’, ‘Z’], default ‘X’ 
    '''

    use_dupli_faces_scale = None
    '''Scale dupli based on face size 

    :type:  boolean, default False 
    '''

    use_dupli_frames_speed = None
    '''Set dupliframes to use the current frame instead of parent curve’s evaluation time 

    :type:  boolean, default False 
    '''

    use_dupli_vertices_rotation = None
    '''Rotate dupli according to vertex normal 

    :type:  boolean, default False 
    '''

    use_dynamic_topology_sculpting = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    use_extra_recalc_data = None
    '''Refresh this object’s data again on frame changes, dependency graph hack 

    :type:  boolean, default False 
    '''

    use_extra_recalc_object = None
    '''Refresh this object again on frame changes, dependency graph hack 

    :type:  boolean, default False 
    '''

    use_shape_key_edit_mode = None
    '''Apply shape keys in edit mode (for Meshes only) 

    :type:  boolean, default False 
    '''

    use_slow_parent = None
    '''Create a delay in the parent relationship (beware: this isn’t renderfarm safe and may be invalid after jumping around the timeline) 

    :type:  boolean, default False 
    '''

    vertex_groups = None
    '''Vertex groups of the object 

    :type:  VertexGroups bpy_prop_collection of VertexGroup, (readonly) 
    '''

    children = None
    '''All the children of this object (readonly) '''

    users_group = None
    '''The groups this object is in (readonly) '''

    users_scene = None
    '''The scenes this object is in (readonly) '''

    def convert_space(self,
                      pose_bone=None,
                      matrix=(0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                              0.0, 0.0, 0.0, 0.0, 0.0, 0.0),
                      from_space='WORLD',
                      to_space='WORLD'):
        '''Convert (transform) the given matrix from one space to another 

        :param pose_bone: Bone to use to define spaces (may be None, in which case only the two ‘WORLD’ and ‘LOCAL’ spaces are usable) 
        :type pose_bone: PoseBone, (optional)
        :param matrix: The matrix to transform 
        :type matrix: float array of 16 items in [-inf, inf], (optional)
        :param from_space: The space in which ‘matrix’ is currentlyWORLD World Space, The most gobal space in Blender.POSE Pose Space, The pose space of a bone (its armature’s object space).LOCAL_WITH_PARENT Local With Parent, The local space of a bone’s parent bone.LOCAL Local Space, The local space of an object/bone. 
        :type from_space: enum in ['WORLD', 'POSE', 'LOCAL_WITH_PARENT', 'LOCAL'], (optional)
        :param to_space: The space to which you want to transform ‘matrix’WORLD World Space, The most gobal space in Blender.POSE Pose Space, The pose space of a bone (its armature’s object space).LOCAL_WITH_PARENT Local With Parent, The local space of a bone’s parent bone.LOCAL Local Space, The local space of an object/bone. 
        :type to_space: enum in ['WORLD', 'POSE', 'LOCAL_WITH_PARENT', 'LOCAL'], (optional)
        :rtype: float array of 16 items in [-inf, inf] 
        :return:  The transformed matrix 
        '''
        pass

    def calc_matrix_camera(self, x=1, y=1, scale_x=1.0, scale_y=1.0):
        '''Generate the camera projection matrix of this object (mostly useful for Camera and Lamp types) 

        :param x: Width of the render area 
        :type x: int in [0, inf], (optional)
        :param y: Height of the render area 
        :type y: int in [0, inf], (optional)
        :param scale_x: Width scaling factor 
        :type scale_x: float in [1e-06, inf], (optional)
        :param scale_y: height scaling factor 
        :type scale_y: float in [1e-06, inf], (optional)
        :rtype: float array of 16 items in [-inf, inf] 
        :return:  The camera projection matrix 
        '''
        pass

    def camera_fit_coords(self, scene, coordinates):
        '''Compute the coordinate (and scale for ortho cameras) given object should be to ‘see’ all given coordinates 

        :param scene: Scene to get render size information from, if available 
        :type scene: Scene
        :param coordinates: Coordinates to fit in 
        :type coordinates: float array of 1 items in [-inf, inf], (never None)
        '''
        pass

    def to_mesh(self,
                scene,
                apply_modifiers,
                settings,
                calc_tessface=True,
                calc_undeformed=False):
        '''Create a Mesh data-block with modifiers applied 

        :param scene: Scene within which to evaluate modifiers 
        :type scene: Scene, (never None)
        :param apply_modifiers: Apply modifiers 
        :type apply_modifiers: boolean
        :param settings: Modifier settings to applyPREVIEW Preview, Apply modifier preview settings.RENDER Render, Apply modifier render settings. 
        :type settings: enum in ['PREVIEW', 'RENDER']
        :param calc_tessface: Calculate Tessellation, Calculate tessellation faces 
        :type calc_tessface: boolean, (optional)
        :param calc_undeformed: Calculate Undeformed, Calculate undeformed vertex coordinates 
        :type calc_undeformed: boolean, (optional)
        :rtype: Mesh 
        :return:  Mesh created from object, remove it if it is only used for export 
        '''
        pass

    def dupli_list_create(self, scene, settings='VIEWPORT'):
        '''Create a list of dupli objects for this object, needs to be freed manually with free_dupli_list to restore the objects real matrix and layers 

        :param scene: Scene within which to evaluate duplis 
        :type scene: Scene, (never None)
        :param settings: Generate texture coordinates for renderingVIEWPORT Viewport, Generate duplis using viewport settings.PREVIEW Preview, Generate duplis using preview settings.RENDER Render, Generate duplis using render settings. 
        :type settings: enum in ['VIEWPORT', 'PREVIEW', 'RENDER'], (optional)
        '''
        pass

    def dupli_list_clear(self):
        '''Free the list of dupli objects 

        '''
        pass

    def find_armature(self):
        '''Find armature influencing this object as a parent or via a modifier 

        :rtype: Object 
        :return:  Armature object influencing this object or NULL 
        '''
        pass

    def shape_key_add(self, name="Key", from_mix=True):
        '''Add shape key to this object 

        :param name: Unique name for the new keyblock 
        :type name: string, (optional, never None)
        :param from_mix: Create new shape from existing mix of shapes 
        :type from_mix: boolean, (optional)
        :rtype: ShapeKey 
        :return:  New shape keyblock 
        '''
        pass

    def shape_key_remove(self, key):
        '''Remove a Shape Key from this object 

        :param key: Keyblock to be removed 
        :type key: ShapeKey, (never None)
        '''
        pass

    def ray_cast(self, origin, direction, distance=1.70141e+38):
        '''Cast a ray onto in object space 

        :param distance: Maximum distance 
        :type distance: float in [0, inf], (optional)
        '''
        pass

    def closest_point_on_mesh(self, origin, distance=1.84467e+19):
        '''Find the nearest point in object space 

        :param distance: Maximum distance 
        :type distance: float in [0, inf], (optional)
        '''
        pass

    def is_visible(self, scene):
        '''Determine if object is visible in a given scene 

        :rtype: boolean 
        :return:  Object visibility 
        '''
        pass

    def is_modified(self, scene, settings):
        '''Determine if this object is modified from the base mesh data 

        :param settings: Modifier settings to applyPREVIEW Preview, Apply modifier preview settings.RENDER Render, Apply modifier render settings. 
        :type settings: enum in ['PREVIEW', 'RENDER']
        :rtype: boolean 
        :return:  Object visibility 
        '''
        pass

    def is_deform_modified(self, scene, settings):
        '''Determine if this object is modified by a deformation from the base mesh data 

        :param settings: Modifier settings to applyPREVIEW Preview, Apply modifier preview settings.RENDER Render, Apply modifier render settings. 
        :type settings: enum in ['PREVIEW', 'RENDER']
        :rtype: boolean 
        :return:  Object visibility 
        '''
        pass

    def update_from_editmode(self):
        '''Load the objects edit-mode data into the object data 

        :rtype: boolean 
        :return:  Success 
        '''
        pass

    def cache_release(self):
        '''Release memory used by caches associated with this object. Intended to be used by render engines only 

        '''
        pass


class ObjectActuator:
    '''Actuator to control the object movement '''

    angular_velocity = None
    '''Angular velocity 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    damping = None
    '''Number of frames to reach the target velocity 

    :type:  int in [-32768, 32767], default 0 
    '''

    derivate_coefficient = None
    '''Not required, high values can cause instability 

    :type:  float in [-inf, inf], default 0.0 
    '''

    force = None
    '''Force 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    force_max_x = None
    '''Upper limit for X force 

    :type:  float in [-inf, inf], default 0.0 
    '''

    force_max_y = None
    '''Upper limit for Y force 

    :type:  float in [-inf, inf], default 0.0 
    '''

    force_max_z = None
    '''Upper limit for Z force 

    :type:  float in [-inf, inf], default 0.0 
    '''

    force_min_x = None
    '''Lower limit for X force 

    :type:  float in [-inf, inf], default 0.0 
    '''

    force_min_y = None
    '''Lower limit for Y force 

    :type:  float in [-inf, inf], default 0.0 
    '''

    force_min_z = None
    '''Lower limit for Z force 

    :type:  float in [-inf, inf], default 0.0 
    '''

    integral_coefficient = None
    '''Low value (0.01) for slow response, high value (0.5) for fast response 

    :type:  float in [-inf, inf], default 0.0 
    '''

    linear_velocity = None
    '''Linear velocity (in Servo mode it sets the target relative linear velocity, it will be achieved by automatic application of force - Null velocity is a valid target) 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    mode = None
    '''Specify the motion system 

    :type:  enum in [‘OBJECT_NORMAL’, ‘OBJECT_SERVO’, ‘OBJECT_CHARACTER’], default ‘OBJECT_NORMAL’ 
    '''

    offset_location = None
    '''Location 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    offset_rotation = None
    '''Rotation 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    proportional_coefficient = None
    '''Typical value is 60x integral coefficient 

    :type:  float in [-inf, inf], default 0.0 
    '''

    reference_object = None
    '''Reference object for velocity calculation, leave empty for world reference 

    :type:  Object 
    '''

    torque = None
    '''Torque 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    use_add_character_location = None
    '''Toggle between ADD and SET character location 

    :type:  boolean, default False 
    '''

    use_add_linear_velocity = None
    '''Toggles between ADD and SET linV 

    :type:  boolean, default False 
    '''

    use_character_jump = None
    '''Make the character jump using the settings in the physics properties 

    :type:  boolean, default False 
    '''

    use_local_angular_velocity = None
    '''Angular velocity is defined in local coordinates 

    :type:  boolean, default False 
    '''

    use_local_force = None
    '''Force is defined in local coordinates 

    :type:  boolean, default False 
    '''

    use_local_linear_velocity = None
    '''Velocity is defined in local coordinates 

    :type:  boolean, default False 
    '''

    use_local_location = None
    '''Location is defined in local coordinates 

    :type:  boolean, default False 
    '''

    use_local_rotation = None
    '''Rotation is defined in local coordinates 

    :type:  boolean, default False 
    '''

    use_local_torque = None
    '''Torque is defined in local coordinates 

    :type:  boolean, default False 
    '''

    use_servo_limit_x = None
    '''Set limit to force along the X axis 

    :type:  boolean, default False 
    '''

    use_servo_limit_y = None
    '''Set limit to force along the Y axis 

    :type:  boolean, default False 
    '''

    use_servo_limit_z = None
    '''Set limit to force along the Z axis 

    :type:  boolean, default False 
    '''


class ObjectBase:
    '''An object instance in a scene '''

    layers = None
    '''Layers the object base is on 

    :type:  boolean array of 20 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    layers_local_view = None
    '''3D local view layers the object base is on 

    :type:  boolean array of 8 items, default (False, False, False, False, False, False, False, False), (readonly) 
    '''

    object = None
    '''Object this base links to 

    :type:  Object, (readonly) 
    '''

    select = None
    '''Object base selection state 

    :type:  boolean, default False 
    '''

    def layers_from_view(self, view):
        '''Sets the object layers from a 3D View (use when adding an object in local view) 

        '''
        pass


class ObjectConstraints:
    '''Collection of object constraints '''

    active = None
    '''Active Object constraint 

    :type:  Constraint 
    '''

    def new(self, type):
        '''Add a new constraint to this object 

        :param type: Constraint type to addCAMERA_SOLVER Camera Solver.FOLLOW_TRACK Follow Track.OBJECT_SOLVER Object Solver.COPY_LOCATION Copy Location, Copy the location of a target (with an optional offset), so that they move together.COPY_ROTATION Copy Rotation, Copy the rotation of a target (with an optional offset), so that they rotate together.COPY_SCALE Copy Scale, Copy the scale factors of a target (with an optional offset), so that they are scaled by the same amount.COPY_TRANSFORMS Copy Transforms, Copy all the transformations of a target, so that they move together.LIMIT_DISTANCE Limit Distance, Restrict movements to within a certain distance of a target (at the time of constraint evaluation only).LIMIT_LOCATION Limit Location, Restrict movement along each axis within given ranges.LIMIT_ROTATION Limit Rotation, Restrict rotation along each axis within given ranges.LIMIT_SCALE Limit Scale, Restrict scaling along each axis with given ranges.MAINTAIN_VOLUME Maintain Volume, Compensate for scaling one axis by applying suitable scaling to the other two axes.TRANSFORM Transformation, Use one transform property from target to control another (or same) property on owner.TRANSFORM_CACHE Transform Cache, Look up the transformation matrix from an external file.CLAMP_TO Clamp To, Restrict movements to lie along a curve by remapping location along curve’s longest axis.DAMPED_TRACK Damped Track, Point towards a target by performing the smallest rotation necessary.IK Inverse Kinematics, Control a chain of bones by specifying the endpoint target (Bones only).LOCKED_TRACK Locked Track, Rotate around the specified (‘locked’) axis to point towards a target.SPLINE_IK Spline IK, Align chain of bones along a curve (Bones only).STRETCH_TO Stretch To, Stretch along Y-Axis to point towards a target.TRACK_TO Track To, Legacy tracking constraint prone to twisting artifacts.ACTION Action, Use transform property of target to look up pose for owner from an Action.CHILD_OF Child Of, Make target the ‘detachable’ parent of owner.FLOOR Floor, Use position (and optionally rotation) of target to define a ‘wall’ or ‘floor’ that the owner can not cross.FOLLOW_PATH Follow Path, Use to animate an object/bone following a path.PIVOT Pivot, Change pivot point for transforms (buggy).RIGID_BODY_JOINT Rigid Body Joint, Use to define a Rigid Body Constraint (for Game Engine use only).SHRINKWRAP Shrinkwrap, Restrict movements to surface of target mesh. 
        :type type: enum in ['CAMERA_SOLVER', 'FOLLOW_TRACK', 'OBJECT_SOLVER', 'COPY_LOCATION', 'COPY_ROTATION', 'COPY_SCALE', 'COPY_TRANSFORMS', 'LIMIT_DISTANCE', 'LIMIT_LOCATION', 'LIMIT_ROTATION', 'LIMIT_SCALE', 'MAINTAIN_VOLUME', 'TRANSFORM', 'TRANSFORM_CACHE', 'CLAMP_TO', 'DAMPED_TRACK', 'IK', 'LOCKED_TRACK', 'SPLINE_IK', 'STRETCH_TO', 'TRACK_TO', 'ACTION', 'CHILD_OF', 'FLOOR', 'FOLLOW_PATH', 'PIVOT', 'RIGID_BODY_JOINT', 'SHRINKWRAP']
        :rtype: Constraint 
        :return:  New constraint 
        '''
        pass

    def remove(self, constraint):
        '''Remove a constraint from this object 

        :param constraint: Removed constraint 
        :type constraint: Constraint, (never None)
        '''
        pass

    def clear(self):
        '''Remove all constraint from this object 

        '''
        pass


class ObjectModifiers:
    '''Collection of object modifiers '''

    def new(self, name, type):
        '''Add a new modifier 

        :param name: New name for the modifier 
        :type name: string, (never None)
        :param type: Modifier type to addDATA_TRANSFER Data Transfer.MESH_CACHE Mesh Cache.MESH_SEQUENCE_CACHE Mesh Sequence Cache.NORMAL_EDIT Normal Edit.UV_PROJECT UV Project.UV_WARP UV Warp.VERTEX_WEIGHT_EDIT Vertex Weight Edit.VERTEX_WEIGHT_MIX Vertex Weight Mix.VERTEX_WEIGHT_PROXIMITY Vertex Weight Proximity.ARRAY Array.BEVEL Bevel.BOOLEAN Boolean.BUILD Build.DECIMATE Decimate.EDGE_SPLIT Edge Split.MASK Mask.MIRROR Mirror.MULTIRES Multiresolution.REMESH Remesh.SCREW Screw.SKIN Skin.SOLIDIFY Solidify.SUBSURF Subdivision Surface.TRIANGULATE Triangulate.WIREFRAME Wireframe, Generate a wireframe on the edges of a mesh.ARMATURE Armature.CAST Cast.CORRECTIVE_SMOOTH Corrective Smooth.CURVE Curve.DISPLACE Displace.HOOK Hook.LAPLACIANSMOOTH Laplacian Smooth.LAPLACIANDEFORM Laplacian Deform.LATTICE Lattice.MESH_DEFORM Mesh Deform.SHRINKWRAP Shrinkwrap.SIMPLE_DEFORM Simple Deform.SMOOTH Smooth.WARP Warp.WAVE Wave.CLOTH Cloth.COLLISION Collision.DYNAMIC_PAINT Dynamic Paint.EXPLODE Explode.FLUID_SIMULATION Fluid Simulation.OCEAN Ocean.PARTICLE_INSTANCE Particle Instance.PARTICLE_SYSTEM Particle System.SMOKE Smoke.SOFT_BODY Soft Body.SURFACE Surface. 
        :type type: enum in ['DATA_TRANSFER', 'MESH_CACHE', 'MESH_SEQUENCE_CACHE', 'NORMAL_EDIT', 'UV_PROJECT', 'UV_WARP', 'VERTEX_WEIGHT_EDIT', 'VERTEX_WEIGHT_MIX', 'VERTEX_WEIGHT_PROXIMITY', 'ARRAY', 'BEVEL', 'BOOLEAN', 'BUILD', 'DECIMATE', 'EDGE_SPLIT', 'MASK', 'MIRROR', 'MULTIRES', 'REMESH', 'SCREW', 'SKIN', 'SOLIDIFY', 'SUBSURF', 'TRIANGULATE', 'WIREFRAME', 'ARMATURE', 'CAST', 'CORRECTIVE_SMOOTH', 'CURVE', 'DISPLACE', 'HOOK', 'LAPLACIANSMOOTH', 'LAPLACIANDEFORM', 'LATTICE', 'MESH_DEFORM', 'SHRINKWRAP', 'SIMPLE_DEFORM', 'SMOOTH', 'WARP', 'WAVE', 'CLOTH', 'COLLISION', 'DYNAMIC_PAINT', 'EXPLODE', 'FLUID_SIMULATION', 'OCEAN', 'PARTICLE_INSTANCE', 'PARTICLE_SYSTEM', 'SMOKE', 'SOFT_BODY', 'SURFACE']
        :rtype: Modifier 
        :return:  Newly created modifier 
        '''
        pass

    def remove(self, modifier):
        '''Remove an existing modifier from the object 

        :param modifier: Modifier to remove 
        :type modifier: Modifier, (never None)
        '''
        pass

    def clear(self):
        '''Remove all modifiers from the object 

        '''
        pass


class ObjectSolverConstraint:
    '''Lock motion to the reconstructed object movement '''

    camera = None
    '''Camera to which motion is parented (if empty active scene camera is used) 

    :type:  Object 
    '''

    clip = None
    '''Movie Clip to get tracking data from 

    :type:  MovieClip 
    '''

    object = None
    '''Movie tracking object to follow 

    :type:  string, default “”, (never None) 
    '''

    use_active_clip = None
    '''Use active clip defined in scene 

    :type:  boolean, default False 
    '''


class ObstacleFluidSettings:
    '''Fluid simulation settings for obstacles in the simulation '''

    impact_factor = None
    '''This is an unphysical value for moving objects - it controls the impact an obstacle has on the fluid, =0 behaves a bit like outflow (deleting fluid), =1 is default, while >1 results in high forces (can be used to tweak total mass) 

    :type:  float in [-2, 10], default 0.0 
    '''

    partial_slip_factor = None
    '''Amount of mixing between no- and free-slip, 0 is no slip and 1 is free slip 

    :type:  float in [0, 1], default 0.0 
    '''

    slip_type = None
    '''

    :type:  enum in [‘NOSLIP’, ‘PARTIALSLIP’, ‘FREESLIP’], default ‘NOSLIP’ 
    '''

    use = None
    '''Object contributes to the fluid simulation 

    :type:  boolean, default False 
    '''

    use_animated_mesh = None
    '''Export this mesh as an animated one (slower and enforces No Slip, only use if really necessary [e.g. armatures or parented objects], animated pos/rot/scale F-Curves do not require it) 

    :type:  boolean, default False 
    '''

    volume_initialization = None
    '''Volume initialization type (WARNING: complex volumes might require too much memory and break simulation) 

    :type:  enum in [‘VOLUME’, ‘SHELL’, ‘BOTH’], default ‘VOLUME’ 
    '''


class OceanModifier:
    '''Simulate an ocean surface '''

    bake_foam_fade = None
    '''How much foam accumulates over time (baked ocean only) 

    :type:  float in [0, inf], default 0.0 
    '''

    choppiness = None
    '''Choppiness of the wave’s crest (adds some horizontal component to the displacement) 

    :type:  float in [0, inf], default 0.0 
    '''

    damping = None
    '''Damp reflected waves going in opposite direction to the wind 

    :type:  float in [0, 1], default 0.0 
    '''

    depth = None
    '''Depth of the solid ground below the water surface 

    :type:  float in [-inf, inf], default 0.0 
    '''

    filepath = None
    '''Path to a folder to store external baked images 

    :type:  string, default “”, (never None) 
    '''

    foam_coverage = None
    '''Amount of generated foam 

    :type:  float in [-inf, inf], default 0.0 
    '''

    foam_layer_name = None
    '''Name of the vertex color layer used for foam 

    :type:  string, default “”, (never None) 
    '''

    frame_end = None
    '''End frame of the ocean baking 

    :type:  int in [0, inf], default 0 
    '''

    frame_start = None
    '''Start frame of the ocean baking 

    :type:  int in [0, inf], default 0 
    '''

    geometry_mode = None
    '''Method of modifying geometry 

    :type:  enum in [‘GENERATE’, ‘DISPLACE’], default ‘GENERATE’ 
    '''

    is_cached = None
    '''Whether the ocean is using cached data or simulating 

    :type:  boolean, default False, (readonly) 
    '''

    random_seed = None
    '''Seed of the random generator 

    :type:  int in [0, inf], default 0 
    '''

    repeat_x = None
    '''Repetitions of the generated surface in X 

    :type:  int in [1, 1024], default 0 
    '''

    repeat_y = None
    '''Repetitions of the generated surface in Y 

    :type:  int in [1, 1024], default 0 
    '''

    resolution = None
    '''Resolution of the generated surface 

    :type:  int in [1, 1024], default 0 
    '''

    size = None
    '''Surface scale factor (does not affect the height of the waves) 

    :type:  float in [0, inf], default 0.0 
    '''

    spatial_size = None
    '''Size of the simulation domain (in meters), and of the generated geometry (in BU) 

    :type:  int in [-inf, inf], default 0 
    '''

    time = None
    '''Current time of the simulation 

    :type:  float in [0, inf], default 0.0 
    '''

    use_foam = None
    '''Generate foam mask as a vertex color channel 

    :type:  boolean, default False 
    '''

    use_normals = None
    '''Output normals for bump mapping - disabling can speed up performance if its not needed 

    :type:  boolean, default False 
    '''

    wave_alignment = None
    '''How much the waves are aligned to each other 

    :type:  float in [0, 10], default 0.0 
    '''

    wave_direction = None
    '''Main direction of the waves when they are (partially) aligned 

    :type:  float in [-inf, inf], default 0.0 
    '''

    wave_scale = None
    '''Scale of the displacement effect 

    :type:  float in [0, inf], default 0.0 
    '''

    wave_scale_min = None
    '''Shortest allowed wavelength 

    :type:  float in [0, inf], default 0.0 
    '''

    wind_velocity = None
    '''Wind speed 

    :type:  float in [-inf, inf], default 0.0 
    '''


class OceanTexData:
    '''Ocean Texture settings '''

    ocean_object = None
    '''Object containing the ocean modifier 

    :type:  Object 
    '''

    output = None
    '''The data that is output by the texture 

    :type:  enum in [‘DISPLACEMENT’, ‘FOAM’, ‘JPLUS’, ‘EMINUS’, ‘EPLUS’], default ‘DISPLACEMENT’ 
    '''


class OceanTexture:
    '''Settings for the Ocean texture '''

    ocean = None
    '''The ocean data associated with this texture 

    :type:  OceanTexData, (readonly) 
    '''

    users_material = None
    '''Materials that use this texture (readonly) '''

    users_object_modifier = None
    '''Object modifiers that use this texture (readonly) '''


class Operator:
    '''Storage of an operator being executed, or registered after execution '''

    bl_description = None
    '''

    :type:  string, default “” 
    '''

    bl_idname = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_label = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_options = None
    '''Options for this operator type 

    :type:  enum set in {‘REGISTER’, ‘UNDO’, ‘BLOCKING’, ‘MACRO’, ‘GRAB_CURSOR’, ‘PRESET’, ‘INTERNAL’}, default {‘REGISTER’} 
    '''

    bl_translation_context = None
    '''

    :type:  string, default “Operator” 
    '''

    has_reports = None
    '''Operator has a set of reports (warnings and errors) from last execution 

    :type:  boolean, default False, (readonly) 
    '''

    layout = None
    '''

    :type:  UILayout, (readonly) 
    '''

    macros = None
    '''

    :type:  bpy_prop_collection of Macro, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (readonly, never None) 
    '''

    options = None
    '''Runtime options 

    :type:  OperatorOptions, (readonly, never None) 
    '''

    properties = None
    '''

    :type:  OperatorProperties, (readonly, never None) 
    '''

    bl_property = None
    '''The name of a property to use as this operators primary property. Currently this is only used to select the default property when expanding an operator into a menu. :type: string '''

    def report(self, type, message):
        '''report 

        :param type: Type 
        :type type: enum set in {'DEBUG', 'INFO', 'OPERATOR', 'PROPERTY', 'WARNING', 'ERROR', 'ERROR_INVALID_INPUT', 'ERROR_INVALID_CONTEXT', 'ERROR_OUT_OF_MEMORY'}
        :param message: Report Message 
        :type message: string, (never None)
        '''
        pass

    def execute(self, context):
        '''Execute the operator 

        :rtype: enum set in {‘RUNNING_MODAL’, ‘CANCELLED’, ‘FINISHED’, ‘PASS_THROUGH’, ‘INTERFACE’} 
        :return:  resultRUNNING_MODAL Running Modal, Keep the operator running with blender.CANCELLED Cancelled, When no action has been taken, operator exits.FINISHED Finished, When the operator is complete, operator exits.PASS_THROUGH Pass Through, Do nothing and pass the event on.INTERFACE Interface, Handled but not executed (popup menus). 
        '''
        pass

    def check(self, context):
        '''Check the operator settings, return True to signal a change to redraw 

        :rtype: boolean 
        :return:  result 
        '''
        pass

    def invoke(self, context, event):
        '''Invoke the operator 

        :rtype: enum set in {‘RUNNING_MODAL’, ‘CANCELLED’, ‘FINISHED’, ‘PASS_THROUGH’, ‘INTERFACE’} 
        :return:  resultRUNNING_MODAL Running Modal, Keep the operator running with blender.CANCELLED Cancelled, When no action has been taken, operator exits.FINISHED Finished, When the operator is complete, operator exits.PASS_THROUGH Pass Through, Do nothing and pass the event on.INTERFACE Interface, Handled but not executed (popup menus). 
        '''
        pass

    def modal(self, context, event):
        '''Modal operator function 

        :rtype: enum set in {‘RUNNING_MODAL’, ‘CANCELLED’, ‘FINISHED’, ‘PASS_THROUGH’, ‘INTERFACE’} 
        :return:  resultRUNNING_MODAL Running Modal, Keep the operator running with blender.CANCELLED Cancelled, When no action has been taken, operator exits.FINISHED Finished, When the operator is complete, operator exits.PASS_THROUGH Pass Through, Do nothing and pass the event on.INTERFACE Interface, Handled but not executed (popup menus). 
        '''
        pass

    def draw(self, context):
        '''Draw function for the operator 

        '''
        pass

    def cancel(self, context):
        '''Called when the operator is canceled 

        '''
        pass

    def as_keywords(self, ignore=()):
        '''Return a copy of the properties as a dictionary 

        '''
        pass


class OperatorFileListElement:
    name = None
    '''Name of a file or directory within a file list 

    :type:  string, default “”, (never None) 
    '''


class OperatorMacro:
    '''Storage of a sub operator in a macro after it has been added '''

    properties = None
    '''

    :type:  OperatorProperties, (readonly, never None) 
    '''


class OperatorMousePath:
    '''Mouse path values for operators that record such paths '''

    loc = None
    '''Mouse location 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    time = None
    '''Time of mouse location 

    :type:  float in [-inf, inf], default 0.0 
    '''


class OperatorOptions:
    '''Runtime options '''

    is_grab_cursor = None
    '''True when the cursor is grabbed 

    :type:  boolean, default False, (readonly) 
    '''

    is_invoke = None
    '''True when invoked (even if only the execute callbacks available) 

    :type:  boolean, default False, (readonly) 
    '''

    use_cursor_region = None
    '''Enable to use the region under the cursor for modal execution 

    :type:  boolean, default False 
    '''


class OperatorProperties:
    '''Input properties of an Operator '''

    pass


class OperatorStrokeElement:
    is_start = None
    '''

    :type:  boolean, default False 
    '''

    location = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    mouse = None
    '''

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    pen_flip = None
    '''

    :type:  boolean, default False 
    '''

    pressure = None
    '''Tablet pressure 

    :type:  float in [0, 1], default 0.0 
    '''

    size = None
    '''Brush Size in screen space 

    :type:  float in [0, inf], default 0.0 
    '''

    time = None
    '''

    :type:  float in [0, inf], default 0.0 
    '''


class OrController:
    '''Controller passing on events based on a logical OR operation '''

    pass


class OutflowFluidSettings:
    '''Fluid simulation settings for objects removing fluids from the simulation '''

    use = None
    '''Object contributes to the fluid simulation 

    :type:  boolean, default False 
    '''

    use_animated_mesh = None
    '''Export this mesh as an animated one (slower and enforces No Slip, only use if really necessary [e.g. armatures or parented objects], animated pos/rot/scale F-Curves do not require it) 

    :type:  boolean, default False 
    '''

    volume_initialization = None
    '''Volume initialization type (WARNING: complex volumes might require too much memory and break simulation) 

    :type:  enum in [‘VOLUME’, ‘SHELL’, ‘BOTH’], default ‘VOLUME’ 
    '''


class OverDropSequence:
    '''Over Drop Sequence '''

    input_1 = None
    '''First input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_2 = None
    '''Second input for the effect strip 

    :type:  Sequence, (never None) 
    '''

    input_count = None
    '''

    :type:  int in [0, inf], default 0, (readonly) 
    '''


class PARTICLE_UL_particle_systems:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index, flt_flag):
        '''

        '''
        pass


class PHYSICS_UL_dynapaint_surfaces:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class PackedFile:
    '''External file packed into the .blend file '''

    data = None
    '''Raw data (bytes, exact content of the embedded file) 

    :type:  string, default “”, (readonly, never None) 
    '''

    size = None
    '''Size of packed file in bytes 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''


class Paint:
    brush = None
    '''Active Brush 

    :type:  Brush 
    '''

    cavity_curve = None
    '''Editable cavity curve 

    :type:  CurveMapping, (readonly, never None) 
    '''

    input_samples = None
    '''Average multiple input samples together to smooth the brush stroke 

    :type:  int in [0, inf], default 0 
    '''

    palette = None
    '''Active Palette 

    :type:  Palette 
    '''

    show_brush = None
    '''

    :type:  boolean, default False 
    '''

    show_brush_on_surface = None
    '''

    :type:  boolean, default False 
    '''

    show_low_resolution = None
    '''For multires, show low resolution while navigating the view 

    :type:  boolean, default False 
    '''

    tile_offset = None
    '''Stride at which tiled strokes are copied 

    :type:  float array of 3 items in [0.01, inf], default (0.0, 0.0, 0.0) 
    '''

    tile_x = None
    '''Tile along X axis 

    :type:  boolean, default False 
    '''

    tile_y = None
    '''Tile along Y axis 

    :type:  boolean, default False 
    '''

    tile_z = None
    '''Tile along Z axis 

    :type:  boolean, default False 
    '''

    use_cavity = None
    '''Mask painting according to mesh geometry cavity 

    :type:  boolean, default False 
    '''

    use_symmetry_feather = None
    '''Reduce the strength of the brush where it overlaps symmetrical daubs 

    :type:  boolean, default False 
    '''

    use_symmetry_x = None
    '''Mirror brush across the X axis 

    :type:  boolean, default False 
    '''

    use_symmetry_y = None
    '''Mirror brush across the Y axis 

    :type:  boolean, default False 
    '''

    use_symmetry_z = None
    '''Mirror brush across the Z axis 

    :type:  boolean, default False 
    '''


class PaintCurve:
    pass


class Palette:
    colors = None
    '''

    :type:  PaletteColors bpy_prop_collection of PaletteColor, (readonly) 
    '''


class PaletteColor:
    color = None
    '''

    :type:  float array of 3 items in [0, 1], default (0.0, 0.0, 0.0) 
    '''

    strength = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''

    weight = None
    '''

    :type:  float in [0, 1], default 0.0 
    '''


class PaletteColors:
    '''Collection of palette colors '''

    active = None
    '''

    :type:  PaletteColor 
    '''

    def new(self):
        '''Add a new color to the palette 

        :rtype: PaletteColor 
        :return:  The newly created color 
        '''
        pass

    def remove(self, color):
        '''Remove a color from the palette 

        :param color: The color to remove 
        :type color: PaletteColor, (never None)
        '''
        pass

    def clear(self):
        '''Remove all colors from the palette 

        '''
        pass


class Panel:
    '''Panel containing UI elements '''

    bl_category = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_context = None
    '''The context in which the panel belongs to. (TODO: explain the possible combinations bl_context/bl_region_type/bl_space_type) 

    :type:  string, default “”, (never None) 
    '''

    bl_idname = None
    '''If this is set, the panel gets a custom ID, otherwise it takes the name of the class used to define the panel. For example, if the class name is “OBJECT_PT_hello”, and bl_idname is not set by the script, then bl_idname = “OBJECT_PT_hello” 

    :type:  string, default “”, (never None) 
    '''

    bl_label = None
    '''The panel label, shows up in the panel header at the right of the triangle used to collapse the panel 

    :type:  string, default “”, (never None) 
    '''

    bl_options = None
    '''Options for this panel type 

    :type:  enum set in {‘DEFAULT_CLOSED’, ‘HIDE_HEADER’}, default {‘DEFAULT_CLOSED’} 
    '''

    bl_region_type = None
    '''The region where the panel is going to be used in 

    :type:  enum in [‘WINDOW’, ‘HEADER’, ‘CHANNELS’, ‘TEMPORARY’, ‘UI’, ‘TOOLS’, ‘TOOL_PROPS’, ‘PREVIEW’], default ‘WINDOW’ 
    '''

    bl_space_type = None
    '''The space where the panel is going to be used in 

    :type:  enum in [‘EMPTY’, ‘VIEW_3D’, ‘TIMELINE’, ‘GRAPH_EDITOR’, ‘DOPESHEET_EDITOR’, ‘NLA_EDITOR’, ‘IMAGE_EDITOR’, ‘SEQUENCE_EDITOR’, ‘CLIP_EDITOR’, ‘TEXT_EDITOR’, ‘NODE_EDITOR’, ‘LOGIC_EDITOR’, ‘PROPERTIES’, ‘OUTLINER’, ‘USER_PREFERENCES’, ‘INFO’, ‘FILE_BROWSER’, ‘CONSOLE’], default ‘EMPTY’ 
    '''

    bl_translation_context = None
    '''

    :type:  string, default “*”, (never None) 
    '''

    layout = None
    '''Defines the structure of the panel in the UI 

    :type:  UILayout, (readonly) 
    '''

    text = None
    '''XXX todo 

    :type:  string, default “”, (never None) 
    '''

    use_pin = None
    '''

    :type:  boolean, default False 
    '''

    def draw(self, context):
        '''Draw UI elements into the panel UI layout 

        '''
        pass

    def draw_header(self, context):
        '''Draw UI elements into the panel’s header UI layout 

        '''
        pass


class ParentActuator:
    mode = None
    '''

    :type:  enum in [‘SETPARENT’, ‘REMOVEPARENT’], default ‘SETPARENT’ 
    '''

    object = None
    '''Set this object as parent 

    :type:  Object 
    '''

    use_compound = None
    '''Add this object shape to the parent shape (only if the parent shape is already compound) 

    :type:  boolean, default False 
    '''

    use_ghost = None
    '''Make this object ghost while parented 

    :type:  boolean, default False 
    '''


class Particle:
    '''Particle in a particle system '''

    alive_state = None
    '''

    :type:  enum in [‘DEAD’, ‘UNBORN’, ‘ALIVE’, ‘DYING’], default ‘DEAD’ 
    '''

    angular_velocity = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    birth_time = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    die_time = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    hair_keys = None
    '''

    :type:  bpy_prop_collection of ParticleHairKey, (readonly) 
    '''

    is_exist = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    is_visible = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    lifetime = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    location = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    particle_keys = None
    '''

    :type:  bpy_prop_collection of ParticleKey, (readonly) 
    '''

    prev_angular_velocity = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    prev_location = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    prev_rotation = None
    '''

    :type:  float array of 4 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''

    prev_velocity = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    rotation = None
    '''

    :type:  float array of 4 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''

    size = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    velocity = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    def uv_on_emitter(self, modifier):
        '''Obtain uv for particle on derived mesh 

        :param modifier: Particle modifier 
        :type modifier: ParticleSystemModifier, (never None)
        :rtype: float array of 2 items in [-inf, inf] 
        :return:  uv 
        '''
        pass


class ParticleBrush:
    '''Particle editing brush '''

    count = None
    '''Particle count 

    :type:  int in [1, 1000], default 0 
    '''

    curve = None
    '''

    :type:  CurveMapping, (readonly) 
    '''

    length_mode = None
    '''

    :type:  enum in [‘GROW’, ‘SHRINK’], default ‘GROW’ 
    '''

    puff_mode = None
    '''

    :type:  enum in [‘ADD’, ‘SUB’], default ‘ADD’ 
    '''

    size = None
    '''Radius of the brush in pixels 

    :type:  int in [1, 32767], default 0 
    '''

    steps = None
    '''Brush steps 

    :type:  int in [1, 32767], default 0 
    '''

    strength = None
    '''Brush strength 

    :type:  float in [0.001, 1], default 0.0 
    '''

    use_puff_volume = None
    '''Apply puff to unselected end-points (helps maintain hair volume when puffing root) 

    :type:  boolean, default False 
    '''


class ParticleDupliWeight:
    '''Weight of a particle dupliobject in a group '''

    count = None
    '''The number of times this object is repeated with respect to other objects 

    :type:  int in [0, 32767], default 0 
    '''

    name = None
    '''Particle dupliobject name 

    :type:  string, default “”, (readonly, never None) 
    '''


class ParticleEdit:
    '''Properties of particle editing mode '''

    brush = None
    '''

    :type:  ParticleBrush, (readonly) 
    '''

    default_key_count = None
    '''How many keys to make new particles with 

    :type:  int in [2, 32767], default 0 
    '''

    draw_step = None
    '''How many steps to draw the path with 

    :type:  int in [1, 10], default 0 
    '''

    emitter_distance = None
    '''Distance to keep particles away from the emitter 

    :type:  float in [0, inf], default 0.0 
    '''

    fade_frames = None
    '''How many frames to fade 

    :type:  int in [1, 100], default 0 
    '''

    is_editable = None
    '''A valid edit mode exists 

    :type:  boolean, default False, (readonly) 
    '''

    is_hair = None
    '''Editing hair 

    :type:  boolean, default False, (readonly) 
    '''

    object = None
    '''The edited object 

    :type:  Object, (readonly) 
    '''

    select_mode = None
    '''Particle select and display mode 

    :type:  enum in [‘PATH’, ‘POINT’, ‘TIP’], default ‘PATH’ 
    '''

    shape_object = None
    '''Outer shape to use for tools 

    :type:  Object 
    '''

    show_particles = None
    '''Draw actual particles 

    :type:  boolean, default False 
    '''

    tool = None
    '''

    :type:  enum in [‘NONE’, ‘COMB’, ‘SMOOTH’, ‘ADD’, ‘LENGTH’, ‘PUFF’, ‘CUT’, ‘WEIGHT’], default ‘COMB’ 
    '''

    type = None
    '''

    :type:  enum in [‘PARTICLES’, ‘SOFT_BODY’, ‘CLOTH’], default ‘PARTICLES’ 
    '''

    use_auto_velocity = None
    '''Calculate point velocities automatically 

    :type:  boolean, default False 
    '''

    use_default_interpolate = None
    '''Interpolate new particles from the existing ones 

    :type:  boolean, default False 
    '''

    use_emitter_deflect = None
    '''Keep paths from intersecting the emitter 

    :type:  boolean, default False 
    '''

    use_fade_time = None
    '''Fade paths and keys further away from current frame 

    :type:  boolean, default False 
    '''

    use_preserve_length = None
    '''Keep path lengths constant 

    :type:  boolean, default False 
    '''

    use_preserve_root = None
    '''Keep root keys unmodified 

    :type:  boolean, default False 
    '''


class ParticleFluidSettings:
    '''Fluid simulation settings for objects storing fluid particles generated by the simulation '''

    alpha_influence = None
    '''Amount of particle alpha change, inverse of size influence: 0=off (all same alpha), 1=full (larger particles get lower alphas, smaller ones higher values) 

    :type:  float in [0, 2], default 0.0 
    '''

    filepath = None
    '''Directory (and/or filename prefix) to store and load particles from 

    :type:  string, default “”, (never None) 
    '''

    particle_influence = None
    '''Amount of particle size scaling: 0=off (all same size), 1=full (range 0.2-2.0), >1=stronger 

    :type:  float in [0, 2], default 0.0 
    '''

    show_tracer = None
    '''Show tracer particles 

    :type:  boolean, default False 
    '''

    use_drops = None
    '''Show drop particles 

    :type:  boolean, default False 
    '''

    use_floats = None
    '''Show floating foam particles 

    :type:  boolean, default False 
    '''


class ParticleHairKey:
    '''Particle key for hair particle system '''

    co = None
    '''Location of the hair key in object space 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    co_local = None
    '''Location of the hair key in its local coordinate system, relative to the emitting face 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    time = None
    '''Relative time of key over hair length 

    :type:  float in [0, inf], default 0.0 
    '''

    weight = None
    '''Weight for cloth simulation 

    :type:  float in [0, 1], default 0.0 
    '''

    def co_object(self, object, modifier, particle):
        '''Obtain hairkey location with particle and modifier data 

        :param object: Object 
        :type object: Object, (never None)
        :param modifier: Particle modifier 
        :type modifier: ParticleSystemModifier, (never None)
        :param particle: hair particle 
        :type particle: Particle, (never None)
        :rtype: float array of 3 items in [-inf, inf] 
        :return:  Co, Exported hairkey location 
        '''
        pass


class ParticleInstanceModifier:
    '''Particle system instancing modifier '''

    axis = None
    '''Pole axis for rotation 

    :type:  enum in [‘X’, ‘Y’, ‘Z’], default ‘X’ 
    '''

    object = None
    '''Object that has the particle system 

    :type:  Object 
    '''

    particle_system_index = None
    '''

    :type:  int in [1, 10], default 0 
    '''

    position = None
    '''Position along path 

    :type:  float in [0, 1], default 0.0 
    '''

    random_position = None
    '''Randomize position along path 

    :type:  float in [0, 1], default 0.0 
    '''

    show_alive = None
    '''Show instances when particles are alive 

    :type:  boolean, default False 
    '''

    show_dead = None
    '''Show instances when particles are dead 

    :type:  boolean, default False 
    '''

    show_unborn = None
    '''Show instances when particles are unborn 

    :type:  boolean, default False 
    '''

    use_children = None
    '''Create instances from child particles 

    :type:  boolean, default False 
    '''

    use_normal = None
    '''Create instances from normal particles 

    :type:  boolean, default False 
    '''

    use_path = None
    '''Create instances along particle paths 

    :type:  boolean, default False 
    '''

    use_preserve_shape = None
    '''Don’t stretch the object 

    :type:  boolean, default False 
    '''

    use_size = None
    '''Use particle size to scale the instances 

    :type:  boolean, default False 
    '''


class ParticleKey:
    '''Key location for a particle over time '''

    angular_velocity = None
    '''Key angular velocity 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    location = None
    '''Key location 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    rotation = None
    '''Key rotation quaternion 

    :type:  float array of 4 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''

    time = None
    '''Time of key over the simulation 

    :type:  float in [0, inf], default 0.0 
    '''

    velocity = None
    '''Key velocity 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''


class ParticleSettings:
    '''Particle settings, reusable by multiple particle systems '''

    active_dupliweight = None
    '''

    :type:  ParticleDupliWeight, (readonly) 
    '''

    active_dupliweight_index = None
    '''

    :type:  int in [0, inf], default 0 
    '''

    active_texture = None
    '''Active texture slot being displayed 

    :type:  Texture 
    '''

    active_texture_index = None
    '''Index of active texture slot 

    :type:  int in [0, 17], default 0 
    '''

    adaptive_angle = None
    '''How many degrees path has to curve to make another render segment 

    :type:  int in [0, 45], default 0 
    '''

    adaptive_pixel = None
    '''How many pixels path has to cover to make another render segment 

    :type:  int in [0, 50], default 0 
    '''

    angular_velocity_factor = None
    '''Angular velocity amount (in radians per second) 

    :type:  float in [-200, 200], default 0.0 
    '''

    angular_velocity_mode = None
    '''What axis is used to change particle rotation with time 

    :type:  enum in [‘NONE’, ‘VELOCITY’, ‘HORIZONTAL’, ‘VERTICAL’, ‘GLOBAL_X’, ‘GLOBAL_Y’, ‘GLOBAL_Z’, ‘RAND’], default ‘NONE’ 
    '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    apply_effector_to_children = None
    '''Apply effectors to children 

    :type:  boolean, default False 
    '''

    apply_guide_to_children = None
    '''

    :type:  boolean, default False 
    '''

    bending_random = None
    '''Random stiffness of hairs 

    :type:  float in [0, 1], default 0.0 
    '''

    billboard_align = None
    '''In respect to what the billboards are aligned 

    :type:  enum in [‘X’, ‘Y’, ‘Z’, ‘VIEW’, ‘VEL’], default ‘X’ 
    '''

    billboard_animation = None
    '''How to animate billboard textures 

    :type:  enum in [‘NONE’, ‘AGE’, ‘FRAME’, ‘ANGLE’], default ‘NONE’ 
    '''

    billboard_object = None
    '''Billboards face this object (default is active camera) 

    :type:  Object 
    '''

    billboard_offset = None
    '''

    :type:  float array of 2 items in [-100, 100], default (0.0, 0.0) 
    '''

    billboard_offset_split = None
    '''How to offset billboard textures 

    :type:  enum in [‘NONE’, ‘LINEAR’, ‘RANDOM’], default ‘NONE’ 
    '''

    billboard_size = None
    '''Scale billboards relative to particle size 

    :type:  float array of 2 items in [0.001, 10], default (0.0, 0.0) 
    '''

    billboard_tilt = None
    '''Tilt of the billboards 

    :type:  float in [-1, 1], default 0.0 
    '''

    billboard_tilt_random = None
    '''Random tilt of the billboards 

    :type:  float in [0, 1], default 0.0 
    '''

    billboard_uv_split = None
    '''Number of rows/columns to split UV coordinates for billboards 

    :type:  int in [1, 100], default 0 
    '''

    billboard_velocity_head = None
    '''Scale billboards by velocity 

    :type:  float in [0, 10], default 0.0 
    '''

    billboard_velocity_tail = None
    '''Scale billboards by velocity 

    :type:  float in [0, 10], default 0.0 
    '''

    boids = None
    '''

    :type:  BoidSettings, (readonly) 
    '''

    branch_threshold = None
    '''Threshold of branching 

    :type:  float in [0, 1], default 0.0 
    '''

    brownian_factor = None
    '''Amount of random, erratic particle movement 

    :type:  float in [0, 200], default 0.0 
    '''

    child_length = None
    '''Length of child paths 

    :type:  float in [0, 1], default 0.0 
    '''

    child_length_threshold = None
    '''Amount of particles left untouched by child path length 

    :type:  float in [0, 1], default 0.0 
    '''

    child_nbr = None
    '''Number of children/parent 

    :type:  int in [0, 100000], default 0 
    '''

    child_parting_factor = None
    '''Create parting in the children based on parent strands 

    :type:  float in [0, 1], default 0.0 
    '''

    child_parting_max = None
    '''Maximum root to tip angle (tip distance/root distance for long hair) 

    :type:  float in [0, 180], default 0.0 
    '''

    child_parting_min = None
    '''Minimum root to tip angle (tip distance/root distance for long hair) 

    :type:  float in [0, 180], default 0.0 
    '''

    child_radius = None
    '''Radius of children around parent 

    :type:  float in [0, 10], default 0.0 
    '''

    child_roundness = None
    '''Roundness of children around parent 

    :type:  float in [0, 1], default 0.0 
    '''

    child_size = None
    '''A multiplier for the child particle size 

    :type:  float in [0.001, 100000], default 0.0 
    '''

    child_size_random = None
    '''Random variation to the size of the child particles 

    :type:  float in [0, 1], default 0.0 
    '''

    child_type = None
    '''Create child particles 

    :type:  enum in [‘NONE’, ‘SIMPLE’, ‘INTERPOLATED’], default ‘NONE’ 
    '''

    clump_curve = None
    '''Curve defining clump tapering 

    :type:  CurveMapping, (readonly) 
    '''

    clump_factor = None
    '''Amount of clumping 

    :type:  float in [-1, 1], default 0.0 
    '''

    clump_noise_size = None
    '''Size of clump noise 

    :type:  float in [1e-05, 100000], default 0.0 
    '''

    clump_shape = None
    '''Shape of clumping 

    :type:  float in [-0.999, 0.999], default 0.0 
    '''

    collision_group = None
    '''Limit colliders to this Group 

    :type:  Group 
    '''

    color_maximum = None
    '''Maximum length of the particle color vector 

    :type:  float in [0.01, 100], default 0.0 
    '''

    count = None
    '''Total number of particles 

    :type:  int in [0, 10000000], default 0 
    '''

    courant_target = None
    '''The relative distance a particle can move before requiring more subframes (target Courant number); 0.01-0.3 is the recommended range 

    :type:  float in [0.0001, 10], default 0.1 
    '''

    create_long_hair_children = None
    '''Calculate children that suit long hair well 

    :type:  boolean, default False 
    '''

    cycles = None
    '''Cycles hair settings 

    :type:  CyclesCurveSettings, (readonly) 
    '''

    damping = None
    '''Amount of damping 

    :type:  float in [0, 1], default 0.0 
    '''

    distribution = None
    '''How to distribute particles on selected element 

    :type:  enum in [‘JIT’, ‘RAND’, ‘GRID’], default ‘JIT’ 
    '''

    drag_factor = None
    '''Amount of air-drag 

    :type:  float in [0, 1], default 0.0 
    '''

    draw_color = None
    '''Draw additional particle data as a color 

    :type:  enum in [‘NONE’, ‘MATERIAL’, ‘VELOCITY’, ‘ACCELERATION’], default ‘NONE’ 
    '''

    draw_method = None
    '''How particles are drawn in viewport 

    :type:  enum in [‘NONE’, ‘RENDER’, ‘DOT’, ‘CIRC’, ‘CROSS’, ‘AXIS’], default ‘NONE’ 
    '''

    draw_percentage = None
    '''Percentage of particles to display in 3D view 

    :type:  int in [0, 100], default 0 
    '''

    draw_size = None
    '''Size of particles on viewport in pixels (0=default) 

    :type:  int in [0, 1000], default 0 
    '''

    draw_step = None
    '''How many steps paths are drawn with (power of 2) 

    :type:  int in [0, 10], default 0 
    '''

    dupli_group = None
    '''Show Objects in this Group in place of particles 

    :type:  Group 
    '''

    dupli_object = None
    '''Show this Object in place of particles 

    :type:  Object 
    '''

    dupli_weights = None
    '''Weights for all of the objects in the dupli group 

    :type:  bpy_prop_collection of ParticleDupliWeight, (readonly) 
    '''

    effect_hair = None
    '''Hair stiffness for effectors 

    :type:  float in [0, 1], default 0.0 
    '''

    effector_amount = None
    '''How many particles are effectors (0 is all particles) 

    :type:  int in [0, 10000], default 0 
    '''

    effector_weights = None
    '''

    :type:  EffectorWeights, (readonly) 
    '''

    emit_from = None
    '''Where to emit particles from 

    :type:  enum in [‘VERT’, ‘FACE’, ‘VOLUME’], default ‘VERT’ 
    '''

    factor_random = None
    '''Give the starting velocity a random variation 

    :type:  float in [0, 200], default 0.0 
    '''

    fluid = None
    '''

    :type:  SPHFluidSettings, (readonly) 
    '''

    force_field_1 = None
    '''

    :type:  FieldSettings, (readonly) 
    '''

    force_field_2 = None
    '''

    :type:  FieldSettings, (readonly) 
    '''

    frame_end = None
    '''Frame number to stop emitting particles 

    :type:  float in [-500000, 500000], default 0.0 
    '''

    frame_start = None
    '''Frame number to start emitting particles 

    :type:  float in [-500000, 500000], default 0.0 
    '''

    grid_random = None
    '''Add random offset to the grid locations 

    :type:  float in [0, 1], default 0.0 
    '''

    grid_resolution = None
    '''The resolution of the particle grid 

    :type:  int in [1, 250], default 0 
    '''

    hair_length = None
    '''Length of the hair 

    :type:  float in [0, 1000], default 0.0 
    '''

    hair_step = None
    '''Number of hair segments 

    :type:  int in [2, 50], default 0 
    '''

    hexagonal_grid = None
    '''Create the grid in a hexagonal pattern 

    :type:  boolean, default False 
    '''

    integrator = None
    '''Algorithm used to calculate physics, from the fastest to the most stable/accurate: Midpoint, Euler, Verlet, RK4 (Old) 

    :type:  enum in [‘EULER’, ‘VERLET’, ‘MIDPOINT’, ‘RK4’], default ‘EULER’ 
    '''

    invert_grid = None
    '''Invert what is considered object and what is not 

    :type:  boolean, default False 
    '''

    is_fluid = None
    '''Particles were created by a fluid simulation 

    :type:  boolean, default False, (readonly) 
    '''

    jitter_factor = None
    '''Amount of jitter applied to the sampling 

    :type:  float in [0, 2], default 0.0 
    '''

    keyed_loops = None
    '''Number of times the keys are looped 

    :type:  int in [1, 10000], default 0 
    '''

    keys_step = None
    '''

    :type:  int in [0, 32767], default 0 
    '''

    kink = None
    '''Type of periodic offset on the path 

    :type:  enum in [‘NO’, ‘CURL’, ‘RADIAL’, ‘WAVE’, ‘BRAID’, ‘SPIRAL’], default ‘NO’ 
    '''

    kink_amplitude = None
    '''The amplitude of the offset 

    :type:  float in [-100000, 100000], default 0.0 
    '''

    kink_amplitude_clump = None
    '''How much clump affects kink amplitude 

    :type:  float in [0, 1], default 0.0 
    '''

    kink_amplitude_random = None
    '''Random variation of the amplitude 

    :type:  float in [0, 1], default 0.0 
    '''

    kink_axis = None
    '''Which axis to use for offset 

    :type:  enum in [‘X’, ‘Y’, ‘Z’], default ‘X’ 
    '''

    kink_axis_random = None
    '''Random variation of the orientation 

    :type:  float in [0, 1], default 0.0 
    '''

    kink_extra_steps = None
    '''Extra steps for resolution of special kink features 

    :type:  int in [1, inf], default 0 
    '''

    kink_flat = None
    '''How flat the hairs are 

    :type:  float in [0, 1], default 0.0 
    '''

    kink_frequency = None
    '''The frequency of the offset (1/total length) 

    :type:  float in [-100000, 100000], default 0.0 
    '''

    kink_shape = None
    '''Adjust the offset to the beginning/end 

    :type:  float in [-0.999, 0.999], default 0.0 
    '''

    length_random = None
    '''Give path length a random variation 

    :type:  float in [0, 1], default 0.0 
    '''

    lifetime = None
    '''Life span of the particles 

    :type:  float in [1, 500000], default 0.0 
    '''

    lifetime_random = None
    '''Give the particle life a random variation 

    :type:  float in [0, 1], default 0.0 
    '''

    line_length_head = None
    '''Length of the line’s head 

    :type:  float in [0, 100000], default 0.0 
    '''

    line_length_tail = None
    '''Length of the line’s tail 

    :type:  float in [0, 100000], default 0.0 
    '''

    lock_billboard = None
    '''Lock the billboards align axis 

    :type:  boolean, default False 
    '''

    lock_boids_to_surface = None
    '''Constrain boids to a surface 

    :type:  boolean, default False 
    '''

    mass = None
    '''Mass of the particles 

    :type:  float in [1e-08, 100000], default 0.0 
    '''

    material = None
    '''Index of material slot used for rendering particles 

    :type:  int in [1, 32767], default 0 
    '''

    material_slot = None
    '''Material slot used for rendering particles 

    :type:  enum in [‘DUMMY’], default ‘DUMMY’ 
    '''

    normal_factor = None
    '''Let the surface normal give the particle a starting velocity 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    object_align_factor = None
    '''Let the emitter object orientation give the particle a starting velocity 

    :type:  float array of 3 items in [-200, 200], default (0.0, 0.0, 0.0) 
    '''

    object_factor = None
    '''Let the object give the particle a starting velocity 

    :type:  float in [-200, 200], default 0.0 
    '''

    particle_factor = None
    '''Let the target particle give the particle a starting velocity 

    :type:  float in [-200, 200], default 0.0 
    '''

    particle_size = None
    '''The size of the particles 

    :type:  float in [0.001, 100000], default 0.0 
    '''

    path_end = None
    '''End time of drawn path 

    :type:  float in [-inf, inf], default 0.0 
    '''

    path_start = None
    '''Starting time of drawn path 

    :type:  float in [-inf, inf], default 0.0 
    '''

    phase_factor = None
    '''Rotation around the chosen orientation axis 

    :type:  float in [-1, 1], default 0.0 
    '''

    phase_factor_random = None
    '''Randomize rotation around the chosen orientation axis 

    :type:  float in [0, 2], default 0.0 
    '''

    physics_type = None
    '''Particle physics type 

    :type:  enum in [‘NO’, ‘NEWTON’, ‘KEYED’, ‘BOIDS’, ‘FLUID’], default ‘NO’ 
    '''

    react_event = None
    '''The event of target particles to react on 

    :type:  enum in [‘DEATH’, ‘COLLIDE’, ‘NEAR’], default ‘DEATH’ 
    '''

    reactor_factor = None
    '''Let the vector away from the target particle’s location give the particle a starting velocity 

    :type:  float in [-10, 10], default 0.0 
    '''

    regrow_hair = None
    '''Regrow hair for each frame 

    :type:  boolean, default False 
    '''

    render_step = None
    '''How many steps paths are rendered with (power of 2) 

    :type:  int in [0, 20], default 0 
    '''

    render_type = None
    '''How particles are rendered 

    :type:  enum in [‘NONE’, ‘HALO’, ‘LINE’, ‘PATH’, ‘OBJECT’, ‘GROUP’, ‘BILLBOARD’], default ‘NONE’ 
    '''

    rendered_child_count = None
    '''Number of children/parent for rendering 

    :type:  int in [0, 100000], default 0 
    '''

    rotation_factor_random = None
    '''Randomize particle orientation 

    :type:  float in [0, 1], default 0.0 
    '''

    rotation_mode = None
    '''Particle orientation axis (does not affect Explode modifier’s results) 

    :type:  enum in [‘NONE’, ‘NOR’, ‘NOR_TAN’, ‘VEL’, ‘GLOB_X’, ‘GLOB_Y’, ‘GLOB_Z’, ‘OB_X’, ‘OB_Y’, ‘OB_Z’], default ‘NONE’ 
    '''

    roughness_1 = None
    '''Amount of location dependent rough 

    :type:  float in [0, 100000], default 0.0 
    '''

    roughness_1_size = None
    '''Size of location dependent rough 

    :type:  float in [0.01, 100000], default 0.0 
    '''

    roughness_2 = None
    '''Amount of random rough 

    :type:  float in [0, 100000], default 0.0 
    '''

    roughness_2_size = None
    '''Size of random rough 

    :type:  float in [0.01, 100000], default 0.0 
    '''

    roughness_2_threshold = None
    '''Amount of particles left untouched by random rough 

    :type:  float in [0, 1], default 0.0 
    '''

    roughness_curve = None
    '''Curve defining roughness 

    :type:  CurveMapping, (readonly) 
    '''

    roughness_end_shape = None
    '''Shape of end point rough 

    :type:  float in [0, 10], default 0.0 
    '''

    roughness_endpoint = None
    '''Amount of end point rough 

    :type:  float in [0, 100000], default 0.0 
    '''

    show_guide_hairs = None
    '''Show guide hairs 

    :type:  boolean, default False 
    '''

    show_hair_grid = None
    '''Show hair simulation grid 

    :type:  boolean, default False 
    '''

    show_health = None
    '''Draw boid health 

    :type:  boolean, default False 
    '''

    show_number = None
    '''Show particle number 

    :type:  boolean, default False 
    '''

    show_size = None
    '''Show particle size 

    :type:  boolean, default False 
    '''

    show_unborn = None
    '''Show particles before they are emitted 

    :type:  boolean, default False 
    '''

    show_velocity = None
    '''Show particle velocity 

    :type:  boolean, default False 
    '''

    simplify_rate = None
    '''Speed of simplification 

    :type:  float in [0, 1], default 0.0 
    '''

    simplify_refsize = None
    '''Reference size in pixels, after which simplification begins 

    :type:  int in [1, 32767], default 0 
    '''

    simplify_transition = None
    '''Transition period for fading out strands 

    :type:  float in [0, 1], default 0.0 
    '''

    simplify_viewport = None
    '''Speed of Simplification 

    :type:  float in [0, 0.999], default 0.0 
    '''

    size_random = None
    '''Give the particle size a random variation 

    :type:  float in [0, 1], default 0.0 
    '''

    subframes = None
    '''Subframes to simulate for improved stability and finer granularity simulations (dt = timestep / (subframes + 1)) 

    :type:  int in [0, 1000], default 0 
    '''

    tangent_factor = None
    '''Let the surface tangent give the particle a starting velocity 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    tangent_phase = None
    '''Rotate the surface tangent 

    :type:  float in [-1, 1], default 0.0 
    '''

    texture_slots = None
    '''Texture slots defining the mapping and influence of textures 

    :type:  ParticleSettingsTextureSlots bpy_prop_collection of ParticleSettingsTextureSlot, (readonly) 
    '''

    time_tweak = None
    '''A multiplier for physics timestep (1.0 means one frame = 1/25 seconds) 

    :type:  float in [0, 100], default 0.0 
    '''

    timestep = None
    '''The simulation timestep per frame (seconds per frame) 

    :type:  float in [0.0001, 100], default 0.0 
    '''

    trail_count = None
    '''Number of trail particles 

    :type:  int in [1, 100000], default 0 
    '''

    type = None
    '''Particle Type 

    :type:  enum in [‘EMITTER’, ‘HAIR’], default ‘EMITTER’ 
    '''

    use_absolute_path_time = None
    '''Path timing is in absolute frames 

    :type:  boolean, default False 
    '''

    use_adaptive_subframes = None
    '''Automatically set the number of subframes 

    :type:  boolean, default False 
    '''

    use_advanced_hair = None
    '''Use full physics calculations for growing hair 

    :type:  boolean, default False 
    '''

    use_clump_curve = None
    '''Use a curve to define clump tapering 

    :type:  boolean, default False 
    '''

    use_clump_noise = None
    '''Create random clumps around the parent 

    :type:  boolean, default False 
    '''

    use_dead = None
    '''Show particles after they have died 

    :type:  boolean, default False 
    '''

    use_die_on_collision = None
    '''Particles die when they collide with a deflector object 

    :type:  boolean, default False 
    '''

    use_dynamic_rotation = None
    '''Particle rotations are affected by collisions and effectors 

    :type:  boolean, default False 
    '''

    use_emit_random = None
    '''Emit in random order of elements 

    :type:  boolean, default False 
    '''

    use_even_distribution = None
    '''Use even distribution from faces based on face areas or edge lengths 

    :type:  boolean, default False 
    '''

    use_global_dupli = None
    '''Use object’s global coordinates for duplication 

    :type:  boolean, default False 
    '''

    use_group_count = None
    '''Use object multiple times in the same group 

    :type:  boolean, default False 
    '''

    use_group_pick_random = None
    '''Pick objects from group randomly 

    :type:  boolean, default False 
    '''

    use_hair_bspline = None
    '''Interpolate hair using B-Splines 

    :type:  boolean, default False 
    '''

    use_modifier_stack = None
    '''Emit particles from mesh with modifiers applied (must use same subsurf level for viewport and render for correct results) 

    :type:  boolean, default False 
    '''

    use_multiply_size_mass = None
    '''Multiply mass by particle size 

    :type:  boolean, default False 
    '''

    use_parent_particles = None
    '''Render parent particles 

    :type:  boolean, default False 
    '''

    use_react_multiple = None
    '''React multiple times 

    :type:  boolean, default False 
    '''

    use_react_start_end = None
    '''Give birth to unreacted particles eventually 

    :type:  boolean, default False 
    '''

    use_render_adaptive = None
    '''Draw steps of the particle path 

    :type:  boolean, default False 
    '''

    use_render_emitter = None
    '''Render emitter Object also 

    :type:  boolean, default False 
    '''

    use_rotation_dupli = None
    '''Use object’s rotation for duplication (global x-axis is aligned particle rotation axis) 

    :type:  boolean, default False 
    '''

    use_rotations = None
    '''Calculate particle rotations 

    :type:  boolean, default False 
    '''

    use_roughness_curve = None
    '''Use a curve to define roughness 

    :type:  boolean, default False 
    '''

    use_scale_dupli = None
    '''Use object’s scale for duplication 

    :type:  boolean, default False 
    '''

    use_self_effect = None
    '''Particle effectors affect themselves 

    :type:  boolean, default False 
    '''

    use_simplify = None
    '''Remove child strands as the object becomes smaller on the screen 

    :type:  boolean, default False 
    '''

    use_simplify_viewport = None
    '''

    :type:  boolean, default False 
    '''

    use_size_deflect = None
    '''Use particle’s size in deflection 

    :type:  boolean, default False 
    '''

    use_strand_primitive = None
    '''Use the strand primitive for rendering 

    :type:  boolean, default False 
    '''

    use_velocity_length = None
    '''Multiply line length by particle speed 

    :type:  boolean, default False 
    '''

    use_whole_group = None
    '''Use whole group at once 

    :type:  boolean, default False 
    '''

    userjit = None
    '''Emission locations / face (0 = automatic) 

    :type:  int in [0, 1000], default 0 
    '''

    virtual_parents = None
    '''Relative amount of virtual parents 

    :type:  float in [0, 1], default 0.0 
    '''


class ParticleSettingsTextureSlot:
    '''Texture slot for textures in a Particle Settings data-block '''

    clump_factor = None
    '''Amount texture affects child clump 

    :type:  float in [-inf, inf], default 0.0 
    '''

    damp_factor = None
    '''Amount texture affects particle damping 

    :type:  float in [-inf, inf], default 0.0 
    '''

    density_factor = None
    '''Amount texture affects particle density 

    :type:  float in [-inf, inf], default 0.0 
    '''

    field_factor = None
    '''Amount texture affects particle force fields 

    :type:  float in [-inf, inf], default 0.0 
    '''

    gravity_factor = None
    '''Amount texture affects particle gravity 

    :type:  float in [-inf, inf], default 0.0 
    '''

    kink_amp_factor = None
    '''Amount texture affects child kink amplitude 

    :type:  float in [-inf, inf], default 0.0 
    '''

    kink_freq_factor = None
    '''Amount texture affects child kink frequency 

    :type:  float in [-inf, inf], default 0.0 
    '''

    length_factor = None
    '''Amount texture affects child hair length 

    :type:  float in [-inf, inf], default 0.0 
    '''

    life_factor = None
    '''Amount texture affects particle life time 

    :type:  float in [-inf, inf], default 0.0 
    '''

    mapping = None
    '''

    :type:  enum in [‘FLAT’, ‘CUBE’, ‘TUBE’, ‘SPHERE’], default ‘FLAT’ 
    '''

    mapping_x = None
    '''

    :type:  enum in [‘NONE’, ‘X’, ‘Y’, ‘Z’], default ‘NONE’ 
    '''

    mapping_y = None
    '''

    :type:  enum in [‘NONE’, ‘X’, ‘Y’, ‘Z’], default ‘NONE’ 
    '''

    mapping_z = None
    '''

    :type:  enum in [‘NONE’, ‘X’, ‘Y’, ‘Z’], default ‘NONE’ 
    '''

    object = None
    '''Object to use for mapping with Object texture coordinates 

    :type:  Object 
    '''

    rough_factor = None
    '''Amount texture affects child roughness 

    :type:  float in [-inf, inf], default 0.0 
    '''

    size_factor = None
    '''Amount texture affects physical particle size 

    :type:  float in [-inf, inf], default 0.0 
    '''

    texture_coords = None
    '''Texture coordinates used to map the texture onto the background 

    :type:  enum in [‘GLOBAL’, ‘OBJECT’, ‘UV’, ‘ORCO’, ‘STRAND’], default ‘GLOBAL’ 
    '''

    time_factor = None
    '''Amount texture affects particle emission time 

    :type:  float in [-inf, inf], default 0.0 
    '''

    use_map_clump = None
    '''Affect the child clumping 

    :type:  boolean, default False 
    '''

    use_map_damp = None
    '''Affect the particle velocity damping 

    :type:  boolean, default False 
    '''

    use_map_density = None
    '''Affect the density of the particles 

    :type:  boolean, default False 
    '''

    use_map_field = None
    '''Affect the particle force fields 

    :type:  boolean, default False 
    '''

    use_map_gravity = None
    '''Affect the particle gravity 

    :type:  boolean, default False 
    '''

    use_map_kink_amp = None
    '''Affect the child kink amplitude 

    :type:  boolean, default False 
    '''

    use_map_kink_freq = None
    '''Affect the child kink frequency 

    :type:  boolean, default False 
    '''

    use_map_length = None
    '''Affect the child hair length 

    :type:  boolean, default False 
    '''

    use_map_life = None
    '''Affect the life time of the particles 

    :type:  boolean, default False 
    '''

    use_map_rough = None
    '''Affect the child rough 

    :type:  boolean, default False 
    '''

    use_map_size = None
    '''Affect the particle size 

    :type:  boolean, default False 
    '''

    use_map_time = None
    '''Affect the emission time of the particles 

    :type:  boolean, default False 
    '''

    use_map_velocity = None
    '''Affect the particle initial velocity 

    :type:  boolean, default False 
    '''

    uv_layer = None
    '''UV map to use for mapping with UV texture coordinates 

    :type:  string, default “”, (never None) 
    '''

    velocity_factor = None
    '''Amount texture affects particle initial velocity 

    :type:  float in [-inf, inf], default 0.0 
    '''


class ParticleSettingsTextureSlots:
    '''Collection of texture slots '''

    pass


class ParticleSystem:
    '''Particle system in an object '''

    active_particle_target = None
    '''

    :type:  ParticleTarget, (readonly) 
    '''

    active_particle_target_index = None
    '''

    :type:  int in [0, inf], default 0 
    '''

    billboard_normal_uv = None
    '''UV map to control billboard normals 

    :type:  string, default “”, (never None) 
    '''

    billboard_split_uv = None
    '''UV map to control billboard splitting 

    :type:  string, default “”, (never None) 
    '''

    billboard_time_index_uv = None
    '''UV map to control billboard time index (X-Y) 

    :type:  string, default “”, (never None) 
    '''

    child_particles = None
    '''Child particles generated by the particle system 

    :type:  bpy_prop_collection of ChildParticle, (readonly) 
    '''

    child_seed = None
    '''Offset in the random number table for child particles, to get a different randomized result 

    :type:  int in [0, inf], default 0 
    '''

    cloth = None
    '''Cloth dynamics for hair 

    :type:  ClothModifier, (readonly, never None) 
    '''

    dt_frac = None
    '''The current simulation time step size, as a fraction of a frame 

    :type:  float in [0.00990099, 1], default 0.0, (readonly) 
    '''

    has_multiple_caches = None
    '''Particle system has multiple point caches 

    :type:  boolean, default False, (readonly) 
    '''

    invert_vertex_group_clump = None
    '''Negate the effect of the clump vertex group 

    :type:  boolean, default False 
    '''

    invert_vertex_group_density = None
    '''Negate the effect of the density vertex group 

    :type:  boolean, default False 
    '''

    invert_vertex_group_field = None
    '''Negate the effect of the field vertex group 

    :type:  boolean, default False 
    '''

    invert_vertex_group_kink = None
    '''Negate the effect of the kink vertex group 

    :type:  boolean, default False 
    '''

    invert_vertex_group_length = None
    '''Negate the effect of the length vertex group 

    :type:  boolean, default False 
    '''

    invert_vertex_group_rotation = None
    '''Negate the effect of the rotation vertex group 

    :type:  boolean, default False 
    '''

    invert_vertex_group_roughness_1 = None
    '''Negate the effect of the roughness 1 vertex group 

    :type:  boolean, default False 
    '''

    invert_vertex_group_roughness_2 = None
    '''Negate the effect of the roughness 2 vertex group 

    :type:  boolean, default False 
    '''

    invert_vertex_group_roughness_end = None
    '''Negate the effect of the roughness end vertex group 

    :type:  boolean, default False 
    '''

    invert_vertex_group_size = None
    '''Negate the effect of the size vertex group 

    :type:  boolean, default False 
    '''

    invert_vertex_group_tangent = None
    '''Negate the effect of the tangent vertex group 

    :type:  boolean, default False 
    '''

    invert_vertex_group_velocity = None
    '''Negate the effect of the velocity vertex group 

    :type:  boolean, default False 
    '''

    is_editable = None
    '''Particle system can be edited in particle mode 

    :type:  boolean, default False, (readonly) 
    '''

    is_edited = None
    '''Particle system has been edited in particle mode 

    :type:  boolean, default False, (readonly) 
    '''

    is_global_hair = None
    '''Hair keys are in global coordinate space 

    :type:  boolean, default False, (readonly) 
    '''

    name = None
    '''Particle system name 

    :type:  string, default “”, (never None) 
    '''

    parent = None
    '''Use this object’s coordinate system instead of global coordinate system 

    :type:  Object 
    '''

    particles = None
    '''Particles generated by the particle system 

    :type:  bpy_prop_collection of Particle, (readonly) 
    '''

    point_cache = None
    '''

    :type:  PointCache, (readonly, never None) 
    '''

    reactor_target_object = None
    '''For reactor systems, the object that has the target particle system (empty if same object) 

    :type:  Object 
    '''

    reactor_target_particle_system = None
    '''For reactor systems, index of particle system on the target object 

    :type:  int in [1, 32767], default 0 
    '''

    seed = None
    '''Offset in the random number table, to get a different randomized result 

    :type:  int in [0, inf], default 0 
    '''

    settings = None
    '''Particle system settings 

    :type:  ParticleSettings, (never None) 
    '''

    targets = None
    '''Target particle systems 

    :type:  bpy_prop_collection of ParticleTarget, (readonly) 
    '''

    use_hair_dynamics = None
    '''Enable hair dynamics using cloth simulation 

    :type:  boolean, default False 
    '''

    use_keyed_timing = None
    '''Use key times 

    :type:  boolean, default False 
    '''

    vertex_group_clump = None
    '''Vertex group to control clump 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_density = None
    '''Vertex group to control density 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_field = None
    '''Vertex group to control field 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_kink = None
    '''Vertex group to control kink 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_length = None
    '''Vertex group to control length 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_rotation = None
    '''Vertex group to control rotation 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_roughness_1 = None
    '''Vertex group to control roughness 1 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_roughness_2 = None
    '''Vertex group to control roughness 2 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_roughness_end = None
    '''Vertex group to control roughness end 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_size = None
    '''Vertex group to control size 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_tangent = None
    '''Vertex group to control tangent 

    :type:  string, default “”, (never None) 
    '''

    vertex_group_velocity = None
    '''Vertex group to control velocity 

    :type:  string, default “”, (never None) 
    '''

    def set_resolution(self, scene=None, object=None, resolution='PREVIEW'):
        '''Set the resolution to use for the number of particles 

        :param scene: Scene 
        :type scene: Scene, (optional)
        :param object: Object 
        :type object: Object, (optional)
        :param resolution: Resolution settings to applyPREVIEW Preview, Apply modifier preview settings.RENDER Render, Apply modifier render settings. 
        :type resolution: enum in ['PREVIEW', 'RENDER'], (optional)
        '''
        pass

    def co_hair(self, object, particle_no=0, step=0):
        '''Obtain cache hair data 

        :param object: Object 
        :type object: Object, (never None)
        :param particle_no: Particle no 
        :type particle_no: int in [-inf, inf], (optional)
        :param step: step no 
        :type step: int in [-inf, inf], (optional)
        :rtype: float array of 3 items in [-inf, inf] 
        :return:  Co, Exported hairkey location 
        '''
        pass

    def uv_on_emitter(self, modifier, particle, particle_no=0, uv_no=0):
        '''Obtain uv for all particles 

        :param modifier: Particle modifier 
        :type modifier: ParticleSystemModifier, (never None)
        :param particle: Particle 
        :type particle: Particle, (never None)
        :param particle_no: Particle no 
        :type particle_no: int in [-inf, inf], (optional)
        :param uv_no: UV no 
        :type uv_no: int in [-inf, inf], (optional)
        :rtype: float array of 2 items in [-inf, inf] 
        :return:  uv 
        '''
        pass

    def mcol_on_emitter(self, modifier, particle, particle_no=0, vcol_no=0):
        '''Obtain mcol for all particles 

        :param modifier: Particle modifier 
        :type modifier: ParticleSystemModifier, (never None)
        :param particle: Particle 
        :type particle: Particle, (never None)
        :param particle_no: Particle no 
        :type particle_no: int in [-inf, inf], (optional)
        :param vcol_no: vcol no 
        :type vcol_no: int in [-inf, inf], (optional)
        :rtype: float array of 3 items in [0, inf] 
        :return:  mcol 
        '''
        pass


class ParticleSystemModifier:
    '''Particle system simulation modifier '''

    particle_system = None
    '''Particle System that this modifier controls 

    :type:  ParticleSystem, (readonly, never None) 
    '''


class ParticleSystems:
    '''Collection of particle systems '''

    active = None
    '''Active particle system being displayed 

    :type:  ParticleSystem, (readonly) 
    '''

    active_index = None
    '''Index of active particle system slot 

    :type:  int in [0, inf], default 0 
    '''


class ParticleTarget:
    '''Target particle system '''

    alliance = None
    '''

    :type:  enum in [‘FRIEND’, ‘NEUTRAL’, ‘ENEMY’], default ‘NEUTRAL’ 
    '''

    duration = None
    '''

    :type:  float in [0, 30000], default 0.0 
    '''

    is_valid = None
    '''Keyed particles target is valid 

    :type:  boolean, default False 
    '''

    name = None
    '''Particle target name 

    :type:  string, default “”, (readonly, never None) 
    '''

    object = None
    '''The object that has the target particle system (empty if same object) 

    :type:  Object 
    '''

    system = None
    '''The index of particle system on the target object 

    :type:  int in [1, inf], default 0 
    '''

    time = None
    '''

    :type:  float in [0, 30000], default 0.0 
    '''


class PathCompare:
    '''Match paths against this value '''

    path = None
    '''

    :type:  string, default “”, (never None) 
    '''

    use_glob = None
    '''Enable wildcard globbing 

    :type:  boolean, default False 
    '''


class PathCompareCollection:
    '''Collection of paths '''

    pass


class PivotConstraint:
    '''Rotate around a different point '''

    head_tail = None
    '''Target along length of bone: Head=0, Tail=1 

    :type:  float in [0, 1], default 0.0 
    '''

    offset = None
    '''Offset of pivot from target (when set), or from owner’s location (when Fixed Position is off), or the absolute pivot point 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    rotation_range = None
    '''Rotation range on which pivoting should occur 

    :type:  enum in [‘ALWAYS_ACTIVE’, ‘NX’, ‘NY’, ‘NZ’, ‘X’, ‘Y’, ‘Z’], default ‘NX’ 
    '''

    subtarget = None
    '''

    :type:  string, default “”, (never None) 
    '''

    target = None
    '''Target Object, defining the position of the pivot when defined 

    :type:  Object 
    '''

    use_bbone_shape = None
    '''Follow shape of B-Bone segments when calculating Head/Tail position 

    :type:  boolean, default False 
    '''

    use_relative_location = None
    '''Offset will be an absolute point in space instead of relative to the target 

    :type:  boolean, default False 
    '''


class PointCache:
    '''Point cache for physics simulations '''

    compression = None
    '''Compression method to be used 

    :type:  enum in [‘NO’, ‘LIGHT’, ‘HEAVY’], default ‘NO’ 
    '''

    filepath = None
    '''Cache file path 

    :type:  string, default “”, (never None) 
    '''

    frame_end = None
    '''Frame on which the simulation stops 

    :type:  int in [1, 500000], default 0 
    '''

    frame_start = None
    '''Frame on which the simulation starts 

    :type:  int in [-500000, 500000], default 0 
    '''

    frame_step = None
    '''Number of frames between cached frames 

    :type:  int in [1, 20], default 0 
    '''

    index = None
    '''Index number of cache files 

    :type:  int in [-1, 100], default 0 
    '''

    info = None
    '''Info on current cache status 

    :type:  string, default “”, (readonly, never None) 
    '''

    is_baked = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    is_baking = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    is_frame_skip = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    is_outdated = None
    '''

    :type:  boolean, default False, (readonly) 
    '''

    name = None
    '''Cache name 

    :type:  string, default “”, (never None) 
    '''

    point_caches = None
    '''Point cache list 

    :type:  PointCaches bpy_prop_collection of PointCache, (readonly) 
    '''

    use_disk_cache = None
    '''Save cache files to disk (.blend file must be saved first) 

    :type:  boolean, default False 
    '''

    use_external = None
    '''Read cache from an external location 

    :type:  boolean, default False 
    '''

    use_library_path = None
    '''Use this file’s path for the disk cache when library linked into another file (for local bakes per scene file, disable this option) 

    :type:  boolean, default False 
    '''


class PointCaches:
    '''Collection of point caches '''

    active_index = None
    '''

    :type:  int in [0, inf], default 0 
    '''


class PointDensity:
    '''Point density settings '''

    color_ramp = None
    '''

    :type:  ColorRamp, (readonly) 
    '''

    falloff = None
    '''Method of attenuating density by distance from the point 

    :type:  enum in [‘STANDARD’, ‘SMOOTH’, ‘SOFT’, ‘CONSTANT’, ‘ROOT’, ‘PARTICLE_AGE’, ‘PARTICLE_VELOCITY’], default ‘STANDARD’ 
    '''

    falloff_curve = None
    '''

    :type:  CurveMapping, (readonly) 
    '''

    falloff_soft = None
    '''Softness of the ‘soft’ falloff option 

    :type:  float in [0.01, inf], default 0.0 
    '''

    falloff_speed_scale = None
    '''Multiplier to bring particle speed within an acceptable range 

    :type:  float in [0.001, 100], default 0.0 
    '''

    noise_basis = None
    '''Noise formula used for turbulence 

    :type:  enum in [‘BLENDER_ORIGINAL’, ‘ORIGINAL_PERLIN’, ‘IMPROVED_PERLIN’, ‘VORONOI_F1’, ‘VORONOI_F2’, ‘VORONOI_F3’, ‘VORONOI_F4’, ‘VORONOI_F2_F1’, ‘VORONOI_CRACKLE’, ‘CELL_NOISE’], default ‘BLENDER_ORIGINAL’ 
    '''

    object = None
    '''Object to take point data from 

    :type:  Object 
    '''

    particle_cache_space = None
    '''Coordinate system to cache particles in 

    :type:  enum in [‘OBJECT_LOCATION’, ‘OBJECT_SPACE’, ‘WORLD_SPACE’], default ‘OBJECT_LOCATION’ 
    '''

    particle_color_source = None
    '''Data to derive color results from 

    :type:  enum in [‘CONSTANT’, ‘PARTICLE_AGE’, ‘PARTICLE_SPEED’, ‘PARTICLE_VELOCITY’], default ‘CONSTANT’ 
    '''

    particle_system = None
    '''Particle System to render as points 

    :type:  ParticleSystem 
    '''

    point_source = None
    '''Point data to use as renderable point density 

    :type:  enum in [‘PARTICLE_SYSTEM’, ‘OBJECT’], default ‘PARTICLE_SYSTEM’ 
    '''

    radius = None
    '''Radius from the shaded sample to look for points within 

    :type:  float in [0.001, inf], default 0.0 
    '''

    speed_scale = None
    '''Multiplier to bring particle speed within an acceptable range 

    :type:  float in [0.001, 100], default 0.0 
    '''

    turbulence_depth = None
    '''Level of detail in the added turbulent noise 

    :type:  int in [0, 30], default 0 
    '''

    turbulence_influence = None
    '''Method for driving added turbulent noise 

    :type:  enum in [‘STATIC’, ‘PARTICLE_VELOCITY’, ‘PARTICLE_AGE’, ‘GLOBAL_TIME’], default ‘STATIC’ 
    '''

    turbulence_scale = None
    '''Scale of the added turbulent noise 

    :type:  float in [0.01, inf], default 0.0 
    '''

    turbulence_strength = None
    '''Strength of the added turbulent noise 

    :type:  float in [0.01, inf], default 0.0 
    '''

    use_falloff_curve = None
    '''Use a custom falloff curve 

    :type:  boolean, default False 
    '''

    use_turbulence = None
    '''Add directed noise to the density at render-time 

    :type:  boolean, default False 
    '''

    vertex_attribute_name = None
    '''Vertex attribute to use for color 

    :type:  string, default “”, (never None) 
    '''

    vertex_cache_space = None
    '''Coordinate system to cache vertices in 

    :type:  enum in [‘OBJECT_LOCATION’, ‘OBJECT_SPACE’, ‘WORLD_SPACE’], default ‘OBJECT_LOCATION’ 
    '''

    vertex_color_source = None
    '''Data to derive color results from 

    :type:  enum in [‘CONSTANT’, ‘VERTEX_COLOR’, ‘VERTEX_WEIGHT’, ‘VERTEX_NORMAL’], default ‘CONSTANT’ 
    '''


class PointDensityTexture:
    '''Settings for the Point Density texture '''

    point_density = None
    '''The point density settings associated with this texture 

    :type:  PointDensity, (readonly) 
    '''

    users_material = None
    '''Materials that use this texture (readonly) '''

    users_object_modifier = None
    '''Object modifiers that use this texture (readonly) '''


class PointLamp:
    '''Omnidirectional point lamp '''

    compression_threshold = None
    '''Deep shadow map compression threshold 

    :type:  float in [0, 1], default 0.0 
    '''

    constant_coefficient = None
    '''Constant distance attenuation coefficient 

    :type:  float in [0, inf], default 0.0 
    '''

    falloff_curve = None
    '''Custom Lamp Falloff Curve 

    :type:  CurveMapping, (readonly) 
    '''

    falloff_type = None
    '''Intensity Decay with distance 

    :type:  enum in [‘CONSTANT’, ‘INVERSE_LINEAR’, ‘INVERSE_SQUARE’, ‘INVERSE_COEFFICIENTS’, ‘CUSTOM_CURVE’, ‘LINEAR_QUADRATIC_WEIGHTED’], default ‘CONSTANT’ 
    '''

    ge_shadow_buffer_type = None
    '''The shadow mapping algorithm used 

    :type:  enum in [‘SIMPLE’, ‘VARIANCE’], default ‘SIMPLE’ 
    '''

    linear_attenuation = None
    '''Linear distance attenuation 

    :type:  float in [0, 1], default 0.0 
    '''

    linear_coefficient = None
    '''Linear distance attenuation coefficient 

    :type:  float in [0, inf], default 0.0 
    '''

    quadratic_attenuation = None
    '''Quadratic distance attenuation 

    :type:  float in [0, 1], default 0.0 
    '''

    quadratic_coefficient = None
    '''Quadratic distance attenuation coefficient 

    :type:  float in [0, inf], default 0.0 
    '''

    shadow_adaptive_threshold = None
    '''Threshold for Adaptive Sampling (Raytraced shadows) 

    :type:  float in [0, 1], default 0.0 
    '''

    shadow_buffer_bias = None
    '''Shadow buffer sampling bias 

    :type:  float in [0.001, 5], default 0.0 
    '''

    shadow_buffer_bleed_bias = None
    '''Bias for reducing light-bleed on variance shadow maps 

    :type:  float in [0, 1], default 0.0 
    '''

    shadow_buffer_clip_end = None
    '''Shadow map clip end, beyond which objects will not generate shadows 

    :type:  float in [0, 9999], default 0.0 
    '''

    shadow_buffer_clip_start = None
    '''Shadow map clip start, below which objects will not generate shadows 

    :type:  float in [0, 9999], default 0.0 
    '''

    shadow_buffer_samples = None
    '''Number of shadow buffer samples 

    :type:  int in [1, 16], default 0 
    '''

    shadow_buffer_size = None
    '''Resolution of the shadow buffer, higher values give crisper shadows but use more memory 

    :type:  int in [128, 10240], default 0 
    '''

    shadow_buffer_soft = None
    '''Size of shadow buffer sampling area 

    :type:  float in [0, 100], default 0.0 
    '''

    shadow_buffer_type = None
    '''Type of shadow buffer 

    :type:  enum in [‘REGULAR’, ‘HALFWAY’, ‘IRREGULAR’, ‘DEEP’], default ‘REGULAR’ 
    '''

    shadow_color = None
    '''Color of shadows cast by the lamp 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    shadow_filter_type = None
    '''Type of shadow filter (Buffer Shadows) 

    :type:  enum in [‘BOX’, ‘TENT’, ‘GAUSS’], default ‘BOX’ 
    '''

    shadow_method = None
    '''

    :type:  enum in [‘NOSHADOW’, ‘RAY_SHADOW’], default ‘NOSHADOW’ 
    '''

    shadow_ray_sample_method = None
    '''Method for generating shadow samples: Adaptive QMC is fastest, Constant QMC is less noisy but slower 

    :type:  enum in [‘ADAPTIVE_QMC’, ‘CONSTANT_QMC’], default ‘ADAPTIVE_QMC’ 
    '''

    shadow_ray_samples = None
    '''Number of samples taken extra (samples x samples) 

    :type:  int in [1, 64], default 0 
    '''

    shadow_sample_buffers = None
    '''Number of shadow buffers to render for better AA, this increases memory usage 

    :type:  enum in [‘BUFFERS_1’, ‘BUFFERS_4’, ‘BUFFERS_9’], default ‘BUFFERS_1’ 
    '''

    shadow_soft_size = None
    '''Light size for ray shadow sampling (Raytraced shadows) 

    :type:  float in [0, inf], default 0.0 
    '''

    use_auto_clip_end = None
    '''Automatic calculation of clipping-end, based on visible vertices 

    :type:  boolean, default False 
    '''

    use_auto_clip_start = None
    '''Automatic calculation of clipping-start, based on visible vertices 

    :type:  boolean, default False 
    '''

    use_only_shadow = None
    '''Cast shadows only, without illuminating objects 

    :type:  boolean, default False 
    '''

    use_shadow = None
    '''

    :type:  boolean, default False 
    '''

    use_shadow_layer = None
    '''Objects on the same layers only cast shadows 

    :type:  boolean, default False 
    '''

    use_sphere = None
    '''Set light intensity to zero beyond lamp distance 

    :type:  boolean, default False 
    '''


class PointerProperty:
    '''RNA pointer property to point to another RNA struct '''

    fixed_type = None
    '''Fixed pointer type, empty if variable type 

    :type:  Struct, (readonly) 
    '''


class PolygonFloatProperties:
    '''Collection of float properties '''

    def new(self, name="Float Prop"):
        '''Add a float property layer to Mesh 

        :param name: Float property name 
        :type name: string, (optional, never None)
        :rtype: MeshPolygonFloatPropertyLayer 
        :return:  The newly created layer 
        '''
        pass


class PolygonIntProperties:
    '''Collection of int properties '''

    def new(self, name="Int Prop"):
        '''Add a integer property layer to Mesh 

        :param name: Int property name 
        :type name: string, (optional, never None)
        :rtype: MeshPolygonIntPropertyLayer 
        :return:  The newly created layer 
        '''
        pass


class PolygonStringProperties:
    '''Collection of string properties '''

    def new(self, name="String Prop"):
        '''Add a string property layer to Mesh 

        :param name: String property name 
        :type name: string, (optional, never None)
        :rtype: MeshPolygonStringPropertyLayer 
        :return:  The newly created layer 
        '''
        pass


class Pose:
    '''A collection of pose channels, including settings for animating bones '''

    animation_visualization = None
    '''Animation data for this data-block 

    :type:  AnimViz, (readonly, never None) 
    '''

    bone_groups = None
    '''Groups of the bones 

    :type:  BoneGroups bpy_prop_collection of BoneGroup, (readonly) 
    '''

    bones = None
    '''Individual pose bones for the armature 

    :type:  bpy_prop_collection of PoseBone, (readonly) 
    '''

    ik_param = None
    '''Parameters for IK solver 

    :type:  IKParam, (readonly) 
    '''

    ik_solver = None
    '''Selection of IK solver for IK chain 

    :type:  enum in [‘LEGACY’, ‘ITASC’], default ‘LEGACY’ 
    '''


class PoseBone:
    '''Channel defining pose data for a bone in a Pose '''

    bbone_curveinx = None
    '''X-axis handle offset for start of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_curveiny = None
    '''Y-axis handle offset for start of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_curveoutx = None
    '''X-axis handle offset for end of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_curveouty = None
    '''Y-axis handle offset for end of the B-Bone’s curve, adjusts curvature 

    :type:  float in [-5, 5], default 0.0 
    '''

    bbone_custom_handle_end = None
    '''Bone that serves as the end handle for the B-Bone curve 

    :type:  PoseBone 
    '''

    bbone_custom_handle_start = None
    '''Bone that serves as the start handle for the B-Bone curve 

    :type:  PoseBone 
    '''

    bbone_rollin = None
    '''Roll offset for the start of the B-Bone, adjusts twist 

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    bbone_rollout = None
    '''Roll offset for the end of the B-Bone, adjusts twist 

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    bbone_scalein = None
    '''Scale factor for start of the B-Bone, adjusts thickness (for tapering effects) 

    :type:  float in [0, 5], default 1.0 
    '''

    bbone_scaleout = None
    '''Scale factor for end of the B-Bone, adjusts thickness (for tapering effects) 

    :type:  float in [0, 5], default 1.0 
    '''

    bone = None
    '''Bone associated with this PoseBone 

    :type:  Bone, (readonly, never None) 
    '''

    bone_group = None
    '''Bone Group this pose channel belongs to 

    :type:  BoneGroup 
    '''

    bone_group_index = None
    '''Bone Group this pose channel belongs to (0=no group) 

    :type:  int in [-32768, 32767], default 0 
    '''

    child = None
    '''Child of this pose bone 

    :type:  PoseBone, (readonly) 
    '''

    constraints = None
    '''Constraints that act on this PoseChannel 

    :type:  PoseBoneConstraints bpy_prop_collection of Constraint, (readonly) 
    '''

    custom_shape = None
    '''Object that defines custom draw type for this bone 

    :type:  Object 
    '''

    custom_shape_scale = None
    '''Adjust the size of the custom shape 

    :type:  float in [0, 1000], default 0.0 
    '''

    custom_shape_transform = None
    '''Bone that defines the display transform of this custom shape 

    :type:  PoseBone 
    '''

    head = None
    '''Location of head of the channel’s bone 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0), (readonly) 
    '''

    ik_linear_weight = None
    '''Weight of scale constraint for IK 

    :type:  float in [0, 1], default 0.0 
    '''

    ik_max_x = None
    '''Maximum angles for IK Limit 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    ik_max_y = None
    '''Maximum angles for IK Limit 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    ik_max_z = None
    '''Maximum angles for IK Limit 

    :type:  float in [0, 3.14159], default 0.0 
    '''

    ik_min_x = None
    '''Minimum angles for IK Limit 

    :type:  float in [-3.14159, 0], default 0.0 
    '''

    ik_min_y = None
    '''Minimum angles for IK Limit 

    :type:  float in [-3.14159, 0], default 0.0 
    '''

    ik_min_z = None
    '''Minimum angles for IK Limit 

    :type:  float in [-3.14159, 0], default 0.0 
    '''

    ik_rotation_weight = None
    '''Weight of rotation constraint for IK 

    :type:  float in [0, 1], default 0.0 
    '''

    ik_stiffness_x = None
    '''IK stiffness around the X axis 

    :type:  float in [0, 0.99], default 0.0 
    '''

    ik_stiffness_y = None
    '''IK stiffness around the Y axis 

    :type:  float in [0, 0.99], default 0.0 
    '''

    ik_stiffness_z = None
    '''IK stiffness around the Z axis 

    :type:  float in [0, 0.99], default 0.0 
    '''

    ik_stretch = None
    '''Allow scaling of the bone for IK 

    :type:  float in [0, 1], default 0.0 
    '''

    is_in_ik_chain = None
    '''Is part of an IK chain 

    :type:  boolean, default False, (readonly) 
    '''

    location = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    lock_ik_x = None
    '''Disallow movement around the X axis 

    :type:  boolean, default False 
    '''

    lock_ik_y = None
    '''Disallow movement around the Y axis 

    :type:  boolean, default False 
    '''

    lock_ik_z = None
    '''Disallow movement around the Z axis 

    :type:  boolean, default False 
    '''

    lock_location = None
    '''Lock editing of location in the interface 

    :type:  boolean array of 3 items, default (False, False, False) 
    '''

    lock_rotation = None
    '''Lock editing of rotation in the interface 

    :type:  boolean array of 3 items, default (False, False, False) 
    '''

    lock_rotation_w = None
    '''Lock editing of ‘angle’ component of four-component rotations in the interface 

    :type:  boolean, default False 
    '''

    lock_rotations_4d = None
    '''Lock editing of four component rotations by components (instead of as Eulers) 

    :type:  boolean, default False 
    '''

    lock_scale = None
    '''Lock editing of scale in the interface 

    :type:  boolean array of 3 items, default (False, False, False) 
    '''

    matrix = None
    '''Final 4x4 matrix after constraints and drivers are applied (object space) 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    matrix_basis = None
    '''Alternative access to location/scale/rotation relative to the parent and own rest bone 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    matrix_channel = None
    '''4x4 matrix, before constraints 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0), (readonly) 
    '''

    motion_path = None
    '''Motion Path for this element 

    :type:  MotionPath, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (never None) 
    '''

    parent = None
    '''Parent of this pose bone 

    :type:  PoseBone, (readonly) 
    '''

    rotation_axis_angle = None
    '''Angle of Rotation for Axis-Angle rotation representation 

    :type:  float array of 4 items in [-inf, inf], default (0.0, 0.0, 1.0, 0.0) 
    '''

    rotation_euler = None
    '''Rotation in Eulers 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    rotation_mode = None
    '''

    :type:  enum in [‘QUATERNION’, ‘XYZ’, ‘XZY’, ‘YXZ’, ‘YZX’, ‘ZXY’, ‘ZYX’, ‘AXIS_ANGLE’], default ‘QUATERNION’ 
    '''

    rotation_quaternion = None
    '''Rotation in Quaternions 

    :type:  float array of 4 items in [-inf, inf], default (1.0, 0.0, 0.0, 0.0) 
    '''

    scale = None
    '''

    :type:  float array of 3 items in [-inf, inf], default (1.0, 1.0, 1.0) 
    '''

    tail = None
    '''Location of tail of the channel’s bone 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0), (readonly) 
    '''

    use_bbone_custom_handles = None
    '''Use custom reference bones as handles for B-Bones instead of next/previous bones, leave these blank to use only B-Bone offset properties to control the shape 

    :type:  boolean, default False 
    '''

    use_bbone_relative_end_handle = None
    '''Treat custom end handle position as a relative value 

    :type:  boolean, default False 
    '''

    use_bbone_relative_start_handle = None
    '''Treat custom start handle position as a relative value 

    :type:  boolean, default False 
    '''

    use_custom_shape_bone_size = None
    '''Scale the custom object by the bone length 

    :type:  boolean, default False 
    '''

    use_ik_limit_x = None
    '''Limit movement around the X axis 

    :type:  boolean, default False 
    '''

    use_ik_limit_y = None
    '''Limit movement around the Y axis 

    :type:  boolean, default False 
    '''

    use_ik_limit_z = None
    '''Limit movement around the Z axis 

    :type:  boolean, default False 
    '''

    use_ik_linear_control = None
    '''Apply channel size as IK constraint if stretching is enabled 

    :type:  boolean, default False 
    '''

    use_ik_rotation_control = None
    '''Apply channel rotation as IK constraint 

    :type:  boolean, default False 
    '''

    basename = None
    '''The name of this bone before any ‘.’ character (readonly) '''

    center = None
    '''The midpoint between the head and the tail. (readonly) '''

    children = None
    '''(readonly) '''

    children_recursive = None
    '''A list of all children from this bone. (readonly) '''

    children_recursive_basename = None
    '''Returns a chain of children with the same base name as this bone. Only direct chains are supported, forks caused by multiple children with matching base names will terminate the function and not be returned. (readonly) '''

    length = None
    '''The distance from head to tail, when set the head is moved to fit the length. '''

    parent_recursive = None
    '''A list of parents, starting with the immediate parent (readonly) '''

    vector = None
    '''The direction this bone is pointing. Utility function for (tail - head) (readonly) '''

    x_axis = None
    '''Vector pointing down the x-axis of the bone. (readonly) '''

    y_axis = None
    '''Vector pointing down the y-axis of the bone. (readonly) '''

    z_axis = None
    '''Vector pointing down the z-axis of the bone. (readonly) '''

    def evaluate_envelope(self, point):
        '''Calculate bone envelope at given point 

        :param point: Point, Position in 3d space to evaluate 
        :type point: float array of 3 items in [-inf, inf]
        :rtype: float in [-inf, inf] 
        :return:  Factor, Envelope factor 
        '''
        pass

    def parent_index(self, parent_test):
        '''The same as ‘bone in other_bone.parent_recursive’ but saved generating a list. 

        '''
        pass

    def translate(self, vec):
        '''Utility function to add vec to the head and tail of this bone 

        '''
        pass


class PoseBoneConstraints:
    '''Collection of pose bone constraints '''

    active = None
    '''Active PoseChannel constraint 

    :type:  Constraint 
    '''

    def new(self, type):
        '''Add a constraint to this object 

        :param type: Constraint type to addCAMERA_SOLVER Camera Solver.FOLLOW_TRACK Follow Track.OBJECT_SOLVER Object Solver.COPY_LOCATION Copy Location, Copy the location of a target (with an optional offset), so that they move together.COPY_ROTATION Copy Rotation, Copy the rotation of a target (with an optional offset), so that they rotate together.COPY_SCALE Copy Scale, Copy the scale factors of a target (with an optional offset), so that they are scaled by the same amount.COPY_TRANSFORMS Copy Transforms, Copy all the transformations of a target, so that they move together.LIMIT_DISTANCE Limit Distance, Restrict movements to within a certain distance of a target (at the time of constraint evaluation only).LIMIT_LOCATION Limit Location, Restrict movement along each axis within given ranges.LIMIT_ROTATION Limit Rotation, Restrict rotation along each axis within given ranges.LIMIT_SCALE Limit Scale, Restrict scaling along each axis with given ranges.MAINTAIN_VOLUME Maintain Volume, Compensate for scaling one axis by applying suitable scaling to the other two axes.TRANSFORM Transformation, Use one transform property from target to control another (or same) property on owner.TRANSFORM_CACHE Transform Cache, Look up the transformation matrix from an external file.CLAMP_TO Clamp To, Restrict movements to lie along a curve by remapping location along curve’s longest axis.DAMPED_TRACK Damped Track, Point towards a target by performing the smallest rotation necessary.IK Inverse Kinematics, Control a chain of bones by specifying the endpoint target (Bones only).LOCKED_TRACK Locked Track, Rotate around the specified (‘locked’) axis to point towards a target.SPLINE_IK Spline IK, Align chain of bones along a curve (Bones only).STRETCH_TO Stretch To, Stretch along Y-Axis to point towards a target.TRACK_TO Track To, Legacy tracking constraint prone to twisting artifacts.ACTION Action, Use transform property of target to look up pose for owner from an Action.CHILD_OF Child Of, Make target the ‘detachable’ parent of owner.FLOOR Floor, Use position (and optionally rotation) of target to define a ‘wall’ or ‘floor’ that the owner can not cross.FOLLOW_PATH Follow Path, Use to animate an object/bone following a path.PIVOT Pivot, Change pivot point for transforms (buggy).RIGID_BODY_JOINT Rigid Body Joint, Use to define a Rigid Body Constraint (for Game Engine use only).SHRINKWRAP Shrinkwrap, Restrict movements to surface of target mesh. 
        :type type: enum in ['CAMERA_SOLVER', 'FOLLOW_TRACK', 'OBJECT_SOLVER', 'COPY_LOCATION', 'COPY_ROTATION', 'COPY_SCALE', 'COPY_TRANSFORMS', 'LIMIT_DISTANCE', 'LIMIT_LOCATION', 'LIMIT_ROTATION', 'LIMIT_SCALE', 'MAINTAIN_VOLUME', 'TRANSFORM', 'TRANSFORM_CACHE', 'CLAMP_TO', 'DAMPED_TRACK', 'IK', 'LOCKED_TRACK', 'SPLINE_IK', 'STRETCH_TO', 'TRACK_TO', 'ACTION', 'CHILD_OF', 'FLOOR', 'FOLLOW_PATH', 'PIVOT', 'RIGID_BODY_JOINT', 'SHRINKWRAP']
        :rtype: Constraint 
        :return:  New constraint 
        '''
        pass

    def remove(self, constraint):
        '''Remove a constraint from this object 

        :param constraint: Removed constraint 
        :type constraint: Constraint, (never None)
        '''
        pass


class Property:
    '''RNA property definition '''

    description = None
    '''Description of the property for tooltips 

    :type:  string, default “”, (readonly, never None) 
    '''

    icon = None
    '''Icon of the item 

    :type:  enum in [‘NONE’, ‘QUESTION’, ‘ERROR’, ‘CANCEL’, ‘TRIA_RIGHT’, ‘TRIA_DOWN’, ‘TRIA_LEFT’, ‘TRIA_UP’, ‘ARROW_LEFTRIGHT’, ‘PLUS’, ‘DISCLOSURE_TRI_DOWN’, ‘DISCLOSURE_TRI_RIGHT’, ‘RADIOBUT_OFF’, ‘RADIOBUT_ON’, ‘MENU_PANEL’, ‘BLENDER’, ‘GRIP’, ‘DOT’, ‘COLLAPSEMENU’, ‘X’, ‘GO_LEFT’, ‘PLUG’, ‘UI’, ‘NODE’, ‘NODE_SEL’, ‘FULLSCREEN’, ‘SPLITSCREEN’, ‘RIGHTARROW_THIN’, ‘BORDERMOVE’, ‘VIEWZOOM’, ‘ZOOMIN’, ‘ZOOMOUT’, ‘PANEL_CLOSE’, ‘COPY_ID’, ‘EYEDROPPER’, ‘LINK_AREA’, ‘AUTO’, ‘CHECKBOX_DEHLT’, ‘CHECKBOX_HLT’, ‘UNLOCKED’, ‘LOCKED’, ‘UNPINNED’, ‘PINNED’, ‘SCREEN_BACK’, ‘RIGHTARROW’, ‘DOWNARROW_HLT’, ‘DOTSUP’, ‘DOTSDOWN’, ‘LINK’, ‘INLINK’, ‘PLUGIN’, ‘HELP’, ‘GHOST_ENABLED’, ‘COLOR’, ‘LINKED’, ‘UNLINKED’, ‘HAND’, ‘ZOOM_ALL’, ‘ZOOM_SELECTED’, ‘ZOOM_PREVIOUS’, ‘ZOOM_IN’, ‘ZOOM_OUT’, ‘RENDER_REGION’, ‘BORDER_RECT’, ‘BORDER_LASSO’, ‘FREEZE’, ‘STYLUS_PRESSURE’, ‘GHOST_DISABLED’, ‘NEW’, ‘FILE_TICK’, ‘QUIT’, ‘URL’, ‘RECOVER_LAST’, ‘FULLSCREEN_ENTER’, ‘FULLSCREEN_EXIT’, ‘BLANK1’, ‘LAMP’, ‘MATERIAL’, ‘TEXTURE’, ‘ANIM’, ‘WORLD’, ‘SCENE’, ‘EDIT’, ‘GAME’, ‘RADIO’, ‘SCRIPT’, ‘PARTICLES’, ‘PHYSICS’, ‘SPEAKER’, ‘TEXTURE_SHADED’, ‘VIEW3D’, ‘IPO’, ‘OOPS’, ‘BUTS’, ‘FILESEL’, ‘IMAGE_COL’, ‘INFO’, ‘SEQUENCE’, ‘TEXT’, ‘IMASEL’, ‘SOUND’, ‘ACTION’, ‘NLA’, ‘SCRIPTWIN’, ‘TIME’, ‘NODETREE’, ‘LOGIC’, ‘CONSOLE’, ‘PREFERENCES’, ‘CLIP’, ‘ASSET_MANAGER’, ‘OBJECT_DATAMODE’, ‘EDITMODE_HLT’, ‘FACESEL_HLT’, ‘VPAINT_HLT’, ‘TPAINT_HLT’, ‘WPAINT_HLT’, ‘SCULPTMODE_HLT’, ‘POSE_HLT’, ‘PARTICLEMODE’, ‘LIGHTPAINT’, ‘SCENE_DATA’, ‘RENDERLAYERS’, ‘WORLD_DATA’, ‘OBJECT_DATA’, ‘MESH_DATA’, ‘CURVE_DATA’, ‘META_DATA’, ‘LATTICE_DATA’, ‘LAMP_DATA’, ‘MATERIAL_DATA’, ‘TEXTURE_DATA’, ‘ANIM_DATA’, ‘CAMERA_DATA’, ‘PARTICLE_DATA’, ‘LIBRARY_DATA_DIRECT’, ‘GROUP’, ‘ARMATURE_DATA’, ‘POSE_DATA’, ‘BONE_DATA’, ‘CONSTRAINT’, ‘SHAPEKEY_DATA’, ‘CONSTRAINT_BONE’, ‘CAMERA_STEREO’, ‘PACKAGE’, ‘UGLYPACKAGE’, ‘BRUSH_DATA’, ‘IMAGE_DATA’, ‘FILE’, ‘FCURVE’, ‘FONT_DATA’, ‘RENDER_RESULT’, ‘SURFACE_DATA’, ‘EMPTY_DATA’, ‘SETTINGS’, ‘RENDER_ANIMATION’, ‘RENDER_STILL’, ‘LIBRARY_DATA_BROKEN’, ‘BOIDS’, ‘STRANDS’, ‘LIBRARY_DATA_INDIRECT’, ‘GREASEPENCIL’, ‘LINE_DATA’, ‘GROUP_BONE’, ‘GROUP_VERTEX’, ‘GROUP_VCOL’, ‘GROUP_UVS’, ‘RNA’, ‘RNA_ADD’, ‘OUTLINER_OB_EMPTY’, ‘OUTLINER_OB_MESH’, ‘OUTLINER_OB_CURVE’, ‘OUTLINER_OB_LATTICE’, ‘OUTLINER_OB_META’, ‘OUTLINER_OB_LAMP’, ‘OUTLINER_OB_CAMERA’, ‘OUTLINER_OB_ARMATURE’, ‘OUTLINER_OB_FONT’, ‘OUTLINER_OB_SURFACE’, ‘OUTLINER_OB_SPEAKER’, ‘RESTRICT_COLOR_OFF’, ‘RESTRICT_COLOR_ON’, ‘RESTRICT_VIEW_OFF’, ‘RESTRICT_VIEW_ON’, ‘RESTRICT_SELECT_OFF’, ‘RESTRICT_SELECT_ON’, ‘RESTRICT_RENDER_OFF’, ‘RESTRICT_RENDER_ON’, ‘OUTLINER_DATA_EMPTY’, ‘OUTLINER_DATA_MESH’, ‘OUTLINER_DATA_CURVE’, ‘OUTLINER_DATA_LATTICE’, ‘OUTLINER_DATA_META’, ‘OUTLINER_DATA_LAMP’, ‘OUTLINER_DATA_CAMERA’, ‘OUTLINER_DATA_ARMATURE’, ‘OUTLINER_DATA_FONT’, ‘OUTLINER_DATA_SURFACE’, ‘OUTLINER_DATA_SPEAKER’, ‘OUTLINER_DATA_POSE’, ‘MESH_PLANE’, ‘MESH_CUBE’, ‘MESH_CIRCLE’, ‘MESH_UVSPHERE’, ‘MESH_ICOSPHERE’, ‘MESH_GRID’, ‘MESH_MONKEY’, ‘MESH_CYLINDER’, ‘MESH_TORUS’, ‘MESH_CONE’, ‘MESH_CAPSULE’, ‘LAMP_POINT’, ‘LAMP_SUN’, ‘LAMP_SPOT’, ‘LAMP_HEMI’, ‘LAMP_AREA’, ‘META_EMPTY’, ‘META_PLANE’, ‘META_CUBE’, ‘META_BALL’, ‘META_ELLIPSOID’, ‘META_CAPSULE’, ‘SURFACE_NCURVE’, ‘SURFACE_NCIRCLE’, ‘SURFACE_NSURFACE’, ‘SURFACE_NCYLINDER’, ‘SURFACE_NSPHERE’, ‘SURFACE_NTORUS’, ‘CURVE_BEZCURVE’, ‘CURVE_BEZCIRCLE’, ‘CURVE_NCURVE’, ‘CURVE_NCIRCLE’, ‘CURVE_PATH’, ‘COLOR_RED’, ‘COLOR_GREEN’, ‘COLOR_BLUE’, ‘TRIA_RIGHT_BAR’, ‘TRIA_DOWN_BAR’, ‘TRIA_LEFT_BAR’, ‘TRIA_UP_BAR’, ‘FORCE_FORCE’, ‘FORCE_WIND’, ‘FORCE_VORTEX’, ‘FORCE_MAGNETIC’, ‘FORCE_HARMONIC’, ‘FORCE_CHARGE’, ‘FORCE_LENNARDJONES’, ‘FORCE_TEXTURE’, ‘FORCE_CURVE’, ‘FORCE_BOID’, ‘FORCE_TURBULENCE’, ‘FORCE_DRAG’, ‘FORCE_SMOKEFLOW’, ‘NODE_INSERT_ON’, ‘NODE_INSERT_OFF’, ‘MODIFIER’, ‘MOD_WAVE’, ‘MOD_BUILD’, ‘MOD_DECIM’, ‘MOD_MIRROR’, ‘MOD_SOFT’, ‘MOD_SUBSURF’, ‘HOOK’, ‘MOD_PHYSICS’, ‘MOD_PARTICLES’, ‘MOD_BOOLEAN’, ‘MOD_EDGESPLIT’, ‘MOD_ARRAY’, ‘MOD_UVPROJECT’, ‘MOD_DISPLACE’, ‘MOD_CURVE’, ‘MOD_LATTICE’, ‘CONSTRAINT_DATA’, ‘MOD_ARMATURE’, ‘MOD_SHRINKWRAP’, ‘MOD_CAST’, ‘MOD_MESHDEFORM’, ‘MOD_BEVEL’, ‘MOD_SMOOTH’, ‘MOD_SIMPLEDEFORM’, ‘MOD_MASK’, ‘MOD_CLOTH’, ‘MOD_EXPLODE’, ‘MOD_FLUIDSIM’, ‘MOD_MULTIRES’, ‘MOD_SMOKE’, ‘MOD_SOLIDIFY’, ‘MOD_SCREW’, ‘MOD_VERTEX_WEIGHT’, ‘MOD_DYNAMICPAINT’, ‘MOD_REMESH’, ‘MOD_OCEAN’, ‘MOD_WARP’, ‘MOD_SKIN’, ‘MOD_TRIANGULATE’, ‘MOD_WIREFRAME’, ‘MOD_DATA_TRANSFER’, ‘MOD_NORMALEDIT’, ‘REC’, ‘PLAY’, ‘FF’, ‘REW’, ‘PAUSE’, ‘PREV_KEYFRAME’, ‘NEXT_KEYFRAME’, ‘PLAY_AUDIO’, ‘PLAY_REVERSE’, ‘PREVIEW_RANGE’, ‘ACTION_TWEAK’, ‘PMARKER_ACT’, ‘PMARKER_SEL’, ‘PMARKER’, ‘MARKER_HLT’, ‘MARKER’, ‘SPACE2’, ‘SPACE3’, ‘KEYINGSET’, ‘KEY_DEHLT’, ‘KEY_HLT’, ‘MUTE_IPO_OFF’, ‘MUTE_IPO_ON’, ‘VISIBLE_IPO_OFF’, ‘VISIBLE_IPO_ON’, ‘DRIVER’, ‘SOLO_OFF’, ‘SOLO_ON’, ‘FRAME_PREV’, ‘FRAME_NEXT’, ‘NLA_PUSHDOWN’, ‘IPO_CONSTANT’, ‘IPO_LINEAR’, ‘IPO_BEZIER’, ‘IPO_SINE’, ‘IPO_QUAD’, ‘IPO_CUBIC’, ‘IPO_QUART’, ‘IPO_QUINT’, ‘IPO_EXPO’, ‘IPO_CIRC’, ‘IPO_BOUNCE’, ‘IPO_ELASTIC’, ‘IPO_BACK’, ‘IPO_EASE_IN’, ‘IPO_EASE_OUT’, ‘IPO_EASE_IN_OUT’, ‘VERTEXSEL’, ‘EDGESEL’, ‘FACESEL’, ‘LOOPSEL’, ‘ROTATE’, ‘CURSOR’, ‘ROTATECOLLECTION’, ‘ROTATECENTER’, ‘ROTACTIVE’, ‘ALIGN’, ‘SMOOTHCURVE’, ‘SPHERECURVE’, ‘ROOTCURVE’, ‘SHARPCURVE’, ‘LINCURVE’, ‘NOCURVE’, ‘RNDCURVE’, ‘PROP_OFF’, ‘PROP_ON’, ‘PROP_CON’, ‘SCULPT_DYNTOPO’, ‘PARTICLE_POINT’, ‘PARTICLE_TIP’, ‘PARTICLE_PATH’, ‘MAN_TRANS’, ‘MAN_ROT’, ‘MAN_SCALE’, ‘MANIPUL’, ‘SNAP_OFF’, ‘SNAP_ON’, ‘SNAP_NORMAL’, ‘SNAP_GRID’, ‘SNAP_VERTEX’, ‘SNAP_EDGE’, ‘SNAP_FACE’, ‘SNAP_VOLUME’, ‘SNAP_INCREMENT’, ‘STICKY_UVS_LOC’, ‘STICKY_UVS_DISABLE’, ‘STICKY_UVS_VERT’, ‘CLIPUV_DEHLT’, ‘CLIPUV_HLT’, ‘SNAP_PEEL_OBJECT’, ‘GRID’, ‘PASTEDOWN’, ‘COPYDOWN’, ‘PASTEFLIPUP’, ‘PASTEFLIPDOWN’, ‘SNAP_SURFACE’, ‘AUTOMERGE_ON’, ‘AUTOMERGE_OFF’, ‘RETOPO’, ‘UV_VERTEXSEL’, ‘UV_EDGESEL’, ‘UV_FACESEL’, ‘UV_ISLANDSEL’, ‘UV_SYNC_SELECT’, ‘BBOX’, ‘WIRE’, ‘SOLID’, ‘SMOOTH’, ‘POTATO’, ‘ORTHO’, ‘LOCKVIEW_OFF’, ‘LOCKVIEW_ON’, ‘AXIS_SIDE’, ‘AXIS_FRONT’, ‘AXIS_TOP’, ‘NDOF_DOM’, ‘NDOF_TURN’, ‘NDOF_FLY’, ‘NDOF_TRANS’, ‘LAYER_USED’, ‘LAYER_ACTIVE’, ‘SORTALPHA’, ‘SORTBYEXT’, ‘SORTTIME’, ‘SORTSIZE’, ‘LONGDISPLAY’, ‘SHORTDISPLAY’, ‘GHOST’, ‘IMGDISPLAY’, ‘SAVE_AS’, ‘SAVE_COPY’, ‘BOOKMARKS’, ‘FONTPREVIEW’, ‘FILTER’, ‘NEWFOLDER’, ‘OPEN_RECENT’, ‘FILE_PARENT’, ‘FILE_REFRESH’, ‘FILE_FOLDER’, ‘FILE_BLANK’, ‘FILE_BLEND’, ‘FILE_IMAGE’, ‘FILE_MOVIE’, ‘FILE_SCRIPT’, ‘FILE_SOUND’, ‘FILE_FONT’, ‘FILE_TEXT’, ‘RECOVER_AUTO’, ‘SAVE_PREFS’, ‘LINK_BLEND’, ‘APPEND_BLEND’, ‘IMPORT’, ‘EXPORT’, ‘EXTERNAL_DATA’, ‘LOAD_FACTORY’, ‘LOOP_BACK’, ‘LOOP_FORWARDS’, ‘BACK’, ‘FORWARD’, ‘FILE_HIDDEN’, ‘FILE_BACKUP’, ‘DISK_DRIVE’, ‘MATPLANE’, ‘MATSPHERE’, ‘MATCUBE’, ‘MONKEY’, ‘HAIR’, ‘ALIASED’, ‘ANTIALIASED’, ‘MAT_SPHERE_SKY’, ‘WORDWRAP_OFF’, ‘WORDWRAP_ON’, ‘SYNTAX_OFF’, ‘SYNTAX_ON’, ‘LINENUMBERS_OFF’, ‘LINENUMBERS_ON’, ‘SCRIPTPLUGINS’, ‘SEQ_SEQUENCER’, ‘SEQ_PREVIEW’, ‘SEQ_LUMA_WAVEFORM’, ‘SEQ_CHROMA_SCOPE’, ‘SEQ_HISTOGRAM’, ‘SEQ_SPLITVIEW’, ‘IMAGE_RGB’, ‘IMAGE_RGB_ALPHA’, ‘IMAGE_ALPHA’, ‘IMAGE_ZDEPTH’, ‘IMAGEFILE’, ‘BRUSH_ADD’, ‘BRUSH_BLOB’, ‘BRUSH_BLUR’, ‘BRUSH_CLAY’, ‘BRUSH_CLAY_STRIPS’, ‘BRUSH_CLONE’, ‘BRUSH_CREASE’, ‘BRUSH_DARKEN’, ‘BRUSH_FILL’, ‘BRUSH_FLATTEN’, ‘BRUSH_GRAB’, ‘BRUSH_INFLATE’, ‘BRUSH_LAYER’, ‘BRUSH_LIGHTEN’, ‘BRUSH_MASK’, ‘BRUSH_MIX’, ‘BRUSH_MULTIPLY’, ‘BRUSH_NUDGE’, ‘BRUSH_PINCH’, ‘BRUSH_SCRAPE’, ‘BRUSH_SCULPT_DRAW’, ‘BRUSH_SMEAR’, ‘BRUSH_SMOOTH’, ‘BRUSH_SNAKE_HOOK’, ‘BRUSH_SOFTEN’, ‘BRUSH_SUBTRACT’, ‘BRUSH_TEXDRAW’, ‘BRUSH_TEXFILL’, ‘BRUSH_TEXMASK’, ‘BRUSH_THUMB’, ‘BRUSH_ROTATE’, ‘BRUSH_VERTEXDRAW’, ‘MATCAP_01’, ‘MATCAP_02’, ‘MATCAP_03’, ‘MATCAP_04’, ‘MATCAP_05’, ‘MATCAP_06’, ‘MATCAP_07’, ‘MATCAP_08’, ‘MATCAP_09’, ‘MATCAP_10’, ‘MATCAP_11’, ‘MATCAP_12’, ‘MATCAP_13’, ‘MATCAP_14’, ‘MATCAP_15’, ‘MATCAP_16’, ‘MATCAP_17’, ‘MATCAP_18’, ‘MATCAP_19’, ‘MATCAP_20’, ‘MATCAP_21’, ‘MATCAP_22’, ‘MATCAP_23’, ‘MATCAP_24’, ‘VIEW3D_VEC’, ‘EDIT_VEC’, ‘EDITMODE_VEC_DEHLT’, ‘EDITMODE_VEC_HLT’, ‘DISCLOSURE_TRI_RIGHT_VEC’, ‘DISCLOSURE_TRI_DOWN_VEC’, ‘MOVE_UP_VEC’, ‘MOVE_DOWN_VEC’, ‘X_VEC’, ‘SMALL_TRI_RIGHT_VEC’, ‘KEYTYPE_KEYFRAME_VEC’, ‘KEYTYPE_BREAKDOWN_VEC’, ‘KEYTYPE_EXTREME_VEC’, ‘KEYTYPE_JITTER_VEC’, ‘KEYTYPE_MOVING_HOLD_VEC’, ‘COLORSET_01_VEC’, ‘COLORSET_02_VEC’, ‘COLORSET_03_VEC’, ‘COLORSET_04_VEC’, ‘COLORSET_05_VEC’, ‘COLORSET_06_VEC’, ‘COLORSET_07_VEC’, ‘COLORSET_08_VEC’, ‘COLORSET_09_VEC’, ‘COLORSET_10_VEC’, ‘COLORSET_11_VEC’, ‘COLORSET_12_VEC’, ‘COLORSET_13_VEC’, ‘COLORSET_14_VEC’, ‘COLORSET_15_VEC’, ‘COLORSET_16_VEC’, ‘COLORSET_17_VEC’, ‘COLORSET_18_VEC’, ‘COLORSET_19_VEC’, ‘COLORSET_20_VEC’], default ‘NONE’, (readonly) 
    '''

    identifier = None
    '''Unique name used in the code and scripting 

    :type:  string, default “”, (readonly, never None) 
    '''

    is_animatable = None
    '''Property is animatable through RNA 

    :type:  boolean, default False, (readonly) 
    '''

    is_argument_optional = None
    '''True when the property is optional in a Python function implementing an RNA function 

    :type:  boolean, default False, (readonly) 
    '''

    is_enum_flag = None
    '''True when multiple enums 

    :type:  boolean, default False, (readonly) 
    '''

    is_hidden = None
    '''True when the property is hidden 

    :type:  boolean, default False, (readonly) 
    '''

    is_library_editable = None
    '''Property is editable from linked instances (changes not saved) 

    :type:  boolean, default False, (readonly) 
    '''

    is_never_none = None
    '''True when this value can’t be set to None 

    :type:  boolean, default False, (readonly) 
    '''

    is_output = None
    '''True when this property is an output value from an RNA function 

    :type:  boolean, default False, (readonly) 
    '''

    is_readonly = None
    '''Property is editable through RNA 

    :type:  boolean, default False, (readonly) 
    '''

    is_registered = None
    '''Property is registered as part of type registration 

    :type:  boolean, default False, (readonly) 
    '''

    is_registered_optional = None
    '''Property is optionally registered as part of type registration 

    :type:  boolean, default False, (readonly) 
    '''

    is_required = None
    '''False when this property is an optional argument in an RNA function 

    :type:  boolean, default False, (readonly) 
    '''

    is_runtime = None
    '''Property has been dynamically created at runtime 

    :type:  boolean, default False, (readonly) 
    '''

    is_skip_save = None
    '''True when the property is not saved in presets 

    :type:  boolean, default False, (readonly) 
    '''

    name = None
    '''Human readable name 

    :type:  string, default “”, (readonly, never None) 
    '''

    srna = None
    '''Struct definition used for properties assigned to this item 

    :type:  Struct, (readonly) 
    '''

    subtype = None
    '''Semantic interpretation of the property 

    :type:  enum in [‘NONE’, ‘FILE_PATH’, ‘DIR_PATH’, ‘PIXEL’, ‘UNSIGNED’, ‘PERCENTAGE’, ‘FACTOR’, ‘ANGLE’, ‘TIME’, ‘DISTANCE’, ‘COLOR’, ‘TRANSLATION’, ‘DIRECTION’, ‘MATRIX’, ‘EULER’, ‘QUATERNION’, ‘XYZ’, ‘COLOR_GAMMA’, ‘COORDINATES’, ‘LAYER’, ‘LAYER_MEMBERSHIP’], default ‘NONE’, (readonly) 
    '''

    translation_context = None
    '''Translation context of the property’s name 

    :type:  string, default “”, (readonly, never None) 
    '''

    type = None
    '''Data type of the property 

    :type:  enum in [‘BOOLEAN’, ‘INT’, ‘FLOAT’, ‘STRING’, ‘ENUM’, ‘POINTER’, ‘COLLECTION’], default ‘BOOLEAN’, (readonly) 
    '''

    unit = None
    '''Type of units for this property 

    :type:  enum in [‘NONE’, ‘LENGTH’, ‘AREA’, ‘VOLUME’, ‘ROTATION’, ‘TIME’, ‘VELOCITY’, ‘ACCELERATION’, ‘CAMERA’], default ‘NONE’, (readonly) 
    '''


class PropertyActuator:
    '''Actuator to handle properties '''

    mode = None
    '''

    :type:  enum in [‘ASSIGN’, ‘ADD’, ‘COPY’, ‘TOGGLE’, ‘LEVEL’], default ‘ASSIGN’ 
    '''

    object = None
    '''Copy from this Object 

    :type:  Object 
    '''

    object_property = None
    '''Copy this property 

    :type:  string, default “”, (never None) 
    '''

    property = None
    '''The name of the property 

    :type:  string, default “”, (never None) 
    '''

    value = None
    '''The name of the property or the value to use (use “” around strings) 

    :type:  string, default “”, (never None) 
    '''


class PropertyGroup:
    '''Group of ID properties '''

    name = None
    '''Unique name used in the code and scripting 

    :type:  string, default “”, (never None) 
    '''


class PropertyGroupItem:
    '''Property that stores arbitrary, user defined properties '''

    collection = None
    '''

    :type:  bpy_prop_collection of PropertyGroup, (readonly) 
    '''

    double = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    double_array = None
    '''

    :type:  float array of 1 items in [-inf, inf], default (0.0) 
    '''

    float = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    float_array = None
    '''

    :type:  float array of 1 items in [-inf, inf], default (0.0) 
    '''

    group = None
    '''

    :type:  PropertyGroup, (readonly) 
    '''

    idp_array = None
    '''

    :type:  bpy_prop_collection of PropertyGroup, (readonly) 
    '''

    int = None
    '''

    :type:  int in [-inf, inf], default 0 
    '''

    int_array = None
    '''

    :type:  int array of 1 items in [-inf, inf], default (0,) 
    '''

    string = None
    '''

    :type:  string, default “”, (never None) 
    '''


class PropertySensor:
    '''Sensor to detect values and changes in values of properties '''

    evaluation_type = None
    '''Type of property evaluation 

    :type:  enum in [‘PROPEQUAL’, ‘PROPNEQUAL’, ‘PROPINTERVAL’, ‘PROPCHANGED’, ‘PROPLESSTHAN’, ‘PROPGREATERTHAN’], default ‘PROPEQUAL’ 
    '''

    property = None
    '''

    :type:  string, default “”, (never None) 
    '''

    value = None
    '''Check for this value in types in Equal, Not Equal, Less Than and Greater Than types 

    :type:  string, default “”, (never None) 
    '''

    value_max = None
    '''Maximum value in Interval type 

    :type:  string, default “”, (never None) 
    '''

    value_min = None
    '''Minimum value in Interval type 

    :type:  string, default “”, (never None) 
    '''


class PythonConstraint:
    '''Use Python script for constraint evaluation '''

    has_script_error = None
    '''The linked Python script has thrown an error 

    :type:  boolean, default False, (readonly) 
    '''

    target_count = None
    '''Usually only 1-3 are needed 

    :type:  int in [-inf, inf], default 0 
    '''

    targets = None
    '''Target Objects 

    :type:  bpy_prop_collection of ConstraintTarget, (readonly) 
    '''

    text = None
    '''The text object that contains the Python script 

    :type:  Text 
    '''

    use_targets = None
    '''Use the targets indicated in the constraint panel 

    :type:  boolean, default False 
    '''


class PythonController:
    '''Controller executing a python script '''

    mode = None
    '''Python script type (textblock or module - faster) 

    :type:  enum in [‘SCRIPT’, ‘MODULE’], default ‘SCRIPT’ 
    '''

    module = None
    '''Module name and function to run, e.g. “someModule.main” (internal texts and external python files can be used) 

    :type:  string, default “”, (never None) 
    '''

    text = None
    '''Text data-block with the python script 

    :type:  Text 
    '''

    use_debug = None
    '''Continuously reload the module from disk for editing external modules without restarting 

    :type:  boolean, default False 
    '''


class RENDERLAYER_UL_linesets:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class RENDERLAYER_UL_renderlayers:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class RENDERLAYER_UL_renderviews:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class RadarSensor:
    '''Sensor to detect objects in a cone shaped radar emanating from the current object '''

    angle = None
    '''Opening angle of the radar cone 

    :type:  float in [0, 3.13985], default 0.0 
    '''

    axis = None
    '''Along which axis the radar cone is cast 

    :type:  enum in [‘XAXIS’, ‘YAXIS’, ‘ZAXIS’, ‘NEGXAXIS’, ‘NEGYAXIS’, ‘NEGZAXIS’], default ‘XAXIS’ 
    '''

    distance = None
    '''Depth of the radar cone 

    :type:  float in [0, 10000], default 0.0 
    '''

    property = None
    '''Only look for objects with this property (blank = all objects) 

    :type:  string, default “”, (never None) 
    '''


class RandomActuator:
    chance = None
    '''Pick a number between 0 and 1, success if it’s below this value 

    :type:  float in [0, 1], default 0.0 
    '''

    distribution = None
    '''Choose the type of distribution 

    :type:  enum in [‘BOOL_CONSTANT’, ‘BOOL_UNIFORM’, ‘BOOL_BERNOUILLI’, ‘INT_CONSTANT’, ‘INT_UNIFORM’, ‘INT_POISSON’, ‘FLOAT_CONSTANT’, ‘FLOAT_UNIFORM’, ‘FLOAT_NORMAL’, ‘FLOAT_NEGATIVE_EXPONENTIAL’], default ‘BOOL_CONSTANT’ 
    '''

    float_max = None
    '''Choose a number from a range: upper boundary of the range 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    float_mean = None
    '''A normal distribution: mean of the distribution 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    float_min = None
    '''Choose a number from a range: lower boundary of the range 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    float_value = None
    '''Always return this number 

    :type:  float in [0, 1], default 0.0 
    '''

    half_life_time = None
    '''Negative exponential dropoff 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    int_max = None
    '''Choose a number from a range: upper boundary of the range 

    :type:  int in [-1000, 1000], default 0 
    '''

    int_mean = None
    '''Expected mean value of the distribution 

    :type:  float in [0.01, 100], default 0.0 
    '''

    int_min = None
    '''Choose a number from a range: lower boundary of the range 

    :type:  int in [-1000, 1000], default 0 
    '''

    int_value = None
    '''Always return this number 

    :type:  int in [-inf, inf], default 0 
    '''

    property = None
    '''Assign the random value to this property 

    :type:  string, default “”, (never None) 
    '''

    seed = None
    '''Initial seed of the random generator, use Python for more freedom (choose 0 for not random) 

    :type:  int in [0, 500000], default 0 
    '''

    standard_derivation = None
    '''A normal distribution: standard deviation of the distribution 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    use_always_true = None
    '''Always false or always true 

    :type:  boolean, default False 
    '''


class RandomSensor:
    '''Sensor to send random events '''

    seed = None
    '''Initial seed of the generator (choose 0 for not random) 

    :type:  int in [0, 1000], default 0 
    '''


class RaySensor:
    '''Sensor to detect intersections with a ray emanating from the current object '''

    axis = None
    '''Along which axis the ray is cast 

    :type:  enum in [‘XAXIS’, ‘YAXIS’, ‘ZAXIS’, ‘NEGXAXIS’, ‘NEGYAXIS’, ‘NEGZAXIS’], default ‘YAXIS’ 
    '''

    material = None
    '''Only look for objects with this material (blank = all objects) 

    :type:  string, default “”, (never None) 
    '''

    property = None
    '''Only look for objects with this property (blank = all objects) 

    :type:  string, default “”, (never None) 
    '''

    range = None
    '''Sense objects no farther than this distance 

    :type:  float in [0.01, 10000], default 0.0 
    '''

    ray_type = None
    '''Toggle collision on material or property 

    :type:  enum in [‘PROPERTY’, ‘MATERIAL’], default ‘PROPERTY’ 
    '''

    use_x_ray = None
    '''Toggle X-Ray option (see through objects that don’t have the property) 

    :type:  boolean, default False 
    '''


class Region:
    '''Region in a subdivided screen area '''

    height = None
    '''Region height 

    :type:  int in [0, 32767], default 0, (readonly) 
    '''

    id = None
    '''Unique ID for this region 

    :type:  int in [-32768, 32767], default 0, (readonly) 
    '''

    type = None
    '''Type of this region 

    :type:  enum in [‘WINDOW’, ‘HEADER’, ‘CHANNELS’, ‘TEMPORARY’, ‘UI’, ‘TOOLS’, ‘TOOL_PROPS’, ‘PREVIEW’], default ‘WINDOW’, (readonly) 
    '''

    view2d = None
    '''2D view of the region 

    :type:  View2D, (readonly, never None) 
    '''

    width = None
    '''Region width 

    :type:  int in [0, 32767], default 0, (readonly) 
    '''

    x = None
    '''The window relative vertical location of the region 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    y = None
    '''The window relative horizontal location of the region 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    def tag_redraw(self):
        '''tag_redraw 

        '''
        pass


class RegionView3D:
    '''3D View region data '''

    is_perspective = None
    '''

    :type:  boolean, default False 
    '''

    lock_rotation = None
    '''Lock view rotation in side views 

    :type:  boolean, default False 
    '''

    perspective_matrix = None
    '''Current perspective matrix (window_matrix * view_matrix) 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0), (readonly) 
    '''

    show_sync_view = None
    '''Sync view position between side views 

    :type:  boolean, default False 
    '''

    use_box_clip = None
    '''Clip objects based on what’s visible in other side views 

    :type:  boolean, default False 
    '''

    view_camera_offset = None
    '''View shift in camera view 

    :type:  float array of 2 items in [-inf, inf], default (0.0, 0.0) 
    '''

    view_camera_zoom = None
    '''Zoom factor in camera view 

    :type:  float in [-30, 600], default 0.0 
    '''

    view_distance = None
    '''Distance to the view location 

    :type:  float in [0, inf], default 0.0 
    '''

    view_location = None
    '''View pivot location 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    view_matrix = None
    '''Current view matrix 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) 
    '''

    view_perspective = None
    '''View Perspective 

    :type:  enum in [‘PERSP’, ‘ORTHO’, ‘CAMERA’], default ‘ORTHO’ 
    '''

    view_rotation = None
    '''Rotation in quaternions (keep normalized) 

    :type:  float array of 4 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0) 
    '''

    window_matrix = None
    '''Current window matrix 

    :type:  float array of 16 items in [-inf, inf], default (0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0), (readonly) 
    '''

    def update(self):
        '''Recalculate the view matrices 

        '''
        pass


class RemeshModifier:
    '''Generate a new surface with regular topology that follows the shape of the input mesh '''

    mode = None
    '''

    :type:  enum in [‘BLOCKS’, ‘SMOOTH’, ‘SHARP’], default ‘BLOCKS’ 
    '''

    octree_depth = None
    '''Resolution of the octree; higher values give finer details 

    :type:  int in [1, 12], default 0 
    '''

    scale = None
    '''The ratio of the largest dimension of the model over the size of the grid 

    :type:  float in [0, 0.99], default 0.0 
    '''

    sharpness = None
    '''Tolerance for outliers; lower values filter noise while higher values will reproduce edges closer to the input 

    :type:  float in [-inf, inf], default 0.0 
    '''

    threshold = None
    '''If removing disconnected pieces, minimum size of components to preserve as a ratio of the number of polygons in the largest component 

    :type:  float in [0, 1], default 0.0 
    '''

    use_remove_disconnected = None
    '''

    :type:  boolean, default False 
    '''

    use_smooth_shade = None
    '''Output faces with smooth shading rather than flat shaded 

    :type:  boolean, default False 
    '''


class RenderEngine:
    '''Render engine '''

    bl_idname = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_label = None
    '''

    :type:  string, default “”, (never None) 
    '''

    bl_use_exclude_layers = None
    '''

    :type:  boolean, default False 
    '''

    bl_use_postprocess = None
    '''

    :type:  boolean, default False 
    '''

    bl_use_preview = None
    '''

    :type:  boolean, default False 
    '''

    bl_use_save_buffers = None
    '''

    :type:  boolean, default False 
    '''

    bl_use_shading_nodes = None
    '''

    :type:  boolean, default False 
    '''

    bl_use_shading_nodes_custom = None
    '''

    :type:  boolean, default True 
    '''

    bl_use_spherical_stereo = None
    '''

    :type:  boolean, default False 
    '''

    bl_use_texture_preview = None
    '''

    :type:  boolean, default False 
    '''

    camera_override = None
    '''

    :type:  Object, (readonly) 
    '''

    is_animation = None
    '''

    :type:  boolean, default False 
    '''

    is_preview = None
    '''

    :type:  boolean, default False 
    '''

    layer_override = None
    '''

    :type:  boolean array of 20 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    render = None
    '''

    :type:  RenderSettings, (readonly) 
    '''

    resolution_x = None
    '''

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    resolution_y = None
    '''

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    tile_x = None
    '''

    :type:  int in [0, inf], default 0 
    '''

    tile_y = None
    '''

    :type:  int in [0, inf], default 0 
    '''

    use_highlight_tiles = None
    '''

    :type:  boolean, default False 
    '''

    def update(self, data=None, scene=None):
        '''Export scene data for render 

        '''
        pass

    def render(self, scene=None):
        '''Render scene into an image 

        '''
        pass

    def bake(self, scene, object, pass_type, pass_filter, object_id,
             pixel_array, num_pixels, depth, result):
        '''Bake passes 

        :param pass_type: Pass, Pass to bake 
        :type pass_type: enum in ['COMBINED', 'AO', 'SHADOW', 'NORMAL', 'UV', 'EMIT', 'ENVIRONMENT', 'DIFFUSE', 'GLOSSY', 'TRANSMISSION', 'SUBSURFACE']
        :param pass_filter: Pass Filter, Filter to combined, diffuse, glossy, transmission and subsurface passes 
        :type pass_filter: int in [0, inf]
        :param object_id: Object Id, Id of the current object being baked in relation to the others 
        :type object_id: int in [0, inf]
        :param num_pixels: Number of Pixels, Size of the baking batch 
        :type num_pixels: int in [0, inf]
        :param depth: Pixels depth, Number of channels 
        :type depth: int in [0, inf]
        '''
        pass

    def view_update(self, context=None):
        '''Update on data changes for viewport render 

        '''
        pass

    def view_draw(self, context=None):
        '''Draw viewport render 

        '''
        pass

    def update_script_node(self, node=None):
        '''Compile shader script node 

        '''
        pass

    def tag_redraw(self):
        '''Request redraw for viewport rendering 

        '''
        pass

    def tag_update(self):
        '''Request update call for viewport rendering 

        '''
        pass

    def begin_result(self, x, y, w, h, layer="", view=""):
        '''Create render result to write linear floating point render layers and passes 

        :param x: X 
        :type x: int in [0, inf]
        :param y: Y 
        :type y: int in [0, inf]
        :param w: Width 
        :type w: int in [0, inf]
        :param h: Height 
        :type h: int in [0, inf]
        :param layer: Layer, Single layer to get render result for 
        :type layer: string, (optional, never None)
        :param view: View, Single view to get render result for 
        :type view: string, (optional, never None)
        :rtype: RenderResult 
        :return:  Result 
        '''
        pass

    def update_result(self, result):
        '''Signal that pixels have been updated and can be redrawn in the user interface 

        :param result: Result 
        :type result: RenderResult
        '''
        pass

    def end_result(self, result, cancel=False, do_merge_results=False):
        '''All pixels in the render result have been set and are final 

        :param result: Result 
        :type result: RenderResult
        :param cancel: Cancel, Don’t mark tile as done, don’t merge results unless forced 
        :type cancel: boolean, (optional)
        :param do_merge_results: Merge Results, Merge results even if cancel=true 
        :type do_merge_results: boolean, (optional)
        '''
        pass

    def test_break(self):
        '''Test if the render operation should been canceled, this is a fast call that should be used regularly for responsiveness 

        :rtype: boolean 
        :return:  Break 
        '''
        pass

    def active_view_get(self):
        '''active_view_get 

        :rtype: string, (never None) 
        :return:  View, Single view active 
        '''
        pass

    def active_view_set(self, view=""):
        '''active_view_set 

        :param view: View, Single view to set as active 
        :type view: string, (optional, never None)
        '''
        pass

    def camera_shift_x(self, camera, use_spherical_stereo=False):
        '''camera_shift_x 

        :param use_spherical_stereo: Spherical Stereo 
        :type use_spherical_stereo: boolean, (optional)
        :rtype: float in [0, inf] 
        :return:  Shift X 
        '''
        pass

    def camera_model_matrix(self,
                            camera,
                            use_spherical_stereo=False,
                            r_model_matrix):
        '''camera_model_matrix 

        :param use_spherical_stereo: Spherical Stereo 
        :type use_spherical_stereo: boolean, (optional)
        :param r_model_matrix: Model Matrix, Normalized camera model matrix 
        :type r_model_matrix: float array of 16 items in [-inf, inf]
        '''
        pass

    def use_spherical_stereo(self, camera):
        '''use_spherical_stereo 

        :rtype: boolean 
        :return:  Spherical Stereo 
        '''
        pass

    def update_stats(self, stats, info):
        '''Update and signal to redraw render status text 

        :param stats: Stats 
        :type stats: string, (never None)
        :param info: Info 
        :type info: string, (never None)
        '''
        pass

    def frame_set(self, frame, subframe):
        '''Evaluate scene at a different frame (for motion blur) 

        :param frame: Frame 
        :type frame: int in [-inf, inf]
        :param subframe: Subframe 
        :type subframe: float in [0, 1]
        '''
        pass

    def update_progress(self, progress):
        '''Update progress percentage of render 

        :param progress: Percentage of render that’s done 
        :type progress: float in [0, 1]
        '''
        pass

    def update_memory_stats(self, memory_used=0.0, memory_peak=0.0):
        '''Update memory usage statistics 

        :param memory_used: Current memory usage in megabytes 
        :type memory_used: float in [0, inf], (optional)
        :param memory_peak: Peak memory usage in megabytes 
        :type memory_peak: float in [0, inf], (optional)
        '''
        pass

    def report(self, type, message):
        '''Report info, warning or error messages 

        :param type: Type 
        :type type: enum set in {'DEBUG', 'INFO', 'OPERATOR', 'PROPERTY', 'WARNING', 'ERROR', 'ERROR_INVALID_INPUT', 'ERROR_INVALID_CONTEXT', 'ERROR_OUT_OF_MEMORY'}
        :param message: Report Message 
        :type message: string, (never None)
        '''
        pass

    def error_set(self, message):
        '''Set error message displaying after the render is finished 

        :param message: Report Message 
        :type message: string, (never None)
        '''
        pass

    def bind_display_space_shader(self, scene):
        '''Bind GLSL fragment shader that converts linear colors to display space colors using scene color management settings 

        '''
        pass

    def unbind_display_space_shader(self):
        '''Unbind GLSL display space shader, must always be called after binding the shader 

        '''
        pass

    def support_display_space_shader(self, scene):
        '''Test if GLSL display space shader is supported for the combination of graphics card and scene settings 

        :rtype: boolean 
        :return:  Supported 
        '''
        pass


class RenderLayer:
    exclude_ambient_occlusion = None
    '''Exclude AO pass from combined 

    :type:  boolean, default False, (readonly) 
    '''

    exclude_emit = None
    '''Exclude emission pass from combined 

    :type:  boolean, default False, (readonly) 
    '''

    exclude_environment = None
    '''Exclude environment pass from combined 

    :type:  boolean, default False, (readonly) 
    '''

    exclude_indirect = None
    '''Exclude indirect pass from combined 

    :type:  boolean, default False, (readonly) 
    '''

    exclude_reflection = None
    '''Exclude raytraced reflection pass from combined 

    :type:  boolean, default False, (readonly) 
    '''

    exclude_refraction = None
    '''Exclude raytraced refraction pass from combined 

    :type:  boolean, default False, (readonly) 
    '''

    exclude_shadow = None
    '''Exclude shadow pass from combined 

    :type:  boolean, default False, (readonly) 
    '''

    exclude_specular = None
    '''Exclude specular pass from combined 

    :type:  boolean, default False, (readonly) 
    '''

    invert_zmask = None
    '''For Zmask, only render what is behind solid z values instead of in front 

    :type:  boolean, default False, (readonly) 
    '''

    layers = None
    '''Scene layers included in this render layer 

    :type:  boolean array of 20 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False), (readonly) 
    '''

    layers_exclude = None
    '''Exclude scene layers from having any influence 

    :type:  boolean array of 20 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False), (readonly) 
    '''

    layers_zmask = None
    '''Zmask scene layers for solid faces 

    :type:  boolean array of 20 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False), (readonly) 
    '''

    light_override = None
    '''Group to override all other lights in this render layer 

    :type:  Group, (readonly) 
    '''

    material_override = None
    '''Material to override all other materials in this render layer 

    :type:  Material, (readonly) 
    '''

    name = None
    '''Render layer name 

    :type:  string, default “”, (readonly, never None) 
    '''

    passes = None
    '''

    :type:  RenderPasses bpy_prop_collection of RenderPass, (readonly) 
    '''

    use = None
    '''Disable or enable the render layer 

    :type:  boolean, default False, (readonly) 
    '''

    use_all_z = None
    '''Fill in Z values for solid faces in invisible layers, for masking 

    :type:  boolean, default False, (readonly) 
    '''

    use_ao = None
    '''Render AO in this Layer 

    :type:  boolean, default False, (readonly) 
    '''

    use_edge_enhance = None
    '''Render Edge-enhance in this Layer (only works for Solid faces) 

    :type:  boolean, default False, (readonly) 
    '''

    use_freestyle = None
    '''Render stylized strokes in this Layer 

    :type:  boolean, default False, (readonly) 
    '''

    use_halo = None
    '''Render Halos in this Layer (on top of Solid) 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_ambient_occlusion = None
    '''Deliver AO pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_color = None
    '''Deliver shade-less color pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_combined = None
    '''Deliver full combined RGBA buffer 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_diffuse = None
    '''Deliver diffuse pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_diffuse_color = None
    '''Deliver diffuse color pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_diffuse_direct = None
    '''Deliver diffuse direct pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_diffuse_indirect = None
    '''Deliver diffuse indirect pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_emit = None
    '''Deliver emission pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_environment = None
    '''Deliver environment lighting pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_glossy_color = None
    '''Deliver glossy color pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_glossy_direct = None
    '''Deliver glossy direct pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_glossy_indirect = None
    '''Deliver glossy indirect pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_indirect = None
    '''Deliver indirect lighting pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_material_index = None
    '''Deliver material index pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_mist = None
    '''Deliver mist factor pass (0.0-1.0) 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_normal = None
    '''Deliver normal pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_object_index = None
    '''Deliver object index pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_reflection = None
    '''Deliver raytraced reflection pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_refraction = None
    '''Deliver raytraced refraction pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_shadow = None
    '''Deliver shadow pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_specular = None
    '''Deliver specular pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_subsurface_color = None
    '''Deliver subsurface color pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_subsurface_direct = None
    '''Deliver subsurface direct pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_subsurface_indirect = None
    '''Deliver subsurface indirect pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_transmission_color = None
    '''Deliver transmission color pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_transmission_direct = None
    '''Deliver transmission direct pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_transmission_indirect = None
    '''Deliver transmission indirect pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_uv = None
    '''Deliver texture UV pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_vector = None
    '''Deliver speed vector pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_pass_z = None
    '''Deliver Z values pass 

    :type:  boolean, default False, (readonly) 
    '''

    use_sky = None
    '''Render Sky in this Layer 

    :type:  boolean, default False, (readonly) 
    '''

    use_solid = None
    '''Render Solid faces in this Layer 

    :type:  boolean, default False, (readonly) 
    '''

    use_strand = None
    '''Render Strands in this Layer 

    :type:  boolean, default False, (readonly) 
    '''

    use_zmask = None
    '''Only render what’s in front of the solid z values 

    :type:  boolean, default False, (readonly) 
    '''

    use_ztransp = None
    '''Render Z-Transparent faces in this Layer (on top of Solid and Halos) 

    :type:  boolean, default False, (readonly) 
    '''

    def load_from_file(self, filename, x=0, y=0):
        '''Copies the pixels of this renderlayer from an image file 

        :param filename: Filename, Filename to load into this render tile, must be no smaller than the renderlayer 
        :type filename: string, (never None)
        :param x: Offset X, Offset the position to copy from if the image is larger than the render layer 
        :type x: int in [0, inf], (optional)
        :param y: Offset Y, Offset the position to copy from if the image is larger than the render layer 
        :type y: int in [0, inf], (optional)
        '''
        pass


class RenderLayers:
    '''Collection of render layers '''

    active = None
    '''Active Render Layer 

    :type:  SceneRenderLayer, (never None) 
    '''

    active_index = None
    '''Active index in render layer array 

    :type:  int in [0, 32767], default 0 
    '''

    def new(self, name):
        '''Add a render layer to scene 

        :param name: New name for the render layer (not unique) 
        :type name: string, (never None)
        :rtype: SceneRenderLayer 
        :return:  Newly created render layer 
        '''
        pass

    def remove(self, layer):
        '''Remove a render layer 

        :param layer: Render layer to remove 
        :type layer: SceneRenderLayer, (never None)
        '''
        pass


class RenderPass:
    channel_id = None
    '''

    :type:  string, default “”, (readonly, never None) 
    '''

    channels = None
    '''

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    debug_type = None
    '''

    :type:  enum in [‘BVH_TRAVERSAL_STEPS’, ‘BVH_TRAVERSED_INSTANCES’, ‘RAY_BOUNCES’], default ‘BVH_TRAVERSAL_STEPS’, (readonly) 
    '''

    name = None
    '''

    :type:  string, default “”, (readonly, never None) 
    '''

    rect = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    type = None
    '''

    :type:  enum in [‘COMBINED’, ‘Z’, ‘COLOR’, ‘DIFFUSE’, ‘SPECULAR’, ‘SHADOW’, ‘AO’, ‘REFLECTION’, ‘NORMAL’, ‘VECTOR’, ‘REFRACTION’, ‘OBJECT_INDEX’, ‘UV’, ‘MIST’, ‘EMIT’, ‘ENVIRONMENT’, ‘MATERIAL_INDEX’, ‘DIFFUSE_DIRECT’, ‘DIFFUSE_INDIRECT’, ‘DIFFUSE_COLOR’, ‘GLOSSY_DIRECT’, ‘GLOSSY_INDIRECT’, ‘GLOSSY_COLOR’, ‘TRANSMISSION_DIRECT’, ‘TRANSMISSION_INDIRECT’, ‘TRANSMISSION_COLOR’, ‘SUBSURFACE_DIRECT’, ‘SUBSURFACE_INDIRECT’, ‘SUBSURFACE_COLOR’], default ‘COMBINED’, (readonly) 
    '''

    view_id = None
    '''

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''


class RenderPasses:
    '''Collection of render passes '''

    def find_by_type(self, pass_type, view):
        '''Get the render pass for a given type and view 

        :param pass_type: Pass 
        :type pass_type: enum in ['COMBINED', 'Z', 'COLOR', 'DIFFUSE', 'SPECULAR', 'SHADOW', 'AO', 'REFLECTION', 'NORMAL', 'VECTOR', 'REFRACTION', 'OBJECT_INDEX', 'UV', 'MIST', 'EMIT', 'ENVIRONMENT', 'MATERIAL_INDEX', 'DIFFUSE_DIRECT', 'DIFFUSE_INDIRECT', 'DIFFUSE_COLOR', 'GLOSSY_DIRECT', 'GLOSSY_INDIRECT', 'GLOSSY_COLOR', 'TRANSMISSION_DIRECT', 'TRANSMISSION_INDIRECT', 'TRANSMISSION_COLOR', 'SUBSURFACE_DIRECT', 'SUBSURFACE_INDIRECT', 'SUBSURFACE_COLOR']
        :param view: View, Render view to get pass from 
        :type view: string, (never None)
        :rtype: RenderPass 
        :return:  The matching render pass 
        '''
        pass


class RenderResult:
    '''Result of rendering, including all layers and passes '''

    layers = None
    '''

    :type:  bpy_prop_collection of RenderLayer, (readonly) 
    '''

    resolution_x = None
    '''

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    resolution_y = None
    '''

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    views = None
    '''

    :type:  bpy_prop_collection of RenderView, (readonly) 
    '''

    def load_from_file(self, filename):
        '''Copies the pixels of this render result from an image file 

        :param filename: File Name, Filename to load into this render tile, must be no smaller than the render result 
        :type filename: string, (never None)
        '''
        pass


class RenderSettings:
    '''Rendering settings for a Scene data-block '''

    alpha_mode = None
    '''Representation of alpha information in the RGBA pixels 

    :type:  enum in [‘SKY’, ‘TRANSPARENT’], default ‘SKY’ 
    '''

    antialiasing_samples = None
    '''Amount of anti-aliasing samples per pixel 

    :type:  enum in [‘5’, ‘8’, ‘11’, ‘16’], default ‘5’ 
    '''

    bake = None
    '''

    :type:  BakeSettings, (readonly, never None) 
    '''

    bake_aa_mode = None
    '''

    :type:  enum in [‘5’, ‘8’, ‘11’, ‘16’], default ‘5’ 
    '''

    bake_bias = None
    '''Bias towards faces further away from the object (in blender units) 

    :type:  float in [0, 1000], default 0.0 
    '''

    bake_distance = None
    '''Maximum distance from active object to other object (in blender units) 

    :type:  float in [0, 1000], default 0.0 
    '''

    bake_margin = None
    '''Extends the baked result as a post process filter 

    :type:  int in [0, 64], default 0 
    '''

    bake_normal_space = None
    '''Choose normal space for baking 

    :type:  enum in [‘CAMERA’, ‘WORLD’, ‘OBJECT’, ‘TANGENT’], default ‘CAMERA’ 
    '''

    bake_quad_split = None
    '''Choose the method used to split a quad into 2 triangles for baking 

    :type:  enum in [‘AUTO’, ‘FIXED’, ‘FIXED_ALT’], default ‘AUTO’ 
    '''

    bake_samples = None
    '''Number of samples used for ambient occlusion baking from multires 

    :type:  int in [64, 1024], default 256 
    '''

    bake_type = None
    '''Choose shading information to bake into the image 

    :type:  enum in [‘FULL’, ‘AO’, ‘SHADOW’, ‘NORMALS’, ‘TEXTURE’, ‘DISPLACEMENT’, ‘DERIVATIVE’, ‘VERTEX_COLORS’, ‘EMIT’, ‘ALPHA’, ‘MIRROR_INTENSITY’, ‘MIRROR_COLOR’, ‘SPEC_INTENSITY’, ‘SPEC_COLOR’], default ‘FULL’ 
    '''

    bake_user_scale = None
    '''Instead of automatically normalizing to 0..1, apply a user scale to the derivative map 

    :type:  float in [0, 1000], default 0.0 
    '''

    border_max_x = None
    '''Maximum X value for the render border 

    :type:  float in [0, 1], default 0.0 
    '''

    border_max_y = None
    '''Maximum Y value for the render border 

    :type:  float in [0, 1], default 0.0 
    '''

    border_min_x = None
    '''Minimum X value for the render border 

    :type:  float in [0, 1], default 0.0 
    '''

    border_min_y = None
    '''Minimum Y value for the render border 

    :type:  float in [0, 1], default 0.0 
    '''

    display_mode = None
    '''Select where rendered images will be displayed 

    :type:  enum in [‘SCREEN’, ‘AREA’, ‘WINDOW’, ‘NONE’], default ‘SCREEN’ 
    '''

    dither_intensity = None
    '''Amount of dithering noise added to the rendered image to break up banding 

    :type:  float in [0, 2], default 0.0 
    '''

    edge_color = None
    '''Edge color 

    :type:  float array of 3 items in [0, inf], default (0.0, 0.0, 0.0) 
    '''

    edge_threshold = None
    '''Threshold for drawing outlines on geometry edges 

    :type:  int in [0, 255], default 0 
    '''

    engine = None
    '''Engine to use for rendering 

    :type:  enum in [‘BLENDER_RENDER’], default ‘BLENDER_RENDER’ 
    '''

    ffmpeg = None
    '''FFmpeg related settings for the scene 

    :type:  FFmpegSettings, (readonly) 
    '''

    field_order = None
    '''Order of video fields (select which lines get rendered first, to create smooth motion for TV output) 

    :type:  enum in [‘EVEN_FIRST’, ‘ODD_FIRST’], default ‘EVEN_FIRST’ 
    '''

    file_extension = None
    '''The file extension used for saving renders 

    :type:  string, default “”, (readonly, never None) 
    '''

    filepath = None
    '''Directory/name to save animations, # characters defines the position and length of frame numbers 

    :type:  string, default “”, (never None) 
    '''

    filter_size = None
    '''Width over which the reconstruction filter combines samples 

    :type:  float in [0.5, 1.5], default 0.0 
    '''

    fps = None
    '''Framerate, expressed in frames per second 

    :type:  int in [1, 120], default 0 
    '''

    fps_base = None
    '''Framerate base 

    :type:  float in [0.1, 120], default 0.0 
    '''

    frame_map_new = None
    '''How many frames the Map Old will last 

    :type:  int in [1, 900], default 0 
    '''

    frame_map_old = None
    '''Old mapping value in frames 

    :type:  int in [1, 900], default 0 
    '''

    has_multiple_engines = None
    '''More than one rendering engine is available 

    :type:  boolean, default False, (readonly) 
    '''

    image_settings = None
    '''

    :type:  ImageFormatSettings, (readonly, never None) 
    '''

    is_movie_format = None
    '''When true the format is a movie 

    :type:  boolean, default False, (readonly) 
    '''

    layers = None
    '''

    :type:  RenderLayers bpy_prop_collection of SceneRenderLayer, (readonly) 
    '''

    line_thickness = None
    '''Line thickness in pixels 

    :type:  float in [0, 10000], default 0.0 
    '''

    line_thickness_mode = None
    '''Line thickness mode for Freestyle line drawing 

    :type:  enum in [‘ABSOLUTE’, ‘RELATIVE’], default ‘ABSOLUTE’ 
    '''

    motion_blur_samples = None
    '''Number of scene samples to take with motion blur 

    :type:  int in [1, 32], default 0 
    '''

    motion_blur_shutter = None
    '''Time taken in frames between shutter open and close 

    :type:  float in [0, inf], default 0.0 
    '''

    motion_blur_shutter_curve = None
    '''Curve defining the shutter’s openness over time 

    :type:  CurveMapping, (readonly) 
    '''

    octree_resolution = None
    '''Resolution of raytrace accelerator, use higher resolutions for larger scenes 

    :type:  enum in [‘64’, ‘128’, ‘256’, ‘512’], default ‘64’ 
    '''

    pixel_aspect_x = None
    '''Horizontal aspect ratio - for anamorphic or non-square pixel output 

    :type:  float in [1, 200], default 0.0 
    '''

    pixel_aspect_y = None
    '''Vertical aspect ratio - for anamorphic or non-square pixel output 

    :type:  float in [1, 200], default 0.0 
    '''

    pixel_filter_type = None
    '''Reconstruction filter used for combining anti-aliasing samples 

    :type:  enum in [‘BOX’, ‘TENT’, ‘QUADRATIC’, ‘CUBIC’, ‘CATMULLROM’, ‘GAUSSIAN’, ‘MITCHELL’], default ‘BOX’ 
    '''

    preview_start_resolution = None
    '''Resolution to start rendering preview at, progressively increasing it to the full viewport size 

    :type:  int in [8, 16384], default 64 
    '''

    raytrace_method = None
    '''Type of raytrace accelerator structure 

    :type:  enum in [‘AUTO’, ‘OCTREE’, ‘VBVH’, ‘SIMD_SVBVH’, ‘SIMD_QBVH’], default ‘AUTO’ 
    '''

    resolution_percentage = None
    '''Percentage scale for render resolution 

    :type:  int in [1, 32767], default 0 
    '''

    resolution_x = None
    '''Number of horizontal pixels in the rendered image 

    :type:  int in [4, 65536], default 0 
    '''

    resolution_y = None
    '''Number of vertical pixels in the rendered image 

    :type:  int in [4, 65536], default 0 
    '''

    sequencer_gl_preview = None
    '''Method to draw in the sequencer view 

    :type:  enum in [‘BOUNDBOX’, ‘WIREFRAME’, ‘SOLID’, ‘TEXTURED’, ‘MATERIAL’, ‘RENDERED’], default ‘BOUNDBOX’ 
    '''

    sequencer_gl_render = None
    '''Method to draw in the sequencer view 

    :type:  enum in [‘BOUNDBOX’, ‘WIREFRAME’, ‘SOLID’, ‘TEXTURED’, ‘MATERIAL’, ‘RENDERED’], default ‘BOUNDBOX’ 
    '''

    simplify_ao_sss = None
    '''Global approximate AO and SSS quality factor 

    :type:  float in [0, 1], default 0.0 
    '''

    simplify_child_particles = None
    '''Global child particles percentage 

    :type:  float in [0, 1], default 0.0 
    '''

    simplify_child_particles_render = None
    '''Global child particles percentage during rendering 

    :type:  float in [0, 1], default 0.0 
    '''

    simplify_shadow_samples = None
    '''Global maximum shadow samples 

    :type:  int in [0, 32767], default 0 
    '''

    simplify_subdivision = None
    '''Global maximum subdivision level 

    :type:  int in [0, 32767], default 0 
    '''

    simplify_subdivision_render = None
    '''Global maximum subdivision level during rendering 

    :type:  int in [0, 32767], default 0 
    '''

    stamp_background = None
    '''Color to use behind stamp text 

    :type:  float array of 4 items in [0, 1], default (0.0, 0.0, 0.0, 0.0) 
    '''

    stamp_font_size = None
    '''Size of the font used when rendering stamp text 

    :type:  int in [8, 64], default 0 
    '''

    stamp_foreground = None
    '''Color to use for stamp text 

    :type:  float array of 4 items in [0, 1], default (0.0, 0.0, 0.0, 0.0) 
    '''

    stamp_note_text = None
    '''Custom text to appear in the stamp note 

    :type:  string, default “”, (never None) 
    '''

    stereo_views = None
    '''

    :type:  bpy_prop_collection of SceneRenderView, (readonly) 
    '''

    threads = None
    '''Number of CPU threads to use simultaneously while rendering (for multi-core/CPU systems) 

    :type:  int in [1, 1024], default 0 
    '''

    threads_mode = None
    '''Determine the amount of render threads used 

    :type:  enum in [‘AUTO’, ‘FIXED’], default ‘AUTO’ 
    '''

    tile_x = None
    '''Horizontal tile size to use while rendering 

    :type:  int in [8, 65536], default 0 
    '''

    tile_y = None
    '''Vertical tile size to use while rendering 

    :type:  int in [8, 65536], default 0 
    '''

    use_antialiasing = None
    '''Render and combine multiple samples per pixel to prevent jagged edges 

    :type:  boolean, default False 
    '''

    use_bake_antialiasing = None
    '''Enables Anti-aliasing 

    :type:  boolean, default False 
    '''

    use_bake_clear = None
    '''Clear Images before baking 

    :type:  boolean, default False 
    '''

    use_bake_lores_mesh = None
    '''Calculate heights against unsubdivided low resolution mesh 

    :type:  boolean, default False 
    '''

    use_bake_multires = None
    '''Bake directly from multires object 

    :type:  boolean, default False 
    '''

    use_bake_normalize = None
    '''With displacement normalize to the distance, with ambient occlusion normalize without using material settings 

    :type:  boolean, default False 
    '''

    use_bake_selected_to_active = None
    '''Bake shading on the surface of selected objects to the active object 

    :type:  boolean, default False 
    '''

    use_bake_to_vertex_color = None
    '''Bake to vertex colors instead of to a UV-mapped image 

    :type:  boolean, default False 
    '''

    use_bake_user_scale = None
    '''Use a user scale for the derivative map 

    :type:  boolean, default False 
    '''

    use_border = None
    '''Render a user-defined border region, within the frame size (note that this disables save_buffers and full_sample) 

    :type:  boolean, default False 
    '''

    use_compositing = None
    '''Process the render result through the compositing pipeline, if compositing nodes are enabled 

    :type:  boolean, default False 
    '''

    use_crop_to_border = None
    '''Crop the rendered frame to the defined border size 

    :type:  boolean, default False 
    '''

    use_edge_enhance = None
    '''Create a toon outline around the edges of geometry 

    :type:  boolean, default False 
    '''

    use_envmaps = None
    '''Calculate environment maps while rendering 

    :type:  boolean, default False 
    '''

    use_fields = None
    '''Render image to two fields per frame, for interlaced TV output 

    :type:  boolean, default False 
    '''

    use_fields_still = None
    '''Disable the time difference between fields 

    :type:  boolean, default False 
    '''

    use_file_extension = None
    '''Add the file format extensions to the rendered file name (eg: filename + .jpg) 

    :type:  boolean, default False 
    '''

    use_free_image_textures = None
    '''Free all image textures from memory after render, to save memory before compositing 

    :type:  boolean, default False 
    '''

    use_freestyle = None
    '''Draw stylized strokes using Freestyle 

    :type:  boolean, default False 
    '''

    use_full_sample = None
    '''Save for every anti-aliasing sample the entire RenderLayer results (this solves anti-aliasing issues with compositing) 

    :type:  boolean, default False 
    '''

    use_game_engine = None
    '''Current rendering engine is a game engine 

    :type:  boolean, default False, (readonly) 
    '''

    use_instances = None
    '''Instance support leads to effective memory reduction when using duplicates 

    :type:  boolean, default False 
    '''

    use_local_coords = None
    '''Vertex coordinates are stored locally on each primitive (increases memory usage, but may have impact on speed) 

    :type:  boolean, default False 
    '''

    use_lock_interface = None
    '''Lock interface during rendering in favor of giving more memory to the renderer 

    :type:  boolean, default False 
    '''

    use_motion_blur = None
    '''Use multi-sampled 3D scene motion blur 

    :type:  boolean, default False 
    '''

    use_multiview = None
    '''Use multiple views in the scene 

    :type:  boolean, default False 
    '''

    use_overwrite = None
    '''Overwrite existing files while rendering 

    :type:  boolean, default False 
    '''

    use_persistent_data = None
    '''Keep render data around for faster re-renders 

    :type:  boolean, default False 
    '''

    use_placeholder = None
    '''Create empty placeholder files while rendering frames (similar to Unix ‘touch’) 

    :type:  boolean, default False 
    '''

    use_raytrace = None
    '''Pre-calculate the raytrace accelerator and render raytracing effects 

    :type:  boolean, default False 
    '''

    use_render_cache = None
    '''Save render cache to EXR files (useful for heavy compositing, Note: affects indirectly rendered scenes) 

    :type:  boolean, default False 
    '''

    use_save_buffers = None
    '''Save tiles for all RenderLayers and SceneNodes to files in the temp directory (saves memory, required for Full Sample) 

    :type:  boolean, default False 
    '''

    use_sequencer = None
    '''Process the render (and composited) result through the video sequence editor pipeline, if sequencer strips exist 

    :type:  boolean, default False 
    '''

    use_sequencer_gl_preview = None
    '''

    :type:  boolean, default False 
    '''

    use_sequencer_gl_textured_solid = None
    '''Draw face-assigned textures in solid draw method 

    :type:  boolean, default False 
    '''

    use_shading_nodes = None
    '''Active render engine uses new shading nodes system 

    :type:  boolean, default False, (readonly) 
    '''

    use_shadows = None
    '''Calculate shadows while rendering 

    :type:  boolean, default False 
    '''

    use_simplify = None
    '''Enable simplification of scene for quicker preview renders 

    :type:  boolean, default False 
    '''

    use_simplify_triangulate = None
    '''Disable non-planar quads being triangulated 

    :type:  boolean, default False 
    '''

    use_single_layer = None
    '''Only render the active layer 

    :type:  boolean, default False 
    '''

    use_spherical_stereo = None
    '''Active render engine supports spherical stereo rendering 

    :type:  boolean, default False, (readonly) 
    '''

    use_sss = None
    '''Calculate sub-surface scattering in materials rendering 

    :type:  boolean, default False 
    '''

    use_stamp = None
    '''Render the stamp info text in the rendered image 

    :type:  boolean, default False 
    '''

    use_stamp_camera = None
    '''Include the name of the active camera in image metadata 

    :type:  boolean, default False 
    '''

    use_stamp_date = None
    '''Include the current date in image metadata 

    :type:  boolean, default False 
    '''

    use_stamp_filename = None
    '''Include the .blend filename in image metadata 

    :type:  boolean, default False 
    '''

    use_stamp_frame = None
    '''Include the frame number in image metadata 

    :type:  boolean, default False 
    '''

    use_stamp_labels = None
    '''Draw stamp labels (“Camera” in front of camera name, etc.) 

    :type:  boolean, default False 
    '''

    use_stamp_lens = None
    '''Include the active camera’s lens in image metadata 

    :type:  boolean, default False 
    '''

    use_stamp_marker = None
    '''Include the name of the last marker in image metadata 

    :type:  boolean, default False 
    '''

    use_stamp_memory = None
    '''Include the peak memory usage in image metadata 

    :type:  boolean, default False 
    '''

    use_stamp_note = None
    '''Include a custom note in image metadata 

    :type:  boolean, default False 
    '''

    use_stamp_render_time = None
    '''Include the render time in image metadata 

    :type:  boolean, default False 
    '''

    use_stamp_scene = None
    '''Include the name of the active scene in image metadata 

    :type:  boolean, default False 
    '''

    use_stamp_sequencer_strip = None
    '''Include the name of the foreground sequence strip in image metadata 

    :type:  boolean, default False 
    '''

    use_stamp_strip_meta = None
    '''Use metadata from the strips in the sequencer 

    :type:  boolean, default False 
    '''

    use_stamp_time = None
    '''Include the rendered frame timecode as HH:MM:SS.FF in image metadata 

    :type:  boolean, default False 
    '''

    use_textures = None
    '''Use textures to affect material properties 

    :type:  boolean, default False 
    '''

    use_world_space_shading = None
    '''Use world space interpretation of lighting data for node materials 

    :type:  boolean, default False 
    '''

    views = None
    '''

    :type:  RenderViews bpy_prop_collection of SceneRenderView, (readonly) 
    '''

    views_format = None
    '''

    :type:  enum in [‘STEREO_3D’, ‘MULTIVIEW’], default ‘STEREO_3D’ 
    '''

    def frame_path(self, frame=-2147483648, preview=False, view=""):
        '''Return the absolute path to the filename to be written for a given frame 

        :param frame: Frame number to use, if unset the current frame will be used 
        :type frame: int in [-inf, inf], (optional)
        :param preview: Preview, Use preview range 
        :type preview: boolean, (optional)
        :param view: View, The name of the view to use to replace the “%” chars 
        :type view: string, (optional, never None)
        :rtype: string, (never None) 
        :return:  File Path, The resulting filepath from the scenes render settings 
        '''
        pass


class RenderSlot:
    '''Parameters defining the render slot '''

    name = None
    '''Render slot name 

    :type:  string, default “”, (never None) 
    '''


class RenderSlots:
    '''Collection of the render slots '''

    active = None
    '''Active render slot of the image 

    :type:  RenderSlot 
    '''

    active_index = None
    '''Index of an active render slot of the image 

    :type:  int in [0, 8], default 0 
    '''


class RenderView:
    name = None
    '''

    :type:  string, default “”, (readonly, never None) 
    '''


class RenderViews:
    '''Collection of render views '''

    active = None
    '''Active Render View 

    :type:  SceneRenderView, (never None) 
    '''

    active_index = None
    '''Active index in render view array 

    :type:  int in [0, 32767], default 0 
    '''

    def new(self, name):
        '''Add a render view to scene 

        :param name: New name for the marker (not unique) 
        :type name: string, (never None)
        :rtype: SceneRenderView 
        :return:  Newly created render view 
        '''
        pass

    def remove(self, view):
        '''Remove a render view 

        :param view: Render view to remove 
        :type view: SceneRenderView, (never None)
        '''
        pass


class RigidBodyConstraint:
    '''Constraint influencing Objects inside Rigid Body Simulation '''

    breaking_threshold = None
    '''Impulse threshold that must be reached for the constraint to break 

    :type:  float in [0, inf], default 10.0 
    '''

    disable_collisions = None
    '''Disable collisions between constrained rigid bodies 

    :type:  boolean, default False 
    '''

    enabled = None
    '''Enable this constraint 

    :type:  boolean, default False 
    '''

    limit_ang_x_lower = None
    '''Lower limit of X axis rotation 

    :type:  float in [-6.28319, 6.28319], default -0.785398 
    '''

    limit_ang_x_upper = None
    '''Upper limit of X axis rotation 

    :type:  float in [-6.28319, 6.28319], default 0.785398 
    '''

    limit_ang_y_lower = None
    '''Lower limit of Y axis rotation 

    :type:  float in [-6.28319, 6.28319], default -0.785398 
    '''

    limit_ang_y_upper = None
    '''Upper limit of Y axis rotation 

    :type:  float in [-6.28319, 6.28319], default 0.785398 
    '''

    limit_ang_z_lower = None
    '''Lower limit of Z axis rotation 

    :type:  float in [-6.28319, 6.28319], default -0.785398 
    '''

    limit_ang_z_upper = None
    '''Upper limit of Z axis rotation 

    :type:  float in [-6.28319, 6.28319], default 0.785398 
    '''

    limit_lin_x_lower = None
    '''Lower limit of X axis translation 

    :type:  float in [-inf, inf], default -1 
    '''

    limit_lin_x_upper = None
    '''Upper limit of X axis translation 

    :type:  float in [-inf, inf], default 1.0 
    '''

    limit_lin_y_lower = None
    '''Lower limit of Y axis translation 

    :type:  float in [-inf, inf], default -1 
    '''

    limit_lin_y_upper = None
    '''Upper limit of Y axis translation 

    :type:  float in [-inf, inf], default 1.0 
    '''

    limit_lin_z_lower = None
    '''Lower limit of Z axis translation 

    :type:  float in [-inf, inf], default -1 
    '''

    limit_lin_z_upper = None
    '''Upper limit of Z axis translation 

    :type:  float in [-inf, inf], default 1.0 
    '''

    motor_ang_max_impulse = None
    '''Maximum angular motor impulse 

    :type:  float in [0, inf], default 1.0 
    '''

    motor_ang_target_velocity = None
    '''Target angular motor velocity 

    :type:  float in [-inf, inf], default 1.0 
    '''

    motor_lin_max_impulse = None
    '''Maximum linear motor impulse 

    :type:  float in [0, inf], default 1.0 
    '''

    motor_lin_target_velocity = None
    '''Target linear motor velocity 

    :type:  float in [-inf, inf], default 1.0 
    '''

    object1 = None
    '''First Rigid Body Object to be constrained 

    :type:  Object 
    '''

    object2 = None
    '''Second Rigid Body Object to be constrained 

    :type:  Object 
    '''

    solver_iterations = None
    '''Number of constraint solver iterations made per simulation step (higher values are more accurate but slower) 

    :type:  int in [1, 1000], default 10 
    '''

    spring_damping_x = None
    '''Damping on the X axis 

    :type:  float in [0, 1], default 0.5 
    '''

    spring_damping_y = None
    '''Damping on the Y axis 

    :type:  float in [0, 1], default 0.5 
    '''

    spring_damping_z = None
    '''Damping on the Z axis 

    :type:  float in [0, 1], default 0.5 
    '''

    spring_stiffness_x = None
    '''Stiffness on the X axis 

    :type:  float in [0, inf], default 10.0 
    '''

    spring_stiffness_y = None
    '''Stiffness on the Y axis 

    :type:  float in [0, inf], default 10.0 
    '''

    spring_stiffness_z = None
    '''Stiffness on the Z axis 

    :type:  float in [0, inf], default 10.0 
    '''

    type = None
    '''Type of Rigid Body Constraint 

    :type:  enum in [‘FIXED’, ‘POINT’, ‘HINGE’, ‘SLIDER’, ‘PISTON’, ‘GENERIC’, ‘GENERIC_SPRING’, ‘MOTOR’], default ‘POINT’ 
    '''

    use_breaking = None
    '''Constraint can be broken if it receives an impulse above the threshold 

    :type:  boolean, default False 
    '''

    use_limit_ang_x = None
    '''Limit rotation around X axis 

    :type:  boolean, default False 
    '''

    use_limit_ang_y = None
    '''Limit rotation around Y axis 

    :type:  boolean, default False 
    '''

    use_limit_ang_z = None
    '''Limit rotation around Z axis 

    :type:  boolean, default False 
    '''

    use_limit_lin_x = None
    '''Limit translation on X axis 

    :type:  boolean, default False 
    '''

    use_limit_lin_y = None
    '''Limit translation on Y axis 

    :type:  boolean, default False 
    '''

    use_limit_lin_z = None
    '''Limit translation on Z axis 

    :type:  boolean, default False 
    '''

    use_motor_ang = None
    '''Enable angular motor 

    :type:  boolean, default False 
    '''

    use_motor_lin = None
    '''Enable linear motor 

    :type:  boolean, default False 
    '''

    use_override_solver_iterations = None
    '''Override the number of solver iterations for this constraint 

    :type:  boolean, default False 
    '''

    use_spring_x = None
    '''Enable spring on X axis 

    :type:  boolean, default False 
    '''

    use_spring_y = None
    '''Enable spring on Y axis 

    :type:  boolean, default False 
    '''

    use_spring_z = None
    '''Enable spring on Z axis 

    :type:  boolean, default False 
    '''


class RigidBodyJointConstraint:
    '''For use with the Game Engine '''

    axis_x = None
    '''Rotate pivot on X axis 

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    axis_y = None
    '''Rotate pivot on Y axis 

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    axis_z = None
    '''Rotate pivot on Z axis 

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    child = None
    '''Child object 

    :type:  Object 
    '''

    limit_angle_max_x = None
    '''

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    limit_angle_max_y = None
    '''

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    limit_angle_max_z = None
    '''

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    limit_angle_min_x = None
    '''

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    limit_angle_min_y = None
    '''

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    limit_angle_min_z = None
    '''

    :type:  float in [-6.28319, 6.28319], default 0.0 
    '''

    limit_max_x = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    limit_max_y = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    limit_max_z = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    limit_min_x = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    limit_min_y = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    limit_min_z = None
    '''

    :type:  float in [-inf, inf], default 0.0 
    '''

    pivot_type = None
    '''

    :type:  enum in [‘BALL’, ‘HINGE’, ‘CONE_TWIST’, ‘GENERIC_6_DOF’], default ‘BALL’ 
    '''

    pivot_x = None
    '''Offset pivot on X 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    pivot_y = None
    '''Offset pivot on Y 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    pivot_z = None
    '''Offset pivot on Z 

    :type:  float in [-1000, 1000], default 0.0 
    '''

    show_pivot = None
    '''Display the pivot point and rotation in 3D view 

    :type:  boolean, default False 
    '''

    target = None
    '''Target Object 

    :type:  Object 
    '''

    use_angular_limit_x = None
    '''Use minimum/maximum X angular limit 

    :type:  boolean, default False 
    '''

    use_angular_limit_y = None
    '''Use minimum/maximum Y angular limit 

    :type:  boolean, default False 
    '''

    use_angular_limit_z = None
    '''Use minimum/maximum Z angular limit 

    :type:  boolean, default False 
    '''

    use_limit_x = None
    '''Use minimum/maximum X limit 

    :type:  boolean, default False 
    '''

    use_limit_y = None
    '''Use minimum/maximum y limit 

    :type:  boolean, default False 
    '''

    use_limit_z = None
    '''Use minimum/maximum z limit 

    :type:  boolean, default False 
    '''

    use_linked_collision = None
    '''Disable collision between linked bodies 

    :type:  boolean, default False 
    '''


class RigidBodyObject:
    '''Settings for object participating in Rigid Body Simulation '''

    angular_damping = None
    '''Amount of angular velocity that is lost over time 

    :type:  float in [0, 1], default 0.1 
    '''

    collision_groups = None
    '''Collision Groups Rigid Body belongs to 

    :type:  boolean array of 20 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    collision_margin = None
    '''Threshold of distance near surface where collisions are still considered (best results when non-zero) 

    :type:  float in [0, 1], default 0.04 
    '''

    collision_shape = None
    '''Collision Shape of object in Rigid Body Simulations 

    :type:  enum in [‘BOX’, ‘SPHERE’, ‘CAPSULE’, ‘CYLINDER’, ‘CONE’, ‘CONVEX_HULL’, ‘MESH’], default ‘BOX’ 
    '''

    deactivate_angular_velocity = None
    '''Angular Velocity below which simulation stops simulating object 

    :type:  float in [0, inf], default 0.5 
    '''

    deactivate_linear_velocity = None
    '''Linear Velocity below which simulation stops simulating object 

    :type:  float in [0, inf], default 0.4 
    '''

    enabled = None
    '''Rigid Body actively participates to the simulation 

    :type:  boolean, default False 
    '''

    friction = None
    '''Resistance of object to movement 

    :type:  float in [0, inf], default 0.5 
    '''

    kinematic = None
    '''Allow rigid body to be controlled by the animation system 

    :type:  boolean, default False 
    '''

    linear_damping = None
    '''Amount of linear velocity that is lost over time 

    :type:  float in [0, 1], default 0.04 
    '''

    mass = None
    '''How much the object ‘weighs’ irrespective of gravity 

    :type:  float in [0.001, inf], default 1.0 
    '''

    mesh_source = None
    '''Source of the mesh used to create collision shape 

    :type:  enum in [‘BASE’, ‘DEFORM’, ‘FINAL’], default ‘BASE’ 
    '''

    restitution = None
    '''Tendency of object to bounce after colliding with another (0 = stays still, 1 = perfectly elastic) 

    :type:  float in [0, inf], default 0.0 
    '''

    type = None
    '''Role of object in Rigid Body Simulations 

    :type:  enum in [‘ACTIVE’, ‘PASSIVE’], default ‘ACTIVE’ 
    '''

    use_deactivation = None
    '''Enable deactivation of resting rigid bodies (increases performance and stability but can cause glitches) 

    :type:  boolean, default True 
    '''

    use_deform = None
    '''Rigid body deforms during simulation 

    :type:  boolean, default False 
    '''

    use_margin = None
    '''Use custom collision margin (some shapes will have a visible gap around them) 

    :type:  boolean, default False 
    '''

    use_start_deactivated = None
    '''Deactivate rigid body at the start of the simulation 

    :type:  boolean, default False 
    '''


class RigidBodyWorld:
    '''Self-contained rigid body simulation environment and settings '''

    constraints = None
    '''Group containing rigid body constraint objects 

    :type:  Group 
    '''

    effector_weights = None
    '''

    :type:  EffectorWeights, (readonly) 
    '''

    enabled = None
    '''Simulation will be evaluated 

    :type:  boolean, default False 
    '''

    group = None
    '''Group containing objects participating in this simulation 

    :type:  Group 
    '''

    point_cache = None
    '''

    :type:  PointCache, (readonly, never None) 
    '''

    solver_iterations = None
    '''Number of constraint solver iterations made per simulation step (higher values are more accurate but slower) 

    :type:  int in [1, 1000], default 10 
    '''

    steps_per_second = None
    '''Number of simulation steps taken per second (higher values are more accurate but slower) 

    :type:  int in [1, 32767], default 60 
    '''

    time_scale = None
    '''Change the speed of the simulation 

    :type:  float in [0, 100], default 1.0 
    '''

    use_split_impulse = None
    '''Reduce extra velocity that can build up when objects collide (lowers simulation stability a little so use only when necessary) 

    :type:  boolean, default False 
    '''

    def convex_sweep_test(self, object, start, end):
        '''Sweep test convex rigidbody against the current rigidbody world 

        :param object: Rigidbody object with a convex collision shape 
        :type object: Object, (never None)
        '''
        pass


class Rotation:
    pass


class SCENE_UL_keying_set_paths:
    def draw_item(self, context, layout, data, item, icon, active_data,
                  active_propname, index):
        '''

        '''
        pass


class SPHFluidSettings:
    '''Settings for particle fluids physics '''

    buoyancy = None
    '''Artificial buoyancy force in negative gravity direction based on pressure differences inside the fluid 

    :type:  float in [0, 10], default 0.0 
    '''

    factor_radius = None
    '''Interaction radius is a factor of 4 * particle size 

    :type:  boolean, default False 
    '''

    factor_repulsion = None
    '''Repulsion is a factor of stiffness 

    :type:  boolean, default False 
    '''

    factor_rest_length = None
    '''Spring rest length is a factor of 2 * particle size 

    :type:  boolean, default False 
    '''

    factor_stiff_viscosity = None
    '''Stiff viscosity is a factor of normal viscosity 

    :type:  boolean, default False 
    '''

    fluid_radius = None
    '''Fluid interaction radius 

    :type:  float in [0, 20], default 0.0 
    '''

    linear_viscosity = None
    '''Linear viscosity 

    :type:  float in [0, 100], default 0.0 
    '''

    plasticity = None
    '''How much the spring rest length can change after the elastic limit is crossed 

    :type:  float in [0, 100], default 0.0 
    '''

    repulsion = None
    '''How strongly the fluid tries to keep from clustering (factor of stiffness) 

    :type:  float in [0, 100], default 0.0 
    '''

    rest_density = None
    '''Fluid rest density 

    :type:  float in [0, 10000], default 0.0 
    '''

    rest_length = None
    '''Spring rest length (factor of particle radius) 

    :type:  float in [0, 2], default 0.0 
    '''

    solver = None
    '''The code used to calculate internal forces on particles 

    :type:  enum in [‘DDR’, ‘CLASSICAL’], default ‘DDR’ 
    '''

    spring_force = None
    '''Spring force 

    :type:  float in [0, 100], default 0.0 
    '''

    spring_frames = None
    '''Create springs for this number of frames since particles birth (0 is always) 

    :type:  int in [0, 100], default 0 
    '''

    stiff_viscosity = None
    '''Creates viscosity for expanding fluid 

    :type:  float in [0, 100], default 0.0 
    '''

    stiffness = None
    '''How incompressible the fluid is (speed of sound) 

    :type:  float in [0, 1000], default 0.0 
    '''

    use_factor_density = None
    '''Density is calculated as a factor of default density (depends on particle size) 

    :type:  boolean, default False 
    '''

    use_initial_rest_length = None
    '''Use the initial length as spring rest length instead of 2 * particle size 

    :type:  boolean, default False 
    '''

    use_viscoelastic_springs = None
    '''Use viscoelastic springs instead of Hooke’s springs 

    :type:  boolean, default False 
    '''

    yield_ratio = None
    '''How much the spring has to be stretched/compressed in order to change it’s rest length 

    :type:  float in [0, 1], default 0.0 
    '''


class Scaling:
    pass


class Scene:
    '''Scene data block, consisting in objects and defining time and render related settings '''

    active_clip = None
    '''Active movie clip used for constraints and viewport drawing 

    :type:  MovieClip 
    '''

    active_layer = None
    '''Active scene layer index 

    :type:  int in [-inf, inf], default 0, (readonly) 
    '''

    animation_data = None
    '''Animation data for this data-block 

    :type:  AnimData, (readonly) 
    '''

    audio_distance_model = None
    '''Distance model for distance attenuation calculation 

    :type:  enum in [‘NONE’, ‘INVERSE’, ‘INVERSE_CLAMPED’, ‘LINEAR’, ‘LINEAR_CLAMPED’, ‘EXPONENT’, ‘EXPONENT_CLAMPED’], default ‘NONE’ 
    '''

    audio_doppler_factor = None
    '''Pitch factor for Doppler effect calculation 

    :type:  float in [0, inf], default 0.0 
    '''

    audio_doppler_speed = None
    '''Speed of sound for Doppler effect calculation 

    :type:  float in [0.01, inf], default 0.0 
    '''

    audio_volume = None
    '''Audio volume 

    :type:  float in [0, 100], default 0.0 
    '''

    background_set = None
    '''Background set scene 

    :type:  Scene 
    '''

    camera = None
    '''Active camera, used for rendering the scene 

    :type:  Object 
    '''

    cursor_location = None
    '''3D cursor location 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    cycles = None
    '''Cycles render settings 

    :type:  CyclesRenderSettings, (readonly) 
    '''

    cycles_curves = None
    '''Cycles hair rendering settings 

    :type:  CyclesCurveRenderSettings, (readonly) 
    '''

    depsgraph = None
    '''Dependencies in the scene data 

    :type:  Depsgraph, (readonly) 
    '''

    display_settings = None
    '''Settings of device saved image would be displayed on 

    :type:  ColorManagedDisplaySettings, (readonly) 
    '''

    frame_current = None
    '''Current Frame, to update animation data from python frame_set() instead 

    :type:  int in [-500000, 500000], default 0 
    '''

    frame_current_final = None
    '''Current frame with subframe and time remapping applied 

    :type:  float in [-500000, 500000], default 0.0, (readonly) 
    '''

    frame_end = None
    '''Final frame of the playback/rendering range 

    :type:  int in [0, 500000], default 0 
    '''

    frame_preview_end = None
    '''Alternative end frame for UI playback 

    :type:  int in [-inf, inf], default 0 
    '''

    frame_preview_start = None
    '''Alternative start frame for UI playback 

    :type:  int in [-inf, inf], default 0 
    '''

    frame_start = None
    '''First frame of the playback/rendering range 

    :type:  int in [0, 500000], default 0 
    '''

    frame_step = None
    '''Number of frames to skip forward while rendering/playing back each frame 

    :type:  int in [0, 500000], default 0 
    '''

    frame_subframe = None
    '''

    :type:  float in [-inf, inf], default 0.0, (readonly) 
    '''

    game_settings = None
    '''

    :type:  SceneGameData, (readonly, never None) 
    '''

    gravity = None
    '''Constant acceleration in a given direction 

    :type:  float array of 3 items in [-inf, inf], default (0.0, 0.0, 0.0) 
    '''

    grease_pencil = None
    '''Grease Pencil data-block 

    :type:  GreasePencil 
    '''

    is_nla_tweakmode = None
    '''Whether there is any action referenced by NLA being edited (strictly read-only) 

    :type:  boolean, default False, (readonly) 
    '''

    keying_sets = None
    '''Absolute Keying Sets for this Scene 

    :type:  KeyingSets bpy_prop_collection of KeyingSet, (readonly) 
    '''

    keying_sets_all = None
    '''All Keying Sets available for use (Builtins and Absolute Keying Sets for this Scene) 

    :type:  KeyingSetsAll bpy_prop_collection of KeyingSet, (readonly) 
    '''

    layers = None
    '''Visible layers - Shift-Click/Drag to select multiple layers 

    :type:  boolean array of 20 items, default (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False) 
    '''

    lock_frame_selection_to_range = None
    '''Don’t allow frame to be selected with mouse outside of frame range 

    :type:  boolean, default False 
    '''

    node_tree = None
    '''Compositing node tree 

    :type:  NodeTree, (readonly) 
    '''

    object_bases = None
    '''

    :type:  SceneBases bpy_prop_collection of ObjectBase, (readonly) 
    '''

    objects = None
    '''

    :type:  SceneObjects bpy_prop_collection of Object, (readonly) 
    '''

    orientations = None
    '''

    :type:  bpy_prop_collection of TransformOrientation, (readonly) 
    '''

    render = None
    '''

    :type:  RenderSettings, (readonly, never None) 
    '''

    rigidbody_world = None
    '''

    :type:  RigidBodyWorld, (readonly) 
    '''

    safe_areas = None
    '''

    :type:  DisplaySafeAreas, (readonly, never None) 
    '''

    sequence_editor = None
    '''

    :type:  SequenceEditor, (readonly) 
    '''

    sequencer_colorspace_settings = None
    '''Settings of color space sequencer is working in 

    :type:  ColorManagedSequencerColorspaceSettings, (readonly) 
    '''

    show_keys_from_selected_only = None
    '''Consider keyframes for active Object and/or its selected bones only (in timeline and when jumping between keyframes) 

    :type:  boolean, default False 
    '''

    sync_mode = None
    '''How to sync playback 

    :type:  enum in [‘NONE’, ‘FRAME_DROP’, ‘AUDIO_SYNC’], default ‘NONE’ 
    '''

    timeline_markers = None
    '''Markers used in all timelines for the current scene 

    :type:  TimelineMarkers bpy_prop_collection of TimelineMarker, (readonly) 
    '''

    tool_settings = None
    '''

    :type:  ToolSettings, (readonly, never None) 
    '''

    unit_settings = None
    '''Unit editing settings 

    :type:  UnitSettings, (readonly, never None) 
    '''

    use_audio = None
    '''Play back of audio from Sequence Editor will be muted 

    :type:  boolean, default False 
    '''

    use_audio_scrub = None
    '''Play audio from Sequence Editor while scrubbing 

    :type:  boolean, default False 
    '''

    use_audio_sync = None
    '''Play back and sync with audio clock, dropping frames if frame display is too slow 

    :type:  boolean, default False 
    '''

    use_frame_drop = None
    '''Play back dropping frames if frame display is too slow 

    :type:  boolean, default False 
    '''

    use_gravity = None
    '''Use global gravity for all dynamics 

    :type:  boolean, default False 
    '''

    use_nodes = None
    '''Enable the compositing node tree 

    :type:  boolean, default False 
    '''

    use_preview_range = None
    '''Use an alternative start/end frame range for animation playback and OpenGL renders instead of the Render properties start/end frame range 

    :type:  boolean, default False 
    '''

    use_stamp_note = None
    '''User defined note for the render stamping 

    :type:  string, default “”, (never None) 
    '''

    view_settings = None
    '''Color management settings applied on image before saving 

    :type:  ColorManagedViewSettings, (readonly) 
    '''

    world = None
    '''World used for rendering the scene 

    :type:  World 
    '''

    def sequence_editor_create(self):
        '''Ensure sequence editor is valid in this scene 

        :rtype: SequenceEditor 
        :return:  New sequence editor data or NULL 
        '''
        pass

    def sequence_editor_clear(self):
        '''Clear sequence editor in this scene 

        '''
        pass

    def statistics(self):
        '''statistics 

        :rtype: string, (never None) 
        :return:  Statistics 
        '''
        pass

    def frame_set(self, frame, subframe=0.0):
        '''Set scene frame updating all objects immediately 

        :param frame: Frame number to set 
        :type frame: int in [-500000, 500000]
        :param subframe: Sub-frame time, between 0.0 and 1.0 
        :type subframe: float in [0, 1], (optional)
        '''
        pass

    def update(self):
        '''Update data tagged to be updated from previous access to data or operators 

        '''
        pass

    def uvedit_aspect(self, object):
        '''Get uv a